/* **COPYRIGHT******************************************************************
    INTEL CONFIDENTIAL
    Copyright (C) 2017 Intel Corporation
    Copyright (C), 1994-2004 Aware Inc. All Rights Reserved.
******************************************************************COPYRIGHT** */
/* **DISCLAIMER*****************************************************************
    The source code contained or described herein and all documents related
    to the source code ("Material") are owned by Intel Corporation or its
    suppliers or licensors. Title to the Material remains with Intel
    Corporation or its suppliers and licensors. The Material may contain
    trade secrets and proprietary and confidential information of Intel
    Corporation and its suppliers and licensors, and is protected by
    worldwide copyright and trade secret laws and treaty provisions. No part
    of the Material may be used, copied, reproduced, modified, published,
    uploaded, posted, transmitted, distributed, or disclosed in any way
    without Intel's prior express written permission.

    No license under any patent, copyright, trade secret or other
    intellectual property right is granted to or conferred upon you by
    disclosure or delivery of the Materials, either expressly, by
    implication, inducement, estoppel or otherwise. Any license under
    such intellectual property rights must be express and approved by
    Intel in writing.
*****************************************************************DISCLAIMER** */
/*
 *------------------------------------------------------------------------
 *
 *   Aware DMT Technology. Proprietary and Confidential.
 *
 *   40 Middlesex Turnpike, Bedford, MA 01730-1413 USA
 *   Phone (781) 276 - 4000
 *   Fax   (781) 276 - 4001
 *
 *   RxSymbolReAlign.c
 *
 *
 *------------------------------------------------------------------------
 */
// ***********************************************************************************************************
// RxSymbolReAlign.c
//
// History
//
//
// 24/09/2019 Chih-Wen
//            Bug fix for several issues related to V43 and T1413 setup.
//            SEARCH PATTERN: XDSLRTFW-4227
//
// ***********************************************************************************************************
#include <string.h>
#include "common.h"
#include "vdsl_state.h"
#include "gdata.h"
#include "ghs.h"
#include "rx_ops.h"
#include "states.h"
#include "fifo.h"
#include "ghs.h"
#include "ghs_cpe.h"
#include "mul.h"
#include "cmv.h"
#include "IRI_Iof.h"

/*^^^
 *------------------------------------------------------------------------
 *
 *  Name : GetSymbol()
 *
 *  Abstract :
 *
 *  Extract symbol data from designated Rx Handshake carrier tones.
 *                     .
 *
 *  Returns: None
 *
 *  Global Variables Used:
 *      gs_RxToneBuf[]    - frequency domain samples
 *      gta_RxSymbolBuf[] - complex valued DMT symbols for designated
 *                     carrier bins in Rx.
 *
 *  Notes :
 *
 *------------------------------------------------------------------------
 *^^^
 */
void GetSymbol(int16 s_SubBit )
{
   int16 i;

   for ( i = 0; i < NUM_CARRIER; i++ )
   {
      gta_RxSymbolBuf[i][s_SubBit].s_X = gpsa_RxToneBuf[ 2*gpsa_RxCarSet[i] ];
      gta_RxSymbolBuf[i][s_SubBit].s_Y = gpsa_RxToneBuf[ 2*gpsa_RxCarSet[i] + 1 ];
   }
}

/*
 *------------------------------------------------------------------------
 *
 *  Name : void TryAnotherToneSet(void)
 *
 *  Abstract :
 *
 *  Parameters:  None
 *
 *  Returns:     None
 *
 *  Global Variables Used:
 *
 *  Notes :
 *
 *------------------------------------------------------------------------
 */
extern int16 gs_numtrials;

void TryAnotherToneSet(void)
{
   int16 i, s_temp;
   int16 s_PrevRxToneOffset;

   s_temp = ~(int16)(1<< gs_GhsCarSetActual);

   // Check if any other tone set enabled
   if ((gs_GhsCarSetCfgInternal & s_temp) == 0) //XDSLRTFW-3515
   {
      return;
   }

   gs_numtrials++;

   if (gs_numtrials >= 1)
   {
      gs_numtrials = 0;

      ///////////////////////////////////
      // Try another tone set
      ////////////////////////////////////
      while(1)
      {
         gs_GhsCarSetActual++;

         if ( gs_GhsCarSetActual >= NUM_CARRIER_SETS)
         {
            gs_GhsCarSetActual = 0;
         }

         // Check if the bitmask for the same is enabled
         s_temp = (int16) (1 << gs_GhsCarSetActual);

         if (gs_GhsCarSetCfgInternal & s_temp) //XDSLRTFW-3515
         {
            break;
         }
      }

      //XDSLRTFW-2288 (Start)
      s_PrevRxToneOffset = gs_RxToneOffset;

      // Initialize GHS carrier sets
      // Set offset for loading gpsa_RxToneBuf from RTV buffer 0
      // The highest DS GHS carriers are {257, 383, 511} implies 511 is
      // out of the RTV buffer range since DetectTone use upto 4 tones
      // on either side of the tone to be detected
      // The lowest DS GHS carriers are {12, 14, 64}
      // Hence we shift the tonegroup up by setting gs_RxToneOffset to 8 so
      // that we can decode all DS carrier sets with a single Rx Tone Offset
      // Note gs_NumChannelsPerGroup is 512 and 256 for 4KHz and 8KHz mode respectively.
      gs_RxToneOffset = 0;
      //XDSLRTFW-4227 (START)  
      //The "gs_GhsCarSetActual" is the bit index. The format below should be 1 left shifted by "gs_GhsCarSetActual". 
      if ((uint16)(1<<gs_GhsCarSetActual) & (CNFG_GHS_CAR_SET_V43 | CNFG_GHS_CAR_SET_V43P | CNFG_GHS_CAR_SET_V43I) )
      //XDSLRTFW-4227 (END)
      {
         gs_RxToneOffset = FIRST_TONE_RTV_BUFF_FOR_V43_GHS_TONE_DET; //XDSLRTFW-2425 (Start_End)
      }

      if (gs_RxToneOffset != s_PrevRxToneOffset )
      {
         AddFunctionToFifo(gp_RxLoadingFunctionFifo, SetRxToneOffset);
      }
      //XDSLRTFW-2288 (End)

      for (i=0; i<NUM_CARRIER; i++)
      {
         gsa_DnCarSet[i] = gsa_DnCarSetTable[gs_GhsCarSetActual][i];
         gsa_UpCarSet[i] = gsa_UpCarSetTable[gs_GhsCarSetActual][i];
         // Adjust Rx carrier set by Rx Tone Offset.
         gsa_DnCarSet[i] -= gs_RxToneOffset;
      }
   }
}



/*^^^
 *-------------------------------------------------------------------
 *
 *  Name : MOVING_AVG
 *
 *  Abstract :
 *
 *  Calculates a moving average of complex valued data in a circula buffer
 *
 *  Parameters :
 *      ta_Buf[]    -   circular buffer containing complex valued data
 *      s_BufLen    -   total length of ta_Buf
 *      s_Idx       -   starting index of average
 *      s_WindowLen -   length of moving average window
 *
 *  Returns :
 *      complex valued moving average result
 *
 *  Notes :
 *      The data buffer ta_Buf is complex valued and circular.
 *
 *-------------------------------------------------------------------
 *^^^
 */

Complex_t MovingAvg( Complex_t ta_Buf[], int16 s_BufLen, int16 s_Idx, int16 s_WindowLen )
{

   int16      i;
   Complex_t t_Result;


   t_Result.s_X = t_Result.s_Y = 0;    // initialize

   for ( i = 0; i < s_WindowLen; i++ )
   {
      t_Result.s_X += ta_Buf[s_Idx].s_X >> 2;    // no rounding to prevent overflow
      t_Result.s_Y += ta_Buf[s_Idx].s_Y >> 2;    // in case the value is saturated
      s_Idx--;
      if ( s_Idx < 0 )    // circulate index
      {
         s_Idx = s_BufLen - 1;
      }
   }

   return (t_Result);

}   // MovingAvg


/*^^^
 *------------------------------------------------------------------------
 *
 *  Name : TestPhaseTransition()
 *
 *  Abstract :
 *
 *  To test phase transition. Take moving averages for current bit symbol
 *  and last bit symbol then vote on hard decision. Note that
 *  gta_RxSymbolBuf is a circular buffer.
 *
 *  Returns: s_Vote
 *
 *  Global Variables Used:
 *      gta_RxSymbolBuf[] - complex valued DMT symbols for designated
 *                     carrier bins in Rx.
 *      gt_RxCurrentSymbol[] - current bit symbol for RX
 *      gt_RxLastSymbol[] - last bit symbol for RX
 *  Notes :
 *
 *------------------------------------------------------------------------
 *^^^
 */
int16 TestPhaseTransition( int16 s_SubBit )
{

   int16  i;
   int16 s_Vote;
   int16 s_WindowLen;
   int16 s_StartIdx;
   int32 l_Product, l_Product1, l_Product2;

   s_Vote      = 0;
   s_WindowLen = RX_SYMBOL_BUF_SIZE >> 1;  // half the buffer length
   s_StartIdx  = ( s_SubBit - s_WindowLen + RX_SYMBOL_BUF_SIZE ) & MODULO_BY_8_MASK;

   for ( i = 0; i < NUM_CARRIER; i++ )
   {
      gt_RxCurrentSymbol[i] = MovingAvg( gta_RxSymbolBuf[i], RX_SYMBOL_BUF_SIZE, s_SubBit, s_WindowLen );
      gt_RxLastSymbol[i]    = MovingAvg( gta_RxSymbolBuf[i], RX_SYMBOL_BUF_SIZE, s_StartIdx, s_WindowLen );

      // Real part of a complex product ( Current_Symbol * Conjugate(Last_Symbol) ) is positive?
      MULS16(l_Product1, gt_RxCurrentSymbol[i].s_X, gt_RxLastSymbol[i].s_X);
      MULS16(l_Product2, gt_RxCurrentSymbol[i].s_Y, gt_RxLastSymbol[i].s_Y);
      l_Product = l_Product1 + l_Product2;
      if ( l_Product >= 0 )
      {
         s_Vote--;   // no phase change if positive
      }
      else
      {
         s_Vote++;   // inverted phase if negative
      }
   }

   return(s_Vote);

}   /* TestPhaseTransition */

/********************************************************************************************

  Prototype: void RxAvgCarrierTones(int16 s_reset)

  Description: Function to compute average of HS carriers over FOUR frames.

  Input Arguments:

   s_reset 0 = add to average buffer gt_RxCurrentSymbol
         non-zero = reset average buffer

  Output Arguments:

   N/A

********************************************************************************************/
void RxAvgCarrierTones(int16 s_reset)
{
   int16 i;

   if(s_reset)
   {
      memset(&gt_RxCurrentSymbol,0,NUM_CARRIER*sizeof(Complex_t));
   }
   else
   {
      for ( i = 0; i < NUM_CARRIER; i++ )
      {
         gt_RxCurrentSymbol[i].s_X += (gpsa_RxToneBuf[ 2*gpsa_RxCarSet[i] ]) >> 2;
         gt_RxCurrentSymbol[i].s_Y += (gpsa_RxToneBuf[ 2*gpsa_RxCarSet[i] + 1 ]) >> 2;
      }
   }
}

/*^^^
 *------------------------------------------------------------------------
 *
 *  Name: RxSymbolReAlign
 *
 *  Description:
 *      Realigns bit boundaries using incoming FLAG and GALF characters
 *
 *  Prototype: void RxSymbolReAlign(void);
 *
 *  Input Arguments: none
 *
 *  Output Arguments: none
 *
 *  Returns: none
 *
 *  Global Variables Used:
 *      gsa_FFT_InBuf[]                - input of IFFT, used in RxBit()
 *      gs_RxToneBuf[]                 - frequency domain samples, used in RxBit()
 *      guc_RxOctet                    - (I/O) current octet being transmitted
 *      gs_RxState                     - (I) current RX state
 *      gs_RxSubState                  - (I/O) current RX substate
 *      gs_RxSubStateCnt               - (I/O) # symbols in current RX substate
 *      gl_RxSymbolCount               - (I) # symbol periods in current RX state
 *      gs_RxNextState                 - (O) RX state that will begin next symbol
 *                                       period
 *      gs_PGA_required_In_GHS         - (0) initial PGA value for next Ghs session
 *                                           (in case Ghs session failed)
 *
 *  Substates:
 *      RX_SYMBOL_REALIGN_FILL_BUFFER  - Fills up gt_RxSymbolBuf.
 *      RX_SYMBOL_REALIGN_DETECT_ZERO  - Detects SYMBOLS_PER_BIT DMT symbols with no
 *                                       phase reversal (a ZERO bit).
 *      RX_SYMBOL_REALIGN_DETECT_ONE   - Detects a DMT symbol with a phase reversal
 *                                       (beginning of a ONE bit).
 *      RX_SYMBOL_REALIGN_RECEIVE_BITS - the RxBit function is called as additional
 *                                       DMT symbols are acquired, and the resulting
 *                                       bits are stored in guc_RxOctet until a total
 *                                       of SYMBOLS_PER_OCTET DMT symbols have been
 *                                       aquired.
 *
 *  Notes: none
 *
 *------------------------------------------------------------------------
 *^^^
 */

/* =============================================== */
/* substates */
/* =============================================== */
#define RX_SYMBOL_REALIGN_FILL_BUFFER  (0)
#define RX_SYMBOL_REALIGN_DETECT_ZERO  (1)
#define RX_SYMBOL_REALIGN_DETECT_ONE   (2)
#define RX_SYMBOL_REALIGN_RECEIVE_BITS (3)


void RxSymbolReAlign(void)
{

   int16 s_SubBit;
   int16 s_Vote;
   int16 s_SymCnt;     /*  local symbol count, which will copy gl_RxSymbolCount or gs_RxGhsSymCnt */

   if (gs_RxState == R_C_GALF1_RX)
   {
      s_SymCnt = (int16) gl_RxSymbolCount;
   }
   else /*  (gs_RxState == R_C_HS_MSG_RX) */
   {
      s_SymCnt = gs_RxGhsSymCnt;
   }

   switch (gs_ReAlignSubState)
   {

      /* =============================================== */
      /* fill up gt_RxSymbolBuf */
      /* =============================================== */
   case RX_SYMBOL_REALIGN_FILL_BUFFER:

      s_SubBit = s_SymCnt & MODULO_BY_8_MASK;
      gs_RxSubStateCnt++;
      GetSymbol( s_SubBit );

      if ( gs_RxSubStateCnt >= SYMBOLS_PER_BIT )
      {
         gs_RxSubStateCnt   = 0;
         gs_ReAlignSubState = RX_SYMBOL_REALIGN_DETECT_ZERO;
      }

      break;

      /* =============================================== */
      /* detect zero bit (no phase change) */
      /* =============================================== */
   case RX_SYMBOL_REALIGN_DETECT_ZERO:

      s_SubBit = s_SymCnt & MODULO_BY_8_MASK;
      GetSymbol( s_SubBit );
      s_Vote = TestPhaseTransition( s_SubBit );

      if ( s_Vote < 0 )
      {
         gs_RxSubStateCnt++;
      }
      else
      {
         gs_RxSubStateCnt = 0;
      }

      /* ---- check for SYMBOLS_PER_BIT consecutive symbos without phase reversal ---- */
      if (gs_RxSubStateCnt >= SYMBOLS_PER_BIT)
      {
         gs_RxSubStateCnt   = 0;
         gs_ReAlignSubState = RX_SYMBOL_REALIGN_DETECT_ONE;
      }

      break;

      /* =============================================== */
      /* detect first phase reversal */
      /* =============================================== */
   case RX_SYMBOL_REALIGN_DETECT_ONE:

      s_SubBit = s_SymCnt & MODULO_BY_8_MASK;
      GetSymbol( s_SubBit );
      s_Vote = TestPhaseTransition( s_SubBit );

      /* ---- if first phase reversal (binary 1 of a bit) detected, ---- */
      /* ---- initiailize for next substate                         ---- */
      if ( s_Vote >= 0 )
      {
         // Initialize the received octet
         guc_RxOctet = 0;

         // Reset averaging buffer
         RxAvgCarrierTones(1);

         if ( gs_RxState == R_C_GALF1_RX )
         {
            gs_RxSubStateCnt = 2 - SYMBOLS_PER_BIT;   // we assume first phase change is
         }
         // the LSB of the previous GALF
         else //  if ( gs_RxState == R_C_HS_MSG_RX )
         {
            gs_RxSubStateCnt = SYMBOLS_PER_BIT + 2;   // we assume the first bit of a flag,
         }
         // which is 0, is already received at this point.

         // Start decoding octet. Note that the '1' we just detected will be decoded again
         // and put into guc_RxOctet. When we are trying to detect GALF (in R_C_GALF1_RX) this
         // bit is eventually discarded because it belongs to the previous octet. When we are
         // trying to detect FLAG it is preserved. This is controlled by the initial value
         // of gs_RxSubStateCnt above.
         s_SubBit = gs_RxSubStateCnt & MODULO_BY_8_MASK;
         RxBit(gpsa_RxCarSet,  s_SubBit );
         gs_ReAlignSubState = RX_SYMBOL_REALIGN_RECEIVE_BITS;
      }

      break;

      /* =============================================== */
      /* after the detection of the first phase reversal */
      /* =============================================== */
   case RX_SYMBOL_REALIGN_RECEIVE_BITS:

      gs_RxSubStateCnt++;

      s_SubBit = gs_RxSubStateCnt & MODULO_BY_8_MASK;

      RxBit(gpsa_RxCarSet,  s_SubBit );

      /* ---- one full octet received ---- */
      if ( gs_RxSubStateCnt == (SYMBOLS_PER_OCTET - 1) )
      {

         switch ( gs_RxState )
         {

         case R_C_GALF1_RX:
            if ( guc_RxOctet == GALF )
            {
               // mark the start of GHS phase only at READY to GHS transition
               if (gsa_IndirectStat0[0] == STAT_ReadyState)
               {
                  // log start of GHS session
                  gl_HandshakePhaseSymCnt_start = gl_TotalTxSymbolCount;
               }
               /* Set the Macro State CMV. */
               gsa_IndirectStat0[0] = STAT_GhsState;
               gusa_MONI_ModemStat_Status[0] = MONI_STAT_GHS;
               // XDSLRTFW-4241 GHs-Tone detection fails when disconnecting DSL line (long training time/PGA) (Start)
               // XDSLRTFW-3764 (Start_End)
               // Note: gs_PGA_required_In_GHS is intended to be used as initial PGA value for GHS.
               //       But gs_PGA_required_In_GHS is also used as output. Now for fails in the GHS start-up, e.g. timeout,
               //       no exception gets be triggered and direct jump to start of GHS is performed, i.e. Silent state.
               //       Now the problem occurs that the previous trained PGA value is used as initial PGA value for GHS,
               //       which can lead to stuck, when the PGA was trained on showtime or training signal due to different point
               //       in times between CPE and CO drop.
               //       But the fix leads to handshake timeouts and therefore to longer train times in multiline vectoring setups.
               //       Reason is that the BDCM is doing an intentional handshake drop after receiving the CLR to load the vectoring
               //       binary. When now the old PGA value is used the link is more robust against detection of XTALK handshake signals
               //       from the other links.
               //       !!Decision was made to go for the time being with the old implementation!!
               {
                  // Store PGA setting after detecting a valid CO signal (GALF signal). This could reduce the possibility of Ghs Stuck
                  // described under XDSLRTFW-3764 Note above.
                  gs_PGA_set_GHS = gs_PGA_set;
                  gs_PGA_required_In_GHS = gs_PGA_set;
               }
               //XDSLRTFW-3613 Start
               gs_Stored_AGC1_Gain_Set = gs_AGC1_Gain_Set; //PGA gain
               gs_Stored_AGC2_Gain_Set = gs_AGC2_Gain_Set; //PREFI gain
               //XDSLRTFW-3613 End
               // XDSLRTFW-4241 GHs-Tone detection fails when disconnecting DSL line (long training time/PGA) (End)

               // alignment is done switch to next state
               gs_RxNextState = R_C_FLAG1_RX;
               gpF_RxStateFunc = (PtrToFunc)RCFlag1RxF;

               //XDSLRTFW-2288 (Start)
#ifdef GHS_TN_DBG
               //Only for debug to capture AFE regs.
               if((TESTArray[TEST_GHS_LOG] & HALT_AFTER_TONE_DETECTION) != 0)
               {
                  gft_PauseOff = 0;
                  Pause(0x0BB1);
               }
#endif
               //XDSLRTFW-2288 (End)
            }
            else    /*   if failed, retry symbol alignment */
            {
               gs_RxSubStateCnt   = 0;
               gs_ReAlignSubState = RX_SYMBOL_REALIGN_FILL_BUFFER;
            }

            break;

         default: /*  case R_C_HS_MSG_RX: */
            if ( guc_RxOctet == FLAG )
            {
               gft_SymbolReAlignRequired = FALSE;
               gs_RxGhsSymCnt = -1; /*  resetting counter upon locking on a FLAG */
               gs_RxSubStateCnt   = 0; /*  reset counter */
            }
            else    /*   if failed, retry symbol alignment */
            {
               gs_RxSubStateCnt   = 0;
               gs_ReAlignSubState = RX_SYMBOL_REALIGN_FILL_BUFFER;
            }

            break;

         }   /*  switch (gs_RxState) */

      }

      break;

   }    /*  switch (gs_RxSubState) */

}   /*  RxSymbolReAlign */

#undef RX_SYMBOL_REALIGN_FILL_BUFFER
#undef RX_SYMBOL_REALIGN_DETECT_ZERO
#undef RX_SYMBOL_REALIGN_DETECT_ONE
#undef RX_SYMBOL_REALIGN_RECEIVE_BITS



