/* **COPYRIGHT******************************************************************
    INTEL CONFIDENTIAL
    Copyright (C) 2017 Intel Corporation
    Copyright (C), 1994-2011 Aware Inc. All Rights Reserved.
******************************************************************COPYRIGHT** */
/* **DISCLAIMER*****************************************************************
    The source code contained or described herein and all documents related
    to the source code ("Material") are owned by Intel Corporation or its
    suppliers or licensors. Title to the Material remains with Intel
    Corporation or its suppliers and licensors. The Material may contain
    trade secrets and proprietary and confidential information of Intel
    Corporation and its suppliers and licensors, and is protected by
    worldwide copyright and trade secret laws and treaty provisions. No part
    of the Material may be used, copied, reproduced, modified, published,
    uploaded, posted, transmitted, distributed, or disclosed in any way
    without Intel's prior express written permission.

    No license under any patent, copyright, trade secret or other
    intellectual property right is granted to or conferred upon you by
    disclosure or delivery of the Materials, either expressly, by
    implication, inducement, estoppel or otherwise. Any license under
    such intellectual property rights must be express and approved by
    Intel in writing.
*****************************************************************DISCLAIMER** */
/*
*-------------------------------------------------------------------------------
*
*   Aware DMT Technology. Proprietary and Confidential.
*
*   40 Middlesex Turnpike, Bedford, MA 01730-1413
*   Phone (781) 276 - 4000
*   Fax   (781) 276 - 4001
*
*   filename: hs_init_train.c
*
*   This file contains handshake functions used in pre/post GHS initialization.
*   Merge 25/05/2012 : VENKATESH S N
*   XDSLRTFW-456 : CORRECT_ADSL_OCTETS_IN_VDSL_GHS_MSG
*   Issue: DSLAM gets to GHS Messaging loop with MultiMode enabled on VR9
*   Cause: ADSL Octets sent as part of CLR with VDSL Octets do not have the
*          right termination octets
*   Fix:   Correct the Octets sent for ADSL with VDSL octets
*
*-------------------------------------------------------------------------------
*/
// ***********************************************************************************************************
// hs_init_train.c
//
// History
// 26/06/2014 Fuss: Extended O-P-Vector1 phase, i.e. enable the feature indication in Ghs by default
//            Grep for XDSLRTFW-1797
// ************************************************************************************************************

#include <string.h>                    //for memset and memcpy
#include "common.h"
#include "gdata.h"
#include "ghs.h"
#include "cmv.h"
#include "cmv_Data.h"
#include "fcs.h"
#include "eoc.h"

#include "vdsl_const.h"

static void InitializeInfoField_TransceiverIdCpe(InfoField_t *t_I);
void InitializeStandardInfoField_ADSL(ADSL_Type_t *pt_ADSL_I, InfoField_t *t_Info);
void InitializeStandardInfoField_VDSL2(VDSL2_Type_t *pt_VDSL2_I);

/*^^^
 *-------------------------------------------------------------------
 *
 *  Name : INITIALIZE_HSC
 *
 *  Abstract : Initialize Handshake Control structure
 *
 *  Parameters:
 *      pt_hsc - pointer to a HandShakeControl_t structure
 *
 *  Returns: None
 *
 *  Global Variables Used:
 *
 *  Notes :
 *
 *-------------------------------------------------------------------
 *^^^
 */
void InitializeHSC( HandShakeControl_t *pt_hsc )
{

   // First check which initial message CPE will start in handshake session
   if ((OPTNArray[OPTN_GhsControl] & OPTN_GHS_INIT_MASK)==OPTN_RTInitTxMsg_CLR)
   {
      gt_hsc.s_InitialMsg = START_WITH_CLR;
   }
   else if ((OPTNArray[OPTN_GhsControl] & OPTN_GHS_INIT_MASK)==OPTN_RTInitTxMsg_MR)
   {
      gt_hsc.s_InitialMsg = START_WITH_MR;
   }
   else if ((OPTNArray[OPTN_GhsControl] & OPTN_GHS_INIT_MASK)==OPTN_RTInitTxMsg_MS)
   {
      gt_hsc.s_InitialMsg = START_WITH_MS;
   }
   else if ((OPTNArray[OPTN_GhsControl] & OPTN_GHS_INIT_MASK)==OPTN_RTInitTxMsg_MP)
   {
      gt_hsc.s_InitialMsg = START_WITH_MP;
   }
   else
   {
      gt_hsc.s_InitialMsg = START_WITH_CLR;  // Default: CPE sends CLR
   }

   pt_hsc->s_State             = R_HS_INITIAL;
   pt_hsc->s_NextState         = R_HS_INITIAL;

   pt_hsc->s_DecodeResult      = DECODE_IDLE;
   pt_hsc->l_SelectedMode      = NOT_SELECTED;
   pt_hsc->s_TxMsgComplete     = FALSE;
   pt_hsc->s_RxMsgComplete     = FALSE;
   if(gs_HandShake_Cnt == 0)
   {
      pt_hsc->s_CL_Exchanged      = FALSE;
   }
   pt_hsc->s_TxNSBCnt          = 0;
   pt_hsc->s_RxNSBCnt          = 0;

}   // InitializeHSC

/*^^^
 *-------------------------------------------------------------------
 *
 *  Name : InitializeStandardInfoField
 *
 *  Abstract :
 *
 *  InitializeInfoField() - Initialize an Information Field structure
 *
 *  Parameters:
 *      InfoField_t *t_Info : information structure to be initialized
 *
 *  Returns: None
 *
 *  Global Variables Used:
 *
 *  Notes :
 *
 *-------------------------------------------------------------------
 *^^^
 */

void InitializeStandardInfoField(InfoField_t * t_I)
{

   int16 i;

   //==== Identification parameter field ===================
   if (gul_dbgGHSControl & FORCE_NON_STANDARD_INFO)
   {
      t_I->uc_ID_NPar1 = NON_STANDARD;  // force non standard info
   }
   else
   {
      t_I->uc_ID_NPar1 = 0;  // no parameters set
   }

   t_I->uc_ID_SPar1 = 0;
   t_I->uc_ID_SPar1_02 = 0;
   t_I->uc_ID_SPar1_03 = 0;
   t_I->uc_ID_SPar1_04 = 0;

   // ID_SPar1
   //---------------
   // Net data rate and data flow
   //t_Info->uc_ID_SPar1 = NDR_UP | NDR_DN | DFC_UP | DFC_DN;

   // Transceiver Id
   if(TESTArray[TEST_XDSLRTFW_CONFIG] & TEST_XDSLRTFW_CONFIG_XTALK_CONNECT)
   {
      InitializeInfoField_TransceiverIdCpe(t_I);
   }


   // ID_SPar1_03
   //---------------
#ifdef VDSL_BONDING
   t_I->uc_BondingNP2 = 0;
   t_I->uc_BondingSP2 = 0;
   // if bonding is enabled, then initialize the Standard Info Field
   // Note: For CPE this is done in function PrepareCLR(void)!
#endif /* VDSL_BONDING */


   //==== Standard Information Field ===================

   // NPar1: V.8, V.8bis, Silent Period, G.997.1
   t_I->uc_SI_NPar1 = SILENT_PERIOD;            // needs to be set in a CL or CLR message

   //XDSLRTFW-1184: BugFix_ALL_ALL_IFTN_No_Connect_In_AVDSL_Mode (Start)
   t_I->uc_SI_SPar1 = 0;
   t_I->uc_SI_SPar1_02 = 0;
   t_I->uc_SI_SPar1_03 = 0;
   t_I->uc_SI_SPar1_04 = 0;
   t_I->uc_SI_SPar1_05 = 0;

   // Note: This IF-case is needed to cover Bonding and TransceiverId fix with
   //       a new handshake session together with the ADSL-modes fix. This saves
   //       a second CLR-CL-ACK1 transaction.
   //       Further, if this code would not be in than the handshake session gets
   //       be changed by the CO that the CPE has to do the ModeSelect. Since the
   //       code was changed for TransceiverId the exception E_CODE_GHS_RETRAIN is
   //       now thrown.
   if ((guc_ClrControl & CLR_SEND_NOT_ADSL_MODES) == 0)
   {
      // SPar1 Octet 1: G992_1_A, G992_1_B, G992_1_C, G992_2_A/B, G992_2_C, G992_1_H
      t_I->uc_SI_SPar1 = guc_SI_SPar1_Optn0;

      // SPar1 Octet 3: G992_3_A, G992_3_B, G992_3_I, G992_3_J, G992_4_A, G992_4_I
      t_I->uc_SI_SPar1_03 = guc_SI_SPar3_Optn0;

      // SPar1 Octet 4: G992_5_A, G992_5_B, G992_5_I
      t_I->uc_SI_SPar1_04 = guc_SI_SPar4_Optn0;

      //XDSLRTFW-456 : CORRECT_ADSL_OCTETS_IN_VDSL_GHS_MSG (START)
      // SPar1 Octet 5: G992_5_M, G993_1, G993_2
      t_I->uc_SI_SPar1_05 = guc_SI_SPar5_Optn0;
      //XDSLRTFW-456 : CORRECT_ADSL_OCTETS_IN_VDSL_GHS_MSG (END)

      InitializeStandardInfoField_ADSL(&t_I->t_ADSL_Info, t_I);
   }
   //XDSLRTFW-1184: BugFix_ALL_ALL_IFTN_No_Connect_In_AVDSL_Mode (End)

   // SPar1 Octet 5: G992_5_M, G993_2
   if (gsa_Optn0_ModeControl[1] & OPTN_ConfigMode_G993_2)
   {
      t_I->uc_SI_SPar1_05 |= G993_2;   // Set to G.993.2
   }

   //===== Spar1 Octet 5: G.993.2 VDSL2 ===========
   if(t_I->uc_SI_SPar1_05 & G993_2)
   {
      //Initialize the standard information field for VDSL2
      InitializeStandardInfoField_VDSL2(&t_I->t_VDSL2_Info);
   }

   // SPar1 Octet 5: G9701
   if (gsa_Optn0_ModeControl[1] & OPTN_ConfigMode_G9701)
   {
      t_I->uc_SI_SPar1_05 |= G9701;   // Set to G.9701
   }

   //===== Spar1 Octet 5: G.993.2 VDSL2 ===========
   if(t_I->uc_SI_SPar1_05 & G9701)
   {
      //Initialize the standard information field for VDSL2
      InitializeStandardInfoField_GFast(&t_I->t_G_Fast_Info, t_I);
   }

   //==== Non-Standard Information Field ====
   {
      t_I->uc_NumBlock       = 0;                 // number of non-standard blocks
      for ( i = 0; i < MAX_NUM_NS_BLOCK; i++ )
      {
         t_I->ta_NS_Info[i].uc_NSLen = 0;
      }
   }
}   // InitializeInfoField


//----------------------------
// The 30 bit VTU ID shall consist of 2 parts:
// - A first part 16 bits (the MSBs) derived from the serial number and vendor ID,
//   i.e. 16bit CRC (HDLC FCS Frame check sequence) over the mentioned octets.
// - A second part 14 bits (the LSBs) using a random number generator.
//
void InitializeInfoField_TransceiverIdCpe(InfoField_t *t_Info)
{
   // ID Spar(1) transceiver ID information
   if ((guc_ClrControl & CLR_SEND_NOT_TRANCSEIVER_ID) == 0)
   {
      t_Info->uc_ID_SPar1 |= TRANSCEIVER_ID;
   }

   //---------------------------------
   // --- Remote side transceiver ID
   // --- ID Par(2) block
   //---------------------------------
   // NPar2
   // Note: Contains no information. Therefore the no memory is allocated for this
   //       parameter. It is hard set in fct. "FormStandard()", file "hs_mesg.c".
//   t_Info->uc_ID_NPar2_TransId = 0;

   // SPar2
   t_Info->uc_ID_SPar2_TransId = CPE_TRANSCEIVER_ID;

   // Npar(3)
   //==== Frame Check Sequence ====
   // 1) The FCS shall be calculated over all bits of the VTU vendor ID followed by the VTU serial number,
   //    i.e. 16bit.
   //    1.1) VTU vendor ID
   //         It shall consist of the T.35 country code (2 octets) followed by the provider code (4 octets),
   //         as indicated in the Vendor ID information block during the G.994.1 Handshake phase of
   //         initialization (see Table 7/G.994.1).
   //    NOTE: This is excluding the "Vendor-specific information" of the Vendor ID information block.
   //    1.2) VTU R serial number
   //         It shall be the equipment serial number that is part of the G.997.1 XTU R serial number
   //         (see clause 7.4.8/G.997.1).
   //          XTU-R serial number is 32 ASCII characters.
   //          All parts shall be encoded in the given order
   //             "<equipment serial number> <equipment model> <equipment firmware version>"
   //          and separated by space characters (0x20 = 32).
   //    NOTE: This is excluding the equipment model and the equipment firmware version that are
   //          also part of the G.997.1 XTU R serial number.
   //
   //    Overview:
   //           FCS over
   //             2 Bytes Country Code
   //             4 Bytes Provider Code
   //             x Bytes equipment serial number (max 32byte)
   //
   // 2) The register used to calculate the CRC shall be initialized to the value 0xFFFF
   {
      unsigned int i;
      uint16 us_FCS;
      uint8 *puc_Data;

      us_FCS = 0xFFFF;    // initialize to all binary 1's

      // FCS over
      //   2 Bytes Country Code
      //   4 Bytes Provider Code
      {
         uint8 uca_DataForFcs[6];

         // Prepare data
         puc_Data = &uca_DataForFcs[0];
         *puc_Data++ = (uint8) (t_Info->us_Country >> 8);
         *puc_Data++ = (uint8) (t_Info->us_Country);

         for(i = 0; i < 4; i++ )
         {
            *puc_Data++ = (uint8) (t_Info->ul_Provider >> ((3-i) * 8));
         }

         // Calculate FCS
         puc_Data = &uca_DataForFcs[0];
         for(i = 0; i < 6; i++)
         {
            us_FCS = calcCrc16(us_FCS, *puc_Data++);
         }
      }

      // FCS over:
      //     x Bytes equipment serial number (max 32byte)
      //
      // Data is coming from API: INFOMap[INFO_CPESystemSerialNum], with INFO_CPESystemSerialNum = 62
      // CMD_SysVendorSerialNum_R_Set: Message Identifier - 0x3E43
      //                               CMV INFO 62
      i = 0;
      while((guca_ne_SystemSerialNum[i] != 0x20) && (i < SERIAL_NUM_LENGTH))
      {
         us_FCS = calcCrc16(us_FCS, guca_ne_SystemSerialNum[i]);
         i++;
      }

      us_FCS = (uint16)(~us_FCS);            // take 1's compliment
      t_Info->ul_CpeTransId = (us_FCS << 14);
   }

   // The second part has length 14 bits and shall be generated by the VTU by means
   // of a vendor discretionary random number generator. The randomly generated number
   // shall change from one G.994.1 session to the next.
   //
   // Note: There are two kinds of random numbers
   //       - "True" random numbers -> physical phenomenon like Noise (Noise generators)
   //       - pseudo-random numbers -> computational algorithms called pseudorandom number generators
   //
   // The 14bit random numer is only needed for lab set-ups where every CPE has the same MAC-address.
   // Otherwise the TransceiverID would be unique due to the MAC-address.
   // So no big effort is spend to implement a pseudorandom number generators like a "Xorshift-Generator".
   {
      t_Info->ul_CpeTransId |= (gl_InitRxHsTonesPower & 0x3fff);
   }

   gul_CpeTransId = t_Info->ul_CpeTransId;
}


#ifdef VDSL_BONDING
/*^^^
 *-------------------------------------------------------------------
 *
 *  Name : InitializeInfoField_Bonding_NPar3
 *
 *  Abstract :
 *      InitializeInfoField_Bonding_NPar3() - Initialize NPar3 Discovery and PME aggregation fields
 *
 *  Parameters:
 *      InfoField_t *t_I : information structure to be initialized
 *
 *  Returns: None
 *
 *-------------------------------------------------------------------
 *^^^
 */
void InitializeInfoField_Bonding_NPar3(InfoField_t *t_I)
{
   // Discovery register
   t_I->uca_BondingDisc[0] = 0;
   t_I->uca_BondingDisc[1] = (gt_Bonding_Continue.sa_DiscCode[0] >> 10) & 0x3F;
   t_I->uca_BondingDisc[2] = (gt_Bonding_Continue.sa_DiscCode[0] >>  4) & 0x3F;
   t_I->uca_BondingDisc[3] = (gt_Bonding_Continue.sa_DiscCode[0] <<  2) & 0x3C
                             | (gt_Bonding_Continue.sa_DiscCode[1] >>  14) & 0x03 ;
   t_I->uca_BondingDisc[4] = (gt_Bonding_Continue.sa_DiscCode[1] >>  8) & 0x3F;
   t_I->uca_BondingDisc[5] = (gt_Bonding_Continue.sa_DiscCode[1] >>  2) & 0x3F;
   t_I->uca_BondingDisc[6] = (gt_Bonding_Continue.sa_DiscCode[1] <<  4) & 0x30
                             | (gt_Bonding_Continue.sa_DiscCode[2] >>  12) & 0x0F ;
   t_I->uca_BondingDisc[7] = (gt_Bonding_Continue.sa_DiscCode[2] >>  6) & 0x3F;
   t_I->uca_BondingDisc[8] = (gt_Bonding_Continue.sa_DiscCode[2] >>  0) & 0x3F;

   // PME aggregation register
   t_I->uca_BondingAggr[0] = (gt_Bonding_Continue.sa_PMEAggr_data[0] >>  14) & 0x03 ;
   t_I->uca_BondingAggr[1] = (gt_Bonding_Continue.sa_PMEAggr_data[0] >>  8) & 0x3F;
   t_I->uca_BondingAggr[2] = (gt_Bonding_Continue.sa_PMEAggr_data[0] >>  2) & 0x3F;
   t_I->uca_BondingAggr[3] = (gt_Bonding_Continue.sa_PMEAggr_data[0] <<  4) & 0x30
                             | (gt_Bonding_Continue.sa_PMEAggr_data[1] >>  12) & 0x0F ;
   t_I->uca_BondingAggr[4] = (gt_Bonding_Continue.sa_PMEAggr_data[1] >>  6) & 0x3F;
   t_I->uca_BondingAggr[5] = (gt_Bonding_Continue.sa_PMEAggr_data[1] >>  0) & 0x3F;

   t_I->uc_BondingId = gus_Bonding_PME_ID & 0x1F ;     // XDSLRTFW-2382 (Start_End)
}

/*^^^
 *-------------------------------------------------------------------
 *
 *  Name : StoreInfoField_Bonding_NPar3
 *
 *  Abstract :
 *      StoreInfoField_Bonding_NPar3() - Store NPar3 Discovery and PME aggregation fields into (CMV-mapped) variable
 *
 *  Parameters:
 *      InfoField_t *t_I : information structure to read from
 *       uint8 uc_FieldToStore:  which NPar3 field to store (bitfield - so multiple bits may be set)
 *
 *  Returns: None
 *
 *-------------------------------------------------------------------
 *^^^
 */
void StoreInfoField_Bonding_NPar3(InfoField_t *t_I, uint8 uc_FieldToStore)
{
   if (uc_FieldToStore & BONDING_PME_DISCOVERY)
   {
      // PME discovery command
      gt_Bonding_DiscAggr_Status.sa_DiscCode[0] = (t_I->uca_BondingDisc[1]<<10)
            | (t_I->uca_BondingDisc[2]<<4)
            | (t_I->uca_BondingDisc[3]>>2);
      gt_Bonding_DiscAggr_Status.sa_DiscCode[1] =   (t_I->uca_BondingDisc[3]<<14)
            | (t_I->uca_BondingDisc[4]<<8)
            | (t_I->uca_BondingDisc[5]<<2)
            | (t_I->uca_BondingDisc[6]>>4);
      gt_Bonding_DiscAggr_Status.sa_DiscCode[2] = (t_I->uca_BondingDisc[6]<<12)
            | (t_I->uca_BondingDisc[7]<<6)
            | (t_I->uca_BondingDisc[8]<<0);
   }
   if (uc_FieldToStore & BONDING_PME_AGGREGATION)
   {
      // PME aggregation command
      gt_Bonding_DiscAggr_Status.sa_PMEAggr_data[0] = (t_I->uca_BondingAggr[0]<<14)
            | (t_I->uca_BondingAggr[1]<<8)
            | (t_I->uca_BondingAggr[2]<<2)
            | (t_I->uca_BondingAggr[3]>>4);
      gt_Bonding_DiscAggr_Status.sa_PMEAggr_data[1] = (t_I->uca_BondingAggr[3]<<12)
            | (t_I->uca_BondingAggr[4]<<6)
            | (t_I->uca_BondingAggr[5]<<0);
   }
}
#endif //#ifdef VDSL_BONDING


void InitializeStandardInfoField_ADSL(ADSL_Type_t *pt_ADSL_I, InfoField_t *t_Info)
{
   // XDSLRTFW-456 : CORRECT_ADSL_OCTETS_IN_VDSL_GHS_MSG (START)
   //G992.1A.NPar(2)
   if(t_Info->uc_SI_SPar1 & G992_1_A)
      pt_ADSL_I->uc_SI_NPar2_01 = G9921A_ATM |G9921A_R_ACK1| G9921A_R_ACK2 | CLEAR_EOC_G997_1;

   //G992.1B.NPar(2)
   if(t_Info->uc_SI_SPar1 & G992_1_B)
      pt_ADSL_I->uc_SI_NPar2_02 = G9921A_ATM |G9921A_R_ACK1| G9921A_R_ACK2 | CLEAR_EOC_G997_1 | UPSTREAM_1_32_DMT;

   //XDSLRTFW-2137 (Start)
   /********************************************************************
       G992.2 CLR
   *********************************************************************/
   if(t_Info->uc_SI_SPar1 & G992_2_AB)
   {
      pt_ADSL_I->uc_SI_NPar2_06 = G9922AB_R_ACK1 | G9922AB_R_ACK2 | G9922AB_RS16 | G9922AB_CLEAR_EOC_G997_1;
   }
   //XDSLRTFW-2137 (End)

   /********************************************************************
       G9923 / G9925 CLR
   *********************************************************************/
   // Note: AnxL is considered at this point already in the uc_SI_SPar1_03 bit "G992_3_A" setting.
   if(t_Info->uc_SI_SPar1_03 & G992_3_A)
      pt_ADSL_I->uc_SI_NPar2_03 = G9923AL_RESERVE0_4;

   if(t_Info->uc_SI_SPar1_03 & G992_3_B)
      pt_ADSL_I->uc_SI_NPar2_04 = G9923AL_RESERVE0_4 | UPSTREAM_1_32_BIS;

   if(t_Info->uc_SI_SPar1_03 & G992_3_I)
      pt_ADSL_I->uc_SI_NPar2_03 = G9923AL_RESERVE0_4;

   if(t_Info->uc_SI_SPar1_03 & G992_3_J)
      pt_ADSL_I->uc_SI_NPar2_05 = G9923AL_RESERVE0_4 | PSD_SHAPE_SUPPORT;

   if(t_Info->uc_SI_SPar1_04 & G992_5_A)
      pt_ADSL_I->uc_SI_NPar2_03 = G9923AL_RESERVE0_4;

   if(t_Info->uc_SI_SPar1_04 & G992_5_B)
      pt_ADSL_I->uc_SI_NPar2_04 = G9923AL_RESERVE0_4 | UPSTREAM_1_32_BIS;

   if(t_Info->uc_SI_SPar1_04 & G992_5_I)
      pt_ADSL_I->uc_SI_NPar2_03 = G9923AL_RESERVE0_4;

   if(t_Info->uc_SI_SPar1_04 & G992_3_M)
      pt_ADSL_I->uc_SI_NPar2_05 = G9923AL_RESERVE0_4 | PSD_SHAPE_SUPPORT;

   if(t_Info->uc_SI_SPar1_04 & G992_5_J)
      pt_ADSL_I->uc_SI_NPar2_05 = G9923AL_RESERVE0_4 | PSD_SHAPE_SUPPORT;

   if(t_Info->uc_SI_SPar1_05 & G992_5_M)
      pt_ADSL_I->uc_SI_NPar2_05 = G9923AL_RESERVE0_4 | PSD_SHAPE_SUPPORT;

   //XDSLRTFW-456 : CORRECT_ADSL_OCTETS_IN_VDSL_GHS_MSG (End)
}


/*^^^
 *-------------------------------------------------------------------
 *
 *  Name : InitializeStandardInfoField_VDSL2()
 *
 *  Abstract :
 *
 *  Initialize an Information Field structure for VDSL2 CO modem
 *
 *  Input argument:
 *
 *  Output argument:
 *     pt_VDSL2_I  : pointer to information structure to be initialized
 *
 *  Returns: None
 *
 *  Global Variables Used:
 *      gs_NumOfRxBands               -- (I) no. of RX bands
 *      gsa_RxBandLeftChannel         -- (I) RX band left edges tone indices
 *      gsa_RxBandRightChannel         -- (I) RX band Right edges tone indices
 *      gs_NumOfTxBands               -- (I) no. of TX bands
 *      gsa_TxBandLeftChannel         -- (I) TX band left edges tone indices
 *      gsa_TxBandRightChannel         -- (I) TX band Right edges tone indices
 *      gs_NumOfRFIBands            -- (I) no. of RFI bands
 *      gsa_RFIBandLeftChannel         -- (I) RFI band left edges tone indices
 *      gsa_RFIBandRightChannel         -- (I) RFI band Right edges tone indices
 *      gs_TxLog2IfftLength            -- (I) FFT/IFFT size represented in log2(2N)
 *      gs_TxCELength                  -- (I) initial Tx CE length
 *  Notes :
 *
 *-------------------------------------------------------------------
 *^^^
 */

void InitializeStandardInfoField_VDSL2(VDSL2_Type_t *pt_VDSL2_I)
{
   int16 s_m_param;
   int16 index = 0;


   //SI - NPar2
   //Set it according to the capability list
   pt_VDSL2_I->uc_SI_NPar2 = (uint8)(gul_OperationModeCapable_VDSL2 & NEG_DELIMIT_2);            // NEG_DELIMIT_2 = 0x3F
   if (gt_ProfileControl.ul_AnxAUS0PsdSupported & CNFG_V2_ANXA_US0_ADLU_ALL_MASK)
   {
      pt_VDSL2_I->uc_SI_NPar2 |= V2_ADL_MODE;
   }
   else
   {
      pt_VDSL2_I->uc_SI_NPar2 &= (~V2_ADL_MODE);
   }

   if (gul_OperationModeCapable_VDSL2 & V2_EQ_FEXT_UPBO)
   {
      pt_VDSL2_I->uc_SI_NPar2 |= V2_EQ_FEXT_UPBO;
   }
   else
   {
      pt_VDSL2_I->uc_SI_NPar2 &= (~V2_EQ_FEXT_UPBO);
   }

   // XDSLRTFW-487_VR9_VRX318_VDSL2_All_AELEM_Support (START)
   if (gul_OperationModeCapable_VDSL2 & (V2_ELE_METHOD1 << 24))
   {
      pt_VDSL2_I->uc_SI_NPar2_02 |= V2_ELE_METHOD1;
   }
   // XDSLRTFW-487_VR9_VRX318_VDSL2_All_AELEM_Support (END)

   if (gul_OperationModeCapable_VDSL2 & (V2_VECTOR_FULL_FRIENDLY << 24))
   {
      //XDSLRTFW-2874 (Start_End)
      pt_VDSL2_I->uc_SI_NPar2_02 |= (V2_VECTOR_FULL_FRIENDLY | V2_VECTOR_FULL_FRIENDLY_PILOT_SEQ_LEN_MULTI4_MASK |
                                     V2_VECTOR_FULL_FRIENDLY_FDPS_US_MASK);
   }

   //SI - SPar2 octet 1
   //Set it according to the capability list
   pt_VDSL2_I->uc_SI_SPar2_01 = (uint8)((gul_OperationModeCapable_VDSL2 >> 8) & NEG_DELIMIT_2);  // NEG_DELIMIT_2 = 0x3F
   if (gt_ProfileControl.us_ProfileSupported & CNFG_V2_PROFILE_ALL_MASK)
   {
      pt_VDSL2_I->uc_SI_SPar2_01 |= V2_PROFILES;
   }
   else
   {
      pt_VDSL2_I->uc_SI_SPar2_01 &= (~V2_PROFILES);
   }


   //SI - SPar2 octet 2
   pt_VDSL2_I->uc_SI_SPar2_02 = (uint8)(gul_OperationModeCapable_VDSL2 >> 16) & NEG_DELIMIT_2;   // NEG_DELIMIT_2 = 0x3F
   if (gt_ProfileControl.ul_AnxAUS0PsdSupported & CNFG_V2_ANXA_US0_ALL_MASK)
   {
      pt_VDSL2_I->uc_SI_SPar2_02 |= V2_ANNEX_A_US0_MASKS;
   }
   else
   {
      pt_VDSL2_I->uc_SI_SPar2_02 &= (~V2_ANNEX_A_US0_MASKS);
   }

   if (gt_ProfileControl.us_AnxBUS0PsdSupported & CNFG_V2_ANXB_US0_ALL_MASK)
   {
      pt_VDSL2_I->uc_SI_SPar2_02 |= V2_ANNEX_B_US0_MASKS;
   }
   else
   {
      pt_VDSL2_I->uc_SI_SPar2_02 &= (~V2_ANNEX_B_US0_MASKS);
   }

   if (gt_ProfileControl.us_AnxCUS0PsdSupported & CNFG_V2_ANXC_US0_ALL_MASK)
   {
      pt_VDSL2_I->uc_SI_SPar2_02 |= V2_ANNEX_C_US0_MASKS;
   }
   else
   {
      pt_VDSL2_I->uc_SI_SPar2_02 &= (~V2_ANNEX_C_US0_MASKS);
   }

   // Enable support of full vectoring (G.993.5 DS+US) operation
   // Note: The standard doesn't allow to indicate both full vectoring and any vectoring-
   //       friendly operation at the same time!
   if (gus_DsmCtrlCmv == G9935_FULL_VECTORING_SUPPORT)
   {
      pt_VDSL2_I->uc_SI_SPar2_02 |= V2_G_VECTOR_MASKS;  // Table 11.68.0.1 G.994.1
   }
   else
   {
      pt_VDSL2_I->uc_SI_SPar2_02 &= (~V2_G_VECTOR_MASKS);
   }
   // XDSLRTFW-1693 (Start_End)
   // Note: This code "(pt_VDSL2_I->uc_SI_SPar2_02 & V2_G_VECTOR_MASKS)" must be removed,
   //       when more bits are defined in the octet! Further the code commented out below
   //       (grep for XDSLRTFW-1693), must be commented in again!
   // Note: Comments why the if-case "(pt_VDSL2_I->uc_SI_SPar2_02 & V2_G_VECTOR_MASKS)" is correct:
   //       - Vectoring friendly doesn't mean G.993.5 capability.
   //       - G.998.4 Amd3 was done for allowing G.998.4 w/ extended memory in G.993.2 operation
   //         if and only if VTU-R has full G.993.5 capability.
   if ((gt_ReTXParams.t_ReTXConfigCMV.us_DsUsReTxCapable & (CNFG_CPE_DS_US_RETX_CAPABLE | CNFG_CPE_DS_RETX_CAPABLE)) &&
       (gt_ReTXParams.t_ReTXConfigCMV.us_DsUsReTxCapable & CNFG_RETX_EXTENDED_MEMORY_AMD3) &&
       (pt_VDSL2_I->uc_SI_SPar2_02 & V2_G_VECTOR_MASKS))
   {
      pt_VDSL2_I->uc_SI_SPar2_02 |= V2_9984_EXTENSIONS_MASKS;
   }

   //----------------- SI NPar3 ----------------------------
   // Profiles
   if(pt_VDSL2_I->uc_SI_SPar2_01 & V2_PROFILES)
   {
      // Octet 1
      pt_VDSL2_I->uca_Profiles[0] = (uint8)(gt_ProfileControl.us_ProfileSupported & (CNFG_V2_PROFILE_12xALL_MASK |
                                            CNFG_V2_PROFILE_8xALL_MASK));
      // Octet 2
         pt_VDSL2_I->uca_Profiles[1] = (uint8)((gt_ProfileControl.us_ProfileSupported >> 6) &
                                            ((CNFG_V2_PROFILE_35B_MASK | CNFG_V2_PROFILE_30A_MASK | CNFG_V2_PROFILE_17A_MASK) >> 6));
   }

   //FFT/IFFT size
   if(pt_VDSL2_I->uc_SI_SPar2_01 & V2_DFT_SIZE)
   {
      // This has a different format in VDSL2
      //gs_TxLog2IfftLength;
      pt_VDSL2_I->uc_fft_size = US_LOG2_FFT_LENGTH_8192;
      if (TX_LOG2_MAX_FFT_LENGTH > US_LOG2_FFT_LENGTH_8192)
      {
         pt_VDSL2_I->uc_fft_size |= 0x10;
      }
   }

   //Initial sample length of cyclic extension
   if(pt_VDSL2_I->uc_SI_SPar2_01 & V2_CE_LENGTH)
   {
      s_m_param = 1 << (gs_m - 2);

      pt_VDSL2_I->uca_CELength[0] = (s_m_param & NEG_DELIMIT_2);
      pt_VDSL2_I->uca_CELength[1] = ((s_m_param >> 6) & NEG_DELIMIT_2);
      pt_VDSL2_I->uca_CELength[2] = ((s_m_param >> 12) & NEG_DELIMIT_2);
   }

   // Annex A US0 PSD
   if(pt_VDSL2_I->uc_SI_SPar2_02 & V2_ANNEX_A_US0_MASKS)
   {
      // Octet 1
      pt_VDSL2_I->uca_US0_A_bands[0] = (uint8)(gt_ProfileControl.ul_AnxAUS0PsdSupported & 0x3F);
      // Octet 2
      pt_VDSL2_I->uca_US0_A_bands[1] = (uint8)((gt_ProfileControl.ul_AnxAUS0PsdSupported>>6) & 0x7);
      pt_VDSL2_I->uca_US0_A_bands[1] |= (uint8)((gt_ProfileControl.ul_AnxAUS0PsdSupported>>17) & 0x8);
      // Octet 3
      pt_VDSL2_I->uca_US0_A_bands[2] = (uint8)((gt_ProfileControl.ul_AnxAUS0PsdSupported>>9) & 0x3F);
      // Octet 4
      pt_VDSL2_I->uca_US0_A_bands[3] = (uint8)((gt_ProfileControl.ul_AnxAUS0PsdSupported>>15) & 0x7);
      pt_VDSL2_I->uca_US0_A_bands[3] |= (uint8)((gt_ProfileControl.ul_AnxAUS0PsdSupported>>18) & 0x8);
      // Octet 5
      pt_VDSL2_I->uca_US0_A_bands[4] = (uint8)((gt_ProfileControl.ul_AnxAUS0PsdSupported>>18) & 0x3);
   }

   // Annex B US0 PSD
   if(pt_VDSL2_I->uc_SI_SPar2_02 & V2_ANNEX_B_US0_MASKS)
   {
      // Octet 1
      pt_VDSL2_I->uca_US0_B_bands[0] = (uint8)(gt_ProfileControl.us_AnxBUS0PsdSupported & 0x7);
      // Octet 2
      pt_VDSL2_I->uca_US0_B_bands[1] = (uint8)((gt_ProfileControl.us_AnxBUS0PsdSupported>>3) & 0x3);
   }

   // Annex C US0 PSD
   if(pt_VDSL2_I->uc_SI_SPar2_02 & V2_ANNEX_C_US0_MASKS)
   {
      pt_VDSL2_I->uca_US0_C_bands[0] = (uint8)(gt_ProfileControl.us_AnxCUS0PsdSupported & 0x3);
      pt_VDSL2_I->uca_US0_C_bands[1] = (uint8)((gt_ProfileControl.us_AnxCUS0PsdSupported & 0x30)>>4);
      pt_VDSL2_I->uca_US0_C_bands[2] = (uint8)((gt_ProfileControl.us_AnxCUS0PsdSupported & 0x3000)>>12);
      // nothing to be done as the NPar3 octet for ANNEX_C_US0 is reserved for future
      // for ANNEX_C_US0 support yet.
   }

   // XDSLRTFW-1797 (Start_End)
   // G.993.2 Vectoring
   if(pt_VDSL2_I->uc_SI_SPar2_02 &  V2_G_VECTOR_MASKS)
   {
      // refer to Table 11.68.10 of G.994.1
      //XDSLRTFW-2820 (Start_End)
      pt_VDSL2_I->uc_Gvector = (V2_OP_VEC1_DURATION_EXT_MASK|V2_FDPS_US_MASK|V2_PILOT_SEQ_LEN_MULTI4_MASK|
                                   V2_GVECTOR_US_MASK|V2_GVECTOR_DS_MASK);

      if(TESTArray[TEST_XDSLRTFW_CONFIG] & TEST_XDSLRTFW_CONFIG_FLAG_ONLY)
      {
         pt_VDSL2_I->uc_Gvector |= V2_OP_VEC1_FLAG_TONES_MASK;
      }
   }

   // XDSLRTFW-1693 (Start_End)
   // G.998.4 Extensions
   if(pt_VDSL2_I->uc_SI_SPar2_02 & V2_9984_EXTENSIONS_MASKS)
   {
// Note: This code must be enabled, when more bits are defined in the octet!
//      if(pt_VDSL2_I->uc_SI_SPar2_02 & V2_G_VECTOR_MASKS)
      {
         pt_VDSL2_I->uc_G9984_Extensions = V2_G9984_ANNEXD_SUPPORT;
      }
//      else
//      {
//         pt_VDSL2_I->uc_GVecCapable = 0;
//      }
   }
}


/*^^^
 * ------------------------------------------------------------------
 * Name : InitializeStandardInfoField_GFast
 *
 * Abstract : Initialize an Information Field structure for G.Fast CO modem
 *
 * Comments: Initialize the various bits of G.Fast structure based on the configuration or capabilites.
 *
 * -----------------------------------------------------------------
 *^^^
 */
void InitializeStandardInfoField_GFast(G_Fast_Type_t *pt_G_Fast_I, InfoField_t *t_Info)
{
   // SI NPar2 Octet1
   // According to table 11.69 of ITU-T G.994.1 standard

   pt_G_Fast_I->uc_SI_NPar2 = (gul_OperationModeCapable_G_Fast & 0x3F);
   pt_G_Fast_I->uc_SI_SPar2_01 = ((gul_OperationModeCapable_G_Fast >> 8) & 0x3F);
   pt_G_Fast_I->uc_SI_SPar2_02 = ((gul_OperationModeCapable_G_Fast >> 16) & 0x3F);

   // Profiles
   if (pt_G_Fast_I->uc_SI_SPar2_01 & G_Fast_Profiles)
   {
      pt_G_Fast_I->uc_Profiles |= G_Fast_Profile_106;
   }
   else
   {
      pt_G_Fast_I->uc_Profiles &= ~G_Fast_Profile_106;
   }

   // Duration of Channel Discovery 1-1
   if ( pt_G_Fast_I->uc_SI_SPar2_01 & G_Fast_Duration_Channel_Discovery)
   {
      pt_G_Fast_I->uc_duration_channel_discovery = 0x3;
   }
   else
   {
      pt_G_Fast_I->uc_duration_channel_discovery = 0;
   }

   // Number of symbols in TDD  frame
   if (pt_G_Fast_I->uc_SI_SPar2_02 & G_Fast_Symbol_Period_TDD_Frame)
   {
      pt_G_Fast_I->uc_symbol_period_TDD_frame =36;
   }
}

/*^^^
 *-------------------------------------------------------------------
 *
 *  Name : PreHandshakeCodeswap()
 *
 *  Abstract :
 *
 *  Performs pre-handshake codeswap.
 *
 *  Input argument:
 *
 *  Output argument:
 *
 *  Returns: None
 *
 *  Global Variables Used:
 *
 *  Notes :
 *
 *-------------------------------------------------------------------
 *^^^
 */
FlagT PreHandshakeCodeswap(FlagT ft_trigger_codeswap)
{
   if (ft_trigger_codeswap)
   {
      FreeSwapHandle(&guc_PrimPageHandle);
      guc_RxSwapActivity = CODESWAP_LOAD(VDSL_GHS_2_PM_SWAPPAGE);
   }

   /* =============================================== */
   /* State machine to request swap page            */
   /* =============================================== */
   switch(guc_RxSwapActivity)
   {
   case CODESWAP_LOAD(VDSL_GHS_2_PM_SWAPPAGE):
      if ((guc_PrimPageHandle = RequestSwap(VDSL_GHS_2_PM_SWAPPAGE)) != INVALID_CODESWAP_HANDLE)
      {
         guc_RxSwapActivity = CODESWAP_CHECK(VDSL_GHS_2_PM_SWAPPAGE);
      }
      break;

   case CODESWAP_CHECK(VDSL_GHS_2_PM_SWAPPAGE):
      if (PollForCodeSwapDone(VDSL_GHS_2_PM_SWAPPAGE, guc_PrimPageHandle) == SWAP_DONE)
      {
         guc_RxSwapActivity = CODESWAP_DO_NOTHING;
         return (TRUE);
      }
      break;
   }

   return (FALSE);
}

