/* **COPYRIGHT******************************************************************
    INTEL CONFIDENTIAL
    Copyright (C) 2017 Intel Corporation
    Copyright (C), 1994-2006 Aware Inc. All Rights Reserved.
******************************************************************COPYRIGHT** */
/* **DISCLAIMER*****************************************************************
    The source code contained or described herein and all documents related
    to the source code ("Material") are owned by Intel Corporation or its
    suppliers or licensors. Title to the Material remains with Intel
    Corporation or its suppliers and licensors. The Material may contain
    trade secrets and proprietary and confidential information of Intel
    Corporation and its suppliers and licensors, and is protected by
    worldwide copyright and trade secret laws and treaty provisions. No part
    of the Material may be used, copied, reproduced, modified, published,
    uploaded, posted, transmitted, distributed, or disclosed in any way
    without Intel's prior express written permission.

    No license under any patent, copyright, trade secret or other
    intellectual property right is granted to or conferred upon you by
    disclosure or delivery of the Materials, either expressly, by
    implication, inducement, estoppel or otherwise. Any license under
    such intellectual property rights must be express and approved by
    Intel in writing.
*****************************************************************DISCLAIMER** */
/***********************************************************************
 *
 * Aware DMT Technology. Proprietary and Confidential.
 *
 * ADDRESS:          40 Middlesex Turnpike, Bedford, MA 01730-1413 USA
 * TELEPHONE:        781.276.4000
 * FAX:              781.276.4001
 * WEB:              http://www.aware.com
 *
 * FILE:             hs_mesg.c
 *
 * DESCRIPTION:      Message handling functions used in Handshaking.
 *
  **********************************************************************/
//*****************************************************************************************/
// hs_mesg.c
//  XDSLRTFW-468 : CORRECT_VDSL_OCTETS_LOG_FOR_DEBUG
//    Issue with Octet logging for debuuging.
//    Casting uca_Msg to Unsigned INT8
//
//  XDSLRTFW-456 : CORRECT_ADSL_OCTETS_IN_VDSL_GHS_MSG
//    Issue with Sending Wrong ADSL Octets in VDSL Code when configured
//    MultiMode: Fixed with this checkin
//
//    06/05/2013 Varun : Added code(Debug option) to capture raw data of all Tx and Rx(All GHs Rx messages in
//                 particular) training messages (G.Hs + training)
//                 Grep for "XDSLRTFW-598 FEATURE_ALL_ALL_ALL_Debug_Buffer"
//
// 10/07/2013 Sriram Shastry : Bug in check condition for setting up error code to indicate CO perfers ADSL mode
//            Grep for XDSLRTFW-1005 : BugFix_ALL_ALL_ALL_Hsk_LookforADSLAnnexbits
// 03/07/2014 Fuss : Skip reserved bit of ID_SPar1 octet 1 (Note: In array it is element 0)
//            Grep for XDSLRTFW-1892
// 26/06/2014 Fuss: Extended O-P-Vector1 phase, i.e. setting hostmessages
//            Grep for XDSLRTFW-1797
// 17/11/2015 Fuss: O-Signature fail with "E_CODE_INVALID_FE_IDFT_SIZE" against new 35b capable DSLAMs
//            Grep for XDSLRTFW-2512
//*****************************************************************************************/
#include <string.h>
#include "common.h"
#include "gdata.h"
#include "ghs.h"
#include "cmv_data.h"
#include "cmv.h"
#include "fcs.h"
#include "DebugBuffer.h"
#include "profile.h"


void PackBandInfoTo16Bit(uint8 *puca_BandSrc, int16 *pusa_BandDst, uint16 us_NumBands, uint8 uc_Offset);
void UnpackMsgBufBandInfo(uint8 *uca_M, int16 *ps_idx, uint8 *puca_UpackedBandInfod, uint8 *puc_NumBands, uint8 uc_vdsl_type);
void Unpack32BitTo8BitArray(uint32 *pul_Src, uint8 *puca_Dst);
void PackBandInfo2MsgBuf(uint8 uca_M[], int16 *ps_idx, uint8 *uca_BandInfo, uint8 uc_NumBands);
void DebugGHSMessages(int8 uc_Type, int16 s_MsgLen, uint8 *uca_Msg, uint16 us_Delimiter);

//*************************************************
//
#ifdef INCLUDE_NON_CACHE_FUNCTIONS
//
//*************************************************

void DebugGHSMessages(int8 uc_Type, int16 s_MsgLen, uint8 *uca_Msg, uint16 us_Delimiter)
{
   uint16 i, us_Msg;
#ifdef DEBUG_TRAIL
   // Add delimiters
   DebugTrail1(2,DEBUG_TRAIL_GHS_ENABLE,0x0000, us_Delimiter, (int16)gl_RxSymbolCount);
#endif // DEBUG_TRAIL
   // If valid message (i.e. not a FCS check log)
   if (s_MsgLen)
   {
      // Add message type and message length
      us_Msg = (s_MsgLen<<8) | uc_Type;
#ifdef DEBUG_TRAIL
      DebugTrail1(1,DEBUG_TRAIL_GHS_ENABLE,0x0000, us_Msg);
#endif // DEBUG_TRAIL
      // Add message bytes
      for (i = 0; i < s_MsgLen; i+=2)
      {
//XDSLRTFW-468 : CORRECT_VDSL_OCTETS_LOG_FOR_DEBUG (START)
         // If message length is odd, stuff a 0 as the last byte
         if (i == (s_MsgLen-1))
         {
            us_Msg = ((uint16)0<<8) | (uint16)uca_Msg[i];
         }
         else
         {
            us_Msg = ((uint16)uca_Msg[i+1]<<8) | (uint16)uca_Msg[i];
         }
#ifdef DEBUG_TRAIL
         DebugTrail1(1,DEBUG_TRAIL_GHS_ENABLE,0x0000, us_Msg);
#endif // DEBUG_TRAIL
//XDSLRTFW-468 : CORRECT_VDSL_OCTETS_LOG_FOR_DEBUG (END)
      }
   }
}

void PackBandInfo2MsgBuf(uint8 uca_M[], int16 *ps_idx, uint8 *uca_BandInfo, uint8 uc_NumBands)
{
   int16 i, idx, s_tones;

   idx = *ps_idx;

   //G993.2 GHS spec. specifies that each tone is represented by 13 bits (i.e., 3 bytes).

   s_tones = uc_NumBands*6;

   for(i=0; i<s_tones; i++)
   {
      uca_M[idx++] = uca_BandInfo[i];
   }

   uca_M[idx-1] |= DELIMIT_2;

   *ps_idx = idx;
}

void PackBandInfoTo16Bit(uint8 *puca_BandSrc, int16 *pusa_BandDst, uint16 us_NumBands, uint8 uc_Offset)
{
   unsigned int i;
   unsigned int ArrayIdx;

   for(i = 0; i < us_NumBands; i++)
   {
      ArrayIdx = ((i*6) + uc_Offset);
      pusa_BandDst[i] = ((puca_BandSrc[ArrayIdx] << 12) |
                         (puca_BandSrc[ArrayIdx+1] << 6) |
                         (puca_BandSrc[ArrayIdx+2]));
   }
}

void Unpack32BitTo8BitArray(uint32 *pul_Src, uint8 *puca_Dst)
{
   unsigned int i;

   for (i = 0; i < 4; i++)
   {
      puca_Dst[i] = (uint8) (*pul_Src >> ((3-i) * 8));
   }
}


/*^^^
 *-------------------------------------------------------------------
 *
 *  Name: VDSL_UnpackBandsInfo
 *
 *  Abstract:
 *
 *  This function unpacks data from the input message buffer and
 *  put them into the output word array in the RX info structure.
 *  Each pair of the input octets are put into one output 16-bit word in the way such that
 *   the x LSBs of the first octet is placed in the bit 6+x-1 to 6 locations of the output word
 *  and the 6 LSBs of the second input octet is placed in the 6 LSBs of the output octet,
 *  until delimiter is encountered.
 *
 *
 *  Parameters:
 *      uint8 *uca_M : the pointer to the input message buffer
 *      int16 *p_idx : the pointer to index into the input buffer uc_M[]
 *      uint8 *pusa_OutDataBuf: the pointer to the first word of output word array
 *      uint8 s_NumOfInputData: the number of input octet pairs
 *
 *   Return: None
 *
 *  Global Variables Used: None
 *
 *  Notes:
 *
 *-------------------------------------------------------------------
 *^^^
 */
void UnpackMsgBufBandInfo(uint8 *uca_M, int16 *ps_idx, uint8 *puca_UpackedBandInfod, uint8 *puc_NumBands, uint8 uc_vdsl_type)
{
   int16 i, j, k, idx, s_NumBands;
   int16 s_NumBytesPerBand;

   idx = *ps_idx;
   s_NumBands = 0;


   // Unpack the maximum of 16 bands
   for(i=0; i<16; i++)
   {

      if(uc_vdsl_type == 2)
      {
         s_NumBytesPerBand = 6;
      }
      else
      {
         s_NumBytesPerBand = 4;
      }

      //Each band has "s_NumBytesPerBand" octets
      k = i*s_NumBytesPerBand;
      for(j=0; j<s_NumBytesPerBand; j++)
      {

         // Check if done with all the used bands set
         if ((uca_M[idx] & DELIMIT_2) != 0)
         {
            // If not all s_NumBytesPerBand octets are received, set all octets to 0 and exit, discard the rest of octets
            // in this message
            if ( j != (s_NumBytesPerBand-1))
            {

               idx -= j;
               goto _return;
            }
            else
            {
               // Force exiting the loop
               i = 16;
            }

         }

         puca_UpackedBandInfod[k++] = uca_M[idx++] & NEG_DELIMIT_2;
      }

      s_NumBands++;
   }

_return:

   *ps_idx = idx;
   *puc_NumBands = (uint8)s_NumBands;
}


/*
 *------------------------------------------------------------------------
 *
 *  Name : SkipExtraOctetAndCheckSegmentation
 *
 *  Abstract :   Function to ignore extra octets if there are any. This function is
 *            also used for checking whether current message has been segmented.
 *            If current message index is larger than message length, and the
 *            number of decoded NPar3 blocks is smaller than the number of active
 *            NPar3 blocks, then this message has been segmented, and we need to
 *            exit function DecodeStandard() with an "INCOMPLETE_STANDARD" return.
 *
 *  Input Arguments:
 *      uint8 uca_M[]       - array of received message octets
 *      idx                 - index of the starting position of the message array
 *      s_MsgLen            - total number of octets in uca_M
 *      s_delimiter         - octet delimiter
 *      s_ActiveBlocks      - total number of NPar3 active blocks to be decoded
 *      s_DecodedBlocks      - current number of NPar3 block decoded
 *
 *  Returns:
 *      idx                 - index of the ending position of the message array
 *                       if message has not been segmented.
 *
 *      0               - If message has been segmented.
 *
 *  Global Variables Used:
 *
 *  Notes:
 *
 *------------------------------------------------------------------------
 */
int16 SkipExtraOctetAndCheckSegmentation(uint8 uca_M[], int16 idx, int16 s_MsgLen, int16 s_delimiter, int16 s_ActiveBlocks, int16 s_DecodedBlocks)
{

   // check for segmented message
   if ( (idx > s_MsgLen) && (s_DecodedBlocks <= s_ActiveBlocks) )
   {
      return (0);
   }

   while ( (uca_M[idx-1] & s_delimiter) == 0 )
   {
      // skip extra octets
      idx++;
      // check for segmented message after skipping extra octets
      if ( (idx > s_MsgLen) && (s_DecodedBlocks <= s_ActiveBlocks) )
      {
         return (0);
      }
   }
   return(idx);
}


/*
 *------------------------------------------------------------------------
 *
 *  Name : SkipCountExtraOctetAndCheckSegmentation
 *
 *  Abstract :   Function to ignore extra octets if there are any. This function is
 *            also used for checking whether current message has been segmented.
 *            If current message index is larger than message length, and the
 *            number of decoded NPar3 blocks is smaller than the number of active
 *            NPar3 blocks, then this message has been segmented, and we need to
 *            exit function DecodeStandard() with an "INCOMPLETE_STANDARD" return.
 *              In this function, we also count how many bits are set to 1 in the
 *              ignored bytes and save the number in gus_ExtraBits.
 *
 *  Input Arguments:
 *      uint8 uca_M[]       - array of received message octets
 *      idx                 - index of the starting position of the message array
 *      s_MsgLen            - total number of octets in uca_M
 *      s_delimiter         - octet delimiter
 *      s_ActiveBlocks      - total number of NPar3 active blocks to be decoded
 *      s_DecodedBlocks      - current number of NPar3 block decoded
 *
 *  Returns:
 *      idx                 - index of the ending position of the message array
 *                       if message has not been segmented.
 *
 *      0               - If message has been segmented.
 *
 *  Global Variables Used:
 *
 *  Notes:
 *
 *------------------------------------------------------------------------
 */
int16 SkipCountExtraOctetAndCheckSegmentation(uint8 uca_M[], int16 idx, int16 s_MsgLen, int16 s_delimiter, int16 s_ActiveBlocks, int16 s_DecodedBlocks)
{
   int16 i;

   // check for segmented message
   if ( (idx > s_MsgLen) && (s_DecodedBlocks <= s_ActiveBlocks) )
   {
      return (0);
   }

   while ( (uca_M[idx-1] & s_delimiter) == 0 )
   {
      for(i=0; i<7; i++)
      {
         if(uca_M[idx]>>i & 0x1)
         {
            gus_ExtraBits++;
         }
      }

      // skip extra octets
      idx++;
      // check for segmented message after skipping extra octets
      if ( (idx > s_MsgLen) && (s_DecodedBlocks <= s_ActiveBlocks) )
      {
         return (0);
      }
   }
   return(idx);
}




/*^^^
 *-------------------------------------------------------------------
 *
 *  Name : CopyInfoField
 *
 *  Abstract :
 *
 *  CopyInfoField() - Copy source Information Field structure to a
 *  to a target Information Field
 *
 *  Parameters:
 *      InfoField *t_Source : information to be copied from
 *      InfoField *t_Target : information to be copied to
 *
 *  Returns: None
 *
 *  Global Variables Used:
 *
 *  Notes :
 *
 *-------------------------------------------------------------------
 *^^^
 */

void CopyInfoField(InfoField_t t_Source[], InfoField_t *t_Target)
{

   memcpy(t_Target, t_Source, sizeof(InfoField_t));

}   // CopyInfoField


/*^^^
 *-------------------------------------------------------------------
 *
 *  Name: SetArray2Struct
 *
 *  Abstract:
 *
 *  This function gets the data from the received message buffer and
 *  put them into the RX info structure
 *
 *
 *  Parameters:
 *      uint8 *uca_M : the pointer to the input message buffer
 *      int16 *p_idx : the pointer to index into the input buffer uc_M[]
 *      uint8 *puca_OutDataBuf: the pointer to the first octet of output data
 *      uint8 s_NumOfInputData: the number of input octets
 *
 *   Return: None
 *
 *  Global Variables Used: None
 *
 *  Notes:
 *
 *-------------------------------------------------------------------
 *^^^
 */

void SetArray2Struct(uint8 *uca_M, int16 *ps_idx, uint8 *puca_OutDataBuf, int16 s_NumOfInputData)
{
   int16 i, idx;
   uint8 uc_temp;

   idx = *ps_idx;

   for(i=0; i<s_NumOfInputData; i++)
   {
      uc_temp = uca_M[idx++] & (uint8)NEG_DELIMIT_2;
      if (uc_temp != (uint8)NEG_DELIMIT_2)
      {
         puca_OutDataBuf[i] = uc_temp;
      }

      if ( (uca_M[idx-1] & DELIMIT_2) != 0 )
      {
         break;
      }
   }

   *ps_idx = idx;
}

/*^^^
 *-------------------------------------------------------------------
 *
 *  Name: Common_CL_InfoSave
 *
 *  Abstract:
 *
 *  This function saves Common info from CL or CLR message
 *
 *
 *  Parameters:
 *
 *   Return: None
 *
 *  Global Variables Used: None
 *
 *  Notes:
 *
 *-------------------------------------------------------------------
 *^^^
 */
void Common_CL_InfoSave(void)
{
   G994_VendorInformation_t *pt_VendorInformation;

   // Initialization of local variables
   pt_VendorInformation = &gt_fe_VendorInformation;

   // In 997.1 rev3, see Table 7-4 Vendor ID information block (8 octets)
   gus_fe_CountryCode = gpt_RxInfo->us_Country;
   pt_VendorInformation->uca_CountryCode[0] = gpt_RxInfo->us_Country >> 8;
   pt_VendorInformation->uca_CountryCode[1] = (uint8)gpt_RxInfo->us_Country;

   // Save G994.1 Vendor ID
   gul_fe_G994VendorID = gpt_RxInfo->ul_Provider;
   Unpack32BitTo8BitArray(&gpt_RxInfo->ul_Provider, &pt_VendorInformation->uca_VendorId[0]);

   pt_VendorInformation->uca_VendorSpecific[0] = gpt_RxInfo->us_VendorInfo >> 8;
   pt_VendorInformation->uca_VendorSpecific[1] = (uint8)gpt_RxInfo->us_VendorInfo;
   gus_fe_G994VendorSpecific = gpt_RxInfo->us_VendorInfo;

   // Non-Standard-Field
   gus_fe_CountryCode_NS = gpt_RxInfo->us_Country_NS;
   gul_fe_G994VendorID_NS = gpt_RxInfo->ul_Provider_NS;

   // Check if the CL message contains the support of RX tones > 6000.
   if (gpt_RxInfo->ul_Provider_NS == BRCM_VENDOR_ID)
   {
      NonStandardBlock_t *pt_NSB;
      uint8 *pData;

      pt_NSB = &(gpt_RxInfo->ta_NS_Info[0]);

      if (pt_NSB->uc_NSLen == 0x08)
      {
         pData = &pt_NSB->uca_NSVendorSpec[0];
         // No tone ordering nor 1-bit allocation constraints apply:
         //    - NSIF block with NSIF ID = 0x6, defined above, is absent
         //    - NSIF value in the non-standard information block has its LSB set to 0
         if((*pData++ == 0x06) && (*pData & 0x01))
         {
            gul_35bLiteConfig |= EN_BDCM_DS_TONES_GT_6000;
         }
      }
   }

   // Save FE operating modes
   gt_FE_OperatingModes.uca_SupportedModes[0] = gpt_RxInfo->uc_SI_SPar1;
   gt_FE_OperatingModes.uca_SupportedModes[1] = gpt_RxInfo->uc_SI_SPar1_02;
   gt_FE_OperatingModes.uca_SupportedModes[2] = gpt_RxInfo->uc_SI_SPar1_03;
   gt_FE_OperatingModes.uca_SupportedModes[3] = gpt_RxInfo->uc_SI_SPar1_04;
   gt_FE_OperatingModes.uca_SupportedModes[4] = gpt_RxInfo->uc_SI_SPar1_05;
   gt_FE_OperatingModes.uca_SupportedModes[5] = 0;
#ifdef VDSL_BONDING
   if ( ((gpt_RxInfo->uc_ID_SPar1_03 & BONDING)!=0) && ((gpt_RxInfo->uc_BondingNP2 & BONDING_ETHERNET)!=0) )
   {
      gt_Bonding_DiscAggr_Status.s_PAF_status |= 0x1;      //Remote PAF supported
   }
#endif

}

/*^^^
 *-------------------------------------------------------------------
 *
 *  Name: VDSL2_CL_InfoSave
 *
 *  Abstract:
 *
 *  This function saves VDSL2 specific info from CL or CLR message
 *
 *
 *  Parameters:
 *
 *   Return: None
 *
 *  Global Variables Used: None
 *
 *  Notes:  CMD_HS_StandardInfoFE_VDSL2Get
 *          Message Identifier: 0xCC03
 *          CMV: INFO 204
 *
 *-------------------------------------------------------------------
 *^^^
 */
// !!!!!!!!!!!!!!
// I would show always all capabilities transmitted from the counter part. Why masking must be done? To be discussed!
// !!!!!!!!!!!!!!
void VDSL2_CL_InfoSave(void)
{
   int16 i;
   uint8 *puc_tmpptr;

   // Save FE VDSL2 standard info

   // SI Npar2, Spar2_01 settings
   gt_VDSL2_FE_StdInfo_CL.us_Npar2Spar2 = gpt_RxInfo->t_VDSL2_Info.uc_SI_NPar2;
   gt_VDSL2_FE_StdInfo_CL.us_Npar2Spar2 |= ((uint16)gpt_RxInfo->t_VDSL2_Info.uc_SI_SPar2_01 << 8);

   // Profiles
   gt_VDSL2_FE_StdInfo_CL.us_ProfileSupported = (uint16)(gpt_RxInfo->t_VDSL2_Info.uca_Profiles[0] &
         ((V2_PROFILE_8D|V2_PROFILE_8C|V2_PROFILE_8B|V2_PROFILE_8A) |
          (V2_PROFILE_12B|V2_PROFILE_12A)));
   gt_VDSL2_FE_StdInfo_CL.us_ProfileSupported |= (uint16)((gpt_RxInfo->t_VDSL2_Info.uca_Profiles[1] &
         NEG_DELIMIT_2) << 6);
// Note: Up to now following profiles are defined in second Ghs octet:
//            V2_PROFILE_17A
//            V2_PROFILE_30A
//            V2_PROFILE_35B

   // Annex A US0 PSD
   gt_VDSL2_FE_StdInfo_CL.ul_AnxAUS0PsdSupported = ((uint32)gpt_RxInfo->t_VDSL2_Info.uca_US0_A_bands[0] & 0x3f);
   gt_VDSL2_FE_StdInfo_CL.ul_AnxAUS0PsdSupported |= (((uint32)gpt_RxInfo->t_VDSL2_Info.uca_US0_A_bands[1] & 0x7)<<6);
   gt_VDSL2_FE_StdInfo_CL.ul_AnxAUS0PsdSupported |= (((uint32)gpt_RxInfo->t_VDSL2_Info.uca_US0_A_bands[2] & 0x3f)<<9);
   gt_VDSL2_FE_StdInfo_CL.ul_AnxAUS0PsdSupported |= (((uint32)gpt_RxInfo->t_VDSL2_Info.uca_US0_A_bands[3] & 0x7)<<15);
   gt_VDSL2_FE_StdInfo_CL.ul_AnxAUS0PsdSupported |= (((uint32)gpt_RxInfo->t_VDSL2_Info.uca_US0_A_bands[4] & 0x3)<<18);
   gt_VDSL2_FE_StdInfo_CL.ul_AnxAUS0PsdSupported |= (((uint32)gpt_RxInfo->t_VDSL2_Info.uca_US0_A_bands[1] & 0x8)<<17);
   gt_VDSL2_FE_StdInfo_CL.ul_AnxAUS0PsdSupported |= (((uint32)gpt_RxInfo->t_VDSL2_Info.uca_US0_A_bands[3] & 0x8)<<18);

   // Annex B US0 PSD
   gt_VDSL2_FE_StdInfo_CL.us_AnxBUS0PsdSupported = ((uint16)gpt_RxInfo->t_VDSL2_Info.uca_US0_B_bands[0] & 0x7);
   gt_VDSL2_FE_StdInfo_CL.us_AnxBUS0PsdSupported |= (((uint16)gpt_RxInfo->t_VDSL2_Info.uca_US0_B_bands[1] & 0x3)<<3);

   // Annex C US0 PSD
   gt_VDSL2_FE_StdInfo_CL.us_AnxCUS0PsdSupported = ((uint16)(gpt_RxInfo->t_VDSL2_Info.uca_US0_C_bands[0] &
         (V2_US0_C_25_276_B|V2_US0_C_25_138_B)));
   gt_VDSL2_FE_StdInfo_CL.us_AnxCUS0PsdSupported |= ((uint16)((gpt_RxInfo->t_VDSL2_Info.uca_US0_C_bands[1] &
         (V2_US0_C_25_276_CO|V2_US0_C_25_138_CO)) <<4));
   gt_VDSL2_FE_StdInfo_CL.us_AnxCUS0PsdSupported |= ((uint16)((gpt_RxInfo->t_VDSL2_Info.uca_US0_C_bands[2] &
         (V2_US0_C_IN_17A|V2_US0_C_IN_12B))<<12));
   // G.993.2 Vectoring and G.998.4 Extensions
   {
      uint16 us_G9935_G9984_Supported;

      // XDSLRTFW-1797 (Start_End)
      us_G9935_G9984_Supported = ((uint16)(gpt_RxInfo->t_VDSL2_Info.uc_Gvector & (V2_OP_VEC1_FLAG_TONES_MASK |
                                           V2_OP_VEC1_DURATION_EXT_MASK |
                                           V2_FDPS_US_MASK |
                                           V2_PILOT_SEQ_LEN_MULTI4_MASK |
                                           V2_GVECTOR_US_MASK |
                                           V2_GVECTOR_DS_MASK)));

      //XDSLRTFW-1693 (Start_End)
      // ITU-T G.998.4 Annex D support (CMV: RtxExtMemSup8 - G.998.4 Anx DSupported Bit 8 (VDSL only))
      // Note: If set to ONE, this bit indicates that the VTU-O supports ITU-T G.998.4 Annex D.
      //       This bit may only be set to ONE if the VTU-O transceiver is ITU-T G.993.5
      //       capable, but the bit "ITU-T G.993.5" is set to ZERO in the Spar(2) octet 2
      //       of ITU-T G.993.2; otherwise this bit shall be set to ZERO.
      us_G9935_G9984_Supported |= ((uint16)(gpt_RxInfo->t_VDSL2_Info.uc_G9984_Extensions & NEG_DELIMIT_2) << 8);

      gt_VDSL2_FE_StdInfo_CL.us_G9935_G9984_Supported = us_G9935_G9984_Supported;
   }

   // IDFT size
   // Note: Masking of the value is done in function "DecodeStandard_VDSL2()"
   gt_VDSL2_FE_StdInfo_CL.us_IDFTSize = (uint16)gpt_RxInfo->t_VDSL2_Info.uc_fft_size;
   //Get the IFFT size used by the remote modem
   gs_FeModemLog2IfftSize = (uint16)gpt_RxInfo->t_VDSL2_Info.uc_fft_size;

   // CE length ("m"-value gets be transmitted, with m = [2 to 16])
   // Note: m = 5 is mandatory!
   puc_tmpptr = (uint8*)(void *)&gt_VDSL2_FE_StdInfo_CL.us_CELengthSupported[0];
   for(i=0; i<3; i++)
   {
      *puc_tmpptr++ = gpt_RxInfo->t_VDSL2_Info.uca_CELength[i];
   }
}


/*^^^
 *-------------------------------------------------------------------
 *
 *  Name: Compare_CL_CLR
 *
 *  Abstract: Compare CL/CLR content after the exchange
 *          Decide how to respond to the received message
 *
 *  Returns: Flag indicate whether we should resend CLR or not, it also indicates what causes resending CLR so that
 *           we can check internally
 *           0x0000:  Do not need to resend CLR
 *           0x0008:  CLR/CL have no common mode which will trigger clear down.
 *
 *  Global Variables Used:
 *          gpt_RxInfo             - (I) received CL message
 *          gpt_TxInfo             - (O) message to be transmitted in response
 *          gpt_TxInfoSave         - (O) a duplicate of the received message
 *
 *-------------------------------------------------------------------
 *^^^
 */

FlagT Compare_CL_CLR(void)
{
   FlagT ft_retCode;
   InfoField_t *pt_RxCL;   // received capability list of the other party
   InfoField_t *pt_TxCL;   // transmitted capability list of CPE


   pt_RxCL = gpt_RxInfo;
   pt_TxCL = gpt_TxInfoSave;

   // initialize the return code with  "No common mode"
   ft_retCode = 8;

   // Check if VDSL2 is supported at both ends via SI_SPar1 octet 5/2
   if((pt_RxCL->uc_SI_SPar1_05 & pt_TxCL->uc_SI_SPar1_05) & G993_2)
   {
      ft_retCode = 0;
   }

   return (ft_retCode);
}


//*************************************************
//
#endif // #ifdef INCLUDE_NON_CACHE_FUNCTIONS
//
//*************************************************


//*************************************************
//
#ifdef INCLUDE_CACHE_FUNCTIONS
//
//*************************************************

/*
 *------------------------------------------------------------------------
 *
 *  Name : FormStandard
 *
 *  Abstract :
 *
 *  Enclose standard information field in an arrary.
 *
 *  Parameters:
 *      const InfoField *t_Info : information to be packed
 *      uint8 uca_M[]           : output arrary containing the information
 *      idx                     : index of the starting position of the message array
 *
 *  Returns:
 *      idx                     : index of the ending position of the message array
 *
 *  Global Variables Used:
 *
 *
 *  Notes : DELIMIT_1 is used to set bit 8 of an octet to indicate the last octet
 *         in the NPar(1) and SPar(1) blocks.
 *         DELIMIT_2 is used to set bit 7 of an octet to indicate the last octet
 *         in the NPar(2) and SPar(2) blocks.
 *         DELIMIT_2 | DELIMIT_1 is used to set bits 8 and 7 of an octet to indicate
 *         only NPar(2) octets.
 *
 *------------------------------------------------------------------------
 */

int16 FormStandard_ADSL(ADSL_Type_t *pt_Info_ADSL, InfoField_t *t_Info, uint8 uca_M[], int16 idx);
int16 FormStandard_VDSL2(VDSL2_Type_t *pt_Info, uint8 uca_M[], int16 idx);
int16 FormStandard_G_Fast(G_Fast_Type_t *pt_Info_G_Fast, InfoField_t *t_Info, uint8 uca_M[], int16 idx);

int16 FormStandard(InfoField_t *t_Info, uint8 uca_M[], int16 idx)
{
   uint8 uc_Delimit = DELIMIT_2 | DELIMIT_1;
   uint16 us_temp;
   int16 i;
   uint8 *puc_BytePtr;

   //==== Service and Channel parameters (for CL, CLR, MS only) ====

   //---- ID_NPar1 ------
   //--------------------
   uca_M[idx++] = t_Info->uc_ID_NPar1 | DELIMIT_1;

   //---- ID_SPar1 ------
   //--------------------
   if(t_Info->uc_ID_SPar1_03 & 0x7F)
   {
      // If bonding bit is set, we have to send up to ID_SPar1_03
      // irrespective of the number if received ID_Spar1's
      uca_M[idx++] = t_Info->uc_ID_SPar1;
      uca_M[idx++] = t_Info->uc_ID_SPar1_02;
      uca_M[idx++] = (t_Info->uc_ID_SPar1_03 | DELIMIT_1);
   }
   else if((t_Info->uc_ID_SPar1_02 & 0x3F))
   {
      uca_M[idx++] = t_Info->uc_ID_SPar1;
      uca_M[idx++] = (t_Info->uc_ID_SPar1_02 | DELIMIT_1);
   }
   else
   {
      uca_M[idx++] = (t_Info->uc_ID_SPar1 | DELIMIT_1);
   }

   //---- ID_Par2 ------
   //--------------------
   // NPar2 - net data rate upstream
   if(t_Info->uc_ID_SPar1 & NDR_UP)
   {
      uca_M[idx++] = t_Info->uc_UpMaxNDR;
      uca_M[idx++] = t_Info->uc_UpMinNDR;
      uca_M[idx++] = (t_Info->uc_UpAvgNDR | uc_Delimit);
   }
   // NPar2 - net data rate downstream
   if(t_Info->uc_ID_SPar1 & NDR_DN)
   {
      uca_M[idx++] = t_Info->uc_DnMaxNDR;
      uca_M[idx++] = t_Info->uc_DnMinNDR;
      uca_M[idx++] = (t_Info->uc_DnAvgNDR | uc_Delimit);
   }
   // NPar2 - data flow characteristics upstream
   if(t_Info->uc_ID_SPar1 & DFC_UP)
   {
      uca_M[idx++] = t_Info->uc_UpMaxLat;
      uca_M[idx++] = (t_Info->uc_UpAvgLat | uc_Delimit);
   }
   // NPar2 - data flow characteristics downstream
   if(t_Info->uc_ID_SPar1 & DFC_DN)
   {
      uca_M[idx++] = t_Info->uc_DnMaxLat;
      uca_M[idx++] = (t_Info->uc_DnAvgLat | uc_Delimit);
   }
   // NPar2 - xTU-R splitter information
   if(t_Info->uc_ID_SPar1 & R_SPLITTER)
   {
      uca_M[idx++] = (t_Info->uc_ID_NPar2_O11 | uc_Delimit);
   }
   // NPar2 - xTU-C splitter information
   if(t_Info->uc_ID_SPar1 & C_SPLITTER)
   {
      uca_M[idx++] = (t_Info->uc_ID_NPar2_O12 | uc_Delimit);
   }
   // Par2 - Transceiver ID information
   if(t_Info->uc_ID_SPar1 & TRANSCEIVER_ID)
   {
      // NPar2
//      uca_M[idx++] = t_Info->uc_ID_NPar2_TransId | DELIMIT_2;
      uca_M[idx++] = DELIMIT_2;
      // SPar2
      uca_M[idx++] = (t_Info->uc_ID_SPar2_TransId | DELIMIT_2);

      // NPar3 - Network side transceiver ID
      if(t_Info->uc_ID_SPar2_TransId & CO_TRANSCEIVER_ID)
      {
         // NPar3 octets
         for (i = 0; i < NUM_TRANSCEIVER_ID; i++ )
         {
            uca_M[idx++] = (uint8) ((t_Info->ul_CoTransId >> (((NUM_TRANSCEIVER_ID-1)-i) * PAR2_NUM_BIT)) & NEG_DELIMIT_2);
         }
         // End of PAR3 block
         uca_M[idx-1] |= DELIMIT_2;
      }
      // NPar3 - Remote side transceiver ID
      if(t_Info->uc_ID_SPar2_TransId & CPE_TRANSCEIVER_ID)
      {
         // NPar3 octets
         for (i = 0; i < NUM_TRANSCEIVER_ID; i++ )
         {
            uca_M[idx++] = (uint8) ((t_Info->ul_CpeTransId >> (((NUM_TRANSCEIVER_ID-1)-i) * PAR2_NUM_BIT)) & NEG_DELIMIT_2);
         }
         // End of PAR3 block
         uca_M[idx-1] |= DELIMIT_2;
      }

      // End of PAR2 block, i.e. Transceiver ID information
      uca_M[idx-1] |= DELIMIT_1;
   }

   // Pack both t_Info->uc_ID_SPar1_02 and t_Info->uc_ID_SPar1_03 into one word
   us_temp = ((t_Info->uc_ID_SPar1_02 & RPL_MASK_OCTET2) | (t_Info->uc_ID_SPar1_03 << 6));
   // NPAR2 blocks for every SPar1_02/03 bit set
   // TX attenuation: UpA43, DnA43, UpB43, DnB43, UpC43 and DnC43
   //                 UpA4, DnA4, UpA43c and DnA43c
   puc_BytePtr = &(t_Info->uc_UpA43Att);
   for(i=0; i < RPL_BUF_SIZE_0; i++)
   {
      if(us_temp & 1)
      {
         uca_M[idx++] = (*puc_BytePtr++ | uc_Delimit);
      }
      us_temp >>= 1;
   }

#ifdef VDSL_BONDING
   //Form bonding Par2/3 block
   if(t_Info->uc_ID_SPar1_03 & BONDING)
   {
      uca_M[idx++] = t_Info->uc_BondingNP2 | DELIMIT_2;
      uca_M[idx++] = t_Info->uc_BondingSP2 | DELIMIT_2;

//      DebugTrail1(1,DEBUG_TRAIL_BONDING_TRAIL_ENABLE,0,(int16)0xFE00);

      // Remote discovery register
      if(t_Info->uc_BondingSP2 & BONDING_PME_DISCOVERY)
      {
//         DebugTrail1(1,DEBUG_TRAIL_BONDING_TRAIL_ENABLE,0,(int16)0xFE01);

         for (i=0; i<9 ; i++)
         {
//            DebugTrail1(1,DEBUG_TRAIL_BONDING_TRAIL_ENABLE,0,(int16)uca_M[idx]);
            uca_M[idx++] = t_Info->uca_BondingDisc[i];
         }
         uca_M[idx-1] |= DELIMIT_2;
      }

      // PME Aggr register
      if(t_Info->uc_BondingSP2 & BONDING_PME_AGGREGATION)
      {
//         DebugTrail1(1,DEBUG_TRAIL_BONDING_TRAIL_ENABLE,0,(int16)0xFE02);

         for (i=0; i<6 ; i++)
         {
//            DebugTrail1(1,DEBUG_TRAIL_BONDING_TRAIL_ENABLE,0,(int16)uca_M[idx]);
            uca_M[idx++] = t_Info->uca_BondingAggr[i];
         }
         uca_M[idx-1] |= DELIMIT_2;
      }

      // PME ID register
      if(t_Info->uc_BondingSP2 & BONDING_PME_IDENTIFICATION)
      {
         uca_M[idx++] = t_Info->uc_BondingId | DELIMIT_2;
      }

      uca_M[idx-1] |= DELIMIT_1;
   }
#endif
   //Pack both t_Info->uc_ID_SPar1_03 (post bonding bit) and t_Info->uc_ID_SPar1_04 into one word
   us_temp = (((t_Info->uc_ID_SPar1_03 & NEG_DELIMIT_1) >> 5) | (t_Info->uc_ID_SPar1_04 << 2));
   // NPAR2 blocks for every SPar1_03/04 bit set
   // TX attenuation: UpJ43 and DnJ433
   //                 UpB43c, DnB43c, UpV43 and DnV43
   puc_BytePtr = &(t_Info->uc_UpJ43Att);
   for(i=0; i < RPL_BUF_SIZE_1; i++)
   {
      if(us_temp & 1)
      {
         uca_M[idx++] = (*puc_BytePtr++ | uc_Delimit);
      }
      us_temp >>= 1;
   }


   //==== Standard information ====
   uca_M[idx++] = t_Info->uc_SI_NPar1 | DELIMIT_1;

   // Octet uc_SI_SPar1 contains information about G.992.1 and G.992.2 modes
   // Octet uc_SI_SPar1_02 contains information about G.991.2 and VDSL1 modes
   // Octet uc_SI_SPar1_03 contains information about G.992.3 and G.992.4 modes
   // Octet uc_SI_SPar1_04 contains information about G.992.5 modes
   // Octet uc_SI_SPar1_05 contains information about G.992.5M / G.993.2 modes
   // Note: Since we are in the VDSL binary all octets must be transmitted to
   //       indicate VDSL2!
   uca_M[idx++] = t_Info->uc_SI_SPar1;                   // G.992.1,G.992.2
   uca_M[idx++] = t_Info->uc_SI_SPar1_02;                // G.991.2 , VDSL1
   uca_M[idx++] = t_Info->uc_SI_SPar1_03;                // G.992.3 , G.992.4
   uca_M[idx++] = t_Info->uc_SI_SPar1_04;                // G.992.5 , IEEE 802.3
   uca_M[idx++] = (t_Info->uc_SI_SPar1_05 | DELIMIT_1);  // G.992.5M - Annex M,G.993.2

   //Form the ADSL standard information
   idx = FormStandard_ADSL(&t_Info->t_ADSL_Info, t_Info, uca_M, idx);

   //Form the VDSL2 standard information
   if ( t_Info->uc_SI_SPar1_05 & G993_2)
   {
      idx = FormStandard_VDSL2(&t_Info->t_VDSL2_Info, uca_M, idx);

   }

   // Form the G.Fast standard information
   if ( t_Info->uc_SI_SPar1_05 & G9701)
   {
      idx = FormStandard_G_Fast(&t_Info->t_G_Fast_Info, t_Info, uca_M, idx);
   }
   return (idx);
}   // FormStandard


/*
 *------------------------------------------------------------------------
 *
 *  Name : FormNonStandard
 *
 *  Abstract :
 *
 *  Enclose non-standard information field in an arrary.
 *
 *  Parameters:
 *      const InfoField *t_Info : information to be packed
 *      uint8 uca_M[]           : output arrary containing the information
 *      idx                     : index of the starting position of the message array
 *
 *  Returns:
 *      idx                     : index of the ending position of the message array
 *
 *  Global Variables Used:
 *      gt_hsc.s_TxNSBCnt       : Number of non-standard blocks transmitted so far
 *
 *  Notes : each non-standard block needs be less than or equal to 64 octets
 *          except for the first block that needs to be less than or equal to 63 octets.
 *
 *------------------------------------------------------------------------
 */

int16 FormNonStandard(InfoField_t *t_Info, uint8 uca_M[], int16 idx)
{
   int16 i;
   const NonStandardBlock_t *pt_NSB;

   if ( (t_Info->uc_ID_NPar1 & NON_STANDARD) != 0 )
   {
      //---- attach number of total blocks at the beginning of the first block ----
      if ( gt_hsc.s_TxNSBCnt == 0)
      {
         uca_M[idx++] = t_Info->uc_NumBlock;
      }

      //---- pack all the blocks assuming its within max message limit ----
      while (gt_hsc.s_TxNSBCnt < t_Info->uc_NumBlock)
      {
         pt_NSB = &( t_Info->ta_NS_Info[ gt_hsc.s_TxNSBCnt ] );
         uca_M[idx++] = pt_NSB->uc_NSLen;

         //---- 2 country code octets, with the most significant octet first ----
         uca_M[idx++] = (uint8) ( t_Info->us_Country >> 8 );
         uca_M[idx++] = (uint8) ( t_Info->us_Country );

         //---- 4 vendor ID octets, with the most significant octet first ----
         Unpack32BitTo8BitArray(&t_Info->ul_Provider, &uca_M[idx]);
         idx += 4;

         //---- Vendor specific information ----
         for ( i = 0; i < (pt_NSB->uc_NSLen - 6); i++ )
         {
            uca_M[idx++] = pt_NSB->uca_NSVendorSpec[i];
         }

         gt_hsc.s_TxNSBCnt++;

      }   // while
   }   // if

   return ( idx );

}   // FormNonStandard

int16 FormStandard_ADSL(ADSL_Type_t *pt_Info_ADSL, InfoField_t *t_Info, uint8 uca_M[], int16 idx)
{
   // Note: The used NPar2 must be according to InitializeStandardInfoField_ADSL() !!!
   //XDSLRTFW-456 : CORRECT_ADSL_OCTETS_IN_VDSL_GHS_MSG (START)

   //G992.1A.NPar(2)
   if(t_Info->uc_SI_SPar1 & G992_1_A)
   {
      uca_M[idx++] = pt_Info_ADSL->uc_SI_NPar2_01 | DELIMIT_1| DELIMIT_2;
   }

   //G992.1B.NPar(2)
   if(t_Info->uc_SI_SPar1 & G992_1_B)
   {
      uca_M[idx++] = pt_Info_ADSL->uc_SI_NPar2_02 | DELIMIT_1| DELIMIT_2;
   }

   //XDSLRTFW-2137 (Start)
   /********************************************************************
      G992_2 CLR
   *********************************************************************/
   if(t_Info->uc_SI_SPar1 & G992_2_AB)
   {
      uca_M[idx++] = pt_Info_ADSL->uc_SI_NPar2_06 | DELIMIT_1| DELIMIT_2;
   }
   //XDSLRTFW-2137 (End)

   /********************************************************************
      G992_3 / G992_5 CLR
   *********************************************************************/
   if(t_Info->uc_SI_SPar1_03 & G992_3_A)
   {
      uca_M[idx++] = pt_Info_ADSL->uc_SI_NPar2_03 | DELIMIT_1| DELIMIT_2;
   }

   if(t_Info->uc_SI_SPar1_03 & G992_3_B)
   {
      uca_M[idx++] = pt_Info_ADSL->uc_SI_NPar2_04 | DELIMIT_1| DELIMIT_2;
   }

   if(t_Info->uc_SI_SPar1_03 & G992_3_I)
   {
      uca_M[idx++] = pt_Info_ADSL->uc_SI_NPar2_03 | DELIMIT_1| DELIMIT_2;
   }

   if(t_Info->uc_SI_SPar1_03 & G992_3_J)
   {
      uca_M[idx++] = pt_Info_ADSL->uc_SI_NPar2_05 | DELIMIT_1| DELIMIT_2;
   }

   if(t_Info->uc_SI_SPar1_04 & G992_5_A)
   {
      uca_M[idx++] = pt_Info_ADSL->uc_SI_NPar2_03 | DELIMIT_1| DELIMIT_2;
   }

   if(t_Info->uc_SI_SPar1_04 & G992_5_B)
   {
      uca_M[idx++] = pt_Info_ADSL->uc_SI_NPar2_04 | DELIMIT_1| DELIMIT_2;
   }

   if(t_Info->uc_SI_SPar1_04 & G992_5_I)
   {
      uca_M[idx++] = pt_Info_ADSL->uc_SI_NPar2_03 | DELIMIT_1| DELIMIT_2;
   }

   if(t_Info->uc_SI_SPar1_04 & G992_3_M)
   {
      uca_M[idx++] = pt_Info_ADSL->uc_SI_NPar2_05 | DELIMIT_1| DELIMIT_2;
   }

   if(t_Info->uc_SI_SPar1_04 & G992_5_J)
   {
      uca_M[idx++] = pt_Info_ADSL->uc_SI_NPar2_05 | DELIMIT_1| DELIMIT_2;
   }

   if(t_Info->uc_SI_SPar1_05 & G992_5_M)
   {
      uca_M[idx++] = pt_Info_ADSL->uc_SI_NPar2_05 | DELIMIT_1| DELIMIT_2;
   }

   return (idx);
}


int16 FormStandard_VDSL2(VDSL2_Type_t *pt_Info, uint8 uca_M[], int16 idx)
{
   int16   i;

   // XDSLRTFW-487_VR9_VRX318_VDSL2_All_AELEM_Support (START)
   if(!(pt_Info->uc_SI_NPar2_02 == 0))
   {
      // uc_SI_NPar2_02 is non-zero, we need to transmit it too:
      uca_M[idx++] = pt_Info->uc_SI_NPar2;
      uca_M[idx++] = pt_Info->uc_SI_NPar2_02 | DELIMIT_2;
   }
   else
      // XDSLRTFW-487_VR9_VRX318_VDSL2_All_AELEM_Support (END)
   {
      uca_M[idx++] = pt_Info->uc_SI_NPar2 | DELIMIT_2;
   }

   uca_M[idx++] = pt_Info->uc_SI_SPar2_01;
   uca_M[idx++] = pt_Info->uc_SI_SPar2_02 | DELIMIT_2;

   //---- SI_NPar3 ----
   if ( (pt_Info->uc_SI_SPar2_01 & V2_PROFILES) != 0 )
   {
      uca_M[idx++] = pt_Info->uca_Profiles[0] & NEG_DELIMIT_2;
      uca_M[idx++] = (pt_Info->uca_Profiles[1] & NEG_DELIMIT_2) | DELIMIT_2;
   }

   //RFI bands
   if ( (pt_Info->uc_SI_SPar2_01 & V2_RFI_BANDS) != 0 )
   {
      PackBandInfo2MsgBuf(uca_M, &idx, pt_Info->uca_RFIBands, (uint8)gs_NumOfRFIBands);
   }

   //DFT size
   if ( (pt_Info->uc_SI_SPar2_01 & V2_DFT_SIZE) != 0 )
   {
      uca_M[idx++] = (pt_Info->uc_fft_size | DELIMIT_2);
   }

   //CE Length
   if ( (pt_Info->uc_SI_SPar2_01 & V2_CE_LENGTH) != 0 )
   {
      uca_M[idx++] = pt_Info->uca_CELength[0];
      uca_M[idx++] = pt_Info->uca_CELength[1];
      uca_M[idx++] = (pt_Info->uca_CELength[2] | DELIMIT_2);
   }

   // Annex A US0 PSD
   if ( (pt_Info->uc_SI_SPar2_02 & V2_ANNEX_A_US0_MASKS) != 0 )
   {
      for (i = 0; i < 4; i++)
      {
         uca_M[idx++] = pt_Info->uca_US0_A_bands[i];
      }
      uca_M[idx++] = (pt_Info->uca_US0_A_bands[i] | DELIMIT_2);
   }

   // Annex B US0 PSD
   if ( (pt_Info->uc_SI_SPar2_02 & V2_ANNEX_B_US0_MASKS) != 0 )
   {
      for (i = 0; i < 1; i++)
      {
         uca_M[idx++] = pt_Info->uca_US0_B_bands[i];
      }
      uca_M[idx++] = (pt_Info->uca_US0_B_bands[i] | DELIMIT_2);
   }

   // Annex C US0 PSD
   if ( (pt_Info->uc_SI_SPar2_02 & V2_ANNEX_C_US0_MASKS) != 0 )
   {
      for (i = 0; i < 2; i++)
      {
         uca_M[idx++] = pt_Info->uca_US0_C_bands[i];
      }
      uca_M[idx++] = (pt_Info->uca_US0_C_bands[i] | DELIMIT_2);
   }
//#ifdef MTK_VECTORING_SUPPORT
   // G.993.2 Vectoring
   if ( (pt_Info->uc_SI_SPar2_02 & V2_G_VECTOR_MASKS) != 0 )
   {
      uca_M[idx++] = (pt_Info->uc_Gvector | DELIMIT_2);
   }
//#endif

   // XDSLRTFW-1693 (Start_End)
   if ((pt_Info->uc_SI_SPar2_02 & V2_9984_EXTENSIONS_MASKS) != 0)
   {
      uca_M[idx++] = (pt_Info->uc_G9984_Extensions | DELIMIT_2);
   }

   uca_M[idx-1] |= DELIMIT_1;

   return(idx);

}

int16 FormStandard_G_Fast(G_Fast_Type_t *pt_Info_G_Fast, InfoField_t *t_Info, uint8 uca_M[], int16 idx)
{
   if ( t_Info->uc_SI_SPar1_05 & G9701)
   {
      uca_M[idx++] = pt_Info_G_Fast->uc_SI_NPar2;

      // SPar2 octet 1
      uca_M[idx++] = pt_Info_G_Fast->uc_SI_SPar2_01;

      // SPar2 octet 2
      uca_M[idx++] = pt_Info_G_Fast->uc_SI_SPar2_02;

      // SPar2 octet 3
      uca_M[idx++] = pt_Info_G_Fast->uc_SI_SPar2_03 | DELIMIT_2;

      // NPar3 octets
      // Profile
      if (pt_Info_G_Fast->uc_SI_SPar2_01 & G_Fast_Profiles)
      {
         uca_M[idx++] = pt_Info_G_Fast->uc_Profiles | DELIMIT_2;
      }

      // Duration of Channel discovery 1-1
      if ( pt_Info_G_Fast->uc_SI_SPar2_01 & G_Fast_Duration_Channel_Discovery)
      {
         uca_M[idx++] = pt_Info_G_Fast->uc_duration_channel_discovery | DELIMIT_2;
      }

      // Number of symbols in TDD  frame
      if (pt_Info_G_Fast->uc_SI_SPar2_02 & G_Fast_Symbol_Period_TDD_Frame)
      {
         uca_M[idx++] = pt_Info_G_Fast->uc_symbol_period_TDD_frame | DELIMIT_2;
      }
      uca_M[idx-1] |= DELIMIT_1;
   }
   return (idx);
}

/*
 *------------------------------------------------------------------------
 *
 *  Name : DecodeNonStandard
 *
 *  Abstract : Check FCS and decode received non-standard information
 *
 *  Input Arguments:
 *      uint8 uca_M[]       - array of received message octets
 *      InfoField_t *t_I    - pointer to a InfoField structure that holds
 *                            the decoded information
 *      idx                 - index of the starting position of the message array
 *      s_MsgLen            - total number of octets in uca_M
 *
 *  Returns:
 *      idx                 - index of the ending position of the message array
 *                            It is set to zero if the current message is not complete.
 *
 *  Global Variables Used:
 *      gt_hsc.s_TxNSBCnt : Number of non-standard blocks transmitted so far
 *
 *  Notes:
 *
 *------------------------------------------------------------------------
 */

int16 DecodeNonStandard(uint8 uca_M[], int16 idx, int16 s_MsgLen, InfoField_t *t_Info)
{
   int16               i;
   int16              s_NSB_Cnt;
   NonStandardBlock_t *pt_NSB;


   s_NSB_Cnt = 0;  // initialize
   t_Info->uc_NumBlock = uca_M[idx++];

   while (s_NSB_Cnt < t_Info->uc_NumBlock)
   {
      pt_NSB = &(t_Info->ta_NS_Info[s_NSB_Cnt]);
      // Security code so that no memory corruption can occur in case
      // the CO is transmitting more NSF blocks than CPE can handle!
      if (s_NSB_Cnt >= MAX_NUM_NS_BLOCK)
      {
         pt_NSB = &( t_Info->ta_NS_Info[ (MAX_NUM_NS_BLOCK-1) ] );
      }

      pt_NSB->uc_NSLen = uca_M[idx++];

      //---- 2 country code octets, with the most significant octet first ----
      t_Info->us_Country_NS = uca_M[idx++];
      t_Info->us_Country_NS <<= 8;
      t_Info->us_Country_NS |= uca_M[idx++];

      //---- 4 vendor ID octets, with the least significant octet first ----

      t_Info->ul_Provider_NS = uca_M[idx++];
      for (i = 0; i < 3; i++)
      {
         t_Info->ul_Provider_NS <<= 8;
         t_Info->ul_Provider_NS |= uca_M[idx++];
      }
      //---- Vendor specific information ----
      {
         uint8 uc_LoopCnt;

         uc_LoopCnt = (pt_NSB->uc_NSLen - NS_HEADER_LENGTH);
         // Security code so that no memory corruption can occur in case
         // the CO is transmitting more VENDOR_INFO bytes than CPE can handle!
         if (uc_LoopCnt > MAX_NS_VENDOR_INFO_LEN)
         {
            uc_LoopCnt = MAX_NS_VENDOR_INFO_LEN;
         }
         for (i = 0; i < uc_LoopCnt; i++)
         {
            pt_NSB->uca_NSVendorSpec[i] = uca_M[idx++];
         }
      }

      s_NSB_Cnt++;
      //---- check for incomplete message ----
      if (idx > s_MsgLen)
      {
         idx = 0;
         break;
      }

   }   // while

   return (idx);

}   // DecodeNonStandard
/*
 *------------------------------------------------------------------------
 *
 *  Name : DecodeMessage
 *
 *  Abstract :
 *
 *  DecodeMessage() - Check FCS and decode received message
 *
 *  Input Arguments:
 *      uint8 uca_Msg[]     - array of received message octets
 *      InfoField_t *t_I    - pointer to a InfoField structure that holds
 *                            the decoded information
 *
 *  Returns:
 *      returns the following error code if there is an error:
 *      DECODE_SUCCESS          - decoding successfully done
 *      FCS_ERROR               - FCS error
 *      INVALID_MESSAGE         - invalid message
 *      INCOMPLETE_STANDARD     - only a partial standard information is received
 *      INCOMPLETE_NON_STANDARD - only a partial non-standard information is received
 *
 *
 *  Global Variables Used:
 *      gt_hsc.s_RxNSBCnt     - used in DecodeNonStandard()
 *
 *  Notes:      DecodeMessage() is called from HSMsgHandler(), which is
 *            executed in the background during state C_R_HS_MSG_RX.
 *
 *------------------------------------------------------------------------
 */

void BgDecodeMessage(void)
{
   // Clear the RX message structure if ignore MS debug bit not set
   if (!(gul_dbgGHSControl & IGNORE_FAREND_MS))
   {
      memset(gpt_RxInfo, 0, sizeof(InfoField_t));
   }

   //Decode message
   DecodeMessage(gpuca_RxMsgBuf, gpt_RxInfo);

   gft_CL_CLR_Comparison = 0;

   // Save VDSL2 info from CL
   if ((gt_hsc.s_DecodeResult == DECODE_SUCCESS)
         && (gpt_RxInfo->uc_Type == M_CL) )
   {
      Common_CL_InfoSave();

      if(gpt_RxInfo->uc_SI_SPar1_05 & G993_2)
      {
         VDSL2_CL_InfoSave();
      }

      gft_CL_CLR_Comparison = Compare_CL_CLR();
      /*  placehold for fw initiated retrain */
      if(gul_fe_G994VendorID == 0xDEADBEAF)
      {
         gft_FW_Initiate_Retrain = 1;
      }

   }

   gs_RxBkgdProcessFlag = BG_GHS_TASK_DONE;
}

void DecodeMessage(uint8 uca_Msg[], InfoField_t *t_Info)
{

   int16   idx, i;
   uint16  us_FCS;
   int16   s_MsgLen;
   uint8   *uca_M;

   //==========================================
   //==== Octet transparency and FCS check ====
   //==== then append to gpuca_DecodeBuf[]. ====
   //==========================================
   uca_M = &gpuca_DecodeBuf[ gs_DecodeBufCnt ]; // setting where to append
   idx = i = 0;        // initialize indices
   us_FCS = 0xFFFF;    // initialize to all binary 1's

   while ( uca_Msg[idx] != FLAG )
   {

      //---- octet transparency ----
      if ( uca_Msg[idx] == 0x7D )
      {
         idx++;
         uca_M[i] = uca_Msg[idx++] ^ 0x20;
      }
      else
      {
         uca_M[i] = uca_Msg[idx++];
      }

      //---- FCS check ----
      us_FCS = calcCrc16(us_FCS, uca_M[i]);
      i++;

      //---- to ensure maximum message length (one segment) and decode buffer length ----
      if ( (i >= MAX_MESSAGE_LEN + FCS_LEN) || (gs_DecodeBufCnt + i >= DECODE_BUF_LEN) )
      {
         break;
      }

   }   // while

   // for debug purpose, log GHS messages recieved
   DebugGHSMessages(uca_M[0], (int16)i, uca_M, 0x8888);
#ifdef DEBUG_TRACES
   // XDSLRTFW-598 FEATURE_ALL_ALL_ALL_Debug_Buffer(START_END)
   LogMessages(4,(uint16)uca_M[0], (uint16)i, uca_M, 0x88888888);
#endif //#ifdef DEBUG_TRACES
      DSH_SendEvent(DSH_EVT_GHS_RX,(uint8)i,&uca_M[0]);


   //---- check for invalid message ----
   if ( i+gs_DecodeBufCnt < MIN_MESSAGE_LEN )
   {
      gt_hsc.s_DecodeResult = INVALID_MESSAGE;
      DebugGHSMessages(uca_M[0], 0, uca_M, 0x8889);   //Log Invalid message length error indicator
#ifdef DEBUG_TRACES
      // XDSLRTFW-598 FEATURE_ALL_ALL_ALL_Debug_Buffer(START_END)
      LogMessages(1,0x88898889);
#endif //#ifdef DEBUG_TRACES
      return;
   }

   //---- Frame Check Sequence (FCS) ----
   else if (us_FCS != MAGIC_FCS_CHECK)
   {
      gt_hsc.s_DecodeResult = FCS_ERROR;
      DebugGHSMessages(uca_M[0], 0, uca_M, 0x888A);   //Log FCS Error indicator
#ifdef DEBUG_TRACES
      // XDSLRTFW-598 FEATURE_ALL_ALL_ALL_Debug_Buffer(START_END)
      LogMessages(1,0x888A888A);
#endif //#ifdef DEBUG_TRACES
      return;
   }

   gft_RTXCtrlFlag |= GHS_NEW_RX_MSG_BIT;

   //***********************************************
   //**** proceed if no error
   //***********************************************

   gs_RxMsgBufCnt   = i - FCS_LEN;         // number of data octets in current message segment NOT including FCS
   gs_DecodeBufCnt += gs_RxMsgBufCnt;      // number of data octets in a total message

   s_MsgLen         = gs_DecodeBufCnt;
   uca_M            = &gpuca_DecodeBuf[0];  // decoding starts from the beginning

   //==========================================================
   //==== for a standard message frame (the first segment) ====
   //==========================================================
   idx = 0;


   //---- Message type and Revision number fields ----
   t_Info->uc_Type = uca_M[idx++];

   t_Info->uc_Rev  = uca_M[idx++];

   //Count the number of consecutive REQ-RTX messages received
   if(t_Info->uc_Type == M_REQ_RTX)
   {
      guc_RxHSReqRtx_cnt++;
   }
   else
   {
      guc_RxHSReqRtx_cnt = 0;
   }

   //---- Vendor ID (required only for CL and CLR messages) ----
   switch (t_Info->uc_Type)
   {
   case M_CL  :
   case M_CLR :
      //---- 2 country code octets, with the most significant octet first ----
      t_Info->us_Country = uca_M[idx++];
      t_Info->us_Country <<= 8;
      t_Info->us_Country |= uca_M[idx++];

      //---- 4 vendor ID octets, with the most significant octet first ----
      t_Info->ul_Provider = uca_M[idx++];
      for ( i = 0; i < 3; i++ )
      {

         t_Info->ul_Provider <<= 8;
         t_Info->ul_Provider |= uca_M[idx++];
      }

      //---- 2 octets of vendor specific information ----
      t_Info->us_VendorInfo = uca_M[idx++];
      t_Info->us_VendorInfo <<= 8;
      t_Info->us_VendorInfo |= uca_M[idx++];

      //XDSLRTFW-1983
      //Store for Use in ADSL if ATM-PTm Mis match is found
      gus_VendorSpecific_Info = t_Info->us_VendorInfo;
      break;

   case M_REQ_RTX:
      /* ---- 2 octets of retransmission information block ---- */
      t_Info->uc_last_received_msg_type = uca_M[idx++];
      t_Info->uc_request_msg_segment_index = uca_M[idx++];
      break;

   default:
      break;

   }   //  switch (t_Info->uc_Type)

   //---- Service and Channel parameters, Standard and Non-Standard Information ----
   //---- only for MS, MP, CL, CLR messges                                          ----
   switch (t_Info->uc_Type)
   {
   case M_MS :
      // Skip MS interpretation if ignore MS debug bit set
      if (gul_dbgGHSControl & IGNORE_FAREND_MS)
      {
         break;
      }

   case M_CL :
   case M_CLR:
   case M_MP :
      idx = DecodeStandard( uca_M, idx, s_MsgLen, t_Info ); // parameters and standard

      if ( idx == 0 )
      {

         // If received CLR/MS is incomplete, set gt_hsc.s_DecodeResult to INCOMPLETE_STANDARD
         // in order to start transmitting ACK(2)s to the CPE.
         gt_hsc.s_DecodeResult = INCOMPLETE_STANDARD;
         DebugGHSMessages(uca_M[0], 0, uca_M, 0x888B);   //Log Incomplete Standard Error indicator
#ifdef DEBUG_TRACES
         // XDSLRTFW-598 FEATURE_ALL_ALL_ALL_Debug_Buffer(START_END)
         LogMessages(1,0x888B888B);
#endif //#ifdef DEBUG_TRACES
         guc_RxHSMsg_SegmentCnt++;  // reset Rx msg segment counter
         return;               // note that gs_DecodeBufCnt is not reset here
         // in order to properly append the next segment of the message.
      }

      if ( (t_Info->uc_ID_NPar1 & NON_STANDARD) != 0 )
      {
         idx = DecodeNonStandard( uca_M, idx, s_MsgLen, t_Info ); // non standard
         if ( idx == 0 )
         {
            gt_hsc.s_DecodeResult = INCOMPLETE_NON_STANDARD;
            DebugGHSMessages(uca_M[0], 0, uca_M, 0x888C);   //Log Incomplete Non-standard Error indicator
#ifdef DEBUG_TRACES
            // XDSLRTFW-598 FEATURE_ALL_ALL_ALL_Debug_Buffer(START_END)
            LogMessages(1,0x888C888C);
#endif //#ifdef DEBUG_TRACES
            guc_RxHSMsg_SegmentCnt++;  // reset Rx msg segment counter
            return;               // note that gs_DecodeBufCnt is not reset here
            // in order to properly append the next segment of the message.
         }
      }
      else
      {
         t_Info->uc_NumBlock = 0;
      }

      break;

   default:
      break;

   } // switch (t_Info->uc_Type)

   // reset for next message
   gt_hsc.s_DecodeResult = DECODE_SUCCESS;
   gs_DecodeBufCnt = 0;
   guc_RxHSMsg_SegmentCnt = 0;  // reset Rx msg segment counter

   return;
}   // DecodeMessage


/*
 *------------------------------------------------------------------------
 *
 *  Name : DecodeStandard
 *
 *  Abstract : decode standard information
 *
 *  Input Arguments:
 *      uint8 uca_M[]       - array of received message octets
 *      InfoField_t *t_I    - pointer to a InfoField structure that holds
 *                            the decoded information
 *      idx                 - index of the starting position of the message array
 *      s_MsgLen            - total number of octets in uca_M
 *
 *  Returns:
 *      idx                 - index of the ending position of the message array
 *
 *  Global Variables Used:
 *
 *  Notes:
 *
 *------------------------------------------------------------------------
 */
int16 DecodeStandard(uint8 uca_M[], int16 idx, int16 s_MsgLen, InfoField_t *t_Info)
{

   int16 j;
   uint16 us_temp;
   //uint8 uc_onebit;
   int16 s_SPar2ActiveBlocks;
   int16 s_SPar2DecodedBlocks;
   uint8 *puc_BytePtr;

#ifdef VDSL_BONDING
   int16 i;
#endif

   //***************************************************************
   //**** Service and Channel parameters (for CL, CLR, MS only) ****
   //***************************************************************

   s_SPar2ActiveBlocks = 0;
   s_SPar2DecodedBlocks = 0;

   t_Info->uc_ID_NPar1 = uca_M[idx++] & NEG_DELIMIT_1;
   idx = SkipExtraOctetAndCheckSegmentation(uca_M, idx, s_MsgLen, DELIMIT_1, s_SPar2ActiveBlocks, s_SPar2DecodedBlocks);
   if (idx == 0)
   {
      return(0);
   }

   t_Info->uc_ID_SPar1 = uca_M[idx++] & NEG_DELIMIT_1;

   if ((uca_M[idx-1] & DELIMIT_1) == 0)
   {
      t_Info->uc_ID_SPar1_02 = uca_M[idx++] & NEG_DELIMIT_1;

      if ((uca_M[idx-1] & DELIMIT_1) == 0)
      {
         t_Info->uc_ID_SPar1_03 = uca_M[idx++] & NEG_DELIMIT_1;

         if ((uca_M[idx-1] & DELIMIT_1) == 0)
         {
            t_Info->uc_ID_SPar1_04 = uca_M[idx++] & NEG_DELIMIT_1;
         }
      }
   }

   // If there are extra ID_Spar1 octets, ignore them. At the same time, use gus_ExtraBits to save
   // the number of bits set to 1 in the ignored octets.
   gus_ExtraBits =0;
   idx = SkipCountExtraOctetAndCheckSegmentation(uca_M, idx, s_MsgLen, DELIMIT_1, s_SPar2ActiveBlocks, s_SPar2DecodedBlocks);
   if (idx == 0)
   {
      return(0);
   }

   //----- ID_SPar1 - Par2 blocks (ID_NPar2) -------
   // net data rate upstream
   if ( (t_Info->uc_ID_SPar1 & NDR_UP) != 0 )
   {
      SetArray2Struct(uca_M, &idx, &t_Info->uc_UpMaxNDR, 3);

      idx = SkipExtraOctetAndCheckSegmentation(uca_M, idx, s_MsgLen, DELIMIT_1, s_SPar2ActiveBlocks, s_SPar2DecodedBlocks);
      if (idx == 0)
      {
         return(0);
      }
   }

   // net data rate downstream
   if ( (t_Info->uc_ID_SPar1 & NDR_DN) != 0 )
   {
      SetArray2Struct(uca_M, &idx, &t_Info->uc_DnMaxNDR, 3);

      idx = SkipExtraOctetAndCheckSegmentation(uca_M, idx, s_MsgLen, DELIMIT_1, s_SPar2ActiveBlocks, s_SPar2DecodedBlocks);
      if (idx == 0)
      {
         return(0);
      }
   }

   // data flow characteristics upstream
   if ( (t_Info->uc_ID_SPar1 & DFC_UP) != 0 )
   {
      SetArray2Struct(uca_M, &idx, &t_Info->uc_UpMaxLat, 2);

      idx = SkipExtraOctetAndCheckSegmentation(uca_M, idx, s_MsgLen, DELIMIT_1, s_SPar2ActiveBlocks, s_SPar2DecodedBlocks);
      if (idx == 0)
      {
         return(0);
      }
   }

   // data flow characteristics downstream
   if ( (t_Info->uc_ID_SPar1 & DFC_DN) != 0 )
   {
      SetArray2Struct(uca_M, &idx, &t_Info->uc_DnMaxLat, 2);

      idx = SkipExtraOctetAndCheckSegmentation(uca_M, idx, s_MsgLen, DELIMIT_1, s_SPar2ActiveBlocks, s_SPar2DecodedBlocks);
      if (idx == 0)
      {
         return(0);
      }
   }

   // xTU-R splitter information
   if ( (t_Info->uc_ID_SPar1 & R_SPLITTER) != 0 )
   {
      t_Info->uc_ID_NPar2_O11 = uca_M[idx++] & NEG_DELIMIT_2;

      idx = SkipExtraOctetAndCheckSegmentation(uca_M, idx, s_MsgLen, DELIMIT_1, s_SPar2ActiveBlocks, s_SPar2DecodedBlocks);
      if (idx == 0)
      {
         return(0);
      }
   }

   // xTU-C splitter information
   if ( (t_Info->uc_ID_SPar1 & C_SPLITTER) != 0 )
   {
      t_Info->uc_ID_NPar2_O12 = uca_M[idx++] & NEG_DELIMIT_2;

      idx = SkipExtraOctetAndCheckSegmentation(uca_M, idx, s_MsgLen, DELIMIT_1, s_SPar2ActiveBlocks, s_SPar2DecodedBlocks);
      if (idx == 0)
      {
         return(0);
      }
   }

   // Transceiver ID information (30bit)
   if((t_Info->uc_ID_SPar1 & TRANSCEIVER_ID) != 0)
   {
      // NPar2
//      t_Info->uc_ID_NPar2_TransId = uca_M[idx++] & NEG_DELIMIT_2;
      idx++;
      // Check end of NPAR2 block, i.e. DELIMIT_2
      idx = SkipExtraOctetAndCheckSegmentation(uca_M, idx, s_MsgLen, DELIMIT_2, s_SPar2ActiveBlocks, s_SPar2DecodedBlocks);
      if(idx == 0)
      {
         return(0);
      }

      // SPar2 / NPar3
      if ((uca_M[idx-1] & DELIMIT_1) == 0)
      {
         // SPar2
         t_Info->uc_ID_SPar2_TransId = uca_M[idx++] & NEG_DELIMIT_2;
         // Check end of SPAR2 block, i.e. DELIMIT_2
         idx = SkipExtraOctetAndCheckSegmentation(uca_M, idx, s_MsgLen, DELIMIT_2, s_SPar2ActiveBlocks, s_SPar2DecodedBlocks);
         if(idx == 0)
         {
            return(0);
         }

         // NPar3 - Network side transceiver ID
         if(t_Info->uc_ID_SPar2_TransId & CO_TRANSCEIVER_ID)
         {
            t_Info->ul_CoTransId = (uca_M[idx++] & NEG_DELIMIT_2);
            for(i = 1; i < NUM_TRANSCEIVER_ID; i++)
            {
               t_Info->ul_CoTransId <<= PAR2_NUM_BIT;
               t_Info->ul_CoTransId |= (uca_M[idx++] & NEG_DELIMIT_2);
            }
            // Check end of NPAR3 block, i.e. DELIMIT_2
            idx = SkipExtraOctetAndCheckSegmentation(uca_M, idx, s_MsgLen, DELIMIT_2, s_SPar2ActiveBlocks, s_SPar2DecodedBlocks);
            if(idx == 0)
            {
               return(0);
            }
         }
         // NPar3 - Remote side transceiver ID
         if(t_Info->uc_ID_SPar2_TransId & CPE_TRANSCEIVER_ID)
         {
            t_Info->ul_CpeTransId = (uca_M[idx++] & NEG_DELIMIT_2);
            for(i = 1; i < NUM_TRANSCEIVER_ID; i++)
            {
               t_Info->ul_CpeTransId <<= PAR2_NUM_BIT;
               t_Info->ul_CpeTransId |= (uca_M[idx++] & NEG_DELIMIT_2);
            }
            // Check end of NPAR3 block, i.e. DELIMIT_2
            idx = SkipExtraOctetAndCheckSegmentation(uca_M, idx, s_MsgLen, DELIMIT_2, s_SPar2ActiveBlocks, s_SPar2DecodedBlocks);
            if(idx == 0)
            {
               return(0);
            }
         }
      }

      //------------------------------
      //--- Skip unsupported NPar(3) blocks of unsupported SPar(2) bits.
      //------------------------------
      // Upon finishing processing the known bits, we just skip until we see DELIMIT_1, i.e.
      // end of Transceiver ID Par2 block consiting of NPar2/SPar2 and NPar3 blocks.
      idx = SkipExtraOctetAndCheckSegmentation(uca_M, idx, s_MsgLen, DELIMIT_1, s_SPar2ActiveBlocks, s_SPar2DecodedBlocks);
      if(idx == 0)
      {
         return(0);
      }
   }

   //----- ID_SPar1_02 - Par2 blocks (ID_NPar2) -------
   //----- Note: ID_SPar1_02 octet has 6 bits(relative power level for US/DS A43, B43 and C43) ------
   //-----       of interest and 1 reserved bit..                                              ------
   puc_BytePtr = &(t_Info->uc_UpA43Att);
   us_temp = (t_Info->uc_ID_SPar1_02 & 0x3F);

   for(j=0; j<6; j++)
   {
      if(us_temp&1)
      {
         *puc_BytePtr++ =  uca_M[idx++] & NEG_DELIMIT_2;

         idx = SkipExtraOctetAndCheckSegmentation(uca_M, idx, s_MsgLen, DELIMIT_1, s_SPar2ActiveBlocks, s_SPar2DecodedBlocks);
         if (idx == 0)
         {
            return(0);
         }
      }
      else
      {
         puc_BytePtr++;
      }

      us_temp >>= 1;
   }

   // Need to skip octets corresponding to the reserved bit of t_Info->uc_ID_SPar1_02
   // in case it is set in the future.
   if((t_Info->uc_ID_SPar1_02 & 0x40) != 0)
   {
      idx++;

      idx = SkipExtraOctetAndCheckSegmentation(uca_M, idx, s_MsgLen, DELIMIT_1, s_SPar2ActiveBlocks, s_SPar2DecodedBlocks);
      if (idx == 0)
      {
         return(0);
      }
   }

   //----- ID_SPar1_03 - Par2 blocks (ID_NPar2) -------
   //----- Note: ID_SPar1_03 octet has 6 bits for relative power level(US/DS A4, A43c and J43) ------
   //-----       and 1 bit for bonding(bit5).                                                  ------
   puc_BytePtr = &(t_Info->uc_UpA4Att);
   us_temp = (t_Info->uc_ID_SPar1_03 & 0xF);

   for(j=0; j<4; j++)
   {
      if(us_temp&1)
      {
         *puc_BytePtr++ =  uca_M[idx++] & NEG_DELIMIT_2;

         idx = SkipExtraOctetAndCheckSegmentation(uca_M, idx, s_MsgLen, DELIMIT_1, s_SPar2ActiveBlocks, s_SPar2DecodedBlocks);
         if (idx == 0)
         {
            return(0);
         }
      }
      else
      {
         puc_BytePtr++;
      }

      us_temp >>= 1;
   }


#ifdef VDSL_BONDING
   if ( (t_Info->uc_ID_SPar1_03 & BONDING) != 0)
   {
//      DebugTrail1(1,DEBUG_TRAIL_BONDING_TRAIL_ENABLE,0,(int16)0xFF00);

      // Bonding NPar2
      t_Info->uc_BondingNP2 = uca_M[idx++] & NEG_DELIMIT_2;

      idx = SkipExtraOctetAndCheckSegmentation(uca_M, idx, s_MsgLen, DELIMIT_2, s_SPar2ActiveBlocks, s_SPar2DecodedBlocks);
      if (idx == 0)
      {
         return(0);
      }

      // Bonding Spar2
      if ( (uca_M[idx-1] & DELIMIT_1) == 0 )
      {
         t_Info->uc_BondingSP2 = uca_M[idx++] & NEG_DELIMIT_2;

         idx = SkipExtraOctetAndCheckSegmentation(uca_M, idx, s_MsgLen, DELIMIT_2, s_SPar2ActiveBlocks, s_SPar2DecodedBlocks);
         if (idx == 0)
         {
            return(0);
         }

         // We care (for the time being) only for the last 3 LSB's of Bonding SPar2, so no need to count extra bits etc.
         // Upon finishing processing the 3 known bits, we just skip until we see DELIMIT_1 (end of Bonding whole Par2/3 block)

         // BONDING_PME_DISCOVERY NPar3
         if ( (t_Info->uc_BondingSP2 & BONDING_PME_DISCOVERY) != 0)
         {

//            DebugTrail1(1,DEBUG_TRAIL_BONDING_TRAIL_ENABLE,0,(int16)0xFF01);

            for (i=0; i<9; i++)
            {

//               DebugTrail1(1,DEBUG_TRAIL_BONDING_TRAIL_ENABLE,0,(int16)uca_M[idx]);

               t_Info->uca_BondingDisc[i] = uca_M[idx++] & NEG_DELIMIT_2;
               // If -for any reason- NPar2 ends unexpectedly, just exit this block
               if (uca_M[idx-1] & DELIMIT_2)
               {
                  break;
               }
            }
            idx = SkipExtraOctetAndCheckSegmentation(uca_M, idx, s_MsgLen, DELIMIT_2, s_SPar2ActiveBlocks, s_SPar2DecodedBlocks);
            if (idx == 0)
            {
               return(0);
            }
         }

         // BONDING_PME_AGGREGATION NPar3
         if ( (t_Info->uc_BondingSP2 & BONDING_PME_AGGREGATION) != 0)
         {

//            DebugTrail1(1,DEBUG_TRAIL_BONDING_TRAIL_ENABLE,0,(int16)0xFF02);

            for (i=0; i<6; i++)
            {

//               DebugTrail1(1,DEBUG_TRAIL_BONDING_TRAIL_ENABLE,0,(int16)uca_M[idx]);

               t_Info->uca_BondingAggr[i] = uca_M[idx++] & NEG_DELIMIT_2;
               // If -for any reason- NPar2 ends unexpectedly, just exit this block
               if (uca_M[idx-1] & DELIMIT_2)
               {
                  break;
               }
            }
            idx = SkipExtraOctetAndCheckSegmentation(uca_M, idx, s_MsgLen, DELIMIT_2, s_SPar2ActiveBlocks, s_SPar2DecodedBlocks);
            if (idx == 0)
            {
               return(0);
            }
         }

         // BONDING_PME_IDENTIFICATION NPar3
         // Note: This parameter is CPE only. Therefore the CPE should never receive any PME_ID
         //       parameter. Code is normally not needed, but with the actual implementation it
         //       is more robust.
         if ( (t_Info->uc_BondingSP2 & BONDING_PME_IDENTIFICATION) != 0)
         {

//            DebugTrail1(2,DEBUG_TRAIL_BONDING_TRAIL_ENABLE,0,(int16)0xFF03, (int16)uca_M[idx]);

            t_Info->uc_BondingId = uca_M[idx++] & NEG_DELIMIT_2;
         }

         idx = SkipExtraOctetAndCheckSegmentation(uca_M, idx, s_MsgLen, DELIMIT_1, s_SPar2ActiveBlocks, s_SPar2DecodedBlocks);
         if (idx == 0)
         {
            return(0);
         }

      }// end Bonding Spar2

   }// end Bonding

#else
   //Need to skip octets corresponding to the bonding bit (bit 5)of t_Info->uc_ID_SPar1_03
   us_temp = (t_Info->uc_ID_SPar1_03 & 0x10)>>4;

   if(us_temp&1)
   {
      idx++;

      idx = SkipExtraOctetAndCheckSegmentation(uca_M, idx, s_MsgLen, DELIMIT_1, s_SPar2ActiveBlocks, s_SPar2DecodedBlocks);
      if (idx == 0)
      {
         return(0);
      }
   }
#endif

   //Pack bits 6 and 7 of t_Info->uc_ID_SPar1_03 and bits1-4 of t_Info->uc_ID_SPar1_04 into one word
   puc_BytePtr = &(t_Info->uc_UpJ43Att);

   us_temp = (((t_Info->uc_ID_SPar1_03 & 0x60)>>5)|((t_Info->uc_ID_SPar1_04 & 0xF)<<2));

   for(j=0; j<6; j++)
   {
      if(us_temp&1)
      {
         *puc_BytePtr++ =  uca_M[idx++] & NEG_DELIMIT_2;

         idx = SkipExtraOctetAndCheckSegmentation(uca_M, idx, s_MsgLen, DELIMIT_1, s_SPar2ActiveBlocks, s_SPar2DecodedBlocks);
         if (idx == 0)
         {
            return(0);
         }
      }
      else
      {
         puc_BytePtr++;
      }

      us_temp >>= 1;
   }


   // Need to skip octets corresponding to the reserved bits of t_Info->uc_ID_SPar1_04
   // in case they are set in the future.
   us_temp = (t_Info->uc_ID_SPar1_04 & 0x70)>>4;

   for(j=0; j<3; j++)
   {
      if(us_temp&1)
      {
         idx++;

         idx = SkipExtraOctetAndCheckSegmentation(uca_M, idx, s_MsgLen, DELIMIT_1, s_SPar2ActiveBlocks, s_SPar2DecodedBlocks);
         if (idx == 0)
         {
            return(0);
         }
      }

      us_temp >>= 1;
   }

   //If we have fifth or more octets for ID_SPar1, we need to ignore the corresponding octets here.
   for(j=0; j<gus_ExtraBits; j++)
   {
      idx++;
      idx = SkipExtraOctetAndCheckSegmentation(uca_M, idx, s_MsgLen, DELIMIT_1, s_SPar2ActiveBlocks, s_SPar2DecodedBlocks);

      if (idx == 0)
      {
         return(0);
      }
   }

   //******************************
   //**** Standard information ****
   //******************************
   // NPar1 octets
   t_Info->uc_SI_NPar1 = uca_M[idx++] & NEG_DELIMIT_1;
   idx = SkipExtraOctetAndCheckSegmentation(uca_M, idx, s_MsgLen, DELIMIT_1, s_SPar2ActiveBlocks, s_SPar2DecodedBlocks);
   if (idx == 0)
   {
      return(0);
   }

   // SPar1 octets
   t_Info->uc_SI_SPar1 = uca_M[idx] & NEG_DELIMIT_1;
   if ((uca_M[idx++] & DELIMIT_1) == 0)
   {
      t_Info->uc_SI_SPar1_02 = uca_M[idx] & NEG_DELIMIT_1;
      if ((uca_M[idx++] & DELIMIT_1) == 0)
      {
         t_Info->uc_SI_SPar1_03 = uca_M[idx] & NEG_DELIMIT_1;
         if ((uca_M[idx++] & DELIMIT_1) == 0)
         {
            t_Info->uc_SI_SPar1_04 = uca_M[idx] & NEG_DELIMIT_1;
            if ((uca_M[idx++] & DELIMIT_1) == 0)
            {
               t_Info->uc_SI_SPar1_05 = uca_M[idx++] & NEG_DELIMIT_1;
            }
         }
      }
   }

   //If there are sixth or more octets for SI_SPar1 added in the future, we need to ignore them.
   // At the same time, save the number of bits set to 1 into  gus_ExtraBits.
   gus_ExtraBits = 0;
   idx = SkipCountExtraOctetAndCheckSegmentation(uca_M, idx, s_MsgLen, DELIMIT_1, s_SPar2ActiveBlocks, s_SPar2DecodedBlocks);
   if (idx == 0)
   {
      return(0);
   }

   //skipping octets
   {
      uint32 ul_temp;
      /*
      //need to skip the octets corresponding to
      SPar1 G992_1_A, G992_1_B, G992_1_C, G992_2_A, G992_2_C, G992_1_H, G992_1_I
      SPar2 G991_2_A, G991_2_B, T1_MCM_VDSL, T1_SCM_VDSL, ETSI_MCM_VDSL, ETSI_SCM_VDSL, Reserved for ITU-T
      SPar3 G992_3_A, G992_3_B, G992_3_I, G992_3_J, G992_4_A, G992_4_I, G992_4_C
      SPar4 G992_5_A, G992_5_B, G992_5_I, G992_3_M, G992_5_J, IEEE_802_3AH_TL, IEEE_802_3AH_TS
      */
      //XDSLRTFW-2249 (Start)
      ul_temp  = (t_Info->uc_SI_SPar1 & 0x7F);
      ul_temp |= ((t_Info->uc_SI_SPar1_02 & 0x7F)<<7);
      ul_temp |= ((t_Info->uc_SI_SPar1_03 & 0x7F)<<14);
      ul_temp |= ((t_Info->uc_SI_SPar1_04 & 0x7F)<<21);

      for(j=0; j<(4*7); j++)
      {
         if(ul_temp&1)
         {
            idx++;

            idx = SkipExtraOctetAndCheckSegmentation(uca_M, idx, s_MsgLen, DELIMIT_1, s_SPar2ActiveBlocks, s_SPar2DecodedBlocks);
            if (idx == 0)
            {
               return(0);
            }
         }

         ul_temp >>= 1;
      }
      //XDSLRTFW-2249 (End)
   }

   /*
   //need to skip the octets corresponding to
   G992_5_M
   G993_1_ANSI_T1_424
   G993_1_ANNEX_I
   G993_1_VAR_SILENCE
   T1_ENHANCED_SHDSL
   G992_5_C
   */
   us_temp = ((t_Info->uc_SI_SPar1_05 & 0x3F));
   for(j=0; j<5; j++)
   {
      if(us_temp&1)
      {
         idx++;

         idx = SkipExtraOctetAndCheckSegmentation(uca_M, idx, s_MsgLen, DELIMIT_1, s_SPar2ActiveBlocks, s_SPar2DecodedBlocks);
         if (idx == 0)
         {
            return(0);
         }
      }

      us_temp >>= 1;
   }

   //Decode G993_2 bit
   if ( t_Info->uc_SI_SPar1_05 & G993_2)
   {
      idx = DecodeStandard_VDSL2(&t_Info->t_VDSL2_Info, uca_M, idx, s_MsgLen);

      if (idx == 0)
      {
         return(0);
      }
   }


   //need to skip the octets corresponding to the reserved bit for future mode.
   if (t_Info->uc_SI_SPar1_05 & 0x40)
   {
      idx++;
      idx = SkipExtraOctetAndCheckSegmentation(uca_M, idx, s_MsgLen, DELIMIT_1, s_SPar2ActiveBlocks, s_SPar2DecodedBlocks);

      if (idx == 0)
      {
         return(0);
      }

   }

   //If we have sixth or more octets for SI_SPar1, we need to ignore the corresponding octets here.
   for(j=0; j<gus_ExtraBits; j++)
   {
      idx++;
      idx = SkipExtraOctetAndCheckSegmentation(uca_M, idx, s_MsgLen, DELIMIT_1, s_SPar2ActiveBlocks, s_SPar2DecodedBlocks);

      if (idx == 0)
      {
         return(0);
      }
   }

   return (idx);

}   // DecodeStandard


/*^^^
*-----------------------------------------------------------------------------
 *
 *  Name: DecodeStandard_VDSL2
 *
 *  Abstract:
 *
 *  DecodeStandard_VDSL2() - decodes the standard field section of the VDSL2
 *   modes from an array to a structure.
 *
 *  Input argument:
 *      uint8 *uca_M : information to be decoded from.
 *      int16  idx:      index into the buffer uca_M[]
 *      int16 s_MsgLen:   total number of octets in uca_M
 *
 *  Output argument:
 *      pt_Info - pointer to the decoded information strucuture
 *
 *  Returns:
 *      idx -- the updated index into the buffer uca_M[]
 *
 *  Global Variables Used: None
 *
 *  Notes:
 *
 *-----------------------------------------------------------------------------
 *^^^
 */
int16 DecodeStandard_VDSL2(VDSL2_Type_t *pt_info, uint8 uca_M[], int16 idx, int16 s_MsgLen)
{
   unsigned int i;
   int16 s_SPar2ActiveBlocks, s_SPar2DecodedBlocks;

   s_SPar2DecodedBlocks = 0;
   s_SPar2ActiveBlocks = 0;

   //Get NPar2
   pt_info->uc_SI_NPar2 = uca_M[idx++] & NEG_DELIMIT_2;
   // XDSLRTFW-487_VR9_VRX318_VDSL2_All_AELEM_Support (START)
   if ( (uca_M[idx-1] & DELIMIT_2) == 0 )
   {
      pt_info->uc_SI_NPar2_02 = uca_M[idx++] & NEG_DELIMIT_2;
   }
   // XDSLRTFW-487_VR9_VRX318_VDSL2_All_AELEM_Support (END)

   idx = SkipExtraOctetAndCheckSegmentation(uca_M, idx, s_MsgLen, DELIMIT_2, s_SPar2ActiveBlocks, s_SPar2DecodedBlocks);
   if (idx == 0)
   {
      return(0);
   }

   /* ---- check if level 2 parameter ends here ---- */
   if ( (uca_M[idx-1] & DELIMIT_1) != 0 )
   {
      return(idx);
   }

   //Get SPar2
   pt_info->uc_SI_SPar2_01 = uca_M[idx++] & NEG_DELIMIT_2;
   pt_info->uc_SI_SPar2_02 = uca_M[idx++] & NEG_DELIMIT_2;

   idx = SkipExtraOctetAndCheckSegmentation(uca_M, idx, s_MsgLen, DELIMIT_2, s_SPar2ActiveBlocks, s_SPar2DecodedBlocks);
   if (idx == 0)
   {
      return(0);
   }

   // Determine how many active NPar(3) blocks we need to decode in this message.
   // This is necessary in order to determine whether a message has been segmented.
   {
      uint16 us_SI_SPar2_VDSL2;

      us_SI_SPar2_VDSL2 = ((pt_info->uc_SI_SPar2_02 << 6) | pt_info->uc_SI_SPar2_01);
      for(i=0; i<12; i++)
      {
         if(us_SI_SPar2_VDSL2 & (1<<i))
         {
            s_SPar2ActiveBlocks++;
         }
      }
   }

   // VDSL2 profiles
   if ( (pt_info->uc_SI_SPar2_01 & V2_PROFILES) != 0 )
   {
      pt_info->uca_Profiles[0] = uca_M[idx++] & NEG_DELIMIT_2;

      if ( (uca_M[idx-1] & DELIMIT_2) == 0 )
      {
         pt_info->uca_Profiles[1] = uca_M[idx++] & NEG_DELIMIT_2;
         s_SPar2DecodedBlocks++;
      }

      idx = SkipExtraOctetAndCheckSegmentation(uca_M, idx, s_MsgLen, DELIMIT_2, s_SPar2ActiveBlocks, s_SPar2DecodedBlocks);
      if (idx == 0)
      {
         return(0);
      }
   }

   // Used band in US
   // If carrier masking feature is on, unpack the US band here. Because
   // O-SIGNATURE message will contain carrier mask band information instead of US band information.
   if ( (pt_info->uc_SI_SPar2_01 & V2_US_BANDS) != 0 )
   {
      UnpackMsgBufBandInfo(uca_M, &idx, pt_info->uca_UsedUsBands, &(pt_info->uc_NumUsBands), 2);
      s_SPar2DecodedBlocks++;

      idx = SkipExtraOctetAndCheckSegmentation(uca_M, idx, s_MsgLen, DELIMIT_2, s_SPar2ActiveBlocks, s_SPar2DecodedBlocks);
      if (idx == 0)
      {
         return(0);
      }

      gs_NumOfTxBands = pt_info->uc_NumUsBands;
      PackBandInfoTo16Bit(&pt_info->uca_UsedUsBands[0], &gsa_TxBandRightChannel[0], pt_info->uc_NumUsBands, (uint8) 0);
      PackBandInfoTo16Bit(&pt_info->uca_UsedUsBands[0], &gsa_TxBandLeftChannel[0], pt_info->uc_NumUsBands, (uint8) 3);
   }

   // Used band in DS
   if ( (pt_info->uc_SI_SPar2_01 & V2_DS_BANDS) != 0 )
   {
      UnpackMsgBufBandInfo(uca_M, &idx, pt_info->uca_UsedDsBands, &(pt_info->uc_NumDsBands), 2);
      s_SPar2DecodedBlocks++;

      idx = SkipExtraOctetAndCheckSegmentation(uca_M, idx, s_MsgLen, DELIMIT_2, s_SPar2ActiveBlocks, s_SPar2DecodedBlocks);
      if (idx == 0)
      {
         return(0);
      }

      // unpack the DS band info used for initial pilot tone selection
      // the actual band info is finalized based on O-SIGNATURE message
      // note that UnpackMsgBufBandInfo() call above doesn't actually unpack
      gs_NumOfEstRxBands = pt_info->uc_NumDsBands;
      PackBandInfoTo16Bit(&pt_info->uca_UsedDsBands[0], &gsa_EstRxBandRightChannel[0], pt_info->uc_NumDsBands, (uint8) 0);
      PackBandInfoTo16Bit(&pt_info->uca_UsedDsBands[0], &gsa_EstRxBandLeftChannel[0], pt_info->uc_NumDsBands, (uint8) 3);
   }

   // RFI band
   if ( (pt_info->uc_SI_SPar2_01 & V2_RFI_BANDS) != 0 )
   {
      UnpackMsgBufBandInfo(uca_M, &idx, pt_info->uca_RFIBands, &(pt_info->uc_NumRFIBands), 2);
      s_SPar2DecodedBlocks++;

      idx = SkipExtraOctetAndCheckSegmentation(uca_M, idx, s_MsgLen, DELIMIT_2, s_SPar2ActiveBlocks, s_SPar2DecodedBlocks);
      if (idx == 0)
      {
         return(0);
      }

      // Unpack the RFI band info.
      // Note that UnpackMsgBufBandInfo() call above doesn't actually unpack.
      gs_NumOfRFIBands = pt_info->uc_NumRFIBands;
      gt_RFIBandControl.us_NumberOfBands = pt_info->uc_NumRFIBands;
      PackBandInfoTo16Bit(&pt_info->uca_RFIBands[0], &gsa_RFIBandRightChannel[0], pt_info->uc_NumRFIBands, (uint8) 0);
      PackBandInfoTo16Bit(&pt_info->uca_RFIBands[0], &gsa_RFIBandLeftChannel[0], pt_info->uc_NumRFIBands, (uint8) 3);

      for (i=0; i < pt_info->uc_NumRFIBands; i++)
      {


         gt_RFIBandControl.ut_BandRecord[i].us_FirstTone = gsa_RFIBandLeftChannel[i];
         gt_RFIBandControl.ut_BandRecord[i].us_LastTone  = gsa_RFIBandRightChannel[i];
      }
   }

   //DFT size
   if ( (pt_info->uc_SI_SPar2_01 & V2_DFT_SIZE) != 0 )
   {
      pt_info->uc_fft_size = uca_M[idx++] & NEG_DELIMIT_2;
      s_SPar2DecodedBlocks++;

      idx = SkipExtraOctetAndCheckSegmentation(uca_M, idx, s_MsgLen, DELIMIT_2, s_SPar2ActiveBlocks, s_SPar2DecodedBlocks);
      if (idx == 0)
      {
         return(0);
      }
   }

   //CE length
   if ( (pt_info->uc_SI_SPar2_01 & V2_CE_LENGTH) != 0 )
   {
      pt_info->uca_CELength[0] = uca_M[idx++] & NEG_DELIMIT_2;

      // ---- check if level 2 parameter ends here
      if ( (uca_M[idx-1] & DELIMIT_2) == 0 )
      {
         pt_info->uca_CELength[1] = uca_M[idx++] & NEG_DELIMIT_2;

         if ( (uca_M[idx-1] & DELIMIT_2) == 0 )
         {
            pt_info->uca_CELength[2] = uca_M[idx++] & NEG_DELIMIT_2;
         }

         s_SPar2DecodedBlocks++;
      }

      idx = SkipExtraOctetAndCheckSegmentation(uca_M, idx, s_MsgLen, DELIMIT_2, s_SPar2ActiveBlocks, s_SPar2DecodedBlocks);
      if (idx == 0)
      {
         return(0);
      }

   }

   // Annex A US0 PSD
   if ( (pt_info->uc_SI_SPar2_02 & V2_ANNEX_A_US0_MASKS) != 0 )
   {

      for(i=0; i<5; i++)
      {
         pt_info->uca_US0_A_bands[i] = uca_M[idx++] & NEG_DELIMIT_2;

         if( (uca_M[idx-1] & DELIMIT_2) != 0 )
         {
            break;
         }
      }

      s_SPar2DecodedBlocks++;

      idx = SkipExtraOctetAndCheckSegmentation(uca_M, idx, s_MsgLen, DELIMIT_2, s_SPar2ActiveBlocks, s_SPar2DecodedBlocks);
      if (idx == 0)
      {
         return(0);
      }

   }

   // Annex B US0 PSD
   if ( (pt_info->uc_SI_SPar2_02 & V2_ANNEX_B_US0_MASKS) != 0 )
   {

      for(i=0; i<2; i++)
      {
         pt_info->uca_US0_B_bands[i] = uca_M[idx++] & NEG_DELIMIT_2;

         if( (uca_M[idx-1] & DELIMIT_2) != 0 )
         {
            break;
         }
      }

      s_SPar2DecodedBlocks++;

      idx = SkipExtraOctetAndCheckSegmentation(uca_M, idx, s_MsgLen, DELIMIT_2, s_SPar2ActiveBlocks, s_SPar2DecodedBlocks);
      if (idx == 0)
      {
         return(0);
      }

   }

   // Annex C US0 PSD
   if ( (pt_info->uc_SI_SPar2_02 & V2_ANNEX_C_US0_MASKS) != 0 )
   {

      for(i=0; i<3; i++)
      {
         pt_info->uca_US0_C_bands[i] = uca_M[idx++] & NEG_DELIMIT_2;

         if( (uca_M[idx-1] & DELIMIT_2) != 0 )
         {
            break;
         }
      }

      s_SPar2DecodedBlocks++;

      idx = SkipExtraOctetAndCheckSegmentation(uca_M, idx, s_MsgLen, DELIMIT_2, s_SPar2ActiveBlocks, s_SPar2DecodedBlocks);
      if (idx == 0)
      {
         return(0);
      }
   }

//#ifdef MTK_VECTORING_SUPPORT
   // G.993.2 Vectoring
   if ( (pt_info->uc_SI_SPar2_02 & V2_G_VECTOR_MASKS) != 0 )
   {
      pt_info->uc_Gvector = uca_M[idx++] & NEG_DELIMIT_2;
      s_SPar2DecodedBlocks++;

      idx = SkipExtraOctetAndCheckSegmentation(uca_M, idx, s_MsgLen, DELIMIT_2, s_SPar2ActiveBlocks, s_SPar2DecodedBlocks);
      if (idx == 0)
      {
         return(0);
      }
   }
//#endif

   // XDSLRTFW-1693 (Start_End)
   // G.998.4 Extensions
   if ( (pt_info->uc_SI_SPar2_02 & V2_9984_EXTENSIONS_MASKS) != 0 )
   {
      pt_info->uc_G9984_Extensions = uca_M[idx++] & NEG_DELIMIT_2;
      s_SPar2DecodedBlocks++;

      idx = SkipExtraOctetAndCheckSegmentation(uca_M, idx, s_MsgLen, DELIMIT_2, s_SPar2ActiveBlocks, s_SPar2DecodedBlocks);
      if (idx == 0)
         return(0);
   }

   return (idx);
} //DecodeStandard_VDSL2


/*
 *------------------------------------------------------------------------
 *
 *  Name : FormMessage
 *
 *  Abstract :
 *
 *  FormMessage() - Enclose an information field in an array.
 *  Block delimiters, FCS, and octet transparency are handled.
 *
 *  Parameters:
 *      const InfoField *t_Info : information to be packed
 *      uint8 uca_Msg[]         : output arrary containing the information
 *
 *  Returns:    Number of octets encoded into an array including FCS, octets
 *              inserted for octet transparency, and prefix and postfix FLAGs.
 *
 *  Global Variables Used:
 *      gt_hsc.s_TxNSBCnt : Number of non-standard blocks transmitted so far
 *
 *  Notes :      FormMessage() is called from HSMsgHandler(), which is executed
 *            in the background during state C_R_HS_MSG_RX.
 *
 *
 *------------------------------------------------------------------------
 */

int16 FormMessage(InfoField_t *t_Info, uint8 uca_Msg[])
{

   int16         idx, i;
   int16        s_MsgLen;
   uint8        uca_M[ MAX_MESSAGE_LEN + FCS_LEN ];
   uint16       us_FCS;                      // Frame Check Sequence

   idx = 0;



   //===============================================================================
   //All the message corresponding to one message type is formed
   //in the first-path and stored in the message buffer,
   //if M_ACK2 is received, directly go the FCS calculation state
   //================================================================================
   if ( gpt_RxInfo->uc_Type != M_ACK2 )
   {
      //Indicate a new message has been formed
      gft_RTXCtrlFlag |= GHS_NEW_TX_MSG_BIT;

      //Initialize the message buffer counts
      gus_NumberOfMsgOctetsSent = 0;
      gus_TotalNumberOfMsgOctets = 0;
      guc_TxHSMsg_SegmentCnt = 0;
      gt_hsc.s_TxNSBCnt = 0;  // reset nonstandard block counter

      //---- Message type field ----
      gpuca_EncodeBuf[idx++] = t_Info->uc_Type;

      //---- Revision number ----
      gpuca_EncodeBuf[idx++] = t_Info->uc_Rev;

      //==== Vendor ID (required only for CL and CLR messages) ====
      if ((t_Info->uc_Type == M_CL) || (t_Info->uc_Type == M_CLR))
      {
         //---- 2 country code octets, with the most significant octet first ----
         gpuca_EncodeBuf[idx++] = (uint8) ( t_Info->us_Country >> 8 );
         gpuca_EncodeBuf[idx++] = (uint8) ( t_Info->us_Country );

         //---- 4 vendor ID octets, with the most significant octet first ----
         Unpack32BitTo8BitArray(&t_Info->ul_Provider, &gpuca_EncodeBuf[idx]);
         idx += 4;

         //---- 2 octets of vendor specific information ----
         gpuca_EncodeBuf[idx++] = (uint8) ( t_Info->us_VendorInfo >> 8 );
         gpuca_EncodeBuf[idx++] = (uint8) ( t_Info->us_VendorInfo );

      }  //if (t_Info->uc_Type == M_CL)

      //=== Retransmission message ============================================
      if(t_Info->uc_Type == M_REQ_RTX)
      {
         gpuca_EncodeBuf[idx++] = t_Info->uc_last_received_msg_type;
         gpuca_EncodeBuf[idx++] = t_Info->uc_request_msg_segment_index;

         //Count the number of consecutive REQ-RTX being sent
         guc_TxHSReqRtx_cnt++;
      }
      else
      {
         guc_TxHSReqRtx_cnt = 0;
      }

      //---- Service and Channel parameters, Standard and Non-Standard Information ----
      //---- only for MS, CL, CLR messges                                          ----
      switch (t_Info->uc_Type)
      {
      case M_CL  :
      case M_CLR :
      case M_MS  :
      case M_MP  :

         idx = FormStandard( t_Info, gpuca_EncodeBuf, idx); // parameters and standard

         if ( (t_Info->uc_ID_NPar1 & NON_STANDARD) != 0 )
         {
            if (gul_dbgGHSControl & FORCE_NON_STANDARD_INFO)
            {
               if ((gsa_Optn0_ModeControl[1] & OPTN_ConfigMode_IKAN) &&
                     (gsa_Optn0_ModeControl[1] & OPTN_ConfigMode_G993_2))
               {
                  gpuca_EncodeBuf[idx++] = 0x03;

                  gpuca_EncodeBuf[idx++] = 0x07;
                  gpuca_EncodeBuf[idx++] = 0xB5;
                  gpuca_EncodeBuf[idx++] = 0x00;
                  gpuca_EncodeBuf[idx++] = 0x49;
                  gpuca_EncodeBuf[idx++] = 0x4B;
                  gpuca_EncodeBuf[idx++] = 0x4E;
                  gpuca_EncodeBuf[idx++] = 0x53;
                  gpuca_EncodeBuf[idx++] = 0x00;

                  gpuca_EncodeBuf[idx++] = 0x0A;
                  gpuca_EncodeBuf[idx++] = 0xB5;
                  gpuca_EncodeBuf[idx++] = 0x00;
                  gpuca_EncodeBuf[idx++] = 0x49;
                  gpuca_EncodeBuf[idx++] = 0x4B;
                  gpuca_EncodeBuf[idx++] = 0x4E;
                  gpuca_EncodeBuf[idx++] = 0x53;
                  gpuca_EncodeBuf[idx++] = 0x00;
                  gpuca_EncodeBuf[idx++] = 0x00;
                  gpuca_EncodeBuf[idx++] = 0x00;
                  gpuca_EncodeBuf[idx++] = 0x2C;

                  gpuca_EncodeBuf[idx++] = 0x0F;
                  gpuca_EncodeBuf[idx++] = 0xB5;
                  gpuca_EncodeBuf[idx++] = 0x00;
                  gpuca_EncodeBuf[idx++] = 0x49;
                  gpuca_EncodeBuf[idx++] = 0x4B;
                  gpuca_EncodeBuf[idx++] = 0x4E;
                  gpuca_EncodeBuf[idx++] = 0x53;
                  gpuca_EncodeBuf[idx++] = 0x00;
                  if (gl_InitRxHsTonesPower > 0x8000000)
                  {
                     gpuca_EncodeBuf[idx++] = 0x08;   //Default Bands Upto 2KFT (DS1/US1/DS2/US2)
                  }
                  else if (gl_InitRxHsTonesPower > 0x3000000)
                  {
                     gpuca_EncodeBuf[idx++] = 0x09;   //Drop US2 @ ~2KFT (DS1/US1/DS2)
                  }
                  else if (gl_InitRxHsTonesPower > 0x2000000)
                  {
                     gpuca_EncodeBuf[idx++] = 0x0B;   //Drop DS2 @ ~3KFT (US0/DS1/US1)
                  }
                  else
                  {
                     gpuca_EncodeBuf[idx++] = 0x11;   //Drop US1 Beyond 3.5KFT (US0/DS1)
                  }
                  gpuca_EncodeBuf[idx++] = 0x00;
                  gpuca_EncodeBuf[idx++] = 0xD2;
                  gpuca_EncodeBuf[idx++] = 0x32;
                  gpuca_EncodeBuf[idx++] = 0x02;   //0x04 when US0
                  gpuca_EncodeBuf[idx++] = 0x00;   //0x07 when US0
                  gpuca_EncodeBuf[idx++] = 0x01;
                  gpuca_EncodeBuf[idx++] = 0x80;
               }
            }
            else
            {
               idx = FormNonStandard( t_Info, gpuca_EncodeBuf, idx ); // non standard
            }
         }

         break;

      case M_MR  :
         if (gul_dbgGHSControl & FORCE_VENDOR_SPECIFIC_MODE_REQUEST)
         {
            // Force vendor specific mode request used by Ikanos CPE
            if (gsa_Optn0_ModeControl[1] & OPTN_ConfigMode_IKAN)
            {
               gpuca_EncodeBuf[idx++] = 0x00;
               gpuca_EncodeBuf[idx++] = 0x00;
               gpuca_EncodeBuf[idx++] = 0x00;
               gpuca_EncodeBuf[idx++] = 0x00;   //0x2B   //Ikanos CPE uses 0x2B here which seems to enable Trellis on US
               gpuca_EncodeBuf[idx++] = 0x01;   //0x0B
               gpuca_EncodeBuf[idx++] = 0xB4;   //0x6D
               gpuca_EncodeBuf[idx++] = 0xD2;
               gpuca_EncodeBuf[idx++] = 0x30;
               gpuca_EncodeBuf[idx++] = 0x02;   //0x04
               gpuca_EncodeBuf[idx++] = 0x00;   //0x07
               gpuca_EncodeBuf[idx++] = 0x01;
               gpuca_EncodeBuf[idx++] = 0x80;
            }
         }
         break;

      default :
         break;
      } //switch (t_Info->uc_Type)

      //set the total number of octets in this message
      gus_TotalNumberOfMsgOctets = idx;

   }   //if ( gpt_RxInfo.uc_Type != M_ACK2 )

   //If the number of sent octets is less than the total number of octets in a message
   if(gus_NumberOfMsgOctetsSent < gus_TotalNumberOfMsgOctets)
   {

      //========= Decide the length of this message segment ========
      //Limit the size of this segment by the maximum message segment length allowed by the standard
      idx = gus_TotalNumberOfMsgOctets - gus_NumberOfMsgOctetsSent;
      if ( idx <= MAX_MESSAGE_LEN )
      {
         s_MsgLen = idx;
      }
      else
      {
         s_MsgLen = MAX_MESSAGE_LEN;

         //Make sure the size of the last segment greater than the minimum message length allowed by the standard
         if ((idx - MAX_MESSAGE_LEN) < MIN_MESSAGE_LEN)
         {
            s_MsgLen -= MIN_MESSAGE_LEN;
         }
      }

      //==== Frame Check Sequence ====
      us_FCS   = 0xFFFF;    // initialize to all binary 1's

      //Move s_MsgLen number of octets from the big message buffer to a local buffer
      //and calculate FSC
      idx = 0;
      for ( i = gus_NumberOfMsgOctetsSent; i < gus_NumberOfMsgOctetsSent + s_MsgLen; i++ )
      {
         uca_M[idx++] = gpuca_EncodeBuf[i];
         us_FCS = calcCrc16(us_FCS, gpuca_EncodeBuf[i]);
      }

      us_FCS = ~us_FCS;   // take 1's compliment

      //Send LSB first, and MSB second
      uca_M[idx++] = (uint8) ( us_FCS & 0xFF);
      uca_M[idx++] = (uint8) ( us_FCS >> 8 );

      //Update the count which counts the number of message octets being sent out
      gus_NumberOfMsgOctetsSent += s_MsgLen;

      //Indicate that if the whole message has been sent or not
      if(gus_NumberOfMsgOctetsSent < gus_TotalNumberOfMsgOctets)
      {
         gft_WholeTxMsgComplete = FALSE;
      }
      else
      {
         gft_WholeTxMsgComplete = TRUE;
      }

      //Increment TX segment count
      guc_TxHSMsg_SegmentCnt++;

      //==== Octet transparency ====
      idx = OctetTransparency(uca_M, idx, uca_Msg);


   } //if(gus_NumberOfMsgOctetsSent < gus_TotalNumberOfMsgOctets)

   // for debug purpose, log GHS messages transmitted
   DebugGHSMessages(t_Info->uc_Type, idx, uca_Msg, 0x7777);
#ifdef DEBUG_TRACES
   // XDSLRTFW-598 FEATURE_ALL_ALL_ALL_Debug_Buffer(START_END)
   LogMessages(4,(uint16)t_Info->uc_Type,(uint16)idx, uca_Msg, 0x77777777);
#endif //#ifdef DEBUG_TRACES
      DSH_SendEvent(DSH_EVT_GHS_TX,(uint8)idx,&uca_Msg[0]);
   //---- return the whole length (No. of octets) of the message ----
   return ( idx );

}   // FormMessage


//*************************************************
//
#endif // ifdef INCLUDE_CACHE_FUNCTIONS
//
//*************************************************



