/* **COPYRIGHT******************************************************************
    INTEL CONFIDENTIAL
    Copyright (C) 2017 Intel Corporation
    Copyright (C), 1994-2011 Aware Inc. All Rights Reserved.
******************************************************************COPYRIGHT** */
/* **DISCLAIMER*****************************************************************
    The source code contained or described herein and all documents related
    to the source code ("Material") are owned by Intel Corporation or its
    suppliers or licensors. Title to the Material remains with Intel
    Corporation or its suppliers and licensors. The Material may contain
    trade secrets and proprietary and confidential information of Intel
    Corporation and its suppliers and licensors, and is protected by
    worldwide copyright and trade secret laws and treaty provisions. No part
    of the Material may be used, copied, reproduced, modified, published,
    uploaded, posted, transmitted, distributed, or disclosed in any way
    without Intel's prior express written permission.

    No license under any patent, copyright, trade secret or other
    intellectual property right is granted to or conferred upon you by
    disclosure or delivery of the Materials, either expressly, by
    implication, inducement, estoppel or otherwise. Any license under
    such intellectual property rights must be express and approved by
    Intel in writing.
*****************************************************************DISCLAIMER** */
/*
 *-------------------------------------------------------------------------
 *
 *   Aware DMT Technology. Proprietary and Confidential.
 *
 *   40 Middlesex Turnpike, Bedford, MA 01730-1413
 *   Phone (781) 276 - 4000
 *   Fax   (781) 276 - 4001
 *
 *   C_HS_RESP.C
 *
 *   Functions that are used to decide how to respond to received messages
 *   in a G.hs session.
 *
 *-------------------------------------------------------------------------
 */

// ***********************************************************************************************************
// hs_resp.c
//
// History
// 25/04/2013 Fuss: MS silent bit evaluatated for bonding
//            Grep for XDSLRTFW-915: : BugFix_All_VDSL2_All_BondingMsSilentBitHandling
// 16/05/2012 Vinjam: Code Pointers connected to Downstream Virtual Noise feature
//            Grep for XDSLRTFW-439: Feature_All_DS_All_All_SupportDsTxRefVirtualNoise
// 10/07/2013 Sriram : Endless G.Hs training session in case of missing common mode in CO / CPE capabilities
//            Grep for XDSLRTFW-961 :BugFix_ALL_ALL_ALL_Hsk_NAKEF_Error
// 19/09/2013 Fuss: Change variables to default values for VDSL2/Vectroing, so that full hybrid training gets performed.
//            Grep for XDSLRTFW-1181: BugFix_DS_VDSL2_All_VecDSRateRunToRunVariation
// 19/09/2013 Abu: Change Slow and fast PLL settings to default VDSL2 values
//            Grep for XDSLRTFW-1182: Bugfix_US_VDSL2_ALL_LowUsRate
// 16/12/2013 Varun : Modified the code to send the ILV memory info always if Amend2 is enabled
//                    Grep for XDSLRTFW-1441
// 30/04/2014 Fuss: Bonding issues seen NVLT-G and VGV Arcadyan modem, KPN
//            Grep for XDSLRTFW-1695
// 30/04/2014 Fuss: Ghs NoCommonMode message from CO is not supported
//            Grep for XDSLRTFW-1730
// 26/06/2014 Fuss: Extended O-P-Vector1 phase, i.e. Evaluation of MS message information + setting hostmessages
//            Grep for XDSLRTFW-1797
// 08/07/2014 Fuss: Extended O-P-Vector1 phase, i.e. timeout adaptation
//            Grep for XDSLRTFW-1872
// 11/08/2014 Fuss: New echo-algo for hybrid selection gets be used also for plain vdsl
//            Grep for XDSLRTFW-1488
// 11/08/2014 Fuss: calibration of BJ Hybrid and oPOTS config
//            Grep for XDSLRTFW-1898
//
// 04/12/2014  Anantha Ramu: SRA disabled for Intra DTU Interleaving(US & DS).
//               Grep for XDSLRTFW-1617
// 13/08/2015 Stefan: set hardcoded EFM/PTM mode if VDSL is started in G.Vectoring mode
//            Grep for XDSLRTFW-2456 Long_train_time_in_VDSL_when_switching_from_ADSL_to_VDSL_Vectoring
// 26/11/2015 Anantha Ramu: Added modifications for SRA with DS Intra DTU Interleaving.
//                          Grep for "XDSLRTFW-2162".
// 26/11/2015 Anantha Ramu: Added modifications for SRA with US Intra DTU Interleaving.
//                          Grep for "XDSLRTFW-2341".
//
// 08/06/2018 Hanyu: XDSLRTFW-3833 Added code for US data rate improvement against EVLT-K/BRCM with non-35B profiles at CTL/AT&T
//   1. Boosted additional 0.5dBm/Hz US1 PSD for all COs to improve US data rates on 3kft and longer loops.
//   2. Excluded XDSLRTFW-1181/1182 for all COs except IFX in non-vectoring mode to improve US data rate at short loops and reduce DS run to run variation.
//      Search Pattern: XDSLRTFW-3833 XDSLRTFW-735 IOP_A_US_VDSL2_EVLT_K_BRCM_USdatarate
//      Search Pattern: XDSLRTFW-3833 XDSLRTFW-735 IOP_A_US_VDSL2_VINAX_USdatarate
//
// 30/08/2018 Sriram Shastry
//            XDSLRTFW-3841 - VRX518 stuck in line state 300 in VDSL-ATM, VDSL-PTM and AUTOMODE
//            GHs is stuck in an endless loop if the previous GHs session was terminated by CO
//            by sending a "NAK-EF" command (The Rx-Message structure is not re-initialized - so
//            GHs will be terminated by FW automatically once the tone-detection phase is passed
//            and the message handler HSMsgHandler gets started.
//            Resetting (at least the header) of the Rx-Msg structure resolves the problem
// Grep for XDSLRTFW-3841
// ************************************************************************************************************

#include <string.h>
#include <stdio.h>
#include "common.h"
#include "gdata.h"
#include "ghs.h"
#include "cmv.h"
#include "vdsl_xception.h"
#include "vdsl_state.h"
#include "pll.h"
#include "vdsl_const.h"
#include "ppe_memmap.h"

void SelectHighestBit(uint8 *puca_DstTx, uint8 *puca_SrcRxCL, uint8 *puca_SrcTxCL, uint8 uc_NumBytes);


/*^^^
 *-------------------------------------------------------------------
 *
 *  Name : HSMsgHandler
 *
 *  Abstract : Decide how to respond to the received message.
 *
 *  Returns: None
 *
 *  Global Variables Used:
 *         gt_hsc               - (I/O) GHS state information
 *          gpt_RxInfo              - (I) pointer to received message
 *          gpt_TxInfo              - (O) pointer to message to be transmitted in response
 *         gs_RxBkgdProcessFlag   - (O) flag to indicate the background process is done or not
 *
 *  Notes :
 *
 *-------------------------------------------------------------------
 *^^^
 */


void HSMsgHandler(void)
{

   guc_NotUnderstood  = FALSE;

   /********************************************************************/
   //---- if FCS error, transmit NAK_EF and return to the beginning ----
   /********************************************************************/
   if (gt_hsc.s_DecodeResult == FCS_ERROR)
   {
      //Response to errored frame
      ResponseToErrorFrame();
      //Reset this result flag so this part will not be called again
      gt_hsc.s_DecodeResult = DECODE_IDLE;

      //XDSLRTFW-3841(Start)
      // Always transition back to the Ready state if a GHs session cannot be finished properly
      gsa_IndirectStat0[0] = STAT_ReadyState;
      gusa_MONI_ModemStat_Status[0] = MONI_STAT_READY;


      // VRX518 stuck in line state 300 in ADSL-ATM, ADSL-PTM and AUTOMODE
      // The message type of structure gpt_RxInfo needs to be cleared.
      // If it is not cleared the next GHs session will automatically stop here before decoding
      // the first CL message as we call the Ghs message handler immediately after having
      // detected the 1st CO FLAG signal (after receiving activation tones).
      // Here we check for the Rx-Message header even if have not yet received any CO GHs message
      // in the ongoing GHs session.
      gpt_RxInfo->uc_Type = 0x00;
      //XDSLRTFW-3841(End)


      goto _return;
   }

   //Update the last correctly transmitted and received message type if a new message is received
   if(gft_RTXCtrlFlag & GHS_NEW_RX_MSG_BIT)
   {
      UpdateMsgTypeForRTX();

      guc_NewRxMsgType = gpt_RxInfo->uc_Type;

      //clear RX message big
      gft_RTXCtrlFlag &= ~GHS_NEW_RX_MSG_BIT;
   }
   else
      //Indicate no new message is received (used for LeaveTrail)
   {
      guc_NewRxMsgType = M_NO_MSG;
   }

   /*  **************************************** */
   /*  **** common responses of all states **** */
   /*  **************************************** */

   /* ==== handling NAK_EF, NAK_CD, and not-understood message ==== */
   if( gpt_RxInfo->uc_Type == M_NAK_EF)
   {
      //XDSLRTFW-1901: ADSL/VDSL ACTIVATION STATE MACHINE (START_END)
      gsa_IndirectStat0[2]=VDSL2_NAK_EF;
      /* ---- NAK_EF handling ( errored frame received by ATU-R ) ---- */
      /* ---- go back to the beginning as specified in G.hs spec  ---- */
      gt_hsc.l_SelectedMode = RESTART_HS_FarEnd;   //XDSLRTFW-961 :BugFix_ALL_ALL_ALL_Hsk_NAKEF_Error(Start_End)

      //XDSLRTFW-3841(Start)
      // Always transition back to the Ready state if a GHs session cannot be finished properly
      gsa_IndirectStat0[0] = STAT_ReadyState;
      gusa_MONI_ModemStat_Status[0] = MONI_STAT_READY;

      // VRX518 stuck in line state 300 in ADSL-ATM, ADSL-PTM and AUTOMODE
      // The message type of structure gpt_RxInfo needs to be cleared.
      // If it is not cleared the next GHs session will automatically stop here before decoding
      // the first CL message as we call the Ghs message handler immediately after having
      // detected the 1st CO FLAG signal (after receiving activation tones).
      // Here we check for the Rx-Message header even if have not yet received any CO GHs message
      // in the ongoing GHs session.
      gpt_RxInfo->uc_Type = 0x00;
      //XDSLRTFW-3841(End)


      gt_hsc.s_NextState    = R_HS_EXIT;

      goto _return;
   }
   else if (gpt_RxInfo->uc_Type == M_NAK_CD)
   {
      //XDSLRTFW-1901: ADSL/VDSL ACTIVATION STATE MACHINE (START_END)
      gsa_IndirectStat0[2]=VDSL2_NAK_CD;
      //---- NAK_CD handling ( message not understoodd by ATU-R )        ----
      // ---- go back to HS_INITIAL state to resume another transaction ----
      Do_ClearDown();
      gt_hsc.l_SelectedMode = RESTART_VIA_CLEARDOWN;       /*  Select mode of operation based on xmitted MS message */

      goto _return;
   }

   //---- REQ_RTX handling ( errored segment frame retransmission request received by ATU-R )
   else if (gpt_RxInfo->uc_Type == M_REQ_RTX)
   {
      ResponseToRTX();
      goto _return;
   }

   {
      uint16 us_JumpTemp;

      us_JumpTemp = gt_hsc.s_State - R_HS_INITIAL;
      // Do not execute R_HS_RETRANSMIT, since patents are available for this feature!!
      if(us_JumpTemp > (R_HS_SEND_MP - R_HS_INITIAL))
      {
         us_JumpTemp = (R_HS_SEND_MP - R_HS_INITIAL);
      }
      (*MsgHandler_states[us_JumpTemp])();
   }
//   (*MsgHandler_states[gt_hsc.s_State - R_HS_INITIAL])();


   /* ---- Check whether Message not understood ---- */
   if ( guc_NotUnderstood == TRUE )
   {
      if ( gpt_RxInfo->uc_Rev <= gpt_TxInfo->uc_Rev )
      {
         gpt_TxInfo->uc_Type  = M_NAK_CD;

         gt_hsc.s_NextState = R_HS_NOT_UNDERSTOOD;

      }
      else
      {
         gpt_TxInfo->uc_Type  = M_NAK_NS;
         //XDSLRTFW-1901: ADSL/VDSL ACTIVATION STATE MACHINE (START_END)
         gsa_IndirectStat0[2]=VDSL2_NAK_NS;
         gt_hsc.s_NextState = R_HS_NOT_SUPPORTED;
      }

      gs_TxMsgBufCnt     = FormMessage( gpt_TxInfo, gpuca_TxMsgBuf );

   }

_return:

   gs_RxBkgdProcessFlag = BG_GHS_TASK_DONE;
}

/*^^^
 *-------------------------------------------------------------------
 *
 *  Name : ResponseToErrorFrame
 *
 *  Abstract : This function decides how to response to the frame which contains CRC error
 *            by either replying with NAK_EF or REQ-RTX
 *
 *  Returns: None
 *
 *  Global Variables Used:
 *         gt_hsc               - (I/O) GHS state information
 *          gpt_TxInfo              - (O) pointer to message to be transmitted in response
 *         gft_RTXCtrlFlag         - (I) flag to control retransmission
 *         guc_last_rx_msg_type   - (I) the last correctly received frame type
 *         guc_RxHSMsg_SegmentCnt   - (I) segment index of last received errored frame
 *         gpuca_TxMsgBuf         - (O) the TX message frame containing the responding message
 *         gs_TxMsgBufCnt         - (O) the number of byte in TX message
 *  Notes :
 *
 *-------------------------------------------------------------------
 *^^^
 */
//Response to errored frame
void ResponseToErrorFrame(void)
{

   //If the retransmission is disabled, send NAK_EF message back
   if((gft_RTXCtrlFlag&GHS_ENABLE_REQ_RTX_BIT) == 0)
   {
      gpt_TxInfo->uc_Type  = M_NAK_EF;
      //XDSLRTFW-1901: ADSL/VDSL ACTIVATION STATE MACHINE (START_END)
      gsa_IndirectStat0[2]=VDSL2_NAK_EF;
      gt_hsc.s_NextState = R_HS_RESTART;
   }
   else
   {
      //If the number of consecutive REQ-RTX messages is less then 3, sends another REQ-RTX
      if(guc_TxHSReqRtx_cnt < 3)
      {
         gpt_TxInfo->uc_Type  = M_REQ_RTX;
         gpt_TxInfo->uc_last_received_msg_type = guc_last_rx_msg_type;
         gpt_TxInfo->uc_request_msg_segment_index = guc_RxHSMsg_SegmentCnt;

         //do not change gt_hsc.s_NextState
      }
      //Otherwise, send NAK-CD message
      else
      {

         // transmit NAC_CD to do clear down
         gpt_TxInfo->uc_Type = M_NAK_CD;
         //XDSLRTFW-1901: ADSL/VDSL ACTIVATION STATE MACHINE (START_END)
         gsa_IndirectStat0[2]=VDSL2_NAK_CD;
         gt_hsc.s_NextState  = R_HS_NOT_UNDERSTOOD;
      }
   }

   gs_TxMsgBufCnt = FormMessage(gpt_TxInfo, gpuca_TxMsgBuf);
}

/*^^^
 *-------------------------------------------------------------------
 *
 *  Name : ResponseToRTX
 *
 *  Abstract : This function handles the response to the REQ-RTX message
 *            It will either start the retransmission of previous message
 *            or send NAK-CD to deny retransmission request.
 *
 *  Returns: None
 *
 *  Global Variables Used:
 *          guc_RxHSReqRtx_cnt      - (I/O) no. of REQ-RTX received
 *          gpt_RxInfo              - (I) the strucuture of the received message information
 *          gpt_TxInfo              - (I) the strucuture of the previous transmitted message information
 *          guc_TxHSMsg_SegmentCnt  - (I) no. of TX segments currently being transmitted
 *          gs_TxMsgBufCnt         - (I) the last transmitted TX message length
 *          gt_hsc               - (O) the structure of GHS state information
 *
 *  Notes :
 *
 *-------------------------------------------------------------------
 *^^^
 */

void ResponseToRTX(void)
{

   //If no. of received REQ_RTX is greater than the limit,
   //or if the message does not match to the last correctly transmitted message type,
   //or if segment number is not match to the last transmitted segment number
   //we deny retransmission request by sending M_NAK_CD
   //(Note: to save the memory, we do not support retransmission of any segment prior to
   // the last transmitted segment)
   if ((guc_RxHSReqRtx_cnt > MAX_REQ_RTX_NUMBER)
         || (gpt_RxInfo->uc_last_received_msg_type != guc_last_tx_msg_type)
         || (gpt_RxInfo->uc_request_msg_segment_index != (guc_TxHSMsg_SegmentCnt-1)))
   {
      // transmit NAC_CD to do clear down
      gpt_TxInfo->uc_Type = M_NAK_CD;
      gs_TxMsgBufCnt      = FormMessage( gpt_TxInfo, gpuca_TxMsgBuf );

      gt_hsc.s_NextState  = R_HS_NOT_UNDERSTOOD;

   }
   //This implementation only supports to retransmit the last transmited segment, but not those
   //prior to it.
   else
   {
      //Set the message length to the last one being transmitted
      gs_TxMsgBufCnt = gs_LastTxMsgBufCnt;
   }
} //void ResponseToRTX(void)

//Update the last correctly transmitted and received message type
void UpdateMsgTypeForRTX(void)
{

   //if a new message is received
   //Update the last correctly transmitted message type
   if(gft_RTXCtrlFlag & GHS_NEW_TX_MSG_BIT)
   {
      if((gpt_RxInfo->uc_Type != M_NAK_EF) && (gpt_RxInfo->uc_Type != M_REQ_RTX))
      {
         guc_last_tx_msg_type = gpt_TxInfo->uc_Type;
      }

      //Reset the flag
      gft_RTXCtrlFlag &= ~GHS_NEW_TX_MSG_BIT;
   }

   //Update the last correctly received message type
   guc_last_rx_msg_type = gpt_RxInfo->uc_Type;
} //void UpdateMsgTypeForRTX(void)

/*^^^
 *-------------------------------------------------------------------
 *
 *  Name : PrepareMS
 *
 *  Abstract : Sets up parameters for an MS message
 *
 *  Returns: None
 *
 *  Global Variables Used:
 *
 *  Notes :
 *
 *-------------------------------------------------------------------
 *^^^
 */

//---- macro to indicate no common mode ----
void IndicateNoCommonMode(void)
{
   gpt_TxInfo->uc_ID_SPar1 = 0;
   gpt_TxInfo->uc_ID_NPar1 &= ~NON_STANDARD;
   gpt_TxInfo->uc_SI_NPar1  = 0;
   gpt_TxInfo->uc_SI_SPar1  = 0;
}


void PrepareMS_VDSL2(VDSL2_Type_t *pt_info,  VDSL2_Type_t *pt_RxCL,  VDSL2_Type_t *pt_TxCL);


void PrepareMS(void)
{
   InfoField_t *pt_RxCL;   // received capability list of the other party
   InfoField_t *pt_TxCL;   // transmitted capability list of CO

   pt_RxCL = gpt_RxInfoSave;
   pt_TxCL = gpt_TxInfoSave;

   gt_hsc.s_NextState    = R_HS_SEND_MS;

   gpt_TxInfo->uc_Type = M_MS;
   // Note: uc_Rev is set at init and must stay set as it is!
   // gpt_TxInfo->uc_Rev = REVISION2;

   //Clear all variables in one path to save program memory
   {
      uint8 *puc_Ptr;

      puc_Ptr = &gpt_TxInfo->uc_ID_NPar1;
      while (puc_Ptr < &gpt_TxInfo->uc_NumBlock)
      {
         *puc_Ptr++ = 0;
      }
   }

   //---- select common mode if CLR and CL have been exchanged ---
   if (gt_hsc.s_CL_Exchanged == TRUE)
   {
      if((pt_TxCL->uc_ID_SPar1 & TRANSCEIVER_ID) &&
            (pt_RxCL->uc_ID_SPar1 & TRANSCEIVER_ID) && (pt_RxCL->uc_ID_SPar2_TransId & CO_TRANSCEIVER_ID))
      {
         gpt_TxInfo->uc_ID_SPar1 = TRANSCEIVER_ID;
         gpt_TxInfo->uc_ID_SPar2_TransId = gpt_TxInfoSave->uc_ID_SPar2_TransId;
         gpt_TxInfo->ul_CoTransId = gpt_TxInfoSave->ul_CoTransId;
         gpt_TxInfo->ul_CpeTransId = gpt_TxInfoSave->ul_CpeTransId;
      }

      // Check if VDSL2 supported at both ends
      // If clause commented out, since VDSL2 is the only supported mode
      //if(pt_RxCL->uc_SI_SPar1_05 & pt_TxCL->uc_SI_SPar1_05 & G993_2)
      //{
      gpt_TxInfo->uc_SI_SPar1_05 = G993_2;
      PrepareMS_VDSL2(&(gpt_TxInfo->t_VDSL2_Info), &(pt_RxCL->t_VDSL2_Info), &(pt_TxCL->t_VDSL2_Info));
      //}
   }

}   // PrepareMS


/*^^^
 *-------------------------------------------------------------------
 *
 *  Name: PrepareMS_VDSL2
 *
 *  Abstract:
 *
 *  PrepareMS_VDSL2() - Form the VDSL2 specific NPar2 and NPar3 parameters
 *   for a MS message, based on CL and CLR information.
 *
 *  Input argumemts:
 *      VDSL2_Type_t *pt_RxCL    : pointer to the structure of CLR information received
 *      VDSL2_Type_t *pt_TxCL    : pointer to the structure of CL information transmitted
 *
 *  Output argument:
 *      VDSL2_Type_t *pt_info    : pointer to the MS information to be transmitted
 *
 *
 *  Returns: None
 *
 *  Global Variables Used: None
 *
 *  Notes:
 *
 *-------------------------------------------------------------------
*^^^
*/
void SelectHighestBit(uint8 *puca_DstTx, uint8 *puca_SrcRxCL, uint8 *puca_SrcTxCL, uint8 uc_NumBytes)
{
   uint8 *puc_Dst;
   unsigned int i;

   // Select and set highest common profile.
   i = uc_NumBytes;
   do
   {
      i--;
      puc_Dst = &puca_DstTx[i];
      *puc_Dst = (puca_SrcRxCL[i] & puca_SrcTxCL[i]);
   }
   while((i > 0) && !(*puc_Dst));

   i = DELIMIT_2;
   do
   {
      i = i >> 1;
      if(*puc_Dst & i)
      {
         *puc_Dst = i;
         i = 0;
      }
   }
   while(i > 0);
}

void PrepareMS_VDSL2(VDSL2_Type_t *pt_info,  VDSL2_Type_t *pt_RxCL,  VDSL2_Type_t *pt_TxCL)
{
//   FlagT ft_ghsStdNoncompliant, ft_ghsRetrain, ft_CheckForUS0PSDMaskEnabled;
   FlagT ft_ghsStdNoncompliant, ft_ghsRetrain;
   uint8 uc_TempVar;
   signed int i;

   ft_ghsStdNoncompliant = FALSE;
   ft_ghsRetrain = FALSE;
//   ft_CheckForUS0PSDMaskEnabled = FALSE;
   uc_TempVar = 0;

   //===========================================================================
   //Decide Standard Information NPar2
   //===========================================================================
   pt_info->uc_SI_NPar2 = (pt_RxCL->uc_SI_NPar2 & pt_TxCL->uc_SI_NPar2);

   //Select the loop diagnostic mode if either the last CLR or CL message
   //has set this bit
   if((pt_RxCL->uc_SI_NPar2 & V2_LOOP_DIAG) || (pt_TxCL->uc_SI_NPar2 & V2_LOOP_DIAG))
   {
      pt_info->uc_SI_NPar2 |= V2_LOOP_DIAG;
   }

   // Note: The vectoring friendly mode has to be set later dependant on full vectoring!
   pt_info->uc_SI_NPar2_02 = (uint8)((pt_RxCL->uc_SI_NPar2_02 & pt_TxCL->uc_SI_NPar2_02) & ~(V2_VECTOR_FULL_FRIENDLY|V2_VECTOR_FRIENDLY));

   //===========================================================================
   //Decide Standard Information SPar2_01 and SPar2_02
   //===========================================================================
   // SPar2_01
   // Note:
   //    Profiles                    Always set to ONE.
   //    Bands Upstream              Always set to ZERO.
   //    Bands Downstream            Always set to ZERO
   //    RFI Bands                   Always set to ZERO.
   //    Initial IDFT Size (2N)      Always set to ZERO.
   //    CE Lengths                  Shall be set to ONE if and only if both the last previous CLR and the last
   //                                previous CL messages have set this bit to ONE. If set to ONE, indicates that the
   //                                initial CE length to be used by both the VTU-O and the VTU-R shall be
   //                                communicated in the corresponding NPar(3) field. If set to ZERO, the
   //                                mandatory value shall be used.
   pt_info->uc_SI_SPar2_01 = (pt_RxCL->uc_SI_SPar2_01 & pt_TxCL->uc_SI_SPar2_01);
   pt_info->uc_SI_SPar2_01 &= ~(V2_US_BANDS | V2_DS_BANDS | V2_RFI_BANDS | V2_DFT_SIZE);

   // SPar2_02
   // Note:
   //    Annex A US0 (NOTE)          May be set to ONE if and only if both the last previous CLR and the last
   //                                previous CL messages have set this bit to ONE.
   //    Annex B US0 (NOTE)          May be set to ONE if and only if both the last previous CLR and the last
   //                                previous CL messages have set this bit to ONE.
   //    Annex C US0 (NOTE)          May be set to ONE if and only if both the last previous CLR and the last
   //                                previous CL messages have set this bit to ONE.
   //    G.993.5 Vectoring
   //
   //    (NOTE - One and only one of these bits shall be set to ONE.)
   pt_info->uc_SI_SPar2_02 = (pt_RxCL->uc_SI_SPar2_02 & pt_TxCL->uc_SI_SPar2_02) ;

   if(!(pt_info->uc_SI_SPar2_02 & V2_G_VECTOR_MASKS))
   {
      uc_TempVar = ((pt_RxCL->uc_SI_NPar2_02 & pt_TxCL->uc_SI_NPar2_02) & (V2_VECTOR_FULL_FRIENDLY|V2_VECTOR_FRIENDLY));
      // Only one friendly bit is allowed to be set! Further the "ITU-T G.993.5" bit shall be set to ZERO.
      if(uc_TempVar & V2_VECTOR_FULL_FRIENDLY)
      {
         uc_TempVar &= (~V2_VECTOR_FRIENDLY);
      }
   }

   pt_info->uc_SI_NPar2_02 |= uc_TempVar;

   //===========================================================================
   // Decide NPar3
   //===========================================================================
   // Choose the CO preferred profile
   {
      uint16 us_Profiles;

      us_Profiles = 0;
      if(pt_info->uc_SI_SPar2_01 & V2_PROFILES)
      {
         us_Profiles = ((pt_RxCL->uca_Profiles[1] << 6) | pt_RxCL->uca_Profiles[0]);
         us_Profiles &= ((pt_TxCL->uca_Profiles[1] << 6) | pt_TxCL->uca_Profiles[0]);
         us_Profiles &= gt_ProfileControl.us_ProfilePreferred;

         gt_ProfileAct.us_ProfileSelected = us_Profiles;

         pt_info->uca_Profiles[0] = (us_Profiles & ((V2_PROFILE_8D|V2_PROFILE_8C|V2_PROFILE_8B|V2_PROFILE_8A) |
                                     (V2_PROFILE_12B|V2_PROFILE_12A)));
         pt_info->uca_Profiles[1] = ((us_Profiles >> 6) & (V2_PROFILE_35B|V2_PROFILE_30A|V2_PROFILE_17A));
      }

      // Profile selection is mandatory
      if (us_Profiles == CNFG_V2_PROFILE_NONE_MASK)
      {
         if(gt_hsc.s_InitialMsg == START_WITH_MP)
         {
            pt_info->uc_SI_SPar2_01 |= V2_PROFILES;

            // Select and set highest common profile.
            SelectHighestBit(&pt_info->uca_Profiles[0], &pt_RxCL->uca_Profiles[0], &pt_TxCL->uca_Profiles[0], 2);
            gt_ProfileAct.us_ProfileSelected = ((pt_info->uca_Profiles[1] << 6) | pt_info->uca_Profiles[0]);
         }
         else
         {
            uint32 ul_ExceptionCode;

            // Far end supports at least 1 profile - this didn't match our preferred one, hence retrain
            if (pt_RxCL->uc_SI_SPar2_01 & V2_PROFILES)
            {
               // Far end supports at least 1 profile - this didn't match our preferred one, hence retrain
               ft_ghsRetrain = TRUE;
               ul_ExceptionCode = E_CODE_GHS_RETRAIN;
            }
            else
            {
               // Far end does not support any profile - while this is mandatory, hence declare non-compliance
               ft_ghsStdNoncompliant = TRUE;
               ul_ExceptionCode = E_CODE_GHS_STD_NONCOMPLIANT;
            }

            EnterFailStates(ul_ExceptionCode);
         }
      }
   }


   if (!ft_ghsStdNoncompliant && !ft_ghsRetrain)
   {
      if(pt_info->uc_SI_SPar2_01 & V2_CE_LENGTH)
      {
         int16 s_m_value;

         s_m_value = ((pt_RxCL->uca_CELength[2] << 12) | (pt_RxCL->uca_CELength[1] << 6) | pt_RxCL->uca_CELength[0]);
         s_m_value &= ((pt_TxCL->uca_CELength[2] << 12) | (pt_TxCL->uca_CELength[1] << 6) | pt_TxCL->uca_CELength[0]);
         // m = 5 is the default value, bit4 (0x08 in octet 1)
         // Note: Handshake is counting from 1 onwards!
         s_m_value &= 0x0008;

         if(s_m_value)
         {
            pt_info->uca_CELength[0] = (uint8)0x08;
         }
         else
         {
            // Select and set highest common CELength.
            SelectHighestBit(&pt_info->uca_CELength[0], &pt_RxCL->uca_CELength[0], &pt_TxCL->uca_CELength[0], 3);
         }
      }

      // For now, since most of the far end's do not diligently indicate support of the
      // US0 PSD masks we remove the check that verifies whether our preferred US0 PSD mask
      // is the one that far end has enabled.Implies we also no longer have the retrain functionality
      // provided to the ME incase its US0 PSD mask is not supported by the far end.
      // Note: Assumes the ME sets preference to only US0 PSD mask.
//      if (!ft_CheckForUS0PSDMaskEnabled)
      {
         uint8 uc_CountBits;

         uc_CountBits = 0;
         for (i = 0; i < 3; i++)
         {
            if(pt_info->uc_SI_SPar2_02 & (1<<i))
            {
               uc_CountBits++;
            }
         }

         if(uc_CountBits > 1)
         {
            // Set SPar bit irrespetive of whether far end sends support for it in CLR message
            pt_info->uc_SI_SPar2_02 &= ~(V2_ANNEX_A_US0_MASKS | V2_ANNEX_B_US0_MASKS | V2_ANNEX_C_US0_MASKS);
            if (gt_ProfileControl.ul_AnxAUS0PsdPreferred)
            {
               pt_info->uc_SI_SPar2_02 |= V2_ANNEX_A_US0_MASKS;
            }
            else if (gt_ProfileControl.us_AnxBUS0PsdPreferred)
            {
               pt_info->uc_SI_SPar2_02 |= V2_ANNEX_B_US0_MASKS;
            }
            else if (gt_ProfileControl.us_AnxCUS0PsdPreferred)
            {
               pt_info->uc_SI_SPar2_02 |= V2_ANNEX_C_US0_MASKS;
            }
            else
            {
               pt_info->uc_SI_SPar2_02 |= V2_ANNEX_A_US0_MASKS;
            }
         }
      }

      // Choose the CO preferred Annex A US0 PSD
      if(pt_info->uc_SI_SPar2_02 & V2_ANNEX_A_US0_MASKS)
      {
//         uint32 ul_AnxAUS0Psd;
//         uint32 *pul_AnxAUS0PsdPreferred;
//
//         pul_AnxAUS0PsdPreferred = &gt_ProfileControl.ul_AnxAUS0PsdPreferred;
//
//         // Bits 1-2 of octet 5 shall always be set to ZERO.
//         // Octet 1
//         ul_AnxAUS0Psd = ((*pul_AnxAUS0PsdPreferred & 0x01ff) | ((*pul_AnxAUS0PsdPreferred >> 11) & 0x0200) |
//                          ((*pul_AnxAUS0PsdPreferred & 0x3FE00) << 1) | ((*pul_AnxAUS0PsdPreferred >> 2) & 0x80000));
//         ul_AnxAUS0Psd &= (pt_RxCL->uca_US0_A_bands[0] | ((pt_RxCL->uca_US0_A_bands[1] & 0x0f) << 6) |
//                           (pt_RxCL->uca_US0_A_bands[2] << 10) | ((pt_RxCL->uca_US0_A_bands[1] & 0x0f) << 16));
//
//         if(ul_AnxAUS0Psd)
//         {
//            pt_info->uca_US0_A_bands[0] = (ul_AnxAUS0Psd & 0x3f);
//            pt_info->uca_US0_A_bands[2] = ((ul_AnxAUS0Psd >> 10) & 0x3f);
//            pt_info->uca_US0_A_bands[1] = ((ul_AnxAUS0Psd >> 6) & 0x0f);
//            pt_info->uca_US0_A_bands[3] = ((ul_AnxAUS0Psd >> 16) & 0x0f);
//            gt_ProfileAct.ul_AnxAUS0PsdSelected = (*pul_AnxAUS0PsdPreferred & ~(CNFG_V2_ANXA_US0_IN_12B_MASK|CNFG_V2_ANXA_US0_IN_17A_MASK));
//         }
//         else
         {
            // Select highest AnxAUS0Psd
            SelectHighestBit(&pt_info->uca_US0_A_bands[0], &pt_RxCL->uca_US0_A_bands[0], &pt_TxCL->uca_US0_A_bands[0], 4);
         }
      }

      // Choose the CO preferred Annex B US0 PSD
      if(pt_info->uc_SI_SPar2_02 & V2_ANNEX_B_US0_MASKS)
      {
         uint8 uc_AnxBUS0Psd;

         // Bits 1-2 of octet 2 shall always be set to ZERO.
         uc_AnxBUS0Psd = (uint8)(gt_ProfileControl.us_AnxBUS0PsdPreferred & 0x7);
         uc_AnxBUS0Psd &= pt_RxCL->uca_US0_B_bands[0];
         uc_AnxBUS0Psd &= pt_TxCL->uca_US0_B_bands[0];

         if(uc_AnxBUS0Psd)
         {
            pt_info->uca_US0_B_bands[0] = uc_AnxBUS0Psd;
            gt_ProfileAct.us_AnxBUS0PsdSelected = uc_AnxBUS0Psd;
         }
         else
         {
            // Select and set highest common AnxBUS0Psd.
            SelectHighestBit(&pt_info->uca_US0_B_bands[0], &pt_RxCL->uca_US0_B_bands[0], &pt_TxCL->uca_US0_B_bands[0], 1);
         }
      }

      // Choose the CO preferred Annex C US0 PSD
      if(pt_info->uc_SI_SPar2_02 & V2_ANNEX_C_US0_MASKS)
      {
         uint16 us_AnxCUS0Psd;

         us_AnxCUS0Psd = ((pt_RxCL->uca_US0_C_bands[1] << 4) |
                          (pt_RxCL->uca_US0_C_bands[0] & (V2_US0_C_25_276_B| V2_US0_C_25_138_B)));
         us_AnxCUS0Psd &= ((pt_TxCL->uca_US0_C_bands[1] << 4) |
                           (pt_TxCL->uca_US0_C_bands[0] & (V2_US0_C_25_276_B| V2_US0_C_25_138_B)));
         us_AnxCUS0Psd &= gt_ProfileControl.us_AnxCUS0PsdPreferred;

         if(us_AnxCUS0Psd)
         {
            gt_ProfileAct.us_ProfileSelected = us_AnxCUS0Psd;
            pt_info->uca_US0_C_bands[0] = (us_AnxCUS0Psd & (V2_US0_C_25_276_B| V2_US0_C_25_138_B));
            pt_info->uca_US0_C_bands[1] = ((us_AnxCUS0Psd >> 4) & (V2_US0_C_25_276_CO | V2_US0_C_25_138_CO));
         }
         else
         {
            // Select and set highest common AnxBUS0Psd.
            SelectHighestBit(&pt_info->uca_US0_C_bands[0], &pt_RxCL->uca_US0_C_bands[0], &pt_TxCL->uca_US0_C_bands[0], 2);
         }
      }

//#ifdef MTK_VECTORING_SUPPORT
      // Configure Vectoring
      if(pt_info->uc_SI_SPar2_02 & V2_G_VECTOR_MASKS)
      {
         // Enable Vectoring in MS if CO and CPE both Support it
         pt_info->uc_Gvector = (pt_RxCL->uc_Gvector & pt_TxCL->uc_Gvector);
      }
//#endif
   }




}

/*^^^
 *-------------------------------------------------------------------
 *
 *  Name : SelectMode
 *
 *  Abstract : Selects mode of operation based on exchanged MS message
 *
 *  Parameters:
 *      InfoField_t *pt_MS_Info : information structure to be
 *                          initialized.
 *  Returns: l_SelectedMode
 *
 *  Global Variables Used:
 *      gus_OperationModeStatus_DMT: (O) Selected Mode of operations
 *      gus_OperationModeStatus_BIS: (O) Selected Mode of operations
 *      gus_OperationModeStatus_PLUS: (O) Selected Mode of operations
 *
 *  Notes :
 *
 *-------------------------------------------------------------------
 *^^^
 */

int32 PostGhsConfigVDSL2(VDSL2_Type_t *pt_Info);

int32 SelectMode(InfoField_t *pt_MS_Info)
{
   int32 l_SelectedMode;

   l_SelectedMode = NOT_SELECTED;

// XDSLRTFW-915: : BugFix_All_VDSL2_All_BondingMsSilentBitHandling (start)
   //======== VDSL 2 Mode ==================
   // Note: Force mode if ignore MS debug bit set
#ifdef VDSL_BONDING
   // XDSLRTFW-1695 (start_end)
   if(((pt_MS_Info->uc_SI_NPar1 & SILENT_PERIOD) != 0) &&
         ((pt_MS_Info->uc_ID_SPar1_03 & BONDING) && (pt_MS_Info->uc_BondingNP2 & BONDING_ETHERNET)) &&
         (!(gul_dbgGHSControl & IGNORE_FAREND_MS)))
   {
      l_SelectedMode  = SILENCE;
   }
   else if((pt_MS_Info->uc_SI_SPar1_05 & G993_2) ||
           ((gul_dbgGHSControl & IGNORE_FAREND_MS) && (gsa_Optn0_ModeControl[1] & OPTN_ConfigMode_G993_2)))
#else
   if((pt_MS_Info->uc_SI_SPar1_05 & G993_2) ||
         ((gul_dbgGHSControl & IGNORE_FAREND_MS) && (gsa_Optn0_ModeControl[1] & OPTN_ConfigMode_G993_2)))
#endif
   {
      // Set operation mode status for VDSL2
      gus_ModemOperationMode_Status = MODEM_OPERATION_MODE_VDSL2;
      gsa_Stat1_Mode[1] = STAT_ConfigMode_G993_2;

      // Force mode if ignore MS debug bit set
      if (gul_dbgGHSControl & IGNORE_FAREND_MS)
      {
         l_SelectedMode = G993_2;
      }
//#ifdef VDSL_BONDING
//      else if(((pt_MS_Info->uc_SI_NPar1 & SILENT_PERIOD) != 0) &&
//               ((pt_MS_Info->uc_ID_SPar1_03 & BONDING) && (pt_MS_Info->uc_BondingNP2 & BONDING_ETHERNET)))
//      {
//         l_SelectedMode  = SILENCE;
//      }
//#endif
      else
      {
         l_SelectedMode = PostGhsConfigVDSL2(&(pt_MS_Info->t_VDSL2_Info));
      }
   }

   //======== Silence ==================
   else if((pt_MS_Info->uc_SI_NPar1 & SILENT_PERIOD) != 0)
   {
      l_SelectedMode  = SILENCE;
   }
   else if (pt_MS_Info->uc_SI_SPar1_05 == G9701)
   {
      l_SelectedMode = G9701;
   }
   else
   {
      // XDSLRTFW-1695 / XDSLRTFW-1730(start_end)
      // XDSLRTFW-3104 : Wrong switch to ADSL FW during 35b Bonding testing
      //                 Details: In case of 35b and bonding the FW indicates "E_CODE_GHS_S_XDSL_MODE",
      //                          which leads to a Switch from VDSL to ADSL. Reason is that BDCM sends
      //                          now also a Non-Standard-Field in case of a NoCommonMode message,
      //                          which is not Standard conform. Therefore this check was commented out.
      // Check NoCommonMode MS message
//      if(((pt_MS_Info->uc_ID_NPar1 & NON_STANDARD)== 0) &&
        if ((pt_MS_Info->uc_SI_NPar1 == 0) &&
            (pt_MS_Info->uc_SI_SPar1 == 0) &&
            (pt_MS_Info->uc_SI_SPar1_02 == 0) &&
            (pt_MS_Info->uc_SI_SPar1_03 == 0) &&
            (pt_MS_Info->uc_SI_SPar1_04 == 0) &&
            (pt_MS_Info->uc_SI_SPar1_05 == 0))
      {
         l_SelectedMode = NO_COMMON_MODE;
      }
   }
// XDSLRTFW-915: : BugFix_All_VDSL2_All_BondingMsSilentBitHandling (end)


   return(l_SelectedMode);

}   // SelectMode



// XDSLRTFW-1797 (Start)
int32 PostGhsConfigVDSL2(VDSL2_Type_t *pt_Info)
{
   int32 l_SelectedMode = G993_2;
   int16 s_selectedProfile;

   gus_VectoringOptionsEnabled = 0;
   gt_ProfileAct.us_GvectorSelected = 0;

   // XDSLRTFW-1693 (Start_End)
   gus_ReTxOptionsEnabled = (gt_ReTXParams.t_ReTXConfigCMV.us_DsUsReTxCapable & 0xFF); //XDSLRTFW-2668 (Start_End)

   // Set the operational mode status
   //---------------------------------
   // Note: OperationModeX = [OperationModeCapable, OperationModeStatus]
   // gul_OperationModeX_VDSL2 bits[7:0]   - standard info NPar2 octet 1 for VDSL2
   // gul_OperationModeX_VDSL2 bits[15:8]  - standard info SPar2 octet 1 for VDSL2
   // gul_OperationModeX_VDSL2 bits[23:16] - standard info SPar2 octet 1 for VDSL2
   // gul_OperationModeX_VDSL2 bits[31:24] - standard info NPar2 octet 2 for VDSL2
   {
      uint8 uc_msb, uc_lsb;
      uint8 uc_lsb2;

// XDSLRTFW-1898 (Start_End)
      uc_lsb = (gul_OperationModeCapable_VDSL2 & 0xFF) & (pt_Info->uc_SI_NPar2);
      // XDSLRTFW-487_VR9_VRX318_VDSL2_All_AELEM_Support (START)
      uc_lsb2 = ((gul_OperationModeCapable_VDSL2 >> 24) & 0xFF) & (pt_Info->uc_SI_NPar2_02);
      // XDSLRTFW-487_VR9_VRX318_VDSL2_All_AELEM_Support (END)
      uc_msb = ((gul_OperationModeCapable_VDSL2 >> 8) & 0xFF) & (pt_Info->uc_SI_SPar2_01);

      //the diag. mode is selected by the remote modem, the local modem must go to the diag. mode
      if(pt_Info->uc_SI_NPar2 & V2_LOOP_DIAG)
      {
         uc_lsb |= V2_LOOP_DIAG;
      }

      gul_OperationModeStatus_VDSL2 = (uint32)(((uint32)uc_lsb)|((uint32)uc_lsb2<<24)|((uint32)uc_msb<<8));
   }

   // This information is only available in the CL message, but needed for training too.
   // Since it is related to bandplan and profile, the best place for evaluation is here.
   if(gft_DisableUseUS0Only == 0)
   {
      // Note: Previously this was gs_NumOfEstTxBand! -> mr addr_pointer 290 (reading from end)
      if((gpt_RxInfoSave->t_VDSL2_Info.uc_NumUsBands == 0) ||
         (TESTArray[TEST_JIRA2257_CONFIG] & TEST_JIRA2257_GHS_US0))
      {
         // XDSLRTFW-3627
         // The concept was changed for VRx518, i.e. no configuration after handshake anymore
         // that is taking care about this info.
         // Therefore the flag is used as info "who" and "when" TX US0 only indication was performed.
         gus_UseUS0OnlyCntrl = TX_US0_ONLY_INDICATION_GHS;
      }
   }

   // Build-up the two handshake profile octets into one 16bit variable
   s_selectedProfile = ((pt_Info->uca_Profiles[0]) & ((V2_PROFILE_8D|V2_PROFILE_8C|V2_PROFILE_8B|V2_PROFILE_8A) |
                        (V2_PROFILE_12B|V2_PROFILE_12A)));
   s_selectedProfile |= ((pt_Info->uca_Profiles[1] & (V2_PROFILE_35B|V2_PROFILE_30A|V2_PROFILE_17A)) << 6);

   // The profile bit should always be set in MS message &&
   // at least one band plan should be selected
   if(((pt_Info->uc_SI_SPar2_01 & V2_PROFILES) == 0) || (s_selectedProfile == 0))
   {
      l_SelectedMode = NOT_SELECTED;
   }
   else
   {
      // Extract the CE length from the handshake MS message
      if (pt_Info->uc_SI_SPar2_01 & V2_CE_LENGTH)
      {
         int16 s_m_value, i, s_temp;

         s_m_value = ((pt_Info->uca_CELength[2] << 12) | (pt_Info->uca_CELength[1] << 6) | pt_Info->uca_CELength[0]);

         // Find the bit position set in the octet
         for (i = 0; i < 15; i++)
         {
            s_temp = (s_m_value >> i) & 0x1;

            if (s_temp != 0)
            {
               break;
            }
         }

         // We can't handle any "m" value other than what we support
         // Hence fail in this case
         if ( gs_m != (i + 2))
         {
            l_SelectedMode = NOT_SELECTED;
         }
      }

// XDSLRTFW-1040: Bugfix_All_ALL_ALL_SelectedProfileVDSL2Get (START)
      //---  CMD_HS_SelectedProfileVDSL2Get ---
      //     Message Identifier: 0xCD03
      //     CMV: INFO 205
      //---------------------------------------
      //store the selected profile
      gt_ProfileAct.us_ProfileSelected = s_selectedProfile;

      if (gt_ProfileAct.us_ProfileSelected & (V2_PROFILE_35B << 6))
      {
         // Profile 35b (CNFG_V2_PROFILE_35B_MASK)
         // The bit 5 indicates the use of the extended IDFT Size with profile 35b. If set to
         // ONE, it indicates that the IDFT size to be used with profiles 35b is
         // 2N=2x8192. If set to ZERO, it indicates the profile 35b is not supported.
         // The 4 LSBs indicate the IDFT size to be used with any profile other than 35b,
         // encoded as a number from 7 to 13 representing n, where IDFT sSize 2N = 2^n. If
         // no profile other than 35b is supported, the 4 LSBs shall be set to ZERO.
         if (gs_FeModemLog2IfftSize & V2_INITIAL_IDFT_SIZE_35B)
         {
            gs_FeModemLog2IfftSize = 14;
         }
      }
      else
      {
         // Secure code for CNFG_V2_PROFILE_8xALL_MASK, CNFG_V2_PROFILE_12xALL_MASK
         // CNFG_V2_PROFILE_17A_MASK and CNFG_V2_PROFILE_30A_MASK.
         if(gs_FeModemLog2IfftSize > 13)
         {
            gs_FeModemLog2IfftSize = 13;
         }
      }

      // store the selected Annex A US0 PSD
      gt_ProfileAct.ul_AnxAUS0PsdSelected = ((uint32)pt_Info->uca_US0_A_bands[0] & 0x3f);
      gt_ProfileAct.ul_AnxAUS0PsdSelected |= (((uint32)pt_Info->uca_US0_A_bands[1] & 0x7)<<6);
      gt_ProfileAct.ul_AnxAUS0PsdSelected |= (((uint32)pt_Info->uca_US0_A_bands[2] & 0x3f)<<9);
      gt_ProfileAct.ul_AnxAUS0PsdSelected |= (((uint32)pt_Info->uca_US0_A_bands[3] & 0x7)<<15);
      gt_ProfileAct.ul_AnxAUS0PsdSelected |= (((uint32)pt_Info->uca_US0_A_bands[1] & 0x8)<<20);
      gt_ProfileAct.ul_AnxAUS0PsdSelected |= (((uint32)pt_Info->uca_US0_A_bands[3] & 0x8)<<21);

      // store the selected Annex B US0 PSD
      gt_ProfileAct.us_AnxBUS0PsdSelected = ((uint16)pt_Info->uca_US0_B_bands[0] & 0x7);

      // store the selected Annex C US0 PSD
      gt_ProfileAct.us_AnxCUS0PsdSelected = ((uint16)pt_Info->uca_US0_C_bands[0] & 0x3);
      gt_ProfileAct.us_AnxCUS0PsdSelected |= ((uint16)pt_Info->uca_US0_C_bands[1] & 0x3)<<4;

      // store the selected vectoring (G.993.5) modes and options
      {
         uint8 uc_GvectorTemp;

         // Evaluate vectoring
         uc_GvectorTemp = (pt_Info->uc_Gvector & (V2_OP_VEC1_FLAG_TONES_MASK |
                           V2_OP_VEC1_DURATION_EXT_MASK |
                           V2_FDPS_US_MASK |
                           V2_PILOT_SEQ_LEN_MULTI4_MASK |
                           V2_GVECTOR_US_MASK |
                           V2_GVECTOR_DS_MASK));

         if((uc_GvectorTemp & (V2_GVECTOR_US_MASK|V2_GVECTOR_DS_MASK)) == (V2_GVECTOR_US_MASK|V2_GVECTOR_DS_MASK))
         {
            gt_ProfileAct.us_GvectorSelected = 0x01;
         }
         else if(uc_GvectorTemp & V2_GVECTOR_DS_MASK)
         {
            gt_ProfileAct.us_GvectorSelected = 0x02;
         }
         gus_VectoringOptionsEnabled |= (uc_GvectorTemp & (V2_GVECTOR_US_MASK|V2_GVECTOR_DS_MASK));

         // Evaluate vectoring options
         uc_GvectorTemp = ((uc_GvectorTemp & (V2_OP_VEC1_FLAG_TONES_MASK |
                                              V2_OP_VEC1_DURATION_EXT_MASK |
                                              V2_FDPS_US_MASK |
                                              V2_PILOT_SEQ_LEN_MULTI4_MASK)) << 2); // shift by two for the friendly modes

         gus_VectoringOptionsEnabled |= uc_GvectorTemp;
         gt_ProfileAct.us_GvectorSelected |= (uc_GvectorTemp << 4);

         // Evaluate vectoring friendly
         // Note: CLR - If V2_VECTOR_FRIENDLY set to ONE, the G.993.5 SPar(2) bit shall be set to ZERO.
         //       MS  - If V2_VECTOR_FULL_FRIENDLY set to ONE, the "G.993.5-friendly G.993.2 operation
         //             in the downstream direction" (see Annex X) bit shall be set to ZERO.
         //       Therefore both bits set one is not allowed. But to have a smart implementation the link will
         //       be done with VECTOR_FULL_FRIENDLY.
         gt_ProfileAct.us_GvectorSelected |= ((pt_Info->uc_SI_NPar2_02 & V2_VECTOR_FRIENDLY) << 3);
         gt_ProfileAct.us_GvectorSelected |= (pt_Info->uc_SI_NPar2_02 & V2_VECTOR_FULL_FRIENDLY);

         if((pt_Info->uc_SI_NPar2_02 & (V2_VECTOR_FULL_FRIENDLY|V2_VECTOR_FRIENDLY) == V2_VECTOR_FRIENDLY))
         {
            // V2_VECTOR_FRIENDLY mode is not supported at all by firmware -> a fail must occur!
            l_SelectedMode = NOT_SELECTED;
         }
         else if(pt_Info->uc_SI_NPar2_02 & V2_VECTOR_FULL_FRIENDLY)
         {
            if(gul_OperationModeCapable_VDSL2 & (V2_VECTOR_FULL_FRIENDLY << 24))
            {
               gus_VectoringOptionsEnabled |= VEC_OPTIONS_FULL_FRIENDLY_MASK;

               //XDSLRTFW-2874 (Start)
               if (pt_Info->uc_SI_NPar2_02 & V2_VECTOR_FULL_FRIENDLY_PILOT_SEQ_LEN_MULTI4_MASK)
               {
                  gus_VectoringOptionsEnabled |= VEC_OPTIONS_PILOT_SEQ_LEN_MULTI4_MASK;
                  gt_ProfileAct.us_GvectorSelected |= (VEC_OPTIONS_PILOT_SEQ_LEN_MULTI4_MASK << 4);
               }

               if (pt_Info->uc_SI_NPar2_02 & V2_VECTOR_FULL_FRIENDLY_FDPS_US_MASK)
               {
                  gus_VectoringOptionsEnabled |= VEC_OPTIONS_FDPS_US_MASK;
                  gt_ProfileAct.us_GvectorSelected |= (VEC_OPTIONS_FDPS_US_MASK << 4);
               }
               //XDSLRTFW-2874 (End)
            }
            else
            {
               // V2_VECTOR_FULL_FRIENDLY mode is not supported, i.e. configured!
               l_SelectedMode = NOT_SELECTED;
            }
         }
// XDSLRTFW-1040: Bugfix_All_ALL_ALL_SelectedProfileVDSL2Get (end)


         // Note: This code part has to be executed for DS, DS+US and full friendly vectoring!
         //       US only is a not allowed configuration.
         //       Therefore the check on DS vectoring covers all cases!
         if (gus_VectoringOptionsEnabled & (VEC_OPTIONS_FULL_FRIENDLY_MASK | VEC_OPTIONS_DS_MASK))
         {
            gft_DSVectoringEnabled = TRUE;

            // XDSLRTFW-2456 Long_train_time_in_VDSL_when_switching_from_ADSL_to_VDSL_Vectoring
            // Set TC mode hardcoded to PTM if VDSL is started in Vectoring mode
            // CMD_TC_StatusGet
            // Reports the used TC-Mode.
            // Supported settings:
            //    0x0 UNKNOWN_TC TC mode is unknown.
            //    0x1 EFM_TC EFM TC
            //    0x2 ATM_TC ATM TC
            gus_DSL_TC_Status = TPS_TC_TYPE_PTM; // CMD_TC_StatusGet / DSL_TC_STATUS / DSL 14

            gft_FastLosEnabled = FALSE;
            if(TESTArray[TEST_XDSLRTFW_CONFIG] & TEST_XDSLRTFW_CONFIG_FLOS)
            {
               gft_FastLosEnabled = TRUE;
            }

            if (TESTArray[TEST_XDSLRTFW_CONFIG] & TEST_XDSLRTFW_CONFIG_XTALK_CONNECT)
            {
               // Feature is only needed when US vectoring is enabled!
               if((gpt_RxInfo->uc_ID_SPar1 & TRANSCEIVER_ID) && (gpt_TxInfoSave->uc_ID_SPar1 & TRANSCEIVER_ID) &&
                     (gus_VectoringOptionsEnabled & (VEC_OPTIONS_FULL_FRIENDLY_MASK | V2_GVECTOR_US_MASK)))
               {
                  gus_VectoringOptionsEnabled |= VEC_OPTIONS_TRANSCEIVER_ID_MASK;
               }
            }

            // XDSLRTFW-1242: Bugfix_ALL_VDSL2_ALL_CleanupGlobalVectoringTimeouts (Start)
            // XDSLRTFW-1872 (Start_End)
            // Set state timeout for OP-Vector1
            if (gus_VectoringOptionsEnabled & VEC_OPTIONS_OP_VEC1_DURATION_EXT_MASK)
            {
               gl_TxStateTimeOutCount = (OP_VECTOR1_STATE_TIMEOUT_LEN + O_P_VECTOR1_MIN_LENGTH);
               gl_RxStateTimeOutCount = OP_VECTOR1_STATE_TIMEOUT_LEN;
            }
            else
            {
               gl_TxStateTimeOutCount = VECTORING_STATE_TIMEOUT_LEN;
               gl_RxStateTimeOutCount = VECTORING_STATE_TIMEOUT_LEN;
            }
            // XDSLRTFW-1242: Bugfix_ALL_VDSL2_ALL_CleanupGlobalVectoringTimeouts (End)

            // DSM_Vectoring_Check: Our Vectoring CO doesn't send the correct periodic signal as of now.
            // Hence don't calc SNR in periodic phase as of now.
//            gs_PeriodicSnrCalcEnableFlag = FALSE;
            //XDSLRTFW-1621 DeltInVectoring (Start)
            if (gul_OperationModeStatus_VDSL2 & V2_LOOP_DIAG)
            {
               gft_DeltInVectoringMode = 1;
            }
            //XDSLRTFW-1621 DeltInVectoring (End)
         }
         // XDSLRTFW-1182: Bugfix_US_VDSL2_ALL_LowUsRate (Start)
         else // for non Vectoring
         {
          //XDSLRTFW-3833 XDSLRTFW-735 IOP_A_US_VDSL2_EVLT_K_BRCM_USdatarate (START)
          //XDSLRTFW-3833 XDSLRTFW-735 IOP_A_US_VDSL2_VINAX_USdatarate (START)
            if ( (gt_ProfileAct.us_ProfileSelected & CNFG_V2_PROFILE_35B_MASK) ||
               ((gt_Dsl_Operator_Select.us_Operator_Select != AT_T) && (gt_Dsl_Operator_Select.us_Operator_Select != CTL)) ||
               ((gul_fe_G994VendorID == IFX_VENDOR_ID)) ) // for non Vectoring Exclude all COs except IFX at CTL/AT&T
          {
            // Change Slow and fast PLL settings to default VDSL2 values.
            gs_NumFramesFastPLLStabilization_ChDiscV1  = PLL_STABILIZATION_LEN_650;
            gs_NumFramesSlowPLLStabilization_ChDiscV1  = PLL_STABILIZATION_LEN_DIS;
            gs_NumFramesFastPLLStabilization_ChDisc1   = PLL_STABILIZATION_LEN_650;
//            gs_NumFramesSlowPLLStabilization_ChDisc1   = PLL_STABILIZATION_LEN_DIS;

            gs_NumFramesFastPLLStabilization_TrainV1   = PLL_STABILIZATION_LEN_650;
            gs_NumFramesSlowPLLStabilization_TrainV1   = PLL_STABILIZATION_LEN_DIS;
            gs_NumFramesFastPLLStabilization_Train1    = PLL_STABILIZATION_LEN_650;
            // XDSLRTFW-3856 - Commented the code out!
            //                 Now the PLL slow stabilization is always used. This shows an positive effect on upstream performance
            //                 at least against BDCM.
            //                 The variable "gs_NumFramesSlowPLLStabilization_Train1" is initialized to "PLL_STABILIZATION_LEN_300".
            //                 Therefore the disabling at this place must be removed to get a common code for all loop length and to remove
            //                 the code for XDSLRTFW-2190/XDSLRTFW-2270 under "case R_O_P_TRAIN_RX_PGA_ECHO:" in transceiver training.
            //                 Reason why it was not done from scratch, is not clear anymore. But there was once a no link problem
            //                 due to a train time issue, which is not available any more due to the bigger RTV and faster SNR/PSD
            //                 measurement.
//            gs_NumFramesSlowPLLStabilization_Train1    = PLL_STABILIZATION_LEN_DIS;

            // Change Ref Pilot tone reset (tracking) to the default VDSL2 handling, i.e. Vectoring is skipping
            // the reset in different phases. These bits are located in the high byte!
            // XDSLRTFW-1181: BugFix_DS_VDSL2_All_VecDSRateRunToRunVariation (Start)
            // Change Hybird training to default VDSL2 training, i.e. test all available hybrid settings +
            // cap search.
            // XDSLRTFW-1488 (Start_End)
            {
               // Note: Some defines are Vectoring only and has no influence for VDSL2 at all!
               //       - TEST_XDSLRTFW_CONFIG_SKIP_OPVEC1_LOSS
               //       - TEST_XDSLRTFW_CONFIG_FLOS
               //
               TESTArray[TEST_XDSLRTFW_CONFIG] &= (TEST_XDSLRTFW_CONFIG_VEC_HYBRID_ECHO | TEST_XDSLRTFW_CONFIG_VEC_HYBRID |
                                                   TEST_XDSLRTFW_CONFIG_XTALK_CONNECT);
//               TESTArray[TEST_XDSLRTFW_CONFIG] &= (TEST_XDSLRTFW_CONFIG_FLOS);
            }
            // XDSLRTFW-1181: BugFix_DS_VDSL2_All_VecDSRateRunToRunVariation (End)
          }
          //XDSLRTFW-3833 XDSLRTFW-735 IOP_A_US_VDSL2_EVLT_K_BRCM_USdatarate (END)
          //XDSLRTFW-3833 XDSLRTFW-735 IOP_A_US_VDSL2_VINAX_USdatarate (END)

            // Do not transmit additional ReTx parameters in R-MSG2
            //XDSLRTFW-1441(START_END)
            //gul_ItlvMemSizeReTxRMsg2 = 0;//commented as the paramenter is sent always even in non-vectoring mode if amend2 is enabled
         }
         // XDSLRTFW-1182: Bugfix_US_VDSL2_ALL_LowUsRate (End)

         // XDSLRTFW-1181: BugFix_DS_VDSL2_All_VecDSRateRunToRunVariation (Start/End)
         if(!(TESTArray[TEST_XDSLRTFW_CONFIG] & TEST_XDSLRTFW_CONFIG_VEC_HYBRID))
         {
            guc_HybridAlgoCtrl = HYB_ALGO_SNR_BASED;
         }
         else
         {
            // If bit is not set the new vectoring hybrid-algo (Echo based) gets disabled.
            // Note: This code can stay common so that it can be used for VDSL2 too!
            if(!(TESTArray[TEST_XDSLRTFW_CONFIG] & TEST_XDSLRTFW_CONFIG_VEC_HYBRID_ECHO))
            {
               guc_HybridAlgoCtrl = HYB_ALGO_GAIN_BASED;
            }
         }
      }

      // XDSLRTFW-2668 (Start)
      // XDSLRTFW-1693 (Start_End)
      // G.998.4 Extensions per Amd3 or explicitly for G.993.5 Vectoring
      // Note: Operation according to this Annex D can be activated in 3 ways:
      //       - If ITU-T G.993.5 vectoring is selected in at least one direction
      //         (as indicated in the ITU-T G.994.1 MS message), then operation of ITU-T G.998.4
      //         shall comply with this Annex D.
      //       - If the ITU-T G.998.4 extension "G.998.4 Annex D support" NPar(3) is set to ONE (see Table
      //         11.68.11 of ITU-T G.994.1), then operation of ITU-T G.998.4 shall comply with this Annex D.
      //       - If explicitly forced by API -> to be backward compatible with previous releases where extended
      //         memory was used to get better rates in both G.993.2 and G.993.5 mode of operation
      // Note: Avinax has following implementation:
      //       With FS5.1 the performance is based on that also in the non-vectoring case the MDO-bytes
      //       are added to the R-MSG2, although no Amd3 is implemented on the CO side.
      //       Same is done for the FS5.2, although the Amd3 is implemented. At the end the Amd3 is don't care
      //       against LTQ CPE.
      //       To be backwards compatible, the LTQ CPE must send always the MDO-bytes against our LTQ CO!
      if ((pt_Info->uc_G9984_Extensions & V2_G9984_ANNEXD_SUPPORT) ||
          (gus_VectoringOptionsEnabled & (VEC_OPTIONS_US_MASK | VEC_OPTIONS_DS_MASK)) ||
          (gt_ReTXParams.t_ReTXConfigCMV.us_DsUsReTxCapable & CNFG_RETX_EXTENDED_MEMORY_FORCE_SOC) ||
          (gul_fe_G994VendorID == IFX_VENDOR_ID))
      {
         gus_ReTxOptionsEnabled |= (G9984_ANNEXD_SUPPORT);
      }
      // XDSLRTFW-2668 (End)
   }

//XDSLRTFW-439: Feature_All_DS_All_All_SupportDsTxRefVirtualNoise (Start)
   // Set the DS virtual noise status after handshaking. For CO, it's final. CPE still needs to check the O-Signature message.
   gft_DSVirtualNoiseStatus = 0;
   if (gul_OperationModeStatus_VDSL2 & V2_VIRTUAL_NOISE_DS)
   {
      gft_DSVirtualNoiseStatus = 1;
   }
//XDSLRTFW-439: Feature_All_DS_All_All_SupportDsTxRefVirtualNoise (End)

   // Check if the MS message contains the support of RX tones > 6000.
   // Note: CO is including a non-standard information block in its CL and MS to indicate support of RX tones > 6000.
   if (gul_35bLiteConfig & EN_BDCM_DS_TONES_GT_6000)
   {
      // Reset the indication, because it must be also indicated in the MS msg.
      gul_35bLiteConfig &= ~(EN_BDCM_DS_TONES_GT_6000);

      // Check if a Non-Standard-Information Field was received with the MS!
      if ((gt_ProfileAct.us_ProfileSelected & (V2_PROFILE_35B << 6)) && ((gpt_RxInfo->uc_ID_NPar1 & NON_STANDARD) != 0))
      {
         if (gpt_RxInfo->ul_Provider_NS == BRCM_VENDOR_ID)
         {
            NonStandardBlock_t *pt_NSB;
            uint8 *pData;

            pt_NSB = &(gpt_RxInfo->ta_NS_Info[0]);

            if (pt_NSB->uc_NSLen == 0x08)
            {
               pData = &pt_NSB->uca_NSVendorSpec[0];
               // No tone ordering nor 1-bit allocation constraints apply:
               //    - NSIF block with NSIF ID = 0x6, defined above, is absent
               //    - NSIF value in the non-standard information block has its LSB set to 0
               if((*pData++ == 0x06) && (*pData & 0x01))
               {
                  gul_35bLiteConfig |= EN_BDCM_DS_TONES_GT_6000;
               }
            }
         }
      }
   }

   // This code shall be located at the end of the function, so that the host message
   // gets still be filled!
   if (gt_PortModeControl.uc_DualPortModeCurrent == DUAL_PORT_MODE)
   {
      // In dual port mode, if the selected profile is not an 8x profile,
      // fail with no selected profile.
      if ((s_selectedProfile & CNFG_V2_PROFILE_8xALL_MASK) == 0)
      {
         return(NOT_SUPPORTED_IN_DUALPORT);
      }

      // If VR9 CPE supports bonding but CO does not bond, then fail with
      // error code in order to switch to single port mode
      if ((gft_SinglePort_if_CO_notBonding) && ((gt_Bonding_DiscAggr_Status.s_PAF_status & 0x1) == 0))
      {
         return(CO_NOT_BONDABLE);
      }
   }

   return(l_SelectedMode);
}
// XDSLRTFW-1797 (End)

/*^^^
 *-------------------------------------------------------------------
 *
 *  Name : Do_ClearDown
 *
 *  Abstract : fill transmit buffer with FLAGS and GALFS
 *
 *  Returns: None
 *
 *  Global Variables Used:
 *          gpuca_TxMsgBuf   - (O) transmit buffer
 *          gs_TxMsgBufCnt  - (O) total number of xmitting octets
 *
 *  Notes :
 *
 *-------------------------------------------------------------------
 *^^^
 */
void Do_ClearDown(void)
{
   int i;

   /* ---- form message array ---- */
   for ( i = 0; i < gs_num_cleardown_flags; i++ )   /*  xmitting FLAGs */
   {
      gpuca_TxMsgBuf[i] = FLAG;
   }
   for ( i = 0; i < NUM_CLEARDOWN_GALF; i++ )   /*  xmitting GALFs */
   {
      gpuca_TxMsgBuf[i+gs_num_cleardown_flags] = GALF;
   }

   gs_TxMsgBufCnt = gs_num_cleardown_flags + NUM_CLEARDOWN_GALF;

   gt_hsc.s_NextState    = R_HS_EXIT;   /*  indicate end of HS message exchange */

}

/*^^^
 *-------------------------------------------------------------------
 *
 *  Name : HndshkTimeout
 *
 *  Abstract : fill transmit buffer with FLAGS and GALFS
 *
 *  Returns: None
 *
 *  Global Variables Used:
 *
 *  Notes :
 *
 *-------------------------------------------------------------------
 *^^^
 */
void HndshkTimeout(void)
{
   if (!(gul_dbgSnifferControl & SNIFFER_DISABLE_GHSTIMEOUTCHECK))
   {
      if (gft_GotoFailStateOnTimeOut_FromHandshake == FALSE)
      {
         if (gs_RxState != R_C_HS_MSG_RX)
         {
            TryAnotherToneSet();
         }
         gs_RxNextState = R_C_SILENT1_RX;
         gpF_RxStateFunc = (PtrToFunc)RCSilent1RxF;

         /* ---- if timeout or tone loss in the Rx side, Tx also go back to the beginning ---- */
         gs_TxNextState = R_SILENT0_TX;
         gpF_TxStateFunc = (PtrToFunc)RSilent0TxF;
      }
      else
      {
         EnterFailStates(E_CODE_GHS_TONE_LOSS_OR_TIME_OUT);
      }
   }
}
