/* **COPYRIGHT******************************************************************
    INTEL CONFIDENTIAL
    Copyright (C) 2017 Intel Corporation
    Copyright (C), 1994-2006 Aware Inc. All Rights Reserved.
******************************************************************COPYRIGHT** */
/* **DISCLAIMER*****************************************************************
    The source code contained or described herein and all documents related
    to the source code ("Material") are owned by Intel Corporation or its
    suppliers or licensors. Title to the Material remains with Intel
    Corporation or its suppliers and licensors. The Material may contain
    trade secrets and proprietary and confidential information of Intel
    Corporation and its suppliers and licensors, and is protected by
    worldwide copyright and trade secret laws and treaty provisions. No part
    of the Material may be used, copied, reproduced, modified, published,
    uploaded, posted, transmitted, distributed, or disclosed in any way
    without Intel's prior express written permission.

    No license under any patent, copyright, trade secret or other
    intellectual property right is granted to or conferred upon you by
    disclosure or delivery of the Materials, either expressly, by
    implication, inducement, estoppel or otherwise. Any license under
    such intellectual property rights must be express and approved by
    Intel in writing.
*****************************************************************DISCLAIMER** */
/****************************************************************************
;   Aware DMT Technology. Proprietary and Confidential.
;
;   40 Middlesex Turnpike, Bedford, MA 01730-1413
;   Phone (781) 276 - 4000
;   Fax   (781) 276 - 4001
;
;
;   File Name: eoc_tx.c
;
;   Functions for implementing VDSL embedded operations channel (EOC).
;
;***************************************************************************/
// ***********************************************************************************************************
// ovhd_tx.c
//
// History
//
//
//  10/09/2013 Sooraj: Added Support of DS-SRA with ReTx
//                 Grep for XDSLRTFW-1032_New_framing_for_SRA_in_DS-ReTx_mode
//                 Grep for XDSLRTFW-1031_OLR Type-5 request for DS-SRA in DS-ReTx mode
// 07/08/2014  Fuss : OHC stuck for segmented Rx testparameter Blk-Read (seen against NVLT-C)
//                    Grep for XDSLRTFW-1929
// 26/11/2015 Anantha Ramu: Added modifications for SRA with DS Intra DTU Interleaving.
//                          Grep for "XDSLRTFW-2162".
//
// 13/09/2017  Hanyu:  Ported Bhadra's code to fix the DS 0-dB margin reporting issue against EVLT-F/CNXT.
//                     Added the CMV control of operator select AT_T and CTL with CNXT CO vendor ID.
//                     Grep for XDSLRTFW-3470 XDSLRTFW-2724 IOP_A_DS_VDSL2_EVLT_F_CNXT_0dBMarginReport
//
// 24/08/2018 Stefan/Sriram : XDSLRTFW-3932-Setting of OLR-counters may result in unpredictable memory corruption
// Due to missing Input Validation in function Update_OLRPMCnt it might happen that calling this function with a
// non-valid Input may result in a unexpected and unpredictable Memory corruption.
// Grep for  XDSLRTFW-3932
// ************************************************************************************************************

#include <string.h>
#include "common.h"
#include "OvhdMsg_IOf.h"
#include "eoc.h"
#include "HDLC.h"
#include "fcs.h"
#include "cmv.h"
#include "gdata.h"
#include "vdsl_xception.h"
#include "dsp_op.h"
#include "cmv_Data.h"
#include "DebugBuffer.h"


int16 FillBinsInfo(uint8 *puca_msg_buffer, uint8 uc_priority, uint8 uc_message_type);
extern ReTX_ShowtimeParams_t gt_ReTxShowtimeParams;


//*************************************************
//
#ifdef INCLUDE_NON_CACHE_FUNCTIONS
//
//*************************************************



void Poll_ClearEOCAck(void);

/*^^^
 *------------------------------------------------------------------------------
 *
 *  Name : void CheckEocTimeOut(void)
 *
 *  Description: This subroutine checks of there is a HLC message timeout
 *
 *  Prototype:
 *     void CheckEocTimeOut(void)
 *
 *
 *  Output Arguments:
 *
 *  Global Variables Used:
 *
 *
 *------------------------------------------------------------------------
 *^^^
 */

void CheckEocTimeOut(void)
{
   int32 l_PriorityIndex;
   TxEocControl_t *pt_TxEocControlInfo;

   for (l_PriorityIndex = 0; l_PriorityIndex < MAX_OVHD_PRIORITY; l_PriorityIndex++)
   {
      pt_TxEocControlInfo = &gta_TxEocControlInfo[l_PriorityIndex];

      if (pt_TxEocControlInfo->ft_ack_expected)
      {
         // For priority 0 message, timeout after 400ms
         // For priority 1 message, timeout after 800ms
         // For priority 2 message, timeout after 1s
         OvhdMsg_Timeout_Handler(l_PriorityIndex);
      }
   }
}


//*************************************************
//
#endif // #ifdef INCLUDE_NON_CACHE_FUNCTIONS
//
//*************************************************







//*************************************************
//
#ifdef INCLUDE_CACHE_FUNCTIONS
//
//*************************************************


void Poll_ClearEOCAck(void);

/*^^^
 *------------------------------------------------------------------------
 *
 *  Name : TxOvhdProcessor(void)
 *
 *  Description:  This subroutine computes the EOC byte to be transmitted and
 *              stores it in a SW fifo.
 *
 *  Prototype:
 *      void  TxOvhdProcessor(void)
 *
 *  Input Arguments:
 *      None
 *
 *  Output Arguments:
 *
 *  Return:
 *      None
 *
 *  Global Variables Used:
 *         gt_TxEocMsgFifo
 *
 *  Notes:
 *
 *------------------------------------------------------------------------
 *^^^
 */
#define MSG_START 2  // msg starts after address and control byte

void TxOvhdProcessor(void)
{
   uint8 uc_new_or_old_bit;
   int32 l_PriorityIndex;
   TxOvhdMsgInfoStruct_t t_TxOvhdMsgInfo;    // will contain the info about the message to be generated
   TxEocControl_t *pt_TxEocControlInfo=NULL;

   // Initialize it to invalid number
   uint8 uc_comm_or_resp_bit = 2;

   uint16 us_tx_eoc_msg_length = 0;

   int i;


   gt_TxEocMsg.gus_FCS = 0xFFFF;
   // XDSLRTFW-650 / XDSLRTFW-1384 (Start_End)
   PM_L3_State_Machine();

   // Form the message and fill the FIFO only if last message has been transmitted
   // Only one ovhd message can be sent at any time. We currently can't handle the complicated scenario when the
   // higher priority messeges comes in before the lower priority message completes its transmission yet.

   if (gt_TxEocMsgFifo.IndexOfOldest == gt_TxEocMsgFifo.NextAvailable)
   {

      // Check if Ack needs to be sent for the received clear EOC
      // message from the other side
      Poll_ClearEOCAck();

      // Check for timeouts
      // Mei: move the following check to the foreground process
      //CheckEocTimeOut();

      // Form OVHD MESSAGES
      // Only one ovhd message can be sent, the higher ovhd msg send first. We currently can't handle
      // complicated higher priority messeges coming when lower priority message is still transmitting yet.
      for (l_PriorityIndex = 0; l_PriorityIndex < MAX_OVHD_PRIORITY; l_PriorityIndex++)
      {
         // check any message needs to be sent
         if (TxEocNextSendMessage(l_PriorityIndex, &t_TxOvhdMsgInfo) == TRUE)
         {

            // IMPORTANT
            //
            // As per spec, an ATU-R should not initiate an OLR if it has transmitted
            // an L2 grant and is waiting for a sync indicator

            // As the gt_RxOLRPMVars.uc_rxOLRPMState = OLR_RECV_SYNC_INDICATOR
            // the case when the ATU-R is expecting a sync indicator from the other end
            // for a L0 to L2 transsition : state being L2_RECV_SYNC_INDICATOR is
            // automatically ruled out

            // Initialize EOC msg buffer each time when we try to form a new message
            memset(guca_tx_eoc_msg_buffer, 0, MAX_TX_EOC_BUFFER_SIZE);

            // Process overhead message based on its priority category
            switch(l_PriorityIndex)
            {

            case OVHD_HIGH_PRIORITY:
               uc_comm_or_resp_bit = Tx_prior0_ovhd_msg_handler(&guca_tx_eoc_msg_buffer[MSG_START], &us_tx_eoc_msg_length, &t_TxOvhdMsgInfo);
               us_tx_eoc_msg_length += MSG_START;
               break;

            case OVHD_NORMAL_PRIORITY:
               uc_comm_or_resp_bit = Tx_prior1_ovhd_msg_handler(&guca_tx_eoc_msg_buffer[MSG_START], &us_tx_eoc_msg_length, &t_TxOvhdMsgInfo);
               us_tx_eoc_msg_length += MSG_START;
               break;

            case OVHD_LOW_PRIORITY:
               uc_comm_or_resp_bit = Tx_prior2_ovhd_msg_handler(&guca_tx_eoc_msg_buffer[MSG_START], &us_tx_eoc_msg_length, &t_TxOvhdMsgInfo);
               us_tx_eoc_msg_length += MSG_START;
               break;
            }

            //Note us_tx_eoc_msg_length excludes the address byte, control byte, but not 2 FCS bytes
            EocTrail(EOC_TRAIL_TXDELIM, guca_tx_eoc_msg_buffer[MSG_START], guca_tx_eoc_msg_buffer[MSG_START+1], us_tx_eoc_msg_length);

            // from now on, this pointer points to the priority that has a pending message to be sent
            pt_TxEocControlInfo = &gta_TxEocControlInfo[l_PriorityIndex];

            // Guarantee that we always transmit the higher priority message first
            break;

         } // TxEocNextSendMessage

      } // for

      /********************************/
      /* TRANSMIT MESSAGE            */
      /********************************/
      if (uc_comm_or_resp_bit < 2)
      {

         // Toggle the LSB for every new message sent
         uc_new_or_old_bit = gt_TxEocMsg.cntrl_field & 0x1;
         uc_new_or_old_bit = (~uc_new_or_old_bit) & 0x1;

         // Fill in the message command/response field
         gt_TxEocMsg.cntrl_field = (uc_comm_or_resp_bit << 1);

         // Fill the message priority based on the command designator
         // BUG: we should add case for  EOC_CMD_G9935_ERRORFB_DESIG here for high prio // solved
         if ((guca_tx_eoc_msg_buffer[MSG_START] == RECONFIG_CMD_DESIG) || (guca_tx_eoc_msg_buffer[MSG_START] == EOC_CMD_G9935_ERRORFB_DESIG))
         {
            gt_TxEocMsg.message_priority = OVHD_HIGH_PRIORITY;
         }
         else if ( (guca_tx_eoc_msg_buffer[MSG_START] == PMD_TEST_PAR_RD_CMD_DESIG)
                   || (guca_tx_eoc_msg_buffer[MSG_START] == NSF_FACILITY_LOW_CMD_DESIG) )
         {
            gt_TxEocMsg.message_priority = OVHD_LOW_PRIORITY;
         }
         else
         {
            // commands EOC_CMD_DESIG, TIME_CMD_DESIG, INVENTORY_CMD_DESIG, CNTL_PAR_RD_CMD_DESIG, MGMT_CNTR_RD_CMD_DESIG
            // POWER_MGMT_CMD_DESIG, CLEAR_EOC_CMD_DESIG,NSF_FACILITY_CMD_DESIG, EOC_CMD_G9935_UPDATE_PILOTSEQ_DESIG have normal priority
            gt_TxEocMsg.message_priority = OVHD_NORMAL_PRIORITY;
         }

         // considering possible ovhd msg segmentation
         if (gsa_Tx_OvhdMsgSegIndx[gt_TxEocMsg.message_priority] >= 0)
         {
            gt_TxEocMsg.cntrl_field |= (gsa_Tx_OvhdMsgSegIndx[gt_TxEocMsg.message_priority])<<3;

            if (gfta_FirstOvhdMsgSeg[gt_TxEocMsg.message_priority])
            {
               // first segmented message
               gt_TxEocMsg.cntrl_field |= (1<<7);
               gfta_FirstOvhdMsgSeg[gt_TxEocMsg.message_priority] = FALSE;
            }
            pt_TxEocControlInfo->lRxSymCnt = gl_RxSymbolCount; // time stamp
            pt_TxEocControlInfo->us_param_rd_carrier_index = t_TxOvhdMsgInfo.us_param_rd_carrier_index;
            pt_TxEocControlInfo->us_param_rd_start_carrier_index = t_TxOvhdMsgInfo.us_param_rd_start_carrier_index;
            pt_TxEocControlInfo->us_param_rd_stop_carrier_index = t_TxOvhdMsgInfo.us_param_rd_stop_carrier_index;
            pt_TxEocControlInfo->ft_MECmd = t_TxOvhdMsgInfo.ft_MECmd;
            // we expect Segment ACK or Inverted Sync symbol
            pt_TxEocControlInfo->ft_ack_expected = TRUE;

            if (gsa_Tx_OvhdMsgSegIndx[gt_TxEocMsg.message_priority] == 0)
            {
               // last segmented message
               gt_TxEocMsg.cntrl_field |= (1<<7);

               //reset the ovhd segment info for the new msg
               gsa_Tx_OvhdMsgSegIndx[gt_TxEocMsg.message_priority] = -1;
               gsa_Rx_OvhdMsgSegIndx[gt_TxEocMsg.message_priority] = -1;
               gfta_FirstOvhdMsgSeg[gt_TxEocMsg.message_priority] = FALSE;

               // for Priority 2, we shouldn't wait for ACK for the last segmented msg
               if (gt_TxEocMsg.message_priority == 2)
               {
                  pt_TxEocControlInfo->ft_ack_expected = FALSE;
               }
            }
         }
         else
         {
            // Fill in the new or old message bit field
            gt_TxEocMsg.cntrl_field |= uc_new_or_old_bit;
         }

         // protect EOC Tx msg counter from overflow
         if (gus_EocTxIndex == 0xFFFF)
         {
            gus_EocTxIndex = 0;
         }
         else
         {
            gus_EocTxIndex++;
         }

         gs_Prev_TxOvhdMsg_length = 4; // accounts for opening HDLC_FLAG, priority, ctrl_field, closing HDLC_FLAG

         // Add the message to gt_TxEocMsgFifo
         AddMessageToOvhdFifo(&gt_TxEocMsgFifo, HDLC_FLAG);

         AddMessageToOvhdFifo(&gt_TxEocMsgFifo, gt_TxEocMsg.message_priority);
         gt_TxEocMsg.gus_FCS = calcCrc16(gt_TxEocMsg.gus_FCS, gt_TxEocMsg.message_priority);
         guca_tx_eoc_msg_buffer[0] = gt_TxEocMsg.message_priority;

         AddMessageToOvhdFifo(&gt_TxEocMsgFifo, gt_TxEocMsg.cntrl_field);
         gt_TxEocMsg.gus_FCS = calcCrc16(gt_TxEocMsg.gus_FCS, gt_TxEocMsg.cntrl_field);
         guca_tx_eoc_msg_buffer[1] = gt_TxEocMsg.cntrl_field;

         // Compute the FCS
         for (i=MSG_START; i< us_tx_eoc_msg_length; i++)
         {
            // Compute FCS before adding the escape octets
            gt_TxEocMsg.gus_FCS = calcCrc16(gt_TxEocMsg.gus_FCS, guca_tx_eoc_msg_buffer[i]);
         }

         gt_TxEocMsg.gus_FCS = ~gt_TxEocMsg.gus_FCS;   //  take 1's compliment

         // Add the FCS computed to the message buffer
         guca_tx_eoc_msg_buffer[us_tx_eoc_msg_length] = (uint8)(gt_TxEocMsg.gus_FCS & 0xFF);// FCS low octet
         guca_tx_eoc_msg_buffer[us_tx_eoc_msg_length + 1] = (uint8)(gt_TxEocMsg.gus_FCS >> 8); // FCS high octet

         // Add the transparency octets before transmitting
         {
            uint8 uc_tx_eoc_byte;

            for (i=MSG_START; i< us_tx_eoc_msg_length + 2; i++)
            {
               uc_tx_eoc_byte = guca_tx_eoc_msg_buffer[i];

               // Octet transparency
               if ((uc_tx_eoc_byte == 0x7E) || (uc_tx_eoc_byte == 0x7D))
               {
                  // A 0x7E or a 0x7D in the message is coded as 0x7D and 0x5E
                  AddMessageToOvhdFifo(&gt_TxEocMsgFifo, 0x7D);
                  uc_tx_eoc_byte = uc_tx_eoc_byte ^ 0x20;
                  gs_Prev_TxOvhdMsg_length++;
               }

               AddMessageToOvhdFifo(&gt_TxEocMsgFifo, uc_tx_eoc_byte);
               gs_Prev_TxOvhdMsg_length++;
            }
         }

         // Stream overhead channel message content
         DSH_SendStream(DSH_OCH_TX_MESSAGES,us_tx_eoc_msg_length,&guca_tx_eoc_msg_buffer[0]);
         // Complete Tx EOC message is available for processing
         gft_StartTxEOC = TRUE;

         // If a commnad is expected
         if((gt_TxEocMsg.cntrl_field & 0x2) == 0)
         {
            pt_TxEocControlInfo->lRxSymCnt = gl_RxSymbolCount; // time stamp
            pt_TxEocControlInfo->uc_ExpectedDesignator = guca_tx_eoc_msg_buffer[MSG_START];
            pt_TxEocControlInfo->uc_ExpectedMsgType = guca_tx_eoc_msg_buffer[MSG_START+1];
            pt_TxEocControlInfo->us_param_rd_carrier_index = t_TxOvhdMsgInfo.us_param_rd_carrier_index;
            pt_TxEocControlInfo->us_param_rd_start_carrier_index = t_TxOvhdMsgInfo.us_param_rd_start_carrier_index;
            pt_TxEocControlInfo->us_param_rd_stop_carrier_index = t_TxOvhdMsgInfo.us_param_rd_stop_carrier_index;
            pt_TxEocControlInfo->ft_MECmd = t_TxOvhdMsgInfo.ft_MECmd;
            // Based on priority and message type, set the flag for
            // acknowledge expected
            pt_TxEocControlInfo->ft_ack_expected = TRUE;

            // Set the expected test param id for test param messages sent
            if (pt_TxEocControlInfo->uc_ExpectedDesignator == PMD_TEST_PAR_RD_CMD_DESIG)
            {
               pt_TxEocControlInfo->uc_ExpectedTestParamId = guca_tx_eoc_msg_buffer[MSG_START+2];
            }

            /////////////////////////////////////////////////////////////////////////////////////////////////
            // Initilaize timer and set the timeout length for this message
            /////////////////////////////////////////////////////////////////////////////////////////////////
            // Standard specifies "Timeouts are based starting from the last octet of a
            // request message sent to last octet of a response message received"
            /////////////////////////////////////////////////////////////////////////////////////////////////

            // Initailize timer to 0
            pt_TxEocControlInfo->us_ovhd_timer = 0;
         }

      } // if (uc_comm_or_resp_bit < 2)
   } // if (gt_TxEocMsgFifo.IndexOfOldest == gt_TxEocMsgFifo.NextAvailable)

   // if (gs_G9935ErrorFeedbackState == G9935_ERRORFB_SOCMSG_SENT)
   // {

      // DSM_Vectoring_Debug:
      // gs_Debug0++;

      //if ((gs_RxSuperFrmCnt - gs_RxSuperFrmCnt_ForEOC) == 1)
      //{
         //gs_NoOfEocTransmitted1++;
      //}
      //else if((gs_RxSuperFrmCnt - gs_RxSuperFrmCnt_ForEOC) == 2)
      //{
         //gs_NoOfEocTransmitted2++;
      //}
      //else if((gs_RxSuperFrmCnt - gs_RxSuperFrmCnt_ForEOC) == 3)
      //{
         //gs_NoOfEocTransmitted3++;
      //}

      //gs_EocDebug[gs_RxPMDFrameCount>>4]++;
      //Debug EOC end

      // REVIEW_STEFAN: Why do we overwrite in ovhd_tx the gs_G9935ErrorFeedbackState?
      // I guess that this should be done:
      //    - in Synchro6 if Vectoring in showtime is supported
      // STATEMACHINE in showtime:
      //       G9935_ERRORFB_ENABLE:            "IDLE STATE" (ERB transmission is turned on - but no ERB processing is required right now)
      //       G9935_ERRORFB_DUMPED:            send out ERB in the following Superframe
      //       G9935_ERRORFB_FORM_MSG:          calculate and transmit ERB
      //       G9935_ERRORFB_SOCMSG_SENT:       ??? WHY DO WE NEED THIS STATE ???
      //gs_G9935ErrorFeedbackState = G9935_ERRORFB_ENABLE;  // disable only for debugging

      //DSM_Vectoring_Debug:
      //if (gs_PauseControl == 0x3A)
      //Pause(gs_PauseControl);
   // }

}



/*^^^
 *------------------------------------------------------------------------
 *
 *  Name : FlagT TxEocNextSendMessage(int32 l_PriorityIndex, TxOvhdMsgInfoStruct_t *pt_NextTxOvhdMsgInfo)
 *
  *  Description:  This subroutine reads the HDLC message from the queue and transmits it
 *
 *  Prototype:   FlagT TxEocNextSendMessage(int32 l_PriorityIndex, TxOvhdMsgInfoStruct_t *pt_NextTxOvhdMsgInfo)
 *
 *  Input Arguments:
 *
 *  Output Arguments:
 *
 *  Return:
 *
 *  Global Variables Used:
 *
  *------------------------------------------------------------------------
 *^^^
 */

FlagT TxEocNextSendMessage(int32 l_PriorityIndex, TxOvhdMsgInfoStruct_t *pt_NextTxOvhdMsgInfo)
{
   int32 l_ReadIndex, l_NextReadIndex;
   TxEocControl_t *pt_TxEocControlInfo;
   TxOvhdMsgInfoStruct_t *pt_TxOvhdMsgInfo=NULL;


   pt_TxEocControlInfo = &gta_TxEocControlInfo[l_PriorityIndex];

   l_ReadIndex = pt_TxEocControlInfo->uc_ReadIndex;
   l_NextReadIndex = l_ReadIndex;

   // Queue is not empty if ReadIndex and WriteIndex are different.
   // Note: When a timeout is running for an already transmitted CMD or Segment Resp no new
   //       queued CMD or Segment Resp is allowed to be send.
   //       But plain Resp can be transmitted which are pending!
   while (l_ReadIndex != pt_TxEocControlInfo->uc_WriteIndex)
   {
      // Get queue entry which is pending
      pt_TxOvhdMsgInfo = &gta_TxOvhdMessage[l_PriorityIndex][l_NextReadIndex];

      // advance l_ReadIndex ahead of time
      if (l_ReadIndex == EOC_QUEUE_LENGTH - 1)
      {
         l_ReadIndex = 0;
      }
      else
      {
         l_ReadIndex++;
      }

      if (pt_TxEocControlInfo->ft_ack_expected == TRUE) // waiting for response
      {
         // Check if in middle of segmentation
         if ((pt_TxOvhdMsgInfo->uc_Segmented)  ||
               (pt_TxOvhdMsgInfo->uc_Source == AUTO_CMD_SOURCE))
         {
            // currently waiting for an segmented ack/response/cmd from the other end
            // don't want to interrupt the segmentation sequence; get out
            l_NextReadIndex = pt_TxEocControlInfo->uc_WriteIndex;     // indicate sending nothing
            break;
         }
         else if ((pt_TxOvhdMsgInfo->uc_Source == RESPONSE_SOURCE) && (pt_TxOvhdMsgInfo->uc_Segmented == FALSE))
         {
            // this response requires no timer. ok to send this response
            break;
         }
      }

#ifdef OVHD_MSG_SEGMENTATION
      // XDSLRTFW-1929 (Start_End)
      else if ((gsa_Tx_OvhdMsgSegIndx[l_PriorityIndex] >= 0) && (l_PriorityIndex != OVHD_LOW_PRIORITY)) // in middle of segmentation
      {
         if (pt_TxOvhdMsgInfo->uc_Segmented)
         {
            // currently, each priority only has one type of segmented message supported.
            // so, this must be the next segmented msg in the sequence.
            break;
         }
      }
#endif
      else
      {
         // send whatever pending
         break;
      }
      l_NextReadIndex = l_ReadIndex;
   }

   // is there a valid pending message to be sent?
   if (l_NextReadIndex != pt_TxEocControlInfo->uc_WriteIndex)
   {
      // copy it the "to-be-sent" entry
      *pt_NextTxOvhdMsgInfo = *pt_TxOvhdMsgInfo;

      // shift/push all queue entries from (original ReadIndex to l_NextReadIndex - 1) to
      // (original ReadIndex + 1 to l_NextReadIndex) to
      while (l_NextReadIndex != pt_TxEocControlInfo->uc_ReadIndex)
      {
         if (l_NextReadIndex == 0)
         {
            l_ReadIndex = EOC_QUEUE_LENGTH - 1;
         }
         else
         {
            l_ReadIndex = l_NextReadIndex - 1;
         }
         gta_TxOvhdMessage[l_PriorityIndex][l_NextReadIndex] = gta_TxOvhdMessage[l_PriorityIndex][l_ReadIndex];
         l_NextReadIndex = l_ReadIndex;
      }

      // update ReadIndex. No acceess control/semaphore needed because this field
      // is only updated by the BG code (in fact, by this function exclusively).
      if (l_NextReadIndex++ == EOC_QUEUE_LENGTH - 1)
      {
         pt_TxEocControlInfo->uc_ReadIndex = 0;
      }
      else
      {
         pt_TxEocControlInfo->uc_ReadIndex = (uint8)l_NextReadIndex;
      }

      // the MsgTypeCount is accessible to both FG and BG, lock while updating
      pt_TxEocControlInfo->uc_Semaphore = TRUE;
      pt_TxEocControlInfo->uca_MsgTypeCount[pt_NextTxOvhdMsgInfo->uc_Source]--;
      pt_TxEocControlInfo->uc_Semaphore = FALSE;

      if (pt_NextTxOvhdMsgInfo->uc_Source == RESPONSE_SOURCE)
      {
         pt_TxEocControlInfo->s_TotalRespSent++;
      }
      else
      {
         //must be CMD
         pt_TxEocControlInfo->s_TotalCmdSent++;
      }
      return(TRUE);
   }

   return(FALSE);
}



/*^^^
 *------------------------------------------------------------------------------
 *
 *  Name : Tx_prior0_ovhd_msg_handler(uint8 *puca_msg_buffer, uint16 *pus_msg_length)
 *
 *  Description:  This subroutine forms the OLR message
 *
 *  Prototype:
 *     uint8 Tx_prior0_ovhd_msg_handler(uint8 *puca_msg_buffer, uint16 *pus_msg_length)
 *
 *  Input Arguments:
 *
 *  Output Arguments:
 *
 *  Global Variables Used:
 *
 *  Notes: These can be sent both by CO as well as RT
 *
 *------------------------------------------------------------------------
 *^^^
 */
// XDSRTFW-442: Feature_All_All_All_All_Vdsl2Sra [Start]
uint8 Tx_prior0_ovhd_msg_handler(uint8 *puca_msg_buffer, uint16 *pus_msg_length, TxOvhdMsgInfoStruct_t *pt_TxOvhdMsgInfo)
{
   uint8 uc_comm_or_resp_bit = 2;
   uint8 uc_mod_msg_type;
   uint8 *puca_local;

   int16 s_Nf, s_lp,s_bc, s_msg_length = 5;
   VDSL2Config_t *pt_Config;

   int16 j = 0;

   pt_Config = &gt_rx_config_v2;

   // Fil in the message designator and type
   puca_msg_buffer[0] = pt_TxOvhdMsgInfo->uc_message_designator;
   puca_msg_buffer[1] = pt_TxOvhdMsgInfo->uc_message_type;

   if (pt_TxOvhdMsgInfo->uc_message_designator == EOC_CMD_G9935_ERRORFB_DESIG)
   {
      uc_comm_or_resp_bit = Form_G9935_ErrorFB_msg( &puca_msg_buffer[0], pus_msg_length, pt_TxOvhdMsgInfo);

      //gs_G9935ErrorFeedbackState = G9935_ERRORFB_SOCMSG_SENT;
   }
   else
   {
      puca_local = &puca_msg_buffer[2];

      uc_mod_msg_type = (uint8)((uint16)pt_TxOvhdMsgInfo->uc_message_type - 3);

      if (uc_mod_msg_type <= 5)
      {
         //increment the counter
         uint8 uc_message_type = pt_TxOvhdMsgInfo->uc_message_type;
         // XDSLRTFW-1032_New_framing_for_SRA_in_DS-ReTx_mode (START)
         if (pt_TxOvhdMsgInfo->uc_message_type == SRA_RETX_REQ)
         {
            // Update the SRA request count as per the location in gt_UsOLRPMCnt
            uc_message_type = SRA_REQ;
         }
         // XDSLRTFW-1032_New_framing_for_SRA_in_DS-ReTx_mode (END)
         Update_OLRPMCnt(&gt_DsOLRPMCnt, uc_message_type, SIMPLE_REQ_CNT);

         uc_comm_or_resp_bit = 0; // command

         // Add Lp value for SRA request
         // Fill in the actual Lp values for active latency path: LP0
         if (pt_TxOvhdMsgInfo->uc_message_type == SRA_REQ)
         {
            // XDSLRTFW-2630 : SRA in 35B got constantly rejected due to wrong s_msg_length, i.e.
            //                 extended length of L (three bytes instead of 2) was not considered.
            {
               uint8 uc_NLpFramingOctets;

               uc_NLpFramingOctets = 7;
               // ANXQ_SUPPORT
               if (gt_ProfileAct.us_ProfileSelected & CNFG_V2_PROFILE_35B_MASK)
               {
                  uc_NLpFramingOctets += 1;                         // uc_NLpFramingOctets = 8
               }
               s_msg_length = 5 + (uc_NLpFramingOctets * pt_Config->s_Nlp);
            }

            // Latency path descriptor
            for ( s_lp = 0; s_lp < pt_Config->s_Nlp; s_lp++)
            {
               /* Lp value */
               {
                  uint32 ul_Temp;

                  ul_Temp = pt_Config->ul_Reconfig_Lp[s_lp];
                  // ANXQ_SUPPORT
                  // Note: The "& 0xFF" was done for better readability, but
                  //       it is not need. The cast "(uint8)" is doing exactly
                  //       the same!
                  if (gt_ProfileAct.us_ProfileSelected & CNFG_V2_PROFILE_35B_MASK)
                  {
                     *puca_local++ = (uint8)((ul_Temp >> 16) & 0xFF);
                  }
                  *puca_local++ = (uint8)((ul_Temp >> 8) & 0xFF);
                  *puca_local++ = (uint8)(ul_Temp & 0xFF);
               }

               /* Dp value */
               *puca_local++ = (uint8)(pt_Config->s_Reconfig_Dp[s_lp] >> 8);
               *puca_local++ = (uint8)(pt_Config->s_Reconfig_Dp[s_lp] & 0xFF);

               /* Tp octet */
               *puca_local++ = (uint8)pt_Config->s_Reconfig_Tp[s_lp];

               /* Gp octet */
               *puca_local++ = (uint8)pt_Config->s_Reconfig_Gp[s_lp];

               //XDSLRTFW-1697(START_END)
               /* Bpn value */
               s_bc = gt_rx_TPS_Map.s_LPtoABC[s_lp];
               *puca_local++ = (uint8)pt_Config->sa_Reconfig_Bpn[s_lp][s_bc];
            }
         }
         // XDSLRTFW-1031_OLR Type-5 request for DS-SRA in DS-ReTx mode (START)
         else if (pt_TxOvhdMsgInfo->uc_message_type == SRA_RETX_REQ)
         {
            // When SRA and retransmission are simultaneously enabled, the modems use OLR Request Type 5
            // to initiate an SRA request. These OLR commands replace the OLR Request Type 3 (SRA)
            // when retransmission is enabled.
            // XDSLRTFW-2630 : SRA in 35B got constantly rejected due to wrong s_msg_length, i.e.
            //                 extended length of L (three bytes instead of 2) was not considered.
            s_msg_length = 14;

            /* L1 value */
            {
               uint32 ul_Temp;

               ul_Temp = pt_Config->ul_Reconfig_Lp[LP1];

               // ANXQ_SUPPORT
               // Note: The "& 0xFF" was done for better readability, but
               //       it is not need. The cast "(uint8)" is doing exactly
               //       the same!
               if (gt_ProfileAct.us_ProfileSelected & CNFG_V2_PROFILE_35B_MASK)
               {
                  s_msg_length += 1;                                    // s_msg_length = 15;

                  *puca_local++ = (uint8)((ul_Temp >> 16) & 0xFF);
               }
               *puca_local++ = (uint8)((ul_Temp >> 8) & 0xFF);
               *puca_local++ = (uint8)(ul_Temp & 0xFF);
            }

            /* B10 value */
            *puca_local++ = (uint8)pt_Config->sa_Reconfig_Bpn[LP1][ABC0];

            /* M1 value */
            *puca_local++ = (uint8)gt_ReTxShowtimeParams.s_Reconfig_Mp;

            /* R1 value */
            *puca_local++ = (uint8)gt_ReTxShowtimeParams.s_Reconfig_Rp;

            /* RS CW per DTU */
            *puca_local = gt_ReTxShowtimeParams.uc_CWsPerDtu_Q;
//XDSLRTFW-2162 (Start_End)
            if(guc_ReTx_DS_D  > 1)
            {
               *puca_local |= 0x80;
            }
            *puca_local++;

            /* padding octets per DTU */
            *puca_local++ = gt_ReTxShowtimeParams.uc_DtuPaddingSize_V;

            /* Delay in DTU between two consecutive transmissions of a DTU */
            *puca_local++ = gt_ReTxShowtimeParams.uc_RPMS_Qtx;

            /* Look-back value used to calculate the values communicated in the RRC */
            *puca_local++ = MIN(31, gt_ReTxShowtimeParams.uc_RPMS_Qtx);
         }
         // XDSLRTFW-1031_OLR Type-5 request for DS-SRA in DS-ReTx mode (END)
         // Start filling the bitswap information
         s_Nf = FillBinsInfo(puca_local, 0, pt_TxOvhdMsgInfo->uc_message_type);

         // Write the tx message length
         *pus_msg_length = s_msg_length + (s_Nf<<2);
         //XDSLRTFW-1379[Start]
         if(TESTArray[TEST_ReconfigControl] & TEST_AbandonSRA )
         {
            gft_BlockSRA = FALSE;
         }
         //XDSLRTFW-1379[End]
      }
      else if(pt_TxOvhdMsgInfo->uc_message_type >= DEFER_BITSWAP)
      {
         uc_comm_or_resp_bit = 0x1; // response

         *puca_local++ = pt_TxOvhdMsgInfo->uc_defer_reason_code; // Reason code

         // Write the tx message length
         *pus_msg_length = 3;

      }
   }
   return(uc_comm_or_resp_bit);
}
// XDSRTFW-442: Feature_All_All_All_All_Vdsl2Sra [End]

/*^^^
 *------------------------------------------------------------------------------
 *
 *  Name : Tx_prior1_ovhd_msg_handler
 *
 *  Description: This subroutine forms the mesage of priority 1
 *
 *  Prototype:
 *   uint8 Tx_prior1_ovhd_msg_handler(void)
 *
 *  Input Arguments:
 *
 *  Output Arguments:
 *      Returns the command /response code based on what the message type is
 *
 *  Global Variables Used:
 *
 *
 *------------------------------------------------------------------------
 *^^^
 */
uint8 Tx_prior1_ovhd_msg_handler(uint8 *puca_msg_buffer, uint16 *pus_msg_length, TxOvhdMsgInfoStruct_t *pt_TxOvhdMsgInfo)
{
   uint8 uc_comm_or_resp_bit = 2;      // Initialize with an invalid value

   puca_msg_buffer[0] = pt_TxOvhdMsgInfo->uc_message_designator;
   puca_msg_buffer[1] = pt_TxOvhdMsgInfo->uc_message_type;

   // Form the message
   switch(pt_TxOvhdMsgInfo->uc_message_designator)
   {

   case EOC_CMD_DESIG:
      uc_comm_or_resp_bit = Form_EOC_msg( &puca_msg_buffer[0], pus_msg_length, pt_TxOvhdMsgInfo);
      break;

   case TIME_CMD_DESIG:
      uc_comm_or_resp_bit = Form_time_msg( &puca_msg_buffer[0], pus_msg_length, pt_TxOvhdMsgInfo);
      break;

   case INVENTORY_CMD_DESIG:
      uc_comm_or_resp_bit = Form_Inventory_msg( &puca_msg_buffer[0], pus_msg_length, pt_TxOvhdMsgInfo);
      break;

   case MGMT_CNTR_RD_CMD_DESIG:
      uc_comm_or_resp_bit = Form_mgmt_counter_read_msg( &puca_msg_buffer[0], pus_msg_length, pt_TxOvhdMsgInfo);
      break;
      // XDSLRTFW-650 / XDSLRTFW-1384 (Start)
   case POWER_MGMT_CMD_DESIG:
      uc_comm_or_resp_bit = Form_Power_Mgmt_msg( &puca_msg_buffer[0], pus_msg_length, pt_TxOvhdMsgInfo);
      break;
      // XDSLRTFW-650 / XDSLRTFW-1384 (End)
   case CLEAR_EOC_CMD_DESIG:
      uc_comm_or_resp_bit = Form_Clear_EOC_msg( &puca_msg_buffer[0], pus_msg_length, pt_TxOvhdMsgInfo);
      break;

   case NSF_FACILITY_CMD_DESIG:
      uc_comm_or_resp_bit = Form_nsf_facility_msg( &puca_msg_buffer[0], pus_msg_length, pt_TxOvhdMsgInfo);
      break;

//#ifdef MTK_VECTORING_SUPPORT
   case EOC_CMD_G9935_UPDATE_PILOTSEQ_DESIG:
      uc_comm_or_resp_bit = Form_G9935_PilotUpdateResp_msg( &puca_msg_buffer[0], pus_msg_length, pt_TxOvhdMsgInfo);
      break;
//#endif

   }

   return(uc_comm_or_resp_bit);
}

// XDSLRTFW-650 / XDSLRTFW-1384 (Start)
#if 1
/*^^^
 *------------------------------------------------------------------------------
 *
 *  Name : Form_Power_Mgmt_msg(uint8 *puca_msg_buffer, uint16 *pus_msg_length)
 *
 *  Description:  This subroutine forms the management counter read message to be sent
 *
 *  Prototype:
 *      uint8 Form_Power_Mgmt_msg(uint8 *puca_msg_buffer, uint16 *pus_msg_length)
 *
 *  Input Arguments:
 *
 *  Output Arguments:
 *
 *  Global Variables Used:
 *
 *  Notes:
 *
 *------------------------------------------------------------------------
 *^^^
 */
uint8 Form_Power_Mgmt_msg(uint8 *puca_msg_buffer, uint16 *pus_msg_length,
                          TxOvhdMsgInfoStruct_t *pt_TxOvhdMsgInfo)
{
   uint8 uc_comm_or_resp_bit = 2;
   int32 idx;
   uint8 uc_message_type, uc_defer_reason_code;

   idx = 2;
   uc_message_type = pt_TxOvhdMsgInfo->uc_message_type;
   uc_defer_reason_code = pt_TxOvhdMsgInfo->uc_defer_reason_code;
   gsa_Tx_OvhdMsgSegIndx[OVHD_NORMAL_PRIORITY] = -1;

   switch(uc_message_type)
   {

   case STATE_TRANS_REQ:
      uc_comm_or_resp_bit = 0;
      //Required for Simple PM Commands
      puca_msg_buffer[idx++] = pt_TxOvhdMsgInfo->uc_TestParameterId;
      break;

   case STATE_TRANS_GRANT:
      uc_comm_or_resp_bit = 1;
      break;

   case STATE_TRANS_REJECT:
      uc_comm_or_resp_bit = 1;
      /* Send 1 octet for the reason code */
      puca_msg_buffer[idx++] = uc_defer_reason_code;

      break;

   }

   *pus_msg_length = (int16)idx;

   return(uc_comm_or_resp_bit);

}

#endif
// XDSLRTFW-650 / XDSLRTFW-1384 (End)
/*^^^
 *------------------------------------------------------------------------------
 *
 *  Name : Tx_prior2_ovhd_msg_handler
 *
 *  Description: This subroutine forms the mesage of priority 2
 *
 *  Prototype:
 *   uint8 Tx_prior2_ovhd_msg_handler(void)
 *
 *  Input Arguments:
 *
 *  Output Arguments:
 *      Returns the command /response code based on what the message type is
 *
 *  Global Variables Used:
 *
 *
 *------------------------------------------------------------------------
 *^^^
 */
uint8 Tx_prior2_ovhd_msg_handler(uint8 *puca_msg_buffer, uint16 *pus_msg_length, TxOvhdMsgInfoStruct_t *pt_TxOvhdMsgInfo)
{
   // Initialize with an invalid value
   uint8 uc_comm_or_resp_bit = 2;

   puca_msg_buffer[0] = pt_TxOvhdMsgInfo->uc_message_designator;
   puca_msg_buffer[1] = pt_TxOvhdMsgInfo->uc_message_type;

   // Form the EOC message
   if(pt_TxOvhdMsgInfo->uc_message_designator == NSF_FACILITY_LOW_CMD_DESIG)
   {
      uc_comm_or_resp_bit = Form_nsf_facility_msg(&puca_msg_buffer[0], pus_msg_length, pt_TxOvhdMsgInfo);
   }
   else if(pt_TxOvhdMsgInfo->uc_message_designator == PMD_TEST_PAR_RD_CMD_DESIG)
   {
      // Form the test parameter read message
      uc_comm_or_resp_bit = Form_test_param_read_msg(&puca_msg_buffer[0], pus_msg_length, pt_TxOvhdMsgInfo);
   }

   return(uc_comm_or_resp_bit);
}

/*^^^
 *------------------------------------------------------------------------------
 *
 *  Name : int16 FillBinsInfo(uint8 *puca_msg_buffer, uint8 uc_priority, uint8 uc_message_type)
 *
 *  Description: This subroutine fills the bins info field of the transmit OLR message
 *
 *  Prototype:
 *     int16 FillBinsInfo(uint8 *puca_msg_buffer, uint8 uc_priority, uint8 uc_message_type)
 *
 *  Input Arguments:
 *
 *  Output Arguments:
 *
 *  Global Variables Used:
 *
 *
 *------------------------------------------------------------------------
 *^^^
 */
int16 FillBinsInfo(uint8 *puca_msg_buffer, uint8 uc_priority, uint8 uc_message_type)
{
   int16 i, k;
   int32 l_InvGain;
   int16 s_RxFineGain;
   int16 s_CurBand;
   uint8 uc_CarrierMaskHigh;

   k = 0;

   // ANXQ_SUPPORT
   uc_CarrierMaskHigh = 0xF;
   if (gt_ProfileAct.us_ProfileSelected & CNFG_V2_PROFILE_35B_MASK)
   {
      uc_CarrierMaskHigh = 0x1F;
   }

   // We will only do the inside part once for the whole ovhd msg, i.e, the first msg segment if we need to do
   // msg segmentation
   if (gsa_Tx_OvhdMsgSegIndx[uc_priority] == -1)
   {
      // 2 bytes for Nf
      puca_msg_buffer[k++] = (uint8)((gs_NumOfBitSwapTones >> 8) & 0xFF);
      puca_msg_buffer[k++] = (uint8)(gs_NumOfBitSwapTones & 0xFF);
   }

   // Form the message based on the tones changed
   // Note: 4 octets formatted as [000i iiii iiii iiii gggg gggg gggg bbbb], with
   //       gi (12 bits) and the bi (4 bits) values of the subcarrier index i (12 or 13 (35b) bits).
   for (s_CurBand=0; s_CurBand < gs_NumOfRxBands; s_CurBand++)
   {
         for (i=gsa_RxBandLeftChannel[s_CurBand]; i <= gsa_RxBandRightChannel[s_CurBand]; i++)
         {
            // Per standard the max allowed number of tones to be changed is (Nf <= 128).
            // This is controlled by guca_RxBitswapToneSet!
            if(IS_TONEFLAGSET(guca_RxBitswapToneSet, i))
            {
               puca_msg_buffer[k++] = (uint8)((i>>8) & uc_CarrierMaskHigh);
               puca_msg_buffer[k++] = (uint8)(i & 0xFF);

               s_RxFineGain = ghpsa_RxFineGains_Inactive[i];

               puca_msg_buffer[k++] = (uint8)(s_RxFineGain >> 8);      // Bin: Gain
               puca_msg_buffer[k++] = (uint8)((ghpuca_RxBat_Inactive[i] & 0xF) | (s_RxFineGain & 0xF0)) ; // DS Bin: # Bits

               //Compute the inverse of the linear gain in Q3.13 format to be stored in HW
               if (s_RxFineGain != 0)
               {
                  l_InvGain = NormAndDivide_32by16bit(((int32)1<<30), s_RxFineGain);
                  l_InvGain = round(l_InvGain, 4);

                  s_RxFineGain = sature16(l_InvGain);
               }

               ghpsa_RxFineGains_Inactive[i] = s_RxFineGain;
            }
         }
   }


      // Put in the Segementation code ( no segmentation)
   if (gsa_Tx_OvhdMsgSegIndx[uc_priority] == -1)
   {
      puca_msg_buffer[k++] = 0xC0;
   }

   return(gs_NumOfBitSwapTones);
}

int16 gs_num_AckSent;

/*^^^
 *------------------------------------------------------------------------------
 *
 *  Name : void Poll_ClearEOCAck(void)
 *
 *  Description: This subroutine checks if a Clear EOC ack needs to be sent
 *
 *  Prototype:
 *       void Poll_ClearEOCAck(void)
 *
 *
 *  Output Arguments:
 *
 *  Global Variables Used:
 *
 *
 *------------------------------------------------------------------------
 *^^^
 */


void Poll_ClearEOCAck(void)
{
   TxOvhdMsgInfoRequest_t t_TxOvhdMsgInfoInput;
   TxEocControl_t *pt_TxEocControlInfo;


   // Check if the last received clear EOC data is read by the host
   // Host needs to clear the data available flag once done with the read
   if ((guc_ClearEOCAckType != 0) && (gt_MONI_CEocStat.us_RxStatus == EOC_CEOC_RXSTAT_RXIDLE))
   {
      memset(&t_TxOvhdMsgInfoInput,0,sizeof(t_TxOvhdMsgInfoInput));

      t_TxOvhdMsgInfoInput.uc_message_designator = CLEAR_EOC_CMD_DESIG;
      t_TxOvhdMsgInfoInput.uc_Source = RESPONSE_SOURCE;

      t_TxOvhdMsgInfoInput.uc_message_type = guc_ClearEOCAckType;

      pt_TxEocControlInfo = &gta_TxEocControlInfo[OVHD_NORMAL_PRIORITY];

      // If clear EOC Ack successfully sent to the other end, reset the flag
      if (EOC_QUEUE_ADDED == TxEocSendRequest(1, t_TxOvhdMsgInfoInput))
      {
         // reset RX status to idle if NACK is sent
         if (guc_ClearEOCAckType == CLEAR_EOC_NACK)
         {
            gt_MONI_CEocStat.us_RxStatus = EOC_CEOC_RXSTAT_RXIDLE;
         }

         gs_num_AckSent++;
         guc_ClearEOCAckType = 0;
      }
   }
}


/*^^^
*-------------------------------------------------------------------
*
*   Prototype: void Update_OLRPMCnt(OLRPMCntInfo_t* pt_OLRPMCnt, int16 s_OLRPMType, int16 s_countType)
*
*   Description:
*      Update couter info of OLRPMCnt_t structure
*
*   Arguments:
*
*
*   Global Variables:
*
*-------------------------------------------------------------------
*^^^
*/
void Update_OLRPMCnt(OLRPMCntInfo_t* pt_OLRPMCnt, int16 s_OLRPMType, int16 s_countType)
{
   int16 *p_addr;
   int16 s_OLRPMCntStructOffset = (int16)(sizeof(OLRPMCnt_t)>>1);
    // XDSLRTFW-3932(Start)
   FlagT ft_INPUT_DATA_ARE_VALID = TRUE;
   int16 s_OLRPMType_Offset = 0;

     // map OLR type to FW structure:
     // typedef struct {
     //    OLRPMCnt_t t_BitswapCnt;       0
     //    OLRPMCnt_t t_SOSCnt;           1
     //    OLRPMCnt_t t_SRACnt;           2
     // }OLRPMCntInfo_t;

     // Supported OLR Types are:
     // REQUEST-TYPE      | OLR-Octet                         | mapped to OLR counter:
     // ------------------+-----------------------------------+------------------------
     // Type 1 (BS)       |  BITSWAP_REQ                 0x04 | t_BitswapCnt
     // Type 2            |  DRR_REQ                     0x05 | not supported
     // Type 3 (SRA)      |  SRA_REQ                     0x06 | t_SRACnt
     // Type 4 (SOS)      |  SOS_REQ                     0x07 | not supported
     // Type 5 (SRA-RETX) |  SRA_RETX_REQ                0x08 | t_SRACnt
     // Type 6 (SOS-RETX) |  SOS_RETX_REQ                0x09 | t_SOSCnt
     // ------------------+-----------------------------------+------------------------
     // DEFER-TYPE        | OLR-Octet                         | mapped to OLR counter:
     // ------------------+-----------------------------------+------------------------
     // Type 1 (BS)       |  DEFER_BITSWAP               0x81 | t_BitswapCnt
     // Type 2            |  DRR_REJECT                  0x82 | not supported
     // Type 3 (SRA)      |  SRA_REJECT                  0x83 | t_SRACnt
     // Type 4 (SOS)      |  SOS_REJECT                  0x84 | not supported
     // Type 5 (SRA-RETX) |  SRA_RETX_REJECT             0x85 | t_SRACnt
     // Type 6 (SOS-RETX) |  SOS_RETX_REJECT             0x86 | t_SOSCnt

     switch(s_OLRPMType)
     {
        // Bitswap requests / defer
        case BITSWAP_REQ:
        case DEFER_BITSWAP:
           s_OLRPMType_Offset = 0;
           break;

        // SOS requests / defer
        case SOS_REQ:
        case SOS_REJECT:
        case SOS_RETX_REQ:
        case SOS_RETX_REJECT:
           s_OLRPMType_Offset = 1;
           break;

 // SRA requests / defer
        case SRA_REQ:
        case SRA_REJECT:
        case SRA_RETX_REQ:
        case SRA_RETX_REJECT:
           s_OLRPMType_Offset = 2;
           break;

        // OLR type is not supported
        default:
           ft_INPUT_DATA_ARE_VALID = FALSE;
           // "INCREASE ERROR-COUNTER";
     }

     // Supported CountTypes are:
     // #define SIMPLE_REQ_CNT           (0)
     // #define UNEXPECTED_OLR_ERROR_CNT (1)
     // #define UTC_RESP_CNT             (2)
     // #define PERFORM_CNT              (3)
     // #define TIMEOUT_CNT              (4)
     if ((s_countType < SIMPLE_REQ_CNT) || (s_countType > TIMEOUT_CNT))
     {
        ft_INPUT_DATA_ARE_VALID = FALSE;
        // "INCREASE ERROR-COUNTER";
     }

     // set address to start of Counter-structure
     p_addr = (int16*)(void *)pt_OLRPMCnt;
     if (ft_INPUT_DATA_ARE_VALID)
     {
        // Add offset to correct OLR counter
        p_addr = p_addr + s_OLRPMType_Offset * s_OLRPMCntStructOffset + s_countType;
        // OLR counter is increased by one
        (*p_addr)++;
     }
     // XDSLRTFW-3932(End)
   // Write OLR Event into debug buffer
   {
      int16 s_MsgRaw[3];

      // OLR Type: 1- BS, 2- NotDef, 3- SRA and 4- SOS
      s_MsgRaw[0] = (int16) s_OLRPMType;

      // #define SIMPLE_REQ_CNT           (0)
      // #define UNEXPECTED_OLR_ERROR_CNT (1)
      // #define UTC_RESP_CNT             (2)
      // #define PERFORM_CNT              (3)
      // #define TIMEOUT_CNT              (4)


      // Counter type: 0 - us_SimpleOLRPMReq_Cnt;
      //               1 - us_ExtendOLRPMReq_Cnt;
      //               2 - us_OLRPMUTCResp_Cnt;
      //               3 - us_ActualOLRPMPerform_Cnt;
      //               4 - us_RespTimeOut_Cnt;
      s_MsgRaw[1] = (int16) s_countType;
      // Counter value corresponding to Counter type
      s_MsgRaw[2] = (int16) (*p_addr);

#ifdef DEBUG_TRACES
      if(pt_OLRPMCnt == &gt_UsOLRPMCnt)
      {
         DebugBuffWriteEv(DEBUG_EVT_OLR_US_STATS, sizeof(s_MsgRaw), s_MsgRaw);
      }
      else
      {
         DebugBuffWriteEv(DEBUG_EVT_OLR_DS_STATS, sizeof(s_MsgRaw), s_MsgRaw);
      }
#endif
   }
}



/*^^^
 *------------------------------------------------------------------------
 *
 *  Name : TxEocSendRequest(int32 l_PriorityIndex, TxOvhdMsgInfoRequest_t t_TxOvhdMsgInfoInput)
 *
 *  Description:  This subroutine sends Tx EOC msgs into Queue
 *
 *  Prototype:
 *      void  TxEocSendRequest(int32 l_PriorityIndex, TxOvhdMsgInfoRequest_t t_TxOvhdMsgInfoInput)
 *
 *  Input Arguments:
 *      l_PriorityIndex : the ovhd msg priority index
 *      t_TxOvhdMsgInfoInput: contains info about the message being added to queue
 *
 *  Output Arguments:
 *
 *  Return:
 *      None
 *
 *
 *  Notes: if the message type exceeds its max allowed count, return FALSE
 *         if not, insert the request to location pointed by WriteIndex, also update the control structure.
 *------------------------------------------------------------------------
 *^^^
 */
int32 TxEocSendRequest(int32 l_PriorityIndex, TxOvhdMsgInfoRequest_t t_TxOvhdMsgInfoInput)
{
   TxOvhdMsgInfoStruct_t *pt_TxOvhdMsgInfo;
   TxEocControl_t *pt_TxEocControlInfo;
   int32 l_Source;
   int32 l_ReturnCode = EOC_QUEUE_FULL;


   pt_TxEocControlInfo = &gta_TxEocControlInfo[l_PriorityIndex];
   l_Source = t_TxOvhdMsgInfoInput.uc_Source;
   // this check is to make that foreground will not corrupt the queue item when
   // this same is called by BG code and being interrupted. It's rare but potential.
   // what's needed protections? uc_WriteIndex and uca_MsgTypeCount
   if (pt_TxEocControlInfo->uc_Semaphore == FALSE)
   {
      uint8 ft_QueueEmpty;

      pt_TxEocControlInfo->uc_Semaphore = TRUE;
      // XDSLRTFW-1929 (Start_End)
      ft_QueueEmpty = TRUE;
      if ((l_PriorityIndex == OVHD_LOW_PRIORITY) &&
            (gsa_Tx_OvhdMsgSegIndx[OVHD_LOW_PRIORITY] > 0) &&
            (l_Source == AUTO_CMD_SOURCE))
      {
         // Block all low prio commands in middle of segmentation
         ft_QueueEmpty = FALSE;
      }

      if ((pt_TxEocControlInfo->uca_MsgTypeCount[l_Source] < pt_TxEocControlInfo->uca_MaxMsgTypeCount[l_Source]) &&
            (ft_QueueEmpty))
      {
         // points to available queue item
         pt_TxOvhdMsgInfo = &gta_TxOvhdMessage[l_PriorityIndex][pt_TxEocControlInfo->uc_WriteIndex];

         // update queue control
         if (pt_TxEocControlInfo->uc_WriteIndex == EOC_QUEUE_LENGTH - 1)
         {
            pt_TxEocControlInfo->uc_WriteIndex = 0;
         }
         else
         {
            pt_TxEocControlInfo->uc_WriteIndex++;
         }

         pt_TxEocControlInfo->uca_MsgTypeCount[l_Source]++;
         pt_TxEocControlInfo->uc_Semaphore = FALSE;

         // update queue entry
         {
            // copy input message info into queue item
            pt_TxOvhdMsgInfo->uc_message_type = t_TxOvhdMsgInfoInput.uc_message_type;
            pt_TxOvhdMsgInfo->uc_defer_reason_code = t_TxOvhdMsgInfoInput.uc_defer_reason_code;
            pt_TxOvhdMsgInfo->uc_message_designator = t_TxOvhdMsgInfoInput.uc_message_designator;
            pt_TxOvhdMsgInfo->uc_Source = (uint8)l_Source;
            pt_TxOvhdMsgInfo->uc_Segmented = t_TxOvhdMsgInfoInput.uc_Segmented;
            pt_TxOvhdMsgInfo->uc_TestParameterId = t_TxOvhdMsgInfoInput.uc_TestParameterId;
            pt_TxOvhdMsgInfo->us_param_rd_carrier_index = t_TxOvhdMsgInfoInput.us_param_rd_carrier_index;
            pt_TxOvhdMsgInfo->us_param_rd_start_carrier_index = t_TxOvhdMsgInfoInput.us_param_rd_start_carrier_index;
            pt_TxOvhdMsgInfo->us_param_rd_stop_carrier_index = t_TxOvhdMsgInfoInput.us_param_rd_stop_carrier_index;
            pt_TxOvhdMsgInfo->ft_MECmd = t_TxOvhdMsgInfoInput.ft_MECmd;

            EocTrail(EOC_TRAIL_TXDELIM, t_TxOvhdMsgInfoInput.uc_message_designator, t_TxOvhdMsgInfoInput.uc_message_type, 0);
         }
         l_ReturnCode = EOC_QUEUE_ADDED;
      }
      else
      {
         EocTrail(EOC_TRAIL_QFULLDELIM, t_TxOvhdMsgInfoInput.uc_message_designator, t_TxOvhdMsgInfoInput.uc_message_type, 0);
         l_ReturnCode = EOC_QUEUE_FULL;
      }

      pt_TxEocControlInfo->uc_Semaphore = FALSE;
      return(l_ReturnCode);
   }
   else
   {
      EocTrail(EOC_TRAIL_BUSYDELIM, t_TxOvhdMsgInfoInput.uc_message_designator, t_TxOvhdMsgInfoInput.uc_message_type, 0);
      return(EOC_QUEUE_BUSY);
   }
}


/*^^^
 *------------------------------------------------------------------------
 *
 *  Name : EOC_AutoMsg_Task
 *
 *  Description: This subroutine sends Autonomous EOC msgs into Queue
 *
 *  Prototype:
 *      void EOC_AutoMsg_Task(void)
 *
 *  Input Arguments:
 *
 *  Output Arguments:
 *
 *  Return:
 *      None
 *
 *
 *  Notes:
 *------------------------------------------------------------------------
 *^^^
 */
void EOC_AutoMsg_Task(void)
{
   TxOvhdMsgInfoRequest_t t_TxOvhdMsgInfoInput;
   int32 txstat;

// XDSLRTFW-1886 : BRCM Based Zyxel CO US Paramters not Acknowledged
   memset(&t_TxOvhdMsgInfoInput,0,sizeof(t_TxOvhdMsgInfoInput));

 //XDSLRTFW-3470 XDSLRTFW-2724 IOP_A_DS_VDSL2_EVLT_F_CNXT_0dBMarginReport (START)
 if ( (gul_fe_G994VendorID != CNXT_VENDOR_ID) ||
      ((gt_Dsl_Operator_Select.us_Operator_Select != AT_T) && (gt_Dsl_Operator_Select.us_Operator_Select != CTL)) )
 {
 //XDSLRTFW-3470 XDSLRTFW-2724 IOP_A_DS_VDSL2_EVLT_F_CNXT_0dBMarginReport (END)
   // LATN, SATN, SNRM, etc.
   t_TxOvhdMsgInfoInput.uc_message_designator = PMD_TEST_PAR_RD_CMD_DESIG;
   t_TxOvhdMsgInfoInput.uc_message_type = SINGLE_READ;
   t_TxOvhdMsgInfoInput.uc_Source = AUTO_CMD_SOURCE;

   txstat = TxEocSendRequest(2 /* priority */, t_TxOvhdMsgInfoInput);

   // mgmt counter read
   t_TxOvhdMsgInfoInput.uc_message_designator = MGMT_CNTR_RD_CMD_DESIG;
   t_TxOvhdMsgInfoInput.uc_message_type = MGMT_CNTR_READ;
   t_TxOvhdMsgInfoInput.uc_Source = AUTO_CMD_SOURCE;

   txstat = TxEocSendRequest(1 /* priority */, t_TxOvhdMsgInfoInput);
 }   //XDSLRTFW-3470 XDSLRTFW-2724 IOP_A_DS_VDSL2_EVLT_F_CNXT_0dBMarginReport (START_END)

   if (!gus_EocInvMsgResp)
   {
      t_TxOvhdMsgInfoInput.uc_message_designator = INVENTORY_CMD_DESIG;
      t_TxOvhdMsgInfoInput.uc_message_type = IDENT_READ;
      t_TxOvhdMsgInfoInput.uc_Source = AUTO_CMD_SOURCE;

      txstat = TxEocSendRequest(1 /* priority */, t_TxOvhdMsgInfoInput);

      t_TxOvhdMsgInfoInput.uc_message_designator = INVENTORY_CMD_DESIG;
      t_TxOvhdMsgInfoInput.uc_message_type = AUX_IDENT_READ;
      t_TxOvhdMsgInfoInput.uc_Source = AUTO_CMD_SOURCE;

      txstat = TxEocSendRequest(1 /* priority */, t_TxOvhdMsgInfoInput);

      return;
   }
// XDSLRTFW-1886 : BRCM Based Zyxel CO US Paramters not Acknowledged

   //XDSLRTFW-3470 XDSLRTFW-2724 IOP_A_DS_VDSL2_EVLT_F_CNXT_0dBMarginReport (START)
   if ( (gul_fe_G994VendorID == CNXT_VENDOR_ID) &&
         ((gt_Dsl_Operator_Select.us_Operator_Select == AT_T) || (gt_Dsl_Operator_Select.us_Operator_Select == CTL)) )
   {
   // mgmt counter read
   t_TxOvhdMsgInfoInput.uc_message_designator = MGMT_CNTR_RD_CMD_DESIG;
   t_TxOvhdMsgInfoInput.uc_message_type = MGMT_CNTR_READ;
   t_TxOvhdMsgInfoInput.uc_Source = AUTO_CMD_SOURCE;
   txstat = TxEocSendRequest(1 /* priority */, t_TxOvhdMsgInfoInput);

   // LATN, SATN, SNRM, etc.
   t_TxOvhdMsgInfoInput.uc_message_designator = PMD_TEST_PAR_RD_CMD_DESIG;
   t_TxOvhdMsgInfoInput.uc_message_type = SINGLE_READ;
   t_TxOvhdMsgInfoInput.uc_Source = AUTO_CMD_SOURCE;
   txstat = TxEocSendRequest(2 /* priority */, t_TxOvhdMsgInfoInput);
   }
   //XDSLRTFW-3470 XDSLRTFW-2724 IOP_A_DS_VDSL2_EVLT_F_CNXT_0dBMarginReport (END)
}

/*^^^
 *------------------------------------------------------------------------
 *
 *  Name : EOC_Poll_FEReadTrigger(void)
 *
 *  Description: Polls MONI 8 to check for FE Read Trigger.  Performs a
 *               PMD-Test-Param Block-Read to get FE Data
 *
 *  Prototype:
 *      void EOC_Poll_FEReadTrigger(void)
 *
 *  Input Arguments:
 *
 *  Output Arguments:
 *
 *  Return:
 *      None
 *
 *
 *  Notes:
 *------------------------------------------------------------------------
 *^^^
 */
extern FlagT gft_EocNack;
extern int16 gs_EocGroupSize ;
extern int16 gs_EocGroupSizeHLOG ;
extern int16 gs_EocGroupSizeSNR ;
#define BRCM_TEST_PARAM_READ_COUNT       (32)
#define BRCM_TEST_PARAM_HLOG_READ_COUNT  (20)
#define TEST_PARAM_BLOCK_READ_COUNT      (32)
#define INITIAL_TIME_BEFORE_START        (30)  //3 Seconds
#define TIME_500MSEC                     (5)   //500msec
#define TIME_100MSEC                     (1)   //100ms
#define TIME_2_SECS                      (20)  //2 Seconds
#define TIME_1_SECS                      (10)  //1 Seconds
#define TIME_10_SECS                     (100) //10 Seconds
#define HLOG_VECTOR_TEST_PARAM           (1)
#define QLN_VECTOR_TEST_PARAM            (2)
#define SNR_VECTOR_TEST_PARAM            (3)
void EOC_Poll_FEReadTrigger(void)
{

   uint32 ul_US_Hlogf_FE = 0;
   uint32 ul_US_Qln_FE = 0;
   uint32 ul_US_Snr_FE = 0;
   uint16 us_US_Hlogf_FE0 = 0;
   uint16 us_US_Qln_FE0 = 0;
   uint16 us_US_Snr_FE0 = 0;
   uint16 us_US_Hlogf_FE1 = 0;
   uint16 us_US_Qln_FE1 = 0;
   uint16 us_US_Snr_FE1 = 0;
   static int16 s_endIx = -1;
   static int16 s_endIx_BRCM = -1;
   static uint16 us_100msCnt = 0;
   static int16 s_timeInitial = INITIAL_TIME_BEFORE_START;
   static int16 s_TestParam = HLOG_VECTOR_TEST_PARAM;
   static int16 s_bandIndex = 0;
   static int16 s_ReadLength = BRCM_TEST_PARAM_HLOG_READ_COUNT ;
   int16 s_TxBandLeftChannel = 0;
   int16 s_TxBandLeftChannelNxt = 0;
   int16 s_TxBandRightChannel = 0;
   int16 s_TxBandRightChannelLast = 0;
   int16 s_FeParamsPollingTime  = 0;
   static FlagT ft_TransferData = 0;

   TxOvhdMsgInfoRequest_t t_TxOvhdMsgInfoInput;
   int32 txstat;
   //int16 i,j;

   s_FeParamsPollingTime = gus_FeParamTimerInterval;

   if (gft_TestParam_Req == VEC_BLOCK_READ_CMD)
   {
      //Group Size Change for SNR
      if(s_TestParam >= SNR_VECTOR_TEST_PARAM)
      {
         gs_EocGroupSize = gs_EocGroupSizeSNR;
      }
      else
      {
         gs_EocGroupSize = gs_EocGroupSizeHLOG;
      }
   }//VECTOR BLOCK READ Code



   //This part of code is introduced to Delay
   //and space the requests with 500msec duration
   //between each vector block read commands
   //For BRCM We need all the Vector Blocks to be
   //read at 500msec duration
   //Also a wait period of 10 seconds before we read
   //SNR Updates periodically
   if ( (!gft_EocME_FEBlockReadPending) &&
         ( (gft_TestParam_Req == VEC_BLOCK_READ_CMD) ||
           (gft_TestParam_Req == BLOCK_READ_CMD) ) )
   {
      if (++us_100msCnt < s_timeInitial) // time-out (clear flag) after 2 seconds
      {
         return; // block until current FE read request is done
      }
      else
      {
         us_100msCnt = 0;
         ft_TransferData = 0; //We are Sceduling a data read periodically 10 Seconds
         //XDSLRTFW-2316 (Start)
         if (gft_TestParam_Req == VEC_BLOCK_READ_CMD)
         {
            s_timeInitial = TIME_100MSEC;
         }
         else // BLOCK_READ_CMD
         {
            s_timeInitial = TIME_500MSEC;
         }
         //XDSLRTFW-2316 (End)
      }
   }

   //Original code
   //Used to Block any more request when CPE
   //has sent the vector block read command
   if (gft_EocME_FEBlockReadPending)
   {
      if (++us_100msCnt >= TIME_2_SECS) // time-out (clear flag) after 2 seconds
      {
         gft_EocME_FEBlockReadPending = 0;
         s_endIx = -1; // start over
      }
      else
      {
         return; // block until current FE read request is done
      }
   }

   //This flag is used to block any more
   //Block read until PMD copy all the data to API/SDRAM
   //This flag is not used now as MONI 8 transfer are not used anymore
   if (gft_SendFEDataFlag)
   {
      return; // block until pending auto msg is sent
   }

   //gt_FEData.us_StartIndex = gt_MONI_FEReadTrigger.us_StartIndex;
   //gt_FEData.us_EndIndex =  gt_MONI_FEReadTrigger.us_EndIndex;
   // valid request?
   if( (gt_FEData.us_EndIndex >= gt_FEData.us_StartIndex) &&
         ((gt_FEData.us_EndIndex < EOC_MAX_SCG) && (gt_FEData.us_StartIndex < EOC_MAX_SCG)) )
   {
      if (-1 == s_endIx) // new request?
      {
         memset(&gt_FEData,0,sizeof(gt_FEData));
      } // new request?
   } // valid request?
   else
   {
      s_endIx = -1; // reset index
      return; // nothing to do
   }

   memset(&t_TxOvhdMsgInfoInput,0,sizeof(t_TxOvhdMsgInfoInput));

   if (-1 == s_endIx) // 1st time?
   {
      s_bandIndex = 0;
      //SNR Reads need Group Size Medley
      if(s_TestParam >= SNR_VECTOR_TEST_PARAM)
      {
         s_TxBandLeftChannel = (gsa_TxBandLeftChannel[s_bandIndex]>>gs_EocGroupSize);
         s_TxBandLeftChannelNxt = (gsa_TxBandLeftChannel[s_bandIndex+1]>>gs_EocGroupSize);
         s_TxBandRightChannel = (gsa_TxBandRightChannel[s_bandIndex]>>gs_EocGroupSize);
      }
      else
      {
         s_TxBandLeftChannel = (gsa_TxBandLeftChannelOSignature[s_bandIndex]>>gs_EocGroupSize);
         s_TxBandLeftChannelNxt = (gsa_TxBandLeftChannelOSignature[s_bandIndex+1]>>gs_EocGroupSize);
         s_TxBandRightChannel = (gsa_TxBandRightChannelOSignature[s_bandIndex]>>gs_EocGroupSize);
      }
      //Block Read for IFX
      //Always starts from index 0
      //Use Block Read Command for IFX where the length to be read
      //would be Start to End of MONI 8 for each HLOG/QLN/SNR
      gt_FEData.us_StartIndex = gt_MONI_FEReadTrigger.us_StartIndex;
      gt_FEData.us_EndIndex =  gt_MONI_FEReadTrigger.us_EndIndex;

      //Vector Block read for BRCM
      //Always starts from In band starting index
      if(gft_TestParam_Req == VEC_BLOCK_READ_CMD)
      {
         gt_FEData.us_StartIndex = s_TxBandLeftChannel ;
         gt_FEData.us_EndIndex = gt_FEData.us_StartIndex + s_ReadLength ;
         //In case of US0, we need this check as in band will be less than Group 18 indices
         if ( ((gt_FEData.us_EndIndex > s_TxBandRightChannel) && (gt_FEData.us_EndIndex < s_TxBandLeftChannelNxt)) ||
               ((gt_FEData.us_EndIndex > s_TxBandRightChannel) && (gus_UseUS0OnlyCntrl & (TX_US0_ONLY_INDICATION|TX_US0_ONLY_INDICATION_GHS))) //US0 Only s_TxBandLeftChannelNxt is 0, so first check is sufficient
            )
         {
            gt_FEData.us_EndIndex = s_TxBandRightChannel;
         }
      }
      else
      {
         s_ReadLength = TEST_PARAM_BLOCK_READ_COUNT;
      }


      s_endIx = MIN(gt_FEData.us_EndIndex,
                    gt_FEData.us_StartIndex+EOC_NUM_SCGs_FEDATA-1);

/*
   //Make sure we request all ways even number. As we pad octets
      //to store in SDRAM
      if( ((gt_FEData.us_EndIndex - gt_FEData.us_StartIndex + 1) & 0x1) &&
            (gft_TestParam_Req == VEC_BLOCK_READ_CMD) )
      {
         s_endIx += 1;
      }
*/
      gt_FEData.us_EndIndex = s_endIx;

      gus_EocFEData_AutoMsgIx = 0; // Index field of the FE Data auto-msg
   }
   //Next Set of read and no Nack in the response for the prev request
   else if ( (s_endIx != -1) && (gft_EocNack == 0) )
   {
      //For Block Read
      //If the Segment in MONI 8 is done, switch off Trigger
      //Else, read the remaining part of the data till the
      //End Index in MONI 8
      if(gft_TestParam_Req == BLOCK_READ_CMD)
      {
         if (s_endIx >= (EOC_MAX_SCG-1))
         {
            //not used as MONI 8 is disabled
            //ft_TransferData = 1;
            //gt_MONI_FEReadTrigger.us_trigger = 0; // all done

            s_endIx_BRCM = -1; // reset index: This is to transfer Data to API
            s_timeInitial = s_FeParamsPollingTime; //TIME_10_SECS; //10 seconds
            s_endIx = -1; // reset index
            s_bandIndex = 0;
            s_ReadLength = TEST_PARAM_BLOCK_READ_COUNT;
            return;
         }
         else {
            gt_FEData.us_StartIndex = s_endIx+1;
            gt_FEData.us_EndIndex = gt_FEData.us_StartIndex + s_ReadLength;
         }
      }
      else
      {
         //Check if start cross the Band end tone, assign the next read from next band
         gt_FEData.us_StartIndex = s_endIx+1;

         //Group Size Change for SNR
         if(s_TestParam >= SNR_VECTOR_TEST_PARAM)
         {
            s_TxBandLeftChannel = (gsa_TxBandLeftChannel[s_bandIndex]>>gs_EocGroupSize);
            s_TxBandLeftChannelNxt = (gsa_TxBandLeftChannel[s_bandIndex+1]>>gs_EocGroupSize);
            s_TxBandRightChannel = (gsa_TxBandRightChannel[s_bandIndex]>>gs_EocGroupSize);
            s_TxBandRightChannelLast = ((gsa_TxBandRightChannel[gs_NumOfTxBands-1])>>gs_EocGroupSize);
         }
         else
         {
            s_TxBandLeftChannel = (gsa_TxBandLeftChannelOSignature[s_bandIndex]>>gs_EocGroupSize);
            s_TxBandLeftChannelNxt = (gsa_TxBandLeftChannelOSignature[s_bandIndex+1]>>gs_EocGroupSize);
            s_TxBandRightChannel = (gsa_TxBandRightChannelOSignature[s_bandIndex]>>gs_EocGroupSize);
            s_TxBandRightChannelLast = ((gsa_TxBandRightChannelOSignature[gs_NumOfTxBandsOSignature-1])>>gs_EocGroupSize);
         }

         if(  (gt_FEData.us_StartIndex > s_TxBandRightChannel) && (gt_FEData.us_StartIndex < s_TxBandLeftChannelNxt)  )
            // for US0 only s_TxBandLeftChannelNxt is 0, so we don't enter this loop. This is ok, since we don't want to increment s_bandIndex (there is no next band)
         {
            s_bandIndex++;
            gt_FEData.us_StartIndex = s_TxBandLeftChannelNxt;
         } //Check if the start is above the last valid tone in-band

         if(gt_FEData.us_StartIndex > s_TxBandRightChannelLast)
         {
            //Move the Test Param to next and if SNR is read once, make repeated SNR reads
            s_endIx = -1; // reset index
            s_TestParam ++;
            s_bandIndex = 0;
            s_timeInitial = TIME_1_SECS;

            //Only Hlog is read with 18 indices as HLOG has
            //to pack 2 byte per index, so reading 36 byte per request
            //For QLN and SNR its 30 indices
            s_ReadLength = BRCM_TEST_PARAM_READ_COUNT;
            //XDSLRTFW-2316 (Start)
            if(s_TestParam > QLN_VECTOR_TEST_PARAM )
            {
               //Make sure we wait more than 10s after HLOG and QLN
               s_timeInitial = TIME_10_SECS; //10 seconds
               s_endIx_BRCM = -1; // reset index: This is to transfer Data to API
            }
            //XDSLRTFW-2316 (End)
            if(s_TestParam > SNR_VECTOR_TEST_PARAM )
            {
               //Flag to transfer data
               //not used as MONI 8 is disabled
               //ft_TransferData = 1;

               //For all consecutive SNR reads, space
               //requests every 10 seconds
               s_endIx_BRCM = -1; // reset index: This is to transfer Data to API
               s_timeInitial = s_FeParamsPollingTime; //TIME_10_SECS; //10 seconds
               s_TestParam = SNR_VECTOR_TEST_PARAM;
            }
            gt_FEData.us_StartIndex = gt_FEData.us_EndIndex;
            return;
         }
         gt_FEData.us_EndIndex   = gt_FEData.us_StartIndex + s_ReadLength;
         //Cap End Index to Band Limits
         if(s_TestParam >= SNR_VECTOR_TEST_PARAM)
         {
            s_TxBandLeftChannel = (gsa_TxBandLeftChannel[s_bandIndex]>>gs_EocGroupSize);
            s_TxBandLeftChannelNxt = (gsa_TxBandLeftChannel[s_bandIndex+1]>>gs_EocGroupSize);
            s_TxBandRightChannel = (gsa_TxBandRightChannel[s_bandIndex]>>gs_EocGroupSize);
            s_TxBandRightChannelLast = ((gsa_TxBandRightChannel[gs_NumOfTxBands-1])>>gs_EocGroupSize);
         }
         else
         {
            s_TxBandLeftChannel = (gsa_TxBandLeftChannelOSignature[s_bandIndex]>>gs_EocGroupSize);
            s_TxBandLeftChannelNxt = (gsa_TxBandLeftChannelOSignature[s_bandIndex+1]>>gs_EocGroupSize);
            s_TxBandRightChannel = (gsa_TxBandRightChannelOSignature[s_bandIndex]>>gs_EocGroupSize);
            s_TxBandRightChannelLast = ((gsa_TxBandRightChannelOSignature[gs_NumOfTxBandsOSignature-1])>>gs_EocGroupSize);
         }

         if (gt_FEData.us_EndIndex > s_TxBandRightChannelLast)
         {
            gt_FEData.us_EndIndex = s_TxBandRightChannelLast;
         }
         else if ( (gt_FEData.us_EndIndex > s_TxBandRightChannel) && (gt_FEData.us_EndIndex < s_TxBandLeftChannelNxt) )
            // for US0 only s_TxBandLeftChannelNxt is 0, so loop is not entered. Is ok since EndIndex is already correctly limited before
         {
            gt_FEData.us_EndIndex = s_TxBandRightChannel;
         }

      } //BRCM Vecndor ID

      //Check for Max Size
      if(gt_FEData.us_StartIndex > (EOC_MAX_SCG-1))
      {
         gt_FEData.us_StartIndex = (EOC_MAX_SCG-1);
      }

      //Check for Max Size
      if(gt_FEData.us_EndIndex > (EOC_MAX_SCG-1))
      {
         gt_FEData.us_EndIndex = (EOC_MAX_SCG-1);
      }

/*
      //Make sure we request all ways even number. As we pad octets
      //to store in SDRAM
      if((gt_FEData.us_EndIndex - gt_FEData.us_StartIndex + 1) & 0x1)
      {
         gt_FEData.us_EndIndex += 1;
      }
*/
      s_endIx = MIN(gt_FEData.us_EndIndex,
                    gt_FEData.us_StartIndex+EOC_NUM_SCGs_FEDATA-1);

      gt_FEData.us_EndIndex = s_endIx;
   }
   else
   {
      //In case of NACK from DSLAM,
      //CPE repeats the message twice
      //In case we do not see the response, next Test Param is requested
      //delayed by 2 Seconds
      if(gft_EocNack == 2)
      {
         //XDSLRTFW-2316 (Start)
         //Continue query till the end
         gft_EocNack = 0;
         //XDSLRTFW-2316 (End)
         return;
      }

   }

   //Block read request for IFX
   //if(gul_fe_G994VendorID == IFX_VENDOR_ID)
   if(gft_TestParam_Req == BLOCK_READ_CMD)
   {
      t_TxOvhdMsgInfoInput.uc_message_designator = PMD_TEST_PAR_RD_CMD_DESIG;
      t_TxOvhdMsgInfoInput.uc_Source = AUTO_CMD_SOURCE;
      t_TxOvhdMsgInfoInput.uc_message_type = guc_BLOCK_READ;
      t_TxOvhdMsgInfoInput.us_param_rd_start_carrier_index = gt_FEData.us_StartIndex;
      t_TxOvhdMsgInfoInput.us_param_rd_stop_carrier_index = gt_FEData.us_EndIndex;
      t_TxOvhdMsgInfoInput.ft_MECmd = 1;
   }
   else
   {
      //Vectored Block read
      switch(s_TestParam)
      {
      case HLOG_VECTOR_TEST_PARAM:
         t_TxOvhdMsgInfoInput.uc_message_designator = PMD_TEST_PAR_RD_CMD_DESIG;
         t_TxOvhdMsgInfoInput.uc_Source = AUTO_CMD_SOURCE;
         t_TxOvhdMsgInfoInput.uc_message_type = VECTOR_BLOCK_READ;
         t_TxOvhdMsgInfoInput.uc_TestParameterId = TEST_PARAM_HLOG_ID;
         t_TxOvhdMsgInfoInput.us_param_rd_start_carrier_index = gt_FEData.us_StartIndex;
         t_TxOvhdMsgInfoInput.us_param_rd_stop_carrier_index = gt_FEData.us_EndIndex;
         t_TxOvhdMsgInfoInput.ft_MECmd = 1;
         break;
      case QLN_VECTOR_TEST_PARAM:
         t_TxOvhdMsgInfoInput.uc_message_designator = PMD_TEST_PAR_RD_CMD_DESIG;
         t_TxOvhdMsgInfoInput.uc_Source = AUTO_CMD_SOURCE;
         t_TxOvhdMsgInfoInput.uc_message_type = VECTOR_BLOCK_READ;
         t_TxOvhdMsgInfoInput.uc_TestParameterId = TEST_PARAM_QLN_ID;
         t_TxOvhdMsgInfoInput.us_param_rd_start_carrier_index = gt_FEData.us_StartIndex;
         t_TxOvhdMsgInfoInput.us_param_rd_stop_carrier_index = gt_FEData.us_EndIndex;
         t_TxOvhdMsgInfoInput.ft_MECmd = 1;
         break;
      case SNR_VECTOR_TEST_PARAM:
      default:
         t_TxOvhdMsgInfoInput.uc_message_designator = PMD_TEST_PAR_RD_CMD_DESIG;
         t_TxOvhdMsgInfoInput.uc_Source = AUTO_CMD_SOURCE;
         t_TxOvhdMsgInfoInput.uc_message_type = VECTOR_BLOCK_READ;
         t_TxOvhdMsgInfoInput.uc_TestParameterId = TEST_PARAM_SNR_ID;
         t_TxOvhdMsgInfoInput.us_param_rd_start_carrier_index = gt_FEData.us_StartIndex;
         t_TxOvhdMsgInfoInput.us_param_rd_stop_carrier_index = gt_FEData.us_EndIndex;
         t_TxOvhdMsgInfoInput.ft_MECmd = 1;
         break;
      }
   } //IF(BRCM)

   txstat = TxEocSendRequest(2 /* priority */, t_TxOvhdMsgInfoInput);
   if (txstat == EOC_QUEUE_ADDED)
   {
      gft_EocME_FEBlockReadPending = 1;
      us_100msCnt = 0;
   }

}


/*^^^
 *------------------------------------------------------------------------
 *
 *  Name : EOC_Poll_CEocTrigger
 *
 *  Description: Polls MONI 9 to check for msg trigger.  Sends a Clear-EOC
 *               message containing the msg payload
 *
 *  Prototype:
 *       void EOC_Poll_CEocTrigger(void)
 *
 *  Input Arguments:
 *
 *  Output Arguments:
 *
 *  Return:
 *      None
 *
 *
 *  Notes:
 *------------------------------------------------------------------------
 *^^^
 */
void EOC_Poll_CEocTrigger(void)
{
#ifdef CLEAR_EOC
   TxOvhdMsgInfoRequest_t t_TxOvhdMsgInfoInput;
   int32 txstat;


   if (gft_SendCEocStatFlag || gft_SendCEocMsgFlag)
   {
      return;   // block until pending auto-msg is sent
   }

   if ((gt_MONI_CEocStat.us_TxStatus == EOC_CEOC_TXSTAT_TXERROR) || (gt_MONI_CEocStat.us_TxStatus == EOC_CEOC_TXSTAT_TXINPROGRESS))
   {
      return;   // In host-controlled mode, in case of TXERROR, FW will not clear TxStatus
   }
   // and block from sending another message.

   if (!gt_MONI_CEocTrigger.b_TxTrigger || !gt_INFO_CEocTxBuf.us_NumBytes)
   {
      gt_MONI_CEocStat.us_TxStatus = EOC_CEOC_TXSTAT_TXIDLE;
      return;
   }

   memset(&t_TxOvhdMsgInfoInput,0,sizeof(t_TxOvhdMsgInfoInput));

   t_TxOvhdMsgInfoInput.uc_message_designator = CLEAR_EOC_CMD_DESIG;
   t_TxOvhdMsgInfoInput.uc_message_type = CLEAR_EOC_MSG;
   t_TxOvhdMsgInfoInput.uc_Source = AUTO_CMD_SOURCE;
   t_TxOvhdMsgInfoInput.ft_MECmd = 1;

   if (gt_INFO_CEocTxBuf.us_NumBytes > MAX_ME_EOC_MSG_SIZE)
   {
      gt_INFO_CEocTxBuf.us_NumBytes = MAX_ME_EOC_MSG_SIZE;
   }

   txstat = TxEocSendRequest(1 /* priority */, t_TxOvhdMsgInfoInput);

   if (txstat == EOC_QUEUE_ADDED )
   {
      gt_MONI_CEocStat.us_TxStatus = EOC_CEOC_TXSTAT_TXINPROGRESS;
   }
#endif
}

/*^^^
 *------------------------------------------------------------------------
 *
 *  Name : EOC_Poll_CorruptCRCTrigger
 *
 *  Description:
 *
 *
 *  Prototype:
 *       void EOC_Poll_CorruptCRCTrigger(void)
 *
 *  Input Arguments:
 *
 *  Output Arguments:
 *
 *  Return:
 *      None
 *
 *
 *  Notes:
 *------------------------------------------------------------------------
 *^^^
 */
void EOC_Poll_CorruptCRCTrigger(void)
{
}

/*^^^
 *------------------------------------------------------------------------
 *
 *  Name : EOC_Poll_TimeSyncTrigger
 *
 *  Description:
 *
 *
 *  Prototype:
 *       void EOC_Poll_TimeSyncTrigger(void)
 *
 *  Input Arguments:
 *
 *  Output Arguments:
 *
 *  Return:
 *      None
 *
 *
 *  Notes:
 *------------------------------------------------------------------------
 *^^^
 */
void EOC_Poll_TimeSyncTrigger(void)
{
}

/*^^^
 *------------------------------------------------------------------------
 *
 *  Name : EOC_AutoMsg_Task_100ms
 *
 *  Description:  This subroutine sends Autonomous EOC msgs into Queue
 *
 *  Prototype:
 *      void EOC_AutoMsg_Task_100ms(void)
 *
 *  Input Arguments:
 *
 *  Output Arguments:
 *
 *  Return:
 *      None
 *
 *
 *  Notes:
 *------------------------------------------------------------------------
 *^^^
 */
void EOC_AutoMsg_Task_100ms(void)
{
   gt_G997_Cntrl.s_EOC_AutoMsg_100ms_State++;

   // execution spread over several frames to avoid FG MIPS peaks
   switch (gt_G997_Cntrl.s_EOC_AutoMsg_100ms_State)
   {
   case 2:
      if( OPTNArray[OPTN_FeParamControl] & OPTN_FeTestParamEnable)
      {
         EOC_Poll_FEReadTrigger();
      }
      break;

   case 3:
      EOC_Poll_CEocTrigger();
      break;

   case 4:
      EOC_Poll_CorruptCRCTrigger();
      break;

   case 5:
      EOC_Poll_TimeSyncTrigger();

      // reset state variable to stop state machine operation
      gt_G997_Cntrl.s_EOC_AutoMsg_100ms_State = 0;
      break;
   }
}


//*************************************************
//
#endif // ifdef INCLUDE_CACHE_FUNCTIONS
//
//*************************************************


