/* **COPYRIGHT******************************************************************
    INTEL CONFIDENTIAL
    Copyright (C) 2017 Intel Corporation
    Copyright (C), 1994-2004 Aware Inc. All Rights Reserved.
******************************************************************COPYRIGHT** */
/* **DISCLAIMER*****************************************************************
    The source code contained or described herein and all documents related
    to the source code ("Material") are owned by Intel Corporation or its
    suppliers or licensors. Title to the Material remains with Intel
    Corporation or its suppliers and licensors. The Material may contain
    trade secrets and proprietary and confidential information of Intel
    Corporation and its suppliers and licensors, and is protected by
    worldwide copyright and trade secret laws and treaty provisions. No part
    of the Material may be used, copied, reproduced, modified, published,
    uploaded, posted, transmitted, distributed, or disclosed in any way
    without Intel's prior express written permission.

    No license under any patent, copyright, trade secret or other
    intellectual property right is granted to or conferred upon you by
    disclosure or delivery of the Materials, either expressly, by
    implication, inducement, estoppel or otherwise. Any license under
    such intellectual property rights must be express and approved by
    Intel in writing.
*****************************************************************DISCLAIMER** */
/*
*-------------------------------------------------------------------------------
*
*   Aware DMT Technology. Proprietary and Confidential.
*
*   40 Middlesex Turnpike, Bedford, MA 01730-1413
*   Phone (781) 276 - 4000
*   Fax   (781) 276 - 4001
*
*   filename: rx_ib.c
*
*   This file contains functions that performs RX IB processing.
*
*-------------------------------------------------------------------------------
*/

// ***********************************************************************************************************
// rx_ib.c
//
// History
//
// 10/08/2018 Sriram Shastry:  XDSLRTFW-3906 : Rx-IB streaming is not correct in VDSL
// Debug stream sends out previous data instead of the latest data
//     Grep for XDSLRTFW-3906
//
// ***********************************************************************************************************

#include "common.h"
#include "gdata.h"
#include "g997.h"
#include "profile.h"

extern int16 gs_FeLprLosDelay;

/*
*-------------------------------------------------------------------------------
*
*   Prototype: void UpdateAtmPrimitive_FE(int16 s_lp, int16 s_abc)
*
*   This function updates Far-end ATM primitives.
*
*   Input Arguments:
*      s_lp: 0 = INLV/LP0, 1 = FAST/LP1
*      s_abc: ATM BC #, 0 = ABC0, 1 = ABC1
*
*   Output Arguments:
*
*   Returns:
*
*   Global Variables:
*      guca_RxIbMsgBuf[MAX_IB_BYTES_PER_SFRAME]: 3 IB bytes to be transmitted
*
*-------------------------------------------------------------------------------
*/

void UpdateAtmPrimitive_FE(int16 s_lp, int16 s_abc)
{

   /*************************************************************************************/
   //SYNC - OCD/NCD - LCD defect state machine
   /*************************************************************************************/
   //First update the NCD/ OCD anomaly condition of the far end.
   if (gt_RxIbData.uc_fncd_ind[s_lp] == PRESENT)
   {
      gt_RxIbData.uc_fncd_anom[s_lp] = PRESENT;
      gt_RxIbData.uc_focd_anom[s_lp] = PRESENT;
   }
   else
   {
      gt_RxIbData.uc_fncd_anom[s_lp] = TERMINATED;
      gt_RxIbData.uc_focd_anom[s_lp] = TERMINATED;
   }

   //Now start processing far end anomaly/ defect states.
   switch(gt_RxIbData.uc_atm_state[s_lp])
   {
   case SYNC:
      if (gt_RxIbData.uc_fncd_ind[s_lp] == PRESENT)    // Lost cell delineation
      {
         gt_RxIbData.s_fncd_cnt[s_lp] = 1;
         gt_RxIbData.uc_atm_state[s_lp] = OCD;   //Transition 1
         gt_RxIbData.uc_fncd_anom[s_lp] = PRESENT;
         gt_RxIbData.uc_focd_anom[s_lp] = PRESENT;
      }
      else
      {
         gt_RxIbData.s_fncd_cnt[s_lp] = 0;
         gt_RxIbData.uc_atm_state[s_lp] = SYNC;   //Transition 7
         gt_RxIbData.uc_fncd_anom[s_lp] = TERMINATED;
         gt_RxIbData.uc_focd_anom[s_lp] = TERMINATED;
         gt_RxIbData.uc_flcd_def[s_lp] = TERMINATED;
      }
      break;

   case OCD:
      if (gt_RxIbData.uc_fncd_ind[s_lp] == PRESENT)   //Transition 2
      {
         gt_RxIbData.s_fncd_cnt[s_lp]++;
      }
      else
      {
         // NCD and OCD are identical. NCD Anom is optional per G993 Section 10.5.2.1.1
         if (gt_RxIbData.s_fncd_cnt[s_lp] < LCD_SUPERFRAMES)
         {
            gt_RxIbData.s_fncd_cnt[s_lp] = 0;
            gt_RxIbData.uc_atm_state[s_lp] = SYNC;      //Transition 8
            gt_RxIbData.uc_focd_anom[s_lp] = TERMINATED;
            gt_RxIbData.uc_fncd_anom[s_lp] = TERMINATED;
         }
      }

      if (gt_RxIbData.s_fncd_cnt[s_lp] >= LCD_SUPERFRAMES)
      {
         // Cap the s_fncd_cnt at LCD_SUPERFRAMES
         gt_RxIbData.s_fncd_cnt[s_lp] = LCD_SUPERFRAMES;

         if (gt_RxIbData.uc_frdi_def == TERMINATED) //G993.1  Section 10.5.2.1.4
         {
            gt_RxIbData.uc_focd_anom[s_lp] = TERMINATED;   //Transition 3
            gt_RxIbData.uc_fncd_anom[s_lp] = TERMINATED;
            gt_RxIbData.uc_flcd_def[s_lp] = PRESENT;
            gt_RxIbData.uc_atm_state[s_lp] = LCD;
         }
         else
         {
            gt_RxIbData.uc_focd_anom[s_lp] = PRESENT;   //Transition 9
            gt_RxIbData.uc_fncd_anom[s_lp] = PRESENT;
            gt_RxIbData.uc_atm_state[s_lp] = OCD;
         }
      }
      break;

   case LCD:
      if (gt_RxIbData.uc_fncd_ind[s_lp] == TERMINATED)
      {
         if (gt_RxIbData.s_fncd_cnt[s_lp] > 0)
         {
            gt_RxIbData.s_fncd_cnt[s_lp]--;
         }

         if (gt_RxIbData.s_fncd_cnt[s_lp] == 0)      //Transition 6
         {
            gt_RxIbData.uc_flcd_def[s_lp] = TERMINATED;  //G993.1  Section 10.5.2.1.4
            gt_RxIbData.uc_atm_state[s_lp] = SYNC;
         }
         else
         {
            //Transition 4 - No action required.
         }
      }
      else
      {
         gt_RxIbData.uc_focd_anom[s_lp] = TERMINATED;   // because atm state cannot be
         gt_RxIbData.uc_fncd_anom[s_lp] = TERMINATED;   // in LCD and OCD (or NCD) at
         // the same time

         gt_RxIbData.s_fncd_cnt[s_lp]++;      //Transition 5
         if (gt_RxIbData.s_fncd_cnt[s_lp] >= LCD_SUPERFRAMES)
         {
            gt_RxIbData.s_fncd_cnt[s_lp] = LCD_SUPERFRAMES;
         }
      }
      break;
   }
}

/*
*-------------------------------------------------------------------------------
*
*   Prototype: void UpdateLineFailurePrimitive_FE()
*
*   This function updates Far-end ATM primitives.
*
*   Input Arguments:
*      s_lp: 0 = INLV/LP0, 1 = FAST/LP1
*      s_bc: ATM BC #, 0 = ABC0, 1 = ABC1
*
*   Output Arguments:
*
*   Returns:
*
*   Global Variables:
*      guca_RxIbMsgBuf[MAX_IB_BYTES_PER_SFRAME]: 3 IB bytes to be transmitted
*
*-------------------------------------------------------------------------------
*/

void UpdateLineFailurePrimitive_FE()
{
   int16 i, s_lp,s_bc, s_ib_cnt;

   /*************************************************************************************/
   // Counter Updates
   /*************************************************************************************/

   for (s_lp=0; s_lp<NUM_DATA_PATHS; s_lp++)
   {
      // get ATM BC # assigned to this latency path
      s_bc = gt_rx_TPS_Map.s_LPtoABC[s_lp];
   }
   /*************************************************************************************/
   // LOS processing
   /*************************************************************************************/
   gt_RxIbData.uc_flos_hist = ((gt_RxIbData.uc_flos_hist << 1) | gt_RxIbData.uc_flos_ind) & 0x3F;
   s_ib_cnt = 0;
   // count previous superframes with los indicated at far-end
   for (i=0; i<LOS_SUPERFRAMES; i++)
   {
      s_ib_cnt += (gt_RxIbData.uc_flos_hist >> i) & 0x01;
   }

   //G993.1  Section 10.5.1.4
   if (gt_RxIbData.uc_flos_def == PRESENT)
   {
      // far-end LOS defect terminates if less than 2 of the previous
      // 6 superframes indicate LOS
      if (s_ib_cnt < LOS_TERMINATE_SFRAMES)
      {
         gt_RxIbData.uc_flos_def = TERMINATED;

#ifdef DEBUG_TRAIL
         // Log LOS defect termination into ShowTime Events Debug Trail
         if (gt_debugTrailControl.s_ShowtimeEventControl & DEBUG_TRAIL_SHOW_EVENTS_FE_LOS_ENABLE)
         {
            DebugTrail1(3,DEBUG_TRAIL_SHOWTIME_EVENTS_ENABLE,0,
                        (int16)DEBUG_TRAIL_SHOW_EVENTS_FE_LOS_DEF_OFF,
                        (int16)gl_RxSymbolCount,
                        (int16)(gl_RxSymbolCount>>16));
         }
#endif // DEBUG_TRAIL
         //@todo, what is a useful value to print here? on Avinax they print power level
         DSH_SendEvent(DSH_EVT_LOS,1,&gt_RxIbData.uc_flos_def);
      }
      else
      {
         gt_RxIbData.uc_flos_def = PRESENT;
      }
   }
   else
   {
      // far-end LOS defect occurs when 4 or more of the previous
      // 6 superframes indicate LOS
      if (s_ib_cnt >= LOS_PRESENT_SFRAMES)
      {
         gt_RxIbData.uc_flos_def = PRESENT;

#ifdef DEBUG_TRAIL
         // Log LOS defect termination into ShowTime Events Debug Trail
         if (gt_debugTrailControl.s_ShowtimeEventControl & DEBUG_TRAIL_SHOW_EVENTS_FE_LOS_ENABLE)
         {
            DebugTrail1(3,DEBUG_TRAIL_SHOWTIME_EVENTS_ENABLE,0,
                        (int16)DEBUG_TRAIL_SHOW_EVENTS_FE_LOS_DEF_ON,
                        (int16)gl_RxSymbolCount,
                        (int16)(gl_RxSymbolCount>>16));
         }
#endif // DEBUG_TRAIL
         //@todo, what is a useful value to print here? on Avinax they print power level
         DSH_SendEvent(DSH_EVT_LOS,1,&gt_RxIbData.uc_flos_def);
      }
   }

   // G993.1  Section 10.5.1.6
   if ((gt_RxIbData.uc_flos_def==PRESENT) &&
         (gt_g997_FailureState_FE.t_LOF.us_NextState == FAILED_STATE))
   {
      gt_g997_FailureState_FE.t_LOS.us_NextState = FAILED_STATE;
      gs_g997_Failure_Flag_FE |= (gt_g997_FailureState_FE.t_LOS.us_FailureBitNumber);
   }

   /*************************************************************************************/
   // RDI processing
   /*************************************************************************************/
   if (gt_RxIbData.uc_frdi_def == PRESENT)
   {
      //G993.1  Section 10.5.1.6
      if ((gt_RxIbData.uc_flos_def == PRESENT) ||
            (gt_g997_FailureState_FE.t_LOS.us_NextState == FAILED_STATE))
      {
         // One of the few places where Transition 10 of the Failure State Machine
         // implemented in ProcessG997Failures() is encountered
         gt_g997_FailureState_FE.t_LOF.us_NextState = TRIGGERED_STATE;
         gt_g997_FailureState_FE.t_LOF.ul_Duration = 1;
      }
   }

   //G993.1  Section 10.5.1.6
   if (((gt_g997_FailureState_FE.t_LOS.us_NextState == FAILED_STATE) ||
         (gt_RxIbData.uc_flos_def == PRESENT)) &&
         (gt_g997_FailureState_FE.t_LOF.us_NextState == FAILED_STATE))
   {
      gt_g997_FailureState_FE.t_LOF.us_NextState = STEADY_STATE;
      gs_g997_Failure_Flag_FE &= ~(gt_g997_FailureState_FE.t_LOF.us_FailureBitNumber);
   }

   /*************************************************************************************/
   // FPO processing
   /*************************************************************************************/

   //G993.1  Section 10.5.3.2.

   /*************************************************************************************/
   // LPR processing
   /*************************************************************************************/

   //G993.1  Section 10.5.3.2
   if ((gt_g997_FailureState_FE.t_LOS.us_NextState == FAILED_STATE) &&
         (gt_g997_FailureState_FE.t_LPR.us_NextState != STEADY_STATE))
   {
      if (gt_TxIbData.uc_los_def == PRESENT)
      {
         gt_g997_FailureState_FE.t_LPR.us_NextState = FAILED_STATE;
      }
   }


   /*************************************************************************************/

}

/*
*-------------------------------------------------------------------------------
*
*   Prototype: void UpdateG997Counters(void)
*
*   This function updates Far-end ATM primitives.
*
*   Input Arguments:
*
*   Output Arguments:
*
*   Returns:
*
*   Global Variables:
*
*-------------------------------------------------------------------------------
*/
void UpdateG997Counters(void)
{
   int16 s_lp, s_bc;

   //Update the G997 Line counters, primitives and anomalies
   UpdateLineFailurePrimitive_FE();

   //Update G997 ATM counters and anomalies
   for (s_lp=0; s_lp<NUM_DATA_PATHS; s_lp++)
   {
      // First get ATM BC # assigned to this latency path
      s_bc = gt_rx_TPS_Map.s_LPtoABC[s_lp];
      // Now update the counters, primitives and anomalies
      if (s_bc != DISABLED_ABC)
      {
         UpdateAtmPrimitive_FE(s_lp, s_bc);
      }
   }

}

/*
*-------------------------------------------------------------------------------
*
*   Prototype: void RxIbHandler(void)
*
*   This function processes 3 octets of indicator bits read during
*   the previous superframe.
*
*   Input Arguments:
*
*   Output Arguments:
*
*   Returns:
*
*   Global Variables:
*      guca_RxIbMsgBuf[MAX_IB_BYTES_PER_SFRAME]: 3 IB bytes to be received
*
*-------------------------------------------------------------------------------
*/
void RxIbHandler(void)
{
   static uint8 uc_PrevRxIb = 0xE0;
   uint8 uc_temp;

   // If previous frame is clean (no CRC, LOS, SEF), then
   //  process the indicator bits from that frame
   if ((gt_TxIbData.uc_be_anom[0] == TERMINATED) &&
         (gt_TxIbData.uc_los_def == TERMINATED) &&
         (gt_TxIbData.uc_sef_def == TERMINATED))
   {
      // Process the received IB bits for VDSL2
      gt_RxIbData.uc_flos_ind = (uint8)(1-((guca_RxIbMsgBuf[0] >> 7) & 0x1));
      gt_RxIbData.uc_frdi_def = (uint8)(1-((guca_RxIbMsgBuf[0] >> 6) & 0x1));

      // the FE LPR "Dying Gasp" defect will be set based on the received indicator
      // bit.  The FE LPR Failure is defined as a FE LPR defect followed by a NE LOS Failure.
      // Since a valid FE LPR indicator will be quickly followed by an LOS, we set the FE LPR defect
      // here and clear it only if the NE LOS test is negative (i.e. the signal level is good).
      // This extends the FE LPR defect and allows the standard ProcessG997Failures function to process
      // the FE LPR Failure.
      if ( (guca_RxIbMsgBuf[0] & (1<<5)) == 0)
      {

         // gs_FeLprLosDelay is set to delay the LOS check
         // A negative LOS is used to clear the FE LPR, but the NE LOS processing
         // is slow relative relative to the overhead (i.e. IB) frame rate.  This counter
         // assures that the LOS associated with a
         // LPR will be registered before the condition is checked to clear the FE LPR.
         gs_FeLprLosDelay = 2;
         gt_RxIbData.uc_flpr_def = PRESENT;

      }

      gt_RxIbData.uc_fncd_ind[INLV] = (uint8)(1-((guca_RxIbMsgBuf[2] >> 7) & 0x1));
      gt_RxIbData.uc_fncd_ind[FAST] = (uint8)(1-((guca_RxIbMsgBuf[2] >> 3) & 0x1));

      uc_temp = (guca_RxIbMsgBuf[0] & 0xE0);
#ifdef DEBUG_TRAIL
      // If FE IB logging enabled and the IB bits change
      // then log the data
      if ((gt_debugTrailControl.s_ShowtimeEventControl & DEBUG_TRAIL_SHOW_EVENTS_FE_IB_CHANGE_ENABLE) &&
            (uc_temp != uc_PrevRxIb))
      {
         DebugTrail1(4,DEBUG_TRAIL_SHOWTIME_EVENTS_ENABLE,0,
                     (int16)DEBUG_TRAIL_SHOW_EVENTS_FE_IB_CHANGE,
                     (int16)gl_RxSymbolCount,
                     (int16)(gl_RxSymbolCount>>16),
                     (int16)((uc_PrevRxIb<<8) | uc_temp));
      }
#endif // DEBUG_TRAIL
      if (uc_temp != uc_PrevRxIb)
      {
         DSH_SendEvent(DSH_EVT_RX_IBIT_CHANGE,1,&uc_temp);  //XDSLRTFW-3906 Bug_DS_Rx_IB_Streaming_PollLatest_Data(Start_End)
      }
      uc_PrevRxIb = uc_temp;

   }   //   if ((gt_TxIbData.uc_be_anom[0] == TERMINATED) &&
   //      (gt_TxIbData.uc_los_def == TERMINATED) &&
   //      (gt_TxIbData.uc_sef_def == TERMINATED))


   //Update the G997 Line counters, primitives and anomalies
   UpdateG997Counters();


#ifdef DEBUG_ATM_FAILURES
   printIBStatus();
#endif // DEBUG_ATM_FAILURES

   // perform G997 superframe task
   G997_SuperFrame_Task();

   // clear flag to initiate RX IB processing
   gus_ShowtimeControl &= ~MASK_PROCESS_RX_IB;
}
