/* **COPYRIGHT******************************************************************
    INTEL CONFIDENTIAL
    Copyright (C) 2017 Intel Corporation
    Copyright (C), 1994-2004 Aware Inc. All Rights Reserved.
******************************************************************COPYRIGHT** */
/* **DISCLAIMER*****************************************************************
    The source code contained or described herein and all documents related
    to the source code ("Material") are owned by Intel Corporation or its
    suppliers or licensors. Title to the Material remains with Intel
    Corporation or its suppliers and licensors. The Material may contain
    trade secrets and proprietary and confidential information of Intel
    Corporation and its suppliers and licensors, and is protected by
    worldwide copyright and trade secret laws and treaty provisions. No part
    of the Material may be used, copied, reproduced, modified, published,
    uploaded, posted, transmitted, distributed, or disclosed in any way
    without Intel's prior express written permission.

    No license under any patent, copyright, trade secret or other
    intellectual property right is granted to or conferred upon you by
    disclosure or delivery of the Materials, either expressly, by
    implication, inducement, estoppel or otherwise. Any license under
    such intellectual property rights must be express and approved by
    Intel in writing.
*****************************************************************DISCLAIMER** */
/*
*-------------------------------------------------------------------------------
*
*   Aware DMT Technology. Proprietary and Confidential.
*
*   40 Middlesex Turnpike, Bedford, MA 01730-1413
*   Phone (781) 276 - 4000
*   Fax   (781) 276 - 4001
*
*   filename: g997.c
*
*   This file contains functions that implement G997 functionality.
*
*-------------------------------------------------------------------------------
*/

// ***********************************************************************************************************
// g997.c
//
// History
//
// 24/12/2012 Ram: Redefining FECS,ES,SES based on new definiton of CRC in ReTx mode of operation
//                 Refer Sec 11.3.1, 11.3.3 of G.998.4 (G.Inp/ReTx spec)
//                 Grep for 'XDSLRTFW-419: BugFix_DS_VDSL2_ALL_Redfine_FECS_ES_SES_In_ReTx_Mode'
// 09/05/2013 Kannan: Observed link drop in showtime at 17Mhz profile against Avinax when ReTx was enabled.
//                This issue was observed when data rate step size is changed from 4Kbps to 1Kbps.
//                We need to consdier LSW & MSW to get complete data rate. Now the data rate is in the
//                steps of 1kbps, hence need to multiply with 1000 to get data rate in bps.
//                Grep for "XDSLRTFW-773 XDSLRTFW-914: BugFix_DS_ALL_ALL_RetxMode17MhzLinkdrop"
// 20/06/2013 Anantha Ramu: EFTR calculated for reporting to API, in CMV CNTR 43.
//                          Grep for XDSLRTFW-974 Fix_DS_VDSL2_All_Reporting_ReTx_Counters
//
// 24/07/2013 Ram: Incorporated review comments from Stefan. Look at JIRA 1098 for review comments.
//                 Rates reported in CNTR 43 (e.g. EFTR) should be in Kbps, modified the code accordingly.
//                 Since "RateMSW" and "RateLSW" are now in uint16 format, probable bug (due to sign extension
//                 when bit 15 is set to 1) during computation of NDR (ul_Temp variable), ul_EFTR etc is avoided.
//                 Grep for XDSLRTFW-1098: BugFix_DS_VDSL2_ALL_SES_Not_Incrementing_In_ReTx
// 08/11/2013 Venkatesh : The new Flag ReInitTrigger will be set only if the
//                 Re-Init duration is exceeded. The Flag is used as Re-Init and Failure
//                 Indication period is different now
//                 Grep : XDSLRTFW-955
// 29/01/2014 Abu: XDSLRTFW-1535 CMV Controlled Link Drop
//                 Grep : XDSLRTFW-1535 CMV_Controlled_Link_Drop
//
// 17/10/2014 Palaksha
//    Implementation of "XDSLRTFW-1211: VR9 RETX, Minimum error-free throughput (EFTR_min) extension according to G.998.4 Corrigendum 4"
//    grep for XDSLRTFW-1211_Enh_All_VDSL2_EFTRmin_Corr4Ginp
//
// 20/12/2018 Abu Rahman
//            XDSLRTFW-4051 Fast downstream up shift SRA
//            This code supports Fast downstream up shift SRA
//            Maximum bits loaded/tone = 5
//            This feature is enabled by DTAG bit (CMV DSL 21 0 1) only against vectoring and BRCM based DSLAM.
//
//            Some other controls are done by CNFG 93 cmv
//            CNFG 93 0:
//             bit 0:
//                - value 0x8000: disable feature "Fast downstream up shift SRA"
//                - value 0x0000: enable  feature "Fast downstream up shift SRA" for BRCM CO and any  vector engine
//                - value 0x0001: enable  feature "Fast downstream up shift SRA" for BRCM CO and BRCM vector engine

//            CNFG 93 1:
//             - value 0xFFFF: do not disable feature "Fast downstream up shift SRA" in showtime
//             - value      X: disable feature "Fast downstream up shift SRA" after X sec of Showtime
//                             (supported range: 0-32767 sec, debug range 0-65534 sec)
//            CNFG 93 2:
//             - value 0xFFFF: Never disable DS-SRAs in Showtime
//             - value      Y: Disable DS SRAs if CPE receive 3 consecutive SRA rejections with rejection code SRA "not enabled"
//                             (0x3) after Y sec of Showtime (supported range: 0-32767 sec, debug range 0-65534 sec)
//            Example:
//            DSL 21[0] : 1 -DTAG bit enabled
//            CNFG 93[0]: 0 - meaning feature is enabled for all kind of VCE
//            CNFG 93[1]: 100 - Fast DS up shift SRA is enabled for 100 sec in Showtime
//            CNFG 93[2]: 200 - After 200 sec of Showtime if three consecutive SRA rejection come with SRA not enabled
//                              exception code (0x3) then DS SRA request will be disabled
//            Default CMV configuration: if DTAG bit is enabled
//            CNFG 93[0] = 0
//            CNFG 93[1] = 255
//            CNFG 93[2] = 255
//            Grep for XDSLRTFW-4005

// ************************************************************************************************************

#include <string.h>
#include "common.h"
#include "gdata.h"
#include "g997.h"
#include "eoc.h"
#include "profile.h"
#include "CustomerTasks.h"


#include "cmv_Data.h"

#include "BitswapDecision_VDSL2.h"

extern FlagT gft_ReInitTrigger;

#ifdef INCLUDE_CACHE_FUNCTIONS
/*
*-------------------------------------------------------------------------------
*
*   Prototype:
*  void UpdateSecCounter_running_inhibited(int32* pl_cnt, void* pl_cnt_curr_reading, int32* pl_cnt_last_10_sec, uint8 reset, uint8 ft_changed_from_unavail_to_avail)
*
*   This function increments counters inhibited by UAS and SES, such as CRC, FEC counters.
*
*
*   Input Arguments:
*   pl_cnt: pointer to the error counter to be incremented;
*   pl_cnt_curr_reading: pointer to a list of variables containing number of errors happened in the current second;
*   pl_cnt_last_10_sec: pointer to a list of variables containing total number of errors happend in the last 10 seconds;
*   reset: indicate whether pl_cnt_last_10_sec is going to be reset or be modified;
*   ft_changed_from_unavail_to_avail: a variable indicates whether the UAS status has been changed from Unavailable to Available in this second;
*
*
*   Output Arguments:
*   Returns:
*
*   Global Variables:
*   gt_g997_LinePerf_Sec_NE.l_SES_L_count; SES count in this second
*   gt_g997_LinePerf_help.s_UAS_L_State;   UAS status in this second;
*
*  Notes: Initially the second parameter was of type int32 but to suppress/remove a warning
 *         it has been changed to type void. And suitable changes are made in the function
 *         and wherever it is being employed.
*-------------------------------------------------------------------------------
*/

void UpdateSecCounter_running_inhibited(int32* pl_cnt, int32* pl_cnt_curr_reading, int32* pl_cnt_last_10_sec, uint8 reset, uint8 ft_changed_from_unavail_to_avail)
{
   if(gt_g997_LinePerf_Sec_NE.l_SES_L_count==0) // no SES is declared in this second.
   {
      //compute the CRCs happened in this second


      if (gt_g997_LinePerf_help.s_UAS_L_State == UAS_STATE_LINE_AVAIL) // CRC counts in this second should be added to the CRC counter
      {

         if(ft_changed_from_unavail_to_avail)
         {
            // upon this status change the CRCs in last 10 seconds should be added to CRC counter too.
            //   for(i=0;i<n_cnt;i++)
            {
               *(pl_cnt) += *(pl_cnt_last_10_sec);

               if(reset)
               {
                  *(pl_cnt_last_10_sec)=0;
               }
            }

         }

         //   for(i=0;i<n_cnt;i++)
         {
            *(pl_cnt) += *(pl_cnt_curr_reading);

         }

      }
      else  // if UAS== UNAVAIL and SES==0
      {
         // CRCs in this second should be added to the last 10 seconds counter, but not to the CRC counter yet
         if(reset)
         {
            *(pl_cnt_last_10_sec) += *(pl_cnt_curr_reading);
         }

      }


   }
   else
   {
      // SES declared in this second;
      if (gt_g997_LinePerf_help.s_UAS_L_State == UAS_STATE_LINE_UNAVAIL) //unavail and ses=1, reset 10 sec CRC buffer.
      {
         //   for(i=0;i<n_cnt;i++)
         {
            if(reset)
            {
               *(pl_cnt_last_10_sec)=0;
            }
         }
      }

   }

//   for(i=0;i<n_cnt;i++)
//      *(pl_cnt_curr_reading[i])=0;
}

/*
*-------------------------------------------------------------------------------
*
*   Prototype: void UpdateSecCounter(int16 *ps_cnt, int16 *ps_SecCnt) and
*            void UpdateSecCounter_running(int16 *ps_cnt, int16 *ps_SecCnt)

*   The UpdateSecCounter_running function increments ps_SecCnt by 1 if ps_cnt != 0.
*   The UpdateSecCounter_running  does the same but also resets ps_cnt.
*
*   Input Arguments:
*
*   Output Arguments:
*
*   Returns:
*
*   Global Variables:
*
*-------------------------------------------------------------------------------
*/

void UpdateSecCounter_running(int32 *ps_cnt, int32 *ps_SecCnt)
{
   if (*ps_cnt)
   {
      (*ps_SecCnt)++;
   }
}


void UpdateSecCounter(int32 *ps_cnt, int32 *ps_SecCnt)
{
   if (*ps_cnt)
   {
      (*ps_SecCnt)++;
   }
   *ps_cnt = 0;
}

/*
*-------------------------------------------------------------------------------
*
*   Prototype:
*  void UpdateSecCounter_running_delayed(int32 *pl_cnt, int32 *pl_SecCnt, void* delay_buf)
*
*   This function increments SES or ES counter with a 10-second delay.
*
*
*   Input Arguments:
*   ps_cnt:   error counter of the current second;
*   ps_SecCnt: pointer to the counter to be incremented;
*   delay_buf: pointer to the variable which records the 10 seconds history of the counter to be incremented.
*
*
*   Output Arguments:
*   Returns:
*
*   Global Variables:
*
*  Notes: Initially the last parameter was of type uint16 but to suppress/remove a warning
 *        it has been changed to type void. And suitable changes are made in the function
 *        and wherever it is being employed.
*-------------------------------------------------------------------------------
*/

void UpdateSecCounter_running_delayed(int32 *pl_cnt, int32 *pl_SecCnt, uint16* delay_buf)
{
   // increment the counter after 10 seconds delay
   if ((*delay_buf)&0x0200)
   {
      pl_SecCnt[0]++;

   }

   // record the error of current second
   if (*pl_cnt)
   {
      (*delay_buf)|=0x0001;
   }


   // shift the delay buf
   (*delay_buf)=(*delay_buf)<<1;
}

/*
*-------------------------------------------------------------------------------
*
*   Prototype: void SetBit(int32 s_in, int32 s_thres, int32 *ps_res, int32 *ps_i)
*
*   This function sets the ps_i bit in ps_res if s_in > s_thres,
*   then increment (*ps_i) by 1.
*
*   Input Arguments:
*
*   Output Arguments:
*
*   Returns:
*
*   Global Variables:
*
*-------------------------------------------------------------------------------
*/

void SetBit(int32 s_in, int32 s_thres, int32 *ps_res, int32 *ps_i)
{
   if (s_in > s_thres)
   {
      (*ps_res) |= 1 << (*ps_i);
   }
   (*ps_i)++;
}

// XDSRTFW-442: Feature_All_All_All_All_Vdsl2Sra [Start]
void ResetSRAStateMachine(void)
{
   gl_SRATotalLp_DeltaCnt = 0;
   gl_SRATotalLp_Delta = 0;

   gs_NoOfDsSraAfterSnrUpdate = 0; //XDSLRTFW-4051

   guc_SRAState = SRA_INIT;

}

/*
*-------------------------------------------------------------------------------
*
*   Prototype: void MonitorMarginForSRA(void)
*
*   This function monitors the average margin in 1 sec intervals and sets the
*   initiate SRA flag if margin is below (or above) upshift (or downshift) SRA
*   threshold value for a fixed interval of time.
*
*   Input Arguments: None
*
*   Output Arguments: None
*
*   Returns: None
*
*   Global Variables: gft_USRAInitiate, gft_DSRAInitiate, gs_UTIME_Count,
*                     gs_DTIME_Count
*-------------------------------------------------------------------------------
*/
void MonitorMarginForSRA(void)
{
   int16 ShiftSNRM;


   // Check if average SNR margin is greater than the UPSHIFT margin
   ShiftSNRM = (int16)gt_RxAutoSRA_Ctrl.us_RA_USNRM;
   if(gt_LineStatusDS.s_SnrMargin > ShiftSNRM)
   {
      gs_UTIME_Count++;         //count number of seconds upshift condition is true

      // Clear out any old downshift margin triggers if set, since margin just got better
      gft_DSRAInitiate = FALSE;
   }
   else
   {
      //reset counter if SNR margin falls below upshift margin threshold
      gs_UTIME_Count = 0;

      // Clear out any old upshift margin triggers if set, since margin is now below the threshold
      // Note: It is not allowed to switch off the SRA, when the real upshift margin is achieved!
      //       At this point the TR115 requirement must be considered, i.e. - 1 dB from the upshift snr margin.
      //       Actually the FW uses guc_SraShiftOffset = SRA_TR115_SHIFT_OFFSET_1DOT3DB!
      if ((gft_USRAInitiate == TRUE) && (guc_SraShiftControl & (SHIFT_TO_SHIFT_THRESH|SHIFT_TO_PROTECT_IMPROVE)))
      {
         ShiftSNRM -= guc_SraShiftOffset;
      }
      else
      {
         ShiftSNRM = gt_SnrMgnConfig.s_TARSNRMds;
      }
//      if ((gt_LineStatusDS.s_SnrMargin <= ShiftSNRM) || ((gl_SRATotalLp_Delta > 0) && (gus_BitloadErrorCode == CHGBAT_RETCODE_CANT_CHANGE_LP4)))
      if (gt_LineStatusDS.s_SnrMargin <= ShiftSNRM)
      {
         gft_USRAInitiate = FALSE;
      }
   }


   // Check if average SNR margin is less than the DOWNSHIFT margin
   ShiftSNRM = (int16)gt_RxAutoSRA_Ctrl.us_RA_DSNRM;
   if(gt_LineStatusDS.s_SnrMargin < ShiftSNRM)
   {
      gs_DTIME_Count++;         //count number of seconds downshift condition is true

      // Clear out any old upshift margin triggers if set, since margin just got too bad
      gft_USRAInitiate = FALSE;
   }
   else
   {
      //reset counter if SNR margin rises above downshift margin in any frame
      gs_DTIME_Count = 0;

      // Clear out any old upshift margin triggers if set, since margin is now below the threshold
      // Note: It is not allowed to switch off the SRA, when the real downshift margin is achieved!
      //       At this point the TR115 requirement must be considered, i.e. + 1 dB from the downshift snr margin.
      //       Actually the FW uses guc_SraShiftOffset = SRA_TR115_SHIFT_OFFSET_1DOT3DB!
      if ((gft_DSRAInitiate == TRUE) && (guc_SraShiftControl & (SHIFT_TO_SHIFT_THRESH|SHIFT_TO_RATE_IMPROVE)))
      {
         ShiftSNRM += guc_SraShiftOffset;
      }
      else
      {
         ShiftSNRM = gt_SnrMgnConfig.s_TARSNRMds;
      }
      if (gt_LineStatusDS.s_SnrMargin >= ShiftSNRM)
      {
         gft_DSRAInitiate = FALSE;
      }
   }


   // Turn off SNR margin based new triggers, if
   // - an old SRA task is in progress (and still valid, i.e. gft_USRAInitiate = 1, gft_DSRAInitiate = 1) or
   // - the SNR margin was not updated since last SRA completed (i.e. gft_SNRUpdated_PostSRA = 0)
   if ((gft_EnableRxAutoSRA) && (!(gft_USRAInitiate || gft_DSRAInitiate || (!gft_SNRUpdated_PostSRA))))
   {
      // Initiate upshift or downshift Auto SRA
      if(gs_UTIME_Count > gt_RxAutoSRA_Ctrl.us_RA_UTIME)
      {
         gft_USRAInitiate = TRUE;
         // Try initiate an SRA only if there is still room to increase further.
         // Note: This is for IFEC mode only!
         if ((gt_ReTXParams.uc_OMSG1_DsReTxEnabled != RETX_SELECTED) && (gt_rx_config_v2.ul_Lp[gs_data_path] >= gul_AutoSRALmax))
         {
            gft_USRAInitiate = FALSE;
         }
      }
      else if(gs_DTIME_Count > gt_RxAutoSRA_Ctrl.us_RA_DTIME)
      {
         gft_DSRAInitiate = TRUE;
         // Try initiate an SRA only if there is still room to decrease further.
         // Note: This is for IFEC mode only!
         if ((gt_ReTXParams.uc_OMSG1_DsReTxEnabled != RETX_SELECTED) && (gt_rx_config_v2.ul_Lp[gs_data_path] <= gul_AutoSRALmin))
         {
            gft_DSRAInitiate = FALSE;
         }
      }

      // Reset SRA state machine for a fresh start
      ResetSRAStateMachine();
   }

}
// XDSRTFW-442: Feature_All_All_All_All_Vdsl2Sra [End]

//XDSLRTFW-2509 Start
/*****************************************************************************
;  Prototype:
;     void MonitorPllFreqOffset(void);
;
;  Description:
;       Captures gl_pll_freq_offset of PLL every seconds (upto 32 seconds)
;         The array gsa_pll_freq_offset_ppm_q4 holds 32 second data (in 12.4 q format)
;  Input Arguments:  None
;  Output Arguments: None
;  Global Variables: none
;
;*************************************************************************************/

void MonitorPllFreqOffset(void)
{
   //gl_pll_freq_offset:             14.18 ppm (e.g. 1ppm: 2^18 = 262144)
   //gsa_pll_freq_offset_ppm_q4 to be in 12.4 format
   // 1ppm gsa_pll_freq_offset_ppm_q4/16 will give values in terms of PPM
   gsa_pll_freq_offset_ppm_q4[gs_PllRunningIndex++] =  (int16) (gl_pll_freq_offset >> 14); //12.4
   //Roll over the index to ZERO if 32
   if (gs_PllRunningIndex == 32)
   {
      gs_PllRunningIndex =0;
   }

}
//XDSLRTFW-2509 End

/*
*-------------------------------------------------------------------------------
*
*   Prototype: void G997_1sec_Task(void)
*
*   This function performs task which has to be done every second.
*   Task mainly updates the System Performance counter.
*
*   Input Arguments:
*
*   Output Arguments:
*
*   Returns:
*
*   Global Variables:
*
*-------------------------------------------------------------------------------
*/
#define DEFAULT_ESE_THRESHOLD (10)
void G997_1sec_Task(void)
{
   int32 s_count, i, crc_lp0, crc_lp1, fec_lp0, fec_lp1, max_crc, max_fec;
   static uint8 uc_ShowtimeCountersTimer=0;
   int16 s_temp_delay_buf;
   int16 s_SESx_threshold = DEFAULT_ESE_THRESHOLD;
   uint32 ul_Temp;
   uint32 ul_ETR;
   uint32 ul_EFTR = 0;
   int16 s_lp = 0;

//XDSLRTFW-974 Fix_DS_VDSL2_All_Reporting_ReTx_Counters (Start)
   //uint32 temp;
//XDSLRTFW-974 Fix_DS_VDSL2_All_Reporting_ReTx_Counters (End)
   //XDSLRTFW-2509 Start_End
   MonitorPllFreqOffset();
// XDSRTFW-442: Feature_All_All_All_All_Vdsl2Sra [Start]
   if(gft_EnableRxAutoSRA) // Monitoring SRA margin is not needed while SRA is disabled
   {
      MonitorMarginForSRA();
   }

// XDSRTFW-442: Feature_All_All_All_All_Vdsl2Sra [End]




   /*********************************************************************************
      //Update Near End SES counters
   *********************************************************************************/

   // preprocessing for SES: LOS || SEF || CRC > CRCS_TO_DECLARE_SES
   if (gt_g997_LinePerf_help.l_1Sec_CRC8_L_count >= (CRCS_TO_DECLARE_SES<<8))
   {
      gt_g997_LinePerf_Sec_NE.l_SES_L_count++;   //G997.1 Section 7.2.1.1.3
   }


   //--------------------------------------------------------------------------------
   // showtime - RETX stats
   //--------------------------------------------------------------------------------
   if (gt_ReTXParams.uc_OMSG1_DsReTxEnabled == RETX_SELECTED)
   {


      //XDSLRTFW-1098: BugFix_DS_VDSL2_ALL_SES_Not_Incrementing_In_ReTx (Start)
      // calculate ERTR
      //XDSLRTFW-773 XDSLRTFW-914: BugFix_DS_ALL_ALL_RetxMode17MhzLinkdrop (Start)
      //Need to consdier LSW & MSW to get complete data rate and the data rate is in the steps of 1kbps, hence
      //need to multiply with 1000 to get data rate in bps.
      ul_Temp = ((t_DSRateInfo.t_CurRate[LP1].us_RateMSW << 16) | t_DSRateInfo.t_CurRate[LP1].us_RateLSW);
      //XDSLRTFW-773 XDSLRTFW-914: BugFix_DS_ALL_ALL_RetxMode17MhzLinkdrop(End)

      if (gt_ReTXStats.us_New_EFTRMin_Avail == FALSE)
      {
         //gt_ReTXStats.us_New_EFTRMin_Avail == FALSE means observation period starts
         //EFTRmin = NDR on Every start of observation period
         if ((gus_DebugVar1 & 1) == 0)
         {
            gt_ReTXStats.ul_EFTR_min = ul_Temp;
         }
      }

      //ul_Temp is in Kbps
      ul_EFTR = ((ul_Temp*1000) - gt_ReTXStats.ul_ErroredBitsPerSec)/1000; //ul_EFTR should be in Kbps

//XDSLRTFW-974 Fix_DS_VDSL2_All_Reporting_ReTx_Counters (Start)
      //EFTR reported to API = ceil(EFTR) in kbps
      //temp = ul_EFTR + 999;
      //gt_ReTXStats.ul_EFTR_API = temp/1000; //EFTR reported to API in units of kbps

      gt_ReTXStats.ul_EFTR = ul_EFTR; //ul_EFTR is in kbps
//XDSLRTFW-974 Fix_DS_VDSL2_All_Reporting_ReTx_Counters (End)

      gt_ReTXStats.ul_ErroredBitsPerSec = 0;

      //XDSLRTFW-1909 (Start)
      // note that t_ReTXConfigCMV.us_OTPS_DS_LEFTR_thresh is defined as the "percent" of NDR
      // Since we are multiplying the received value (i.e. us_OTPS_DS_LEFTR_thresh) by 10 in
      // func InterpOTpsMsg_VDSL2() (to handle the special case of "leftr_thresh == 0"),
      // we will divide by 1000 here.
      ul_Temp = (ul_Temp*gt_ReTXParams.t_ReTXConfigCMV.us_OTPS_DS_LEFTR_thresh)/1000;
      //XDSLRTFW-1909 (End)

      // note that ul_ReTX_ReceiverETR is in Kbps
      ul_ETR = MAX(gt_ReTXStats.ul_ReTX_ReceiverETR >> 1, ul_Temp);

      if (ul_EFTR < ul_ETR)
      {
         gt_ReTXStats.ul_LEFTR_Cntr++;
      }

      //XDSLRTFW-419: BugFix_DS_VDSL2_ALL_Redfine_FECS_ES_SES_In_ReTx_Mode (Start)
      // Sec 11.3.3 of G.998.4: seftr occurs when EFTR < ETR/2
      // note that ul_ReTX_ReceiverETR is in Kbps
      if (ul_EFTR < (gt_ReTXStats.ul_ReTX_ReceiverETR >> 1))
      {
         // SES is declared if, during 1 sec interval, there is one or more seftr
         // in near-end receive direction.
         gt_g997_LinePerf_Sec_NE.l_SES_L_count++;   //G997.1 Section 7.2.1.1.3
         guc_seftrPrecFollowing = 2;      //XDSLRTFW-1211_Enh_All_VDSL2_EFTRmin_Corr4Ginp (Start_End)
      }
      else
      {
         guc_EftrLtETRby2OverObsPeriod = FALSE;
      }

      //XDSLRTFW-419: BugFix_DS_VDSL2_ALL_Redfine_FECS_ES_SES_In_ReTx_Mode (End)
      //XDSLRTFW-1098: BugFix_DS_VDSL2_ALL_SES_Not_Incrementing_In_ReTx (End)
   }
//XDSLRTFW-2094/2050
   /*********************************************************************************
       //Update Near End OOS defect : 3rd Consecutive second
   *********************************************************************************/
   if (gt_ReTXParams.uc_OMSG1_DsReTxEnabled == RETX_SELECTED)
      s_lp = 1;
   else
      s_lp = 0;

   //If TC-SYNC LOSS is seen in this Second, Count this anomaly
   if( gt_TxIbData.uc_cd_flag[s_lp] == FALSE ) {
      gt_TxIbData.uc_oos_def = PRESENT;
   } else if( gt_TxIbData.uc_cd_flag[s_lp] == TRUE ) {
      gt_TxIbData.uc_oos_def = TERMINATED;
   }
//XDSLRTFW-956 : ENH_DS_ALL_ALL_SESx_ESx_LOMx
   /*********************************************************************************
      //Update Near End ES defect
   *********************************************************************************/
   if (gt_g997_LinePerf_Sec_NE.l_ES_L_count)
   {
      gt_TxIbData.uc_es_def = PRESENT;
   }
   else
   {
      gt_TxIbData.uc_es_def = TERMINATED;
   }

//XDSLRTFW-956 : ENH_DS_ALL_ALL_SESx_ESx_LOMx
   /*********************************************************************************
      //Update Near End ESE defect
   *********************************************************************************/
   if (gt_g997_LinePerf_Sec_NE.l_SES_L_count)
   {
      gt_TxIbData.uc_ese_def = PRESENT;
   }
   else
   {
      gt_TxIbData.uc_ese_def = TERMINATED;
   }

   //orrigendum 4 of  G.INP (G.998.4 )
   //performance monitoring parameter minimum error-free throughput (EFTR_min) calculation, excluding the following seconds:
   //(i)   Seconds in which the values of EFTR are less than ETR/2;  guc_seftrPrecFollowing = 2; ensures this
   //(ii)  Seconds in which EFTR is not defined;
   //(iii)  the single second preceding a second with seftr defect;  guc_seftrPrecFollowing = 2; ensures this
   //(iv)  the single second following a second with seftr defect. guc_seftrPrecFollowing = 2; ensures this

   if (guc_seftrPrecFollowing != 0)
   {
      guc_seftrPrecFollowing--;
      //gt_ReTXStats.us_New_EFTRMin_Avail = FALSE;
   }
   else
   {
      if (gt_ReTXStats.ul_EFTR_min > ul_EFTR)
      {
         gt_ReTXStats.ul_EFTR_min = ul_EFTR;
      }

      gt_ReTXStats.us_New_EFTRMin_Avail = TRUE;
      guc_seftrPrecFollowing = 0;
   }


   /*********************************************************************************
      //Update Near End UAS counters
   *********************************************************************************/

   gft_changed_from_unavail_to_avail=0;
   gft_changed_from_avail_to_unavail=0;
   //gt_G997_Cntrl.s_LinkState==G997_LINK_AVAIL;
   // If line currently available: Check to see if we need to declare line unavailable.
   if (gt_g997_LinePerf_help.s_UAS_L_State == UAS_STATE_LINE_AVAIL)
   {
      // Increment the SES count if LINK is unavailable or if SES count is nonzero
      // Else reset the count
      if ((gt_g997_LinePerf_Sec_NE.l_LOSS_L_count>0) || (gt_g997_LinePerf_Sec_NE.l_SES_L_count !=0))
      {
         gt_g997_LinePerf_help.s_UAS_L_TriggerCnt++;
      }
      else
      {
         gt_g997_LinePerf_help.s_UAS_L_TriggerCnt = 0;
      }

      //XDSLRTFW-956 : ENH_DS_ALL_ALL_SESx_ESx_LOMx
      // In case of 10 contiguous SES_L count
      //If SESx criteria is set for ReInit, use this threshold
      if(gsa_MONI_ShowFailure_NE[1] & 0x1)
      {
         s_SESx_threshold = gt_ReInit_Threshold.s_SESx_ReInit_Threshold;
      }
      //XDSLRTFW-956 : ENH_DS_ALL_ALL_SESx_ESx_LOMx

      if (gt_g997_LinePerf_help.s_UAS_L_TriggerCnt >= s_SESx_threshold)
      {
         // Include the last 10 SES-L in the UAS time
//         gt_g997_LinePerf_Current_Previous_TR1_NE.t_Current.l_UAS_L_count += gt_g997_LinePerf_help.s_UAS_L_TriggerCnt;
//         gt_g997_LinePerf_Running_NE.l_UAS_L_count += gt_g997_LinePerf_help.s_UAS_L_TriggerCnt;

         // Subtract the last 10 SES seconds from the SES cnt as they are accounted for as
         // line unavailable seconds

         //The following two lines are taken out to implement delayed count.
         //gt_g997_LinePerf_Current_Previous_TR1_NE.t_Current.l_SES_L_count -= 10;
         //gt_g997_LinePerf_Running_NE.l_SES_L_count -= 10;

         gft_changed_from_avail_to_unavail=1;

         // Reset the count as we switch to Unavailable state
         gt_g997_LinePerf_help.s_UAS_L_State = UAS_STATE_LINE_UNAVAIL;
         gt_g997_LinePerf_help.s_UAS_L_TriggerCnt = 0;
      }
   }
   // If line currently unavailable: Check to see if we need to declare line available.
   else
   {

      if (gt_g997_LinePerf_help.s_UAS_L_State == UAS_STATE_LINE_UNAVAIL)
      {

         //XDSLRTFW-981
         // Increment the line unavailable seconds while the line is declared as unavailable
         // Update the Running UAS Counter in UNAVAIL state till this moves to AVAIL state
         // In this case the SES and ES increments would not be seen at API due to Delayed 10Sec

         gt_g997_LinePerf_Current_Previous_TR1_NE.t_Current.l_UAS_L_count ++;
         gt_g997_LinePerf_Running_NE.l_UAS_L_count ++ ;

         // Reset the trigger count in case of SES seconds or link not available
         if ((gt_g997_LinePerf_Sec_NE.l_LOSS_L_count>0) || (gt_g997_LinePerf_Sec_NE.l_SES_L_count!=0))
         {
            gt_g997_LinePerf_help.s_UAS_L_TriggerCnt = 0;
         }
         else
         {
            gt_g997_LinePerf_help.s_UAS_L_TriggerCnt++;
         }

         // If 10 consecutive no- SES seconds, declare line back as available
         if (gt_g997_LinePerf_help.s_UAS_L_TriggerCnt >= 10)
         {
            // Subtract the last 10 no-SES seconds from the line unavailable seconds count
            //   gt_g997_LinePerf_Current_Previous_TR1_NE.t_Current.l_UAS_L_count -= gt_g997_LinePerf_help.s_UAS_L_TriggerCnt;

            //   gt_g997_LinePerf_Running_NE.l_UAS_L_count -= gt_g997_LinePerf_help.s_UAS_L_TriggerCnt;

            // Declare the line as available
            // Reset the trigger count as state is changed
            gt_g997_LinePerf_help.s_UAS_L_State = UAS_STATE_LINE_AVAIL;
            gt_g997_LinePerf_help.s_UAS_L_TriggerCnt = 0;
            gft_changed_from_unavail_to_avail=1;
         }
      }
   }

   /*********************************************************************************
      //Update Far End UAS counters
   *********************************************************************************/

   // run customer-specific 1 sec tasks
   Customer_1sec_Task();

   /*********************************************************************************
   //Update CRC, FEC, FEC uncorrectable, FECS counters which are inhibited by UAS and SES.
   *********************************************************************************/
//======= CRC counters ====================================

   if(guc_EnableInhibition)
   {


      UpdateSecCounter_running_inhibited(&gt_g997_ChPerf_CRC_TR1_Curr_Prev_NE.t_Current.la_CV_C_count[0], &gt_CRC_Curr_Reading.la_CV_C_count[0], &gt_CRC_Last_10_Sec.la_CV_C_count[0], 0, gft_changed_from_unavail_to_avail);
      UpdateSecCounter_running_inhibited(&gt_g997_ChPerf_CRC_Running_NE.la_CV_C_count[0], &gt_CRC_Curr_Reading.la_CV_C_count[0], &gt_CRC_Last_10_Sec.la_CV_C_count[0], 1, gft_changed_from_unavail_to_avail);
      gt_CRC_Curr_Reading.la_CV_C_count[0]=0;
      UpdateSecCounter_running_inhibited(&gt_g997_ChPerf_CRC_TR1_Curr_Prev_NE.t_Current.la_CV_C_count[1], &gt_CRC_Curr_Reading.la_CV_C_count[1], &gt_CRC_Last_10_Sec.la_CV_C_count[1], 0, gft_changed_from_unavail_to_avail);
      UpdateSecCounter_running_inhibited(&gt_g997_ChPerf_CRC_Running_NE.la_CV_C_count[1], &gt_CRC_Curr_Reading.la_CV_C_count[1], &gt_CRC_Last_10_Sec.la_CV_C_count[1], 1, gft_changed_from_unavail_to_avail);
      gt_CRC_Curr_Reading.la_CV_C_count[1]=0;

      //============= FEC counters =============================


      UpdateSecCounter_running_inhibited(&gt_g997_ChPerf_FEC_TR1_Curr_Prev_NE.t_Current.la_FEC_C_count[0],(int32 *)(void *) &gul_FEC_Curr_Reading[0],(int32 *)(void *) &gul_FEC_Last_10_Sec[0], 0, gft_changed_from_unavail_to_avail);
      UpdateSecCounter_running_inhibited(&gt_g997_ChPerf_FEC_Running_NE.la_FEC_C_count[0],(int32 *)(void *) &gul_FEC_Curr_Reading[0],(int32 *)(void *) &gul_FEC_Last_10_Sec[0], 1, gft_changed_from_unavail_to_avail);
      gul_FEC_Curr_Reading[0]=0;
      UpdateSecCounter_running_inhibited(&gt_g997_ChPerf_FEC_TR1_Curr_Prev_NE.t_Current.la_FEC_C_count[1],(int32 *)(void *) &gul_FEC_Curr_Reading[1],(int32 *)(void *) &gul_FEC_Last_10_Sec[1], 0, gft_changed_from_unavail_to_avail);
      UpdateSecCounter_running_inhibited(&gt_g997_ChPerf_FEC_Running_NE.la_FEC_C_count[1],(int32 *)(void *) &gul_FEC_Curr_Reading[1],(int32 *)(void *) &gul_FEC_Last_10_Sec[1], 1, gft_changed_from_unavail_to_avail);
      gul_FEC_Curr_Reading[1]=0;


//============== FEC uncorrectable CW cunter =========================


      UpdateSecCounter_running_inhibited(&gt_g997_ChPerf_FEC_TR1_Curr_Prev_NE.t_Current.la_FECV_C_count[0],(int32 *)(void *) &gul_FEC_Uncorr_Curr_Reading[0],(int32 *)(void *) &gul_FEC_Uncorr_Last_10_Sec[0], 0, gft_changed_from_unavail_to_avail);
      UpdateSecCounter_running_inhibited(&gt_g997_ChPerf_FEC_Running_NE.la_FECV_C_count[0],(int32 *)(void *) &gul_FEC_Uncorr_Curr_Reading[0],(int32 *)(void *) &gul_FEC_Uncorr_Last_10_Sec[0], 1, gft_changed_from_unavail_to_avail);
      gul_FEC_Uncorr_Curr_Reading[0]=0;
      UpdateSecCounter_running_inhibited(&gt_g997_ChPerf_FEC_TR1_Curr_Prev_NE.t_Current.la_FECV_C_count[1],(int32 *)(void *) &gul_FEC_Uncorr_Curr_Reading[1],(int32 *)(void *) &gul_FEC_Uncorr_Last_10_Sec[1], 0, gft_changed_from_unavail_to_avail);
      UpdateSecCounter_running_inhibited(&gt_g997_ChPerf_FEC_Running_NE.la_FECV_C_count[1],(int32 *)(void *) &gul_FEC_Uncorr_Curr_Reading[1],(int32 *)(void *) &gul_FEC_Uncorr_Last_10_Sec[1], 1, gft_changed_from_unavail_to_avail);
      gul_FEC_Uncorr_Curr_Reading[1]=0;


//========= FECS counter ==============================
      if(gul_FECS_Curr_Reading!=0)
      {
         gul_FECS_Curr_Reading=1;
      }

      UpdateSecCounter_running_inhibited(&gt_g997_LinePerf_Running_NE.l_FECS_L_count,(int32 *)(void *) &gul_FECS_Curr_Reading,(int32 *)(void *) &gul_FECS_Last_10_Sec, 0, gft_changed_from_unavail_to_avail);
      UpdateSecCounter_running_inhibited(&gt_g997_LinePerf_Current_Previous_TR1_NE.t_Current.l_FECS_L_count,(int32 *)(void *) &gul_FECS_Curr_Reading,(int32 *)(void *) &gul_FECS_Last_10_Sec, 1, gft_changed_from_unavail_to_avail);
      gul_FECS_Curr_Reading=0;

   }
   /*********************************************************************************
      //Update Near and Far End LOSS/ FECS/ ES/ SES and UAS running counters
   *********************************************************************************/
   if(guc_EnableInhibition)
   {
      // Update the Running UAS Counter in UNAVAIL state till this moves to AVAIL state
      // In this case the SES and ES increments would not be seen at API due to Delayed 10Sec
      if(gt_g997_LinePerf_help.s_UAS_L_State == UAS_STATE_LINE_UNAVAIL)
      {
         gt_g997_LinePerf_Sec_NE.l_SES_L_count=0;
         gt_g997_LinePerf_Sec_NE.l_ES_L_count=0;
      }



      if(gft_changed_from_avail_to_unavail)
      {
         gs_SES_Delay_Buf=0;
         gs_ES_Delay_Buf=0;
      }
   }

   if(!guc_EnableInhibition)
   {
      UpdateSecCounter_running(&gt_g997_LinePerf_Sec_NE.l_FECS_L_count, &gt_g997_LinePerf_Running_NE.l_FECS_L_count);
      UpdateSecCounter_running(&gt_g997_LinePerf_Sec_NE.l_ES_L_count, &gt_g997_LinePerf_Running_NE.l_ES_L_count);
      UpdateSecCounter_running(&gt_g997_LinePerf_Sec_NE.l_SES_L_count, &gt_g997_LinePerf_Running_NE.l_SES_L_count);
   }
   else
   {
      s_temp_delay_buf=gs_SES_Delay_Buf;
      UpdateSecCounter_running_delayed(&gt_g997_LinePerf_Sec_NE.l_SES_L_count, &gt_g997_LinePerf_Running_NE.l_SES_L_count,&gs_SES_Delay_Buf);
      UpdateSecCounter_running_delayed(&gt_g997_LinePerf_Sec_NE.l_SES_L_count, &gt_g997_LinePerf_Current_Previous_TR1_NE.t_Current.l_SES_L_count,(uint16 *)(void *) &s_temp_delay_buf);

      //Reset after all update is done
      gt_g997_LinePerf_Sec_NE.l_SES_L_count=0;

      s_temp_delay_buf=gs_ES_Delay_Buf;
      UpdateSecCounter_running_delayed(&gt_g997_LinePerf_Sec_NE.l_ES_L_count, &gt_g997_LinePerf_Running_NE.l_ES_L_count, &gs_ES_Delay_Buf);
      UpdateSecCounter_running_delayed(&gt_g997_LinePerf_Sec_NE.l_ES_L_count, &gt_g997_LinePerf_Current_Previous_TR1_NE.t_Current.l_ES_L_count,(uint16 *)(void *) &s_temp_delay_buf);

      //Reset after all update is done
      gt_g997_LinePerf_Sec_NE.l_ES_L_count=0;
   }



   /*********************************************************************************
      //Update Near End LOSS/ FECS/ ES/ SES counters
   *********************************************************************************/

   // increment second counter if at least on indication was present
   UpdateSecCounter_running(&gt_g997_LinePerf_Sec_NE.l_LOSS_L_count, &gt_g997_LinePerf_Running_NE.l_LOSS_L_count);
   UpdateSecCounter(&gt_g997_LinePerf_Sec_NE.l_LOSS_L_count, &gt_g997_LinePerf_Current_Previous_TR1_NE.t_Current.l_LOSS_L_count);

   //clear this auxiliary LOSS counter every second after being used.
   gt_g997_LinePerf_Sec_NE.l_LOSS_L_count=0;

   if(!guc_EnableInhibition)
   {
      UpdateSecCounter(&gt_g997_LinePerf_Sec_NE.l_FECS_L_count, &gt_g997_LinePerf_Current_Previous_TR1_NE.t_Current.l_FECS_L_count);
      UpdateSecCounter(&gt_g997_LinePerf_Sec_NE.l_ES_L_count, &gt_g997_LinePerf_Current_Previous_TR1_NE.t_Current.l_ES_L_count);
      UpdateSecCounter(&gt_g997_LinePerf_Sec_NE.l_SES_L_count, &gt_g997_LinePerf_Current_Previous_TR1_NE.t_Current.l_SES_L_count);
   }

   // Update the counter to count the first 10 seconds in showtime
   if (gt_g997_LinePerf_help.s_10secCnt != -1)
   {
      gt_g997_LinePerf_help.s_10secCnt++;

      //XDSLRTFW-956 : ENH_DS_ALL_ALL_SESx_ESx_LOMx
      // Count the first 10 seconds in showtime without LOS/LOF
      if ((gsa_g997_Failure_Flag_NE[0] & 0x3) == 0)
      {
         gs_first10secs_in_Showtime++;
      }
      else
      {
         gs_first10secs_in_Showtime = 0;
      }

   }


   // Check if the first 10 seconds are LOS/LOF free
   if ((gt_g997_LinePerf_help.s_10secCnt == gs_first10secs_in_Showtime) &&
         (gt_g997_LinePerf_help.s_10secCnt == 10))
   {
      // If the first 10 seconds in showtime are error free
      // set the flag to send an autonomous message to the host
      gft_ErrorFree_10sInShowtime = TRUE;

      // Reset the count for the current link
      gs_first10secs_in_Showtime = -1;
      gt_g997_LinePerf_help.s_10secCnt = -1;

   }

   /*********************************************************************************
   * Update and track allowed (configured) time in sec to disable the Fast downstream
   * upshift SRA
   * gt_FastDsSraConfig.us_TimeAfterShowtimeToDisableFastSra = -1 : Fast downstream
   * up shift feature will never be disabled
   *********************************************************************************/

   // XDSLRTFW-4051 Fast downstream up shift SRA(start)
   if((gus_FastDsSraControl & FAST_DS_UPSHIFT_SRA_ENABLED) && (gt_FastDsSraConfig.us_TimeAfterShowtimeToDisableFastSra != 0xFFFF))
   {
      if (gt_FastDsSraConfig.us_TimeAfterShowtimeToDisableFastSra == 0)
      {
         gus_FastDsSraControl &= (~(uint16)(FAST_DS_UPSHIFT_SRA_ENABLED));
      }
      else
      {
         gt_FastDsSraConfig.us_TimeAfterShowtimeToDisableFastSra--;
      }
   }

   if ((gus_FastDsSraControl & FAST_DS_SRA_USED_IN_PRESENT_LINK) &&
       ((gt_FastDsSraConfig.us_TimeAfterShowtimeToCheckForRejection_3 != 0) && (gt_FastDsSraConfig.us_TimeAfterShowtimeToCheckForRejection_3 != 0xFFFF)) )
   {
      gt_FastDsSraConfig.us_TimeAfterShowtimeToCheckForRejection_3--;
   }
   // XDSLRTFW-4051 Fast downstream up shift SRA(End)



   /*********************************************************************************
      //Update Far End LOSS/ FECS/ ES/ SES counters
   *********************************************************************************/

   gt_g997_LinePerf_help.l_1Sec_CRC8_L_count = 0;

   // Set the alarm indicators, this is slightly from above because the order of bits is
   // different

   // NE Alarm Indicators
   i = 0;
   s_count = 0;
   crc_lp0 = gt_g997_ChPerf_CRC_TR1_Curr_Prev_NE.t_Current.la_CV_C_count[0];
   crc_lp1 = gt_g997_ChPerf_CRC_TR1_Curr_Prev_NE.t_Current.la_CV_C_count[1];
   fec_lp0 = gt_g997_ChPerf_FEC_TR1_Curr_Prev_NE.t_Current.la_FEC_C_count[0];
   fec_lp1 = gt_g997_ChPerf_FEC_TR1_Curr_Prev_NE.t_Current.la_FEC_C_count[1];
   max_crc = (crc_lp0 > crc_lp1 ? crc_lp0 : crc_lp1);
   max_fec = (fec_lp0 > fec_lp1 ? fec_lp0 : fec_lp1);
   SetBit(gt_g997_LinePerf_Current_Previous_TR1_NE.t_Current.l_FECS_L_count, (int32)gt_g997_LinePerfThresholdControl.t_NE_Alarm.l_FECS_L_count, &s_count, &i);
   SetBit(gt_g997_LinePerf_Current_Previous_TR1_NE.t_Current.l_ES_L_count, (int32)gt_g997_LinePerfThresholdControl.t_NE_Alarm.l_ES_L_count, &s_count, &i);
   SetBit(gt_g997_LinePerf_Current_Previous_TR1_NE.t_Current.l_SES_L_count, (int32)gt_g997_LinePerfThresholdControl.t_NE_Alarm.l_SES_L_count, &s_count, &i);
   SetBit(gt_g997_LinePerf_Current_Previous_TR1_NE.t_Current.l_LOSS_L_count, (int32)gt_g997_LinePerfThresholdControl.t_NE_Alarm.l_LOSS_L_count, &s_count, &i);
   SetBit(gt_g997_LinePerf_Current_Previous_TR1_NE.t_Current.l_UAS_L_count, (int32)gt_g997_LinePerfThresholdControl.t_NE_Alarm.l_UAS_L_count, &s_count, &i);
   SetBit(max_crc, gt_g997_LinePerfThresholdControl.t_NE_Alarm.l_CRC_L_count, &s_count, &i);
   SetBit(max_fec, gt_g997_LinePerfThresholdControl.t_NE_Alarm.l_FEC_L_count, &s_count, &i);
   gt_G997_TR1AlarmIndicators.us_NEAlarmIndicators = s_count;

   // FE Alarm Indicators
   i = 0;
   s_count = 0;
   crc_lp0 = gt_g997_ChPerf_CRC_TR1_Curr_Prev_FE.t_Current.la_CV_CFE_count[0];
   crc_lp1 = gt_g997_ChPerf_CRC_TR1_Curr_Prev_FE.t_Current.la_CV_CFE_count[1];
   fec_lp0 = gt_g997_ChPerf_FEC_TR1_Curr_Prev_FE.t_Current.la_FEC_CFE_count[0];
   fec_lp1 = gt_g997_ChPerf_FEC_TR1_Curr_Prev_FE.t_Current.la_FEC_CFE_count[1];
   max_crc = (crc_lp0 > crc_lp1 ? crc_lp0 : crc_lp1);   // either LP with CRCs over threshold
   max_fec = (fec_lp0 > fec_lp1 ? fec_lp0 : fec_lp1);   // either LP with FECs over threshold
   SetBit(gt_g997_LinePerf_Current_Previous_TR1_FE.t_Current.l_FECS_LFE_count, (int32)gt_g997_LinePerfThresholdControl.t_FE_Alarm.l_FECS_LFE_count, &s_count, &i);
   SetBit(gt_g997_LinePerf_Current_Previous_TR1_FE.t_Current.l_ES_LFE_count, (int32)gt_g997_LinePerfThresholdControl.t_FE_Alarm.l_ES_LFE_count, &s_count, &i);
   SetBit(gt_g997_LinePerf_Current_Previous_TR1_FE.t_Current.l_SES_LFE_count, (int32)gt_g997_LinePerfThresholdControl.t_FE_Alarm.l_SES_LFE_count, &s_count, &i);
   SetBit(gt_g997_LinePerf_Current_Previous_TR1_FE.t_Current.l_LOSS_LFE_count, (int32)gt_g997_LinePerfThresholdControl.t_FE_Alarm.l_LOSS_LFE_count, &s_count, &i);
   SetBit(gt_g997_LinePerf_Current_Previous_TR1_FE.t_Current.l_UAS_LFE_count, (int32)gt_g997_LinePerfThresholdControl.t_FE_Alarm.l_UAS_LFE_count, &s_count, &i);
   SetBit(max_crc, gt_g997_LinePerfThresholdControl.t_FE_Alarm.l_CRC_LFE_count, &s_count, &i);
   SetBit(max_fec, gt_g997_LinePerfThresholdControl.t_FE_Alarm.l_FEC_LFE_count, &s_count, &i);
   gt_G997_TR1AlarmIndicators.us_FEAlarmIndicators = s_count;


   uc_ShowtimeCountersTimer++;
#ifdef DEBUG_TRAIL

   //--------------------------------------------------------------------------------
   // Debug Trail - Showtime Counters
   //--------------------------------------------------------------------------------

   if (uc_ShowtimeCountersTimer == (gt_debugTrailControl.s_ShowtimeCounterPeriod & 0x00FF))
   {
      uc_ShowtimeCountersTimer = 0;
      DebugTrail1(3,0x0000,DEBUG_TRAIL_SHOWTIME_CNTRS_ENABLE,
                  (int16)0xCCCC,
                  (int16)gl_RxSymbolCount,
                  (int16)(gl_RxSymbolCount >> 16));

      if (gt_debugTrailControl.s_ShowtimeCounterControl & DEBUG_TRAIL_SHOW_CNTRS_NE_CRC_ENABLE)
      {
         DebugTrail1(3,0x0000,DEBUG_TRAIL_SHOWTIME_CNTRS_ENABLE,
                     (int16)DEBUG_TRAIL_SHOW_CNTRS_NE_CRC,
                     (int16)(gt_g997_ChPerf_CRC_TR1_Curr_Prev_NE.t_Current.la_CV_C_count[0]),
                     (int16)(gt_g997_ChPerf_CRC_TR1_Curr_Prev_NE.t_Current.la_CV_C_count[0] >> 16));
      }

      if (gt_debugTrailControl.s_ShowtimeCounterControl & DEBUG_TRAIL_SHOW_CNTRS_FE_CRC_ENABLE)
      {
         DebugTrail1(3,0x0000,DEBUG_TRAIL_SHOWTIME_CNTRS_ENABLE,
                     (int16)DEBUG_TRAIL_SHOW_CNTRS_FE_CRC,
                     (int16)(gt_g997_ChPerf_CRC_TR1_Curr_Prev_FE.t_Current.la_CV_CFE_count[0]),
                     (int16)(gt_g997_ChPerf_CRC_TR1_Curr_Prev_FE.t_Current.la_CV_CFE_count[0] >> 16));
      }


      if (gt_debugTrailControl.s_ShowtimeCounterControl & DEBUG_TRAIL_SHOW_CNTRS_NE_FEC_ENABLE)
      {
         DebugTrail1(3,0x0000,DEBUG_TRAIL_SHOWTIME_CNTRS_ENABLE,
                     (int16)DEBUG_TRAIL_SHOW_CNTRS_NE_FEC,
                     (int16)(gt_g997_ChPerf_FEC_TR1_Curr_Prev_NE.t_Current.la_FEC_C_count[0]),
                     (int16)(gt_g997_ChPerf_FEC_TR1_Curr_Prev_NE.t_Current.la_FEC_C_count[0] >> 16));
      }

      if (gt_debugTrailControl.s_ShowtimeCounterControl & DEBUG_TRAIL_SHOW_CNTRS_FE_FEC_ENABLE)
      {
         DebugTrail1(3,0x0000,DEBUG_TRAIL_SHOWTIME_CNTRS_ENABLE,
                     (int16)DEBUG_TRAIL_SHOW_CNTRS_FE_FEC,
                     (int16)(gt_g997_ChPerf_FEC_TR1_Curr_Prev_FE.t_Current.la_FEC_CFE_count[0]),
                     (int16)(gt_g997_ChPerf_FEC_TR1_Curr_Prev_FE.t_Current.la_FEC_CFE_count[0] >> 16));
      }

      if (gt_debugTrailControl.s_ShowtimeCounterControl & DEBUG_TRAIL_SHOW_CNTRS_NE_SNRM_ENABLE)
      {
         DebugTrail1(7,0x0000,DEBUG_TRAIL_SHOWTIME_CNTRS_ENABLE,
                     (int16)DEBUG_TRAIL_SHOW_CNTRS_NE_SNRM,
                     gt_LineStatusDS.s_SnrMargin,
                     gt_AttenSnrMPerRxBand.t_AttenSnrM[0].s_SNRM,
                     gt_AttenSnrMPerRxBand.t_AttenSnrM[1].s_SNRM,
                     gt_AttenSnrMPerRxBand.t_AttenSnrM[2].s_SNRM,
                     gt_AttenSnrMPerRxBand.t_AttenSnrM[3].s_SNRM,
                     gt_AttenSnrMPerRxBand.t_AttenSnrM[4].s_SNRM);
      }

      if (gt_debugTrailControl.s_ShowtimeCounterControl & DEBUG_TRAIL_SHOW_CNTRS_FE_SNRM_ENABLE)
      {
         DebugTrail1(7,0x0000,DEBUG_TRAIL_SHOWTIME_CNTRS_ENABLE,
                     (int16)DEBUG_TRAIL_SHOW_CNTRS_FE_SNRM,
                     gt_LineStatusUS.s_SnrMargin,
                     gt_AttenSnrMPerRxBand_FE.t_AttenSnrM[0].s_SNRM,
                     gt_AttenSnrMPerRxBand_FE.t_AttenSnrM[1].s_SNRM,
                     gt_AttenSnrMPerRxBand_FE.t_AttenSnrM[2].s_SNRM,
                     gt_AttenSnrMPerRxBand_FE.t_AttenSnrM[3].s_SNRM,
                     gt_AttenSnrMPerRxBand_FE.t_AttenSnrM[4].s_SNRM);
      }


   }
#endif // DEBUG_TRAIL
}

// XDSLRTFW-1535 CMV_Controlled_Link_Drop (start)
/*
*-------------------------------------------------------------------------------
*
*   Prototype:   void CmvControlledLinkDrop(void)
*
*  This function is responsible for cmv controlled showtime link drop by making
*  either Tx gain to zero or Rx gain to zero.
*
*   CMV (test 42 0) contains two information
*   15 LSB bits : configures/indicates the time in second when Tx or Rx gain will set to zero
*   MSB :  if 1 --> function makes Tx gain to zero after x second mentioned in 15 LSB bits
*         if 0 --> function makes Rx gain to zero after x second mentioned in 15 LSB bits
*
*   Input Arguments: cmv test 42 0
*
*   Output Arguments: none
*
*   Returns: none
*
*   Global Variables: none
*
*-------------------------------------------------------------------------------
*/


void CmvControlledLinkDrop(void)
{
   uint16 s_TempCnt;
   uint16 s_TempSign;


   s_TempCnt = TESTArray[TEST_CMVCTRL_LINKDROP] & 0x7FFF;
   s_TempSign = TESTArray[TEST_CMVCTRL_LINKDROP] & 0x8000;

   if (s_TempCnt != 0)
   {
      if(s_TempCnt > 1)
      {
         s_TempCnt--;
      }

      if (s_TempCnt == 1) // time to link drop
      {
         if (s_TempSign)
         {
            // Set Tx variable gain to 0 to shut off signal
            WriteCoreReg((uint32)V_TX_VARGAIN_ADDR, (uint32)0);
            WriteCoreReg((uint32)V_TX_AVARGAIN_ADDR, (uint32)0);
         }
         else
         {
            // Set Rx variable gain to zero
            WriteCoreReg((uint32)V_RX_VARGAIN_ADDR, (uint32)0);
            WriteCoreReg((uint32)V_RX_AVARGAIN_ADDR, (uint32)0);
         }
      }
      TESTArray[TEST_CMVCTRL_LINKDROP] = s_TempSign  | s_TempCnt; // reconstract the CMV for further use in next second
   }

   return;
}

// XDSLRTFW-1535 CMV_Controlled_Link_Drop (end)



void G997_1sec_Task_VDSL2(void)
{
   G997_1sec_Task();

   // Check for TR1 Alarms
   G997_CheckTR1FailureChange();

   // XDSLRTFW-1535 CMV_Controlled_Link_Drop (start)
   // if(gft_DSVectoringEnabled)
   CmvControlledLinkDrop();
   // XDSLRTFW-1535 CMV_Controlled_Link_Drop (end)

   gs_G997_1sec_Task_Done_Flag = TRAINING_DONE;
}

#endif  // INCLUDE_CACHE_FUNCTIONS





#ifdef INCLUDE_NON_CACHE_FUNCTIONS
/*
*-------------------------------------------------------------------------------
*
*   Prototype: void G997_SuperFrame_Task(void)
*
*   This function performs task which have to be done once per superframe.
*   The reason is the width of the hardware counter.
*
*   Input Arguments:
*
*   Output Arguments:
*
*   Returns:
*
*   Global Variables:
*
*-------------------------------------------------------------------------------
*/

void G997_SuperFrame_Task(void)
{
   G997_CheckRxFailure();
   // missing implementation of G997_CheckUsMargin()
   G997_CheckUsMargin();

   if (gt_G997_Cntrl.s_InitialCrc_HoldOff > 0)
      // CRC count hold off is not implemented for now
   {
      gt_G997_Cntrl.s_InitialCrc_HoldOff--;
   }
}

/*
*-------------------------------------------------------------------------------
*
*   Prototype: void G997_FailureCounterUpdate(int16* ps_count, FlagT ft_present)
*
*   This function updates a failure counter once per superframe.
*
*   Input Arguments:
*      ps_count: 16-bit counter address
*      ft_present: Flag for anomaly or defect
*
*   Output Arguments:
*      ps_count: 16-bit counter address
*
*   Returns:
*
*   Global Variables:
*
*-------------------------------------------------------------------------------
*/

void G997_FailureCounterUpdate(int16* ps_count, FlagT ft_present)
{
   int16 s_count;

   s_count = *ps_count;

   // if defect or anomaly present, decrement counter and check for overflow
   if (!ft_present)
   {
      if (s_count > 0)
      {
         s_count = 0;
      }
      s_count--;
      if (s_count <= -32768)
      {
         s_count=0;
      }
   }
   else
   {
      // increment the counter and check for overflow
      if (s_count < 0)
      {
         s_count = 0;
      }
      s_count++;
      if (s_count >= 32767)
      {
         s_count=0;
      }
   }
   *ps_count = s_count;
}



/*
*-------------------------------------------------------------------------------
*
*   Prototype: void G997_CheckRxFailure(void)
*
*   This function sets s_CurFailure by comparing the failure count
*   with the thresholds.
*
*   Input Arguments:
*
*   Output Arguments:
*
*   Returns:
*
*   Global Variables:
*
*-------------------------------------------------------------------------------
*/

void G997_CheckRxFailure(void)
{
   int16  s_BC, s_lp;

   // declare failure if defect lasts more than 2.5 sec (def counter > ActiveThreshold)
   // reset failure if defect doesn't exist for more than 10 sec  (-def counter > InactiveThreshold)

   // ATM Failure:
   for (s_lp=0; s_lp<NUM_DATA_PATHS; s_lp++)
   {
      s_BC = gt_rx_TPS_Map.s_LPtoABC[s_lp];
      if (s_BC != DISABLED_ABC)
      {
         ProcessG997Failures(&gt_g997_AtmFailure_FE.t_LCD[s_BC],gt_RxIbData.uc_flcd_def[s_lp],
                             &gs_g997_Failure_Flag_FE,gs_MONI_ShowFailure_FE,gs_PMDFramesPerRxIBStructure);
         ProcessG997Failures(&gt_g997_AtmFailure_FE.t_NCD[s_BC],gt_RxIbData.uc_fncd_anom[s_lp],
                             &gs_g997_Failure_Flag_FE,gs_MONI_ShowFailure_FE,gs_PMDFramesPerRxIBStructure);

         //Failure State in the Latency Path LPx needs to be reflected back into the Bearer Channel BCx failure flag
         //that is hooked to PLAM CMV 1. Only two latency paths are supported.
         if (gs_g997_Failure_Flag_FE & NCD_LP0_FAILURE_BIT)
         {
            if (s_BC==0)
            {
               gs_g997_Failure_Flag_FE |= NCD_BC0_FAILURE_BIT;
            }
            else
            {
               gs_g997_Failure_Flag_FE |= NCD_BC1_FAILURE_BIT;
            }
         }
         else
         {
            if (s_BC==0)
            {
               gs_g997_Failure_Flag_FE &= ~NCD_BC0_FAILURE_BIT;
            }
            else
            {
               gs_g997_Failure_Flag_FE &= ~NCD_BC1_FAILURE_BIT;
            }
         }

         if (gs_g997_Failure_Flag_FE & NCD_LP1_FAILURE_BIT)
         {
            if (s_BC==0)
            {
               gs_g997_Failure_Flag_FE |= NCD_BC0_FAILURE_BIT;
            }
            else
            {
               gs_g997_Failure_Flag_FE |= NCD_BC1_FAILURE_BIT;
            }
         }
         else
         {
            if (s_BC==0)
            {
               gs_g997_Failure_Flag_FE &= ~NCD_BC0_FAILURE_BIT;
            }
            else
            {
               gs_g997_Failure_Flag_FE &= ~NCD_BC1_FAILURE_BIT;
            }
         }

         if (gs_g997_Failure_Flag_FE & LCD_LP0_FAILURE_BIT)
         {
            if (s_BC==0)
            {
               gs_g997_Failure_Flag_FE |= LCD_BC0_FAILURE_BIT;
            }
            else
            {
               gs_g997_Failure_Flag_FE |= LCD_BC1_FAILURE_BIT;
            }
         }
         else
         {
            if (s_BC==0)
            {
               gs_g997_Failure_Flag_FE &= ~LCD_BC0_FAILURE_BIT;
            }
            else
            {
               gs_g997_Failure_Flag_FE &= ~LCD_BC1_FAILURE_BIT;
            }
         }

         if (gs_g997_Failure_Flag_FE & LCD_LP1_FAILURE_BIT)
         {
            if (s_BC==0)
            {
               gs_g997_Failure_Flag_FE |= LCD_BC0_FAILURE_BIT;
            }
            else
            {
               gs_g997_Failure_Flag_FE |= LCD_BC1_FAILURE_BIT;
            }
         }
         else
         {
            if (s_BC==0)
            {
               gs_g997_Failure_Flag_FE &= ~LCD_BC0_FAILURE_BIT;
            }
            else
            {
               gs_g997_Failure_Flag_FE &= ~LCD_BC1_FAILURE_BIT;
            }
         }
      }
   }

   // Far End Line failures
   //Remember, there are no indicator bits for far end LOL, ESE. Moreover, they are required to be
   //implemented only for the CO.
   ProcessG997Failures(&gt_g997_FailureState_FE.t_LOS,gt_RxIbData.uc_flos_def,
                       &gs_g997_Failure_Flag_FE,gs_MONI_ShowFailure_FE,gs_PMDFramesPerRxIBStructure);
   ProcessG997Failures(&gt_g997_FailureState_FE.t_LOF,gt_RxIbData.uc_frdi_def,
                       &gs_g997_Failure_Flag_FE,gs_MONI_ShowFailure_FE,gs_PMDFramesPerRxIBStructure);

}


/*------------------------------------------------------------------------------------------------------
*
*   Prototype: void G997_CheckTxFailure(void)
*
*
*   Input Arguments:
*
*   Output Arguments:
*
*   Returns:
*
*   Global Variables:
*
*-------------------------------------------------------------------------------
*/
void G997_CheckTxFailure(void)
{
   int16 s_BC, s_lp;

   // declare failure if defect lasts more than 2.5 sec (def counter > ActiveThreshold)
   // reset failure if defect doesn't exist for more than 10 sec  (-def counter > InactiveThreshold)

   // ATM Failure:
   for (s_lp=0; s_lp<ATM_NUM_BC; s_lp++)
   {
      s_BC = gt_tx_TPS_Map.s_LPtoABC[s_lp];

      if ( (gta_DsBearerChanAct[s_BC].s_TypeEnabled == TPS_TC_TYPE_ATM) &&
            (s_BC != DISABLED_ABC) )
      {
         //XDSLRTFW-956 : ENH_DS_ALL_ALL_SESx_ESx_LOMx
         ProcessG997Failures(&gt_g997_AtmFailure_NE.t_LCD[s_BC],gt_TxIbData.uc_lcd_def[s_lp],
                             &gsa_g997_Failure_Flag_NE[0],gsa_MONI_ShowFailure_NE[0],gs_PMDFramesPerTxIBStructure);
         ProcessG997Failures(&gt_g997_AtmFailure_NE.t_NCD[s_BC],gt_TxIbData.uc_ncd_anom[s_lp],
                             &gsa_g997_Failure_Flag_NE[0],gsa_MONI_ShowFailure_NE[0],gs_PMDFramesPerTxIBStructure);

         //Failure State in the Latency Path LPx needs to be reflected back into the Bearer Channel BCx failure flag
         //that is hooked to PLAM CMV 0. Only two latency paths/ Bearer Channel are supported
         if (gsa_g997_Failure_Flag_NE[0] & NCD_LP0_FAILURE_BIT)
         {
            if (s_BC==0)
            {
               gsa_g997_Failure_Flag_NE[0] |= NCD_BC0_FAILURE_BIT;
            }
            else
            {
               gsa_g997_Failure_Flag_NE[0] |= NCD_BC1_FAILURE_BIT;
            }
         }
         else
         {
            if (s_BC==0)
            {
               gsa_g997_Failure_Flag_NE[0] &= ~NCD_BC0_FAILURE_BIT;
            }
            else
            {
               gsa_g997_Failure_Flag_NE[0] &= ~NCD_BC1_FAILURE_BIT;
            }
         }

         if (gsa_g997_Failure_Flag_NE[0] & NCD_LP1_FAILURE_BIT)
         {
            if (s_BC==0)
            {
               gsa_g997_Failure_Flag_NE[0] |= NCD_BC0_FAILURE_BIT;
            }
            else
            {
               gsa_g997_Failure_Flag_NE[0] |= NCD_BC1_FAILURE_BIT;
            }
         }
         else
         {
            if (s_BC==0)
            {
               gsa_g997_Failure_Flag_NE[0] &= ~NCD_BC0_FAILURE_BIT;
            }
            else
            {
               gsa_g997_Failure_Flag_NE[0] &= ~NCD_BC1_FAILURE_BIT;
            }
         }


      }
   }
//XDSLRTFW-956 : ENH_DS_ALL_ALL_SESx_ESx_LOMx
   // Near End Line failures
   ProcessG997Failures(&gt_g997_FailureState_NE.t_LOS,gt_TxIbData.uc_los_def,
                       &gsa_g997_Failure_Flag_NE[0],gsa_MONI_ShowFailure_NE[0],gs_PMDFramesPerTxIBStructure);

   // FE LPR is defined as a FE LPR primitive (2 or 3 FE LPR IBs) followed by an LOS
   // The FE LPR defect is set based on this primitive and can only be cleared by the non-LOS condition.
   // The FE LPR failure is triggred when the FE LPR defect and NE LOS failure are both present.
   // Set the FE LPR simultaneously with NE LOS to avoid automsg/autorestart problems.

   if ((gsa_g997_Failure_Flag_NE[0] & LOS_FAILURE_BIT) && (gt_RxIbData.uc_flpr_def == PRESENT))
   {
      gs_g997_Failure_Flag_FE = gs_g997_Failure_Flag_FE | LPR_FAILURE_BIT;
   }

   ProcessG997Failures(&gt_g997_FailureState_NE.t_LOF,(uint8)(gt_TxIbData.uc_sef_def & !gt_TxIbData.uc_los_def),
                       &gsa_g997_Failure_Flag_NE[0],gsa_MONI_ShowFailure_NE[0],gs_PMDFramesPerTxIBStructure);
   ProcessG997Failures(&gt_g997_FailureState_NE.t_LPR,gt_TxIbData.uc_lpr_def,
                       &gsa_g997_Failure_Flag_NE[0],gsa_MONI_ShowFailure_NE[0],gs_PMDFramesPerTxIBStructure);
   ProcessG997Failures(&gt_g997_FailureState_NE.t_LOM,gt_TxIbData.uc_lom_fail,
                       &gsa_g997_Failure_Flag_NE[0],gsa_MONI_ShowFailure_NE[0],gs_PMDFramesPerTxIBStructure); //snv
   ProcessG997Failures(&gt_g997_FailureState_NE.t_ESE,gt_TxIbData.uc_ese_def,
                       &gsa_g997_Failure_Flag_NE[0],gsa_MONI_ShowFailure_NE[0],gs_PMDFramesPerTxIBStructure);
   ProcessG997Failures(&gt_g997_FailureState_NE.t_SESx,gt_TxIbData.uc_ese_def,
                       &gsa_g997_Failure_Flag_NE[1],gsa_MONI_ShowFailure_NE[1],gs_PMDFramesPerTxIBStructure);
   ProcessG997Failures(&gt_g997_FailureState_NE.t_ESx,gt_TxIbData.uc_es_def,
                       &gsa_g997_Failure_Flag_NE[1],gsa_MONI_ShowFailure_NE[1],gs_PMDFramesPerTxIBStructure);
   ProcessG997Failures(&gt_g997_FailureState_NE.t_PRO,gt_TxIbData.uc_pro_fail,
                       &gsa_g997_Failure_Flag_NE[0],gsa_MONI_ShowFailure_NE[0],gs_PMDFramesPerTxIBStructure);
   //XDSLRTFW-2050/2094
   ProcessG997Failures(&gt_g997_FailureState_NE.t_OOS,gt_TxIbData.uc_oos_def,
                       &gsa_g997_Failure_Flag_NE[0],gsa_MONI_ShowFailure_NE[0],gs_PMDFramesPerTxIBStructure);
//XDSLRTFW-956 : ENH_DS_ALL_ALL_SESx_ESx_LOMx
}


/*
*-------------------------------------------------------------------------------
*
*   Prototype: void G997_CheckUsMargin(void)
*
*   This function compares the current SNR with the configured
*   miminum SNR and issues a retrain.
*
*   Input Arguments:
*
*   Output Arguments:
*
*   Returns:
*
*   Global Variables:
*
*-------------------------------------------------------------------------------
*/

void G997_CheckUsMargin(void)
{
}

/*
*-------------------------------------------------------------------------------
*
*   Prototype: void SetTwoBits(int16 s_in, int16 s_act_thres, int16 s_inact_thres,
*      int16 *ps_act_res, int16 *ps_inact_res, int16 *ps_i)
*
*   This function sets the ps_i bit in ps_act_res if s_in > s_act_thres,
*   then set the ps_i bit in ps_inact_res if -s_in > s_inact_thres,
*   then increment increment (*ps_i) by 1.
*
*   Input Arguments:
*
*   Output Arguments:
*
*   Returns:
*
*   Global Variables:
*
*-------------------------------------------------------------------------------
*/

void SetTwoBits(int16 s_in, int16 s_act_thres, int16 s_inact_thres, int16 *ps_act_res, int16 *ps_inact_res, int16 *ps_i)
{
   if (s_in >= 0 && s_in > s_act_thres)
   {
      (*ps_act_res) |= 1 << (*ps_i);
   }
   else if (s_in <= 0 && (-s_in > s_inact_thres))
   {
      (*ps_inact_res) |= 1 << (*ps_i);
   }
   (*ps_i)++;
}






/*
*-------------------------------------------------------------------------------
*
*   Prototype: void ProcessG997Failure(G997_Failure_State_t* Defect, uint16 uc_AnomDefIndicator, uint16 us_FailFlag)
*
*   This function tracks the state of a given defect per Superframe. Four states
*   are defined. The INIT state in which the device has been initialized. The Steady
*   state in which the device enteres after initialization and when no defect is
*   present. The TRIGGERED state in which the defect has been triggered for a
*   period less than the failure set period. The device can stay in this state
*   until either the defect goes away before the failure set period is up or until
*   the failure set period elapses. In the latter case, the state changes to
*   FAILED state. The device stays in this state as long as the failure
*   condition persists or else the failure condition goes away for a period greater
*   than the failure reset period. In the latter case, the device goes back into the
*   STEADY state.
*
*   Input Arguments:   State Variable for Failure, Condition of Anomaly, Bit mapped Failure Flag
*                  for PLAM CMV0 or 1
*
*   Output Arguments:   None
*
*   Returns:         New Defect State, Counters and Failure condition update
*
*   Global Variables:   gt_g997_FailureState_NE or gt_g997_FailureState_FE (pt_Defect input)
*                  gt_TxIbData or gt_RxIbData (uc_AnomDefIndicator input)
*                  gsa_g997_Failure_Flag_NE or gs_g997_Failure_Flag_FE (us_FailFlag input)
*
*-------------------------------------------------------------------------------
*/
void ProcessG997Failures(G997_Failure_State_t* pt_Defect, uint8 uc_AnomDefIndicator, uint16 *pus_FailFlag,
                         uint16 us_ReInitTriggerFlag, int16 s_PMDFramesPerIBStructure)
{
   switch (pt_Defect->us_NextState)
   {
   case STEADY_STATE:
      if (uc_AnomDefIndicator == PRESENT) //Transition 2
      {
         if (pt_Defect->us_PrevState == STEADY_STATE)
         {
            pt_Defect->us_NextState = TRIGGERED_STATE;
            pt_Defect->ul_Duration = 1;
            pt_Defect->ul_Duration_reset = 0;
         }
      }
      else
      {
         pt_Defect->us_PrevState = STEADY_STATE; //Transition 9
         pt_Defect->us_NextState = STEADY_STATE;
         pt_Defect->ul_Duration = 0;
      }
      break;

   case TRIGGERED_STATE:
      if (uc_AnomDefIndicator == PRESENT)
      {
         //XDSLRTFW-2050
         if(pt_Defect->ul_Duration_reset != 0)
         {
            pt_Defect->ul_Duration = 1;
            pt_Defect->ul_Duration_reset = 0;
         }
         //XDSLRTFW-955 : Implementation to split between the Failure Indication
         // and Re-init period
         // Trigger Re-Init if consecutive failure indication is seen and if enabled by CMV MODEM_MONITOR 5 or CMV MONI 5
         if ((pt_Defect->ul_Duration >= pt_Defect->ul_ReInitPeriod) && ((pt_Defect->us_FailureBitNumber) & us_ReInitTriggerFlag))
         {
            pt_Defect->us_PrevState = TRIGGERED_STATE; //Transition 5
            pt_Defect->us_NextState = FAILED_STATE;
            pt_Defect->ul_Duration = 1;
            gft_ReInitTrigger = 1;
            *pus_FailFlag |= (pt_Defect->us_FailureBitNumber); // Sets up the failure flag in the CMV
         }
         else
         {
            //XDSLRTFW-955 : Implementation to split between the Failure Indication
            // and Re-init period
            if (pt_Defect->ul_Duration >= pt_Defect->ul_FailureSetPeriod)
            {
               *pus_FailFlag |= (pt_Defect->us_FailureBitNumber);   // Sets up the failure flag in the CMV
            }

            pt_Defect->us_PrevState = TRIGGERED_STATE; //Transition 4
            pt_Defect->us_NextState = TRIGGERED_STATE;
            pt_Defect->ul_Duration +=  s_PMDFramesPerIBStructure;
         }
      }

      if (uc_AnomDefIndicator == TERMINATED)
      {
         if (pt_Defect->ul_Duration <= pt_Defect->ul_FailureSetPeriod)
         {
            pt_Defect->us_PrevState = TRIGGERED_STATE; //Transition 3
            pt_Defect->us_NextState = STEADY_STATE;
            pt_Defect->ul_Duration = 0;
            *pus_FailFlag &= ~(pt_Defect->us_FailureBitNumber); // Clears the failure flag in the CMV
         }
         else if (pt_Defect->ul_Duration > pt_Defect->ul_FailureSetPeriod)
         {
            pt_Defect->ul_Duration_reset +=  s_PMDFramesPerIBStructure;
            if (pt_Defect->ul_Duration_reset >= pt_Defect->ul_FailureResetPeriod)
            {
               pt_Defect->us_PrevState = TRIGGERED_STATE; //Transition 4
               pt_Defect->us_NextState = STEADY_STATE;
               pt_Defect->ul_Duration_reset = 0;
               pt_Defect->ul_Duration = 0;
               *pus_FailFlag &= ~(pt_Defect->us_FailureBitNumber); // Clears the failure flag in the CMV

            }
         }
      }
      if ((pt_Defect->us_PrevState == FAILED_STATE) && (pt_Defect->us_NextState == TRIGGERED_STATE))
      {
         pt_Defect->us_PrevState = TRIGGERED_STATE;   //Transition 10 - Special condition for some
         //compound conditions
         pt_Defect->ul_Duration = 1;
      }
      break;

   case FAILED_STATE:
      if (uc_AnomDefIndicator == TERMINATED)
      {
         if (pt_Defect->ul_Duration >= pt_Defect->ul_FailureResetPeriod)
         {
            pt_Defect->us_PrevState = FAILED_STATE; //Transition 8
            pt_Defect->us_NextState = STEADY_STATE;
            pt_Defect->ul_Duration = 0;
            *pus_FailFlag &= ~(pt_Defect->us_FailureBitNumber); // Clears the failure flag in the CMV
         }
         else
         {
            pt_Defect->us_PrevState = FAILED_STATE; //Transition 6
            pt_Defect->us_NextState = FAILED_STATE;
            pt_Defect->ul_Duration +=  s_PMDFramesPerIBStructure;
         }
      }
      else
      {
         pt_Defect->ul_Duration = 0;
      }


      //if (uc_AnomDefIndicator == PRESENT)
      //{
      //   pt_Defect->us_PrevState = FAILED_STATE; //Transition 7
      //   pt_Defect->us_NextState = FAILED_STATE;
      //   pt_Defect->ul_Duration++;
      //}
      break;
   }
}

#ifdef DEBUG_ATM_FAILURES
void printIBStatus()
{
   uint16   s_lp;
   FILE   *G997Failures;

   G997Failures = fopen("g997fail.txt","a");

   fprintf(G997Failures,"\nNear End Data ........  \n");
   fprintf(G997Failures,"***********************\n");
   fprintf(G997Failures,"Transmit Symbol Count -->> %d  \n",gl_TxSymbolCount);
   for (s_lp=0; s_lp<NUM_DATA_PATHS; s_lp++)
   {
      fprintf(G997Failures,"gt_TxIbData.uc_be_anom[%d]  = %d  \n",s_lp,gt_TxIbData.uc_be_anom[s_lp]);
      fprintf(G997Failures,"gt_TxIbData.uc_fec_anom[%d] = %d  \n",s_lp,gt_TxIbData.uc_fec_anom[s_lp]);
      fprintf(G997Failures,"gt_TxIbData.uc_hec_anom[%d] = %d  \n",s_lp,gt_TxIbData.uc_hec_anom[s_lp]);
      fprintf(G997Failures,"gt_TxIbData.uc_cd_flag[%d]  = %d  \n",s_lp,gt_TxIbData.uc_cd_flag[s_lp]);
      fprintf(G997Failures,"gt_TxIbData.uc_ncd_anom[%d] = %d  \n",s_lp,gt_TxIbData.uc_ncd_anom[s_lp]);
      fprintf(G997Failures,"gt_TxIbData.uc_ocd_anom[%d] = %d  \n",s_lp,gt_TxIbData.uc_ocd_anom[s_lp]);
      fprintf(G997Failures,"gt_TxIbData.uc_lcd_def[%d]  = %d  \n",s_lp,gt_TxIbData.uc_lcd_def[s_lp]);
      fprintf(G997Failures,"gt_TxIbData.uc_atm_state[%d]= %d  \n",s_lp,gt_TxIbData.uc_atm_state[s_lp]);
      fprintf(G997Failures,"gt_TxIbData.s_ocd_cnt[%d]   = %d  \n",s_lp,gt_TxIbData.s_ocd_cnt[s_lp]);
   }
   fprintf(G997Failures,"gt_TxIbData.uc_pro_fail = %d  \n",gt_TxIbData.uc_pro_fail);
   fprintf(G997Failures,"gt_TxIbData.uc_sef_def  = %d  \n",gt_TxIbData.uc_sef_def);
   fprintf(G997Failures,"gt_TxIbData.uc_ese_def  = %d  \n",gt_TxIbData.uc_ese_def);
   fprintf(G997Failures,"gt_TxIbData.uc_fpo_def  = %d  \n",gt_TxIbData.uc_fpo_def);
   fprintf(G997Failures,"gt_TxIbData.uc_lol_fail = %d  \n",gt_TxIbData.uc_lol_fail);
   fprintf(G997Failures,"gt_TxIbData.uc_lom_fail = %d  \n",gt_TxIbData.uc_lom_fail);
   fprintf(G997Failures,"gt_TxIbData.uc_los_def  = %d  \n",gt_TxIbData.uc_los_def);
   fprintf(G997Failures,"gt_TxIbData.uc_lpr_def  = %d  \n",gt_TxIbData.uc_lpr_def);
   fprintf(G997Failures,"gsa_g997_Failure_Flag_NE = %d  \n",gsa_g997_Failure_Flag_NE);

   fprintf(G997Failures,"\nFar End Data ........  \n");
   fprintf(G997Failures,"Receive Symbol Count -->> %d  \n",gl_RxSymbolCount);
   fprintf(G997Failures,"#######################\n");
   for (s_lp=0; s_lp<NUM_DATA_PATHS; s_lp++)
   {
      fprintf(G997Failures,"gt_RxIbData.uc_fncd_ind[%d]  = %d  \n",s_lp,gt_RxIbData.uc_fncd_ind[s_lp]);
      fprintf(G997Failures,"gt_RxIbData.uc_fncd_anom[%d] = %d  \n",s_lp,gt_RxIbData.uc_fncd_anom[s_lp]);
      fprintf(G997Failures,"gt_RxIbData.uc_focd_anom[%d] = %d  \n",s_lp,gt_RxIbData.uc_focd_anom[s_lp]);
      fprintf(G997Failures,"gt_RxIbData.s_fncd_cnt[%d]   = %d  \n",s_lp,gt_RxIbData.s_fncd_cnt[s_lp]);
      fprintf(G997Failures,"gt_RxIbData.uc_flcd_def[%d]  = %d  \n",s_lp,gt_RxIbData.uc_flcd_def[s_lp]);
      fprintf(G997Failures,"gt_RxIbData.uc_atm_state[%d] = %d  \n",s_lp,gt_RxIbData.uc_atm_state[s_lp]);
      fprintf(G997Failures,"gt_RxIbData.uc_fncd_hist[%d] = %d  \n",s_lp,gt_RxIbData.uc_fncd_hist[s_lp]);
   }

   fprintf(G997Failures,"gt_RxIbData.uc_frdi_def  = %d  \n",gt_RxIbData.uc_frdi_def);
   fprintf(G997Failures,"gt_RxIbData.uc_flol_fail = %d  \n",gt_RxIbData.uc_flol_fail);
   fprintf(G997Failures,"gt_RxIbData.uc_flos_def  = %d  \n",gt_RxIbData.uc_flos_def);
   fprintf(G997Failures,"gt_RxIbData.uc_flos_hist = %d  \n",gt_RxIbData.uc_flos_hist);
   fprintf(G997Failures,"gt_RxIbData.uc_flos_ind  = %d  \n",gt_RxIbData.uc_flos_ind);
   fprintf(G997Failures,"gt_RxIbData.uc_flpr_def  = %d  \n",gt_RxIbData.uc_flpr_def);
   fprintf(G997Failures,"gs_g997_Failure_Flag_FE = %d  \n",gs_g997_Failure_Flag_FE);
   fclose(G997Failures);
}
#endif // DEBUG_ATM_FAILURES

#endif // INCLUDE_NON_CACHE_FUNCTIONS
