/* **COPYRIGHT******************************************************************
    INTEL CONFIDENTIAL
    Copyright (C) 2017 Intel Corporation
    Copyright (C), 1994-2006 Aware Inc. All Rights Reserved.
******************************************************************COPYRIGHT** */
/* **DISCLAIMER*****************************************************************
    The source code contained or described herein and all documents related
    to the source code ("Material") are owned by Intel Corporation or its
    suppliers or licensors. Title to the Material remains with Intel
    Corporation or its suppliers and licensors. The Material may contain
    trade secrets and proprietary and confidential information of Intel
    Corporation and its suppliers and licensors, and is protected by
    worldwide copyright and trade secret laws and treaty provisions. No part
    of the Material may be used, copied, reproduced, modified, published,
    uploaded, posted, transmitted, distributed, or disclosed in any way
    without Intel's prior express written permission.

    No license under any patent, copyright, trade secret or other
    intellectual property right is granted to or conferred upon you by
    disclosure or delivery of the Materials, either expressly, by
    implication, inducement, estoppel or otherwise. Any license under
    such intellectual property rights must be express and approved by
    Intel in writing.
*****************************************************************DISCLAIMER** */
/**********************************************************************
 *
 * Aware BIS Technology. Proprietary and Confidential.
 *
 * ADDRESS:          40 Middlesex Turnpike, Bedford, MA 01730-1413 USA
 * TELEPHONE:        781.276.4000
 * FAX:              781.276.4001
 * WEB:              http://www.aware.com
 *
 * FILE:             GetQlnHlogForCarrierGroup.c
 * DESCRIPTION:      Functions to get the QLN or HLOG for subcarrier group.
 *
 **********************************************************************/
#include "common.h"
#include "gdata.h"
#include "vdsl_xception.h"
#include "cmv.h"



/*^^^
 *------------------------------------------------------------------------------
 *
 *  Name: GetCarrierGroupIdx
 *
 *  Description: For the given subcarrier group index, determine if
 *  this subgroup contains any out of band tones. If it does, return
 *  a special value "OUT_OF_BAND" to indicate this to the calling function.
 *  If this tone is a in-band tone, this function converts the input subgroup index
 *  to an output subgroup index which matches to the QLN or HLOG table.
 *
 *  Prototype: uint16 GetCarrierGroupIdx(const uint16 us_CarrierGroupIdx_In)
 *
 *  Input Arguments:
 *      us_CarrierGroupIdx_In -- subcarrier group index in the range of 0..511
 *
 *  Output Arguments:
 *
 *   return:
 *      the output subcarrier group index or 0xFFFF for the special value of "OUT_OF_BAND"
 *
 *  Global Variables Used:
 *      guca_RxSupportedToneSet[] -- (I) array to indicated the supported set
 *      ghpsa_RxFineGains_Inactive -- (I) pointer to the RX fine gain
 *
 *  Notes:
 *
 *------------------------------------------------------------------------
 *^^^
 */
uint16 GetCarrierGroupIdx(uint16 us_CarrierGroupIdx_In,
                          uint16 us_out_of_band_indicator,
                          int16 s_Log2G_old,
                          int16 s_Log2G_new)
{
   uint16 us_toneIdx, us_startToneIdx, us_endToneIdx, us_CarrierGroupIdx_out;
   int16 s_G_new, s_G_old;

   s_G_old = 1<<s_Log2G_old;
   s_G_new = 1<<s_Log2G_new;

   // compute the tone indices based on the subcarrier group index, i.e., k*G
   us_startToneIdx = us_CarrierGroupIdx_In << s_Log2G_new;

   if(us_out_of_band_indicator == OUT_OF_RANGE_QLN)
   {
      us_endToneIdx = us_startToneIdx + s_G_new;
   }
   else
   {
      us_endToneIdx = us_startToneIdx + 1;
   }

   //Check if there is any tone in this group, which is inactive
   for (us_toneIdx = us_startToneIdx; us_toneIdx < us_endToneIdx; us_toneIdx++)
   {
      //If this group contains at least one tone which is inactive, return
      if(IS_TONEFLAGSET(guca_RxSupportedToneSet, us_toneIdx) == 0)
      {
         return((uint16)OUT_OF_BAND);
      }

      if(ghpsa_RxFineGains_Inactive[us_toneIdx] == 0)
      {
         return((uint16)OUT_OF_BAND);
      }
   }

   //Perform the index conversion
   us_CarrierGroupIdx_out = us_CarrierGroupIdx_In >> (s_Log2G_old-s_Log2G_new);

   return(us_CarrierGroupIdx_out);

}

/*^^^
 *------------------------------------------------------------------------------
 *
 *  Name: ResampleQln
 *
 *  Description:
 *  According to the VDSL2 specification, the G used to sample QLN is computed
 *  based on the highest tone indes of SUPPORTEDSET. Since the QLN computed
 *  originally is based on G of 8. This function resamples the QLN to
 *  meet the correct G value requirement. It also checks for any outband/out-range
 *  tone group and set the corresponding entries properly.
 *
 *  Prototype: ResampleQln(int16 s_Log2G)
 *
 *  Input Arguments:
 *      s_log2G -- log2(G), where G should be the one computed based on SUPPORTEDSET.
 *
 *  Output Arguments:
 *
 *  return:
 *
 *
 *  Global Variables Used:
 *      guca_QLN[] -- (I/O) array to store QLN
 *  Notes:
 *
 *------------------------------------------------------------------------
 *^^^
 */
void ResampleQln(int16 s_Log2G)
{
   int16 j;
   uint16 k;

   signed int s_MaxLog2G;

   s_MaxLog2G = SUBCAR_GROUP_SIZE_LOG2_MAX_DS; //SUBCAR_GROUP_SIZE_LOG2_8
   if (gt_ProfileAct.us_ProfileSelected & CNFG_V2_PROFILE_35B_MASK)
   {
      s_MaxLog2G = SUBCAR_GROUP_SIZE_LOG2_MAX_DS_35B;
   }

   // In case of 8K tones, QLN is calculated for every 8th tone and but we need QLN in for every 16 tone
   if(gs_RxLog2FftLength == 14)
   {

      s_MaxLog2G  = SUBCAR_GROUP_SIZE_LOG2_16;
   }

   //We cannot perform the conversion from a smaller G to a larger G
   if(s_Log2G <= s_MaxLog2G)
   {
      for(j=511; j>=0; j--)
      {
         k = GetCarrierGroupIdx(j, OUT_OF_RANGE_QLN, s_MaxLog2G, s_Log2G);

         if(k != (uint16)OUT_OF_BAND)
         {
            guca_QLN[j] = guca_QLN[k];
         }
         else
         {
            guca_QLN[j] = ((uint8)OUT_OF_RANGE_QLN);
         }
      }
   }
   else
   {
      EnterFailStates(E_CODE_QLN_G_CONVERSION);
   }
}

/*^^^
 *------------------------------------------------------------------------------
 *
 *  Name: ResampleHlog
 *
 *  Description:
 *  According to the VDSL2 specification, the G used to sample HLOG is computed
 *  based on the highest tone indes of SUPPORTEDSET. Since the QLOG computed
 *  originally is based on G of 8. This function resamples the HLOG to
 *  meet the correct G value requirement. It also checks for any outband/out-range
 *  tone group and set the corresponding entries properly.
 *
 *  Prototype: ResampleQln(int16 s_Log2G)
 *
 *  Input Arguments:
 *      s_log2G -- log2(G), where G should be the one computed based on SUPPORTEDSET.
 *
 *  Output Arguments:
 *
 *  return:
 *
 *
 *  Global Variables Used:
 *      gsa_RxHlog[] -- (I/O) array to store HLOG
 *  Notes:
 *
 *------------------------------------------------------------------------
 *^^^
 */
void ResampleHlog(int16 s_Log2G)
{
   int16 j;
   uint16 k;

   signed int s_MaxLog2G;

   s_MaxLog2G = SUBCAR_GROUP_SIZE_LOG2_MAX_DS;
   if (gt_ProfileAct.us_ProfileSelected & CNFG_V2_PROFILE_35B_MASK)
   {
      s_MaxLog2G = SUBCAR_GROUP_SIZE_LOG2_MAX_DS_35B;
   }

   //We cannot perform the conversion from a smaller G ot a larger G
   if(s_Log2G > s_MaxLog2G)
   {
      EnterFailStates(E_CODE_HLOG_G_CONVERSION);
   }

   for(j=511; j>=0; j--)
   {
      k = GetCarrierGroupIdx(j, OUT_OF_RANGE_HLOG, s_MaxLog2G, s_Log2G);

      if(k != (uint16)OUT_OF_BAND)
      {
         gsa_RxHlog[j] = gsa_RxHlog[k];
      }
      else
      {
         gsa_RxHlog[j] = ((int16)OUT_OF_RANGE_HLOG);
      }
   }

}

