/* **COPYRIGHT******************************************************************
    INTEL CONFIDENTIAL
    Copyright (C) 2017 Intel Corporation
    Copyright (C), 1994-2003 Aware Inc. All Rights Reserved.
******************************************************************COPYRIGHT** */
/* **DISCLAIMER*****************************************************************
    The source code contained or described herein and all documents related
    to the source code ("Material") are owned by Intel Corporation or its
    suppliers or licensors. Title to the Material remains with Intel
    Corporation or its suppliers and licensors. The Material may contain
    trade secrets and proprietary and confidential information of Intel
    Corporation and its suppliers and licensors, and is protected by
    worldwide copyright and trade secret laws and treaty provisions. No part
    of the Material may be used, copied, reproduced, modified, published,
    uploaded, posted, transmitted, distributed, or disclosed in any way
    without Intel's prior express written permission.

    No license under any patent, copyright, trade secret or other
    intellectual property right is granted to or conferred upon you by
    disclosure or delivery of the Materials, either expressly, by
    implication, inducement, estoppel or otherwise. Any license under
    such intellectual property rights must be express and approved by
    Intel in writing.
*****************************************************************DISCLAIMER** */
/****************************************************************************
;   Aware DMT Technology. Proprietary and Confidential.
;
;   40 Middlesex Turnpike, Bedford, MA 01730-1413
;   Phone (781) 276 - 4000
;   Fax   (781) 276 - 4001
;
;   File Name: PSDShaping.c
;
;   Functions for PSD shaping
;
'
*****************************************************************************/

// ************************************************************************************************************************
// PSDShaping.c
//
// History
// 28/02/2014 : Tranmitted PSD should be less than or equal -80 dbm/Hz at RFI bands. If the calculated value is less than -80dbm Hz then don't
//              modify the PSD else CAP it to -80dbm/Hz
//              Search for XDSLRTFW-1602
//
//*************************************************************************************************************************

#include <string.h>
#include "vdsl_compiler.h"
#include "typedef.h"
#include "gdata.h"
#include "dsp_op.h"
#include "vdsl_xception.h"
#include "vdsl_const.h"
#include "sys_const.h"
#include "TxPSDControl.h"
#include "decimalgain.h"
#include "cmv.h"

//Implementation Margin 0.7dB is desired; But we saw US0 violate by 0.9dB and US1/2/3 by 0.2dB; So flat reduce by 0.2dB an US0 by 0.7dB
int16 gs_TxPsd_us0_ReduceByMinus0pt7 = 7; //reduce by 0.7dB; -ve value means Boost PSD, +ve value means Reduce PSD
//int16 *gpsa_DebugBuf5Dk_16K = (int16 *)0x5D000; Commented may be needed fr future debugging
//extern int16 gsa_dbg_StateTrail[];

/*^^^
 *------------------------------------------------------------------------
 *
 *
 *  Description:  Gets adjacent breakpoints given breakpoint index
 *
 *
 *  Prototype:
 *      void GetAdjacentBreakpts(int16 s_brkptIdxIn, int16 s_NumBrkptsIn, PSDRecord_t *pt_PSDRecIn, int16 s_minToneIdxIn, int16 s_maxToneIdxIn,
 *            int16 *ps_leftToneIdxOut, int16 *ps_leftToneAttenLevelOut, int16 *ps_rightToneIdxOut, int16 *ps_rightToneAttenLevelOut)
 *
 *
 *  Input Arguments:
 *      None
 *
 *  Output Arguments:
 *      None
 *
 *   Return:
 *
 *  Global Variables Used:
 *
 *  Notes:
 *
 *------------------------------------------------------------------------
 *^^^
 */
void GetAdjacentBreakpts(int16 s_brkptIdxIn, int16 s_NumBrkptsIn, PSDRecord_t *pt_PSDRecIn,
                         int16 s_minToneIdxIn, int16 s_maxToneIdxIn,
                         int16 *ps_leftToneIdxOut, int16 *ps_leftToneAttenLevelOut,
                         int16 *ps_rightToneIdxOut, int16 *ps_rightToneAttenLevelOut)
{
   //Get the tone index and psd level of a breakpoint
   if (s_brkptIdxIn==0)
   {
      // For flat extension of log_PSD to frequencies lower than first breakpoint
      *ps_leftToneIdxOut = s_minToneIdxIn;
      *ps_leftToneAttenLevelOut = pt_PSDRecIn[0].s_PSDLevelOfTone;

      *ps_rightToneIdxOut = pt_PSDRecIn[s_brkptIdxIn].us_IndexOfTone;
      *ps_rightToneAttenLevelOut = pt_PSDRecIn[s_brkptIdxIn].s_PSDLevelOfTone;
   }
   else
   {
      *ps_leftToneIdxOut = pt_PSDRecIn[s_brkptIdxIn-1].us_IndexOfTone;
      *ps_leftToneAttenLevelOut = pt_PSDRecIn[s_brkptIdxIn-1].s_PSDLevelOfTone;

      if (s_brkptIdxIn == s_NumBrkptsIn)
      {
         // For flat extension of log_PSD to frequencies higher than last breakpoint
         *ps_rightToneIdxOut = s_maxToneIdxIn+1;
         *ps_rightToneAttenLevelOut = pt_PSDRecIn[s_brkptIdxIn-1].s_PSDLevelOfTone;
      }
      else
      {
         // For interpolation between two breakpoints.
         // Need to also access the next PSD breakpoint record
         *ps_rightToneIdxOut = pt_PSDRecIn[s_brkptIdxIn].us_IndexOfTone;
         *ps_rightToneAttenLevelOut = pt_PSDRecIn[s_brkptIdxIn].s_PSDLevelOfTone;
      }
   }
}

//XDSLRTFW-2260 (Start)
/*^^^
 *------------------------------------------------------------------------
 *
 *
 *  Description:  Gets adjacent breakpoints given breakpoint index
 *
 *
 *  Prototype:
 *      void GetAdjacentBreakpts2(int16 s_brkptIdxIn, int16 s_NumBrkptsIn, PSDRecord_t *pt_PSDRecIn, int16 s_minToneIdxIn, int16 s_maxToneIdxIn,
 *            int16 *ps_leftToneIdxOut, int16 *ps_leftToneAttenLevelOut, int16 *ps_rightToneIdxOut, int16 *ps_rightToneAttenLevelOut)
 *
 *
 *  Input Arguments:
 *      None
 *
 *  Output Arguments:
 *      None
 *
 *   Return:
 *
 *  Global Variables Used:
 *
 *  Notes:  this function is identical to GetAdjacentBreakpts2 except one input sturcture. Used PSDCompRecord_t structure uses less memory!
 *
 *------------------------------------------------------------------------
 *^^^
 */
void GetAdjacentBreakpts2(int16 s_brkptIdxIn, int16 s_NumBrkptsIn, PSDCompRecord_t *pt_PSDRecIn, int16 s_minToneIdxIn, int16 s_maxToneIdxIn, int16 *ps_leftToneIdxOut, int16 *ps_leftToneAttenLevelOut, int16 *ps_rightToneIdxOut, int16 *ps_rightToneAttenLevelOut)
{
   //Get the tone index and psd level of a breakpoint
   if (s_brkptIdxIn==0)
   {
      // For flat extension of log_PSD to frequencies lower than first breakpoint
      *ps_leftToneIdxOut = s_minToneIdxIn;
      *ps_leftToneAttenLevelOut = pt_PSDRecIn[0].s_PSDLevelOfTone;

      *ps_rightToneIdxOut = pt_PSDRecIn[s_brkptIdxIn].us_IndexOfTone;
      *ps_rightToneAttenLevelOut = pt_PSDRecIn[s_brkptIdxIn].s_PSDLevelOfTone;
   }
   else
   {
      *ps_leftToneIdxOut = pt_PSDRecIn[s_brkptIdxIn-1].us_IndexOfTone;
      *ps_leftToneAttenLevelOut = pt_PSDRecIn[s_brkptIdxIn-1].s_PSDLevelOfTone;

      if (s_brkptIdxIn == s_NumBrkptsIn)
      {
         // For flat extension of log_PSD to frequencies higher than last breakpoint
         *ps_rightToneIdxOut = s_maxToneIdxIn+1;
         *ps_rightToneAttenLevelOut = pt_PSDRecIn[s_brkptIdxIn-1].s_PSDLevelOfTone;
      }
      else
      {
         // For interpolation between two breakpoints.
         // Need to also access the next PSD breakpoint record
         *ps_rightToneIdxOut = pt_PSDRecIn[s_brkptIdxIn].us_IndexOfTone;
         *ps_rightToneAttenLevelOut = pt_PSDRecIn[s_brkptIdxIn].s_PSDLevelOfTone;
      }
   }

}
//XDSLRTFW-2260 (End)
/*^^^
 *------------------------------------------------------------------------
 *
 *
 *  Description:  Computes attenuation level given tone index and adjacent breakpoints
 *
 *
 *  Prototype:
 *      int16 CalcAttenLevel(s_toneIdx, s_leftToneIdx, s_leftToneAttenLevel, s_rightToneIdx, s_rightToneAttenLevel);
 *
 *
 *  Input Arguments:
 *      None
 *
 *  Output Arguments:
 *      None
 *
 *   Return:
 *      Attenuation level @ tone index
 *
 *  Global Variables Used:
 *
 *  Notes:
 *
 *------------------------------------------------------------------------
 *^^^
 */
int16 CalcAttenLevel(int16 s_toneIdx, int16 s_leftToneIdx, int16 s_leftToneAttenLevel, int16 s_rightToneIdx, int16 s_rightToneAttenLevel)
{
   int32 l_numer;
   int16 s_denom, s_attenLevel;

   l_numer = (s_toneIdx - s_leftToneIdx) * (s_rightToneAttenLevel - s_leftToneAttenLevel);
   s_denom = (s_rightToneIdx - s_leftToneIdx);

   s_attenLevel = s_leftToneAttenLevel;
   if (s_denom)   // assure we have a non-zero divisor
   {
      if (l_numer > 0)
      {
         s_attenLevel = s_attenLevel + (int16)NormAndDivide_32by16bit_with_rnd(l_numer, s_denom);
      }
      else
      {
         s_attenLevel = s_attenLevel - (int16)NormAndDivide_32by16bit_with_rnd(-l_numer, s_denom);
      }
   }

   return(s_attenLevel);
}

//XDSLRTFW-2059 (START)
/******************************************************************************/
int16 PSDCompensation(int16 s_ToneIdx, uint16 us_NoOfCompBreakPoint, PSDCompDescriptorTable_t *pt_PSDDesc)
{
   int16 i;
   int16 s_attenLevel;
   int16 s_leftToneIdx, s_leftToneAttenLevel, s_rightToneIdx, s_rightToneAttenLevel;
   PSDCompRecord_t *pt_PSDRec;

   pt_PSDRec = &(pt_PSDDesc->ut_PSDRecord[0]);

   /* Identify the PSDRecord_t[] band within which the tone lies */
   for (i=0; i < us_NoOfCompBreakPoint; i++)
   {
      if (pt_PSDDesc->ut_PSDRecord[i].us_IndexOfTone >= s_ToneIdx)
      {
         break;
      }
   }

   /* Get the adjacent breakpoints given breakpoint index */
   GetAdjacentBreakpts2(i, us_NoOfCompBreakPoint, pt_PSDRec, 0, (int16)(gs_TxNumTones-1),
                        &s_leftToneIdx, &s_leftToneAttenLevel, &s_rightToneIdx, &s_rightToneAttenLevel);

   /* Compute attenuation level given tone index and adjacent breakpoints */
   s_attenLevel = CalcAttenLevel(s_ToneIdx, s_leftToneIdx, s_leftToneAttenLevel, s_rightToneIdx, s_rightToneAttenLevel);

   return(s_attenLevel);
}
//XDSLRTFW-2059 (END)


//XDSLRTFW-2260 (Start)
/*^^^
 *------------------------------------------------------------------------
 *
 *
 *  Description:  Computes attenuation level given tone index and adjacent breakpoints
 *
 *
 *  Prototype:
 *      int16 PSDCompensationStopAndHold(int16 s_ToneIdx,uint16 us_NoOfCompBreakPoint, PSDCompDescriptorTable32_t *pt_PSDDesc)
 *
 *
 *  Input Arguments:
 *      None
 *
 *  Output Arguments:
 *      None
 *
 *   Return:
 *      Attenuation level @ tone index
 *
 *  Note : this function uses Stop and Hold logic. Left tones below the first breakpoint uses
 *         flat extension from the first break point. Right tones above the last breakpoint
 *         uses the flat extension of last breakpoint.
 *
 *  Notes:
 *
 *------------------------------------------------------------------------
 *^^^
 */
int16 PSDCompensationStopAndHold(int16 s_ToneIdx,uint16 us_NoOfCompBreakPoint, PSDCompDescriptorTable32_t *pt_PSDDesc)
{
   int16 i;
   int16 s_attenLevel;
   //int16 s_leftToneIdx, s_leftToneAttenLevel, s_rightToneIdx, s_rightToneAttenLevel;
   PSDCompRecord_t *pt_PSDRec;

   pt_PSDRec = &(pt_PSDDesc->ut_PSDRecord[0]);

   /* Identify the PSDRecord_t[] band within which the tone lies */
   for (i=0; i < us_NoOfCompBreakPoint; i++)
   {
      if (pt_PSDDesc->ut_PSDRecord[i].us_IndexOfTone >= s_ToneIdx)
      {
         break;
      }
   }

   i--;

   // flat extension for left most tones
   if (i < 0)
   {
      i = 0;
   }

   s_attenLevel = pt_PSDDesc->ut_PSDRecord[i].s_PSDLevelOfTone;

   return(s_attenLevel);
}
//XDSLRTFW-2260 (End)

//XDSLRTFW-2059 (Start)
int16 PSDCompensationStopAndHold2(int16 s_ToneIdx,uint16 us_NoOfCompBreakPoint, PSDCompDescriptorTable_t *pt_PSDDesc)
{
   int16 i;
   int16 s_attenLevel;
   //int16 s_leftToneIdx, s_leftToneAttenLevel, s_rightToneIdx, s_rightToneAttenLevel;
   PSDCompRecord_t *pt_PSDRec;

   pt_PSDRec = &(pt_PSDDesc->ut_PSDRecord[0]);

   /* Identify the PSDRecord_t[] band within which the tone lies */
   for (i=0; i < us_NoOfCompBreakPoint; i++)
   {
      if (pt_PSDDesc->ut_PSDRecord[i].us_IndexOfTone >= s_ToneIdx)
      {
         break;
      }
   }

   i--;

   // flat extension for left most tones
   if (i < 0)
   {
      i = 0;
   }

   s_attenLevel = pt_PSDDesc->ut_PSDRecord[i].s_PSDLevelOfTone;

   return(s_attenLevel);
}
// //XDSLRTFW-2059 (End)


#define RFI_PSD_LEVEL_80    (800)   //80dBm/Hz PSD level in 0.1 dB format
#define RFI_PSD_LEVEL_83    (835)   //83.5dBm/Hz PSD level in 0.1 dB format

/*****************************************************************************
;   Subroutine Name: int16 ComputeLinearPSD(int16 s_ToneIdx, PSDDescriptorTable_t *pt_PSDDesc)
;
;   This subroutine computes the PSD shaping before applying the received transmit gains
;
;   Prototype:
;      int16 ComputeLinearPSD(int16 s_ToneIdx, PSDDescriptorTable_t *pt_PSDDesc)
;
;   Input Arguments:
;
;   Output Arguments:
;
;   Return:
;
;******************************************************************************/
int16 ComputeLinearPSD(int16 s_ToneIdx, PSDDescriptorTable_t *pt_PSDDesc)
{
   int16 i, s_linearPSD, s_mant, s_exp;
   int16 s_attenLevel;
   int16 s_compensationLevel = 0;
   int16 s_leftToneIdx, s_leftToneAttenLevel, s_rightToneIdx, s_rightToneAttenLevel;
   int32 l_dB_in;
   PSDRecord_t *pt_PSDRec;
   uint16 us_TxPSD_Mode;
   //int16  s_attenLevel_P=0,s_compensationLevel_P=0; //Debug
   int16 s_ExternalPSDCompensation = 0;
   PSDCompDescriptorTable32_t *pt_PSDCompDesc32;
   PSDCompDescriptorTable_t *pt_PSDCompDesc;

   //XDSLRTFW-2260 (Start)
   pt_PSDCompDesc32 = &(gt_UsPsdCompensationTable);
   //XDSLRTFW-2260 (End)
   pt_PSDRec = &(pt_PSDDesc->ut_PSDRecord[0]);

   //XDSLRTFW-2059 (Start)
   pt_PSDCompDesc = &(gt_PSDCompData.t_PSDCompTable);
   //XDSLRTFW-2059 (End)

   /* Identify the PSDRecord_t[] band within which the tone lies */
   for (i=0; i < pt_PSDDesc->us_NumberOfTones; i++)
   {
      if (pt_PSDDesc->ut_PSDRecord[i].us_IndexOfTone >= s_ToneIdx)
      {
         break;
      }
   }

   /* Get the adjacent breakpoints given breakpoint index */
   GetAdjacentBreakpts(i, pt_PSDDesc->us_NumberOfTones, pt_PSDRec, 0, (int16)(gs_TxNumTones-1),
                       &s_leftToneIdx, &s_leftToneAttenLevel, &s_rightToneIdx, &s_rightToneAttenLevel);

   /* Compute attenuation level given tone index and adjacent breakpoints */
   s_attenLevel = CalcAttenLevel(s_ToneIdx, s_leftToneIdx, s_leftToneAttenLevel, s_rightToneIdx, s_rightToneAttenLevel);

   /* Get the actual psd attenuations for given tone index with upbo */
   if (gt_TxPsdControl.pt_RefPsdDescIn)
   {
      s_attenLevel = GetUPBOAttenLevel(s_ToneIdx, s_attenLevel);
   }
   //s_attenLevel_P = s_attenLevel;//Debug
   //              | gt_PSDCompData.us_FilterCompIdx
   // gs_tx_afe_gain_setting_dB                     |   0 |  -1 |  -2 | ... | -10
   // --------------------------------------        +-----+-----+-----+-----+-----
   // OPTN_BYPASS_Filter_Select                 (0) |   0 |   1 |   2 | ... |  10
   // OPTN_POTS_Filter_Select                   (1) |  20 |  21 |  22 | ... |  30
   // OPTN_POTS_DOUBLE_Filter_Select            (2) |  40 |  41 |  42 | ... |  50
   // OPTN_ISDN_Filter_Select                   (3) |  60 |  61 |  62 | ... |  70
   // OPTN_35B_VDSL_Filter_Select               (4) |  80 |  81 |  82 | ... |  90
   // OPTN_35B_VDSL_POTS_DOUBLE_Filter_Select   (5) | 100 | 101 | 102 | ... | 110
   // OPTN_35B_VDSL_ISDN_Filter_Select          (6) | 120 | 121 | 122 | ... | 130
   // OPTN_POTS_LP_Filter_Select                (7) | 140 | 140 | 140 | ... | 140   => All US0 Filter have 0dB Compensation as these filters are pre-compensated ones
   // OPTN_ISDN_LP_Filter_Select                (8) | 141 | 141 | 141 | ... | 141   => All US0 Filter have 0dB Compensation as these filters are pre-compensated ones
   // OPTN_POTS_DOUBLE_LP_Filter_Select         (9) | 142 | 142 | 142 | ... | 142   => All US0 Filter have 0dB Compensation as these filters are pre-compensated ones
   // OPTN_POTS_QUAD_LP_Filter_Select          (10) | 143 | 143 | 143 | ... | 143   => All US0 Filter have 0dB Compensation as these filters are pre-compensated ones
   // OPTN_35B_POTS_LP_Filter_Select           (11) | 144 | 144 | 144 | ... | 144   => All US0 Filter have 0dB Compensation as these filters are pre-compensated ones
   // OPTN_35B_ISDN_LP_Filter_Select           (12) | 145 | 145 | 145 | ... | 145   => All US0 Filter have 0dB Compensation as these filters are pre-compensated ones
   // OPTN_35B_POTS_DOUBLE_LP_Filter_Select    (13) | 146 | 146 | 146 | ... | 146   => All US0 Filter have 0dB Compensation as these filters are pre-compensated ones

   // get TxPSD Mode (bits 8-11 of TESTArray[TEST_TX_PSD_CONTROL])
   //-----------------------+----------+--------+--------+--------+--------+
   // Mode                  | kl0      | US0    | US1    | US2    | US3    |
   //-----------------------+----------+--------+--------+--------+--------+
   // R12_DEFAULT           | use default R12 code - US PSD feature dis.   |
   //-----------------------+----------+--------+--------+--------+--------+
   // BRCM                  | all      | 0.0 dB | 0.5 dB | 1.5 dB | 1.5 dB |
   //-----------------------+----------+--------+--------+--------+--------+
   // KPN                   | < 3.0 dB | 0.0 dB | 0.5 dB | 1.5 dB | 1.5 dB |
   //                       | < 4.0 dB | 0.0 dB | 0.4 dB | 1.2 dB | 1.2 dB |
   //                       | < 6.0 dB | 0.0 dB | 0.3 dB | 0.9 dB | 0.9 dB |
   //                       | < 8.0 dB | 0.0 dB | 0.2 dB | 0.6 dB | 0.6 dB |
   //                       | <10.0 dB | 0.0 dB | 0.1 dB | 0.3 dB | 0.3 dB |
   //                       | >10.0 dB | 0.0 dB | 0.0 dB | 0.0 dB | 0.0 dB |
   //-----------------------+----------+--------+--------+--------+--------+
   // DEBUG                 | all      | 0.0 dB | 0.0 dB | 0.0 dB | 0.0 dB | // This mode may be used for debugging
   //-----------------------+----------+--------+--------+--------+--------+
   us_TxPSD_Mode = (TESTArray[TEST_TX_PSD_CONTROL] >> 8) & 0x000F;

   // R12 default code
   if ( us_TxPSD_Mode == TEST_TX_PSD_CONTROL_R12_DEFAULT_MODE )
   {
      // TODO: A cleanup should be done here like in R4(R11) code. I.e. combine the if-else
      // clauses regarding TEST_TX_PSD_CONTROL_R12_DEFAULT_MODE.

      //XDSLRTFW-2059 (START)
      if (pt_PSDCompDesc->ut_PSDRecord != 0)
      {
         if (TESTArray[TEST_TX_PSD_CONTROL] & TEST_TX_PSD_CONTROL_SEL_PSDCompensation )
         {
            s_compensationLevel =  PSDCompensation(s_ToneIdx, gt_PSDCompData.us_NumberOfCompBreakPoint, pt_PSDCompDesc);
            //s_compensationLevel_P = s_compensationLevel; //Debug
            s_attenLevel = s_attenLevel + s_compensationLevel;
            //Planning Removing this code as it is not required, but since US0 PSD violations are seen @Munich FUSED board, using offset to remove psd violations  //XDSLRTFW-3746

            if (TESTArray[TEST_TX_PSD_CONTROL] & TEST_TX_PSD_CONTROL_US0_BAND) //XDSLRTFW-3302
            {
               if (s_ToneIdx <= US0_ISDNDOUBLE_POTSQUAD_MAX_TONE)
               {
                  if (gs_TxIIRFilterSelect < OPTN_POTS_LP_Filter_Select) // For all VDSL filters(Not for US0 filters)
                  {
                     s_attenLevel = s_attenLevel+gs_TxPsd_us0_ReduceByMinus0pt7;
                  }
                  //   if (gs_TxIIRFilterSelect == OPTN_ISDN_Filter_Select)
                  //   {
                  //      s_attenLevel = s_attenLevel + 15;   // remove 1.5dB PSD boosting for ISDN US0 band
                  //   }
                  //   else if (gs_TxIIRFilterSelect >= OPTN_POTS_LP_Filter_Select) //35b mode US0 band Raise Edge violation(ISDN)
                  //   {
                  //      s_attenLevel = s_attenLevel - gs_TxPsdCtrl_dbg; // remove 4.5dB PSD boosting for POTS US0 band
                  //   }
                  //   else
                  //   {
                  //      s_attenLevel = s_attenLevel + 17;   // remove 1.7dB PSD boosting for non ISDN US0 band
                  //   }
                  //   gula_DebugVarsPalak[8] = (uint32) ((s_attenLevel << 16) | 0x1218);
               }
               //Earier Tx PSD tests were done with Non-Fused VRX518 devices. Because of this, Tx PSD test results showed PSD violations in No US0 VDSL profiles
               // Now Tx PSD tests were done with Fused VRX518 devices(With fusing info applied, Tx Path charactertiscs will be close to simulations model),
               // Tx PSD results show close 0.5dB less Tx PSD compared standard values. So
               //XDSLRTFW-3678 remove the fix "-0.5dB TX PSD offset for us1/us2/us3 in case of No US0(was done to remove the Tx PSD violations in No US0 case)"
               //if ( (gft_US0BandUsed == 0) && (s_ToneIdx > US0_ISDNDOUBLE_POTSQUAD_MAX_TONE) )
               //{
               //   s_attenLevel = s_attenLevel + 5; // remove .5dB PSD boosting us1/us2/us3 in case of No US0
               //   gula_DebugVarsPalak[9] = (uint32) ((s_ToneIdx << 16) | 0x1219);
               //}
            }
            gula_DebugVarsPalak[0] = (uint32) ((s_compensationLevel << 16) | 0x1210);
         }
         else
         {
            s_compensationLevel =  PSDCompensationStopAndHold2(s_ToneIdx,gt_PSDCompData.us_NumberOfCompBreakPoint, pt_PSDCompDesc);

            if (s_ToneIdx <= US0_ISDNDOUBLE_POTSQUAD_MAX_TONE)
            {
               if (gs_TxIIRFilterSelect == OPTN_ISDN_Filter_Select)
               {
                  s_attenLevel = s_attenLevel + s_compensationLevel + 5;  // remove 0.5dB PSD boosting for ISDN US0 band
               }
               else
               {
                  s_attenLevel = s_attenLevel + s_compensationLevel + 7;  // remove 0.7dB PSD boosting for non ISDN US0 band
               }
            }
            else
            {
               s_attenLevel = s_attenLevel + s_compensationLevel;

            }
         }
      }
   }
   // Following code is used for:
   // us_TxPSD_Mode = BRCM
   // us_TxPSD_Mode = KPN
   // us_TxPSD_Mode = DEBUG
   else
   {
      //XDSLRTFW-2059 (START)
      if (pt_PSDCompDesc->ut_PSDRecord != 0)
      {
         // US0 only
         // ALWAYS ON !!!!!!!!!!!!!!!!
         if (TESTArray[TEST_TX_PSD_CONTROL] & TEST_TX_PSD_CONTROL_SEL_PSDCompensation )
         {
            s_compensationLevel =  PSDCompensation(s_ToneIdx, gt_PSDCompData.us_NumberOfCompBreakPoint, pt_PSDCompDesc);
            s_attenLevel = s_attenLevel + s_compensationLevel;
         }
         // for all US configuration which are NOT US0-only
         else
         {
            s_compensationLevel =  PSDCompensationStopAndHold2(s_ToneIdx,gt_PSDCompData.us_NumberOfCompBreakPoint, pt_PSDCompDesc);
            s_attenLevel = s_attenLevel + s_compensationLevel;
         }
      }

      if (s_ToneIdx <= US0_ISDNDOUBLE_POTSQUAD_MAX_TONE)
      {
         // US0 boost:
         s_attenLevel = s_attenLevel - gt_TxPsdControl.s_US_PSD_Boost_0dB1[0]; // boost for 552kHz <= f
      }
      else if (s_ToneIdx < FIRST_BIN_US2)
      {
         // US1 boost:
         s_attenLevel = s_attenLevel - gt_TxPsdControl.s_US_PSD_Boost_0dB1[1]; // boost for 552kHz < f < 7.05MHz
      }
      else
      {
         // US2 / US3 boost:
         s_attenLevel = s_attenLevel - gt_TxPsdControl.s_US_PSD_Boost_0dB1[2]; // boost for f >= 7.05MHz
      }
   }
   //XDSLRTFW-2059 (END)

   //XDSLRTFW-2260 (Start)
   //Tx PSD compensation from external input (cnfg 88 0)
   if(pt_PSDCompDesc32->us_NumberOfTones)
   {
      //External Tx compensation function gt_UsPsdCompensationTable uses 4.3125Hz tone spacing
      s_ExternalPSDCompensation = PSDCompensationStopAndHold(s_ToneIdx << gs_frame_rate_is_8khz, pt_PSDCompDesc32->us_NumberOfTones, pt_PSDCompDesc32);
   }

   s_attenLevel = s_attenLevel+s_ExternalPSDCompensation;
   //XDSLRTFW-2260 (End)

   //Convert from attenuation in 0.1 dB format to gain in Q8.8 format
   l_dB_in = -(s_attenLevel - gs_min_atten_dB);

   //Disable the Code as it is not needed now Kept the code as it can be copied at place for capturing if needed
   //if (gus_DebugControlVRX518)
   //{
   //   gpsa_DebugBuf5Dk_16K[s_ToneIdx] = (int16)l_dB_in;
   //gpsa_DebugBuf5Dk_16K[s_ToneIdx] = (int16)s_attenLevel;
   // gula_DebugVarsPalak[19] = ((s_ToneIdx << 16) | gs_TxState);
   //}

   //XDSLRTFW-1602(START)
   //For RFI bands, limit the PSD to -83.5dbm/Hz if the PSD at the tone is greater than -83.5(As per the Standard)
   for(int k=0; k<gs_NumOfRFIBands; k++)
   {
      for (int j = gsa_RFIBandLeftChannel[k]; j <= gsa_RFIBandRightChannel[k]; j++)
      {
         if(s_ToneIdx==j)
         {
            //check for Tx direction and enter if (calculated PSD level > -80dbm/Hz)
            if ((*(ghpuca_TxExtGains_Inactive+j) & 0x10) && ((gt_TxPsdControl.s_MaxNomPsdIn+s_attenLevel) < RFI_PSD_LEVEL_83))

            {
               //Final PSD = -(gt_TxPsdControl.s_MaxNomPsdOut + l_dB_in)
               l_dB_in=-(RFI_PSD_LEVEL_83-gt_TxPsdControl.s_MaxNomPsdOut) ;

               // break out of both the loops as one tone per call can match
               k =gs_NumOfRFIBands;
               break;
            }
         }

      }
   }
   //XDSLRTFW-1602(END)

   l_dB_in = (l_dB_in<<8)/10;   //Convert to Q8.8 format

   if (l_dB_in > 32767)      // And cap this to be in the range of 16 bit integer
   {
      l_dB_in = 32767;
   }
   else if (l_dB_in < -32768)
   {
      l_dB_in = -32768;
   }

   //Convert gain to Q3.13 linear format
   db_to_linear((int16)l_dB_in, &s_mant, &s_exp);

   // s_exp should always be <= 0 since l_dB_in should be [-128.0, 0.0] dB
   s_exp = -s_exp;   // To convert to # of right shifts
   s_exp += 2;      // To convert from Q1.15 to Q3.13 format

   // round result and return
   s_linearPSD = (int16) ((s_mant+((1<<s_exp)>>1)) >> s_exp);

//   if ( (gus_DebugControlVRX518 & 0x1000) && (s_ToneIdx < 3000) )
//   {
//      gsa_dbg_StateTrail[s_ToneIdx] = (int16) s_compensationLevel_P;
//      gsa_dbg_StateTrail[s_ToneIdx+3000] = (int16) s_attenLevel_P;
//      gsa_dbg_StateTrail[s_ToneIdx] = (int16) l_dB_in
//      gsa_dbg_StateTrail[s_ToneIdx+3000] = (int16) s_linearPSD;
//    gula_DebugVarsPalak[19] = ((gs_TxState << 16) | s_ToneIdx);
//   }

   return(s_linearPSD);
}
