/* **COPYRIGHT******************************************************************
    INTEL CONFIDENTIAL
    Copyright (C) 2017 Intel Corporation
    Copyright (C), 1994-2006 Aware Inc. All Rights Reserved.
******************************************************************COPYRIGHT** */
/* **DISCLAIMER*****************************************************************
    The source code contained or described herein and all documents related
    to the source code ("Material") are owned by Intel Corporation or its
    suppliers or licensors. Title to the Material remains with Intel
    Corporation or its suppliers and licensors. The Material may contain
    trade secrets and proprietary and confidential information of Intel
    Corporation and its suppliers and licensors, and is protected by
    worldwide copyright and trade secret laws and treaty provisions. No part
    of the Material may be used, copied, reproduced, modified, published,
    uploaded, posted, transmitted, distributed, or disclosed in any way
    without Intel's prior express written permission.

    No license under any patent, copyright, trade secret or other
    intellectual property right is granted to or conferred upon you by
    disclosure or delivery of the Materials, either expressly, by
    implication, inducement, estoppel or otherwise. Any license under
    such intellectual property rights must be express and approved by
    Intel in writing.
*****************************************************************DISCLAIMER** */
/*
 *------------------------------------------------------------------------
 *
 *   Aware DMT Technology. Proprietary and Confidential.
 *
 *   40 Middlesex Turnpike, Bedford, MA 01730-1413 USA
 *   Phone (781) 276 - 4000
 *   Fax   (781) 276 - 4001
 *
 *   SharedFuncs.c
 *
 *   This file contains functions shared by different part of program
 *------------------------------------------------------------------------
 */

#include "common.h"
#include "gdata.h"
#include "fifo.h"
#include "IRI_Iof.h"
#include "mul.h"
#include "vdsl_state.h"
#include "vdsl_xception.h"
#include "str_memmap.h"
#include "SharedFuncs.h"


#ifdef  INCLUDE_NON_CACHE_FUNCTIONS
/*^^^
 *---------------------------------------------------------------------------------
 *
 *  Name : CalcTxRxFrameOffset
 *
 *  Description:  This function is used to compute the number of samples that
 *                TX is ahead of RX (if the result is negative, that means the RX frame start is ahead the TX
 *                frame start).
 *
 *  Prototype:  int CalcTxRxFrameOffset(void)
 *
 *  Input Arguments:
 *
 *  Output Arguments:
 *
 *  Return: TX and RX frame offset (in IFFT samples)
 *
 *  Global Variables Used:
 *      gs_TxTimerVal           -- (I) TX timer value (in DSP clock cycles)
 *      gs_TxTimerVal           -- (I) RX timer value (in DSP clock cycles)
 *      gs_frame_rate_is_8khz   -- (I) the frame rate indication flag
 *
 *----------------------------------------------------------------------------------
 *^^^
 */
int16 CalcTxRxFrameOffset(void)
{
   int16 s_tx_lead_rx_amount;

   int16 s_FrameLength, s_temp;
   int16 s_TxFrameStartOffset, s_RxFrameStartOffset;
   int16 s_TxCPLength, s_TxBetaLength, s_RemoteTxCPLength, s_RemoteTxBetaLength;
   int16 s_RxCPLength;

   // All the predefined TA variables are defined assuming max tone IDFT size of the Chip, i.e.
   //     VRx510 = IFFT 16384 (8192 tones)/ 8192 (4096 tones)
   //              Note: Actual VRx518 35b: TX - IFFT 8192 and RX - FFT 16384
   //     VR9/VRx318 = IFFT 8192 (4096 tones).
   //
   // Therefore first express all variables in the same max sample rate.
   int16 s_MaxLog2IFFTLength_Reference = TX_LOG2_MAX_FFT_LENGTH;                   // Should be out of GHS.

   // Use info from Strymon Register directly, which is in units of the current IFFT size.
   // !!! Note: The TX and RX buffer swap time occurs at the beginning of each cyclic prefix!!!!!!
   s_tx_lead_rx_amount = (gl_txRxFrameOffset & V_FRAME_SKEW_MASK);

   // Convert s_tx_lead_rx_amount to the IFFT sampling rate corresponding to max sample rate.
   s_temp = (s_MaxLog2IFFTLength_Reference - gs_TxLog2IfftLength);
   s_tx_lead_rx_amount = s_tx_lead_rx_amount << s_temp;

   // In VDSL2, the frame start point used for the timing advance is defined to be
   // floor(beta/2) samples after the first sample of the cyclic prefix.
   // So the offset must be re-adjusted as below.
   s_TxBetaLength = gs_TxBetaLength << s_temp;
   s_TxCPLength = (gs_TxCPLength << s_temp);

   // Consider the difference between RX configuration and the real used far end TX configuration!
   s_temp = (s_MaxLog2IFFTLength_Reference - gs_RxLog2FftLength);
   if (s_temp >= 0)
   {
      s_RxCPLength = gs_RxCPLength << s_temp;
   }
   else
   {
      s_RxCPLength = gs_RxCPLength >> (-s_temp);
   }

   // Note: The remote CP and Beta values must belong to the DS TX side, i.e. not already be normalized to DS RX (CPE).
   s_temp = (s_MaxLog2IFFTLength_Reference - gs_FeModemLog2IfftSize);
   if (s_temp >= 0)
   {
      s_RemoteTxCPLength = gs_RemoteTxCPLength << s_temp;
      s_RemoteTxBetaLength = gs_RemoteTxBetaLength << s_temp;
   }
   else
   {
      s_RemoteTxCPLength = gs_RemoteTxCPLength >> (-s_temp);
      s_RemoteTxBetaLength = gs_RemoteTxBetaLength >> (-s_temp);
   }

   // TX symbol TA reference point is floor(beta/2) samples after the first sample
   // of cyclic prefix.
   {
      s_TxFrameStartOffset = -(s_TxBetaLength>>1);                                             // beta_us/2
      s_RxFrameStartOffset = (-s_RxCPLength + s_RemoteTxCPLength) - (s_RemoteTxBetaLength>>1); // beta_ds/2

      // This if-case is needed for "sleep"-state.
      if(gft_SkipCpAdjust == 0)
      {
         int16 s_TaOffset;

         // Note: positive value - TX frame start is ahead of RX frame start
         //       negative value - TX frame start is behind of RX frame start
         s_TaOffset = (s_TxFrameStartOffset - s_RxFrameStartOffset);
         if (gs_DbgRecnfgCpCsBeta & TX_TIMING_ADVANCE_ZERO_EN)
         {
            s_TaOffset = 0;
         }
         // Note:For BDCM 35b an additional offset is needed!
         s_TaOffset = gs_TimingAdvanceOffset - s_TaOffset;

         if(gs_TimingAdvance == 0)
         {
            int16 s_TargetTimingAdvance;

            // NOTE: entering this substate,
            //       - gs_TimingAdvance and gs_TargetTimingAdvance are in samples at the US IDFT sampling rate
            //       - s_TxRxOffset and TotalFilterDelay = (gs_TimingAdvanceFixedDelay + gs_TimingAdvanceTxIirDelay + gs_TimingAdvanceRxIirDelay) are
            //         in samples at 4096 IDFT sample rate
            //
            // First convert all variables in samples expressed at the 4096 IDFT sample rate.
            // Note: In VDSL2, gs_TargetTimingAdvance is obtained either from
            //               - O-Signature message or
            //               - Default value (if the O-Signature sets the initial TA value to 0x7FFF).
            //
            //Change the target timing advance value for test purpose
            if(gs_DbgTargetTimingAdvance >= 0)
            {
               gs_TargetTimingAdvance = gs_DbgTargetTimingAdvance;
            }
            s_temp = s_MaxLog2IFFTLength_Reference-gs_TxLog2IfftLength;
            s_TargetTimingAdvance = gs_TargetTimingAdvance << s_temp;

            // Correction of
            //    - TA reference point difference
            //    - "TotalFilterDelay = (gs_TimingAdvanceFixedDelay + gs_TimingAdvanceTxIirDelay + gs_TimingAdvanceRxIirDelay)"
            //       (i.e. the Rx, Tx path processing delay due to filters)
            //    - "gs_TargetTimingAdvance" (i.e. channel delay)
            // XDSLRTFW-3737 (Start_End)
            s_TaOffset += (s_TargetTimingAdvance + ((gs_TimingAdvanceFixedDelay + gs_TimingAdvanceTxIirDelay + gs_TimingAdvanceRxIirDelay) & 0xFFFE));
         }
         // Note: s_tx_lead_rx_amount is random value between [0 to FrameLength]!
         //       Therefore s_tx_lead_rx_amount is normally a bigger negative value or a
         //       small positve value
         // Note: To get a TA of zero the full actual s_tx_lead_rx_amount must be shifted, i.e. make it negative!
         //       Afterwards correct the CP-Beta difference.
         //       Case 1: TX frame start is ahead of RX frame start  -> It must be shifted to the left,  i.e. +TA.
         //       Case 2: TX frame start is behind of RX frame start -> It must be shifted to the right, i.e. -TA.
         s_tx_lead_rx_amount = s_TaOffset - s_tx_lead_rx_amount;
      }

      // CELength = (m*N/32) = 5*4096/32 = 640 or 5*8192/32 = 1280
      // Note: CELength must be also on the max sample rate.
      // Compute Frame Length = max FFTLength + CELength, with CELength = m*N/32
      // - 17a: Frame Length = 8192+640 = 8832, with temp = 5*(8192/64) = 640
      //        -> 8832/2 = 4416
      s_FrameLength = (1<<s_MaxLog2IFFTLength_Reference);
      {
         int32 l_Acc;

         s_temp = (s_FrameLength>>6);
         MULS16(l_Acc, gs_m, s_temp);
         s_FrameLength += (int16)l_Acc;
      }

      // Decide how to shift the frame!
      if (gs_DbgRecnfgCpCsBeta & TX_TIMING_ADVANCE_DELAY_EN)
      {
         if (gs_TimingAdvanceTrailIndex == 0)
         {
            if (s_tx_lead_rx_amount > 0) // (int16)(640*4))
            {
                s_tx_lead_rx_amount -= s_FrameLength;
            }
         }
         else
         {
            s_temp = (s_FrameLength >> 1);
            if (s_tx_lead_rx_amount < -s_temp)
            {
               s_tx_lead_rx_amount += s_FrameLength;
            }
            else if (s_tx_lead_rx_amount > s_temp)
            {
               s_tx_lead_rx_amount -= s_FrameLength;
            }
         }
      }
      else
      {
         s_temp = (s_FrameLength >> 1);
         if (s_tx_lead_rx_amount < -s_temp)
         {
            s_tx_lead_rx_amount += s_FrameLength;
         }
         else if (s_tx_lead_rx_amount > s_temp)
         {
            s_tx_lead_rx_amount -= s_FrameLength;
         }
      }
   }


   return(s_tx_lead_rx_amount);

}
#endif //  INCLUDE_NON_CACHE_FUNCTIONS




#ifdef INCLUDE_CACHE_FUNCTIONS

/*^^^
*-------------------------------------------------------------------
*
*  Prototype: int16 ClusterIndex(int16 s_channel)
*
*  Description:
*      This function compute the cluster index (each cluster contains tone from m*10 to m*10+9)
*  Input:
*
*  s_channel -- the input channel number
*
*  Output:
*       None
*
*  Return:
*       floor(s_channel/10)
*
*-------------------------------------------------------------------
*^^^
*/
int16 ClusterIndex(int16 s_channel)
{
   int32 l_Acc;
   int16 s_cluster_index;

   //Compute the first cluster index by floor(s_channel/10)
   //= s_channel*(0.1*(2^14))/(2^14)=(gs_leftChannel*1638)>>14, where 1638=0.1*(2^14)
   l_Acc = s_channel*1638;
   l_Acc >>= 14;
   s_cluster_index = (int16)l_Acc;

   //Take care of special case
   if(((s_cluster_index*10)+TONE_CLUSTER_END_INDEX) < s_channel)
   {
      s_cluster_index++;
   }

   return(s_cluster_index);
}


/*^^^
 *------------------------------------------------------------------------
 *
 *  Name : SelectBand
 *
 *  Prototype:  void SelectBand(int16 *psa_SnrBuf, int16 s_SelectBandSize, int16 s_SnrThresh, int16 *ps_SelectBandStart, int16 *ps_AvgSnr)
 *
 *  This function selects the band with best metric.
 *
 *   It is assured that *ALL* the tones in the band selected are in the Rx
 *   supported set AND that all the tones have a metric (snr) that is greater
 *   than a threshold
 *
 *  Input Arguments:
 *      psa_SnrBuf -- (I) pointer to array of per tone metric
 *      s_SelectBandSize -- (I) num of tones in band to be selected
 *      s_SnrThresh -- (I) threshold SNR which resets metric (in Q8.8)
 *
 *  Output Arguments:
 *      ps_SelectBandStart -- (O) the first tone index of selected band
 *      ps_AvgSnr -- (O) average SNR within s_SelectBandSize
 *
 *   Return:
 *      NONE
 *
 *  Global Variables Used:
 *      guca_RxSupportedToneSet -- (I) supported tone sets
 *
 *  Notes:
 *
 *------------------------------------------------------------------------
 *^^^
 */
void SelectBand(int16 *psa_SnrBuf, int16 s_SelectBandSize, int16 s_SnrThresh, int16 *ps_SelectBandStart, int16 *ps_AvgSnr)
{
   int16 i, s_ch, s_BandStart;
   int32 l_Metric, l_BestBandMetric;

   l_BestBandMetric = -1; // initialize to min
   *ps_SelectBandStart = -1;

   for (i=0; i<gs_NumOfRxBands; i++)
   {
      // start fresh from the band start
      s_BandStart = gsa_RxBandLeftChannel[i];
      l_Metric = 0;
      for (s_ch=gsa_RxBandLeftChannel[i]; s_ch<=gsa_RxBandRightChannel[i]; s_ch++)
      {
         // no need to look further
         if (s_ch > gs_HighestAllowedRxTone)
         {
            break;
         }

         if ((!IS_TONEFLAGSET(guca_RxSupportedToneSet, s_ch)) ||
               (psa_SnrBuf[s_ch] < s_SnrThresh))
         {
            s_BandStart = s_ch + 1;
            l_Metric = 0;
         }
         else
         {
            l_Metric += psa_SnrBuf[s_ch];

            if ((s_ch-s_BandStart+1) >= s_SelectBandSize)
            {
               // Start tone of band with best SNR metric
               if (l_Metric > l_BestBandMetric)
               {
                  l_BestBandMetric = l_Metric;
                  *ps_SelectBandStart = s_BandStart;
               }

               l_Metric -= psa_SnrBuf[s_BandStart];
               s_BandStart++;
            }
         }
      }
   }

   *ps_AvgSnr = l_BestBandMetric / s_SelectBandSize;
}

/*^^^
 *------------------------------------------------------------------------
 *
 *  Name : BgSelectRxMsgDecodeTones
 *
 *  Prototype:  void BgSelectRxMsgDecodeTones(void)
 *
 *  This function selects the tone cluster(s) which will be used for decoding
 *  the RX message
 *
 *  Input Arguments:
 *      NONE
 *
 *  Output Arguments:
 *      NONE
 *
 *   Return:
 *      NONE
 *
 *  Global Variables Used:
 *      gs_NumOfToneClustersForMsgDecode -- (O) the number of tone clusters used for message decoding
 *      gsa_RxBandRightChannel[] -- (I) array of right band edges
 *      gsa_RxBandLeftChannel[] -- (I) array of left band edges
 *      gs_RxToneOffset   -- (O) the first tone index to be loaded to RxToneBuf[]
 *      gs_RxNumChannelsForMsgDecode -- (O) the total number of consecutive tones used for message decoding.
 *  Notes:
 *
 *------------------------------------------------------------------------
 *^^^
 */
void BgSelectRxMsgDecodeTones(void)
{
   int16 i, s_AvgSnr, s_NumOfToneClusters, s_MinBandStart;
   int16 s_BandSize0, s_NewNumOfToneClusters, s_NewBandSize, s_NewBandStart, s_NewAvgSnr;


   // Find NUM_TONE_CLUSTERS_FOR_MSG_DECODING tone clusters with good SNRs for msg decoding.
   // These clusters must reside within one group of NUM_CHANNELS_PER_GROUP contigous tones.
   // But these clusters don't have to be consecutive.
   // In this implementation, we force these clusters to be consecutive to simplify the computation.
   gpsa_MeasuredSnrBuf = gsa_SnrBuf;
   gs_SelectBandStart = -1;
   s_NumOfToneClusters = NUM_TONE_CLUSTERS_FOR_MSG_DECODING;

   // Clear the pilot tones without data-on-pilot enabled from supported set to avoid bit-loading
   ClearSetPilotTonesInSupportedSet((int16)CLEAR_TONES_SUPPORTED_SET);

   // keep trying until we select the start tone index, or # of clusters is reduced to 1
   while ((gs_SelectBandStart == -1) && (s_NumOfToneClusters >= 1))
   {
      gs_SelectBandSize = (s_NumOfToneClusters+1)*NUM_CHANNELS_PER_CLUSTER;
      SelectBand(gpsa_MeasuredSnrBuf, gs_SelectBandSize, (int16)(10<<8), &gs_SelectBandStart, &s_AvgSnr);
      s_NumOfToneClusters--;
   }
   gs_NumOfToneClustersForMsgDecode = s_NumOfToneClusters + 1;
   gs_RxNumChannelsForMsgDecode = gs_NumOfToneClustersForMsgDecode*NUM_CHANNELS_PER_CLUSTER;

   // if selected band size is greater than the first band size, then
   // examine the possiblity of using the first band with the smaller # of clusters
   // this is particularily necessary for CO enabled with narrow US band 0 in the long loop
   s_BandSize0 = gsa_RxBandRightChannel[0] - gsa_RxBandLeftChannel[0] + 1;
   if (gs_SelectBandSize > s_BandSize0)
   {
      s_NewNumOfToneClusters = (s_BandSize0/NUM_CHANNELS_PER_CLUSTER) - 1;
      if (s_NewNumOfToneClusters > 0)
      {
         s_NewBandSize = (s_NewNumOfToneClusters+1)*NUM_CHANNELS_PER_CLUSTER;
         SelectBand(gpsa_MeasuredSnrBuf, s_NewBandSize, (int16)(10<<8), &s_NewBandStart, &s_NewAvgSnr);
         // if new band gives better SNR
         if (s_NewAvgSnr > s_AvgSnr)
         {
            gs_NumOfToneClustersForMsgDecode = s_NewNumOfToneClusters;
            gs_RxNumChannelsForMsgDecode = s_NewNumOfToneClusters*NUM_CHANNELS_PER_CLUSTER;
            gs_SelectBandSize = s_NewBandSize;
            gs_SelectBandStart = s_NewBandStart;
         }
      }
   }

   // if we cannot find a good tone index with the minimum window size,
   // which is 2*NUM_CHANNELS_PER_CLUSTER, then try with lower SNR threshold
   // to find the best start tone index, and set # of tone cluster to its minimum (1)
   if (gs_SelectBandStart == -1)
   {
      gs_SelectBandSize = NUM_CHANNELS_PER_CLUSTER;
      SelectBand(gpsa_MeasuredSnrBuf, gs_SelectBandSize, (int16)(5<<8), &gs_SelectBandStart, &s_AvgSnr);
      // check if the start tone is set too high
      // this is possible especially when SNR is increasing (due to discrimination filter)
      // since we used the minimum window size instead of minimum + 1
      s_MinBandStart = (gsa_RxBandRightChannel[0] - 10)/10;
      s_MinBandStart *= 10;
      if ((gs_SelectBandStart < gsa_RxBandRightChannel[0]) && (gs_SelectBandStart > s_MinBandStart))
      {
         gs_SelectBandStart = s_MinBandStart;
      }
      gs_NumOfToneClustersForMsgDecode = 1;
      gs_RxNumChannelsForMsgDecode = NUM_CHANNELS_PER_CLUSTER;
   }

   // overwrite tone offset for debug purpose
   if (gs_dbg_ToneOffset != 0)
   {
      gs_SelectBandStart = gs_dbg_ToneOffset;
   }

   //Set the RX tone offset for capturing tones in RTV buffer
   gs_RxToneOffset = (gs_SelectBandStart+9)/10;
   gs_RxToneOffset *= 10;
   AddFunctionToFifo(gp_RxLoadingFunctionFifo, SetRxToneOffset);

   //Store all cluster starting tone index relative to gs_RxToneOffset.
   for (i=0; i < gs_NumOfToneClustersForMsgDecode; i++)
   {
      gsa_SOCMsgRxToneClusterStarts[i] = i*NUM_CHANNELS_PER_CLUSTER;
   }

   // Add back the pilot tone indexes
   ClearSetPilotTonesInSupportedSet((int16)SET_TONES_SUPPORTED_SET);


   gs_RxBkgdProcessFlag = TRAINING_DONE;
}

/*^^^
 *------------------------------------------------------------------------
 *
 *  Name : BgStreamRxEchoPsd
 *
 *  Prototype:  void BgStreamRxEchoPsd(void)
 *
 *  This function scales the RxEchoPsd for comparison in Tracee
 *  the RX message
 *
 *  Input Arguments:
 *      NONE
 *
 *  Output Arguments:
 *      NONE
 *
 *   Return:
 *      NONE
 *
 *  Global Variables Used:
 *      gpsa_MeasuredSnrBuf                      -- (I) pointer to measured PSD buffer
 *      gs_PGA_set                               -- (I) current PGA setting, in dB (Q8.8)
 *      gs_HybridGain                            -- (I) Gain from currently used Hybrid, in dB (Q8.8)
 *      gt_DfeAfeGainSettings.s_Rx_VGWin_Gain_dB -- (I) RxVarGain 0dB during Gain training, after PGA2 train = Avg-Peak power, in dB (Q8.8)
 *
 *  Notes:
 *
 *------------------------------------------------------------------------
 *^^^
 */
void BgStreamRxEchoPsd(void)
{
   int16 i;
   int16 s_RxPathGain;

   // Code to scale the RxEchoPsd for comparison.
   // Note:
   // - Hybrid gain is per loop same. But for loop reach comparision it should be considered too.
   // - The scaling is not done for RxEchoPsd stored in SRAM array "gsa_DiscPsdBuf_EchoOn", which is used from DCT.
   //   Reason is that also IV-team is using this RxEchoPsd with their own scaling.
   s_RxPathGain = (gs_PGA_set+gs_HybridGain+gt_DfeAfeGainSettings.s_Rx_VGWin_Gain_dB);
   for (i=0; i < gs_RxNumTones; i++)
   {
      gpsa_MeasuredSnrBuf[i] = gpsa_MeasuredSnrBuf[i] - s_RxPathGain;
   }
   DSH_SendStream(DHS_RX_PSD,gs_RxNumTones*sizeof(int16),(void*)gpsa_MeasuredSnrBuf);

   gs_RxBkgdProcessFlag = TRAINING_DONE;
}
#endif // INCLUDE_CACHE_FUNCTIONS
