/* **COPYRIGHT******************************************************************
    INTEL CONFIDENTIAL
    Copyright (C) 2017 Intel Corporation
    Copyright (C), 1994-2004 Aware Inc. All Rights Reserved.
******************************************************************COPYRIGHT** */
/* **DISCLAIMER*****************************************************************
    The source code contained or described herein and all documents related
    to the source code ("Material") are owned by Intel Corporation or its
    suppliers or licensors. Title to the Material remains with Intel
    Corporation or its suppliers and licensors. The Material may contain
    trade secrets and proprietary and confidential information of Intel
    Corporation and its suppliers and licensors, and is protected by
    worldwide copyright and trade secret laws and treaty provisions. No part
    of the Material may be used, copied, reproduced, modified, published,
    uploaded, posted, transmitted, distributed, or disclosed in any way
    without Intel's prior express written permission.

    No license under any patent, copyright, trade secret or other
    intellectual property right is granted to or conferred upon you by
    disclosure or delivery of the Materials, either expressly, by
    implication, inducement, estoppel or otherwise. Any license under
    such intellectual property rights must be express and approved by
    Intel in writing.
*****************************************************************DISCLAIMER** */
/*
 *-------------------------------------------------------------------------
 *
 *   Aware DMT Technology. Proprietary and Confidential.
 *
 *   40 Middlesex Turnpike, Bedford, MA 01730-1413
 *   Phone (781) 276 - 4000
 *   Fax   (781) 276 - 4001
 *
 *   RX_OPS.C
 *
 *   Receive State operations
 *
 *-------------------------------------------------------------------------
 */


#include "common.h"
#include "gdata.h"
#include "ghs.h"
#include "dsp_op.h"
#include "mul.h"
#include "cmv.h"

#define RXPWR_GUARD_BITS        8
//XDSLRTFW-636
#define GHS_GUARD_BITS_RXPWR    (4)
#define DETECT_MIN              0x000A

/*^^^
 *------------------------------------------------------------------------
 *
 *  Name : RxPower
 *
 *  Description:  Computes the total power over the receiver bandwidth
 *
 *  Prototype:
 *      int32 RxPower(int16 *psa_inbuf,int16 s_first_channel, int16 s_last_channel)
 )
 *
 *  Input Arguments:
 *      int16 *psa_inbuf - pointer to input time samples
 *      int16  s_first_channel   - first channel in the power calculation
 *      int16  s_last_channel   - last channel in the power calculation
 *                int16  s_guard_bits     - number of  bits to round
 *
 *  Output Arguments:
 *
 *  Return:
 *      int32 l_rx_power    - total power in receiver band
 *
 *
 *  Global Variables Used:
 *
 *  Notes:
 *
 *------------------------------------------------------------------------
 *^^^
 */

int32 RxPower(int16 *psa_inbuf,int16 s_first_channel, int16 s_last_channel, int16 s_guard_bits)
{

   int32   l_rx_power;
   int32   l_ipower;
   int32   l_ipower_x;
   int32   l_ipower_y;
   int16   sX, sY;
   int16    i;
   l_rx_power = 0;
   for (i = s_first_channel; i <= s_last_channel; i++)
   {
      sX = psa_inbuf[2 * i];
      sY = psa_inbuf[2 * i + 1];

      MULS16(l_ipower_x,sX,sX);
      MULS16(l_ipower_y,sY,sY);

      l_ipower = l_ipower_x + l_ipower_y;

      l_ipower = round(l_ipower, s_guard_bits);
      l_rx_power = l_add(l_rx_power, l_ipower);
   }

   return l_rx_power;

}


/*****************************************************************************
;   Subroutine Name: DetectTone(...)
;
;   This subroutine detects if the data in the input buffer contains the
;   tone of interest. The function returns 1 if the tone is detected,
;   0 otherwise. The algorithm used for the tone detection is based on the
;   following equation:
;
;      Power_of_Tone
;   ----------------------   > Detection_Threshold
;      Average_Power
;
;   where Power_of-Tone is the power of the tone to be detected,
;   Average_Power is the average power of tones around the detected tone, excluding
;   the detected tone. Average power is measured over a window of tones as
;   s_avg_power = s_rx_power / (s_last_tone-s_first_tone)
;
;   Prototype:
;      int16 DetectTone(int16 *psa_inbuf, int16 s_tone_idx, int16 s_first_tone, int16 s_last_tone, int16 s_threshold)
;
;   Input Arguments:
;      psa_inbuf      -- pointer to the input buffer containing received DMT tones
;      s_tone_idx      -- index of the tone to be detected;
;      s_first_tone   -- the index of the first tone used in the detection
;      s_last_tone      -- the index of the last tone used in the detection
;      s_threshold      -- detection threshold in Q8.8 format
;
;   Output Arguments:
;      none
;
;   Return:
;      0 -- the tone is not detected;
;      1 -- the tone is detected.
;
;   Global Variables:
;      none
;
*******************************************************************************/
int16 DetectTone(int16 *psa_inbuf, int16 s_tone_idx, int16 s_first_tone, int16 s_last_tone, int16 s_threshold)
{

   int16 s_tone_power, s_rx_power;
   int32 l_tone_power, l_rx_totpower, l_rx_power;
   int16 s_retval, s_exp1, s_exp2;
   int16 s_guard_bits = GHS_GUARD_BITS_RXPWR;
#ifdef DEBUG_TRAIL
// Debug code for DCT - start
//if(gs_RxState == 161)
//{
//   int16 *psaRxTones;
//   int16  i;
//
//   psaRxTones = psa_inbuf;
//   for (i = s_first_tone; i <= s_last_tone; i++)
//   {
//        gsa_StatesTrail[gs_Debug1++] = psaRxTones[2 * i];
//        gsa_StatesTrail[gs_Debug1++] = psaRxTones[2 * i + 1];
//   }
////   gsa_StatesTrail[gs_Debug1++] = psaRxTones[2 * 66];
////   gsa_StatesTrail[gs_Debug1++] = psaRxTones[2 * (66 + 1)];
////   gsa_StatesTrail[gs_Debug1++] = psaRxTones[2 * 80];
////   gsa_StatesTrail[gs_Debug1++] = psaRxTones[2 * (80 + 1)];
////   gsa_StatesTrail[gs_Debug1++] = psaRxTones[2 * 88];
////   gsa_StatesTrail[gs_Debug1++] = psaRxTones[2 * (88 + 1)];
//
//   if(gs_Debug1 >= 512)
//      gs_Debug1 = 0;
//}
// Debug code for DCT - end
#endif // DEBUG_TRAIL
//XDSLRTFW-636
   if ((TESTArray[TEST_Control3] & TEST_GHS_Fix) == 0)
   {
      s_guard_bits = RXPWR_GUARD_BITS;
   }

   l_rx_totpower = RxPower(psa_inbuf, s_first_tone, s_last_tone,s_guard_bits);

   //If the total power is less than the minimum power threshold, declare no tone
   if((l_rx_totpower) <= DETECT_MIN)
   {
      s_retval = 0;      // no detection declared
#ifdef DEBUG_TRAIL
      // Debug code for DCT - start
//if(gs_RxState == 161)
//{
//   gsa_StatesTrail[gs_Debug1++] = (DETECT_MIN & 0xFFFF);
//   gsa_StatesTrail[gs_Debug1++] = ((0 >> 16)& 0xFFFF);
//   gsa_StatesTrail[gs_Debug1++] = (DETECT_MIN & 0xFFFF);
//   gsa_StatesTrail[gs_Debug1++] = ((0 >> 16)& 0xFFFF);
//   if(gs_Debug1 >= 512)
//      gs_Debug1 = 0;
//}
// Debug code for DCT - end
#endif // DEBUG_TRAIL
      return(s_retval);
   }

   // Compute power of desired tone and the rx power
//XDSLRTFW-636
   l_tone_power = RxPower(psa_inbuf, s_tone_idx, s_tone_idx,s_guard_bits);
   l_rx_power = l_add(l_rx_totpower, -l_tone_power);
#ifdef DEBUG_TRAIL
   // Debug code for DCT - start
//if(gs_RxState == 161)
//{
//   gsa_StatesTrail[gs_Debug1++] = (l_tone_power & 0xFFFF);
//   gsa_StatesTrail[gs_Debug1++] = ((l_tone_power >> 16)& 0xFFFF);
//   gsa_StatesTrail[gs_Debug1++] = (l_rx_power & 0xFFFF);
//   gsa_StatesTrail[gs_Debug1++] = ((l_rx_power >> 16)& 0xFFFF);
//   if(gs_Debug1 >= 512)
//      gs_Debug1 = 0;
//}
// Debug code for DCT - end
#endif // DEBUG_TRAIL
   //Convert both l_tone_power and l_rx_power to 16-bit value
   s_exp1 = s_exp2 = 31;

   if(l_tone_power != 0)
   {
      s_exp1 = norm_l(l_tone_power);
   }
   if(l_rx_power != 0)
   {
      s_exp2 = norm_l(l_rx_power);
   }

   if(s_exp1 < s_exp2)
   {
      l_tone_power <<= s_exp1;
      l_rx_power   <<= s_exp1;
   }
   else
   {
      l_tone_power <<= s_exp2;
      l_rx_power   <<= s_exp2;
   }
   s_tone_power = (int16)(l_tone_power>>16);
   s_rx_power   = (int16)(l_rx_power>>16);

   //====================================================================
   // Detection declared if:
   // s_tone_power    > s_threshold * s_avg_power
   //                 > s_threshold * (s_rx_power / (s_last_tone-s_first_tone))
   // (s_last_tone-s_first_tone) * s_tone_power > s_threshold * s_rx_power
   //
   //====================================================================
   {
      int16  i;
      i = s_last_tone - s_first_tone;
      l_tone_power = (int32)s_tone_power*i;
   }
   l_rx_power = (int32)s_threshold*s_rx_power;

   //Since s_threshold is in Q8.8 format, perform the following rounding
   //and then right-shift 8 bits
   l_rx_power = round(l_rx_power, 8);

   if (l_tone_power > l_rx_power)
   {
      s_retval = 1;   // detection declared
   }
   else
   {
      s_retval = 0;   // no detection declared
   }

   return(s_retval);

}

/*^^^
 *------------------------------------------------------------------------
 *
 *  Name : RxPowerGHS
 *
 *  Description:  Computes the average power of 3 handshake tones
 *
 *  Prototype:
 *      int32 RxPowerGHS(const int16 sa_CarSet[], int16 s_RightShift)
 *
 *  Input Arguments:
 *      sa_CarSet[] - Frequency bins of the carrier set to be used
 *      s_RightShift -- number of right shift
 *
 *  Output Arguments:
 *      None.
 *
 *  Return:
 *      average power of 3 received handshake tones
 *
 *  Global Variables Used:
 *      gpsa_RxToneBuf - received frequency domain samples
 *
 *  Notes:
 *
 *------------------------------------------------------------------------
 *^^^
 */

int32 RxPowerGHS(int16 sa_CarSet[], int16 s_RightShift)
{
   int16 i, j;
   int32 l_Power;
   int32 l_AvgPower = 0;

   for(i=0; i<NUM_CARRIER; i++)
   {
      j = 2 * sa_CarSet[i];

      l_Power = (int32)gpsa_RxToneBuf[j] * gpsa_RxToneBuf[j];
      l_Power += (int32)gpsa_RxToneBuf[j+1] * gpsa_RxToneBuf[j+1];

      //approximate the number of carriers as the power of 2 greater than the number of carriers
      l_Power >>= s_RightShift;
      l_AvgPower += l_Power;
   }

   return l_AvgPower;
}

/*
 *------------------------------------------------------------------------
 *
 *  Name : DetectMultiTones
 *
 *  Abstract :
 *
 *  DetectMultiTones() - Detect tones in given frequency bins of the carrier set.
 *  mojority of the carrier set tones (e.g. 2 out of 3) have to be present to return 1,
 *  otherwise return 0.
 *
 *  Parameters:
 *     int16 sa_CarSet[] - Frequency bins of the carrier set to be used
 *     int16 s_NumCar      - No. of the carrier frequencies
 *     int16 s_threshold - detection scale
 *
 *  Returns:
 *      1   - tone present in all the desired bins
 *      0   - tone not present in all the desired bins
 *
 *  Global Variables Used:
 *      gpsa_RxToneBuf   - (O) frequency domains samples
 *
 *  Notes :
 *
 *------------------------------------------------------------------------
 */

int16 DetectMultiTones(int16 sa_CarSet[], int16 s_NumCar, int16 s_threshold)
{

   int16 i;
   int16 s_FirstTone, s_LastTone;
   int16 s_ToneFlag;

   //Compute RX tones if they haven't been computed in GetRxTone()

   //---- detect tones ----
   s_ToneFlag = 0;
   for ( i = 0; i < s_NumCar; i++ )
   {
      s_FirstTone = sa_CarSet[i] - HALF_DETECT_BAND;
      s_LastTone  = sa_CarSet[i] + HALF_DETECT_BAND;
      s_ToneFlag += DetectTone(gpsa_RxToneBuf, sa_CarSet[i], s_FirstTone, s_LastTone, s_threshold );
   }


   if ( s_ToneFlag >= 2 )  // 2-out-of-3 decision
   {
      s_ToneFlag = 1;
   }
   else
   {
      s_ToneFlag = 0;
   }

   return ( s_ToneFlag );

}   // DetectMultiTones
