/* **COPYRIGHT******************************************************************
    INTEL CONFIDENTIAL
    Copyright (C) 2017 Intel Corporation
    Copyright (C), 1994-2007 Aware Inc. All Rights Reserved.
******************************************************************COPYRIGHT** */
/* **DISCLAIMER*****************************************************************
    The source code contained or described herein and all documents related
    to the source code ("Material") are owned by Intel Corporation or its
    suppliers or licensors. Title to the Material remains with Intel
    Corporation or its suppliers and licensors. The Material may contain
    trade secrets and proprietary and confidential information of Intel
    Corporation and its suppliers and licensors, and is protected by
    worldwide copyright and trade secret laws and treaty provisions. No part
    of the Material may be used, copied, reproduced, modified, published,
    uploaded, posted, transmitted, distributed, or disclosed in any way
    without Intel's prior express written permission.

    No license under any patent, copyright, trade secret or other
    intellectual property right is granted to or conferred upon you by
    disclosure or delivery of the Materials, either expressly, by
    implication, inducement, estoppel or otherwise. Any license under
    such intellectual property rights must be express and approved by
    Intel in writing.
*****************************************************************DISCLAIMER** */
/*
*-------------------------------------------------------------------------------
*
*   Aware DMT Technology. Proprietary and Confidential.
*
*   40 Middlesex Turnpike, Bedford, MA 01730-1413
*   Phone (781) 276 - 4000
*   Fax   (781) 276 - 4001
*
*   filename: Framing_VDSL2.c
*
*   This file contains functions to initialize showtime variables.
*
*-------------------------------------------------------------------------------
*/
#include <string.h>
#include "common.h"
#include "gdata.h"
#include "cmv_Data.h"
#include "fifo.h"
#include "InitShowTimeTx.h"
#include "TxDataPumpSetUp.h"
#include "ToneReorder.h"
#include "eoc.h"
#include "vdsl_xception.h"
#include "dsp_op.h"
#include "mul.h"
#include "Framing_VDSL2.h"
#include "vdsl_state.h"
#include "GenFramingParams_VDSL2.h"
#include "cmv.h"

//****************************************************************************
// Framing_VDSL2.c
//   History :
//
//  22/07/2013 Varun : Incorporated the code changes to support US ReTx
//                     Grep for "XDSLRTFW-1078 Feature_US_VDSL2_ALL_UsReTx"
//***************************************************************************

/**************************************************************************************
*   Prototype: int16  ComputeCWSize(VDSL2Config_t *pt_Config, int16 s_lp)
*
*   Description: This function computes the codeword size NFECp in the following equation:
*            Kp = (sum Bpn) + ceil(Gp/Tp)
*
*            NFECp = Mp * Kp + Rp
*
*   Input arguments:
*         pt_Config   -   point to the showtime config structure
*         s_lp       -   path type: LP0 or LP1
*
*   Output arguments:
*         s_CodewordSize - codeword size
*
*   Return:
*
****************************************************************************************/

int16 ComputeCWSize(VDSL2Config_t *pt_Config, int16 s_lp)
{
   int16 s_bc, s_MuxFrameSize;
   int16 s_num, s_den, s_CodewordSize;

   //Compute the mux frame size: Kp = (sum Bpn) + ceil(Gp/Tp)
   s_num = pt_Config->s_Gp[s_lp];
   s_den = pt_Config->s_Tp[s_lp];

   //XDSLRTFW-1078 Feature_US_VDSL2_ALL_UsReTx(START)
   //For ReTx LP1 Gp1/Tp1=1 if GP1=0 and Tp1=0
   if((s_lp==LP1)&& (gt_ReTXParams.uc_UsReTxStatus == US_RETX_IN_USE) && (s_num==0) && (s_den==0))
   {
      s_MuxFrameSize = 1;

   } //XDSLRTFW-1078 Feature_US_VDSL2_ALL_UsReTx(END)
   else
   {
      s_MuxFrameSize = ceil16(s_num, s_den);
   }

   for (s_bc=0; s_bc < NUM_BEARER_CHANNELS; s_bc++)
   {
      s_MuxFrameSize += pt_Config->sa_Bpn[s_lp][s_bc];
   }

   //Compute the codeword size: NFECp = Mp * Kp + Rp
   s_CodewordSize = pt_Config->s_Mp[s_lp] * s_MuxFrameSize + pt_Config->s_Rp[s_lp];

   return(s_CodewordSize);

}

/**************************************************************************************
*   Prototype: void Compute_PERBp(VDSL2Config_t *pt_Config, int16 s_lp, int16 s_NumTones)
*
*
*   Description: This function computes the PERBp in the following equation:
*
*            PERBp = (Tp * Nfecp/Mp) * floor (Q * Mp/ (Tp * Nfecp)
*
*            where Q = QMAX if DRp >= DR0 Kbps
*                    = QMAX * DRp/DR0 if DRp < DR0
*                  with QMAX = 17000 and DR0 = 7880
*
*   Input arguments:
*         pt_Config   -   point to the showtime config structure
*         s_lp       -   path type: LP0 or LP1
*         s_NumTones
*
*   Output arguments:
*         s_CodewordSize - codeword size
*
*   Return:
*
****************************************************************************************/
void Compute_PERBp(VDSL2Config_t *pt_Config, int16 s_lp, int16 s_NumTones)
{
   int16 s_temp, s_temp1;
   int16 s_k = 0;
   int32 l_Acc,l_Acc2,l_Q;
   uint16 us_Qmax, us_DRmax;


   if (gul_dbgMiscControl & DISABLE_ACCURATE_PERBp_COMPUTATION_FIX)
   {
      //l_Acc = (int32)pt_Config->ul_Lp[s_lp] * s_DataFrameRate;  // note: assume "s_temp" is at most 8000 symbols/sec, we can still use a signed-32 bit number
      //l_Acc = Lp*fs should always be a +'ve number!
      MULS32xU16(l_Acc, pt_Config->ul_Lp[s_lp], (uint16)gs_DataFrameRate);

      // Convert the line rate to Kbps: DRp = Lp * Fs
      pt_Config->l_DRp[s_lp] = NormAndDivide_32by16bit(l_Acc, (int16)1000);
   }
   else
   {
      // Using gs_DataFrameRate causes a loss of precision, which is unacceptable for PERBp computation
      // Hence, we **ASSUME** a DMT frame rate of (4 or 8)Khz i.e. we assume gs_m of 5
      // DRp = Lp * Fs = (Lp * (4 or 8) * 256) / 257 Kbps
      MULS32xU16(l_Acc, pt_Config->ul_Lp[s_lp], 1024);
      if (gs_frame_rate_is_8khz == 1)
      {
         l_Acc <<= 1;
      }
      pt_Config->l_DRp[s_lp] = l_Acc / 257;
   }

   us_Qmax = QMAX;
   us_DRmax = DR0;
   if ((gt_ProfileAct.us_ProfileSelected & CNFG_V2_PROFILE_35B_MASK) &&
         (&gt_rx_config_v2 == pt_Config))
   {
      us_Qmax = QMAX_35B;
      us_DRmax = DR0_35B;
   }

   // Compute the Qhat value
   if (pt_Config->l_DRp[s_lp] >= (int32)us_DRmax)
   {
      l_Q = (int32)us_Qmax;
   }
   else
   {
      // Qhat = Q * DRp / DR0
      if (gul_dbgMiscControl & DISABLE_ACCURATE_PERBp_COMPUTATION_FIX)
      {
         // OLD Inaccurate Computation- sacrifices accuracy for using the generic gs_DataFrameRate, which accounts for any CE size or gs_m
         l_Acc = (int32)QMAX * pt_Config->l_DRp[s_lp];
         l_Q = NormAndDivide_32by16bit(l_Acc, (int16)DR0);
      }
      else
      {
         // Note: This is also valid for AnxQ (35b)
         // Using gs_DataFrameRate causes a loss of precision, which is unacceptable for PERBp computation
         // Hence, we ASSUME a DMT frame rate of (4 or 8)Khz i.e. we assume gs_m of 5
         // Qhat   = 17000 * DRp / 7880
         //      = 17000 * Lp * Fs / 7880
         //      = 17000 * Lp * (4 or 8) * 256 / (7880 * 257)
         //      = 1700 * Lp * (1 or 2) * 256 / (197 * 257)
         //      = 1700 * Lp * (1 or 2) * 256 / (197 * 257)
         MULS32xU16(l_Acc, pt_Config->ul_Lp[s_lp], (QMAX/10));
         l_Acc2 = l_Acc << 8;      // Will NOT overflow because Lp is bounded by condition Lp < (7880/Fs)
         if (gs_frame_rate_is_8khz == 1)
         {
            l_Acc2 <<= 1;
         }
         l_Q = l_Acc2 / (((DR0/10)/4)*257);
      }
   }

   // s_temp = k ( where k <= 64) as Tp = k * Mp always
   s_k = 0;
   s_temp = pt_Config->s_Tp[s_lp];



   while ((s_temp >= pt_Config->s_Mp[s_lp]) && (s_temp != 0) && (pt_Config->s_Mp[s_lp] > 0))
   {
      s_temp -= pt_Config->s_Mp[s_lp];
      s_k++;
   }


   // s_temp = k * Nfecp
   s_temp1 = s_k * pt_Config->s_Nfecp[s_lp];

   // floor (Q * Mp/ (Tp * Nfecp) = floor (Q / (k * Nfecp)
   if (s_temp1)
   {
      s_temp1 = floor32(l_Q, s_temp1);
   }

   s_temp = s_k * pt_Config->s_Nfecp[s_lp];

   // Compute the PERBp parameter
   pt_Config->l_PERBp[s_lp] = (int32)(s_temp * s_temp1);
}


/**************************************************************************************
*   Prototype: void Compute_VDSL2Framing(VDSL2Config_t *pt_Config, BC_LP_TPS_Map_t   *pt_TPS_Map, int16 *s_PMDFramesPerIBStructure)
*
*   Description: This function computes the derived framing parameters for VDSL2 Tx and Rx
*
*   Input arguments:
*         pt_Config
*
*   Output arguments:
*         s_CodewordSize - codeword size
*
*   Return:
*
****************************************************************************************/
void Compute_VDSL2Framing(VDSL2Config_t *pt_Config, BC_LP_TPS_Map_t   *pt_TPS_Map, int16 *s_PMDFramesPerIBStructure)
{
   int16 s_temp, s_lp;
   int16 s_shift;
   int32 l_Acc;
   //Feature_US_VDSL2_ALL_UsReTx(START)
   int16 s_Temp_PMDFramesPerIBStructure[NUM_DATA_PATHS];
   //Feature_US_VDSL2_ALL_UsReTx(END)
   int16 s_lpInit = LP0;               // In showtime, ReTx updates only LP1.

   if(gs_RxState == R_O_SHOWTIME_RX) // Training
   {
      s_lpInit = LP1;               // In showtime, ReTx updates only LP1.
   }

   for (s_lp=s_lpInit; s_lp<NUM_DATA_PATHS; s_lp++)
   {
      // Compute the codeword size for each latency path
      pt_Config->s_Nfecp[s_lp] =  ComputeCWSize(pt_Config, s_lp);

      //======================================================
      // Compute q = N/I
      //======================================================
      pt_Config->s_q[s_lp] = 0;

      s_temp = pt_Config->s_Nfecp[s_lp];

      // Skip all computations over the unused latency path
      if (s_temp == 0)
      {
         continue;
      }



      while ((s_temp >= pt_Config->s_Ip[s_lp]) && (pt_Config->s_Ip[s_lp] > 0))
      {
         s_temp -= pt_Config->s_Ip[s_lp];
         pt_Config->s_q[s_lp]++;
      }

      //Make sure that Nfecp contains an integer multiple of Ip
      // and Ip > 0
      if ((s_temp != 0) || (pt_Config->s_Ip[s_lp] == 0))
      {
#ifndef STANDALONE_FRAMING_TEST
         EnterFailStates(E_CODE_WRONG_INLV_BLOCK_SIZE);
#else
         return;
#endif //STANDALONE_FRAMING_TEST
      }

      //======================================================
      // Compute the PERBp parameter in bytes
      //======================================================
      Compute_PERBp(pt_Config, s_lp, gs_TxNumTones);

      //======================================================
      // Compute PERp = 8*PERBp/(Lp*fs) = 8*PERBp/DRp in ms
      //======================================================
      //  NOTE1: Make sure we call Compute_PERBp() before computing PERp.
      //  NOTE2: pt_Config->l_DRp[s_lp] is computed in Compute_PERBp()
      //         Also, DRp is computed based on the output of ComputeSymbolRate(), which might not be exactly
      //         equal to the value computed based on fs = 4 or 8 ksymbols/s.
      //  NOTE3: In VDSL2 standard, PERp can be < 1 ms.  It's because the highest possible PERBp value
      //         is QMAX when DRp is > DR0 (7880 kbps).  In such case, PERp would be reported as 0 ms
      //         because we round down PERp in this computation.
      l_Acc =   (pt_Config->l_PERBp[s_lp]<<3)/pt_Config->l_DRp[s_lp];

      if ((l_Acc > 0x00007FFF) || (l_Acc < 0))   // this is to catch any error in computing l_PERBp or l_DRp.
      {
         pt_Config->s_PERp[s_lp] = 0x7FFF;
      }
      else
      {
         pt_Config->s_PERp[s_lp] = (int16)l_Acc;
      }

      //======================================================
      // Compute the rate at which we should be processing the IBits
      // in terms of the number of PMD frames:
      //    s_PMDFramesPerIBStructure = Period of IBits-OH / Period of a Symbol
      //                              = PERp/SYMp
      //         where SYMp = 1/4 msec for frame rate of 4KHz
      //                    = 1/8 msec for frame rate of 8KHz
      //               PERp = (8*PERBp/DRp)
      //======================================================
      // For frame rate of 4 KHz, the constant for multiplying "PERBp/DRp" is 32 (2^5);
      // while the threshold based on MaxPERp of 20 ms is 80 Frames/IBStructs
      s_shift = 5;
      s_temp = VDSL2_MAX_NUM_PMD_FRAMES_PER_IB_STRUCT;

      if (gs_frame_rate_is_8khz == 1)
      {
         // For frame rate of 8 KHz, both the constant and threshold have to be doubled.
         s_shift = s_shift + 1;  // constant is now 64 (2^6)
         s_temp  = s_temp << 1;  // threshold is now 160 Frames/IBStructs
      }


      l_Acc =   (pt_Config->l_PERBp[s_lp]<<s_shift)/pt_Config->l_DRp[s_lp];

      //Feature_US_VDSL2_ALL_UsReTx(START)

      if ((l_Acc > s_temp) || (l_Acc <= 0))   // this is to catch any error in computing l_PERBp or l_DRp.
      {
         s_Temp_PMDFramesPerIBStructure[s_lp]=s_temp;
      }
      else
      {
         s_Temp_PMDFramesPerIBStructure[s_lp]=(int16)l_Acc;
      }

      //copy the calulated value to actual variable..
      //if ReTx Enabled then copy only LP0 value as IB bits are only present in LP0
      //XDSLRTFW-1386
      //Update this Frames Per IB Structure only with LP0 value for both
      //ReTx and Normal Mode
      if(s_lp == LP0)
      {
         *s_PMDFramesPerIBStructure = s_Temp_PMDFramesPerIBStructure[LP0];
      }
      //XDSLRTFW-1386

      //Feature_US_VDSL2_ALL_UsReTx(END)

      //======================================================
      // Compute the Up parameter: Up = (PERBp * Mp ) / (Nfecp * Tp)
      //======================================================
      MULS16(l_Acc, pt_Config->s_Tp[s_lp], pt_Config->s_Nfecp[s_lp]);
      s_temp = (int16)l_Acc;
      MULS32x16(l_Acc, pt_Config->l_PERBp[s_lp], pt_Config->s_Mp[s_lp]);

      pt_Config->s_Up[s_lp] = (int16)NormAndDivide_32by16bit(l_Acc, s_temp);

      //======================================================
      // Compute SEQp value for each latency path
      // SEQp = Up * Gp
      //======================================================
      MULS16(l_Acc, pt_Config->s_Up[s_lp], pt_Config->s_Gp[s_lp]);
      pt_Config->s_SEQp[s_lp] = (int16)l_Acc;

      //======================================================
      // Compute the value of MSGc = SEQp - 6
      //======================================================
      if (s_lp == pt_TPS_Map->s_IBITSlp)
      {
         pt_Config->s_MSGc = pt_Config->s_SEQp[s_lp] - 6;
      }

      //======================================================
      // Compute the switchpoint between OH1 to OH2
      // Switchpoint = Gp - Tp * floor(Gp/Tp)
      //======================================================
      if (pt_Config->s_Tp[s_lp])
      {
         s_temp = floor16(pt_Config->s_Gp[s_lp], pt_Config->s_Tp[s_lp]);
      }

      MULS16(l_Acc, s_temp, pt_Config->s_Tp[s_lp]);

      pt_Config->s_OHSWITCHp[s_lp] = pt_Config->s_Gp[s_lp] - (int16)l_Acc;
      pt_Config->s_OHRate2p[s_lp] = s_temp;

      if (pt_Config->s_Tp[s_lp])
      {
         s_temp = ceil16(pt_Config->s_Gp[s_lp], pt_Config->s_Tp[s_lp]);
      }

      pt_Config->s_OHRate1p[s_lp] = s_temp;

      //======================================================
      // Compute INPp value for each latency path
      // INP = (8 * Dp * floor(Rp/(2*q))/Lp
      //======================================================

      //s_temp = 2*q
      s_temp = pt_Config->s_q[s_lp]<<1;

      //s_temp = floor(Rp/(2*q))
      s_temp = floor16(pt_Config->s_Rp[s_lp], s_temp);

      //s_temp = Dp * floor(Rp/(2*q)
      MULS16(l_Acc, pt_Config->s_Dp[s_lp], s_temp)

      //INP = (8 * Dp * floor(Rp/(2*q))/Lp in Q8.8 format
      pt_Config->s_INPp[s_lp] = (int16)((l_Acc << 11)/pt_Config->ul_Lp[s_lp]);

   }
   //XDSLRTFW-1055 Feature_US_VDSL2_ALL_UsReTx_TxOvhdTestMgmtDiag (Start)
   if(gt_ReTXParams.uc_UsReTxStatus == US_RETX_IN_USE)
   {
      gt_ReTXParams.uc_US_H=(gt_tx_config_v2.s_Nfecp[LP1] - gt_tx_config_v2.s_Rp[LP1]);
   }
   //XDSLRTFW-1055 Feature_US_VDSL2_ALL_UsReTx_TxOvhdTestMgmtDiag (End)
}

/*
*-------------------------------------------------------------------------------
*
*   Prototype: int16 GetFifoReadWriteStep(int16 s_D, int16 s_I)
*
*   This function returns "read-step" or "write-step" - # of FIFO's to be skipped
*   between reads (ILVB to TxDTB) or writes (RxDTB to DILVB), which is uniquely
*   determined by interleave depth (D) and the interlever bolck size (I).
*
*   Input Arguments:
*      s_D: interleave depth
*      s_I: Interleaver block size
*
*   Output Arguments:
*
*   Returns:
*      i s.t. i*D mod N == 1
*
*   Global Variables:
*
*-------------------------------------------------------------------------------
*/

int16 GetFifoReadWriteStep(int16 s_D, int16 s_I)
{
   int16 i;
   int32 l_temp;

   for (i=0; i<s_I; i++)
   {
      // l_temp = i*D mod I
      MULS16(l_temp, i, s_D);

      while (l_temp > s_I)
      {
         l_temp -= s_I;
      }

      if (l_temp == 1)
      {
         break;
      }
   }

   return (i);
}

/*
*-------------------------------------------------------------------------------
*
*   Prototype: void ComputeRatesFS6(int16 s_Path, VDSL2Config_t *pt_Config, int32 *pl_NetRate)
*
*   This function computes the net data rate in unit of 4kbps.
*
*   Input Arguments:
*      pt_Config -- pointer to the structure of the framing parameters
*      s_Path      -- latency path
*
*   Output Arguments:
*      pl_NetDataRate -- pointer of the returned net data rate.
*
*   Returns:
*
*   Global Variables:
*
*-------------------------------------------------------------------------------
*/
void ComputeRatesFS6(int16 s_Path, VDSL2Config_t *pt_Config, int32 *pl_NetRate)
{
   int16 s_temp1, s_temp2;
   int32 l_LineRate, l_OHRate, l_TotalRate, l_a0, l_a1;

   //------------------------------------------------------------------------
   // The computation of TotalRate, OHRate & NetRate is based on the one
   // performed in FormVDSL2FramingParams().
   //
   // Note: It's confirmed from Matlab that the accuracy is within +/- 1 kbps.
   //------------------------------------------------------------------------
   // wng - FIXME - 1) check if we can use the DRp value from gt_rx_config_v2 instead of computing l_LineRate here.
   //               2) change to use MULS/MULU instruction to optimize code & mips count.
   //
   // l_LineRate = ul_Lp*s_fs;
   l_LineRate =  pt_Config->ul_Lp[s_Path] << (2+gs_frame_rate_is_8khz);


   // s_temp1 = k = Tp/Mp
   s_temp1 = pt_Config->s_Tp[s_Path]/pt_Config->s_Mp[s_Path];
   // s_temp2 = k*CWSize
   s_temp2 = s_temp1*pt_Config->s_Nfecp[s_Path];

   // convert OHRate to Q28.4
   l_a0 = (l_LineRate*pt_Config->s_Gp[s_Path])<<4;
   // OHRate = Lp*fs*Gp/(k*CWSize)  (in Q28.4)
   l_OHRate = l_a0/s_temp2;

   // convert TotalRate to signed Q28.4 format
   l_a1 = (l_LineRate*(pt_Config->s_Nfecp[s_Path]-pt_Config->s_Rp[s_Path]))<<4;
   // TotalRate = Lp*fs*(CWSize-Rp)/CWSize (in Q28.4)
   l_TotalRate = l_a1/pt_Config->s_Nfecp[s_Path];

   // NetRate = TotalRate - OHRate (in Q28.4)
   *pl_NetRate = l_TotalRate - l_OHRate;
   //-------------------------------------------------------------------------
}

/*
*-------------------------------------------------------------------------------
*
*   Prototype: void ComputeDelay(VDSL2Config_t *pt_Config, int16 s_lp, uint16 *pus_delay)
*
*   This function computes the delay in milliseconds according to the following equation:
*
*      Delay = Sp*(Dp-1)*(1-q/Nfecp)/(q*fs)
*             = 8*Nfecp*(Dp-1)*(1-q/Nfecp)/(q*Lp*fs)
*             = 8*(Dp-1)*(Nfecp-q)/(q*DRp)
*
*   Input Arguments:
*      pt_Config -- pointer to the structure of the framing parameters
*      s_lp      -- latency path
*
*   Output Arguments:
*      pus_delay -- pointer of the computed delay value.
*
*   Returns:
*
*   Global Variables:
*
*-------------------------------------------------------------------------------
*/
void ComputeDelay(VDSL2Config_t *pt_Config, int16 s_lp, uint16 *pus_delay)
{
   int32 l_temp, l_LineRate;
   uint16 us_delay;


   //=====================================
   //Compute 8*(Nfecp - q)*(Dp - 1)
   //=====================================

   //l_temp = (Nfecp -q)*(Dp - 1)
   MULU16(l_temp, (pt_Config->s_Nfecp[s_lp] - pt_Config->s_q[s_lp]), (pt_Config->s_Dp[s_lp] - 1));

   //l_temp *= 8, extra 2 shifts is to increase the precision and will be cancelled later
   l_temp <<= (3+2);

   //=====================================
   //Compute delay = l_temp/(q*DRp)
   //=====================================

   //Compute q*DRp
   MULS32x16(l_LineRate, pt_Config->l_DRp[s_lp], pt_Config->s_q[s_lp]);

   us_delay = 0;

   //To avoid stuck problem, execute this while loop only when the linerate is positive.
   if(l_LineRate > 0)
   {
      while(l_temp >= l_LineRate)
      {
         l_temp -= l_LineRate;
         us_delay++;
      }
   }

   //Perform rounding
   us_delay += (1<<1);

   //shift out two bits
   us_delay >>= 2;

   *pus_delay = us_delay;
}

/**************************************************************************************
*   Prototype: int16 ComputeSymbolRate(int16 s_NumTones, int16 s_CELength)
*
*   Description: This function computes the fs in the following equation:
*            fs = Nsc*8625/(2*Nsc+CELength)
*
*   Input arguments:
*         s_NumTones
*         s_CELength
*
*   Output arguments:
*         frame rate - in symbols/second
*
*   Return:
*
****************************************************************************************/
int16 ComputeSymbolRate(int16 s_NumTones, int16 s_CELength)
{
   int16 s_n, s_temp;
   int32 l_Acc;

   // Compute the frame rate : fs = Nsc*8625/(2*Nsc+CELength)
   l_Acc = s_NumTones*8625;
   s_temp = (2*s_NumTones + s_CELength);
   s_n = NormAndDivide_32by16bit(l_Acc, s_temp);

   if (gs_frame_rate_is_8khz == 1)
   {
      s_n = s_n << 1;
   }

   //scale frame rate by 256/257 to count for the synch symbol insertion
   if(gft_AccurateFrameRateEnable == TRUE)
   {
      uint32 ul_Acc;

      //0x7F80 = (2^15)*256/257
      ul_Acc = 0x7F80 * s_n;
      s_n = (int16)(ul_Acc>>15);

   }

   return(s_n);

}
