/* **COPYRIGHT******************************************************************
    INTEL CONFIDENTIAL
    Copyright (C) 2017 Intel Corporation
    Copyright (C), 1994-2006 Aware Inc. All Rights Reserved.
******************************************************************COPYRIGHT** */
/* **DISCLAIMER*****************************************************************
    The source code contained or described herein and all documents related
    to the source code ("Material") are owned by Intel Corporation or its
    suppliers or licensors. Title to the Material remains with Intel
    Corporation or its suppliers and licensors. The Material may contain
    trade secrets and proprietary and confidential information of Intel
    Corporation and its suppliers and licensors, and is protected by
    worldwide copyright and trade secret laws and treaty provisions. No part
    of the Material may be used, copied, reproduced, modified, published,
    uploaded, posted, transmitted, distributed, or disclosed in any way
    without Intel's prior express written permission.

    No license under any patent, copyright, trade secret or other
    intellectual property right is granted to or conferred upon you by
    disclosure or delivery of the Materials, either expressly, by
    implication, inducement, estoppel or otherwise. Any license under
    such intellectual property rights must be express and approved by
    Intel in writing.
*****************************************************************DISCLAIMER** */
/*
*-------------------------------------------------------------------------------
*
*   Aware DMT Technology. Proprietary and Confidential.
*
*   40 Middlesex Turnpike, Bedford, MA 01730-1413
*   Phone (781) 276 - 4000
*   Fax   (781) 276 - 4001
*
*   filename: InitShowTimeTx.c
*
*   This file contains functions to initialize showtime variables.
*
*-------------------------------------------------------------------------------
*/

// ***********************************************************************************************************
// InitShowTimeTx.c
//
// History
//
// 05/12/2012 Ram: Aligning to Msg Spec 1.8:- INFO 68, has no variable 'Line Rate'
//                 Grep for 'XDSLRTFW-466: BugFix_DS_VDSL2_ALL_Align_As_Per_CMV_INFO_68'
//
// 10/10/2017 Abu Rahman
// XDSLRTFW-3071: US & DS ATP reporting is swapped
//     Variables contains the DS ActATP and US ActATP were wrongly (swapped) populated.
//     - Fixed the variable initialization.
//     - Fixed variables population for DS ActATP and US ActATP
//     - Fixed HDLC messages relating to these variables both for Showtime and diag mode
//     - Changed two structure names
//       1. old name: gt_G997_NeSignalAndMarginParams, new name: gt_LineStatusDS
//       2. old name: gt_G997_FeSignalAndMarginParams, new name: gt_LineStatusUS
//     Grep for XDSLRTFW-3071
// ************************************************************************************************************

#include <string.h>
#include "common.h"
#include "gdata.h"
#include "cmv_Data.h"
#include "fifo.h"
#include "InitShowTimeTx.h"
#include "TxDataPumpSetUp.h"
#include "ToneReorder.h"
#include "eoc.h"
#include "vdsl_xception.h"
#include "dsp_op.h"
#include "mul.h"
#include "cmv.h"
#include "states.h"
#include "Snr_b.h"
#include "ConvertToDB.h"
#include "data_alloc.h"
#include "profile.h"

#include "LL_IOf.h"
extern RRC_Stat_t gt_RrcStat;
extern ReTX_Params_t gt_ReTXParams;

void ClearTxZephyrRegs(void);

#ifdef INCLUDE_NON_CACHE_FUNCTIONS
/*
*-------------------------------------------------------------------------------
*
*   Prototype: void TxShowInitHandler(void)
*
*   This function initializes variables and cores for showtime TX processing.
*
*   Input Arguments:
*
*   Output Arguments:
*
*   Returns:
*
*   Global Variables:
*
*-------------------------------------------------------------------------------
*/

void TxShowInitHandler(void)
{
   switch (guc_TxShowInitState)
   {

   case TX_SHOW_INIT_VARS:
      // initialize showtime variables & BG tasks
      guc_BgTxShowInitState = TRAINING_IN_PROGRESS;
      AddFunctionToBkgdFifo((PtrToBkgdFunc)BgInitShowTimeTx);
      guc_TxShowInitState = TX_SHOW_INIT_VARS_WAIT;
      break;

   case TX_SHOW_INIT_VARS_WAIT:
      // wait until background task is done
      if (guc_BgTxShowInitState == TRAINING_DONE)
      {
         if (gul_ilvb_size == 0)
         {
            EnterFailStates(E_CODE_ILV_INIT_FAIL);
         }

         if (gft_ToneOrderOK == FAIL)
         {
            EnterFailStates(E_CODE_TONE_ORDERING_ERROR);
         }

         guc_TxDataPumpState = TX_CONFIG_CORE_START;
         guc_TxShowInitState = TX_SHOW_INIT_CORES;

         // reset all Zephyr-FCI TX PRAM LP0/1 prior to running TxDataPumpSetup.
         ClearTxZephyrRegs();

      }
      break;

   case TX_SHOW_INIT_CORES:
      // initialize cores if variable initialization is done
      if (guc_TxDataPumpState != TX_CONFIG_CORE_DONE)
      {
         AddFunctionToFifo(gp_TxLoadingFunctionFifo, TxDataPumpSetUp);
      }
      else
      {
         guc_TxShowInitState = TX_SHOW_INIT_DONE;
      }
      break;
   }
}
#endif // ifndef INCLUDE_NON_CACHE_FUNCTIONS

#ifdef INCLUDE_CACHE_FUNCTIONS
/*
*-------------------------------------------------------------------------------
*
*   Prototype: void InitTxVarsCommon( BC_LP_TPS_Map_t *pt_Config)
*
*   This function initializes the BC to TPS mapping.
*
*
*   Output Arguments:
*
*   Returns:
*
*   Global Variables:
*
*-------------------------------------------------------------------------------
*/

void InitTxVarsCommon( BC_LP_TPS_Map_t *pt_Config)
{
   int16 s_bc, s_lp, s_TpsMode;

   // Initialize per BC variables
   for (s_bc=0; s_bc<NUM_BEARER_CHANNELS; s_bc++)
   {
      s_TpsMode = gta_UsBearerChanAct[s_bc].s_TypeEnabled;
      if (s_TpsMode == TPS_TC_TYPE_PTM)
      {
         // Since there is currently no PTM, map PTM aot ATM
         // but set PhysicalTransportMode to PTM
         if (s_bc == BC0)
         {
            pt_Config->s_BCtoTPS[BC0] = MAP2ABC0;
            gus_PhysicalTransportMode |= TX_BC0_PTM_MASK;
         }
         else
         {
            pt_Config->s_BCtoTPS[BC1] = MAP2ABC1;
            gus_PhysicalTransportMode |= TX_BC1_PTM_MASK;
         }
         gft_TcTypeSelected = TPS_TC_TYPE_PTM;
      }
      else if (s_TpsMode == TPS_TC_TYPE_ATM)
      {
         // if BC is ATM, always connect BC0 to ATM0, BC1 to ATM1
         if (s_bc == BC0)
         {
            pt_Config->s_BCtoTPS[BC0] = MAP2ABC0;
            gus_PhysicalTransportMode |= TX_BC0_ATM0_MASK;
         }
         else
         {
            pt_Config->s_BCtoTPS[BC1] = MAP2ABC1;
            gus_PhysicalTransportMode |= TX_BC1_ATM1_MASK;
         }
         gft_TcTypeSelected = TPS_TC_TYPE_ATM;
      }
      else if (s_TpsMode == TPS_TC_TYPE_STM)
      {
         // there exists only 1 STM channel
         if (s_bc == BC0)
         {
            pt_Config->s_BCtoTPS[BC0] = MAP2EBC;
            gus_PhysicalTransportMode |= TX_BC0_STM_MASK;
         }
         else
         {
            pt_Config->s_BCtoTPS[BC1] = MAP2EBC;
            gus_PhysicalTransportMode |= TX_BC1_STM_MASK;
         }

         gft_TcTypeSelected = TPS_TC_TYPE_STM;
      }
      else
      {
         pt_Config->s_BCtoTPS[s_bc] = MAP2NONE;
      }
   }

   // initialize per LP variables
   pt_Config->s_LPtoABC[LP0] = DISABLED_ABC;
   pt_Config->s_LPtoABC[LP1] = DISABLED_ABC;

   for (s_bc=0; s_bc<NUM_BEARER_CHANNELS; s_bc++)
   {
      s_lp = pt_Config->s_BCtoLP[s_bc];

      if (s_lp != DISABLED_LP)
      {
         if (pt_Config->s_BCtoTPS[s_bc] == MAP2ABC0)
         {
            pt_Config->s_LPtoABC[s_lp] = ABC0;
         }
         else if (pt_Config->s_BCtoTPS[s_bc] == MAP2ABC1)
         {
            pt_Config->s_LPtoABC[s_lp] = ABC1;
         }
      }
   }

}


/*
*-------------------------------------------------------------------------------
*
*   Prototype: int16 CalcTxActAtp(int16 s_MaxNomPsdIn, int16 *hpsa_TxFineGainsIn)
*
*      This function calculates the Actual Aggregate Transmit Power.  It is called
*   following Bit Loading to account for the Fine Gain allocation.
*
*      The calculation relies on an earlier calculation of the Aggregate Transmit Power
*   that accounts for the PSD shaping and supported set.  The calculation in
*   this functions calculates the updated IFFT input power, compares it to the
*   previous IFFT input power, and then adjusts the previously calculated transmit
*   power accordingly.
*
*   Input Arguments:
*
*      int16 s_MaxNomPsdIn - no longer used
*      *hpsa_TxFineGainsIn - pointer to tx fine gains
*
*   Output Arguments:
*
*   Returns:
*
*   Global Variables:
*
*      gs_tx_power_dBm_line   - previously calculated transmit power
*      gs_avg_ifft_in_dB      - previously calculated IFFT input power
*
*-------------------------------------------------------------------------------
*/
#define TEN_LOG_DELTA_F     (9305)          /* 36.35 = 10*log10(4312.5) in Q8.8 */
#define C_10LOG10_2         (771)           /* 10log10(2) in Q8.8 */

int16 CalcTxActAtp(int16 s_MaxNomPsdIn, int16 *hpsa_TxFineGainsIn)
{
   /* Local Variables */
   int16 i, j, s_actAtp;
   int32 l_acc;

   uint32 ul_acc;
   int16 s_avg_ifft_in_dB_post_bitload;
   int16 s_tx_power_dBm_line_post_bitload;


   //Accumulate an average ms gain for the IFFT fine gains.
   ul_acc = 0;
   for(i=0; i<gs_NumOfTxBands; i++)
   {
      for (j = gsa_TxBandLeftChannel[i]; j <= gsa_TxBandRightChannel[i]; j++)
      {

         // ghpsa_TxFineGains_Inactive[] is 3.13, with max value 0x2000 (=2^13).
         // maximum number of tones in average is 2^(gs_TxLog2IfftLength-1),
         // so max acc is 2^(13+13+gs_TxLog2IfftLength-1 - scl), where products are shifted
         // down scl bits to avoid overflowing acc.  For max acc = 2^31, set scl = (gs_TxLog2IfftLength-1)-5.

         ul_acc += (((int32)ghpsa_TxFineGains_Inactive[j]) * ((int32)ghpsa_TxFineGains_Inactive[j]) >> (gs_TxLog2IfftLength-1-5));
      }
   }
   //Scale so the average is in 6.26 format.
   ul_acc >>= 5;

   //Calculate the average signal level at the ifft input.
   //This will be used as a basis for ifft scaleback settings.
   //ConvertToDB treats input as 0.32, whereas ul_acc is in 6.26,
   //so adjust output down by 2*10*log10(2^13), in 8.8 format (which is output format).

   s_avg_ifft_in_dB_post_bitload = ConvertToDB(ul_acc) - (int16) (CONST_GAIN_8192_DB<<1);            //Avg fine gains
   //s_avg_ifft_in_dB_post_bitload += ConvertToDB(TX_NEG165DBM*TX_NEG165DBM) - CONST_GAIN_2pow30_DB;
   s_avg_ifft_in_dB_post_bitload += TX_NEG165DBM_DB - CONST_GAIN_2pow30_DB;   //QAM costellation level

   s_tx_power_dBm_line_post_bitload = gs_tx_power_dBm_line - (gs_avg_ifft_in_dB - s_avg_ifft_in_dB_post_bitload);

   // Convert ACTATP to 0.1 dB format and bound it to
   // standard specified allowable dynamic range
   l_acc = (int32)s_tx_power_dBm_line_post_bitload * 10;
   s_actAtp = round(l_acc, 8);
   if (s_actAtp < -310 || s_actAtp > 310)
   {
      s_actAtp = OUT_OF_RANGE_ACTATP;
   }

   return(s_actAtp);
}

/*
*-------------------------------------------------------------------------------
*
*   Prototype: int16 CalcTxActPSD(void)
*
*      This function calculates the Actual Aggregate Transmit PSD
*
*   Input Arguments:
*
*  gt_LineStatusUS structure contains US information.
*  gt_LineStatusUS.s_ActualAggregateXmtPwr contains US ACT ATP
*   Output Arguments:
*
*   Returns:
*
*   Global Variables:
*
*
*-------------------------------------------------------------------------------
*/
int16 CalcTxActPSD(void)
{
   int16 s_ActualAggregateXmtPwr;
   int16 s_bandwidth, s_logdelta_f;
   int16 s_ActualAggregatePSD;
   // XDSLRTFW-3071 US & DS ATP reporting is swapped (Start)
   // gt_LineStatusUS.s_ActualAggregateXmtPwr contains US ACT ATP
   // Skip calculation if transmit power is out of range value
   if ( gt_LineStatusUS.s_ActualAggregateXmtPwr == OUT_OF_RANGE_ACTATP)
   {
      return(OUT_OF_RANGE_ACTPSD);
   }

   // Convert from 0.1 dB step to Q8.8 format values
   s_ActualAggregateXmtPwr = (gt_LineStatusUS.s_ActualAggregateXmtPwr << 8)/10;
    // XDSLRTFW-3071 US & DS ATP reporting is swapped (End)

   // Calculate 10*log(delta_f) in Q8.8 format
   if (gs_frame_rate_is_8khz)
   {
      s_logdelta_f = 10076;
   }
   else
   {
      s_logdelta_f = 9305;
   }

   // Compute bandwidth
   s_bandwidth = ConvertToDB((int32)gs_TxNumTonesUsed) + s_logdelta_f;

   // Compute aggregate actual PSD from Q8.8 to 0.1 dB steps
   s_ActualAggregatePSD = ((s_ActualAggregateXmtPwr - s_bandwidth)*10) >> 8;

   // Check for valid range of ACTPSD
   if ((s_ActualAggregatePSD > 0) || (s_ActualAggregatePSD < -900))
   {
      s_ActualAggregatePSD = OUT_OF_RANGE_ACTPSD;
   }

   return(s_ActualAggregatePSD);
}
/*
*-------------------------------------------------------------------------------
*
*   Prototype: void BgInitShowTimeTx(void)
*
*   This function initializes background function to call InitShowTimeTx().
*
*   Input Arguments:
*
*   Output Arguments:
*
*   Returns:
*
*   Global Variables:
*
*-------------------------------------------------------------------------------
*/
void BgInitShowTimeTx(void)
{

#ifndef HW_TRT
   TwelveBitsPair_t temp24;
   TwelveBitsPair_t nexttemp24;
   int16 j;
   int16 s_first, s_second;
   int16 s_start, s_end, s_band;
   int16 *psa_TRT;
   uint8 uca_temp[6];
   uint16 us_word;
#endif


   InitShowTimeTxVars_VDSL2();


   // XDSLRTFW-3071 US & DS ATP reporting is swapped (Start)
   // gt_LineStatusUS structure contains US information.
   // gt_LineStatusUS.s_ActualAggregateXmtPwr contains US(Tx) ACT ATP
   gt_LineStatusUS.s_ActualAggregateXmtPwr = CalcTxActAtp((int16)-gt_TxPsdControl.s_MaxNomPsdOut, ghpsa_TxFineGains_Inactive);

   // Tx Actual aggregate PSD value is never used in VDSL2 code.
   gt_LineStatusUS.s_ActualAggregatePSD = OUT_OF_RANGE_ACTPSD;
   // XDSLRTFW-3071 US & DS ATP reporting is swapped (End)

   // Do the Tx Tone ordering only for showtime only test
   if (TESTArray[TEST_InitState] == TEST_ShowtimeInitState)
   {
      // initialize TRT
      gft_ToneOrderOK = ToneOrdering(TX, ghpuca_TxBat_Inactive, ghpuca_TxExtGains_Inactive, gs_TxNumTones, gs_TxMaxConstSize, ghpsa_TxToneOrder_Inactive);
   }

   // configure TX Interleave FIFO Management buffers
   if (gus_ModemOperationMode_Status & MODEM_OPERATION_MODE_VDSL2)
   {
#ifndef HW_TRT
      // Write TRT
      psa_TRT = &(ghpsa_TxToneOrder_Inactive[gs_TxNumTones - gs_TxNumTonesUsed]);

      // Find the correct offset to write to the FDQ array
      s_band = gs_RxTRT_write_band;
      s_start = gs_RxTRT_write_offset;

      // Save the read band ofset for Rx TRT for use in showtime
      gs_TxTRT_read_band = gs_RxTRT_write_band;
      gs_TxTRT_read_offset = gs_RxTRT_write_offset;

      s_end = (gsa_RxBandLeftChannel[s_band] << 1) - 1;
      if (s_end == -1)
      {
         s_end = s_end + 8192;
      }

      // NOTE: Number of tones is always even

      // Pack 2 tones of information ie. 3 bytes into a word
      for (j = 0; j<gs_TxNumTonesUsed; j = j+2)
      {
         // Encode two 12 bit values into 3 bytes message
         s_first = psa_TRT[j];
         s_second = psa_TRT[j+1];
         InitTwelveBitsPair(s_first, s_second, &temp24);

         j = j+2;

         // Clear the last values
         nexttemp24.uc_Byte0 =0;
         nexttemp24.uc_Byte1 =0;
         nexttemp24.uc_Byte2 =0;

         // Read next tone set
         if ( j < gs_TxNumTonesUsed)
         {
            s_first = psa_TRT[j];
            s_second = psa_TRT[j+1];
            InitTwelveBitsPair(s_first, s_second, &nexttemp24);

         }

         uca_temp[0] = temp24.uc_Byte0;
         uca_temp[1] = temp24.uc_Byte1;
         uca_temp[2] = temp24.uc_Byte2;

         uca_temp[3] = nexttemp24.uc_Byte0;
         uca_temp[4] = nexttemp24.uc_Byte1;
         uca_temp[5] = nexttemp24.uc_Byte2;

         for (i= 0; i < 6; i++)
         {
            // save the packed values in the FDQ memory
            us_word = (uca_temp[i++] << 8);
            us_word |= uca_temp[i];
            WriteTRT(us_word, &s_start, &s_end, &s_band);
         }

      }
#endif
      //Reorder the TX TRT for the HW processing
      gft_ToneOrderOK = ReorderTRT_Tx();

      //XDSLRTFW-466: BugFix_DS_VDSL2_ALL_Align_As_Per_CMV_INFO_68 (Start)
      //Commenting out as INFO 68 in Msg Spec 1.8 does not have variable 'Line Rate'
      //gt_LineStatusUS.ul_LineRate = gul_TxLineRate;   // Setting the Rx Line Rate to the cmv
      //XDSLRTFW-466: BugFix_DS_VDSL2_ALL_Align_As_Per_CMV_INFO_68 (End)

      if(gft_ToneOrderOK == FAIL)
      {
         EnterFailStates(E_CODE_TONE_ORDERING_ERROR);
      }

#ifndef HW_TRT
      // Save the last offset for use later
      gs_TxTRT_write_offset = s_start;
      gs_TxTRT_write_band = s_band;
#endif
      gul_ilvb_size = ConfigZephyrIlvTxIFM_VDSL2();
   }


   // init gt_RrcStat
   if (gt_ReTXParams.uc_OMSG1_DsReTxEnabled == RETX_SELECTED)
   {
      // these fields must have been initialized during training
      // gt_RrcStat.uc_CWsPerDtu = 30;

      gt_RrcStat.ull_64MostRecentDtuStats = 0;
      gt_RrcStat.ft_RrcStatUpdated = 0;

      // AbsDtuCnt is set to 0x1F so that upon the first DTU received
      // in showtime, it will be incremented to 0
      gt_RrcStat.uc_AbsDtuNum = 0x1F;
   }

   guc_BgTxShowInitState = TRAINING_DONE;
}

/*
*-------------------------------------------------------------------------------
*
*   Prototype: void TxIbInit()
*
*   This function initializes TxIbData structures.
*
*
*   Output Arguments:
*
*   Returns:
*
*   Global Variables:
*
*-------------------------------------------------------------------------------
*/
void TxIbInit(void)
{
   int16   s_lp;

   //Initialize the elements of the Tx IB Bits data structure
   for (s_lp=0; s_lp<NUM_DATA_PATHS; s_lp++)
   {
      gt_TxIbData.uc_atm_state[s_lp] = OCD;
      gt_TxIbData.s_ocd_cnt[s_lp] = 0;
      gt_TxIbData.uc_be_anom[s_lp] = TERMINATED;
      gt_TxIbData.uc_fec_anom[s_lp] = TERMINATED;
      gt_TxIbData.uc_hec_anom[s_lp] = TERMINATED;
      gt_TxIbData.uc_lcd_def[s_lp] = TERMINATED;
      gt_TxIbData.uc_ncd_anom[s_lp] = TERMINATED;
      gt_TxIbData.uc_ocd_anom[s_lp] = TERMINATED;
   }
   gt_TxIbData.uc_pro_fail = TERMINATED;
   gt_TxIbData.uc_sef_def = TERMINATED;
   gt_TxIbData.uc_ese_def = TERMINATED;
   gt_TxIbData.uc_fpo_def = TERMINATED;
   gt_TxIbData.uc_lol_fail = TERMINATED;
   gt_TxIbData.uc_lom_fail = TERMINATED;
   gt_TxIbData.uc_los_def = TERMINATED;
   gt_TxIbData.uc_lpr_def = TERMINATED;

}


#endif // INCLUDE_CACHE_FUNCTIONS
