/* **COPYRIGHT******************************************************************
    INTEL CONFIDENTIAL
    Copyright (C) 2017 Intel Corporation
    Copyright (C), 1994-2006 Aware Inc. All Rights Reserved.
******************************************************************COPYRIGHT** */
/* **DISCLAIMER*****************************************************************
    The source code contained or described herein and all documents related
    to the source code ("Material") are owned by Intel Corporation or its
    suppliers or licensors. Title to the Material remains with Intel
    Corporation or its suppliers and licensors. The Material may contain
    trade secrets and proprietary and confidential information of Intel
    Corporation and its suppliers and licensors, and is protected by
    worldwide copyright and trade secret laws and treaty provisions. No part
    of the Material may be used, copied, reproduced, modified, published,
    uploaded, posted, transmitted, distributed, or disclosed in any way
    without Intel's prior express written permission.

    No license under any patent, copyright, trade secret or other
    intellectual property right is granted to or conferred upon you by
    disclosure or delivery of the Materials, either expressly, by
    implication, inducement, estoppel or otherwise. Any license under
    such intellectual property rights must be express and approved by
    Intel in writing.
*****************************************************************DISCLAIMER** */
/*
*-------------------------------------------------------------------------
*
*   Aware DMT Technology. Proprietary and Confidential.
*
*   40 Middlesex Turnpike, Bedford, MA 01730-1413 USA
*   Phone (781) 276 - 4000
*   Fax   (781) 276 - 4001
*
*   sleep.c
*
*   This file contains functions to allow the modem to pause at and
*   resume from Tx and Rx States and Substates.
*
*-------------------------------------------------------------------------
*/

#include <string.h>
#include "common.h"
#include "gdata.h"

#include "afe.h"

#include "mtkernel.h"
#include "states.h"
#include "cmv_data.h"
#include "fifo.h"
#include "cri_iof.h"
#include "IRI_Iof.h"
#include "IRI_sync.h"
#include "V_STR_IOf.h"
#include "interrupt_control.h"
#include "ModulateSocMessages.h"
#include "FrameAlign.h"
#include "DSLEngin.h"
#include "fdq.h"
#include "accum32.h"
#include "sys_const.h"
#include "noiseacc.h"
#include "acc48_ops.h"
#include "PsdHandler.h"
#include "FrameAlignmentHandler.h"
#include "TimingAdvanceHandler.h"
#include "psd_b.h"
#include "snr_b.h"
#include "cmv.h"
#include "ghs.h"
#include "pll.h"
#include "FdqHandler.h"
#include "SnrHandler.h"
#include "dsp_op.h"
#include "MfdqHandler.h"
#include "TxPSDControl.h"
#include "PGAHandler.h"
#include "vdsl_xception.h"
#include "HybridTrainingHandler.h"
#include "vdsl_state.h"
#include "printtrail.h"
#include "SharedFuncs.h"
#include "DshInterface.h"

#include "DmtScope_MsgHandler.h"
#include "DmtScope_PsdMeasure.h"

#include "vecpwr.h"
#include "sleep_iof.h"
#include "AFED_ReadWriteModify.h"
#include "AFED_Constants.h"


static uint8 guca_octet[2] = {0x7F, 0x7F};


/*------------------------------------------------------------------------
 *
 *  Name : RunPrbsGenerator
 *
 *   This function is used to implement the following PRBS sequence 1 byte at a time:
 *
 *      d(n) = d(n-9) xor d(n-11).
 *
 *
 *  Prototype:  void RunPrbsGenerator(uint16 *us_PrbsState, uint8 *uc_octet)
 *
 *  Input Arguments:
 *      uint16 *us_PrbsState - pointer to the PRBS state

 *  Output Arguments:
 *      uint8 *uc_octet   - pointer to the return byte
 *
 *   Return:
 *      None
 *
 *  Global Variables Used:
 *      None
 *
 *
 *------------------------------------------------------------------------
 *^^^
 */
static void RunPrbsGenerator(uint16 *us_PrbsState, uint8 *uc_octet)
{
   uint16 us_tmp0, us_tmp1, us_tmp2;

   us_tmp0 = *us_PrbsState;         // Save PRBS state: samples [d(n-1) to d(n-16)]
   *uc_octet = (uint8)us_tmp0;         // Return samples [d(n+7) ... d(n+1) d(n)]

   us_tmp1 = us_tmp0 >> 5;            // Samples [d(n-4) ... d(n-10) d(n-11)] in LSByte
   us_tmp2 = us_tmp0 >> 7;            // Samples [d(n-2) ... d(n-8) d(n-9)] in LSByte
   us_tmp2 = (us_tmp2 ^ us_tmp1);      // d(m) = d(m-9) xor d(m-11) for m=n to n+7

   us_tmp0 >>= 8;                  // Shift off oldest 8 samples
   us_tmp2 <<= 8;                  // Generate 8 newest samples
   *us_PrbsState = us_tmp2 | us_tmp0;   // Save PRBS state
}


/*^^^
*----------------------------------------------------------------------------------------
*
*  Name : Tx_Rx_VarGainScalar
*
*   Prototype:
*       void Tx_Rx_VarGainScalar(int16* s_Gn, int16* s_GnExp,
*                           PwrConfigParam_t* pt_Pwr,uint8 uc_Tx_Rx)
*
*   Abstract:
*      This function scales the digital Tx/ Rx gain based on the CMV. This CMV is connected
*      to the global structure gt_PwrConfigParam. The s_Gn parameter is the resulting
*      gain in Q1.15 format and the s_GnExp is the exponent. Therefore, the total gain
*      is s_Gn x 2^s_GnExp. The last parameter, uc_Tx_Rx is to be used as a flag. Use 'T'
*      to indicate the transmit direction and everything else for 'R'. This is needed
*      because the G.997 spec talks of Upstream and downstream which is clearly different
*      for the CO and CPE. Downstream for CO is Tx and Upstream for CPE is Tx etc.
*
*   Input Parameters:
*
*   Global Variables: gt_PwrConfigParam
*
*----------------------------------------------------------------------------------------
^^^*/


void Tx_Rx_VarGainScalar(int16* s_Gn, int16* s_GnExp, PwrConfigParam_t* pt_Pwr,uint8 uc_Tx_Rx)
{
   int16 gain=0, l_LogGain;
   int32 temp;

   /*   The G997 spec. specifies Power as Upstream and Downstream and not as Rx and Tx, thus we
      need to differentiate between CO and CPE. For CO, Upstream = Receive
      and downstream = Transmit. For CPE, Upstream = Transmit and downstream = Receive. */

   /*   Obtain the gain value from the CMV. Gain is specified in 0.1 dB increments and the nominal
      value is different for CO mode and CPE mode. Therefore gain has to be massaged accordingly*/
   if (gus_ModemMode == CO_MODE)
   {
      if (uc_Tx_Rx == 'T')
      {
         gain = pt_Pwr->s_Dn_MaxNomPSD;   // For the transmit direction
      }
      else
      {
         gain = pt_Pwr->s_Up_MaxNomPSD;   // For the receive direction
      }

      gain = gain/10 - VDSL_CO_NOMINAL_GAIN;
   }
   if (gus_ModemMode == CPE_MODE) // Just to ensure
   {
      if (uc_Tx_Rx == 'T')
      {
         gain = pt_Pwr->s_Up_MaxNomPSD;   // For the transmit direction for CPE
      }
      else
      {
         gain = pt_Pwr->s_Dn_MaxNomPSD;   // For the receive direction for CPE
      }
      gain = gain/10 - VDSL_CPE_NOMINAL_GAIN;
   }

   /* Fix the exponent part of the gain. Exponent is adjusted by 1 for each 6dB step*/
   if (gain >= 6)
      while (gain >= 6)
      {
         gain -= 6;
         (*s_GnExp)--;
      }
   else if (gain <=-6)
      while (gain <=-6)
      {
         gain += 6;
         (*s_GnExp)++;
      }

   //Convert gs_TxVarGain_dB to a linear multiplicative gain.
   //Note that the function ConvertLogToLinear(uint16 *pusa_LinearPSDOut, uint16 pusa_LogPSDIn)
   //converts a Q7.9 dB value (a negative dB value, passed as a positive number)
   //into the corresponding a Q3.13 Linear Value (10^(-pusa_LogPSDIn/20)).

   // Convert to 7.9 format.
   l_LogGain = (uint16) ((int32)gain << 9);
   if (l_LogGain<0)
   {
      l_LogGain = -l_LogGain;
      ConvertLogToLinear((uint16 *)(void *)s_Gn,l_LogGain);
   }
   else
   {
      ConvertLogToLinear((uint16 *)(void *)s_Gn,l_LogGain);
      // Invert, to reflect that l_LogGain is positive, but ConvertLogToLinear treated it as negative.
      (*s_Gn) = (int16) (((int32) (1<<13) * (1<<13)) / (int32)(*s_Gn));
   }

   // Multiply this Linear gain with the unity gain. Remember s_Gn is in Q3.13 format and
   // UNITY_TX_VAR_GAIN is in Q1.15 format
   temp = (int32) UNITY_TX_VAR_GAIN * (int32)(*s_Gn);

   //Now turn this Q4.28 number into a Q4.12 number
   *s_Gn = (int16) round(temp,16);

   //Now turn this Q4.12 into Q1.15 number
   *s_Gn <<=3;
   return ;
}

/*^^^
 *------------------------------------------------------------------------
 *
 *  Name : ClearTransmitGains
 *
 *  Description:
 *
 *      This function clears the inactive GST tables based on the
 *   Band Descriptor block.  The subsequent QT processing will zero
 *   out the IFFT input.  This function is used during GHS and TONE
 *   generation, in which
 *
 *      (1) the IFFT is placed into register start mode
 *      (2) the QT clears the IFFT input
 *      (3) the ARC loads the desired IFFT intput
 *      (4) the ARC triggers the IFFT
 *
 *  Prototype:
 *      void Clear TransmitGains(void)
 *
 *  Input Arguments:
 *
 *  Output Arguments:
 *
 *  Return:
 *
 *  Global Variables Used:
 *      gs_NumOfTxBands -- (I) no. of TX frequency bands
 *      gsa_TxBandLeftChannel -- (I) an array of left band edges
 *      gsa_TxBandRightChannel -- (I) an array of right band edges
 *
 *  Notes:
 *
 *------------------------------------------------------------------------
 *^^^
 */
void ClearTransmitGains(void)
{
   int16 i, j;

   for(i=0; i<gs_NumOfTxBands; i++)
   {
      // Set EGT and GST for Tx tones between left and right edges of bands inclusive.
      for (j = gsa_TxBandLeftChannel[i]; j <= gsa_TxBandRightChannel[i]; j++)
      {
         // Set GST values to unity
         *(ghpsa_TxFineGains_Inactive+j) = 0x000;
      }
   }

   guc_ClearTransmitGainsState = TRAINING_DONE;
}

void BgVectorPowerHandler(void)
{
   gl_Pa += VectorPower((void*)gla_SharedBuffer_X, 0, 4096, (RX_LOG2_MAX_FFT_LENGTH));
   gl_Pa += VectorPower((void*)gla_SharedBuffer_Y, 0, 4096, (RX_LOG2_MAX_FFT_LENGTH));
   gs_PgaHandlerState = PGA_DONE;
}


#ifdef BIT_PATTERN_TEST
void BgFindPathDelayUseRamp(void)
{
   int16 i;

   gs_PathDelay = -1;

   for (i = 0; i< gs_TxIfftLength; i++ )
   {
      if (gpsa_RxTimeBuffer[i]==0)
      {
         continue;
      }
      else if ((gpsa_RxTimeBuffer[i] == 1) || (gpsa_RxTimeBuffer[i+1] == 2)) // only check two entries
      {
         gs_PathDelay = i;
      }
   }
   guc_FindPathDelayState = TRAINING_DONE;
}


void BgFindSampleDiff(void)
{
   int16 i;

   for (i = gs_BitPatternToneOffset*2; i< gs_BitPatternToneOffset*2+1024; i++ )
   {

      if (gpsa_RxTimeBuffer[i+gs_PathDelay] != gpsa_IfftBuffer[i])
      {
         gul_NumDiff ++;
#ifdef DEBUG_TRAIL
         gsa_StatesTrail[gs_DiffLogIndex ] = gpsa_IfftBuffer[i];
         gs_DiffLogIndex = (gs_DiffLogIndex+1) & (STATES_TRAIL_LENGTH-1);
         gsa_StatesTrail[gs_DiffLogIndex ] = gpsa_RxTimeBuffer[i+gs_PathDelay];
         gs_DiffLogIndex = (gs_DiffLogIndex+1) & (STATES_TRAIL_LENGTH-1);
#endif // DEBUG_TRAIL
      }
   }
   guc_FindSampleDiffState = TRAINING_DONE;
}

void FindSampleDiff(void)
{
   int i;

   for (i = gs_BitPatternToneOffset*2; i< gs_BitPatternToneOffset*2 - gs_PathDelay; i++ )
   {

      if (gpsa_RxToneBuf[i+gs_PathDelay] != gpsa_IfftBuffer[i])
      {
         gul_NumDiff ++;
#ifdef DEBUG_TRAIL
         gsa_StatesTrail[gs_DiffLogIndex ] = gpsa_IfftBuffer[i];
         gs_DiffLogIndex = (gs_DiffLogIndex+1) & (STATES_TRAIL_LENGTH-1);
         gsa_StatesTrail[gs_DiffLogIndex ] = gpsa_RxToneBuf[i+gs_PathDelay];
         gs_DiffLogIndex = (gs_DiffLogIndex+1) & (STATES_TRAIL_LENGTH-1);
#endif // DEBUG_TRAIL
      }
   }
}


void FindPulseLocation(void)
{
   int16 i;
   int32 l_x, l_x_max;

   gs_PathDelay = -1;
   l_x_max = 0;

   for (i = 0; i< gs_TxIfftLength; i++ )
   {
      //Compute the absolute value
      l_x = gpsa_RxTimeBuffer[i];
      if(l_x < 0)
      {
         l_x = -l_x;
      }

      if(l_x > l_x_max)
      {
         gs_PathDelay = i;
         l_x_max = l_x;
      }
   }
} //void FindPulseLocation(void)

void BgCalcEchoPathDelay(void)
{
   int16 s_tx_lead_rx_amount;

   //Find the pulse location in the RX sample frame
   //the location is stored in gs_PathDelay
   FindPulseLocation();

   //Compute the TX and RX frame offset
   gft_SkipCpAdjust = 1;

   //Note the returned s_tx_lead_rx_amount is in samples corresponding to 4096 tone FFT size
   s_tx_lead_rx_amount = CalcTxRxFrameOffset();

   gft_SkipCpAdjust = 0;


   //Add the TX/RX frame offset to the pulse location to get the total delay
   if(gs_PathDelay >= 0)
   {
      //Express the delay in samples corresponding to 4096 tone FFT size
      gs_PathDelay <<= (TX_LOG2_MAX_FFT_LENGTH-gs_TxLog2IfftLength);
      gs_PathDelay += s_tx_lead_rx_amount;
   }

   guc_FindPathDelayState = TRAINING_DONE;
}


#endif //BIT_PATTERN_TEST
/*^^^
 *------------------------------------------------------------------------
 *
 *  Name : RSleepTxF
 *
 *  Prototype:  void RSleepTxF(void)
 *
 *      Quiescent state to provide transmit signal control
 *
 *  Input Arguments:
 *
 *  Output Arguments:
 *
 *   Return:
 *      None
 *
 *  Global Variables Used:
 *
 *  Notes:
 *
 *------------------------------------------------------------------------
 *^^^
 */
uint16 gus_aferdaddr = 0x308; //AGC1_GAIN
uint16 gus_rd_data = 0;
uint8 gft_RunPrbsGeneratorFlag = 0;
void RSleepTxF(void)
{
   uint8 uc_octet=0;
   static uint16 us_PrbsState = 0x07FF;   // Initialize PRBS: d(n) = 1 for n=1 to 11, 0 for n=12 to 16
   // Note: d(1) is the lsbit
   uint16 ReadDataDummy;
   uint16 FcsiInterface = FCSI_DSL;

   // Run PRBS to generate an HDLC octet each frame
   if (gft_RunPrbsGeneratorFlag == 1)
   {
      RunPrbsGenerator(&us_PrbsState, &uc_octet);
      ModulateHdlcOctets(&uc_octet, 1);
   }

#ifdef BIT_PATTERN_TEST
   if(gs_LoadGhsToneFlag ==(BITMAP_LOAD_BIT_PATTERN | BITMAP_REGSTART_IFFT))
   {
      AddFunctionToFifo(gp_TxLoadingFunctionFifo, LoadBitPatternToIfftBuf);
   }

#endif //BIT_PATTERN_TEST

   switch (TESTArray[TEST_TxSubState])
   {

   case -1: // Default substate
      break;

   case 0: // Send QUIET
   case 5: // Send TONE (should run sending TRAIN/MEDLEY before this to set the TX gain properly)

      guc_ClearTransmitGainsState = TRAINING_IN_PROGRESS;
      AddFunctionToBkgdFifo((PtrToBkgdFunc)ClearTransmitGains);

      gs_TxSubState = TESTArray[TEST_TxSubState];  //VR9_BRINGUP
      TESTArray[TEST_TxSubState] = 11;

      break;

   case 11: // Background part of Send QUIET

      if (guc_ClearTransmitGainsState == TRAINING_DONE)
      {
         AddFunctionToFifo(gp_TxLoadingFunctionFifo, LoadIridiaTransmitGains);

         switch(gs_TxSubState)
         {
         case 0: //send QUIET

            //disable tone loading, enable register start IFFT bit
            gs_LoadGhsToneFlag = (BITMAP_REGSTART_IFFT);
            break;

         case 5: //send TONE

            //Enable loading single tone, enable register start IFFT bit
            gs_LoadGhsToneFlag = (BITMAP_LOAD_SINGLE_TONE | BITMAP_REGSTART_IFFT);
            break;
         }

         TESTArray[TEST_TxSubState] = -1;

      } //if (guc_ClearTransmitGainsState == TRAINING_DONE)
      break;

   case 1: //Send TRAIN (REVERB)
   case 2: //Send SEGUE
   case 3: //Send MEDLEY

      // Setup TxPSDControl control structure used for Transmit Signal
      gt_TxPsdControl.s_MaxNomPsdIn = gt_PwrConfigParam.s_Up_MaxNomPSD;

      /* No PSD celing to be applied in VDSL1 mode, simply use the ME */
      /* configured Max PSD as Tx PSD */
      gt_TxPsdControl.s_PsdCeiling = gt_PwrConfigParam.s_Up_MaxNomPSD;
      gt_TxPsdControl.pt_MaxPsdDescIn = (PSDDescriptorTable_t*)(void *)&gt_MaxUsPSDDescriptorTable;
      gt_TxPsdControl.pt_RefPsdDescIn = NULL;

      gt_TxPsdControl.pt_ActPsdDescOut = (PSDDescriptorTable_t*)(void *)&gt_UsREFPSDDescriptorTable;
      // Set MAXNOMATP
      gt_TxPsdControl.s_MaxAtp = gt_PwrConfigParam.s_Up_MaxNomAggrPwr;

      guc_SetTransmitGainsState = TRAINING_IN_PROGRESS;

      //Disable the direct copy
      AddFunctionToFifo(gp_TxLoadingFunctionFifo,DisableDCI);

      AddFunctionToBkgdFifo((PtrToBkgdFunc)SetTransmitGains);

      //disable tone loading
      gs_LoadGhsToneFlag = 0;

      gs_TxSubState = TESTArray[TEST_TxSubState];  //VR9_BRINGUP
      gs_TxSubStateCnt = 0;
      TESTArray[TEST_TxSubState] = 0x1A0;

      break;

   case 0x1A0: //Background part of Send TRAIN, MEDLEY

      if (guc_SetTransmitGainsState == TRAINING_DONE)
      {
         // Send the structure of gt_TxPsdControl as debug stream    (size of DSH_MAX_PSD_DESC_IN = 386 bytes considering MAX_NUM_PSD_POINTS] = 64)
         DSH_SendStream(DSH_TX_PSD_CONTROL,36,&gt_TxPsdControl);
         DSH_SendStream(DSH_MAX_PSD_DESC_IN,386,gt_TxPsdControl.pt_MaxPsdDescIn);
         DSH_SendStream(DSH_Max_PSD_DESC_OUT,386,gt_TxPsdControl.pt_ActPsdDescOut);
         if (gs_TxSubStateCnt == 0)
         {
            gs_TxSubStateCnt++;

            //Set the message bytes to HDLC flag
            gs_TxNumBytesPerSymbol = 1;
            ModulateHdlcOctets(guca_octet, gs_TxNumBytesPerSymbol);

            switch(gs_TxSubState)
            {
            case 1:   //Send REVERB
               gft_ReverbSegue = REVERB;
               AddFunctionToFifo(gp_TxLoadingFunctionFifo, SetUpTrainingSignal);
               break;

            case 2: //Send SEGUE (not supported)

               gft_ReverbSegue = SEGUE;
               AddFunctionToFifo(gp_TxLoadingFunctionFifo, SetUpTrainingSignal);
               break;

            case 3: //Send Medley
               AddFunctionToFifo(gp_TxLoadingFunctionFifo, SetUpMedleySignal);
               break;

            default:
               break;
            }

            // Setup for Loading of Transmit Gains
            guc_LoadTransmitGainsState = LTG_LOAD_IRIDIA;

         } ////if (gs_TxSubStateCnt == 0)

         //Reconfig the HW to change the Tx Gains,
         if (guc_LoadTransmitGainsState != LTG_LOAD_DONE)
         {
            LoadTransmitGains();
         }
         else
         {
            TESTArray[TEST_TxSubState] = -1;
         }
      }
      break;

   case 4: /* MTPR test */

      TESTArray[TEST_TxSubState] = -1;
      break;

   case 6: /* Send two frequency bands of REVERB, no matter disjoint or not. */

      /* default TESTArray[Test_EncoderGain] = ENCGAIN2 */
      TESTArray[TEST_TxSubState] = -1;
      break;

   case 7: /* Send ECT (Echo canceller training sequence */

      TESTArray[TEST_TxSubState] = -1;
      break;

   case 8: // Set IFFT scalebacks

      // Updates HW IFFT scaling according to current value of gs_TxLog2FftScaling.
      TESTArray[TEST_TxSubState] = -1;
      break;

   case 9:

      /* Reset hardware overflow counter to be 0 */
      TESTArray[TEST_TxSubState] = -1;
      break;

   case 10:

      TESTArray[TEST_TxSubState] = -1;
      //g_collect_tx = 1;

      break;

      // case 11:   RESEREVED. DO NOT USE. Already used above..


   case 15: // Timing Advance

      // Initialize vars for Timing Advance
      gs_TimingAdvanceHandlerState = TIMING_ADVANCE_INIT;

      TESTArray[TEST_TxSubState] = 150;

      break;

   case 150:
      if (gs_TimingAdvanceHandlerState != TIMING_ADVANCE_DONE)
      {
         TimingAdvanceHandler();
      }
      else
      {
         TESTArray[TEST_TxSubState] = -1;
      }

      break;

   case 17:  /* Enable PARR */

      TESTArray[TEST_TxSubState] = -1;
      break;

   case 18:  /*Disable PARR */

      TESTArray[TEST_TxSubState] = -1;
      break;

   case 20:  /* Tx CE Control */

      // Adjust TX CE
      AddFunctionToFifo(gp_TxLoadingFunctionFifo, InsertTxCE);

      TESTArray[TEST_TxSubState] = -1;
      break;

   case 21:   /* Tx Algignment Control */

      gs_TxFrameAlignRemain = 0;

      gs_TimingAdvance = TESTArray[TEST_TimingAdvance];

      // Adjust Tx Alignment
      AddFunctionToFifo(gp_TxLoadingFunctionFifo, AdjustTxTimingAdvance);

      TESTArray[TEST_TxSubState] = 210;
      break;

   case 210:   /* Tx Algignment Control */

      //For the 6.2 HW, it is not allowed to make a strymon frame less than IFFT size
      //so it may take more than one frame to do TA adjustment if gs_TimingAdvance is greater than
      //CE Length. Note the variable gs_TxFrameAlignRemain stores the remaining adjustment
      //after each call to AdjustTxTimingAdvance()
      if(gs_TxFrameAlignRemain > 0)
      {
         // queue loading function to make Timing Advance adjustment
         AddFunctionToFifo(gp_TxLoadingFunctionFifo, AdjustTxTimingAdvance);
      }
      else
      {

         // Adjust Tx Alignment
         AddFunctionToFifo(gp_TxLoadingFunctionFifo, ResetTxTimingAdvance);

         TESTArray[TEST_TxSubState] = -1;
      }
      break;

   case 22:   /* Get Tx/Rx Alignment */

      // queue loading function to measure Tx/Rx Counters, which
      // are returned in gl_TxTimerVal and gl_RxTimerVal
      AddFunctionToFifo(gp_TxLoadingFunctionFifo, GetTxRxCounters);

      TESTArray[TEST_TxSubState] = -1;
      break;


   case 30:   /* Enable window Loop back testing */

      // Enable Loopback
      AddFunctionToFifo(gp_TxLoadingFunctionFifo, EnableTimeDomainLoopback);
      TESTArray[TEST_TxSubState] = -1;
      break;

   case 31:   /* Disable window Loop back testing */

      // Disable Loopback
      AddFunctionToFifo(gp_TxLoadingFunctionFifo, DisableTimeDomainLoopback);

      TESTArray[TEST_TxSubState] = -1;
      break;

      //Enable loopback at the AFE interface
   case 34:
      AddFunctionToFifo(gp_TxLoadingFunctionFifo, EnableTxToRxLoopback);

      TESTArray[TEST_TxSubState] = -1;
      break;

      //Disable loopback at the AFE interface
   case 35:
      AddFunctionToFifo(gp_TxLoadingFunctionFifo, DisableTxToRxLoopback);

      TESTArray[TEST_TxSubState] = -1;
      break;

      //Enable loopback at the SerDes
   case 341:
      AddFunctionToFifo(gp_TxLoadingFunctionFifo, EnableSerDesLoopback);

      TESTArray[TEST_TxSubState] = -1;
      break;

      //Disable loopback at the SerDes
   case 351:
      AddFunctionToFifo(gp_TxLoadingFunctionFifo, DisableSerDesLoopback);

      TESTArray[TEST_TxSubState] = -1;
      break;

#ifdef BIT_PATTERN_TEST
   case 401:
      //configure unity VAR Gain
      AddFunctionToFifo(gp_TxLoadingFunctionFifo, SetTxVarGain);
      TESTArray[TEST_TxSubState] = -1;
      break;

   case 402:
      //Disable IFFT and digit ordering at QT interface
      //Disable FFT and digit ordering at DSP interface.
      AddFunctionToFifo(gp_TxLoadingFunctionFifo,Disable_IFFT);
      AddFunctionToFifo(gp_TxLoadingFunctionFifo,Disable_FFT);
      TESTArray[TEST_TxSubState] = -1;
      break;

   case 403:
      //Enable load of bit test pattern into IFFT buffer, enable register start IFFT bit
      gs_LoadGhsToneFlag = (BITMAP_LOAD_BIT_PATTERN | BITMAP_REGSTART_IFFT);
      TESTArray[TEST_TxSubState] = -1;
      break;

   case 404:
      //Enable to read the IFFT buffer
      gft_EnableGetIfftBuf = 1;
      gft_TriggerGetIfft  = 1;
      TESTArray[TEST_TxSubState] =  405;
      break;

   case 405:
      //disable read the IFFT buffer
      gft_EnableGetIfftBuf = 0;
      TESTArray[TEST_TxSubState] = -1;
      break;

   case 406:
      //Disable load of bit test pattern into IFFT buffer, enable register start IFFT bit
      gs_LoadGhsToneFlag =0;
      TESTArray[TEST_TxSubState] = -1;

#endif //#ifdef BIT_PATTERN_TEST

      /* To be enabled when needed for testing

         case 32:  //Test Transition from Synchro1 to O/R-P-Periodic state and then to Synchro2

            if(gs_TxSubStateCnt == 0)
            {
               AddFunctionToFifo(gp_TxLoadingFunctionFifo,   Enable_IFFT_Reordering);
               AddFunctionToFifo(gp_TxLoadingFunctionFifo,   InsertTxCE);
               AddFunctionToFifo(gp_TxLoadingFunctionFifo,   InsertRxCE);

               GenerateSynSymLookUpData((int16)(gs_TxGain));

               SetTransmitGains();
               AddFunctionToFifo(gp_TxLoadingFunctionFifo, SetUpTrainingSignal);
               AddFunctionToFifo(gp_TxLoadingFunctionFifo,LoadTransmitGains);

               //Disable the loading IFFT buffer feature
               gs_LoadGhsToneFlag = 0;

         }
            //wait for a few of symbols
            else if(gs_TxSubStateCnt == 2)
            {
               gs_TxSubStateCnt = -1;
               TESTArray[TEST_TxSubState] = 320;
            }
            gs_TxSubStateCnt++;

            break;

         case 320: //sending O-P-Synchro1

            if((gs_TxSubStateCnt == 0) || (gs_TxSubStateCnt == 10))
            {
      #ifdef CAPTURE_AFE_TX_OUT
               gs_CaptureAfeTxOutFlag = 1;
               gs_CaptureAfeRxInFlag = 0;
      #endif

               GenerateSynSymLookUpData((int16)(-gs_TxGain));
      }
            //For 3 middle symbols, //set all tones to (+,+)
            else if(gs_TxSubStateCnt == 5)
            {
               GenerateSynSymLookUpData(gs_TxGain);
            }
            //After sending the 15-th symbol, go to O-P-Quiet/O-P-Pilot state if the lineprobe
            //is requested, go to O-P-Peridic state if the lineprobe is not requested
            else if(gs_TxSubStateCnt == 14)
            {
               gs_TxSubStateCnt = -1;
               TESTArray[TEST_TxSubState] = 321;
            }
            gs_TxSubStateCnt++;
            break;

         case 321:   //sending O-P-Periodic1

            if(gs_TxSubStateCnt == 0)
            {
               //Value of 11 should be modulated on each DS tone
               GenerateSynSymLookUpData((int16)(-gs_TxGain));
            }
            else if(gs_TxSubStateCnt == 1)
            {
                  //queue the TC task for programing the HW core
               //For the first symbol, set CP length = beta, leaving window on and turn off CS
               AddFunctionToFifo(gp_TxLoadingFunctionFifo, ChangeTxCE_ForFirstSym);
            }

            else if(gs_TxSubStateCnt == 2)
            {
               AddFunctionToFifo(gp_TxLoadingFunctionFifo, DisableTxCE);

               gs_TimingAdvance = gs_TxIfftLength + gs_TxCELength;
            }
            else
            {

               //CE should be enabled after the number of samples being sent equals to an integer multiple of
               //(2N+Lce), which we call a special boundary
               //Note: where gs_TimingAdvance is set to gs_TxIfftLength + gs_TxCELength
               if((((gs_TxSubStateCnt+1)*gs_TxIfftLength)%gs_TimingAdvance) == 0)
               {
                  //Go to the next TX state to send R-P-Synchro2
                  gs_TxSubStateCnt = -1;
                  TESTArray[TEST_TxSubState] = 322;
               }
            }

            gs_TxSubStateCnt++;
            break;

         case 322: //sending O-P-Synchro2

            if(gs_TxSubStateCnt == 0)
            {
               //This is for the last symbol of the last state
               AddFunctionToFifo(gp_TxLoadingFunctionFifo, ChangeTxCE_ForLastSym);
            }
            else if(gs_TxSubStateCnt == 1)
            {
               //queue the TC task for programing the HW core to insert
                //cyclic extension
                AddFunctionToFifo(gp_TxLoadingFunctionFifo, InsertTxCE);

               //programing the HW core to restore the original frame length
               AddFunctionToFifo(gp_TxLoadingFunctionFifo, ConfigTxFrameSize);

            }

            //For the first 5 and last 5 symbols, set all the tones to (-,-)
            if((gs_TxSubStateCnt == 0) || (gs_TxSubStateCnt == 10))
            {
               GenerateSynSymLookUpData((int16)(-gs_TxGain));
            }
            //For 3 middle symbols, //set all tones to (+,+)
            else if(gs_TxSubStateCnt == 5)
            {
               GenerateSynSymLookUpData(gs_TxGain);
            }
            else if(gs_TxSubStateCnt == 14)
            {

      #ifdef CAPTURE_AFE_TX_OUT
               //disable printout
               gs_CaptureAfeTxOutFlag = 0;
      #endif
            }


            gs_TxSubStateCnt++;

            break;
      */

      //Test SSC Interface for AFE configuration
      //Specify the AFE config Clock speed through the global variable gul_SSC_Speed

   case 37: //VR9_BRINGUP

      TESTArray[TEST_TxSubState] = -1;
      break;

   case 38: //VR9_BRINGUP
      wait_ms(10);

      TESTArray[TEST_TxSubState] = -1;
      break;

   case 39: //VR9_BRINGUP

   gus_rd_data = VRX5AFE_FCSI_READ(FcsiInterface, gus_aferdaddr,&ReadDataDummy);
      TESTArray[TEST_TxSubState] = -1;

   case 40: //VR9_BRINGUP
      VRX5AFE_FCSI_RMW(FcsiInterface, gus_aferdaddr, 0xFFFF, gus_rd_data);

      TESTArray[TEST_TxSubState] = -1;
      break;

   case 50: //518 Test Code

      int16 s_compensationLevel, s_ToneIdx=0;
      PSDCompDescriptorTable_t *pt_PSDCompDesc;


      //pt_PSDCompDesc =  (PSDDescriptorTable_t) (void *) (&gt_PSDCompData.t_PSDCompTable);
      pt_PSDCompDesc = (PSDCompDescriptorTable_t *) (void *) &(gt_PSDCompData.t_PSDCompTable);

      gt_UsTSSIPSDDescriptorTable.us_NumberOfTones = 18;
      gt_UsTSSIPSDDescriptorTable.ut_PSDRecord[0].us_IndexOfTone = 1;
      gt_UsTSSIPSDDescriptorTable.ut_PSDRecord[0].s_PSDLevelOfTone = 405;
      gt_UsTSSIPSDDescriptorTable.ut_PSDRecord[1].us_IndexOfTone = 3;
      gt_UsTSSIPSDDescriptorTable.ut_PSDRecord[1].s_PSDLevelOfTone = 525;
      gt_UsTSSIPSDDescriptorTable.ut_PSDRecord[2].us_IndexOfTone = 7;
      gt_UsTSSIPSDDescriptorTable.ut_PSDRecord[2].s_PSDLevelOfTone = 580;
      gt_UsTSSIPSDDescriptorTable.ut_PSDRecord[3].us_IndexOfTone = 14;
      gt_UsTSSIPSDDescriptorTable.ut_PSDRecord[3].s_PSDLevelOfTone = 610;
      gt_UsTSSIPSDDescriptorTable.ut_PSDRecord[4].us_IndexOfTone = 31;
      gt_UsTSSIPSDDescriptorTable.ut_PSDRecord[4].s_PSDLevelOfTone = 620;
      gt_UsTSSIPSDDescriptorTable.ut_PSDRecord[5].us_IndexOfTone = 32;
      gt_UsTSSIPSDDescriptorTable.ut_PSDRecord[5].s_PSDLevelOfTone = 167;
      gt_UsTSSIPSDDescriptorTable.ut_PSDRecord[6].us_IndexOfTone = 60;
      gt_UsTSSIPSDDescriptorTable.ut_PSDRecord[6].s_PSDLevelOfTone = 167;
      gt_UsTSSIPSDDescriptorTable.ut_PSDRecord[7].us_IndexOfTone = 61;
      gt_UsTSSIPSDDescriptorTable.ut_PSDRecord[7].s_PSDLevelOfTone = 620;
      gt_UsTSSIPSDDescriptorTable.ut_PSDRecord[8].us_IndexOfTone = 870;
      gt_UsTSSIPSDDescriptorTable.ut_PSDRecord[8].s_PSDLevelOfTone = 167;
      gt_UsTSSIPSDDescriptorTable.ut_PSDRecord[9].us_IndexOfTone = 871;
      gt_UsTSSIPSDDescriptorTable.ut_PSDRecord[9].s_PSDLevelOfTone = 167;
      gt_UsTSSIPSDDescriptorTable.ut_PSDRecord[10].us_IndexOfTone = 1205;
      gt_UsTSSIPSDDescriptorTable.ut_PSDRecord[10].s_PSDLevelOfTone = 182;
      gt_UsTSSIPSDDescriptorTable.ut_PSDRecord[11].us_IndexOfTone = 1206;
      gt_UsTSSIPSDDescriptorTable.ut_PSDRecord[11].s_PSDLevelOfTone = 620;
      gt_UsTSSIPSDDescriptorTable.ut_PSDRecord[12].us_IndexOfTone = 1971;
      gt_UsTSSIPSDDescriptorTable.ut_PSDRecord[12].s_PSDLevelOfTone = 620;
      gt_UsTSSIPSDDescriptorTable.ut_PSDRecord[13].us_IndexOfTone = 1972;
      gt_UsTSSIPSDDescriptorTable.ut_PSDRecord[13].s_PSDLevelOfTone = 203;
      gt_UsTSSIPSDDescriptorTable.ut_PSDRecord[14].us_IndexOfTone = 2319;
      gt_UsTSSIPSDDescriptorTable.ut_PSDRecord[14].s_PSDLevelOfTone = 210;
      gt_UsTSSIPSDDescriptorTable.ut_PSDRecord[15].us_IndexOfTone = 2782;
      gt_UsTSSIPSDDescriptorTable.ut_PSDRecord[15].s_PSDLevelOfTone = 210;
      gt_UsTSSIPSDDescriptorTable.ut_PSDRecord[16].us_IndexOfTone = 2783;
      gt_UsTSSIPSDDescriptorTable.ut_PSDRecord[16].s_PSDLevelOfTone = 620;
      gt_UsTSSIPSDDescriptorTable.ut_PSDRecord[17].us_IndexOfTone = 8191;
      gt_UsTSSIPSDDescriptorTable.ut_PSDRecord[17].s_PSDLevelOfTone = 620;

      //gt_PSDCompData.t_PSDCompTable = { { 32, 0 }, { 44, 3 }, { 60, 2 }, { 866, -1 }, { 1208, -2 }, { 1968, -6 }, { 2235, -7 }, { 2628, -8 }, { 2786, -10 } }
      gt_PSDCompData.us_NumberOfCompBreakPoint = 9;
      gt_PSDCompData.t_PSDCompTable.ut_PSDRecord[0].us_IndexOfTone = 32;
      gt_PSDCompData.t_PSDCompTable.ut_PSDRecord[0].s_PSDLevelOfTone = 0;
      gt_PSDCompData.t_PSDCompTable.ut_PSDRecord[1].us_IndexOfTone = 44;
      gt_PSDCompData.t_PSDCompTable.ut_PSDRecord[1].s_PSDLevelOfTone = 3;
      gt_PSDCompData.t_PSDCompTable.ut_PSDRecord[2].us_IndexOfTone = 60;
      gt_PSDCompData.t_PSDCompTable.ut_PSDRecord[2].s_PSDLevelOfTone = 2;
      gt_PSDCompData.t_PSDCompTable.ut_PSDRecord[3].us_IndexOfTone = 866;
      gt_PSDCompData.t_PSDCompTable.ut_PSDRecord[3].s_PSDLevelOfTone = -1;
      gt_PSDCompData.t_PSDCompTable.ut_PSDRecord[4].us_IndexOfTone = 1208;
      gt_PSDCompData.t_PSDCompTable.ut_PSDRecord[4].s_PSDLevelOfTone = -2;
      gt_PSDCompData.t_PSDCompTable.ut_PSDRecord[5].us_IndexOfTone = 1968;
      gt_PSDCompData.t_PSDCompTable.ut_PSDRecord[5].s_PSDLevelOfTone = -6;
      gt_PSDCompData.t_PSDCompTable.ut_PSDRecord[6].us_IndexOfTone = 2235;
      gt_PSDCompData.t_PSDCompTable.ut_PSDRecord[6].s_PSDLevelOfTone = -7;
      gt_PSDCompData.t_PSDCompTable.ut_PSDRecord[6].us_IndexOfTone = 2628;
      gt_PSDCompData.t_PSDCompTable.ut_PSDRecord[6].s_PSDLevelOfTone = -8;
      gt_PSDCompData.t_PSDCompTable.ut_PSDRecord[6].us_IndexOfTone = 2786;
      gt_PSDCompData.t_PSDCompTable.ut_PSDRecord[6].s_PSDLevelOfTone = -10;

      gula_DebugVarsPalak[18] = 0x1234;


      for (s_ToneIdx = 0; s_ToneIdx < 4096; s_ToneIdx++)
      {
         s_compensationLevel = PSDCompensation(s_ToneIdx, gt_PSDCompData.us_NumberOfCompBreakPoint, pt_PSDCompDesc);
         //gsa_FG[s_ToneIdx] = s_compensationLevel;
         //gsa_dbg_StateTrail[s_ToneIdx] = s_compensationLevel;
      }
      gula_DebugVarsPalak[19] = s_ToneIdx;
      break;

   }


}

/*^^^
 *------------------------------------------------------------------------
 *
 *  Name : RSleepRxF
 *
 *  Prototype:  void RSleepRxF(void)
 *
 *      Quiescent state to provide configuration control
 *
 *  Input Arguments:
 *
 *  Output Arguments:
 *
 *   Return:
 *      None
 *
 *  Global Variables Used:
 *
 *  Notes:
 *
 *------------------------------------------------------------------------
 *^^^
 */
// Mentioned in MessageSpec
#define MEASURE_PSD                 14
#define FDQ_TRAINING                16
#define MEASURE_SNR                 17
#define PGA_TRAINING                19

// Not mentioned in MessageSpec
#define CE_ADJUST_RX                20
#define FRAME_ALIGN_RX              21

#define ACE_MODE_SET                27
#define HYBRID_SET                  28


void EnableDecLpbk(void);

void RSleepRxF(void)
{
   int32 l_acc_snr;
   int16 i,j;

   //update gs_RxSubState to reflect the TEST RxSubState
   gs_RxSubState = TESTArray[TEST_RxSubState];

   switch (TESTArray[TEST_RxSubState])
   {

   case -1: // Default substate
      gs_RxSubStateCnt = 0;
      break;

   case 0: /* Initialize for frequency domain data - FFT output */

      // Enable Rx PRBS Descrambling
      AddFunctionToFifo(gp_RxLoadingFunctionFifo, EnableRxDescrambling);


      // Enable GetRxTones in RxProcessTones fcn
      gft_EnableGetRxTones = TRUE;

      // Set RTV selection to FFT output
      gs_RtvSelect = FFT_OUTPUT;

      TESTArray[TEST_RxSubState] = -1;

      break;


   case 2: /* Disable FDQ */

      // Set RTV selection to FFT output
      gs_RtvSelect = FFT_OUTPUT;
      TESTArray[TEST_RxSubState] = -1;
      break;

   case 3: /* Enable FDQ */

      // Set RTV selection to FFT output
      gs_RtvSelect = SFDQ_OUTPUT;
      TESTArray[TEST_RxSubState] = -1;
      break;

   case 4: /* Disable TDQ */

      TESTArray[TEST_RxSubState] = -1;
      break;

   case 5: /* Enable TDQ */

      TESTArray[TEST_RxSubState] = -1;
      break;

   case 6: /* Disable DEC */

      TESTArray[TEST_RxSubState] = -1;
      break;

   case 7: /* Enable DEC */

      TESTArray[TEST_RxSubState] = -1;
      break;


   case 9: /* Enable AEC */

      TESTArray[TEST_RxSubState] = -1;
      break;

   case 10: /* Perform frame alignment (using gus_syncoffset) */

      TESTArray[TEST_RxSubState] = -1;
      break;

   case 11: // Calculate PGA and set PGA

      gs_PGA_required = TESTArray[TEST_PGA_required]; // Q8.8 format

      guc_PgaState = TRAINING_IN_PROGRESS;
      AddFunctionToBkgdFifo((PtrToBkgdFunc)AFED_BgSingleStagePGASetting);

      TESTArray[TEST_RxSubState] = 110;
      break;

   case 110:
      if (guc_PgaState == TRAINING_DONE)
      {
         TESTArray[TEST_RxSubState] = -1;
      }
      break;


   case 13: /* Average received signal over # frames */

      if (gs_RxSubStateCnt == 1)
      {
         TESTArray[TEST_RxSubState] = -1;
      }
      else
      {
         if (gs_RxSubStateCnt == 0)
         {
            gl_Pa = 0;
         }

         // Initialize vars for PGA calculation & PsdHandler
         gs_PgaHandlerState = PGA_INIT;

         AddFunctionToBkgdFifo((PtrToBkgdFunc)BgVectorPowerHandler);

         TESTArray[TEST_RxSubState] = 130;
      }
      break;

   case 130: /* Average received signal over # frames */

      if (gs_PgaHandlerState == PGA_DONE)
      {
         gs_RxSubStateCnt++;
         TESTArray[TEST_RxSubState] = 13;
      }
      break;

   case MEASURE_PSD:

      // Initialize vars for PSD calculation
      gs_AlgHandlerState = PSD_INIT;
      gt_PsdConfig.s_AlgCalcType = (PSD_CALC_TYPE_INITMSRDBUFF_AND_INC);

      gs_RxSubStateCnt = 0;
      // use gsa_SnrBuf to store PSD
      gpsa_MeasuredSnrBuf = gsa_SnrBuf;

      //Note this CMV is mapped to a off chip buffer in VR9 regular mode at the beginning
      //of training. We perform the initialization here to make sure that sleep state operation
      //is not changed
      INFOMap[INFO_PsdBuf] = &gsa_SnrBuf[0];

      TESTArray[TEST_RxSubState] = 140;

      break;

   case 140:
      if (gs_AlgHandlerState != PSD_CALC_DONE)
      {
         // DMTscope - measure psd
         PsdMeasure();
      }
      else
      {

         TESTArray[TEST_RxSubState] = 141;

      }
      break;

   case 141:
      // DMTscope - calibrate measured psd
      gs_AlgHandlerState = 0;
      AddFunctionToBkgdFifo((PtrToBkgdFunc)PsdCalibrate);
      TESTArray[TEST_RxSubState] = 142;
      break;
   case 142:
      // DMTscope - when finished send notification to the host
      if (gs_AlgHandlerState == 1)
      {
         DmtScopeActionFinished();
         TESTArray[TEST_RxSubState] = -1;
      }
      break;

   case 15: // Frame Alignment

      // Initialize vars for Frame Alignment
      gs_CumulativeAlignmentOffset = 0;
      gs_AlgHandlerState = FRAMEALIGNMENT_SELECT_TONES;
      gs_RxSubStateCnt = 0;

      TESTArray[TEST_RxSubState] = 150;

      break;

   case 150:
      if (gs_AlgHandlerState != FRAMEALIGNMENT_DONE)
      {
         FrameAlignmentHandler();
      }
      else
      {
         TESTArray[TEST_RxSubState] = -1;
      }

      break;

   case FDQ_TRAINING:
      // Reverb or Medley FDQ Training

      //Initialize vars to perform FDQ training
      gs_AlgHandlerState = FDQ_TRAIN_INIT;
      gs_RxSubStateCnt = 0;

      //gs_RxNumBytesPerSymbol = 1; // For Reverb FDQ
      //gs_RxNumBytesPerSymbol = 2; // For Medley FDQ
      TESTArray[TEST_RxSubState] = 160;

      break;

   case 160:
      if (gs_AlgHandlerState != FDQ_TRAIN_DONE)
      {
         FdqHandler();
      }
      else
      {
         // DMTscope - when finished send notification to the host
         DmtScopeActionFinished();
         TESTArray[TEST_RxSubState] = -1;

      }
      break;

   case MEASURE_SNR:

      //Initialize vars for SNR calculation
      gs_AlgHandlerState = gs_AlgHandlerState = SNR_INIT;
      gs_RxSubStateCnt = 0;
      gpsa_MeasuredSnrBuf = gsa_SnrBuf;
      TESTArray[TEST_RxSubState] = 170;

      break;

   case 170:
      if (gs_AlgHandlerState != SNR_CALC_DONE)
      {
         SnrHandler();
      }
      else
      {
         // Get average snr, over all active rx subchannels, leave in gs_avg_snr.
         // Also leave the number of active subchannels in gs_num_active_rx_ch
         l_acc_snr = 0;
         gs_num_active_rx_ch = 0;
         for(i=0; i<gs_NumOfRxBands; i++)
            for (j = gsa_RxBandLeftChannel[i]; j <= gsa_RxBandRightChannel[i]; j++)
            {
               l_acc_snr += gsa_SnrBuf[j];
               gs_num_active_rx_ch++;
            }
         // gs_avg_snr is in 8.8 format
         gs_avg_snr = (int16) (l_acc_snr/gs_num_active_rx_ch);

         // DMTscope - when finished send notification to the host
         DmtScopeActionFinished();
         TESTArray[TEST_RxSubState] = -1;

      }
      break;

   case 18: // MFDQ
      gft_EnableMfdq = 1;
      gt_MfdqConfig.s_AlgHandlerState = MFDQ_TRAIN_INIT;
      TESTArray[TEST_RxSubState] = 180;
      break;

   case 180:
      if (gt_MfdqConfig.s_AlgHandlerState != MFDQ_TRAIN_DONE)
      {
         MfdqHandler();
      }
      else
      {
         TESTArray[TEST_RxSubState] = -1;
      }
      break;

   case PGA_TRAINING:
      // calc&load, and AFE (PGA) reg writes

      // Initialize vars for PGA calculation & PsdHandler
      gs_PgaHandlerState = PGA_INIT;
      gs_RxSubStateCnt = 0;
      // XDSLRTFW-3716 (Start_End)
      guc_PgaState = TRAINING_DONE;

      // Adaptation of Sleep state "Pga Training" to the real one used during TransceiverTraining,
      // i.e. with PGA and ACE switch down, plus special hybrid handling.
      {
         // Enable special hybrid handling for PGA
         gft_StartHybTrn = 1;
         // Enable Ace switch down handling for PREFI
         gl_HybTrSymcnt_start = 1;
         gs_HybTrFinishUp = 1;
         // Enable PGA switch down handling for PREFI instead of ACE,
         // if only 1dB is missing.
         gl_HybTrSymcnt_end = 1;
      }

      // use gsa_SnrBuf to store PSD
      gpsa_MeasuredSnrBuf = gsa_SnrBuf;
      TESTArray[TEST_RxSubState] = 190;

      break;

   case 190:
      if (gs_PgaHandlerState != PGA_DONE)
      {
         PGAHandler();
      }
      else
      {
         guc_PgaState = TRAINING_IN_PROGRESS;
         AddFunctionToFifo(gp_RxLoadingFunctionFifo, AFED_GetPgaGains);

         TESTArray[TEST_RxSubState] = 191;
      }

      break;

   case 191:
      if (guc_PgaState == TRAINING_DONE)
      {
         // Leave state with default config
         gft_StartHybTrn = 0;
         gl_HybTrSymcnt_start = 0;
         gs_HybTrFinishUp = 0;
         gl_HybTrSymcnt_end = 0;
         guc_PgaState = TRAINING_WAITING;

         // DMTscope - when finished send notification to the host
         DmtScopeActionFinished();
         TESTArray[TEST_RxSubState] = -1;
      }

      break;


   case CE_ADJUST_RX:  /* Rx CE Control */

      // Adjust RX CE
      AddFunctionToFifo(gp_RxLoadingFunctionFifo, InsertRxCE);

      TESTArray[TEST_RxSubState] = -1;
      break;

   case FRAME_ALIGN_RX:   /* Rx Algignment Control */

      gs_RxFrameAlignRemain = 0;

      gs_AlignmentOffset = TESTArray[TEST_AlignmentOffset];

      // Adjust Rx Alignment
      AddFunctionToFifo(gp_RxLoadingFunctionFifo, AdjustAlignment);

      TESTArray[TEST_RxSubState] = 210;
      break;

   case 210:   /* Rx Algignment Control */

      //For the 6.2 HW, it is not allowed to make a strymon frame less than FFT size
      //so it may take more than one frame to do frame adjustment if gs_AlignmentOffset is greater than
      //CE Length. Note the variable gs_RxFrameAlignRemain stores the remaining adjustment
      //after each call to AdjustAlignment()
      if(gs_RxFrameAlignRemain != 0)
      {
         AddFunctionToFifo(gp_RxLoadingFunctionFifo, AdjustAlignment);
         break;
      }

      // Adjust Rx Alignment
      AddFunctionToFifo(gp_RxLoadingFunctionFifo, ResetRxAlign);

      TESTArray[TEST_RxSubState] = -1;

      break;

   case ACE_MODE_SET: // ACE mode set
      // For sleep state, the choosing(forcing of ACE ) to be done here
      {
         int16 s_PrefiAceMode;

         s_PrefiAceMode = (TESTArray[TEST_DS_ACE] & ACE_MODE_MASK);
         if (s_PrefiAceMode != ACE_MODE_MASK)
         {
            gs_PrefiAceMode_Selected = s_PrefiAceMode;
         }
      }
      TESTArray[TEST_RxSubState] = -1;

      break;

   case HYBRID_SET:
      // Force a specific hybrid index
      // Note: For sleep state the fct.'s InitModem_PostHandshake() and ConfigTaskLayerForPostGhs()
      //       are called. Here the variables gt_ProfileAct.us_ProfileSelected, TESTArray[TEST_MaxNumHybrSettings] and
      //       gusa_HybridsForProfileSelected are set.
      gs_MaxNumHybrSettings = TESTArray[TEST_MaxNumHybrSettings];

      if(gs_HybTrIndexDbg >= 0)
      {
         // Stay in range of the selected profile hybrid table, i.e. if the
         // configuration is to big always the last entry of the table will be used!
         if (gs_HybTrIndexDbg >= gs_MaxNumHybrSettings)
         {
            gs_HybTrIndexDbg = (gs_MaxNumHybrSettings - 1);
         }

         gs_HybTrIndex = gs_HybTrIndexDbg;
      }

      gs_HybTrFinishUp = 0;
      guc_PgaState = TRAINING_IN_PROGRESS;
      AddFunctionToFifo(gp_RxLoadingFunctionFifo, AFED_SetHybrid);

      gs_RxSubStateCnt = 0;
      TESTArray[TEST_RxSubState] = 280;
      break;

   case 280:
      if (guc_PgaState == TRAINING_DONE)
      {
         guc_PgaState = TRAINING_WAITING;
         gs_HybTrIndex = 0;
         TESTArray[TEST_RxSubState] = -1;
      }

      break;


   case 29: // PLL Test. CPE side
      // Enable GetRxTones in RxProcessTones fcn
      gft_EnableGetRxTones = TRUE;

      // Set RTV selection to FFT output
      gs_RtvSelect = FFT_OUTPUT;

      if (TESTArray[TEST_SrcFRNominalOffset])    // Init SRC in order to use PLL
      {

         AddFunctionToFifo(gp_RxLoadingFunctionFifo, EnableSRCFilter);
         gl_pll_freq_offset = (int32)(-TESTArray[TEST_SrcFRNominalOffset]) << 14;
      }

      // XDSLRTFW-3280 - Start - PLL improvement / pilot tone selection improvement
      gt_PilotConfig.ta_PilotTones[gt_PilotConfig.te_UsedPTArrayIdx].s_PilotToneIdx = TESTArray[TEST_RxTestPilotTone];
      AddFunctionToFifo(gp_RxLoadingFunctionFifo, SetRxPilotToneIndex);
#ifdef DEBUG_TRAIL
      // Turn off statestrail logging
      // Turn on PLL phase logging
      gt_debugTrailControl.s_logType[0] = 0;
      gt_debugTrailControl.s_logType[1] = 0;
#endif // DEBUG_TRAIL


      gs_RxSubStateCnt = 0;
      TESTArray[TEST_RxSubState] = 290;
      break;

   case 290: // PLL Test. CPE side

      gs_RxSubStateCnt++;
      if (gs_RxSubStateCnt == 2)
      {
         // Configure PLL
         gft_EnableNoiseShaping = 1;
         gft_UpdateTimingRecoveryHWEnable = 1;
         gft_EnablePLL  = 1; // Turn on PLL

         gt_PilotConfig.ta_PilotTones[gt_PilotConfig.te_UsedPTArrayIdx].s_PllRefTone_Re =  0x7FFF;
         gt_PilotConfig.ta_PilotTones[gt_PilotConfig.te_UsedPTArrayIdx].s_PllRefTone_Im =  0x7FFF;

         // Set PLL loop filter parameters
         ResetPLL(gt_PilotConfig.te_UsedPTArrayIdx, (int16)gs_Kp_Fast, (int16)gs_Ki_Fast, (int16)PLL_HALF_PI_RADIANS);
         // XDSLRTFW-3280 - End - PLL improvement / pilot tone selection improvement
         TESTArray[TEST_RxSubState] = -1;
      }

      break;

      //Enable loopback at the DEC sum junction by setting the DEC filter to unity
      //Note this is only work in ADSL mode

#ifdef BIT_PATTERN_TEST
#define BIT_PATTERN_ERROR_FAIL_TO_LOAD_RAMP (1)
#define   BIT_PATTERN_ERROR_DELAY_NOT_FOUND  (2)
#define BIT_PATTERN_ERROR_FAIL_TO_LOAD_PULSE (3)

   case 401://configure loopback mode (serdes or 3glp)
      AddFunctionToFifo(gp_RxLoadingFunctionFifo, EnableSerDesLoopback);

      TESTArray[TEST_RxSubState] = -1;
      break;


   case 402: //to obtain the rx time-domainoutput
      gft_EnableGetRxTime = 1;
      gft_TriggerGetRxTime = 1;

      TESTArray[TEST_RxSubState] = 403;
      break;

   case 403://diable collection of rx time
      gft_EnableGetRxTime = 0;

      TESTArray[TEST_RxSubState] = -1;
      break;

   case 404://determine the delay between tx input and rx output
      if (gus_BitPatternIndex == 1)
      {
         guc_FindPathDelayState = TRAINING_IN_PROGRESS;
         AddFunctionToBkgdFifo((PtrToBkgdFunc)BgFindPathDelayUseRamp);
      }
      else
      {
         gus_BitPatternError = BIT_PATTERN_ERROR_FAIL_TO_LOAD_RAMP;
      }
      TESTArray[TEST_RxSubState] = -1;
      break;

   case 405: //comparison
      if (guc_FindPathDelayState == TRAINING_DONE)
      {
         guc_FindSampleDiffState = TRAINING_IN_PROGRESS;
         AddFunctionToBkgdFifo((PtrToBkgdFunc)BgFindSampleDiff);
      }
      else
      {
         gus_BitPatternError = BIT_PATTERN_ERROR_DELAY_NOT_FOUND;
      }
      TESTArray[TEST_RxSubState] = -1;
      break;

   case 406:
      //Disable PRBS rotation
      AddFunctionToFifo(gp_RxLoadingFunctionFifo,DisableRxPRBS);
      TESTArray[TEST_RxSubState] = -1;
      break;

   case 407:
      if (gs_CompareLog_Enable == 1)
      {
         FindSampleDiff();
      }
      else
      {
         TESTArray[TEST_RxSubState] = -1;
      }
      break;


   case 408: //find the delay between tx input and rx output using the single pulse
      //Note: for this test, we have to set gus_BitPatternIndex to 4
      if(gus_BitPatternIndex == 4)
      {
         // queue loading function to measure Tx/Rx timer, which
         // are returned in gl_TxTimerVal and gl_RxTimerVal
         AddFunctionToFifo(gp_TxLoadingFunctionFifo, GetTxRxCounters);

         guc_FindPathDelayState = TRAINING_IN_PROGRESS;
         AddFunctionToBkgdFifo((PtrToBkgdFunc)BgCalcEchoPathDelay);
      }
      else
      {
         gus_BitPatternError = BIT_PATTERN_ERROR_FAIL_TO_LOAD_PULSE;
      }

      TESTArray[TEST_RxSubState] = -1;

      break;

#endif //BIT_PATTERN_TEST
   }
}


