/** @file
  Platform Flash Access library to update different system firmware components

@copyright
  INTEL CONFIDENTIAL
  Copyright 2017 - 2019 Intel Corporation.

  The source code contained or described herein and all documents related to the
  source code ("Material") are owned by Intel Corporation or its suppliers or
  licensors. Title to the Material remains with Intel Corporation or its suppliers
  and licensors. The Material may contain trade secrets and proprietary and
  confidential information of Intel Corporation and its suppliers and licensors,
  and is protected by worldwide copyright and trade secret laws and treaty
  provisions. No part of the Material may be used, copied, reproduced, modified,
  published, uploaded, posted, transmitted, distributed, or disclosed in any way
  without Intel's prior express written permission.

  No license under any patent, copyright, trade secret or other intellectual
  property right is granted to or conferred upon you by disclosure or delivery
  of the Materials, either expressly, by implication, inducement, estoppel or
  otherwise. Any license under such intellectual property rights must be
  express and approved by Intel in writing.

  Unless otherwise agreed by Intel in writing, you may not remove or alter
  this notice or any other notice embedded in Materials by Intel or
  Intel's suppliers or licensors in any way.

  This file contains a 'Sample Driver' and is licensed as such under the terms
  of your license agreement with Intel or your vendor. This file may be modified
  by the user, subject to the additional terms of the license agreement.

@par Specification Reference:

**/

#include <PiDxe.h>

#include <Library/BaseLib.h>
#include <Library/BaseMemoryLib.h>
#include <Library/DebugLib.h>
#include <Library/PlatformFlashAccessLib.h>
#include <Library/MemoryAllocationLib.h>
#include <Library/UefiBootServicesTableLib.h>
#include <Library/UefiLib.h>
#include <Library/PciSegmentLib.h>
#include <Library/IoLib.h>
#include <Library/FWUpdateLib.h>
#include <Library/DxeMeLib.h>
#include <Library/TimerLib.h>
#include <Library/BiosUpdateHelpersLib.h>

#include <Protocol/Spi.h>
#include <Library/HobLib.h>

#include <BiosGuard.h>
#include <Register/HeciRegs.h>

//
// Used in SPI update function
//
#define UPDATE_BLOCK_SIZE                      SIZE_4KB
//
// Used in BiosGuard update function
//
// BGUPC header 8 + PKCS1 1.5 Certificate 516 (256+4+256) = 524 bytes
#define BIOSGUARD_CERTIFICATE_SIZE             524

//
// Used in ME update function
//
#define FW_UPDATE_DISABLED                     0
#define FW_UPDATE_ENABLED                      1
#define ME_FWSTS2_TIMEOUT_COUNTER              150


PCH_SPI_PROTOCOL                               *mSpiProtocol = NULL;
//
// Use to display progress of sending ME FW
//
EFI_FIRMWARE_MANAGEMENT_UPDATE_IMAGE_PROGRESS  mMeDisplayProgress = NULL;
UINTN                                          mMeSendProgressStartPercentage = 0;
UINTN                                          mMeSendProgressEndPercentage = 100;

/**
  Initialize PCH SpiProtocol interface module pointer

  @param EFI_SUCCESS      PCH SpiProtocol interface module pointer is initialized successfully
  @param Others           Fail to locate PCH SpiProtocol interface

**/
EFI_STATUS
InitializeSpiProtocolInterface (
  VOID
  )
{
  EFI_STATUS                   Status;

  Status = EFI_SUCCESS;

  if (mSpiProtocol == NULL) {
    Status = gBS->LocateProtocol (
                    &gPchSpiProtocolGuid,
                    NULL,
                    (VOID **)&mSpiProtocol
                    );
    ASSERT_EFI_ERROR (Status);
  }

  return Status;
}

/**
  Update a block (with UPDATE_BLOCK_SIZE) on SPI flash.

  This function updates one block to FlashRegionType region with give Address in below steps:
  1. Read one block from the destination and compare it with Buffer.
     If the content is identical, no further action is needed. Return EFI_SUCCESS directly.
  2. Erase one block from the destination.
  3. Write one block to the destination with Buffer.
  4. Read the block back from destination and compare it to ensure the write operation is successful.

  @param[in]  FlashRegionType          The flash region to be updated.
  @param[in]  Address                  Starting address within the FlashRegionType region to be updated.
  @param[in]  Buffer                   A pointer to a buffer contains the update data.

  @retval     EFI_SUCCESS.             Operation is successful.
  @retval     EFI_OUT_OF_RESOURCES     Failed to allocate needed memory buffer.
  @retval     EFI_VOLUME_CORRUPTED     The block is not updated as expected.
  @retval     Others                   If there is any device errors.

**/
EFI_STATUS
EFIAPI
SpiFlashUpdateBlock (
  IN FLASH_REGION_TYPE                 FlashRegionType,
  IN UINT32                            FlashAddress,
  IN UINT8                             *Buffer
  )
{
  EFI_STATUS                           Status;
  UINT32                               NumBytes;
  UINT8                                *CompareBuffer;

  Status = InitializeSpiProtocolInterface ();
  if (EFI_ERROR (Status)) {
    return EFI_NOT_READY;
  }

  NumBytes = UPDATE_BLOCK_SIZE;
  CompareBuffer = NULL;

  CompareBuffer = AllocateZeroPool (NumBytes);
  if (CompareBuffer == NULL) {
    Status = EFI_OUT_OF_RESOURCES;
    goto Done;
  }

  //
  // Compare Buffer block with the destination
  //
  Status = mSpiProtocol->FlashRead (
                           mSpiProtocol,
                           FlashRegionType,
                           FlashAddress,
                           NumBytes,
                           CompareBuffer
                           );

  if (EFI_ERROR (Status)) {
    DEBUG ((DEBUG_ERROR, "SpiFlashUpdateBlock: FlashRead failed at Region %d, Address 0x%x\n", FlashRegionType, FlashAddress));
    goto Done;
  }

  if (CompareMem (CompareBuffer, Buffer, NumBytes) == 0) {
    //
    // No need to update this block
    //
    Status = EFI_SUCCESS;
    goto Done;
  }

  //
  // Erase the block
  //
  Status = mSpiProtocol->FlashErase (
                           mSpiProtocol,
                           FlashRegionType,
                           FlashAddress,
                           NumBytes
                           );

  if (EFI_ERROR (Status)) {
    DEBUG ((DEBUG_ERROR, "SpiFlashUpdateBlock: FlashErase failed at Region %d, Address 0x%x\n", FlashRegionType, FlashAddress));
    goto Done;
  }

  //
  // Wrtie the block
  //
  Status = mSpiProtocol->FlashWrite (
                           mSpiProtocol,
                           FlashRegionType,
                           FlashAddress,
                           NumBytes,
                           Buffer
                           );

  if (EFI_ERROR (Status)) {
    DEBUG ((DEBUG_ERROR, "SpiFlashUpdateBlock: FlashErase failed at Region %d, Address 0x%x\n", FlashRegionType, FlashAddress));
    goto Done;
  }

  //
  // Read buffer back to verify udpate status
  //
  ZeroMem (CompareBuffer, NumBytes);
  Status = mSpiProtocol->FlashRead (
                           mSpiProtocol,
                           FlashRegionType,
                           FlashAddress,
                           NumBytes,
                           CompareBuffer
                           );

  if (CompareMem (CompareBuffer, Buffer, NumBytes) != 0) {
    //
    // Block is not updated as expected.
    //
    Status = EFI_VOLUME_CORRUPTED;
    DEBUG ((DEBUG_ERROR, "SpiFlashUpdateBlock: Update failed at Region %d, Address 0x%x\n", FlashRegionType, FlashAddress));
  }

Done:
  if (CompareBuffer !=  NULL) {
    FreePool (CompareBuffer);
  }

  return Status;
}

/**
  Read from SPI flash.

  This function updates one block to FlashRegionType region with give Address in below steps:
  1. Read one block from the destination and compare it with Buffer.
     If the content is identical, no further action is needed. Return EFI_SUCCESS directly.
  2. Erase one block from the destination.
  3. Write one block to the destination with Buffer.
  4. Read the block back from destination and compare it to ensure the write operation is successful.

  @param[in]  FlashRegionType          The flash region to be updated.
  @param[in]  Address                  Starting address within the FlashRegionType region to be updated.
  @param[in]  Buffer                   A pointer to a buffer contains the update data.

  @retval     EFI_SUCCESS.             Operation is successful.
  @retval     EFI_OUT_OF_RESOURCES     Failed to allocate needed memory buffer.
  @retval     EFI_VOLUME_CORRUPTED     The block is not updated as expected.
  @retval     Others                   If there is any device errors.

**/
EFI_STATUS
EFIAPI
SpiFlashRead (
  IN FLASH_REGION_TYPE                 FlashRegionType,
  IN UINT32                            FlashAddress,
  IN UINTN                             NumBytes,
  IN UINT8                             *Buffer
  )
{
  EFI_STATUS  Status;

  Status = InitializeSpiProtocolInterface ();
  if (EFI_ERROR (Status)) {
    return EFI_NOT_READY;
  }

  //
  // Compare Buffer block with the destination
  //
  Status = mSpiProtocol->FlashRead (
                           mSpiProtocol,
                           FlashRegionType,
                           FlashAddress,
                           (UINT32)NumBytes,
                           Buffer
                           );
  return Status;
}

/**
  Update a buffer to SPI flash.
  The size of the buffer must be aligned to UPDATE_BLOCK_SIZE.

  @param[in]  FlashRegionType          The flash region to be updated.
  @param[in]  Address                  Starting address within the FlashRegionType region to be updated.
                                       It must be 4K aligned.
  @param[in]  Buffer                   A pointer to a buffer contains the update data.
  @param[in]  Length                   Number of bytes in Buffer.
                                       It must be aligned to UPDATE_BLOCK_SIZE.
  @param[in]  Progress                 A function used report the progress of the
                                       firmware update.  This is an optional parameter
                                       that may be NULL.
  @param[in]  StartPercentage          The start completion percentage value that may
                                       be used to report progress during the flash
                                       write operation.
  @param[in]  EndPercentage            The end completion percentage value that may
                                       be used to report progress during the flash
                                       write operation.

  @retval     EFI_SUCCESS              Operation is successful.
  @retval     EFI_INVALID_PARAMETER    Buffer is NULL or FlashAddress/Length is not well aligned.
  @retval     EFI_OUT_OF_RESOURCES     Failed to allocate needed memory buffer.
  @retval     EFI_VOLUME_CORRUPTED     SPI flash is not updated as expected.
  @retval     Others                   If there is any device errors.

**/
EFI_STATUS
EFIAPI
SpiFlashUpdate (
  IN FLASH_REGION_TYPE                              FlashRegionType,
  IN UINT32                                         FlashAddress,
  IN UINT8                                          *Buffer,
  IN UINT32                                         Length,
  IN EFI_FIRMWARE_MANAGEMENT_UPDATE_IMAGE_PROGRESS  Progress,        OPTIONAL
  IN UINTN                                          StartPercentage,
  IN UINTN                                          EndPercentage
  )
{
  EFI_STATUS                           Status;
  UINTN                                Index;
  EFI_PHYSICAL_ADDRESS                 Address;
  UINTN                                CountOfBlocks;
  UINT8                                *Buf;

  DEBUG ((DEBUG_INFO, "SpiFlashUpdate - Region  %d\n", (UINTN)FlashRegionType));
  DEBUG ((DEBUG_INFO, "SpiFlashUpdate - Address 0x%x\n", FlashAddress));
  DEBUG ((DEBUG_INFO, "SpiFlashUpdate - Length  0x%x(%d)\n", Length));

  if ((Buffer == NULL) || ((Length % UPDATE_BLOCK_SIZE) != 0) || ((FlashAddress % SIZE_4KB) != 0)) {
    ASSERT (FALSE);
    return EFI_INVALID_PARAMETER;
  }

  Status        = EFI_SUCCESS;
  Index         = 0;
  CountOfBlocks = (UINTN) (Length / UPDATE_BLOCK_SIZE);
  Address       = FlashAddress;
  Buf           = Buffer;

  for (Index = 0; Index < CountOfBlocks; Index++) {
    if (Progress != NULL) {
      Progress (StartPercentage + ((Index * (EndPercentage - StartPercentage)) / CountOfBlocks));
    }

    Status = SpiFlashUpdateBlock (FlashRegionType, (UINT32) Address, Buf);
    if (EFI_ERROR (Status)) {
      DEBUG ((DEBUG_ERROR, "SpiFlashUpdate failed (%r) at Region %d - 0x%x\n", Status, FlashRegionType, Address));
      break;
    }

    Address += UPDATE_BLOCK_SIZE;
    Buf += UPDATE_BLOCK_SIZE;
  }

  if (Progress != NULL) {
    Progress (EndPercentage);
  }

  return Status;
}

/**
  Perform EC firmware update.

  @param[in] Offset                  Offset relative to EC region to be programmed.
  @param[in] EcBinPtr                Pointer to EC FW binary.
  @param[in] EcBinSize               Size of EC FW binary.
  @param[in] Progress                A function used report the progress of the
                                     firmware update.  This is an optional parameter
                                     that may be NULL.
  @param[in] StartPercentage         The start completion percentage value that may
                                     be used to report progress during the flash
                                     write operation.
  @param[in] EndPercentage           The end completion percentage value that may
                                     be used to report progress during the flash
                                     write operation.

  @retval EFI_SUCCESS                EC FW is updated successfully.
  @retval Others                     The update operation fails.

**/
EFI_STATUS
UpdateEcFirmware (
  IN UINT32                                         Offset,
  IN UINT8                                          *EcBinPtr,
  IN UINTN                                          EcBinSize,
  IN EFI_FIRMWARE_MANAGEMENT_UPDATE_IMAGE_PROGRESS  Progress,        OPTIONAL
  IN UINTN                                          StartPercentage,
  IN UINTN                                          EndPercentage
  )
{
  DEBUG ((DEBUG_INFO, "UpdateEcFirmware: Direct SPI.\n"));

  return SpiFlashUpdate (
           FlashRegionEC,
           Offset,
           EcBinPtr,
           (UINT32) EcBinSize,
           Progress,
           StartPercentage,
           EndPercentage
           );

}

/**
  Display the ME update progress.

  @param BytesSent               Bytes already sent to ME
  @param BytestobeSent           Bytes to be sent to ME

**/
VOID DisplaySendMeFwStatus (
  UINT32                                BytesSent,
  UINT32                                BytestobeSent
  )
{
  UINTN  Percentage;

  Percentage = (BytesSent * (mMeSendProgressEndPercentage - mMeSendProgressStartPercentage)) / BytestobeSent;

  if (mMeDisplayProgress != NULL) {
    mMeDisplayProgress (Percentage);
  }
}

/**
  Update ME Firmware by HECI interface.
  Use the ME FW Update API (FWUpdateLib.lib) provided by ME team to perform the update.

  @param[in] ImageBuffer             Pointer to ME FWU FwImage.
  @param[in] ImageLength             Size of FwImage.
  @param[in] Progress                A function used report the progress of the
                                     firmware update.  This is an optional parameter
                                     that may be NULL.
  @param[in] StartPercentage         The start completion percentage value that may
                                     be used to report progress during the flash
                                     write operation.
  @param[in] EndPercentage           The end completion percentage value that may
                                     be used to report progress during the flash
                                     write operation.

  @retval EFI_SUCCESS                ME FW is updated successfully.
  @retval Others                     The update operation fails.

**/
EFI_STATUS
UpdateMeByHeci (
  IN UINT8                                          *ImageBuffer,
  IN UINTN                                          ImageLength,
  IN EFI_FIRMWARE_MANAGEMENT_UPDATE_IMAGE_PROGRESS  Progress,        OPTIONAL
  IN UINTN                                          StartPercentage,
  IN UINTN                                          EndPercentage
  )
{
  EFI_STATUS                           Status;
  UINT32                               FwUpdateApiStatus;
  UINT16                               EnabledState;
  BOOLEAN                              InProgress;
  UINT32                               CurrentPercent;
  UINT32                               FwUpdateStatus;
  UINT32                               NeededResetType;
  UINT32                               UpdateTimer;
  UINT32                               PreviousPercent;

  DEBUG ((DEBUG_INFO, "UpdateMeByHeci: ME Image Length %d(%xh)\n", ImageLength, ImageLength));

  EnabledState    = 0;
  InProgress      = FALSE;
  CurrentPercent  = 0;
  FwUpdateStatus  = 0;
  NeededResetType = 0;
  UpdateTimer     = 0;
  PreviousPercent = 0;

  //
  // Check FWU enabled state
  //
  FwUpdateApiStatus = FwuEnabledState (&EnabledState);
  if (FwUpdateApiStatus != EFI_SUCCESS) {
    DEBUG ((DEBUG_ERROR, "UpdateMeByHeci: FwuEnabledState failed: %r.\n", FwUpdateApiStatus));
    return EFI_DEVICE_ERROR;
  }

  DEBUG ((DEBUG_INFO, "UpdateMeByHeci: FwuEnabledState is 0x%x.\n", EnabledState));

  switch (EnabledState) {
    case FW_UPDATE_DISABLED:
      DEBUG ((DEBUG_ERROR, "UpdateMeByHeci: FWUpdate is disabled.\n"));
      return EFI_DEVICE_ERROR;

    case FW_UPDATE_ENABLED:
      break;

    default:
      break;
  }

  if (Progress != NULL) {
    mMeDisplayProgress = Progress;
    mMeSendProgressStartPercentage = StartPercentage;
    //
    // Assign 1/5 of progress bar to SendProgress
    //
    mMeSendProgressEndPercentage = StartPercentage + ((EndPercentage - StartPercentage) / 5);
  }

  //
  // Starting FWU full update. Send image to FW Update Client
  //
  DEBUG ((DEBUG_INFO, "UpdateMeByHeci: Executing Full FW Update.\n"));
  FwUpdateApiStatus = FwuFullUpdateFromBuffer (ImageBuffer, (UINT32) ImageLength, NULL, &DisplaySendMeFwStatus);
  if (FwUpdateApiStatus != EFI_SUCCESS) {
    DEBUG ((DEBUG_ERROR, "UpdateMeByHeci: FwuFullUpdateFromBuffer failed: %r.\n", FwUpdateApiStatus));
    return EFI_DEVICE_ERROR;
  }

  //
  // Image was sent to FW Update client
  // Poll the FW Update progress until finished
  //
  DEBUG ((DEBUG_INFO, "UpdateMeByHeci: Waiting for FW Update progress to be finished.\n"));
  do {
    FwUpdateApiStatus = FwuCheckUpdateProgress (
                          &InProgress,
                          &CurrentPercent,
                          &FwUpdateStatus,
                          &NeededResetType
                          );
    if (FwUpdateApiStatus != EFI_SUCCESS) {
      DEBUG ((DEBUG_ERROR, "UpdateMeByHeci: FwuCheckUpdateProgress failed: %r.\n", FwUpdateApiStatus));
      break;
    }

    if (!InProgress) {
      DEBUG ((DEBUG_INFO, "UpdateMeByHeci: FWU Update finished successfully: %r.\n", FwUpdateApiStatus));
      if (Progress != NULL) {
        Progress (EndPercentage);
      }
      break;
    }

    // Update is in progress
    DEBUG ((DEBUG_INFO, "UpdateMeByHeci: Current percent: %d\n", CurrentPercent));
    gBS->Stall (250000); // wait 250 milliseconds before polling again

    // If 30 seconds passed
    if (UpdateTimer >= 30000) {
      // If percent didn't change in 30 seconds
      if (CurrentPercent == PreviousPercent) {
        DEBUG ((DEBUG_ERROR, "UpdateMeByHeci: FwuCheckUpdateProgress timeout.\n"));
        Status = EFI_TIMEOUT;
        break;
      }
      // Percent changed
      PreviousPercent = CurrentPercent;
      UpdateTimer = 0;
    } else {
      // Increase timer
      UpdateTimer += 250;
    }

    if (Progress != NULL) {
      Progress (mMeSendProgressEndPercentage + (CurrentPercent * (EndPercentage - mMeSendProgressEndPercentage)) / 100);
    }
  } while (TRUE);


  DEBUG ((DEBUG_INFO, "UpdateMeByHeci: FwUpdateApiStatus: 0x%x (%d).\n", FwUpdateApiStatus, FwUpdateApiStatus));
  DEBUG ((DEBUG_INFO, "UpdateMeByHeci: FwUpdateStatus   : 0x%x (%d).\n", FwUpdateStatus, FwUpdateStatus));
  DEBUG ((DEBUG_INFO, "UpdateMeByHeci: NeededResetType  : 0x%x.\n", NeededResetType));
  if ((FwUpdateApiStatus != EFI_SUCCESS) || (FwUpdateStatus != EFI_SUCCESS)) {
    Status = EFI_DEVICE_ERROR;
  } else {
    Status = EFI_SUCCESS;
  }

  return Status;
}

/**
  Perform ME firmware update.

  @param[in] MeBinPtr                Pointer to ME FW binary.
  @param[in] MeBinSize               Size of ME FW binary.
  @param[in] Progress                A function used report the progress of the
                                     firmware update.  This is an optional parameter
                                     that may be NULL.
  @param[in] StartPercentage         The start completion percentage value that may
                                     be used to report progress during the flash
                                     write operation.
  @param[in] EndPercentage           The end completion percentage value that may
                                     be used to report progress during the flash
                                     write operation.

  @retval EFI_SUCCESS                ME FW is updated successfully.
  @retval Others                     The update operation fails.

**/
EFI_STATUS
UpdateMeFirmware (
  IN UINT8                                          *MeBinPtr,
  IN UINTN                                          MeBinSize,
  IN EFI_FIRMWARE_MANAGEMENT_UPDATE_IMAGE_PROGRESS  Progress,        OPTIONAL
  IN UINTN                                          StartPercentage,
  IN UINTN                                          EndPercentage
  )
{
  EFI_STATUS                      Status;

  DEBUG ((DEBUG_INFO, "UpdateMeFirmware: MeBinSize = %x\n", MeBinSize));

  DEBUG ((DEBUG_INFO, "UpdateMeFirmware: Update ME firmware with HECI interface\n"));
  //
  // Use the HECI method to perform Me update.
  //
  Status = UpdateMeByHeci (
             MeBinPtr,
             MeBinSize,
             Progress,
             StartPercentage,
             EndPercentage
             );

  return Status;
}

/**
  Perform ISH PDT configuration update

  @param[in] PdtBinPtr               Pointer to PDT binary.
  @param[in] PdtBinSize              Size of PDT binary.
  @param[in] Progress                A function used report the progress of the
                                     firmware update.  This is an optional parameter
                                     that may be NULL.
  @param[in] StartPercentage         The start completion percentage value that may
                                     be used to report progress during the flash
                                     write operation.
  @param[in] EndPercentage           The end completion percentage value that may
                                     be used to report progress during the flash
                                     write operation.

  @retval EFI_SUCCESS                PDT is updated successfully.
  @retval Others                     The update operation fails.

**/
EFI_STATUS
UpdateIshPdt (
  IN UINT8                                          *PdtBinPtr,
  IN UINTN                                          PdtBinSize,
  IN EFI_FIRMWARE_MANAGEMENT_UPDATE_IMAGE_PROGRESS  Progress,        OPTIONAL
  IN UINTN                                          StartPercentage,
  IN UINTN                                          EndPercentage
  )
{
  EFI_STATUS              Status;
  UINT8                   *PaddedPayload;
  UINTN                   PaddedPayloadSize;
  UINT32                  FwUpdateApiStatus;

  DEBUG ((DEBUG_INFO, "UpdateIshPdt: PdtBinSize = 0x%x\n", PdtBinSize));
  ASSERT (PdtBinSize <= SIZE_4KB);

  PaddedPayload = NULL;

  //
  // Payload size must be word aligned and padded with zero byte if size is not even.
  //
  PaddedPayloadSize = ALIGN_VALUE (PdtBinSize, 2);
  DEBUG ((DEBUG_INFO, "UpdateIshPdt: Padded PdtBinSize = 0x%x\n", PaddedPayloadSize));

  PaddedPayload = AllocateZeroPool (PaddedPayloadSize);
  if (PaddedPayload == NULL) {
    DEBUG ((DEBUG_ERROR, "UpdateIshPdt: Ran out of memory resource.\n"));
    return EFI_OUT_OF_RESOURCES;
  }

  CopyMem (PaddedPayload, PdtBinPtr, PdtBinSize);

  if (Progress != NULL) {
    Progress (StartPercentage);
  }

  FwUpdateApiStatus = FwuSetIshConfig (PaddedPayload, (UINT32) PaddedPayloadSize);
  DEBUG ((DEBUG_INFO, "UpdateIshPdt FwuStatus = 0x%x\n", FwUpdateApiStatus));

  if (FwUpdateApiStatus != EFI_SUCCESS) {
    Status = EFI_DEVICE_ERROR;
  } else {
    Status = EFI_SUCCESS;
  }

  if (Progress != NULL) {
    Progress (EndPercentage);
  }

  FreePool (PaddedPayload);

  return Status;
}
/**
  Update BIOS via BiosGuard service.

  @param[in]  SystemFirmwareImage     Points to the System Firmware (BIOS) image.
  @param[in]  SystemFirmwareImageSize The length of the System Firmware image in bytes.
  @param[out] BgupImage               Points to BgupImage which would be sent to BiosGaurd ACM along with SystemFirmwareImage.
                                      BgupImage contains BGSL to update BIOS and certificate for authentication.
  @param[out] BgupImageSize           The length of the BgupImage in bytes.

  @retval EFI_SUCCESS                 The System Firmware image is updated successfully.
  @retval EFI_SECURITY_VIOLATION      The update operation fails due to SVN check error.
  @retval EFI_DEVICE_ERROR            The update operation fails.
  @retval EFI_INVALID_PARAMETER       BgupImage is NULL or BgupImageSize is unexpected.
  @retval Others                      The update operation fails.

**/
EFI_STATUS
PerformBiosGuardUpdate (
  IN VOID                         *SystemFirmwareImage,
  IN UINTN                        SystemFirmwareImageSize,
  IN VOID                         *BgupImage,
  IN UINTN                        BgupImageSize
  )
{
  EFI_STATUS                           Status;
  BIOSGUARD_HOB                        *BiosGuardHobPtr;
  EFI_PHYSICAL_ADDRESS                 BiosGuardMemAddress;
  UINT32                               BiosGuardMemSize;
  EFI_PHYSICAL_ADDRESS                 BgupCertificate;
  EFI_PHYSICAL_ADDRESS                 BgIoTrapAddress;
  UINT8                                *BiosGuardMemAddressPtr;
  UINT8                                *BgupCertificatePtr;
  UINT64                               BiosGuardStatus;
  UINT8                                *BgUpFileBuffer;
  UINTN                                BgUpFileBufferSize;

  ASSERT (SystemFirmwareImage != NULL);
  ASSERT (BgupImage != NULL);
  ASSERT (BgupImageSize != 0);

  if ((BgupImage == NULL) || (BgupImageSize == 0) || (BgupImageSize < (sizeof (BGUP_HEADER) + BIOSGUARD_CERTIFICATE_SIZE))) {
    DEBUG ((DEBUG_ERROR, "PerformBiosGuardUpdate: BIOS Bgup file error! Abort BIOS Firmware Update\n"));
    return EFI_INVALID_PARAMETER;
  }

  BiosGuardHobPtr          = NULL;
  BiosGuardMemAddressPtr   = NULL;
  BgupCertificatePtr       = NULL;
  BiosGuardMemAddress      = 0;
  BiosGuardMemSize         = 0;
  BgIoTrapAddress          = 0;
  Status                   = EFI_NOT_STARTED;
  BgUpFileBuffer           = NULL;
  BgUpFileBufferSize       = 0;

  //
  // Get BIOS Guard Hob
  //
  BiosGuardHobPtr = GetFirstGuidHob (&gBiosGuardHobGuid);
  if (BiosGuardHobPtr == NULL) {
    DEBUG ((DEBUG_ERROR, "BIOS Guard HOB not available\n"));
    return EFI_DEVICE_ERROR;
  }

  BiosGuardMemAddress    = (EFI_PHYSICAL_ADDRESS) BiosGuardHobPtr->BiosGuardMemAddress;
  BiosGuardMemSize       = (UINT32) LShiftU64 (BiosGuardHobPtr->BiosGuardMemSize, 20);
  BgIoTrapAddress        = (EFI_PHYSICAL_ADDRESS) BiosGuardHobPtr->BiosGuardIoTrapAddress;

  BgupCertificate        = (EFI_PHYSICAL_ADDRESS) (BiosGuardMemAddress + BiosGuardMemSize - BGUPC_MEMORY_OFFSET);
  BiosGuardMemAddressPtr = (UINT8 *) BiosGuardMemAddress;
  BgupCertificatePtr     = (UINT8 *) BgupCertificate;

  DEBUG ((DEBUG_INFO, "PerformBiosGuardUpdate: BIOS Guard SPI Flow\n"));
  BgUpFileBuffer = BgupImage;
  BgUpFileBufferSize = BgupImageSize;

  DEBUG ((DEBUG_INFO, "PerformBiosGuardUpdate: BgUpFileBufferSize        = %d\n", BgUpFileBufferSize));
  DEBUG ((DEBUG_INFO, "PerformBiosGuardUpdate: (BgUpFileBufferSize - %d) = %d\n", BIOSGUARD_CERTIFICATE_SIZE, (BgUpFileBufferSize - BIOSGUARD_CERTIFICATE_SIZE)));
  DEBUG ((DEBUG_INFO, "PerformBiosGuardUpdate: BiosBinSize               = 0x%08X\n", SystemFirmwareImageSize));

  CopyMem (BiosGuardMemAddressPtr, BgUpFileBuffer, (BgUpFileBufferSize - BIOSGUARD_CERTIFICATE_SIZE));
  CopyMem ((BiosGuardMemAddressPtr + (BgUpFileBufferSize - BIOSGUARD_CERTIFICATE_SIZE)), SystemFirmwareImage, SystemFirmwareImageSize);
  CopyMem (BgupCertificatePtr, (BgUpFileBuffer + (BgUpFileBufferSize - BIOSGUARD_CERTIFICATE_SIZE)), BIOSGUARD_CERTIFICATE_SIZE);
  IoRead8 (BgIoTrapAddress);
  CopyMem (&BiosGuardStatus, BgupCertificatePtr, sizeof (UINT64));

  DEBUG ((DEBUG_INFO, "PerformBiosGuardUpdate: BIOS GUARD Bios Update Execution Result = 0x%016llX\n", BiosGuardStatus));

  if ((BiosGuardStatus & 0xFFFF) != 0) {
    switch (BiosGuardStatus) {
      case ERR_BAD_SVN:
        Status = EFI_SECURITY_VIOLATION;
        break;
      default:
        Status = EFI_DEVICE_ERROR;
        break;
    }
  } else {
    Status = EFI_SUCCESS;
  }

  return Status;
}

/**
  Update uCode via BiosGuard service.

  @param[in]  SystemFirmwareImage     Points to the System Firmware (BIOS) image.
  @param[in]  SystemFirmwareImageSize The length of the System Firmware image in bytes.
  @param[out] BgupImage               Points to BgupImage which would be sent to BiosGaurd ACM along with SystemFirmwareImage.
                                      BgupImage contains BGSL to update BIOS and certificate for authentication.
  @param[out] BgupImageSize           The length of the BgupImage in bytes.

  @retval EFI_SUCCESS                 The System Firmware image is updated successfully.
  @retval EFI_SECURITY_VIOLATION      The update operation fails due to SVN check error.
  @retval EFI_DEVICE_ERROR            The update operation fails.
  @retval EFI_INVALID_PARAMETER       BgupImage is NULL or BgupImageSize is unexpected.
  @retval Others                      The update operation fails.

**/
EFI_STATUS
PerformBiosGuardUpdateuCode (
  IN VOID                         *SystemFirmwareImage,
  IN UINTN                        SystemFirmwareImageSize,
  IN VOID                         *BgupImage,
  IN UINTN                        BgupImageSize
  )
{
  EFI_STATUS                           Status;
  BIOSGUARD_HOB                        *BiosGuardHobPtr;
  EFI_PHYSICAL_ADDRESS                 BiosGuardMemAddress;
  UINT32                               BiosGuardMemSize;
  EFI_PHYSICAL_ADDRESS                 BgupCertificate;
  EFI_PHYSICAL_ADDRESS                 BgIoTrapAddress;
  UINT8                                *BiosGuardMemAddressPtr;
  UINT8                                *BgupCertificatePtr;
  UINT64                               BiosGuardStatus;
  UINT8                                *BgUpFileBuffer;
  UINTN                                BgUpFileBufferSize;

  ASSERT (SystemFirmwareImage != NULL);
  ASSERT (BgupImage != NULL);
  ASSERT (BgupImageSize != 0);

  if (BgupImageSize < (sizeof (BGUP_HEADER) + BIOSGUARD_CERTIFICATE_SIZE)) {
    DEBUG ((DEBUG_ERROR, "PerformBiosGuardUpdate: BIOS Bgup file error! Abort BIOS Firmware Update\n"));
    return EFI_INVALID_PARAMETER;
  }

  BiosGuardHobPtr          = NULL;
  BiosGuardMemAddressPtr   = NULL;
  BgupCertificatePtr       = NULL;
  BiosGuardMemAddress      = 0;
  BiosGuardMemSize         = 0;
  BgIoTrapAddress          = 0;
  Status                   = EFI_NOT_STARTED;
  BgUpFileBuffer           = NULL;
  BgUpFileBufferSize       = 0;

  //
  // Get BIOS Guard Hob
  //
  BiosGuardHobPtr = GetFirstGuidHob (&gBiosGuardHobGuid);
  if (BiosGuardHobPtr == NULL) {
    DEBUG ((DEBUG_ERROR, "BIOS Guard HOB not available\n"));
    return EFI_DEVICE_ERROR;
  }

  BiosGuardMemAddress    = (EFI_PHYSICAL_ADDRESS) BiosGuardHobPtr->BiosGuardMemAddress;
  BiosGuardMemSize       = (UINT32) LShiftU64 (BiosGuardHobPtr->BiosGuardMemSize, 20);
  BgIoTrapAddress        = (EFI_PHYSICAL_ADDRESS) BiosGuardHobPtr->BiosGuardIoTrapAddress;

  BgupCertificate        = (EFI_PHYSICAL_ADDRESS) (BiosGuardMemAddress + BiosGuardMemSize - BGUPC_MEMORY_OFFSET);
  BiosGuardMemAddressPtr = (UINT8 *) BiosGuardMemAddress;
  BgupCertificatePtr     = (UINT8 *) BgupCertificate;

  DEBUG ((DEBUG_INFO, "PerformBiosGuardUpdate: BIOS Guard SPI Flow\n"));
  BgUpFileBuffer = BgupImage;
  BgUpFileBufferSize = BgupImageSize;

  DEBUG ((DEBUG_INFO, "PerformBiosGuardUpdate: BgUpFileBufferSize        = %d\n", BgUpFileBufferSize));
  DEBUG ((DEBUG_INFO, "PerformBiosGuardUpdate: (BgUpFileBufferSize - %d) = %d\n", BIOSGUARD_CERTIFICATE_SIZE, (BgUpFileBufferSize - BIOSGUARD_CERTIFICATE_SIZE)));
  DEBUG ((DEBUG_INFO, "PerformBiosGuardUpdate: BiosBinSize               = 0x%08X\n", SystemFirmwareImageSize));

  CopyMem (BiosGuardMemAddressPtr, BgUpFileBuffer, (BgUpFileBufferSize - BIOSGUARD_CERTIFICATE_SIZE));
  CopyMem ((BiosGuardMemAddressPtr + (BgUpFileBufferSize - BIOSGUARD_CERTIFICATE_SIZE)), SystemFirmwareImage, SystemFirmwareImageSize);
  CopyMem (BgupCertificatePtr, (BgUpFileBuffer + (BgUpFileBufferSize - BIOSGUARD_CERTIFICATE_SIZE)), BIOSGUARD_CERTIFICATE_SIZE);
  IoRead8 (BgIoTrapAddress);
  CopyMem (&BiosGuardStatus, BgupCertificatePtr, sizeof (UINT64));

  DEBUG ((DEBUG_INFO, "PerformBiosGuardUpdate: BIOS GUARD Bios Update Execution Result = 0x%016llX\n", BiosGuardStatus));

  if ((BiosGuardStatus & 0xFFFF) != 0) {
    switch (BiosGuardStatus) {
      case ERR_BAD_SVN:
        Status = EFI_SECURITY_VIOLATION;
        break;
      default:
        Status = EFI_DEVICE_ERROR;
        break;
    }
  } else if ((BiosGuardStatus & 0xFFFF0000) != 0) {
    //
    // BGUP execution succeeds but function exits with extra error code
    //   1  Errase error
    //   2  Write  error
    //   3  Region error
    //
    DEBUG ((DEBUG_INFO, "PerformBiosGuardUpdate: extra data returns BIOS GUARD script If = %x\n", (BiosGuardStatus & 0xFFFF0000) >> 16));
    Status = EFI_DEVICE_ERROR;
  } else {
    Status = EFI_SUCCESS;
  }

  return Status;
}

/**
  Perform BIOS region update.

  This function is intended to support both non-BiosGuard and BiosGuard cases.
  When BiosGuard is supported and enabled on the platform, BiosGuardUpdate argument
  is supposed to be TRUE and BgupImage is the Bios Guard Update Package along with
  input BIOS data buffer.
  Considering BIOS region might be updated by stage, the input argument FlashAddress
  indicates the relative or absolute BIOS region address (depends on FlashAddressType)
  to be flashed.

  @param[in] FlashAddress      The offset relative to BIOS region to be accessed.
  @param[in] Buffer            The pointer to the data buffer.
  @param[in] Length            The length of data buffer in bytes.
  @param[in] BiosGuardUpdate   Indicates if this update should be performed by
                               BiosGuard service or not.
  @param[in] BgupImage         If BiosGuardUpdate is TRUE, the pointer to the
                               BGUP image corresponding to Buffer used in
                               BiosGuard Update.
  @param[in] BgupImageSize     The length of BgupImage in bytes.
  @param[in] Progress          A function used report the progress of the
                               firmware update.  This is an optional parameter
                               that may be NULL.
  @param[in] StartPercentage   The start completion percentage value that may
                               be used to report progress during the flash
                               write operation.
  @param[in] EndPercentage     The end completion percentage value that may
                               be used to report progress during the flash
                               write operation.

  @retval EFI_SUCCESS           The operation returns successfully.
  @retval EFI_WRITE_PROTECTED   The flash device is read only.
  @retval EFI_DEVICE_ERROR      The flash update fails.
  @retval EFI_UNSUPPORTED       The flash device access is unsupported.
  @retval EFI_INVALID_PARAMETER The input parameter is not valid.
  @retval Others                Flash update failure

**/
EFI_STATUS
EFIAPI
UpdateBiosFirmware (
  IN EFI_PHYSICAL_ADDRESS                           FlashAddress,
  IN VOID                                           *Buffer,
  IN UINTN                                          Length,
  IN BOOLEAN                                        BiosGuardUpdate,
  IN VOID                                           *BgupImage,
  IN UINTN                                          BgupImageSize,
  IN EFI_FIRMWARE_MANAGEMENT_UPDATE_IMAGE_PROGRESS  Progress,        OPTIONAL
  IN UINTN                                          StartPercentage,
  IN UINTN                                          EndPercentage
  )
{
  EFI_STATUS          Status;

  DEBUG ((DEBUG_INFO, "UpdateBiosFirmware - 0x%x - 0x%x\n", (UINTN)FlashAddress, Length));

  if (BiosGuardUpdate) {
    DEBUG ((DEBUG_INFO, "Update via BiosGuard service\n"));

    if (Progress != NULL) {
      Progress (StartPercentage);
    }

    Status = PerformBiosGuardUpdate (
               Buffer,
               Length,
               BgupImage,
               BgupImageSize
               );

    if (Progress != NULL) {
      Progress (EndPercentage);
    }

  } else {
    DEBUG ((DEBUG_INFO, "Update via PCH SPI protocol\n"));
    Status = SpiFlashUpdate (
               FlashRegionBios,
               (UINT32) FlashAddress,
               (UINT8 *) Buffer,
               (UINT32) Length,
               Progress,
               StartPercentage,
               EndPercentage
               );
  }

  return Status;
}

/**
  Perform microcode write opreation dedicated for SPI update

  @param FlashAddress      The address of flash device to be accessed.
  @param Buffer            The pointer to the data buffer.
  @param Length            The length of data buffer in bytes.

  @retval EFI_SUCCESS           The operation returns successfully.
  @retval EFI_WRITE_PROTECTED   The flash device is read only.
  @retval EFI_UNSUPPORTED       The flash device access is unsupported.
  @retval EFI_INVALID_PARAMETER The input parameter is not valid.
**/
EFI_STATUS
EFIAPI
MicrocodeFlashWrite(
  IN EFI_PHYSICAL_ADDRESS         FlashAddress,
  IN VOID                         *Buffer,
  IN UINTN                        Length
  )
{
  EFI_PHYSICAL_ADDRESS         AlignedFlashAddress;
  VOID                         *AlignedBuffer;
  UINTN                        AlignedLength;
  UINTN                        OffsetHead;
  UINTN                        OffsetTail;
  EFI_STATUS                   Status;


  DEBUG((EFI_D_INFO, "MicrocodeFlashWrite - 0x%x - 0x%x\n", (UINTN)FlashAddress, Length));

  //
  // Need make buffer 64K aligned to support ERASE
  //
  // [Aligned] FlashAddress Aligned]
  // |                      |                                 |
  // V                     V                                V
  // +--------------+========+------------+
  // | OffsetHeader | Length | OffsetTail |
  // +--------------+========+------------+
  // ^
  // |<-----------AlignedLength----------->
  // |
  // AlignedFlashAddress
  //
  OffsetHead = FlashAddress & (SIZE_4KB - 1);
  OffsetTail = (FlashAddress + Length) & (SIZE_4KB - 1);
  if (OffsetTail != 0) {
    OffsetTail = SIZE_4KB - OffsetTail;
  }

  if ((OffsetHead != 0) || (OffsetTail != 0)) {
    AlignedFlashAddress = FlashAddress - OffsetHead;
    AlignedLength = Length + OffsetHead + OffsetTail;

    AlignedBuffer = AllocatePool(AlignedLength);
    if (AlignedBuffer == NULL) {
      return EFI_OUT_OF_RESOURCES;
    }
    //
    // Save original buffer. We can't use MMIO read because TopSwap bit may be toggled
    // We must use SPI Read
    //
    //
    // 1. Copy content in Header alignment area
    //
    if (OffsetHead != 0) {
      Status = SpiFlashRead(
                 FlashRegionBios,
                 (UINT32)AlignedFlashAddress,
                 OffsetHead,
                 (UINT8 *)AlignedBuffer
                 );
      ASSERT(!EFI_ERROR(Status));
    }

    //
    // 2. Copy content in Tail alignment area
    //
    if (OffsetTail != 0) {
      Status = SpiFlashRead(
                 FlashRegionBios,
                 (UINT32)(AlignedFlashAddress + OffsetHead + Length),
                 OffsetTail,
                 (UINT8 *)AlignedBuffer + OffsetHead + Length
                 );
      ASSERT(!EFI_ERROR(Status));
    }

    //
    // Override new buffer. Source is from memory. No need to use SPI
    //
    CopyMem((UINT8 *)AlignedBuffer + OffsetHead, Buffer, Length);
  } else {
    AlignedFlashAddress = FlashAddress;
    AlignedBuffer = Buffer;
    AlignedLength = Length;
  }

  Status = SpiFlashUpdate (
             FlashRegionBios,
             (UINT32) AlignedFlashAddress,
             (UINT8 *) AlignedBuffer,
             (UINT32) AlignedLength,
             NULL,
             0,
             90
             );

  if ((OffsetHead != 0) || (OffsetTail != 0)) {
    FreePool (AlignedBuffer);
  }
  return Status;
}
