/** @file
  TbtRetimerNvmUpdateLib instance to support TBT Retimer firmware update

@copyright
  INTEL CONFIDENTIAL
  Copyright 2019 Intel Corporation.

  The source code contained or described herein and all documents related to the
  source code ("Material") are owned by Intel Corporation or its suppliers or
  licensors. Title to the Material remains with Intel Corporation or its suppliers
  and licensors. The Material may contain trade secrets and proprietary and
  confidential information of Intel Corporation and its suppliers and licensors,
  and is protected by worldwide copyright and trade secret laws and treaty
  provisions. No part of the Material may be used, copied, reproduced, modified,
  published, uploaded, posted, transmitted, distributed, or disclosed in any way
  without Intel's prior express written permission.

  No license under any patent, copyright, trade secret or other intellectual
  property right is granted to or conferred upon you by disclosure or delivery
  of the Materials, either expressly, by implication, inducement, estoppel or
  otherwise. Any license under such intellectual property rights must be
  express and approved by Intel in writing.

  Unless otherwise agreed by Intel in writing, you may not remove or alter
  this notice or any other notice embedded in Materials by Intel or
  Intel's suppliers or licensors in any way.

  This file contains a 'Sample Driver' and is licensed as such under the terms
  of your license agreement with Intel or your vendor. This file may be modified
  by the user, subject to the additional terms of the license agreement.

@par Specification Reference:
**/
#include <PiDxe.h>

#include <Library/DebugLib.h>
#include <Library/BaseLib.h>
#include <Library/BaseMemoryLib.h>
#include <Library/MemoryAllocationLib.h>
#include <Library/UefiBootServicesTableLib.h>
#include <Library/TcssRetimerNvmUpdateLib.h>

#include "TbtNvmDrvRetimerThruHr.h"
#include "TbtNvmDrvUtils.h"
#include "TbtNvmDrvYflLib.h"

#define IECS_CMD_ADDR      8
#define IECS_DATA_ADDR     9
#define IECS_MSG_OUT_RDATA 18

// The commands below are 4cc encoded
#define IECS_BOPS_CMD 0x53504f42
#define IECS_BLKW_CMD 0x574b4c42
#define IECS_AUTH_CMD 0x48545541
#define IECS_PCYC_CMD 0x43594350
#define IECS_AFRR_CMD 0x52524641

// This is number of accesses to remote IECS CMD before declaring timeout
#define TBT_TOTAL_OF_ACCESSES_TO_IECS_CMD_UNTIL_TIMEOUT 300
#define TBT_NVM_MAX_DWS_TO_WRITE                        16
#define TBT_TOTAL_NUM_OF_RECOVERIES_DURING_IMAGE_WRITE  1

/// Success status from LC
#define TBT_AUTH_SUCCCESS                               1
#define TBT_NVM_VERSION_OFFSET_DW                       2
#define TBT_TIME_TO_WAIT_FOR_AUTH_US                    1000 * 1000 * 5 // 5s
#define TBT_NVM_VERSION_MASK                            (MASK (23, 0))

STATIC
TBT_STATUS
TbtNvmDrvReadIECS_9 (
  IN TARGET_DEVICE *DevCom
  )
{
  TBT_STATUS  Status;
  UINT32      ReadData;
  UINT32      AccessCount;

  ASSERT (DevCom != NULL);
  if (DevCom == NULL) {
    return TBT_STATUS_INVALID_PARAM;
  }

  AccessCount = 0;

  // Wait for command to be completed
  do {
    Status = DevCom->ReadIecsReg (DevCom, IECS_DATA_ADDR, &ReadData);
    if (TBT_STATUS_ERR (Status)) {
      DEBUG ((
        DEBUG_ERROR,
        "TbtNvmDrvReadIECS_9: Got an error while reading from IECS_9 register, Status - %d, AccessCount = %d. Exiting...\n",
        Status,
        AccessCount
        ));

      return TBT_STATUS_NON_RECOVERABLE_ERROR;
    }
    AccessCount++;
  } while (AccessCount < TBT_TOTAL_OF_ACCESSES_TO_IECS_CMD_UNTIL_TIMEOUT);

  DEBUG ((DEBUG_INFO, "TbtNvmDrvReadIECS_9: AccessCount = %d, ReadData = %x\n", AccessCount, ReadData));
  return TBT_STATUS_SUCCESS;
}

/**
  Waits for completion of IECS command by polling the IECS CMD remote register

  @param[in] DevCom Pointer to the device interface
  @param[in] Cmd IECS CMD which was send

  @retval TBT_STATUS_SUCCESS Command was successefully send
  @retval TBT_STATUS_NON_RECOVERABLE_ERROR A device error has accured.
  @retval TBT_STATUS_RETRY LC reported error, the command might be retried
**/
STATIC
TBT_STATUS
TbtNvmDrvWaitForIecsCmdCpl (
  IN TARGET_DEVICE *DevCom,
  UINT32           Cmd
  )
{
  TBT_STATUS  Status;
  UINT32      ReadData;
  UINT32      AccessCount;

  ASSERT (DevCom != NULL);
  ASSERT (Cmd != 0);

  if (DevCom == NULL || Cmd == 0) {
    return TBT_STATUS_INVALID_PARAM;
  }

  AccessCount = 0;

  // Wait for command to be completed
  do {
    Status = DevCom->ReadIecsReg (DevCom, IECS_CMD_ADDR, &ReadData);
    if (TBT_STATUS_ERR (Status)) {
      DEBUG ((DEBUG_ERROR, "TbtNvmDrvWaitForIecsCmdCpl: Got an error while reading from IECS_CMD register, Status - %d, AccessCount = %d, Cmd = %x. Exiting...\n", Status, AccessCount, Cmd));
      return TBT_STATUS_NON_RECOVERABLE_ERROR;
    }
    AccessCount++;
  } while ((ReadData == Cmd) && (AccessCount < TBT_TOTAL_OF_ACCESSES_TO_IECS_CMD_UNTIL_TIMEOUT));

  if (ReadData == Cmd) {
    DEBUG ((DEBUG_ERROR, "TbtNvmDrvWaitForIecsCmdCpl: Remote IECS CMD is timeouted. AccessCount = %d, ReadData = %x Exiting...\n", AccessCount, ReadData));
    return TBT_STATUS_NON_RECOVERABLE_ERROR;
  }
  if (ReadData != 0) { // ERR
    DEBUG ((DEBUG_ERROR, "TbtNvmDrvWaitForIecsCmdCpl: Got error from remote LC while writing the block. AccessCount = %d, ReadData = %x, Cmd = %x\n", AccessCount, ReadData, Cmd));
    TbtNvmDrvReadIECS_9 (DevCom);
    return TBT_STATUS_RETRY;
  }

  return TBT_STATUS_SUCCESS;
}

/**
  Sends IECS command and waits for completion.

  @param[in] DevCom Pointer to the device interface
  @param[in] Cmd IECS CMD to send

  @retval TBT_STATUS_SUCCESS Command was successefully send
  @retval TBT_STATUS_NON_RECOVERABLE_ERROR A device error has accured.
  @retval TBT_STATUS_RETRY LC reported error, the command might be retried

**/
TBT_STATUS
TbtNvmDrvSendCmd (
  IN TARGET_DEVICE *DevCom,
  UINT32           Cmd
  )
{
  TBT_STATUS       Status;

  DEBUG ((DEBUG_VERBOSE, "Sending CMD 0x%x\n", Cmd));

  Status = DevCom->WriteIecsReg (DevCom, IECS_CMD_ADDR, &Cmd, 1);
  if (TBT_STATUS_ERR (Status)) {
    DEBUG ((DEBUG_ERROR, "TbtNvmDrvSendCmd: Fail to write to IECS CMD reg, Status = %d.\n", Status));
    return TBT_STATUS_NON_RECOVERABLE_ERROR;
  }
  // Wait for command to be completed
  return TbtNvmDrvWaitForIecsCmdCpl (DevCom, Cmd);
}

/**
  Writes data block to the target device's NVM

  @param[in] DevCom Pointer to the device interface
  @param[in] Data to send
  @param[in] Length data length in DW

  @retval TBT_STATUS_SUCCESS Command was successefully send
  @retval TBT_STATUS_NON_RECOVERABLE_ERROR A device error has accured.
  @retval TBT_STATUS_RETRY LC reported error, the command might be retried
**/
STATIC
TBT_STATUS
TbtNvmDrvDeviceWrBlk (
  IN TARGET_DEVICE  *DevCom,
  UINT32            *Data,
  UINT8             Length
  )
{
  TBT_STATUS Status;

  ASSERT(DevCom != NULL);
  ASSERT(Length <= TBT_NVM_MAX_DWS_TO_WRITE);
  ASSERT(Length > 0);
  ASSERT(Data != NULL);
  if ((DevCom == NULL) || (Length == 0) || (Length > TBT_NVM_MAX_DWS_TO_WRITE) || (Data == NULL)) {
    return TBT_STATUS_INVALID_PARAM;
  }

  DEBUG ((DEBUG_VERBOSE, "Writing block with length - 0x%x\n", Length));

  // Write data
  Status = DevCom->WriteIecsReg (DevCom, IECS_MSG_OUT_RDATA, Data, Length);
  if (TBT_STATUS_ERR (Status)) {
    DEBUG ((DEBUG_ERROR, "TbtNvmDrvDeviceWrBlk: ERROR! The data to REMOTE_WDATA register wasn't written. Status - %d. Exiting...\n", Status));
    return TBT_STATUS_NON_RECOVERABLE_ERROR;
  }

  // Send the command
  Status = TbtNvmDrvSendCmd (DevCom, IECS_BLKW_CMD);
  DEBUG ((DEBUG_VERBOSE, "Finished writing block with length - 0x%x (%d)\n", Length, Status));
  return Status;
}

/**
  Writes offset in the NVM, to start from it

  @param[in] DevCom Pointer to the device interface
  @param[in] Offset in NVM in DW resolution

  @retval TBT_STATUS_SUCCESS Command was successefully send
  @retval TBT_STATUS_NON_RECOVERABLE_ERROR A device error has accured.
  @retval TBT_STATUS_RETRY LC reported error, the command might be retried
**/
STATIC
TBT_STATUS
TbtNvmDrvDeviceWrOffset (
  IN TARGET_DEVICE *DevCom,
  IN UINT32        Offset
  )
{
  TBT_STATUS         Status;
  UINT32             OffsetInB;

  ASSERT(DevCom != NULL);
  if (DevCom == NULL) {
    return TBT_STATUS_INVALID_PARAM;
  }

  DEBUG ((DEBUG_VERBOSE, "Write device offset - 0x%x\n", Offset));

  OffsetInB = Offset << 2;

  // Write data
  Status = DevCom->WriteIecsReg (DevCom, IECS_DATA_ADDR, &OffsetInB, 1);
  if (TBT_STATUS_ERR (Status)) {
    DEBUG ((DEBUG_ERROR, "TbtNvmDrvDeviceWrBlk: ERROR! The data to IECS_DATA register wasn't written. Status - %d. Exiting...\n", Status));
    return TBT_STATUS_NON_RECOVERABLE_ERROR;
  }
  // Send the command
  Status = TbtNvmDrvSendCmd (DevCom, IECS_BOPS_CMD);
  DEBUG ((DEBUG_VERBOSE, "Finished writing device offset - 0x%x (%d)\n", Offset, Status));
  return Status;
}

/**
  Initiate Auth at the device, wait for it to complete and check the status

  @param[in] DevCom Pointer to the device interface

  @retval TBT_STATUS_SUCCESS If auth passed
  @retval TBT_STATUS_TIMEOUT If device is not responding
  @retval TBT_STATUS_NON_RECOVERABLE_ERROR On any other communication error
  @retval TBT_STATUS_GENERIC_ERR If auth is failed

**/
STATIC
TBT_STATUS
TbtNvmDrvDeviceExecAuth (
  IN TARGET_DEVICE *DevCom
  )
{
  UINT32            ReadData;
  TBT_STATUS        Status;
  UINT32            WriteData;

  DEBUG ((DEBUG_INFO, "Sending Auth command\n"));

  WriteData = 0;
  Status = DevCom->WriteIecsReg (DevCom, IECS_DATA_ADDR, &WriteData, 1); // Turn Auth on
  if (TBT_STATUS_ERR (Status)) {
    DEBUG ((DEBUG_ERROR, "TbtNvmDrvDeviceExecAuth: Got an error while writing to IECS_DATA register, status - %d. Exiting...\n", Status));
    return TBT_STATUS_NON_RECOVERABLE_ERROR;
  }

  WriteData = IECS_AUTH_CMD;
  Status = DevCom->WriteIecsReg (DevCom, IECS_CMD_ADDR, &WriteData, 1); // Send Auth command
  if (TBT_STATUS_ERR (Status)) {
    DEBUG ((DEBUG_ERROR, "TbtNvmDrvDeviceExecAuth: Got an error while writing to IECS_CMD register, status - %d. Exiting...\n", Status));
    return TBT_STATUS_NON_RECOVERABLE_ERROR;
  }
  DEBUG ((DEBUG_INFO, "Waiting %d seconds for authentication to complete...\n", TBT_TIME_TO_WAIT_FOR_AUTH_US / 1000000));
  gBS->Stall(TBT_TIME_TO_WAIT_FOR_AUTH_US);

  DevCom->IndicateState (DevCom, AFTER_AUTH);

  Status = TbtNvmDrvWaitForIecsCmdCpl (DevCom, IECS_AUTH_CMD);
  if (TBT_STATUS_ERR (Status)) {
    DEBUG ((DEBUG_ERROR, "TbtNvmDrvDeviceExecAuth: Authentication failed!!! status from LC - %d. Exiting...\n", Status));
    return TBT_STATUS_NON_RECOVERABLE_ERROR;
  }

  Status = DevCom->ReadIecsReg (DevCom, IECS_DATA_ADDR, &ReadData);
  if (TBT_STATUS_ERR (Status)) {
    DEBUG ((DEBUG_ERROR, "TbtNvmDrvDeviceExecAuth: Got an error while reading from IECS_DATA register, status - %d. Exiting...\n", Status));
    return TBT_STATUS_NON_RECOVERABLE_ERROR;
  }
  if (ReadData != TBT_AUTH_SUCCCESS) {
    DEBUG ((DEBUG_ERROR, "TbtNvmDrvDeviceExecAuth: Auth is failed rdata = %x. Exiting...\n", ReadData));
    return TBT_STATUS_NON_RECOVERABLE_ERROR;
  }
  DEBUG ((DEBUG_INFO, "TbtNvmDrvDeviceExecAuth: Auth is succeeded!!!\n"));
  return TBT_STATUS_SUCCESS;
}

STATIC
TBT_STATUS
TbtDrvReadDwFromNvm (
  IN TARGET_DEVICE  *DevCom,
  IN UINT32         Offset,
  OUT UINT32        *RdData
  )
{
  TBT_STATUS Status;

  ASSERT (DevCom != NULL);
  if (DevCom == NULL) {
    return TBT_STATUS_INVALID_PARAM;
  }

  DEBUG ((DEBUG_INFO, "Reading DW from NVM, offset 0x%x\n", Offset));

  // Write Offset to IECS_DATA register
  Status = DevCom->WriteIecsReg (DevCom, IECS_DATA_ADDR, &Offset, 1);
  if (TBT_STATUS_ERR (Status)) {
    DEBUG ((DEBUG_ERROR, "TbtDrvReadDwFromNvm: ERROR! The data to IECS_DATA register wasn't written. Status - %d. Exiting...\n", Status));
    return TBT_STATUS_NON_RECOVERABLE_ERROR;
  }
  // Send the command
  Status = TbtNvmDrvSendCmd (DevCom, IECS_AFRR_CMD);
  if (TBT_STATUS_ERR (Status)) {
    DEBUG ((DEBUG_ERROR, "TbtDrvReadDwFromNvm: ERROR! The command wasn't send. Status - %d. Exiting...\n", Status));
    return TBT_STATUS_NON_RECOVERABLE_ERROR;
  }
  // Read data from MSG_OUT
  Status = DevCom->ReadIecsReg (DevCom, IECS_MSG_OUT_RDATA, RdData);
  if (TBT_STATUS_ERR (Status)) {
    DEBUG ((DEBUG_ERROR, "TbtDrvReadDwFromNvm: ERROR! The read from RDATA has failed. Status - %d. Exiting...\n", Status));
    return TBT_STATUS_NON_RECOVERABLE_ERROR;
  }

  DEBUG ((DEBUG_INFO, "Finished reading DW from NVM offset - 0x%x\n", Offset));

  return TBT_STATUS_SUCCESS;
}

/**
  Reads NVM version from the target device

  @param[in]  DevCom Pointer to the device interface
  @param[out] Pointer where to put the read version

  @retval EFI_INVALID_PARAMETER
  @retval EFI_DEVICE_ERROR
  @retval EFI_SUCCESS

**/
EFI_STATUS
TbtDrvReadNvmVersion (
  IN TARGET_DEVICE *DevCom,
  OUT UINT32       *Version
  )
{
  TBT_STATUS       Status;

  ASSERT (DevCom != NULL);
  if (DevCom == NULL) {
    return EFI_INVALID_PARAMETER;
  }

  DEBUG ((DEBUG_INFO, "Reading nvm version\n"));

  Status = TbtDrvReadDwFromNvm (DevCom, TBT_NVM_VERSION_OFFSET_DW << 2, Version);
  if (TBT_STATUS_ERR (Status)) {
    DEBUG ((DEBUG_ERROR, "TbtDrvReadNvmVersion: ERROR! Couldn't read from NVM. Status %d\n", Status));
    return EFI_DEVICE_ERROR;
  }

  DEBUG ((DEBUG_INFO, "Finished reading NVM version - 0x%x\n", *Version));

  return EFI_SUCCESS;
}

/**
  Createa and initialize the required data structure to access an Retimer device.
  Besides, set up a communication with it if needed.

  @param[in]   DevAddress         The address info of target Retimer device.
  @param[out]  RetimerDevice      Pointer to a Retimer device instance corresponding to DevAddress.

  @retval EFI_SUCCESS             RetimerDevice is created successfully and is ready to accept commands.
  @retval Others                  An error occurred attempting to access the Retimer device.

**/
EFI_STATUS
EFIAPI
CreateRetimerDevInstance (
  IN   RETIMER_DEV_ADDRESS                            *DevAddress,
  OUT  RETIMER_DEV_INSTNACE                           *RetimerDevice
  )
{
  EFI_STATUS           Status;
  TARGET_DEVICE        *Device;
  PCIE_BDF             PciLocation;

  ASSERT (DevAddress != NULL);
  ASSERT (RetimerDevice != NULL);

  if ((DevAddress == NULL) || (RetimerDevice == NULL)) {
    return EFI_INVALID_PARAMETER;
  }

  //
  // Initialize device start
  //
  Device = NULL;
  Status = EFI_SUCCESS;

  PciLocation.Bus      = DevAddress->Bus;
  PciLocation.Device   = DevAddress->Device;
  PciLocation.Function = DevAddress->Function;
  Device = TbtNvmDrvRetimerThruHrCtor (
             PciLocation,
             DevAddress->Channel.Port,
             DevAddress->RetimerIndex,
             TbtNvmDrvYflForcePwrFunc
             );

  if (Device == NULL) {
    DEBUG ((DEBUG_ERROR, "Failed to allocate memory for Device\n"));
    Status = EFI_DEVICE_ERROR;
  }

  *RetimerDevice = Device;
  return Status;
}

/**
  Release the data structure corresponding to given RetimerDevice.
  Besides, close the communication with it if needed.

  @param[in]  RetimerDevice       Retimer device instance corresponding to DevAddress.

  @retval EFI_SUCCESS             RetimerDevice is released successfully.
  @retval Others                  An error occurred attempting to access the Retimer device.

**/
EFI_STATUS
EFIAPI
DestroyRetimerDevInstance (
  IN  RETIMER_DEV_INSTNACE                            RetimerDevice
  )
{
  TARGET_DEVICE        *Device;

  Device = (TARGET_DEVICE *) RetimerDevice;

  if (Device != NULL) {
    Device->Destroy (Device);
  }

  return EFI_SUCCESS;
}

/**
  Read Retimer NVM FW version on given RetimerDevice.

  Note: Caller is supposed to invoke CreateRetimerDevInstance() first to ensure the data structure
        of RetimerDevice is created properly and the communication to the target device is settled down.
        DestroyRetimerDevInstance() should be invoked after the caller is done with all Reimter accesses.

  @param[in]   RetimerDevice      Retimer device instance corresponding to DevAddress.
  @param[out]  Version            Retimer NVM FW version.

  @retval EFI_SUCCESS             Retimer FW version is successfully get.
  @retval Others                  An error occurred attempting to access the Retimer firmware

**/
EFI_STATUS
EFIAPI
ReadRetimerNvmVersion (
  IN   RETIMER_DEV_INSTNACE                           RetimerDevice,
  OUT  UINT32                                         *Version
  )
{
  EFI_STATUS           Status;
  TARGET_DEVICE        *Device;
  UINT32               RetimerVersion;

  if (Version == NULL) {
    ASSERT (FALSE);
    return EFI_INVALID_PARAMETER;
  }

  RetimerVersion = 0;
  Device = (TARGET_DEVICE *) RetimerDevice;

  Status = TbtDrvReadNvmVersion (Device, &RetimerVersion);
  if (EFI_ERROR (Status)) {
    DEBUG ((DEBUG_ERROR, "TbtDrvReadNvmVersion error %r.\n", Status));
    return EFI_DEVICE_ERROR;
  }

  DEBUG ((DEBUG_INFO, "Current NVM version (before mask) is 0x%x\n", RetimerVersion));
  RetimerVersion = RetimerVersion & TBT_NVM_VERSION_MASK;
  DEBUG ((DEBUG_INFO, "Current NVM version (after mask) is 0x%x\n", RetimerVersion));

  *Version = RetimerVersion;
  return EFI_SUCCESS;
}

/**
  Update Retimer NVM Firmware on given RetimerDevice.

  Note: Caller is supposed to invoke CreateRetimerDevInstance() first to ensure the data structure
        of RetimerDevice is created properly and the communication to the target device is settled down.
        DestroyRetimerDevInstance() should be invoked after the caller is done with all Reimter accesses.

  @param[in]  RetimerDevice       Retimer device instance corresponding to DevAddress.
  @param[in]  RetimerImage        Pointer to an image buffer contains Retimer FW to be updated with.
  @param[in]  ImageSize           The size of RetimerImage memory buffer.
  @param[in]  Progress            A function used to report the progress of updating the firmware
                                  device with the new firmware image.
  @param[in]  StartPercentage     The start completion percentage value that may be used to report progress
                                  during the flash write operation. The value is between 1 and 100.
  @param[in]  EndPercentage       The end completion percentage value that may be used to report progress
                                  during the flash write operation.

  @retval EFI_SUCCESS             Retimer Firmware is successfully updated.
  @retval Others                  An error occurred attempting to access the Retimer firmware

**/
EFI_STATUS
EFIAPI
UpdateRetimerNvmFirmware (
  IN  RETIMER_DEV_INSTNACE                           RetimerDevice,
  IN  UINT8                                          *RetimerImage,
  IN  UINTN                                          ImageSize,
  IN  EFI_FIRMWARE_MANAGEMENT_UPDATE_IMAGE_PROGRESS  Progress,          OPTIONAL
  IN  UINTN                                          StartPercentage,
  IN  UINTN                                          EndPercentage
  )
{
  EFI_STATUS           Status;
  UINT32               *BufferPointer;
  TARGET_DEVICE        *Device;
  UINT32               Offset;
  TBT_STATUS           TbtStatus;
  UINT32               RecoveryCount;
  UINT8                writeLength;
  UINTN                DisplayLength;

  ASSERT (RetimerDevice != NULL);
  ASSERT (RetimerImage != NULL);
  ASSERT (EndPercentage > StartPercentage);

  if ((RetimerDevice == NULL) || (RetimerImage == NULL) || (EndPercentage < StartPercentage)) {
    return EFI_INVALID_PARAMETER;
  }

  DisplayLength = EndPercentage - StartPercentage;

  Device = (TARGET_DEVICE *) RetimerDevice;

  BufferPointer = (UINT32 *) RetimerImage;
  ImageSize = (UINT32) ImageSize;
  ImageSize -= (BufferPointer[0] & TBT_NVM_VERSION_MASK);
  BufferPointer += ((BufferPointer[0] & TBT_NVM_VERSION_MASK) / 4); // Point to digital
  ImageSize /= 4;

  for (RecoveryCount = 0; RecoveryCount < TBT_TOTAL_NUM_OF_RECOVERIES_DURING_IMAGE_WRITE; RecoveryCount++) {
    // Write BOPS once at the start
    TbtStatus = TbtNvmDrvDeviceWrOffset (Device, 0x0);
    if (TBT_STATUS_ERR (TbtStatus)) {
      if (TBT_STATUS_FATAL_ERR (TbtStatus)) {
        DEBUG ((DEBUG_ERROR, "Got a FATAL error, exiting...\n"));
        Status = EFI_DEVICE_ERROR;
        goto finish_set_image;
      }
      continue;  // retry image write from the start
    }

    for (Offset = 0; Offset < ImageSize; Offset += TBT_NVM_MAX_DWS_TO_WRITE) {
      if ((Progress != NULL) & (DisplayLength > 0) && ((Offset & 0xFF) == 0)) {
        //
        // Display current progress
        //
        Progress (StartPercentage + ((Offset * (DisplayLength)) / ImageSize));
      }
      writeLength = (ImageSize - Offset) >= TBT_NVM_MAX_DWS_TO_WRITE ? TBT_NVM_MAX_DWS_TO_WRITE :
        (UINT8)(ImageSize - Offset);

      if ((Offset & 0xFF) == 0) {
        DEBUG ((DEBUG_INFO, "Writing offset - 0x%x\n", Offset));
      }

      TbtStatus = TbtNvmDrvDeviceWrBlk (Device, BufferPointer + Offset, writeLength);
      if (TBT_STATUS_ERR (TbtStatus)) {
        if (TBT_STATUS_FATAL_ERR (TbtStatus)) {
          DEBUG ((DEBUG_ERROR, "Got a FATAL error, exiting...\n"));
          Status = EFI_DEVICE_ERROR;
          goto finish_set_image;
        }
        break;  // retry image write from the start
                // TODO more fine tuned recovery is possible
      }
    }

    if (!TBT_STATUS_ERR (TbtStatus)) {  // If got here without errors => Success
      DEBUG ((DEBUG_INFO, "Image write is done!!!\n"));
      Status = EFI_SUCCESS;
      break;
    }
    else {
      DEBUG ((DEBUG_ERROR, "Got an error while writing the image. As a recovery, starting again\n"));
    }
  }
  if (RecoveryCount >= TBT_TOTAL_NUM_OF_RECOVERIES_DURING_IMAGE_WRITE) {
    DEBUG ((DEBUG_ERROR, "Image write wasn't successefull due to a device error\n"));
    Status = EFI_DEVICE_ERROR;
    goto finish_set_image;
  }

  TbtStatus = TbtNvmDrvDeviceExecAuth (Device);
  if (TBT_STATUS_ERR (TbtStatus)) {
    Status = EFI_DEVICE_ERROR;
    goto finish_set_image;
  }

  DEBUG ((DEBUG_INFO, "Image was updated successefully and passed the authentication.\n"));

  Status = EFI_SUCCESS;

finish_set_image:

  if (Progress != NULL) {
    Progress (EndPercentage);
  }

  return Status;
}
