/** @file
  Implements the interface for TbtNvmDrvHr class.
  This class represents TBT HR abstraction.

@copyright
  INTEL CONFIDENTIAL
  Copyright 2019 - 2020 Intel Corporation.

  The source code contained or described herein and all documents related to the
  source code ("Material") are owned by Intel Corporation or its suppliers or
  licensors. Title to the Material remains with Intel Corporation or its suppliers
  and licensors. The Material may contain trade secrets and proprietary and
  confidential information of Intel Corporation and its suppliers and licensors,
  and is protected by worldwide copyright and trade secret laws and treaty
  provisions. No part of the Material may be used, copied, reproduced, modified,
  published, uploaded, posted, transmitted, distributed, or disclosed in any way
  without Intel's prior express written permission.

  No license under any patent, copyright, trade secret or other intellectual
  property right is granted to or conferred upon you by disclosure or delivery
  of the Materials, either expressly, by implication, inducement, estoppel or
  otherwise. Any license under such intellectual property rights must be
  express and approved by Intel in writing.

  Unless otherwise agreed by Intel in writing, you may not remove or alter
  this notice or any other notice embedded in Materials by Intel or
  Intel's suppliers or licensors in any way.

  This file contains a 'Sample Driver' and is licensed as such under the terms
  of your license agreement with Intel or your vendor. This file may be modified
  by the user, subject to the additional terms of the license agreement.

@par Specification Reference:

**/

#include <Library/UefiBootServicesTableLib.h>
#include "TcI2cmDrvHr.h"
#include "Crc32c.h"
#include "TcI2cmDrvDma.h"

#define TBT_EE_ARC_EN_BIT_OFFSET 0
#define TBT_EE_ARC_EN_XOR_BIT_OFFSET 1

struct _TBT_HR_IMPL {
  EFI_PCI_IO_PROTOCOL    *pPciIoProto;
  TBT_DMA                *pDma;
  FORCE_PWR_HR           ForcePwrFunc;
};

typedef struct {
  UINT32 reqCode;
  UINT32 crc;
} DRV_RDY_CMD;

/*
* SW->FW commands, supported by this implementation
* CC = Command Code
*/
enum {
  CC_DRV_READY = 3
};

/*
* FW->SW responses, supported by this implementation
* RC = response code
*/
enum {
  RC_DRV_READY = 3
};

/**
  Send driver ready using DMA.

  @param[in]  Dma  DMA instance to work with.
**/
STATIC
TBT_STATUS
TbtNvmDrvHrSendDrvReady(
  TBT_DMA *Dma
  )
{
  DRV_RDY_CMD message;
  UINT32 crc32;
  EFI_STATUS status;
  // Prepare the driver ready message to be send
  message.reqCode = SwapBytes32(CC_DRV_READY);
  status = CalCrc32c((unsigned char *)&message, OFFSET_OF(DRV_RDY_CMD, crc), &crc32);
  message.crc = SwapBytes32(crc32);
  // Send the message
  status = Dma->TxCfgPkt(Dma, PDF_SW_TO_FW_COMMAND, sizeof(message), (UINT8 *)&message);
  if (TBT_STATUS_ERR(status)) {
    DEBUG ((DEBUG_ERROR, "TbtNvmDrvHrSendDrvReady: The driver ready command is not sent successefully. Status - %d. Exiting...\n", status));
    return TBT_STATUS_NON_RECOVERABLE_ERROR;
  }
  // Make sure response is received
  status = Dma->RxCfgPkt(Dma, PDF_FW_TO_SW_RESPONSE, NULL, NULL);
  if (TBT_STATUS_ERR(status)) {
    DEBUG ((DEBUG_ERROR, "TbtNvmDrvHrSendDrvReady: Didn't receive response for driver ready command. Status - %d. Exiting...\n", status));
    return TBT_STATUS_NON_RECOVERABLE_ERROR;
  }
  DEBUG ((DEBUG_ERROR, "TbtNvmDrvHrSendDrvReady: Sent successefully\n"));
  return TBT_STATUS_SUCCESS;
}

/* Write request pointer to reuse in TbtNvmDrvHrWriteCioReg only */
STATIC WRITE_CONFIGURATION_REGISTERS_REQUEST *mWriteDevReq = NULL;
/* Buffer to be used in TbtNvmDrvHrWriteCioReg. Predefined as optimization */
STATIC WRITE_CONFIGURATION_REGISTERS_REQUEST mWriteReqSwapped;

/**
  Write register to TBT CIO config space.

  @param[in]  This    Pointer to the TBT_HR structure on which the method is invoked.
  @param[in]  RegNum  Register offset in TBT device config space.
  @param[in]  RegNum  Register number.
  @param[in]  Length  Length of data to write.
  @param[in]  Data    Pointer to data to write.
**/
STATIC
TBT_STATUS
TbtNvmDrvHrWriteCioReg (
  IN TBT_HR   *This,
  IN UINT16   RegNum,
  IN UINT8    Length,
  IN UINT32*  Data
  )
{
  UINT32 len;
  UINT32 crc32;
  EFI_STATUS status;
  UINT16 dataPointer;
  UINT16 expectedLength;

  DEBUG ((DEBUG_ERROR, "HrWriteCioReg: RegNum - 0x%x, Length - 0x%x\n", RegNum, Length));
  DEBUG ((DEBUG_ERROR, "HrWriteCioReg: Data - "));
  for (UINTN i = 0; i < Length; i++) {
    DEBUG ((DEBUG_ERROR, "%x ", *(Data + i)));
  }
  DEBUG ((DEBUG_ERROR, "\n"));

  ASSERT(Length > 0 && Length <= 16);

  if (mWriteDevReq == NULL) {  // If not allocated yet
    mWriteDevReq = TcssNvmDrvAllocateMem(sizeof(WRITE_CONFIGURATION_REGISTERS_REQUEST));
    if (mWriteDevReq == NULL) {
      DEBUG ((DEBUG_ERROR, "TbtNvmDrvHrWriteCioReg: Couldn't allocate memory for write request struct. Exiting...\n"));
      return TBT_STATUS_NON_RECOVERABLE_ERROR;
    }
    // Initialize constant values for read request
    mWriteDevReq->RouteString = 0x0;    // HR
    mWriteDevReq->SequenceNumber = 0;   // Always use the same one, no two outstanding requests anyway;
    mWriteDevReq->ConfigurationSpace = DEVICE_CONFIG_SPACE;
    mWriteDevReq->Port = 0x0;
  }
  mWriteDevReq->DwIndex = RegNum;
  mWriteDevReq->Length = Length;
  for (dataPointer = 0; dataPointer < Length; dataPointer++) {
    mWriteDevReq->WrData[dataPointer] = *(Data + dataPointer);
  }
  // dataPointer - points after the data

  // Swap the data
  len = OFFSET_OF(WRITE_CONFIGURATION_REGISTERS_REQUEST, WrData) + dataPointer * 4; // Length in Byte without CRC
  TcssNvmDrvSwapEndianess((UINT32 *)mWriteDevReq, len / 4, (UINT32 *)&mWriteReqSwapped);
  // Calculate CRC
  status = CalCrc32c((unsigned char *)&mWriteReqSwapped, len, &crc32);
  mWriteReqSwapped.WrData[dataPointer] = SwapBytes32(crc32);    // Insert CRC
  // Transmit the request
  status = This->Impl->pDma->TxCfgPkt(
    This->Impl->pDma,
    PDF_WRITE_CONFIGURATION_REGISTERS,
    OFFSET_OF(WRITE_CONFIGURATION_REGISTERS_REQUEST, WrData) + Length * 4 + 4,    // RS + CMD + WRDATA + CRC
    (UINT8 *)&mWriteReqSwapped
  );
  if (TBT_STATUS_ERR(status)) {
    DEBUG ((DEBUG_ERROR, "TbtNvmDrvHrWriteCioReg: Couldn't transmit DMA frame to write to reg - 0x%x, length - 0x%x. Exiting...\n", RegNum, Length));
    goto error_while_writing;
  }
  expectedLength = (UINT16)sizeof(WRITE_CONFIGURATION_REGISTERS_RESPONSE);
  // Receive the response
  status = This->Impl->pDma->RxCfgPkt(
    This->Impl->pDma,
    PDF_WRITE_CONFIGURATION_REGISTERS,
    &expectedLength,
    NULL                                              // No data in the response
  );
  if (TBT_STATUS_ERR(status)) {
    DEBUG ((DEBUG_ERROR, "TbtNvmDrvHrReadCioDevReg: TbtNvmDrvHrWriteCioReg: Couldn't receive response while writing to reg - 0x%x, length - 0x%x. Exiting...\n", RegNum, Length));
    goto error_while_writing;
  }

  return TBT_STATUS_SUCCESS;

error_while_writing:
  This->Impl->pDma->DbgPrint(This->Impl->pDma);
  return TBT_STATUS_NON_RECOVERABLE_ERROR;
}

/* Read request pointer to reuse in TbtNvmDrvHrReadCioDevReg only */
STATIC READ_CONFIGURATION_REGISTERS_REQUEST *mReadDevReq = NULL;

/* Buffer to be used in TbtNvmDrvHrReadCioDevReg. Predefined as optimization */
STATIC READ_CONFIGURATION_REGISTERS_REQUEST mReadReqSwapped;

/* Buffer to be used in TbtNvmDrvHrReadCioDevReg. Predefined as optimization */
STATIC READ_CONFIGURATION_REGISTERS_RESPONSE mReadReqResponse;

/**
  Reads the a register with a given address from TBT device config space.
  Operation:
    Prepare the read request
    Swap bytes
    Calculate CRC and swap it too

  @param[in]   This    Pointer to the TBT_HR structure on which the method is invoked.
  @param[in]   RegNum  Register offset in TBT device config space.
  @param[out]  Data    Read data, one DW.
**/
STATIC
TBT_STATUS
TbtNvmDrvHrReadCioDevReg (
  IN TBT_HR   *This,
  IN UINT16   RegNum,
  OUT UINT32  *Data
  )
{
  UINT32 len;
  UINT32 crc32;
  EFI_STATUS status;
  UINT32 readData;
  UINT16 expectedLength;

  DEBUG ((DEBUG_ERROR, "HrReadCioDevReg: RegNum - 0x%x\n", RegNum));

  if (mReadDevReq == NULL) {  // If not allocated yet
    mReadDevReq = TcssNvmDrvAllocateMem(sizeof(READ_CONFIGURATION_REGISTERS_REQUEST));
    if (mReadDevReq == NULL) {
      DEBUG ((DEBUG_ERROR, "Couldn't allocate memory for read request struct\n"));
      return TBT_STATUS_NON_RECOVERABLE_ERROR;
    }
    // Initialize constant values for read request
    mReadDevReq->RouteString = 0x0;    // HR
    mReadDevReq->SequenceNumber = 0;   // Always use the same one, no two outstanding requests anyway;
    mReadDevReq->ConfigurationSpace = DEVICE_CONFIG_SPACE;
    mReadDevReq->Port = 0x0;
    mReadDevReq->Length = 1;          // Always read only one register
  }
  mReadDevReq->DwIndex = RegNum;
  // Swap the data
//  len = OFFSET_OF(READ_CONFIGURATION_REGISTERS_REQUEST, Crc); // Length in Byte
// Calculate len in an alternative way to pass KW check
  len = (UINT32) ((UINTN) &(mReadReqSwapped.Crc) - (UINTN) &mReadReqSwapped);
  TcssNvmDrvSwapEndianess((UINT32 *)mReadDevReq, len / 4, (UINT32 *)&mReadReqSwapped);
  // Calculate CRC
  status = CalCrc32c((unsigned char *)&mReadReqSwapped, len, &crc32);
  mReadReqSwapped.Crc = SwapBytes32(crc32);
  // Transmit the request
  status = This->Impl->pDma->TxCfgPkt(
    This->Impl->pDma,
    PDF_READ_CONFIGURATION_REGISTERS,
    (UINT16)(len + 4),                                // Length + CRC
    (UINT8 *)&mReadReqSwapped
    );
  if (TBT_STATUS_ERR(status)) {
    DEBUG ((DEBUG_ERROR, "Couldn't transmit DMA frame to read from reg - 0x%x. Exiting...\n", RegNum));
    goto error_while_reading;
  }
  expectedLength = (UINT16)OFFSET_OF(READ_CONFIGURATION_REGISTERS_RESPONSE, Crc) + 4;
  // Receive the response
  status = This->Impl->pDma->RxCfgPkt(
    This->Impl->pDma,
    PDF_READ_CONFIGURATION_REGISTERS,
    &expectedLength,
    (UINT8 *)&mReadReqResponse
    );
  if (TBT_STATUS_ERR(status)) {
    DEBUG ((DEBUG_ERROR, "Couldn't receive response while reading from reg - 0x%x. Exiting...\n", RegNum));
    goto error_while_reading;
  }

  // Prepare the read data for return
  readData = mReadReqResponse.ReadData;
  readData = SwapBytes32(readData);
  *Data = readData;
  DEBUG ((DEBUG_ERROR, "Read data is - 0x%x\n", readData));
  return TBT_STATUS_SUCCESS;

error_while_reading:
  This->Impl->pDma->DbgPrint(This->Impl->pDma);
  return TBT_STATUS_NON_RECOVERABLE_ERROR;
}

/**
  TBT HR de-allocator.

  @param  This  Pointer to the TBT_HR structure on which the method is invoked.
**/
STATIC
VOID
TbtNvmDrvHrDtor (
  TBT_HR *This
  )
{
  DEBUG ((DEBUG_ERROR, "TbtNvmDrvHrDtor was called.\n"));
  This->Impl->pDma->Dtor(This->Impl->pDma);
  if (This->Impl->ForcePwrFunc != NULL) {
    This->Impl->ForcePwrFunc(This->Impl->pPciIoProto, FALSE);   // Disable the force power to HR
  }
  TcssNvmDrvDeAllocateMem(This->Impl);
  TcssNvmDrvDeAllocateMem(This);
}

/**
  TBT HR constructor.

  @param[in]  PcieBdf       PCIe BDF identifier.
  @param[in]  ForcePwrFunc  Force power function, will be performed on a found TBT device. If NULL -> no force pwr.

  @retval The TBT_HR object pointer.
**/
TBT_HR*
TbtNvmDrvHrCtor (
  IN PCIE_BDF     PcieBdf,
  IN FORCE_PWR_HR ForcePwrFunc OPTIONAL
  )
{
  EFI_STATUS  status;
  UINTN       handleCount;
  EFI_HANDLE  *handles;
  UINT32      i;
  TBT_HR_IMPL *pHrImpl;
  UINT32      curPciId = 0x0;
  UINTN       segmentNumber;
  UINTN       busNumber;
  UINTN       deviceNumber;
  UINTN       functionNumber;
  UINT64      attributes;
  TBT_HR      *pHr;
  UINT32      dataHolder;

  // Initialize UEFI PCI protocol
  status = gBS->LocateHandleBuffer(
             ByProtocol,
             &gEfiPciIoProtocolGuid,
             NULL,
             &handleCount,
             &handles
             );
  if (EFI_ERROR(status)) {
    DEBUG ((DEBUG_ERROR, "TbtNvmDrvHrCtor: LocateHandleBuffer of gEfiPciIoProtocolGuid failed, status=%d\n", status));
    return NULL;
  }

  pHrImpl = TcssNvmDrvAllocateMem(sizeof(TBT_HR_IMPL));
  if (!pHrImpl) {
    DEBUG ((DEBUG_ERROR, "TbtNvmDrvHrCtor: TcssNvmDrvAllocateMem failed, status=%d\n", EFI_OUT_OF_RESOURCES));
    return NULL;
  }
  // Search for the device with a given BDF
  for (i = 0; i < handleCount; i++) {
    status = gBS->HandleProtocol(handles[i], &gEfiPciIoProtocolGuid, &pHrImpl->pPciIoProto);
    if (EFI_ERROR(status)) {
      DEBUG ((DEBUG_ERROR, "TbtNvmDrvHrCtor: HandleProtocol of gEfiPciIoProtocolGuid failed, status=%d\n", status));
      continue;
    }
    pHrImpl->pPciIoProto->GetLocation(
                            pHrImpl->pPciIoProto,
                            &segmentNumber,
                            &busNumber,
                            &deviceNumber,
                            &functionNumber
                            );
    if (EFI_ERROR(status)) {
      DEBUG ((DEBUG_ERROR, "TbtNvmDrvHrCtor: GetLocation failed, status=%d\n", status));
      continue;
    }
    if (
      busNumber != PcieBdf.Bus ||
      deviceNumber != PcieBdf.Device ||
      functionNumber != PcieBdf.Function
      ) {
      status = EFI_NOT_FOUND;
      continue;
    }
    status = pHrImpl->pPciIoProto->Pci.Read(pHrImpl->pPciIoProto, EfiPciIoWidthUint32, 0, 1, &curPciId);
    if (EFI_ERROR(status)) {
      DEBUG ((DEBUG_ERROR, "TbtNvmDrvHrCtor: Pci.Read failed, status=%d\n", status));
      continue;
    }
    DEBUG ((DEBUG_INFO, "TbtNvmDrvHrCtor: Found Thunderbolt Controller %x: %04x:%02x:%02x.%02x\n",
      curPciId, segmentNumber, busNumber, deviceNumber, functionNumber));

    status = EFI_SUCCESS;
    break;
  } // end of search for the HR

  if (EFI_ERROR(status)) {
    DEBUG ((DEBUG_ERROR, "TbtNvmDrvHrCtor: PCI device with bdf = 0x%x,0x%x,0x%x is not found!, status=%d\n",
      PcieBdf.Bus,
      PcieBdf.Device,
      PcieBdf.Function,
      status
      ));
    goto free_hr_impl;
  }

  // Get the PCI Command options that are supported by this controller.
  status = pHrImpl->pPciIoProto->Attributes(
                                   pHrImpl->pPciIoProto,
                                   EfiPciIoAttributeOperationSupported,
                                   0,
                                   &attributes
                                   );
  if (EFI_ERROR(status)) {
    DEBUG ((DEBUG_ERROR, "TbtNvmDrvHrCtor: Attributes get is failed, status=%d\n", status));
    goto free_hr_impl;
  }

  // Set the PCI Command options to enable device memory mapped IO,
  // port IO, and bus mastering.
  status = pHrImpl->pPciIoProto->Attributes(
                                   pHrImpl->pPciIoProto,
                                   EfiPciIoAttributeOperationEnable,
                                   attributes & (EFI_PCI_DEVICE_ENABLE | EFI_PCI_IO_ATTRIBUTE_DUAL_ADDRESS_CYCLE),
                                   NULL
                                   );
  if (EFI_ERROR(status)) {
    DEBUG ((DEBUG_ERROR, "TbtNvmDrvHrCtor: Attributes set failed, status=%d\n", status));
    goto free_hr_impl;
  }

  // Force power the TBT
  if (ForcePwrFunc != NULL) {
    status = ForcePwrFunc(pHrImpl->pPciIoProto, TRUE);
    if (TBT_STATUS_ERR(status)) {
      DEBUG ((DEBUG_ERROR, "TbtNvmDrvHrCtor: Could not perform device %x force pwr, status=%d\n", curPciId, status));
      goto free_hr_impl;
    }
  }
  pHrImpl->ForcePwrFunc = ForcePwrFunc;

  Crc32cInit();

  // Initiate DMA and config it
  pHrImpl->pDma = TbtNvmDrvDmaCtor(pHrImpl->pPciIoProto);
  if (!pHrImpl->pDma) {
    DEBUG ((DEBUG_ERROR, "TbtNvmDrvHrCtor: Could not init DMA for device %x\n", curPciId));
    goto free_hr_impl;
  }

  // Send driver ready
  dataHolder = pHrImpl->pDma->ReadMmio(pHrImpl->pDma, TBT_ARC_DEBUG_REG_MMIO_OFFSET);
  if ((dataHolder & TBT_EE_ARC_EN_BIT_OFFSET) ^ (dataHolder & TBT_EE_ARC_EN_XOR_BIT_OFFSET)) {  // If ARC is enabled
    DEBUG ((DEBUG_ERROR, "Sending driver ready\n"));
    status = TbtNvmDrvHrSendDrvReady(pHrImpl->pDma);
    if (TBT_STATUS_ERR(status)) {
      DEBUG ((DEBUG_ERROR, "TbtNvmDrvHrCtor: Failed to send driver ready, status=%d", status));
      pHrImpl->pDma->DbgPrint(pHrImpl->pDma);
      goto free_dma;
    }
  }

  // Allocate the interface and assign it
  pHr = TcssNvmDrvAllocateMem(sizeof(TBT_HR));
  if (!pHr) {
    DEBUG ((DEBUG_ERROR, "TbtNvmDrvHrCtor: Could not allocate memory for TBT_HR\n"));
    goto free_dma;
  }
  pHr->Impl = pHrImpl;
  pHr->ReadCioDevReg  = TbtNvmDrvHrReadCioDevReg;
  pHr->WriteCioDevReg = TbtNvmDrvHrWriteCioReg;
  pHr->Dtor           = TbtNvmDrvHrDtor;

  return pHr;

free_dma:
  pHrImpl->pDma->Dtor(pHrImpl->pDma);
free_hr_impl:
  TcssNvmDrvDeAllocateMem(pHrImpl);
  return NULL;
}
