/** @file
  Chasm Falls Resiliency init Dxe driver.

@copyright
  INTEL CONFIDENTIAL
  Copyright 2019 - 2020 Intel Corporation.

  The source code contained or described herein and all documents related to the
  source code ("Material") are owned by Intel Corporation or its suppliers or
  licensors. Title to the Material remains with Intel Corporation or its suppliers
  and licensors. The Material may contain trade secrets and proprietary and
  confidential information of Intel Corporation and its suppliers and licensors,
  and is protected by worldwide copyright and trade secret laws and treaty
  provisions. No part of the Material may be used, copied, reproduced, modified,
  published, uploaded, posted, transmitted, distributed, or disclosed in any way
  without Intel's prior express written permission.

  No license under any patent, copyright, trade secret or other intellectual
  property right is granted to or conferred upon you by disclosure or delivery
  of the Materials, either expressly, by implication, inducement, estoppel or
  otherwise. Any license under such intellectual property rights must be
  express and approved by Intel in writing.

  Unless otherwise agreed by Intel in writing, you may not remove or alter
  this notice or any other notice embedded in Materials by Intel or
  Intel's suppliers or licensors in any way.

  This file contains a 'Sample Driver' and is licensed as such under the terms
  of your license agreement with Intel or your vendor. This file may be modified
  by the user, subject to the additional terms of the license agreement.

@par Specification Reference:
**/
#include <Library/UefiRuntimeServicesTableLib.h>
#include <Library/BaseLib.h>
#include <Library/BaseMemoryLib.h>
#include <Library/DebugLib.h>
#include <Library/MemoryAllocationLib.h>
#include <Library/PcdLib.h>
#include <Library/UefiLib.h>
#include <Library/HobLib.h>
#include <Library/BaseCryptLib.h>
#include <Library/UefiBootServicesTableLib.h>
#include <Library/SpiAccessLib.h>
#include <Library/PlatformFlashAccessLib.h>
#include <Library/CpuPlatformLib.h>

#include <Protocol/FirmwareVolume2.h>
#include <BiosGuard.h>

#include <Protocol/SimpleFileSystem.h>
#include <Protocol/DevicePath.h>
#include <Protocol/PciIo.h>
#include <Protocol/Spi.h>

#include <Guid/FileSystemInfo.h>
#include <Guid/FileInfo.h>
#include <Guid/FmpCapsule.h>
#include <Guid/SysFwUpdateProgress.h>

#include <IndustryStandard/Pci.h>

#include <Uefi.h>
#include <Uefi/UefiSpec.h>
#include <SetupVariable.h>
#include <Protocol/FirmwareVolume2.h>
#include <Library/IoLib.h>
#include <Register/PchRegsLpc.h>
//#include <Library/ConfigBlockLib.h>

EFI_HANDLE                               mBackUpFileSystemHandle = NULL;
SYSTEM_FIRMWARE_UPDATE_PROGRESS          mPreviousUpdateProgress = {0};

/**
  Set/Clear TopSwap bit by request. This is used in Non-BiosGuard BIOS update scenerio.

  @param[in]      Enable        TRUE: Set TopSwap bit; FALSE: Clear TopSwap bit.

  @retval EFI_UNSUPPORTED       Current platform does not support the SwSmi functions to access TopSwap bit.
  @retval EFI_SUCCESS           Triggered TopSwap enable/disable SMI by request.

**/
EFI_STATUS
TopSwapControl (
  IN BOOLEAN           Enable
  )
{
  ASSERT (PcdGet8 (PcdTopSwapEnableSwSmi) != 0xFF);
  ASSERT (PcdGet8 (PcdTopSwapDisableSwSmi) != 0xFF);

  if (Enable) {
    DEBUG ((DEBUG_INFO, "Enable TopSwap via SwSmi (0x%x)\n", PcdGet8 (PcdTopSwapEnableSwSmi)));
    IoWrite8 (R_PCH_IO_APM_CNT, PcdGet8 (PcdTopSwapEnableSwSmi));
  } else {
    DEBUG ((DEBUG_INFO, "Disable TopSwap via SwSmi (0x%x)\n", PcdGet8 (PcdTopSwapDisableSwSmi)));
    IoWrite8 (R_PCH_IO_APM_CNT, PcdGet8 (PcdTopSwapDisableSwSmi));
  }

  return EFI_SUCCESS;
}

/**
  Dump raw data.

  @param[in]  Data  raw data
  @param[in]  Size  raw data size

**/
VOID
InternalDumpData (
  IN UINT8   *Data8,
  IN UINTN   DataSize
  )
{
  DEBUG_CODE_BEGIN();

  UINTN      Index;

  for (Index = 0; Index < DataSize; Index++) {
    if (Index % 0x10 == 0) {
      DEBUG ((DEBUG_INFO, "\n%08X:", Index));
    }
    DEBUG ((DEBUG_INFO, " %02X", *Data8++));
  }
  DEBUG ((DEBUG_INFO, "\n"));

  DEBUG_CODE_END();
}

/**
  Calculate SHA256 Hash for OBB to check OBB is the original one

  @param[in]  Data   data
  @param[in]  Size   data size
  @param[out] Digest SHA256 digest

**/
VOID
CreateSha256Hash (
  IN  UINT8     *Data,
  IN  UINTN     Size,
  OUT UINT8     *Digest
  )
{
  UINTN       CtxSize;
  VOID        *HashCtx;

  CtxSize = Sha256GetContextSize ();
  HashCtx = AllocatePool (CtxSize);
  ASSERT (HashCtx != NULL);
  Sha256Init (HashCtx);
  Sha256Update (HashCtx, Data, Size);
  Sha256Final (HashCtx, Digest);
  InternalDumpData (Digest, 32);

  FreePool (HashCtx);
}

/**
  Connect storage controllers to read back up relevant files for resiliency support.
  Currently only NVME and onboard SATA controller are supported.
  After this called, storage will be ready for access.

**/
VOID
ConnectPlatformController (
  VOID
  )
{
  EFI_STATUS           Status;
  UINTN                Index;
  UINTN                HandleCount;
  EFI_HANDLE           *HandleBuffer;
  PCI_TYPE00           PciData;
  EFI_PCI_IO_PROTOCOL  *PciIo;
  UINTN                Segment;
  UINTN                Bus;
  UINTN                Device;
  UINTN                Function;

  HandleCount  = 0;
  HandleBuffer = NULL;

  Status = gBS->LocateHandleBuffer (
                  ByProtocol,
                  &gEfiPciIoProtocolGuid,
                  NULL,
                  &HandleCount,
                  &HandleBuffer
                  );
  if (EFI_ERROR (Status)) {
    return;
  }

  for (Index = 0; Index < HandleCount; Index++) {
    Status = gBS->HandleProtocol (
                    HandleBuffer[Index],
                    &gEfiPciIoProtocolGuid,
                    (VOID *) &PciIo
                    );
    ASSERT_EFI_ERROR (Status);
    PciIo->GetLocation (PciIo, &Segment, &Bus, &Device, &Function);
    Status = PciIo->Pci.Read (
                          PciIo,
                          EfiPciIoWidthUint8,
                          0,
                          sizeof (PciData),
                          &PciData
                          );
    if (Bus != 0) {
      //
      //  Locate all NVME controllers and connect them.
      //
      if ((PciData.Hdr.ClassCode[2] == PCI_CLASS_MASS_STORAGE) &&
          (PciData.Hdr.ClassCode[1] == PCI_CLASS_MASS_STORAGE_SOLID_STATE) &&
          (PciData.Hdr.ClassCode[0] == PCI_IF_MASS_STORAGE_SOLID_STATE_ENTERPRISE_NVMHCI)) {
        Status = gBS->ConnectController (HandleBuffer[Index], NULL, NULL, TRUE);
        if (EFI_ERROR (Status)) {
          DEBUG ((DEBUG_ERROR, "Connect NVME Controller on PCI %d/%d/%d - %r\n", Bus, Device, Function, Status));
        }
      }
    } else {
      //
      //  Locate Sata controllers and connect them.
      //
      if ((PciData.Hdr.ClassCode[2] == PCI_CLASS_MASS_STORAGE) &&
          (PciData.Hdr.ClassCode[1] == PCI_CLASS_MASS_STORAGE_SATADPA)) {
        Status = gBS->ConnectController (HandleBuffer[Index], NULL, NULL, TRUE);
        if (EFI_ERROR (Status)) {
          DEBUG ((DEBUG_ERROR, "Connect Sata Controller on PCI %d/%d/%d - %r\n", Bus, Device, Function, Status));
        }
      }
      if ((PciData.Hdr.ClassCode[2] == PCI_CLASS_MASS_STORAGE) &&
          (PciData.Hdr.ClassCode[1] == PCI_CLASS_MASS_STORAGE_RAID)) {
        //
        // In case that RAID controller has been connected before update function being invoked.
        //
        Status = gBS->DisconnectController (
                        HandleBuffer[Index],
                        NULL,
                        NULL
                        );
        if (EFI_ERROR (Status)) {
          DEBUG ((DEBUG_ERROR, "Disconnect RAID Controller on PCI %d/%d/%d - %r\n", Bus, Device, Function, Status));
        }
      }
    }
  }
  if (HandleBuffer) {
    FreePool (HandleBuffer);
  }
  return;
}

/**
  Find out Efi System Partition in storage to save backup files.

  @param[out] FileSystemHandle    Pointer to the file system handle would be used to keep
                                  Seamless Recovery backup files.

  @retval EFI_SUCCESS             The file system is found.
  @retval Others                  Cannot find an available file system.

**/
EFI_STATUS
SearchBackupFileSystem (
  OUT   EFI_HANDLE       *FileSystemHandle
  )
{
  EFI_STATUS                            Status;
  EFI_HANDLE                            *HandleArray;
  UINTN                                 HandleArrayCount;
  EFI_DEVICE_PATH_PROTOCOL              *DevicePath;
  UINTN                                 Index;
  EFI_SIMPLE_FILE_SYSTEM_PROTOCOL       *Fs;
  EFI_FILE                              *Root;
  EFI_FILE_SYSTEM_INFO                  *SysInfo;
  UINTN                                 SysInfoSize;

  DEBUG ((DEBUG_INFO, "SearchBackupFileSystem - entry\n"));

  *FileSystemHandle = NULL;
  HandleArray       = NULL;
  //
  // Search all EFI system partitions
  //
  Status = gBS->LocateHandleBuffer (ByProtocol, &gEfiPartTypeSystemPartGuid, NULL, &HandleArrayCount, &HandleArray);
  if (EFI_ERROR (Status)) {
    DEBUG ((DEBUG_ERROR, "Cannot find ESP partition. Status = %r\n", Status));
    return Status;
  }

  DEBUG ((DEBUG_INFO, "ESP handle count is: %d\n", HandleArrayCount));

  for (Index = 0; (Index < HandleArrayCount) && (*FileSystemHandle == NULL); Index++) {
    Status = gBS->HandleProtocol (HandleArray[Index], &gEfiDevicePathProtocolGuid, (VOID **)&DevicePath);
    if (EFI_ERROR (Status)) {
      DEBUG ((DEBUG_ERROR, "Cannot locate DevicePath protocol. Status = %r\n", Status));
      continue;
    }

    //
    // Get the SFS protocol from the handle
    //
    Status = gBS->HandleProtocol (HandleArray[Index], &gEfiSimpleFileSystemProtocolGuid, (VOID **)&Fs);
    if (EFI_ERROR (Status)) {
      DEBUG ((DEBUG_ERROR, "Cannot locate SFS protocol. Status = %r\n", Status));
      continue;
    }

    //
    // Open the root directory, get EFI_FILE_PROTOCOL
    //
    Status = Fs->OpenVolume (Fs, &Root);
    if (EFI_ERROR (Status)) {
      DEBUG ((DEBUG_ERROR, "Cannot open volume. Status = %r\n", Status));
      continue;
    }

    SysInfo     = NULL;
    SysInfoSize = 0;
    Status = Root->GetInfo (Root, &gEfiFileSystemInfoGuid, &SysInfoSize, SysInfo);
    if (Status == EFI_BUFFER_TOO_SMALL) {
      SysInfo = AllocateZeroPool (SysInfoSize);
      if (SysInfo == NULL) {
        DEBUG ((DEBUG_ERROR, "System memory is out of resource to allocate file system info buffer.\n"));
        Root->Close (Root);
        break;
      }
      Status = Root->GetInfo (Root, &gEfiFileSystemInfoGuid, &SysInfoSize, SysInfo);
      if (Status == EFI_SUCCESS) {
        DEBUG ((DEBUG_INFO, "File system info:\n"));
        DEBUG ((DEBUG_INFO, "FreeSpace:0x%x bytes\n", SysInfo->FreeSpace));
        DEBUG ((DEBUG_INFO, "BlockSize:0x%x bytes\n", SysInfo->BlockSize));
        DEBUG ((DEBUG_INFO, "ReadOnly:%x\n", SysInfo->ReadOnly));
        if ((SysInfo->FreeSpace >= SIZE_16MB) && (!SysInfo->ReadOnly)) {
          DEBUG ((DEBUG_INFO, "Available ESP found\n"));
          *FileSystemHandle = HandleArray[Index];
          Status = EFI_SUCCESS;
        }
      }
      FreePool (SysInfo);
    }

    Root->Close (Root);
  }

  if (*FileSystemHandle == NULL) {
    Status = EFI_NOT_FOUND;
  }

  FreePool (HandleArray);
  return Status;
}

/**
  Initialize mBackUpFileSystemHandle module variable

  @retval EFI_SUCCESS             Backup file system is found and assign to mBackUpFileSystemHandle
  @retval Others                  Cannot find an available file system to initialize mBackUpFileSystemHandle.

**/
EFI_STATUS
InitializeBackupFileSystem (
  VOID
  )
{
  EFI_STATUS             Status;

  if (mBackUpFileSystemHandle != NULL) {
    //
    // BackupFilesystem has been initialized.
    //
    return EFI_SUCCESS;
  }

  //
  // Connect storage and check free space.
  //
  ConnectPlatformController ();
  Status = SearchBackupFileSystem (&mBackUpFileSystemHandle);
  if (EFI_ERROR (Status)) {
    DEBUG ((DEBUG_ERROR, "Cannot find storage file system to support seamless recovery. Status = %r\n", Status));
  }

  return Status;
}

/**
  Delete a file from an assigned file system.

  @param[in] FileSystemHandle    Handle of the file system that file would be deleted from.
  @param[in] FileName            Pointer to file name.

  @retval EFI_SUCCESS            File does not exist or deleted the file successfully.
  @retval Others                 Failed to delete the file.

**/
EFI_STATUS
DeleteFile (
  IN   EFI_HANDLE       FileSystemHandle,
  IN   CHAR16           *FileName
  )
{
  EFI_STATUS                            Status;
  EFI_SIMPLE_FILE_SYSTEM_PROTOCOL       *FileSystem;
  EFI_FILE                              *Root;
  EFI_FILE                              *FileHandle;

  DEBUG ((DEBUG_INFO, "DeleteBackupFile - entry\n"));

  Status = gBS->HandleProtocol (FileSystemHandle, &gEfiSimpleFileSystemProtocolGuid, (VOID **)&FileSystem);
  if (EFI_ERROR (Status)) {
    DEBUG ((DEBUG_ERROR, "Cannot locate SFS protocol. Status = %r\n", Status));
    return Status;
  }

  //
  // Open the root directory, get EFI_FILE_PROTOCOL
  //
  Status = FileSystem->OpenVolume (FileSystem, &Root);
  if (EFI_ERROR (Status)) {
    DEBUG ((DEBUG_ERROR, "Cannot open volume. Status = %r\n", Status));
    return Status;
  }

  Status = Root->Open (Root, &FileHandle, FileName, EFI_FILE_MODE_READ | EFI_FILE_MODE_WRITE, 0);
  if (Status == EFI_NOT_FOUND) {
    DEBUG ((DEBUG_INFO, "File %s does not exist. No need to delete\n", FileName));
    return EFI_SUCCESS;
  } else if (EFI_ERROR (Status)) {
    DEBUG ((DEBUG_ERROR, "Cannot open file: %s. Status = %r\n", FileName, Status));
    return Status;
  }

  if (FileHandle == NULL) {
    Status = EFI_UNSUPPORTED;
    DEBUG ((DEBUG_ERROR, "Failed to open root dir on partition for writing. Stautus = %r\n", Status));
    return Status;
  }

  Status = FileHandle->Delete (FileHandle);
  DEBUG ((DEBUG_INFO, "Delete %s %r\n", FileName, Status));

  Root->Close (Root);

  return Status;
}

/**
  Write a file to an assigned file system.

  @param[in] FileSystemHandle    Handle of the file system that file would be write into.
  @param[in] FileName            Pointer to file name.
  @param[in] FileBuffer          The buffer to be written into file system.
  @param[in] FileSize            The size of FileBuffer.

  @retval EFI_SUCCESS            Wrote the file successfully.
  @retval Others                 Failed to write the file.

**/
EFI_STATUS
WriteBackupFile (
  IN   EFI_HANDLE       FileSystemHandle,
  IN   CHAR16           *FileName,
  IN   UINT8            *FileBuffer,
  IN   UINTN            FileSize
  )
{
  EFI_STATUS                            Status;
  EFI_SIMPLE_FILE_SYSTEM_PROTOCOL       *FileSystem;
  EFI_FILE                              *Root;
  EFI_FILE                              *FileHandle;
  UINTN                                 WriteSize;

  DEBUG ((DEBUG_INFO, "WriteBackupFile - entry\n"));

  Status = gBS->HandleProtocol (FileSystemHandle, &gEfiSimpleFileSystemProtocolGuid, (VOID **)&FileSystem);
  if (EFI_ERROR (Status)) {
    DEBUG ((DEBUG_ERROR, "Cannot locate SFS protocol. Status = %r\n", Status));
    return Status;
  }

  //
  // Open the root directory, get EFI_FILE_PROTOCOL
  //
  Status = FileSystem->OpenVolume (FileSystem, &Root);
  if (EFI_ERROR (Status)) {
    DEBUG ((DEBUG_ERROR, "Cannot open volume. Status = %r\n", Status));
    return Status;
  }

  Status = Root->Open (Root, &FileHandle, FileName, EFI_FILE_MODE_READ | EFI_FILE_MODE_WRITE | EFI_FILE_MODE_CREATE, 0);

  if (EFI_ERROR (Status)) {
    DEBUG ((DEBUG_INFO, "Cannot open file: %s. Status = %r\n", FileName, Status));
    return Status;
  }

  if (FileHandle == NULL) {
    Status = EFI_UNSUPPORTED;
    DEBUG ((DEBUG_ERROR, "Failed to open root dir on partition for writing. Stautus = %r\n", Status));
    return Status;
  }

  do {
    WriteSize = (FileSize > SIZE_4KB) ? SIZE_4KB : FileSize;
    Status = FileHandle->Write (FileHandle, &WriteSize, FileBuffer);
    if (EFI_ERROR (Status)) {
      break;
    }
    FileSize = FileSize - WriteSize;
    FileBuffer = FileBuffer + WriteSize;
  } while (FileSize > 0);

  DEBUG ((DEBUG_INFO, "Write %s %r\n", FileName, Status));
  FileHandle->Close (FileHandle);
  Root->Close (Root);

  return Status;
}

EFI_STATUS
ReadBackupFile (
  IN  EFI_FILE *FileSystemHandle,
  IN  CHAR16   *FileName,
  OUT VOID     **Buffer,
  OUT UINTN    *BufferSize
  )
{
  EFI_STATUS                            Status;
  EFI_SIMPLE_FILE_SYSTEM_PROTOCOL       *FileSystem;
  EFI_FILE                              *Root;
  EFI_FILE                              *FileHandle;
  UINTN                                 FileInfoSize;
  EFI_FILE_INFO                         *FileInfo;
  EFI_GUID                              FileInfoGuid = EFI_FILE_INFO_ID;

  DEBUG ((DEBUG_INFO, "ReadBackupFile - entry\n"));

  Status = gBS->HandleProtocol (FileSystemHandle, &gEfiSimpleFileSystemProtocolGuid, (VOID **)&FileSystem);
  if (EFI_ERROR (Status)) {
    DEBUG ((DEBUG_ERROR, "Cannot locate SFS protocol. Status = %r\n", Status));
    return Status;
  }

  //
  // Open the root directory, get EFI_FILE_PROTOCOL
  //
  Status = FileSystem->OpenVolume (FileSystem, &Root);
  if (EFI_ERROR (Status)) {
    DEBUG ((DEBUG_ERROR, "Cannot open volume. Status = %r\n", Status));
    return Status;
  }

  Status = Root->Open (Root, &FileHandle, FileName, EFI_FILE_MODE_READ, 0);

  if (EFI_ERROR (Status)) {
    DEBUG ((DEBUG_INFO, "Cannot open file: %s. Status = %r\n", FileName, Status));
    return Status;
  }

  if (FileHandle == NULL) {
    Status = EFI_UNSUPPORTED;
    DEBUG ((DEBUG_ERROR, "Failed to open root dir on partition for reading. Stautus = %r\n", Status));
    return Status;
  }

  DEBUG ((DEBUG_INFO, "FileInfoRead\n"));

  FileInfoSize = 0;
  FileInfo = NULL;

  Status = FileHandle->GetInfo (
                         FileHandle,
                         &FileInfoGuid,
                         &FileInfoSize,
                         NULL
                         );

  if (EFI_ERROR (Status)) {
    if (Status != EFI_BUFFER_TOO_SMALL) {
      DEBUG ((DEBUG_ERROR, "FileRead fail, GetInfo error, Status: %r\n", Status));
      return Status;
    }
  }

  if (Buffer == NULL || BufferSize == NULL) {
    DEBUG ((DEBUG_INFO, "FileInfoRead only, FileInfo Status: %r\n", Status));
    return EFI_SUCCESS;
  }

  DEBUG ((DEBUG_INFO, "FileRead\n"));
  FileInfo = AllocatePool (FileInfoSize);
  if (FileInfo == NULL) {
    DEBUG ((DEBUG_ERROR, "FileRead fail, AllocatePool(FileInfoSize: %x) error\n", FileInfoSize));
    return EFI_OUT_OF_RESOURCES;
  }

  Status = FileHandle->GetInfo (
                         FileHandle,
                         &FileInfoGuid,
                         &FileInfoSize,
                         FileInfo
                         );
  if (EFI_ERROR (Status)) {
    DEBUG ((DEBUG_ERROR, "FileRead fail, GetInfo error, Status: %r\n", Status));
    goto done;
  }

  *BufferSize = (UINT32) FileInfo->FileSize;
  if (*BufferSize != 0) {
    *Buffer = AllocateZeroPool (*BufferSize);
    if (*Buffer == NULL) {
      DEBUG ((DEBUG_ERROR, "FileRead fail, AllocatePool(FileSize: %x) error\n", *BufferSize));
      Status = EFI_OUT_OF_RESOURCES;
      goto done;
    }

    Status = FileHandle->Read (
                           FileHandle,
                           BufferSize,
                           *Buffer
                           );
    if (EFI_ERROR (Status)) {
      DEBUG ((DEBUG_ERROR, "FileRead fail, Read error, Status: %r\n", Status));
      goto done;
    }

  } else {
    DEBUG ((DEBUG_INFO, "File size is 0, set return Buffer to NULL.\n"));
    *Buffer = NULL;
  }

  Status = EFI_SUCCESS;

done:
  if (FileInfo != NULL) {
    FreePool (FileInfo);
  }

  DEBUG ((DEBUG_INFO, "FileRead done, Status: %r, BufferSize: %x\n", Status, *BufferSize));
  return Status;
}

/**
  ResiliencyCallBackFunction

  Chasm Falls Resiliency Feature to determine the newly updated BIOS Image health and
  then determine the rollback or sync-up for SPI IBBR region and Disk OBBR file.

  @param[in] Event     - A pointer to the Event that triggered the callback.
  @param[in] Context   - A pointer to private data registered with the callback function.
**/
VOID
EFIAPI
ResiliencyCallBackFunction (
  IN EFI_EVENT    Event,
  IN VOID         *Context
  )
{
  EFI_STATUS     Status;
  UINTN          VariableSize;
  UINT8          *Buffer;
  UINTN          Length;
  UINT8          *Buffer1;
  UINT8          Sha256[SHA256_DIGEST_SIZE];

  DEBUG((DEBUG_INFO, "ResiliencyCallBackFunction START\n"));

  Status = EFI_SUCCESS;
  Buffer1 = NULL;
  ZeroMem (&mPreviousUpdateProgress, sizeof (SYSTEM_FIRMWARE_UPDATE_PROGRESS));
  VariableSize = sizeof (SYSTEM_FIRMWARE_UPDATE_PROGRESS);

  Status = gRT->GetVariable (
                  SYSFW_UPDATE_PROGRESS_VARIABLE_NAME,
                  &gSysFwUpdateProgressGuid,
                  NULL,
                  &VariableSize,
                  &mPreviousUpdateProgress
                  );

  if ((Status != EFI_NOT_FOUND) && (Status != EFI_SUCCESS)) {
    DEBUG ((DEBUG_INFO, "Get UpdateProgress variable = %r.\n", Status));
    ASSERT_EFI_ERROR (Status);
  }

  DEBUG ((DEBUG_INFO, "Last boot UpdateProgress component = 0x%x.\n", mPreviousUpdateProgress.Component));
  DEBUG ((DEBUG_INFO, "Last boot UpdateProgress progress  = 0x%x.\n", mPreviousUpdateProgress.Progress));
#if FixedPcdGetBool(PcdResiliencyEnable) == 1
  {
    Status = InitializeBackupFileSystem ();
    if (EFI_ERROR (Status)) {
      return;
    }
    Status = ReadBackupFile (mBackUpFileSystemHandle, SYSBIOS_UPDATE_CAPSULE_BACKUP_FILE_NAMEN1, (void **) &Buffer, &Length);
    if ( !EFI_ERROR (Status) ) {
      //
      // old file existed, corruption detected(topswap enabled) or capsule update happened and trigger resiliency
      //
      if (SpiIsTopSwapEnabled() || (mPreviousUpdateProgress.Component == UpdatingResiliency && mPreviousUpdateProgress.Progress == 0)) {

        if (!SpiIsTopSwapEnabled ()) {
          //
          // capsule update with resiliency case, good update
          //
          DEBUG ((DEBUG_INFO, "IBB SyncUp via PCH SPI protocol. Source Base = %X\n", (UINTN) FixedPcdGet32(PcdFlashIbbBase)));
          //
          // write spi ibb to esp, then read ibb from esp and sync up it to ibbr
          //
          Status = WriteBackupFile (
                     mBackUpFileSystemHandle,
                     SYSBIOS_UPDATE_CAPSULE_DIGEST_VARIABLE_TEMP,
                     (UINT8 *) (UINTN) FixedPcdGet32 (PcdFlashIbbBase),
                     (UINTN) FixedPcdGet32 (PcdFlashIbbSize)
                     );
          Status = ReadBackupFile (mBackUpFileSystemHandle, SYSBIOS_UPDATE_CAPSULE_DIGEST_VARIABLE_TEMP, (void **) &Buffer, &Length);
          Status = SpiFlashUpdate (
                     FlashRegionBios,
                     (UINT32) FixedPcdGet32(PcdFlashIbbROffset),
                     (UINT8 *) Buffer,
                     (UINT32) FixedPcdGet32(PcdFlashIbbSize),
                     NULL,
                     0,
                     0
                     );

          DEBUG ((DEBUG_INFO, "OBB SyncUp from SPI to Disk\n"));
          Buffer = (UINT8 *) (UINTN) (FixedPcdGet32(PcdFlashObbBase));
          Length = FixedPcdGet32(PcdFlashObbSize);
          Status = WriteBackupFile (mBackUpFileSystemHandle, SYSBIOS_UPDATE_CAPSULE_BACKUP_FILE_NAMEN1, (UINT8 *) (UINTN) Buffer, Length);
          DeleteFile (mBackUpFileSystemHandle, SYSBIOS_UPDATE_CAPSULE_DIGEST_VARIABLE_TEMP);
          DeleteFile (mBackUpFileSystemHandle, SYSBIOS_UPDATE_CAPSULE_BACKUP_FILE_NAMEN);

          DeleteFile (mBackUpFileSystemHandle, SYSFW_UPDATE_CURRENT_FVADV_BACKUP_FILE_NAME);
          DeleteFile (mBackUpFileSystemHandle, SYSFW_UPDATE_CURRENT_FVOPT_BACKUP_FILE_NAME);
          DeleteFile (mBackUpFileSystemHandle, SYSFW_UPDATE_CURRENT_FVUEFI_BACKUP_FILE_NAME);
          DeleteFile (mBackUpFileSystemHandle, SYSFW_UPDATE_CURRENT_FVOS_BACKUP_FILE_NAME);
          Status = WriteBackupFile (
                     mBackUpFileSystemHandle,
                     SYSFW_UPDATE_CURRENT_FVADV_BACKUP_FILE_NAME,
                     (UINT8 *) (UINTN) FixedPcdGet32 (PcdFlashFvAdvancedBase),
                     (UINTN) FixedPcdGet32 (PcdFlashFvAdvancedSize)
                     );
          if (EFI_ERROR (Status)) {
            DEBUG ((DEBUG_ERROR, "Failed to write FvAdvanced.fv(%r)\n", Status));
            return;
          }

          Status = WriteBackupFile (
                     mBackUpFileSystemHandle,
                     SYSFW_UPDATE_CURRENT_FVOPT_BACKUP_FILE_NAME,
                     (UINT8 *) (UINTN) FixedPcdGet32 (PcdFlashFvOptionalBase),
                     (UINTN) FixedPcdGet32 (PcdFlashFvOptionalSize)
                     );
          if (EFI_ERROR (Status)) {
            DEBUG ((DEBUG_ERROR, "Failed to write FvOptional.fv(%r)\n", Status));
            return;
          }

          Status = WriteBackupFile (
                     mBackUpFileSystemHandle,
                     SYSFW_UPDATE_CURRENT_FVUEFI_BACKUP_FILE_NAME,
                     (UINT8 *) (UINTN) FixedPcdGet32 (PcdFlashFvUefiBootBase),
                     (UINTN) FixedPcdGet32 (PcdFlashFvUefiBootSize)
                     );
          if (EFI_ERROR (Status)) {
            DEBUG ((DEBUG_ERROR, "Failed to write FvUefiBoot.fv(%r)\n", Status));
            return;
          }

          Status = WriteBackupFile (
                     mBackUpFileSystemHandle,
                     SYSFW_UPDATE_CURRENT_FVOS_BACKUP_FILE_NAME,
                     (UINT8 *) (UINTN) FixedPcdGet32 (PcdFlashFvOsBootBase),
                     (UINTN) FixedPcdGet32 (PcdFlashFvOsBootSize)
                     );
          if (EFI_ERROR (Status)) {
            DEBUG ((DEBUG_ERROR, "Failed to write FvOsBoot.fv(%r)\n", Status));
            return;
          }

          ZeroMem (Sha256, SHA256_DIGEST_SIZE);
          CreateSha256Hash ((UINT8 *) (UINTN) Buffer, Length, Sha256);

          Status = gRT->SetVariable (
                         SYSBIOS_UPDATE_CAPSULE_DIGEST_VARIABLE_NAMEN1,
                         &gSysFwUpdateProgressGuid,
                         EFI_VARIABLE_NON_VOLATILE | EFI_VARIABLE_BOOTSERVICE_ACCESS,
                         SHA256_DIGEST_SIZE,
                         Sha256
                         );
          if (EFI_ERROR (Status)) {
            DEBUG ((DEBUG_ERROR, "Failed to sync message digest of the saved Capsule image.\n"));
            return;
          }
        } else {
          //
          // bad update with top swap enabled, top swap only works for mmio read spi, not work for write
          //
          DEBUG ((DEBUG_INFO, "IBB RollBack via PCH SPI protocol. Source Base = %X\n", (UINTN) FixedPcdGet32(PcdFlashIbbRBase)));
          Status = WriteBackupFile (
                     mBackUpFileSystemHandle,
                     SYSBIOS_UPDATE_CAPSULE_DIGEST_VARIABLE_TEMP,
                     (UINT8 *) (UINTN) FixedPcdGet32 (PcdFlashIbbBase),
                     (UINTN) FixedPcdGet32 (PcdFlashIbbSize)
                     );
          Status = ReadBackupFile (mBackUpFileSystemHandle, SYSBIOS_UPDATE_CAPSULE_DIGEST_VARIABLE_TEMP, (void **) &Buffer, &Length);
          Status = SpiFlashUpdate (
                     FlashRegionBios,
                     (UINT32) FixedPcdGet32(PcdFlashIbbOffset),
                     (UINT8 *) Buffer,
                     (UINT32) FixedPcdGet32(PcdFlashIbbSize),
                     NULL,
                     0,
                     0
                     );

          DEBUG ((DEBUG_INFO, "Obb file digest:\n"));
          Status = ReadBackupFile (mBackUpFileSystemHandle, SYSBIOS_UPDATE_CAPSULE_BACKUP_FILE_NAMEN1, (void **) &Buffer, &Length);
          ZeroMem (Sha256, SHA256_DIGEST_SIZE);
          CreateSha256Hash ((UINT8 *) Buffer, Length, Sha256);

          VariableSize = SHA256_DIGEST_SIZE;
          Status = gRT->GetVariable (
                          SYSBIOS_UPDATE_CAPSULE_DIGEST_VARIABLE_NAMEN1,
                          &gSysFwUpdateProgressGuid,
                          NULL,
                          &VariableSize,
                          Buffer1
                          );
          if (EFI_ERROR (Status)) {
            DEBUG ((DEBUG_ERROR, "Failed to get message digest of the saved Capsule image.\n"));
            return;
          }

          if (CompareMem (Buffer1, Sha256, VariableSize) != 0) {
            DEBUG ((DEBUG_ERROR, "Obb image loaded from media is corrupted.\n"));
            return;
          }

          DEBUG ((DEBUG_INFO, "OBB RollBack via PCH SPI protocol\n"));
          Status = SpiFlashUpdate (
                     FlashRegionBios,
                     (UINT32) FixedPcdGet32(PcdFlashObbOffset),
                     (UINT8 *) Buffer,
                     (UINT32) FixedPcdGet32(PcdFlashObbSize),
                     NULL,
                     0,
                     0
                     );
          DEBUG ((DEBUG_INFO, "OBB RollBack in Disk\n"));
          DeleteFile (mBackUpFileSystemHandle, SYSBIOS_UPDATE_CAPSULE_BACKUP_FILE_NAMEN);
        }
        //
        // Delete progress variable
        //
        Status = gRT->SetVariable (
                        SYSFW_UPDATE_PROGRESS_VARIABLE_NAME,
                        &gSysFwUpdateProgressGuid,
                        0,
                        0,
                        NULL
                        );
        DeleteFile (mBackUpFileSystemHandle, SYSBIOS_UPDATE_CAPSULE_BACKUP_FILE_NAMEN);
        DeleteFile (mBackUpFileSystemHandle, SYSFW_UPDATE_NEW_FVADV_BACKUP_FILE_NAME);
        DeleteFile (mBackUpFileSystemHandle, SYSFW_UPDATE_NEW_FVOPT_BACKUP_FILE_NAME);
        DeleteFile (mBackUpFileSystemHandle, SYSFW_UPDATE_NEW_FVUEFI_BACKUP_FILE_NAME);
        DeleteFile (mBackUpFileSystemHandle, SYSFW_UPDATE_NEW_FVOS_BACKUP_FILE_NAME);
        TopSwapControl (FALSE);
        gRT->ResetSystem (EfiResetCold, EFI_SUCCESS, 0, NULL);
      }
    } else {
      //
      // old file not existing --> backup OBB
      //
      ZeroMem (Sha256, SHA256_DIGEST_SIZE);
      DEBUG ((DEBUG_INFO, "Create Capsule digest:\n"));
      CreateSha256Hash ((UINT8 *) (UINTN) FixedPcdGet32 (PcdFlashObbBase), FixedPcdGet32 (PcdFlashObbSize), Sha256);
      //
      // save first obb to esp
      //
      Status = WriteBackupFile (mBackUpFileSystemHandle,
                 SYSBIOS_UPDATE_CAPSULE_BACKUP_FILE_NAMEN1,
                 (UINT8 *) (UINTN) FixedPcdGet32 (PcdFlashObbBase),
                 (UINTN) FixedPcdGet32 (PcdFlashObbSize));

      Status = gRT->SetVariable (
                 SYSBIOS_UPDATE_CAPSULE_DIGEST_VARIABLE_NAMEN1,
                 &gSysFwUpdateProgressGuid,
                 EFI_VARIABLE_NON_VOLATILE | EFI_VARIABLE_BOOTSERVICE_ACCESS,
                 SHA256_DIGEST_SIZE,
                 Sha256
                 );
      if (EFI_ERROR (Status)) {
        DEBUG ((DEBUG_ERROR, "Failed to sync message digest of the saved Capsule image.\n"));
        return;
      }
      //
      // save separate obb 4 FVs into ESP
      //
      Status = WriteBackupFile (
                 mBackUpFileSystemHandle,
                 SYSFW_UPDATE_CURRENT_FVADV_BACKUP_FILE_NAME,
                 (UINT8 *) (UINTN) FixedPcdGet32 (PcdFlashFvAdvancedBase),
                 (UINTN) FixedPcdGet32 (PcdFlashFvAdvancedSize)
                 );
      if (EFI_ERROR (Status)) {
        DEBUG ((DEBUG_ERROR, "Failed to write backup obb FvAdvanced.fv(%r)\n", Status));
        return;
      }
      Status = WriteBackupFile (
                 mBackUpFileSystemHandle,
                 SYSFW_UPDATE_CURRENT_FVOPT_BACKUP_FILE_NAME,
                 (UINT8 *) (UINTN) FixedPcdGet32 (PcdFlashFvOptionalBase),
                 (UINTN) FixedPcdGet32 (PcdFlashFvOptionalSize)
                  );
      if (EFI_ERROR (Status)) {
        DEBUG ((DEBUG_ERROR, "Failed to write backup obb FvOptional.fv(%r)\n", Status));
        return;
      }
      Status = WriteBackupFile (
                 mBackUpFileSystemHandle,
                 SYSFW_UPDATE_CURRENT_FVUEFI_BACKUP_FILE_NAME,
                 (UINT8 *) (UINTN) FixedPcdGet32 (PcdFlashFvUefiBootBase),
                 (UINTN) FixedPcdGet32 (PcdFlashFvUefiBootSize)
                 );
      if (EFI_ERROR (Status)) {
        DEBUG ((DEBUG_ERROR, "Failed to write backup obb FvUefiBoot.fv(%r)\n", Status));
        return;
      }

      Status = WriteBackupFile (
                 mBackUpFileSystemHandle,
                 SYSFW_UPDATE_CURRENT_FVOS_BACKUP_FILE_NAME,
                 (UINT8 *) (UINTN) FixedPcdGet32 (PcdFlashFvOsBootBase),
                 (UINTN) FixedPcdGet32 (PcdFlashFvOsBootSize)
                 );
      if (EFI_ERROR (Status)) {
        DEBUG ((DEBUG_ERROR, "Failed to write backup obb FvOsBoot.fv(%r)\n", Status));
        return;
      }
    }
  }
#endif
  DEBUG((DEBUG_INFO, "ResiliencyCallBackFunction End\n"));
}

/**
  Initialize Resiliency Support for Chasm Falls

  @retval EFI_SUCCESS    Resiliency Support is initialized successfully
  @retval EFI_NOT_FOUND  Resiliency Support is not initialized successfully
**/

EFI_STATUS
EFIAPI
ResiliencyDxeEntryPoint (
  IN EFI_HANDLE           ImageHandle,
  IN EFI_SYSTEM_TABLE     *SystemTable
  )
{
  EFI_STATUS                Status;
  EFI_EVENT                 BeforeEndOfDxeEvent;

  Status                    = EFI_SUCCESS;
  BeforeEndOfDxeEvent       = NULL;

  DEBUG ((DEBUG_INFO, "ResiliencyDxeEntryPoint START\n"));

  DEBUG ((DEBUG_INFO, "Register an Before EndOfDxe Callback Function for Resiliency Health Check \n"));
  Status = gBS->CreateEventEx (
                  EVT_NOTIFY_SIGNAL,
                  TPL_CALLBACK,
                  ResiliencyCallBackFunction,
                  NULL,
                  &gPlatformBeforeEndOfDxeEventGroupGuid,
                  &BeforeEndOfDxeEvent
                  );
  ASSERT_EFI_ERROR (Status);
  if (EFI_ERROR (Status)) {
    DEBUG ((DEBUG_ERROR, "Failed to Register an EndOfDxe CallBack function for Resiliency, Status: %d\n", Status));
  }

  DEBUG ((DEBUG_INFO, "ResiliencyDxeEntryPoint END\n"));
  return Status;
}
