## @file
# Generate RAW data for Version FFS file with 32-bit version, 32-bit lowest
# supported version, and null-terminated Unicode version string.
#
# Copyright (c) 2019, Intel Corporation. All rights reserved.<BR>
# This program and the accompanying materials
# are licensed and made available under the terms and conditions of the BSD License
# which accompanies this distribution.  The full text of the license may be found at
# http://opensource.org/licenses/bsd-license.php
#
# THE PROGRAM IS DISTRIBUTED UNDER THE BSD LICENSE ON AN "AS IS" BASIS,
# WITHOUT WARRANTIES OR REPRESENTATIONS OF ANY KIND, EITHER EXPRESS OR IMPLIED.
#

'''
GenerateVersionFfsRawData
'''

import argparse
import struct

#
# Globals for help information
#
__prog__        = 'GenMicrocodeVersion'
__copyright__   = 'Copyright (c) 2019, Intel Corporation. All rights reserved.'
__description__ = 'Generate RAW data contents with verion, LVS, and Version Stting.\n'

if __name__ == '__main__':
    def convert_arg_line_to_args(arg_line):
        for arg in arg_line.split():
            if not arg.strip():
                continue
            yield arg

    def Validate32BitInteger (Argument):
        try:
            Value = int (Argument, 0)
        except:
            Message = '{Argument} is not a valid integer value.'.format (Argument = Argument)
            raise argparse.ArgumentTypeError (Message)
        if Value < 0:
            Message = '{Argument} is a negative value.'.format (Argument = Argument)
            raise argparse.ArgumentTypeError (Message)
        if Value > 0xffffffff:
            Message = '{Argument} is larger than 32-bits.'.format (Argument = Argument)
            raise argparse.ArgumentTypeError (Message)
        return Value

    #
    # Create command line argument parser object
    #
    parser = argparse.ArgumentParser (
                        prog = __prog__,
                        description = __description__ + __copyright__,
                        conflict_handler = 'resolve',
                        fromfile_prefix_chars = '@'
                        )
    parser.convert_arg_line_to_args = convert_arg_line_to_args

    #
    # Add required arguments for this command
    #
    parser.add_argument("-o", "--output", dest = 'OutputFile', type = argparse.FileType('wb'), required = True,
                        help = "Output filename.")
    parser.add_argument ("--fw-version", dest = 'FwVersion', type = Validate32BitInteger, required = True,
                         help = 'The 32-bit version of the binary payload (e.g. 0x11223344 or 5678).')
    parser.add_argument ("--lsv", dest = 'LowestSupportedVersion', type = Validate32BitInteger, required = True,
                         help = 'The 32-bit lowest supported version of the binary payload (e.g. 0x11223344 or 5678).')
    parser.add_argument ("--fw-version-string", dest = 'VersionString', required = True,
                         help = 'The version string of the binary payload (e.g. "Version 0.1.2.3").')

    #
    # Add optional arguments common to all operations
    #
    parser.add_argument ("-v", "--verbose", dest = 'Verbose', action = "store_true",
                         help = "Turn on verbose output with informational messages printed, including capsule headers and warning messages.")
    parser.add_argument ("-q", "--quiet", dest = 'Quiet', action = "store_true",
                         help = "Disable all messages except fatal errors.")
    parser.add_argument ("--debug", dest = 'Debug', type = int, metavar = '[0-9]', choices = range (0, 10), default = 0,
                         help = "Set debug level")

    #
    # Parse command line arguments
    #
    args, remaining = parser.parse_known_args()

    #
    # Generate buffer with FFS Version file RAW Data
    #
    RawData = b''
    RawData = RawData + struct.pack('<I', args.FwVersion)
    RawData = RawData + struct.pack('<I', args.LowestSupportedVersion)
    RawData = RawData + args.VersionString.encode('utf-16-le')
    RawData = RawData + b'\0\0'

    #
    # Write output file
    #
    args.OutputFile.write(RawData)
    args.OutputFile.close()
