/** @file
  RST One Click switching flow support module.
  Include this in the build if you want to allow automatic switching Optane configuration on your platform

@copyright
  INTEL CONFIDENTIAL
  Copyright 2017 - 2019 Intel Corporation.

  The source code contained or described herein and all documents related to the
  source code ("Material") are owned by Intel Corporation or its suppliers or
  licensors. Title to the Material remains with Intel Corporation or its suppliers
  and licensors. The Material may contain trade secrets and proprietary and
  confidential information of Intel Corporation and its suppliers and licensors,
  and is protected by worldwide copyright and trade secret laws and treaty
  provisions. No part of the Material may be used, copied, reproduced, modified,
  published, uploaded, posted, transmitted, distributed, or disclosed in any way
  without Intel's prior express written permission.

  No license under any patent, copyright, trade secret or other intellectual
  property right is granted to or conferred upon you by disclosure or delivery
  of the Materials, either expressly, by implication, inducement, estoppel or
  otherwise. Any license under such intellectual property rights must be
  express and approved by Intel in writing.

  Unless otherwise agreed by Intel in writing, you may not remove or alter
  this notice or any other notice embedded in Materials by Intel or
  Intel's suppliers or licensors in any way.

  This file contains a 'Sample Driver' and is licensed as such under the terms
  of your license agreement with Intel or your vendor. This file may be modified
  by the user, subject to the additional terms of the license agreement.

@par Specification Reference:
**/

#include <Library/RstCrLib.h>
#include <Library/PchPciBdfLib.h>
#include "RstOneClickEnable.h"

/**
  Entry point for RstOneClickEnable module

  @param ImageHandle  Handle for this image
  @param SystemTable  Pointer to system table
**/
EFI_STATUS
EFIAPI
RstOneClickEnableEntryPoint (
  IN EFI_HANDLE        ImageHandle,
  IN EFI_SYSTEM_TABLE  *SystemTable
  )
{
  RST_CONFIG_VARIABLE  RstConfigVariable;
  UPDATE_OPERATION     UpdateOperation;
  UINTN                Size;
  UINT32               RstConfigVarAttr;
  EFI_STATUS           Status;
  RST_MODE             RstMode;

  DEBUG ((DEBUG_INFO, "RstOneClickEnableEntryPoint() Start\n"));

  RstMode = PchGetSupportedRstMode ();

  if (RstMode == RstUnsupported) {
    return EFI_UNSUPPORTED;
  }

  gRS = SystemTable->RuntimeServices;

  Size = sizeof (RST_CONFIG_VARIABLE);
  Status = gRS->GetVariable (
                  RstConfigVariableName,
                  &gRstConfigVariableGuid,
                  &RstConfigVarAttr,
                  &Size,
                  &RstConfigVariable
                  );

  if (EFI_ERROR (Status)) {
    DEBUG ((DEBUG_INFO, "Failed to fetch RstConfigVariable\n Atempting to create RstConfigVariable\n"));
    Status = CreateRstConfigVariable ();
    if (EFI_ERROR (Status)) {
      DEBUG ((DEBUG_INFO, "Failed to create RstConfigVariable\n"));
      return Status;
    } else {
      DEBUG ((DEBUG_INFO, "RstConfigVariable created successfully\n"));
      return EFI_SUCCESS;
    }
  }

  UpdateOperation = GetRequiredUpdateOperation (RstConfigVariable);

  //
  //  CreateRstConfigVariable can also be used for cleaning the variable
  //
  Status = CreateRstConfigVariable ();
  if (EFI_ERROR (Status)) {
    DEBUG ((DEBUG_INFO, "Failed to clean RstConfigVariable\n"));
    return Status;
  }

  if (UpdateOperation != None) {
    Status = UpdateSystemConfiguration (UpdateOperation);
    if (EFI_ERROR (Status)) {
      DEBUG ((DEBUG_INFO, "Failed to update system configuration\n"));
      return Status;
    }
  }

  return EFI_SUCCESS;
}

/**
  Returns required update operation

  @param  RstConfigVariable  An instance of RST_CONFIG_VARIABLE passed from OS

  @retval  UPDATE_OPERATION  Required update operation to perform
**/
UPDATE_OPERATION
GetRequiredUpdateOperation (
  IN RST_CONFIG_VARIABLE  RstConfigVariable
  )
{
  if (RstConfigVariable.ModeSwitch) {
    return RaidSwitch;
  } else if (RstConfigVariable.RemapEnable){
    return RemapEnable;
  } else {
    return None;
  }

}

/**
  Creates an instance of RST_CONFIG_VARIABLE.
  Since this function initializes variable to 0 it can also be used to reseting the variable

  @retval EFI_STATUS  Returns status of SetVariable runtime service
**/
EFI_STATUS
CreateRstConfigVariable (
  VOID
  )
{
  RST_CONFIG_VARIABLE  RstConfigVariable;
  UINT32  RstConfigVarAttr;
  EFI_STATUS  Status;

  ZeroMem (&RstConfigVariable, sizeof (RstConfigVariable));

  RstConfigVarAttr = EFI_VARIABLE_NON_VOLATILE | EFI_VARIABLE_BOOTSERVICE_ACCESS | EFI_VARIABLE_RUNTIME_ACCESS;

  Status = gRS->SetVariable (
                  RstConfigVariableName,
                  &gRstConfigVariableGuid,
                  RstConfigVarAttr,
                  sizeof (RstConfigVariable),
                  &RstConfigVariable
                  );

  return Status;
}

/**
  Updates system configuration based on RST_CONFIG_VARIABLE recieved from OS

  @param RstConfigVariable  A pointer to an instance of RST_CONFIG_VARIABLE received from OS

  @retval EFI_STATUS  Returns EFI_SUCCESS only if update wasn't necessary, otherwise returns last error
**/
EFI_STATUS
UpdateSystemConfiguration (
  IN UPDATE_OPERATION  UpdateOperation
  )
{
  EFI_STATUS        Status;

  Status = UpdatePchSetupVariable (UpdateOperation);
  if (EFI_ERROR (Status)) {
    DEBUG((DEBUG_INFO, "Failed to update PchSetup variable\n"));
    return Status;
  }
  gRS->ResetSystem (
         EfiResetWarm,
         EFI_SUCCESS,
         0,
         NULL
         );

  return EFI_SUCCESS;

}

/**
  Updates PchSetup variable based on information passed in RST_CONFIG_VARIABLE

  @retval EFI_STATUS  Returns EFI_SUCCESS if PchSetup was updated successfully, returns last error otherwise
**/
EFI_STATUS
UpdatePchSetupVariable (
  IN UPDATE_OPERATION  UpdateOperation
  )
{
  PCH_SETUP   PchSetup;
  UINT32      PchSetupAttributes;
  UINTN       Size;
  EFI_STATUS  Status;

  Size = sizeof (PCH_SETUP);
  Status = gRS->GetVariable (
                  L"PchSetup",
                  &gPchSetupVariableGuid,
                  &PchSetupAttributes,
                  &Size,
                  &PchSetup
                  );
  if (EFI_ERROR (Status)) {
    return Status;
  }

  switch (UpdateOperation) {
    case RaidSwitch:
      SwitchSataControllerToRaid (&PchSetup);
      break;
    case RemapEnable:
      if (IsIntegratedSataControllerInRaidMode ()) {
        EnableRemapOnAllCapableRp (&PchSetup);
      } else {
        DEBUG((DEBUG_ERROR, "Integrated SATA not in RAID mode, can't enable remapping\n"));
      }
      break;
    case None:
    default:
      break;
  }

  Status = gRS->SetVariable (
                  L"PchSetup",
                  &gPchSetupVariableGuid,
                  PchSetupAttributes,
                  Size,
                  &PchSetup
                  );
  if (EFI_ERROR (Status)) {
    return Status;
  }

  return EFI_SUCCESS;
}

/**
  Checks if the integrated SATA controller is in RAID mode

  @retval BOOLEAN  True if SATA controller is in RAID mode
**/
BOOLEAN
IsIntegratedSataControllerInRaidMode (
  VOID
  )
{
  UINT8  SataMode;

  SataMode = PciSegmentRead8 (SataPciCfgBase (SATA_1_CONTROLLER_INDEX) + R_PCI_SCC_OFFSET);

  if (SataMode == PCI_CLASS_MASS_STORAGE_RAID) {
    return TRUE;
  } else {
    return FALSE;
  }
}

/**
  Enables NVMe remapping on all remap capable root ports

  @param PchSetup  An instance of PchSetup to be updated
**/
VOID
EnableRemapOnAllCapableRp (
  IN OUT PCH_SETUP  *PchSetup
  )
{
  PCIE_STORAGE_INFO_HOB  *PcieStorageInfoHob;
  VOID                   *Hob;
  UINT32                 RpNumber;

  DEBUG((DEBUG_INFO, "Enabling remapping on all capable root ports\n"));

  Hob = NULL;
  PcieStorageInfoHob = NULL;
  Hob = GetFirstGuidHob (&gPchPcieStorageDetectHobGuid);
  if (Hob == NULL) {
    DEBUG((DEBUG_ERROR, "Failed to get storage info hob\n"));
    return;
  }

  PcieStorageInfoHob = (PCIE_STORAGE_INFO_HOB*) GET_GUID_HOB_DATA (Hob);

  for (RpNumber = 0; RpNumber < GetPchMaxPciePortNum (); RpNumber++) {
    if (CheckIfRootPortIsRemapCapable (PcieStorageInfoHob, RpNumber)) {
      //
      //  Check if remap has not been already enabled on this PCIe controller
      //
      if (RpNumber > 1 && PchSetup->RstPcieRemapEnabled[RpNumber - 2] != 1) {
        DEBUG ((DEBUG_INFO, "Enabling remapping on RP# %d\n", RpNumber));
        PchSetup->RstPcieRemapEnabled[RpNumber] = 1;
      }
    }
  }

}

/**
  Checks if given root port is remap capable based on information in SETUP_VOLATILE_DATA

  @param  PcieStorageInfoHob  A pointer to an instance of SETUP_VOLATILE_DATA
  @param  RpNumber            Root port number to check

  @retval  BOOLEAN  Returns TRUE if root port is remap capable, returns false otherwise
**/
BOOLEAN
CheckIfRootPortIsRemapCapable (
  PCIE_STORAGE_INFO_HOB  *PcieStorageInfoHob,
  UINT32  RpNumber
  )
{
  if (PcieStorageInfoHob->PcieStorageLinkWidth[RpNumber] == 0 || PcieStorageInfoHob->PcieStorageProgrammingInterface[RpNumber] == 0 || PcieStorageInfoHob->RstCycleRouterMap[RpNumber / 4] == RST_PCIE_STORAGE_CR_INVALID) {
    return FALSE;
  }

  return TRUE;
}

/**
  Switches SATA controller to RAID mode

  @param PchSetup  A pointer to PCH_SETUP instance
**/
VOID
SwitchSataControllerToRaid (
  OUT  PCH_SETUP  *PchSetup
  )
{
  PchSetup->SataInterfaceMode = SATA_MODE_RAID;
  DEBUG ((DEBUG_INFO, "Sata controller interface chaged to RAID\n"));

}

