/** @file
  This file is SampleCode of the library for Intel CPU PEI Policy initialization.

@copyright
  INTEL CONFIDENTIAL
  Copyright 2009 - 2020 Intel Corporation.

  The source code contained or described herein and all documents related to the
  source code ("Material") are owned by Intel Corporation or its suppliers or
  licensors. Title to the Material remains with Intel Corporation or its suppliers
  and licensors. The Material may contain trade secrets and proprietary and
  confidential information of Intel Corporation and its suppliers and licensors,
  and is protected by worldwide copyright and trade secret laws and treaty
  provisions. No part of the Material may be used, copied, reproduced, modified,
  published, uploaded, posted, transmitted, distributed, or disclosed in any way
  without Intel's prior express written permission.

  No license under any patent, copyright, trade secret or other intellectual
  property right is granted to or conferred upon you by disclosure or delivery
  of the Materials, either expressly, by implication, inducement, estoppel or
  otherwise. Any license under such intellectual property rights must be
  express and approved by Intel in writing.

  Unless otherwise agreed by Intel in writing, you may not remove or alter
  this notice or any other notice embedded in Materials by Intel or
  Intel's suppliers or licensors in any way.

  This file contains a 'Sample Driver' and is licensed as such under the terms
  of your license agreement with Intel or your vendor. This file may be modified
  by the user, subject to the additional terms of the license agreement.

@par Specification Reference:
**/
#include "PeiCpuPolicyUpdate.h"
#include <Library/SiPolicyLib.h>
#include <Library/ConfigBlockLib.h>
#include <Library/PmcLib.h>
#include <Library/PchCycleDecodingLib.h>
#include <Library/PciSegmentLib.h>
#include <Register/PchRegs.h>
#include <Library/SpiLib.h>
#include <Library/SpiAccessLib.h>
#include <Library/PreSiliconEnvDetectLib.h>
#if FixedPcdGet8(PcdFspModeSelection) == 1
#include <FspmUpd.h>
#endif
#include <Ppi/Spi.h>
#include <Library/HobLib.h>
#include <Library/FirmwareBootMediaLib.h>
#include <PolicyUpdateMacro.h>
#include <Guid/SysFwUpdateProgress.h>
#include <Register/CommonMsr.h>
#include <Register/TglMsr.h>

#define GET_OCCUPIED_SIZE(ActualSize, Alignment) \
  ((ActualSize) + (((Alignment) - ((ActualSize) & ((Alignment) - 1))) & ((Alignment) - 1)))

/**
  Calculate the smallest supported PRMRR size that will be
  reported in response to a given EPC size request from OS

  @param[in] EpcSize           - EPC size

  @retval    UINT32            - PRMRR size that is just bigger than the requested EPC
**/
UINT32
ConvertEpcToPrmrr (
  UINT32 EpcSize
  )
{
  UINT32 EpcAlignedPowerOfTwo;
  UINT32 SupportedPrmrr;

  if (EpcSize == 0) {
    return 0;
  }

  ///
  /// Round EPC size to the nearest power of two
  ///
  EpcAlignedPowerOfTwo = GetPowerOfTwo32(EpcSize);

  ///
  /// If given EpcSize is not power of two
  ///
  if (EpcAlignedPowerOfTwo < EpcSize) {
    EpcAlignedPowerOfTwo = (UINT32) (UINTN) LShiftU64(EpcAlignedPowerOfTwo,1);
  }

  /*
    Bits 0 through 31 indicate which sizes are supported. Each bit
    represents one size. A bit at position n indicates whether
    size 2^(n+20) bytes is supported
  */
  SupportedPrmrr = (UINT32) (UINTN) LShiftU64 (AsmReadMsr64 (TGL_MSR_PRMRR_VALID_CONFIG),20);

  ///
  /// Removing sizes that are lower then EPC round size
  ///
  SupportedPrmrr = SupportedPrmrr & ~(EpcAlignedPowerOfTwo - 1);

  ///
  /// If EPC rounding is bigger than highest supported PRMRR size,
  /// then EPC size that was set by OS is not supported
  ///
  if (SupportedPrmrr < EpcAlignedPowerOfTwo) {
    DEBUG ((DEBUG_INFO, "ConvertEpcToPrmrr: Requested EPC size (rounded up to be aligned to power of two) by OS 0x%016llX, is bigger than supported PRMRRs size 0x%016llX\n" ,EpcAlignedPowerOfTwo ,SupportedPrmrr));
    return 0;
  }

  ///
  /// If PRMRR supported size is bigger then EPC size rounded to two
  /// then return the next lowest PRMRR size supported
  ///
  if ((SupportedPrmrr & EpcAlignedPowerOfTwo) == 0) {
    return (UINT32) (UINTN) (LShiftU64(BIT0, LowBitSet32(SupportedPrmrr)));
  }
  return (SupportedPrmrr & EpcAlignedPowerOfTwo);
}

/**
  Function will update the policy for PRMRR size

  @param[in]  VariableServices     Variable Service Ppi
  @param[in]  CpuSetup             The pointer of CPU_SETUP
  @param[in]  FspmUpd              A VOID pointer
  @param[in, out]  CpuSecurityPreMemConfig - will update
                   CpuSecurityPreMemConfig->PrmrrSize
**/
VOID
UpdatePrmrrSize (
  IN     EFI_PEI_READ_ONLY_VARIABLE2_PPI *VariableServices,
  IN     CPU_SETUP                       *CpuSetup,
  IN OUT VOID                            *FspmUpd,
  IN OUT CPU_SECURITY_PREMEM_CONFIG      *CpuSecurityPreMemConfig
  )
{
  /**
   If PrmrrSize is 0 then it is not supported or PRMRR size
   is 0 in setup options then allocate a PRMRR memory size of 1MB,
   if required only for C6DRAM powergating feature.
   **/
#if FixedPcdGet8(PcdFspModeSelection) == 1
  if (((FSPM_UPD *) FspmUpd)->FspmConfig.PrmrrSize == 0 &&
      ((FSPM_UPD *) FspmUpd)->FspmConfig.EnableC6Dram &&
      IsC6dramSupported ()) {
#else
  if (CpuSecurityPreMemConfig->PrmrrSize == 0 &&
      CpuSecurityPreMemConfig->EnableC6Dram &&
      IsC6dramSupported ()) {
#endif
    UPDATE_POLICY (((FSPM_UPD *) FspmUpd)->FspmConfig.PrmrrSize, CpuSecurityPreMemConfig->PrmrrSize, SIZE_1MB);
  }

  ///
  /// Check if PRMRR size is not supported.
  /// Bits 0 through 31 indicate which sizes are supported. Each bit represents one size.
  /// A bit at position n indicates whether size 2^(n+20) bytes is supported
  ///
#if FixedPcdGet8(PcdFspModeSelection) == 1
  if ((LShiftU64 (AsmReadMsr64 (TGL_MSR_PRMRR_VALID_CONFIG), 20) & ((FSPM_UPD *) FspmUpd)->FspmConfig.PrmrrSize) == 0x0) {
    DEBUG ((DEBUG_INFO, "InitializePrm: PRMRR setup size 0x%X\n", ((FSPM_UPD *) FspmUpd)->FspmConfig.PrmrrSize));
#else
  if ((LShiftU64 (AsmReadMsr64 (TGL_MSR_PRMRR_VALID_CONFIG), 20) & CpuSecurityPreMemConfig->PrmrrSize) == 0x0) {
    DEBUG ((DEBUG_INFO, "InitializePrm: PRMRR setup size 0x%X\n", CpuSecurityPreMemConfig->PrmrrSize));
#endif
    UPDATE_POLICY (((FSPM_UPD *) FspmUpd)->FspmConfig.PrmrrSize, CpuSecurityPreMemConfig->PrmrrSize, 0);
  }
}

/**
  Get the BIOS Guard Module pointer.

  @param[in, out] ModulePtr  - Input is a NULL pointer,
                               and output points BIOS Guard module address if found.
  @param[out]     ModuleSize - UINT32 Input Output the BIOS Guard module size

  @retval EFI_SUCCESS        - BIOS Guard Module found.
  @retval EFI_NOT_FOUND      - BIOS Guard Module size and/or Address equal to 0.
  @retval Others             - BIOS Guard Module not found.
**/
EFI_STATUS
UpdateBiosGuardModulePtr (
  IN OUT EFI_PHYSICAL_ADDRESS   *ModulePtr,
  OUT    UINT32                 *ModuleSize
  )
{
  EFI_STATUS                    Status;
  EFI_FIRMWARE_VOLUME_HEADER    *FvHeader;
  EFI_FV_FILE_INFO              BiosGuardFileInfo;
  EFI_PEI_FILE_HANDLE           *FfsHeader;
  EFI_GUID                      *BiosGuardModuleGuidPtr;
  UINT32                        BiosGuardModuleAddr;
  UINTN                         BiosGuardModuleSize;
  CPU_STEPPING                  CpuSteppingId;
  CPU_FAMILY                    CpuFamily;

  DEBUG ((DEBUG_INFO, "Update BIOS Guard Module Pointer and Size\n"));

  Status     = EFI_SUCCESS;
  BiosGuardModuleAddr = 0;
  BiosGuardModuleSize = 0;
  CpuSteppingId = GetCpuStepping ();
  CpuFamily     = GetCpuFamily();
  ///
  /// Locate Firmware Volume header
  ///
  FvHeader = (EFI_FIRMWARE_VOLUME_HEADER *) (UINTN) FixedPcdGet32 (PcdFlashFvFirmwareBinariesBase);

  if ((CpuFamily == EnumCpuTglUltUlx) && (CpuSteppingId == EnumTglA0)) {
    DEBUG((DEBUG_INFO, "BIOS Guard ACM for TGL U/Y Ax Stepping is selected \n"));
    BiosGuardModuleGuidPtr = &gBiosGuardModuleGuid;
  } else if (IsSimicsEnvironment ()) {
    DEBUG((DEBUG_INFO, "BIOS Guard ACM for TGL Simics Stepping is selected \n"));
    BiosGuardModuleGuidPtr = &gBiosGuardModuleGuid;
  } else {
    DEBUG((DEBUG_INFO, "BIOS Guard ACM for TGL-H, TGLU/Y B0 Stepping is selected \n"));
    BiosGuardModuleGuidPtr = &gBiosGuardModuleGuidBx;
  }

  Status = PeiServicesFfsFindFileByName (BiosGuardModuleGuidPtr, FvHeader, (VOID **)&FfsHeader);

  if (EFI_ERROR (Status)) {
    DEBUG ((DEBUG_ERROR, "Error returned by PeiServicesFfsFindFileByName while retrieving BIOS Guard Module Address\n"));
    ASSERT_EFI_ERROR (Status);
    return Status;
  }

  BiosGuardModuleAddr = (UINT32)((UINT8 *) FfsHeader + sizeof (EFI_FFS_FILE_HEADER));
  DEBUG ((DEBUG_ERROR, "BIOS Guard Module Location: 0x%X\n", BiosGuardModuleAddr));

  Status = PeiServicesFfsGetFileInfo (FfsHeader, &BiosGuardFileInfo);
  if (EFI_ERROR (Status)) {
    DEBUG ((DEBUG_ERROR, "Error returned by PeiServicesFfsGetFileInfo while retrieving BIOS Guard Module Size\n"));
  }

  BiosGuardModuleSize = (UINTN) BiosGuardFileInfo.BufferSize;
  DEBUG(( DEBUG_ERROR, "BIOS Guard Module size: 0x%X\n", BiosGuardModuleSize));

  if ((BiosGuardModuleAddr == 0) || (BiosGuardModuleSize == 0)) {
    DEBUG ((DEBUG_ERROR, "BiosGuardModuleSize = 0x%X and/or BiosGuardModule = 0x%X equal to zero\n", BiosGuardModuleAddr, BiosGuardModuleSize));
    ASSERT_EFI_ERROR (Status);
    return EFI_NOT_FOUND;
  }

  *ModulePtr  = BiosGuardModuleAddr;
  *ModuleSize = BiosGuardModuleSize;
  return Status;
}

/**
  Check if BIOS Guard Module is Debug Signed

  @param[in] ModulePtr - Input is the BIOS Guard module address.

  @retval TRUE  - BIOS Guard Module is Debug Signed
  @retval FALSE - BIOS Guard Module is Production Signed
**/
BOOLEAN
IsBiosGuardModuleDebugSigned2 (
  IN EFI_PHYSICAL_ADDRESS   ModulePtr
  )
{
  UINT16                  RevisionIdMajor;

  ///
  /// Offset from beginning 142 Bytes (Size of BIOS HEADER (128 Bytes) + Offset of Internal Header Major Revision ID (14 Bytes))
  ///
  RevisionIdMajor = *(UINT16*)(UINTN)(ModulePtr + BIOSGUARD_INTERNAL_HEADER_MODULE_ID_OFFSET);
  DEBUG ((DEBUG_INFO, "BIOS Guard Revision ID (Major) = %x\n", RevisionIdMajor));

  ///
  /// Check if Module is Debug Signed. Bit 15 being set means it is Debug Signed.
  ///
  return ((RevisionIdMajor & BIT15) != 0) ? TRUE : FALSE;
}

/**
  Create and initialize BIOS Guard HOB

  @retval  EFI_SUCCESS  HOB Creating successful.
  @retval  Others       HOB Creating failed.
**/
EFI_STATUS
BiosGuardHobInit (
  VOID
  )
{
  EFI_STATUS                       Status;
  BIOSGUARD_HOB                    *BiosGuardHobPtr;
  UINT8                            BgpdtPkeySlot0[] = { 0x6a, 0x94, 0x81, 0x87, 0xfd, 0x8c, 0x51, 0x0a, 0x2a, 0x74, 0x9c, 0x7b, 0xe3, 0xf7, 0x5c, 0x3a, 0xbf, 0xce, 0x60, 0x4f, 0x68, 0xeb, 0xc6, 0x8f, 0x69, 0x98, 0x8c, 0xc5, 0xdb, 0xe3, 0x29, 0x70 };
  UINT8                            BgpdtPkeySlot1[] = { 0x1c, 0x51, 0x3a, 0xc5, 0x1c, 0x3d, 0x63, 0x10, 0xef, 0xed, 0xfe, 0xc5, 0xf3, 0x4b, 0xf2, 0xe0, 0x9b, 0x22, 0xe8, 0x06, 0xab, 0xd3, 0x19, 0x2e, 0xfa, 0xfb, 0x6e, 0xd9, 0x36, 0x0d, 0x68, 0x18 };
  UINT8                            BgpdtPkeySlot2[] = { 0x19, 0x0b, 0x33, 0xf8, 0xde, 0x3a, 0xa7, 0x9b, 0x57, 0xad, 0xb2, 0x45, 0x86, 0x0e, 0x7f, 0x0e, 0x40, 0x62, 0x80, 0x22, 0x8f, 0x04, 0x92, 0xec, 0x87, 0x44, 0x81, 0xd9, 0xef, 0xed, 0x9f, 0xa3 };
  UINT8                            EcStatus;
  UINT8                            PlatIdStr[] = "TIGERLAKE";
  PCH_SPI_PPI                      *SpiPpi;
  UINT32                           BaseAddr;
  UINT32                           RegionSize;
  UINT32                           Index = 0;

  RegionSize = 0;
  BaseAddr   = 0;

  Status = PeiServicesLocatePpi (
             &gPchSpiPpiGuid,
             0,
             NULL,
             (VOID **)&SpiPpi
             );

  DEBUG ((DEBUG_INFO, "PlatformInit: LocatePpi gPchSpiPpiGuid = %r\n", Status));
  if (!EFI_ERROR (Status)) {
    //
    // Find the base address for flash
    //
    Status = SpiPpi->GetRegionAddress (SpiPpi, FlashRegionBios, &BaseAddr, &RegionSize);
    DEBUG ((DEBUG_INFO, "BIOS Region: %r - Start Address: 0x%x - Size: 0x%x\n", Status, BaseAddr, RegionSize));
    if (EFI_ERROR (Status)) {
      DEBUG ((DEBUG_ERROR, "Unable to find base address for flash\n"));
      ASSERT_EFI_ERROR (Status);
      return Status;
    }
  }

  Status = PeiServicesCreateHob (EFI_HOB_TYPE_GUID_EXTENSION, sizeof (BIOSGUARD_HOB), (VOID **)&BiosGuardHobPtr);
  if (EFI_ERROR (Status)) {
    DEBUG ((DEBUG_ERROR, "Unable to create BIOS Guard Hob Pointer\n"));
    ASSERT_EFI_ERROR (Status);
    return Status;
  }

  ZeroMem (&(BiosGuardHobPtr->Bgpdt), (sizeof (BIOSGUARD_HOB) - sizeof (EFI_HOB_GUID_TYPE)));
  BiosGuardHobPtr->EfiHobGuidType.Name = gBiosGuardHobGuid;

  BiosGuardHobPtr->Bgpdt.BgpdtMajVer = BGPDT_MAJOR_VERSION;
  BiosGuardHobPtr->Bgpdt.BgpdtMinVer = BGPDT_MINOR_VERSION;
  BiosGuardHobPtr->Bgpdt.BiosSvn     = 0x24220000;

  BiosGuardHobPtr->Bgpdt.LastSfam = 0;
  BiosGuardHobPtr->Bgpdt.SfamData[0].FirstByte = BaseAddr + FixedPcdGet32 (PcdFlashNvStorageSize);
  BiosGuardHobPtr->Bgpdt.SfamData[0].LastByte  = (BaseAddr + RegionSize) - 1;

  CopyMem (&BiosGuardHobPtr->Bgpdt.PkeySlot0[0], &BgpdtPkeySlot0[0], sizeof (BgpdtPkeySlot0));
  CopyMem (&BiosGuardHobPtr->Bgpdt.PkeySlot1[0], &BgpdtPkeySlot1[0], sizeof (BgpdtPkeySlot1));
  CopyMem (&BiosGuardHobPtr->Bgpdt.PkeySlot2[0], &BgpdtPkeySlot2[0], sizeof (BgpdtPkeySlot2));
  BiosGuardHobPtr->BiosGuardLog.LastPage       = MAX_BIOSGUARD_LOG_PAGE - 1;
  BiosGuardHobPtr->BiosGuardLog.LoggingOptions = BIOSGUARD_LOG_OPT_DEBUG | BIOSGUARD_LOG_OPT_FLASH_ERROR | BIOSGUARD_LOG_OPT_FLASH_ERASE | BIOSGUARD_LOG_OPT_FLASH_WRITE | BIOSGUARD_LOG_OPT_BRANCH_TRACE | BIOSGUARD_LOG_OPT_STEP_TRACE;
  BiosGuardHobPtr->BiosGuardLog.Version        = BIOSGUARD_LOG_VERSION;

  if (PcdGetBool (PcdEcPresent)) {
    DEBUG ((DEBUG_INFO, "EC is Present\n"));
    BiosGuardHobPtr->Bgpdt.BiosGuardAttr      |= EnumEcPresent;
    BiosGuardHobPtr->Bgpdt.EcCmd               = EC_C_PORT;
    BiosGuardHobPtr->Bgpdt.EcData              = EC_D_PORT;
    Status = SendEcCommand (0xB0);
    if (Status == EFI_SUCCESS) {
      Status = ReceiveEcData (&EcStatus);
      if (((EcStatus & 0xF8) == 0) && (EcStatus & BIT0)) {
        if (((EcStatus & (BIT2 | BIT1)) == (BIT2 | BIT1)) || (((EcStatus & (BIT2 | BIT1)) == 0))) {
          DEBUG ((DEBUG_INFO, "EC FW supports BiosGuard\n"));
          BiosGuardHobPtr->Bgpdt.BiosGuardAttr |= EnumEcBiosGuardProtected;
          BiosGuardHobPtr->Bgpdt.EcCmdGetSvn   = 0xB3;
          BiosGuardHobPtr->Bgpdt.EcCmdOpen     = 0xB4;
          BiosGuardHobPtr->Bgpdt.EcCmdClose    = 0xB5;
          BiosGuardHobPtr->Bgpdt.EcCmdPortTest = 0xB6;
          BiosGuardHobPtr->BgupHeader.EcSvn    = 0x00010000;
        } else {
          DEBUG ((DEBUG_ERROR, "EC FW Error\n"));
        }
      } else {
        DEBUG ((DEBUG_WARN, "EC FW do not support BiosGuard\n"));
      }
    } else {
      DEBUG ((DEBUG_ERROR, "SendEcCommand (EcCmdDiscovery) Failed\n"));
    }
  }
  BiosGuardHobPtr->Bgpdt.BiosGuardAttr |= EnumFlashwearoutProtection;
  BiosGuardHobPtr->Bgpdt.BiosGuardAttr |= EnumFtuEnable;

  BiosGuardHobPtr->BiosGuardModulePtr  = 0;
  BiosGuardHobPtr->BiosGuardModuleSize = 0;

  Status = UpdateBiosGuardModulePtr (&BiosGuardHobPtr->BiosGuardModulePtr, &BiosGuardHobPtr->BiosGuardModuleSize);
  if (EFI_ERROR (Status)) {
    DEBUG ((DEBUG_ERROR, "UpdateBiosGuardModulePtr not found\n"));
    ASSERT_EFI_ERROR (Status);
    return Status;
  }

  DEBUG ((DEBUG_INFO, "------------------ BIOS Guard Module ------------------\n"));
  BiosGuardHobPtr->Bgpdt.BgModSvn = *(UINT32*)(UINTN)(BiosGuardHobPtr->BiosGuardModulePtr + BIOSGUARD_INTERNAL_HEADER_SVN_OFFSET);
  DEBUG ((DEBUG_INFO, " BIOS Guard Module:: SVN : 0x%X\n", *(UINT32*)(UINTN)(BiosGuardHobPtr->BiosGuardModulePtr + BIOSGUARD_INTERNAL_HEADER_SVN_OFFSET)));
  DEBUG ((DEBUG_INFO, " BIOS Guard Module:: Product ID : 0x%X\n", *(UINT16*)(UINTN)(BiosGuardHobPtr->BiosGuardModulePtr + BIOSGUARD_PRODUCT_ID_OFFSET)));
  DEBUG ((DEBUG_INFO, " BIOS Guard Module:: Build Number : 0x%X\n", *(UINT16*)(UINTN)(BiosGuardHobPtr->BiosGuardModulePtr + BIOSGUARD_BUILD_NUMBER_OFFSET)));

  BiosGuardHobPtr->Bgpdt.BgpdtSize = (sizeof (BGPDT) - sizeof (BiosGuardHobPtr->Bgpdt.SfamData) + ((BiosGuardHobPtr->Bgpdt.LastSfam + 1) * sizeof (SFAM_DATA)));
  CopyMem (&BiosGuardHobPtr->Bgpdt.PlatId[0], &PlatIdStr[0], sizeof (PlatIdStr));

  BiosGuardHobPtr->BgupHeader.Version        = BGUP_HDR_VERSION;
  CopyMem (&BiosGuardHobPtr->BgupHeader.PlatId[0], &PlatIdStr[0], sizeof (PlatIdStr));
  BiosGuardHobPtr->BgupHeader.PkgAttributes  = 0;
  BiosGuardHobPtr->BgupHeader.PslMajorVer    = PSL_MAJOR_VERSION;
  BiosGuardHobPtr->BgupHeader.PslMinorVer    = PSL_MINOR_VERSION;
  BiosGuardHobPtr->BgupHeader.BiosSvn        = BiosGuardHobPtr->Bgpdt.BiosSvn;
  BiosGuardHobPtr->BgupHeader.VendorSpecific = 0x808655AA;

  DEBUG ((DEBUG_INFO, "------------------ BIOS GUARD Config ------------------\n"));
  DEBUG ((DEBUG_INFO, " BIOS_GUARD_CONFIG:: BGUP_HEADER : Version : 0x%X\n", BiosGuardHobPtr->BgupHeader.Version));
  DEBUG ((DEBUG_INFO, " BIOS_GUARD_CONFIG:: BGUP_HEADER : PlatId[16] :\n"));
  for (Index = 0; Index < 16; Index++) {
      DEBUG ((DEBUG_INFO, " 0x%X", BiosGuardHobPtr->BgupHeader.PlatId[Index]));
  }

  DEBUG ((DEBUG_INFO, " \n"));
  DEBUG ((DEBUG_INFO, " BIOS_GUARD_CONFIG:: BGUP_HEADER : PkgAttributes : 0x%X\n", BiosGuardHobPtr->BgupHeader.PkgAttributes));
  DEBUG ((DEBUG_INFO, " BIOS_GUARD_CONFIG:: BGUP_HEADER : PslMajorVer : 0x%X\n", BiosGuardHobPtr->BgupHeader.PslMajorVer));
  DEBUG ((DEBUG_INFO, " BIOS_GUARD_CONFIG:: BGUP_HEADER : PslMinorVer : 0x%X\n", BiosGuardHobPtr->BgupHeader.PslMinorVer));
  DEBUG ((DEBUG_INFO, " BIOS_GUARD_CONFIG:: BGUP_HEADER : ScriptSectionSize : 0x%X\n", BiosGuardHobPtr->BgupHeader.ScriptSectionSize));
  DEBUG ((DEBUG_INFO, " BIOS_GUARD_CONFIG:: BGUP_HEADER : DataSectionSize : 0x%X\n", BiosGuardHobPtr->BgupHeader.DataSectionSize));
  DEBUG ((DEBUG_INFO, " BIOS_GUARD_CONFIG:: BGUP_HEADER : BiosSvn : 0x%X\n", BiosGuardHobPtr->BgupHeader.BiosSvn));
  DEBUG ((DEBUG_INFO, " BIOS_GUARD_CONFIG:: BGUP_HEADER : EcSvn : 0x%X\n", BiosGuardHobPtr->BgupHeader.EcSvn));
  DEBUG ((DEBUG_INFO, " BIOS_GUARD_CONFIG:: BGUP_HEADER : VendorSpecific : 0x%X\n", BiosGuardHobPtr->BgupHeader.VendorSpecific));
  DEBUG ((DEBUG_INFO, " BIOS_GUARD_CONFIG:: BGPDT : BgpdtSize : 0x%X\n", BiosGuardHobPtr->Bgpdt.BgpdtSize));
  DEBUG ((DEBUG_INFO, " BIOS_GUARD_CONFIG:: BGPDT : BgpdtMajVer : 0x%X\n", BiosGuardHobPtr->Bgpdt.BgpdtMajVer));
  DEBUG ((DEBUG_INFO, " BIOS_GUARD_CONFIG:: BGPDT : BgpdtMinVer : 0x%X\n", BiosGuardHobPtr->Bgpdt.BgpdtMinVer));
  DEBUG ((DEBUG_INFO, " BIOS_GUARD_CONFIG:: BGPDT : PlatId[16] :\n"));
  for (Index = 0; Index < 16; Index++) {
      DEBUG ((DEBUG_INFO, " 0x%X", BiosGuardHobPtr->Bgpdt.PlatId[Index]));
  }

  DEBUG ((DEBUG_INFO, " \n"));
  DEBUG ((DEBUG_INFO, " BIOS_GUARD_CONFIG:: BGPDT : PkeySlot0[32] :\n"));
  for (Index = 0; Index < 32; Index++) {
    if ((Index == 15) || (Index == 31)) {
      DEBUG ((DEBUG_INFO, " 0x%X\n", BiosGuardHobPtr->Bgpdt.PkeySlot0[Index]));
    } else {
      DEBUG ((DEBUG_INFO, " 0x%X", BiosGuardHobPtr->Bgpdt.PkeySlot0[Index]));
    }
  }

  DEBUG ((DEBUG_INFO, " \n"));
  DEBUG ((DEBUG_INFO, " BIOS_GUARD_CONFIG:: BGPDT : PkeySlot1[32] :\n"));
  for (Index = 0; Index < 32; Index++) {
    if ((Index == 15) || (Index == 31)) {
      DEBUG ((DEBUG_INFO, " 0x%X\n", BiosGuardHobPtr->Bgpdt.PkeySlot1[Index]));
    } else {
      DEBUG ((DEBUG_INFO, " 0x%X", BiosGuardHobPtr->Bgpdt.PkeySlot1[Index]));
    }
  }

  DEBUG ((DEBUG_INFO, " \n"));
  DEBUG ((DEBUG_INFO, " BIOS_GUARD_CONFIG:: BGPDT : PkeySlot2[32] :\n"));
  for (Index = 0; Index < 32; Index++) {
    if ((Index == 15) || (Index == 31)) {
      DEBUG ((DEBUG_INFO, " 0x%X\n", BiosGuardHobPtr->Bgpdt.PkeySlot2[Index]));
    } else {
      DEBUG ((DEBUG_INFO, " 0x%X", BiosGuardHobPtr->Bgpdt.PkeySlot2[Index]));
    }
  }

  DEBUG ((DEBUG_INFO, " \n"));
  DEBUG ((DEBUG_INFO, " BIOS_GUARD_CONFIG:: BGPDT : BgModSvn : 0x%X\n", BiosGuardHobPtr->Bgpdt.BgModSvn));
  DEBUG ((DEBUG_INFO, " BIOS_GUARD_CONFIG:: BGPDT : BiosSvn : 0x%X\n", BiosGuardHobPtr->Bgpdt.BiosSvn));
  DEBUG ((DEBUG_INFO, " BIOS_GUARD_CONFIG:: BGPDT : ExecLim : 0x%X\n", BiosGuardHobPtr->Bgpdt.ExecLim));
  DEBUG ((DEBUG_INFO, " BIOS_GUARD_CONFIG:: BGPDT : BiosGuardAttr : 0x%X\n", BiosGuardHobPtr->Bgpdt.BiosGuardAttr));
  DEBUG ((DEBUG_INFO, " BIOS_GUARD_CONFIG:: BGPDT : EcCmd : 0x%X\n", BiosGuardHobPtr->Bgpdt.EcCmd));
  DEBUG ((DEBUG_INFO, " BIOS_GUARD_CONFIG:: BGPDT : EcData : 0x%X\n", BiosGuardHobPtr->Bgpdt.EcData));
  DEBUG ((DEBUG_INFO, " BIOS_GUARD_CONFIG:: BGPDT : EcCmdGetSvn : 0x%X\n", BiosGuardHobPtr->Bgpdt.EcCmdGetSvn));
  DEBUG ((DEBUG_INFO, " BIOS_GUARD_CONFIG:: BGPDT : EcCmdOpen : 0x%X\n", BiosGuardHobPtr->Bgpdt.EcCmdOpen));
  DEBUG ((DEBUG_INFO, " BIOS_GUARD_CONFIG:: BGPDT : EcCmdClose : 0x%X\n", BiosGuardHobPtr->Bgpdt.EcCmdClose));
  DEBUG ((DEBUG_INFO, " BIOS_GUARD_CONFIG:: BGPDT : EcCmdPortTest : 0x%X\n", BiosGuardHobPtr->Bgpdt.EcCmdPortTest));
  DEBUG ((DEBUG_INFO, " BIOS_GUARD_CONFIG:: BGPDT : LastSfam : 0x%X\n", BiosGuardHobPtr->Bgpdt.LastSfam));
  DEBUG ((DEBUG_INFO, " BIOS_GUARD_CONFIG:: BGPDT : SfamData[64] :\n"));

  //
  // Change the array size according to MAX_SFAM_COUNT
  //
  for (Index = 0; Index <= BiosGuardHobPtr->Bgpdt.LastSfam; Index++) {
    if ((Index == 15) || (Index == 31) || (Index == 47) || (Index == 63) ) {
      DEBUG ((DEBUG_INFO, " 0x%X 0x%X\n", BiosGuardHobPtr->Bgpdt.SfamData[Index].FirstByte,
                                          BiosGuardHobPtr->Bgpdt.SfamData[Index].LastByte));
    } else {
      DEBUG ((DEBUG_INFO, " 0x%X 0x%X,", BiosGuardHobPtr->Bgpdt.SfamData[Index].FirstByte,
                                         BiosGuardHobPtr->Bgpdt.SfamData[Index].LastByte));
    }
  }
  DEBUG ((DEBUG_INFO, " \n"));
  BiosGuardHobPtr->BiosGuardMemAddress = 0;
  BiosGuardHobPtr->BiosGuardMemSize = 0;
  BiosGuardHobPtr->BiosGuardIoTrapAddress = 0;
  BiosGuardHobPtr->BiosGuardIoTrapLength = 0;

  return EFI_SUCCESS;
}

/**
  This function performs CPU PEI Policy initialization in Pre-memory.

  @retval EFI_SUCCESS              The PPI is installed and initialized.
  @retval EFI ERRORS               The PPI is not successfully installed.
  @retval EFI_OUT_OF_RESOURCES     Do not have enough resources to initialize the driver
**/
EFI_STATUS
EFIAPI
UpdatePeiCpuPolicyPreMem (
  VOID
  )
{
  EFI_STATUS                      Status;
  EFI_PEI_READ_ONLY_VARIABLE2_PPI *VariableServices;
  UINTN                           VariableSize;
  SETUP_DATA                      SetupData;
  CPU_SETUP                       CpuSetup;
  EFI_BOOT_MODE                   BootMode;
  VOID                            *FspmUpd;
  SI_PREMEM_POLICY_PPI            *SiPreMemPolicyPpi;
  CPU_SECURITY_PREMEM_CONFIG      *CpuSecurityPreMemConfig;
  CPU_TXT_PREMEM_CONFIG           *CpuTxtPreMemConfig;
  CPU_CONFIG_LIB_PREMEM_CONFIG    *CpuConfigLibPreMemConfig;
  OVERCLOCKING_PREMEM_CONFIG      *OverClockingPreMemConfig;
  UINT32                          MaxLogicProcessors;
  BIOSGUARD_HOB                   *BiosGuardHobPtr;
  UINT16                          BiosSize;
  UINT16                          BiosMemSizeInMb;
  UINT8                           BiosGuardToolsInterface;
  FW_BOOT_MEDIA_TYPE              FwBootMediaType;
  UINTN                           VarSize;
  SYSTEM_FIRMWARE_UPDATE_PROGRESS UpdateProgress;
  MSR_CORE_THREAD_COUNT_REGISTER  MsrCoreThreadCount;
  UINT8                           VfPointIndex;

  DEBUG ((DEBUG_INFO, "Update PeiCpuPolicyUpdate Pre-Mem Start\n"));

  FspmUpd                     = NULL;
  SiPreMemPolicyPpi           = NULL;
  CpuTxtPreMemConfig          = NULL;
  CpuSecurityPreMemConfig     = NULL;
  CpuConfigLibPreMemConfig    = NULL;
  OverClockingPreMemConfig    = NULL;
  BiosGuardHobPtr             = NULL;
  BiosSize                    = 0;
  BiosMemSizeInMb             = 0;
  BiosGuardToolsInterface     = FALSE;
  FwBootMediaType             = FwBootMediaMax;

#if FixedPcdGet8(PcdFspModeSelection) == 1
  FspmUpd = (FSPM_UPD *) PcdGet32 (PcdFspmUpdDataAddress);
  ASSERT (FspmUpd != NULL);
#else
  Status = PeiServicesLocatePpi (&gSiPreMemPolicyPpiGuid, 0, NULL, (VOID **) &SiPreMemPolicyPpi);
  ASSERT_EFI_ERROR (Status);
  Status = GetConfigBlock ((VOID *) SiPreMemPolicyPpi, &gCpuSecurityPreMemConfigGuid, (VOID *) &CpuSecurityPreMemConfig);
  ASSERT_EFI_ERROR (Status);
  Status = GetConfigBlock ((VOID *) SiPreMemPolicyPpi, &gCpuTxtPreMemConfigGuid, (VOID *) &CpuTxtPreMemConfig);
  ASSERT_EFI_ERROR (Status);
  Status = GetConfigBlock ((VOID *) SiPreMemPolicyPpi, &gCpuConfigLibPreMemConfigGuid, (VOID *) &CpuConfigLibPreMemConfig);
  ASSERT_EFI_ERROR (Status);
  Status = GetConfigBlock ((VOID *) SiPreMemPolicyPpi, &gOverclockingPreMemConfigGuid, (VOID *) &OverClockingPreMemConfig);
  ASSERT_EFI_ERROR (Status);
#endif

  //
  // Make sure ReadOnlyVariablePpi is available
  //
  Status = PeiServicesLocatePpi (
             &gEfiPeiReadOnlyVariable2PpiGuid,
             0,
             NULL,
             (VOID **) &VariableServices
             );
  ASSERT_EFI_ERROR (Status);

  //
  // Get Setup Variable
  //
  VariableSize = sizeof (CPU_SETUP);
  Status = VariableServices->GetVariable (
                               VariableServices,
                               L"CpuSetup",
                               &gCpuSetupVariableGuid,
                               NULL,
                               &VariableSize,
                               &CpuSetup
                               );
  ASSERT_EFI_ERROR (Status);

  VariableSize = sizeof (SETUP_DATA);
  Status = VariableServices->GetVariable (
                               VariableServices,
                               L"Setup",
                               &gSetupVariableGuid,
                               NULL,
                               &VariableSize,
                               &SetupData
                               );
  ASSERT_EFI_ERROR (Status);

  //
  // Get current boot mode
  //
  Status = PeiServicesGetBootMode (&BootMode);
  ASSERT_EFI_ERROR (Status);

  UPDATE_POLICY (((FSPM_UPD *) FspmUpd)->FspmConfig.EnableC6Dram, CpuSecurityPreMemConfig->EnableC6Dram, CpuSetup.EnableC6Dram);

  ///
  /// Function will update PrmrrSize
  ///
  UpdatePrmrrSize (
    VariableServices,
    &CpuSetup,
    (VOID *) FspmUpd,
    CpuSecurityPreMemConfig
    );

  //
  // Update TXT Platform Policy
  //
  UPDATE_POLICY (((FSPM_UPD *) FspmUpd)->FspmConfig.Txt,              CpuSecurityPreMemConfig->Txt,         CpuSetup.Txt);
  UPDATE_POLICY (((FSPM_UPD *) FspmUpd)->FspmConfig.TxtDprMemorySize, CpuTxtPreMemConfig->TxtDprMemorySize, CpuSetup.DprSize * 0x100000);

  //
  // Check PWR FLR
  //
  if (PmcIsPowerFailureDetected ()) {
    PmcClearPowerFailureStatus ();
  }

  //
  // When the SUS_PWR_FLR bit is set, it indicates VRMPWRGD signal from the CPU VRM went low.
  // Software must clear this bit if set.
  //
  if (PmcIsSusPowerFailureDetected ()) {
    PmcClearSusPowerFailureStatus ();
  }

  if (CpuSetup.SkipStopPbet == 1) {
    UPDATE_POLICY (((FSPM_UPD *) FspmUpd)->FspmConfig.SkipStopPbet, CpuSecurityPreMemConfig->SkipStopPbet, TRUE);
  } else {
    UPDATE_POLICY (((FSPM_UPD *) FspmUpd)->FspmConfig.SkipStopPbet, CpuSecurityPreMemConfig->SkipStopPbet, FALSE);
  }

  ///
  /// Set BIOS Guard Tools Interface policy
  ///
  if ((BootMode == BOOT_ON_FLASH_UPDATE) || (BootMode == BOOT_IN_RECOVERY_MODE)) {
    UPDATE_POLICY (((FSPM_UPD *) FspmUpd)->FspmConfig.BiosGuardToolsInterface, CpuSecurityPreMemConfig->BiosGuardToolsInterface, CPU_FEATURE_ENABLE);
  } else {
    UPDATE_POLICY (((FSPM_UPD *) FspmUpd)->FspmConfig.BiosGuardToolsInterface, CpuSecurityPreMemConfig->BiosGuardToolsInterface, CpuSetup.BiosGuardToolsInterface);
  }

  SpiServiceInit ();

  //
  // Update BIOS Guard Policy
  // - Disable if SAF Mode is Active
  //
  if (SpiIsSafModeActive ()) {
    DEBUG ((DEBUG_INFO, "SAF mode is active. BIOS Guard has been disabled due to incompatibilty with SAF Mode\n"));
    UPDATE_POLICY (((FSPM_UPD *) FspmUpd)->FspmConfig.BiosGuard, CpuSecurityPreMemConfig->BiosGuard, CPU_FEATURE_DISABLE);
  } else {

  VarSize = sizeof (SYSTEM_FIRMWARE_UPDATE_PROGRESS);
  Status = VariableServices->GetVariable (
                               VariableServices,
                               SYSFW_UPDATE_PROGRESS_VARIABLE_NAME,
                               &gSysFwUpdateProgressGuid,
                               NULL,
                               &VarSize,
                               &UpdateProgress
                               );
  if (!(UpdateProgress.Component == UpdatingResiliency && UpdateProgress.Progress == 0)) {
  } else {
      UPDATE_POLICY (((FSPM_UPD *) FspmUpd)->FspmConfig.BiosGuard, CpuSecurityPreMemConfig->BiosGuard, CPU_FEATURE_DISABLE);
  }
#if FixedPcdGetBool(PcdBiosGuardEnable) == 1
    if (!(UpdateProgress.Component == UpdatingResiliency && UpdateProgress.Progress == 0)) {
      UPDATE_POLICY (((FSPM_UPD *) FspmUpd)->FspmConfig.BiosGuard, CpuSecurityPreMemConfig->BiosGuard, CPU_FEATURE_ENABLE);
    }
#else
    DEBUG ((DEBUG_INFO, "BIOS Guard PCD and Policy are disabled\n"));
    UPDATE_POLICY (((FSPM_UPD *) FspmUpd)->FspmConfig.BiosGuard, CpuSecurityPreMemConfig->BiosGuard, CPU_FEATURE_DISABLE);
#endif
  }

  Status = GetFirmwareBootMediaType (&FwBootMediaType);
  if (Status != EFI_SUCCESS) {
    ASSERT_EFI_ERROR (Status);
    DEBUG ((DEBUG_ERROR, "FirmwareBootMediaLib Error: Cannot Get Boot Media Type\n"));
  } else {
    if (FwBootMediaType == FwBootMediaSpi) {
      DEBUG ((DEBUG_INFO, "Firmware Boot Media is SPI \n"));
#if FixedPcdGet8(PcdFspModeSelection) == 1
      if (((FSPM_UPD *) FspmUpd)->FspmConfig.BiosGuard == TRUE) {
#else
      if (CpuSecurityPreMemConfig->BiosGuard == TRUE) {
#endif
        Status = BiosGuardHobInit ();
        ASSERT_EFI_ERROR (Status);
      }

      BiosGuardHobPtr = GetFirstGuidHob (&gBiosGuardHobGuid);
      if (BiosGuardHobPtr == NULL) {
        DEBUG ((DEBUG_ERROR, "BIOS Guard HOB not available\n"));
      }
      else {
#if FixedPcdGet8(PcdFspModeSelection) == 1 // #if API Mode
        BiosSize = ((FSPM_UPD *) FspmUpd)->FspmConfig.BiosSize;
        BiosGuardToolsInterface |= (UINT8)(((FSPM_UPD *) FspmUpd)->FspmConfig.BiosGuardToolsInterface);
#else // Dispatch Mode
        BiosSize = CpuSecurityPreMemConfig->BiosSize;
        BiosGuardToolsInterface |= (UINT8)CpuSecurityPreMemConfig->BiosGuardToolsInterface;
#endif // #endif API Mode

        ///
        /// Check if BiosGuardToolsInterface policy is enabled
        /// If Enabled, BGUP data will use the memory allocated within DPR
        /// If Disabled, the size allocated to BIOS Guard in DPR is 0MB, BGUP data will be stored in TSEG
        ///
        if (BiosGuardToolsInterface == TRUE) {
          ///
          /// The size allocated for DPR needs to account for the size of the BIOS ROM plus the size of the
          /// BGUP data: BGUP Header, BGUP Script,and BGUP Certificate.
          /// The size allocated needs to aligned to 1MB due to 1MB granularity of DPR.
          ///
          if ((BiosSize % 1024) == 0) {
            BiosMemSizeInMb = (BiosSize / 1024) + 1;
          } else {
            ///
            /// For a BiosSize that is not exactly 1MB, additional space will need to be added
            /// With consideration of the BGUP data, we can add an additional 2MB for a BIOS
            /// with greater than X.5MB, where X represents whole number value of the BiosSize
            ///
            if ((BiosSize % 1024) < 512) {
              BiosMemSizeInMb = (BiosSize / 1024) + 1;
            } else {
              BiosMemSizeInMb = (BiosSize / 1024) + 2;
            }
          }
          ///
          /// Upon MTRR shortage, BiosMemSizeInMb can be 2MB aligned at the expense of losing memory.
          ///
          if ((BiosMemSizeInMb % 2) == 1) {
            BiosMemSizeInMb = BiosMemSizeInMb + 1;
          }
        }

        BiosGuardHobPtr->BiosGuardMemSize = (UINT8)BiosMemSizeInMb;
      }
    }
  }

#if FixedPcdGetBool(PcdFspWrapperEnable) == 0
  CpuSecurityPreMemConfig->ResetAux = CpuSetup.ResetAux;
#endif

  //
  // Update CpuConfigLibPreMem Config Block data
  //
  UPDATE_POLICY (((FSPM_UPD *) FspmUpd)->FspmConfig.HyperThreading,          CpuConfigLibPreMemConfig->HyperThreading,          CpuSetup.HyperThreading);
  UPDATE_POLICY (((FSPM_UPD *) FspmUpd)->FspmConfig.BootFrequency,           CpuConfigLibPreMemConfig->BootFrequency,           CpuSetup.BootFrequency);
  UPDATE_POLICY (((FSPM_UPD *) FspmUpd)->FspmConfig.ActiveCoreCount,         CpuConfigLibPreMemConfig->ActiveCoreCount1,        CpuSetup.ActiveCoreCount);
  UPDATE_POLICY (((FSPM_UPD *) FspmUpd)->FspmConfig.FClkFrequency,           CpuConfigLibPreMemConfig->FClkFrequency,           CpuSetup.EpocFclkFreq);
  UPDATE_POLICY (((FSPM_UPD *) FspmUpd)->FspmConfig.JtagC10PowerGateDisable, CpuConfigLibPreMemConfig->JtagC10PowerGateDisable, CpuSetup.JtagC10PowerGateDisable);
  UPDATE_POLICY (((FSPM_UPD *) FspmUpd)->FspmConfig.BistOnReset,             CpuConfigLibPreMemConfig->BistOnReset,             CpuSetup.BistOnReset);
  UPDATE_POLICY (((FSPM_UPD *) FspmUpd)->FspmConfig.VmxEnable,               CpuConfigLibPreMemConfig->VmxEnable,               CpuSetup.VT);
  UPDATE_POLICY (((FSPM_UPD *) FspmUpd)->FspmConfig.TmeEnable,               CpuConfigLibPreMemConfig->TmeEnable,               CpuSetup.TmeEnable);
  UPDATE_POLICY (((FSPM_UPD *) FspmUpd)->FspmConfig.CpuCrashLogEnable,       CpuConfigLibPreMemConfig->CrashLogEnable,          SetupData.EnableCrashLog);
  UPDATE_POLICY (((FSPM_UPD *) FspmUpd)->FspmConfig.DebugInterfaceEnable,    CpuConfigLibPreMemConfig->DebugInterfaceEnable,    CpuSetup.DebugInterfaceEnable);
  if (BiosGuardHobPtr != NULL) {
    if (IsBiosGuardModuleDebugSigned2 (BiosGuardHobPtr->BiosGuardModulePtr)) {
      UPDATE_POLICY (((FSPM_UPD *) FspmUpd)->FspmConfig.DebugInterfaceEnable, CpuConfigLibPreMemConfig->DebugInterfaceEnable, CPU_FEATURE_ENABLE);
    }
  }
  UPDATE_POLICY (((FSPM_UPD *) FspmUpd)->FspmConfig.DebugInterfaceLockEnable, CpuConfigLibPreMemConfig->DebugInterfaceLockEnable, CpuSetup.DebugInterfaceLockEnable);
  if (CpuSetup.CpuRatioOverride) {
    UPDATE_POLICY (((FSPM_UPD *) FspmUpd)->FspmConfig.CpuRatio, CpuConfigLibPreMemConfig->CpuRatio, CpuSetup.CpuRatio);
  } else {
    UPDATE_POLICY (((FSPM_UPD *) FspmUpd)->FspmConfig.CpuRatio, CpuConfigLibPreMemConfig->CpuRatio, 0);
  }

  //
  // Update OverClockingPreMemConfig Config Block data
  //
  UPDATE_POLICY (((FSPM_UPD *) FspmUpd)->FspmConfig.OcSupport, OverClockingPreMemConfig->OcSupport, CpuSetup.OverclockingSupport);
  UPDATE_POLICY (((FSPM_UPD *) FspmUpd)->FspmConfig.OcLock,    OverClockingPreMemConfig->OcLock,    CpuSetup.OverclockingLock);

  if (CpuSetup.OverclockingSupport) {
    //
    // IA Core
    //
    UPDATE_POLICY (((FSPM_UPD *) FspmUpd)->FspmConfig.CoreVoltageMode, OverClockingPreMemConfig->CoreVoltageMode, CpuSetup.CoreVoltageMode);

#if FixedPcdGet8(PcdFspModeSelection) == 1
    if (((FSPM_UPD *) FspmUpd)->FspmConfig.CoreVoltageMode == OC_LIB_OFFSET_ADAPTIVE) {
#else
    if (OverClockingPreMemConfig->CoreVoltageMode == OC_LIB_OFFSET_ADAPTIVE) {
#endif
      UPDATE_POLICY (((FSPM_UPD *) FspmUpd)->FspmConfig.CoreVoltageOverride, OverClockingPreMemConfig->CoreVoltageOverride, 0);
      UPDATE_POLICY (((FSPM_UPD *) FspmUpd)->FspmConfig.CoreVoltageAdaptive, OverClockingPreMemConfig->CoreVoltageAdaptive, CpuSetup.CoreExtraTurboVoltage);
#if FixedPcdGet8(PcdFspModeSelection) == 1
    } else if (((FSPM_UPD *) FspmUpd)->FspmConfig.CoreVoltageMode == OC_LIB_OFFSET_OVERRIDE) {
#else
    } else if (OverClockingPreMemConfig->CoreVoltageMode == OC_LIB_OFFSET_OVERRIDE) {
#endif
      UPDATE_POLICY (((FSPM_UPD *) FspmUpd)->FspmConfig.CoreVoltageOverride, OverClockingPreMemConfig->CoreVoltageOverride, CpuSetup.CoreVoltageOverride);
      UPDATE_POLICY (((FSPM_UPD *) FspmUpd)->FspmConfig.CoreVoltageAdaptive, OverClockingPreMemConfig->CoreVoltageAdaptive, 0);
    } else {
      UPDATE_POLICY (((FSPM_UPD *) FspmUpd)->FspmConfig.CoreVoltageOverride, OverClockingPreMemConfig->CoreVoltageOverride, 0);
      UPDATE_POLICY (((FSPM_UPD *) FspmUpd)->FspmConfig.CoreVoltageAdaptive, OverClockingPreMemConfig->CoreVoltageAdaptive, 0);
    }

    UPDATE_POLICY (((FSPM_UPD *) FspmUpd)->FspmConfig.CoreVfPointOffsetMode, OverClockingPreMemConfig->CoreVfPointOffsetMode, CpuSetup.CoreVfPointOffsetMode);
    UPDATE_POLICY (((FSPM_UPD *) FspmUpd)->FspmConfig.CoreVfPointCount, OverClockingPreMemConfig->CoreVfPointCount, CpuSetup.CoreVfPointCount);

    if (CpuSetup.CoreVfPointOffsetMode == 0) {
      if (CpuSetup.CoreVoltageOffsetPrefix == 1) {
        UPDATE_POLICY (((FSPM_UPD *) FspmUpd)->FspmConfig.CoreVoltageOffset, OverClockingPreMemConfig->CoreVoltageOffset, (INT16)(~CpuSetup.CoreVoltageOffset + 1));
      } else {
        UPDATE_POLICY (((FSPM_UPD *) FspmUpd)->FspmConfig.CoreVoltageOffset, OverClockingPreMemConfig->CoreVoltageOffset, CpuSetup.CoreVoltageOffset);
      }
    } else if (CpuSetup.CoreVfPointOffsetMode == 1) {
      for (VfPointIndex = 0; VfPointIndex < CPU_OC_MAX_VF_POINTS; VfPointIndex++) {
        if (CpuSetup.CoreVfPointOffsetPrefix[VfPointIndex] == 1) {
          UPDATE_POLICY (((FSPM_UPD *) FspmUpd)->FspmConfig.CoreVfPointOffset[VfPointIndex], OverClockingPreMemConfig->CoreVfPointOffset[VfPointIndex], (INT16)(~CpuSetup.CoreVfPointOffset[VfPointIndex] + 1));
        } else {
          UPDATE_POLICY (((FSPM_UPD *) FspmUpd)->FspmConfig.CoreVfPointOffset[VfPointIndex], OverClockingPreMemConfig->CoreVfPointOffset[VfPointIndex], CpuSetup.CoreVfPointOffset[VfPointIndex]);
        }
      }
    }

    //
    // Ring
    //
    UPDATE_POLICY (((FSPM_UPD *) FspmUpd)->FspmConfig.RingVoltageMode, OverClockingPreMemConfig->RingVoltageMode, CpuSetup.RingVoltageMode);

#if FixedPcdGet8(PcdFspModeSelection) == 1
    if (((FSPM_UPD *) FspmUpd)->FspmConfig.RingVoltageMode == OC_LIB_OFFSET_ADAPTIVE) {
#else
    if (OverClockingPreMemConfig->RingVoltageMode == OC_LIB_OFFSET_ADAPTIVE) {
#endif
      UPDATE_POLICY (((FSPM_UPD *) FspmUpd)->FspmConfig.RingVoltageOverride, OverClockingPreMemConfig->RingVoltageOverride, 0);
      UPDATE_POLICY (((FSPM_UPD *) FspmUpd)->FspmConfig.RingVoltageAdaptive, OverClockingPreMemConfig->RingVoltageAdaptive, CpuSetup.RingExtraTurboVoltage);
#if FixedPcdGet8(PcdFspModeSelection) == 1
    } else if (((FSPM_UPD *) FspmUpd)->FspmConfig.RingVoltageMode == OC_LIB_OFFSET_OVERRIDE) {
#else
    } else if (OverClockingPreMemConfig->RingVoltageMode == OC_LIB_OFFSET_OVERRIDE) {
#endif
      UPDATE_POLICY (((FSPM_UPD *) FspmUpd)->FspmConfig.RingVoltageOverride, OverClockingPreMemConfig->RingVoltageOverride, CpuSetup.RingVoltageOverride);
      UPDATE_POLICY (((FSPM_UPD *) FspmUpd)->FspmConfig.RingVoltageAdaptive, OverClockingPreMemConfig->RingVoltageAdaptive, 0);
    } else {
      UPDATE_POLICY (((FSPM_UPD *) FspmUpd)->FspmConfig.RingVoltageOverride, OverClockingPreMemConfig->RingVoltageOverride, 0);
      UPDATE_POLICY (((FSPM_UPD *) FspmUpd)->FspmConfig.RingVoltageAdaptive, OverClockingPreMemConfig->RingVoltageAdaptive, 0);
    }

    if (CpuSetup.RingVoltageOffsetPrefix == 1) {
      // Offset is negative, need to convert
      UPDATE_POLICY (((FSPM_UPD *) FspmUpd)->FspmConfig.RingVoltageOffset, OverClockingPreMemConfig->RingVoltageOffset, (INT16)(~CpuSetup.RingVoltageOffset + 1));
    } else {
      UPDATE_POLICY (((FSPM_UPD *) FspmUpd)->FspmConfig.RingVoltageOffset, OverClockingPreMemConfig->RingVoltageOffset, CpuSetup.RingVoltageOffset);
    }
    UPDATE_POLICY (((FSPM_UPD *) FspmUpd)->FspmConfig.PerCoreHtDisable,     OverClockingPreMemConfig->PerCoreHtDisable,     CpuSetup.PerCoreHtDisable);
    UPDATE_POLICY (((FSPM_UPD *) FspmUpd)->FspmConfig.CoreMaxOcRatio,       OverClockingPreMemConfig->CoreMaxOcRatio,       CpuSetup.CoreMaxOcRatio);
    UPDATE_POLICY (((FSPM_UPD *) FspmUpd)->FspmConfig.CorePllVoltageOffset, OverClockingPreMemConfig->CorePllVoltageOffset, CpuSetup.CorePllVoltageOffset);
    UPDATE_POLICY (((FSPM_UPD *) FspmUpd)->FspmConfig.GtPllVoltageOffset,   OverClockingPreMemConfig->GtPllVoltageOffset,   CpuSetup.GtPllVoltageOffset);
    UPDATE_POLICY (((FSPM_UPD *) FspmUpd)->FspmConfig.RingPllVoltageOffset, OverClockingPreMemConfig->RingPllVoltageOffset, CpuSetup.RingPllVoltageOffset);
    UPDATE_POLICY (((FSPM_UPD *) FspmUpd)->FspmConfig.SaPllVoltageOffset,   OverClockingPreMemConfig->SaPllVoltageOffset,   CpuSetup.SaPllVoltageOffset);
    UPDATE_POLICY (((FSPM_UPD *) FspmUpd)->FspmConfig.McPllVoltageOffset,   OverClockingPreMemConfig->McPllVoltageOffset,   CpuSetup.McPllVoltageOffset);
    UPDATE_POLICY (((FSPM_UPD *) FspmUpd)->FspmConfig.RingMaxOcRatio,       OverClockingPreMemConfig->RingMaxOcRatio,       CpuSetup.RingMaxOcRatio);
    UPDATE_POLICY (((FSPM_UPD *) FspmUpd)->FspmConfig.RingDownBin,          OverClockingPreMemConfig->RingDownBin,          CpuSetup.RingDownBin);
    UPDATE_POLICY (((FSPM_UPD *) FspmUpd)->FspmConfig.Avx2RatioOffset,      OverClockingPreMemConfig->Avx2RatioOffset,      CpuSetup.Avx2RatioOffset);
    UPDATE_POLICY (((FSPM_UPD *) FspmUpd)->FspmConfig.Avx3RatioOffset,      OverClockingPreMemConfig->Avx3RatioOffset,      CpuSetup.Avx3RatioOffset);
    UPDATE_POLICY (((FSPM_UPD *) FspmUpd)->FspmConfig.TjMaxOffset,          OverClockingPreMemConfig->TjMaxOffset,          CpuSetup.TjMaxOffset);
    UPDATE_POLICY (((FSPM_UPD *) FspmUpd)->FspmConfig.BclkAdaptiveVoltage,  OverClockingPreMemConfig->BclkAdaptiveVoltage,  CpuSetup.BclkAdaptiveVoltageEnable);
    UPDATE_POLICY (((FSPM_UPD *) FspmUpd)->FspmConfig.FivrFaults,           OverClockingPreMemConfig->FivrFaults,           CpuSetup.FivrFaults);
    UPDATE_POLICY (((FSPM_UPD *) FspmUpd)->FspmConfig.FivrEfficiency,       OverClockingPreMemConfig->FivrEfficiency,       CpuSetup.FivrEfficiency);
    UPDATE_POLICY (((FSPM_UPD *) FspmUpd)->FspmConfig.Avx2VoltageScaleFactor,    OverClockingPreMemConfig->Avx2VoltageScaleFactor,    CpuSetup.Avx2VoltageScaleFactor);
    UPDATE_POLICY (((FSPM_UPD *) FspmUpd)->FspmConfig.Avx512VoltageScaleFactor,  OverClockingPreMemConfig->Avx512VoltageScaleFactor,  CpuSetup.Avx512VoltageScaleFactor);
  }

  ///
  /// Set PcdCpuMaxLogicalProcessorNumber to max number of logical processors enabled
  /// Read MSR_CORE_THREAD_COUNT (0x35) to check the total active Threads
  ///
  MsrCoreThreadCount.Uint64 = AsmReadMsr64 (MSR_CORE_THREAD_COUNT);
  MaxLogicProcessors = MsrCoreThreadCount.Bits.Threadcount;
  DEBUG ((DEBUG_INFO, "MaxLogicProcessors = %d\n", MaxLogicProcessors));

#if (FixedPcdGetBool(PcdFspWrapperEnable) == 0) || (FixedPcdGet8(PcdFspModeSelection) == 1)
  PcdSet32S (PcdCpuMaxLogicalProcessorNumber, MaxLogicProcessors);
#else
  PcdSetEx32S (&gUefiCpuPkgTokenSpaceGuid, PcdCpuMaxLogicalProcessorNumber, MaxLogicProcessors);
#endif

  return EFI_SUCCESS;
}
