/** @file
  CPU Setup Routines

@copyright
  INTEL CONFIDENTIAL
  Copyright 2009 - 2020 Intel Corporation.

  The source code contained or described herein and all documents related to the
  source code ("Material") are owned by Intel Corporation or its suppliers or
  licensors. Title to the Material remains with Intel Corporation or its suppliers
  and licensors. The Material may contain trade secrets and proprietary and
  confidential information of Intel Corporation and its suppliers and licensors,
  and is protected by worldwide copyright and trade secret laws and treaty
  provisions. No part of the Material may be used, copied, reproduced, modified,
  published, uploaded, posted, transmitted, distributed, or disclosed in any way
  without Intel's prior express written permission.

  No license under any patent, copyright, trade secret or other intellectual
  property right is granted to or conferred upon you by disclosure or delivery
  of the Materials, either expressly, by implication, inducement, estoppel or
  otherwise. Any license under such intellectual property rights must be
  express and approved by Intel in writing.

  Unless otherwise agreed by Intel in writing, you may not remove or alter
  this notice or any other notice embedded in Materials by Intel or
  Intel's suppliers or licensors in any way.

  This file contains a 'Sample Driver' and is licensed as such under the terms
  of your license agreement with Intel or your vendor. This file may be modified
  by the user, subject to the additional terms of the license agreement.

@par Specification Reference:
**/

#include <PlatformNvRamHookLib.h>
#include <Library/CpuPlatformLib.h>
#include <Library/RngLib.h>
#include <Library/CpuMailboxLib.h>
#include <Protocol/DxeSmmReadyToLock.h>
#include <Library/UefiLib.h>
#include <Library/BaseMemoryLib.h>
#include <Guid/MdeModuleHii.h>
#include <Library/BaseLib.h>
#include "Base.h"
#include <SetupPrivate.h>
#include "Platform.h"
#include "PlatformBoardId.h"
#include <Register/Cpuid.h>
#include <Register/Msr.h>
#include <CpuAccess.h>
#include <Protocol/CpuInfo.h>
#include <Library/PciSegmentLib.h>
#include <Library/BootGuardLib.h>
#include <Library/MeFwStsLib.h>
#include <Register/PchRegs.h>
#include <Library/SpiAccessLib.h>
#include <Protocol/MpService.h>
#include <Library/VoltageRegulatorCommands.h>
#include <CpuDataHob.h>
#include <Register/CommonMsr.h>
#include <Register/ArchMsr.h>
#include <CmosMap.h>
#include <Library/CmosAccessLib.h>
#include <Register/SaRegsHostBridge.h>
#if FixedPcdGetBool(PcdITbtEnable) == 1
#include <TcssDataHob.h>
#endif

#define OC_LIB_CMD_GET_FAVORED_CORE         0x1C

static EFI_HII_HANDLE     gHiiHandle;

static UINT8              mCoreRatioFinal1     = 0;
static UINT8              mCoreRatioFinal2     = 0;
static UINT8              mCoreRatioFinal3     = 0;
static UINT8              mCoreRatioFinal4     = 0;
static UINT8              mCoreRatioFinal5     = 0;
static UINT8              mCoreRatioFinal6     = 0;
static UINT8              mCoreRatioFinal7     = 0;
static UINT8              mCoreRatioFinal8     = 0;
EFI_MP_SERVICES_PROTOCOL  *mMpService;

typedef struct _MAILBOX_READ {
  UINT32                   *MailboxDataPtr;
  UINT32                   *MailboxStatus;
} MAILBOX_READ;

GLOBAL_REMOVE_IF_UNREFERENCED EFI_STRING_ID FuseFreqString [] = {
  STRING_TOKEN (STR_P0_FUSED_MAX_CORE_VALUE),
  STRING_TOKEN (STR_P1_FUSED_MAX_CORE_VALUE),
  STRING_TOKEN (STR_P2_FUSED_MAX_CORE_VALUE),
  STRING_TOKEN (STR_P3_FUSED_MAX_CORE_VALUE),
  STRING_TOKEN (STR_P4_FUSED_MAX_CORE_VALUE),
  STRING_TOKEN (STR_P5_FUSED_MAX_CORE_VALUE),
  STRING_TOKEN (STR_P6_FUSED_MAX_CORE_VALUE),
  STRING_TOKEN (STR_P7_FUSED_MAX_CORE_VALUE)
};
extern UINT8 AdvancedBin[];

GLOBAL_REMOVE_IF_UNREFERENCED CPU_DATA_HOB                *mCpuDataHob;

/**
   This function update Bit 4 of CMOS index 0x2A (index reg: 0x70, Data : 0x71)
   based on TXT setup option.
   If TXT is Enabled Bit 4 will be 1, else bit 4 will be 0

   @param[IN]       EFI_FORM_CALLBACK_PROTOCOL   *This
   @param[IN]       UINT16                       KeyValue
   @param[IN]       EFI_IFR_DATA_ARRAY           *Data,
   @param[IN]       EFI_HII_CALLBACK_PACKET      **Packet

   @retval          EFI_SUCCESS
**/
EFI_STATUS
EFIAPI
TxtPolicyCallBackFunction (
  IN CONST EFI_HII_CONFIG_ACCESS_PROTOCOL *This,
  IN EFI_BROWSER_ACTION                   Action,
  IN EFI_QUESTION_ID                      KeyValue,
  IN UINT8                                Type,
  IN EFI_IFR_TYPE_VALUE                   *Value,
  OUT EFI_BROWSER_ACTION_REQUEST          *ActionRequest
  );

/**
  Update the min, max, and default values for CpuRatio.

  @param[in] CpuRatioDefault      The CPU MNTR default.

  @retval EFI_SUCCESS             Values updated successfully.
  @retval EFI_NOT_FOUND           Failed to update it.
**/
EFI_STATUS
InitCpuMntrDefaultVfr (
  UINT8      CpuRatioDefault
  );

/**
  Returns number of active CPU Cores

  @retval UINT8 - Number of active CPU Cores.
**/
UINT8 NumCpuCores (
  VOID
  )
{
  MSR_CORE_THREAD_COUNT_REGISTER  MsrCoreThreadCount;

  MsrCoreThreadCount.Uint64 = AsmReadMsr64 (MSR_CORE_THREAD_COUNT);
  return (UINT8) MsrCoreThreadCount.Bits.Corecount;
}

/**
  Get number of supported threads per core.

  @retval  UINT8- Number of Threads per core.
**/
UINT8 NumSupportedThreadsPerCore (
  VOID
  )
{
  CPUID_EXTENDED_TOPOLOGY_EBX    CpuidExtendedTopologyEbx;
  AsmCpuidEx (CPUID_EXTENDED_TOPOLOGY, CPUID_EXTENDED_TOPOLOGY_LEVEL_TYPE_INVALID, NULL, &CpuidExtendedTopologyEbx.Uint32, NULL, NULL);
  return (UINT8) CpuidExtendedTopologyEbx.Bits.LogicalProcessors;
}

/**
  Get number of supported Cpu Cores per package.

  @retval UINT8 - Number of supported Cpu Cores per package.
**/
UINT8 NumSupportedCpuCores (
  VOID
  )
{
  CPUID_EXTENDED_TOPOLOGY_EBX    CpuidExtendedTopologyEbx;
  AsmCpuidEx (CPUID_EXTENDED_TOPOLOGY, CPUID_EXTENDED_TOPOLOGY_LEVEL_TYPE_SMT, NULL, &CpuidExtendedTopologyEbx.Uint32, NULL, NULL);
  return (UINT8) CpuidExtendedTopologyEbx.Bits.LogicalProcessors / NumSupportedThreadsPerCore ();
}

/**
  This function to be used with StartUpAllAps for retrieving
  the Max Core Frequency of all cores.

  @param[in] VOID *Buffer
**/
VOID
MailboxReadFavoredCore (
  IN OUT VOID *Buffer
  )
{
  EFI_STATUS   Status;
  MAILBOX_READ *MailboxReadParameters;
  UINTN        CoreNumber;
  UINTN        ApNumber;
  UINT32       FuseCoreMax;
  UINT32       MailboxStatus;

  MailboxReadParameters = (MAILBOX_READ *) Buffer;

  if (IsSecondaryThread ()) {
    return;
  }
  CoreNumber = 0;
  ApNumber = 0;
  Status = mMpService->WhoAmI (mMpService, &ApNumber);
  ASSERT_EFI_ERROR (Status);

  if (ApNumber != 0) {
    CoreNumber = (ApNumber / 2);
  }

  MailboxRead (MAILBOX_TYPE_OC, OC_LIB_CMD_GET_FAVORED_CORE, &FuseCoreMax, &MailboxStatus);
  MailboxReadParameters->MailboxDataPtr [CoreNumber] = FuseCoreMax;
  MailboxReadParameters->MailboxStatus [CoreNumber] = MailboxStatus;
}

/**
  Initialize Turbo Core Ratio defaults.

  @param[in] EFI_EVENT    Event
  @param[in] VOID         *Context
**/
VOID
InitTurboRatioDefault (
  IN EFI_EVENT Event,
  IN VOID      *Context
  )
{
  EFI_STATUS                         Status;
  MSR_TURBO_RATIO_LIMIT_REGISTER     TurboRatioLimit;
  UINTN                              VariableSize;
  UINT32                             Attributes;
  UINT8                              MaxBusRatio;
  UINT8                              MinBusRatio;

  MaxBusRatio = 0;
  MinBusRatio = 0;

  VariableSize = sizeof (CPU_SETUP);
  Status = gRT->GetVariable (
                  L"CpuSetup",
                  &gCpuSetupVariableGuid,
                  &Attributes,
                  &VariableSize,
                  &mCpuSetup
                  );
  ASSERT_EFI_ERROR (Status);

  if (mCpuSetup.IsTurboRatioDefaultsInitalized == 0) {
    ///
    /// Get Maximum Non-Turbo bus ratio (HFM) from Platform Info MSR Bits[15:8]
    ///
    GetBusRatio (&MaxBusRatio, &MinBusRatio);
    mCpuSetup.IsTurboRatioDefaultsInitalized = 1;
    mCpuSetup.FlexRatioOverrideDefault = MaxBusRatio;

    TurboRatioLimit.Uint64 = AsmReadMsr64 (MSR_TURBO_RATIO_LIMIT);
    mCpuSetup.RatioLimit1Default = (UINT8) TurboRatioLimit.Bits.MaxTurbo1Core;
    mCpuSetup.RatioLimit2Default = (UINT8) TurboRatioLimit.Bits.MaxTurbo2Core;
    mCpuSetup.RatioLimit3Default = (UINT8) TurboRatioLimit.Bits.MaxTurbo3Core;
    mCpuSetup.RatioLimit4Default = (UINT8) TurboRatioLimit.Bits.MaxTurbo4Core;
    mCpuSetup.RatioLimit5Default = (UINT8) TurboRatioLimit.Bits.MaxTurbo5Core;
    mCpuSetup.RatioLimit6Default = (UINT8) TurboRatioLimit.Bits.MaxTurbo6Core;
    mCpuSetup.RatioLimit7Default = (UINT8) TurboRatioLimit.Bits.MaxTurbo7Core;
    mCpuSetup.RatioLimit8Default = (UINT8) TurboRatioLimit.Bits.MaxTurbo8Core;
    mCpuSetup.RatioLimit1 = mCpuSetup.RatioLimit1Default;
    mCpuSetup.RatioLimit2 = mCpuSetup.RatioLimit2Default;
    mCpuSetup.RatioLimit3 = mCpuSetup.RatioLimit3Default;
    mCpuSetup.RatioLimit4 = mCpuSetup.RatioLimit4Default;
    mCpuSetup.RatioLimit5 = mCpuSetup.RatioLimit5Default;
    mCpuSetup.RatioLimit6 = mCpuSetup.RatioLimit6Default;
    mCpuSetup.RatioLimit7 = mCpuSetup.RatioLimit7Default;
    mCpuSetup.RatioLimit8 = mCpuSetup.RatioLimit8Default;

    Status = gRT->SetVariable (
                    L"CpuSetup",
                    &gCpuSetupVariableGuid,
                    Attributes,
                    VariableSize,
                    &mCpuSetup
                    );
    ASSERT_EFI_ERROR (Status);
  }
}

/**
  Initialize AcheckRequest defaults. Always reset to Disable
**/
VOID
InitTxtAcheckDefault (
  VOID
  )
{
  EFI_STATUS            Status;
  UINT32                VariableAttr;
  UINTN                 VariableSize;

  VariableAttr = 0;
  VariableSize = sizeof (CPU_SETUP);
  Status = gRT->GetVariable (
                  L"CpuSetup",
                  &gCpuSetupVariableGuid,
                  &VariableAttr,
                  &VariableSize,
                  &mCpuSetup
                  );
  if (EFI_ERROR (Status)) {
    VariableAttr = EFI_VARIABLE_NON_VOLATILE | EFI_VARIABLE_BOOTSERVICE_ACCESS | EFI_VARIABLE_RUNTIME_ACCESS;
  }
  ASSERT_EFI_ERROR (Status);

  mCpuSetup.AcheckRequest = 0;

  Status = gRT->SetVariable (
                  L"CpuSetup",
                  &gCpuSetupVariableGuid,
                  VariableAttr,
                  VariableSize,
                  &mCpuSetup
                  );
  ASSERT_EFI_ERROR (Status);
}

/**
   This function update Bit 4 of CMOS index 0x2A (index reg: 0x70, Data : 0x71)
   based on TXT setup option.
   If TXT is Enabled Bit 4 will be 1, else bit 4 will be 0

   @param[IN]       EFI_FORM_CALLBACK_PROTOCOL   *This
   @param[IN]       UINT16                       KeyValue
   @param[IN]       EFI_IFR_DATA_ARRAY           *Data,
   @param[IN]       EFI_HII_CALLBACK_PACKET      **Packet

   @retval          EFI_SUCCESS
**/
EFI_STATUS
EFIAPI
TxtPolicyCallBackFunction (
  IN CONST EFI_HII_CONFIG_ACCESS_PROTOCOL *This,
  IN EFI_BROWSER_ACTION                   Action,
  IN EFI_QUESTION_ID                      KeyValue,
  IN UINT8                                Type,
  IN EFI_IFR_TYPE_VALUE                   *Value,
  OUT EFI_BROWSER_ACTION_REQUEST          *ActionRequest
  )
{
  EFI_STATUS              Status;
  CPU_SETUP               *CpuSetup;
  UINTN                   VarSize;
  UINT8                   TxtStatus;
  UINT8                   NewTxtStatus;
  EFI_STRING              RequestString;

  Status = EFI_SUCCESS;
  RequestString = NULL;

  //
  // Only process KEY_TXT_POLICY_FIT callback
  //
  if (KeyValue != KEY_TXT_POLICY_FIT) {
    return EFI_UNSUPPORTED;
  }

  switch (Action) {
  case EFI_BROWSER_ACTION_DEFAULT_STANDARD:
    //
    // Set Txt knob value to the default value (Disabled)
    //
    Value->u8 = 0;
    break;

  case EFI_BROWSER_ACTION_CHANGED:
  case EFI_BROWSER_ACTION_CHANGING:
    break;

  default:
    return EFI_UNSUPPORTED;
    break;
  }

  VarSize   = sizeof (CPU_SETUP);
  CpuSetup  = AllocatePool (VarSize);
  ASSERT (CpuSetup != NULL);

  if (CpuSetup == NULL) {
    DEBUG ((DEBUG_ERROR, "TxtPolicyCallBackFunction: Out of resources allocating CpuSetup!\n"));
    return EFI_OUT_OF_RESOURCES;
  }

  //
  // GetBrowserData by VarStore Name (Setup)
  //
  if (!HiiGetBrowserData (&gCpuSetupVariableGuid, L"CpuSetup", VarSize, (UINT8 *) CpuSetup)) {
    FreePool (CpuSetup);
    DEBUG ((DEBUG_ERROR, "TxtPolicyCallBackFunction: CpuSetup not found!\n"));
    return EFI_NOT_FOUND;
  }

  //
  // Read current TXT status from CMOS
  //
  NewTxtStatus = TxtStatus = CmosRead8 (FIT_REC_TXT_POLICY_TYPE_A);

  //
  // Handle restoring TXT to default state (DISABLED)
  //
  if (Action == EFI_BROWSER_ACTION_DEFAULT_STANDARD) {
    CpuSetup->Txt = FALSE;
  }

  if (CpuSetup->Txt == FALSE) {
    DEBUG ((DEBUG_INFO, "TxtPolicyCallBackFunction: Setting TXT to disabled.\n"));
    NewTxtStatus &= ~BIT4;
  } else {
    DEBUG ((DEBUG_INFO, "TxtPolicyCallBackFunction: Setting TXT to enabled.\n"));
    NewTxtStatus |=  BIT4;
  }

  //
  // Sync CMOS with new Txt Status
  //
  if (NewTxtStatus != TxtStatus) {
    CmosWrite8 (FIT_REC_TXT_POLICY_TYPE_A, NewTxtStatus);
  }

  //
  // Update menu string with new value
  //
  RequestString = HiiConstructRequestString (RequestString, OFFSET_OF (CPU_SETUP, Txt), sizeof (CpuSetup->Txt));

  if (RequestString != NULL) {
    if (!HiiSetBrowserData (&gCpuSetupVariableGuid, L"CpuSetup", VarSize, (UINT8 *) CpuSetup, RequestString)) {
      DEBUG ((DEBUG_ERROR, "TxtPolicyCallBackFunction: Setting Hii string ERROR!\n"));
      Status = EFI_NOT_FOUND;
    }
    ASSERT_EFI_ERROR (Status);
    FreePool (RequestString);
  }

  FreePool (CpuSetup);
  return Status;
}

/**
  Initialize Debug CPU Disabled (DCD) Bit information
**/
VOID
InitDCDInfo (
  VOID
  )
{
  EFI_STATUS            Status;
  UINT32                VariableAttr;
  UINTN                 VariableSize;

  VariableAttr = 0;
  VariableSize = sizeof (CPU_SETUP);
  Status = gRT->GetVariable (
                  L"CpuSetup",
                  &gCpuSetupVariableGuid,
                  &VariableAttr,
                  &VariableSize,
                  &mCpuSetup
                  );
  if (EFI_ERROR (Status)) {
    VariableAttr = EFI_VARIABLE_NON_VOLATILE | EFI_VARIABLE_BOOTSERVICE_ACCESS | EFI_VARIABLE_RUNTIME_ACCESS;
  }
  ASSERT_EFI_ERROR (Status);

  if (IsCpuDebugDisabled ()) {
    mCpuSetup.DebugCpuDisabled = 1;
  } else {
    mCpuSetup.DebugCpuDisabled = 0;
  }

  Status = gRT->SetVariable (
                  L"CpuSetup",
                  &gCpuSetupVariableGuid,
                  VariableAttr,
                  VariableSize,
                  &mCpuSetup
                  );
  ASSERT_EFI_ERROR (Status);
}

/**
  Initialize CPU Package Cx limit
**/
VOID
InitPkgCxLimit (
  VOID
  )
{
  EFI_STATUS    Status;
  UINT32        VariableAttr;
  UINTN         VariableSize;
#if FixedPcdGetBool(PcdITbtEnable) == 1
  TCSS_DATA_HOB *TcssHob;

  TcssHob = NULL;
#endif
  VariableAttr = 0;
#if FixedPcdGetBool(PcdITbtEnable) == 1
  ///
  /// Locate HOB for TCSS Data
  ///
  TcssHob = (TCSS_DATA_HOB *) GetFirstGuidHob (&gTcssHobGuid);
  if (TcssHob == NULL) {
    DEBUG ((DEBUG_ERROR, "SA TSCC Data HOB not found\n"));
    return;
  }

  ///
  /// Check IOM ready state
  /// - Limit Package C state to PC2 when IOM is not ready
  /// - Keep user setting when IOM is ready, and return
  ///
  if (TcssHob->TcssData.IOMReady == 1) {
    return;
  }
#endif
  VariableSize = sizeof (CPU_SETUP);
  Status = gRT->GetVariable (
                  L"CpuSetup",
                  &gCpuSetupVariableGuid,
                  &VariableAttr,
                  &VariableSize,
                  &mCpuSetup
                  );
  if (EFI_ERROR (Status)) {
    VariableAttr = EFI_VARIABLE_NON_VOLATILE | EFI_VARIABLE_BOOTSERVICE_ACCESS | EFI_VARIABLE_RUNTIME_ACCESS;
  }
  ASSERT_EFI_ERROR (Status);

  DEBUG ((DEBUG_INFO, "CpuSetup warming: Limit Package C state to PC2 when IOM is not ready\n"));
  mCpuSetup.PkgCStateLimit = 1; //Limit Package C state to PC2

  Status = gRT->SetVariable (
                  L"CpuSetup",
                  &gCpuSetupVariableGuid,
                  VariableAttr,
                  VariableSize,
                  &mCpuSetup
                  );
ASSERT_EFI_ERROR (Status);

}

///
///

/**
  Display the CPU VR menu programmed defaults.
**/
VOID
InitCpuVrConfig (
  VOID
  )
{
  EFI_STATUS                  Status;
  UINT32                      MailboxData;
  UINT32                      MailboxCmd;
  UINT32                      MailboxStatus;
  UINT32                      MailboxType;
  UINT64                      TempAcLoadline;
  UINT64                      TempDcLoadline;
  UINT8                       TempVrAddress;
  UINTN                       VrIndex;
  UINTN                       VariableSize;
  UINT32                      Attributes;

  VariableSize = sizeof (CPU_SETUP);
  Status = gRT->GetVariable (
                  L"CpuSetup",
                  &gCpuSetupVariableGuid,
                  &Attributes,
                  &VariableSize,
                  &mCpuSetup
                  );
  ASSERT_EFI_ERROR (Status);

  DEBUG ((DEBUG_INFO, "InitCpuVrConfig ()"));

  if (mCpuDataHob == NULL) {
    return;
  }
  ///
  /// CPU VR MSR mailbox
  ///
  MailboxType = MAILBOX_TYPE_VR_MSR;

  for (VrIndex = mCpuDataHob->MinVrIndex; VrIndex < mCpuDataHob->MaxVrIndex; VrIndex++) {
    TempVrAddress = mCpuDataHob->VrAddress[VrIndex];

    if ((mCpuSetup.VrConfigEnable == 1) && (mCpuDataHob->SvidEnabled == 1)) {
      ///
      /// AC / DC Loadline
      ///
      MailboxCmd = READ_ACDC_LOADLINE_CMD | ((TempVrAddress & VR_ADDRESS_MASK) << VR_ADDRESS_OFFSET);
      MailboxData = 0;
      Status = MailboxRead (MailboxType, MailboxCmd, &MailboxData, &MailboxStatus);
      if (MailboxStatus != PCODE_MAILBOX_CC_SUCCESS) {
        DEBUG ((DEBUG_ERROR, "VR: Error Reading AC/DC Loadline. EFI_STATUS = %r, Mailbox Status = %X\n", Status, MailboxStatus));
      }
      TempAcLoadline = (MailboxData & AC_LOADLINE_MASK);
      TempDcLoadline = RShiftU64 ((MailboxData & DC_LOADLINE_MASK), DC_LOADLINE_OFFSET);
      ///
      ///  Loadline is 1/100 mOhm units. Mailbox interface requires Loadline in U-4.20 Ohms format.
      ///  After multiplying by 2^20, we must divide the result by 100,000 to convert to Ohms.
      ///  Adding half of divisor to dividend to account for rounding errors in fixed point arithmetic.

      TempAcLoadline = MultU64x32 (TempAcLoadline, 100000);
      mCpuSetup.AcLoadline = (UINT16) ((TempAcLoadline + (1 << 19)) >> 20);
      TempDcLoadline = MultU64x32 (TempDcLoadline, 100000);
      mCpuSetup.DcLoadline = (UINT16) ((TempDcLoadline + (1 << 19)) >> 20);

      ///
      /// PS Cutoff Current
      ///
      MailboxCmd = READ_PSI_CUTOFF_CMD | ((TempVrAddress & VR_ADDRESS_MASK) << VR_ADDRESS_OFFSET);
      MailboxData = 0;
      Status = MailboxRead (MailboxType, MailboxCmd, &MailboxData, &MailboxStatus);
      if (MailboxStatus != PCODE_MAILBOX_CC_SUCCESS) {
        DEBUG ((DEBUG_ERROR, "VR: Error Reading PS Cutoff Current. EFI_STATUS = %r, Mailbox Status = %X\n", Status, MailboxStatus));
      }
      mCpuSetup.Psi1Threshold = (UINT16) (MailboxData & PSI_THRESHOLD_MASK);
      mCpuSetup.Psi2Threshold = (UINT16) ((MailboxData & PSI2_THRESHOLD_OFFSET_MASK) >> PSI2_THRESHOLD_OFFSET);
      mCpuSetup.Psi3Threshold = (UINT16) ((MailboxData & PSI3_THRESHOLD_OFFSET_MASK) >> PSI3_THRESHOLD_OFFSET);

      ///
      /// IMON Config
      /// -Policy Imon offset is defined in 1/1000 increments
      /// -Policy Imon slope is defined in 1/100 increments
      /// -Mailbox ImonOffset = (PlatPolicyImonOffset * 2^8)/1000
      /// -Mailbox ImonSlope = (PlatPolicyImonSlope * 2^15)/100
      /// -Adding half of divisor to dividend to account for rounding errors in fixed point arithmetic.
      ///
      MailboxCmd = READ_IMON_CONFIG_CMD | ((TempVrAddress & VR_ADDRESS_MASK) << VR_ADDRESS_OFFSET);
      MailboxData = 0;
      Status = MailboxRead (MailboxType, MailboxCmd, &MailboxData, &MailboxStatus);
      if (MailboxStatus != PCODE_MAILBOX_CC_SUCCESS) {
        DEBUG ((DEBUG_ERROR, "VR: Error Reading IMON Config. EFI_STATUS = %r, Mailbox Status = %X\n", Status, MailboxStatus));
      }
      mCpuSetup.ImonOffset = (UINT16) (((MailboxData & VR_IMON_OFFSET_MASK) * 1000 + (1 << 7)) >> 8);
      mCpuSetup.ImonSlope = (UINT16) ((((MailboxData & VR_IMON_SLOPE_MASK) >> VR_IMON_SLOPE_OFFSET) * 100 + (1 << 14)) >> 15);

      ///
      /// Icc Max
      ///
      MailboxCmd = READ_VR_ICC_MAX_CMD | ((TempVrAddress & VR_ADDRESS_MASK) << VR_ADDRESS_OFFSET);
      MailboxData = 0;
      Status = MailboxRead (MailboxType, MailboxCmd, &MailboxData, &MailboxStatus);
      if (MailboxStatus != PCODE_MAILBOX_CC_SUCCESS) {
        DEBUG ((DEBUG_ERROR, "VR: Error Reading ICC Max. EFI_STATUS = %r, Mailbox Status = %X\n", Status, MailboxStatus));
      }
      mCpuSetup.IccMax = (UINT16) (MailboxData & 0xFFFF);

      ///
      /// VR TDC Settings
      /// -Mailbox TDC Current Limit defined as U15.12.3, Range 0-4095A
      ///    -Policy defined in 1/8 A increments
      ///
      MailboxCmd = READ_VR_TDC_CONFIG_CMD | ((TempVrAddress & VR_ADDRESS_MASK) << VR_TDC_ADDRESS_OFFSET);
      MailboxData = 0;
      Status = MailboxRead (MailboxType, MailboxCmd, &MailboxData, &MailboxStatus);
      if (MailboxStatus != PCODE_MAILBOX_CC_SUCCESS) {
        DEBUG ((DEBUG_ERROR, "VR: Error Reading VR TDC Config. EFI_STATUS = %r, Mailbox Status = %X\n", Status, MailboxStatus));
      }
      mCpuSetup.TdcCurrentLimit = (UINT16) ((MailboxData & VR_TDC_CURRENT_LIMIT_MASK));

    }
  }

  Status = gRT->SetVariable (
                  L"CpuSetup",
                  &gCpuSetupVariableGuid,
                  Attributes,
                  VariableSize,
                  &mCpuSetup
                  );
}

/**
  Initialize CPU strings and VR topology.

  @param[in] EFI_EVENT    Event
  @param[in] VOID         *Context
**/
VOID
InitCPUInfo (
  IN EFI_EVENT Event,
  IN VOID      *Context
  )
{
  EFI_STATUS                           Status;
  UINTN                                VariableSize;
  UINT8                                Index;
  SETUP_CPU_FEATURES                   SetupCpuFeatures;
  CHAR8                                String[15];
  MSR_BIOS_SIGN_ID_REGISTER            MsrSignId;
  UINT16                               PowerLimitInteger;
  UINT16                               PowerLimitFraction;
  UINT16                               PowerLimitInteger2;
  UINT16                               PowerLimitFraction2;
  UINT8                                PowerUnit;
  UINT8                                CoreRatio;
  CPU_INFO                             *GetCpuInfo;
  CPUID_VERSION_INFO_ECX               Ecx;
  UINT8                                CtdpLevels;
  UINT8                                CtdpRatio;
  UINT16                               CtdpTdp;
  UINT32                               CtdpAddress;
  CPU_INFO_PROTOCOL                    *DxeCpuInfo;
  UINT32                               CpuSetupVolVarAttr;
  CPU_SETUP_VOLATILE_DATA              CpuSetupVolData;
  UINTN                                MchBar;
  UINT64                               MmioValue64;
  UINT32                               RegEcx;
  UINT32                               RfiNominalValue;
  UINT32                               RfiTotalFreqInKhz;
  UINT16                               RfiFreqMhz;
  UINT16                               RfiFreqKhz;
  CONST CHAR8                          *GenerationString = NULL;
  CONST CHAR8                          *SkuString = NULL;
  MSR_TURBO_RATIO_LIMIT_REGISTER       TurboRatioLimit;
  MSR_CORE_THREAD_COUNT_REGISTER       MsrCoreThreadCount;
  MAILBOX_READ                         MailboxReadParameters;
  UINTN                                NumberOfCores;
  UINTN                                ProcNum;
  UINTN                                LogProcNum;
  UINTN                                BspIndex;
  UINT8                                MaxBusRatio;
  UINT8                                MinBusRatio;
  MSR_IA32_DEBUG_INTERFACE_REGISTER    DebugInterfaceReg;
  VOID                                 *Hob;
  MSR_PACKAGE_POWER_SKU_REGISTER       PackagePowerSkuMsr;
  MSR_PACKAGE_POWER_SKU_UNIT_REGISTER  PackagePowerSkuUnitMsr;
  MSR_PACKAGE_RAPL_LIMIT_REGISTER      PackageRaplLimitMsr;
  MSR_CONFIG_TDP_NOMINAL_REGISTER      ConfigTdpNominalMsr;
  MSR_CONFIG_TDP_LEVEL1_REGISTER       ConfigTdpLevel1Msr;
  MSR_CONFIG_TDP_LEVEL1_REGISTER       ConfigTdpLevel2Msr;
  MSR_TURBO_ACTIVATION_RATIO_REGISTER  TurboActivationRatioMsr;

  MaxBusRatio                            = 0;
  MinBusRatio                            = 0;
  PowerLimitInteger = PowerLimitFraction = 0;
  PowerUnit                              = 0;
  GetCpuInfo                             = NULL;
  RegEcx                                 = 0;

  mCpuDataHob = NULL;

  VariableSize  = sizeof (SETUP_VOLATILE_DATA);
  Status = gRT->GetVariable (
                  L"SetupVolatileData",
                  &gSetupVariableGuid,
                  NULL,
                  &VariableSize,
                  &mSetupVolatileData
                  );
  ASSERT_EFI_ERROR (Status);
  if (EFI_ERROR (Status)) {
    return ;
  }

  ///
  /// Locate DxeCpuInfo protocol instance and gather CPU information
  ///
  Status = gBS->LocateProtocol (&gCpuInfoProtocolGuid, NULL, (VOID **) &DxeCpuInfo);
  if (EFI_ERROR (Status)) {
    DEBUG ((DEBUG_ERROR,"Failed to locate DxeCpuInfo Protocol\n"));
    return;
  }

  GetBusRatio (&MaxBusRatio, &MinBusRatio);
  GetCpuInfo               = DxeCpuInfo->CpuInfo;
  GetCpuInfo->BrandString  = DxeCpuInfo->CpuInfo->BrandString;
  GetCpuInfo->CacheInfo    = DxeCpuInfo->CpuInfo->CacheInfo;
  GetCpuInfo->IntendedFreq = (10000 * MaxBusRatio) / 100;
  GetCpuInfo->Voltage      = 0;    //Voltage is variable, and no information os available.

  ///
  /// Update current Debug interface MSR enable/disable state
  ///
  AsmCpuid (CPUID_VERSION_INFO, NULL, NULL, &Ecx.Uint32, NULL);
  if (Ecx.Bits.SDBG == 1) {
    DebugInterfaceReg.Uint64 = AsmReadMsr64 (MSR_IA32_DEBUG_INTERFACE);
  }

  if (DebugInterfaceReg.Bits.DebugOccurred) {
    InitString (
      gHiiHandle,
      STRING_TOKEN (STR_DEBUG_MSR_INTERFACE_STATUS_VALUE),
      L"%a",
      "Enabled"
      );
  } else {
    InitString (
      gHiiHandle,
      STRING_TOKEN (STR_DEBUG_MSR_INTERFACE_STATUS_VALUE),
      L"%a",
      "Disabled"
      );
  }

  InitString (
    gHiiHandle,
    STRING_TOKEN (STR_PROCESSOR_VERSION_VALUE),
    L"%a",
    GetCpuInfo->BrandString
    );

  InitString (
    gHiiHandle,
    STRING_TOKEN (STR_PROCESSOR_SPEED_VALUE),
    L"%d MHz",
    GetCpuInfo->IntendedFreq
    );

  InitString (
    gHiiHandle,
    STRING_TOKEN (STR_PROCESSOR_ID_VALUE),
    L"0x%X",
    GetCpuInfo->CpuSignature
    );

  InitString (
    gHiiHandle,
    STRING_TOKEN (STR_PROCESSOR_PACKAGE_VALUE),
    L"%a",
    "Not Implemented Yet"
    );

  //
  // Update the Microcode Revision
  //
  MsrSignId.Uint64 = AsmReadMsr64 (MSR_BIOS_SIGN_ID);
  if (MsrSignId.Bits.PatchId != 0) {
    InitString (
    gHiiHandle,
    STRING_TOKEN (STR_PROCESSOR_MICROCODE_VALUE),
    L"%x",
    MsrSignId.Bits.PatchId
    );
  }

  InitString (
    gHiiHandle,
    STRING_TOKEN (STR_PROCESSOR_COUNT_VALUE),
    L"%dCore(s) / %dThread(s)",
    GetCpuInfo->NumCores,
    GetCpuInfo->NumCores * GetCpuInfo->NumHts
    );

  for (Index = 0; Index <= GetCpuInfo->MaxCacheSupported; ++Index) {
    switch (GetCpuInfo->CacheInfo[Index].Level) {
    case 1:
      //
      // L1 cache size is per core. Display the size per core times number of cores.
      // If there's only 1 enabled core, simply display the size per core.
      //
      if (GetCpuInfo->CacheInfo[Index].Type == 1) {
        if (GetCpuInfo->NumCores == 1) {
          InitString (
            gHiiHandle,
            STRING_TOKEN (STR_PROCESSOR_L1_DATA_CACHE_VALUE),
            L"%d KB",
            GetCpuInfo->CacheInfo[Index].Size
            );
        } else {
          InitString (
            gHiiHandle,
            STRING_TOKEN (STR_PROCESSOR_L1_DATA_CACHE_VALUE),
            L"%d KB x %d",
            GetCpuInfo->CacheInfo[Index].Size,
            GetCpuInfo->NumCores
            );
        }
      } else if (GetCpuInfo->CacheInfo[Index].Type == 2) {
        if (GetCpuInfo->NumCores == 1) {
          InitString (
            gHiiHandle,
            STRING_TOKEN (STR_PROCESSOR_L1_INSTR_CACHE_VALUE),
            L"%d KB",
            GetCpuInfo->CacheInfo[Index].Size
            );
        } else {
          InitString (
            gHiiHandle,
            STRING_TOKEN (STR_PROCESSOR_L1_INSTR_CACHE_VALUE),
            L"%d KB x %d",
            GetCpuInfo->CacheInfo[Index].Size,
            GetCpuInfo->NumCores
            );
        }
      }
      break;

    case 2:
      //
      // L2 cache size is per core. Display the size per core times number of cores.
      // If there's only 1 enabled core, simply display the size per core.
      //
      if (GetCpuInfo->NumCores == 1) {
        InitString (
          gHiiHandle,
          STRING_TOKEN (STR_PROCESSOR_L2_CACHE_VALUE),
          L"%d KB",
          GetCpuInfo->CacheInfo[Index].Size
          );
      } else {
        InitString (
          gHiiHandle,
          STRING_TOKEN (STR_PROCESSOR_L2_CACHE_VALUE),
          L"%d KB x %d",
          GetCpuInfo->CacheInfo[Index].Size,
          GetCpuInfo->NumCores
          );
      }
      break;

    case 3:
      //
      // L3 cache size is not per core. It is shared between cores.
      //
      InitString (
        gHiiHandle,
        STRING_TOKEN (STR_PROCESSOR_L3_CACHE_VALUE),
        L"%d MB",
        GetCpuInfo->CacheInfo[Index].Size / 1024
        );
      break;

    case 4:
      InitString (
        gHiiHandle,
        STRING_TOKEN (STR_PROCESSOR_L4_CACHE_VALUE),
        L"%d MB",
        GetCpuInfo->CacheInfo[Index].Size / 1024
        );
      break;
    }
  }

  GenerationString = GetGenerationString ();
  SkuString = GetSkuString ();
  if ((GenerationString != NULL) && (SkuString != NULL)) {
    InitString (
      gHiiHandle,
      STRING_TOKEN (STR_PROCESSOR_VALUE),
      L"%a %a",
      GenerationString,
      SkuString
    );
  } else {
    InitString (
      gHiiHandle,
      STRING_TOKEN (STR_PROCESSOR_VALUE),
      L"%a",
      "Unknown"
    );
  }

  VariableSize = sizeof (SETUP_CPU_FEATURES);
  Status = gRT->GetVariable (
                  L"SetupCpuFeatures",
                  &gSetupVariableGuid,
                  NULL,
                  &VariableSize,
                  &SetupCpuFeatures
                  );
  ASSERT_EFI_ERROR (Status);
  if (EFI_ERROR (Status)) {
    return ;
  }

  if (SetupCpuFeatures.VTAvailable) {
    AsciiSPrint (String, 15, "Supported");
  } else {
    AsciiSPrint (String, 15, "Not Supported");
  }

  InitString (
    gHiiHandle,
    STRING_TOKEN (STR_PROCESSOR_VMX_VALUE),
    L"%a",
    String
    );

  if (SetupCpuFeatures.TXTAvailable) {
    AsciiSPrint (String, 15, "Supported");
  } else {
    AsciiSPrint (String, 15, "Not Supported");
  }

  InitString (
    gHiiHandle,
    STRING_TOKEN (STR_PROCESSOR_SMX_VALUE),
    L"%a",
    String
    );

  //
  // Package Power SKU Unit
  //
  PackagePowerSkuUnitMsr.Uint64  = AsmReadMsr64 (MSR_PACKAGE_POWER_SKU_UNIT);
  PowerUnit = (UINT8) PackagePowerSkuUnitMsr.Bits.PwrUnit;
  PowerUnit = (UINT8) LShiftU64 (2, (PowerUnit - 1));

  if (PowerUnit == 0 ) {
    ASSERT (FALSE);
    return;
  }

  //
  // Min and Max Turbo Power Limit
  //
  PackagePowerSkuMsr.Uint64 = AsmReadMsr64 (MSR_PACKAGE_POWER_SKU);
  PowerLimitFraction = PowerLimitInteger = (UINT16) PackagePowerSkuMsr.Bits.PkgMaxPwr;

  //
  // 0 for Max Power Limit means no limit, so set the value to 0x7FFF (max value possible)
  //
  if (PowerLimitInteger == 0 ) {
    PowerLimitFraction = PowerLimitInteger= 0x7FFF;
  }

  PowerLimitInteger = PowerLimitInteger / PowerUnit;
  PowerLimitFraction = PowerLimitFraction % PowerUnit;

  // Now convert fractional part into 3 decimal place
  PowerLimitFraction = (PowerLimitFraction * 1000 ) / PowerUnit;

  InitString (
    gHiiHandle,
    STRING_TOKEN (STR_MAX_TURBO_POWER_LIMIT_VALUE),
    L"%d.%d",
    PowerLimitInteger,PowerLimitFraction
    );

  PowerLimitFraction = PowerLimitInteger = (UINT16) PackagePowerSkuMsr.Bits.PkgMinPwr;

  PowerLimitInteger = PowerLimitInteger / PowerUnit;
  PowerLimitFraction = PowerLimitFraction % PowerUnit;

  // Now convert fractional part into 3 decimal place
  PowerLimitFraction = (PowerLimitFraction * 1000 ) / PowerUnit;

  InitString (
    gHiiHandle,
    STRING_TOKEN (STR_MIN_TURBO_POWER_LIMIT_VALUE),
    L"%d.%d",
    PowerLimitInteger,PowerLimitFraction
    );

  //
  // Package TDP Limit
  //
  PowerLimitFraction = PowerLimitInteger = (UINT16) PackagePowerSkuMsr.Bits.PkgTdp;

  PowerLimitInteger = PowerLimitInteger / PowerUnit;
  PowerLimitFraction = PowerLimitFraction % PowerUnit;

  // Now convert fractional part into 3 decimal place
  PowerLimitFraction = (PowerLimitFraction * 1000 ) / PowerUnit;

  InitString (
    gHiiHandle,
    STRING_TOKEN (STR_TDP_LIMIT_VALUE),
    L"%d.%d",
    PowerLimitInteger,PowerLimitFraction
    );

  //
  // Turbo Power Limit
  //
  PackageRaplLimitMsr.Uint64 = AsmReadMsr64 (MSR_PACKAGE_RAPL_LIMIT);
  //
  // Long duration power limit
  //
  PowerLimitFraction = PowerLimitInteger = (UINT16) PackageRaplLimitMsr.Bits.PkgPwrLim1;

  PowerLimitInteger = PowerLimitInteger / PowerUnit;
  PowerLimitFraction = PowerLimitFraction % PowerUnit;

  // Now convert fractional part into 3 decimal place
  PowerLimitFraction = (PowerLimitFraction * 1000 ) / PowerUnit;

  InitString (
    gHiiHandle,
    STRING_TOKEN (STR_LONG_DUR_PWR_LIMIT_VALUE),
    L"%d.%d",
    PowerLimitInteger,
    PowerLimitFraction
    );

  //
  // Short duration power limit
  //    Notice, we are using TempMsr.Dwords.High, so reusing Power Limit 1 Mask
  //
  PowerLimitFraction = PowerLimitInteger = (UINT16) PackageRaplLimitMsr.Bits.PkgPwrLim2;

  PowerLimitInteger = PowerLimitInteger / PowerUnit;
  PowerLimitFraction = PowerLimitFraction % PowerUnit;

  // Now convert fractional part into 3 decimal place
  PowerLimitFraction = (PowerLimitFraction * 1000 ) / PowerUnit;

  InitString (
    gHiiHandle,
    STRING_TOKEN (STR_SHORT_DUR_PWR_LIMIT_VALUE),
    L"%d.%d",
    PowerLimitInteger,
    PowerLimitFraction
    );

  //
  // Turbo ratio limit for 1Core, 2Core, 3Core, 4Core, 5Core, 6Core, 7Core, 8Core
  //
  TurboRatioLimit.Uint64 = AsmReadMsr64 (MSR_TURBO_RATIO_LIMIT);
  CoreRatio = (UINT8) TurboRatioLimit.Bits.MaxTurbo1Core;
  if (mCoreRatioFinal1 == 0) {
    mCoreRatioFinal1 = CoreRatio;
  }

  InitString (
    gHiiHandle,
    STRING_TOKEN (STR_1C_RATIO_VALUE),
    L"%d",
    CoreRatio
    );

  CoreRatio = (UINT8) TurboRatioLimit.Bits.MaxTurbo2Core;
  if (mCoreRatioFinal2 == 0) {
    mCoreRatioFinal2 = CoreRatio;
  }

  InitString (
    gHiiHandle,
    STRING_TOKEN (STR_2C_RATIO_VALUE),
    L"%d",
    CoreRatio
    );

  CoreRatio = (UINT8) TurboRatioLimit.Bits.MaxTurbo3Core;
  if (mCoreRatioFinal3 == 0) {
    mCoreRatioFinal3 = CoreRatio;
  }

  InitString (
    gHiiHandle,
    STRING_TOKEN (STR_3C_RATIO_VALUE),
    L"%d",
    CoreRatio
    );

  CoreRatio = (UINT8) TurboRatioLimit.Bits.MaxTurbo4Core;
  if (mCoreRatioFinal4 == 0) {
    mCoreRatioFinal4 = CoreRatio;
  }
  InitString (
    gHiiHandle,
    STRING_TOKEN (STR_4C_RATIO_VALUE),
    L"%d",
    CoreRatio
    );

  CoreRatio = (UINT8) TurboRatioLimit.Bits.MaxTurbo5Core;
  if (mCoreRatioFinal5 == 0) {
    mCoreRatioFinal5 = CoreRatio;
  }
  InitString (
    gHiiHandle,
    STRING_TOKEN (STR_5C_RATIO_VALUE),
    L"%d",
    CoreRatio
    );

  CoreRatio = (UINT8) TurboRatioLimit.Bits.MaxTurbo6Core;
  if (mCoreRatioFinal6 == 0) {
    mCoreRatioFinal6 = CoreRatio;
  }
  InitString (
    gHiiHandle,
    STRING_TOKEN (STR_6C_RATIO_VALUE),
    L"%d",
    CoreRatio
    );

  CoreRatio = (UINT8) TurboRatioLimit.Bits.MaxTurbo7Core;
  if (mCoreRatioFinal7 == 0) {
    mCoreRatioFinal7 = CoreRatio;
  }
  InitString (
    gHiiHandle,
    STRING_TOKEN (STR_7C_RATIO_VALUE),
    L"%d",
    CoreRatio
    );

  CoreRatio = (UINT8) TurboRatioLimit.Bits.MaxTurbo8Core;
  if (mCoreRatioFinal8 == 0) {
    mCoreRatioFinal8 = CoreRatio;
  }
  InitString (
    gHiiHandle,
    STRING_TOKEN (STR_8C_RATIO_VALUE),
    L"%d",
    CoreRatio
    );

  //
  // Get the number of configurable TDP Levels supported
  //
  CtdpLevels = GetConfigTdpLevels ();
  InitString (
    gHiiHandle,
    STRING_TOKEN (STR_CTDP_LEVELS_VALUE),
    L"%d",
    CtdpLevels + 1
    );

  if (CtdpLevels != 0) {

    //
    // Get Nominal Ratio
    //
    ConfigTdpNominalMsr.Uint64 = AsmReadMsr64 (MSR_CONFIG_TDP_NOMINAL);
    CtdpRatio = (UINT8) ConfigTdpNominalMsr.Bits.TdpRatio;

    // Package TDP Limit
    PowerLimitFraction                      = PowerLimitInteger = (UINT8) PackagePowerSkuMsr.Bits.PkgTdp;
    PowerLimitInteger                       = PowerLimitInteger / PowerUnit;
    PowerLimitFraction                      = PowerLimitFraction % PowerUnit;
    // Now convert fractional part into 3 decimal place
    PowerLimitFraction = (PowerLimitFraction * 1000) / PowerUnit;

    InitString (
      gHiiHandle,
      STRING_TOKEN (STR_CTDP_NOMINAL_VALUE),
      L"Ratio:%d TAR:%d PL1:%d.%dW",
      CtdpRatio,
      CtdpRatio - 1,
      PowerLimitInteger,
      PowerLimitFraction
      );

    //
    // Get Level1 Ratio and TDP
    //
    ConfigTdpLevel1Msr.Uint64   = AsmReadMsr64 (MSR_CONFIG_TDP_LEVEL1);
    CtdpRatio = (UINT8) ConfigTdpLevel1Msr.Bits.TdpRatio;

    PowerLimitFraction                      = PowerLimitInteger = (UINT8) ConfigTdpLevel1Msr.Bits.PkgTdp;
    PowerLimitInteger                       = PowerLimitInteger / PowerUnit;
    PowerLimitFraction                      = PowerLimitFraction % PowerUnit;
    // Now convert fractional part into 3 decimal place
    PowerLimitFraction = (PowerLimitFraction * 1000) / PowerUnit;

    InitString (
      gHiiHandle,
      STRING_TOKEN (STR_CTDP_LEVEL1_VALUE),
      L"Ratio:%d TAR:%d PL1:%d.%dW",
      CtdpRatio,
      CtdpRatio - 1,
      PowerLimitInteger,
      PowerLimitFraction
      );

    //
    // Get Level2 Ratio and TDP
    //
    ConfigTdpLevel2Msr.Uint64   = AsmReadMsr64 (MSR_CONFIG_TDP_LEVEL2);
    CtdpRatio = (UINT8) ConfigTdpLevel2Msr.Bits.TdpRatio;

    PowerLimitFraction                      = PowerLimitInteger = (UINT8) ConfigTdpLevel2Msr.Bits.PkgTdp;
    PowerLimitInteger                       = PowerLimitInteger / PowerUnit;
    PowerLimitFraction                      = PowerLimitFraction % PowerUnit;
    // Now convert fractional part into 3 decimal place
    PowerLimitFraction = (PowerLimitFraction * 1000) / PowerUnit;

    InitString (
      gHiiHandle,
      STRING_TOKEN (STR_CTDP_LEVEL2_VALUE),
      L"Ratio:%d TAR:%d PL1:%d.%dW",
      CtdpRatio,
      CtdpRatio - 1,
      PowerLimitInteger,
      PowerLimitFraction
      );

    //
    // Get MMIO Power Limit 1
    //
    CtdpAddress = (UINT32) ((UINTN) PcdGet64 (PcdMchBaseAddress) + 0x59A0);
    CtdpTdp = (UINT16) ((*((UINT32*) ((UINTN) CtdpAddress))) & 0x7FFF);

    PowerLimitFraction                      = PowerLimitInteger = CtdpTdp;
    PowerLimitInteger                       = PowerLimitInteger / PowerUnit;
    PowerLimitFraction                      = PowerLimitFraction % PowerUnit;
    // Now convert fractional part into 3 decimal place
    PowerLimitFraction = (PowerLimitFraction * 1000) / PowerUnit;

    //
    // Get MSR Power Limit 1
    //
    PackageRaplLimitMsr.Uint64 = AsmReadMsr64 (MSR_PACKAGE_RAPL_LIMIT);
    PowerLimitFraction2 = PowerLimitInteger2 = (UINT16) PackageRaplLimitMsr.Bits.PkgPwrLim1;
    PowerLimitInteger2                      = PowerLimitInteger2 / PowerUnit;
    PowerLimitFraction2                     = PowerLimitFraction2 % PowerUnit;
    // Now convert fractional part into 3 decimal place
    PowerLimitFraction2 = (PowerLimitFraction2 * 1000) / PowerUnit;

    InitString (
      gHiiHandle,
      STRING_TOKEN (STR_CTDP_PWR_LIMIT1_VALUE),
      L"%d.%dW (MSR:%d.%d)",
      PowerLimitInteger,
      PowerLimitFraction,
      PowerLimitInteger2,
      PowerLimitFraction2
      );

    //
    // Get MMIO Power Limit 2
    //
    CtdpAddress = (UINT32) ((UINTN) PcdGet64 (PcdMchBaseAddress) + 0x59A4);
    CtdpTdp = (UINT16) ((*((UINT32*) ((UINTN) CtdpAddress))) & 0x7FFF);

    PowerLimitFraction                      = PowerLimitInteger = CtdpTdp;
    PowerLimitInteger                       = PowerLimitInteger / PowerUnit;
    PowerLimitFraction                      = PowerLimitFraction % PowerUnit;
    // Now convert fractional part into 3 decimal place
    PowerLimitFraction = (PowerLimitFraction * 1000) / PowerUnit;

    //
    // Get MSR Power Limit 2
    //
    PowerLimitFraction2                     = PowerLimitInteger2 = (UINT16) PackageRaplLimitMsr.Bits.PkgPwrLim2;
    PowerLimitInteger2                      = PowerLimitInteger2 / PowerUnit;
    PowerLimitFraction2                     = PowerLimitFraction2 % PowerUnit;
    // Now convert fractional part into 3 decimal place
    PowerLimitFraction2 = (PowerLimitFraction2 * 1000) / PowerUnit;

    InitString (
      gHiiHandle,
      STRING_TOKEN (STR_CTDP_PWR_LIMIT2_VALUE),
      L"%d.%dW (MSR:%d.%d)",
      PowerLimitInteger,
      PowerLimitFraction,
      PowerLimitInteger2,
      PowerLimitFraction2
      );

    //
    // Get TAR value
    //
    TurboActivationRatioMsr.Uint64   = AsmReadMsr64 (MSR_TURBO_ACTIVATION_RATIO);
    if (TurboActivationRatioMsr.Bits.TurboActivationRatioLock) {
      InitString (
        gHiiHandle,
        STRING_TOKEN (STR_CTDP_TAR_VALUE),
        L"%d (Locked)",
        TurboActivationRatioMsr.Bits.MaxNonTurboRatio
        );
    } else {
      InitString (
        gHiiHandle,
        STRING_TOKEN (STR_CTDP_TAR_VALUE),
        L"%d (Unlocked)",
        TurboActivationRatioMsr.Bits.MaxNonTurboRatio
        );
    }
  }

  VariableSize = sizeof (CPU_SETUP_VOLATILE_DATA);
  Status = gRT->GetVariable (
                  L"CpuSetupVolatileData",
                  &gCpuSetupVariableGuid,
                  &CpuSetupVolVarAttr,
                  &VariableSize,
                  &CpuSetupVolData
                  );
  if (EFI_ERROR (Status)) {
    CpuSetupVolVarAttr = EFI_VARIABLE_BOOTSERVICE_ACCESS | EFI_VARIABLE_RUNTIME_ACCESS;
  }
  ASSERT_EFI_ERROR (Status);

 ///
 /// Get CPU VR topology
 ///
  Hob = (CPU_DATA_HOB *) GetFirstGuidHob (&gCpuDataHobGuid);
  mCpuDataHob = (CPU_DATA_HOB *)((UINTN)Hob + sizeof (EFI_HOB_GUID_TYPE));

  if (mCpuDataHob == NULL) {
    DEBUG((DEBUG_ERROR, "CPU Data HOB not available\n"));
    ASSERT (mCpuDataHob != NULL);
  }

  //
  // Lock the VR submenu if it does not use SVID.
  // VR Type. 1 - No SVID VR, 0 - SVID VR
  //
  if (mCpuDataHob == NULL) {
    CpuSetupVolData.VccInVrLocked = 0;
  } else {
    CpuSetupVolData.VccInVrLocked = ~(mCpuDataHob->SvidEnabled);
  }

  Status = gRT->SetVariable (
                  L"CpuSetupVolatileData",
                  &gCpuSetupVariableGuid,
                  CpuSetupVolVarAttr,
                  sizeof (CpuSetupVolData),
                  &CpuSetupVolData
                  );
  ASSERT_EFI_ERROR (Status);

  ///
  /// Calculate RFI nominal frequency.
  ///
  RfiNominalValue = 0;
  MchBar = PciSegmentRead32 (PCI_SEGMENT_LIB_ADDRESS (SA_SEG_NUM, SA_MC_BUS, SA_MC_DEV, SA_MC_FUN, R_SA_MCHBAR)) & ~BIT0;
  MmioValue64 = ((UINT64)MmioRead32 (MchBar + R_RFI_STATUS_0_0_0_MCHBAR_PCU) &~BIT0) + LShiftU64 ((UINT64)MmioRead32 (MchBar + R_RFI_STATUS_0_0_0_MCHBAR_PCU + 4), 32);


  ///
  /// Register definition varies by SKU
  ///
    RfiNominalValue = (UINT32) ((MmioValue64) & B_RFI_CURRENT_FREQ_MASK);
  ///
  /// Get XTAL clock frequency
  ///
  AsmCpuid(CPUID_TIME_STAMP_COUNTER, NULL, NULL, &RegEcx, NULL);

  ///
  /// Convert value to frequency in KHz by multiplying by ClockFreq / 128.
  /// So multiply by either 38400 / 128 = 300, or 24000 / 128 = 375 / 2.
  ///
  if (RegEcx == CLOCK_FREQUENCY_38MHz) {
    RfiTotalFreqInKhz = (UINT32) (RfiNominalValue * 300);
  } else if (RegEcx == CLOCK_FREQUENCY_24MHz) {
    RfiTotalFreqInKhz = (UINT32) ((RfiNominalValue * 375) / 2);
  } else {
    DEBUG ((DEBUG_ERROR, "Unexpected Clock Frequency! RegEcx: %d\n", RegEcx));
    RfiTotalFreqInKhz = 0;
  }

  RfiFreqMhz = (UINT16) (RfiTotalFreqInKhz / 1000);
  RfiFreqKhz = (UINT16) (RfiTotalFreqInKhz % 1000);

  InitString (
    gHiiHandle,
    STRING_TOKEN (STR_CPU_RFI_NOM_FREQ_VALUE),
    L"%d.%dMHz",
    RfiFreqMhz,
    RfiFreqKhz
    );

  ///
  /// In TGL platform, only TGL H patch is able to support the OCMB commmand 0x1C
  ///
  if (EnumCpuHalo == GetCpuSku ()) {
    //
    // Locate the MP services protocol
    // Find the MP Protocol. This is an MP platform, so MP protocol must be there.
    //
    Status = gBS->LocateProtocol (&gEfiMpServiceProtocolGuid, NULL, (VOID **) &mMpService);
    ASSERT_EFI_ERROR (Status);
    if (EFI_ERROR (Status)) {
      return;
    }

    MsrCoreThreadCount.Uint64 = AsmReadMsr64 (MSR_CORE_THREAD_COUNT);

    NumberOfCores = (UINTN) MsrCoreThreadCount.Bits.Corecount;
    LogProcNum    = (UINTN) MsrCoreThreadCount.Bits.Threadcount;

    MailboxReadParameters.MailboxDataPtr = AllocateZeroPool (sizeof (UINT32) * NumberOfCores);
    MailboxReadParameters.MailboxStatus  = AllocateZeroPool (sizeof (EFI_STATUS) * NumberOfCores);

    if ((MailboxReadParameters.MailboxDataPtr == NULL) || (MailboxReadParameters.MailboxStatus == NULL)) {
      return;
    }

    Status = mMpService->WhoAmI (mMpService, &BspIndex);
    //
    //  Read Favored Core Information from each core index
    //
    for (Index = 0; Index < LogProcNum; Index++) {
      if (Index == BspIndex) {
        MailboxReadFavoredCore ((VOID*) &MailboxReadParameters);
      } else {
        mMpService->StartupThisAP (
                      mMpService,
                      (EFI_AP_PROCEDURE) MailboxReadFavoredCore,
                      Index,
                      NULL,
                      0,
                      (VOID*) &MailboxReadParameters,
                      NULL
                      );
      }
    }

    for (ProcNum = 0; ProcNum < NumberOfCores; ProcNum++) {
      InitString (
        gHiiHandle,
        STRING_TOKEN (FuseFreqString [ProcNum]),
        L"%d",
        ((UINT32) RShiftU64 (MailboxReadParameters.MailboxDataPtr [ProcNum], 8) & 0xFF)
        );
    }
    FreePool (MailboxReadParameters.MailboxDataPtr);
    FreePool (MailboxReadParameters.MailboxStatus);
  }
}

/**
  Cpu setup callback

  @param[in] EFI_EVENT    Event
  @param[in] VOID         *Context
**/
VOID
EFIAPI
CpuSetupCallback (
  IN EFI_EVENT Event,
  IN VOID      *Context
  )
{
  EFI_STATUS                Status;
  UINTN                     VariableSize;

  VariableSize = sizeof (CPU_SETUP);
  Status = gRT->GetVariable (
                  L"CpuSetup",
                  &gCpuSetupVariableGuid,
                  NULL,
                  &VariableSize,
                  &mCpuSetup
                  );
  ASSERT_EFI_ERROR (Status);
  if (EFI_ERROR (Status)) {
    return ;
  }

  SaveCpuRatio (mCpuSetup.CpuRatio);
  ///
  /// Save the Alias Check request to use in early PEI phase
  ///
  SaveTxtAliasCheck (mCpuSetup.AcheckRequest);

}

/**
  Initialize CPU strings.

  @param[in] EFI_HII_HANDLE   HiiHandle
  @param[in] UINT16           Class
**/
VOID
InitCPUStrings (
  EFI_HII_HANDLE HiiHandle,
  UINT16         Class
  )
{
  EFI_EVENT             SetupNvramCallbackEvt;
  VOID                  *SetupNvramCallbackReg;
  CONST CHAR8           *RevisionTableString = NULL;

  if ((Class == MAIN_FORM_SET_CLASS) || (Class == ADVANCED_FORM_SET_CLASS)) {
    DEBUG ((DEBUG_INFO, "<InitCPUStrings>"));
    gHiiHandle  = HiiHandle;

    InitTurboRatioDefault (NULL, NULL);
    InitTxtAcheckDefault ();
    InitDCDInfo ();
    InitCPUInfo (NULL, NULL);
///
///

    SetupNvramCallbackEvt = EfiCreateProtocolNotifyEvent (
                              &gSetupNvramUpdateGuid,
                              TPL_CALLBACK,
                              CpuSetupCallback,
                              NULL,
                              &SetupNvramCallbackReg
                              );
    ASSERT (SetupNvramCallbackEvt != NULL);
  }

  if (Class == MAIN_FORM_SET_CLASS) {
  RevisionTableString = GetRevisionTable ();
        InitString (
          HiiHandle,
          STRING_TOKEN (STR_PROCESSOR_STEPPING_VALUE),
          L"%a",
          RevisionTableString
          );
  }
  InitCpuVrConfig (); ///< Display CPU VR config default programmed values.
  InitPkgCxLimit ();
}

/**
  This function validates the Flex Ratio setup value

  @param[in] EFI_FORM_CALLBACK_PROTOCOL   *This
  @param[in] UINT16                       KeyValue
  @param[in] EFI_IFR_DATA_ARRAY           *Data,
  @param[in] EFI_HII_CALLBACK_PACKET      **Packet

  @retval EFI_SUCCESS        Call Back Function executed successfully
**/
EFI_STATUS
EFIAPI
CpuFormCallBackFunction (
  IN CONST EFI_HII_CONFIG_ACCESS_PROTOCOL *This,
  IN EFI_BROWSER_ACTION                   Action,
  IN EFI_QUESTION_ID                      KeyValue,
  IN UINT8                                Type,
  IN EFI_IFR_TYPE_VALUE                   *Value,
  OUT EFI_BROWSER_ACTION_REQUEST          *ActionRequest
  )
{
  CPU_SETUP               *CpuSetup;
  UINT8                   RatioLimitMin;
  UINT8                   OverclockingBins;
  UINTN                   VarSize;
  UINTN                   VariableSize;
  EFI_STATUS              Status;
  EFI_STRING              RequestString;
  UINT8                   MaxBusRatio;
  UINT8                   MinBusRatio;
  MSR_FLEX_RATIO_REGISTER FlexRatioMsr;

  MaxBusRatio = 0;
  MinBusRatio = 0;

  Status = EFI_UNSUPPORTED;
  if (Action != EFI_BROWSER_ACTION_CHANGING && Action != EFI_BROWSER_ACTION_CHANGED) {
    if (Action == EFI_BROWSER_ACTION_DEFAULT_STANDARD) {
      Status = EFI_SUCCESS;
      switch (KeyValue) {
        case KEY_CpuRatioLimit:
          break;

        case KEY_RatioLimit1:
          Value->u8 = mCpuSetup.RatioLimit1Default;
          break;

        case KEY_RatioLimit2:
          Value->u8 = mCpuSetup.RatioLimit2Default;
          break;

        case KEY_RatioLimit3:
          Value->u8 = mCpuSetup.RatioLimit3Default;
          break;

        case KEY_RatioLimit4:
          Value->u8 = mCpuSetup.RatioLimit4Default;
          break;

        case KEY_RatioLimit5:
          Value->u8 = mCpuSetup.RatioLimit5Default;
          break;

        case KEY_RatioLimit6:
          Value->u8 = mCpuSetup.RatioLimit6Default;
          break;

        case KEY_RatioLimit7:
          Value->u8 = mCpuSetup.RatioLimit7Default;
          break;

        case KEY_RatioLimit8:
          Value->u8 = mCpuSetup.RatioLimit8Default;
          break;

        default:
          Status = EFI_UNSUPPORTED;
      }
    }
    return Status;
  }

  RequestString = NULL;
  Status = EFI_SUCCESS;
  VarSize = sizeof (CPU_SETUP);
  CpuSetup = AllocatePool (VarSize);
  ASSERT (CpuSetup != NULL);
  if (CpuSetup == NULL) {
    return EFI_OUT_OF_RESOURCES;
  }

  // GetBrowserData by VarStore Name (Setup)
  if (!HiiGetBrowserData (&gCpuSetupVariableGuid, L"CpuSetup", VarSize, (UINT8 *) CpuSetup)) {
    Status = EFI_NOT_FOUND;
  }
  ASSERT_EFI_ERROR (Status);

  //
  // Get Maximum Non-Turbo bus ratio (HFM) from Platform Info MSR Bits[15:8]
  //
  GetBusRatio (&MaxBusRatio, &MinBusRatio);

  //
  // Save the Max Non Turbo Ratio
  //
  if (MaxBusRatio > CpuSetup->MaxNonTurboRatio) {
    CpuSetup->MaxNonTurboRatio = MaxBusRatio;
  } else {
    MaxBusRatio = CpuSetup->MaxNonTurboRatio;
  }
  FlexRatioMsr.Uint64       = AsmReadMsr64 (MSR_FLEX_RATIO);
  OverclockingBins         = (UINT8) FlexRatioMsr.Bits.OcBins;

  VariableSize = sizeof (CPU_SETUP);
  Status = gRT->GetVariable (
                  L"CpuSetup",
                  &gCpuSetupVariableGuid,
                  NULL,
                  &VariableSize,
                  &mCpuSetup
                  );
  ASSERT_EFI_ERROR (Status);

  //
  // If Flex Ratio Override is disabled, use the original fused value.
  // Otherwise, use the current Flex Ratio value.
  //
  if (mCpuSetup.CpuRatioOverride == 0) {
    RatioLimitMin = mCpuSetup.FlexRatioOverrideDefault;
  } else {
    RatioLimitMin = mCpuSetup.CpuRatio;
  }

  switch (KeyValue) {
    case KEY_CpuRatioLimit:
      break;

    case KEY_RatioLimit1:
      if (CpuSetup->RatioLimit1 < RatioLimitMin) {
        CpuSetup->RatioLimit1 = mCoreRatioFinal1;
      } else if (OverclockingBins != 0x07) {
        if (CpuSetup->RatioLimit1 > (mCpuSetup.RatioLimit1Default + OverclockingBins)) {
          CpuSetup->RatioLimit1 = mCoreRatioFinal1;
        }
      }
      mCoreRatioFinal1 = CpuSetup->RatioLimit1;
      RequestString = HiiConstructRequestString (RequestString, OFFSET_OF (CPU_SETUP, RatioLimit1), sizeof (CpuSetup->RatioLimit1));
      break;

    case KEY_RatioLimit2:
      if (CpuSetup->RatioLimit2 < RatioLimitMin) {
        CpuSetup->RatioLimit2 = mCoreRatioFinal2;
      } else if (OverclockingBins != 0x07) {
        if (CpuSetup->RatioLimit2 > (mCpuSetup.RatioLimit2Default + OverclockingBins)) {
          CpuSetup->RatioLimit2 = mCoreRatioFinal2;
        }
      }
      mCoreRatioFinal2 = CpuSetup->RatioLimit2;
      RequestString = HiiConstructRequestString (RequestString, OFFSET_OF (CPU_SETUP, RatioLimit2), sizeof (CpuSetup->RatioLimit1));
      break;

    case KEY_RatioLimit3:
      if (CpuSetup->RatioLimit3 < RatioLimitMin) {
        CpuSetup->RatioLimit3 = mCoreRatioFinal3;
      } else if (OverclockingBins != 0x07) {
        if (CpuSetup->RatioLimit3 > (mCpuSetup.RatioLimit3Default + OverclockingBins)) {
          CpuSetup->RatioLimit3 = mCoreRatioFinal3;
        }
      }
      mCoreRatioFinal3 = CpuSetup->RatioLimit3;
      RequestString = HiiConstructRequestString (RequestString, OFFSET_OF (CPU_SETUP, RatioLimit3), sizeof (CpuSetup->RatioLimit1));
      break;

    case KEY_RatioLimit4:
      if (CpuSetup->RatioLimit4 < RatioLimitMin) {
        CpuSetup->RatioLimit4 = mCoreRatioFinal4;
      } else if (OverclockingBins != 0x07) {
        if (CpuSetup->RatioLimit4 > (mCpuSetup.RatioLimit4Default + OverclockingBins)) {
          CpuSetup->RatioLimit4 = mCoreRatioFinal4;
        }
      }
      mCoreRatioFinal4 = CpuSetup->RatioLimit4;
      RequestString = HiiConstructRequestString (RequestString, OFFSET_OF (CPU_SETUP, RatioLimit4), sizeof (CpuSetup->RatioLimit1));
      break;

    case KEY_RatioLimit5:
      if (CpuSetup->RatioLimit5 < RatioLimitMin) {
        CpuSetup->RatioLimit5 = mCoreRatioFinal5;
      } else if (OverclockingBins != 0x07) {
        if (CpuSetup->RatioLimit5 > (mCpuSetup.RatioLimit5Default + OverclockingBins)) {
          CpuSetup->RatioLimit5 = mCoreRatioFinal5;
        }
      }
      mCoreRatioFinal5 = CpuSetup->RatioLimit5;
      RequestString = HiiConstructRequestString (RequestString, OFFSET_OF (CPU_SETUP, RatioLimit5), sizeof (CpuSetup->RatioLimit1));
      break;

   case KEY_RatioLimit6:
      if (CpuSetup->RatioLimit6 < RatioLimitMin) {
        CpuSetup->RatioLimit6 = mCoreRatioFinal6;
      } else if (OverclockingBins != 0x07) {
        if (CpuSetup->RatioLimit6 > (mCpuSetup.RatioLimit6Default + OverclockingBins)) {
          CpuSetup->RatioLimit6 = mCoreRatioFinal6;
        }
      }
      mCoreRatioFinal6 = CpuSetup->RatioLimit6;
      RequestString = HiiConstructRequestString (RequestString, OFFSET_OF (CPU_SETUP, RatioLimit6), sizeof (CpuSetup->RatioLimit1));
      break;

  case KEY_RatioLimit7:
      if (CpuSetup->RatioLimit7 < RatioLimitMin) {
        CpuSetup->RatioLimit7 = mCoreRatioFinal7;
      } else if (OverclockingBins != 0x07) {
        if (CpuSetup->RatioLimit7 > (mCpuSetup.RatioLimit7Default + OverclockingBins)) {
          CpuSetup->RatioLimit7 = mCoreRatioFinal7;
        }
      }
      mCoreRatioFinal7 = CpuSetup->RatioLimit7;
      RequestString = HiiConstructRequestString (RequestString, OFFSET_OF (CPU_SETUP, RatioLimit7), sizeof (CpuSetup->RatioLimit1));
      break;

  case KEY_RatioLimit8:
      if (CpuSetup->RatioLimit8 < RatioLimitMin) {
        CpuSetup->RatioLimit8 = mCoreRatioFinal8;
      } else if (OverclockingBins != 0x07) {
        if (CpuSetup->RatioLimit8 > (mCpuSetup.RatioLimit8Default + OverclockingBins)) {
          CpuSetup->RatioLimit8 = mCoreRatioFinal8;
        }
      }
      mCoreRatioFinal8 = CpuSetup->RatioLimit8;
      RequestString = HiiConstructRequestString (RequestString, OFFSET_OF (CPU_SETUP, RatioLimit8), sizeof (CpuSetup->RatioLimit1));
      break;

    default:
      ASSERT (FALSE);
  }

  if (RequestString != NULL) {
    VarSize = sizeof (CPU_SETUP);
    if (!HiiSetBrowserData (&gCpuSetupVariableGuid, L"CpuSetup", VarSize, (UINT8 *) CpuSetup, RequestString)) {
      Status = EFI_NOT_FOUND;
    }
    ASSERT_EFI_ERROR (Status);
    FreePool (RequestString);
  }

  FreePool (CpuSetup);

  return EFI_SUCCESS;
}

/**
  Initial CPU MNTR default in SETUP variable and VFR.

  @retval EFI_SUCCESS             The initialization is done.
  @retval EFI_NOT_FOUND           Failed to initial CPU MNTR.
**/
EFI_STATUS
EFIAPI
InitCpuMntrDefault (
  VOID
  )
{
  EFI_STATUS    Status;
  UINTN         VariableSize;
  UINT32        Attributes;
  UINT8         MaxBusRatio;
  UINT8         MinBusRatio;

  MaxBusRatio = 0;
  MinBusRatio = 0;

  VariableSize = sizeof (CPU_SETUP);
  Status = gRT->GetVariable (
                  L"CpuSetup",
                  &gCpuSetupVariableGuid,
                  &Attributes,
                  &VariableSize,
                  &mCpuSetup
                  );
  if (EFI_ERROR (Status)) {
    return Status;
  }

  GetBusRatio (&MaxBusRatio, &MinBusRatio);
  if (mCpuSetup.CpuRatioOverride == 0) {
    mCpuSetup.CpuDefaultRatio = MaxBusRatio;
  }
  mCpuSetup.CpuRatio = MaxBusRatio;

  Status = gRT->SetVariable (
                  L"CpuSetup",
                  &gCpuSetupVariableGuid,
                  Attributes,
                  VariableSize,
                  &mCpuSetup
                  );
  Status = InitCpuMntrDefaultVfr (mCpuSetup.CpuDefaultRatio);

  return EFI_SUCCESS;
}


/**
  Update the min, max, and default values for CpuRatio.

  @param[in] CpuRatioDefault      The CPU MNTR default.

  @retval EFI_SUCCESS             Values updated successfully.
  @retval EFI_NOT_FOUND           Failed to update it.
**/
EFI_STATUS
InitCpuMntrDefaultVfr (
  UINT8      CpuRatioDefault
  )
{
  EFI_HII_PACKAGE_HEADER *PackageHdr;
  CHAR8                  *BytePtr;
  EFI_IFR_OP_HEADER      *ParentIfr;
  EFI_IFR_GUID_LABEL     *LabelPtr;
  EFI_IFR_DEFAULT        *DefaultPtr;
  UINTN                  PackageSize;
  EFI_IFR_NUMERIC        *NumericPtr;

  //
  // add Array Length (UINT32) to point to package header.
  //
  PackageHdr = (EFI_HII_PACKAGE_HEADER *) (AdvancedBin + sizeof (UINT32));
  PackageSize = PackageHdr->Length;
  PackageSize = PackageSize & 0x00ffffff;
  BytePtr = (CHAR8 *) (PackageHdr + 1);

  //
  // loop to find CPU RATIO label.
  //
  while (TRUE) {
    if (BytePtr >= (CHAR8 *) PackageHdr + PackageSize) {
      DEBUG ((DEBUG_ERROR, "Couldn't find the target node to patch.\n"));
      return EFI_NOT_FOUND;
    }

    ParentIfr = (EFI_IFR_OP_HEADER *) BytePtr;
    if (ParentIfr->OpCode == EFI_IFR_GUID_OP) {
      LabelPtr = (EFI_IFR_GUID_LABEL *) ParentIfr;
      if (CompareGuid ((EFI_GUID *) (VOID *) &LabelPtr->Guid, &gEfiIfrTianoGuid) && (LabelPtr->Number == LABEL_CPU_RATIO)) {
        BytePtr += sizeof (EFI_IFR_GUID_LABEL);
        break;
      }
    }
    BytePtr += ParentIfr->Length;
  }

  //
  // loop to find CPU ratio numeric OP code.
  //
  while (TRUE) {
    if (BytePtr >= (CHAR8 *) PackageHdr + PackageSize) {
      DEBUG ((DEBUG_ERROR, "Couldn't find the target node to patch.\n"));
      return EFI_NOT_FOUND;
    }

    ParentIfr = (EFI_IFR_OP_HEADER *) BytePtr;
    if (ParentIfr->OpCode == EFI_IFR_NUMERIC_OP) {
      NumericPtr = (EFI_IFR_NUMERIC *) ParentIfr;
      NumericPtr->data.u8.MinValue = GetMaxEfficiencyRatio ();
      NumericPtr->data.u8.MaxValue = CpuRatioDefault;
      break;
    }
    BytePtr += ParentIfr->Length;
  }

  //
  // advance to numeric default OP code.
  //
  while (TRUE) {
    if (BytePtr >= (CHAR8 *) PackageHdr + PackageSize) {
      DEBUG ((DEBUG_ERROR, "Couldn't find the target node to patch.\n"));
      return EFI_NOT_FOUND;
    }

    ParentIfr = (EFI_IFR_OP_HEADER *) BytePtr;
    if (ParentIfr->OpCode == EFI_IFR_DEFAULT_OP) {
      DefaultPtr = (EFI_IFR_DEFAULT *) ParentIfr;
      break;
    }
    BytePtr += ParentIfr->Length;
  }

  //
  // Now the Default value is found. Patch it!
  //
  DefaultPtr->Value.u8 = CpuRatioDefault;
  return EFI_SUCCESS;
}


