/** @file
  Dxe Telemetry Lib implementation.

  Copyright (c) 2018 - 2019, Intel Corporation. All rights reserved.<BR>
  This program and the accompanying materials
  are licensed and made available under the terms and conditions of the BSD License
  which accompanies this distribution. The full text of the license may be found at
  http://opensource.org/licenses/bsd-license.php

  THE PROGRAM IS DISTRIBUTED UNDER THE BSD LICENSE ON AN "AS IS" BASIS,
  WITHOUT WARRANTIES OR REPRESENTATIONS OF ANY KIND, EITHER EXPRESS OR IMPLIED.
**/
#include "DxeTelemetryFviLib.h"

/**
  Create the adapter information FVI block from FVI arrays.

  @param[in] *Data       Pointer to an array of TELEMETRY_VERSION_RECORD_DATA.
  @param[in] Count       Telemetry record counts.
  @param[out] *Block     Pointer to an block of Adapter Information Data.
  @param[out] *BlockSize Pointer to Data block size.

  @retval EFI_SUCCESS           - if the data is successfully reported.
  @retval EFI_OUT_OF_RESOURCES  - if not able to get resources.
  @retval EFI_INVALID_PARAMETER - if Data == NULL.
  @retval EFI_INVALID_PARAMETER - if Block == NULL.
  @retval EFI_INVALID_PARAMETER - if BlockSize == NULL.
  @retval EFI_INVALID_PARAMETER - if Count > MAX_UINT16.
  @retval EFI_INVALID_PARAMETER - if FVI block totalsize > MAX_UINT16.
**/
EFI_STATUS
EFIAPI
CreateTelemetryFviBlock (
  IN TELEMETRY_VERSION_RECORD_DATA    *Data,
  IN UINTN                            Count,
  OUT EFI_AIP_TELEMETRY_VERSION_DATA  **Block,
  OUT UINTN                           *BlockSize
  )
{
  UINTN                             HeaderSize;
  UINTN                             DataSize;
  UINTN                             StringSize;
  UINTN                             TotalSize;
  UINTN                             SavedSize;
  UINTN                             ComponentStrSize;
  EFI_AIP_TELEMETRY_VERSION_DATA    *BlockPtr;
  EFI_AIP_TELEMETRY_VERSION_RECORD  *RecordPtr;
  UINTN                             Index;

  if ((Data == NULL) || (Block == NULL) || (BlockSize == NULL)) {
    return EFI_INVALID_PARAMETER;
  }

  if (Count > MAX_UINT16) {
    DEBUG ((EFI_D_ERROR, "FVI record count exceed maximum of UINT16.\n"));
    return EFI_INVALID_PARAMETER;
  }

  //
  // Collect data and calculate allocate size.
  //
  *BlockSize = 0;
  *Block     = NULL;
  //
  // Total size of strings.
  //
  StringSize = 0;
  for (Index = 0; Index < Count; Index++) {
    //
    // ComponentName should not be NULL.
    //
    ASSERT (Data[Index].ComponentName != NULL);
    StringSize += StrSize (Data[Index].ComponentName);
  }

  DataSize   = Count * OFFSET_OF (EFI_AIP_TELEMETRY_VERSION_RECORD, ComponentStr);
  HeaderSize = OFFSET_OF (EFI_AIP_TELEMETRY_VERSION_DATA, Records);
  TotalSize  = HeaderSize + DataSize + StringSize;

  if (TotalSize > MAX_UINT16) {
    DEBUG ((EFI_D_ERROR, "FVI block size exceed maximum of UINT16 (64KB).\n"));
    return EFI_INVALID_PARAMETER;
  }

  BlockPtr = (EFI_AIP_TELEMETRY_VERSION_DATA *) AllocatePool (TotalSize);
  if (BlockPtr == NULL) {
    return EFI_OUT_OF_RESOURCES;
  }

  //
  // Create Aip Telemetry Block
  //
  BlockPtr->Length      = (UINT16) TotalSize;
  BlockPtr->RecordCount = (UINT16) Count;
  SavedSize = HeaderSize;
  for (Index = 0; Index < Count; Index++) {
    RecordPtr = (EFI_AIP_TELEMETRY_VERSION_RECORD*) ((UINTN) BlockPtr + SavedSize);
    RecordPtr->Version     = Data[Index].Version;
    //
    // Default EsrtId is 0.
    //
    SetMem (&RecordPtr->EsrtId , sizeof (RecordPtr->EsrtId), 0);
    CopyGuid (&RecordPtr->ComponentID, &Data[Index].ComponentID);

    //
    // Connect string to record.
    //
    ASSERT (Data[Index].ComponentName != NULL);
    ComponentStrSize = StrSize (Data[Index].ComponentName);
    CopyMem (RecordPtr->ComponentStr, Data[Index].ComponentName, ComponentStrSize);
    SavedSize += OFFSET_OF (EFI_AIP_TELEMETRY_VERSION_RECORD, ComponentStr) + ComponentStrSize;
  }

  ASSERT (TotalSize == SavedSize);
  *BlockSize = (UINTN) TotalSize;
  *Block     = BlockPtr;

  return EFI_SUCCESS;
}
