/** @file
  This file implements platform firmware version information driver.

@copyright
  INTEL CONFIDENTIAL
  Copyright 2013 - 2020 Intel Corporation.

  The source code contained or described herein and all documents related to the
  source code ("Material") are owned by Intel Corporation or its suppliers or
  licensors. Title to the Material remains with Intel Corporation or its suppliers
  and licensors. The Material may contain trade secrets and proprietary and
  confidential information of Intel Corporation and its suppliers and licensors,
  and is protected by worldwide copyright and trade secret laws and treaty
  provisions. No part of the Material may be used, copied, reproduced, modified,
  published, uploaded, posted, transmitted, distributed, or disclosed in any way
  without Intel's prior express written permission.

  No license under any patent, copyright, trade secret or other intellectual
  property right is granted to or conferred upon you by disclosure or delivery
  of the Materials, either expressly, by implication, inducement, estoppel or
  otherwise. Any license under such intellectual property rights must be
  express and approved by Intel in writing.

  Unless otherwise agreed by Intel in writing, you may not remove or alter
  this notice or any other notice embedded in Materials by Intel or
  Intel's suppliers or licensors in any way.

  This file contains a 'Sample Driver' and is licensed as such under the terms
  of your license agreement with Intel or your vendor. This file may be modified
  by the user, subject to the additional terms of the license agreement.

@par Specification Reference:

**/

#include "PlatformFirmwareVersion.h"

#include <PiDxe.h>
#include <Library/DebugLib.h>
#include <Library/UefiLib.h>
#include <Library/BaseMemoryLib.h>
#include <Library/MemoryAllocationLib.h>
#include <Protocol/AdapterInformation.h>

#include <Library/IoLib.h>
#include <Library/BaseLib.h>
#include <Library/DevicePathLib.h>
#include <Library/UefiBootServicesTableLib.h>
#include <Library/UefiRuntimeServicesTableLib.h>
#include <Guid/EventGroup.h>

#include <Protocol/Smbios.h>
#include <Protocol/LoadedImage.h>
#include <Protocol/FirmwareVolume2.h>

#include <PlatformInfo.h>
#include <Library/PciSegmentLib.h>
#include <IndustryStandard/Pci30.h>
#include <Register/PchRegs.h>


//
// Arrays for Adapter Information
//
GLOBAL_REMOVE_IF_UNREFERENCED TELEMETRY_VERSION_RECORD_DATA mPlatformTelemetryVersionData[] = {
  { LAN_PHY_VERSION,           LAN_PHY_VERSION_COMPONENT_ID,          LAN_PHY_VERSION_STRING,          },
  { SENSOR_FW_VERSION,         SENSOR_FW_VERSION_COMPONENT_ID,        SENSOR_FW_VERSION_STRING,        },
  { DEBUG_MODE_VERSION,        DEBUG_MODE_STATUS_COMPONENT_ID,        DEBUG_MODE_STATUS,               },
  { PERFORMANCE_MODE_VERSION,  PERFORMANCE_MODE_STATUS_COMPONENT_ID,  PERFORMANCE_MODE_STATUS,         },
  { DEBUG_USE_USB_VERSION,     DEBUG_USE_USB_COMPONENT_ID,            DEBUG_USE_USB_STATUS,            },
  { ICC_OVERLOCKING_VERSION,   ICC_OVERLOCKING_VERSION_COMPONENT_ID,  ICC_OVERLOCKING_VERSION_STRING,  },
  { EC_VERSION,                EC_VERSION_COMPONENT_ID,               EC_VERSION_STRING,               },
  { ROYALPARK_VERSION,         ROYALPARK_VERSION_COMPONENT_ID,        ROYALPARK_VERSION_STRING,        },
  { EDKII_VERSION,             EDKII_STABLE_VERSION_COMPONENT_ID,     EDKII_VERSION_STRING,            },
  { PLATFORM_VERSION,          PLATFORM_VERSION_COMPONENT_ID,         PLATFORM_VERSION_STRING,         },
  { CLIENT_SILICON_VERSION,    CLIENT_SILICON_VERSION_COMPONENT_ID,   CLIENT_SILICON_VERSION_STRING,   },
};

/**
  Get a list of supported information types for this instance of the protocol.

  This function returns a list of InformationType GUIDs that are supported on an
  adapter with this instance of EFI_ADAPTER_INFORMATION_PROTOCOL. The list is returned
  in InfoTypesBuffer, and the number of GUID pointers in InfoTypesBuffer is returned in
  InfoTypesBufferCount.

  @param[in]  This                  A pointer to the EFI_ADAPTER_INFORMATION_PROTOCOL instance.
  @param[out] InfoTypesBuffer       A pointer to the array of InformationType GUIDs that are supported
                                    by This.
  @param[out] InfoTypesBufferCount  A pointer to the number of GUIDs present in InfoTypesBuffer.

  @retval EFI_SUCCESS               The list of information type GUIDs that are supported on this adapter was
                                    returned in InfoTypesBuffer. The number of information type GUIDs was
                                    returned in InfoTypesBufferCount.
  @retval EFI_INVALID_PARAMETER     This is NULL.
  @retval EFI_INVALID_PARAMETER     InfoTypesBuffer is NULL.
  @retval EFI_INVALID_PARAMETER     InfoTypesBufferCount is NULL.
  @retval EFI_OUT_OF_RESOURCES      There is not enough pool memory to store the results.

**/
EFI_STATUS
EFIAPI
PlatformAipGetSupportedTypes (
  IN  EFI_ADAPTER_INFORMATION_PROTOCOL  *This,
  OUT EFI_GUID                          **InfoTypesBuffer,
  OUT UINTN                             *InfoTypesBufferCount
  )
{
  if ((This == NULL) || (InfoTypesBuffer == NULL) || (InfoTypesBufferCount == NULL)) {
    return EFI_INVALID_PARAMETER;
  }

  *InfoTypesBuffer = AllocateCopyPool (sizeof (gEfiAdapterInfoTelemetryVersionRecordGuid), &gEfiAdapterInfoTelemetryVersionRecordGuid);
  if (*InfoTypesBuffer == NULL) {
    return EFI_OUT_OF_RESOURCES;
  }
  *InfoTypesBufferCount = 1;

  return EFI_SUCCESS;
}

/**
  Returns the current state information for the adapter.

  This function returns information of type InformationType from the adapter.
  If an adapter does not support the requested informational type, then
  EFI_UNSUPPORTED is returned.

  @param[in]  This                   A pointer to the EFI_ADAPTER_INFORMATION_PROTOCOL instance.
  @param[in]  InformationType        A pointer to an EFI_GUID that defines the contents of InformationBlock.
  @param[out] InformationBlock       The service returns a pointer to the buffer with the InformationBlock
                                     structure which contains details about the data specific to InformationType.
  @param[out] InformationBlockSize   The driver returns the size of the InformationBlock in bytes.

  @retval EFI_SUCCESS                The InformationType information was retrieved.
  @retval EFI_UNSUPPORTED            The InformationType is not known.
  @retval EFI_DEVICE_ERROR           The device reported an error.
  @retval EFI_OUT_OF_RESOURCES       The request could not be completed due to a lack of resources.
  @retval EFI_INVALID_PARAMETER      This is NULL.
  @retval EFI_INVALID_PARAMETER      InformationBlock is NULL.
  @retval EFI_INVALID_PARAMETER      InformationBlockSize is NULL.

**/
EFI_STATUS
EFIAPI
PlatformFviAipGetInfo (
  IN  EFI_ADAPTER_INFORMATION_PROTOCOL  *This,
  IN  EFI_GUID                          *InformationType,
  OUT VOID                              **InformationBlock,
  OUT UINTN                             *InformationBlockSize
  )
{
  EFI_STATUS Status;

  if ((This == NULL) || (InformationBlock == NULL) || (InformationBlockSize == NULL)) {
    return EFI_INVALID_PARAMETER;
  }
  if (!CompareGuid (InformationType, &gEfiAdapterInfoTelemetryVersionRecordGuid)) {
    return EFI_UNSUPPORTED;
  }

  Status = CreateTelemetryFviBlock (
             mPlatformTelemetryVersionData,
             sizeof (mPlatformTelemetryVersionData) / sizeof (TELEMETRY_VERSION_RECORD_DATA),
             (EFI_AIP_TELEMETRY_VERSION_DATA**) InformationBlock,
             InformationBlockSize
             );
  if (EFI_ERROR (Status)) {
    return Status;
  }

  if (*InformationBlock == NULL) {
    return EFI_OUT_OF_RESOURCES;
  }

  return EFI_SUCCESS;
}

/**
  Sets state information for an adapter.

  This function sends information of type InformationType for an adapter.
  If an adapter does not support the requested information type, then EFI_UNSUPPORTED
  is returned.

  @param[in]  This                   A pointer to the EFI_ADAPTER_INFORMATION_PROTOCOL instance.
  @param[in]  InformationType        A pointer to an EFI_GUID that defines the contents of InformationBlock.
  @param[in]  InformationBlock       A pointer to the InformationBlock structure which contains details
                                     about the data specific to InformationType.
  @param[in]  InformationBlockSize   The size of the InformationBlock in bytes.

  @retval EFI_SUCCESS                The information was received and interpreted successfully.
  @retval EFI_UNSUPPORTED            The InformationType is not known.
  @retval EFI_DEVICE_ERROR           The device reported an error.
  @retval EFI_INVALID_PARAMETER      This is NULL.
  @retval EFI_INVALID_PARAMETER      InformationBlock is NULL.
  @retval EFI_WRITE_PROTECTED        The InformationType cannot be modified using EFI_ADAPTER_INFO_SET_INFO().

**/
EFI_STATUS
EFIAPI
PlatformFviAipSetInfo (
  IN  EFI_ADAPTER_INFORMATION_PROTOCOL  *This,
  IN  EFI_GUID                          *InformationType,
  IN  VOID                              *InformationBlock,
  IN  UINTN                             InformationBlockSize
  )
{

  if ((This == NULL) || (InformationBlock == NULL)) {
    return EFI_INVALID_PARAMETER;
  }
  if (!CompareGuid (InformationType, &gEfiAdapterInfoTelemetryVersionRecordGuid)) {
    return EFI_UNSUPPORTED;
  }

  return EFI_SUCCESS;
}

/**
  Set FVI EC Version.

  @attention This function may receive untrusted input.
  Type0Record is external input, so this function will validate its data structure within this buffer before use.

  @param[in] Type0Record               A pointer to the SMBIOS_TABLE_TYPE0 instance.
**/
VOID
SetFviEcVersion (
  IN SMBIOS_TABLE_TYPE0   *Type0Record
  )
{
  if (Type0Record != NULL) {
    mPlatformTelemetryVersionData[EC_VER].Version = (UINT64)Type0Record->EmbeddedControllerFirmwareMajorRelease << VERSION_MAJOR_VERSION_SHIFT
                                                  | (UINT64)Type0Record->EmbeddedControllerFirmwareMinorRelease << VERSION_MINOR_VERSION_SHIFT;
  }
}

/**
  Get EC Version.
**/
VOID
GetEcVersion (
  VOID
  )
{
  EFI_STATUS              Status;
  EFI_SMBIOS_HANDLE       SmbiosHandle;
  EFI_SMBIOS_PROTOCOL     *mSmbios;
  SMBIOS_TABLE_TYPE0      *Type0Record;
  EFI_SMBIOS_TABLE_HEADER *Record;

  Status = gBS->LocateProtocol (
                  &gEfiSmbiosProtocolGuid,
                  NULL,
                  (VOID **)&mSmbios
                  );
  if (!EFI_ERROR (Status)) {
    SmbiosHandle = SMBIOS_HANDLE_PI_RESERVED;
    do {
      Status = mSmbios->GetNext (mSmbios, &SmbiosHandle, NULL, &Record, NULL);
      if (EFI_ERROR(Status)) {
        break;
      }
      if (Record->Type == EFI_SMBIOS_TYPE_BIOS_INFORMATION) {
        Type0Record = (SMBIOS_TABLE_TYPE0 *) Record;
        SetFviEcVersion (Type0Record);
        break;
      }
    } while (Record->Type != EFI_SMBIOS_TYPE_BIOS_INFORMATION);
  }
}

/**
  Get EDKII Version.
**/
VOID
GetEDKIIStableVersion (
  VOID
  )
{
  mPlatformTelemetryVersionData[EDKII_VER].Version = (UINT64)PcdGet8 (PcdEDKIIVersionMajor) << VERSION_MAJOR_VERSION_SHIFT
                                                       | (UINT64)PcdGet8 (PcdEDKIIVersionMinor) << VERSION_MINOR_VERSION_SHIFT
                                                       | (UINT64)PcdGet8 (PcdEDKIIVersionRevision) << VERSION_REVISION_SHIFT
                                                       | (UINT64)PcdGet8 (PcdEDKIIVersionBuild) << VERSION_BUILD_NUM_SHIFT;

}

/**
  Get Platform Version.
**/
VOID
GetPlatformVersion (
  VOID
  )
{
  mPlatformTelemetryVersionData[PLATFORM_VER].Version = (UINT64)PcdGet8 (PcdPlatformVersionMajor) << VERSION_MAJOR_VERSION_SHIFT
                                                      | (UINT64)PcdGet8 (PcdPlatformVersionMinor) << VERSION_MINOR_VERSION_SHIFT
                                                      | (UINT64)PcdGet8 (PcdPlatformVersionRevision) << VERSION_REVISION_SHIFT
                                                      | (UINT64)PcdGet8 (PcdPlatformVersionBuild) << VERSION_BUILD_NUM_SHIFT;
}

/**
  Get Client Silicon Version.
**/
VOID
GetClientSiliconVersion (
  VOID
  )
{
  mPlatformTelemetryVersionData[CLIENT_SILICON_VER].Version = (UINT64)PcdGet8 (PcdClientSiliconVersionMajor) << VERSION_MAJOR_VERSION_SHIFT
                                                            | (UINT64)PcdGet8 (PcdClientSiliconVersionMinor) << VERSION_MINOR_VERSION_SHIFT
                                                            | (UINT64)PcdGet8 (PcdClientSiliconVersionRevision) << VERSION_REVISION_SHIFT
                                                            | (UINT64)PcdGet8 (PcdClientSiliconVersionBuild) << VERSION_BUILD_NUM_SHIFT;
}

/**
  Get Platform Firmware Version Information.

  @param[in] Event    The Event this notify function registered to.
  @param[in] Context  Pointer to the context data registered to the Event.
**/
VOID
EFIAPI
InitializeFviDataCallback (
  IN EFI_EVENT Event,
  IN VOID      *Context
  )
{
  EFI_STATUS                       Status;
  VOID                             *ProtocolPointer;
  EFI_ADAPTER_INFORMATION_PROTOCOL *AdapterInfo;
  EFI_HANDLE                       Handle;

  AdapterInfo = NULL;
  Handle      = NULL;

  Status = gBS->LocateProtocol (&gBdsAllDriversConnectedProtocolGuid, NULL, (VOID **) &ProtocolPointer);
  if (EFI_SUCCESS != Status) {
    return;
  }

  gBS->CloseEvent (Event);

  GetEcVersion ();

  AdapterInfo = AllocateZeroPool (sizeof (EFI_ADAPTER_INFORMATION_PROTOCOL));
  if (AdapterInfo == NULL) {
    DEBUG ((EFI_D_ERROR, "EFI_OUT_OF_RESOURCES to Allocate EFI_ADAPTER_INFORMATION_PROTOCOL. \n"));
    return;
  }

  AdapterInfo->GetInformation    = PlatformFviAipGetInfo;
  AdapterInfo->SetInformation    = PlatformFviAipSetInfo;
  AdapterInfo->GetSupportedTypes = PlatformAipGetSupportedTypes;

  Status = gBS->InstallMultipleProtocolInterfaces (
                  &Handle,
                  &gEfiAdapterInformationProtocolGuid,
                  AdapterInfo,
                  NULL
                  );

  if (EFI_ERROR (Status)) {
    DEBUG ((EFI_D_ERROR, "Platform FVI: faild to install AIP protocol, %r.\n", Status));
  }

  return;
}


/**
  This is driver entry point to register the notification event.

  @param[in] ImageHandle  A handle for the image that is initializing this driver
  @param[in] SystemTable  A pointer to the EFI system table

  @retval    EFI_SUCCESS  The initialization finished successfully.
**/
EFI_STATUS
EFIAPI
PlatformFirmwareVersionDxe (
  IN EFI_HANDLE        ImageHandle,
  IN EFI_SYSTEM_TABLE  *SystemTable
  )
{
  VOID               *Registration;

  GetEDKIIStableVersion ();
  GetPlatformVersion ();
  GetClientSiliconVersion ();

  //
  // Register InitializeFviDataCallback
  //
  EfiCreateProtocolNotifyEvent (
    &gBdsAllDriversConnectedProtocolGuid,
    TPL_CALLBACK,
    InitializeFviDataCallback,
    NULL,
    &Registration
    );

  return EFI_SUCCESS;
}
