 /** @file
  This file contains Cpu Information for specific generation.

@copyright
  INTEL CONFIDENTIAL
  Copyright 2019 - 2020 Intel Corporation.

  The source code contained or described herein and all documents related to the
  source code ("Material") are owned by Intel Corporation or its suppliers or
  licensors. Title to the Material remains with Intel Corporation or its suppliers
  and licensors. The Material may contain trade secrets and proprietary and
  confidential information of Intel Corporation and its suppliers and licensors,
  and is protected by worldwide copyright and trade secret laws and treaty
  provisions. No part of the Material may be used, copied, reproduced, modified,
  published, uploaded, posted, transmitted, distributed, or disclosed in any way
  without Intel's prior express written permission.

  No license under any patent, copyright, trade secret or other intellectual
  property right is granted to or conferred upon you by disclosure or delivery
  of the Materials, either expressly, by implication, inducement, estoppel or
  otherwise. Any license under such intellectual property rights must be
  express and approved by Intel in writing.

  Unless otherwise agreed by Intel in writing, you may not remove or alter
  this notice or any other notice embedded in Materials by Intel or
  Intel's suppliers or licensors in any way.

  This file contains an 'Intel Peripheral Driver' and is uniquely identified as
  "Intel Reference Module" and is licensed for Intel CPUs and chipsets under
  the terms of your license agreement with Intel or your vendor. This file may
  be modified by the user, subject to additional terms of the license agreement.

@par Specification Reference:
**/
#ifndef _CPU_INFO_FRU_LIB_H_
#define _CPU_INFO_FRU_LIB_H_

#include <Uefi.h>
#include <Library/BaseLib.h>
#include <Library/DebugLib.h>
#include <Library/IoLib.h>
#include <CpuRegs.h>
#include <Register/SaRegsHostBridge.h>
#include <Library/PreSiliconEnvDetectLib.h>
#include <Library/PciSegmentLib.h>
#include <Library/BaseMemoryLib.h>
#include <Library/MmPciLib.h>
#include <Library/CpuCommonLib.h>
#include <ConfigBlock.h>
#include <ConfigBlock/CpuConfigLibPreMemConfig.h>

#define SMBIOS_TYPE4_SOCKET_BGA1392           0x3A
#define SMBIOS_TYPE4_SOCKET_BGA1510           0x3B
#define SMBIOS_TYPE4_SOCKET_BGA1528           0x3C

///
/// Override table structure for cTDP and non-cTDP skus.
/// Non-cTDP parts would have '0' data for TDP level information.
///
typedef struct {
  UINTN  CpuIdentifier;
  UINT16 SkuPackageTdp;
  UINTN  MsrPowerLimit1;
  UINTN  MsrPowerLimit2;
  UINTN  CtdpUpPowerLimit1;
  UINTN  CtdpUpPowerLimit2;
  UINTN  CtdpNominalPowerLimit1;
  UINTN  CtdpNominalPowerLimit2;
  UINTN  CtdpDownPowerLimit1;
  UINTN  CtdpDownPowerLimit2;
  UINTN  MsrPowerLimit4;
} PPM_OVERRIDE_TABLE;

/**
  Return CPU Sku

  @param[in]  UINT32             CpuFamilyModel
  @param[in]  UINT16             CpuDid

  @retval     UINT8              CPU Sku
**/
UINT8
GetCpuSkuInfo (
  IN UINT32    CpuFamilyModel,
  IN UINT16    CpuDid
  );

/**
  Get processor generation

  @param[in]  CPU_FAMILY         CpuFamilyModel
  @retval     CPU_GENERATION     Returns the executing thread's processor generation.
**/
CPU_GENERATION
GetCpuSkuGeneration (
  IN CPU_FAMILY         CpuFamilyModel
  );

/**
  Returns Generation string of the respective CPU

  @param[in]   CpuFamilyId

  @retval      Character pointer of Generation string
**/
CONST CHAR8*
GetFruGenerationString (
  IN   UINT32   CpuFamilyId
  );

/**
  Returns Revision Table string

  @param[in]   CpuFamilyId

  @retval      Character pointer of Revision Table string
**/
CONST CHAR8*
GetRevisionTableString (
  UINT32                CpuId
  );

/**
  Determine if CPU supports Intel Turbo Boost Max Technology 3.0 (ITBM).

  @retval Bit is set if ITBM is supported
**/
BOOLEAN
GetItbmSupportedStatus (
  VOID
);

/**
  Determine if CPU supports Telemetry.

  @retval TRUE   if CPU Supports Telemetry.
  @retval FALSE  if CPU doesn't supports Telemetry.
**/
BOOLEAN
IsFruSupportedTelemetry (
  VOID
  );

/**
  This function returns if CPU support SyncFeatures

  @retval TRUE             SyncFeature is supported
  @retval FALSE            SyncFeature is not Supported
**/
BOOLEAN
IsSmmSyncFeatureSupported (
  VOID
  );

/**
  This function returns Number of CBO0 Bank Index.

  @retval Number of CBO0 Bank Index.
**/
UINT8
GetCbo0BankIndex (
  VOID
  );

/**
  Detect if Hetero Core is supported.

  @retval TRUE - Processor Support HeteroCore
  @retval FALSE - Processor doesnt support HeteroCore
**/
BOOLEAN
IsHeteroCoreSupported (
  VOID
  );

/**
  Detect the type of core.

  @retval the core type which is running
**/
UINT8
DetectCoreType (
  VOID
  );

/**
  Returns power limits Table

  @param[in]      NoOfOverrides

  @retval         override table pointer of power limits Table
**/
PPM_OVERRIDE_TABLE*
GetFruPowerLimits (
  IN UINTN        *NoOfOverrides
  );

/**
  This function checks for HWP advanced features like EnableFastMsrHwpReq, EnableHwpAutoEppGrouping, EnableHwpAutoPerCorePstate, EnablePerCorePState.

  @retval TRUE  - Advanced HWP features supported.
  @retval FALSE - Advanced HWP features not supported.
**/
BOOLEAN
IsHwpAdvancedFeatureSupport (
  VOID
  );

/**
  Compare the hyper threading setup option against the CPU strap setting
  and in case of mismatch request a reset.

  @param[in] CpuStrapSetData           - The current strap setting data.
  @param[in] HyperThreading            - hyper threading setting from BIOS Setup option.

  @retval NO_RESET          - No reset is needed.
  @retval WARM_RESET        - Update requires a warm reset.
**/
CPU_RESET_TYPE
CpuHtEnableDisable (
  IN VOID              *CpuStrapSetData,
  IN UINT8             HyperThreading
  );

/**
  Compare the number of active cores setup option against the CPU strap setting
  and in case of mismatch request a reset.

  @param[in] CpuStrapSetData      - The current strap setting.
  @param[in] ActiveCoreCount      - Active big core count.
  @param[in] ActiveSmallCoreCount - Active small core count.

  @retval NO_RESET          - No reset is needed.
  @retval WARM_RESET        - Update requires a warm reset.
  @retval COLD_RESET        - Update requires a cold reset.
**/
CPU_RESET_TYPE
CpuSetActiveCores (
  IN VOID              *CpuStrapSetData,
  IN UINT8             ActiveCoreCount,
  IN UINT8             ActiveSmallCoreCount
  );

/**
  Compare the BIST setup option against the CPU strap setting
  and in case of mismatch request a reset.

  @param[in] CpuStrapSetData    - The current strap setting.
  @param[in] BistOnReset        - BistOnReset CPU Test Config Policy.

  @retval NO_RESET          - No reset is needed.
  @retval WARM_RESET        - Update requires a warm reset.
**/
CPU_RESET_TYPE
CpuBistEnableDisable (
  IN VOID              *CpuStrapSetData,
  IN UINT8             BistOnReset
  );

/**
  Compare the flex multiplier setup options against the CPU strap settings
  and in case of mismatch request a reset.

  @param[in] CpuStrapSetData    - The current strap setting.
  @param[in] CpuRatio           - CpuRatio CPU policy.

  @retval NO_RESET          - No reset is needed.
  @retval WARM_RESET        - Update requires a warm reset.
**/
CPU_RESET_TYPE
CpuProgramFlexMultiplier (
  IN VOID                        *CpuStrapSetData,
  IN UINT8                       CpuRatio
  );

/**
  Compare the boot frequency setup option against the boot ratio strap setting
  and in case of mismatch request a reset.

  @param[in] CpuStrapSetData    - The current strap setting.
  @param[in] BootFrequency      - BootFrequency CPU policy.

  @retval NO_RESET          - No reset is needed.
  @retval WARM_RESET        - Update requires a warm reset.
**/
CPU_RESET_TYPE
CpuBootRatioEnableDisable (
  IN VOID                *CpuStrapSetData,
  IN UINT8               BootFrequency
  );

/**
  Compare the JTAG power gate setup option against the CPU strap setting
  and in case of mismatch request a reset.

  @param[in] CpuStrapSet    - The current strap setting.
  @param[in] JtagC10PowerGateDisable - JtagC10PowerGateDisable CPU policy.

  @retval NO_RESET          - No reset is needed.
  @retval WARM_RESET        - Update requires a warm reset.
**/
CPU_RESET_TYPE
CpuJtagPowerGateEnableDisable (
  IN VOID                    *CpuStrapSetData,
  IN UINT8                   JtagC10PowerGateDisable
  );

/**
  This function is to set the CPU Epoc on early phase.

  @param[in]  CpuConfigData

  @retval EFI_SUCCESS            Early CPU Epoc has been set successfully.
  @retval EFI_UNSUPPORTED        Early CPU Epoc seting is not supported.
  @retval EFI_INVALID_PARAMETER  Early Cpu Epoc returned invalid data.
**/
EFI_STATUS
CpuEarlyEpocSet (
  IN VOID *CpuConfigData
  );

/**
  This function sends PECI related mailbox commands early in boot if needed

  @param[in] CpuConfigLibPreMemConfig Pointer to the cpu config lib premem config block instance

  @retval EFI_SUCCESS              - Function successfully executed.
  @retval Other                    - Error occurred during mailbox commands.
**/
EFI_STATUS
EFIAPI
CpuPeciMailboxCommands (
  IN CPU_CONFIG_LIB_PREMEM_CONFIG *CpuConfigLibPreMemConfig
  );

/**
  Programs Processor Upgrade for type 4 SMBIOS Processor Info HOB.

  @retval Returns Processor Upgrade value for type 4 SMBIOS Processor Info HOB.
**/
UINT8
SmbiosProcessorInfoHobType4 (
  VOID
  );

/**
  Get CPU default C state latency value.

  @param[out] CstateLatencyControl1Irtl  Interrupt Response Time Limit of LatencyContol1
  @param[out] CstateLatencyControl2Irtl  Interrupt Response Time Limit of LatencyContol2
  @param[out] CstateLatencyControl3Irtl  Interrupt Response Time Limit of LatencyContol3
  @param[out] CstateLatencyControl4Irtl  Interrupt Response Time Limit of LatencyContol4
  @param[out] CstateLatencyControl5Irtl  Interrupt Response Time Limit of LatencyContol5
**/
VOID
CpuGetCstateLatencyDefault (
  OUT UINT16 *CstateLatencyControl1Irtl,
  OUT UINT16 *CstateLatencyControl2Irtl,
  OUT UINT16 *CstateLatencyControl3Irtl,
  OUT UINT16 *CstateLatencyControl4Irtl,
  OUT UINT16 *CstateLatencyControl5Irtl
  );

/**
  Return Total Memory Encryption (TME) default policy setting

  @retval Total Memory Encryption (TME) default policy setting
**/
UINT32
TmeDefaultSetting (
  VOID
  );

/**
  Disable or enable CPU Chashlog dump feature by MSR.

  @param[in] BOOLEAN      Policy for CrashLog
**/
VOID
CpuCrashLogMsrEnable (
  IN  BOOLEAN                          CrashLogEnable
  );

/**
  This function is used to Patch SmmSupovrStateLock.

  @retval This corresponds to bit 2 of MSR_SMM_SUPOVR_STATE_LOCK_REGISTER. When set, prevents WRMSR to IA32_SMM_MONITOR_CTL (aka MSEG) MSR.

**/
BOOLEAN
SmmSupovrStateLockPatch (
  VOID
  );

/**
  This function returns the supported Physical Address Size

  @retval returns the supported Physical Address Size.
**/
UINT8
GetMaxPhysicalAddressSizeFru (
  VOID
  );

/**
  Initialize the Core CLOS for LLC Cache QOS mask.

**/
VOID
InitializeCoreClos (
  VOID
  );

/**
  This function return the Heuristics Policy Data for C6DRAM.

  @retval  Return the Heuristics Policy Data for C6DRAM.
**/
UINT8
GetC6DramHeuristicsPolicyData (
  VOID
  );
#endif // _CPU_INFO_FRU_LIB_H_
