/** @file
  PEIM to initialize the cache and program for unlock processor

@copyright
  INTEL CONFIDENTIAL
  Copyright 1999 - 2020 Intel Corporation.

  The source code contained or described herein and all documents related to the
  source code ("Material") are owned by Intel Corporation or its suppliers or
  licensors. Title to the Material remains with Intel Corporation or its suppliers
  and licensors. The Material may contain trade secrets and proprietary and
  confidential information of Intel Corporation and its suppliers and licensors,
  and is protected by worldwide copyright and trade secret laws and treaty
  provisions. No part of the Material may be used, copied, reproduced, modified,
  published, uploaded, posted, transmitted, distributed, or disclosed in any way
  without Intel's prior express written permission.

  No license under any patent, copyright, trade secret or other intellectual
  property right is granted to or conferred upon you by disclosure or delivery
  of the Materials, either expressly, by implication, inducement, estoppel or
  otherwise. Any license under such intellectual property rights must be
  express and approved by Intel in writing.

  Unless otherwise agreed by Intel in writing, you may not remove or alter
  this notice or any other notice embedded in Materials by Intel or
  Intel's suppliers or licensors in any way.

  This file contains an 'Intel Peripheral Driver' and is uniquely identified as
  "Intel Reference Module" and is licensed for Intel CPUs and chipsets under
  the terms of your license agreement with Intel or your vendor. This file may
  be modified by the user, subject to additional terms of the license agreement.

@par Specification
**/

#include <Ppi/MasterBootMode.h>
#include <Library/DebugLib.h>
#include <Library/PeiServicesLib.h>
#include <Library/HobLib.h>
#include <Ppi/SecPlatformInformation.h>
#include <Library/ReportStatusCodeLib.h>
#include <Library/BaseMemoryLib.h>
#include <Library/MemoryAllocationLib.h>
#include "CpuAccess.h"
#include "CpuInitPeim.h"
#include <Library/CpuPlatformLib.h>
#include <Library/TxtPeiLib.h>
#include <Library/CpuCommonLib.h>
#include <Library/CpuInitLib.h>
#include <Library/ConfigBlockLib.h>
#include <Library/PeiOcLib.h>
#include <Library/CpuPowerOnConfigLib.h>
#include <Library/PostCodeLib.h>
#include <Library/PcdLib.h>
#include <Register/Cpuid.h>

#include <Register/ArchitecturalMsr.h>
#include <Register/ArchMsr.h>
#include <Library/MsrFruLib.h>
#include <Library/CpuInfoFruLib.h>
#include <Library/PmcLib.h>
#include <Register/PmcRegs.h>

extern EFI_GUID gHtBistHobGuid;

#if FixedPcdGetBool(PcdTmeLibSupported)== 0x0
#define TME_ACTIVATE_RETRY_LIMIT           10
#endif // FixedPcdGetBool(PcdTmeLibSupported)== 0x0

#if FixedPcdGetBool(PcdFspBinaryEnable) == 0
STATIC EFI_PEI_NOTIFY_DESCRIPTOR mNotifyList[] = {
  {
    (EFI_PEI_PPI_DESCRIPTOR_NOTIFY_CALLBACK | EFI_PEI_PPI_DESCRIPTOR_TERMINATE_LIST),
    &gEfiPeiMasterBootModePpiGuid,
    BuildBistHob
  }
};
#endif

#if FixedPcdGetBool(PcdTmeLibSupported) == 0
//
// PcdTmeLibSupported
//   0 - TmeLib is not present therefore, use below function.
//   1 - TmeLib is present therefore, do not use below function.
//       Use function from TmeLib.
//
/**
  Perform Total Memory Encryption initialization.

  @param[in] TmeEnable      - TME policy enable
  @param[in] TmeExcludeBase - Base physical address to be excluded for TME encryption
  @param[in] TmeExcludeSize - Size of range to be excluded from TME encryption

  @retval VOID - No value to return
**/
VOID
TmeInit (
  IN UINT32 TmeEnable,
  IN UINT64 TmeExcludeBase,
  IN UINT64 TmeExcludeSize
  )
{
  EFI_CPUID_REGISTER             CpuidRegs;
  MSR_TME_CAPABILITY_REGISTER    TmeCapability;
  MSR_TME_ACTIVATE_REGISTER      TmeActivate;
  MSR_TME_ACTIVATE_REGISTER      TmeValidate;
  MSR_TME_EXCLUDE_MASK_REGISTER  ExcludeMask;
  MSR_TME_EXCLUDE_BASE_REGISTER  ExcludeBase;
  BOOLEAN                        TmeLocked;
  BOOLEAN                        TmeSupported;
  EFI_BOOT_MODE                  BootMode;
  EFI_STATUS                     Status;
  UINT8                          Index;
  UINT32                         RegisterVal32;
  BOOLEAN                        WarmReset;

  TmeCapability.Uint64   = 0;
  TmeActivate.Uint64     = 0;
  TmeValidate.Uint64     = 0;
  TmeLocked              = 0;
  TmeSupported           = 0;
  BootMode               = 0;
  Status                 = 0;
  Index                  = 0;
  WarmReset              = FALSE;

  DEBUG ((DEBUG_INFO, "Total Memory Encryption (TME) Initialization\n"));

  ///
  /// Detect TME capability and encryption policy is supported in the hardware
  ///

  ///
  /// Verify TME supported through CPUID.7.0.ECX.13
  ///
  AsmCpuidEx (CPUID_STRUCTURED_EXTENDED_FEATURE_FLAGS, 0, &CpuidRegs.RegEax,&CpuidRegs.RegEbx,&CpuidRegs.RegEcx,&CpuidRegs.RegEdx);
  if ((CpuidRegs.RegEcx & BIT13) != BIT13) {
    DEBUG ((DEBUG_INFO, "TME is not supported by the processor. Skipping TME activation.\n"));
    return;
  }

  ///
  /// Read MSR_TME_CAPABILITY (981H) to determine supported encryption policy
  /// TGL currently only supports AES-XTS
  ///
  TmeCapability.Uint64 = AsmReadMsr64 (MSR_TME_CAPABILITY);
  if (TmeCapability.Bits.AesXts != 0) {
    DEBUG ((DEBUG_INFO, "TME supports AES-XTS encryption policy\n"));
    ///
    /// Set TME policy
    ///
    TmeActivate.Bits.TmePolicy = V_TME_ACTIVATE_TME_POLICY_AES_XTS;

    ///
    /// TME is supported in the hardware
    ///
    TmeSupported = TRUE;
  } else {
    DEBUG ((DEBUG_INFO, "TME does not support selected encryption policy. Skipping TME activation.\n"));
    return;
  }

  if (TmeSupported) {
    ///
    /// Enable TME if BIOS policy is enabled
    ///
    if (TmeEnable) {
      ///
      /// Configure MSR_TME_EXCLUDE_MASK (983H) with TME Mask and Enable bit
      ///
      if (TmeExcludeSize != 0) {
        ExcludeMask.Uint64 = TmeExcludeSize;
        ExcludeMask.Bits.Enable = 1;
        AsmWriteMsr64 (MSR_TME_EXCLUDE_MASK, ExcludeMask.Uint64);
      }

      ///
      /// Configure MSR_TME_EXCLUDE_BASE (984H) with TME Base
      ///
      if (TmeExcludeBase != 0) {
        ExcludeBase.Uint64 = TmeExcludeBase;
        AsmWriteMsr64 (MSR_TME_EXCLUDE_BASE, ExcludeBase.Uint64);
      }

      ///
      /// Set TME Key Select and Save Key for Warm Reset, Standby and Flash Update Mode
      /// Key Select
      ///   - Set for Warm Reset, S3 resume flow and flash update flow to restore
      ///     TME keys from PCH
      ///   - Clear for cold boot to create new TME keys
      /// Save Key
      ///   - Clear for Warm Reset, S3 resume flow and flash update flow
      ///   - Set for cold boot to save key into storage
      ///
      Status = PeiServicesGetBootMode (&BootMode);
      ASSERT_EFI_ERROR (Status);

      ///
      /// Detect Warm Reset boot
      ///
      RegisterVal32 = MmioRead32 ((UINTN) (PmcGetPwrmBase () + R_PMC_PWRM_GEN_PMCON_A));
      if ((((RegisterVal32 & B_PMC_PWRM_GEN_PMCON_A_MEM_SR) != 0) &&
          ((RegisterVal32 & B_PMC_PWRM_GEN_PMCON_A_DISB) != 0))) {
        WarmReset = TRUE;
      }

      if ((BootMode == BOOT_ON_S3_RESUME) ||
          (BootMode == BOOT_ON_FLASH_UPDATE) ||
          (WarmReset == TRUE)) {
        TmeActivate.Bits.KeySelect = 1;
        TmeActivate.Bits.SaveKey = 0;
      } else {
        TmeActivate.Bits.KeySelect = 0;
        TmeActivate.Bits.SaveKey = 1;
      }

      ///
      /// Set TME Enable
      ///
      TmeActivate.Bits.TmeEnable = 1;
    } else {
      ///
      /// Clear TME Enable
      ///
      TmeActivate.Bits.TmeEnable = 0;
      DEBUG ((DEBUG_INFO, "TME is disabled. Skipping TME activation.\n"));
    }

    ///
    /// Configure MSR_TME_ACTIVATE (982H) with TME Enable, Key Select, Save Key, and TME Policy
    /// Lock bit will be set upon successful WRMSR for MSR_TME_ACTIVATE.
    ///   - First SMI will also set the Lock
    ///   - This will also lock MSR_TME_EXCLUDE_MASK and MSR_TME_EXCLUDE_BASE
    ///
    DEBUG ((DEBUG_INFO, "Initialize MSR_TME_ACTIVATE: %016llx\n", TmeActivate.Uint64));

    for (Index = 0; Index < TME_ACTIVATE_RETRY_LIMIT; Index++) {
      AsmWriteMsr64 (MSR_TME_ACTIVATE, TmeActivate.Uint64);

      TmeValidate.Uint64 = AsmReadMsr64 (MSR_TME_ACTIVATE);
      if (TmeValidate.Bits.Lock == 1) {
        DEBUG ((DEBUG_INFO, "MSR_TME_ACTIVATE was initialized and locked: %016llx\n", TmeValidate));
        TmeLocked = TRUE;
        break;
      }
    }

    ///
    /// Ensure Lock bit of MSR_TME_ACTIVATE (982H) is set
    /// Lock bit will not be set if creation of ephemeral number using DRNG action failed.
    ///
    if (!TmeLocked) {
      DEBUG ((DEBUG_INFO, "TME was not enabled. Locking MSR_TME_ACTIVATE.\n"));
      TmeActivate.Bits.TmeEnable = 0;
      AsmWriteMsr64 (MSR_TME_ACTIVATE, TmeActivate.Uint64);
      TmeValidate.Uint64 = AsmReadMsr64 (MSR_TME_ACTIVATE);
      DEBUG ((DEBUG_INFO, "MSR_TME_ACTIVATE: %016llx\n", TmeValidate));
    }
  }
}
#endif // FixedPcdGetBool(PcdTmeLibSupported)== 0x0

/**
  Check some CPU policies are valid for debugging unexpected problem if these
  values are not initialized or assigned incorrect resource.

  @param[in] SiPreMemPolicy    The Si PreMem Policy protocol instance
**/
VOID
CpuValidatePolicy (
  IN  SI_PREMEM_POLICY_PPI   *SiPreMemPolicy
  )
{
  ASSERT (SiPreMemPolicy->TableHeader.Header.Revision == SI_PREMEM_POLICY_REVISION);
}

/**
  This function performs basic initialization for CPU in PEI phase after Policy produced.
  Enable/Disable CPU Debug as per Setup and locks it.

  @param[in] SiPreMemPolicyPpi The Si Pre-Mem Policy PPI instance
**/
EFI_STATUS
CpuOnPolicyInstalled (
  IN  SI_PREMEM_POLICY_PPI *SiPreMemPolicyPpi
  )
{
  EFI_STATUS Status;
  CPU_CONFIG_LIB_PREMEM_CONFIG   *CpuConfigLibPreMemConfig;
  CPU_SECURITY_PREMEM_CONFIG     *CpuSecurityPreMemConfig;
  MSR_IA32_BIOS_SIGN_ID_REGISTER Revision;
  UINT8                          FitErrorCode;
  UINT8                          FitEntryType;

  DEBUG ((DEBUG_INFO, "CpuOnPolicyInstalled Start \n"));
  PostCode (0xC20);

  Status = GetConfigBlock ((VOID *) SiPreMemPolicyPpi, &gCpuConfigLibPreMemConfigGuid, (VOID *) &CpuConfigLibPreMemConfig);
  ASSERT_EFI_ERROR (Status);

  Status = GetConfigBlock ((VOID *) SiPreMemPolicyPpi, &gCpuSecurityPreMemConfigGuid, (VOID *) &CpuSecurityPreMemConfig);
  ASSERT_EFI_ERROR (Status);

#if FixedPcdGetBool(PcdFspBinaryEnable) == 0
  ///
  /// Install Notify
  ///
  Status = PeiServicesNotifyPpi (&mNotifyList[0]);
  ASSERT_EFI_ERROR (Status);
#endif

  DEBUG ((DEBUG_INFO, "XmmInit Start \n"));
  PostCode (0xC2F);
  ///
  /// Init XMM support
  ///
  XmmInit ();

  MsrGetFitBiosError (&FitEntryType, &FitErrorCode);
  switch (FitEntryType) {
    case FIT_HEADER_ENTRY:
      DEBUG((DEBUG_INFO, "Fit Entry Type: Fit Header Entry\n"));
      switch (FitErrorCode) {
        case FIT_SUCCESSFUL:
          DEBUG ((DEBUG_INFO, "Fit Error Code: Successful\n"));
          break;
        case FIT_SIZE_CHECK:
          DEBUG ((DEBUG_INFO, "Fit Error Code: Size check failed\n"));
          break;
        case FIT_RESERVED_FIELD_CHECK:
          DEBUG ((DEBUG_INFO, "Fit Error Code: Reserved field check failed\n"));
          break;
        case FIT_VERSION_AND_TYPE_CHECK:
          DEBUG ((DEBUG_INFO, "Fit Error Code: Version and Type Check failed\n"));
          break;
        default:
          DEBUG ((DEBUG_INFO, "Fit Error Code: Unknown\n"));
          break;
      }
      break;
    case FIT_MICROCODE_UPDATE_ENTRY:
      DEBUG ((DEBUG_INFO, "Fit Entry Type: Microcode Update Entry\n"));
      switch (FitErrorCode) {
        case FIT_SUCCESSFUL:
          DEBUG ((DEBUG_INFO, "Fit Error Code: Successful\n"));
          break;
        case FIT_NO_MICROCODE_UPDATE:
          DEBUG ((DEBUG_INFO, "Fit Error Code: No microcode update found\n"));
          break;
        case FIT_MICROCODE_UPDATE_FAIL:
          DEBUG ((DEBUG_INFO, "Fit Error Code: Microcode update failure\n"));
          break;
        default:
          DEBUG ((DEBUG_INFO, "Fit Error Code: Unknown\n"));
          break;
      }
      break;
    case FIT_STARTUP_ACM_ENTRY:
      DEBUG ((DEBUG_INFO, "Fit Entry Type: Startup ACM Entry\n"));
      switch (FitErrorCode) {
        case FIT_SUCCESSFUL:
          DEBUG ((DEBUG_INFO, "Fit Error Code: Successful\n"));
          break;
        case FIT_STARTUP_ACM_NOT_SUPPORTED:
          DEBUG ((DEBUG_INFO, "Fit Error Code: Startup ACM not supported by CPU\n"));
          break;
        case FIT_FATAL_ERROR_DURING_ACM:
          DEBUG ((DEBUG_INFO, "Fit Error Code: Fatal error during ACM in previous boot\n"));
          break;
        case FIT_CPU_DOES_NOT_SUPPORT_LT:
          DEBUG ((DEBUG_INFO, "Fit Error Code: CPU doesn't support LT\n"));
          break;
        case FIT_BIST_ERRORS:
          DEBUG ((DEBUG_INFO, "Fit Error Code: BIST errors\n"));
          break;
        case FIT_BEYOND_END_OF_FIT:
          DEBUG ((DEBUG_INFO, "Fit Error Code: Beyond end of FIT\n"));
          break;
        case FIT_NO_FIT_ACM_TYPE_MISMATCH:
          DEBUG ((DEBUG_INFO, "Fit Error Code: No FIT ACM type mismatch\n"));
          break;
        case FIT_ACM_BASE_SIZE_AND_CHECKS:
          DEBUG ((DEBUG_INFO, "Fit Error Code: ACM base size and checks\n"));
          break;
        default:
          DEBUG ((DEBUG_INFO, "Fit Error Code: Unknown\n"));
          break;
      }
      break;
    case FIT_GENERAL_CHECKS:
      DEBUG ((DEBUG_INFO, "Fit Entry Type: General Checks\n"));
      switch (FitErrorCode) {
        case FIT_SUCCESSFUL:
          DEBUG ((DEBUG_INFO, "Fit Error Code: Successful\n"));
          break;
        case FIT_DISABLED_BY_CPU:
          DEBUG ((DEBUG_INFO, "Fit Error Code: FIT disabled by CPU\n"));
          break;
        case FIT_POINTER_ERROR:
          DEBUG ((DEBUG_INFO, "Fit Error Code: FIT pointer error (> FFFFFFB0; < FF000000)\n"));
          break;
        case FIT_FIRST_FIT_ENTRY_MISMATCH:
          DEBUG ((DEBUG_INFO, "Fit Error Code: First FIT entry doesn't match \"_FIT_\"\n"));
          break;
        default:
          DEBUG ((DEBUG_INFO, "Fit Error Code: Unknown\n"));
          break;
      }
      break;
    default:
      DEBUG ((DEBUG_INFO, "Fit Entry Type: Unknown\n"));
      DEBUG ((DEBUG_INFO, "Fit Error Code: Unknown\n"));
      break;
  }

  Revision.Uint64 = AsmReadMsr64 (MSR_IA32_BIOS_SIGN_ID);
  DEBUG ((DEBUG_INFO, "Microcode Update: 0x%08X\n", (UINT32) Revision.Bits.MicrocodeUpdateSignature));

  ///
  /// Initializes TXT after Policy PPI produced
  ///
  DEBUG ((DEBUG_INFO, "TxtInit Start \n"));
  PostCode (0xC3F);
  TxtInit ();

  if (CpuSecurityPreMemConfig->SkipStopPbet != TRUE) {
    ///
    /// Disable PBET
    ///
    StopPbeTimer ();
  }
  ///
  /// Init CPU Straps
  ///
  PostCode (0xC4F);

  ///
  /// Initialize CPU PECI Mailbox commands
  ///
  CpuPeciMailboxCommands (CpuConfigLibPreMemConfig);

  SetCpuStrapAndEarlyPowerOnConfig (SiPreMemPolicyPpi);


  ///
  /// Init Overclocking
  ///
  PostCode (0xC5F);
  CpuOcInitPreMem (SiPreMemPolicyPpi);


  //
  // Enable/Disable CPU Debug MSR as per setup and Lock Before Enabling TME (TME is being activated in Memory Init - MrcDone function)
  // If TME is enabled and not in Debug mode, then CPU Debug must be disabled and Locked.
  //
  DEBUG ((DEBUG_INFO, "Invoking LockEnableDisableCpuDebug \n"));
  LockEnableDisableCpuDebug ();

  DEBUG ((DEBUG_INFO, "CPU Pre-Mem Exit \n"));
  PostCode (0xC6F);

  return EFI_SUCCESS;
}

#if FixedPcdGetBool(PcdFspBinaryEnable) == 0
/**
  Build BIST HOB

  @param[in] PeiServices       - Indirect reference to the PEI Services Table.
  @param[in] NotifyDescriptor  - Address of the notification descriptor data structure. Type
                      EFI_PEI_NOTIFY_DESCRIPTOR is defined above.
  @param[in] Ppi               - Address of the PPI that was installed.

  @retval EFI_SUCCESS - Hob built or not necessary
**/
EFI_STATUS
EFIAPI
BuildBistHob (
  IN EFI_PEI_SERVICES          **PeiServices,
  IN EFI_PEI_NOTIFY_DESCRIPTOR *NotifyDescriptor,
  IN VOID                      *Ppi
  )
{
  EFI_STATUS                           Status;
  EFI_BOOT_MODE                        BootMode;
  EFI_SEC_PLATFORM_INFORMATION_PPI     *SecPlatformInformationPpi;
  UINT64                               InformationSize;
  EFI_SEC_PLATFORM_INFORMATION_RECORD  *SecPlatformInformation;
  BIST_HOB_DATA                        BspBistData;
  VOID                                 *Hob;
  SI_PREMEM_POLICY_PPI                 *SiPreMemPolicyPpi;
  UINT32                               BistStatus;
  CPU_CONFIG_LIB_PREMEM_CONFIG         *CpuConfigLibPreMemConfig;

  Status = PeiServicesGetBootMode (&BootMode);
  if (!EFI_ERROR (Status) && (BootMode == BOOT_ON_S3_RESUME)) {
    return EFI_SUCCESS;
  }

  Status = PeiServicesLocatePpi (
             &gEfiSecPlatformInformationPpiGuid,   // GUID
             0,                                    // Instance
             NULL,                                 // EFI_PEI_PPI_DESCRIPTOR
             (VOID ** ) &SecPlatformInformationPpi // PPI
             );

  if (Status == EFI_NOT_FOUND) {
    return EFI_SUCCESS;
  }

  if (EFI_ERROR (Status)) {
    return Status;
  }

  Status = PeiServicesLocatePpi (
             &gSiPreMemPolicyPpiGuid,
             0,
             NULL,
             (VOID **) &SiPreMemPolicyPpi
             );
  ASSERT_EFI_ERROR (Status);

  Status = GetConfigBlock ((VOID *) SiPreMemPolicyPpi, &gCpuConfigLibPreMemConfigGuid, (VOID *) &CpuConfigLibPreMemConfig);
  ASSERT_EFI_ERROR (Status);

  ///
  /// Obtain BIST information for BSP. Information for APs will obtained in DXE phase during MP initialization.
  ///
  ZeroMem (&BspBistData, sizeof (BIST_HOB_DATA));

  SecPlatformInformation = (EFI_SEC_PLATFORM_INFORMATION_RECORD *) (&(BspBistData.Health));
  InformationSize        = sizeof (BspBistData.Health);
  Status = SecPlatformInformationPpi->PlatformInformation (
                                        (CONST EFI_PEI_SERVICES  **) PeiServices,
                                        &InformationSize,
                                        SecPlatformInformation
                                        );

  if (EFI_ERROR (Status)) {
    return Status;
  }

  ///
  /// Dump BIST information to serial log
  ///
  BistStatus = SecPlatformInformation->x64HealthFlags.Uint32;
  DEBUG ((DEBUG_INFO, "BistStatus = 0x%x\n", BistStatus));

  if (BistStatus != 0) {
    DEBUG ((DEBUG_ERROR, "BIST for BSP failed\n"));
    ReportStatusCode (
      EFI_ERROR_MAJOR | EFI_ERROR_CODE,
      EFI_COMPUTING_UNIT_HOST_PROCESSOR | EFI_CU_HP_EC_SELF_TEST
      );
    ASSERT (FALSE);
  } else if (CpuConfigLibPreMemConfig->BistOnReset == 1) {
    DEBUG ((DEBUG_INFO, "BIST for BSP passed\n"));
  }

  BspBistData.ApicId = GetCpuApicId ();
  Hob = BuildGuidDataHob (
          &gHtBistHobGuid,
          &BspBistData,
          sizeof (BIST_HOB_DATA)
          );
  ASSERT (Hob != NULL);

  return Status;
}
#endif

