/** @file
  Data for SMBIOS Cache tables (Type 7).

@copyright
  INTEL CONFIDENTIAL
  Copyright 2013 - 2020 Intel Corporation.

  The source code contained or described herein and all documents related to the
  source code ("Material") are owned by Intel Corporation or its suppliers or
  licensors. Title to the Material remains with Intel Corporation or its suppliers
  and licensors. The Material may contain trade secrets and proprietary and
  confidential information of Intel Corporation and its suppliers and licensors,
  and is protected by worldwide copyright and trade secret laws and treaty
  provisions. No part of the Material may be used, copied, reproduced, modified,
  published, uploaded, posted, transmitted, distributed, or disclosed in any way
  without Intel's prior express written permission.

  No license under any patent, copyright, trade secret or other intellectual
  property right is granted to or conferred upon you by disclosure or delivery
  of the Materials, either expressly, by implication, inducement, estoppel or
  otherwise. Any license under such intellectual property rights must be
  express and approved by Intel in writing.

  Unless otherwise agreed by Intel in writing, you may not remove or alter
  this notice or any other notice embedded in Materials by Intel or
  Intel's suppliers or licensors in any way.

  This file contains an 'Intel Peripheral Driver' and is uniquely identified as
  "Intel Reference Module" and is licensed for Intel CPUs and chipsets under
  the terms of your license agreement with Intel or your vendor. This file may
  be modified by the user, subject to additional terms of the license agreement.

@par Specification Reference:
  System Management BIOS (SMBIOS) Reference Specification v3.0.0
  dated 2015-Feb-12 (DSP0134)
  http://www.dmtf.org/sites/default/files/standards/documents/DSP0134_3.0.0.pdf
**/

#include "SmbiosCpu.h"
#include <Library/CpuInfoFruLib.h>
#include <Ppi/MpServices.h>
#include <Library/PeiServicesLib.h>
#include <Library/PeiServicesTablePointerLib.h>
extern EFI_PEI_MP_SERVICES_PPI  *gMpServicesPpi;

///
/// Cache Information (Type 7)
///
GLOBAL_REMOVE_IF_UNREFERENCED SMBIOS_CACHE_INFO SmbiosCacheInfoData = {
  TO_BE_FILLED,                  ///< ProcessorSocketNumber
  TO_BE_FILLED,                  ///< NumberOfCacheLevels
  STRING_1,                      ///< SocketDesignationStrIndex
  TO_BE_FILLED,                  ///< CacheConfiguration
  TO_BE_FILLED,                  ///< MaxCacheSize
  TO_BE_FILLED,                  ///< InstalledSize
  SMBIOS_TYPE7_SRAM_SYNCHRONOUS, ///< SupportedSramType
  SMBIOS_TYPE7_SRAM_SYNCHRONOUS, ///< CurrentSramType
  0,                             ///< CacheSpeed
  TO_BE_FILLED,                  ///< ErrorCorrectionType
  TO_BE_FILLED,                  ///< SystemCacheType
  TO_BE_FILLED,                  ///< Associativity
  TO_BE_FILLED,                  ///< MaxCacheSize2
  TO_BE_FILLED,                  ///< InstalledSize2
};
GLOBAL_REMOVE_IF_UNREFERENCED SMBIOS_CACHE_INFO_STRING_ARRAY SmbiosCacheInfoStrings = {
  TO_BE_FILLED_STRING, ///< SocketDesignation
};

///
/// Convert Cache Type Field to SMBIOS format
///
#define SMBIOS_CACHE_TYPE_MAX 5
GLOBAL_REMOVE_IF_UNREFERENCED UINT8 SmbiosCacheTypeFieldConverter[SMBIOS_CACHE_TYPE_MAX] = {
  CacheTypeUnknown,
  CacheTypeData,
  CacheTypeInstruction,
  CacheTypeUnified,
  CacheTypeOther
};

GLOBAL_REMOVE_IF_UNREFERENCED CHAR8 *CacheStrings[] = {
  "L1 Cache",
  "L2 Cache",
  "L3 Cache"
};
GLOBAL_REMOVE_IF_UNREFERENCED CHAR8 UnknownString[] = "Unknown";
UINTN                               mNumberOfEnabledProcessors = 0;
UINT16                              mSocketNumber = 0;
UINT8                               mBigCoreCount = 0;
UINT8                               mSmallCoreCount = 0;
UINT16                              mNumberOfProcessors = 0;
BOOLEAN                             mHeteroCoreSupported = FALSE;

/**
  This function gets number of different cores present in the platform

**/
STATIC
VOID
GetNumberOfHybridCores (
  VOID
  )
{
  if (DetectCoreType () == V_BIG_CORE_TYPE) {
    mBigCoreCount++;
  } else {
    mSmallCoreCount++;
  }
}
/**

  This function produces/ Updates the SMBIOS Table with cache information of cores based on High & Low performance cores

**/
VOID
EFIAPI
UpdateSmbiosCacheInfoHobs (
  VOID
  )
{
  UINT8                                 CacheIndex;
  CPUID_CACHE_PARAMS_EAX                Eax;
  CPUID_CACHE_PARAMS_EBX                Ebx;
  UINT32                                RegEcx;
  UINT8                                 CacheLevel[MAX_NUMBER_OF_CACHE_LEVELS];
  UINT8                                 LxCacheType[MAX_NUMBER_OF_CACHE_LEVELS];
  UINT32                                Ways[MAX_NUMBER_OF_CACHE_LEVELS];
  UINT32                                Partitions;
  UINT32                                LineSets;
  UINT32                                Sets;
  UINT32                                MaxCacheSize[MAX_NUMBER_OF_CACHE_LEVELS];
  UINT16                                NumberOfCacheLevels;
  BOOLEAN                               CombineL1DataAndInstruction;
  UINT8                                 Index;
  SMBIOS_TYPE7_CACHE_CONFIGURATION_DATA CacheConfig;
  STATIC BOOLEAN                        StaticSmallCoreCacheDone = FALSE;
  STATIC BOOLEAN                        StaticBigCoreCacheDone = FALSE;
  STATIC UINT8                          StaticCurrentSmallCoreNum = 0;
  STATIC UINT8                          StaticCurrentBigCoreNum = 0;
  UINT32                                MaximumCacheSize2;
  UINT32                                InstalledSize2;
  STATIC BOOLEAN                        StaticCacheCalculateDone = FALSE;
  STATIC UINT16                         StaticMaxCache = 0;
  STATIC UINT16                         StaticInstalledCache = 0;

  if (mHeteroCoreSupported) {
    if (DetectCoreType () == V_BIG_CORE_TYPE) {
      StaticCurrentBigCoreNum++;
    } else {
      StaticCurrentSmallCoreNum++;
    }
  }

  if ((((StaticCurrentSmallCoreNum == 1) && (StaticSmallCoreCacheDone == FALSE)) || ((StaticCurrentBigCoreNum == 1) && (StaticBigCoreCacheDone == FALSE)))|| ((!mHeteroCoreSupported) && (StaticCacheCalculateDone == FALSE))) {  // calculate cache information one time for Low & high performance cores
    ZeroMem (CacheLevel, sizeof (CacheLevel));
    ZeroMem (LxCacheType, sizeof (LxCacheType));
    ZeroMem (Ways, sizeof (Ways));
    ZeroMem (MaxCacheSize, sizeof (MaxCacheSize));

    NumberOfCacheLevels = 0;
    ///
    /// Find the total number of cache levels reported via CPUID instruction.
    /// Save the CPUID output for later use.
    ///
    for (CacheIndex = 0;;CacheIndex++) {
      AsmCpuidEx (CPUID_CACHE_PARAMS, CacheIndex, &Eax.Uint32, &Ebx.Uint32, &RegEcx, NULL);
      if (Eax.Bits.CacheType == 0) {
        break;
      }
      NumberOfCacheLevels++;
      if (NumberOfCacheLevels > MAX_NUMBER_OF_CACHE_LEVELS) {
        DEBUG ((DEBUG_ERROR, "Error producing Cache Info HOBs. Total number of reported cache levels exceeded limit.\n"));
        break;
      }

      CacheLevel[CacheIndex] = (UINT8) Eax.Bits.CacheLevel;
      LxCacheType[CacheIndex] = (UINT8) Eax.Bits.CacheType;

      Ways[CacheIndex] = Ebx.Bits.Ways + 1;
      Partitions = Ebx.Bits.LinePartitions + 1;
      LineSets = Ebx.Bits.LineSize + 1;
      Sets = RegEcx + 1;
      MaxCacheSize[CacheIndex] = (Ways[CacheIndex] * Partitions * LineSets * Sets) / 1024;
    }

    ///
    /// Check if L1 Data cache and L1 instruction cache can be combined.
    ///
    CombineL1DataAndInstruction = FALSE;
    if (NumberOfCacheLevels == 4) {
      if ((CacheLevel[0] == 1) && (CacheLevel[1] == 1) && (Ways[0] == Ways[1]) && (MaxCacheSize[0] == MaxCacheSize[1])) {
        ///
        /// If one is Data cache, and the other is Instruction cache.
        ///
        if (((LxCacheType[0] == 1) && (LxCacheType[1] == 2)) || ((LxCacheType[0] == 2) && (LxCacheType[1] == 1))) {
          CombineL1DataAndInstruction = TRUE;
          LxCacheType[1] = CacheTypeUnified; //Unified
          MaxCacheSize[1] = MaxCacheSize[0] + MaxCacheSize[1];
          NumberOfCacheLevels = 3;
        }
      }
    }

    SmbiosCacheInfoData.ProcessorSocketNumber = mSocketNumber;
    SmbiosCacheInfoData.NumberOfCacheLevels = NumberOfCacheLevels;

    for (CacheIndex = 0; CacheIndex < NumberOfCacheLevels; CacheIndex++) {
      Index = CacheIndex;
      if (CombineL1DataAndInstruction == TRUE) {
        Index = CacheIndex + 1;
      }

      ///
      /// Index can never be >= MAX_NUMBER_OF_CACHE_LEVELS
      ///
      if (Index >= MAX_NUMBER_OF_CACHE_LEVELS) {
        ASSERT (FALSE);
        return;
      }

      ///
      /// Make CacheLevel zero-based for indexing and for SMBIOS Cache Configuration format
      ///
      if (CacheLevel[Index] == 0) {
        ASSERT (FALSE);
        return;
      }
      CacheLevel[Index]--;

      if (CacheLevel[Index] < 3) {
        SmbiosCacheInfoStrings.SocketDesignation = CacheStrings[CacheLevel[Index]];
      } else {
        SmbiosCacheInfoStrings.SocketDesignation = (CHAR8 *)&UnknownString;
      }

      CacheConfig.Level = CacheLevel[Index];
      CacheConfig.Socketed = CACHE_NOTSOCKETED;                    // Not Socketed
      CacheConfig.Reserved1 = 0;
      CacheConfig.Location = CACHE_LOCATION_INTERNAL;              // Internal Cache
      CacheConfig.Enable = CACHE_ENABLE;                           // Cache enabled
      CacheConfig.OperationalMode = CACHE_OPMODE_WRITE_BACK;       // Write Back
      CacheConfig.Reserved2 = 0;
      SmbiosCacheInfoData.CacheConfiguration = *(UINT16 *)&CacheConfig;

      SmbiosCacheInfoData.MaxCacheSize = (UINT16) MaxCacheSize[Index];
      SmbiosCacheInfoData.InstalledSize = (UINT16) MaxCacheSize[Index];
      ///
      /// For L1 and L2 cache, multiply size by the number of cores.
      ///
      if (CacheLevel[Index] < 2) {
        SmbiosCacheInfoData.MaxCacheSize = (SmbiosCacheInfoData.MaxCacheSize * ((UINT8) mNumberOfProcessors - (mHeteroCoreSupported ? StaticCurrentBigCoreNum == 1 ? mSmallCoreCount : mBigCoreCount : 0))) + StaticMaxCache;
        SmbiosCacheInfoData.InstalledSize = (SmbiosCacheInfoData.InstalledSize * ((UINT8) mNumberOfEnabledProcessors - (mHeteroCoreSupported ? StaticCurrentBigCoreNum == 1 ? mSmallCoreCount : mBigCoreCount : 0))) + StaticInstalledCache;
        StaticMaxCache = (Index < 1 ? SmbiosCacheInfoData.MaxCacheSize : 0);
        StaticInstalledCache = (Index < 1 ? SmbiosCacheInfoData.InstalledSize : 0);
        if (Index == 2) {
          (mHeteroCoreSupported ? StaticCurrentBigCoreNum == 1 ? (StaticBigCoreCacheDone = TRUE) : (StaticSmallCoreCacheDone = TRUE) : (StaticCacheCalculateDone = TRUE));
        }
      }

      ///
      /// Calculate the Max Cache Size in MB
      ///
      SmbiosCacheInfoData.MaximumCacheSize2 = 0;
      MaximumCacheSize2 = (SmbiosCacheInfoData.MaxCacheSize & 0x7FFF) / 1024;
      if ((SmbiosCacheInfoData.MaxCacheSize & BIT15) > 0) {
        MaximumCacheSize2 = MaximumCacheSize2 * 64;
        if (MaximumCacheSize2 <= 2047 ) {
          SmbiosCacheInfoData.MaximumCacheSize2 |= BIT31;
        }
      }
      if (MaximumCacheSize2 <= 2047) {
        ///
        /// For Cache sizes of 2047 MB or smaller, the value in MaxCacheSize in given granularity
        /// equals MaximumCacheSize2 and the Granularity bit matches the value of Granilarity bit in MaxCacheSize
        ///
        SmbiosCacheInfoData.MaximumCacheSize2 |= (UINT32) SmbiosCacheInfoData.MaxCacheSize & 0x7FFF;
      } else {
        ///
        /// For Cache sizes greater than 2047 MB, the MaxiCacheSize field is set to 0xFFFF
        /// and the MaximumCacheSize2 field is present, the Granularity bit is set to 1b and the size set as required
        ///
        SmbiosCacheInfoData.MaxCacheSize = 0xFFFF;
        SmbiosCacheInfoData.MaximumCacheSize2 = MaximumCacheSize2 | BIT31;
      }

      ///
      /// Calculate the Installed Cache Size in MB
      ///
      SmbiosCacheInfoData.InstalledSize2 = 0;
      InstalledSize2 = (SmbiosCacheInfoData.InstalledSize & 0x7FFF) / 1024;
      if ((SmbiosCacheInfoData.InstalledSize & BIT15) > 0) {
        InstalledSize2 = InstalledSize2 * 64;
        if (InstalledSize2 <= 2047 ) {
          SmbiosCacheInfoData.InstalledSize2 |= BIT31;
        }
      }
      if (InstalledSize2 <= 2047) {
        ///
        /// For Cache sizes of 2047 MB or smaller, the value in InstalledSize in given granularity
        /// equals InstalledSize2 and the Granularity bit matches the value of Granularity bit in InstalledSize
        ///
        SmbiosCacheInfoData.InstalledSize2 |= (UINT32) SmbiosCacheInfoData.InstalledSize & 0x7FFF;
      } else {
        ///
        /// For Cache sizes greater than 2047 MB, the MaxiCacheSize field is set to 0xFFFF
        /// and the InstalledSize2 field is present, the Granularity bit is set to 1b and the size set as required
        ///
        SmbiosCacheInfoData.InstalledSize = 0xFFFF;
        SmbiosCacheInfoData.InstalledSize2 = InstalledSize2 | BIT31;
      }

      switch (CacheLevel[Index]) {
        case 0:
          SmbiosCacheInfoData.ErrorCorrectionType = CacheErrorParity;
          break;
        case 1:
          SmbiosCacheInfoData.ErrorCorrectionType = CacheErrorSingleBit;
          break;
        case 2:
          SmbiosCacheInfoData.ErrorCorrectionType = CacheErrorMultiBit;
          break;
        default:
          SmbiosCacheInfoData.ErrorCorrectionType = CacheErrorUnknown;
          break;
      }
      ///
      /// If cache type is larger or equal than 5, this is undefined type so mark it as "Other" Cache type.
      ///
      if (LxCacheType[Index] >= SMBIOS_CACHE_TYPE_MAX) {
        LxCacheType[Index] = SMBIOS_CACHE_TYPE_MAX - 1;
      }
      SmbiosCacheInfoData.SystemCacheType = SmbiosCacheTypeFieldConverter[LxCacheType[Index]];
      ///
      /// Convert Associativity Ways to SMBIOS format
      ///
      switch (Ways[Index]) {
        case 2:
          Ways[Index] = CacheAssociativity2Way;
          break;
        case 4:
          Ways[Index] = CacheAssociativity4Way;
          break;
        case 8:
          Ways[Index] = CacheAssociativity8Way;
          break;
        case 12:
          Ways[Index] = CacheAssociativity12Way;
          break;
        case 16:
          Ways[Index] = CacheAssociativity16Way;
          break;
        case 24:
          Ways[Index] = CacheAssociativity24Way;
          break;
        case 32:
          Ways[Index] = CacheAssociativity32Way;
          break;
        case 48:
          Ways[Index] = CacheAssociativity48Way;
          break;
        case 64:
          Ways[Index] = CacheAssociativity64Way;
          break;
        default:
          Ways[Index] = CacheAssociativityOther;
          break;
      }
      SmbiosCacheInfoData.Associativity = (UINT8) Ways[Index];

      AddSmbiosStringsAndBuildGuidDataHob (
        (VOID *) &SmbiosCacheInfoData,
        sizeof (SMBIOS_CACHE_INFO),
        (CHAR8 **) &SmbiosCacheInfoStrings,
        SMBIOS_CACHE_INFO_NUMBER_OF_STRINGS,
        &gSmbiosCacheInfoHobGuid
        );
    }
  }
  return;
}
/**
  This function produces SMBIOS Cache Info HOBs.

  @param[in] ProcessorSocketNumber - The processor socket number corresponding to the Cache Info HOBs to produce.
**/
VOID
InitializeSmbiosCacheInfoHobs (
  IN  UINT16               ProcessorSocketNumber
  )
{
  CONST EFI_PEI_SERVICES                **PeiServices;
  EFI_STATUS                            Status;

  DEBUG ((DEBUG_INFO, "InitializeSmbiosCacheInfoHobs() - Start\n"));
  PeiServices = GetPeiServicesTablePointer ();

  Status = PeiServicesLocatePpi (&gEfiPeiMpServicesPpiGuid, 0, NULL, (VOID **)&gMpServicesPpi);
  ASSERT_EFI_ERROR (Status);

  ///
  /// Determine the number of supported and active cores for L1/L2 total cache size.
  ///
  gMpServicesPpi->GetNumberOfProcessors (PeiServices, gMpServicesPpi, (UINTN *) &mNumberOfProcessors, (UINTN *) &mNumberOfEnabledProcessors);

  mHeteroCoreSupported = IsHeteroCoreSupported ();

  if (mHeteroCoreSupported) {
  ///
  /// Get current no.of cores present in platform
  ///
    GetNumberOfHybridCores ();
    gMpServicesPpi->StartupAllAPs (
                      PeiServices,
                      gMpServicesPpi,
                      (EFI_AP_PROCEDURE) GetNumberOfHybridCores,
                      TRUE,
                      0,
                      NULL
                      );

  DEBUG ((DEBUG_INFO, "No.of cores present in platform: BigCores= %d  SmallCores= %d\n",mBigCoreCount, mSmallCoreCount));
  }
  mSocketNumber = ProcessorSocketNumber;
  ///
  /// Update SMBIOS table 7 with cache information
  ///
  UpdateSmbiosCacheInfoHobs (); // Run on BSP
  if (mHeteroCoreSupported) {
    gMpServicesPpi->StartupAllAPs (
                      PeiServices,
                      gMpServicesPpi,
                      (EFI_AP_PROCEDURE) UpdateSmbiosCacheInfoHobs,
                      TRUE,
                      0,
                      NULL
                      );
  }

  DEBUG ((DEBUG_INFO, "InitializeSmbiosCacheInfoHobs() - End\n"));
  return;
}
