/** @file

  @copyright
  INTEL CONFIDENTIAL
  Copyright 2015 - 2020 Intel Corporation. <BR>

  The source code contained or described herein and all documents related to the
  source code ("Material") are owned by Intel Corporation or its suppliers or
  licensors. Title to the Material remains with Intel Corporation or its suppliers
  and licensors. The Material may contain trade secrets and proprietary and
  confidential information of Intel Corporation and its suppliers and licensors,
  and is protected by worldwide copyright and trade secret laws and treaty
  provisions. No part of the Material may be used, copied, reproduced, modified,
  published, uploaded, posted, transmitted, distributed, or disclosed in any way
  without Intel's prior express written permission.

  No license under any patent, copyright, trade secret or other intellectual
  property right is granted to or conferred upon you by disclosure or delivery
  of the Materials, either expressly, by implication, inducement, estoppel or
  otherwise. Any license under such intellectual property rights must be
  express and approved by Intel in writing.

  Unless otherwise agreed by Intel in writing, you may not remove or alter
  this notice or any other notice embedded in Materials by Intel or
  Intel's suppliers or licensors in any way.
**/

#ifndef _ME_FIA_MUX_LIB_H_
#define _ME_FIA_MUX_LIB_H_

#pragma pack (1)
//
// Define a gap of non used lines
//
// SNR uses 0-7 & 16-23
#define HSIO_GAP_SNR_LOW                 7
#define HSIO_GAP_SNR_HI                  16
// TNR uses 0-15 & 20-23
#define HSIO_GAP_TNR_LOW                 15
#define HSIO_GAP_TNR_HI                  20
// ICX-D uses 0 - 23 (no gap)

//
//  The structure of FIA MUX configuration that can be set in ME.
//
typedef enum {
  FIA_MUX_LANE_DISABLED,
  FIA_MUX_LANE_PCIE_OWNER,
  FIA_MUX_LANE_SATA_OWNER,
  FIA_MUX_LANE_USB_OWNER
} FIA_MUX_SELECTION;

typedef union {
  struct {
    UINT8 FiaMuxSelection    : 2;
    UINT8 FiaMuxHardwareIndex: 6;
  } Data;
  UINT8   Raw;
} FIA_MUX_RECORD;

typedef enum {
  SATA_CONTROLLER_ASSIGNED,
  SATA_CONTROLLER_NOT_ASSIGNED
} SATA_SELECTION;

typedef union {
  struct {
    UINT8 SataSelection    : 2;
    UINT8 SataHardwareIndex: 6;
  } Data;
  UINT8   Raw;
} SATA_RECORD;

typedef enum {
  PCIE_ROOT_PORT_IS_ENABLED,
  PCIE_ROOT_PORT_IS_DISABLED
} PCIE_ROOT_PORT_ENABLED;

typedef enum {
  PCIE_ROOT_PORT_LINK_WIDTH_SET_BY_BICTRL = 0,
  PCIE_ROOT_PORT_LINK_WIDTH_X1 = 0xF
} PCIE_ROOT_PORT_LINK_WIDTH;

typedef union {
  struct {
    UINT8 PcieRootPortEnable    : 1;
    UINT8 PcieRootPortLinkWidth : 4;
    UINT8 Reserved              : 3;
    UINT8 PcieRootPortHardwareIndex;
  } Data;
  UINT16                       Raw;
} PCIE_ROOT_PORT_RECORD;

typedef struct {
  UINT8                        *FiaMuxLaneOverride;       // Table pointer to override per lane switches
  UINT8                        FiaMuxCfgInvalidate;       // FiaMuxInvalidate setting passed in Set message
  UINT8                        FiaMuxRecordsCount;        // Number of FiaMux lanes in configuration
  UINT8                        SataRecordsCount;          // Number of SATA lanes described in configuration
  UINT8                        PcieRootPortRecordsCount;  // Number of RootPort lanes described in configuration
  FIA_MUX_RECORD               *FiaMux;                   // Table pointer to FiaMux description table
  SATA_RECORD                  *Sata;                     // Table pointer to SATA description table
  PCIE_ROOT_PORT_RECORD        *PcieRootPort;             // Table pointer to RootPorts description table
  // Here should be allocated buffers and above pointers should point to
} ME_FIA_CONFIG;

#pragma pack ()

/**
  Check if FIA MUX is supported by ME.

  @retval TRUE         FiaMux messages are allowed
  @retval FALSE        FiaMux messages are not allowed
**/
BOOLEAN
MeFiaMuxAllowed (
  VOID
  );

/**
  Read FIA MUX configuration from ME.

  @param [out] pConfig              Pointer to a buffer for ME FIA MUX configuration
  @param [out] pLanesAllowed        Pointer to a buffer for the number of lanes allowed
  @param [out] OverridesActive      Pointer to a OverridesActive indicator.
                                    If TRUE, set command should be executed

  @retval EFI_NOT_READY         Returned if this API is called before ME signals init
                                complete, or after END_OF_POST was sent.
  @retval EFI_INVALID_PARAMETER Returned if any passed NULL configuration pointer
  @retval EFI_DEVICE_ERROR      Returned if HECI transport is not available.
  @retval EFI_PROTOCOL_ERROR    Unexpected protocol error returned by ME.
  @retval EFI_SUCCESS           FiaMux configuration has been read successfully
**/
EFI_STATUS
EFIAPI
MeFiaMuxConfigGet (
  OUT ME_FIA_CONFIG *pConfig,
  OUT UINT32        *pLanesAllowed,
  OUT BOOLEAN       *OverridesActive
  );


/**
  Sends FIA MUX configuration to ME.

  @param [in] pConfig           Pointer to FIA MUX configuration to set in ME,
                                when NULL, Invalidate field is set to 1 (causing
                                invalidate of configuration stored in flash -
                                configuration from Soft Straps will be used)

  @retval EFI_NOT_READY         Returned if this API is called before ME signals init
                                complete, or after END_OF_POST was sent.
  @retval EFI_DEVICE_ERROR      Returned if HECI transport is not available.
  @retval EFI_PROTOCOL_ERROR    Unexpected protocol error returned by ME.
  @retval EFI_OUT_OF_RESOURCES  Requested configuration refused by ME, because
                                it does not match the SoC SKU.
  @retval EFI_ABORTED           Requested configuration refused by ME, because it is
                                invalid, or nothing to update
  @retval EFI_INVALID_PARAMETER Input parameter is invalid
  @retval EFI_SUCCESS           FiaMux configuration has been sent successfully
**/
EFI_STATUS
EFIAPI
MeFiaMuxConfigSet (
  IN ME_FIA_CONFIG *pConfig
  );

/**
  Get number of HSIO lanes for given CPU

  @param [out]  LaneGapLow  pointer to return low limit of unused lines
  @param [out]  LaneGapHi   pointer to return hi limit of unused lines

  @retval UINT8             Number of Hsio lanes in the system
**/
UINT8
GetHsioLanes (
  OUT UINT8 *LaneGapLow,
  OUT UINT8 *LaneGapHi
  );

/**
  Return of buffers size that should be associated with policy
  This functions helps in buffers allocation for 4 tables:
    FIA_MUX_RECORD, SATA_RECORD, PCIE_ROOT_PORT_RECORD, Override indication
  for number of supported lanes

  @retval UINT32   Buffer size
**/
UINT32
GetFiaMuxBuffersSize (
  VOID
  );

/**
  Sets pointer in ME_FIA_CONFIG to buffer relative to FiaMuxConfig parameter
  This functions helps in ME_FIA_CONFIG buffers organization for 4 tables:
    FIA_MUX_RECORD, SATA_RECORD, PCIE_ROOT_PORT_RECORD, Override indication

  @param [in] FiaMuxConfig       pointer to ME_FIA_CONFIG

  @retval EFI_INVALID_PARAMETER  Passed pointer is not valid
  @retval EFI_SUCCESS            Pointer has been set
**/
EFI_STATUS
SetFiaMuxBufferPointers (
  IN ME_FIA_CONFIG     *FiaMuxConfig
  );

#endif // _ME_FIA_MUX_LIB_H_
