/** @file
  This file contains functions for CPU DMI configuration for SIP16

@copyright
  INTEL CONFIDENTIAL
  Copyright 2019 - 2020 Intel Corporation.

  The source code contained or described herein and all documents related to the
  source code ("Material") are owned by Intel Corporation or its suppliers or
  licensors. Title to the Material remains with Intel Corporation or its suppliers
  and licensors. The Material may contain trade secrets and proprietary and
  confidential information of Intel Corporation and its suppliers and licensors,
  and is protected by worldwide copyright and trade secret laws and treaty
  provisions. No part of the Material may be used, copied, reproduced, modified,
  published, uploaded, posted, transmitted, distributed, or disclosed in any way
  without Intel's prior express written permission.

  No license under any patent, copyright, trade secret or other intellectual
  property right is granted to or conferred upon you by disclosure or delivery
  of the Materials, either expressly, by implication, inducement, estoppel or
  otherwise. Any license under such intellectual property rights must be
  express and approved by Intel in writing.

  Unless otherwise agreed by Intel in writing, you may not remove or alter
  this notice or any other notice embedded in Materials by Intel or
  Intel's suppliers or licensors in any way.

  This file contains an 'Intel Peripheral Driver' and is uniquely identified as
  "Intel Reference Module" and is licensed for Intel CPUs and chipsets under
  the terms of your license agreement with Intel or your vendor. This file may
  be modified by the user, subject to additional terms of the license agreement.

@par Specification Reference:
**/

#include <Base.h>
#include <Uefi/UefiBaseType.h>
#include <IndustryStandard/Pci22.h>
#include <Library/IoLib.h>
#include <Library/DebugLib.h>
#include <Library/BaseLib.h>
#include <Library/BaseMemoryLib.h>
#include <Library/PeiServicesLib.h>
#include <Library/ConfigBlockLib.h>
#include <Library/PciSegmentLib.h>
#include <Ppi/SiPolicy.h>
#include <Library/CpuPlatformLib.h>
#include <Library/PeiCpuDmiInitLib.h>
#include <Register/CpuDmiRegs.h>
#include <Register/CpuDmi16Regs.h>
#include <CpuDmiPreMemConfig.h>
#include <Register/CpuPcieRegs.h>
#include <Library/CpuPcieInitCommon.h>
#include <Library/TimerLib.h>
#include <Library/CpuDmiInfoLib.h>
#include <Library/PeiPchDmiLib.h>
#include <Register/PchFiaRegs.h>
#include <CpuSbInfo.h>
#include <Library/CpuRegbarAccessLib.h>
#include <Library/PeiCpuDmiDekelInitLib.h>

enum CPU_DMI_SPEED {
  CpuDmiAuto,
  CpuDmiGen1,
  CpuDmiGen2,
  CpuDmiGen3,
};


/**
  This function performs basic DMI initialization.
**/
STATIC
VOID
CpuDmiBasicInit (
  IN  UINT32                       DmiBar,
  IN  CPU_DMI_PREMEM_CONFIG        *CpuDmiPreMemConfig
  )
{
  UINT32                        Data32Or;
  DMI_ASPM                      DmiAspm;

  Data32Or = 0;
  DmiAspm = CpuDmiPreMemConfig->DmiAspm;
  // Note : All registers in this API are not called out in BWG - check if this is required for CPU DMI
  //
  // Section 4.1 DownStream Port DMI specific in SIP16 Converged PCIe BWG 0.92 >>>
  //
  //
  // Set SERR# Enable
  //
  MmioAndThenOr32(
      DmiBar + PCI_COMMAND_OFFSET,
      ~0u,
      EFI_PCI_COMMAND_SERR
      );

  //
  // Program the Capabilities Pointer
  //
    MmioAndThenOr32 (
    DmiBar + R_CPU_DMI16_CAPP,
    (UINT32)~B_CPU_DMI16_CAPP_PTR_MASK,
    V_CPU_DMI16_CAPP_PTR
    );

  /*
    Advanced Error Reporting
    1. Set Capability Version, Dxx:Fn +100h[19:16] = 1h
    2. Set Capability ID, Dxx:Fn +100h[15:0] = 0001h
  */
  MmioWrite32 (DmiBar + R_CPU_DMI_AECH, (UINT32) ((V_CPU_DMI_CV << N_CPU_DMI_EXCAP_CV) | V_CPU_DMI_EX_AEC_CID));
  /*
    Enable Secondary PCI Express Extended Capability
    1. Set Capability Version, 0xA30[19:16] = 1h
    2. Set Capability ID,  0xA30[15:0] = 0019h
  */
  MmioWrite32 (DmiBar + R_CPU_DMI_SPEECH, (UINT32) ((V_CPU_DMI_CV << N_CPU_DMI_EXCAP_CV) | V_CPU_DMI_EX_SPE_CID));
  /*
    Data Link Feature Extended Capability Header(DLFECH)
    1. Set Capability Version, 0xA90[19:16] = 1h
    2. Set Capability ID, 0xA90[15:0] = 0025h
  */
  MmioWrite32 (DmiBar + R_CPU_DMI_DLFECH, (UINT32) ((V_CPU_DMI_CV << N_CPU_DMI_EXCAP_CV) | V_CPU_DMI_EX_DLFECH_CID));
  //
  // Section 4.1 DownStream Port DMI specific in SIP16 Converged PCIe BWG 0.92 <<<
  //

  //
  // Section 4.3 Standard PCIe / Downstream Port DMI / RCiEP for 2LM Specific >>>
  //

  //
  // Enable Unsupported Request Reporting
  // Enable Fatal Error Reporting Enable
  // Enable Non-Fatal Error Reporting
  // Enable Correctable Error Reporting
  //
  MmioAndThenOr32 (
    DmiBar + R_CPU_DMI16_DCTL,
    ~0u,
    (B_CPU_DMI16_DCTL_URE | B_CPU_DMI16_DCTL_NEE | B_CPU_DMI16_DCTL_NFE | B_CPU_DMI16_DCTL_CEE)
    );

  //
  // Program L0s Exit Latency to 100b = 512ns to less than 1us
  // Enable L0s and L1 substates
  // Program L1 Exit Latency to 100b = 8us to less than 16us
  //
   if (DmiAspm == DmiAspmL0sL1) {
    //
    // Enable L0s/L1 on DMI
    //
    Data32Or = V_CPU_DMI16_LCAP_APMS_L0S_L1;
  } else if (DmiAspm == DmiAspmL0s) {
    //
    // Enable L0s Entry only
    //
    Data32Or = V_CPU_DMI16_LCAP_APMS_L0S;

  } else if ((DmiAspm == DmiAspmAutoConfig) || (DmiAspm == DmiAspmL1)) {
    //
    // Enable L1 Entry only
    //
    Data32Or = V_CPU_DMI16_LCAP_APMS_L1;
  } else {
    //
    // ASPM Disabled
    //
    Data32Or  = V_CPU_DMI16_LCAP_APMS_DIS;
  }
  Data32Or = (Data32Or << N_CPU_DMI16_LCAP_APMS);
  Data32Or = Data32Or | (CpuDmiPreMemConfig->DmiAspmL1ExitLatency << N_CPU_DMI16_LCAP_EL1);
  DEBUG((DEBUG_INFO, "Basic init Data32 = %x\n", Data32Or));

  MmioAndThenOr32 (
    DmiBar + R_CPU_DMI16_LCAP,
    (UINT32)~(B_CPU_DMI16_LCAP_EL0 | B_CPU_DMI16_LCAP_APMS | B_CPU_DMI16_LCAP_EL1),
    Data32Or | (V_CPU_DMI16_LCAP_EL0_512NS_1US << N_CPU_DMI16_LCAP_EL0)
    );

  //
  // Clear No Soft Reset
  //
  MmioAndThenOr32 (
    DmiBar + R_CPU_DMI16_PMCS,
    (UINT32)~B_CPU_DMI16_PMCS_NSR,
    0u
    );

  //
  // Enable Poisoned TLP Non-Fatal Advisory Error
  // Enable Link Speed Training Policy
  // Set EOI Forwarding Disable
  //
  MmioAndThenOr32 (
    DmiBar + R_CPU_DMI16_MPC2,
    (UINT32)~(B_CPU_DMI16_MPC2_PTNFAE | B_CPU_DMI16_MPC2_EOIFD),
    B_CPU_DMI16_MPC2_PTNFAE | B_CPU_DMI16_MPC2_EOIFD
    );

  //
  // Program Common Clock Exit Latency = 100b => 512ns to less than 1us
  // Enable MCTP Message Bus Number Check
  // Clear Invalid Receive Range Check
  // Set MCTP Support Enable
  //
  MmioAndThenOr32 (
    DmiBar + R_CPU_DMI16_MPC,
    (UINT32)~(B_CPU_DMI16_MPC_CCEL | B_CPU_DMI16_MPC_MMBNCE | B_CPU_DMI16_MPC_IRRCE | B_CPU_DMI16_MPC_MCTPSE),
    (V_CPU_DMI16_MPC_CCEL_512NS_1US << N_CPU_DMI16_MPC_CCEL) | B_CPU_DMI16_MPC_MMBNCE | B_CPU_DMI16_MPC_MCTPSE
    );

  //
  // Configure Transmit Datapath Flush Timer
  //  00b - Wait for 4 clocks prior to initiating rate or powerstate change.
  // Configure Transmit Configuration Change Wait Time
  //  00b - 128ns for GEN1 and GEN2 mode. 256ns for GEN3 mode.
  //
  MmioAndThenOr8 (
    DmiBar + R_CPU_DMI16_PHYCTL2,
    (UINT8) ~(B_CPU_DMI16_PHYCTL2_TDFT | B_CPU_DMI16_PHYCTL2_TXCFGCHGWAIT),
    0
    );

  //
  // Configure IOSF Sideband Interface Idle Counter
  //  00b - Wait for 32 idle clocks before allowing trunk clock gating.
  //
  MmioAndThenOr8 (
    DmiBar + R_CPU_DMI16_IOSFSBCS,
    (UINT8) ~(B_CPU_DMI16_IOSFSBCS_SIID),
    0
    );

  //
  // Enable Completion Time-Out Non-Fatal Advisory Error
  //
  MmioAndThenOr32 (
    DmiBar + R_CPU_DMI16_PCIEDBG,
    ~0u,
    B_CPU_DMI16_PCIEDBG_CTONFAE
    );

  //
  // Enable ASPM Optionality Compliance
  //
  MmioAndThenOr32 (
    DmiBar + R_CPU_DMI16_LCAP,
    (UINT32)~B_CPU_DMI16_LCAP_ASPMOC,
    B_CPU_DMI16_LCAP_ASPMOC
    );

  //
  // Program PME Timeout to 10ms (01b)
  //
  MmioAndThenOr32 (
    DmiBar + R_CPU_DMI16_PCIECFG2,
    (UINT32)~B_CPU_DMI16_PCIECFG2_PMET,
    (V_CPU_DMI16_PCIECFG2_PMET_10MS << N_CPU_DMI16_PCIECFG2_PMET)
    );

  //
  // Section 4.3 Standard PCIe / Downstream Port DMI / RCiEP for 2LM Specific <<<
  //
}

VOID
CpuDmiFiaFinalizeConfigurationAndLock (
  )
 {

    DEBUG ((DEBUG_INFO, "CpuDmiFiaFinalizeConfigurationAndLock \n"));
    //
    // Set PCR[FIA] + 0h bit [17, 16, 15] to [1, 1, 1]
    //
    CpuRegbarAndThenOr32 (
        CPU_SB_PID_DMI_FIA,
        R_PCH_FIA_PCR_CC,
        ~0u,
        B_PCH_FIA_PCR_CC_PTOCGE | B_PCH_FIA_PCR_CC_OSCDCGE | B_PCH_FIA_PCR_CC_SCPTCGE
        );
    DEBUG ((DEBUG_INFO, "Offset 0x00[CC] = 0x%x\n", CpuRegbarRead32 (CPU_SB_PID_DMI_FIA, R_PCH_FIA_PCR_CC)));

    //
    // Set PCR[FIA] + 20h bit [0, 1, 31] to [0, 0, 0]
    //
    CpuRegbarAndThenOr32 (
        CPU_SB_PID_DMI_FIA,
        R_PCH_FIA_PCR_PLLCTL,
        (UINT32)~(B_PCH_FIA_PCR_PLLCTL_CL0PLLFO | B_PCH_FIA_PCR_PLLCTL_CL1PLLFO | B_PCH_FIA_PCR_PLLCTL_PLLACBGD),
        0
        );
    DEBUG ((DEBUG_INFO, "Offset 0x20[PLLCTL] = 0x%x\n", CpuRegbarRead32 (CPU_SB_PID_DMI_FIA, R_PCH_FIA_PCR_PLLCTL)));

    //
    // Set PCR[FIA] + 40h bit [3] to [1]
    //
    CpuRegbarAndThenOr32 (
        CPU_SB_PID_DMI_FIA,
        R_PCH_FIA_PCR_PMC,
        ~0u,
        B_PCH_FIA_PCR_PMC_PRDPGE
        );
    DEBUG ((DEBUG_INFO, "Offset 0x40[PMC] = 0x%x\n", CpuRegbarRead32 (CPU_SB_PID_DMI_FIA, R_PCH_FIA_PCR_PMC)));

    //
    // Set PCR[FIA] + 48h bit [0] to [0]
    //
    CpuRegbarAndThenOr32 (
        CPU_SB_PID_DMI_FIA,
        R_PCH_FIA_PCR_PGCUC,
        (UINT32)~(B_PCH_FIA_PCR_PGCUC_ACC_CLKGATE_DISABLED),
        0
        );
    DEBUG ((DEBUG_INFO, "Offset 0x48[PGCUC] = 0x%x\n", CpuRegbarRead32 (CPU_SB_PID_DMI_FIA, R_PCH_FIA_PCR_PGCUC)));
}


/**
  Get max Dmi link speed supported

  @param[in] DmiBar          DmiBar
  @return    Max link speed
**/
UINT32
CpuDmiGetMaxLinkSpeed (
  IN  UINT32                       DmiBar
  )
{
  return MmioRead32 (DmiBar + R_CPU_DMI16_LCAP) & B_CPU_DMI16_LCAP_MLS;
}
/**
  Limit Link Speed Based on Policy

  @param[in] DmiBar                 DmiBar
**/
VOID
EFIAPI
DmiLimitLinkSpeed (
  IN  UINT32                       DmiBar,
  IN  CPU_DMI_PREMEM_CONFIG        *CpuDmiPreMemConfig
  )
{
  UINT32                             Data32Or;
  UINT32                             Data32And;
  //
  // Set speed capability in DmiBar
  //
  DEBUG ((DEBUG_INFO, "DmiLimitLinkSpeed \n"));
  Data32And = (~((UINT32)(B_CPU_DMI_MPC_PCIESD_MASK)));
  Data32Or = 0;
  DEBUG ((DEBUG_INFO, "DmiSpeed from policy = %x\n", CpuDmiPreMemConfig->DmiMaxLinkSpeed));
  switch (CpuDmiPreMemConfig->DmiMaxLinkSpeed) {
    case CpuDmiGen1:
      Data32Or |= (V_CPU_DMI_MPC_PCIESD_GEN1 << B_CPU_DMI_MPC_PCIESD_OFFSET);
      break;
    case CpuDmiGen2:
      Data32Or |= (V_CPU_DMI_MPC_PCIESD_GEN2 << B_CPU_DMI_MPC_PCIESD_OFFSET);
      break;
    case CpuDmiGen3:
      Data32Or |= (V_CPU_DMI_MPC_PCIESD_GEN3 << B_CPU_DMI_MPC_PCIESD_OFFSET);
      break;
    case CpuDmiAuto:
      break;
  }
  if (Data32Or != 0) {
    DEBUG ((DEBUG_INFO, "Dmi Link Speed is Limited to Gen%x based on policy!!!\n", CpuDmiPreMemConfig->DmiMaxLinkSpeed));
  }
  MmioAndThenOr32 (DmiBar + R_CPU_DMI16_MPC, Data32And, Data32Or);
}

/**
  Configure DMI clock gating.
  This function must be called before ASPM is enabled
**/
STATIC
VOID
ConfigureCpuDmiClockGating (
  IN  UINT32                       DmiBar
  )
{
  //
  // Enable Dynamic Clock Gating on ISM Active
  //
  MmioAndThenOr32 (
    DmiBar + R_CPU_DMI16_CCFG,
    ~0u,
    B_CPU_DMI16_CCFG_DCGEISMA
    );

  //
  // Enable below clock gating settings:
  // - Partition/Trunk Oscillator Clock Gate Enable
  // - Link CLKREQ Enable
  // - Backbone CLKREQ Enable
  // - Shared Resource Dynamic Backbone Clock Gate Enable
  // - Root Port Dynamic Link Clock Gate Enable
  // - Root Port Dynamic Backbone Clock Gate Enable
  //
  MmioAndThenOr8 (
    DmiBar + R_CPU_DMI16_RPDCGEN,
    (UINT8)~0,
    (B_CPU_DMI16_PTOCGE | B_CPU_DMI16_LCLKREQEN | B_CPU_DMI16_BBCLKREQEN |
    B_CPU_DMI16_SRDBCGEN | B_CPU_DMI16_RPDLCGEN | B_CPU_DMI16_RPDBCGEN)
    );

  //
  // Enable Side Clock Partition/Trunk Clock Gating
  // Set IOSF Sideband Interface Idle Counter to 00b - Wait for
  // 32 idle clocks before allowing trunk clock gating.
  //
  MmioAndThenOr8 (
    DmiBar + R_CPU_DMI16_IOSFSBCS,
    (UINT8)~B_CPU_DMI16_IOSFSBCS_SIID,
    B_CPU_DMI16_IOSFSBCS_SCPTCGE
    );

  //
  // Enable Sideband Endpoint Oscillator/Side Clock Gating
  //
  MmioAndThenOr16 (
    DmiBar + R_CPU_DMI16_RPPGEN,
    (UINT8)~0,
    B_CPU_DMI16_RPPGEN_SEOSCGE
    );

  //
  // Program OSC Clock Gate Hysteresis
  //
  MmioAndThenOr32 (
    DmiBar + R_CPU_DMI16_PME_CTL3,
    (UINT8)~B_CPU_DMI16_PME_CTL3_OSCCGH,
    (UINT32) (V_CPU_DMI16_PME_CTL3_OSCCGH_1US << B_CPU_DMI16_PME_CTL3_OSCCGH_OFFSET)
    );
}

/**
  Configure DMI link power management.
  This function must be called before ASPM is enabled
**/
STATIC
VOID
ConfigureCpuDmiLinkPowerManagement (
  IN  UINT32                       DmiBar
  )
{
  //
  // Configure Gen1 and Gen2 Active State L0s Preparation Latency - time that link layer has to
  // indicate IDLE before the link initialization and control logic enters L0s
  // Set it to 0x14 clocks
  //
  MmioAndThenOr32 (
    DmiBar + R_CPU_DMI16_PCIEL0SC,
    (UINT32)~(B_CPU_DMI16_PCIEL0SC_G2ASL0SPL | B_CPU_DMI16_PCIEL0SC_G1ASL0SPL),
    ((0x14 << N_CPU_DMI16_PCIEL0SC_G2ASL0SPL) |
    (0x14 << N_CPU_DMI16_PCIEL0SC_G1ASL0SPL))
    );

  //
  // Section 4.3 Standard PCIe / Downstream Port DMI / RCiEP for 2LM Specific >>>
  //

  //
  // Configure Gen1 and Gen2 Common Clock N_FTS
  // - 0x2C for GEN1
  // - 0x5B for GEN2
  // Configure Gen1 and Gen2 Unique Clock N_FTS
  // - 0x3F for GEN1
  // - 0x7E for GEN2
  //
  MmioAndThenOr32 (
    DmiBar + R_CPU_DMI16_PCIENFTS,
    (UINT32)~(B_CPU_DMI16_PCIENFTS_G2CCNFTS | B_CPU_DMI16_PCIENFTS_G1CCNFTS | B_CPU_DMI16_PCIENFTS_G2UCNFTS | B_CPU_DMI16_PCIENFTS_G1UCNFTS),
    ((0x5B << N_CPU_DMI16_PCIENFTS_G2CCNFTS) |
    (0x2C << N_CPU_DMI16_PCIENFTS_G1CCNFTS) |
    (0x7E << N_CPU_DMI16_PCIENFTS_G2UCNFTS) |
    (0x3F << N_CPU_DMI16_PCIENFTS_G1UCNFTS))
    );

  //
  // Configure Gen3 Unique Clock N_FTS to 0x40
  // Configure Gen3 Common Clock N_FTS to 0x2c
  //
  MmioAndThenOr32 (
    DmiBar + R_CPU_DMI16_G3L0SCTL,
    (UINT32)~(B_CPU_DMI16_G3L0SCTL_G3UCNFTS | B_CPU_DMI16_G3L0SCTL_G3CCNFTS),
    ((0x40 << N_CPU_DMI16_G3L0SCTL_G3UCNFTS) |
    (0x2c << N_CPU_DMI16_G3L0SCTL_G3CCNFTS))
    );

  //
  // Configure L0s Exit Latency to 0x4 (512ns - < 1us) - CPU DPI DMI Value
  // Configure L1s Exit Latency to 0x4 (8us - 16us) - CPU DPI DMI Value
  //
  MmioAndThenOr32 (
    DmiBar + R_CPU_DMI_PCR_LCAP,
    (UINT32) ~(B_CPU_DMI_PCR_LCAP_EL1 | B_CPU_DMI_PCR_LCAP_EL0),
    ((V_CPU_DMI_PCR_LCAP_EL1_8US_16US << N_CPU_DMI_PCR_LCAP_EL1) |
    (V_CPU_DMI_PCR_LCAP_EL0_256NS_512NS << N_CPU_DMI_PCR_LCAP_EL0))
    );
}

/**
  Configure DMI Squelch Power Management.
**/
STATIC
VOID
ConfigureCpuDmiSquelchPowerManagement (
  IN  UINT32                       DmiBar
  )
{
  //
  // Enable Squelch propagation control
  // Program Link Clock Domain Squelch Exit Debounce Timers to 0ns
  //
  MmioAndThenOr32 (
    DmiBar + R_CPU_DMI16_PCIEDBG,
    (UINT32)~B_CPU_DMI16_PCIEDBG_LGCLKSQEXITDBTIMERS,
    (B_CPU_DMI16_PCIEDBG_SPCE | (V_CPU_DMI16_PCIEDBG_LGCLKSQEXITDBTIMERS_0ns << N_CPU_DMI16_PCIEDBG_LGCLKSQEXITDBTIMERS))
    );

  //
  // Disable Squelch polling
  //
  MmioAndThenOr32 (
    DmiBar + R_CPU_DMI16_PWRCTL,
    (UINT32)~(B_CPU_DMI16_PWRCTL_RPL1SQPOL | B_CPU_DMI16_PWRCTL_RPDTSQPOL),
    0u
    );

  //
  // Disable Squelch Off in L0
  //
  MmioAndThenOr32 (
    DmiBar + R_CPU_DMI16_PWRCTL,
    (UINT32)~(B_CPU_DMI16_PCIEDBG_SQOL0),
    0u
    );

  // Enable Low Bandwidth Squelch Settling Timer
  // Set this before enabling any of the squelch power management
  //
  MmioAndThenOr32 (
    DmiBar + R_CPU_DMI16_PCIECFG2,
    (UINT32)~0u,
    B_CPU_DMI16_PCIECFG2_LBWSSTE
    );

  //
  // Program Un-Squelch Sampling Period
  //
  MmioAndThenOr32 (
    DmiBar + R_CPU_DMI16_PCIECFG2,
    (UINT32)~B_CPU_DMI16_PCIEDBG_USSP,
    (V_CPU_DMI16_PCIEDBG_USSP_32ns << N_CPU_DMI16_PCIEDBG_USSP)
    );

  //
  // Program Disabled, Detect, L23_Rdy State,Un-Configured Lane and Down - Configured Lane Squelch Disable = 1
  //
  MmioAndThenOr32 (
    DmiBar + R_CPU_DMI16_PCIEPMECTL,
    (UINT32)~B_CPU_DMI16_PCIEPMECTL_DLSULDLSD,
    (B_CPU_DMI16_PCIEPMECTL_DLSULPPGE | B_CPU_DMI16_PCIEPMECTL_DLSULDLSD)
    );
}

/**
  Configure DMI PLL Shutdown.
**/
STATIC
VOID
ConfigureCpuDmiPllShutdown (
  IN  UINT32                       DmiBar
  )
{
  //
  // Enable PCI Express GEN2 and GEN3 PLL Off
  // If enabled and no devices require the use of the PXP PLL clock outputs,
  // the PXP PLL can be shutdown.
  //
  MmioAndThenOr8 (
    DmiBar + R_CPU_DMI16_PHYCTL2,
    (UINT8)~0,
    (B_CPU_DMI16_PHYCTL2_PXPG3PLLOFFEN | B_CPU_DMI16_PHYCTL2_PXPG2PLLOFFEN)
    );
}

/**
  Configure DMI Power Gating.
  This needs to done before enabling ASPM L1
**/
STATIC
VOID
ConfigureCpuDmiPowerGating (
  IN  UINT32                       DmiBar
  )
{
  UINT32                        Data32And;
  UINT32                        Data32Or;

  //
  // Set Wake PLL On Detect mod-PHY Power Gating Exit Policy
  //
  MmioAndThenOr32 (
    DmiBar + R_CPU_DMI16_PWRCTL,
    (UINT32)~0u,
    B_CPU_DMI16_PWRCTL_WPDMPGEP
    );


  //
  // L1LOW LTR threshold latency value
  //
  MmioAndThenOr32 (
    DmiBar + R_CPU_DMI16_PCIEPMECTL,
    (UINT32) ~B_CPU_DMI16_PCIEPMECTL_L1LTRTLV_MASK,
    (V_CPU_DMI16_PCIEPMECTL_L1LTRTLV << B_CPU_DMI16_PCIEPMECTL_L1LTRTLV_OFFSET)
    );
  DEBUG ((DEBUG_INFO, "R_CPU_DMI16_PCIEPMECTL after write = %x\n", MmioRead32 (DmiBar + R_CPU_DMI16_PCIEPMECTL)));

  //
  // Set Function Disable PHY Power Gating Enable
  //
  MmioAndThenOr32 (
    DmiBar + R_CPU_DMI16_PCIEPMECTL,
    (UINT32)~0u,
    B_CPU_DMI16_PCIEPMECTL_FDPPGE
    );

  //
  // Enable PHY Common Lane Power Gating
  // Enable Function Disable Controller Power Gating
  // Enable Detect State Controller Power Gating
  // Enable L23_Rdy State Controller Power Gating
  // Enable Disabled State Controller Power Gating
  // Enable L1 State Controller Power Gating
  // Enable Power Gating Entry Hysteresis
  // Enable Power Gating Exit Hysteresis
  //
  Data32And = ~(UINT32)(B_CPU_DMI16_PCIEPMECTL2_CPGENH_MASK |B_CPU_DMI16_PCIEPMECTL2_CPGEXH_MASK);
  Data32Or = B_CPU_DMI16_PCIEPMECTL2_PHYCLPGE | B_CPU_DMI16_PCIEPMECTL2_FDCPGE | B_CPU_DMI16_PCIEPMECTL2_DETSCPGE | B_CPU_DMI16_PCIEPMECTL2_L23RDYSCPGE |
             B_CPU_DMI16_PCIEPMECTL2_DISSCPGE | B_CPU_DMI16_PCIEPMECTL2_L1SCPGE | (V_CPU_DMI16_PCIEPMECTL2_CPGENH << B_CPU_DMI16_PCIEPMECTL2_CPGENH_OFFSET) |
            (V_CPU_DMI16_PCIEPMECTL2_CPGEXH << B_CPU_DMI16_PCIEPMECTL2_CPGEXH_OFFSET);
  MmioAndThenOr32 (
    DmiBar + R_CPU_DMI16_PCIEPMECTL2,
    Data32And,
    Data32Or
    );
  DEBUG ((DEBUG_INFO, "R_CPU_DMI16_PCIEPMECTL2 after write = %x\n", MmioRead32 (DmiBar + R_CPU_DMI16_PCIEPMECTL2)));

  //
  // Set PMC Request Enable
  // Clear Sleep Enable
  // Clear Hardware Autonomous Enable
  //
  MmioAndThenOr32 (
    DmiBar + R_CPU_DMI16_PCE,
    (UINT32)~(B_CPU_DMI16_PCE_PMCRE | B_CPU_DMI16_PCE_SE | B_CPU_DMI16_PCE_HAE),
    B_CPU_DMI16_PCE_PMCRE
    );

  //
  // Enable L1.PG Auto Power Gate Enable
  // Set PM Request Controller Power Gating Exit Hysteresis = 5 us
  //
  MmioAndThenOr32 (
    DmiBar + R_CPU_DMI16_PME_CTL3,
    (UINT32)~(B_CPU_DMI16_PME_CTL3_L1PGAUTOPGEN | B_CPU_DMI16_PME_CTL3_PMREQCPGEXH),
    B_CPU_DMI16_PME_CTL3_L1PGAUTOPGEN | V_CPU_DMI16_PME_CTL3_PMREQCPGEXH
    );

  //
  // Program Bank Enable Pulse Width
  // Program Restore Enable Pulse Width
  // Program Entry and Exit Hysteresis
  //
  MmioAndThenOr32 (
    DmiBar + R_CPU_DMI16_HWSNR,
    (UINT32)~(B_CPU_DMI16_HWSNR_BEPW | B_CPU_DMI16_HWSNR_REPW | B_CPU_DMI16_HWSNR_EEH),
    V_CPU_DMI16_HWSNR_BEPW_8CLKS | (V_CPU_DMI16_HWSNR_REPW_2CLKS << N_CPU_DMI16_HWSNR_REPW) | (V_CPU_DMI16_HWSNR_EEH_16CLKS << N_CPU_DMI16_HWSNR_EEH)
    );

  //
  // Program PM_REQ Block Response Time
  //
  MmioAndThenOr32 (
    DmiBar + R_CPU_DMI16_PGCTRL,
    (UINT32)~(B_CPU_DMI16_PGCTRL_PMREQBLKRSPT),
    V_CPU_DMI16_PGCTRL_PMREQBLKRSPT_25US
    );

  //
  // Program PM_REQ Block Response Time
  // Program PM_REQ_Clock Wake Control = 0 (Disable Link Clock Domain, Prim Clock Domain and Side Clock Domain)
  //
  MmioAndThenOr32 (
    DmiBar + R_CPU_DMI16_HWSNR367,
    (UINT32)~(B_CPU_DMI16_HWSNR367_PMREQBLKPGRSPT | B_CPU_DMI16_HWSNR367_PMREQCWC),
    ((V_CPU_DMI16_HWSNR367_PMREQBLKPGRSPT_5US << N_CPU_DMI16_HWSNR367_PMREQBLKPGRSPT) | (V_CPU_DMI16_HWSNR367_PMREQCWC << N_CPU_DMI16_HWSNR367_PMREQCWC))
    );
   DEBUG ((DEBUG_INFO, "R_CPU_DMI16_HWSNR367 after write = %x\n", MmioRead32 (DmiBar + R_CPU_DMI16_HWSNR367)));

  //
  // Program L1 Power Gating LTR Threshold Latency Scale Value = 1024ns
  // Program L1 Power Gating LTR Threshold Latency Value
  // Enable L1 Power Gating LTR
  //
  MmioAndThenOr32 (
    DmiBar + R_CPU_DMI16_HWSNR368,
    (UINT32)~(B_CPU_DMI16_HWSNR368_L1PGLTRTLSV | B_CPU_DMI16_HWSNR368_L1PGLTRTLV | B_CPU_DMI16_HWSNR368_L1PGLTREN),
    (V_CPU_DMI16_HWSNR367_L1PGLTRTLSV_1024NS << N_CPU_DMI16_HWSNR368_L1PGLTRTLSV) | (V_CPU_DMI16_HWSNR367_L1PGLTRTLV << N_CPU_DMI16_HWSNR368_L1PGLTRTLV) | B_CPU_DMI16_HWSNR368_L1PGLTREN
    );

  //
  // Clear Chassis PMC Save and Restore Enable
  //
  MmioAndThenOr32 (
    DmiBar + R_CPU_DMI16_PCIEPMECTL2,
    (UINT32)~B_CPU_DMI16_PCIEPMECTL2_CPMCSRE,
    0u
    );

  //
  // Program CPG Wake Control = 8 us
  //
  MmioAndThenOr32 (
    DmiBar + R_CPU_DMI16_ACRG3,
    (UINT32)~B_CPU_DMI16_ACRG3_CPGWAKECTRL,
    (V_CPU_DMI16_ACRG3_CPGWAKECTRL_8US << N_CPU_DMI16_ACRG3_CPGWAKECTRL)
    );
}


/**
  This function configures ASPM on DMI

  @param[in] SiPolicyPpi The CPU Policy PPI instance
**/
STATIC
VOID
ConfigureCpuDmiAspm (
  IN  UINT32                       DmiBar,
  IN  CPU_DMI_PREMEM_CONFIG        *CpuDmiPreMemConfig
  )
{
  UINT16                        Data16And;
  UINT16                        Data16Or;
  UINT32                        Data32And;
  UINT32                        Data32Or;
  DMI_ASPM                      DmiAspmCtrl;

  //
  // Enable DMI ASPM
  //
  DmiAspmCtrl = CpuDmiPreMemConfig->DmiAspmCtrl;

  Data32And = (UINT32)~B_CPU_DMI16_G3L0SCTL_G3ASL0SPL;
  Data32Or  = (UINT32)(0x28 << N_CPU_DMI16_G3L0SCTL_G3ASL0SPL);
  //
  // Program Gen3 Active State L0s Preparation Latency
  //
  MmioAndThenOr32 (
    DmiBar + R_CPU_DMI16_G3L0SCTL,
    Data32And,
    Data32Or
    );

  Data32And = (UINT32)~B_CPU_DMI16_PCIEL0SC_G2ASL0SPL;
  Data32Or  = (UINT32)(0x14 << N_CPU_DMI16_PCIEL0SC_G2ASL0SPL);
  //
  // Program Gen2 Active State L0s Preparation Latency
  //
  MmioAndThenOr32 (
    DmiBar + R_CPU_DMI16_PCIEL0SC,
    Data32And,
    Data32Or
    );

  Data32And = (UINT32)~B_CPU_DMI16_PCIEL0SC_G1ASL0SPL;
  Data32Or  = (UINT32)(0x14 << N_CPU_DMI16_PCIEL0SC_G1ASL0SPL);
  //
  // Program Gen1 Active State L0s Preparation Latency
  //
  MmioAndThenOr32 (
    DmiBar + R_CPU_DMI16_PCIEL0SC,
    Data32And,
    Data32Or
    );

  Data16And = (UINT16)~B_CPU_DMI16_LCTL_ASPM;

  if ((DmiAspmCtrl == DmiAspmAutoConfig) || (DmiAspmCtrl == DmiAspmL0sL1)) {
    //
    // Enable L0s/L1 on DMI
    //
    Data16Or = V_CPU_DMI16_LCTL_ASPM_L0SL1;
  } else if (DmiAspmCtrl == DmiAspmL0s) {
    //
    // Enable L0s Entry only
    //
    Data16Or = V_CPU_DMI16_LCTL_ASPM_L0S;

  } else if (DmiAspmCtrl == DmiAspmL1) {
    //
    // Enable L1 Entry only
    //
    Data16Or = V_CPU_DMI16_LCTL_ASPM_L1;
  } else {
    //
    // ASPM Disabled
    //
    Data16Or  = V_CPU_DMI16_LCTL_ASPM_DIS;
  }

  //
  // Configue DMI ASPM
  //
  MmioAndThenOr16 (
    DmiBar + R_CPU_DMI16_LCTL,
    Data16And,
    Data16Or
    );

}
/**
  Configure DMI Coalescing
  @param[in] DmiBar         Dmi Base Address
**/
VOID
ConfigureCpuDmiCoalescing (
  IN  UINT32       DmiBar
  )
{
  UINT32     Data32Or;
  UINT32     Data32And;

  //
  // Section 4.25 DownStream Port DMI specific in SIP16 Converged PCIe BWG 0.93 >>>
  //
  Data32And = ~(UINT32)(B_PCIE_COCTL_CT_MASK | B_PCIE_COCTL_ROAOP | B_PCIE_COCTL_PCLM_MASK | B_PCIE_COCTL_NPCLM_MASK);
  Data32Or = B_PCIE_COCTL_PWCE | B_PCIE_COCTL_DDCE | B_PCIE_COCTL_ROAOP | (3 << B_PCIE_COCTL_CT_OFFSET) |
               B_PCIE_COCTL_CTE | (0x3 << B_PCIE_COCTL_PCLM_OFFSET) | (0x3 << B_PCIE_COCTL_NPCLM_OFFSET);
  MmioAndThenOr32 (
    DmiBar + R_PCIE_COCTL,
    Data32And,
    Data32Or
    );
  DEBUG((DEBUG_INFO, "COCTL DMI= %x\n", MmioRead32 (DmiBar + R_PCIE_COCTL)));

  ///
  /// Chain Bit Generation Mode(CBGM)
  ///
  Data32And = ~(UINT32)B_PCIE_ACRG3_CBGM;
  Data32Or  = B_PCIE_ACRG3_CBGM;
  MmioAndThenOr32 (
    DmiBar + R_PCIE_ACRG3,
    Data32And,
    Data32Or
    );
  DEBUG ((DEBUG_INFO, "R_PCIE_ACRG3 DMI= %x\n", MmioRead32 (DmiBar + R_PCIE_ACRG3)));

}
/**
  Configure DMI Atomics
  @param[in] DmiBar         Dmi Base Address
**/
VOID
ConfigureCpuDmiAtomics (
  IN  UINT32       DmiBar
  )
{
  UINT32     Data32And;

  //
  // Section 4.28 DownStream Port DMI specific in SIP16 Converged PCIe BWG 0.93 >>>
  //

  Data32And = (UINT32)~(B_PCIE_DCAP2_AC64BS | B_PCIE_DCAP2_AC32BS | B_PCIE_DCAP2_ARS);
  MmioAnd32(DmiBar + R_PCIE_DCAP2, Data32And);
  DEBUG((DEBUG_INFO, "R_PCIE_DCAP2 DMI after programming Atomics and 10-Bit Requester Supported = %x\n", MmioRead32 (DmiBar + R_PCIE_DCAP2)));


  Data32And = (UINT32)~(B_PCIE_DCTL2_AEB | B_PCIE_DCTL2_ARE);
  MmioAnd32(DmiBar + R_PCIE_DCTL2, Data32And);
  DEBUG ((DEBUG_INFO, "R_PCIE_DCTL2 DMI after programming Atomics = %x\n", MmioRead32 (DmiBar + R_PCIE_DCTL2)));

}

/**
  Configure DMI Server Error Reporting Mode

 @param[in] DmiBar         Dmi Base Address
**/
VOID
ConfigureCpuDmiSerm (
  IN  UINT32       DmiBar
  )
{
  UINT32     Data32And;

  //
  // Section 4.19 DownStream Port DMI specific in SIP16 Converged PCIe BWG 0.93 >>>
  //
  Data32And = (UINT32)~B_PCIE_STRPFUSECFG_SERM;
  MmioAnd32 (DmiBar + R_PCIE_STRPFUSECFG, Data32And);
  DEBUG ((DEBUG_INFO, "R_PCIE_STRPFUSECFG DMI after programming SERM = %x\n", MmioRead32 (DmiBar + R_PCIE_STRPFUSECFG)));
}

/**
  Configure DMI SMI/SCI

 @param[in] DmiBar         Dmi Base Address
**/
VOID
ConfigureCpuDmiSmiSci (
  IN  UINT32       DmiBar
  )
{
  UINT32     Data32And;

  //
  // Section 4.17 DownStream Port DMI specific in SIP16 Converged PCIe BWG 0.93 >>>
  //

  Data32And = (UINT32)~B_PCIE_MPC_PMME;
  MmioAnd32 (DmiBar + R_PCIE_MPC, Data32And);
  DEBUG ((DEBUG_INFO, "R_PCIE_MPC DMI = %x\n", MmioRead32 (DmiBar + R_PCIE_MPC)));

  Data32And = (UINT32)~(B_PCIE_EQCFG1_LERSMIE | B_PCIE_EQCFG1_LERSCIE);
  MmioAnd32 (DmiBar + R_CPU_DMI16_EQCFG1, Data32And);
  DEBUG ((DEBUG_INFO, "R_PCIE_EQCFG1 DMI = %x\n", MmioRead32 (DmiBar + R_CPU_DMI16_EQCFG1)));
}

/**
  Configure DMI Replay Timer
  @param[in] DmiBar         Dmi Base Address
**/
VOID
ConfigureCpuDmiReplayTimer (
  IN  UINT32          DmiBar
  )
{
  UINT32    Data32Or;
  UINT32    Data32And;

  //
  // Section 4.15 DownStream Port DMI specific in SIP16 Converged PCIe BWG 0.93 >>>
  //
  Data32And = ~(UINT32)(B_PCIE_PCIERTP1_G2X1_MASK | B_PCIE_PCIERTP1_G2X2_MASK | B_PCIE_PCIERTP1_G2X4_MASK |
                        B_PCIE_PCIERTP1_G1X1_MASK | B_PCIE_PCIERTP1_G1X2_MASK | B_PCIE_PCIERTP1_G1X4_MASK);
  Data32Or = (0xB << B_PCIE_PCIERTP1_G2X1_OFFSET) | (0x7 << B_PCIE_PCIERTP1_G2X2_OFFSET) | (0x5 << B_PCIE_PCIERTP1_G2X4_OFFSET) |
             (0xF << B_PCIE_PCIERTP1_G1X1_OFFSET) | (0xA << B_PCIE_PCIERTP1_G1X2_OFFSET) | (0x7);
  MmioAndThenOr32 (
    DmiBar + R_PCIE_PCIERTP1,
    Data32And,
    Data32Or
    );
  DEBUG((DEBUG_INFO, "PCIERTP1 DMI= %x\n", MmioRead32 (DmiBar + R_PCIE_PCIERTP1)));

  Data32And = (UINT32)~(B_PCIE_PCIERTP2_G3X1_MASK | B_PCIE_PCIERTP2_G3X2_MASK | B_PCIE_PCIERTP2_G3X4_MASK);
  Data32Or = (0xC << B_PCIE_PCIERTP2_G3X1_OFFSET) | (0x9 << B_PCIE_PCIERTP2_G3X2_OFFSET) | (0x7 << B_PCIE_PCIERTP2_G3X4_OFFSET);
  MmioAndThenOr32 (
    DmiBar + R_PCIE_PCIERTP2,
    Data32And,
    Data32Or
    );
  DEBUG((DEBUG_INFO, "PCIERTP2 DMI= %x\n", MmioRead32 (DmiBar + R_PCIE_PCIERTP2)));

  Data32And = (UINT32)~(B_PCIE_PCIERTP3_G3X8_MASK | B_PCIE_PCIERTP3_G2X8_MASK | B_PCIE_PCIERTP3_G1X8_MASK);
  Data32Or = (0x9 << B_PCIE_PCIERTP3_G3X8_OFFSET) | (0x7 << B_PCIE_PCIERTP3_G2X8_OFFSET) | (0x5 << B_PCIE_PCIERTP3_G1X8_OFFSET);
  MmioAndThenOr32 (
    DmiBar + R_PCIE_PCIERTP3,
    Data32And,
    Data32Or
    );
  DEBUG((DEBUG_INFO, "PCIERTP3 DMI= %x\n", MmioRead32 (DmiBar + R_PCIE_PCIERTP3)));

  Data32And = (UINT32) ~B_PCIE_ACGR3S2_SRT;
  MmioAnd32 (DmiBar + R_PCIE_ACGR3S2, Data32And);
  DEBUG ((DEBUG_INFO, "R_PCIE_ACGR3S2 DMI after setting = %x\n", MmioRead32 (DmiBar + R_PCIE_ACGR3S2)));

}

/**
  Configure DMI Design Specific Configuration
  @param[in] DmiBar         Dmi Base Address
**/
VOID
ConfigureCpuDmiDesignSpecificConfiguration (
  IN  UINT32       DmiBar
  )
{
  UINT32    Data32And;
  UINT32    Data32Or;

  //
  // Section 4.16 DownStream Port DMI specific in SIP16 Converged PCIe BWG 0.93 >>>
  //

  Data32And = (UINT32) ~B_PCIE_PCIECFG2_RLLG3R;
  MmioAnd32 (DmiBar + R_PCIE_PCIECFG2, Data32And);
  DEBUG ((DEBUG_INFO, "R_PCIE_PCIECFG2 DMI after setting = %x\n", MmioRead32 (DmiBar + R_PCIE_PCIECFG2)));

  Data32And = (UINT32)~B_PCIE_CCFG_UNRD_MASK;
  MmioAnd32 (DmiBar + R_PCIE_CCFG, Data32And);
  DEBUG ((DEBUG_INFO, "R_PCIE_CCFG after setting DMI = %x\n", MmioRead32 (DmiBar + R_PCIE_CCFG)));

  Data32And = (UINT32)~(B_PCIE_DC_DCT0C |B_PCIE_DC_COM);
  MmioAnd32 (DmiBar + R_PCIE_DC, Data32And);
  DEBUG ((DEBUG_INFO, "R_PCIE_DC after setting DMI = %x\n", MmioRead32 (DmiBar + R_PCIE_DC)));

  Data32And = (UINT32)~(B_PCIE_PCIEALC_SSRLD | B_PCIE_PCIEALC_SSRRS);
  MmioAnd32 (DmiBar + R_PCIE_PCIEALC, Data32And);

  //cpu.dmi.cfg.dc.dct1c = 0x1
  MmioOr32 (DmiBar + R_PCIE_DC, B_PCIE_DC_DCT1C);

  //cpu.dmi.cfg.ipcs.imps = 0x7
  Data32And = (UINT32)~(B_PCIE_IPCS_IMPS_MASK);
  Data32Or = (V_PCIE_IPCS_IMPS << B_PCIE_IPCS_IMPS_OFFSET);
  MmioAndThenOr32 (DmiBar + R_PCIE_IPCS, Data32And, Data32Or);

  //cpu.dmi.cfg.ccfg.upsd = 0x0
  Data32And = (UINT32) ~(B_PCIE_CCFG_UPSD);
  MmioAnd32 (DmiBar + R_PCIE_CCFG, Data32And);


  Data32And = (UINT32)~(B_PCIE_ACRG3_RRXDME | B_PCIE_ACRG3_ADESKEW_DIS);
  MmioAnd32 (DmiBar + R_PCIE_ACRG3, Data32And);
  DEBUG ((DEBUG_INFO, "R_PCIE_ACRG3 after setting DMI = %x\n", MmioRead32 (DmiBar + R_PCIE_ACRG3)));

  Data32And = (UINT32)~B_PCIE_PWRCTL_DARECE;
  MmioAnd32 (DmiBar + R_PCIE_PWRCTL, Data32And);
  DEBUG ((DEBUG_INFO, "R_PCIE_PWRCTL after setting DMI = %x\n", MmioRead32 (DmiBar + R_PCIE_PWRCTL)));

  Data32And = (UINT32)~B_PCIE_PCIEDBG_LDSWQRP;
  MmioAnd32 (DmiBar + R_PCIE_PCIEDBG, Data32And);
  DEBUG ((DEBUG_INFO, "R_PCIE_PCIEDBG after setting DMI = %x\n", MmioRead32 (DmiBar + R_PCIE_PCIEDBG)));

  Data32And = (UINT32)~(B_PCIE_ADVMCTRL_ACCRM | B_PCIE_ADVMCTRL_RXL0DC | B_PCIE_ADVMCTRL_RRLLCL | B_PCIE_ADVMCTRL_RLLG12R);
  Data32Or = (UINT32)(B_PCIE_ADVMCTRL_INRXL0CTRL | B_PCIE_ADVMCTRL_EIOSDISDS | B_PCIE_ADVMCTRL_EIOSMASKRX | B_PCIE_ADVMCTRL_G3STFER);
  MmioAndThenOr32 (
    DmiBar + R_PCIE_ADVMCTRL,
    Data32And,
    Data32Or
    );
  DEBUG ((DEBUG_INFO, "R_PCIE_ADVMCTRL after setting DMI = %x\n", MmioRead32 (DmiBar + R_PCIE_ADVMCTRL)));

}

/**
  Configure DMI Enabling
  @param[in] DmiBar         Dmi Base Address
**/
VOID
ConfigureCpuDmiEnabling (
  IN  UINT32       DmiBar
  )
{
  UINT32    Data32And;

  //
  // Section 4.26 DownStream Port DMI specific in SIP16 Converged PCIe BWG 0.93 >>>
  //
  Data32And = (UINT32)~(B_PCIE_PCIEALC_BLKPAPC | B_PCIE_PCIEALC_BLKDQDA | B_PCIE_PCIEALC_BLKDQDASD);
  MmioAnd32 (DmiBar + R_PCIE_PCIEALC, Data32And);
  DEBUG ((DEBUG_INFO, "R_PCIE_PCIEALC after setting DMI = %x\n", MmioRead32 (DmiBar + R_PCIE_PCIEALC)));
}

/**
  Configure DMI MultiVC support
  @param[in] DmiBar         Dmi Base Address
**/
VOID
ConfigureCpuDmiMultiVC (
  IN  UINT32                       DmiBar,
  IN  CPU_DMI_PREMEM_CONFIG        *CpuDmiPreMemConfig
  )
{
  UINT32    Data32Or;
  UINT32    Data32And;

  //
  // Section 4.31 DownStream Port DMI specific in SIP16 Converged PCIe BWG 0.93 >>>
  //
  Data32And = (UINT32)~B_PCIE_PVCCR1_EVCC_MASK;
  Data32Or = (UINT32)(V_PCIE_PVCCR1_EVCC_3_VC << B_PCIE_PVCCR1_EVCC_OFFSET);
  MmioAndThenOr32 (
    DmiBar + R_PCIE_PVCCR1,
    Data32And,
    Data32Or
    );
  DEBUG ((DEBUG_INFO, "R_PCIE_PVCCR1 after setting DMI = %x\n", MmioRead32 (DmiBar + R_PCIE_PVCCR1)));

  Data32And = (UINT32)~B_PCIE_V0VCRC_MTS_MASK;
  MmioAnd32 (DmiBar + R_PCIE_V0VCRC, Data32And);
  DEBUG ((DEBUG_INFO, "R_PCIE_V0VCRC after setting DMI = %x\n", MmioRead32 (DmiBar + R_PCIE_V0VCRC)));


  Data32And = (UINT32)~(B_PCIE_VC0CTL_TVM_MASK | V_CPU_DMI_V0CTL_ETVM_MASK);
  MmioAnd32 (DmiBar + R_PCIE_VC0CTL, Data32And);
  DEBUG ((DEBUG_INFO, "R_PCIE_VC0CTL after setting DMI = %x\n", MmioRead32 (DmiBar + R_PCIE_VC0CTL)));

  Data32And = (UINT32)~(B_PCIE_VC1CTL_TVM_MASK | B_PCIE_VC1CTL_ID_MASK | B_PCIE_VC1CTL_EN_OFFSET);
  Data32Or = (UINT32)((B_PCIE_VC1CTL_TVM | V_PCIE_VC1CTL_ID_ONE << B_PCIE_VC1CTL_ID_OFFSET)
      | B_PCIE_VC1CTL_EN);

  Data32Or = (UINT32) (Data32Or & ~(B_PCIE_VC1CTL_EN));
  MmioAndThenOr32 (
    DmiBar + R_PCIE_VC1CTL,
    Data32And,
    Data32Or
    );
  DEBUG ((DEBUG_INFO, "R_PCIE_VC1CTL after setting DMI = %x\n", MmioRead32 (DmiBar + R_PCIE_VC1CTL)));

  Data32And = (UINT32)~(B_CPU_DMI16_VMCTL_TVM_MASK | B_CPU_DMI16_VMCTL_ID_MASK | B_CPU_DMI16_VMCTL_EN_MASK);
  Data32Or  = (UINT32)(B_CPU_DMI16_VMCTL_TVM | (V_CPU_DMI16_VMCTL_ID_TWO << B_CPU_DMI16_VMCTL_ID_OFFSET) | B_CPU_DMI16_VMCTL_EN);
  Data32Or = (UINT32) (Data32Or & ~(B_CPU_DMI16_VMCTL_EN));

  MmioAndThenOr32 (
    DmiBar + R_CPU_DMI16_VMCTL,
    Data32And,
    Data32Or
    );
  DEBUG ((DEBUG_INFO, "DmiBar + R_CPU_DMI16_VMCTL = %x \n", MmioRead32 (DmiBar + R_CPU_DMI16_VMCTL)));
}

/*
  Program registers for Elastic Buffer

  @param[in] DmiBar                 Cpu DmiBar
*/
VOID
EFIAPI
CpuDmiElasticBufferProgramming (
  IN  UINT32       DmiBar
  )
{
  UINT32  Data32Or;
  UINT32  Data32And;
  DEBUG ((DEBUG_INFO, "Cpu Dmi ElasticBufferProgramming Start"));

  Data32And = (UINT32)~(B_CPU_DMI_ACGR3S2_G1EBM | B_CPU_DMI_ACGR3S2_G2EBM | B_CPU_DMI_ACGR3S2_G3EBM | B_CPU_DMI_ACGR3S2_G4EBM);
  Data32Or = (UINT32)(B_CPU_DMI_ACGR3S2_G1EBM | B_CPU_DMI_ACGR3S2_G2EBM);

  MmioAndThenOr32 (DmiBar + R_CPU_DMI_ACGR3S2, Data32And, Data32Or);
  DEBUG ((DEBUG_INFO, "Cpu Dmi ElasticBufferProgramming End"));
  DEBUG ((DEBUG_INFO, "R_CPU_DMI_ACGR3S after setting = %x \n", MmioRead32 (DmiBar + R_CPU_DMI_ACGR3S2)));
}

/**
  Lock SRL bits
  @param[in] DmiBar                 DmiBar
**/
VOID
CpuDmiSetSecuredRegisterLock (
  IN  UINT32                       DmiBar
  )
{
  UINT32                                Data32Or;
  UINT32                                Data32And;

  DEBUG ((DEBUG_INFO, "CpuDmiSetSecuredRegisterLock () Start!\n"));

  Data32And = (UINT32)~B_SA_SPX_PCR_PCD_SRL;
  Data32Or = (UINT32) B_SA_SPX_PCR_PCD_SRL;
  MmioAndThenOr32 (DmiBar +  R_SA_SPX_PCR_SRL, Data32And, Data32Or);
  ///Secure Register Lock (SRL) 0xC8C[0]: 1
  ///Secure Equalization Register Lock (SERL) 0XC8C[8]: 1
  ///Device ID Override Lock (DIDOVR_LOCK) 0xC8C[24]: 1
  Data32Or = (B_CPU_DMI16_LPCR_SRL | B_CPU_DMI16_LPCR_SERL | B_CPU_DMI16_LPCR_DIDOVR_LOCK);
  MmioOr32 (DmiBar + R_CPU_DMI16_LPCR, Data32Or);
  DEBUG ((DEBUG_VERBOSE, "R_CPU_DMI16_LPCR after setting SRL = %x\n", MmioRead32 (DmiBar + R_CPU_DMI16_LPCR)));
  DEBUG ((DEBUG_INFO, "CpuDmiSetSecuredRegisterLock () End!\n"));
}

/**
  This function configures DMI SIP16.

  @param[in] SiPolicyPpi The CPU Policy PPI instance
**/
VOID
CpuDmiInit (
VOID
  )
{
  DEBUG ((DEBUG_INFO, "CpuDmiInit Start\n"));
  UINT64      McD0BaseAddress;
  UINT32      DmiBar;
  EFI_STATUS                    Status;
  SI_PREMEM_POLICY_PPI          *SiPreMemPolicyPpi;
  CPU_DMI_PREMEM_CONFIG         *CpuDmiPreMemConfig;

  SiPreMemPolicyPpi       = NULL;

  //
  // Read DmiBar and pass it to all APIs
  //
  McD0BaseAddress = PCI_SEGMENT_LIB_ADDRESS(SA_SEG_NUM, SA_MC_BUS, SA_MC_DEV, SA_MC_FUN, 0);
  DmiBar = (UINT32)PciSegmentRead32(McD0BaseAddress + R_SA_DMIBAR) &~(1 << N_SA_DMIBAR_DMIBAREN_OFFSET);
  DEBUG ((DEBUG_INFO, "DmiBar = %x \n",DmiBar));

  Status = PeiServicesLocatePpi (
             &gSiPreMemPolicyPpiGuid,
             0,
             NULL,
             (VOID **)&SiPreMemPolicyPpi
             );
  ASSERT_EFI_ERROR (Status);

  CpuDmiPreMemConfig = NULL;
  Status = GetConfigBlock ((VOID *) SiPreMemPolicyPpi, &gCpuDmiPreMemConfigGuid, (VOID *)&CpuDmiPreMemConfig);
  ASSERT_EFI_ERROR (Status);
  DmiDekelInit(CPU_SB_PID_DEKEL_DMIIO);
  CpuDmiBasicInit (DmiBar, CpuDmiPreMemConfig);
  CpuDmiElasticBufferProgramming (DmiBar);
  CpuDmiFiaFinalizeConfigurationAndLock ();
  DmiLimitLinkSpeed (DmiBar, CpuDmiPreMemConfig);
  if ((CpuDmiPreMemConfig->DmiMaxLinkSpeed > 1) && (CpuDmiPreMemConfig->DmiMaxLinkSpeed < 4)){
    PchDmiSetTargetLinkSpeed ((UINT8) CpuDmiPreMemConfig->DmiMaxLinkSpeed);
  }
  ConfigureCpuDmiClockGating (DmiBar);
  ConfigureCpuDmiLinkPowerManagement (DmiBar);
  ConfigureCpuDmiSquelchPowerManagement (DmiBar);
  ConfigureCpuDmiPllShutdown (DmiBar);
  ConfigureCpuDmiPowerGating (DmiBar);
  ConfigureCpuDmiAspm (DmiBar, CpuDmiPreMemConfig);
  ConfigureCpuDmiCoalescing (DmiBar);
  ConfigureCpuDmiAtomics (DmiBar);
  ConfigureCpuDmiSerm (DmiBar);
  ConfigureCpuDmiSmiSci (DmiBar);
  ConfigureCpuDmiReplayTimer (DmiBar);
  ConfigureCpuDmiDesignSpecificConfiguration (DmiBar);
  ConfigureCpuDmiEnabling (DmiBar);
  ConfigureCpuDmiMultiVC (DmiBar, CpuDmiPreMemConfig);
  CpuDmi16EnablePcieRelaxedOrder (DmiBar);
  CpuDmi16EnablePcieLegacyInterrupt (DmiBar);
  //
  // Lock SRL bits after function remapping
  //
  DEBUG((DEBUG_INFO, "Lock the registers\n"));
  CpuDmiSetSecuredRegisterLock (DmiBar);
}
/**
  Enable Relaxed Order for DMI SIP16
**/
VOID
CpuDmi16EnablePcieRelaxedOrder (
  IN  UINT32                       DmiBar
  )
{
  //
  // Enable Forced Relaxed Ordering to always allow downstream completions to pass posted writes.
  // Set Completion Relaxed Ordering Attribute Override Value
  // and Completion Relaxed Ordering Attribute Override Enable
  //
  MmioAndThenOr32 (DmiBar + R_CPU_DMI16_PCIECFG2, ~0u, (B_CPU_DMI16_PCIECFG2_CROAOV | B_CPU_DMI16_PCIECFG2_CROAOE));
}

/**
  Enable Legacy Interrupt for DMI SIP16
**/
VOID
CpuDmi16EnablePcieLegacyInterrupt (
  IN  UINT32                       DmiBar
  )
{
  //
  // BIOS programs this field to 0001b
  //
  MmioAndThenOr32 (DmiBar + R_CPU_DMI_STRPFUSECFG, (UINT32) ~B_CPU_DMI_STRPFUSECFG_PXIP_MASK, (UINT32) BIT0);
}
