/** @file
  This file contains routines used to communicate with FIA 19.0 hardware

@copyright
  INTEL CONFIDENTIAL
  Copyright 2019 Intel Corporation.

  The source code contained or described herein and all documents related to the
  source code ("Material") are owned by Intel Corporation or its suppliers or
  licensors. Title to the Material remains with Intel Corporation or its suppliers
  and licensors. The Material may contain trade secrets and proprietary and
  confidential information of Intel Corporation and its suppliers and licensors,
  and is protected by worldwide copyright and trade secret laws and treaty
  provisions. No part of the Material may be used, copied, reproduced, modified,
  published, uploaded, posted, transmitted, distributed, or disclosed in any way
  without Intel's prior express written permission.

  No license under any patent, copyright, trade secret or other intellectual
  property right is granted to or conferred upon you by disclosure or delivery
  of the Materials, either expressly, by implication, inducement, estoppel or
  otherwise. Any license under such intellectual property rights must be
  express and approved by Intel in writing.

  Unless otherwise agreed by Intel in writing, you may not remove or alter
  this notice or any other notice embedded in Materials by Intel or
  Intel's suppliers or licensors in any way.

  This file contains an 'Intel Peripheral Driver' and is uniquely identified as
  "Intel Reference Module" and is licensed for Intel CPUs and chipsets under
  the terms of your license agreement with Intel or your vendor. This file may
  be modified by the user, subject to additional terms of the license agreement.

@par Specification Reference:
**/
#include <Library/GpioPrivateLib.h>
#include <Library/CpuPcieRpLib.h>

#include "PchFiaLibInternal.h"
#include "PchFia19.h"

/**
  Get FIA lane owner

  @param[in] FiaInst  FIA Instance
  @param[in] LaneNum  lane number

  @retval PCH_FIA_LANE_OWNER  FIA lane owner
**/
PCH_FIA_LANE_OWNER
PchFia19GetLaneOwner (
  IN FIA_INSTANCE FiaInst,
  IN UINT8        LaneNum
  )
{
  UINT32  LosValue;
  UINT32  LosRegisterIndex;

  LosRegisterIndex = LaneNum / 8;
  //
  // Only 5 LOS registers in FIA 19.0
  //
  if (LosRegisterIndex > 4) {
    ASSERT (FALSE);
    return PchFiaOwnerInvalid;
  }

  LosValue = PchPcrRead32 (FiaInst.SbPortId, R_PCH_FIA_19_PCR_LOS1_REG_BASE + (4 * LosRegisterIndex));
  //
  // Invalid LOS register value means that we are in post boot SAI.
  // This code should not be called at this point so ASSERT.
  //
  if (LosValue == 0xFFFFFFFF) {
    ASSERT (FALSE);
    return PchFiaOwnerInvalid;
  }

  return PchFiaDecodeOwner (LosValue >> (4 * (LaneNum % 8)) & B_PCH_FIA_PCR_L0O, LaneNum);
}

/**
  Print FIA LOS registers.

  @param[in] FiaInst  FIA Instance
**/
VOID
PchFia19PrintLosRegisters (
  IN FIA_INSTANCE FiaInst
  )
{
  UINT32  Index;
  UINT32  MaxLosRegister;

  MaxLosRegister = ((PchFiaGetMaxLaneNumEx (FiaInst) - 1) / 8) + 1;
  for (Index = 0; Index < MaxLosRegister; Index++) {
    DEBUG ((DEBUG_INFO, "FIA LOS%d = 0x%08X\n", Index, PchPcrRead32 (FiaInst.SbPortId, R_PCH_FIA_19_PCR_LOS1_REG_BASE + (4 * Index))));
  }
}

/**
  Assigns CLKREQ# to PCH PCIe ports

  @param[in] FiaInst        FIA Instance
  @param[in] ClkReqMap      Mapping between PCH PCIe ports and CLKREQ#
  @param[in] ClkReqMapSize  Size of the map
**/
VOID
PchFia19AssignPchPciePortsClkReq (
  IN FIA_INSTANCE FiaInst,
  IN UINT8        *ClkReqMap,
  IN UINT8        ClkReqMapSize
  )
{
  UINT32  RegisterValue;
  UINT8   RegisterIndex;
  UINT8   ClkReqIndex;
  UINT8   FirstClkReqIndex;
  UINT8   LastClkReqIndex;
  UINT8   ClkReqPosition;

  ASSERT (ClkReqMapSize <= PCH_FIA_19_MAX_PCH_PCIE_CLKREQ);

  for (RegisterIndex = 0; RegisterIndex < PCH_FIA_19_MAX_PDRCRM; RegisterIndex++) {

    RegisterValue = 0;
    FirstClkReqIndex = RegisterIndex * S_PCH_FIA_19_PCR_DRCRM_FIELDS_PER_REG;
    LastClkReqIndex = FirstClkReqIndex + S_PCH_FIA_19_PCR_DRCRM_FIELDS_PER_REG;

    for (ClkReqIndex = FirstClkReqIndex; (ClkReqIndex < LastClkReqIndex) && (ClkReqIndex < PCH_FIA_19_MAX_PCH_PCIE_CLKREQ); ClkReqIndex++) {
      if ((ClkReqIndex < ClkReqMapSize) && (ClkReqMap[ClkReqIndex] != PCH_FIA_NO_CLKREQ)) {
        ClkReqPosition = S_PCH_FIA_19_PCR_DRCRM_BITS_PER_FIELD * (ClkReqIndex % S_PCH_FIA_19_PCR_DRCRM_FIELDS_PER_REG);
        RegisterValue |= ClkReqMap[ClkReqIndex] << ClkReqPosition;
      }
    }

    PchPcrWrite32 (FiaInst.SbPortId, R_PCH_FIA_19_PCR_PDRCRM1 + (RegisterIndex * 4), RegisterValue);
  }

  RegisterIndex = 0;
  RegisterValue = 0;
  for (ClkReqIndex = 0; ClkReqIndex < PCH_FIA_19_MAX_PCH_PCIE_CLKREQ; ClkReqIndex++) {

    if (ClkReqIndex < ClkReqMapSize && ClkReqMap[ClkReqIndex] != PCH_FIA_NO_CLKREQ) {
      RegisterValue |= 1 << (ClkReqIndex % 32);
    }

    if ((ClkReqIndex % 32 == 31) || (ClkReqIndex == PCH_FIA_19_MAX_PCH_PCIE_CLKREQ - 1)) {
      PchPcrWrite32 (FiaInst.SbPortId, R_PCH_FIA_19_PCR_PDRCRE1 + (RegisterIndex * 4), RegisterValue);
      RegisterValue = 0;
      RegisterIndex++;
    }
  }
}

/**
  Assigns CLKREQ# to CPU PCIe ports

  @param[in] FiaInst        FIA Instance
  @param[in] RpIndex        CPU PCIe root port index, 0-based
  @param[in] ClkReqNum      Number of the CLKREQ
**/
VOID
PchFia19AssignCpuPciePortClkReq (
  IN FIA_INSTANCE FiaInst,
  IN UINT32       RpIndex,
  IN UINT8        ClkReqNum
  )
{
  UINT32      ClkReqPosition;
  EFI_STATUS  Status;

  ASSERT (RpIndex <= GetMaxCpuPciePortNum ());

  ClkReqPosition = S_PCH_FIA_19_PCR_DRCRM_BITS_PER_FIELD * RpIndex;

  if (ClkReqNum != PCH_FIA_NO_CLKREQ) {
    PchPcrAndThenOr32 (FiaInst.SbPortId, R_PCH_FIA_19_PCR_CPDRCRM, ~(UINT32) (0x1F << ClkReqPosition), ClkReqNum << ClkReqPosition);
    PchPcrAndThenOr32 (FiaInst.SbPortId, R_PCH_FIA_19_PCR_CPDRCRES, ~(UINT32) 0x0, BIT0 << RpIndex);
    Status = GpioEnableCpuPcieVwClkReqMsgBus (RpIndex);
    ASSERT_EFI_ERROR (Status);
  } else {
    PchPcrAndThenOr32 (FiaInst.SbPortId, R_PCH_FIA_19_PCR_CPDRCRES, ~(UINT32) (BIT0 << RpIndex), 0x0);
  }
}

/**
  Assigns CLKREQ# to CPU PCIe ports

  @param[in] FiaInst        FIA Instance
  @param[in] RpIndex        CPU PCIe root port index, 0-based
  @param[in] ClkReqNum      Number of the CLKREQ
**/
VOID
PchFia19EnableCpuPciePortClkReq (
  IN FIA_INSTANCE FiaInst,
  IN UINT32       RpIndex,
  IN UINT8        ClkReqNum
  )
{
  DEBUG ((DEBUG_INFO, "PchFia19EnableCpuPciePortClkReq Entry\n"));

  ASSERT (RpIndex <= GetMaxCpuPciePortNum ());

  if (ClkReqNum != PCH_FIA_NO_CLKREQ) {
    PchPcrAndThenOr32 (FiaInst.SbPortId, R_PCH_FIA_19_PCR_CPDRCRES, ~(UINT32)0x0, BIT0 << RpIndex);// Use the BIT name to clear the bitmap
  } else {
    PchPcrAndThenOr32 (FiaInst.SbPortId, R_PCH_FIA_19_PCR_CPDRCRES, ~(UINT32)(BIT0 << RpIndex), 0x0);
  }

  DEBUG ((DEBUG_INFO, "PchFia19EnableCpuPciePortClkReq Exit\n"));
}


/**
  Get the status of the CLKREQ

  @param[in] FiaInst  FIA Instance
  @param[in] Rpindex  CPU PCIe index

  @return Status of the CLKREQ signal received by virtual wire
**/
BOOLEAN
PchFia19GetCpuPcieClkReqStatus (
  IN FIA_INSTANCE FiaInst,
  IN UINT32       RpIndex
  )
{
  UINT32  ClkReqStatus;

  ASSERT (RpIndex <= GetMaxCpuPciePortNum ());

  ClkReqStatus = PchPcrRead32 (FiaInst.SbPortId, R_PCH_FIA_19_PCR_CPDRCRES);
  return !!(ClkReqStatus & (B_PCH_FIA_19_PCR_CP0TXCKRQS << RpIndex));
}

/**
  Assigns CLKREQ# to GbE

  @param[in] FiaInst    FIA Instance
  @param[in] ClkReqNum  CLKREQ# number
**/
VOID
PchFia19AssignGbeClkReq (
  IN FIA_INSTANCE FiaInst,
  IN UINT8        ClkReqNum
  )
{
  UINT32  GdrcremValue;

  ASSERT (ClkReqNum <= PCH_FIA_19_MAX_PCH_PCIE_CLKREQ);

  if (ClkReqNum == PCH_FIA_NO_CLKREQ) {
    PchPcrAndThenOr32 (
      FiaInst.SbPortId,
      R_PCH_FIA_19_PCR_GDRCREM,
      ~(UINT32) B_PCH_FIA_19_PCR_GDRCREM_ENABLE,
      0);
  } else {
    GdrcremValue = (ClkReqNum << N_PCH_FIA_19_PCR_GDRCREM_NUMBER) & B_PCH_FIA_19_PCR_GDRCREM_NUMBER;
    GdrcremValue |= B_PCH_FIA_19_PCR_GDRCREM_ENABLE;
    PchPcrWrite32 (FiaInst.SbPortId, R_PCH_FIA_19_PCR_GDRCREM, GdrcremValue);
  }
}

/**
  Configures lower bound of delay between ClkReq assertion and driving RefClk.
  The delay is hardcoded to 15us, due to hardware design. Clocks may not be stable earlier.

  @param[in] FiaInst     FIA Instance
**/
VOID
PchFia19SetClockOutputDelay (
  IN FIA_INSTANCE FiaInst
  )
{
  UINT32 Data32Or;
  UINT32 Data32And;
  UINT32 Index;

  Data32Or = 0;
  Data32And = 0xFFFFFFFF;
  for (Index = 0; Index < S_PCH_FIA_PCR_TREFCLKON_FIELDS_PER_REG; Index++) {
    Data32Or |= (V_PCH_FIA_PCR_TREFCLKON_15us << (Index * S_PCH_FIA_PCR_TREFCLKON_FIELD));
    Data32And &= ~(B_PCH_FIA_PCR_TREFCLKON_FIELD << (Index * S_PCH_FIA_PCR_TREFCLKON_FIELD));
  }
  PchPcrAndThenOr32 (FiaInst.SbPortId, R_PCH_FIA_PCR_TREFCLKONTMR1, Data32And, Data32Or);
  PchPcrAndThenOr32 (FiaInst.SbPortId, R_PCH_FIA_PCR_TREFCLKONTMR2, Data32And, Data32Or);
  PchPcrAndThenOr32 (FiaInst.SbPortId, R_PCH_FIA_PCR_TREFCLKONTMR3, Data32And, Data32Or);
  PchPcrAndThenOr32 (FiaInst.SbPortId, R_PCH_FIA_PCR_TREFCLKONTMR4, Data32And, Data32Or);
}

/**
  Converts FIA 19 lane owner encoding to version independent encoding.

  @param[in] LaneOwner  Lane owner value corresponding to the FIA 19 lane owner encoding
  @param[in] LaneNum    Fia lane number

  @retval PCH_FIA_LANE_OWNER  FIA lane owner in version independent format.
**/
PCH_FIA_LANE_OWNER
PchFiaDecodeOwner (
  IN UINT32  LaneOwner,
  IN UINT8   LaneNum
  )
{
  switch (LaneOwner) {
    case V_PCH_FIA_19_PCR_LANE_OWN_USB3:
      return PchFiaOwnerUsb3;
    case V_PCH_FIA_19_PCR_LANE_OWN_PCIE:
      return PchFiaOwnerPcie;
    case V_PCH_FIA_19_PCR_LANE_OWN_DMI:
      return PchFiaOwnerDmi;
    case V_PCH_FIA_19_PCR_LANE_OWN_SATA:
      return PchFiaOwnerSata;
    case V_PCH_FIA_19_PCR_LANE_OWN_GBE:
      return PchFiaOwnerGbe;
    case V_PCH_FIA_19_PCR_LANE_OWN_UFS:
      return PchFiaOwnerUfs;
    case V_PCH_FIA_19_PCR_LANE_OWN_TSN:
      return PchFiaOwnerTsn;
    default:
      return PchFiaOwnerInvalid;
  }
}
