/** @file
  This file contains ADL specific FIA routines

@copyright
  INTEL CONFIDENTIAL
  Copyright 2019 - 2020 Intel Corporation.

  The source code contained or described herein and all documents related to the
  source code ("Material") are owned by Intel Corporation or its suppliers or
  licensors. Title to the Material remains with Intel Corporation or its suppliers
  and licensors. The Material may contain trade secrets and proprietary and
  confidential information of Intel Corporation and its suppliers and licensors,
  and is protected by worldwide copyright and trade secret laws and treaty
  provisions. No part of the Material may be used, copied, reproduced, modified,
  published, uploaded, posted, transmitted, distributed, or disclosed in any way
  without Intel's prior express written permission.

  No license under any patent, copyright, trade secret or other intellectual
  property right is granted to or conferred upon you by disclosure or delivery
  of the Materials, either expressly, by implication, inducement, estoppel or
  otherwise. Any license under such intellectual property rights must be
  express and approved by Intel in writing.

  Unless otherwise agreed by Intel in writing, you may not remove or alter
  this notice or any other notice embedded in Materials by Intel or
  Intel's suppliers or licensors in any way.

  This file contains an 'Intel Peripheral Driver' and is uniquely identified as
  "Intel Reference Module" and is licensed for Intel CPUs and chipsets under
  the terms of your license agreement with Intel or your vendor. This file may
  be modified by the user, subject to additional terms of the license agreement.

@par Specification Reference:
**/
#include "PchFiaLibInternal.h"
#include <Register/PchPcrRegs.h>

#include "PchFia19.h"

/**
  Returns the FIA-LP Instance

  @return FIA-LP Instance
**/
FIA_INSTANCE
FiaGetInstance (
  VOID
  )
{
  FIA_INSTANCE FiaInst;
  FiaInst.SbPortId = PID_FIA;
  return FiaInst;
}

/**
  Returns the FIA-U Instance

  @return FIA-U Instance
**/
FIA_INSTANCE
FiaGetUInstance (
  VOID
  )
{
  FIA_INSTANCE FiaInst;
  FiaInst.SbPortId = PID_FIAU;
  return FiaInst;
}

/**
  Returns the FIA-PD Instance

  @return FIA-PD Instance
**/
FIA_INSTANCE
FiaGetPdInstance (
  VOID
  )
{
  FIA_INSTANCE FiaInst;
  FiaInst.SbPortId = PID_FIAPD;
  return FiaInst;
}

/**
  Returns the FIA-PGS Instance

  @return FIA-PGS Instance
**/
FIA_INSTANCE
FiaGetPgsInstance (
  VOID
  )
{
  FIA_INSTANCE FiaInst;
  FiaInst.SbPortId = PID_FIAPGS;
  return FiaInst;
}

/**
  Returns a FIA lane number for a given SATA port.

  @param[in]  SataCtrlIndex  SATA controller index
  @param[in]  SataPortIndex  SATA port index
  @param[out] LaneNum        Pointer to the variable that stores lane number.
                             The output value is only valid if this function returns TRUE.

  @return TRUE if given SATA port owns FIA lane, FALSE otherwise
**/
BOOLEAN
PchFiaGetSataLaneNum (
  IN  UINT32  SataCtrlIndex,
  IN  UINT32  SataPortIndex,
  OUT UINT8   *LaneNum
  )
{
  FIA_INSTANCE FiaInst;
  if (IsPchLp ()) {
    FiaInst = FiaGetInstance ();
    switch (SataPortIndex) {
      case 0:
        if (PchFiaGetLaneOwner (FiaInst, 6) == PchFiaOwnerSata) {
          *LaneNum = 6;
          return TRUE;
        } else if (PchFiaGetLaneOwner (FiaInst, 8) == PchFiaOwnerSata) {
          *LaneNum = 8;
          return TRUE;
        }
        return FALSE;
      case 1:
        *LaneNum = 7;
        break;
      default:
        ASSERT (FALSE);
        return FALSE;
    }
  } else {
    ASSERT (IsPchS ());
    FiaInst = FiaGetPgsInstance ();
    switch (SataPortIndex) {
      case 0:
        *LaneNum = 12;
        break;
      case 1:
        *LaneNum = 13;
        break;
      case 2:
        *LaneNum = 14;
        break;
      case 3:
        *LaneNum = 15;
        break;
      case 4:
        *LaneNum = 16;
        break;
      case 5:
        *LaneNum = 17;
        break;
      case 6:
        *LaneNum = 18;
        break;
      case 7:
        *LaneNum = 19;
        break;
      default:
        ASSERT (FALSE);
        return FALSE;
    }
  }

  return (PchFiaGetLaneOwner (FiaInst, *LaneNum) == PchFiaOwnerSata);
}

/**
  Returns a FIA lane number for a GbE controller

  @param[out] LaneNum  FIA lane number that is owned by GbE

  @retval TRUE   Found FIA lane assigned to GbE
  @retval FALSE  No lanes assigned to GbE
**/
BOOLEAN
PchFiaGetGbeLaneNum (
  OUT UINT8  *LaneNum
  )
{
  FIA_INSTANCE FiaInst;
  STATIC UINT8  GbeLanesLp[] = {5, 9, 10};
  STATIC UINT8  GbeLanesS[]  = {2, 6, 14};
  UINT8  *GbeLanes;
  UINT8  GbeLanesSize;
  UINT8  LaneIndex;

  if (IsPchLp ()) {
    FiaInst = FiaGetInstance ();
    GbeLanes = GbeLanesLp;
    GbeLanesSize = ARRAY_SIZE (GbeLanesLp);
  } else {
    ASSERT (IsPchS ());
    FiaInst = FiaGetPgsInstance ();
    GbeLanes = GbeLanesS;
    GbeLanesSize = ARRAY_SIZE (GbeLanesS);
  }

  //
  // Since there is only one instance of GbE controller on the system
  // we return first lane that is owned by GbE.
  //
  for (LaneIndex = 0; LaneIndex < GbeLanesSize; LaneIndex++) {
    if (PchFiaGetLaneOwner (FiaInst, GbeLanes[LaneIndex]) == PchFiaOwnerGbe) {
      *LaneNum = GbeLanes[LaneIndex];
      return TRUE;
    }
  }

  DEBUG ((DEBUG_INFO, "No FIA lanes assigned as GbE\n"));
  return FALSE;
}

/**
  Returns a FIA lane number for a given PCIe lane.

  @param[in]  PciePhysicalLane  Index of the PCIe lane
  @param[out] LaneNum           Pointer to the variable that stores lane number.
                                The output value is only valid if this function returns TRUE.

  @return TRUE if given PciePhysicalLane owns FIA lane, FALSE otherwise
**/
BOOLEAN
PchFiaGetPcieLaneNum (
  IN  UINT32  PciePhysicalLane,
  OUT UINT8   *LaneNum
  )
{
  STATIC UINT8 FirstLaneS_PD[]  = { 15, 11 };
  UINT32  ControllerIndex;
  UINT32  ControllerPhyLane;
  FIA_INSTANCE FiaInst;

  if (IsPchLp ()) {
    *LaneNum = (UINT8) PciePhysicalLane;
    FiaInst = FiaGetInstance ();
  } else {
    ASSERT (IsPchS ());
    if (PciePhysicalLane < 20) {
      *LaneNum = (UINT8) PciePhysicalLane;
      FiaInst = FiaGetPgsInstance ();
    } else {
      ControllerIndex = (PciePhysicalLane - 20) / 4;
      ControllerPhyLane = (PciePhysicalLane - 20) % 4;
      *LaneNum = FirstLaneS_PD[ControllerIndex] - (UINT8) ControllerPhyLane;
      FiaInst = FiaGetPdInstance ();
    }
  }
  return (PchFiaGetLaneOwner (FiaInst, *LaneNum) == PchFiaOwnerPcie);
}

/**
  Returns a FIA lane number for a given USB3 port.
  This function assumes that for ADP-Lp and ADP-S USB3 ports to FIA lane mapping is 1:1

  @param[in]  Usb3PortIndex  USB3 port index
  @param[out] LaneNum        Pointer to the variable that stores lane number.
                             The output value is only valid if this function returns TRUE.

  @return TRUE if given USB3 port owns FIA lane, FALSE otherwise
**/
BOOLEAN
PchFiaGetUsb3LaneNum (
  IN  UINT32  Usb3PortIndex,
  OUT UINT8   *LaneNum
  )
{
  FIA_INSTANCE FiaInst;

  ASSERT (Usb3PortIndex < GetPchXhciMaxUsb3PortNum ());
  *LaneNum = (UINT8) Usb3PortIndex;
  if (IsPchLp ()) {
    FiaInst = FiaGetInstance ();
  } else {
    ASSERT (IsPchS ());
    FiaInst = FiaGetUInstance ();
  }
  return (PchFiaGetLaneOwner (FiaInst, *LaneNum) == PchFiaOwnerUsb3);
}

/**
  Returns number of FIA lanes

  @param[in] FiaInst  FIA Instance

  @return Number of FIA lanes
**/
UINT8
PchFiaGetMaxLaneNumEx (
  IN  FIA_INSTANCE FiaInst OPTIONAL
  )
{
  if (IsPchLp ()) {
    return 12;
  } else {
    ASSERT (IsPchS ());
    switch (FiaInst.SbPortId) {
      case PID_FIAPGS:
        return 20;
      case PID_FIAPD:
        return 16;
      case PID_FIAU:
        return 10;
      default:
        ASSERT (FALSE);
        DEBUG ((DEBUG_INFO, "Invalid FIA Instance \n"));
        return 0;
    }
  }
}

/**
  Return FIA lane owner.

  @param[in] FiaInst  FIA Instance
  @param[in] LaneNum  FIA lane number

  @return  Code of the FIA lane owner, PchFiaOwnerInvalid if lane number wasn't valid
**/
PCH_FIA_LANE_OWNER
PchFiaGetLaneOwner (
  IN  FIA_INSTANCE FiaInst,
  IN  UINT8        LaneNum
  )
{
  if (LaneNum >= PchFiaGetMaxLaneNumEx (FiaInst)) {
    ASSERT (FALSE);
    return PchFiaOwnerInvalid;
  }
  return PchFia19GetLaneOwner (FiaInst, LaneNum);
}

/**
  Returns a FIA lane number for a given UFS lane.

  @param[in]  UfsControllerIndex  Index of the UFS controller
  @param[in]  UfsLaneIndex        Index of the UFS lane on given controller
  @param[out] LaneNum             Optional. Pointer to the variable that stores lane number.
                                  The output value is only valid if this function returns TRUE.

 @return TRUE if given UFS lane owns FIA lane, FALSE otherwise
**/
BOOLEAN
PchFiaGetUfsLaneNum (
  IN  UINT32  UfsControllerIndex,
  IN  UINT32  UfsLaneIndex,
  OUT UINT8   *LaneNum
  )
{
  ASSERT (FALSE);
  return FALSE;
}

/**
  Returns a FIA lane number for a given TSN port.

  @param[in]  TsnPortIndex  TSN port index
  @param[out] LaneNum       Pointer to the variable that stores lane number.
                            The output value is only valid if this function returns TRUE.

  @return TRUE if given TSN port owns FIA lane, FALSE otherwise
**/
BOOLEAN
PchFiaGetTsnLaneNum (
  IN  UINT32  TsnPortIndex,
  OUT UINT8   *LaneNum
  )
{
  ASSERT (FALSE);
  return FALSE;
}

/**
  Print FIA LOS registers.
**/
VOID
PchFiaPrintLosRegisters (
  VOID
  )
{
  if (IsPchLp ()) {
    PchFia19PrintLosRegisters (FiaGetInstance ());
  } else {
    ASSERT (IsPchS ());
    DEBUG ((DEBUG_INFO, "FIAU: \n"));
    PchFia19PrintLosRegisters (FiaGetUInstance ());
    DEBUG ((DEBUG_INFO, "FIAPD: \n"));
    PchFia19PrintLosRegisters (FiaGetPdInstance ());
    DEBUG ((DEBUG_INFO, "FIAPGS: \n"));
    PchFia19PrintLosRegisters (FiaGetPgsInstance ());
  }
}

/**
  Assigns CLKREQ# to PCH PCIe ports

  @param[in] ClkReqMap      Mapping between PCH PCIe ports and CLKREQ#
  @param[in] ClkReqMapSize  Size of the map
**/
VOID
PchFiaAssignPchPciePortsClkReq (
  IN UINT8  *ClkReqMap,
  IN UINT8  ClkReqMapSize
  )
{
  if (IsPchLp ()) {
    PchFia19AssignPchPciePortsClkReq (FiaGetInstance (), ClkReqMap, ClkReqMapSize);
  } else {
    ASSERT (IsPchS ());
    PchFia19AssignPchPciePortsClkReq (FiaGetPgsInstance (), ClkReqMap, ClkReqMapSize);
  }
}

/**
  Assigns CLKREQ# to CPU PCIe ports

  @param[in] RpIndex        CPU PCIe root port index
  @param[in] ClkReqNum      Number of the CLKREQ
**/
VOID
PchFiaAssignCpuPciePortClkReq (
  IN UINT32  RpIndex,
  IN UINT8   ClkReqNum
  )
{
  if (IsPchLp ()) {
    PchFia19AssignCpuPciePortClkReq (FiaGetInstance (), RpIndex, ClkReqNum);
  } else {
    ASSERT (IsPchS ());
    PchFia19AssignCpuPciePortClkReq (FiaGetPgsInstance (), RpIndex, ClkReqNum);
  }
}

/**
Enable CLKREQ# to CPU PCIe ports

@param[in] RpIndex        CPU PCIe root port index
@param[in] ClkReqNum      Number of the CLKREQ
**/
VOID
PchFiaEnableCpuPciePortClkReq (
  IN UINT32  RpIndex,
  IN UINT8   ClkReqNum
  )
{
  DEBUG ((DEBUG_INFO, "PchFiaEnableCpuPciePortClkReq Entry\n"));
  if (IsPchLp ()) {
    PchFia19EnableCpuPciePortClkReq (FiaGetInstance (), RpIndex, ClkReqNum);
  } else {
    ASSERT (IsPchS ());
    PchFia19EnableCpuPciePortClkReq (FiaGetPgsInstance (), RpIndex, ClkReqNum);
  }
  DEBUG ((DEBUG_INFO, "PchFiaEnableCpuPciePortClkReq Exit\n"));
}

/**
  Return the status of the CLKREQ state received with VW msg.

  @param[in] RpIndex  CPU PCIe index.

  @return Status of the CLKREQ.
**/
BOOLEAN
PchFiaGetCpuPcieClkReqStatus (
  IN UINT32  RpIndex
  )
{
  if (IsPchLp ()) {
    return PchFia19GetCpuPcieClkReqStatus (FiaGetInstance (), RpIndex);
  } else {
    ASSERT (IsPchS ());
    return PchFia19GetCpuPcieClkReqStatus (FiaGetPgsInstance (), RpIndex);
  }
}

/**
  Assigns CLKREQ# to GbE

  @param[in]  ClkReqNum  CLKREQ# number
**/
VOID
PchFiaAssignGbeClkReq (
  IN UINT8  ClkReqNum
  )
{
  if (IsPchLp ()) {
    PchFia19AssignGbeClkReq (FiaGetInstance (), ClkReqNum);
  } else {
    ASSERT (IsPchS ());
    PchFia19AssignGbeClkReq (FiaGetPgsInstance (), ClkReqNum);
  }
}

/**
  Configures lower bound of delay between ClkReq assertion and driving RefClk.
**/
VOID
PchFiaSetClockOutputDelay (
  VOID
  )
{
  if (IsPchLp ()) {
    PchFia19SetClockOutputDelay (FiaGetInstance ());
  } else {
    ASSERT (IsPchS ());
    PchFia19SetClockOutputDelay (FiaGetPgsInstance ());
  }
}

/**
  Performs FIA programming required at the end of configuration and locks lockable FIA registers
**/
VOID
PchFiaFinalizeConfigurationAndLock (
  VOID
  )
{
  FIA_INSTANCE FiaInst;
  UINT32       Data32Or;
  UINT8        NbFiaIPs;
  UINT8        Index;

  FIA_INSTANCE (*FiaGetInst []) (VOID) = {FiaGetInstance, FiaGetPdInstance, FiaGetPgsInstance};

  if (IsPchLp ()) {
    NbFiaIPs = 1;
  } else {
    ASSERT (IsPchS ());
    NbFiaIPs = 3;
  }

  for (Index = 0;  Index < NbFiaIPs; Index++) {
    FiaInst = (*FiaGetInst[Index])();
    //
    // Program PCR[FIA] + 20h bit [13:11] to 001, bit [30:28] to [000]
    //
    PchPcrAndThenOr32 (
      FiaInst.SbPortId,
      R_PCH_FIA_PCR_PLLCTL,
      (UINT32) ~(B_PCH_FIA_PCR_PLLCTL_CL0PLLWAIT | B_PCH_FIA_PCR_PLLCTL_PLLCLKVADT),
      (UINT32) (1 << N_PCH_FIA_PCR_PLLCTL_CL0PLLWAIT)
      );

    //
    // Program PCR[FIA] + 18h bit [17:16, 1:0] to [00, 00]
    //
    PchPcrAndThenOr32 (
      FiaInst.SbPortId,
      R_PCH_FIA_PCR_CLSDM,
      (UINT32) ~(B_PCH_FIA_PCR_CLSDM_DMIIPSLSD | B_PCH_FIA_PCR_CLSDM_PCIEGBEIPSLSD),
      0
      );

    //
    // Set PCR[FIA] + 0h bit [17, 16, 15] to [1, 1, 1]
    //
    PchPcrAndThenOr32 (
      FiaInst.SbPortId,
      R_PCH_FIA_PCR_CC,
      ~0u,
      B_PCH_FIA_PCR_CC_PTOCGE | B_PCH_FIA_PCR_CC_OSCDCGE | B_PCH_FIA_PCR_CC_SCPTCGE
      );

    //
    // Set PCR[FIA] + 40h bit [3] to [1]
    //
    Data32Or = B_PCH_FIA_PCR_PMC_PRDPGE;
    PchPcrAndThenOr32 (
      FiaInst.SbPortId,
      R_PCH_FIA_PCR_PMC,
      ~0u,
      Data32Or
      );

    //
    // Set PCR[FIA] + 48h bit [0] to [0]
    //
    PchPcrAndThenOr32 (
      FiaInst.SbPortId,
      R_PCH_FIA_PCR_PGCUC,
      (UINT32) ~(B_PCH_FIA_PCR_PGCUC_ACC_CLKGATE_DISABLED),
      0
      );

    //
    // Set PCR[FIA] + 50h bit [0] to [0]
    //
    PchPcrAndThenOr32 (
      FiaInst.SbPortId,
      R_PCH_FIA_PCR_PGCUCSOC,
      (UINT32) ~(B_PCH_FIA_PCR_PGCUCSOC_ACC_CLKGATE_DISABLED),
      0
      );
  }
}

/**
  Returns number of FIA LOS registers used

  @param[in] FiaIndex        FIA index to specify FIA instance

  @return Number of FIA LOS registers used
**/
UINT32
PchFiaGetMaxLosRegister (
  FIA_INDEX FiaIndex
  )
{
  FIA_INSTANCE FiaInst;

  if (IsPchLp ()) {
    FiaInst = FiaGetInstance ();
  } else {
    ASSERT (IsPchS ());
    switch (FiaIndex) {
      case FiaIndexFiaU:
        FiaInst = FiaGetUInstance ();
        break;
      case FiaIndexFiaPgs:
        FiaInst = FiaGetPgsInstance ();
        break;
      case FiaIndexFiaPd:
        FiaInst = FiaGetPdInstance ();
        break;
      default:
        ASSERT(FALSE);
        FiaInst = FiaGetUInstance ();
        DEBUG ((DEBUG_INFO, "Invalid FIA Instance \n"));
        break;
    }
  }

  return (((PchFiaGetMaxLaneNumEx (FiaInst) - 1) / 8) + 1);
}

/**
  Get FIA LOS register value

  @param[in] FiaIndex        FIA index to specify FIA instance
  @param[in] LosIndex        LOS FIA register index

  @return  LOS FIA register value
**/
UINT32
PchFiaGetLos (
  FIA_INDEX FiaIndex,
  UINT32    LosIndex
  )
{
  UINT32  MaxLosRegister;
  FIA_INSTANCE FiaInst = {0};

  if (IsPchLp ()) {
    FiaInst = FiaGetInstance ();
  } else {
    ASSERT (IsPchS ());
    switch (FiaIndex) {
      case FiaIndexFiaU:
        FiaInst = FiaGetUInstance ();
        break;
      case FiaIndexFiaPgs:
        FiaInst = FiaGetPgsInstance ();
        break;
      case FiaIndexFiaPd:
        FiaInst = FiaGetPdInstance ();
        break;
      default:
        ASSERT(FALSE);
        FiaInst = FiaGetUInstance ();
        DEBUG ((DEBUG_INFO, "Invalid FIA Instance \n"));
        break;
    }
  }

  MaxLosRegister = ((PchFiaGetMaxLaneNumEx (FiaInst) - 1) / 8) + 1;
  if (LosIndex > MaxLosRegister) {
    ASSERT (FALSE);
    LosIndex = MaxLosRegister;
  }

  return PchPcrRead32 (FiaInst.SbPortId, R_PCH_FIA_19_PCR_LOS1_REG_BASE + (4 * LosIndex));
}
