/** @file
  This file provides Fusa Core IDI Parity E2E CTC implementation

@copyright
  INTEL CONFIDENTIAL
  Copyright 2019 - 2020 Intel Corporation.

  The source code contained or described herein and all documents related to the
  source code ("Material") are owned by Intel Corporation or its suppliers or
  licensors. Title to the Material remains with Intel Corporation or its suppliers
  and licensors. The Material may contain trade secrets and proprietary and
  confidential information of Intel Corporation and its suppliers and licensors,
  and is protected by worldwide copyright and trade secret laws and treaty
  provisions. No part of the Material may be used, copied, reproduced, modified,
  published, uploaded, posted, transmitted, distributed, or disclosed in any way
  without Intel's prior express written permission.

  No license under any patent, copyright, trade secret or other intellectual
  property right is granted to or conferred upon you by disclosure or delivery
  of the Materials, either expressly, by implication, inducement, estoppel or
  otherwise. Any license under such intellectual property rights must be
  express and approved by Intel in writing.

  Unless otherwise agreed by Intel in writing, you may not remove or alter
  this notice or any other notice embedded in Materials by Intel or
  Intel's suppliers or licensors in any way.

  This file contains an 'Intel Peripheral Driver' and is uniquely identified as
  "Intel Reference Module" and is licensed for Intel CPUs and chipsets under
  the terms of your license agreement with Intel or your vendor. This file may
  be modified by the user, subject to additional terms of the license agreement.

@par Specification Reference:
**/

#include <Uefi/UefiBaseType.h>
#include <Library/BaseLib.h>
#include <Library/DebugLib.h>
#include <Library/BaseMemoryLib.h>
#include <Library/LocalApicLib.h>
#include "PeiFusaE2eCtcLibInternal.h"
#include "PeiFusaPrivateLibInternal.h"
#include "PeiFusaResultReportingLib.h"

///
/// Fusa Error Injection Configuration Registers
///

/**
  FUSA_PERRINJ_CTRL.PERR_ADDR.
    if set to 1: indicates address injection, in which case the
    later bits are unimportant.
    If set to 0: indicates data injection and BIT_LOC, CONE,
    SLICE are taken into account
**/
#define B_FUSA_PERRINJ_CTRL_PERR_ADDR BIT10

#define B_FUSA_PERRINJ_CTRL_DATA_PARITY_SLICE_SELECT    BIT9      ///<Slice Selection, must match address[6]
#define B_FUSA_PERRINJ_CTRL_DATA_PARITY_CONE_SELECT     BIT8      ///Cone selection
#define B_FUSA_PERRINJ_CTRL_PERR_DATA                   0xFFU     ///One-hot data parity

extern UINT8 *gpTestData;

STATIC UINT32 mBspApicId = 0;
STATIC UINT32 mCurrentThreadApicId = 0;

/**
  Error injection and corresponding traffic routine for Core IDI
  Data

  @param[in] IpIndex - IP index vale for MSR MSR_PERRINJ_AT_IP
  @param[in] InjectionPayload - Payload value for MSR MSR_PERRINJ_CTRL

  @return injection control value after traffic generation
**/
STATIC
UINT32
InjectionAndTrafficCoreIdiData(
  IN UINT32 IpIndex,
  IN UINT32 InjectionPayload
  )
{
  UINT32 Value;
  UINT32 Address;
  UINT32 DataAddress = (UINT32) gpTestData;

  _asm
  {
    lea eax,  InjectionAndTrafficCoreIdiData
    mov Address, eax

    //clflush gpTestData
    mov eax, DataAddress
    clflush byte ptr [eax]
    mfence //cflush is ordred with fence instruction
    //injection
    //update IpIndex
    mov     edx, 0
    mov     eax, IpIndex
    //mov     eax, esi
    mov     ecx, MSR_PERRINJ_AT_IP
    wrmsr
    //update InjectionPayload
    mov     eax, InjectionPayload
    //mov     eax, edi
    mov     ecx, MSR_PERRINJ_CTRL
    wrmsr

    //traffic
    mfence
    //64KB of dword reference
    wbinvd
    mov ecx, 0x400
    mov eax, DataAddress
    DUMMY_DATA_READ:
    prefetcht1 BYTE PTR [eax+256]
    add edx, DWORD ptr [eax]
    add eax, 64
    loop DUMMY_DATA_READ
    mfence
    nop

    //readback the value after injection
    mov     ecx, MSR_PERRINJ_CTRL
    rdmsr
    mov     Value, eax
  }

  if (mCurrentThreadApicId == mBspApicId)
  {
    DEBUG ((DEBUG_INFO, "First read back of control is 0x%x\n", Value));
    DEBUG ((DEBUG_INFO, "TrafficCoreIdiData code address is at 0x%x\n", Address));
  }

  return Value;
}

/**
  Error injection and corresponding traffic routine for Core IDI
  Address

  @param[in] IpIndex - IP index vale for MSR MSR_PERRINJ_AT_IP
  @param[in] InjectionPayload - Payload value for MSR MSR_PERRINJ_CTRL

  @return injection control value after traffic generation
**/
STATIC
UINT32
InjectionAndTrafficCoreIdiAddress(
  IN UINT32 IpIndex,
  IN UINT32 InjectionPayload
  )
{
  UINT32 Value;
  UINT32 DataAddress = (UINT32) gpTestData;
  gpTestData[0] = 0xba;

    _asm
  {
    //update IpIndex
    mov     edx, 0
    mov     eax, IpIndex
    mov     ecx, MSR_PERRINJ_AT_IP
    wrmsr
    //update InjectionPayload
    mov     eax, InjectionPayload
    mov     ecx, MSR_PERRINJ_CTRL
    wrmsr
    mov eax, DataAddress
    //traffic
    mfence
    //clflush gpTestData
    clflush byte ptr [eax]
    mfence

    //readback the value after injection
    mov     ecx, MSR_PERRINJ_CTRL
    rdmsr
    mov     Value, eax

  }

  return Value;
}

/**
  Generic CTC routine for Core IDI Data

  @note Control the debug dump to be within BSP as dumping
        within the AP will cause hang

  @param[in] InjectionPayload - Payload value for MSR MSR_PERRINJ_CTRL
  @param[in] CheckNum - check number assigned to the check
  @param[in,out] pFusaTestResult - pointer to the test result
        buffer for result to be updated

  @return FUSA_LIB_STATUS
**/
STATIC
FUSA_LIB_STATUS
CoreIdiDataParityTestGeneric (
  IN UINT32               InjectionPayload,
  IN UINT8                CheckNum,
  IN OUT FUSA_TEST_RESULT *pFusaTestResult
  )
{
  FUSA_LIB_STATUS LibStatus;
  UINT32 IpIndex = CORE;
  UINT8 McaBankNum = MCA_MLC;
  UINT64 ExpectedMcaStatus = 0x0405ULL;
  UINT64 McaStatusMask = MCACOD_MASK;

  LibStatus = FusaMsrCtcTestGeneric (
                IpIndex,
                InjectionPayload,
                InjectionAndTrafficCoreIdiData,
                McaBankNum,
                McaStatusMask,
                ExpectedMcaStatus,
                CheckNum,
                pFusaTestResult,
                mCurrentThreadApicId == mBspApicId
                );

  return LibStatus;
}

/**
  Generic CTC routine for Core IDI Address

  @note Control the debug dump to be within BSP as dumping
        within the AP will cause hang

  @param[in] InjectionPayload - Payload value for MSR
        MSR_PERRINJ_CTRL
  @param[in] CheckNum - check number assigned to the check
  @param[in,out] pFusaTestResult - pointer to the test result
        buffer for result to be updated

  @return FUSA_LIB_STATUS
**/
STATIC
FUSA_LIB_STATUS
CoreIdiAddressParityTestGeneric (
  IN UINT32               InjectionPayload,
  IN UINT8                CheckNum,
  IN OUT FUSA_TEST_RESULT *pFusaTestResult
  )
{
  FUSA_LIB_STATUS LibStatus;
  UINT32 IpIndex = CORE;
  UINT8 McaBankNum = MCA_MLC;
  UINT64 ExpectedMcaStatus = 0x0405ULL;
  UINT64 McaStatusMask = MCACOD_MASK;

  LibStatus = FusaMsrCtcTestGeneric (
                IpIndex,
                InjectionPayload,
                InjectionAndTrafficCoreIdiAddress,
                McaBankNum,
                McaStatusMask,
                ExpectedMcaStatus,
                CheckNum,
                pFusaTestResult,
                mCurrentThreadApicId == mBspApicId
                );

  return LibStatus;
}

/**
  Perform the permutation needed for Core IDI CTC

  @param[in,out] pFusaTestResult - result buffer where the test
       result to be updated to
  @return FUSA_LIB_STATUS
**/
STATIC
FUSA_LIB_STATUS
CoreIdiDataAddressParityCtc (
  IN OUT FUSA_TEST_RESULT *pFusaTestResult
  )
{
  FUSA_LIB_STATUS LibStatus;

  mCurrentThreadApicId = GetInitialApicId();

  LibStatus = CoreIdiDataParityTestGeneric (
                0xFF, // SLICE = 0, CONE = 0, PERR_DATA[BIT_LOC] = 0xFF
                0U, // check number
                pFusaTestResult
                );
  ASSERT(LibStatus == FusaNoError);

  LibStatus |= CoreIdiDataParityTestGeneric (
                 0x1FF, // SLICE = 0, CONE = 1, PERR_DATA[BIT_LOC] = 0xFF
                 1U, // check number
                 pFusaTestResult
                 );
  ASSERT(LibStatus == FusaNoError);

  LibStatus |= CoreIdiDataParityTestGeneric (
                 0x2FF, // SLICE = 1, CONE = 0, PERR_DATA[BIT_LOC] = 0xFF
                 2U, // check number
                 pFusaTestResult
                 );
  ASSERT(LibStatus == FusaNoError);

  LibStatus |= CoreIdiDataParityTestGeneric (
                 0x3FF, // SLICE = 1, CONE = 1, PERR_DATA[BIT_LOC] = 0xFF
                 3U,
                 pFusaTestResult
                 );
  ASSERT(LibStatus == FusaNoError);

  LibStatus |= CoreIdiAddressParityTestGeneric (
                 B_FUSA_PERRINJ_CTRL_PERR_ADDR,
                 4U, // check number
                 pFusaTestResult
                 );
  ASSERT(LibStatus == FusaNoError);

  return LibStatus;
}

/// Structure of the parameter passed into the Core Idi test routine
typedef struct
{
  FUSA_TEST_RESULT *pFusaTestResult;
  UINT32           CoreNumber;
} CORE_IDI_TEST_PARAM;

/**
  Core IDI Parity Error E2E CTC internal main routine

  @param[in,out] pCoreIdiTestParam - conttain result buffer
       where the test result to be updated to beside core number
       information
**/
VOID
FspDxCheckCoreIdiInternal (
  IN CORE_IDI_TEST_PARAM *pCoreIdiTestParam
  )
{
  FUSA_LIB_STATUS LibStatus;
  FUSA_TEST_RESULT *pFusaTestResult;
  UINT32 CoreNumber;

  UINTN   Cr0;
  UINTN   Cr4;

  ///
  /// Enable XMM and cache
  ///
  Cr0 = AsmReadCr0 ();
  Cr0 |= BIT1;
  Cr0 &= ~BIT30; //make sure cache is not disabled
  AsmWriteCr0 (Cr0);

  Cr4 = AsmReadCr4 ();
  Cr4 |= (UINTN) (BIT9 | BIT10);
  AsmWriteCr4 (Cr4);

  pFusaTestResult = pCoreIdiTestParam->pFusaTestResult;
  CoreNumber = pCoreIdiTestParam->CoreNumber;

  LibStatus = FusaTestAndReporting (
                CoreIdiDataAddressParityCtc,
                FusaTestNumCpu0Idi + CoreNumber,
                5U,
                pFusaTestResult
                );
  ASSERT(LibStatus == FusaNoError);
}

/**
  Perform Core IDI Parity Error E2E CTC.
  The test targets all active unique core. For core not
  detected, their test result is updated to
  FUSA_TEST_DEVICE_NOTAVAILABLE

  @note The test expects to be run within  Memory Type =
        Writeback (MTRR Encoding = 06h), which is the usual
        case of the memory type covering the code/data region
        of the bootloader.

  @param[out] pFusaTestResult - pointer to test result
                         buffer for test from FusaTestNumCpu0Idi
                         to FusaTestNumCpu7Idi

**/
VOID
FspDxCheckCoreIdi (
  OUT FUSA_TEST_RESULT *pFusaTestResult
  )
{
  UINT32 CoreIndex;
  CORE_IDI_TEST_PARAM CoreIdiTestParam;
  CONST UINT32 *UniqueCoreList = UniqueCoreListGet ();

  DEBUG ((DEBUG_INFO, "FspDxCheckCoreIdi\n"));

  mBspApicId = GetInitialApicId ();

  //intialize the test result buffer as some test may not run because of the device not being available
  for (CoreIndex = 0; CoreIndex < MAX_TGL_CORE_COUNT; CoreIndex++) {
    if (( 0 == UniqueCoreList[CoreIndex]) //only BSP will be have processor number 0
        && (CoreIndex != mBspApicId >> 1))
    {
      FUSA_LIB_STATUS LibStatus;
      LibStatus = FusaTestAndReporting (
                    NULL,
                    FusaTestNumCpu0Idi + CoreIndex,
                    5U,
                    &(pFusaTestResult[CoreIndex])
                    );
      ASSERT(LibStatus == FusaNoError);
    }
  }

  CoreIdiTestParam.CoreNumber = mBspApicId >> 1;
  CoreIdiTestParam.pFusaTestResult = &(pFusaTestResult[CoreIdiTestParam.CoreNumber]);
  FspDxCheckCoreIdiInternal (&CoreIdiTestParam); //BSP runnig the test
  DumpResults (CoreIdiTestParam.pFusaTestResult);

  //for TGL as UP, the CoreIndex should max/valid at MAX_TGL_CORE_COUNT-1
  for (CoreIndex = 0; CoreIndex < MAX_TGL_CORE_COUNT; CoreIndex++)
  {
    if (UniqueCoreList[CoreIndex] > 0)
    {
      DEBUG ((DEBUG_INFO, "UniqueCoreList[%d] = %d\n", CoreIndex, UniqueCoreList[CoreIndex] ));
      CoreIdiTestParam.CoreNumber = CoreIndex;
      CoreIdiTestParam.pFusaTestResult = &(pFusaTestResult[CoreIdiTestParam.CoreNumber]);

      RunAtAp (
        (AP_PROCEDURE) FspDxCheckCoreIdiInternal,
        UniqueCoreList[CoreIndex],
        &CoreIdiTestParam
        );

      DumpResults (CoreIdiTestParam.pFusaTestResult);
    }
  }
}
