/** @file
  This file Implement the public interface for  Fusa
   E2E CTC implementation

@copyright
  INTEL CONFIDENTIAL
  Copyright 2019 - 2020 Intel Corporation.

  The source code contained or described herein and all documents related to the
  source code ("Material") are owned by Intel Corporation or its suppliers or
  licensors. Title to the Material remains with Intel Corporation or its suppliers
  and licensors. The Material may contain trade secrets and proprietary and
  confidential information of Intel Corporation and its suppliers and licensors,
  and is protected by worldwide copyright and trade secret laws and treaty
  provisions. No part of the Material may be used, copied, reproduced, modified,
  published, uploaded, posted, transmitted, distributed, or disclosed in any way
  without Intel's prior express written permission.

  No license under any patent, copyright, trade secret or other intellectual
  property right is granted to or conferred upon you by disclosure or delivery
  of the Materials, either expressly, by implication, inducement, estoppel or
  otherwise. Any license under such intellectual property rights must be
  express and approved by Intel in writing.

  Unless otherwise agreed by Intel in writing, you may not remove or alter
  this notice or any other notice embedded in Materials by Intel or
  Intel's suppliers or licensors in any way.

  This file contains an 'Intel Peripheral Driver' and is uniquely identified as
  "Intel Reference Module" and is licensed for Intel CPUs and chipsets under
  the terms of your license agreement with Intel or your vendor. This file may
  be modified by the user, subject to additional terms of the license agreement.

@par Specification Reference:
**/
#include <Uefi/UefiBaseType.h>
#include <Library/BaseLib.h>
#include <Library/DebugLib.h>
#include <Library/BaseMemoryLib.h>
#include <Library/IoLib.h>
#include <Library/PcdLib.h>
#include <Uefi/UefiMultiPhase.h>
#include <Pi/PiBootMode.h>
#include <Pi/PiHob.h>
#include <Library/HobLib.h>
#include <PiPei.h>
#include <Ppi/MpServices.h>
#include <Library/PeiServicesLib.h>
#include <Library/LocalApicLib.h>
#include <Library/PmcLib.h>
#include <Library/MemoryAllocationLib.h>
#include "PeiFusaPrivateLibInternal.h"
#include "PeiFusaResultReportingLib.h"
#include "PeiFusaE2eCtcLibInternal.h"

UINT8 *gpTestData = NULL;

//map unique core to processor number used by MP services, processor number 0 is being ignored as it refers to the BSP
//use array size of 128 so that we do not need to do array index check, although we expect the initial APIC ID max at
//(MAX_TGL_CORE_COUNT*2) -1
STATIC UINT32                   mUniqueCoreList[128];

extern EFI_GUID                 gEfiPeiMpServicesPpiGuid;
STATIC CONST EFI_PEI_SERVICES   **mPeiServices;
STATIC EFI_PEI_MP_SERVICES_PPI  *mMpServicesPpi = NULL;


/**
  To retrieve  mUniqueCoreList

  @retval mUniqueCoreList
**/
CONST UINT32 *
UniqueCoreListGet (
  VOID
  )
{
  return mUniqueCoreList;
}

/**
  To execute routine "Routine" with parameters "Param" at AP
  indicated by ProcessorNumber, in blocking mode.

  @param[in] Routine - routine to be executed
  @param[in] ProcessorNumber - target AP processor number
                         enumerated by the MP services
  @param[in] Param - parameters to be used by the said routine

**/
VOID
RunAtAp (
  IN AP_PROCEDURE Routine,
  IN UINT32       ProcessorNumber,
  IN VOID         *Param
  )
{
  EFI_STATUS Status;
  Status = mMpServicesPpi->StartupThisAP (
                             mPeiServices,
                             mMpServicesPpi,
                             (EFI_AP_PROCEDURE) Routine,
                             ProcessorNumber,
                             0,
                             Param);

  DEBUG ((DEBUG_INFO, "AP number %d calling status = 0x%x\n", ProcessorNumber, Status));
}

/**
  To populate the mUniqueCoreList array with the processor
  number used for multi-processor calling. Array index implies
  core number. Array value zero implies related core being not
  available. This list does not track the BSP because BSP core
  is always available and tested without MP calling.

  @param[in] PeiServices

  @return error status if MpService PPI was not installed

**/
STATIC
EFI_STATUS
CoreInfoPopulate (
    IN  CONST EFI_PEI_SERVICES  **PeiServices
    )
{
  UINTN NumberOfProcessors;
  UINTN NumberOfEnabledProcessors;
  UINT32 ProcessorNumber;
  EFI_STATUS Status;
  EFI_PROCESSOR_INFORMATION       ProcessorInfoBuffer[MAX_TGL_CORE_COUNT*2];

  //store for later RunAtAp usage
  mPeiServices = PeiServices;

  // Locate CpuMpCpu MpService Ppi
  Status = PeiServicesLocatePpi (
             &gEfiPeiMpServicesPpiGuid,
             0,
             NULL,
             (VOID **) &mMpServicesPpi
             );

  if (EFI_SUCCESS != Status) {
    ASSERT_EFI_ERROR (Status);
  } else {
    mMpServicesPpi->GetNumberOfProcessors (
                      PeiServices,
                      mMpServicesPpi,
                      &NumberOfProcessors,
                      &NumberOfEnabledProcessors
                      );

    for (ProcessorNumber = 0; ProcessorNumber < NumberOfProcessors; ProcessorNumber++) {
      Status = mMpServicesPpi->GetProcessorInfo (
                                 PeiServices,
                                 mMpServicesPpi,
                                 ProcessorNumber,
                                 &(ProcessorInfoBuffer[ProcessorNumber])
                                 );
      ASSERT_EFI_ERROR (Status);
    }

    SetMem ((UINT8 *)mUniqueCoreList, sizeof(mUniqueCoreList), 0 );

    if (NumberOfProcessors > MAX_TGL_CORE_COUNT*2) {
      //Multiplied by 2 for SMT support
      //it is not expected the supported TGL to have threads exceeding MAX_TGL_CORE_COUNT*2
      //simply cap it at the supported count to avoid unnecessary array index check
      NumberOfProcessors = MAX_TGL_CORE_COUNT*2;
      ASSERT (FALSE);
    }

    //search for unique core, ProcessorNumber 0 is BSP so we skip it
    for (ProcessorNumber = 1; ProcessorNumber < NumberOfProcessors; ProcessorNumber++)
    {
      UINT8 CoreIndex = (UINT8) (ProcessorInfoBuffer[ProcessorNumber].ProcessorId) >> 1;
      //skip the SMT sharing the same core with BSP
      if (CoreIndex != (UINT8) (ProcessorInfoBuffer[0].ProcessorId) >> 1)
      {
        //if there were 2 active SMT belong to the same core, then we only remember the last one
        mUniqueCoreList[CoreIndex] = ProcessorNumber;
        DEBUG ((DEBUG_INFO, "UniqueCoreList[%d] = %d\n", CoreIndex, ProcessorNumber));
      }
    }
    //now all non-zero UniqueCoreList[] contains the unique Cpu number not the same as BSP
  }

  return Status;
}

#define R_PMC_PWRM_FUSA_STS_CTL                     0x1F30U
#define B_PMC_PWRM_FUSA_STS_CTL_DIAGTEST_EN         BIT2
#define B_PMC_PWRM_FUSA_STS_CTL_DIAGTEST_PCHMODE    BIT1
#define B_PMC_PWRM_FUSA_STS_CTL_DIAGTEST_FEATURE_EN BIT0

/**
  Check if the system is under Fusa diagnostic mode

  @note Bootloader does not rely on this routine for the
        diagnostic mode detection but the existence of the
        gSiFusaInfoGuid HOB

  @retval TRUE if system is under Fusa diagnostic mode
  @retval FALSE otherwise

**/
BOOLEAN
FspDxDiagnosticModeGet (
  VOID
  )
{
  BOOLEAN FusaDiagTestMode;
  UINT32  FusaStsCtl;
  UINT32  PwrmBaseAddress = PmcGetPwrmBase ();

  FusaStsCtl = MmioRead32 ((UINTN) PwrmBaseAddress + R_PMC_PWRM_FUSA_STS_CTL);
  DEBUG ((DEBUG_INFO, "DIAGTEST_FEATURE_EN: %d\nDIAGTEST_EN: %d\nDIAGTEST_PCHMODE: %d\n",
          FusaStsCtl & B_PMC_PWRM_FUSA_STS_CTL_DIAGTEST_FEATURE_EN,
          (FusaStsCtl & B_PMC_PWRM_FUSA_STS_CTL_DIAGTEST_EN) >> 2,
          (FusaStsCtl & B_PMC_PWRM_FUSA_STS_CTL_DIAGTEST_PCHMODE) >> 1));

  if (((FusaStsCtl & B_PMC_PWRM_FUSA_STS_CTL_DIAGTEST_FEATURE_EN) != 0)
      &&((FusaStsCtl & B_PMC_PWRM_FUSA_STS_CTL_DIAGTEST_EN) != 0)
      &&((FusaStsCtl & B_PMC_PWRM_FUSA_STS_CTL_DIAGTEST_PCHMODE) == 0))
  {
    FusaDiagTestMode = TRUE;
  } else {
    FusaDiagTestMode = FALSE;
  }

  DEBUG ((DEBUG_INFO, "Diagnostic Mode detected is %d\n", FusaDiagTestMode));

  return FusaDiagTestMode;
}

/**
  Init and Install Fusa Info Hob

  @retval EFI_OUT_OF_RESOURCES if the HOB resource allocation
          fails
  @retval EFI_SUCCESS otherwise
**/
EFI_STATUS
InstallFusaInfoHob (
  VOID
  )
{
  EFI_STATUS        Status;
  FUSA_INFO_HOB     FusaInfo;
  FUSA_INFO_HOB     *pFusaInfoHob;

  if (FspDxDiagnosticModeGet ()) {
    ZeroMem ( &FusaInfo, sizeof (FusaInfo));
    FusaInfo.Version = FUSA_INFO_VERSION;

    //
    // Create HOB for Fusa INFO
    //
    pFusaInfoHob = BuildGuidDataHob (
                       &gSiFusaInfoGuid,
                       &FusaInfo,
                       sizeof (FUSA_INFO_HOB)
                       );
    if (NULL == pFusaInfoHob) {
      Status = EFI_OUT_OF_RESOURCES;
    } else {
      Status = EFI_SUCCESS;
    }
  } else {
    Status = EFI_SUCCESS;
  }

  return Status;
}

/**
  Main check the checker routine that will perform all the check
  the checker test (exclude MemSS CTC) and update the test
  result HOB.

  If systematic fault exists in such the MpServices PPI was not
  installed, the test result HOB is still being produced with
  zero content from this test, which the bootloader stage can
  detect the issue.

  @note This routine shall be called before BIOS_DONE or CPU
        PRMRR setup, or another word, prior to CpuInit(). Some
        of the test are using memory reference and thus it shall
        be called after system memory is avaialable.

  @pre  MpService PPI was installed

  @param[in] PeiServices    Pointer to PEI Services Table

**/
VOID
FspDxCheck (
  IN  CONST EFI_PEI_SERVICES  **PeiServices
  )
{
  EFI_HOB_GUID_TYPE      *GuidHob;
  FUSA_INFO_HOB        *FusaInfo;
  FUSA_TEST_RESULT     *TestResult;
  EFI_STATUS Status;

  GuidHob = NULL;
  FusaInfo = NULL;

  GuidHob = GetFirstGuidHob (&gSiFusaInfoGuid);
  if (GuidHob != NULL) {
    FusaInfo = (FUSA_INFO_HOB *) GET_GUID_HOB_DATA (GuidHob);
  }

  if (FusaInfo != NULL) {
    TestResult = FusaInfo->FspDxCtcTestResult;

    // dump to see if any bank status seen.
    // by right there shall not be any existing machine check error
    // or it could cause the CTC test to misbehave.
    McaBankStatusDumpAll ();
    McaBankStatusClearAll ();

    gpTestData = (UINT8 *) AllocatePages (EFI_SIZE_TO_PAGES (1028 * 1024));
    if (gpTestData == NULL) {
      ASSERT(FALSE);
      Status = EFI_OUT_OF_RESOURCES;
    } else {
      //init
      Status = CoreInfoPopulate (PeiServices);
    }

    if (EFI_SUCCESS == Status) {
      FspDxCheckCoreIdi (&(TestResult[FusaTestNumCpu0Idi]) );
      FspDxCheckLlc (&(TestResult[FusaTestNumCboSlice0Ingress]) );
      FspDxCheckDip (&(TestResult[FusaTestNumDip]));
      FspDxCheckIoPort (&(TestResult[FusaTestNumIop]));
      FspDxCheckOpiLink (&(TestResult[FusaTestNumOpiLinkIosfData]));
      FspDxCheckMemoryMbist(&(TestResult[FusaTestNumMc0Mbist]));
      FspDxCheckCoreArrayMbist (&(TestResult[FusaTestNumCpu0Mbist]), PeiServices );

      //dump to see if any bank status seen.
      McaBankStatusDumpAll ();
      McaBankStatusClearAll ();
    }
  } else {
    //expect the existence of the HOB
    ASSERT (FALSE);
  }
}

