/** @file
  This file provides Fusa LLC E2E CTC implementation

@copyright
  INTEL CONFIDENTIAL
  Copyright 2020 Intel Corporation.

  The source code contained or described herein and all documents related to the
  source code ("Material") are owned by Intel Corporation or its suppliers or
  licensors. Title to the Material remains with Intel Corporation or its suppliers
  and licensors. The Material may contain trade secrets and proprietary and
  confidential information of Intel Corporation and its suppliers and licensors,
  and is protected by worldwide copyright and trade secret laws and treaty
  provisions. No part of the Material may be used, copied, reproduced, modified,
  published, uploaded, posted, transmitted, distributed, or disclosed in any way
  without Intel's prior express written permission.

  No license under any patent, copyright, trade secret or other intellectual
  property right is granted to or conferred upon you by disclosure or delivery
  of the Materials, either expressly, by implication, inducement, estoppel or
  otherwise. Any license under such intellectual property rights must be
  express and approved by Intel in writing.

  Unless otherwise agreed by Intel in writing, you may not remove or alter
  this notice or any other notice embedded in Materials by Intel or
  Intel's suppliers or licensors in any way.

  This file contains an 'Intel Peripheral Driver' and is uniquely identified as
  "Intel Reference Module" and is licensed for Intel CPUs and chipsets under
  the terms of your license agreement with Intel or your vendor. This file may
  be modified by the user, subject to additional terms of the license agreement.

@par Specification Reference:
**/

#include <Uefi/UefiBaseType.h>
#include <Library/BaseLib.h>
#include <Library/DebugLib.h>
#include <Library/BaseMemoryLib.h>
#include <Library/LocalApicLib.h>

#include "PeiFusaE2eCtcLibInternal.h"
#include "PeiFusaPrivateLibInternal.h"
#include "PeiFusaResultReportingLib.h"


extern UINT8 *gpTestData;

STATIC UINT32 mBspApicId = 0;
STATIC UINT32 mCurrentThreadApicId = 0;

/**
  Error injection and corresponding traffic routine for LLC Data

  @param[in] IpIndex - IP index vale for MSR MSR_PERRINJ_AT_IP
  @param[in] InjectionPayload - Payload value for MSR MSR_PERRINJ_CTRL

  @return injection control value after traffic generation
**/

STATIC
UINT32
InjectionAndTrafficLlcData(
  IN UINT32 IpIndex,
  IN UINT32 InjectionPayload
  )
{
  UINT32 Address;
  UINT32 DataAddress = (UINT32) gpTestData;

  _asm
  {
    lea eax,  InjectionAndTrafficLlcData
    mov Address, eax

    //injection
    //update IpIndex
    mov     edx, 0
    mov     eax, IpIndex
    mov     ecx, MSR_PERRINJ_AT_IP
    wrmsr
    //update InjectionPayload
    mov     eax, InjectionPayload
    mov     ecx, MSR_PERRINJ_CTRL
    wrmsr

    //traffic
    mov ecx, 0x4000
    mov eax, DataAddress
    LLCDATA_DUMMY_DATA_WRITE:
    mov DWORD PTR [eax], 0xdeadbeef
    add eax, 64
    loop LLCDATA_DUMMY_DATA_WRITE
    mov ecx, 0x4000
    mov eax, DataAddress
    LLCDATA_DUMMY_DATA_READ:
    prefetcht2 BYTE PTR [eax+256]
    add edx, DWORD ptr [eax]
    add eax, 64
    loop LLCDATA_DUMMY_DATA_READ
    wbinvd
    nop

    //override the value after injection because LLC control will not flip
    mov     ecx, MSR_PERRINJ_CTRL
    xor eax, eax
    xor edx, edx
    wrmsr
  }

  if (mCurrentThreadApicId == mBspApicId) {
    DEBUG ((DEBUG_INFO, "LLC Test control value will not flip, override it with 0 \n"));
    DEBUG ((DEBUG_INFO, "TrafficLlc code address is at 0x%x\n", Address));
  }

  return 0;
}

/**
  Error injection and corresponding traffic routine for LLC Tag

  @param[in] IpIndex - IP index vale for MSR MSR_PERRINJ_AT_IP
  @param[in] InjectionPayload - Payload value for MSR MSR_PERRINJ_CTRL

  @return injection control value after traffic generation
**/
STATIC
UINT32
InjectionAndTrafficLlcTag(
  IN UINT32 IpIndex,
  IN UINT32 InjectionPayload
  )
{
  UINT32 DataAddress = (UINT32) gpTestData;

  _asm
  {
    //flush 1MB of data
    mov ecx, 0x4000
    mov eax, DataAddress
    LLCTEST_DATA_FLUSH:
    clflush BYTE PTR [eax]
    add eax, 64
    loop LLCTEST_DATA_FLUSH

    //injection
    mov     edx, 0
    mov     eax, IpIndex
    mov     ecx, MSR_PERRINJ_AT_IP
    wrmsr
    mov     eax, InjectionPayload
    mov     ecx, MSR_PERRINJ_CTRL
    wrmsr

    //traffic
    mov ecx, 0x4000
    mov eax, DataAddress
    LLCTAG_DUMMY_DATA_READ:
    prefetcht2 BYTE PTR [eax+256]
    add edx, DWORD ptr [eax]
    add eax, 64
    loop LLCTAG_DUMMY_DATA_READ
    mfence
    nop

    //override the value after injection because LLC control will not flip
    mov ecx, MSR_PERRINJ_CTRL
    xor eax, eax
    xor edx, edx
    wrmsr
  }

  return 0;
}

/**
  Generic CTC routine for LLC Tag

  @note Control the debug dump to be within BSP as dumping
        within the AP will cause hang

  @param[in] InjectionPayload - Payload value for MSR MSR_PERRINJ_CTRL
  @param[in] CoreNumber - core number aka CBO number
  @param[in] CheckNum - check number assigned to the check
  @param[in,out] pFusaTestResult - pointer to the test result
        buffer for result to be updated

  @return FUSA_LIB_STATUS
**/
STATIC
FUSA_LIB_STATUS
LlcTagTestGeneric (
  IN UINT32               InjectionPayload,
  IN UINT8                CoreNumber,
  IN UINT8                CheckNum,
  IN OUT FUSA_TEST_RESULT *pFusaTestResult
  )
{
  FUSA_LIB_STATUS LibStatus;
  UINT32 IpIndex = (CBO0_INGRESS + CoreNumber);
  UINT8 McaBankNum = MCA_CBO0 + CoreNumber;
  UINT64 ExpectedMcaStatus = 0x1102ULL;
  UINT64 MacStatusMask = 0x1103ULL;

  LibStatus = FusaMsrCtcTestGeneric (
                IpIndex,
                InjectionPayload,
                InjectionAndTrafficLlcTag,
                McaBankNum,
                MacStatusMask,
                ExpectedMcaStatus,
                CheckNum,
                pFusaTestResult,
                (mBspApicId >> 1) == CoreNumber
                );

  return LibStatus;
}

/**
  Generic CTC routine for LLC Data

  @note Control the debug dump to be within BSP as dumping
        within the AP will cause hang

  @param[in] InjectionPayload - Payload value for MSR MSR_PERRINJ_CTRL
  @param[in] CoreNumber - core number aka CBO number
  @param[in] CheckNum - check number assigned to the check
  @param[in,out] pFusaTestResult - pointer to the test result
        buffer for result to be updated

  @return FUSA_LIB_STATUS
**/
STATIC
FUSA_LIB_STATUS
LlcDataTestGeneric (
  IN UINT32               InjectionPayload,
  IN UINT8                CoreNumber,
  IN UINT8                CheckNum,
  IN OUT FUSA_TEST_RESULT *pFusaTestResult
  )
{
  FUSA_LIB_STATUS LibStatus;
  UINT32 IpIndex = (CBO0_INGRESS + CoreNumber);
  UINT8 McaBankNum = MCA_CBO0 + CoreNumber;
  UINT64 ExpectedMcaStatus = 0x1102ULL;
  UINT64 MacStatusMask = 0x1103ULL;

  LibStatus = FusaMsrCtcTestGeneric (
                IpIndex,
                InjectionPayload,
                InjectionAndTrafficLlcData,
                McaBankNum,
                MacStatusMask,
                ExpectedMcaStatus,
                CheckNum,
                pFusaTestResult,
                (mBspApicId >> 1) == CoreNumber
                );

  return LibStatus;
}

/**
  Perform the permutation needed for Core IDI CTC

  @param[in,out] pFusaTestResult - result buffer where the test
       result to be updated to
  @return FUSA_LIB_STATUS
**/
STATIC
FUSA_LIB_STATUS
LlcCtc (
  IN OUT FUSA_TEST_RESULT *pFusaTestResult
  )
{
  FUSA_LIB_STATUS LibStatus;

  mCurrentThreadApicId = GetInitialApicId ();

  LibStatus = LlcTagTestGeneric (
                0x42005, // uncorrectable tag error
                (UINT8) (mCurrentThreadApicId >> 1),
                0U, // check number
                pFusaTestResult
                );
  ASSERT (LibStatus == FusaNoError);

  if (mCurrentThreadApicId == mBspApicId) {
    McaBankStatusDumpAll();
  }

  LibStatus |= LlcDataTestGeneric (
                 0xA005, // correctable data error
                 (UINT8) (mCurrentThreadApicId >> 1),
                 1U, // check number
                 pFusaTestResult
                 );
  ASSERT (LibStatus == FusaNoError);

  if (mCurrentThreadApicId == mBspApicId) {
    McaBankStatusDumpAll();
  }

  LibStatus |= LlcDataTestGeneric (
                 0x1A005, // uncorrectable data error
                 (UINT8) (mCurrentThreadApicId >> 1),
                 2U, // check number
                 pFusaTestResult
                 );
  ASSERT (LibStatus == FusaNoError);

  if (mCurrentThreadApicId == mBspApicId) {
    McaBankStatusDumpAll ();
  }

  return LibStatus;
}

/// Structure of the parameter passed into the LLC test routine
typedef struct
{
  FUSA_TEST_RESULT *pFusaTestResult;
  UINT32           CoreNumber;
} LLC_TEST_PARAM;

/**
  LLC Error E2E CTC internal main routine

  @param[in,out] pLlcTestParam - conttain result buffer
       where the test result to be updated to beside core number
       information
**/
VOID
FspDxCheckLlcInternal (
  IN LLC_TEST_PARAM *pLlcTestParam
  )
{
  FUSA_LIB_STATUS LibStatus;
  FUSA_TEST_RESULT *pFusaTestResult;
  UINT32 CoreNumber;

  UINTN   Cr0;
  UINTN   Cr4;
  ///
  /// Enable XMM and cache
  ///
  Cr0 = AsmReadCr0 ();
  Cr0 |= BIT1;
  Cr0 &= ~BIT30; //make sure cache is not disabled
  AsmWriteCr0 (Cr0);

  Cr4 = AsmReadCr4 ();
  Cr4 |= (UINTN) (BIT9 | BIT10);
  AsmWriteCr4 (Cr4);

  pFusaTestResult = pLlcTestParam->pFusaTestResult;
  CoreNumber = pLlcTestParam->CoreNumber;

  LibStatus = FusaTestAndReporting (
                LlcCtc,
                FusaTestNumCboSlice0Ingress + CoreNumber,
                3U,
                pFusaTestResult
                );
  ASSERT (LibStatus == FusaNoError);
}

/**
  Perform LLC ECC Error E2E CTC.
  The test targets all active unique CBO (which in turn maps to
  unique core). For CBO/core not detected, their test result is
  updated to FUSA_TEST_DEVICE_NOTAVAILABLE

  @note The test expects to be run within Memory Type =
        Writeback (MTRR Encoding = 06h), which is the usual
        case of the memory type covering the code/data region
        of the bootloader.

  @param[out] pFusaTestResult - pointer to test result
                         buffer for test from
                         FusaTestNumCboSlice0Ingress to
                         FusaTestNumCboSlice7Ingress

**/
VOID
FspDxCheckLlc (
  OUT FUSA_TEST_RESULT *pFusaTestResult
  )
{
  UINT32 CoreIndex;
  LLC_TEST_PARAM LlcTestParam;
  CONST UINT32 *UniqueCoreList = UniqueCoreListGet ();

  DEBUG ((DEBUG_INFO, "FspDxCheckLlc\n"));

  mBspApicId = GetInitialApicId ();

  //intialize the test result buffer as some test may not run because of the device not being available
  for (CoreIndex = 0; CoreIndex < MAX_TGL_CORE_COUNT; CoreIndex++)
  {
    if (( 0 == UniqueCoreList[CoreIndex]) //only BSP will be have processor number 0
        && (CoreIndex != mBspApicId >> 1))
    {
      FUSA_LIB_STATUS LibStatus;
      LibStatus = FusaTestAndReporting (
                    NULL,
                    FusaTestNumCboSlice0Ingress + CoreIndex,
                    3U,
                    &(pFusaTestResult[CoreIndex])
                    );
      ASSERT(LibStatus == FusaNoError);
    }
  }

  LlcTestParam.CoreNumber = mBspApicId >> 1;
  LlcTestParam.pFusaTestResult = &(pFusaTestResult[LlcTestParam.CoreNumber]);

  FspDxCheckLlcInternal (&LlcTestParam); //BSP runnig the test
  DumpResults (LlcTestParam.pFusaTestResult);

  //for TGL as UP, the CoreIndex should max/valid at MAX_TGL_CORE_COUNT-1
  for (CoreIndex = 0; CoreIndex < MAX_TGL_CORE_COUNT; CoreIndex++) {
    if (UniqueCoreList[CoreIndex] > 0) {
      DEBUG ((DEBUG_INFO, "UniqueCoreList[%d] = %d\n", CoreIndex, UniqueCoreList[CoreIndex] ));

      LlcTestParam.CoreNumber = CoreIndex;
      LlcTestParam.pFusaTestResult = &(pFusaTestResult[LlcTestParam.CoreNumber]);

      RunAtAp (
        (AP_PROCEDURE) FspDxCheckLlcInternal,
        UniqueCoreList[CoreIndex],
        &LlcTestParam
        );
      DumpResults (LlcTestParam.pFusaTestResult);
    }
  }
}


