/** @file
  Header file for Fusa Private Lib Internal functions that
  perform Fusa HW access and CTC generic routines.

@copyright
  INTEL CONFIDENTIAL
  Copyright 2019 Intel Corporation.

  The source code contained or described herein and all documents related to the
  source code ("Material") are owned by Intel Corporation or its suppliers or
  licensors. Title to the Material remains with Intel Corporation or its suppliers
  and licensors. The Material may contain trade secrets and proprietary and
  confidential information of Intel Corporation and its suppliers and licensors,
  and is protected by worldwide copyright and trade secret laws and treaty
  provisions. No part of the Material may be used, copied, reproduced, modified,
  published, uploaded, posted, transmitted, distributed, or disclosed in any way
  without Intel's prior express written permission.

  No license under any patent, copyright, trade secret or other intellectual
  property right is granted to or conferred upon you by disclosure or delivery
  of the Materials, either expressly, by implication, inducement, estoppel or
  otherwise. Any license under such intellectual property rights must be
  express and approved by Intel in writing.

  Unless otherwise agreed by Intel in writing, you may not remove or alter
  this notice or any other notice embedded in Materials by Intel or
  Intel's suppliers or licensors in any way.

  This file contains an 'Intel Peripheral Driver' and is uniquely identified as
  "Intel Reference Module" and is licensed for Intel CPUs and chipsets under
  the terms of your license agreement with Intel or your vendor. This file may
  be modified by the user, subject to additional terms of the license agreement.

@par Specification Reference:
**/
#ifndef _FUSA_PRIVATE_LIB_INTERNAL_H_
#define _FUSA_PRIVATE_LIB_INTERNAL_H_

#include "PeiFusaResultReportingLib.h"

///MCA Bank numbers
#define TGL_CBO0_INDEX 8
#define MCA_IFU 0
#define MCA_DCU 1
#define MCA_DTLB 2
#define MCA_MLC 3
#define MCA_SEC 4
#define MCA_MEE1 5
#define MCA_IOP 6
#define MCA_PCU 7
#define MCA_CBO0 TGL_CBO0_INDEX
#define MCA_CBO1 (TGL_CBO0_INDEX+1)
#define MCA_CBO2 (TGL_CBO0_INDEX+2)
#define MCA_CBO3 (TGL_CBO0_INDEX+3)
#define MCA_CBO4 (TGL_CBO0_INDEX+4)
#define MCA_CBO5 (TGL_CBO0_INDEX+5)
#define MCA_CBO6 (TGL_CBO0_INDEX+6)
#define MCA_CBO7 (TGL_CBO0_INDEX+7)

///IP index for the FuSa MSR MSR_PERRINJ_AT_IP
typedef enum {
  CORE = 1,
  IOP_CS0_IOSF_UP = 2,
  IOP_CS1_CMI = 3,
  IOP_CS2_CII = 4,
  IOP_CS4_RCH = 6,
  IOP_CS5_IOSF_DIP = 7,
  IOP_CS6_IOTRK  = 8,
  IOP_CS7_CMI_UP = 9,
  IOP_CS8_P2PD = 10,
  IOP_CS9_MSG_FIFO = 11,
  IOP_CS10_P2PA = 12,
  IOP_CS11_UP_COMPLETION = 13,
  DIP_IOSF_MCMD = 18,
  DIP_UP_CMD_ADDRESS = 19,
  DIP_CMI_CPL_DATA = 21,
  CBO0_INGRESS = 55,
  CBO1_INGRESS,
  CBO2_INGRESS,
  CBO3_INGRESS,
  CBO4_INGRESS,
  CBO5_INGRESS,
  CBO6_INGRESS,
  CBO7_INGRESS
} FUSA_IP_INDEX_NUMBER;

#define MSCOD_MASK 0xFFFF0000ULL
#define MCACOD_MASK 0xFFFFULL
#define MCASTATUS_VALID BIT63

///<"Index" - Tells uCode which CR should be targeted in the uncore. Thread-Scope
#define MSR_PERRINJ_AT_IP 0x107U
/**
  "Payload" - Data to be written (in the case of WRMSR instruction) or read (in the case of RDMSR instruction)
  to/from uncore CR.  Core-Scope
  */
#define MSR_PERRINJ_CTRL  0x108U

///Common register definition used by CTC with MCA_IOP usage
#define R_SA_MCHBAR_FUSA_MCA_REPORTING_EN_0_0_0_MCHBAR_IMPH                      0x6F30U
#define B_SA_MCHBAR_FUSA_MCA_REPORTING_EN_0_0_0_MCHBAR_IMPH_MCA_REPORTING_EN     BIT0

/**
  Check if it is a supported CPU for this library

  @note Not being used yet

  @retval TRUE - Supported
  @retval FALSE - Not Supported
**/
BOOLEAN
IsCpuFusaSupported (
  VOID
  );

/**
  Enable MCA on the targetted bank index for error reporting
  purpose and clear the rest. if TargetMcaBankNum is 0xFF
  (outside of the supported MCA bank count), then effectively
  disable all MCA reporting

  @param[in] TargetMcaBankNum - Target MCA bank number
**/
VOID
McaReportingEnable(
  IN UINT32 TargetMcaBankNum
  );

/**
  Read the related MCA bank status and compare against the
  expected McaStatus values (auto check for valid bit)

  @param[in] McaBankNum - MCA bank number
  @param[in] McaStatusMask - Mask value to be applied on
        McaStatus before comparison
  @param[in] ExpectedMcaStatus - Expected McaStatus value after
        masked.

  @retval  FUSA_TEST_PASS - if matches expected values
  @retval  FUSA_TEST_FAIL - if not matches expected values
**/

UINT8
McaBankStatusCheck(
  IN UINT8  McaBankNum,
  IN UINT64 McaStatusMask,
  IN UINT64 ExpectedMcaStatus
  );

/**
  Clear all MCA bank status
**/
VOID
McaBankStatusClearAll(
  VOID
  );

/**
  Dump all MCA bank status in debug build
**/
VOID
McaBankStatusDumpAll(
  VOID
  );

/**
  Clear the related MCA bank status, address and misc details

  @param[in] McaBankNum - MCA bank number
**/
VOID
McaBankInfoClear(
  IN UINT8 McaBankNum
  );

/**
  Dump the related MCA bank status, address and misc details in
  debug build

  @param[in] McaBankNum - MCA bank number
**/
VOID
McaBankInfoDump(
  IN UINT8 McaBankNum
  );

/**
  Write the parity error injection control payload into the Fusa
  IP indexed by the IpIndex

  @param[in] IpIndex - IP Index for the Fusa MSR for the error
        injection
  @param[in] InjectionControlPayload - Injection control payload
        for the error injection

  @note Once the Fusa MSR is used, the ucode will disable the
        prefetcher for preventing from messing with the CTC
        mechanism. This step will not be undone because it is
        not expected the system to boot to OS without a reset
        for the CTC path.

**/
VOID
FuSaMsrWrite (
  IN UINT32 IpIndex,
  IN UINT32 InjectionControlPayload
  );

/**
  Read the parity error injection control payload status from
  the Fusa IP indexed by the IpIndex

  @param[in] IpIndex - IP Index for the Fusa MSR for the error
        injection

  @retval  Parity error injection control payload status
**/
UINT32
FuSaMsrRead (
  IN UINT32 IpIndex
  );

/**
  Get MCH Base address

  @param  MchBaseAddress - MCH Base address

  @retval  fusaDeviceError - for unexpected address value
  @retval  fusaNoError - no error otherwise
**/
FUSA_LIB_STATUS
MchBarAddressGet (
  OUT UINTN *MchBaseAddress
  );

/**
  Get DMI Bar address

  @param  DmiBaseAddress - DMI Base address

  @retval  fusaDeviceError - for unexpected address value
  @retval  fusaNoError - no error otherwise
**/
FUSA_LIB_STATUS
DmiBarAddressGet (
  OUT UINTN *DmiBaseAddress
  );

/**
  Get GTTMM Base address

  @param  GttMmBaseAdr - GTTMM Base address

  @retval  fusaDeviceError - for unexpected address value
  @retval  fusaNoError - no error otherwise
**/
FUSA_LIB_STATUS
GttMmBarAddressGet (
  OUT UINTN *GttMmBaseAdr
  );

/**
  Get GM Base address

  @param  GmBaseAdr - GM Base address

  @retval  fusaDeviceError - for unexpected address value
  @retval  fusaNoError - no error otherwise
**/
FUSA_LIB_STATUS
GmBarAddressGet (
  OUT UINTN *GmBaseAdr
  );

/**
  Get PCH Device D31 F0 Base address

  @param  PchD31F0BaseAddress - PCH Device D31 F0 Base address

  @note   VOID return because no error possible
**/
VOID
PchD31F0BaseAddressGet (
  OUT UINTN *PchD31F0BaseAddress
  );

/**
 * Function pointer for the generic injection test traffic
 * routine
 */
//typedef VOID (* const FUSA_TRAFFIC_FUNC)( VOID );

/**
 * Function pointer for the generic injection test traffic
 * routine
 */
typedef UINT32 (* const FUSA_INJECTION_TRAFFIC_FUNC)( UINT32 IpIndex, UINT32 InjectionPayload);

/**
  Generic error injection flow using Fusa MSR

  @param[in] IpIndexInput - IP Index for the Fusa MSR for the
        error injection
  @param[in] InjectionPayload - Injection payload for the error
      injection
  @param[in] InjectionAndTraffic - function pointer for
        performing injection and related traffic
  @param[in] McaBankNum - The MCA Bank that reflect the test
        status
  @param[in] McaStatusMask - MCA status mask to be applied
        before comparing for the ExpectedMcaStatus
  @param[in] ExpectedMcaStatus - Expected MCA status value
  @param[in] CheckNum - check number assigned to this check
  @param[in,out] pFusaTestResult - pointer to the test result
        buffer for result to be updated
  @param[in] DebugFlag - to control the debug print. Main
        purpose is to allow AP to avoid the debug print that
        would cause hang.

  @retval  lib routine calling status from UpdateResults

  @note - combine injection and traffic in one routine so that
        there is not done through function calling and return in
        which the error itself may mess up with function return
        or calling.
**/
FUSA_LIB_STATUS
FusaMsrCtcTestGeneric (
  IN UINT32                       IpIndexInput,
  IN UINT32                       InjectionPayload,
  IN FUSA_INJECTION_TRAFFIC_FUNC  InjectionAndTraffic,
  IN UINT8                        McaBankNum,
  IN UINT64                       McaStatusMask,
  IN UINT64                       ExpectedMcaStatus,
  IN UINT8                        CheckNum,
  IN OUT FUSA_TEST_RESULT         *pFusaTestResult,
  IN BOOLEAN                      DebugFlag
  );

#endif // _FUSA_PRIVATE_LIB_INTERNAL_H_
