/** @file
  Source file for common Graphics Info Lib.

@copyright
  INTEL CONFIDENTIAL
  Copyright 2019 - 2020 Intel Corporation.

  The source code contained or described herein and all documents related to the
  source code ("Material") are owned by Intel Corporation or its suppliers or
  licensors. Title to the Material remains with Intel Corporation or its suppliers
  and licensors. The Material may contain trade secrets and proprietary and
  confidential information of Intel Corporation and its suppliers and licensors,
  and is protected by worldwide copyright and trade secret laws and treaty
  provisions. No part of the Material may be used, copied, reproduced, modified,
  published, uploaded, posted, transmitted, distributed, or disclosed in any way
  without Intel's prior express written permission.

  No license under any patent, copyright, trade secret or other intellectual
  property right is granted to or conferred upon you by disclosure or delivery
  of the Materials, either expressly, by implication, inducement, estoppel or
  otherwise. Any license under such intellectual property rights must be
  express and approved by Intel in writing.

  Unless otherwise agreed by Intel in writing, you may not remove or alter
  this notice or any other notice embedded in Materials by Intel or
  Intel's suppliers or licensors in any way.

  This file contains an 'Intel Peripheral Driver' and is uniquely identified as
  "Intel Reference Module" and is licensed for Intel CPUs and chipsets under
  the terms of your license agreement with Intel or your vendor. This file may
  be modified by the user, subject to additional terms of the license agreement.

@par Specification Reference:
**/
#include <Library/GraphicsInfoLib.h>
#include <Register/IgdRegs.h>
#include <Library/TimerLib.h>
#include <Library/PreSiliconEnvDetectLib.h>

/**
  This function will check if Bus Master and Memory access on 0:2:0 is enabled or not

  @retval TRUE          IGD Bus Master Access and Memory Space access is Enabled
  @retval FALSE         IGD Bus Master Access and Memory Space access is Disable
**/
BOOLEAN
IgfxCmdRegEnabled (
  VOID
)
{
  UINT16  IgfxCmdRegValue;

  IgfxCmdRegValue = (PciSegmentRead16 (PCI_SEGMENT_LIB_ADDRESS (SA_SEG_NUM, IGD_BUS_NUM, IGD_DEV_NUM, IGD_FUN_NUM, 0) + PCI_COMMAND_OFFSET));
  if ((IgfxCmdRegValue & EFI_PCI_COMMAND_BUS_MASTER) && (IgfxCmdRegValue & EFI_PCI_COMMAND_MEMORY_SPACE)) {
    return TRUE;
  }
  return FALSE;
}

/**
  GetIgdBusNumber: Get IGD Bus Number

  @retval PCI bus number for IGD
**/
UINT64
GetIgdBusNumber (
  VOID
  )
{
  return (UINT64)IGD_BUS_NUM;
}

/**
  GetIgdDevNumber: Get IGD Dev Number

  @retval PCI dev number for IGD
**/
UINT64
GetIgdDevNumber (
  VOID
  )
{
  return (UINT64)IGD_DEV_NUM;
}

/**
  GetIgdFunNumber: Get IGD Fun Number

  @retval PCI fun number for IGD
**/
UINT64
GetIgdFuncNumber (
  VOID
  )
{
  return (UINT64)IGD_FUN_NUM;
}

/**
  GetGttMmAdrOffset: Get GTTMMADR Offset value

  @retval PCI Offset for GTTMMADR
**/
UINT64
GetGttMmAdrOffset (
  VOID
  )
{
  return (UINT64)R_SA_IGD_GTTMMADR;
}

/**
  GetGmAdrOffset: Get GMADR Offset value

  @retval PCI Offset for GMADR
**/
UINT64
GetGmAdrOffset (
  VOID
  )
{
  return (UINT64)R_SA_IGD_GMADR;
}

/**
  GetIgdMsacOffset: Get IGD MSAC Offset value

  @retval PCI Offset for IGD Aperture size.
**/
UINT64
GetIgdMsacOffset (
  VOID
  )
{
  return (UINT64)R_SA_IGD_MSAC_OFFSET;
}

/**
  GetIgdDssmRefClockFreqValue: Read DSSM (Display Strap State) register to get CD Clock reference clock frequence value

  @param[in] GttMmAdr         IGD MMIO Base address value

  @retval IGD CD Clock Reference clock frequency value
**/
UINT32
GetIgdDssmRefClockFreqValue (
  IN  UINTN                GttMmAdr
  )
{
  return (UINT32) ((MmioRead32 ((UINTN) (GttMmAdr + R_SA_GTTMMADR_DSSM_OFFSET)) & B_SA_GTTMMADR_DSSM_REFERENCE_FREQ_MASK) >> B_SA_GTTMMADR_DSSM_REFERENCE_FREQ_OFFSET) ;
}

/**
  "Poll Status" for GT Readiness

  @param[in] Base            - Base address of MMIO
  @param[in] Offset          - MMIO Offset
  @param[in] Mask            - Mask
  @param[in] Result          - Value to wait for

  @retval EFI_SUCCESS          Wait Bit Successfully set
  @retval EFI_TIMEOUT          Timed out
**/
EFI_STATUS
PollGtReady (
  IN       UINT64                       Base,
  IN       UINT32                       Offset,
  IN       UINT32                       Mask,
  IN       UINT32                       Result
  )
{
  UINT32  GtStatus;
  UINT32  StallCount;

  StallCount = 0;

  ///
  /// Register read
  ///
  GtStatus = MmioRead32 ((UINTN) Base + Offset);

  while (((GtStatus & Mask) != Result) && (StallCount < GT_WAIT_TIMEOUT)) {
    ///
    /// 10 microSec wait
    ///
    MicroSecondDelay (10);
    StallCount += 10;

    GtStatus = MmioRead32 ((UINTN) Base + Offset);
  }

  if (StallCount < GT_WAIT_TIMEOUT) {
    return EFI_SUCCESS;
  } else {
    if (!IsSimicsEnvironment()) {
      ASSERT ((StallCount < GT_WAIT_TIMEOUT));
    }
    return EFI_TIMEOUT;
  }
}
