/** @file
  PEIM to initialize Early Display.

@copyright
  INTEL CONFIDENTIAL
  Copyright 2015 - 2019 Intel Corporation.

  The source code contained or described herein and all documents related to the
  source code ("Material") are owned by Intel Corporation or its suppliers or
  licensors. Title to the Material remains with Intel Corporation or its suppliers
  and licensors. The Material may contain trade secrets and proprietary and
  confidential information of Intel Corporation and its suppliers and licensors,
  and is protected by worldwide copyright and trade secret laws and treaty
  provisions. No part of the Material may be used, copied, reproduced, modified,
  published, uploaded, posted, transmitted, distributed, or disclosed in any way
  without Intel's prior express written permission.

  No license under any patent, copyright, trade secret or other intellectual
  property right is granted to or conferred upon you by disclosure or delivery
  of the Materials, either expressly, by implication, inducement, estoppel or
  otherwise. Any license under such intellectual property rights must be
  express and approved by Intel in writing.

  Unless otherwise agreed by Intel in writing, you may not remove or alter
  this notice or any other notice embedded in Materials by Intel or
  Intel's suppliers or licensors in any way.

  This file contains an 'Intel Peripheral Driver' and is uniquely identified as
  "Intel Reference Module" and is licensed for Intel CPUs and chipsets under
  the terms of your license agreement with Intel or your vendor. This file may
  be modified by the user, subject to additional terms of the license agreement.

@par Specification Reference:
**/

#include <Library/PeiGraphicsInitLib.h>
#include <Library/PeiDisplayInitLib.h>
#include <Ppi/SiPolicy.h>
#include <Library/PostCodeLib.h>
#include <Register/IgdRegs.h>
#include <Library/GpioNativePads.h>
#include <Library/GpioPrivateLib.h>
#include <Library/TimerLib.h>

/**
  ProgramDisplayWorkaround: Programs Display specific Workarounds

  @param[in]  GtPreMemConfig  - GRAPHICS_PEI_PREMEM_PREMEM_CONFIG to access the GtPreMemConfig related information

  @retval     EFI_SUCCESS     - Display workarounds done
**/
EFI_STATUS
ProgramDisplayWorkaround (
  IN   GRAPHICS_PEI_PREMEM_CONFIG      *GtPreMemConfig
  )
{
  return EFI_SUCCESS;
}

/**
  DisplayNativeGpioInit: Initialize the Display Native Gpio

  @param[in] GtPreMemConfig        - GRAPHICS_PEI_PREMEM_CONFIG to access the GtConfig related information

**/
VOID
DisplayNativeGpioInit (
  IN   GRAPHICS_PEI_PREMEM_CONFIG      *GtPreMemConfig
  )
{
  DEBUG ((DEBUG_INFO, "DisplayNativeGpioInit: Begin \n"));

  //
  // Enable shared and specific pins for eDP/MIPI
  //
  if (GtPreMemConfig->DdiConfiguration.DdiPortAConfig != DdiPortDisabled) {
    GpioSetNativePadByFunction (GPIO_FUNCTION_PANEL_AVDD_EN(0), 0);
    GpioSetNativePadByFunction (GPIO_FUNCTION_PANEL_BKLTEN(0), 0);
    GpioSetNativePadByFunction (GPIO_FUNCTION_PANEL_BKLTCTL(0), 0);
    if (GtPreMemConfig->DdiConfiguration.DdiPortAConfig == DdiPortEdp) {
     GpioSetNativePadByFunction (GPIO_FUNCTION_DDSP_HPD('A'), 0);
    } else if (GtPreMemConfig->DdiConfiguration.DdiPortAConfig == DdiPortMipiDsi) {
     GpioSetNativePadByFunction (GPIO_FUNCTION_MIPI_PANEL_RESET(0), 0);
    }
  }

  if (GtPreMemConfig->DdiConfiguration.DdiPortBConfig != DdiPortDisabled) {
    GpioSetNativePadByFunction (GPIO_FUNCTION_PANEL_AVDD_EN(1), 0);
    GpioSetNativePadByFunction (GPIO_FUNCTION_PANEL_BKLTEN(1), 0);
    GpioSetNativePadByFunction (GPIO_FUNCTION_PANEL_BKLTCTL(1), 0);
    if (GtPreMemConfig->DdiConfiguration.DdiPortBConfig == DdiPortEdp) {
     GpioSetNativePadByFunction (GPIO_FUNCTION_DDSP_HPD('B'), 0);
    } else if (GtPreMemConfig->DdiConfiguration.DdiPortBConfig == DdiPortMipiDsi) {
     GpioSetNativePadByFunction (GPIO_FUNCTION_MIPI_PANEL_RESET(1), 0);
    }
  }

  ///
  /// Enable DDSP_HPD pins for DP HotPlug
  ///
  if (GtPreMemConfig->DdiConfiguration.DdiPortAHpd) {
    GpioSetNativePadByFunction (GPIO_FUNCTION_DDSP_HPD('A'), 0);
  }
  if (GtPreMemConfig->DdiConfiguration.DdiPortBHpd) {
    GpioSetNativePadByFunction (GPIO_FUNCTION_DDSP_HPD('B'), 0);
  }
  if (GtPreMemConfig->DdiConfiguration.DdiPortCHpd) {
    GpioSetNativePadByFunction (GPIO_FUNCTION_DDSP_HPD('C'), 0);
  }
  if (GtPreMemConfig->DdiConfiguration.DdiPort1Hpd) {
    GpioSetNativePadByFunction (GPIO_FUNCTION_DDSP_HPD(1), 0);
  }
  if (GtPreMemConfig->DdiConfiguration.DdiPort2Hpd) {
    GpioSetNativePadByFunction (GPIO_FUNCTION_DDSP_HPD(2), 0);
  }
  if (GtPreMemConfig->DdiConfiguration.DdiPort3Hpd) {
    GpioSetNativePadByFunction (GPIO_FUNCTION_DDSP_HPD(3), 0);
  }
  if (GtPreMemConfig->DdiConfiguration.DdiPort4Hpd) {
    GpioSetNativePadByFunction (GPIO_FUNCTION_DDSP_HPD(4), 0);
  }

  ///
  /// Enable DDP CTRLCLK and CTRLDATA pins OR TBT RX and TX pins
  ///
  //
  // DDI Port A
  //
  if (GtPreMemConfig->DdiConfiguration.DdiPortADdc == DdiDdcEnable) {
    GpioEnableDpInterface (GpioDdpA);
  }
  //
  // DDI Port B
  //
  if (GtPreMemConfig->DdiConfiguration.DdiPortBDdc == DdiDdcEnable) {
    GpioEnableDpInterface (GpioDdpB);
  }
  //
  // DDI Port C
  //
  if (GtPreMemConfig->DdiConfiguration.DdiPortCDdc == DdiDdcEnable) {
    GpioEnableDpInterface (GpioDdpC);
  }
  //
  // DDI Port 1
  //
  if (GtPreMemConfig->DdiConfiguration.DdiPort1Ddc == DdiDdcEnable) {
    GpioEnableDpInterface (GpioDdp1);
  }
  //
  // DDI Port 2
  //
  if (GtPreMemConfig->DdiConfiguration.DdiPort2Ddc == DdiDdcEnable) {
    GpioEnableDpInterface (GpioDdp2);
  }
  //
  // DDI Port 3
  //
  if (GtPreMemConfig->DdiConfiguration.DdiPort3Ddc == DdiDdcEnable) {
    GpioEnableDpInterface (GpioDdp3);
  }
  //
  // DDI Port 4
  //
  if (GtPreMemConfig->DdiConfiguration.DdiPort4Ddc == DdiDdcEnable) {
    GpioEnableDpInterface (GpioDdp4);
  }

  DEBUG ((DEBUG_INFO, "DisplayNativeGpioInit: End \n"));
}

/**
  Poll Run busy clear

  @param[in] Base    - Base address of MMIO
  @param[in] Timeout - Timeout value in microsecond

  @retval TRUE       - Run Busy bit is clear
  @retval FALSE      - Run Busy bit is still set
**/
BOOLEAN
PollRunBusyClear (
  IN    UINT64           Base,
  IN    UINT32           Timeout
  )
{
  UINT32  Value;
  BOOLEAN Status = FALSE;

  //
  // Make timeout an exact multiple of 10 to avoid infinite loop
  //
  if ((Timeout) % 10 != 0) {
    Timeout = (Timeout) + 10 - ((Timeout) % 10);
  }

  while (Timeout != 0) {
    Value = MmioRead32 ((UINTN) Base + 0x138124);
    if (Value & BIT31) {
      //
      // Wait for 10us and try again.
      //
      DEBUG ((DEBUG_INFO, "Interface register run busy bit is still set. Trying again \n"));
      MicroSecondDelay (MAILBOX_WAITTIME);
      Timeout = Timeout - MAILBOX_WAITTIME;
    } else {
      Status = TRUE;
      break;
    }
  }
  ASSERT ((Timeout != 0));

  return Status;
}

/**
  Program the max Cd Clock supported by the platform

  @param[in] GtConfig            - Instance of GRAPHICS_PEI_CONFIG
  @param[in] GttMmAdr            - Base Address of IGFX MMIO BAR

  @retval EFI_SUCCESS            - CD Clock value programmed.
  @retval EFI_INVALID_PARAMETER  - The input parameter is invalid

**/
EFI_STATUS
ProgramCdClkReg (
  IN       GRAPHICS_PEI_CONFIG          *GtConfig,
  IN       UINT32                       GttMmAdr
  )
{
  UINT32         Data32Or;

#ifdef CPU_CFL
  ///
  /// CDCLK_CTL - GttMmAdr + 0x46000
  /// CdClock 1: 450Mhz   - [10:0] = 0x382
  /// CdClock 2: 540Mhz   - [10:0] = 0x436
  /// CdClock 0: 337.5Mhz - [10:0] = 0x2A1
  /// CdClock 3: 675Mhz   - [10:0] = 0x544
  ///
  switch (GtConfig->CdClock) {
    case 0 :
      Data32Or = V_SA_CDCLK_CTL_CD_FREQ_DECIMAL_337_5;
      break;
    case 1 :
      Data32Or = V_SA_CDCLK_CTL_CD_FREQ_DECIMAL_450;
      break;
    case 2 :
      Data32Or = V_SA_CDCLK_CTL_CD_FREQ_DECIMAL_540;
      break;
    case 3 :
      Data32Or = V_SA_CDCLK_CTL_CD_FREQ_DECIMAL_675;
      break;
    default:
      return EFI_INVALID_PARAMETER;
  }
#else
  ///
  /// For CNL, CDCLK_CTL - GttMmAdr + 0x46000
  /// CdClock 0: 168Mhz - [10:0] = 0x14E
  /// CdClock 1: 336Mhz - [10:0] = 0x29E
  /// CdClock 2: 528Mhz - [10:0] = 0x41E
  ///
  switch (GtConfig->CdClock) {
    case 0 :
      Data32Or = V_SA_CDCLK_CTL_CD_FREQ_DECIMAL_168;
      break;
    case 1 :
      Data32Or = V_SA_CDCLK_CTL_CD_FREQ_DECIMAL_336;
      break;
    case 2 :
      Data32Or = V_SA_CDCLK_CTL_CD_FREQ_DECIMAL_528;
      break;
    default:
      return EFI_INVALID_PARAMETER;
  }
#endif // CPU_CFL
  //
  // Program CDCLK register with user selected value so that GOP can read and initialize CD Clock.
  //
  MmioAndThenOr32 (GttMmAdr + R_SA_GTTMMADR_CDCLK_CTL_OFFSET, 0xFFFFF800, Data32Or);

  return EFI_SUCCESS;
}


/**
  Initialize the full CD clock as per Bspec sequence.

  @param[in] GtConfig            - Instance of GRAPHICS_PEI_CONFIG
  @param[in] GtPreMemConfig      - Instance of GRAPHICS_PEI_PREMEM_CONFIG

  @retval EFI_SUCCESS            - CD Clock Init successful.
  @retval EFI_INVALID_PARAMETER  - The input parameter is invalid
  @retval EFI_UNSUPPORTED        - iGfx is not present.
**/
EFI_STATUS
CdClkInit (
  IN  GRAPHICS_PEI_CONFIG             *GtConfig,
  IN  GRAPHICS_PEI_PREMEM_CONFIG      *GtPreMemConfig
  )
{
  UINT32         Data32Or;
  UINT16         WaitTime;
  UINT64         McD2BaseAddress;
  UINT32         GttMmAdr;
  UINT32         VoltageLevel;
#ifdef CPU_CFL
  UINT8          DpModeLinkRate;
  UINT32         CdClkRegValue;
#endif //CPU_CFL

  WaitTime = DISPLAY_CDCLK_TIMEOUT;
  McD2BaseAddress = PCI_SEGMENT_LIB_ADDRESS (SA_SEG_NUM, IGD_BUS_NUM, IGD_DEV_NUM, IGD_FUN_NUM, 0);
  if (PciSegmentRead16 (McD2BaseAddress + PCI_VENDOR_ID_OFFSET) == 0xFFFF) {
    DEBUG ((DEBUG_INFO, "iGFX not enabled - Exit!\n"));
    return EFI_UNSUPPORTED;
  }

  if (GtConfig->SkipCdClockInit) {
    return EFI_SUCCESS;
  }

  GttMmAdr = (PciSegmentRead32 (McD2BaseAddress + R_SA_IGD_GTTMMADR)) & 0xFFFFFFF0;
  if (GttMmAdr == 0) {
    GttMmAdr = GtPreMemConfig->GttMmAdr;
    PciSegmentWrite32 (McD2BaseAddress + R_SA_IGD_GTTMMADR, (UINT32) (GttMmAdr & 0xFFFFFFFF));
    PciSegmentWrite32 (McD2BaseAddress + R_SA_IGD_GTTMMADR + 4, 0);
  }

  if (!IgfxCmdRegEnabled()) {
    ///
    /// Enable Bus Master and Memory access on 0:2:0
    ///
    PciSegmentOr16 (McD2BaseAddress + PCI_COMMAND_OFFSET, (BIT2 | BIT1));
  }


  //
  // Initialize full CDCLK sequence if not initialzed by PEIM Gfx.
  //
  if (!(MmioRead32 (GttMmAdr + R_SA_GTTMMADR_CDCLK_PLL_ENABLE_OFFSET) & B_SA_CDCLK_PLL_ENABLE_BIT)) {
#ifdef CPU_CFL
    ///
    /// CDCLK_CTL - GttMmAdr + 0x46000
    /// CdClock 1: [27:26] = 00b; 450    Mhz - [10:0] = 0x382
    /// CdClock 2: [27:26] = 01b; 540    Mhz - [10:0] = 0x436
    /// CdClock 0: [27:26] = 10b; 337.5  Mhz - [10:0] = 0x2A1
    /// CdClock 3: [27:26] = 11b; 675    Mhz - [10:0] = 0x544
    ///
    switch (GtConfig->CdClock) {
      case 0 :
        Data32Or = V_SA_CDCLK_CTL_CD_FREQ_SELECT_2 | V_SA_CDCLK_CTL_CD_FREQ_DECIMAL_337_5;
        VoltageLevel = V_SA_GTTMMADR_MAILBOX_DATA_LOW_VOLTAGE_LEVEL_0;
        break;
      case 1 :
        Data32Or = V_SA_CDCLK_CTL_CD_FREQ_SELECT_0 | V_SA_CDCLK_CTL_CD_FREQ_DECIMAL_450;
        VoltageLevel = V_SA_GTTMMADR_MAILBOX_DATA_LOW_VOLTAGE_LEVEL_1;
        break;
      case 2 :
        Data32Or = V_SA_CDCLK_CTL_CD_FREQ_SELECT_1 | V_SA_CDCLK_CTL_CD_FREQ_DECIMAL_540;
        VoltageLevel = V_SA_GTTMMADR_MAILBOX_DATA_LOW_VOLTAGE_LEVEL_2;
        break;
      case 3 :
        Data32Or = V_SA_CDCLK_CTL_CD_FREQ_SELECT_3 | V_SA_CDCLK_CTL_CD_FREQ_DECIMAL_675;
        VoltageLevel = V_SA_GTTMMADR_MAILBOX_DATA_LOW_VOLTAGE_LEVEL_3;
        break;
      default:
        return EFI_INVALID_PARAMETER;
    }
#else
    ///
    /// For CNL, CDCLK_CTL - GttMmAdr + 0x46000
    /// CdClock 0: [23:22] = 10b; 168Mhz - [10:0] = 001 0100 1110 = 0x14E
    /// CdClock 1: [23:22] = 00b; 336Mhz - [10:0] = 010 1001 1110 = 0x29E
    /// CdClock 2: [23:22] = 00b; 528Mhz - [10:0] = 100 0001 1110 = 0x41E
    ///
    switch (GtConfig->CdClock) {
      case 0 :
        Data32Or = V_SA_CDCLK_CTL_CD2X_DIVIDE_BY_2 | V_SA_CDCLK_CTL_CD_FREQ_DECIMAL_168;
        VoltageLevel = V_SA_GTTMMADR_MAILBOX_DATA_LOW_VOLTAGE_LEVEL_0;
        break;
      case 1 :
        Data32Or = V_SA_CDCLK_CTL_CD2X_DIVIDE_BY_1 | V_SA_CDCLK_CTL_CD_FREQ_DECIMAL_336;
        VoltageLevel = V_SA_GTTMMADR_MAILBOX_DATA_LOW_VOLTAGE_LEVEL_1;
        break;
      case 2 :
        Data32Or = V_SA_CDCLK_CTL_CD2X_DIVIDE_BY_1 | V_SA_CDCLK_CTL_CD_FREQ_DECIMAL_528;
        VoltageLevel = V_SA_GTTMMADR_MAILBOX_DATA_LOW_VOLTAGE_LEVEL_2;
        break;
      default:
        return EFI_INVALID_PARAMETER;
    }
#endif // CPU_CFL
    //
    // Enable Display Power Well
    //
    EnablePowerWell (GttMmAdr);
    //
    // Inform Power control of upcoming frequency change
    //
    PollRunBusyClear (GttMmAdr, MAILBOX_TIMEOUT); // Poll run-busy before start

    while (WaitTime != 0) { //3ms loop
      MmioWrite32 (GttMmAdr + R_SA_GTTMMADR_MAILBOX_DATA_LOW_OFFSET, 0x00000003);  // mailbox_low       = 0x00000003
      MmioWrite32 (GttMmAdr + R_SA_GTTMMADR_MAILBOX_DATA_HIGH_OFFSET, 0x00000000); // mailbox_high      = 0x00000000
      MmioWrite32 (GttMmAdr + R_SA_GTTMMADR_MAILBOX_INTERFACE_OFFSET, 0x80000007); // mailbox Interface = 0x80000007
      PollRunBusyClear (GttMmAdr, MAILBOX_TIMEOUT);   // Poll Run Busy cleared
      //
      // Check for MailBox Data read status successful
      //
      if ((MmioRead32 (GttMmAdr + R_SA_GTTMMADR_MAILBOX_DATA_LOW_OFFSET) & BIT0) == 1) {
        DEBUG ((DEBUG_INFO, "Mailbox Data low read Successfull \n"));
        break;
      }
      MicroSecondDelay (MAILBOX_WAITTIME);
      WaitTime = WaitTime - MAILBOX_WAITTIME;
    }
    //
    // 3ms Timeout
    //
    if (WaitTime == 0) {
      DEBUG ((DEBUG_INFO, "CDCLK initialization failed , not changing CDCLK \n"));
    } else {
      DEBUG ((DEBUG_INFO, "Enabling CDCLK  \n"));
      //
      // Enable CDCLK PLL and change the CDCLK_CTL register
      //
#ifndef CPU_CFL
      if (((MmioRead32 (GttMmAdr + R_SA_GTTMMADR_DSSM_OFFSET)) & BIT31) == 1) {
        // Program CD Clock PLL Ratio with reference clock frequency = 24MHz
        MmioAndThenOr32 (GttMmAdr + R_SA_GTTMMADR_CDCLK_PLL_ENABLE_OFFSET, B_SA_GTTMMADR_CDCLK_PLL_RATIO_MASK, V_SA_CDCLK_PLL_RATIO_REF_CLK_24_MHZ);
      } else {
        // Program CD Clock PLL Ration with reference clock frequency = 19.2MHz
        MmioAndThenOr32 (GttMmAdr + R_SA_GTTMMADR_CDCLK_PLL_ENABLE_OFFSET, B_SA_GTTMMADR_CDCLK_PLL_RATIO_MASK, V_SA_CDCLK_PLL_RATIO_REF_CLK_19_2MHZ);
      }
#endif

#ifdef CPU_CFL
      ///
      /// If DPLL_CTRL1 (0x6C058) BIT[3:1] == 100b or 101b (2.16 GHz or 4.32 GHz), toggle the frequency select to get the PLL to recover.
      ///
      CdClkRegValue = MmioRead32(GttMmAdr + R_SA_GTTMMADR_CDCLK_CTL_OFFSET);
      DpModeLinkRate = (((MmioRead32(GttMmAdr + R_SA_GTTMMADR_DPLL_CTRL_OFFSET)) >> 1) & 0x07);
      if (DpModeLinkRate == 0x04 || DpModeLinkRate == 0x05) {
        MmioOr32 (GttMmAdr + R_SA_GTTMMADR_CDCLK_CTL_OFFSET, BIT19);
        MmioOr32 (GttMmAdr + R_SA_GTTMMADR_CDCLK_PLL_ENABLE_OFFSET, B_SA_CDCLK_PLL_ENABLE_BIT);
        PollGtReady (GttMmAdr, R_SA_GTTMMADR_CDCLK_PLL_ENABLE_OFFSET, B_SA_CDCLK_PLL_LOCK_BIT, B_SA_CDCLK_PLL_LOCK_BIT);

        MmioAnd32 (GttMmAdr + R_SA_GTTMMADR_CDCLK_CTL_OFFSET, (UINT32)(~(BIT27 | BIT26)));
        MmioOr32 (GttMmAdr + R_SA_GTTMMADR_CDCLK_CTL_OFFSET, (UINT32)((CdClkRegValue & (BIT27 | BIT26))));
        MmioAnd32 (GttMmAdr + R_SA_GTTMMADR_CDCLK_CTL_OFFSET, (UINT32)(~BIT19));
      } else {
#endif //CPU_CFL
      MmioOr32 (GttMmAdr + R_SA_GTTMMADR_CDCLK_PLL_ENABLE_OFFSET, B_SA_CDCLK_PLL_ENABLE_BIT);
      PollGtReady (GttMmAdr, R_SA_GTTMMADR_CDCLK_PLL_ENABLE_OFFSET, B_SA_CDCLK_PLL_LOCK_BIT, B_SA_CDCLK_PLL_LOCK_BIT);
#ifdef CPU_CFL
      }
#endif //CPU_CFL
      MmioAndThenOr32 (GttMmAdr + R_SA_GTTMMADR_CDCLK_CTL_OFFSET, B_SA_GT_CD_CLK_FREQ_MASK, Data32Or);
      //
      //Inform Power controller of the selected freq
      //
      MmioWrite32 (GttMmAdr + R_SA_GTTMMADR_MAILBOX_DATA_LOW_OFFSET, VoltageLevel);
      MmioWrite32 (GttMmAdr + R_SA_GTTMMADR_MAILBOX_DATA_HIGH_OFFSET, 0x00000000);
      MmioWrite32 (GttMmAdr + R_SA_GTTMMADR_MAILBOX_INTERFACE_OFFSET, 0x80000007);
    }
  }
  return EFI_SUCCESS;
}

/**
  Enables Power Well 1 for platform

  @param[in] GttMmAdr            - Base Address of IGFX MMIO BAR

  @retval EFI_SUCCESS            - Power well 1 Enabled
  @retval EFI_UNSUPPORTED        - Power well 1 programming Failed
  @retval EFI_TIMEOUT            - Timed out
**/
EFI_STATUS
EnablePowerWell1 (
  IN  UINT32     GttMmAdr
  )
{
  EFI_STATUS  Status;
  //
  // Poll for PG0 Fuse distribution status
  //
  Status = PollGtReady (GttMmAdr, R_SA_GTTMMADR_FUSE_STATUS_OFFSET, B_SA_GTTMMADR_FUSE_STATUS_PG0_DIST_STATUS, B_SA_GTTMMADR_FUSE_STATUS_PG0_DIST_STATUS);
  if (Status != EFI_SUCCESS) {
    return EFI_UNSUPPORTED;
  }
  //
  // Enable PG1
  //
  MmioOr32 (GttMmAdr + R_SA_GTTMMADR_PWR_WELL_CTL_OFFSET, B_SA_GTTMMADR_PWR_WELL_CTL_PG_1_ENABLE);
  //
  // Poll for PG1 state
  //
  Status = PollGtReady (GttMmAdr, R_SA_GTTMMADR_PWR_WELL_CTL_OFFSET, B_SA_GTTMMADR_PWR_WELL_CTL_PG_1_STATE, B_SA_GTTMMADR_PWR_WELL_CTL_PG_1_STATE);
  return Status;
}

/**
  Enables Power Well 2 for platform

  @param[in] GttMmAdr            - Base Address of IGFX MMIO BAR

  @retval EFI_SUCCESS            - Power well 2 Enabled
  @retval EFI_UNSUPPORTED        - Power well 2 programming Failed
  @retval EFI_TIMEOUT            - Timed out
**/
EFI_STATUS
EnablePowerWell2 (
  IN  UINT32     GttMmAdr
  )
{
  EFI_STATUS  Status;
  //
  // Poll Fuse PG1 distribution status
  //
  Status = PollGtReady (GttMmAdr, R_SA_GTTMMADR_FUSE_STATUS_OFFSET, B_SA_GTTMMADR_FUSE_STATUS_PG1_DIST_STATUS, B_SA_GTTMMADR_FUSE_STATUS_PG1_DIST_STATUS);
  if (Status != EFI_SUCCESS) {
    return EFI_UNSUPPORTED;
  }
  //
  // Enable PG2
  //
  MmioOr32 (GttMmAdr + R_SA_GTTMMADR_PWR_WELL_CTL_OFFSET, B_SA_GTTMMADR_PWR_WELL_CTL_PG_2_ENABLE);
  //
  // Poll for PG2 state
  //
  Status = PollGtReady (GttMmAdr, R_SA_GTTMMADR_PWR_WELL_CTL_OFFSET, B_SA_GTTMMADR_PWR_WELL_CTL_PG_2_STATE, B_SA_GTTMMADR_PWR_WELL_CTL_PG_2_STATE);
  return Status;
}


/**
  Program the Display Power Wells supported by platform

  @param[in] GttMmAdr            - Base Address of IGFX MMIO BAR

  @retval EFI_SUCCESS            - Power well programming finished successfully
  @retval EFI_UNSUPPORTED        - Power well programming failed
  @retval EFI_TIMEOUT            - Timed out
**/
EFI_STATUS
EnablePowerWell (
  IN  UINT32     GttMmAdr
)
{
  EFI_STATUS        Status;

  DEBUG ((DEBUG_INFO, "EnablePowerWell Started !\n"));
  //
  // Enable the power well 1
  //
  Status = EnablePowerWell1 (GttMmAdr);
  if (EFI_ERROR (Status)) {
    DEBUG ((DEBUG_WARN, "EnablePowerWell1 () has failed!\n"));
    return Status;
  }
  //
  // Enable power well 2
  //
  Status = EnablePowerWell2 (GttMmAdr);
  if (EFI_ERROR (Status)) {
    DEBUG ((DEBUG_WARN, "EnablePowerWell2 () has failed!\n"));
    return Status;
  }

  DEBUG ((DEBUG_INFO, "EnablePowerWell Successfull \n"));
  return EFI_SUCCESS;
}

/**
  InitializeDisplayAudio: Initialize display engine for iDisplay Audio programming.

  This function is called by PCH Init Pre-mem code to program CD clock freq.
  CD clock program steps involve enable PG1 & PG2 and P-code notification.

  @retval EFI_SUCCESS             The function completed successfully
  @retval EFI_ABORTED             S3 boot - display already initialized
  @retval EFI_UNSUPPORTED         iGfx disabled, iDisplay Audio not present
  @retval EFI_NOT_FOUND           SaPolicy or temporary GTT base address not found
**/
EFI_STATUS
InitializeDisplayAudio (
  VOID
  )
{
  UINT64                      McD2BaseAddress;
  SI_PREMEM_POLICY_PPI        *SiPreMemPolicyPpi;
  GRAPHICS_PEI_PREMEM_CONFIG  *GtPreMemConfig;
  UINTN                        GttMmAdr;
  EFI_STATUS                   Status;
  UINT8                        Msac;
  LARGE_INTEGER                GmAdrValue;

  DEBUG ((DEBUG_INFO, "InitializeDisplayAudio() Start\n"));


  McD2BaseAddress = PCI_SEGMENT_LIB_ADDRESS (SA_SEG_NUM, IGD_BUS_NUM, IGD_DEV_NUM, IGD_FUN_NUM, 0);
  Msac = PciSegmentRead8 (McD2BaseAddress + R_SA_IGD_MSAC_OFFSET);

  if (PciSegmentRead16 (McD2BaseAddress + PCI_VENDOR_ID_OFFSET) == 0xFFFF) {
    DEBUG ((DEBUG_INFO, "iGFX not enabled - iDisplayAudio not supported - Exit!\n"));
    return EFI_UNSUPPORTED;
  }

  ///
  /// Check if GttMmAdr has been already assigned, initialize if not
  ///
  GttMmAdr = (PciSegmentRead32 (McD2BaseAddress + R_SA_IGD_GTTMMADR)) & 0xFFFFFFF0;
  if (GttMmAdr == 0) {
    ///
    /// Get SA Policy settings through the SaInitConfigBlock PPI
    ///
    Status = PeiServicesLocatePpi (
               &gSiPreMemPolicyPpiGuid,
               0,
               NULL,
               (VOID **) &SiPreMemPolicyPpi
               );
    if (EFI_ERROR (Status) || (SiPreMemPolicyPpi == NULL)) {
      DEBUG ((DEBUG_WARN, "SaPolicy PPI not found - Exit!\n"));
      return EFI_NOT_FOUND;
    }

    Status = GetConfigBlock ((VOID *) SiPreMemPolicyPpi, &gGraphicsPeiPreMemConfigGuid, (VOID *) &GtPreMemConfig);
    ASSERT_EFI_ERROR (Status);

    GttMmAdr = GtPreMemConfig->GttMmAdr;
    if (GttMmAdr == 0) {
      DEBUG ((DEBUG_WARN, "Temporary GttMmAdr Bar is not initialized - Exit!\n"));
      return EFI_NOT_FOUND;
    }

    ///
    /// Program and read back GTT Memory Mapped BAR
    ///
    PciSegmentWrite32 (McD2BaseAddress + R_SA_IGD_GTTMMADR, (UINT32) (GttMmAdr & 0xFF000000));
    GttMmAdr = (PciSegmentRead32 (McD2BaseAddress + R_SA_IGD_GTTMMADR)) & 0xFFFFFFF0;

    PciSegmentAndThenOr8 (McD2BaseAddress + R_SA_IGD_MSAC_OFFSET, (UINT8) ~(BIT4 + BIT3 + BIT2 + BIT1 + BIT0), SA_GT_APERTURE_SIZE_256MB);
    GmAdrValue.Data = GtPreMemConfig->GmAdr64;
    PciSegmentWrite32 (McD2BaseAddress + R_SA_IGD_GMADR, GmAdrValue.Data32.Low);
    PciSegmentWrite32 (McD2BaseAddress + R_SA_IGD_GMADR + 4, GmAdrValue.Data32.High);
  }

  if (!IgfxCmdRegEnabled()) {
    ///
    /// Enable Bus Master and Memory access on 0:2:0 if not enabled
    ///
    PciSegmentOr16 (McD2BaseAddress + PCI_COMMAND_OFFSET, (BIT2 | BIT1));
  }
  //
  // Enable PCH Reset Handshake
  //
  MmioOr32 ((GttMmAdr + R_SA_GTTMMADR_NDE_RSTWRN_OPT_OFFSET), BIT4);
  //
  // Enable Display Power Well
  //
  EnablePowerWell (GttMmAdr);
#ifdef CPU_CFL
  //
  // Enable Misc IO Power
  //
  MmioOr32 (GttMmAdr + R_SA_GTTMMADR_PWR_WELL_CTL_OFFSET, BIT1);
  PollGtReady (GttMmAdr, R_SA_GTTMMADR_PWR_WELL_CTL_OFFSET, BIT0, BIT0);
#else
  //
  // Enable Audio Buffer
  //
  MmioOr32 (GttMmAdr + R_SA_GTTMMADR_AUDIO_PIN_BUF_CTL_OFFSET, B_SA_GTTMMADR_AUDIO_PIN_BUF_CTL_ENABLE);
#endif
  ///
  /// Program Aperture Size MSAC register based on policy
  ///
  PciSegmentWrite8 (McD2BaseAddress + R_SA_IGD_MSAC_OFFSET, Msac);

  DEBUG ((DEBUG_INFO, "InitializeDisplayAudio() End\n"));
  return EFI_SUCCESS;
}

/**
  ConfigureIDispAudioFrequency: Configures iDisplay Audio BCLK frequency and T-Mode

  @param[in] RequestedBclkFrequency     IDisplay Link clock frequency to be set
  @param[in] RequestedTmode             IDisplay Link T-Mode to be set

  @retval EFI_NOT_FOUND                 SA Policy PPI or GT config block not found, cannot initialize GttMmAdr
  @retval EFI_UNSUPPORTED               iDisp link unsupported frequency
  @retval EFI_SUCCESS                   The function completed successfully
**/
EFI_STATUS
ConfigureIDispAudioFrequency (
  IN       HDAUDIO_LINK_FREQUENCY   RequestedBclkFrequency,
  IN       HDAUDIO_IDISP_TMODE      RequestedTmode
  )
{
  UINT64                     McD2BaseAddress;
  SI_POLICY_PPI             *SiPreMemPolicyPpi;
  GRAPHICS_PEI_PREMEM_CONFIG *GtPreMemConfig;
  UINTN                      GttMmAdr;
  UINT32                     Data32And;
  UINT32                     Data32Or;
  EFI_STATUS                 Status;
  UINT8                      Msac;
  LARGE_INTEGER              Val;

  DEBUG ((DEBUG_INFO, "ConfigureIDispAudioFrequency() Start\n"));
  McD2BaseAddress = PCI_SEGMENT_LIB_ADDRESS (SA_SEG_NUM, IGD_BUS_NUM, IGD_DEV_NUM, IGD_FUN_NUM, 0);
  Msac = PciSegmentRead8 (McD2BaseAddress + R_SA_IGD_MSAC_OFFSET);

  if (PciSegmentRead16 (McD2BaseAddress + PCI_VENDOR_ID_OFFSET) == 0xFFFF) {
    DEBUG ((DEBUG_INFO, "iGFX not enabled - frequency switching for iDisplay link not supported - Exit!\n"));
    return EFI_UNSUPPORTED;
  }

  ///
  /// Check if GttMmAdr has been already assigned, initialize if not
  ///
  GttMmAdr = (PciSegmentRead32 (McD2BaseAddress + R_SA_IGD_GTTMMADR)) & 0xFFFFFFF0;
  if (GttMmAdr == 0) {
    ///
    /// Get SA Policy settings through the SaInitConfigBlock PPI
    ///
    Status = PeiServicesLocatePpi (
               &gSiPreMemPolicyPpiGuid,
               0,
               NULL,
               (VOID **) &SiPreMemPolicyPpi
               );
    if (EFI_ERROR (Status) || (SiPreMemPolicyPpi == NULL)) {
      DEBUG ((DEBUG_WARN, "SaPolicy PPI not found - Exit!\n"));
      return EFI_NOT_FOUND;
    }

    Status = GetConfigBlock ((VOID *) SiPreMemPolicyPpi, &gGraphicsPeiPreMemConfigGuid, (VOID *) &GtPreMemConfig);
    ASSERT_EFI_ERROR (Status);

    GttMmAdr = GtPreMemConfig->GttMmAdr;
    if (GttMmAdr == 0) {
      DEBUG ((DEBUG_WARN, "Temporary GttMmAdr Bar is not initialized - Exit!\n"));
      return EFI_NOT_FOUND;
    }

    ///
    /// Program and read back GTT Memory Mapped BAR
    ///
    PciSegmentWrite32 (McD2BaseAddress + R_SA_IGD_GTTMMADR, (UINT32) (GttMmAdr & 0xFF000000));
    GttMmAdr = (PciSegmentRead32 (McD2BaseAddress + R_SA_IGD_GTTMMADR)) & 0xFFFFFFF0;
    Val.Data = GtPreMemConfig->GmAdr64;
    PciSegmentWrite32 (McD2BaseAddress + R_SA_IGD_GMADR, Val.Data32.Low);
    PciSegmentWrite32 (McD2BaseAddress + R_SA_IGD_GMADR + 4, Val.Data32.High);
    PciSegmentAndThenOr8 (McD2BaseAddress + R_SA_IGD_MSAC_OFFSET, (UINT8) ~(BIT4 + BIT3 + BIT2 + BIT1 + BIT0), SA_GT_APERTURE_SIZE_256MB);
  }

  switch (RequestedBclkFrequency) {
    case HdaLinkFreq96MHz:
      //
      // SA IGD: GttMmAdr + 0x65900[4:3] = 10b (96MHz)
      //
      Data32And = (UINT32) ~(B_SA_IGD_AUD_FREQ_CNTRL_48MHZ);
      Data32Or  = (UINT32) B_SA_IGD_AUD_FREQ_CNTRL_96MHZ;
      break;
    case HdaLinkFreq48MHz:
      //
      // SA IGD: GttMmAdr + 0x65900[4:3] = 01b (48MHz)
      //
      Data32And = (UINT32) ~(B_SA_IGD_AUD_FREQ_CNTRL_96MHZ);
      Data32Or  = (UINT32) B_SA_IGD_AUD_FREQ_CNTRL_48MHZ;
      break;
    default:
      DEBUG ((DEBUG_WARN, "SA iGFX: Unsupported iDisplay Audio link frequency - Exit!\n"));
      return EFI_UNSUPPORTED;
  }

  Data32And &= (UINT32) ~(B_SA_IGD_AUD_FREQ_CNTRL_TMODE);
  if (RequestedTmode == HdaIDispMode1T) {
    //
    // SA IGD: 1T Mode [15] = 1b
    //
    Data32Or |= (UINT32) B_SA_IGD_AUD_FREQ_CNTRL_TMODE;
  }

  if (!IgfxCmdRegEnabled()) {
    ///
    /// Enable Bus Master and Memory access on 0:2:0
    ///
    PciSegmentOr16 (McD2BaseAddress + PCI_COMMAND_OFFSET, (BIT2 | BIT1));
  }

  ///
  /// Program iDisplay Audio link frequency and T-mode
  ///
  MmioAndThenOr32 ((UINTN) (GttMmAdr + R_SA_IGD_AUD_FREQ_CNTRL_OFFSET), Data32And, Data32Or);

  ///
  /// Program iDisplay Audio detect Frame sync early
  ///
  MmioAndThenOr32 ((UINTN) (GttMmAdr + R_SA_IGD_AUD_FREQ_CNTRL_OFFSET),
    (UINT32) ~B_SA_IGD_AUD_FREQ_CNTRL_BCLKS,
    V_SA_IGD_AUD_FREQ_CNTRL_2BCLKS << N_SA_IGD_AUD_FREQ_CNTRL_BCLKS);

  DEBUG ((DEBUG_INFO, "SA iGFX: iDisplay Audio link frequency setting: 0x%X\n", MmioRead32 ((UINTN) (GttMmAdr + R_SA_IGD_AUD_FREQ_CNTRL_OFFSET))));

  ///
  /// Program Aperture Size MSAC register based on policy
  ///
  PciSegmentWrite8 (McD2BaseAddress + R_SA_IGD_MSAC_OFFSET, Msac);

  DEBUG ((DEBUG_INFO, "ConfigureIDispAudioFrequency() End\n"));
  return EFI_SUCCESS;
}
