/** @file
  Contains functions that are used outside of the DdrIo Library.

@copyright
  INTEL CONFIDENTIAL
  Copyright 2018 - 2020 Intel Corporation.

  The source code contained or described herein and all documents related to the
  source code ("Material") are owned by Intel Corporation or its suppliers or
  licensors. Title to the Material remains with Intel Corporation or its suppliers
  and licensors. The Material may contain trade secrets and proprietary and
  confidential information of Intel Corporation and its suppliers and licensors,
  and is protected by worldwide copyright and trade secret laws and treaty
  provisions. No part of the Material may be used, copied, reproduced, modified,
  published, uploaded, posted, transmitted, distributed, or disclosed in any way
  without Intel's prior express written permission.

  No license under any patent, copyright, trade secret or other intellectual
  property right is granted to or conferred upon you by disclosure or delivery
  of the Materials, either expressly, by implication, inducement, estoppel or
  otherwise. Any license under such intellectual property rights must be
  express and approved by Intel in writing.

  Unless otherwise agreed by Intel in writing, you may not remove or alter
  this notice or any other notice embedded in Materials by Intel or
  Intel's suppliers or licensors in any way.

  This file contains an 'Intel Peripheral Driver' and is uniquely identified as
  "Intel Reference Module" and is licensed for Intel CPUs and chipsets under
  the terms of your license agreement with Intel or your vendor. This file may
  be modified by the user, subject to additional terms of the license agreement.

@par Specification Reference:
**/
#ifndef _MrcDdrIoApi_h_
#define _MrcDdrIoApi_h_

/// Defines
#define MRC_DDRIO_RX_TO_FIFO_GB_PS  (500)
#define MRC_NUM_CCC_INSTANCES       (8)
#define DCC_CCC_LANE_EN_CLK         (0x4)
#define DCC_CCC_LANE_EN_WCK         (0x8)
#define DCC_CCC_LANE_EN_ALL         (0xC)
/// DCC Parameter Mask
#define DCC_DQ_MASK            (MRC_BIT0)
#define DCC_DQS_MASK           (MRC_BIT1)
#define DCC_CLK_MASK           (MRC_BIT2)
#define DCC_WCK_MASK           (MRC_BIT3)

#define MRC_MAX_RESET_RECOVERY_ATTEMPTS  (10)
/// Functions
/**
  This function runs Comp Calibration.

  @param[in, out] MrcData - Include all MRC global data.

  @retval MrcStatus - mrcSuccess if successful or an error status
**/
MrcStatus
MrcDdrCompCalPre (
  IN OUT MrcParameters *const MrcData
  );

/**
  This function train the current supply for the DCO such that the FLL is able to hit 4 GHz.

  @param[in]  MrcData - Pointer to global data.

  @retval MrcStatus - mrcFail if Inputs->ExitOnFailure is set and we cannot reach 4 GHz; otherwise mrcSuccess.
**/
MrcStatus
MrcFllInit (
  IN  MrcParameters *const  MrcData
  );

/**
  This function runs DDRIO Offset Correction - PreTraining
  (VsxHi FB/FF Offset Correction & VccDLL Per Partition Nbias Vref Offset Correction)

  @param[in, out] MrcData - Include all MRC global data.

  @retval mrcSuccess
**/
MrcStatus
MrcOffsetCorrectionPre (
  IN OUT MrcParameters *const MrcData
  );

/**
  This function returns the tCL delay separation needed from Receive Enable to the RxFifo Read Enable.
  This covers all internal logic delay in the path.

  @param[in]  MrcData - Pointer to global MRC data.
  @param[in]  Controller  - 0-based index instance to select.
  @param[in]  Channel     - 0-based index instance to select.
  @param[in]  DdrType     - Enumeration of MrcDdrType which DDR type is being enabled.
  @param[in]  GearRatio   - Integer number of the current Gear ratio.
**/
UINT8
MrcRcvEn2RxFifoReadTclDelay (
  IN MrcParameters *const MrcData,
  IN UINT32               Controller,
  IN UINT32               Channel,
  IN MrcDdrType           DdrType,
  IN UINT8                GearRatio
  );

/**
  This function configures the DDR IO ODT type to Data and Comp blocks.
  VSS and VTT blocks are one time configured in MrcDdrIoInit.

  @param[in]  MrcData - Pointer to global MRC data.
  @param[in]  NewMode - ODT mode to enable.

  @retval MrcStatus - mrcSuccess if a valid ODT mode is requested, otherwise mrcWrongInputParameter.
**/
MrcStatus
MrcSetIoOdtMode (
  IN  MrcParameters *const  MrcData,
  IN  MRC_ODT_MODE_TYPE     NewMode
  );

/**
  This function trains VccDll for Write, Read and Idle FeedForward timing.
  Idle isnt coded for A0

  @param[in]  MrcData - Pointer to global MRC data.

  @retval MrcStatus - mrcSuccess
**/
MrcStatus
MrcVccDllFFTiming (
  IN  MrcParameters *const  MrcData
  );

/**
  This function trains VccDll Per Module FF Offset Correction

  @param[in]  MrcData - Pointer to global MRC data.

  @retval MrcStatus - mrcSuccess
**/
MrcStatus
MrcVccDllFFOffsetCorrection (
  IN MrcParameters    *const MrcData
  );

/**
  This function runs VssHi Regulator Offset Correction - PreTraining

  @param[in, out] MrcData - Include all MRC global data.
  @param[in] DebugPrint   - To print debug messages or not.
**/
void
MrcVssHiRegulatorOffsetCorrection (
  IN OUT MrcParameters *const MrcData,
  IN     BOOLEAN              DebugPrint
  );

/**
  VccDll KickBackNoise Correction
  This is the main function for this training and its organized as follows :
    Setup
    Call linear search and update the reg field
    Restore per partition
    Restore at the end

  @param[in] MrcData - Include all MRC global data.

  @retval MrcStatus - mrcFail if Inputs->ExitOnFailure is set and we cannot reach 4 GHz; otherwise mrcSuccess.
**/
MrcStatus
MrcKickBackNoiseCorrection (
  IN MrcParameters *const MrcData
  );

/**
  This function returns the target CCC pin parameters for the input Index

  @param[in] Index              - The target group, ony used in for LPDDR5.
                                    0 = CLK
                                    1 = WCK
  @param[out] CccEnFeedbackOut  - GetSet type to use for enabling feedback mode
  @param[out] CccPFallNRiseOut  - GetSet type to use for PFallNRise configuration
  @param[out] CccPRiseNFallOut  - GetSet type to use for PRiseNFall configuration

  @retval mrcSuccess if DCC converges otherwise mrcFail.
**/
void
MrcDccGetTargetCccPins (
  IN  UINT32               Index,
  OUT GSM_GT               *CccEnFeedbackOut,
  OUT GSM_GT               *CccPFallNRiseOut,
  OUT GSM_GT               *CccPRiseNFallOut
  );

/**
  This function sets up DCC steps prior to JEDEC init

  @param[in, out] MrcData - Include all MRC global data.

  @retval mrcSuccess if DCC converges otherwise mrcFail.
**/
MrcStatus
MrcDccInitPreJedec (
  IN OUT MrcParameters *const MrcData
  );

/**
  This function runs DCC steps prior to Early Command Training

  @param[in, out] MrcData - Include all MRC global data.

  @retval mrcSuccess if successful.
**/
MrcStatus
MrcDccPreEct (
  IN OUT MrcParameters *const MrcData
  );

/**
  DCC Training step for running prior to read/write leveling and after command training

  @param[in] MrcData - Include all MRC global data.

  @retval mrcSuccess if DCC converged otherwise mrcFail.
**/
MrcStatus
MrcDccPreLeveling (
  IN MrcParameters *const MrcData
  );

/**
  DCC Training step for running prior to read timing

  @param[in] MrcData - Include all MRC global data.

  @retval mrcSuccess.
**/
MrcStatus
MrcDccPreReadTiming (
  IN MrcParameters *const MrcData
  );

/**
  DCC Training step for running after write leveling

  @param[in] MrcData - Include all MRC global data.

  @retval mrcSuccess if successful
**/
MrcStatus
MrcDccPostWriteLeveling (
  IN MrcParameters *const MrcData
);
/**
  This function handles the Lpddr Overshoot problem

  @param[in, out] MrcData - Include all MRC global data.
**/
VOID
MrcLpddrOvershoot (
  IN OUT MrcParameters *const MrcData
  );

/**
  This routine returns the TX FIFO separation based on technology

  @param[in]  MrcData           - Pointer to MRC global data.

  @retval TxFifoSeparation.
**/

INT32
MrcGetTxFifoSeparation (
  IN OUT MrcParameters *const MrcData
  );

/**
  This routine computes the read and write fifo pointer delays

  @param[in]  MrcData           - Pointer to MRC global data.
  @param[in]  Controller        - Controller to get timings
  @param[in]  Channel           - Controller to get timings
  @param[in]  tCWL              - Write Latency for current channel
  @param[in]  AddTcwl           - Current AddTcwl value
  @param[in]  DecTcwl           - Current DecTcwl value
  @param[out] tCWL4TxDqFifoWrEn - Pointer to the write TX DQ fifo delay
  @param[out] tCWL4TxDqFifoRdEn - Pointer to the read TX DQ fifo delay

  @retval N/A.
**/
VOID
MrcGetTxDqFifoDelays (
  IN OUT MrcParameters *const MrcData,
  IN     UINT32        Controller,
  IN     UINT32        Channel,
  IN     INT32         tCWL,
  IN     UINT32        AddTcwl,
  IN     UINT32        DecTcwl,
     OUT INT64         *tCWL4TxDqFifoWrEn,
     OUT INT64         *tCWL4TxDqFifoRdEn
  );

/**
  DCC Convergence check - Successful convergence is if Status fields return 0x100 (256) +/- 6

  @param[in] MrcData         - Include all MRC global data.
  @param[in] ParamMask       - Bit Mask of parameters to be run.  Bit 0 = DCC_DQ, Bit 1 = DCC_DQS, Bit 2 = DCC_CLK, Bit 3 = DCC_WCK
  @param[in] RankMask        - Which Rank(s) to target
  @param[in] Print           - Whether to print (TRUE) or not (FALSE)

  @retval mrcSuccess if all is converged, mrcFail otherwise.
**/
MrcStatus
MrcDccConvergenceCheck (
  IN MrcParameters *const MrcData,
  IN UINT32               ParamMask,
  IN UINT32               RankMask,
  IN BOOLEAN              Print
  );

/**
  This function runs Reset Recovery

  @param[in, out] MrcData - Include all MRC global data.

  @retval mrcSuccess if MC1 not populated or if no Reset Recovery needed otherwise return mrcResetRecovery
**/
MrcStatus
MrcResetRecovery (
  IN OUT MrcParameters *const MrcData
  );

/**
  This function sets up Phy registers that needs clearing after previous SAGV point restore

  @param[in, out] MrcData - Include all MRC global data.
**/
void
MrcDdrPhyRegisterSettings (
  IN OUT MrcParameters *const MrcData
  );

#endif //_MrcDdrIoApi_h_
