/** @file
  This file contains code related to initializing and configuring the DDRIO.

@copyright
  INTEL CONFIDENTIAL
  Copyright 2018 - 2020 Intel Corporation.

  The source code contained or described herein and all documents related to the
  source code ("Material") are owned by Intel Corporation or its suppliers or
  licensors. Title to the Material remains with Intel Corporation or its suppliers
  and licensors. The Material may contain trade secrets and proprietary and
  confidential information of Intel Corporation and its suppliers and licensors,
  and is protected by worldwide copyright and trade secret laws and treaty
  provisions. No part of the Material may be used, copied, reproduced, modified,
  published, uploaded, posted, transmitted, distributed, or disclosed in any way
  without Intel's prior express written permission.

  No license under any patent, copyright, trade secret or other intellectual
  property right is granted to or conferred upon you by disclosure or delivery
  of the Materials, either expressly, by implication, inducement, estoppel or
  otherwise. Any license under such intellectual property rights must be
  express and approved by Intel in writing.

  Unless otherwise agreed by Intel in writing, you may not remove or alter
  this notice or any other notice embedded in Materials by Intel or
  Intel's suppliers or licensors in any way.

  This file contains an 'Intel Peripheral Driver' and is uniquely identified as
  "Intel Reference Module" and is licensed for Intel CPUs and chipsets under
  the terms of your license agreement with Intel or your vendor. This file may
  be modified by the user, subject to additional terms of the license agreement.

@par Specification Reference:
**/
#include "MrcTypes.h"
#include "MrcInterface.h"
#include "MrcHalRegisterAccess.h"
#include "MrcCommon.h" // Needed for OFFSET_CALC_CH
#include "MrcDdrIoApi.h"
#include "MrcDdrIoApiInt.h"
#include "MrcReset.h"
#include "MrcDdrIoOffsets.h"
#include "Cpgc20TestCtl.h"
#include "MrcChipApi.h"
#include "MrcInterpreter.h"

/// Defines
#define VoltageSelect_VccDDQ   (0)
#define VoltageSelect_Vccdd2   (1)
#define MaxVoltageSelect       (2)
#define DDRIO_MAX_VTT_GEN      (4)
#define VCC_DLL_BYTES          (2)
#define VCCDLL1                (1)
#define VCCDLL2                (2U)
#define VCCDLL3                (3U)
#define VSXHI                  (4)
#define MAX_VSXHI_CODES        (5)
#define MAX_OFFSET_VOLTAGE     (2)
#define MAX_FF_RCVEN_PI        (5)
#define MAX_FF_RCVEN_PRE       (5)
#define MAX_VCCDLL_CODES       (7)
#define MAX_FF_WRITESTAGGER    (2)
#define MRC_NUMBURSTS_FFTIMING (32)
#define MAX_VCCDLL_DATA_PAR    (8)
#define MAX_VCCDLL_PAR         (12)
#define MAX_KICKBACK_CODES     (5)
#define NDEFAULT               (33)
#define MAX_VCCDLL_DATA_PAR    (8)
#define COMPVTARGET_SWINGV     (260)
#define COMPVTARGET_STEPV      (26)
#define THOUSAND_MULTIPLIER    (1000)
#define MAX_TCO_FEEDBACK_BITS  (4)
#define TCO_FEEDBACK_MASK      ((1 << MAX_TCO_FEEDBACK_BITS) - 1)
#define PN_REPEAT_READS        (9)
#define MAX_OPTIMIZED_PI_CB_EN (16)
#define MAX_OPTIMIZED_PI_SWEEP (16)
#define MAX_COARSE             (15)
#define MAX_RXDQS_VREF_OFFSETS (DATA0CH0_CR_DDRDATADQSRANK0_RxDQSVrefOffset_MAX + 1)
#define HALF_DUTY_CYCLE        (256)
#define MAX_TCO_COMP           (DATA0CH0_CR_DDRCRDATATCOCONTROL0_DqTcoCompBit0_MAX + 1)

#define MRC_DCC_CONVERGENCE_RANGE       (6)
#define MRC_MAX_DCC_CONVERGENCE_CYCLES  (100)

#define DCC_DATA_LANE_EN_DQS   (0x300) //DQS (BIT9) DBI (BIT8)
#define DCC_DATA_LANE_EN_DQ    (0x1FF) //DQ (BIT[7:0]) DBI (BIT8)
#define DCC_DATA_LANE_EN_ALL   (0x3FF)

#define MAX_RESET_RECOVERY_WRITES  (5)
/// Constants
const INT8 RxFifoChDelay[MRC_DDR_TYPE_UNKNOWN][MAX_GEARS][MAX_SYS_CHANNEL] = {
// Gear1                              Gear2
// Channel
//   0,  1,  2,  3,  4,  5,  6,  7     0,  1,  2,  3,  4,  5,  6,  7
  {{ 5,  3,  0,  0,  0,  0,  0,  0}, { 9,  7,  0,  0,  0,  0,  0,  0}}, // DDR4
  {{ 5,  5,  3,  3,  0,  0,  0,  0}, { 9,  9,  7,  7,  0,  0,  0,  0}}, // DDR5
  {{ 7,  7,  7,  7,  5,  5,  5,  5}, {11, 11, 11, 11,  9,  9,  9,  9}}, // LPDDR5
  {{ 5,  5,  5,  5,  3,  3,  3,  3}, { 9,  9,  9,  9,  7,  7,  7,  7}}  // LPDDR4
};

/// Enums
typedef enum {
  DllDdrData0,
  DllDdrData1,
  DllDdrData2,
  DllDdrData3,
  DllDdrData4,
  DllDdrData5,
  DllDdrData6,
  DllDdrData7,
  DllDdrData8,
  DllDdrCcc0,
  DllDdrCcc1,
  DllDdrCcc2,
  DllDdrCcc3,
  DllDdrMax
} DLL_PARTITIONS;

typedef enum {
  CaVssHi,
  CtlVssHi,
  ClkVssHi,
  MaxCccVssHi
} CCC_VSSHI;

/// Structs
typedef struct {
  UINT8 Channel;
  UINT8 Byte;
} PHY_PARTITION_BLOCK;

const PHY_PARTITION_BLOCK LpddrDdr4ILDllPartitions[DllDdrMax][VCC_DLL_BYTES] = {

  //   Physical Channel, Phy Byte
    {{ 0, 0}, { 1, 0}}, //DllDdrData0
    {{ 0, 1}, { 1, 1}}, //DllDdrData1
    {{ 0, 2}, { 1, 2}}, //DllDdrData2
    {{ 0, 3}, { 1, 3}}, //DllDdrData3
    {{ 0, 4}, { 1, 4}}, //DllDdrData4
    {{ 0, 5}, { 1, 5}}, //DllDdrData5
    {{ 0, 6}, { 1, 6}}, //DllDdrData6
    {{ 0, 7}, { 1, 7}}, //DllDdrData7
    {{ 0, 8}, { 1, 8}}, //DllDdrData8 ECC Byte for TGL H DDR4
    {{ 0, 0}, { 1, 0}}, //DllDdrCcc0
    {{ 2, 0}, { 3, 0}}, //DllDdrCcc1
    {{ 4, 0}, { 5, 0}}, //DllDdrCcc2
    {{ 6, 0}, { 7, 0}}, //DllDdrCcc3
};

const PHY_PARTITION_BLOCK Ddr4NILDllPartitions[DllDdrMax][VCC_DLL_BYTES] = {
  //   Physical Channel, Phy Byte
    {{ 0, 0}, { 0, 2}}, //DllDdrData0
    {{ 0, 1}, { 0, 3}}, //DllDdrData1
    {{ 0, 4}, { 0, 6}}, //DllDdrData2
    {{ 0, 5}, { 0, 7}}, //DllDdrData3
    {{ 1, 0}, { 1, 2}}, //DllDdrData4
    {{ 1, 1}, { 1, 3}}, //DllDdrData5
    {{ 1, 4}, { 1, 6}}, //DllDdrData6
    {{ 1, 5}, { 1, 7}}, //DllDdrData7
    {{ 0, 8}, { 1, 8}}, //DllDdrData8 ECC byte for TGL H DDR4
    {{ 0, 0}, { 1, 0}}, //DllDdrCcc0
    {{ 2, 0}, { 3, 0}}, //DllDdrCcc1
    {{ 4, 0}, { 5, 0}}, //DllDdrCcc2
    {{ 6, 0}, { 7, 0}}, //DllDdrCcc3
};



/**
  This function updates the Gear specific fields in the DDRIO

  @param[in]  MrcData - Pointer to global MRC data.
  @param[in]  Gear2   - TRUE for Gear2, FALSE for Gear1
**/
VOID
DdrIoConfigGear2 (
  IN  MrcParameters *MrcData,
  IN  BOOLEAN       Gear2
  )
{
  const MrcInput *Inputs;
  MrcOutput      *Outputs;
  INT64          GetSetVal;
  UINT32         Index;
  UINT32         Offset;
  UINT32         Data;
  UINT32         DdrIoGear1;
  UINT8          PiSyncDiv;
  UINT8          CaInc;
  UINT8          CaDiv;
  UINT8          ClkInc;
  UINT8          ClkDiv;
  UINT8          CtlInc;
  UINT8          CtlDiv;
  UINT8          WckInc;
  UINT8          WckDiv;
  BOOLEAN        A0;
  BOOLEAN        DtHaloDdr4;
  BOOLEAN        Lpddr5;
  CH0CCC_CR_DDRCRPINSUSED_STRUCT        CccPinsUsed;
  CH0CCC_CR_DDRCRPINSUSED_P0_STRUCT     CccPinsUsedP0;
  CH0CCC_CR_DDRCRCCCPIDIVIDER_STRUCT    CccPiDivider;
  CH0CCC_CR_DDRCRCCCPIDIVIDER_STRUCT    CccPiDividerOdd;
  CH0CCC_CR_DDRCRCCCPIDIVIDER_STRUCT    *CccPiDivPtr;
  DDRPHY_COMP_CR_VCCDLLDQSDELAY_STRUCT  VccDllDqsDelay;

  Inputs = &MrcData->Inputs;
  Outputs = &MrcData->Outputs;
  DdrIoGear1 = Gear2 ? 0 : 1; // DDRIO Register encoding: 1 = Gear1, 0 = Gear2
  Lpddr5 = (Outputs->DdrType == MRC_DDR_TYPE_LPDDR5);
  A0 = Inputs->A0;
  DtHaloDdr4 = ((Inputs->DtHalo) && (Outputs->DdrType == MRC_DDR_TYPE_DDR4));

  GetSetVal = DdrIoGear1;
  MrcGetSetNoScope (MrcData, GsmIocScramGear1, WriteCached, &GetSetVal);

  MrcGetSetStrobe (MrcData, MAX_CONTROLLER, MAX_CHANNEL, MRC_IGNORE_ARG, MAX_SDRAM_IN_DIMM, GsmIocDataCtlGear1, WriteCached, &GetSetVal);

  // PiSyncDiv
  if (Lpddr5) {
    PiSyncDiv = ((Gear2) ? 2 : 3);
  } else {
    PiSyncDiv = 0;
  }
  // CA
  CaDiv = ((!A0 || Lpddr5 || !Gear2) ? 1 : 0);
  if (Lpddr5) {
    CaInc = ((Gear2) ? 2 : 1);
    if (!A0) {
      CaInc--;
    }
  } else {
    if (A0) {
      CaInc = ((Gear2) ? 0 : 2);
    } else {
      CaInc = ((Gear2) ? 2 : 1);
    }
  }
  // CTL
  CtlDiv = CaDiv;
  if (Lpddr5) {
    CtlInc = ((A0 && Gear2) ? 1 : 0);
  } else {
    // Same encoding as CaInc for this case
    CtlInc = CaInc;
  }
  // CLK
  ClkDiv = ((!Gear2 || Lpddr5) ? 1 : 0);
  if (Lpddr5) {
    ClkInc = ((Gear2) ? 1 : 0);
  } else {
    ClkInc = ((Gear2) ? 0 : 2);
  }
  // WCK
  WckInc = ((Gear2) ? 3 : 2);
  WckDiv = ((Gear2) ? 0 : 1);

  CccPiDivider.Data = 0;
  CccPiDivider.Bits.PiClkDuration = CH0CCC_CR_DDRCRCCCPIDIVIDER_PiClkDuration_MAX; // @todo - Set final value in MrcMcActivate.
  CccPiDivider.Bits.PiSyncDivider = PiSyncDiv;
  CccPiDivider.Bits.WckHalfPreamble = 1; // 4:1 mode
  CccPiDivider.Bits.Pi4IncPreamble = (Gear2) ? 2 : 1;
  CccPiDivider.Bits.Pi4DivEnPreamble = 1;
  // Cmd
  CccPiDivider.Bits.Pi0DivEn  = CaDiv;
  CccPiDivider.Bits.Pi0Inc    = CaInc;
  // Ctl
  CccPiDivider.Bits.Pi1DivEn  = CtlDiv;
  CccPiDivider.Bits.Pi1Inc    = CtlInc;
  // Ctl (Lp, Ddr4 Ccc1) / Cke (Ddr4 Ccc2)
  CccPiDivider.Bits.Pi2DivEn  = CtlDiv;
  CccPiDivider.Bits.Pi2Inc    = CtlInc;
  // Clk - P0 Diverges and is handled below
  CccPiDivider.Bits.Pi3DivEn  = ClkDiv;
  CccPiDivider.Bits.Pi3Inc    = ClkInc;
  // Cke (Lp4) / Wck (Lp5) - P0 Diverges and is handled below.
  CccPiDivider.Bits.Pi4DivEn  = (Lpddr5) ? WckDiv : CtlDiv;
  CccPiDivider.Bits.Pi4Inc    = (Lpddr5) ? WckInc : CtlInc;
  MrcWriteCR (MrcData, CCC_CR_DDRCRCCCPIDIVIDER_REG, CccPiDivider.Data);

  if (DtHaloDdr4) {
    // PI 4 is coded to CTL by default.  Flip it to CLK for odd CCC's.
    CccPiDividerOdd.Data = CccPiDivider.Data;
    CccPiDividerOdd.Bits.Pi4DivEn = ClkDiv;
    CccPiDividerOdd.Bits.Pi4Inc = ClkInc;
    // PI 3 is coded to CLK by default.  Flip it for even CCC's
    CccPiDivider.Bits.Pi3Inc = CtlInc;
    CccPiDivider.Bits.Pi3DivEn = CtlDiv;
  }

  for (Index = 0; Index < MRC_NUM_CCC_INSTANCES; Index++) {
    // For Desktop PI Group 3 & 4 alternate types depending on which CCC instance.
    // -----------------------------------
    // | PiGroup | CLK CCC's | CTL CCC's |
    // |---------|-----------|-----------|
    // |    3    |  1,3,5,7  |  0,2,4,6  |
    // |    4    |  1,3,5,7  |  0,2,4,6  |
    // -----------------------------------
    if (DtHaloDdr4) {
      Offset = OFFSET_CALC_CH (CH0CCC_CR_DDRCRCCCPIDIVIDER_REG, CH1CCC_CR_DDRCRCCCPIDIVIDER_REG, Index);
      CccPiDivPtr = (((Index % 2) == 0) ? &CccPiDivider : &CccPiDividerOdd);
      MrcWriteCR (MrcData, Offset, CccPiDivPtr->Data);
    }

    Offset = OFFSET_CALC_CH (CH0CCC_CR_DDRCRPINSUSED_REG, CH1CCC_CR_DDRCRPINSUSED_REG, Index);
    Data = MrcReadCR (MrcData, Offset);
    if (Inputs->UlxUlt) {
      CccPinsUsed.Data = Data;
      CccPinsUsed.Bits.Gear1 = DdrIoGear1;
      Data = CccPinsUsed.Data;
    } else {
      CccPinsUsedP0.Data = Data;
      CccPinsUsedP0.P0Bits.Gear1 = DdrIoGear1;
      Data = CccPinsUsedP0.Data;
    }
    MrcWriteCR (MrcData, Offset, Data);
  }
  if ((Outputs->RxMode == MrcRxModeUnmatchedRxWPpath) || (Outputs->RxMode == MrcRxModeUnmatchedRxWRload)) {
    VccDllDqsDelay.Data = MrcReadCR (MrcData, DDRPHY_COMP_CR_VCCDLLDQSDELAY_REG);
    VccDllDqsDelay.Bits.Gear1 = DdrIoGear1;
    MrcWriteCR (MrcData, DDRPHY_COMP_CR_VCCDLLDQSDELAY_REG, VccDllDqsDelay.Data);
  }
}

/**
  This function records the result of VssHi Comp Calibration as well as setup for the next iteration

  @param[in, out] MrcData  - Include all MRC global data.
  @param[out] CCCVssHi2    - Array for storing CCC Vdd2
  @param[out] CCCVssHiQ    - Array for storing CCC Vddq
  @param[out] DQVssHiQ     - Array for storing DQ
  @param[in] CCCBlock      - Array for which CCCBlock is for VDDQ and which is for Vdd2
  @param[in] FirstCCCBlock - First populated Channel to read from for CCC Block
  @param[in] Index         - Index of array
  @param[in] RCompCodes    - Array of CompCodes to write for next iteration

**/
VOID
MrcVssHiCompCalSetup (
  IN OUT MrcParameters *const MrcData,
  OUT UINT8      CCCVssHi2[MRC_NUM_OF_VSSHI_COMPS],
  OUT UINT8      CCCVssHiQ[MRC_NUM_OF_VSSHI_COMPS],
  OUT UINT8      DQVssHiQ[MRC_NUM_OF_VSSHI_COMPS],
  IN  CCC_VSSHI  CCCBlock[MaxVoltageSelect],
  IN  UINT32     FirstCCCBlock,
  IN  UINT32     Index,
  IN  UINT8      RCompCodes[MRC_NUM_OF_VSSHI_COMPS]
  )
{
  UINT8   LoopIndex;
  UINT32  CccOffset;
  UINT32  CccBroadcastOffset;
  UINT32  Offset;
  DDRPHY_COMP_CR_DDRCRCACOMP_STRUCT  CccPhyComp;
  CCC_CR_DDRCRCACOMP_STRUCT  CccComp;
  DDRPHY_COMP_CR_DDRCRDATACOMP0_STRUCT  DataComp0;

  for (LoopIndex = 0; LoopIndex < MaxVoltageSelect; LoopIndex++) {
    switch (CCCBlock[LoopIndex]) {
      case CaVssHi:
        Offset = DDRPHY_COMP_CR_DDRCRCACOMP_REG;
        CccOffset = OFFSET_CALC_CH (CH0CCC_CR_DDRCRCACOMP_REG, CH1CCC_CR_DDRCRCACOMP_REG, FirstCCCBlock);
        CccBroadcastOffset = CCC_CR_DDRCRCACOMP_REG;
        break;

      case CtlVssHi:
        Offset = DDRPHY_COMP_CR_DDRCRCTLCOMP_REG;
        CccOffset = OFFSET_CALC_CH (CH0CCC_CR_DDRCRCTLCOMP_REG, CH1CCC_CR_DDRCRCTLCOMP_REG, FirstCCCBlock);
        CccBroadcastOffset = CCC_CR_DDRCRCTLCOMP_REG;
        break;

      case ClkVssHi:
        Offset = DDRPHY_COMP_CR_DDRCRCLKCOMP_REG;
        CccOffset = OFFSET_CALC_CH (CH0CCC_CR_DDRCRCLKCOMP_REG, CH1CCC_CR_DDRCRCLKCOMP_REG, FirstCCCBlock);
        CccBroadcastOffset = CCC_CR_DDRCRCLKCOMP_REG;
        break;

      default:
        Offset = CccOffset = CccBroadcastOffset = MRC_IGNORE_ARG;
        break;
    }
    if (Offset == MRC_IGNORE_ARG) {
      continue;
    }
    CccPhyComp.Data = MrcReadCR (MrcData, Offset);
    if (LoopIndex == VoltageSelect_VccDDQ) {
      CCCVssHiQ[Index] = (UINT8) CccPhyComp.Bits.VssHiFF;
    } else {
      CCCVssHi2[Index] = (UINT8) CccPhyComp.Bits.VssHiFF;
    }
    if (Index < (MRC_NUM_OF_VSSHI_COMPS - 1)) {
      //Setup for the next iteration except last one
      CccPhyComp.Bits.RcompDrvUp = RCompCodes[Index + 1];
      MrcWriteCR (MrcData, Offset, CccPhyComp.Data);
      CccComp.Data = MrcReadCR (MrcData, CccOffset);
      CccComp.Bits.RcompDrvUp = RCompCodes[Index + 1];
      MrcWriteCrMulticast (MrcData, CccBroadcastOffset, CccComp.Data);
    }
  }

  Offset = DDRPHY_COMP_CR_DDRCRDATACOMP0_REG;
  DataComp0.Data = MrcReadCR (MrcData, Offset);
  DQVssHiQ[Index] = (UINT8) DataComp0.Bits.VssHiFF;
  if (Index < (MRC_NUM_OF_VSSHI_COMPS - 1)) {
    //Setup for the next iteration except last one
    DataComp0.Bits.RcompDrvUp = RCompCodes[Index + 1];
    MrcWriteCR (MrcData, Offset, DataComp0.Data);
  }
}

/**
  This function calculates the VssHi Coefficients

  @param[in, out] MrcData - Include all MRC global data.
  @param[in, out] VssHi   - Array for of VssHiFF results
  @param[out] FFRefBit5   - Result for FFRefBit5
  @param[out] FFRefBit4   - Result for FFRefBit4
  @param[out] FFRefBit3   - Result for FFRefBit3
  @param[out] FFRefBit2   - Result for FFRefBit2
  @param[out] FFRefLSB    - Result for FFRefLSB
  @param[out] FFRefStatic - Result for FFRefStatic

**/
VOID
MrcVssHiCoeffCalc (
  IN OUT MrcParameters *const MrcData,
  IN OUT UINT8                VssHi[MRC_NUM_OF_VSSHI_COMPS],
  OUT UINT8                   *FFRefBit5,
  OUT UINT8                   *FFRefBit4,
  OUT UINT8                   *FFRefBit3,
  OUT UINT8                   *FFRefBit2,
  OUT UINT8                   *FFRefLSB,
  OUT UINT8                   *FFRefStatic
  )
{
  INT32  Data32;
  INT32  SubtractionValue;
  INT32  Bit5Value;
  INT32  Bit4Value;
  INT32  Bit3Value;
  INT32  Bit2Value;
  INT32  LSBValue;
  INT32  StaticValue;

  Data32 = VssHi[0] + VssHi[1] + VssHi[2];
  Data32 /= 3;
  VssHi[2] = (UINT8) Data32;
  StaticValue = VssHi[2];
  if (FFRefStatic != NULL) {
    *FFRefStatic = (UINT8) (RANGE (StaticValue, 0, 15));
  }

  Data32 = VssHi[13] + VssHi[12];
  SubtractionValue = VssHi[11] + VssHi[2];
  Bit5Value = Data32 - SubtractionValue;
  Bit5Value /= 2;
  if (FFRefBit5 != NULL) {
    *FFRefBit5 = (UINT8) (RANGE (Bit5Value, 0, 15));
  }

  Data32 = VssHi[11] + VssHi[10];
  SubtractionValue = VssHi[9] + VssHi[2];
  Bit4Value = Data32 - SubtractionValue;
  Bit4Value /= 2;
  if (FFRefBit4 != NULL) {
    *FFRefBit4 = (UINT8) (RANGE (Bit4Value, 0, 7));
  }

  Data32 = VssHi[9] + VssHi[8];
  SubtractionValue = VssHi[7] + VssHi[2];
  Bit3Value = Data32 - SubtractionValue;
  Bit3Value /= 2;
  if (FFRefBit3 != NULL) {
    *FFRefBit3 = (UINT8) (RANGE (Bit3Value, 0, 7));
  }

  Data32 = VssHi[7] + VssHi[6];
  SubtractionValue = VssHi[5] + VssHi[2];
  Bit2Value = Data32 - SubtractionValue;
  Bit2Value /= 2;
  if (FFRefBit2 != NULL) {
    *FFRefBit2 = (UINT8) (RANGE (Bit2Value, 0, 7));
  }

  Data32 = (2 * VssHi[5]) + (2 * VssHi[4]) + VssHi[3];
  SubtractionValue =  VssHi[4] + (2 * VssHi[3]) + (2 * VssHi[2]);
  LSBValue = Data32 - SubtractionValue;
  LSBValue /= 5;
  if (FFRefLSB != NULL) {
    *FFRefLSB = (UINT8) (RANGE (LSBValue, 0, 7));
  }

  return;
}

/**
  This function programs the Drift Limits (Global and Local) based on TargetNUI

  @param[in, out] MrcData            - Include all MRC global data.
  @param[in, out] OrigVccDllDqsDelay - Register value of DDRPHY_COMP_CR_VCCDLLDQSDELAY_REG
  @param[in] FirstController         - First populated Controller to read from
  @param[in] FirstByte               - First populated Byte to read from

**/
VOID
MrcDriftProg (
  IN OUT MrcParameters *const MrcData,
  IN OUT UINT32  *OrigVccDllDqsDelay,
  IN UINT32       FirstController,
  IN UINT32       FirstByte
  )
{
  MrcOutput          *Outputs;
  UINT32             Offset;
  UINT32             TargetNUI;
  UINT32             Numerator;
  UINT32             Denominator;
  UINT32             HalfUi;
  UINT32             Formula;
  UINT32             VTDrift;
  UINT32             SysDrift;
  UINT32             FwdClkMinMax;
  UINT32             UIfs;
  UINT32             DriftLimitGlobal;
  UINT32             DriftLimitLocal;
  UINT32             IamA0;
  BOOLEAN            Lpddr4;
  DATA0CH0_CR_RCOMPDATA0_STRUCT  RcompData0;
  DDRPHY_COMP_CR_VCCDLLDQSDELAY_STRUCT  VccDllDqsDelay;

  VccDllDqsDelay.Data = *OrigVccDllDqsDelay;
  Outputs    = &MrcData->Outputs;
  Lpddr4     = (Outputs->DdrType == MRC_DDR_TYPE_LPDDR4);
  IamA0      = (MrcData->Inputs.A0) ? 1 : 0;
  UIfs       = (Outputs->Qclkps * 1000) >> ((Outputs->Gear2) ? 1 : 0);
  UIfs       = MAX (UIfs, 1); //Divide by zero protection
  VTDrift    = 15000 * THOUSAND_MULTIPLIER / UIfs;
  SysDrift   = 15000 * THOUSAND_MULTIPLIER / UIfs;

  Offset     = OFFSET_CALC_MC_CH (DATA0CH0_CR_RCOMPDATA0_REG, DATA0CH1_CR_RCOMPDATA0_REG, FirstController, DATA1CH0_CR_RCOMPDATA0_REG, FirstByte);
  RcompData0.Data = MrcReadCR (MrcData, Offset);
  TargetNUI = (RcompData0.Bits.DqsNTargetNUI) + 1; //Convert 0-based to 1-based

  Numerator = (TargetNUI * THOUSAND_MULTIPLIER) + (THOUSAND_MULTIPLIER / 2) - (THOUSAND_MULTIPLIER / 4) - ((THOUSAND_MULTIPLIER * 18) / 100);
  Denominator = MAX (THOUSAND_MULTIPLIER + VTDrift + SysDrift, 1);
  FwdClkMinMax = Numerator * THOUSAND_MULTIPLIER / Denominator;
  HalfUi = UIfs * THOUSAND_MULTIPLIER * IamA0 / 2;
  Formula = ((THOUSAND_MULTIPLIER / 4) + ((18 * THOUSAND_MULTIPLIER) / 100) + VTDrift + (SysDrift * FwdClkMinMax / THOUSAND_MULTIPLIER)) * UIfs;
  DriftLimitGlobal = (MAX (HalfUi, Formula)) / UIfs;
  Formula = DriftLimitGlobal - (THOUSAND_MULTIPLIER / 2);
  DriftLimitLocal = MAX (VTDrift, Formula);

  // Cast in PI ticks
  DriftLimitGlobal = DriftLimitGlobal * 64 / THOUSAND_MULTIPLIER;
  DriftLimitLocal = DriftLimitLocal * 64 / THOUSAND_MULTIPLIER;

  // make sure don't overflow
  DriftLimitLocal = MIN (DriftLimitLocal, 15);
  DriftLimitGlobal = MIN (DriftLimitGlobal, 255);

  if (Lpddr4) {
    DriftLimitGlobal = 32;
    DriftLimitLocal = 12;
  }
  VccDllDqsDelay.Bits.DriftLimitGlobal = DriftLimitGlobal;
  VccDllDqsDelay.Bits.DriftLimitLocal =  DriftLimitLocal;

  if (*OrigVccDllDqsDelay != VccDllDqsDelay.Data) {
    *OrigVccDllDqsDelay = VccDllDqsDelay.Data;
    MrcWriteCR (MrcData, DDRPHY_COMP_CR_VCCDLLDQSDELAY_REG, VccDllDqsDelay.Data);
  }
}

/**
  This function runs DVFS Comp Calibration

  @param[in, out] MrcData - Include all MRC global data.

  @retval mrcSuccess
**/
MrcStatus
MrcDdrCompCalDvfs (
  IN OUT MrcParameters *const MrcData
  )
{
  MrcOutput          *Outputs;
  MrcInput           *Inputs;
  const MRC_FUNCTION *MrcCall;
  BOOLEAN            Lpddr;
  UINT8              FFRefBit5;
  UINT8              FFRefBit4;
  UINT8              FFRefBit3;
  UINT8              FFRefBit2;
  UINT8              FFRefLSB;
  UINT8              FFRefStatic;
  static UINT8       RCompCodes[MRC_NUM_OF_VSSHI_COMPS] = {0, 0, 0, 1, 2, 3, 4, 7, 8, 15, 16, 31, 32, 63};
  CCC_VSSHI          CCCBlock[MaxVoltageSelect];
  UINT8              CCCVssHi2[MRC_NUM_OF_VSSHI_COMPS];
  UINT8              CCCVssHiQ[MRC_NUM_OF_VSSHI_COMPS];
  UINT8              DQVssHiQ[MRC_NUM_OF_VSSHI_COMPS];
  UINT32             Index;
  UINT32             MaxIndex;
  UINT32             FirstController;
  UINT32             FirstChannel;
  UINT32             FirstCCCBlock;
  UINT32             Offset;
  UINT32             OrigCaPhyComp;
  UINT32             OrigCtlPhyComp;
  UINT32             OrigClkPhyComp;
  UINT32             OrigCaComp;
  UINT32             OrigCtlComp;
  UINT32             OrigClkComp;
  UINT32             OrigDataRcompDrvUp;
  DATA0CH0_CR_RCOMPDATA0_STRUCT  RcompData0;
  DDRPHY_COMP_CR_DDRCRCOMPOVR0_STRUCT  CompOvr0;
  DDRPHY_COMP_CR_DDRCRCOMPCTL0_STRUCT  CompCtl0;
  DDRPHY_COMP_CR_DDRCRCACOMP_STRUCT  CaPhyComp;
  DDRPHY_COMP_CR_DDRCRCTLCOMP_STRUCT  CtlPhyComp;
  DDRPHY_COMP_CR_DDRCRCLKCOMP_STRUCT  ClkPhyComp;
  CCC_CR_DDRCRCACOMP_STRUCT  CaComp;
  CCC_CR_DDRCRCTLCOMP_STRUCT  CtlComp;
  CCC_CR_DDRCRCLKCOMP_STRUCT  ClkComp;
  DDRPHY_COMP_CR_DDRCRDATACOMP0_STRUCT  DataComp0;
  DDRPHY_COMP_CR_VSXHIFFCOMPREF0_STRUCT  VsxHiFFCompRef0;
  DDRPHY_COMP_CR_VSXHIFFCOMPREF1_STRUCT  VsxHiFFCompRef1;

  Inputs       = &MrcData->Inputs;
  MrcCall      = Inputs->Call.Func;
  Outputs      = &MrcData->Outputs;
  Lpddr        = Outputs->Lpddr;
  MaxIndex     = MRC_NUM_OF_VSSHI_COMPS;

  MrcCall->MrcSetMem (CCCVssHi2, sizeof (CCCVssHi2), 0);
  MrcCall->MrcSetMem (CCCVssHiQ, sizeof (CCCVssHiQ), 0);
  MrcCall->MrcSetMem (DQVssHiQ, sizeof (DQVssHiQ), 0);

  FirstController = (MrcControllerExist (MrcData, cCONTROLLER0)) ? 0 : 1;
  if (Lpddr) {
    FirstChannel = Outputs->Controller[FirstController].FirstPopCh;
  } else {
    FirstChannel = 0;
  }
  FirstCCCBlock = (FirstController * MAX_CHANNEL) + FirstChannel;

  //VsxHi FF Equations (Reading and initializing)
  CompCtl0.Data = MrcReadCR (MrcData, DDRPHY_COMP_CR_DDRCRCOMPCTL0_REG);
  if (CompCtl0.Bits.ClkVoltageSelect == VoltageSelect_VccDDQ) {
    CCCBlock[VoltageSelect_VccDDQ] = ClkVssHi;
  } else if (CompCtl0.Bits.CaVoltageSelect == VoltageSelect_VccDDQ) {
    CCCBlock[VoltageSelect_VccDDQ] = CaVssHi;
  } else if (CompCtl0.Bits.CtlVoltageSelect == VoltageSelect_VccDDQ) {
    CCCBlock[VoltageSelect_VccDDQ] = CtlVssHi;
  } else {
    CCCBlock[VoltageSelect_VccDDQ] = MaxCccVssHi;
  }
  if (CompCtl0.Bits.ClkVoltageSelect == VoltageSelect_Vccdd2) {
    CCCBlock[VoltageSelect_Vccdd2] = ClkVssHi;
  } else if (CompCtl0.Bits.CaVoltageSelect == VoltageSelect_Vccdd2) {
    CCCBlock[VoltageSelect_Vccdd2] = CaVssHi;
  } else if (CompCtl0.Bits.CtlVoltageSelect == VoltageSelect_Vccdd2) {
    CCCBlock[VoltageSelect_Vccdd2] = CtlVssHi;
  } else {
    CCCBlock[VoltageSelect_Vccdd2] = MaxCccVssHi;
  }

  CompOvr0.Data = MrcReadCR (MrcData, DDRPHY_COMP_CR_DDRCRCOMPOVR0_REG);
  CompOvr0.Bits.ClkDrvU = 1;
  CompOvr0.Bits.CtlDrvU = 1;
  CompOvr0.Bits.CmdDrvU = 1;
  CompOvr0.Bits.DqDrvU = 1;
  MrcWriteCR (MrcData, DDRPHY_COMP_CR_DDRCRCOMPOVR0_REG, CompOvr0.Data);
  CaPhyComp.Data = MrcReadCR (MrcData, DDRPHY_COMP_CR_DDRCRCACOMP_REG);
  OrigCaPhyComp = CaPhyComp.Data;
  CtlPhyComp.Data = MrcReadCR (MrcData, DDRPHY_COMP_CR_DDRCRCTLCOMP_REG);
  OrigCtlPhyComp = CtlPhyComp.Data;
  ClkPhyComp.Data = MrcReadCR (MrcData, DDRPHY_COMP_CR_DDRCRCLKCOMP_REG);
  OrigClkPhyComp = ClkPhyComp.Data;
  Offset = OFFSET_CALC_CH (CH0CCC_CR_DDRCRCACOMP_REG, CH1CCC_CR_DDRCRCACOMP_REG, FirstCCCBlock);
  CaComp.Data = MrcReadCR (MrcData, Offset);
  OrigCaComp = CaComp.Data;
  Offset = OFFSET_CALC_CH (CH0CCC_CR_DDRCRCTLCOMP_REG, CH1CCC_CR_DDRCRCTLCOMP_REG, FirstCCCBlock);
  CtlComp.Data = MrcReadCR (MrcData, Offset);
  OrigCtlComp = CtlComp.Data;
  Offset = OFFSET_CALC_CH (CH0CCC_CR_DDRCRCLKCOMP_REG, CH1CCC_CR_DDRCRCLKCOMP_REG, FirstCCCBlock);
  ClkComp.Data = MrcReadCR (MrcData, Offset);
  OrigClkComp = ClkComp.Data;
  RcompData0.Data = MrcReadCR (MrcData, DDRPHY_COMP_CR_DDRCRDATACOMP0_REG);
  DataComp0.Data = RcompData0.Data;
  OrigDataRcompDrvUp = DataComp0.Bits.RcompDrvUp;
  //VsxHi FF Equations (Updating Rcomp fields)
  CaPhyComp.Bits.RcompDrvUp = RCompCodes[0];
  CtlPhyComp.Bits.RcompDrvUp = RCompCodes[0];
  ClkPhyComp.Bits.RcompDrvUp = RCompCodes[0];
  CaComp.Bits.RcompDrvUp = RCompCodes[0];
  CtlComp.Bits.RcompDrvUp = RCompCodes[0];
  ClkComp.Bits.RcompDrvUp = RCompCodes[0];
  DataComp0.Bits.RcompDrvUp = RCompCodes[0];
  //VsxHi FF Equations (Write registers)
  MrcWriteCR (MrcData, DDRPHY_COMP_CR_DDRCRCACOMP_REG, CaPhyComp.Data);
  MrcWriteCrMulticast (MrcData, CCC_CR_DDRCRCACOMP_REG, CaComp.Data);
  MrcWriteCR (MrcData, DDRPHY_COMP_CR_DDRCRCTLCOMP_REG, CtlPhyComp.Data);
  MrcWriteCrMulticast (MrcData, CCC_CR_DDRCRCTLCOMP_REG, CtlComp.Data);
  MrcWriteCR (MrcData, DDRPHY_COMP_CR_DDRCRCLKCOMP_REG, ClkPhyComp.Data);
  MrcWriteCrMulticast (MrcData, CCC_CR_DDRCRCLKCOMP_REG, ClkComp.Data);
  MrcWriteCR (MrcData, DDRPHY_COMP_CR_DDRCRDATACOMP0_REG, DataComp0.Data);
  //Force 1st Comp
  ForceRcomp (MrcData);

  //VssHi (Store Index = 0, Write Out Index = 1)
  Index = 0;
  MrcVssHiCompCalSetup (MrcData, CCCVssHi2, CCCVssHiQ, DQVssHiQ, CCCBlock, FirstCCCBlock, Index, RCompCodes);

  for (Index = 1; Index < MaxIndex; Index++) {
    //Force Comp
    ForceRcomp (MrcData);

    //VsxHi FF Equations (Store Index, Write Out Index+1)
    MrcVssHiCompCalSetup (MrcData, CCCVssHi2, CCCVssHiQ, DQVssHiQ, CCCBlock, FirstCCCBlock, Index, RCompCodes);
  }

  VsxHiFFCompRef0.Data = 0;
  VsxHiFFCompRef1.Data = 0;

  if (CCCBlock[VoltageSelect_VccDDQ] != MaxCccVssHi) {
    //Calculate VssHi Coefficients for CCC VddQ
    MrcVssHiCoeffCalc (MrcData, CCCVssHiQ, &FFRefBit5, &FFRefBit4, &FFRefBit3, &FFRefBit2, &FFRefLSB, &FFRefStatic);
    VsxHiFFCompRef0.Bits.CCCQFFRefBit5 = FFRefBit5;
    VsxHiFFCompRef0.Bits.CCCQFFRefBit4 = FFRefBit4;
    VsxHiFFCompRef0.Bits.CCCQFFRefBit3 = FFRefBit3;
    VsxHiFFCompRef0.Bits.CCCQFFRefBit2 = FFRefBit2;
    VsxHiFFCompRef0.Bits.CCCQFFRefLSB = FFRefLSB;
    VsxHiFFCompRef0.Bits.CCCQFFRefStatic = FFRefStatic;
  }

  if (CCCBlock[VoltageSelect_Vccdd2] != MaxCccVssHi) {
    //Calculate VssHi Coefficients for CCC Vdd2
    MrcVssHiCoeffCalc (MrcData, CCCVssHi2, &FFRefBit5, &FFRefBit4, &FFRefBit3, &FFRefBit2, &FFRefLSB, &FFRefStatic);
    VsxHiFFCompRef0.Bits.CCC2FFRefBit5 = FFRefBit5;
    VsxHiFFCompRef0.Bits.CCC2FFRefBit4 = FFRefBit4;
    VsxHiFFCompRef0.Bits.CCC2FFRefBit3 = FFRefBit3;
    VsxHiFFCompRef1.Bits.CCC2FFRefBit2 = FFRefBit2;
    VsxHiFFCompRef1.Bits.CCC2FFRefLSB = FFRefLSB;
    VsxHiFFCompRef1.Bits.CCC2FFRefStatic = FFRefStatic;
  }

  //Calculate VssHi Coefficients for DQ VddQ
  MrcVssHiCoeffCalc (MrcData, DQVssHiQ, &FFRefBit5, &FFRefBit4, &FFRefBit3, &FFRefBit2, &FFRefLSB, &FFRefStatic);
  VsxHiFFCompRef1.Bits.DQFFRefBit5 = FFRefBit5;
  VsxHiFFCompRef1.Bits.DQFFRefBit4 = FFRefBit4;
  VsxHiFFCompRef1.Bits.DQFFRefBit3 = FFRefBit3;
  VsxHiFFCompRef1.Bits.DQFFRefBit2 = FFRefBit2;
  VsxHiFFCompRef1.Bits.DQFFRefLsb = FFRefLSB;
  VsxHiFFCompRef1.Bits.DQFFRefStatic = FFRefStatic;

  MrcWriteCR (MrcData, DDRPHY_COMP_CR_VSXHIFFCOMPREF0_REG, VsxHiFFCompRef0.Data);
  MrcWriteCR (MrcData, DDRPHY_COMP_CR_VSXHIFFCOMPREF1_REG, VsxHiFFCompRef1.Data);

  CompOvr0.Bits.ClkDrvU = 0;
  CompOvr0.Bits.CtlDrvU = 0;
  CompOvr0.Bits.CmdDrvU = 0;
  CompOvr0.Bits.DqDrvU = 0;
  MrcWriteCR (MrcData, DDRPHY_COMP_CR_DDRCRCOMPOVR0_REG, CompOvr0.Data);
  MrcWriteCR (MrcData, DDRPHY_COMP_CR_DDRCRCACOMP_REG, OrigCaPhyComp);
  MrcWriteCR (MrcData, DDRPHY_COMP_CR_DDRCRCTLCOMP_REG, OrigCtlPhyComp);
  MrcWriteCR (MrcData, DDRPHY_COMP_CR_DDRCRCLKCOMP_REG, OrigClkPhyComp);
  MrcWriteCrMulticast (MrcData, CCC_CR_DDRCRCACOMP_REG, OrigCaComp);
  MrcWriteCrMulticast (MrcData, CCC_CR_DDRCRCTLCOMP_REG, OrigCtlComp);
  MrcWriteCrMulticast (MrcData, CCC_CR_DDRCRCLKCOMP_REG, OrigClkComp);
  RcompData0.Data = MrcReadCR (MrcData, DDRPHY_COMP_CR_DDRCRDATACOMP0_REG);
  RcompData0.Bits.RcompDrvUp = OrigDataRcompDrvUp;
  MrcWriteCR (MrcData, DDRPHY_COMP_CR_DDRCRDATACOMP0_REG, RcompData0.Data);

  //Force Comp
  ForceRcomp (MrcData);

  return mrcSuccess;
}

/**
  This function records the results of RcompDrvUp for VsxHi LVR

  @param[in, out] MrcData        - Include all MRC global data.
  @param[in, out] ClkRcompDrvUp  - CLK RcompDrvUp (Current Total)
  @param[in, out] CaRcompDrvUp   - CA RcompDrvUp (Current Total)
  @param[in, out] CtlRcompDrvUp  - CTL RcompDrvUp (Current Total)
  @param[in, out] NumOfComps     - # of comps stored (used for division to get average)
**/
VOID
MrcVsxHiLVRDataGathering (
  IN OUT MrcParameters *const MrcData,
  IN OUT UINT32  *ClkRcompDrvUp,
  IN OUT UINT32  *CaRcompDrvUp,
  IN OUT UINT32  *CtlRcompDrvUp,
  IN OUT UINT32  *NumOfComps
  )
{
  DDRPHY_COMP_CR_DDRCRCACOMP_STRUCT  CaPhyComp;
  DDRPHY_COMP_CR_DDRCRCLKCOMP_STRUCT  ClkPhyComp;
  DDRPHY_COMP_CR_DDRCRCTLCOMP_STRUCT  CtlPhyComp;

  if (CaRcompDrvUp != NULL) {
    CaPhyComp.Data = MrcReadCR (MrcData, DDRPHY_COMP_CR_DDRCRCACOMP_REG);
    *CaRcompDrvUp += CaPhyComp.Bits.RcompDrvUp;
  }

  if (ClkRcompDrvUp != NULL) {
    ClkPhyComp.Data = MrcReadCR (MrcData, DDRPHY_COMP_CR_DDRCRCLKCOMP_REG);
    *ClkRcompDrvUp += ClkPhyComp.Bits.RcompDrvUp;
  }

  if (CtlRcompDrvUp != NULL) {
    CtlPhyComp.Data = MrcReadCR (MrcData, DDRPHY_COMP_CR_DDRCRCTLCOMP_REG);
    *CtlRcompDrvUp += CtlPhyComp.Bits.RcompDrvUp;
  }

  if (NumOfComps != NULL) {
    *NumOfComps += 1;
  }
}

/**
  This function does the one time compensation for LVR for Rload value.

  @param[in, out] MrcData    - Include all MRC global data.
**/
VOID
MrcVsxHiLVROneTime (
  IN OUT MrcParameters *const MrcData
  )
{
  MrcInput          *Inputs;
  DDRPHY_COMP_CR_VCCDLLREPLICACTRL1_STRUCT        VccDllReplicaCtrl1;
  DDRPHY_COMP_CR_VCCDLLCOMPDLL_STRUCT             VccDllCompDll;
  DDRPHY_COMP_CR_DDRCRCOMPDVFSRLOAD_STRUCT        CompDvfsRload;

  Inputs     = &MrcData->Inputs;

  VccDllReplicaCtrl1.Data = MrcReadCR (MrcData, DDRPHY_COMP_CR_VCCDLLREPLICACTRL1_REG);
  //Dividend is the target Resistance load
  VccDllReplicaCtrl1.Bits.RloadInstances = DIVIDECEIL (800, Inputs->RcompResistor);
  MrcWriteCR (MrcData, DDRPHY_COMP_CR_VCCDLLREPLICACTRL1_REG, VccDllReplicaCtrl1.Data);
  ForceRcomp (MrcData);

  VccDllCompDll.Data = MrcReadCR (MrcData, DDRPHY_COMP_CR_VCCDLLCOMPDLL_REG);
  CompDvfsRload.Data = MrcReadCR (MrcData, DDRPHY_COMP_CR_DDRCRCOMPDVFSRLOAD_REG);

  if (VccDllCompDll.Bits.RloadDqs != CompDvfsRload.Bits.VsxHiLVRBias) {
    CompDvfsRload.Bits.VsxHiLVRBias = VccDllCompDll.Bits.RloadDqs;
    MrcWriteCR (MrcData, DDRPHY_COMP_CR_DDRCRCOMPDVFSRLOAD_REG, CompDvfsRload.Data);
  }

}

/**
  This function runs early DVFS compensation for VsxHi LVR and calculates Error against the
  average RcompDrvUp data for finding optimal value to apply to VsxHiLVRTargetDvfs

  @param[in, out] MrcData     - Include all MRC global data.
  @param[in] CaRcompDrvUpAvg  - Average RcompDrvUp data for Ca.
  @param[in] ClkRcompDrvUpAvg - Average RcompDrvUp data for Clk.
  @param[in] CtlRcompDrvUpAvg - Average RcompDrvUp data for Ctl.

**/
VOID
MrcVsxHiLVREarlyDvfs (
  IN OUT MrcParameters *const MrcData,
  IN UINT32  CaRcompDrvUpAvg,
  IN UINT32  ClkRcompDrvUpAvg,
  IN UINT32  CtlRcompDrvUpAvg
  )
{
  MrcOutput         *Outputs;
  MrcInput          *Inputs;
  UINT8             Counter;
  UINT32            Vdd2Mv;
  UINT32            VccDdqMv;
  UINT32            VccIoMv;
  UINT32            TargetVsxHi;
  UINT32            TargetLow;
  UINT32            TargetHigh;
  INT32             TargetV0;
  INT32             Error;
  INT64             GetSetVal;
  BOOLEAN           ErrorMinimized;
  BOOLEAN           AddClkVoltageSelect;
  BOOLEAN           AddCtlVoltageSelect;
  BOOLEAN           AddCaVoltageSelect;
  DDRPHY_COMP_CR_DDRCRCOMPCTL0_STRUCT  CompCtl0;
  DDRPHY_COMP_CR_DDRCRCOMPDVFSRLOAD_STRUCT  CompDvfsRLoad;
  DDRPHY_COMP_CR_DDRCRCACOMP_STRUCT  PhyComp;

  Outputs     = &MrcData->Outputs;
  Inputs      = &MrcData->Inputs;
  Vdd2Mv      = Outputs->Vdd2Mv;
  Vdd2Mv      = MAX (Vdd2Mv, 1);
  VccDdqMv    = Outputs->VccddqVoltage;
  VccIoMv     = Inputs->VccIomV;
  TargetHigh  = DDRPHY_COMP_CR_DDRCRCOMPDVFSRLOAD_VsxHiLVRTargetDvfs_MAX;
  TargetLow   = DDRPHY_COMP_CR_DDRCRCOMPDVFSRLOAD_VsxHiLVRTargetDvfs_MIN;
  ErrorMinimized = FALSE;

  if (!Outputs->Lpddr) {
    return;
  }

  GetSetVal = 1;
  MrcGetSetNoScope (MrcData, GsmIocScramOvrdPeriodicToDvfsComp, WriteCached, &GetSetVal);

  CompCtl0.Data = MrcReadCR (MrcData, DDRPHY_COMP_CR_DDRCRCOMPCTL0_REG);
  CompDvfsRLoad.Data = MrcReadCR (MrcData, DDRPHY_COMP_CR_DDRCRCOMPDVFSRLOAD_REG);
  TargetVsxHi = Vdd2Mv - (VccDdqMv - (CompCtl0.Bits.CCCVssHiBypassVddqMode ? 0 : Vdd2Mv - VccIoMv));
  TargetVsxHi = RANGE (TargetVsxHi, 100, 450); //Unit is mV

  TargetV0 = (Vdd2Mv + 4 * TargetVsxHi) * 192;  //unit in # of millivolt
  TargetV0 /= (Vdd2Mv * 5); //unitless and *1 granularity
  TargetV0 -= 32;
  TargetV0 = RANGE (TargetV0, 0, 127);

  CompDvfsRLoad.Bits.VsxHiLVRTargetDvfs = (UINT32) TargetV0;
  MrcWriteCR (MrcData, DDRPHY_COMP_CR_DDRCRCOMPDVFSRLOAD_REG, CompDvfsRLoad.Data);
  AddClkVoltageSelect = (CompCtl0.Bits.ClkVoltageSelect == VoltageSelect_VccDDQ);
  AddCtlVoltageSelect = (CompCtl0.Bits.CtlVoltageSelect == VoltageSelect_VccDDQ);
  AddCaVoltageSelect = (CompCtl0.Bits.CaVoltageSelect == VoltageSelect_VccDDQ);

  Counter = 0;
  do {
    //Force DVFS Comp
    ForceRcomp (MrcData);

    Counter++;
    Error = 0;
    if (AddClkVoltageSelect) {
      PhyComp.Data = MrcReadCR (MrcData, DDRPHY_COMP_CR_DDRCRCLKCOMP_REG);
      Error += (INT32) (PhyComp.Bits.RcompDrvUp - ClkRcompDrvUpAvg);
    }
    if (AddCtlVoltageSelect) {
      PhyComp.Data = MrcReadCR (MrcData, DDRPHY_COMP_CR_DDRCRCTLCOMP_REG);
      Error += (INT32) (PhyComp.Bits.RcompDrvUp - CtlRcompDrvUpAvg);
    }
    if (AddCaVoltageSelect) {
      PhyComp.Data = MrcReadCR (MrcData, DDRPHY_COMP_CR_DDRCRCACOMP_REG);
      Error += (INT32) (PhyComp.Bits.RcompDrvUp - CaRcompDrvUpAvg);
    }

    if (Error > 0) {
      //Decrease TargetV0
      TargetHigh = TargetV0;
      TargetV0 += TargetLow;
      TargetV0 /= 2;
    } else if (Error < 0) {
      //Increase TargetV0
      TargetLow = TargetV0;
      TargetV0 += TargetHigh;
      TargetV0 /= 2;
    } else {
      //No change to TargetV0, Done
      TargetLow = TargetHigh = TargetV0;
    }

    if (TargetLow == TargetHigh) {
      //Here when either Error == 0 or when no more range left to sweep
      ErrorMinimized = TRUE;
    } else {
      CompDvfsRLoad.Bits.VsxHiLVRTargetDvfs = (UINT32) TargetV0;
      MrcWriteCR (MrcData, DDRPHY_COMP_CR_DDRCRCOMPDVFSRLOAD_REG, CompDvfsRLoad.Data);
    }
  } while ((ErrorMinimized == FALSE) && (Counter < 10));
  MRC_DEBUG_ASSERT (Counter < 10, &Outputs->Debug, "MrcVsxHiLVREarlyDvfs failed to Minimize Error!\n");

  //Restore POR CR values
  GetSetVal = 0;
  MrcGetSetNoScope (MrcData, GsmIocScramOvrdPeriodicToDvfsComp, WriteCached, &GetSetVal);

  //Clear DvfsModeEn
  ForceRcomp (MrcData);

}

/**
  This function Reads or Clears out NUI related fields

  @param[in, out] MrcData - Include all MRC global data.
  @param[in]      Read    - TRUE if only reading, FALSE if only clearing
  @param[in]      Index   - # of Comp Cycle this is on
**/
void
MrcDdrCompNUI (
  IN OUT MrcParameters *const MrcData,
  IN     BOOLEAN              Read,
  IN     UINT32               Index
  )
{
  DDRPHY_COMP_CR_DDRCRDATACOMP1_STRUCT  RcompData1;
  DDRPHY_COMP_CR_DDRCRDATACOMP0_STRUCT  RcompData0;
  MrcDebug   *Debug;
  MrcOutput  *Outputs;
  UINT32     DbgLevel;
  UINT32     TotalDelayP;
  UINT32     TotalDelayN;
  UINT32     TargetP;
  UINT32     TargetN;
  UINT32     OffsetP;
  UINT32     OffsetN;
  UINT32     TotalDelay;

  Outputs = &MrcData->Outputs;
  Debug = &Outputs->Debug;
  DbgLevel = MSG_LEVEL_NOTE;

  RcompData1.Data = MrcReadCR (MrcData, DDRPHY_COMP_CR_DDRCRDATACOMP1_REG);

  if (Read) {
    RcompData0.Data = MrcReadCR (MrcData, DDRPHY_COMP_CR_DDRCRDATACOMP0_REG);
    MRC_DEBUG_MSG (Debug, DbgLevel, "\n%sNUI_values\n", gStartTagStr);
    MRC_DEBUG_MSG (Debug, DbgLevel, "RcompData1.Bits.RxDqsDelayP = %d\nRcompData1.Bits.RxDqsDelayN = %d\n", RcompData1.Bits.RxDqsDelayP, RcompData1.Bits.RxDqsDelayN);
    MRC_DEBUG_MSG (Debug, DbgLevel, "RcompData1.Bits.DqsPTargetNUI = %d\nRcompData1.Bits.DqsPOffsetNUI = %d\n", RcompData1.Bits.DqsPTargetNUI, RcompData1.Bits.DqsPOffsetNUI);
    MRC_DEBUG_MSG (Debug, DbgLevel, "RcompData0.Bits.DqsNTargetNUI = %d\nRcompData0.Bits.DqsNOffsetNUI = %d\n", RcompData0.Bits.DqsNTargetNUI, RcompData0.Bits.DqsNOffsetNUI);
    MRC_DEBUG_MSG (Debug, DbgLevel, "%sNUI_values\n", gStopTagStr);
    if ((Outputs->Lpddr) && (Index == 4)) {
      TargetP = RcompData1.Bits.DqsPTargetNUI;
      TargetN = RcompData0.Bits.DqsNTargetNUI;
      OffsetP = RcompData1.Bits.DqsPOffsetNUI;
      OffsetN = RcompData0.Bits.DqsNOffsetNUI;
      TotalDelayP = TargetP + 1 - OffsetP;
      TotalDelayN = TargetN + 1 - OffsetN;
      if (Outputs->DdrType == MRC_DDR_TYPE_LPDDR4) {
        if ((TotalDelayP > 2) || (TotalDelayN > 2)) {
          MRC_DEBUG_MSG (Debug, DbgLevel, "LPDDR4 does not support 3UI, forcing to 2UI\n");
        }
        if (TotalDelayP > 2) {
          RcompData1.Bits.DqsPTargetNUI = TargetP = 3;
          RcompData1.Bits.DqsPOffsetNUI = OffsetP = 2;
          MrcWriteCR (MrcData, DDRPHY_COMP_CR_DDRCRDATACOMP1_REG, RcompData1.Data);
          TotalDelayP = 2;
          MRC_DEBUG_MSG (Debug, DbgLevel, "DqsPTargetNUI = %d\nDqsPOffsetNUI = %d\n", RcompData1.Bits.DqsPTargetNUI, RcompData1.Bits.DqsPOffsetNUI);
        }
        if (TotalDelayN > 2) {
          RcompData0.Bits.DqsNTargetNUI = TargetN = 3;
          RcompData0.Bits.DqsNOffsetNUI = OffsetN = 2;
          MrcWriteCR (MrcData, DDRPHY_COMP_CR_DDRCRDATACOMP0_REG, RcompData0.Data);
          TotalDelayN = 2;
          MRC_DEBUG_MSG (Debug, DbgLevel, "DqsNTargetNUI = %d\nDqsNOffsetNUI = %d\n", RcompData0.Bits.DqsNTargetNUI, RcompData0.Bits.DqsNOffsetNUI);
        }
      }

      if ((TotalDelayP != TotalDelayN) || (TargetP != TargetN)) {
        TotalDelay = MAX (TotalDelayP, TotalDelayN);
        if (TotalDelay != TotalDelayP) {
          //Adjust DelayP
          TargetP = TargetN;
          OffsetP = OffsetN;
        } else if (TotalDelay != TotalDelayN) {
          //Adjust DelayN
          TargetN = TargetP;
          OffsetN = OffsetP;
        } else {
          //TotalDelay matches but TargetP and TargetN does not
          TotalDelay = MAX (TargetP, TargetN);
          if (TotalDelay == TargetP) {
            //Adjust DelayN
            TargetN = TargetP;
            OffsetN = OffsetP;
          } else {
            //Adjust DelayP
            TargetP = TargetN;
            OffsetP = OffsetN;
          }
        }
        RcompData1.Bits.DqsPTargetNUI = TargetP;
        RcompData1.Bits.DqsPOffsetNUI = OffsetP;
        MrcWriteCR (MrcData, DDRPHY_COMP_CR_DDRCRDATACOMP1_REG, RcompData1.Data);
        RcompData0.Bits.DqsNTargetNUI = TargetN;
        RcompData0.Bits.DqsNOffsetNUI = OffsetN;
        MrcWriteCR (MrcData, DDRPHY_COMP_CR_DDRCRDATACOMP0_REG, RcompData0.Data);
        MRC_DEBUG_MSG (Debug, DbgLevel, "Mismatch Delay: RcompData0.Bits.DqsNTargetNUI = %d\nRcompData0.Bits.DqsNOffsetNUI = %d\n", TargetN, OffsetN);
        MRC_DEBUG_MSG (Debug, DbgLevel, "RcompData1.Bits.DqsPTargetNUI = %d\nRcompData1.Bits.DqsPOffsetNUI = %d\n", TargetP, OffsetP);
      }
    }
  } else {
    RcompData1.Bits.RxDqsDelayN = 0;
    RcompData1.Bits.RxDqsDelayP = 0;
    MrcWriteCR (MrcData, DDRPHY_COMP_CR_DDRCRDATACOMP1_REG, RcompData1.Data);
  }
}

/**
  This function runs First Comp Calibration - PreTraining
  (VsxHi/Vtt Panic, PDB Cal, Unmatched, View Pins, SideClkPulse)

  @param[in, out] MrcData - Include all MRC global data.

  @retval mrcSuccess
**/
MrcStatus
MrcDdrCompCalPre (
  IN OUT MrcParameters *const MrcData
  )
{
  MrcInput           *Inputs;
  MrcOutput          *Outputs;
  BOOLEAN            HVM;
  BOOLEAN            UnMatched;
  BOOLEAN            Lpddr;
  BOOLEAN            Lpddr4;
  UINT8              Gear2;
  INT8               DqsPTarget;
  INT8               DqsPOffset;
  INT8               DqsNTarget;
  INT8               DqsNOffset;
  INT8               DqsPResult;
  INT8               DqsNResult;
  UINT16             RTargetUp;
  UINT16             RTargetDown;
  UINT16             QclkPs;
  UINT16             TempValU16;
  UINT32             Index;
  UINT32             MaxIndex;
  UINT32             FirstController;
  UINT32             FirstChannel;
  UINT32             FirstByte;
  UINT32             FirstCCCBlock;
  UINT32             Offset;
  UINT32             TempValU32;
  UINT32             Temp1ValU32;
  UINT32             DivisorU32;
  UINT32             OrigCompCtl3DqDrvVrefDn;
  UINT32             OrigCompCtl4DqDrvVrefUp;
  UINT32             CaRcompDrvUp;
  UINT32             ClkRcompDrvUp;
  UINT32             CtlRcompDrvUp;
  UINT32             NumOfComps;
  INT64              GetSetVal;
  BOOLEAN            UyA0;
  BOOLEAN            UlxUlt;
  DDRPHY_COMP_CR_VSSHIPANIC_STRUCT  VssHiPanic;
  DDRPHY_COMP_CR_DDRCRVSXHICOMPDATA_STRUCT  VsxHiCompData;
  DDRPHY_COMP_CR_DDRCRCOMPVTTPANIC_STRUCT  VttPanic;
  DDRPHY_COMP_CR_DDRCRCOMPVTTPANIC2_STRUCT  VttPanic2;
  DATA0CH0_CR_DATACOMPVTT_STRUCT  VttCompData;
  DDRPHY_COMP_CR_DDRCRVIEWCTL_STRUCT  ViewCtl;
  DDRPHY_COMP_CR_DDRCRCOMPCTL3_STRUCT  CompCtl3;
  DDRPHY_COMP_CR_DDRCRCOMPCTL4_STRUCT  CompCtl4;
  DDRPHY_COMP_CR_VCCDLLDQSDELAY_STRUCT  VccDllDqsDelay;
  DATA0CH0_CR_RCOMPDATA0_STRUCT  RcompData0;
  DATA0CH0_CR_RCOMPDATA1_STRUCT  RcompData1;
  DLLDDR_CR_DDRCRVCCDLLFFNBIAS_STRUCT  VccDllFfNbias;
  DDRPHY_COMP_CR_DDRCRCOMPOVR1_STRUCT  DdrCrCompOvr1;

  Inputs       = &MrcData->Inputs;
  Outputs      = &MrcData->Outputs;
  HVM          = FALSE; //@todo Add option to Inputs parameter
  RTargetUp    = 50; //Ohms
  RTargetDown  = 50; //Ohms
  QclkPs       = Outputs->Qclkps;
  UnMatched    = ((Outputs->RxMode == MrcRxModeUnmatchedRxWRload) || (Outputs->RxMode == MrcRxModeUnmatchedRxWPpath));
  Gear2        = (Outputs->Gear2) ? 1 : 0;
  Lpddr        = Outputs->Lpddr;
  Lpddr4       = (Outputs->DdrType == MRC_DDR_TYPE_LPDDR4);
  MaxIndex     = 4;
  UyA0         = Inputs->A0;
  UlxUlt       = Inputs->UlxUlt;
  VccDllDqsDelay.Data = 0;

  FirstController = (MrcControllerExist (MrcData, cCONTROLLER0)) ? cCONTROLLER0 : cCONTROLLER1;
  if (Lpddr) {
    FirstChannel = Outputs->Controller[FirstController].FirstPopCh;
  } else {
    FirstChannel = 0;
  }
  FirstByte = (MAX_BYTE_IN_LP_CHANNEL * FirstChannel); //@todo : For DDR5 MAX_BYTE_IN_CHANNEL != 2 so add that
  FirstCCCBlock = (FirstController * MAX_CHANNEL) + FirstChannel;

  // Clean out DqsXOffsetNUI
  RcompData0.Data = MrcReadCR (MrcData, DDRPHY_COMP_CR_DDRCRDATACOMP0_REG);
  RcompData0.Bits.DqsNOffsetNUI = 0;
  RcompData1.Data = MrcReadCR (MrcData, DDRPHY_COMP_CR_DDRCRDATACOMP1_REG);
  RcompData1.Bits.DqsPOffsetNUI = 0;
  MrcWriteCR (MrcData, DDRPHY_COMP_CR_DDRCRDATACOMP0_REG, RcompData0.Data);
  MrcWriteCR (MrcData, DDRPHY_COMP_CR_DDRCRDATACOMP1_REG, RcompData1.Data);

  //1st Comp (Adjust Panic Drivers, View Pin Comp, PBD Deskew, Unmatched RX, VsxHi FF Equations)
  //Adjust Panic Drivers
  VssHiPanic.Data = MrcReadCR (MrcData, DDRPHY_COMP_CR_VSSHIPANIC_REG);
  VttPanic2.Data = MrcReadCR (MrcData, DDRPHY_COMP_CR_DDRCRCOMPVTTPANIC2_REG);
  VssHiPanic.Bits.PanicPDn2xStep = (HVM) ? 1 : 0;
  VttPanic2.Bits.PanicPDn2xStep = (HVM) ? 1 : 0;
  MrcWriteCR (MrcData, DDRPHY_COMP_CR_VSSHIPANIC_REG, VssHiPanic.Data);
  MrcWriteCR (MrcData, DDRPHY_COMP_CR_DDRCRCOMPVTTPANIC2_REG, VttPanic2.Data);

  //View Pin Comp
  ViewCtl.Data = MrcReadCR (MrcData, DDRPHY_COMP_CR_DDRCRVIEWCTL_REG);
  ViewCtl.Bits.WrEnViewDrv = 1;
  MrcWriteCR (MrcData, DDRPHY_COMP_CR_DDRCRVIEWCTL_REG, ViewCtl.Data);
  CompCtl3.Data = MrcReadCR (MrcData, DDRPHY_COMP_CR_DDRCRCOMPCTL3_REG);
  CompCtl4.Data = MrcReadCR (MrcData, DDRPHY_COMP_CR_DDRCRCOMPCTL4_REG);
  OrigCompCtl3DqDrvVrefDn = CompCtl3.Bits.DqDrvVrefDn;
  OrigCompCtl4DqDrvVrefUp = CompCtl4.Bits.DqDrvVrefUp;
  TempValU32 = (191 * RTargetDown) / (RTargetDown + RTargetUp);
  TempValU32 = MIN (191, TempValU32);
  CompCtl3.Bits.DqDrvVrefDn = TempValU32;
  TempValU32 = (191 * 100) / (100 + RTargetUp);
  TempValU32 = MIN (191, TempValU32);
  CompCtl4.Bits.DqDrvVrefUp = TempValU32;
  MrcWriteCR (MrcData, DDRPHY_COMP_CR_DDRCRCOMPCTL3_REG, CompCtl3.Data);
  MrcWriteCR (MrcData, DDRPHY_COMP_CR_DDRCRCOMPCTL4_REG, CompCtl4.Data);

  //PBD Deskew
  TempValU16 = (UlxUlt) ? 100 : 90;
  GetSetVal = UDIVIDEROUND (QclkPs, TempValU16);
  MrcGetSetNoScope (MrcData, CmdSlewRate, WriteToCache, &GetSetVal);
  GetSetVal = 1;
  MrcGetSetNoScope (MrcData, CmdScompPC, WriteToCache, &GetSetVal);
  MrcFlushRegisterCachedData (MrcData);

  //Unmatched RX
  if (UnMatched) {
    RcompData0.Data = MrcReadCR (MrcData, DDRPHY_COMP_CR_DDRCRDATACOMP0_REG);
    RcompData1.Data = MrcReadCR (MrcData, DDRPHY_COMP_CR_DDRCRDATACOMP1_REG);

    RcompData0.Bits.DqsNTargetNUI = Gear2;
    RcompData1.Bits.DqsPTargetNUI = Gear2;

    MrcWriteCR (MrcData, DDRPHY_COMP_CR_DDRCRDATACOMP0_REG, RcompData0.Data);
    MrcWriteCR (MrcData, DDRPHY_COMP_CR_DDRCRDATACOMP1_REG, RcompData1.Data);

    //Clear
    MrcDdrCompNUI (MrcData, MRC_NUI_CLEAR, 0);

    VccDllDqsDelay.Data = MrcReadCR (MrcData, DDRPHY_COMP_CR_VCCDLLDQSDELAY_REG);
    VccDllDqsDelay.Bits.RxDqsOffsetComp = 0;
    VccDllDqsDelay.Bits.TrainTargetOffsetUI = 1;
    VccDllDqsDelay.Bits.Gear1 = ~Gear2;
    MrcWriteCR (MrcData, DDRPHY_COMP_CR_VCCDLLDQSDELAY_REG, VccDllDqsDelay.Data);

    MrcDriftProg (MrcData, &VccDllDqsDelay.Data, FirstController, FirstByte);
  }

  MrcVsxHiLVROneTime (MrcData);

  //Force 1st Comp
  ForceRcomp (MrcData);

  if (VccDllDqsDelay.Bits.TrainTargetOffsetUI == 1) {
    //Read
    MrcDdrCompNUI (MrcData, MRC_NUI_READ, 0);
  }

  //Adjust Panic Drivers
  if (HVM == FALSE) {
    VsxHiCompData.Data = MrcReadCR (MrcData, DDRPHY_COMP_CR_DDRCRVSXHICOMPDATA_REG);
    DivisorU32 = MAX (VssHiPanic.Bits.VsxHiPanicCompDnMult, 1);
    TempValU32 = VsxHiCompData.Bits.PanicDrvDn / DivisorU32;
    if (TempValU32 > 40) {
      VssHiPanic.Bits.PanicPDn2xStep = 1;
      MrcWriteCR (MrcData, DDRPHY_COMP_CR_VSSHIPANIC_REG, VssHiPanic.Data);
    }
    VttPanic.Data = MrcReadCR (MrcData, DDRPHY_COMP_CR_DDRCRCOMPVTTPANIC_REG);
    VttPanic2.Data = MrcReadCR (MrcData, DDRPHY_COMP_CR_DDRCRCOMPVTTPANIC2_REG);

    Offset = OFFSET_CALC_MC_CH (DATA0CH0_CR_DATACOMPVTT_REG, DATA0CH1_CR_DATACOMPVTT_REG, FirstController, DATA1CH0_CR_DATACOMPVTT_REG, FirstByte);
    VttCompData.Data = MrcReadCR (MrcData, Offset);
    DivisorU32 = MAX (VttPanic.Bits.VttPanicCompDn0Mult, 1);
    TempValU32 = VttCompData.Bits.PanicVttDn0 / DivisorU32;
    DivisorU32 = MAX (VttPanic.Bits.VttPanicCompDn1Mult, 1);
    Temp1ValU32 = VttCompData.Bits.PanicVttDn1 / DivisorU32;
    TempValU32 = MAX (TempValU32, Temp1ValU32);
    if (TempValU32 > 40) {
      VttPanic2.Bits.PanicPDn2xStep = 1;
      MrcWriteCR (MrcData, DDRPHY_COMP_CR_DDRCRCOMPVTTPANIC2_REG, VttPanic2.Data);
    }
  }

  //View Pin Comp
  ViewCtl.Bits.WrEnViewDrv = 0;
  MrcWriteCR (MrcData, DDRPHY_COMP_CR_DDRCRVIEWCTL_REG, ViewCtl.Data);
  CompCtl3.Bits.DqDrvVrefDn = OrigCompCtl3DqDrvVrefDn;
  CompCtl4.Bits.DqDrvVrefUp = OrigCompCtl4DqDrvVrefUp;
  MrcWriteCR (MrcData, DDRPHY_COMP_CR_DDRCRCOMPCTL3_REG, CompCtl3.Data);
  MrcWriteCR (MrcData, DDRPHY_COMP_CR_DDRCRCOMPCTL4_REG, CompCtl4.Data);

  //PBD Deskew
  MrcGetSetNoScope (MrcData, SCompCodeCmd, ReadUncached | PrintValue, &GetSetVal);
  TempValU32 = (UINT32) (GetSetVal);
  GetSetVal = (TempValU32 >= 30) ? 1 : 0;
  MrcGetSetNoScope (MrcData, GsmIocVccDllRxDeskewCal, WriteToCache, &GetSetVal);
  MrcGetSetNoScope (MrcData, GsmIocVccDllTxDeskewCal, WriteToCache, &GetSetVal);
  MrcGetSetNoScope (MrcData, GsmIocVccDllCccDeskewCal, WriteToCache, &GetSetVal);
  MrcGetSetChStrb (MrcData, MAX_CONTROLLER, MAX_CHANNEL, MAX_SDRAM_IN_DIMM, RxPerBitDeskewCal, WriteToCache, &GetSetVal);
  MrcGetSetChStrb (MrcData, MAX_CONTROLLER, MAX_CHANNEL, MAX_SDRAM_IN_DIMM, TxPerBitDeskewCal, WriteToCache, &GetSetVal);
  MrcGetSetCcc (MrcData, MAX_CONTROLLER, MAX_CHANNEL, MRC_IGNORE_ARG, 0, CccPerBitDeskewCal, WriteToCache, &GetSetVal);
  MrcFlushRegisterCachedData (MrcData);
  GetSetVal = 1;
  MrcGetSetNoScope (MrcData, GsmIocForceCmpUpdt, ForceWriteUncached | PrintValue, &GetSetVal);

  // although not related to PBD, Scomp result is also used to program the VccDLL clock doubler
  Offset = OFFSET_CALC_CH (DLLDDRDATA0_CR_DDRCRVCCDLLFFNBIAS_REG, DLLDDRDATA1_CR_DDRCRVCCDLLFFNBIAS_REG, FirstCCCBlock);
  Temp1ValU32 = DIVIDEROUND (((TempValU32 * 5) + 8), 10);
  VccDllFfNbias.Data = MrcReadCR (MrcData, Offset);
  VccDllFfNbias.Bits.SideClkPulse = MIN (Temp1ValU32, 15);
  MrcWriteCrMulticast (MrcData, DLLDDR_CR_DDRCRVCCDLLFFNBIAS_REG, VccDllFfNbias.Data);

  if (UlxUlt) {
    VccDllFfNbias.Bits.NbiasFastStartup = 0;
    MrcWriteCR (MrcData, DLLDDRCOMP_CR_DDRCRVCCDLLFFNBIAS_REG, VccDllFfNbias.Data);
  }

  CaRcompDrvUp = ClkRcompDrvUp = CtlRcompDrvUp = NumOfComps = 0;
  if (!UlxUlt && Lpddr) {
    MrcVsxHiLVRDataGathering (MrcData, &ClkRcompDrvUp, &CaRcompDrvUp, &CtlRcompDrvUp, &NumOfComps);
  }

  for (Index = 1; Index <= MaxIndex; Index++) {
    if (VccDllDqsDelay.Bits.TrainTargetOffsetUI == 1) {
      //Clear
      MrcDdrCompNUI (MrcData, MRC_NUI_CLEAR, Index);
    }

    //Force Comp
    ForceRcomp (MrcData);

    if (VccDllDqsDelay.Bits.TrainTargetOffsetUI == 1) {
      //Read
      MrcDdrCompNUI (MrcData, MRC_NUI_READ, Index);
    }

    if (!UlxUlt && Lpddr) {
      MrcVsxHiLVRDataGathering (MrcData, &ClkRcompDrvUp, &CaRcompDrvUp, &CtlRcompDrvUp, &NumOfComps);
    }

    if ((Index == 4) && UnMatched) {
      RcompData0.Data = MrcReadCR (MrcData, DDRPHY_COMP_CR_DDRCRDATACOMP0_REG);
      RcompData1.Data = MrcReadCR (MrcData, DDRPHY_COMP_CR_DDRCRDATACOMP1_REG);

      DqsNTarget = (INT8) RcompData0.Bits.DqsNTargetNUI;
      DqsPTarget = (INT8) RcompData1.Bits.DqsPTargetNUI;
      DqsNOffset = (INT8) RcompData0.Bits.DqsNOffsetNUI;
      DqsPOffset = (INT8) RcompData1.Bits.DqsPOffsetNUI;

      if (Lpddr4 && UyA0 && ((DqsPTarget > 1) || (DqsNTarget > 1))) {
        DdrCrCompOvr1.Data = MrcReadCR (MrcData, DDRPHY_COMP_CR_DDRCRCOMPOVR1_REG);
        DdrCrCompOvr1.Bits.DqsNFwdClk = 1;
        DdrCrCompOvr1.Bits.DqsPFwdClk = 1;
        MrcWriteCR (MrcData, DDRPHY_COMP_CR_DDRCRCOMPOVR1_REG, DdrCrCompOvr1.Data);

        DqsNTarget = DqsPTarget = 1;
        DqsNOffset = DqsPOffset = 0;
        RcompData1.Bits.RxDqsDelayN = 0;
        RcompData1.Bits.RxDqsDelayP = 0;
      }

      DqsNResult = DqsNTarget - DqsNOffset;
      DqsPResult = DqsPTarget - DqsPOffset;

      if (DqsPResult > DqsNResult) {
        DqsPTarget = DqsNTarget;
        DqsPOffset = DqsNOffset;
      } else if (DqsPResult < DqsNResult) {
        DqsNTarget = DqsPTarget;
        DqsNOffset = DqsPOffset;
      }

      RcompData0.Bits.DqsNOffsetNUI = DqsNOffset;
      RcompData0.Bits.DqsNTargetNUI = DqsNTarget;
      RcompData1.Bits.DqsPOffsetNUI = DqsPOffset;
      RcompData1.Bits.DqsPTargetNUI = DqsPTarget;

      MrcWriteCR (MrcData, DDRPHY_COMP_CR_DDRCRDATACOMP0_REG, RcompData0.Data);
      MrcWriteCR (MrcData, DDRPHY_COMP_CR_DDRCRDATACOMP1_REG, RcompData1.Data);

      VccDllDqsDelay.Bits.RxDqsOffsetComp = 1;
      VccDllDqsDelay.Bits.TrainTargetOffsetUI = 0;
      MrcWriteCR (MrcData, DDRPHY_COMP_CR_VCCDLLDQSDELAY_REG, VccDllDqsDelay.Data);
    }
  }

  //Force Comp
  ForceRcomp (MrcData);

  if (!UlxUlt && Lpddr) {
    NumOfComps = MAX (NumOfComps, 1);
    CaRcompDrvUp /= NumOfComps;
    ClkRcompDrvUp /= NumOfComps;
    CtlRcompDrvUp /= NumOfComps;
    MrcVsxHiLVREarlyDvfs (MrcData, CaRcompDrvUp, ClkRcompDrvUp, CtlRcompDrvUp);
  }

  return mrcSuccess;
}

/**
  This function train the current supply for the DCO such that the FLL is able to hit 4 GHz.

  @param[in]  MrcData - Pointer to global data.

  @retval MrcStatus - mrcFail if Inputs->ExitOnFailure is set and we cannot reach 4 GHz; otherwise mrcSuccess.
**/
MrcStatus
MrcFllInit (
  IN  MrcParameters *const  MrcData
  )
{
  static const UINT8 FllRefClkDivRatio[4] = {3, 4, 6, 8};
  MrcStatus Status;
  MrcOutput *Outputs;
  UINT32    FllVcoFreq;
  UINT32    RefClkPs;
  UINT8     TuneVal;
  UINT8     FastCalWinVal;
  FLL_STATIC_CFG_0_REG_STRUCT FllStaticCfg0;
  FLL_STATIC_CFG_1_REG_STRUCT FllStaticCfg1;
  FLL_CMD_CFG_REG_STRUCT      FllCmdCfg;
  FLL_CMD_CFG_REG_STRUCT      FllCmdCfgSave;
  FLL_DIAG_STAT_REG_STRUCT    FllDiagStat;
  DDRPHY_COMP_CR_DDRCRFLLWIRED_STRUCT             FllWired;
#ifdef MRC_DEBUG_PRINT
  MrcDebug  *Debug;
  Debug = &MrcData->Outputs.Debug;
#endif

  Outputs = &MrcData->Outputs;
  Status  = mrcSuccess;

  // Get current register configuration and save FllCmdCfg
  FllCmdCfgSave.Data = MrcReadCR (MrcData, FLL_CMD_CFG_REG_REG);
  FllCmdCfg.Data = FllCmdCfgSave.Data;

  // Set FLL in closed loop and init for training
  FllStaticCfg0.Data = MrcReadCR (MrcData, FLL_STATIC_CFG_0_REG_REG);
  FllStaticCfg0.Bits.RUNTIME_CAL = 1;
  MrcWriteCR (MrcData, FLL_STATIC_CFG_0_REG_REG, FllStaticCfg0.Data);
  FastCalWinVal = 1 << FllStaticCfg0.Bits.FAST_CAL_WINDOW_VAL;

  FllCmdCfg.Bits.FLL_RATIO = 66;
  MrcWriteCR (MrcData, FLL_CMD_CFG_REG_REG, FllCmdCfg.Data);

  // Calculate FLL RefClk in pS
  FllStaticCfg1.Data = MrcReadCR (MrcData, FLL_STATIC_CFG_1_REG_REG);
  RefClkPs = (1000000 * FllRefClkDivRatio[FllStaticCfg1.Bits.REFCLK_DIVIDE_RATIO_SEL]) / FLL_REF_CLK;

  // Set Wired to use FLL CRI for ratio
  FllWired.Data = MrcReadCR(MrcData, DDRPHY_COMP_CR_DDRCRFLLWIRED_REG);
  FllWired.Bits.FLLCalEnSrc = 1;
  FllWired.Bits.FLLCalRatioSrc = 1;
  MrcWriteCR (MrcData, DDRPHY_COMP_CR_DDRCRFLLWIRED_REG, FllWired.Data);

  MRC_DEBUG_MSG (Debug, MSG_LEVEL_ALGO, "DcoIrefTune:\tFrequency (MHz)\n");
  for (TuneVal = FLL_CMD_CFG_REG_DCO_IREFTUNE_MIN; TuneVal <= FLL_CMD_CFG_REG_DCO_IREFTUNE_MAX; TuneVal++) {
    // Update the new IrefTune value and enable cold relock.

    // Disable FLL
    FllCmdCfg.Bits.FLL_LDO_ENABLE = 1;
    FllCmdCfg.Bits.FLL_OUT_CLK_REQ_OVRD_EN = 1;
    FllCmdCfg.Bits.FLL_OUT_CLK_REQ = 0;
    FllCmdCfg.Bits.FLL_ENABLE = 0;
    FllCmdCfg.Bits.FLL_LDO_ENABLE = 0;
    MrcWriteCR (MrcData, FLL_CMD_CFG_REG_REG, FllCmdCfg.Data);

    FllCmdCfg.Bits.DCO_IREFTUNE = TuneVal;
    MrcWriteCR (MrcData, FLL_CMD_CFG_REG_REG, FllCmdCfg.Data);

    // Relock FLL
    FllCmdCfg.Bits.FLL_LDO_ENABLE_OVRD_EN = 1;
    FllCmdCfg.Bits.FLL_OUT_CLK_REQ_OVRD_EN = 1;
    FllCmdCfg.Bits.FLL_OUT_CLK_REQ = 1;
    MrcWriteCR (MrcData, FLL_CMD_CFG_REG_REG, FllCmdCfg.Data);

    FllStaticCfg0.Bits.CAL_THRESH_HI = 0x0;
    FllStaticCfg0.Bits.CAL_THRESH_LO = 0x0;
    MrcWriteCR (MrcData, FLL_STATIC_CFG_0_REG_REG, FllStaticCfg0.Data);

    // Enable FLL
    //FllCmdCfg.Bits.DCO_IREFTUNE = TuneVal;
    FllCmdCfg.Bits.FLL_ENABLE = 1;
    FllCmdCfg.Bits.FLL_LDO_ENABLE = 1;
    FllCmdCfg.Bits.FLL_OUT_CLK_REQ = 1;
    MrcWriteCR (MrcData, FLL_CMD_CFG_REG_REG, FllCmdCfg.Data);

    // Wait 2k Cycles, read the count and calculate FLL VCO Freq
    MrcWait (MrcData, ((2 * Outputs->Qclkps)));
    FllDiagStat.Data = MrcReadCR (MrcData, FLL_DIAG_STAT_REG_REG);

    // FLL VCO CLK (MHz) = fll_counter_sum / (fast_cal_window_val * refclk_us * 10)
    // Optimizing the * 10 out of TuneVal calculation so drop a 0 here to be in Kilo instead of Mega units.
    FllVcoFreq = (FllDiagStat.Bits.FLL_COUNTER_SUM * 100000) / (FastCalWinVal * RefClkPs);

    // Check if FLL VCO Freq > 4 GHz and break if we reach 4 GHz
    MRC_DEBUG_MSG (Debug, MSG_LEVEL_ALGO, "%u:\t\t%u\n", TuneVal, FllVcoFreq);
    if (FllVcoFreq > f4000) {
      break;
    }
  }

  // Disable FLL Training mode
  FllStaticCfg0.Bits.RUNTIME_CAL = 0;
  MrcWriteCR (MrcData, FLL_STATIC_CFG_0_REG_REG, FllStaticCfg0.Data);

  // Program optimal IrefTune
  FllCmdCfg.Bits.DCO_IREFTUNE = TuneVal;
  FllCmdCfg.Bits.FLL_RATIO = FllCmdCfgSave.Bits.FLL_RATIO;
  FllCmdCfg.Bits.FLL_LDO_ENABLE_OVRD_EN = 0;   // Clear to give control to wired interface
  FllCmdCfg.Bits.FLL_LDO_ENABLE = 0;           // Clear to give control to wired interface
  FllCmdCfg.Bits.FLL_OUT_CLK_REQ_OVRD_EN = 0;  // Clear to give control to wired interface
  MrcWriteCR (MrcData, FLL_CMD_CFG_REG_REG, FllCmdCfg.Data);

  FllWired.Bits.FLLCalEnSrc = 0;
  FllWired.Bits.FLLCalRatioSrc = 0;
  MrcWriteCR (MrcData, DDRPHY_COMP_CR_DDRCRFLLWIRED_REG, FllWired.Data);

  if (TuneVal > FLL_CMD_CFG_REG_DCO_IREFTUNE_MAX) {
    MRC_DEBUG_MSG (Debug, MSG_LEVEL_ERROR, "%s FLL Training could not reach 4GHz in the DcoIrefTune range\n", gErrString);
    if (MrcData->Inputs.ExitOnFailure) {
      Status = mrcFail;
    }
  } else {
    MRC_DEBUG_MSG (Debug, MSG_LEVEL_NOTE, "\n%sDcoIrefTune\nFLL:\t%u\n%sDcoIrefTune\n", gStartTagStr, TuneVal, gStopTagStr);
  }
  MRC_DEBUG_MSG (Debug, MSG_LEVEL_ALGO, "End FLL_CMD_CFG_REG_REG = 0x%x\n", MrcReadCR (MrcData, FLL_CMD_CFG_REG_REG));

  return Status;
}

/**
  This function returns the tCL delay separation needed from Receive Enable to the RxFifo Read Enable.
  This covers all internal logic delay in the path.

  @param[in]  MrcData - Pointer to global MRC data.
  @param[in]  Controller  - 0-based index instance to select.
  @param[in]  Channel     - 0-based index instance to select.
  @param[in]  DdrType     - Enumeration of MrcDdrType which DDR type is being enabled.
  @param[in]  GearRatio   - Integer number of the current Gear ratio.
**/
UINT8
MrcRcvEn2RxFifoReadTclDelay (
  IN MrcParameters *const MrcData,
  IN UINT32               Controller,
  IN UINT32               Channel,
  IN MrcDdrType           DdrType,
  IN UINT8                GearRatio
  )
{
  UINT8 RxDelay;
  UINT8 Gear2;

  Gear2 = ((GearRatio == 2) ? 1 : 0);

  RxDelay = RxFifoChDelay[DdrType][Gear2][(Controller * MrcData->Outputs.MaxChannels) + Channel];

  if (MrcData->Inputs.DtHalo) {
    if (Controller == 1) {
      // Need to increment by 1 for Rxfifo/RcvEn separation.
      RxDelay++;
    }
  }

  return RxDelay;
}

/**
  This function configures the DDR IO ODT type to Data and Comp blocks.
  VSS and VTT blocks are one time configured in MrcDdrIoInit.
  This updates OdtMode in MrcOutput.

  @param[in]  MrcData - Pointer to global MRC data.
  @param[in]  NewMode - ODT mode to enable.

  @retval MrcStatus - mrcSuccess if a valid ODT mode is requested, otherwise mrcWrongInputParameter.
**/
MrcStatus
MrcSetIoOdtMode (
  IN  MrcParameters *const  MrcData,
  IN  MRC_ODT_MODE_TYPE     NewMode
  )
{
  static const UINT8  OdtModeLut[MrcOdtModeMax] = {2,2,1,0}; //MrcOdtModeDefault - unused, MrcOdtModeVtt, MrcOdtModeVddq, MrcOdtModeVss,
  MrcDebug            *Debug;
  MrcOutput           *Outputs;
  MRC_ODT_MODE_TYPE   CurrentMode;
  INT64               VttMode;
  INT64               VddqMode;
  INT64               DataOdtMode;
  INT64               RxVrefValue;
  INT32               RxVrefMinValue;
  INT32               RxVrefMaxValue;
  INT64               GetRxVrefMin;
  INT64               GetRxVrefMax;
  UINT32              Idx;
  UINT32              Offset;
  BOOLEAN             Ddr4;
  BOOLEAN             Lpddr4;
  BOOLEAN             Lpddr4Vss;
  DDRVTT0_CR_DDRCRVTTGENCONTROL_STRUCT  VttGenCtl;

  Outputs     = &MrcData->Outputs;
  Debug       = &Outputs->Debug;
  CurrentMode = Outputs->OdtMode;
  Ddr4        = (Outputs->DdrType == MRC_DDR_TYPE_DDR4);
  Lpddr4      = (Outputs->DdrType == MRC_DDR_TYPE_LPDDR4);
  Lpddr4Vss   = (Lpddr4 && (NewMode == MrcOdtModeVss));

  if ((NewMode >= MrcOdtModeMax) || (NewMode == MrcOdtModeDefault)) {
    MRC_DEBUG_MSG (Debug, MSG_LEVEL_ERROR, "Invalid ODT Mode requested: %d\n", NewMode);
    return mrcWrongInputParameter;
  }
  MRC_DEBUG_MSG (Debug, MSG_LEVEL_NOTE, "New ODT Mode: %s\t Original ODT Mode: %s\n", gIoOdtModeStr[NewMode], gIoOdtModeStr[CurrentMode]);
  MrcGetSetLimits (MrcData, RxVref, &GetRxVrefMin, &GetRxVrefMax, NULL);
  RxVrefMinValue = (INT32) GetRxVrefMin;
  RxVrefMaxValue = (INT32) GetRxVrefMax;

  // default mode if no other mode is enabled.  Thus we clear all other modes.
  VttMode   = (NewMode == MrcOdtModeVtt)  ? 1 : 0;
  VddqMode  = (NewMode == MrcOdtModeVddq) ? 1 : 0;
  if (NewMode == MrcOdtModeVss) {
    RxVrefValue = (RxVrefMaxValue + (3 * RxVrefMinValue)) / 4;  // VSS termination: RxVref = 0.25 * Vddq
  } else {
    RxVrefValue = (RxVrefMinValue + RxVrefMaxValue) / 2;  // Middle of the range for VTT termination
  }
  DataOdtMode = OdtModeLut[NewMode];

  MrcGetSetChStrb (MrcData, MAX_CONTROLLER, MAX_CHANNEL, MAX_SDRAM_IN_DIMM, GsmIocDataOdtMode,  WriteToCache, &DataOdtMode);
  MrcGetSetChStrb (MrcData, MAX_CONTROLLER, MAX_CHANNEL, MAX_SDRAM_IN_DIMM, GsmIocRxVrefMFC,    WriteToCache, &VttMode);

  // Update RxVref to match the new termination type
  // DDR4 will be updated below using MrcSetDefaultRxVrefDdr4
  // LPDDR4 VSS termination will be updated below.
  if (!(Ddr4 || Lpddr4Vss)) {
    MrcGetSetChStrb (MrcData, MAX_CONTROLLER, MAX_CHANNEL, MAX_SDRAM_IN_DIMM, RxVref, WriteToCache | PrintValue, &RxVrefValue);
  }

  MrcGetSetNoScope (MrcData, GsmIocCompVddqOdtEn, WriteToCache, &VddqMode);
  MrcGetSetNoScope (MrcData, GsmIocCompVttOdtEn,  WriteToCache, &VttMode);

  MrcFlushRegisterCachedData (MrcData);

  for (Idx = 0; Idx < DDRIO_MAX_VTT_GEN; Idx++) {
    Offset = DDRVTT0_CR_DDRCRVTTGENCONTROL_REG +
      ((DDRVTT1_CR_DDRCRVTTGENCONTROL_REG - DDRVTT0_CR_DDRCRVTTGENCONTROL_REG) * Idx);
    VttGenCtl.Data = MrcReadCR (MrcData, Offset);
    VttGenCtl.Bits.EnVttOdt = (UINT32) VttMode;
    MrcWriteCR (MrcData, Offset, VttGenCtl.Data);
  }

  Outputs->OdtMode = NewMode;
  MrcSetDefaultRxVref (MrcData, TRUE, TRUE);

  return mrcSuccess;
}

/**
  This function decodes which Channel and Byte is connected to a specific DLL Data/CCC partition.
  Use of the Channel/Byte array in GetSet, NOT MrcGetSet.

  @param[in, out] MrcData - Include all MRC global data.
  @param[in] Partition    - DLL Data/CCC partition.
  @param[out] Channel     - Array of corresponding Channel.
  @param[out] Byte        - Array of corresponding Byte.
  @param[out] TargetRail  - Whether it is VccDLL0 (1) or VccDLL1 (2) or VccDLL2 (3)

  @retval Status - mrcSuccess if partition is expected, otherwise mrcFail
**/
MrcStatus
MrcDllPartitionDecode (
  IN MrcParameters *const MrcData,
  IN DLL_PARTITIONS       Partition,
  OUT UINT32              Channel[VCC_DLL_BYTES],
  OUT UINT32              Byte[VCC_DLL_BYTES],
  OUT UINT32              *TargetRail
  )
{
  MrcOutput            *Outputs;
  MrcInput             *Inputs;
  MrcDebug             *Debug;
  const PHY_PARTITION_BLOCK  *PhyDllPartition;
  MrcStatus            Status;
  MrcDdrType           DdrType;
  UINT32               Index;

  Outputs = &MrcData->Outputs;
  Inputs = &MrcData->Inputs;
  Debug = &Outputs->Debug;
  DdrType = Outputs->DdrType;
  Status = mrcSuccess;

  if (Partition >= DllDdrMax) {
    MRC_DEBUG_MSG (Debug, MSG_LEVEL_ERROR, "%s - DLL Data/CCC Partition is not valid! - %x\n", gErrString, Partition);
    Status = mrcFail;
  }
  if ((Channel == NULL) || (Byte == NULL) || (TargetRail == NULL)) {
    MRC_DEBUG_MSG (Debug, MSG_LEVEL_ERROR, "%s - Missing pointers for return data!\n", gErrString);
    Status = mrcFail;
  }

  if (Status != mrcFail) {
    if ((DdrType == MRC_DDR_TYPE_DDR4) && (Inputs->DqPinsInterleaved == 0)) {
      PhyDllPartition = Ddr4NILDllPartitions[Partition];
    } else {
      PhyDllPartition = LpddrDdr4ILDllPartitions[Partition];
    }
    for (Index = 0; Index < VCC_DLL_BYTES; Index++) {
      Channel[Index] = PhyDllPartition[Index].Channel;
      Byte[Index] = PhyDllPartition[Index].Byte;
    }
    if (Inputs->UlxUlt) {
      *TargetRail = ((Partition < DllDdrData4) || (Partition == DllDdrCcc0) || (Partition == DllDdrCcc1)) ? VCCDLL1 : VCCDLL2;
    } else { //DtHalo
      if (Partition >= DllDdrCcc0) {
        *TargetRail = VCCDLL2;
      } else {
        *TargetRail = ((Partition < DllDdrData4) || (Partition == DllDdrData8)) ? VCCDLL1 : VCCDLL3;
      }
    }
  }
  return Status;
}

/**
  This function freezes VccDll or VsxHi voltage rails

  @param[in]  MrcData - Pointer to global MRC data.
  @param[in]  TargetRail - VccDll Rail North or South.
  @param[in]  Freeze - Freeze or Unfreeze target rail.

**/
VOID
FreezeVccDllVsxHi (
  IN MrcParameters    *const MrcData,
  IN UINT32                  TargetRail,
  IN BOOLEAN                 Freeze
  )
{
  UINT32 Index;
  UINT32 SelCode;
  UINT32 OpenLoop;
  INT64  GetSetEn;
  INT64  GetSetOpenLoop;

  if (TargetRail != VSXHI) {
    Index = TargetRail - VCCDLL1; //Base 0
    SelCode = GsmIocVccDllControlSelCode_V;
    OpenLoop = GsmIocVccDllControlOpenLoop_V;
  } else {
    Index = 1;
    SelCode = GsmIocVsxHiControlSelCode_V;
    OpenLoop = GsmIocVsxHiControlOpenLoop_V;
  }
  GetSetEn = 1;
  GetSetOpenLoop = 0;

  if (Freeze) {
    MrcGetSetVccDll (MrcData, Index, SelCode, ReadCrWriteCached, &GetSetEn);
    GetSetOpenLoop = 1;
  }
  //2 Writes happening to remove chance of race condition between writing OpenLoop = 1 and OutputCode data is reflected back on read.
  MrcGetSetVccDll (MrcData, Index, OpenLoop, ReadCrWriteCached, &GetSetOpenLoop);
  MrcGetSetVccDll (MrcData, Index, OpenLoop, ReadCrForceWriteCached, &GetSetOpenLoop);
}

/**
  Get the max supported VCCDLL register index for the current configuration

  @param[in] MrcData - Include all MRC global data.

  @retval The max VCCDLL register index (VCCDLL2 for UlxUlt, VCCDLL3 for DtHalo)
**/
UINT32
GetMaxVccDllRail (
  IN MrcParameters *const MrcData
  )
{
  return ((MrcData->Inputs.DtHalo) ? VCCDLL3 : VCCDLL2);
}

/**
  This function runs DDRIO Offset Correction - PreTraining
  (VsxHi FB/FF Offset Correction & VccDLL Per Partition Nbias Vref Offset Correction)

  @param[in, out] MrcData - Include all MRC global data.

  @retval mrcSuccess
**/
MrcStatus
MrcOffsetCorrectionPre (
  IN OUT MrcParameters *const MrcData
  )
{
  const MRC_FUNCTION *MrcCall;
  MrcOutput          *Outputs;
  MrcInput           *Inputs;
  MrcDebug           *Debug;
  MrcStatus          Status;
  BOOLEAN            SweepNotFinished;
  BOOLEAN            VccDllBypass;
  BOOLEAN            VssHiBypass;
  BOOLEAN            Lpddr5;
  DLL_PARTITIONS     Partition;
  UINT8              McChBitMask;
  UINT8              Channel;
  UINT8              OrigTxPiOn[MAX_CONTROLLER][MAX_CHANNEL][MAX_SDRAM_IN_DIMM];
  UINT8              Counter;
  UINT8              SweepFailureCount;
  INT16              SweepValue;
  INT16              SweepStep;
  INT16              CurrentStep;
  INT16              PrevStep;
  UINT32             DebugPrint;
  UINT32             Wait4000Qclks;
  UINT32             Index;
  UINT32             MaxVccDllRail;
  UINT32             TargetRail;
  UINT32             TargetRailCheck;
  UINT32             OrigVccDllSampler[VCCDLL3];
  UINT32             Controller;
  UINT32             Byte;
  UINT32             MaxByte;
  UINT32             Offset;
  UINT32             ChannelIndex[VCC_DLL_BYTES];
  UINT32             ByteIndex[VCC_DLL_BYTES];
  INT32              LowSigned;
  INT32              HighSigned;
//  INT32              MiddleSigned;
  INT64              GetSetCompClkOn;
  INT64              GetSetVal;
  INT64              GetSetEn;
  INT64              GetSetDis;
  INT64              GetSetMin;
  INT64              GetSetMax;
  DLLDDRCOMP_CR_DDRCRVCCDLLFFCONTROL_STRUCT  VccDllFFControl;
  DDRVCCDLL0_CR_DDRCRVCCDLLCONTROL_STRUCT  VccDll0Control;
  DDRVCCDLL1_CR_DDRCRVCCDLLCONTROL_STRUCT  VccDll1Control;
  DDRVCCDLL0_CR_DDRCRVCCDLLSAMPLER_STRUCT  VccDllSampler;
  DDRPHY_COMP_CR_VCCDLLTARGET_STRUCT  VccDllTarget;
  DDRPHY_COMP_CR_VCCDLLTARGET_STRUCT  OrigVccDllTarget;
  DLLDDRDATA0_CR_DDRCRVCCDLLFFNBIAS_STRUCT  VccDllFFNBias;
  DLLDDRDATA0_CR_DDRCRVCCDLLFFNBIAS_STRUCT  VccDllFFNBiasComp;
  DDRPHY_COMP_CR_VCCDLLREPLICACTRL2_STRUCT  VccDllReplicaCtrl2;
  DDRPHY_COMP_CR_VSSHITARGET_STRUCT  VssHiTarget;
  DDRVSSHIAFEA_CR_DDRCRVSSHISAMPLER_STRUCT  VssHiSampler;
  DDRPHY_COMP_CR_DDRCRCOMPOVR1_STRUCT  CompOverride1;
  MC0_REQ0_CR_CPGC_SEQ_CTL_STRUCT  CpgcStartTest;
  MC0_REQ0_CR_CPGC_SEQ_CTL_STRUCT  CpgcStopTest;
  MC0_CH0_CR_CADB_CFG_STRUCT  CadbConfig;
  MC0_CH0_CR_CADB_CTL_STRUCT  CadbStartTest;
  MC0_CH0_CR_CADB_CTL_STRUCT  CadbStopTest;

  Outputs       = &MrcData->Outputs;
  Inputs        = &MrcData->Inputs;
  MrcCall       = Inputs->Call.Func;
  Debug         = &Outputs->Debug;
  Wait4000Qclks = (Outputs->Qclkps * 4) * MRC_TIMER_1NS;
  Status        = mrcSuccess;
  DebugPrint    = GSM_PRINT_VAL;
  GetSetEn      = 1;
  GetSetDis     = 0;
  Lpddr5        = (Outputs->DdrType == MRC_DDR_TYPE_LPDDR5);
  MaxByte       = Outputs->SdramCount;
  MaxVccDllRail = GetMaxVccDllRail (MrcData);
  MrcCall->MrcSetMem ((UINT8*) OrigVccDllSampler, sizeof (OrigVccDllSampler), 0);

  CpgcStartTest.Data = CpgcStopTest.Data = CadbStartTest.Data = CadbStopTest.Data = 0;
  CpgcStartTest.Bits.START_TEST = 1;
  CpgcStopTest.Bits.STOP_TEST = 1;
  CadbStartTest.Bits.START_TEST = 1;
  CadbStopTest.Bits.STOP_TEST = 1;

  VccDllFFControl.Data = MrcReadCR (MrcData, DLLDDRCOMP_CR_DDRCRVCCDLLFFCONTROL_REG);
  VccDll0Control.Data = MrcReadCR (MrcData, DDRVCCDLL0_CR_DDRCRVCCDLLCONTROL_REG);
  VccDll1Control.Data = MrcReadCR (MrcData, DDRVCCDLL1_CR_DDRCRVCCDLLCONTROL_REG);
  VssHiSampler.Data = MrcReadCR (MrcData, DDRVSSHIAFEA_CR_DDRCRVSSHISAMPLER_REG);
  if (VccDllFFControl.Bits.Bypass || VccDll0Control.Bits.Bypass || VccDll1Control.Bits.Bypass) {
    VccDllBypass = TRUE;
    MRC_DEBUG_MSG (Debug, MSG_LEVEL_NOTE, "VccDll Bypass!\n");
  } else {
    VccDllBypass = FALSE;
  }
  VssHiBypass = (VssHiSampler.Bits.DdrVssHiGndSelQnnnH) ? TRUE : FALSE;

  McChBitMask = 0;
  for (Controller = 0; Controller < MAX_CONTROLLER; Controller++) {
    for (Channel = 0; Channel < Outputs->MaxChannels; Channel++) {
      //@todo H/S needs function adjustment for different populated rank across channels
      McChBitMask |= SelectReutRanks (MrcData, (UINT8) Controller, Channel, 1, FALSE, 0);
    }
  }

  if (Lpddr5) {
    VccDllReplicaCtrl2.Data = MrcReadCR (MrcData, DDRPHY_COMP_CR_VCCDLLREPLICACTRL2_REG);
    VccDllReplicaCtrl2.Bits.DisableTxDqs = 0;
    MrcWriteCR (MrcData, DDRPHY_COMP_CR_VCCDLLREPLICACTRL2_REG, VccDllReplicaCtrl2.Data);

    GetSetVal = 0;
    MrcGetSetChStrb (MrcData, MAX_CONTROLLER, MAX_CHANNEL, MAX_SDRAM_IN_DIMM, GsmIocDataDisableTxDqs, WriteCached | PrintValue, &GetSetVal);
  }

  if (VccDllBypass == FALSE) {
    //VccDLL Offset Correction
    VccDllFFNBias.Data = MrcReadCR (MrcData, DLLDDRCOMP_CR_DDRCRVCCDLLFFNBIAS_REG);
    VccDllFFNBias.Bits.NbiasFastStartup = 0;
    MrcWriteCR (MrcData, DLLDDRCOMP_CR_DDRCRVCCDLLFFNBIAS_REG, VccDllFFNBias.Data);
    for (Index = 0; Index < MaxVccDllRail; Index++) {
      GetSetVal = 0;
      MrcGetSetVccDll (MrcData, Index, GsmIocVccDllControlPanicEn, WriteToCache | PrintValue, &GetSetVal);
      MrcGetSetVccDll (MrcData, Index, GsmIocVccDllControlSampleDivider, WriteToCache | PrintValue, &GetSetVal);
      GetSetVal = 3;
      MrcGetSetVccDll (MrcData, Index, GsmIocVccDllControlHiBWDivider, WriteToCache | PrintValue, &GetSetVal);
      MrcGetSetVccDll (MrcData, Index, GsmIocVccDllControlLoBWDivider, WriteToCache | PrintValue, &GetSetVal);

      Offset = OFFSET_CALC_CH (DDRVCCDLL0_CR_DDRCRVCCDLLSAMPLER_REG, DDRVCCDLL1_CR_DDRCRVCCDLLSAMPLER_REG, Index);
      if (Offset == DDRVCCDLL2_CR_DDRCRVCCDLLSAMPLER_P0_REG) {
        // This register aliases with a UlxUlt offset
        // Set the NO_ADJ to prevent offset translation in MrcCrOffsetProjAdj
        Offset |= NO_ADJ;
      }

      VccDllSampler.Data = OrigVccDllSampler[Index] = MrcReadCR (MrcData, Offset);
      MRC_DEBUG_MSG (Debug, MSG_LEVEL_NOTE, "OrigVccDllSampler[%d]: 0x%x\n", Index, OrigVccDllSampler[Index]);
      VccDllSampler.Bits.GainBoost = 2;
      if (OrigVccDllSampler[Index] != VccDllSampler.Data) {
        MrcWriteCR (MrcData, Offset, VccDllSampler.Data);
      }
      MRC_DEBUG_MSG (Debug, MSG_LEVEL_NOTE, "VccDllSampler[%d].Panic: %d\n", Index, VccDllSampler.Bits.Panic);
    }
    MrcFlushRegisterCachedData (MrcData);

    VccDllTarget.Data = OrigVccDllTarget.Data = MrcReadCR (MrcData, DDRPHY_COMP_CR_VCCDLLTARGET_REG);
    MRC_DEBUG_MSG (Debug, MSG_LEVEL_NOTE, "VccDllTarget.Bits.CompVTarget: %d\n", VccDllTarget.Bits.CompVTarget);

    GetSetVal = 0;
    for (Index = 0; Index < MAX_SYS_CHANNEL; Index++) {
      //CCC (only 8 Phy Channel)
      GetSet (MrcData, MRC_IGNORE_ARG, MRC_IGNORE_ARG, Index, MRC_IGNORE_ARG, MRC_IGNORE_ARG, MRC_IGNORE_ARG, MRC_IGNORE_ARG, MRC_IGNORE_ARG, DdrLevel, GsmIocCccPiEn, WriteToCache | PrintValue, &GetSetVal);
      GetSet (MrcData, MRC_IGNORE_ARG, MRC_IGNORE_ARG, Index, MRC_IGNORE_ARG, MRC_IGNORE_ARG, MRC_IGNORE_ARG, MRC_IGNORE_ARG, MRC_IGNORE_ARG, DdrLevel, GsmIocCccPiEnOverride, WriteToCache | PrintValue, &GetSetVal);
    }
    MrcFlushRegisterCachedData (MrcData);

    //Save Values
    for (Controller = 0; Controller < MAX_CONTROLLER; Controller++) {
      for (Channel = 0; Channel < Outputs->MaxChannels; Channel++) {
        for (Byte = 0; Byte < MAX_SDRAM_IN_DIMM; Byte++) {
          if (MrcByteExist (MrcData, Controller, Channel, Byte)) {
            MrcGetSetChStrb (MrcData, Controller, Channel, Byte, GsmIocTxPiPwrDnDis, ReadFromCache | PrintValue, &GetSetVal);
            OrigTxPiOn[Controller][Channel][Byte] = (UINT8) GetSetVal;
          }
        }
      }
    }

    //Set Values
    GetSetVal = 1;
    MrcGetSetChStrb (MrcData, MAX_CONTROLLER, MAX_CHANNEL, MAX_SDRAM_IN_DIMM, GsmIocRxDisable, WriteToCache | PrintValue, &GetSetVal);

    GetSetVal = 0;
    MrcGetSetChStrb (MrcData, MAX_CONTROLLER, MAX_CHANNEL, MAX_SDRAM_IN_DIMM, GsmIocTxPiPwrDnDis, WriteToCache | PrintValue, &GetSetVal);
    MrcFlushRegisterCachedData (MrcData);

    MrcGetSetNoScope (MrcData, GsmIocCompClkOn, ReadFromCache, &GetSetCompClkOn);
    GetSetVal = 1;
    MrcGetSetNoScope (MrcData, GsmIocCompClkOn, WriteCached | PrintValue, &GetSetVal);

    GetSetVal = 0;
    MrcGetSetChStrb (MrcData, MAX_CONTROLLER, MAX_CHANNEL, MAX_SDRAM_IN_DIMM, GsmIocInternalClocksOn, WriteCached | DebugPrint, &GetSetVal);
    MrcGetSetNoScope (MrcData, GsmIocEnableSpineGate, WriteCached | DebugPrint, &GetSetVal);

    //Step 2: Comp Nbias Offset Correction
    //Configure VccDLL for feedback loop, done as part of Step 1
/*
    MRC_DEBUG_MSG (Debug, MSG_LEVEL_NOTE, "VccDll_Step2\n");

    VccDllTarget.Bits.ForceCompAmpOn = 1;
    VccDllTarget.Bits.SelectVccDllRail = 3; //VccDLL_Replica
    MrcWriteCR (MrcData, DDRPHY_COMP_CR_VCCDLLTARGET_REG, VccDllTarget.Data);
    MrcWait (MrcData, Wait4000Qclks * 10); //Wait for Nbias to turn on

    //Initialize variables for this Target Rail
    SweepNotFinished = TRUE;
    LowSigned = DLLDDRDATA0_CR_DDRCRVCCDLLFFNBIAS_NbTarget_MIN;
    HighSigned = DLLDDRDATA0_CR_DDRCRVCCDLLFFNBIAS_NbTarget_MAX;
    VccDllFFNBias.Data = MrcReadCR (MrcData, DLLDDRCOMP_CR_DDRCRVCCDLLFFNBIAS_REG);
    MiddleSigned = VccDllFFNBias.Bits.NbTarget;

    //Binary Sweep Target Rail
    while (SweepNotFinished) {
      VccDllFFNBias.Bits.NbTarget = (UINT32) MiddleSigned;
      MrcWriteCR (MrcData, DLLDDRCOMP_CR_DDRCRVCCDLLFFNBIAS_REG, VccDllFFNBias.Data);

      VccDllTarget.Bits.RstCount = 1;
      MrcWriteCR (MrcData, DDRPHY_COMP_CR_VCCDLLTARGET_REG, VccDllTarget.Data);
      MrcWait (MrcData, Wait4000Qclks * 10);

      VccDllTarget.Data = MrcReadCR (MrcData, DDRPHY_COMP_CR_VCCDLLTARGET_REG);
      MRC_DEBUG_MSG (Debug, MSG_LEVEL_NOTE, "Testing %d, CountMSB = 0x%x\n", MiddleSigned, VccDllTarget.Bits.CountMSB);
*/
/*
      if (VccDllTarget.Bits.CountMSB >= 0x8) {
        HighSigned = MiddleSigned - 1;
      } else {
        LowSigned = MiddleSigned + 1;
      }
      if (HighSigned < LowSigned) {
        SweepNotFinished = FALSE;
      }
      MiddleSigned = (HighSigned + LowSigned) / 2;
#ifdef CTE_FLAG
      SweepNotFinished = FALSE;
#endif
    }

#ifdef MRC_DEBUG_PRINT
    MRC_DEBUG_MSG (Debug, MSG_LEVEL_WARNING, "\n%sPreTrain_Offset_Correction_VccDll_Step2\n", gStartTagStr);
    MRC_DEBUG_MSG (Debug, MSG_LEVEL_WARNING, "\t%d (0x%x)\n", VccDllFFNBias.Bits.NbTarget, VccDllFFNBias.Bits.NbTarget);
    MRC_DEBUG_MSG (Debug, MSG_LEVEL_WARNING, "%sPreTrain_Offset_Correction_VccDll_Step2\n", gStopTagStr);
#endif
*/
    //Step 3: Compensation
    MRC_DEBUG_MSG (Debug, MSG_LEVEL_NOTE, "VccDll_Step3\n");
    VccDllTarget.Bits.SelectVccDllRail = 0;
    VccDllTarget.Bits.ForceCompAmpOn = 0;
    VccDllTarget.Bits.OffsetBinSearch = 0;
    VccDllTarget.Bits.CodeLeakSearch = 1;
    MrcWriteCR (MrcData, DDRPHY_COMP_CR_VCCDLLTARGET_REG, VccDllTarget.Data);

    ForceRcomp (MrcData);

    //Step 1: Feedback Offset Correction
    //Configure VccDLL for feedback loop
    GetSetVal = 0;
    for (Index = 0; Index < VCC_DLL_BYTES; Index++) {
      MrcGetSetVccDll (MrcData, Index, GsmIocVccDllControlSelCode_V, ReadCrWriteToCache | DebugPrint, &GetSetVal);
    }
    MrcFlushRegisterCachedData (MrcData);

    //Loop TargetRail
    VccDllTarget.Data = OrigVccDllTarget.Data;
    MRC_DEBUG_MSG (Debug, MSG_LEVEL_NOTE, " CompVTarget %d\n", VccDllTarget.Bits.CompVTarget);

    VccDllTarget.Bits.ForceCompAmpOn = 1;
    for (TargetRail = VCCDLL1; TargetRail <= MaxVccDllRail; TargetRail++) {
      MRC_DEBUG_MSG (Debug, MSG_LEVEL_NOTE, "VccDll_Step1\n");
      // Select which VccDLL rail is used when ForceCompAmpOn is selected.
      // {0: VccDLLComp0, 1: VccDLL1, 2: VccDLL2, 3: VccDLL_Replica, 4: VccDLL3 (SH Only), 5-7: Rsvd}
      VccDllTarget.Bits.SelectVccDllRail = (TargetRail == VCCDLL3) ? 4: TargetRail;
      VccDllTarget.Bits.RstCount = 0;
      MrcWriteCR (MrcData, DDRPHY_COMP_CR_VCCDLLTARGET_REG, VccDllTarget.Data);

      Index = TargetRail - VCCDLL1; //Base 0
      MRC_DEBUG_MSG (Debug, MSG_LEVEL_NOTE, "\tSweeping TargetRail 0x%x\n", TargetRail);

      //Initialize variables for this Target Rail
      SweepNotFinished = TRUE;
      MrcGetSetLimits (MrcData, GsmIocVccDllControlTarget_V, &GetSetMin, &GetSetMax, NULL);
      LowSigned = (INT32) GetSetMin;
      HighSigned = (INT32) GetSetMax;
      //Reading current register value to start the sweep
      MrcGetSetVccDll(MrcData, Index, GsmIocVccDllControlTarget_V, ReadUncached | DebugPrint, &GetSetVal);

      //Linear Sweep Target Rail
      while (SweepNotFinished) {
        MrcGetSetVccDll (MrcData, Index, GsmIocVccDllControlTarget_V, ForceWriteUncached | DebugPrint, &GetSetVal);
        MrcWait (MrcData, Wait4000Qclks * 10);

        VccDllTarget.Bits.RstCount = 1;
        MrcWriteCR (MrcData, DDRPHY_COMP_CR_VCCDLLTARGET_REG, VccDllTarget.Data);
        MrcWait (MrcData, Wait4000Qclks * 10);

        VccDllTarget.Data = MrcReadCR (MrcData, DDRPHY_COMP_CR_VCCDLLTARGET_REG);
        MRC_DEBUG_MSG (Debug, MSG_LEVEL_NOTE, "\tVccDllTarget.Bits.CountMSB = 0x%x\n", VccDllTarget.Bits.CountMSB);
        if (VccDllTarget.Bits.CountMSB >= 0x8) {
          GetSetVal = HighSigned = (INT32) GetSetVal - 1;
        } else {
          GetSetVal = LowSigned = (INT32) GetSetVal + 1;
        }
        if (HighSigned < LowSigned) {
          SweepNotFinished = FALSE;
        }
      }

      //Step 4: Per partition Nbias Vref Offset Correction
      if ((TargetRail == MaxVccDllRail) && Inputs->DtHalo) {
        continue;
      }
      MRC_DEBUG_MSG (Debug, MSG_LEVEL_NOTE, "VccDll_Step4\n");

      VccDllTarget.Data = MrcReadCR (MrcData, DDRPHY_COMP_CR_VCCDLLTARGET_REG);
      MRC_DEBUG_MSG (Debug, MSG_LEVEL_NOTE, " CompVTarget %d\n", VccDllTarget.Bits.CompVTarget);
      VccDllTarget.Bits.ForceCompAmpOn = 1;

      for (Index = 0; Index < MAX_SYS_CHANNEL; Index++) {
        //CCC (only 8 Phy Channel)
        GetSet (MrcData, MRC_IGNORE_ARG, MRC_IGNORE_ARG, Index, MRC_IGNORE_ARG, MRC_IGNORE_ARG, MRC_IGNORE_ARG, MRC_IGNORE_ARG, MRC_IGNORE_ARG, DdrLevel, GsmIocCccPiEn, WriteToCache | DebugPrint, &GetSetDis);
        GetSet (MrcData, MRC_IGNORE_ARG, MRC_IGNORE_ARG, Index, MRC_IGNORE_ARG, MRC_IGNORE_ARG, MRC_IGNORE_ARG, MRC_IGNORE_ARG, MRC_IGNORE_ARG, DdrLevel, GsmIocCccPiEnOverride, WriteToCache | DebugPrint, &GetSetEn);

        for (Controller = 0; Controller < MAX_CONTROLLER; Controller++) {
          //Data (Using Index for Byte/Strobe)
          GetSet (MrcData, MRC_IGNORE_ARG, MRC_IGNORE_ARG, Controller, MRC_IGNORE_ARG, MRC_IGNORE_ARG, Index, MRC_IGNORE_ARG, MRC_IGNORE_ARG, DdrLevel, GsmIocTxPiPwrDnDis, WriteToCache | DebugPrint, &GetSetDis);
        }
      }
      MrcFlushRegisterCachedData (MrcData);

      FreezeVccDllVsxHi (MrcData, TargetRail, TRUE);

      MrcWait (MrcData, Wait4000Qclks * 5);

      VccDllTarget.Bits.SelectVccDllRail = TargetRail;
      VccDllTarget.Bits.RstCount = 0;
      MrcWriteCR (MrcData, DDRPHY_COMP_CR_VCCDLLTARGET_REG, VccDllTarget.Data);

      //Sweep all partitions to find correct partitions associated with this VccDLL and set them up for testing
      for (Partition = DllDdrData0; Partition < DllDdrMax; Partition++) {
        if ((Partition == DllDdrData8) && (Inputs->UlxUlt)) { //Skip ECC byte for ULXULT
          continue;
        }
        Status |= MrcDllPartitionDecode (MrcData, Partition, ChannelIndex, ByteIndex, &TargetRailCheck);
        if (TargetRail != TargetRailCheck) {
          //skip to next partition as this is on different VccDll Rail
          continue;
        }
        MRC_DEBUG_MSG (Debug, MSG_LEVEL_NOTE, "\nPartition %d Setup\n", Partition);

        //Setup for this partition
        for (Index = 0; Index < VCC_DLL_BYTES; Index++) {
          if (Partition < DllDdrCcc0) {
            //Data Partitions
            GetSetVal = 1;
            GetSet (MrcData, MRC_IGNORE_ARG, MRC_IGNORE_ARG, ChannelIndex[Index], MRC_IGNORE_ARG, MRC_IGNORE_ARG, ByteIndex[Index], MRC_IGNORE_ARG, MRC_IGNORE_ARG, DdrLevel, GsmIocTxPiPwrDnDis, WriteToCache | DebugPrint, &GetSetVal);
          } else {
            //CCC Partitions
            GetSetVal = 0x1F;
            GetSet (MrcData, MRC_IGNORE_ARG, MRC_IGNORE_ARG, ChannelIndex[Index], MRC_IGNORE_ARG, MRC_IGNORE_ARG, MRC_IGNORE_ARG, MRC_IGNORE_ARG, MRC_IGNORE_ARG, DdrLevel, GsmIocCccPiEn, WriteToCache | DebugPrint, &GetSetVal);
            GetSetVal = 1;
            GetSet (MrcData, MRC_IGNORE_ARG, MRC_IGNORE_ARG, ChannelIndex[Index], MRC_IGNORE_ARG, MRC_IGNORE_ARG, MRC_IGNORE_ARG, MRC_IGNORE_ARG, MRC_IGNORE_ARG, DdrLevel, GsmIocCccPiEnOverride, WriteToCache | DebugPrint, &GetSetVal);
          }
        }
        MrcFlushRegisterCachedData (MrcData);
      } // for Partition Setup

      MRC_DEBUG_MSG (Debug, MSG_LEVEL_NOTE, "\nStep4 sweeping\n");
      //Setup for sweep all Partitions associated with this VccDLL
      SweepFailureCount = 0;
      Counter = 0;
      SweepStep = PrevStep = 0;
      SweepNotFinished = TRUE;
      SweepValue = 0;
      for (Partition = DllDdrData0; Partition < DllDdrMax; Partition++) {
        if ((Partition == DllDdrData8) && (Inputs->UlxUlt)) { //Skip ECC byte for ULXULT
          continue;
        }
        Status |= MrcDllPartitionDecode (MrcData, Partition, ChannelIndex, ByteIndex, &TargetRailCheck);
        if (TargetRail != TargetRailCheck) {
          //skip to next partition as this is on different VccDll Rail
          continue;
        }
        Offset = OFFSET_CALC_CH (DLLDDRDATA0_CR_DDRCRVCCDLLFFNBIAS_REG, DLLDDRDATA1_CR_DDRCRVCCDLLFFNBIAS_REG, Partition); //Offsets works across Data and CCC partitions
        VccDllFFNBiasComp.Data = MrcReadCR (MrcData, Offset);
        SweepValue = (INT16) VccDllFFNBiasComp.Bits.NbTarget;
      }

      //Sweep the setting
      do {
        if (Counter) {
          SweepValue += SweepStep;
          for (Partition = DllDdrData0; Partition < DllDdrMax; Partition++) {
            if ((Partition == DllDdrData8) && (Inputs->UlxUlt)) { //Skip ECC byte for ULXULT
              continue;
            }
            Status |= MrcDllPartitionDecode (MrcData, Partition, ChannelIndex, ByteIndex, &TargetRailCheck);
            if (TargetRail != TargetRailCheck) {
              //skip to next partition as this is on different VccDll Rail
              continue;
            }
            Offset = OFFSET_CALC_CH (DLLDDRDATA0_CR_DDRCRVCCDLLFFNBIAS_REG, DLLDDRDATA1_CR_DDRCRVCCDLLFFNBIAS_REG, Partition); //Offsets works across Data and CCC partitions
            VccDllFFNBiasComp.Data = MrcReadCR (MrcData, Offset);
            VccDllFFNBiasComp.Bits.NbTarget = (UINT32) SweepValue;
            MrcWriteCR (MrcData, Offset, VccDllFFNBiasComp.Data);
          }
        }
        MrcWait (MrcData, Wait4000Qclks * 5);

        VccDllTarget.Bits.RstCount = 1;
        MrcWriteCR (MrcData, DDRPHY_COMP_CR_VCCDLLTARGET_REG, VccDllTarget.Data);
        MrcWait (MrcData, Wait4000Qclks * 5);

        VccDllTarget.Data = MrcReadCR (MrcData, DDRPHY_COMP_CR_VCCDLLTARGET_REG);
        MRC_DEBUG_MSG (Debug, MSG_LEVEL_NOTE, "Testing %d, CountMSB = 0x%x\n", SweepValue, VccDllTarget.Bits.CountMSB);
        CurrentStep = (VccDllTarget.Bits.CountMSB >= 0x8) ? -1 : 1;
        if (Counter == 0) {
          SweepStep = CurrentStep;
          if (VccDllTarget.Bits.CountMSB == 0) {
            SweepNotFinished = FALSE;
          }
        }

        if (((CurrentStep != SweepStep) && (SweepFailureCount == 0)) || ((PrevStep != CurrentStep) && SweepFailureCount)) {
          //MSB of CountMSB toggled
          SweepNotFinished = FALSE;
        } else if ((SweepValue <= 0) || (SweepValue >= 127)) {
          //SweepValue reached edge of valid range
          if (SweepFailureCount) {
            SweepNotFinished = FALSE;
          }
          SweepFailureCount++;
          //Restart sweep in opposite direction
          SweepStep = (SweepStep == 1) ? -1 : 1;
        } else {
          Counter++;
        }
        PrevStep = CurrentStep;
      } while (SweepNotFinished);
      MRC_DEBUG_MSG (Debug, MSG_LEVEL_NOTE, "Sweep of Step4 took 0x%x iterations and Sweep direction changes %d\n", Counter, SweepFailureCount);
      Status |= (SweepFailureCount >= 2) ? mrcFail : mrcSuccess;

      //Sweep all partitions to find correct partitions associated with this VccDLL and cleanup
      for (Partition = DllDdrData0; Partition < DllDdrMax; Partition++) {
        if ((Partition == DllDdrData8) && (Inputs->UlxUlt)) { //Skip ECC byte for ULXULT
          continue;
        }
        Status |= MrcDllPartitionDecode (MrcData, Partition, ChannelIndex, ByteIndex, &TargetRailCheck);
        if (TargetRail != TargetRailCheck) {
          //skip to next partition as this is on different VccDll Rail
          continue;
        }
        MRC_DEBUG_MSG (Debug, MSG_LEVEL_NOTE, "\nPartition %d Cleanup\n", Partition);
        //Cleanup for this partition
        for (Index = 0; Index < VCC_DLL_BYTES; Index++) {
          if (Partition < DllDdrCcc0) {
            //Data
            GetSetVal = 0;
            GetSet (MrcData, MRC_IGNORE_ARG, MRC_IGNORE_ARG, ChannelIndex[Index], MRC_IGNORE_ARG, MRC_IGNORE_ARG, ByteIndex[Index], MRC_IGNORE_ARG, MRC_IGNORE_ARG, DdrLevel, GsmIocTxPiPwrDnDis, WriteToCache | DebugPrint, &GetSetVal);
          } else {
            //CCC
            GetSetVal = 0;
            GetSet (MrcData, MRC_IGNORE_ARG, MRC_IGNORE_ARG, ChannelIndex[Index], MRC_IGNORE_ARG, MRC_IGNORE_ARG, MRC_IGNORE_ARG, MRC_IGNORE_ARG, MRC_IGNORE_ARG, DdrLevel, GsmIocCccPiEn, WriteToCache | DebugPrint, &GetSetVal);
            GetSet (MrcData, MRC_IGNORE_ARG, MRC_IGNORE_ARG, ChannelIndex[Index], MRC_IGNORE_ARG, MRC_IGNORE_ARG, MRC_IGNORE_ARG, MRC_IGNORE_ARG, MRC_IGNORE_ARG, DdrLevel, GsmIocCccPiEnOverride, WriteToCache | DebugPrint, &GetSetVal);
          }
        }
        MrcFlushRegisterCachedData (MrcData);
      } // for Partition Cleanup
    } // for VccDll rail

#ifdef MRC_DEBUG_PRINT
    MRC_DEBUG_MSG (Debug, MSG_LEVEL_WARNING, "\n%sPreTrain_Offset_Correction_VccDll_Step1\n", gStartTagStr);
    for (TargetRail = VCCDLL1; TargetRail <= MaxVccDllRail; TargetRail++) {
      Index = TargetRail - VCCDLL1; //Base 0
      MrcGetSetVccDll (MrcData, Index, GsmIocVccDllControlTarget_V, ReadFromCache, &GetSetVal);
      MRC_DEBUG_MSG (Debug, MSG_LEVEL_WARNING, "\tTargetRail 0x%x = %d (0x%x)\n", TargetRail, (UINT32) GetSetVal, (UINT32) GetSetVal);
    }
    MRC_DEBUG_MSG (Debug, MSG_LEVEL_WARNING, "%sPreTrain_Offset_Correction_VccDll_Step1\n", gStopTagStr);

    MRC_DEBUG_MSG (Debug, MSG_LEVEL_WARNING, "\n%sPreTrain_Offset_Correction_VccDll_Step4\n", gStartTagStr);
    for (Partition = DllDdrData0; Partition < DllDdrMax; Partition++) {
      if ((Partition == DllDdrData8) && (Inputs->UlxUlt)) { //Skip ECC byte for ULXULT
        continue;
      }
      Offset = OFFSET_CALC_CH (DLLDDRDATA0_CR_DDRCRVCCDLLFFNBIAS_REG, DLLDDRDATA1_CR_DDRCRVCCDLLFFNBIAS_REG, Partition); //Offsets works across Data and CCC partitions
      VccDllFFNBias.Data = MrcReadCR (MrcData, Offset);
      MRC_DEBUG_MSG (Debug, MSG_LEVEL_WARNING, "\tPartition 0x%x = %d (0x%x)\n", Partition, VccDllFFNBias.Bits.NbTarget, VccDllFFNBias.Bits.NbTarget);
    }
    MRC_DEBUG_MSG (Debug, MSG_LEVEL_WARNING, "%sPreTrain_Offset_Correction_VccDll_Step4\n", gStopTagStr);
#endif

    //Cleanup of Step 4
    VccDllFFNBias.Bits.NbiasFastStartup = 0;
    MrcWriteCR (MrcData, DLLDDRCOMP_CR_DDRCRVCCDLLFFNBIAS_REG, VccDllFFNBias.Data);
    //Clean up by clearing VccDLLControl.OpenLoop, VccDLLComp.ForceCompAmpOn and any other clocking overrides used in the individual partitions.Issue TrainReset
    VccDllTarget.Data = MrcReadCR (MrcData, DDRPHY_COMP_CR_VCCDLLTARGET_REG);
    VccDllTarget.Bits.SelectVccDllRail = OrigVccDllTarget.Bits.SelectVccDllRail;
    VccDllTarget.Bits.ForceCompAmpOn = OrigVccDllTarget.Bits.ForceCompAmpOn;
    VccDllTarget.Bits.RstCount = OrigVccDllTarget.Bits.RstCount;
    MrcWriteCR (MrcData, DDRPHY_COMP_CR_VCCDLLTARGET_REG, VccDllTarget.Data);

    for (TargetRail = VCCDLL1; TargetRail <= MaxVccDllRail; TargetRail++) {
      FreezeVccDllVsxHi (MrcData, TargetRail, FALSE);
    }
    for (Index = 0; Index < MaxVccDllRail; Index++) {
      GetSetVal = VccDll0Control.Bits.PanicEn;
      MrcGetSetVccDll (MrcData, Index, GsmIocVccDllControlPanicEn, WriteToCache, &GetSetVal);
      GetSetVal = VccDll0Control.Bits.SampleDivider;
      MrcGetSetVccDll (MrcData, Index, GsmIocVccDllControlSampleDivider, WriteToCache, &GetSetVal);
      GetSetVal = VccDll0Control.Bits.HiBWDivider;
      MrcGetSetVccDll (MrcData, Index, GsmIocVccDllControlHiBWDivider, WriteToCache, &GetSetVal);
      GetSetVal = VccDll0Control.Bits.LoBWDivider;
      MrcGetSetVccDll (MrcData, Index, GsmIocVccDllControlLoBWDivider, WriteToCache, &GetSetVal);

      Offset = OFFSET_CALC_CH (DDRVCCDLL0_CR_DDRCRVCCDLLSAMPLER_REG, DDRVCCDLL1_CR_DDRCRVCCDLLSAMPLER_REG, Index);
      MrcWriteCR (MrcData, Offset, OrigVccDllSampler[Index]);
    }
    MrcFlushRegisterCachedData (MrcData);

    //VccDll is good to rely on from here on
    MrcGetSetNoScope (MrcData, GsmIocCompClkOn, WriteCached | PrintValue, &GetSetCompClkOn);

    //Restore Values
    for (Controller = 0; Controller < MAX_CONTROLLER; Controller++) {
      for (Channel = 0; Channel < Outputs->MaxChannels; Channel++) {
        for (Byte = 0; Byte < MAX_SDRAM_IN_DIMM; Byte++) {
          if (MrcByteExist (MrcData, Controller, Channel, Byte)) {
            GetSetVal = OrigTxPiOn[Controller][Channel][Byte];
            MrcGetSetChStrb (MrcData, Controller, Channel, Byte, GsmIocTxPiPwrDnDis, WriteToCache | PrintValue, &GetSetVal);
          }
        }
      }
    }
    GetSetVal = 0;
    MrcGetSetChStrb (MrcData, MAX_CONTROLLER, MAX_CHANNEL, MAX_SDRAM_IN_DIMM, GsmIocRxDisable, WriteToCache | PrintValue, &GetSetVal);

    for (Index = 0; Index < MAX_SYS_CHANNEL; Index++) {
      //CCC (only 8 Phy Channel)
      GetSetVal = 0;
      GetSet (MrcData, MRC_IGNORE_ARG, MRC_IGNORE_ARG, Index, MRC_IGNORE_ARG, MRC_IGNORE_ARG, MRC_IGNORE_ARG, MRC_IGNORE_ARG, MRC_IGNORE_ARG, DdrLevel, GsmIocCccPiEn, WriteToCache | PrintValue, &GetSetVal);
      GetSet (MrcData, MRC_IGNORE_ARG, MRC_IGNORE_ARG, Index, MRC_IGNORE_ARG, MRC_IGNORE_ARG, MRC_IGNORE_ARG, MRC_IGNORE_ARG, MRC_IGNORE_ARG, DdrLevel, GsmIocCccPiEnOverride, WriteToCache | PrintValue, &GetSetVal);
    }
    MrcFlushRegisterCachedData (MrcData);
    MrcBlockTrainResetToggle (MrcData, FALSE);
    IoReset (MrcData);
    MrcBlockTrainResetToggle (MrcData, TRUE);

  } //VccDllBypass == FALSE

  if (VssHiBypass == FALSE) {

    //VsxHi FB Offset (@todo LPMode = 3)
/*
    GetSetVal = 0;
    MrcGetSetChStrb (MrcData, MAX_CONTROLLER, MAX_CHANNEL, MAX_SDRAM_IN_DIMM, GsmIocInternalClocksOn, WriteCached | DebugPrint, &GetSetVal);
    VssHiTarget.Data = MrcReadCR (MrcData, DDRPHY_COMP_CR_VSSHITARGET_REG);
    VssHiTarget.Bits.ForceCompAmpOn = 1;
    MrcWriteCR (MrcData, DDRPHY_COMP_CR_VSSHITARGET_REG, VssHiTarget.Data);

    //Initialize variables for this sweep
    SweepNotFinished = TRUE;
    LowSigned = DDRVSSHIAFEA_CR_DDRCRVSSHICONTROL_Target_MIN;
    HighSigned = DDRVSSHIAFEA_CR_DDRCRVSSHICONTROL_Target_MAX;
    VssHiControl.Data = MrcReadCR (MrcData, DDRVSSHIAFEA_CR_DDRCRVSSHICONTROL_REG);
    MiddleSigned = VssHiControl.Bits.Target;

    //Sweep Target Binary
    while (SweepNotFinished) {
      VssHiControl.Bits.Target = MiddleSigned;
      MrcWriteCR (MrcData, DDRVSSHIAFEA_CR_DDRCRVSSHICONTROL_REG, VssHiControl.Data);
      ForceRcomp (MrcData);

      VssHiTarget.Bits.RstCount = 1;
      MrcWriteCR (MrcData, DDRPHY_COMP_CR_VSSHITARGET_REG, VssHiTarget.Data);
      MrcWait (MrcData, Wait4000Qclks);

      VssHiTarget.Data = MrcReadCR (MrcData, DDRPHY_COMP_CR_VSSHITARGET_REG);
*/
/*
      if (VssHiTarget.Bits.CountMSB >= 0x8) {
        LowSigned = MiddleSigned + 1;
      } else {
        HighSigned = MiddleSigned - 1;
      }
      if (HighSigned < LowSigned) {
        SweepNotFinished = FALSE;
      }
      MiddleSigned = (HighSigned + LowSigned) / 2;
#ifdef CTE_FLAG
      SweepNotFinished = FALSE;
#endif
    }
    VssHiTarget.Bits.RstCount = 0;
    VssHiTarget.Bits.ForceCompAmpOn = 0;
    MrcWriteCR (MrcData, DDRPHY_COMP_CR_VSSHITARGET_REG, VssHiTarget.Data);
#ifdef MRC_DEBUG_PRINT
    MRC_DEBUG_MSG (Debug, MSG_LEVEL_WARNING, "\n%sPreTrain_Offset_Correction_VssHi_FB\n", gStartTagStr);
    MRC_DEBUG_MSG (Debug, MSG_LEVEL_WARNING, "\t0x%x\n", VssHiControl.Bits.Target);
    MRC_DEBUG_MSG (Debug, MSG_LEVEL_WARNING, "%sPreTrain_Offset_Correction_VssHi_FB\n", gStopTagStr);
#endif
*/
    //VsxHi FF Offset (@todo LPMode = 3)
    MRC_DEBUG_MSG (Debug, MSG_LEVEL_NOTE, "VssHi FF Offset\n");
    GetSetVal = 1;
    MrcGetSetNoScope (MrcData, GsmIocForceCmpUpdt, WriteCached, &GetSetVal);
    MrcWait (MrcData, 100 * MRC_TIMER_1NS);

    //Freeze FB loop
    FreezeVccDllVsxHi (MrcData, VSXHI, TRUE);

    //Override = 1
    CompOverride1.Data = MrcReadCR (MrcData, DDRPHY_COMP_CR_DDRCRCOMPOVR1_REG);
    CompOverride1.Bits.DqVsxHiFF = 1;
    CompOverride1.Bits.CmdVsxHiFF = 1;
    CompOverride1.Bits.CtlVsxHiFF = 1;
    CompOverride1.Bits.ClkVsxHiFF = 1;
    MrcWriteCR (MrcData, DDRPHY_COMP_CR_DDRCRCOMPOVR1_REG, CompOverride1.Data);

    //LPMode = 0 (@todo)
    GetSetVal = 1;
    MrcGetSetChStrb (MrcData, MAX_CONTROLLER, MAX_CHANNEL, MAX_SDRAM_IN_DIMM, GsmIocInternalClocksOn, WriteCached | DebugPrint, &GetSetVal);
    IoReset (MrcData);

    //CCC partitions
    SetupIOTestCADB (MrcData, McChBitMask, 10, NSOE, 1, 0);
    //Enable CADB Always On mode (Override the one in SetupIOTest)
    CadbConfig.Data = 0;
    CadbConfig.Bits.CADB_MODE = Cadb20ModeAlwaysOn;
    CadbConfig.Bits.LANE_DESELECT_EN = 5;        // Drive deselects on CS and CA pins; @todo - not sure this is needed in AlwaysOn mode ?
    CadbConfig.Bits.INITIAL_DSEL_EN = 1;
    CadbConfig.Bits.INITIAL_DSEL_SSEQ_EN = 1;
    Cadb20ConfigRegWrite (MrcData, CadbConfig);
    // Enable CADB on MC
    McCadbEnable (MrcData, TRUE);

    //SelectReutRanks skipped as earlier step is still valid

    //Initialize variables for the CCC Partitions
    SweepNotFinished = TRUE;
    MrcGetSetLimits (MrcData, VssHiFFClk, &GetSetMin, &GetSetMax, NULL);
    LowSigned = (INT32) GetSetMin;
    HighSigned = (INT32) GetSetMax;
    //Binary Sweep
    while (SweepNotFinished) {
      GetSetVal = (LowSigned + HighSigned) / 2;

      //Stop CADB Test
      Cadb20ControlRegWrite (MrcData, McChBitMask, CadbStopTest);

      for (Controller = 0; Controller < MAX_CONTROLLER; Controller++) {
        for (Channel = 0; Channel < MAX_CCC_PER_CONTROLLER; Channel++) {
          MrcGetSetCcc (MrcData, Controller, Channel, MRC_IGNORE_ARG, MRC_IGNORE_ARG, VssHiFFClk, WriteToCache, &GetSetVal);
          MrcGetSetCcc (MrcData, Controller, Channel, MRC_IGNORE_ARG, MRC_IGNORE_ARG, VssHiFFCtl, WriteToCache, &GetSetVal);
          MrcGetSetCcc (MrcData, Controller, Channel, MRC_IGNORE_ARG, MRC_IGNORE_ARG, VssHiFFCa, WriteToCache, &GetSetVal);
        }
      }
      MrcFlushRegisterCachedData (MrcData);
      //ForceCompUpdate to make sure VssHiFF takes
      MrcGetSetNoScope (MrcData, GsmIocForceCmpUpdt, WriteCached, &GetSetEn);

      //Start CADB Test
      Cadb20ControlRegWrite (MrcData, McChBitMask, CadbStartTest);

      VssHiTarget.Bits.RstCount = 1;
      MrcWriteCR (MrcData, DDRPHY_COMP_CR_VSSHITARGET_REG, VssHiTarget.Data);
      MrcWait (MrcData, Wait4000Qclks);

      VssHiTarget.Data = MrcReadCR (MrcData, DDRPHY_COMP_CR_VSSHITARGET_REG);
      if (VssHiTarget.Bits.CountMSB >= 0x8) {
        LowSigned = (INT32) GetSetVal + 1;
      } else {
        HighSigned = (INT32) GetSetVal - 1;
      }

      if (HighSigned < LowSigned) {
        SweepNotFinished = FALSE;
      }
    }
    // Stop CADB Test
    Cadb20ControlRegWrite (MrcData, McChBitMask, CadbStopTest);
    // Stop CADB
    CadbConfig.Data = 0;
    Cadb20ConfigRegWrite (MrcData, CadbConfig);
    // Disable CADB on MC
    McCadbEnable (MrcData, FALSE);

    //Enable CPGC for EndlessWrite
    SetupIOTestBasicVA (MrcData, McChBitMask, 1, NSOE, 0, 0, 10, PatWrEndless, 0, 0);
    Cpgc20ControlRegWrite (MrcData, McChBitMask, CpgcStartTest);

    //Initialize variables for the Data Partitions
    SweepNotFinished = TRUE;
    MrcGetSetLimits (MrcData, VssHiFFData, &GetSetMin, &GetSetMax, NULL);
    LowSigned = (INT32) GetSetMin;
    HighSigned = (INT32) GetSetMax;

    //Binary Sweep
    while (SweepNotFinished) {
      GetSetVal = (LowSigned + HighSigned) / 2;
      for (Controller = 0; Controller < MAX_CONTROLLER; Controller++) {
        for (Channel = 0; Channel < Outputs->MaxChannels; Channel++) {
          for (Byte = 0; Byte < MaxByte; Byte++) {
            MrcGetSetChStrb (MrcData, Controller, Channel, Byte, VssHiFFData, WriteToCache, &GetSetVal);
          }
        }
      }
      MrcFlushRegisterCachedData (MrcData);
      //ForceCompUpdate to make sure VssHiFF takes
      MrcGetSetNoScope (MrcData, GsmIocForceCmpUpdt, WriteCached, &GetSetEn);

      VssHiTarget.Bits.RstCount = 1;
      MrcWriteCR (MrcData, DDRPHY_COMP_CR_VSSHITARGET_REG, VssHiTarget.Data);
      MrcWait (MrcData, Wait4000Qclks);

      VssHiTarget.Data = MrcReadCR (MrcData, DDRPHY_COMP_CR_VSSHITARGET_REG);
      if (VssHiTarget.Bits.CountMSB >= 0x8) {
        LowSigned = (INT32) GetSetVal + 1;
      } else {
        HighSigned = (INT32) GetSetVal - 1;
      }

      if (HighSigned < LowSigned) {
        SweepNotFinished = FALSE;
      }
    }

    //Clean up
    Cpgc20ControlRegWrite (MrcData, McChBitMask, CpgcStopTest);

    GetSetVal = 0;
    MrcGetSetNoScope (MrcData, GsmIocForceCmpUpdt, WriteCached, &GetSetVal);

    //Unfreeze FB
    FreezeVccDllVsxHi (MrcData, VSXHI, FALSE);

#ifdef MRC_DEBUG_PRINT
    MRC_DEBUG_MSG (Debug, MSG_LEVEL_WARNING, "\n%sPreTrain_Offset_Correction_VssHi_FF\n", gStartTagStr);
    MRC_DEBUG_MSG (Debug, MSG_LEVEL_WARNING, "%sPreTrain_Offset_Correction_VssHi_FF\n", gStopTagStr);
#endif
  } // VssHiBypass == FALSE

  return Status;
}

/**
  This function runs VssHi Regulator Offset Correction - PreTraining

  @param[in, out] MrcData - Include all MRC global data.
  @param[in] DebugPrint   - To print debug messages or not.
**/
void
MrcVssHiRegulatorOffsetCorrection (
  IN OUT MrcParameters *const MrcData,
  IN     BOOLEAN              DebugPrint
  )
{
  MrcInput           *Inputs;
  MrcOutput          *Outputs;
  MrcDebug           *Debug;
  MrcDebugMsgLevel   DebugLevel;
  UINT32             Vdd2Mv;
  INT32              TargetV;
  INT32              PanicV;
  INT32              VccIoMvMinus375mV;
  INT32              ErrorTarget1;
  INT32              ErrorHi1;
  INT32              ErrorLo1;
  INT32              ErrorTarget2;
  INT32              ErrorHi2;
  INT32              ErrorLo2;
  BOOLEAN            Fail;
  BOOLEAN            VssHiBypass;
  DDRVSSHIAFEA_CR_DDRCRVSSHISAMPLER_STRUCT  VssHiSampler;
  DDRVSSHIAFEA_CR_DDRCRVSSHICOMPOFFSET_STRUCT  VssHiCompOffset;

  Inputs            = &MrcData->Inputs;
  Outputs           = &MrcData->Outputs;
  Debug             = &Outputs->Debug;
  Vdd2Mv            = Outputs->Vdd2Mv;
  VccIoMvMinus375mV = MAX ((INT32) (Inputs->VccIomV) - 375, 0);
  DebugLevel = DebugPrint ? MSG_LEVEL_NOTE : MSG_LEVEL_NEVER;

  VssHiSampler.Data = MrcReadCR (MrcData, DDRVSSHIAFEA_CR_DDRCRVSSHISAMPLER_REG);
  VssHiBypass = (VssHiSampler.Bits.DdrVssHiGndSelQnnnH) ? TRUE : FALSE;

  if (VssHiBypass == FALSE) {
    TargetV = (VssHiSampler.Bits.Target + 32) * Vdd2Mv / 384;
    PanicV = (VssHiSampler.Bits.Panic) * Vdd2Mv / 384;

    VssHiSampler.Bits.VssHiTargetUsePmos = (TargetV < VccIoMvMinus375mV) ? 1 : 0;
    VssHiSampler.Bits.PanicHiUsePmos = ((TargetV + PanicV) < VccIoMvMinus375mV) ? 1 : 0;
    VssHiSampler.Bits.PanicLoUsePmos = ((TargetV - PanicV) < VccIoMvMinus375mV) ? 1 : 0;

    MrcWriteCR (MrcData, DDRVSSHIAFEA_CR_DDRCRVSSHISAMPLER_REG, VssHiSampler.Data);
    MRC_DEBUG_MSG (Debug, DebugLevel, "Vdd2Mv = %dmV, VccIo - 375 = %dmV, TargetV = %dmV, PanicV = %dmV, DDRVSSHIAFEA_CR_DDRCRVSSHISAMPLER = 0x%x\n", Vdd2Mv, VccIoMvMinus375mV, TargetV, PanicV, VssHiSampler.Data);
    ForceRcomp (MrcData);

    Fail = FALSE;
    VssHiCompOffset.Data = MrcReadCR (MrcData, DDRVSSHIAFEA_CR_DDRCRVSSHICOMPOFFSET_REG);
    ErrorTarget1 = (VssHiCompOffset.Bits.VsshiTargetCompOfst * Vdd2Mv / 384) + TargetV - VccIoMvMinus375mV;
    ErrorHi1 = (VssHiCompOffset.Bits.PanicHiCompOfst * Vdd2Mv / 384) + TargetV + PanicV - VccIoMvMinus375mV;
    ErrorLo1 = (VssHiCompOffset.Bits.PanicLoCompOfst * Vdd2Mv / 384) + TargetV - PanicV - VccIoMvMinus375mV;
    MRC_DEBUG_MSG (Debug, DebugLevel, "VsshiTargetCompOfst = 0x%x, PanicHiCompOfst = 0x%x, PanicLoCompOfst = 0x%x\n", VssHiCompOffset.Bits.VsshiTargetCompOfst, VssHiCompOffset.Bits.PanicHiCompOfst, VssHiCompOffset.Bits.PanicLoCompOfst);
    if (ErrorTarget1 > 0) {
      Fail = TRUE;
      VssHiSampler.Bits.VssHiTargetUsePmos = ~VssHiSampler.Bits.VssHiTargetUsePmos;
      MRC_DEBUG_MSG (Debug, DebugLevel, "1st Try Error on VssHiTargetUsePmos\n");
    }
    if (ErrorHi1 > 0) {
      Fail = TRUE;
      VssHiSampler.Bits.PanicHiUsePmos = ~VssHiSampler.Bits.PanicHiUsePmos;
      MRC_DEBUG_MSG (Debug, DebugLevel, "1st Try Error on PanicHiUsePmos\n");
    }
    if (ErrorLo1 > 0) {
      Fail = TRUE;
      VssHiSampler.Bits.PanicLoUsePmos = ~VssHiSampler.Bits.PanicLoUsePmos;
      MRC_DEBUG_MSG (Debug, DebugLevel, "1st Try Error on PanicLoUsePmos\n");
    }

    if (Fail) {
      MrcWriteCR (MrcData, DDRVSSHIAFEA_CR_DDRCRVSSHISAMPLER_REG, VssHiSampler.Data);
      MRC_DEBUG_MSG (Debug, DebugLevel, "DDRVSSHIAFEA_CR_DDRCRVSSHISAMPLER = 0x%x\n", VssHiSampler.Data);
      ForceRcomp (MrcData);

      Fail = FALSE;
      VssHiCompOffset.Data = MrcReadCR (MrcData, DDRVSSHIAFEA_CR_DDRCRVSSHICOMPOFFSET_REG);
      ErrorTarget2 = 375 - ((VssHiCompOffset.Bits.VsshiTargetCompOfst * Vdd2Mv / 384) + TargetV);
      ErrorHi2 = 375 - ((VssHiCompOffset.Bits.PanicHiCompOfst * Vdd2Mv / 384) + TargetV + PanicV);
      ErrorLo2 = 375 - ((VssHiCompOffset.Bits.PanicLoCompOfst * Vdd2Mv / 384) + TargetV - PanicV);
      MRC_DEBUG_MSG (Debug, DebugLevel, "VsshiTargetCompOfst = 0x%x, PanicHiCompOfst = 0x%x, PanicLoCompOfst = 0x%x\n", VssHiCompOffset.Bits.VsshiTargetCompOfst, VssHiCompOffset.Bits.PanicHiCompOfst, VssHiCompOffset.Bits.PanicLoCompOfst);

      if ((ErrorTarget1 > 0) && (ErrorTarget1 < ErrorTarget2)) {
        Fail = TRUE;
        VssHiSampler.Bits.VssHiTargetUsePmos = ~VssHiSampler.Bits.VssHiTargetUsePmos;
        MRC_DEBUG_MSG (Debug, DebugLevel, "2nd Try Error on VssHiTargetUsePmos\n");
      }
      if ((ErrorHi1 > 0) && (ErrorHi1 < ErrorHi2)) {
        Fail = TRUE;
        VssHiSampler.Bits.PanicHiUsePmos = ~VssHiSampler.Bits.PanicHiUsePmos;
        MRC_DEBUG_MSG (Debug, DebugLevel, "2nd Try Error on PanicHiUsePmos\n");
      }
      if ((ErrorLo1 > 0) && (ErrorLo1 < ErrorLo2)) {
        Fail = TRUE;
        VssHiSampler.Bits.PanicLoUsePmos = ~VssHiSampler.Bits.PanicLoUsePmos;
        MRC_DEBUG_MSG (Debug, DebugLevel, "2nd Try Error on PanicLoUsePmos\n");
      }
      if (Fail) {
        MrcWriteCR (MrcData, DDRVSSHIAFEA_CR_DDRCRVSSHISAMPLER_REG, VssHiSampler.Data);
        MRC_DEBUG_MSG (Debug, DebugLevel, "DDRVSSHIAFEA_CR_DDRCRVSSHISAMPLER = 0x%x\n", VssHiSampler.Data);
        ForceRcomp (MrcData);
      }
    }

#ifdef MRC_DEBUG_PRINT
    MRC_DEBUG_MSG (Debug, MSG_LEVEL_WARNING, "\n%sVssHi_Regulator_Offset_Correction\n", gStartTagStr);
    MRC_DEBUG_MSG (Debug, MSG_LEVEL_WARNING, "DDRVSSHIAFEA_CR_DDRCRVSSHISAMPLER = 0x%x\n", VssHiSampler.Data);
    MRC_DEBUG_MSG (Debug, MSG_LEVEL_WARNING, "VssHiTargetUsePmos = 0x%x\n", VssHiSampler.Bits.VssHiTargetUsePmos);
    MRC_DEBUG_MSG (Debug, MSG_LEVEL_WARNING, "PanicHiUsePmos = 0x%x\n", VssHiSampler.Bits.PanicHiUsePmos);
    MRC_DEBUG_MSG (Debug, MSG_LEVEL_WARNING, "PanicLoUsePmos = 0x%x\n", VssHiSampler.Bits.PanicLoUsePmos);
    MRC_DEBUG_MSG (Debug, MSG_LEVEL_WARNING, "%sVssHi_Regulator_Offset_Correction\n", gStopTagStr);
#endif
  } //VssHiBypass == FALSE

  return;
}

/**
  This function runs the Inner Test Content for  VccDll Read/Write/Idle FF Timing
  Idle isnt implemented for A0

  @param[in]  MrcData - Pointer to global MRC data.
  @param[in]  VccDllTargetOrig - Register value to avoid reading it again.

  @retval Goodness - Subtract VccDllTarget.CountMSB for 7mV from VccDllTarget.CountMSB -7mV
**/
INT8
RunInnerTest (
  IN MrcParameters    *const MrcData,
  IN UINT32           *const VccdllTarget
)
{
  MrcOutput           *Outputs;
  MrcVddSelect        Vccddq;
  UINT8               McChBitMask;
  UINT8               OffsetV;
  UINT8               CountMSB [MAX_OFFSET_VOLTAGE];
  INT8                Goodness;
  INT16               OffsetVoltage [MAX_OFFSET_VOLTAGE];
  DDRPHY_COMP_CR_VCCDLLTARGET_STRUCT              VccDllTarget;
  MC0_REQ0_CR_CPGC_SEQ_CTL_STRUCT  CpgcStartTest;
  MC0_REQ0_CR_CPGC_SEQ_CTL_STRUCT  CpgcStopTest;

  Outputs           = &MrcData->Outputs;
  Vccddq            = Outputs->VccddqVoltage;
  McChBitMask       = Outputs->McChBitMask;
  OffsetVoltage[0]  = 7 * 384;
  OffsetVoltage[1]  = -7 * 384;

  CpgcStartTest.Data = CpgcStopTest.Data = 0;
  CpgcStartTest.Bits.START_TEST = 1;
  CpgcStopTest.Bits.STOP_TEST = 1;

  Cpgc20ControlRegWrite (MrcData, McChBitMask, CpgcStartTest);
  for (OffsetV = 0; OffsetV < MAX_OFFSET_VOLTAGE; OffsetV++) {
    VccDllTarget.Data = *VccdllTarget;
    VccDllTarget.Bits.CompVTarget +=  DIVIDEROUND (OffsetVoltage[OffsetV], Vccddq);
    VccDllTarget.Bits.RstCount = 1;
    MrcWriteCR (MrcData, DDRPHY_COMP_CR_VCCDLLTARGET_REG, VccDllTarget.Data);
    MrcWait (MrcData, 1000); // Wait 1us, Allow DAC to settle before starting test. The new wait function takes delay in ns
    VccDllTarget.Data = MrcReadCR (MrcData, DDRPHY_COMP_CR_VCCDLLTARGET_REG);
    CountMSB[OffsetV] = (UINT8) VccDllTarget.Bits.CountMSB;
  }
  Cpgc20ControlRegWrite (MrcData, McChBitMask, CpgcStopTest);
  Goodness = (UINT8)MrcSE (CountMSB[1], DDRPHY_COMP_CR_VCCDLLTARGET_CountMSB_WID, 8) - (UINT8)MrcSE (CountMSB[0], DDRPHY_COMP_CR_VCCDLLTARGET_CountMSB_WID, 8); // (CountMSB for -7mv) - (CountMSB for 7mv)
  MRC_DEBUG_MSG (&Outputs->Debug, MSG_LEVEL_NOTE, "\n Goodness = %i", Goodness);
  return Goodness;
}

/**
  This function finds the best setting for the following :
  FFWriteStagger during FF Write training
  FFRcvEnPi and FFRcvEnPre during FF Read training. FFRcvEnPi + FFRcvEnPre = 5. So we increment FFRcvEnPi while decrementing FFRcvEnPre and determine which
  combination gives us the maximum Goodness.

  @param[in]  MrcData - Pointer to global MRC data.
  @param[in]  CmdType - Pass the enum RdEndless, WrEndless or Idle (enum doesnt exist currently).
  @param[in]  VccdllTarget - Init value for VccDllTarget register
**/
VOID
VccDllCompWriteReadIdleTuning (
  IN MrcParameters    *const MrcData,
  IN UINT8            CmdType,
  IN UINT32           *const VccdllTarget
)
{
  //MrcInput    *Inputs;
  MrcOutput   *Outputs;
  INT8              GoodnessWr[MAX_FF_WRITESTAGGER];
  INT8              GoodnessRd;
  INT8              Goodness;
//  UINT32            OffsetDataControl6;
  UINT32            Offset;
//  UINT32            RcvEnExtFF;
//  UINT32            FirstController;
  UINT8             FFWriteStagger;
  UINT8             FFRcvEnPi;
  UINT8             FFRcvEnPre;
  UINT8             GoodnessRcvEnPi;
  UINT8             GoodnessRcvEnPre;
  UINT8             Byte;
  DLLDDRDATA0_CR_DDRCRVCCDLLFFCONTROL_STRUCT      VccDllFFControl;
//  DATA0CH0_CR_DDRCRDATACONTROL6_STRUCT            DataControl6;

  //Inputs = &MrcData->Inputs;
  Outputs = &MrcData->Outputs;
  GoodnessRcvEnPi = 0;
  GoodnessRcvEnPre = 0;
  //DataControl6.Data = 0;
  // Read the first populated channel
  // FirstController = (MrcControllerExist (MrcData, cCONTROLLER0)) ? 0 : 1; // This is only needed for DataControl6 Offset read and the Channel is equivalent to controller

  for (Byte = 0; Byte < (MAX_SDRAM_IN_DIMM - 1); Byte++) {
    Offset = DLLDDRDATA0_CR_DDRCRVCCDLLFFCONTROL_REG + (DLLDDRDATA1_CR_DDRCRVCCDLLFFCONTROL_REG - DLLDDRDATA0_CR_DDRCRVCCDLLFFCONTROL_REG) * Byte;
    VccDllFFControl.Data = MrcReadCR (MrcData, Offset);
    if (CmdType == PatWrEndless) {
      for (FFWriteStagger = 0; FFWriteStagger < MAX_FF_WRITESTAGGER; FFWriteStagger++) {
        VccDllFFControl.Bits.FFWriteStagger = FFWriteStagger;
        MrcWriteCR (MrcData, Offset, VccDllFFControl.Data);
        GoodnessWr[FFWriteStagger] = RunInnerTest (MrcData, VccdllTarget);
      }
      if (GoodnessWr[0] > GoodnessWr[1]) {
        FFWriteStagger = 0;
      } else {
        FFWriteStagger = 1;
      }
      // Preserve the max goodness value for each byte
      VccDllFFControl.Bits.FFWriteStagger = FFWriteStagger;

    } else if (CmdType == PatRdEndless) {
      GoodnessRd = MRC_INT8_MIN;
      //if (!(Inputs->A0)) {
        //OffsetDataControl6 = DataControl6Offset (FirstController, Byte); // Assume all channels are programmed as first channel
        //DataControl6.Data = MrcReadCR (MrcData, OffsetDataControl6);
      //}
      for (FFRcvEnPi = 0; FFRcvEnPi < MAX_FF_RCVEN_PI; FFRcvEnPi++) {
        for (FFRcvEnPre = 0; FFRcvEnPre < (MAX_FF_RCVEN_PRE - FFRcvEnPi); FFRcvEnPre++) {
          //if (!(Inputs->A0)) {
            //RcvEnExtFF = RANGE (((5 / FFRcvEnPi) * (DataControl6.Bits.RXTogglePreAmble << ((!Outputs->Gear2) ? 1 : 0))), 0, 3);
          //}
          VccDllFFControl.Bits.FFRcvEnPi = FFRcvEnPi;
          VccDllFFControl.Bits.FFRcvEnPre = FFRcvEnPre;
          MrcWriteCR (MrcData, Offset, VccDllFFControl.Data);
          Goodness = RunInnerTest (MrcData, VccdllTarget);
          if (GoodnessRd < Goodness) {
            GoodnessRd = Goodness;
            if (GoodnessRd == DDRPHY_COMP_CR_VCCDLLTARGET_CountMSB_MAX) {
              MRC_DEBUG_MSG (&Outputs->Debug, MSG_LEVEL_WARNING, "\n Reached Max Goodness = %d", GoodnessRd);
            }
            GoodnessRcvEnPi = FFRcvEnPi;
            GoodnessRcvEnPre = FFRcvEnPre;
          }
        } // FFRcvEnPre
      } // FFRcvEnPi
      VccDllFFControl.Bits.FFRcvEnPi = GoodnessRcvEnPi;
      VccDllFFControl.Bits.FFRcvEnPre = GoodnessRcvEnPre;
    } // Read
    MrcWriteCR (MrcData, Offset, VccDllFFControl.Data);
  } // Byte
}

/**
  This function trains VccDll for Write, Read and Idle FeedForward timing.

  @param[in]  MrcData - Pointer to global MRC data.

  @retval MrcStatus - mrcSuccess
**/
MrcStatus
MrcVccDllFFTiming (
  IN  MrcParameters *const  MrcData
  )
{
  MrcOutput           *Outputs;
  MrcStatus           Status;
  UINT8               NumCL;
  UINT8               VccDllRail;
  UINT8               Wait;
  UINT8               Train;
  UINT32              VccDllTargetOrig;
  DDRPHY_COMP_CR_VCCDLLTARGET_STRUCT           VccDllTarget;


  Status      = mrcSuccess;
  Outputs     = &MrcData->Outputs;
  // BurstLength is in terms of tCK and not in terms of no. of bursts of DQS
  NumCL       = MRC_NUMBURSTS_FFTIMING / (Outputs->BurstLength * 2);

  VccDllTarget.Data = MrcReadCR (MrcData, DDRPHY_COMP_CR_VCCDLLTARGET_REG);
  VccDllTargetOrig = VccDllTarget.Data;
  VccDllTarget.Bits.ForceCompAmpOn = 1;

  // Run each training i.e Write, Read and Idle
  for (Train = PatWrEndless; Train <= PatRdEndless; Train++) {
    // @todo change the condition to !Idle when Idle is coded
    Wait = ((Train == PatWrEndless) || (Train == PatRdEndless))  ? 32 : 128;  // Number of Qclks to wait between CPGC Algos
    SetupIOTestBasicVA (MrcData, Outputs->McChBitMask, 1, NSOE, 0, 0, 9, Train, Wait, NumCL); // This will setup for Traffic - Idle.

    for (VccDllRail = VCCDLL1; VccDllRail <= GetMaxVccDllRail (MrcData); VccDllRail++) {
      VccDllTarget.Bits.SelectVccDllRail = VccDllRail;
      MrcWriteCR (MrcData, DDRPHY_COMP_CR_VCCDLLTARGET_REG, VccDllTarget.Data);
      VccDllCompWriteReadIdleTuning (MrcData, Train, &VccDllTarget.Data);
    }
  }

  // Restore the values of VccDllTarget
  VccDllTarget.Data = VccDllTargetOrig;
  MrcWriteCR (MrcData, DDRPHY_COMP_CR_VCCDLLTARGET_REG, VccDllTarget.Data);
  return Status;
}

/**
  This function runs the test to update the VccDllTarget.countMsb

  @param[in]  MrcData - Pointer to global MRC data.
  @param[in]  TargetRail - VccDll North or South Rail.
  @param[in]  WaitTime - Wait for function call MrcWait.
  @param[in]  VccDllTargetOrig - Save the original value of VccDllTarget before modifying it.

  @retval Count - Returns CountMsb
**/
INT8
GetVccDllVsxHiCompVrefCountMsb (
  IN MrcParameters    *const MrcData,
  IN UINT8                   TargetRail,
  IN UINT32                  WaitTime,
  IN UINT32           *const TargetOrig
  )
{
  DDRPHY_COMP_CR_VCCDLLTARGET_STRUCT   VccDllTarget;
  DDRPHY_COMP_CR_VSSHITARGET_STRUCT    VssHiTarget;
  UINT8                                CountMsb;
  INT8                                 Count;

  Count = 0;
  if (TargetRail != VSXHI) {
    VccDllTarget.Data = *TargetOrig;
    VccDllTarget.Bits.ForceCompAmpOn = 1;
    VccDllTarget.Bits.SelectVccDllRail = TargetRail;
    VccDllTarget.Bits.RstCount = 1;
    MrcWriteCR (MrcData, DDRPHY_COMP_CR_VCCDLLTARGET_REG, VccDllTarget.Data);
    MrcWait (MrcData, WaitTime);
    VccDllTarget.Data = MrcReadCR (MrcData, DDRPHY_COMP_CR_VCCDLLTARGET_REG);
    CountMsb = (UINT8) VccDllTarget.Bits.CountMSB;
    Count = (INT8)MrcSE (CountMsb, DDRPHY_COMP_CR_VCCDLLTARGET_CountMSB_WID, 8);
  } else {
    VssHiTarget.Data = *TargetOrig;
    VssHiTarget.Bits.ForceCompAmpOn = 1;
    VssHiTarget.Bits.SelectVssHiRail= (TargetRail == VSXHI) ? 1 : 0;
    VssHiTarget.Bits.RstCount = 1;
    MrcWriteCR (MrcData, DDRPHY_COMP_CR_VSSHITARGET_REG, VssHiTarget.Data);
    MrcWait (MrcData, WaitTime);
    VssHiTarget.Data = MrcReadCR (MrcData, DDRPHY_COMP_CR_VSSHITARGET_REG);
    CountMsb = (UINT8) VssHiTarget.Bits.CountMSB;
    Count = (INT8)MrcSE (CountMsb, DDRPHY_COMP_CR_VSSHITARGET_CountMSB_WID, 8);
  }
  MRC_DEBUG_MSG (&MrcData->Outputs.Debug, MSG_LEVEL_NOTE, "\n Count 0x%x\n", Count);
  return Count;
}

/**
  This function to finds the transition of CountMsb from -ve to +ve or vice-versa

  @param[in]  MrcData - Pointer to global MRC data.
  @param[in]  Code - Feed Forward Offset Code.
  @param[in]  Partition - VccDll Partition.
  @param[in]  TargetRail - VccDll North or South voltage rail.
  @param[in] VccDllTargetOrig - VccDllTarget setting as per Phyinit.
**/
VOID
LinearSearch (
  IN MrcParameters    *const MrcData,
  IN UINT32                  Code,
  IN UINT8                   Partition,
  IN UINT8                   TargetRail,
  IN UINT32           *const VccDllTargetOrig
  )
{
  MrcInput   *Inputs;
  INT64  TargetVal;
  INT64  Min;
  INT8   Step;
  UINT32 WaitTime;
  UINT32 Channel;
  UINT32 Byte;
  INT64  Max;
  INT16  Result;
  UINT8  MSBValue;
  UINT8  PrevMSBValue;
  UINT8  Par;
  UINT8  ParLimit;
  UINT32 MaxVccDllRail;
  BOOLEAN Done;
  BOOLEAN IdleOrWeak;

  Inputs = &MrcData->Inputs;
  MaxVccDllRail = GetMaxVccDllRail (MrcData);
  TargetVal = 0;
  PrevMSBValue = 0;
  Done = FALSE;
  Step = 0;
  //WaitTime = (MrcData->Outputs.Qclkps * 4) * MRC_TIMER_1NS; // The new wait function takes delay in ns
  WaitTime = (MrcData->Outputs.Qclkps * 4); // remove this and uncomment above after TGL_MRC_PO_Branch is merged
  IdleOrWeak = ((Code == GsmIocFFCodeIdleOffset) || (Code == GsmIocFFCodeWeakOffset)) ? TRUE : FALSE;
  ParLimit = IdleOrWeak ? DllDdrMax : Partition;

  MrcGetSetLimits (MrcData, Code, &Min, &Max, NULL);
  do {
    // This loop will run only once for all other codes other than IdleOrWeak
    for (Par = Partition; Par <= ParLimit; Par++) {
      if ((Partition == DllDdrData8) && (Inputs->UlxUlt)) { //Skip ECC byte for ULXULT
        continue;
      }
      // decode Channel and Byte from Partition to program correct VccDllComp reg
      Channel = (Par > (MAX_VCCDLL_DATA_PAR - 1)) ? (Par - MAX_VCCDLL_DATA_PAR) : MRC_IGNORE_ARG;
      Byte = (Par > (MAX_VCCDLL_DATA_PAR - 1)) ? MRC_IGNORE_ARG : Par;
      GetSet (MrcData, MRC_IGNORE_ARG, MRC_IGNORE_ARG, Channel, MRC_IGNORE_ARG, MRC_IGNORE_ARG, Byte, MRC_IGNORE_ARG, MRC_IGNORE_ARG, MRC_IGNORE_ARG, Code, WriteCached, &TargetVal);
    }
    // ClampCheck code
    if ((TargetVal < Min) || (TargetVal > Max)) {
      Done = TRUE;
    }
    Result = GetVccDllVsxHiCompVrefCountMsb (MrcData, TargetRail, WaitTime, VccDllTargetOrig);
    if (IdleOrWeak) {
      Result += GetVccDllVsxHiCompVrefCountMsb (MrcData, (UINT8) MaxVccDllRail, WaitTime, VccDllTargetOrig); // Result is 16 bits wide because of Binary add for 2 UINT8
    }

    MSBValue = Result ? 1 : 0;
    if (TargetVal == 0) {
      PrevMSBValue = MSBValue;
      Step = (MSBValue) ? -1 : 1;
    } else if (MSBValue != PrevMSBValue) {
      Done = TRUE;
    }
    TargetVal += Step;
  }  while (!Done);

  FreezeVccDllVsxHi (MrcData, TargetRail, TRUE);
  if (IdleOrWeak) {
    FreezeVccDllVsxHi (MrcData, MaxVccDllRail, TRUE);
  }
}

/**
  This function trains VccDll Per Module FF Offset Correction

  @param[in]  MrcData - Pointer to global MRC data.

  @retval MrcStatus - mrcSuccess if successful, else an error status

**/
MrcStatus
MrcVccDllFFOffsetCorrection (
  IN MrcParameters    *const MrcData
  )
{
  const MrcInput    *Inputs;
  MrcOutput         *Outputs;
  MrcStatus         Status;
  INT64             GetSetVal;
  UINT32            Offset;
  UINT32            Data;
  UINT32            ChannelPerRail[VCC_DLL_BYTES];
  UINT32            BytePerRail[VCC_DLL_BYTES];
  UINT8             NumCL;
  UINT8             Channel;
  UINT8             Byte;
  UINT32            TargetRail;
  UINT32            Code;
  UINT32            FirstController;
  UINT32            FirstChannel;
  UINT32            FirstByte;
  UINT32            VccDllTargetOrig;
  UINT8             VccDllRail;
  UINT8             Index;
  UINT8             ParStart;
  UINT8             ParEnd;
  UINT8             Partition;
  INT64             InternalClocks;
  INT64             TxPiOn;
  INT64             RxPiOn;
  INT64             RxDisable;
  INT64             WlWakeCyc;
  INT64             WlSleepCyc;
  INT32             AvgCodePiOffset;
  UINT32            CccPinsUsedSaved[MRC_NUM_CCC_INSTANCES];
  BOOLEAN           DtHalo;
  CH0CCC_CR_DDRCRPINSUSED_STRUCT          *CccPinsUsed;
  CH0CCC_CR_DDRCRPINSUSED_P0_STRUCT       *CccPinsUsedP0;
  MC0_REQ0_CR_CPGC_SEQ_CTL_STRUCT         CpgcStartTest;
  MC0_REQ0_CR_CPGC_SEQ_CTL_STRUCT         CpgcStopTest;
  const MRC_FUNCTION                      *MrcCall;

  static UINT32 const TargetCode [] = {GsmIocFFCodeIdleOffset, GsmIocFFCodeWeakOffset, GsmIocFFCodePiOffset, GsmIocFFCodeCCCDistOffset, GsmIocFFCodePBDOffset,
    GsmIocFFCodeWriteOffset, GsmIocFFCodeReadOffset};

  // @todo Add EccByte DLL Partition for TGL H DDR4
  static UINT8 const ParStartEnd[2][MAX_VCCDLL_CODES] =
  { // Code   Idle   Weak   Pi   CCCDist   Pbd   Write   Read
              {0,     0,    0,     8,       0,     0,     0},
              {0,     0,    7,     11,      11,    7,     7}};

  Status  = mrcSuccess;
  Inputs  = &MrcData->Inputs;
  Outputs = &MrcData->Outputs;
  MrcCall = Inputs->Call.Func;
  DtHalo = Inputs->DtHalo;

  // Assuming Numburst = 32
  NumCL     = MRC_NUMBURSTS_FFTIMING / (Outputs->BurstLength * 2); // BurstLength is in tCK and not in no. of DQS bursts
  Channel = 0; //Initialize to avoid compile errors
  Byte = 0;
  AvgCodePiOffset = 0;

  MrcCall->MrcSetMem ((UINT8*) CccPinsUsedSaved, sizeof (CccPinsUsedSaved), 0);

  CpgcStartTest.Data = CpgcStopTest.Data = 0;
  CpgcStartTest.Bits.START_TEST = 1;
  CpgcStopTest.Bits.STOP_TEST = 1;

  // Read VccDllTarget and VccDllControl before modifying them
  VccDllTargetOrig = MrcReadCR (MrcData, DDRPHY_COMP_CR_VCCDLLTARGET_REG);
  // Read the first populated channel
  FirstController = (MrcControllerExist (MrcData, cCONTROLLER0)) ? 0 : 1;
  FirstChannel = Outputs->Controller[FirstController].FirstPopCh;

  // Save the following fields
  // Assume that the following fields will be same across all channels
  // Following fields are from datacontrol regs, channel is equivalent to controller for offset calc.
  // But these enums go through MrcTranslateSystemToIp translates channel and strobe to corresponding memtech
  MrcGetSetChStrb (MrcData, FirstController, FirstChannel, 0, GsmIocInternalClocksOn, ReadFromCache, &InternalClocks);
  MrcGetSetChStrb (MrcData, FirstController, FirstChannel, 0, GsmIocRxPiPwrDnDis, ReadFromCache, &RxPiOn);
  MrcGetSetChStrb (MrcData, FirstController, FirstChannel, 0, GsmIocTxPiPwrDnDis, ReadFromCache, &TxPiOn);
  MrcGetSetChStrb (MrcData, FirstController, FirstChannel, 0, GsmIocRxDisable, ReadFromCache, &RxDisable);

  // The following enums do not go through Translate function
  if (Outputs->Lpddr) {
    if (FirstController == 1) {
      FirstChannel = FirstChannel + MAX_CHANNEL; // FirstPopCh only returns ch0,1,2,3 so if only Controller 1 is populated, FirstPopCh should be +4
    }
  }
  FirstByte = (MAX_BYTE_IN_LP_CHANNEL * FirstChannel);
  MrcGetSetChStrb (MrcData, MRC_IGNORE_ARG, MRC_IGNORE_ARG, FirstByte, GsmIocWlWakeCyc, ReadFromCache, &WlWakeCyc); // Offset calc only needs Byte
  MrcGetSetChStrb (MrcData, MRC_IGNORE_ARG, MRC_IGNORE_ARG, FirstByte, GsmIocWlSleepCyclesLp, ReadFromCache, &WlSleepCyc);

  for (Code = 0; Code < ARRAY_COUNT (TargetCode); Code++) {
    ParStart = ParStartEnd[0][Code];
    ParEnd = ParStartEnd[1][Code];
    if (TargetCode[Code] == GsmIocFFCodeReadOffset) {
      Status = MrcResetSequence (MrcData);
      if (Status != mrcSuccess) {
        MRC_DEBUG_MSG (&Outputs->Debug, MSG_LEVEL_ERROR, "\n Dram Reset fails at FFCodeReadOffset training \n");
        break;
      }
    }
    for (Partition = ParStart; Partition <= ParEnd; Partition++) {
      if ((Partition == DllDdrData8) && (Inputs->UlxUlt)) { //Skip ECC byte for ULXULT
        continue;
      }
      MrcDllPartitionDecode (MrcData, Partition, ChannelPerRail, BytePerRail, &TargetRail);
      switch (TargetCode[Code]) {  // Setup DDRIO before starting compensation
        case GsmIocFFCodeIdleOffset:
          GetSetVal = 0;
          MrcGetSetChStrb (MrcData, MAX_CONTROLLER, MAX_CHANNEL, MAX_SDRAM_IN_DIMM, GsmIocInternalClocksOn, WriteCached, &GetSetVal);
          // MrcGetSetNoScope (MrcData, GsmIocScramLpMode, WriteToCache | PrintValue, &GetSetDis);
          MrcWeaklockEnDis (MrcData, MRC_DISABLE);
          FreezeVccDllVsxHi (MrcData, VCCDLL1, TRUE);
          FreezeVccDllVsxHi (MrcData, VCCDLL2, TRUE);
          if (DtHalo) {
            FreezeVccDllVsxHi (MrcData, VCCDLL3, TRUE);
          }
          GetSetVal = 1;
          MrcGetSetChStrb (MrcData, MAX_CONTROLLER, MAX_CHANNEL, MAX_SDRAM_IN_DIMM, GsmIocInternalClocksOn, WriteCached, &GetSetVal);
          TargetRail = VCCDLL1;
          break;

        case GsmIocFFCodeWeakOffset:
          // Internal clocks are already off after Idle
          // TargetRail is already VCCDLL1
          FreezeVccDllVsxHi (MrcData, VCCDLL1, TRUE);
          FreezeVccDllVsxHi (MrcData, VCCDLL2, TRUE);
          if (DtHalo) {
            FreezeVccDllVsxHi (MrcData, VCCDLL3, TRUE);
          }
          // MrcGetSetNoScope (MrcData, GsmIocScramLpMode, WriteToCache | PrintValue, &GetSetDis);
          GetSetVal = 3;
          MrcGetSetChStrb (MrcData, MRC_IGNORE_ARG, MRC_IGNORE_ARG, MAX_SDRAM_IN_DIMM, GsmIocWlWakeCyc, WriteCached, &GetSetVal);
          GetSetVal = 5;
          MrcGetSetChStrb (MrcData, MRC_IGNORE_ARG, MRC_IGNORE_ARG, MAX_SDRAM_IN_DIMM, GsmIocWlSleepCyclesLp, WriteCached, &GetSetVal);
          MrcWeaklockEnDis (MrcData, MRC_ENABLE);
          GetSetVal = 1;
          MrcGetSetChStrb (MrcData, MAX_CONTROLLER, MAX_CHANNEL, MAX_SDRAM_IN_DIMM, GsmIocInternalClocksOn, WriteCached, &GetSetVal);
          break;

        case GsmIocFFCodePiOffset:
          MrcWeaklockEnDis (MrcData, MRC_DISABLE);

          GetSetVal = 1;
          for (VccDllRail = 0; VccDllRail < VCC_DLL_BYTES; VccDllRail++) {
            GetSet (MrcData, MRC_IGNORE_ARG, MRC_IGNORE_ARG, ChannelPerRail[VccDllRail], MRC_IGNORE_ARG, MRC_IGNORE_ARG, BytePerRail[VccDllRail], MRC_IGNORE_ARG, MRC_IGNORE_ARG, MRC_IGNORE_ARG, GsmIocInternalClocksOn, WriteCached, &GetSetVal);
          }
          FreezeVccDllVsxHi (MrcData, TargetRail, TRUE);

          for (VccDllRail = 0; VccDllRail < VCC_DLL_BYTES; VccDllRail++) {
            GetSetVal = 1;
            GetSet (MrcData, MRC_IGNORE_ARG, MRC_IGNORE_ARG, ChannelPerRail[VccDllRail], MRC_IGNORE_ARG, MRC_IGNORE_ARG, BytePerRail[VccDllRail], MRC_IGNORE_ARG, MRC_IGNORE_ARG, MRC_IGNORE_ARG, GsmIocRxPiPwrDnDis, WriteCached, &GetSetVal);
          }
          break;

        case GsmIocFFCodeCCCDistOffset:
          MrcWeaklockEnDis (MrcData, MRC_DISABLE);
          GetSetVal = 1;
          for (VccDllRail = 0; VccDllRail < VCC_DLL_BYTES; VccDllRail++) {
            GetSet (MrcData, MRC_IGNORE_ARG, MRC_IGNORE_ARG, ChannelPerRail[VccDllRail], MRC_IGNORE_ARG, MRC_IGNORE_ARG, BytePerRail[VccDllRail], MRC_IGNORE_ARG, MRC_IGNORE_ARG, MRC_IGNORE_ARG, GsmIocInternalClocksOn, WriteCached, &GetSetVal);
          }
          FreezeVccDllVsxHi (MrcData, TargetRail, TRUE);
          for (VccDllRail = 0; VccDllRail < VCC_DLL_BYTES; VccDllRail++) {
            Offset = OFFSET_CALC_CH (CH0CCC_CR_DDRCRPINSUSED_REG, CH1CCC_CR_DDRCRPINSUSED_REG, ChannelPerRail[VccDllRail]);
            CccPinsUsedSaved[ChannelPerRail[VccDllRail]] = MrcReadCR (MrcData, Offset);
            Data = CccPinsUsedSaved[ChannelPerRail[VccDllRail]];
            if (Inputs->UlxUlt) {
              CccPinsUsed = (CH0CCC_CR_DDRCRPINSUSED_STRUCT *) &Data;
              CccPinsUsed->Bits.PiEn = 0x1F;
              CccPinsUsed->Bits.TxEn = 0;
            } else {
              CccPinsUsedP0 = (CH0CCC_CR_DDRCRPINSUSED_P0_STRUCT *) &Data;
              CccPinsUsedP0->P0Bits.PiEn = 0x1F;
              CccPinsUsedP0->P0Bits.TxEn = 0;
            }
            MrcWriteCR (MrcData, Offset, Data);
          }
          break;

        case GsmIocFFCodePBDOffset:
          // PiEn and TxEn are set as per CCCDistOffset, they will be set to init values after this step
          GetSetVal = 1;
          for (VccDllRail = 0; VccDllRail < VCC_DLL_BYTES; VccDllRail++) {
            GetSet (MrcData, MRC_IGNORE_ARG, MRC_IGNORE_ARG, ChannelPerRail[VccDllRail], MRC_IGNORE_ARG, MRC_IGNORE_ARG, BytePerRail[VccDllRail], MRC_IGNORE_ARG, MRC_IGNORE_ARG, MRC_IGNORE_ARG, GsmIocInternalClocksOn, WriteCached, &GetSetVal);
          }
          FreezeVccDllVsxHi (MrcData, TargetRail, TRUE);
          for (VccDllRail = 0; VccDllRail < VCC_DLL_BYTES; VccDllRail++) {
            Offset = OFFSET_CALC_CH (CH0CCC_CR_DDRCRPINSUSED_REG, CH1CCC_CR_DDRCRPINSUSED_REG, ChannelPerRail[VccDllRail]);
            // Set PiEn and TxEn back to Init
            MrcWriteCR (MrcData, Offset, CccPinsUsedSaved[ChannelPerRail[VccDllRail]]);
          }
          break;

        case GsmIocFFCodeWriteOffset:
          GetSetVal = 1;
          for (VccDllRail = 0; VccDllRail < VCC_DLL_BYTES; VccDllRail++) {
            GetSet (MrcData, MRC_IGNORE_ARG, MRC_IGNORE_ARG, ChannelPerRail[VccDllRail], MRC_IGNORE_ARG, MRC_IGNORE_ARG, BytePerRail[VccDllRail], MRC_IGNORE_ARG, MRC_IGNORE_ARG, MRC_IGNORE_ARG, GsmIocInternalClocksOn, WriteCached, &GetSetVal);
          }
          FreezeVccDllVsxHi (MrcData, TargetRail, TRUE);
          // Because order needs to be followed for config therefore TxPiOn cant be set above
          GetSetVal = 1;
          for (VccDllRail = 0; VccDllRail < VCC_DLL_BYTES; VccDllRail++) {
            GetSet (MrcData, MRC_IGNORE_ARG, MRC_IGNORE_ARG, ChannelPerRail[VccDllRail], MRC_IGNORE_ARG, MRC_IGNORE_ARG, BytePerRail[VccDllRail], MRC_IGNORE_ARG, MRC_IGNORE_ARG, MRC_IGNORE_ARG, GsmIocTxPiPwrDnDis, WriteCached, &GetSetVal);
          }
          break;

        case GsmIocFFCodeReadOffset:
          GetSetVal = 1;
          MrcGetSetChStrb (MrcData, MAX_CONTROLLER, MAX_CHANNEL, MAX_SDRAM_IN_DIMM, GsmIocRxDisable, WriteCached, &GetSetVal);
          for (VccDllRail = 0; VccDllRail < VCC_DLL_BYTES; VccDllRail++) {
            GetSetVal = 1;
            GetSet (MrcData, MRC_IGNORE_ARG, MRC_IGNORE_ARG, ChannelPerRail[VccDllRail], MRC_IGNORE_ARG, MRC_IGNORE_ARG, BytePerRail[VccDllRail], MRC_IGNORE_ARG, MRC_IGNORE_ARG, MRC_IGNORE_ARG, GsmIocInternalClocksOn, WriteCached, &GetSetVal);
            GetSet (MrcData, MRC_IGNORE_ARG, MRC_IGNORE_ARG, ChannelPerRail[VccDllRail], MRC_IGNORE_ARG, MRC_IGNORE_ARG, BytePerRail[VccDllRail], MRC_IGNORE_ARG, MRC_IGNORE_ARG, MRC_IGNORE_ARG, GsmIocRxPiPwrDnDis, WriteCached, &GetSetVal);
            GetSetVal = 0;
            GetSet (MrcData, MRC_IGNORE_ARG, MRC_IGNORE_ARG, ChannelPerRail[VccDllRail], MRC_IGNORE_ARG, MRC_IGNORE_ARG, BytePerRail[VccDllRail], MRC_IGNORE_ARG, MRC_IGNORE_ARG, MRC_IGNORE_ARG, GsmIocRxDisable, WriteCached, &GetSetVal);
          }
          FreezeVccDllVsxHi (MrcData, TargetRail, TRUE);
          SetupIOTestBasicVA (MrcData, Outputs->McChBitMask, 1, NSOE, 0, 0, 9, PatRdEndless, 0, NumCL);
          Cpgc20ControlRegWrite (MrcData, Outputs->McChBitMask, CpgcStartTest);
          break;
      } // End of Switch

      LinearSearch (MrcData, TargetCode[Code], Partition, (UINT8) TargetRail, &VccDllTargetOrig);

      // Turn Off Internal clock for that partition for CodePi : CodeRead
      GetSetVal = 0;
      if (!(TargetCode[Code] == GsmIocFFCodeIdleOffset) && !(TargetCode[Code] == GsmIocFFCodeWeakOffset)) {
        for (VccDllRail = 0; VccDllRail < VCC_DLL_BYTES; VccDllRail++) {
          GetSet (MrcData, MRC_IGNORE_ARG, MRC_IGNORE_ARG, ChannelPerRail[VccDllRail], MRC_IGNORE_ARG, MRC_IGNORE_ARG, BytePerRail[VccDllRail], MRC_IGNORE_ARG, MRC_IGNORE_ARG, MRC_IGNORE_ARG, GsmIocInternalClocksOn, WriteCached, &GetSetVal);
        }
      } else { // After Idle and Weak set all InternalClocks Off
        MrcGetSetChStrb (MrcData, MAX_CONTROLLER, MAX_CHANNEL, MAX_SDRAM_IN_DIMM, GsmIocInternalClocksOn, WriteCached, &GetSetVal);
      }
      // Restore per partition
      switch (TargetCode[Code]) {
        case GsmIocFFCodePiOffset:
          GetSetVal = 0;
          GetSet (MrcData, MRC_IGNORE_ARG, MRC_IGNORE_ARG, Channel, MRC_IGNORE_ARG, MRC_IGNORE_ARG, Byte, MRC_IGNORE_ARG, MRC_IGNORE_ARG, MRC_IGNORE_ARG, GsmIocRxPiPwrDnDis, WriteCached, &GetSetVal);
          break;
        case GsmIocFFCodePBDOffset:
          GetSetVal = 0;
          for (Index = 0; Index < MRC_NUM_CCC_INSTANCES; Index++) {
            Offset = OFFSET_CALC_CH (CH0CCC_CR_DDRCRPINSUSED_REG, CH1CCC_CR_DDRCRPINSUSED_REG, Index);
            Data = CccPinsUsedSaved[Index];
            if (Inputs->UlxUlt) {
              CccPinsUsed = (CH0CCC_CR_DDRCRPINSUSED_STRUCT *) &Data;
              CccPinsUsed->Bits.PiEn = 0;
            } else {
              CccPinsUsedP0 = (CH0CCC_CR_DDRCRPINSUSED_P0_STRUCT *) &Data;
              CccPinsUsedP0->P0Bits.PiEn = 0;
            }
            MrcWriteCR (MrcData, Offset, Data);
          }
          break;
        case GsmIocFFCodeWriteOffset:
          GetSetVal = 0;
          for (VccDllRail = 0; VccDllRail < VCC_DLL_BYTES; VccDllRail++) {
            GetSet (MrcData, MRC_IGNORE_ARG, MRC_IGNORE_ARG, ChannelPerRail[VccDllRail], MRC_IGNORE_ARG, MRC_IGNORE_ARG, BytePerRail[VccDllRail], MRC_IGNORE_ARG, MRC_IGNORE_ARG, MRC_IGNORE_ARG, GsmIocTxOn, WriteCached, &GetSetVal);
          }
          break;
        case GsmIocFFCodeReadOffset:
          // stop CPGC
          Cpgc20ControlRegWrite (MrcData, Outputs->McChBitMask, CpgcStopTest);
          GetSetVal = 0;
          for (VccDllRail = 0; VccDllRail < VCC_DLL_BYTES; VccDllRail++) {
            GetSet (MrcData, MRC_IGNORE_ARG, MRC_IGNORE_ARG, ChannelPerRail[VccDllRail], MRC_IGNORE_ARG, MRC_IGNORE_ARG, BytePerRail[VccDllRail], MRC_IGNORE_ARG, MRC_IGNORE_ARG, MRC_IGNORE_ARG, GsmIocRxPiPwrDnDis, WriteCached, &GetSetVal);
          }
          break;
      }
    } // Partition

    // Average all FFCodePiOffset from Data partitions and program the CCC partition with that value
    if (TargetCode[Code] == GsmIocFFCodePiOffset) {
      for (Partition = 0; Partition < MAX_VCCDLL_DATA_PAR; Partition++) {
        if ((Partition == DllDdrData8) && (Inputs->UlxUlt)) { //Skip ECC byte for ULXULT
          continue;
        }
        // Strobe is passed in as Partition, check VccDllDataCCCCompOffset () to see offset calculation
        GetSet (MrcData, MRC_IGNORE_ARG, MRC_IGNORE_ARG, MRC_IGNORE_ARG, MRC_IGNORE_ARG, MRC_IGNORE_ARG, (UINT32) Partition, MRC_IGNORE_ARG, MRC_IGNORE_ARG, MRC_IGNORE_ARG, TargetCode[Code], ReadFromCache, &GetSetVal);
        AvgCodePiOffset += (INT32) GetSetVal;
      }
      AvgCodePiOffset = (AvgCodePiOffset / MAX_VCCDLL_DATA_PAR);
      GetSetVal = (INT64) AvgCodePiOffset;
      for (Partition = MAX_VCCDLL_DATA_PAR; Partition < MAX_VCCDLL_PAR; Partition++) {
        if ((Partition == DllDdrData8) && (Inputs->UlxUlt)) { //Skip ECC byte for ULXULT
          continue;
        }
        // Channel is passed in as Partition, check VccDllDataCCCCompOffset () to see offset calculation. CCC VccDllPartitions are programmed
        GetSet (MrcData, MRC_IGNORE_ARG, MRC_IGNORE_ARG, (UINT32) Partition, MRC_IGNORE_ARG, MRC_IGNORE_ARG, MRC_IGNORE_ARG, MRC_IGNORE_ARG, MRC_IGNORE_ARG, MRC_IGNORE_ARG, TargetCode[Code], WriteCached, &GetSetVal);
      }
    }

    // Restore at the end
    MrcGetSetChStrb (MrcData, MAX_CONTROLLER, MAX_CHANNEL, MAX_SDRAM_IN_DIMM, GsmIocInternalClocksOn, WriteToCache, &InternalClocks);
    MrcGetSetChStrb (MrcData, MAX_CONTROLLER, MAX_CHANNEL, MAX_SDRAM_IN_DIMM, GsmIocTxPiPwrDnDis, WriteToCache, &TxPiOn);
    MrcGetSetChStrb (MrcData, MAX_CONTROLLER, MAX_CHANNEL, MAX_SDRAM_IN_DIMM, GsmIocRxDisable, WriteToCache, &RxDisable);
    MrcGetSetChStrb (MrcData, MAX_CONTROLLER, MAX_CHANNEL, MAX_SDRAM_IN_DIMM, GsmIocRxPiPwrDnDis, WriteToCache, &RxPiOn);
    for (Byte = 0; Byte < (MAX_SDRAM_IN_DIMM - 1); Byte++) {
      MrcGetSetChStrb (MrcData, MRC_IGNORE_ARG, MRC_IGNORE_ARG, Byte, GsmIocWlWakeCyc, WriteToCache, &WlWakeCyc);
      MrcGetSetChStrb (MrcData, MRC_IGNORE_ARG, MRC_IGNORE_ARG, Byte, GsmIocWlSleepCyclesLp, WriteToCache, &WlSleepCyc);
    }
    MrcFlushRegisterCachedData (MrcData);

    for (Index = 0; Index < MRC_NUM_CCC_INSTANCES; Index++) {
      Offset = OFFSET_CALC_CH (CH0CCC_CR_DDRCRPINSUSED_REG, CH1CCC_CR_DDRCRPINSUSED_REG, Index);
      MrcWriteCR (MrcData, Offset, CccPinsUsedSaved[Index]);
    }
    //CleanUp
    // Unfreeze VccDll, assume OpenLoop = 0 in original as its not set unless we are training VccDll
    for (TargetRail = VCCDLL1; TargetRail < GetMaxVccDllRail (MrcData); TargetRail++) {
      FreezeVccDllVsxHi (MrcData, TargetRail, FALSE);
    }
    // Restore VccDllTarget
    MrcWriteCR (MrcData, DDRPHY_COMP_CR_VCCDLLTARGET_REG, VccDllTargetOrig);

  } // Code

  return Status;
}

/**
  This function runs the innerloop for kickbacknoise linear search which detects X for every N.
  N - Number of capacitive noise cancelling legs that is the denominator for the equation for each of the codes run in this training
  X - CompVTargetmax - CompVTargetmin

  @param[in] MrcData - Include all MRC global data.
  @param[in] VccDllTargetOrig - VccDllTarget setting as per Phyinit.
  @param[in] TargetRail - VccDll North or South voltage rail.
  @param[in, out] X = CompVTargetmax - CompVTargetmin

  @retval MrcStatus - mrcSuccess if successful, else an error status
**/
MrcStatus
RunInnerTestKickBackNoise (
  IN MrcParameters *const MrcData,
  IN UINT32        *const VccDllTargetOrig,
  IN UINT32               TargetRail,
  IN OUT UINT32           *X
  )
{
  MrcOutput          *Outputs;
  MrcStatus          Status;
  MrcDebug           *Debug;
  UINT32 WaitTime;
  UINT32 CompVTarget;
  INT32  Step;
  INT32  Limit;
  UINT32 CompVTargetMax;
  UINT32 CompVTargetMin;
  BOOLEAN Done;
  BOOLEAN DoneMax;
  BOOLEAN DoneMin;
  DDRPHY_COMP_CR_VCCDLLTARGET_STRUCT  VccDllTarget;

  Outputs  = &MrcData->Outputs;
  Debug    = &Outputs->Debug;
  Step = 1;
  CompVTargetMax = 0;
  CompVTargetMin = 0;
  Limit = COMPVTARGET_SWINGV / COMPVTARGET_STEPV;
  Done = FALSE;
  DoneMax = FALSE;
  DoneMin = FALSE;
  WaitTime = (MrcData->Outputs.Qclkps * 4) * MRC_TIMER_1NS;
  VccDllTarget.Data = *VccDllTargetOrig;
  CompVTarget = VccDllTarget.Bits.CompVTarget; // Center point
  Status = mrcSuccess;
  do {
    VccDllTarget.Bits.ForceCompAmpOn = 1;
    VccDllTarget.Bits.SelectVccDllRail   = TargetRail;
    VccDllTarget.Bits.CompVTarget = CompVTarget + Step; // Center + 1 or Center -1 depending on the direction of the search
    VccDllTarget.Bits.RstCount = 1;
    MrcWriteCR (MrcData, DDRPHY_COMP_CR_VCCDLLTARGET_REG, VccDllTarget.Data);
    MrcWait (MrcData, WaitTime); // Wait 4000Qclks, Allow DAC to settle before starting test.
    VccDllTarget.Data = MrcReadCR (MrcData, DDRPHY_COMP_CR_VCCDLLTARGET_REG);
    if (VccDllTarget.Bits.CountMSB  == 0) {
      CompVTargetMax = VccDllTarget.Bits.CompVTarget;
      DoneMax = TRUE;
      Step = -1;
    } else if (VccDllTarget.Bits.CountMSB  == 0xF) {
      CompVTargetMin = VccDllTarget.Bits.CompVTarget;
      DoneMin = TRUE;
    } else {
      if (!DoneMax && (Step < Limit)) {
        Step += 1;
      } else if (!DoneMin && (ABS (Step) < Limit)) {
        Step -= 1;
      } else {
        if (!DoneMax && (Step == Limit)) {
          MRC_DEBUG_MSG (Debug, MSG_LEVEL_ERROR, "\n Couldn't find CompVTargetMax, try increasing COMPVTARGET_SWINGV from 25mV to 30mV \n");
          Status = mrcFail;
          break;
        } else if (!DoneMin && (ABS (Step) == Limit)) {
          MRC_DEBUG_MSG (Debug, MSG_LEVEL_ERROR, "\n Couldn't find CompVTargetMin, try increasing COMPVTARGET_SWINGV from 25mV to 30mV \n");
          Status = mrcFail;
          break;
        }
      }
    }
    Done = ((DoneMax == TRUE) && (DoneMin == TRUE));

  } while (!Done);
  if (Status == mrcSuccess) {
    CompVTargetMax -= CompVTargetMin;
    *X = CompVTargetMax; // X must not be -ve
  }
  MRC_DEBUG_MSG (Debug, MSG_LEVEL_NOTE, "CompVTargetMax = %d CompVTargetMin = %d, X = %d\n", CompVTargetMax, CompVTargetMin, X);

  return Status;
}
/**
  This function is used to perform linear search for VccDll Kickback noise correction

  @param[in]     MrcData          - Include all MRC global data.
  @param[in]     Code             - Coupling cap register fields.
  @param[in]     Partition        - VccDll Partition.
  @param[in]     TargetRail       - VccDll North or South voltage rail.
  @param[in]     VccDllTargetOrig - VccDllTarget setting as per Phyinit.
  @param[in,out] NIdle            - Remember BestN for CodeIdle.
  @param[in,out] NWrite           - Remember BestN for CodeWrite.
  @param[in,out] NRead            - Remember BestN for CodeRead.

  @retval Status - mrcSuccess if successful, else an error status
**/
MrcStatus
LinearSearchKickBackNoise (
  IN MrcParameters *const MrcData,
  IN UINT32               Code,
  IN UINT8                Partition,
  IN UINT32               TargetRail,
  IN UINT32        *const VccDllTargetOrig,
  IN OUT UINT8            *NIdle,
  IN OUT UINT8            *NWrite,
  IN OUT UINT8            *NRead
) {
  MrcStatus  Status;
  INT64    GetSetVal;
  UINT32   NewCode;
  UINT32   OrigCode;
  UINT32   Channel;
  UINT32   Byte;
  UINT32    X;
  UINT32    Xcompare;
  UINT8    BestN;
  UINT8    N;
  BOOLEAN  PiOrPBD;

  Xcompare = 126;
  X = 0; // initialize
  Status = mrcSuccess;
  BestN = 1; // dont set to 0 , to avoid any divide by 0 errors
  PiOrPBD = ((Code == GsmIocCapCancelCodePi) || (Code == GsmIocCapCancelCodePBD));
  // decode Channel and Byte from Partition. Channel is only used to program CCC partitions, Byte is used for data par
  Channel = (Partition > (MAX_VCCDLL_DATA_PAR - 1)) ? (Partition - MAX_VCCDLL_DATA_PAR) : MRC_IGNORE_ARG;
  Byte = (Partition > (MAX_VCCDLL_DATA_PAR - 1)) ? MRC_IGNORE_ARG : Partition;

  // Program the register field
  GetSet (MrcData, MRC_IGNORE_ARG, MRC_IGNORE_ARG, Channel, MRC_IGNORE_ARG, MRC_IGNORE_ARG, Byte, MRC_IGNORE_ARG, MRC_IGNORE_ARG, MRC_IGNORE_ARG, Code, ReadFromCache, &GetSetVal);
  OrigCode = (UINT32) GetSetVal;

  if (!PiOrPBD) {
    for (N = (NDEFAULT - 4); N < (NDEFAULT + 5); N++) {
      NewCode = OrigCode * NDEFAULT;
      NewCode /= N;
      GetSetVal = NewCode;
      GetSet (MrcData, MRC_IGNORE_ARG, MRC_IGNORE_ARG, Channel, MRC_IGNORE_ARG, MRC_IGNORE_ARG, Byte, MRC_IGNORE_ARG, MRC_IGNORE_ARG, MRC_IGNORE_ARG, Code, WriteCached, &GetSetVal);
      Status = RunInnerTestKickBackNoise (MrcData, VccDllTargetOrig, TargetRail, &X);
      if (Status != mrcSuccess) {
        return Status;
      }
      if (X < Xcompare) {
        if (Status == mrcSuccess) {
          Xcompare = X;
          BestN = N;
        } else {
          break;
        }
      }
    }
  }

    switch (Code) {
      case GsmIocCapCancelCodeIdle:
        *NIdle = BestN;
        break;

      case GsmIocCapCancelCodeWrite:
        *NWrite = BestN;
        break;

      case GsmIocCapCancelCodeRead:
        *NRead = BestN;
        break;

      case GsmIocCapCancelCodePi:
      case GsmIocCapCancelCodePBD:
        BestN = *NIdle;
        BestN += *NWrite;
        BestN += *NRead;
        BestN /= 3;
        NewCode = OrigCode * NDEFAULT;
        NewCode /= BestN;
        GetSetVal = NewCode;
        GetSet (MrcData, MRC_IGNORE_ARG, MRC_IGNORE_ARG, Channel, MRC_IGNORE_ARG, MRC_IGNORE_ARG, Byte, MRC_IGNORE_ARG, MRC_IGNORE_ARG, MRC_IGNORE_ARG, Code, WriteCached, &GetSetVal);
        break;

      default:
        break;
    }

  return Status;
}

/**
  VccDll KickBackNoise Correction
  This is the main function for this training and its organized as follows :
    Setup
    Call linear search and update the reg field
    Restore per partition
    Restore at the end

  @param[in] MrcData - Include all MRC global data.

  @retval MrcStatus - mrcFail if Inputs->ExitOnFailure is set and we cannot reach 4 GHz; otherwise mrcSuccess.
**/
MrcStatus
MrcKickBackNoiseCorrection (
  IN MrcParameters *const MrcData
  )
{
  MrcInput          *Inputs;
  MrcOutput         *Outputs;
  MrcStatus         Status;
  INT64             GetSetEn;
  INT64             GetSetDis;
  INT64             InternalClocks;
  INT64             TxPiOn;
  INT64             RxPiOn;
  INT64             RxDisable;
  UINT32            Code;
  UINT32            ChannelPerRail[VCC_DLL_BYTES];
  UINT32            BytePerRail[VCC_DLL_BYTES];
  UINT32            TargetRail;
  UINT32            VccDllTargetOrig;
  UINT32            FirstController;
  UINT32            FirstChannel;
  UINT8             NIdle;
  UINT8             NWrite;
  UINT8             NRead;
  UINT8             Partition;
  UINT8             ParStart;
  UINT8             ParEnd;
  UINT8             VccDllRail;
  UINT8             McChBitMask;
  UINT8             Wait;
  UINT8             NumCL;
  MC0_REQ0_CR_CPGC_SEQ_CTL_STRUCT  CpgcStartTest;
  MC0_REQ0_CR_CPGC_SEQ_CTL_STRUCT  CpgcStopTest;

  static UINT32 const TargetCode [] = {GsmIocCapCancelCodeIdle, GsmIocCapCancelCodeWrite, GsmIocCapCancelCodeRead, GsmIocCapCancelCodePi, GsmIocCapCancelCodePBD};

  Inputs      = &MrcData->Inputs;
  Outputs     = &MrcData->Outputs;
  McChBitMask = Outputs->McChBitMask;
  CpgcStartTest.Data = CpgcStopTest.Data = 0;
  CpgcStartTest.Bits.START_TEST = 1;
  CpgcStopTest.Bits.STOP_TEST = 1;
  Wait = 64 / 2;
  NumCL       = 64 / (Outputs->BurstLength * 2);
  GetSetEn  = 1;
  GetSetDis = 0;
  NIdle  = 0;
  NWrite = 0;
  NRead  = 0;
  Status = mrcSuccess;

  // Read VccDllTarget and VccDllControl before modifying them
  VccDllTargetOrig = MrcReadCR (MrcData, DDRPHY_COMP_CR_VCCDLLTARGET_REG);
  // Read the first populated channel
  FirstController = (MrcControllerExist (MrcData, cCONTROLLER0)) ? 0 : 1;
  FirstChannel = Outputs->Controller[FirstController].FirstPopCh;

  // Save the following fields
  // Assume that the following fields will be same across all channels
  // Following fields are from datacontrol regs, channel is equivalent to controller for offset calc.
  // But these enums go through MrcTranslateSystemToIp translates channel and strobe to corresponding memtech
  MrcGetSetChStrb (MrcData, FirstController, FirstChannel, 0, GsmIocInternalClocksOn, ReadFromCache, &InternalClocks);
  MrcGetSetChStrb (MrcData, FirstController, FirstChannel, 0, GsmIocRxPiPwrDnDis, ReadFromCache, &RxPiOn);
  MrcGetSetChStrb (MrcData, FirstController, FirstChannel, 0, GsmIocTxPiPwrDnDis, ReadFromCache, &TxPiOn);
  MrcGetSetChStrb (MrcData, FirstController, FirstChannel, 0, GsmIocRxDisable, ReadFromCache, &RxDisable);

  // @todo Add EccByte DLL Partition for TGL H DDR4
  static UINT8 const ParStartEnd[2][MAX_KICKBACK_CODES] = {
  //Code  Idle  Write   Read  Pi  Pbd
    {      0,     0,     0,   0,    8},
    {      11,    7,     7,   11,   11}};

  for (Code = 0; Code < ARRAY_COUNT (TargetCode); Code++) {
    ParStart = ParStartEnd[0][Code];
    ParEnd = ParStartEnd[1][Code];
    if (TargetCode[Code] == GsmIocCapCancelCodeRead) {
      Status = MrcResetSequence (MrcData);
      if (Status != mrcSuccess) {
        MRC_DEBUG_MSG (&Outputs->Debug, MSG_LEVEL_ERROR, "\n Dram Reset fails at FFCodeReadOffset training \n");
        break;
      }
    }
    for (Partition = ParStart; Partition <= ParEnd; Partition++) {
      if ((Partition == DllDdrData8) && (Inputs->UlxUlt)) { //Skip ECC byte for ULXULT
        continue;
      }
      MrcDllPartitionDecode (MrcData, Partition, ChannelPerRail, BytePerRail, &TargetRail);
      switch (TargetCode[Code]) {  // Setup DDRIO before starting compensation
        case GsmIocCapCancelCodeIdle:
          MrcGetSetChStrb (MrcData, MAX_CONTROLLER, MAX_CHANNEL, MAX_SDRAM_IN_DIMM, GsmIocInternalClocksOn, WriteCached, &GetSetDis);
          MrcWeaklockEnDis (MrcData, MRC_ENABLE);
          FreezeVccDllVsxHi (MrcData, TargetRail, TRUE);
          for (VccDllRail = 0; VccDllRail < VCC_DLL_BYTES; VccDllRail++) {
            GetSet (MrcData, MRC_IGNORE_ARG, MRC_IGNORE_ARG, ChannelPerRail[VccDllRail], MRC_IGNORE_ARG, MRC_IGNORE_ARG, BytePerRail[VccDllRail], MRC_IGNORE_ARG, MRC_IGNORE_ARG, MRC_IGNORE_ARG, GsmIocInternalClocksOn, WriteCached, &GetSetEn);
          }
          // Turn off Internal Clocks of TargetRail at the partition restore of this Code
          break;

        case GsmIocCapCancelCodeWrite:
          FreezeVccDllVsxHi (MrcData, TargetRail, TRUE);
          for (VccDllRail = 0; VccDllRail < VCC_DLL_BYTES; VccDllRail++) {
            GetSet (MrcData, MRC_IGNORE_ARG, MRC_IGNORE_ARG, ChannelPerRail[VccDllRail], MRC_IGNORE_ARG, MRC_IGNORE_ARG, BytePerRail[VccDllRail], MRC_IGNORE_ARG, MRC_IGNORE_ARG, MRC_IGNORE_ARG, GsmIocInternalClocksOn, WriteCached, &GetSetEn);
            GetSet (MrcData, MRC_IGNORE_ARG, MRC_IGNORE_ARG, ChannelPerRail[VccDllRail], MRC_IGNORE_ARG, MRC_IGNORE_ARG, BytePerRail[VccDllRail], MRC_IGNORE_ARG, MRC_IGNORE_ARG, MRC_IGNORE_ARG, GsmIocTxPiPwrDnDis, WriteCached, &GetSetEn);
          }
          SetupIOTestBasicVA (MrcData, McChBitMask, 1, NSOE, 0, 0, 9, PatWrEndless, Wait, NumCL);
          Cpgc20ControlRegWrite (MrcData, McChBitMask, CpgcStartTest);
          break;

        case GsmIocCapCancelCodeRead:
          FreezeVccDllVsxHi (MrcData, TargetRail, TRUE);
          MrcGetSetChStrb (MrcData, MAX_CONTROLLER, MAX_CHANNEL, MAX_SDRAM_IN_DIMM, GsmIocRxDisable, WriteCached, &GetSetEn);
          for (VccDllRail = 0; VccDllRail < VCC_DLL_BYTES; VccDllRail++) {
            GetSet (MrcData, MRC_IGNORE_ARG, MRC_IGNORE_ARG, ChannelPerRail[VccDllRail], MRC_IGNORE_ARG, MRC_IGNORE_ARG, BytePerRail[VccDllRail], MRC_IGNORE_ARG, MRC_IGNORE_ARG, MRC_IGNORE_ARG, GsmIocInternalClocksOn, WriteCached, &GetSetEn);
            GetSet (MrcData, MRC_IGNORE_ARG, MRC_IGNORE_ARG, ChannelPerRail[VccDllRail], MRC_IGNORE_ARG, MRC_IGNORE_ARG, BytePerRail[VccDllRail], MRC_IGNORE_ARG, MRC_IGNORE_ARG, MRC_IGNORE_ARG, GsmIocRxPiPwrDnDis, WriteCached, &GetSetEn);
            GetSet (MrcData, MRC_IGNORE_ARG, MRC_IGNORE_ARG, ChannelPerRail[VccDllRail], MRC_IGNORE_ARG, MRC_IGNORE_ARG, BytePerRail[VccDllRail], MRC_IGNORE_ARG, MRC_IGNORE_ARG, MRC_IGNORE_ARG, GsmIocRxDisable, WriteCached, &GetSetDis);
          }
          SetupIOTestBasicVA (MrcData, McChBitMask, 1, NSOE, 0, 0, 9, PatRdEndless, Wait, NumCL);
          Cpgc20ControlRegWrite (MrcData, McChBitMask, CpgcStartTest);
          break;

        case GsmIocCapCancelCodePi:
          break;
        case GsmIocCapCancelCodePBD:
          break;

      } // End of Switch
      Status = LinearSearchKickBackNoise (MrcData, Code, Partition, TargetRail, &VccDllTargetOrig, &NIdle, &NWrite, &NRead);
      if ((Status != mrcSuccess) && MrcData->Inputs.ExitOnFailure) {
        MRC_DEBUG_MSG (&Outputs->Debug, MSG_LEVEL_ERROR, "\n Linear Search for BestN and smallest X values failed for Code  %d Partition %d \n", Code, Partition);
        return Status;
      }
      // Restore per partition
      switch (TargetCode[Code]) {
        case GsmIocCapCancelCodeIdle:
          for (VccDllRail = 0; VccDllRail < VCC_DLL_BYTES; VccDllRail++) {
            GetSet (MrcData, MRC_IGNORE_ARG, MRC_IGNORE_ARG, ChannelPerRail[VccDllRail], MRC_IGNORE_ARG, MRC_IGNORE_ARG, BytePerRail[VccDllRail], MRC_IGNORE_ARG, MRC_IGNORE_ARG, MRC_IGNORE_ARG, GsmIocInternalClocksOn, WriteCached, &GetSetDis);
          }
          MrcWeaklockEnDis (MrcData, MRC_DISABLE);
          break;

        case GsmIocCapCancelCodeWrite:
          for (VccDllRail = 0; VccDllRail < VCC_DLL_BYTES; VccDllRail++) {
            GetSet (MrcData, MRC_IGNORE_ARG, MRC_IGNORE_ARG, ChannelPerRail[VccDllRail], MRC_IGNORE_ARG, MRC_IGNORE_ARG, BytePerRail[VccDllRail], MRC_IGNORE_ARG, MRC_IGNORE_ARG, MRC_IGNORE_ARG, GsmIocInternalClocksOn, WriteCached, &GetSetDis);
            GetSet (MrcData, MRC_IGNORE_ARG, MRC_IGNORE_ARG, ChannelPerRail[VccDllRail], MRC_IGNORE_ARG, MRC_IGNORE_ARG, BytePerRail[VccDllRail], MRC_IGNORE_ARG, MRC_IGNORE_ARG, MRC_IGNORE_ARG, GsmIocTxPiPwrDnDis, WriteCached, &GetSetDis);
          }
          Cpgc20ControlRegWrite (MrcData, McChBitMask, CpgcStopTest);
          break;

        case GsmIocCapCancelCodeRead:
          for (VccDllRail = 0; VccDllRail < VCC_DLL_BYTES; VccDllRail++) {
            GetSet (MrcData, MRC_IGNORE_ARG, MRC_IGNORE_ARG, ChannelPerRail[VccDllRail], MRC_IGNORE_ARG, MRC_IGNORE_ARG, BytePerRail[VccDllRail], MRC_IGNORE_ARG, MRC_IGNORE_ARG, MRC_IGNORE_ARG, GsmIocInternalClocksOn, WriteCached, &GetSetDis);
            GetSet (MrcData, MRC_IGNORE_ARG, MRC_IGNORE_ARG, ChannelPerRail[VccDllRail], MRC_IGNORE_ARG, MRC_IGNORE_ARG, BytePerRail[VccDllRail], MRC_IGNORE_ARG, MRC_IGNORE_ARG, MRC_IGNORE_ARG, GsmIocRxPiPwrDnDis, WriteCached, &GetSetDis);
          }
          Cpgc20ControlRegWrite (MrcData, McChBitMask, CpgcStopTest);
          break;

        case GsmIocCapCancelCodePi:
        case GsmIocCapCancelCodePBD:
          break;
      } // end of switch
    } // End of Partition

    // Restore at the end of the code
    MrcGetSetChStrb (MrcData, MAX_CONTROLLER, MAX_CHANNEL, MAX_SDRAM_IN_DIMM, GsmIocInternalClocksOn, WriteToCache, &InternalClocks);
    MrcGetSetChStrb (MrcData, MAX_CONTROLLER, MAX_CHANNEL, MAX_SDRAM_IN_DIMM, GsmIocTxPiPwrDnDis, WriteToCache, &TxPiOn);
    MrcGetSetChStrb (MrcData, MAX_CONTROLLER, MAX_CHANNEL, MAX_SDRAM_IN_DIMM, GsmIocRxDisable, WriteToCache, &RxDisable);
    MrcGetSetChStrb (MrcData, MAX_CONTROLLER, MAX_CHANNEL, MAX_SDRAM_IN_DIMM, GsmIocRxPiPwrDnDis, WriteToCache, &RxPiOn);
    MrcFlushRegisterCachedData (MrcData);

    // CleanUp
    // Unfreeze VccDll, assume OpenLoop = 0 in original as its not set unless we are training VccDll
    FreezeVccDllVsxHi (MrcData, 0, FALSE);
    // Restore VccDllTarget
    MrcWriteCR (MrcData, DDRPHY_COMP_CR_VCCDLLTARGET_REG, VccDllTargetOrig);

  } // Code
  return Status;
}

/**
  This function to finds the transition of CountMsb from -ve to +ve or vice-versa

  @param[in]  MrcData - Pointer to global MRC data.
  @param[in]  Code - Feed Forward Offset Code.
  @param[in]  Partition - VccDll Partition
  @param[in]  TargetOrig - VsxHiFFTarget setting as per Phyinit.
  @param[in,out] MIdle - Best M value for CodeIdle.
  @param[in,out] MWrite - Best M value for CodeWrite.
  @param[in,out] MRead - Best M value for CodeRead.
**/
VOID
LinearSearchVsxHi (
  IN MrcParameters    *const MrcData,
  IN UINT32                  Code,
  IN UINT8                   Partition,
  IN UINT32           *const TargetOrig,
  IN OUT UINT32              *MIdle,
  IN OUT UINT32              *MWrite,
  IN OUT UINT32              *MRead
  )
{
  INT64  TargetDefault;
  INT64  TargetVal;
  INT64  Min;
  INT8   Step;
  UINT32 DefaultM;
  UINT32 M;
  UINT32 MTrain;
  UINT32 MScale;
  UINT32 Numerator;
  UINT32 WaitTime;
  UINT32 Channel;
  UINT32 Byte;
  INT64  Max;
  INT16  Result;
  UINT8  MSBValue;
  UINT8  PrevMSBValue;
  BOOLEAN Done;

  PrevMSBValue = 0;
  Done = FALSE;
  Step = 0;
  DefaultM = 0;
  WaitTime = (MrcData->Outputs.Qclkps * 4);

  // decode channel and byte from partition. This is only pertinent if we have multiple partitions
  Channel = (Partition > (MAX_VCCDLL_DATA_PAR - 1)) ? (Partition - MAX_VCCDLL_DATA_PAR) : MRC_IGNORE_ARG;
  Byte = (Partition > (MAX_VCCDLL_DATA_PAR - 1)) ? MRC_IGNORE_ARG : Partition;
  GetSet (MrcData, MRC_IGNORE_ARG, MRC_IGNORE_ARG, Channel, MRC_IGNORE_ARG, MRC_IGNORE_ARG, Byte, MRC_IGNORE_ARG, MRC_IGNORE_ARG, MRC_IGNORE_ARG, Code, ReadCached | PrintValue, &TargetDefault);
  TargetVal = TargetDefault;

  // Lookup Default M
  switch (Code) {
    case GsmIocVssHiFFCodeIdle :
      DefaultM = 18;
      break;
    case GsmIocVssHiFFCodeRead :
    case GsmIocVssHiFFCodeWrite :
      DefaultM = 20;
      break;
    case GsmIocVssHiFFCodePBD :
      DefaultM = 13;
      break;
    case GsmIocVssHiFFCodePi :
      DefaultM = 7;
      break;
    default:
      DefaultM = 1;
      break;
  }

  MRC_DEBUG_ASSERT (DefaultM != 1, &MrcData->Outputs.Debug, "Feed Forward Offset Code (%d) is invalid\n", Code);
  M = DefaultM;
  Numerator = (UINT32) TargetDefault * DefaultM;
  MrcGetSetLimits (MrcData, Code, &Min, &Max, NULL);
  if ((Code != GsmIocVssHiFFCodePBD) && (Code != GsmIocVssHiFFCodePi)) {
    do {
      GetSet (MrcData, MRC_IGNORE_ARG, MRC_IGNORE_ARG, Channel, MRC_IGNORE_ARG, MRC_IGNORE_ARG, Byte, MRC_IGNORE_ARG, MRC_IGNORE_ARG, MRC_IGNORE_ARG, Code, WriteCached | PrintValue, &TargetVal);
      // ClampCheck code
      if ((TargetVal < Min) || (TargetVal > Max)) {
        Done = TRUE;
      }
      Result = GetVccDllVsxHiCompVrefCountMsb (MrcData, VSXHI, WaitTime, TargetOrig);

      MSBValue = Result ? 1 : 0;
      if (TargetVal == TargetDefault) {
        PrevMSBValue = MSBValue;
        Step = (MSBValue) ? -1 : 1;
      } else if (MSBValue != PrevMSBValue) {
        Done = TRUE;  // Dont stop here, do the full sweep and report. Parameterize the sweep
        switch (Code) {
          case GsmIocVssHiFFCodeIdle :
            *MIdle = M;
            break;
          case GsmIocVssHiFFCodeWrite :
            *MWrite = M;
            break;
          case GsmIocVssHiFFCodeRead :
            *MRead = M;
            break;
          default:
            *MIdle = 0;
            *MWrite = 0;
            *MRead = 0;
            break;
        }
      }
      M += Step;
      if (M == 0) {
        M = DefaultM;
        MRC_DEBUG_MSG (&MrcData->Outputs.Debug, MSG_LEVEL_WARNING, "\n M has hit the value of 0 and therefore being restored to its default value");
      }
      TargetVal = Numerator / M;
    }  while (!Done);
  } else {
    MScale = (*MIdle + *MWrite + *MRead) / 3;
    MTrain = MScale * DefaultM;
    if (MTrain == 0) {
      MRC_DEBUG_MSG (&MrcData->Outputs.Debug, MSG_LEVEL_WARNING, "\n MIdle + MWrite + MRead = 0, restoring MTrain to be 1 to avoid divide by 0 fault");
      MTrain = 1;
    }
    TargetVal = Numerator / MTrain ;
    MRC_DEBUG_MSG (&MrcData->Outputs.Debug, MSG_LEVEL_NOTE, "\n MTrain %d, Mscale %d, Numerator %d\n", MTrain, MScale, Numerator);
    GetSet (MrcData, MRC_IGNORE_ARG, MRC_IGNORE_ARG, Channel, MRC_IGNORE_ARG, MRC_IGNORE_ARG, Byte, MRC_IGNORE_ARG, MRC_IGNORE_ARG, MRC_IGNORE_ARG, Code, WriteCached | PrintValue, &TargetVal);
  }
}

/**
  VsxHiFFTuning
  so MRC needs to tune the VsxHi FF legs to compensate

  @param[in] MrcData - Include all MRC global data.

  @retval MrcStatus - mrcFail if Inputs->ExitOnFailure is set and we cannot reach 4 GHz; otherwise mrcSuccess.
**/
MrcStatus
MrcVsxHiPostTrainingComp (
  IN MrcParameters *const MrcData
  )
{
  MrcInput          *Inputs;
  MrcOutput         *Outputs;
  MrcStatus         Status;
  INT64             GetSetEn;
  INT64             GetSetDis;
  INT64             InternalClocks;
  INT64             TxPiOn;
  INT64             RxPiOn;
  INT64             RxDisable;
  UINT32            Code;
  UINT32            ChannelPerRail[VCC_DLL_BYTES];
  UINT32            BytePerRail[VCC_DLL_BYTES];
  UINT32            TargetRail;
  UINT32            VsxHiTargetOrig;
  UINT32            FirstController;
  UINT32            FirstChannel;
  UINT32            MIdle;
  UINT32            MWrite;
  UINT32            MRead;
  UINT8             Partition;
  UINT8             ParStart;
  UINT8             ParEnd;
  UINT8             VccDllRail;
  UINT8             McChBitMask;
  UINT8             Wait;
  UINT8             NumCL;
  MC0_REQ0_CR_CPGC_SEQ_CTL_STRUCT  CpgcStartTest;
  MC0_REQ0_CR_CPGC_SEQ_CTL_STRUCT  CpgcStopTest;
  static UINT32 const TargetCode [] = {GsmIocVssHiFFCodeIdle, GsmIocVssHiFFCodeWrite, GsmIocVssHiFFCodeRead, GsmIocVssHiFFCodePBD, GsmIocVssHiFFCodePi};

  Inputs = &MrcData->Inputs;
  Outputs = &MrcData->Outputs;
  McChBitMask = Outputs->McChBitMask;
  CpgcStartTest.Data = CpgcStopTest.Data = 0;
  CpgcStartTest.Bits.START_TEST = 1;
  CpgcStopTest.Bits.STOP_TEST = 1;
  Wait = 64 / 2;
  NumCL = 64 / (Outputs->BurstLength * 2);
  GetSetEn = 1;
  GetSetDis = 0;
  MIdle = 0;
  MWrite = 0;
  MRead = 0;
  Status = mrcSuccess;

  VsxHiTargetOrig = MrcReadCR (MrcData,  DDRPHY_COMP_CR_VSSHITARGET_REG);
  // Read the first populated channel
  FirstController = (MrcControllerExist (MrcData, cCONTROLLER0)) ? 0 : 1;
  FirstChannel = Outputs->Controller[FirstController].FirstPopCh;

  // Save the following fields
  // Assume that the following fields will be same across all channels
  // Following fields are from datacontrol regs, channel is equivalent to controller for offset calc.
  // But these enums go through MrcTranslateSystemToIp translates channel and strobe to corresponding memtech
  MrcGetSetChStrb (MrcData, FirstController, FirstChannel, 0, GsmIocInternalClocksOn, ReadFromCache, &InternalClocks);
  MrcGetSetChStrb (MrcData, FirstController, FirstChannel, 0, GsmIocRxPiPwrDnDis, ReadFromCache, &RxPiOn);
  MrcGetSetChStrb (MrcData, FirstController, FirstChannel, 0, GsmIocTxPiPwrDnDis, ReadFromCache, &TxPiOn);
  MrcGetSetChStrb (MrcData, FirstController, FirstChannel, 0, GsmIocRxDisable, ReadFromCache, &RxDisable);

  static UINT8 const ParStartEnd[2][MAX_VSXHI_CODES] = {
  // @todo Add EccByte DLL Partition for TGL H DDR4
  //Code Idle  Write Read   Pi  Pbd
         {0,    0,    0,    0,  8},
         {11,   7,    7,    11, 11}};
  for (Code = 0; Code < ARRAY_COUNT (TargetCode); Code++) {
    if (TargetCode[Code] == GsmIocVssHiFFCodeRead) {
      Status = MrcResetSequence (MrcData);
      if (Status != mrcSuccess) {
        MRC_DEBUG_MSG (&Outputs->Debug, MSG_LEVEL_ERROR, "\n Dram Reset fails at VssHiFFCodeRead training \n");
        break;
      }
    }
    ParStart = ParStartEnd[0][Code];
    ParEnd = ParStartEnd[1][Code];
    for (Partition = ParStart; Partition <= ParEnd; Partition++) {
      if ((Partition == DllDdrData8) && (Inputs->UlxUlt)) { //Skip ECC byte for ULXULT
        continue;
      }
      MrcDllPartitionDecode (MrcData, Partition, ChannelPerRail, BytePerRail, &TargetRail);
      switch (TargetCode[Code]) {  // Setup DDRIO before starting compensation
        case GsmIocVssHiFFCodeIdle:
          MrcGetSetChStrb (MrcData, MAX_CONTROLLER, MAX_CHANNEL, MAX_SDRAM_IN_DIMM, GsmIocInternalClocksOn, WriteCached, &GetSetDis);
          MrcWeaklockEnDis (MrcData, MRC_ENABLE);
          FreezeVccDllVsxHi (MrcData, VSXHI, TRUE);
          for (VccDllRail = 0; VccDllRail < VCC_DLL_BYTES; VccDllRail++) {
            GetSet (MrcData, MRC_IGNORE_ARG, MRC_IGNORE_ARG, ChannelPerRail[VccDllRail], MRC_IGNORE_ARG, MRC_IGNORE_ARG, BytePerRail[VccDllRail], MRC_IGNORE_ARG, MRC_IGNORE_ARG, MRC_IGNORE_ARG, GsmIocInternalClocksOn, WriteCached | PrintValue, &GetSetEn);
          }
          break;

        case GsmIocVssHiFFCodeWrite:
          FreezeVccDllVsxHi (MrcData, VSXHI, TRUE);
          for (VccDllRail = 0; VccDllRail < VCC_DLL_BYTES; VccDllRail++) {
            GetSet (MrcData, MRC_IGNORE_ARG, MRC_IGNORE_ARG, ChannelPerRail[VccDllRail], MRC_IGNORE_ARG, MRC_IGNORE_ARG, BytePerRail[VccDllRail], MRC_IGNORE_ARG, MRC_IGNORE_ARG, MRC_IGNORE_ARG, GsmIocInternalClocksOn, WriteCached | PrintValue, &GetSetEn);
            GetSet (MrcData, MRC_IGNORE_ARG, MRC_IGNORE_ARG, ChannelPerRail[VccDllRail], MRC_IGNORE_ARG, MRC_IGNORE_ARG, BytePerRail[VccDllRail], MRC_IGNORE_ARG, MRC_IGNORE_ARG, MRC_IGNORE_ARG, GsmIocTxPiPwrDnDis, WriteCached | PrintValue, &GetSetEn);
          }
          SetupIOTestBasicVA (MrcData, McChBitMask, 1, NSOE, 0, 0, 9, PatWrEndless, Wait, NumCL);
          Cpgc20ControlRegWrite (MrcData, McChBitMask, CpgcStartTest);
          break;

        case GsmIocVssHiFFCodeRead:
          FreezeVccDllVsxHi (MrcData, VSXHI, TRUE);
          MrcGetSetChStrb (MrcData, MAX_CONTROLLER, MAX_CHANNEL, MAX_SDRAM_IN_DIMM, GsmIocRxDisable, WriteCached, &GetSetEn);
          for (VccDllRail = 0; VccDllRail < VCC_DLL_BYTES; VccDllRail++) {
            GetSet (MrcData, MRC_IGNORE_ARG, MRC_IGNORE_ARG, ChannelPerRail[VccDllRail], MRC_IGNORE_ARG, MRC_IGNORE_ARG, BytePerRail[VccDllRail], MRC_IGNORE_ARG, MRC_IGNORE_ARG, MRC_IGNORE_ARG, GsmIocInternalClocksOn, WriteCached | PrintValue, &GetSetEn);
            GetSet (MrcData, MRC_IGNORE_ARG, MRC_IGNORE_ARG, ChannelPerRail[VccDllRail], MRC_IGNORE_ARG, MRC_IGNORE_ARG, BytePerRail[VccDllRail], MRC_IGNORE_ARG, MRC_IGNORE_ARG, MRC_IGNORE_ARG, GsmIocRxPiPwrDnDis, WriteCached | PrintValue, &GetSetEn);
            GetSet (MrcData, MRC_IGNORE_ARG, MRC_IGNORE_ARG, ChannelPerRail[VccDllRail], MRC_IGNORE_ARG, MRC_IGNORE_ARG, BytePerRail[VccDllRail], MRC_IGNORE_ARG, MRC_IGNORE_ARG, MRC_IGNORE_ARG, GsmIocRxDisable, WriteCached | PrintValue, &GetSetDis);
          }
          SetupIOTestBasicVA (MrcData, McChBitMask, 1, NSOE, 0, 0, 9, PatRdEndless, Wait, NumCL);
          Cpgc20ControlRegWrite (MrcData, McChBitMask, CpgcStartTest);
          break;
        case GsmIocVssHiFFCodePBD:
        case GsmIocVssHiFFCodePi:
          // No setup as alternative way is implemented
          break;
      } // switch
      LinearSearchVsxHi (MrcData, Code, Partition, &VsxHiTargetOrig, &MIdle, &MWrite, &MRead);

      // Restore per partition, wouldnt be necessary if the training isnt per partition
      switch (Code) {
        case GsmIocVssHiFFCodeIdle:
          for (VccDllRail = 0; VccDllRail < VCC_DLL_BYTES; VccDllRail++) {
            GetSet (MrcData, MRC_IGNORE_ARG, MRC_IGNORE_ARG, ChannelPerRail[VccDllRail], MRC_IGNORE_ARG, MRC_IGNORE_ARG, BytePerRail[VccDllRail], MRC_IGNORE_ARG, MRC_IGNORE_ARG, MRC_IGNORE_ARG, GsmIocInternalClocksOn, WriteCached | PrintValue, &GetSetDis);
          }
          MrcWeaklockEnDis (MrcData, MRC_DISABLE);
          break;

        case GsmIocVssHiFFCodeWrite:
          for (VccDllRail = 0; VccDllRail < VCC_DLL_BYTES; VccDllRail++) {
            GetSet (MrcData, MRC_IGNORE_ARG, MRC_IGNORE_ARG, ChannelPerRail[VccDllRail], MRC_IGNORE_ARG, MRC_IGNORE_ARG, BytePerRail[VccDllRail], MRC_IGNORE_ARG, MRC_IGNORE_ARG, MRC_IGNORE_ARG, GsmIocInternalClocksOn, WriteCached | PrintValue, &GetSetDis);
            GetSet (MrcData, MRC_IGNORE_ARG, MRC_IGNORE_ARG, ChannelPerRail[VccDllRail], MRC_IGNORE_ARG, MRC_IGNORE_ARG, BytePerRail[VccDllRail], MRC_IGNORE_ARG, MRC_IGNORE_ARG, MRC_IGNORE_ARG, GsmIocTxPiPwrDnDis, WriteCached | PrintValue, &GetSetDis);
          }
          Cpgc20ControlRegWrite (MrcData, McChBitMask, CpgcStopTest);
          break;

        case GsmIocVssHiFFCodeRead:
          for (VccDllRail = 0; VccDllRail < VCC_DLL_BYTES; VccDllRail++) {
            GetSet (MrcData, MRC_IGNORE_ARG, MRC_IGNORE_ARG, ChannelPerRail[VccDllRail], MRC_IGNORE_ARG, MRC_IGNORE_ARG, BytePerRail[VccDllRail], MRC_IGNORE_ARG, MRC_IGNORE_ARG, MRC_IGNORE_ARG, GsmIocInternalClocksOn, WriteCached | PrintValue, &GetSetDis);
            GetSet (MrcData, MRC_IGNORE_ARG, MRC_IGNORE_ARG, ChannelPerRail[VccDllRail], MRC_IGNORE_ARG, MRC_IGNORE_ARG, BytePerRail[VccDllRail], MRC_IGNORE_ARG, MRC_IGNORE_ARG, MRC_IGNORE_ARG, GsmIocRxPiPwrDnDis, WriteCached | PrintValue, &GetSetDis);
          }
          Cpgc20ControlRegWrite (MrcData, McChBitMask, CpgcStopTest);
          break;
      } // switch
    } // Partition
    // Restore at the end of the code
    MrcGetSetChStrb (MrcData, MAX_CONTROLLER, MAX_CHANNEL, MAX_SDRAM_IN_DIMM, GsmIocInternalClocksOn, WriteToCache | PrintValue, &InternalClocks);
    MrcGetSetChStrb (MrcData, MAX_CONTROLLER, MAX_CHANNEL, MAX_SDRAM_IN_DIMM, GsmIocTxPiPwrDnDis, WriteToCache | PrintValue, &TxPiOn);
    MrcGetSetChStrb (MrcData, MAX_CONTROLLER, MAX_CHANNEL, MAX_SDRAM_IN_DIMM, GsmIocRxDisable, WriteToCache | PrintValue, &RxDisable);
    MrcGetSetChStrb (MrcData, MAX_CONTROLLER, MAX_CHANNEL, MAX_SDRAM_IN_DIMM, GsmIocRxPiPwrDnDis, WriteToCache | PrintValue, &RxPiOn);
    MrcFlushRegisterCachedData (MrcData);

    // CleanUp
    // Unfreeze VsxHi, assume OpenLoop = 0 in original as its not set unless we are training VsxHi
    FreezeVccDllVsxHi (MrcData, VSXHI, FALSE);
    // Restore VsxHiTarget
    MrcWriteCR (MrcData,  DDRPHY_COMP_CR_VSSHITARGET_REG, VsxHiTargetOrig);

  } // Code
  return Status;
}

/**
  VsxHiFFTuning
  so MRC needs to tune the VsxHi FF legs to compensate

  @param[in] MrcData - Include all MRC global data.
  @retval MrcStatus - mrcFail if Inputs->ExitOnFailure is set and we cannot reach 4 GHz; otherwise mrcSuccess.
**/
MrcStatus
MrcVsxHiFFTuning (
  IN MrcParameters *const MrcData
  )
{
  MrcOutput         *Outputs;
  MrcStatus         Status;
  UINT32            FirstController;
  UINT32            FirstChannel;
  UINT32            Offset;
  UINT32            VccDllBlock;
  DLLDDRDATA0_CR_DDRCRVCCDLLFFCONTROL_STRUCT  VccDllFFControl;

  Outputs = &MrcData->Outputs;
  Status  = mrcSuccess;
  FirstController = (MrcControllerExist (MrcData, cCONTROLLER0)) ? 0 : 1;
  FirstChannel = Outputs->Controller[FirstController].FirstPopCh;
  VccDllBlock = ((MAX_CCC_PER_CONTROLLER * FirstController) + FirstChannel);
  Offset = OFFSET_CALC_CH (DLLDDRDATA0_CR_DDRCRVCCDLLFFCONTROL_REG, DLLDDRDATA1_CR_DDRCRVCCDLLFFCONTROL_REG, VccDllBlock) ;
  VccDllFFControl.Data = MrcReadCR (MrcData, Offset);
  if (VccDllFFControl.Bits.LocalVsxHiBypass) {
    MRC_DEBUG_MSG (&Outputs->Debug, MSG_LEVEL_NOTE, "\n VsxHiFFTuning Skipped because LocalVsxHiBypass is set\n");
  } else if (VccDllFFControl.Bits.Bypass) {
    MrcWriteCR (MrcData, DLLDDR_CR_DDRCRVCCDLLVSXHIFF_REG, 0);
    MRC_DEBUG_MSG (&Outputs->Debug, MSG_LEVEL_NOTE, "\n VsxHiFFTuning Skipped because VccDllBypass is set. VccDllVsxHiFF\n");
  } else {
    Status = MrcVsxHiPostTrainingComp (MrcData);
  }
  return Status;
}

/**
  This function runs PI Step Size vs CB optimization

  @param[in, out] MrcData     - Include all MRC global data.
  @param[in] FirstCccChannel  - First CCC Channel populated.
  @param[in] OrigCCCDccFsmCtl - Original CCC DCC FSM Control register array.
  **/
void
MrcDccCbOptimization (
  IN OUT MrcParameters *const MrcData,
  IN     UINT32               FirstCccChannel,
  IN     UINT32               OrigCCCDccFsmCtl[MRC_NUM_CCC_INSTANCES]
  )
{
  MrcInput   *Inputs;
  MrcOutput  *Outputs;
  MrcDebug   *Debug;
  const MRC_FUNCTION *MrcCall;
  UINT8      SweepIndex;
  UINT8      SweepStart;
  UINT8      SweepEnd;
  UINT8      ArrayIndex;
  UINT8      Index;
  UINT8      TxDqsIndex;
  UINT8      BestIndex;
  UINT8      CccIndex;
  UINT8      Gear2;
  UINT8      MaxChannels;
  UINT8      MaxSdram;
  UINT8      RankEnd;
  INT16      RiseFallResults[MAX_OPTIMIZED_PI_CB_EN][MAX_OPTIMIZED_PI_SWEEP];
#ifdef MRC_DEBUG_PRINT
  INT16      FallResults[MAX_OPTIMIZED_PI_CB_EN][MAX_OPTIMIZED_PI_SWEEP];
#endif
  UINT32     Offset;
  UINT32     Data32;
  UINT32     Controller;
  UINT32     Channel;
  UINT32     Rank;
  UINT32     Byte;
  UINT32     OrigCCCPiDivider[MRC_NUM_CCC_INSTANCES];
  UINT32     ModCCCPiCoding0[MRC_NUM_CCC_INSTANCES];
  UINT32     OrigCCCPiCoding0[MRC_NUM_CCC_INSTANCES];
  INT32      SmallestPiCbEnResult;
  INT32      IdealStep;
  INT32      ResultStep[MAX_OPTIMIZED_PI_CB_EN][MAX_OPTIMIZED_PI_SWEEP];
  INT64      GetSetVal;
  INT64      GetSetEn;
  INT64      GetSetDis;
  DATA0CH0_CR_DCCFSMCONTROL_STRUCT    DataDccFsmCtl;
  CH0CCC_CR_DCCFSMCONTROL_STRUCT      CCCDccFsmCtl;
  CH0CCC_CR_DDRCRCCCPIDIVIDER_STRUCT  CCCPiDivider;

  Inputs = &MrcData->Inputs;
  MrcCall = Inputs->Call.Func;
  Outputs = &MrcData->Outputs;
  Debug = &Outputs->Debug;
  Gear2 = (Outputs->Gear2) ? 1 : 0;
  MaxChannels = Outputs->MaxChannels;
  MaxSdram = Outputs->SdramCount;
  RankEnd = (Outputs->DdrType == MRC_DDR_TYPE_DDR4) ? MAX_RANK_IN_CHANNEL : 1;
  GetSetEn = 1;
  GetSetDis = 0;

  if (!(Inputs->TrainingEnables2.DCC && Outputs->Gear2)) {
    //Skip this as DCC is not enabled
    return;
  }

  MRC_DEBUG_MSG (Debug, MSG_LEVEL_ALGO, "\n%sCB optimization\n", gStartTagStr);
  MrcCall->MrcSetMem ((UINT8 *) RiseFallResults, sizeof (RiseFallResults), 0);
#ifdef MRC_DEBUG_PRINT
  MrcCall->MrcSetMem ((UINT8 *) FallResults, sizeof (FallResults), 0);
#endif
  Offset = DataDccFsmCtlOffset (FirstCccChannel, 0);
  DataDccFsmCtl.Data = MrcReadCR (MrcData, Offset);

  //Step A-D
  GetSetVal = 1;
  MrcGetSetChStrb (MrcData, MAX_CONTROLLER, MAX_CHANNEL, MAX_SDRAM_IN_DIMM, GsmIocDataDccMeasPoint, WriteToCache | PrintValue, &GetSetVal);
  MrcGetSetChStrb (MrcData, MAX_CONTROLLER, MAX_CHANNEL, MAX_SDRAM_IN_DIMM, GsmIocDataDccSaveFullDcc, WriteToCache | PrintValue, &GetSetDis);
  MrcGetSetChStrb (MrcData, MAX_CONTROLLER, MAX_CHANNEL, MAX_SDRAM_IN_DIMM, GsmIocDataDccSkipCRWrite, WriteToCache | PrintValue, &GetSetEn);
  MrcGetSetChStrb (MrcData, MAX_CONTROLLER, MAX_CHANNEL, MAX_SDRAM_IN_DIMM, GsmIocDataDccRankEn, WriteToCache | PrintValue, &GetSetEn);
  GetSetVal = 0x4; //1 << 2
  MrcGetSetChStrb (MrcData, MAX_CONTROLLER, MAX_CHANNEL, MAX_SDRAM_IN_DIMM, GsmIocDataDccLaneEn, WriteToCache | PrintValue, &GetSetVal);
  MrcFlushRegisterCachedData (MrcData);

  for (CccIndex = 0; CccIndex < MRC_NUM_CCC_INSTANCES; CccIndex++) {
    Offset = OFFSET_CALC_CH (CH0CCC_CR_DCCFSMCONTROL_REG, CH1CCC_CR_DCCFSMCONTROL_REG, CccIndex);
    CCCDccFsmCtl.Data = OrigCCCDccFsmCtl[CccIndex];
    CCCDccFsmCtl.Bits.MeasPoint = 1;
    CCCDccFsmCtl.Bits.SaveFullDcc = 0;
    CCCDccFsmCtl.Bits.SkipCRWrite = 1;
    CCCDccFsmCtl.Bits.RankEn = 1;
    CCCDccFsmCtl.Bits.LaneEn = 0x2; //1 << 1
    MrcWriteCR (MrcData, Offset, CCCDccFsmCtl.Data);

    Offset = OFFSET_CALC_CH (CH0CCC_CR_DDRCRCCCPIDIVIDER_REG, CH1CCC_CR_DDRCRCCCPIDIVIDER_REG, CccIndex);
    CCCPiDivider.Data = OrigCCCPiDivider[CccIndex] = MrcReadCR (MrcData, Offset);
    CCCPiDivider.Bits.Pi0DivEn = 0;
    CCCPiDivider.Bits.Pi1DivEn = 0;
    CCCPiDivider.Bits.Pi2DivEn = 0;
    CCCPiDivider.Bits.Pi3DivEn = 0;
    CCCPiDivider.Bits.Pi4DivEn = 0;
    MrcWriteCR (MrcData, Offset, CCCPiDivider.Data);

    Offset = OFFSET_CALC_CH (CH0CCC_CR_DDRCRCCCPICODING0_REG, CH1CCC_CR_DDRCRCCCPICODING0_REG, CccIndex);
    ModCCCPiCoding0[CccIndex] = OrigCCCPiCoding0[CccIndex] = MrcReadCR (MrcData, Offset);
    ModCCCPiCoding0[CccIndex] &= ~(CH0CCC_CR_DDRCRCCCPICODING0_PiCode0_MSK | CH0CCC_CR_DDRCRCCCPICODING0_PiCode2_MSK);
  }

  MrcGetSetChStrb (MrcData, MAX_CONTROLLER, MAX_CHANNEL, MAX_SDRAM_IN_DIMM, GsmIocRankOverrideEn, WriteNoCache | PrintValue, &GetSetEn);
  MrcGetSetChStrb (MrcData, MAX_CONTROLLER, MAX_CHANNEL, MAX_SDRAM_IN_DIMM, GsmIocRankOverrideVal, WriteNoCache | PrintValue, &GetSetDis);

  //Step E-I
  if (Gear2) {
    SweepStart = 24;
    SweepEnd = 39;
  } else {
    SweepStart = 12;
    SweepEnd = 19;
  }
  for (TxDqsIndex = 0; TxDqsIndex < 2; TxDqsIndex++) {
    Data32 = (TxDqsIndex) ? (32 << Gear2) : 0;
    GetSetVal = Data32;
    MrcGetSetStrobe (MrcData, MAX_CONTROLLER, MAX_CHANNEL, 0, MAX_SDRAM_IN_DIMM, TxDqsDelay, WriteToCache, &GetSetVal);
    // Skipping Flushing as innermost for loop will take care of it.
    for (CccIndex = 0; CccIndex < MRC_NUM_CCC_INSTANCES; CccIndex++) {
      ModCCCPiCoding0[CccIndex] |= (Data32 << CH0CCC_CR_DDRCRCCCPICODING0_PiCode0_OFF);
    }
    MRC_DEBUG_MSG (Debug, MSG_LEVEL_ALGO, "\nTxDqsDelay/PiCode0 = 0x%x\n", Data32);

    for (Index = 0; Index < MAX_OPTIMIZED_PI_CB_EN; Index++) {
      GetSetVal = Index;
      MrcGetSetChStrb (MrcData, MAX_CONTROLLER, MAX_CHANNEL, MAX_SDRAM_IN_DIMM, RxCben, WriteToCache, &GetSetVal);
      for (SweepIndex = SweepStart; SweepIndex <= SweepEnd; SweepIndex++) {
        ArrayIndex = SweepIndex - SweepStart;
        Data32 = SweepIndex;
        GetSetVal = Data32;
        MrcGetSetStrobe (MrcData, MAX_CONTROLLER, MAX_CHANNEL, 0, MAX_SDRAM_IN_DIMM, TxDqDelay, WriteToCache, &GetSetVal);
        MrcFlushRegisterCachedData (MrcData);

        for (CccIndex = 0; CccIndex < MRC_NUM_CCC_INSTANCES; CccIndex++) {
          Offset = OFFSET_CALC_CH (CH0CCC_CR_DDRCRCCCPICODING0_REG, CH1CCC_CR_DDRCRCCCPICODING0_REG, CccIndex);
          MrcWriteCR (MrcData, Offset, ModCCCPiCoding0[CccIndex] | (Data32 << CH0CCC_CR_DDRCRCCCPICODING0_PiCode2_OFF));
        }
        ForceRcomp (MrcData);

        for (Controller = 0; Controller < MAX_CONTROLLER; Controller++) {
          for (Channel = 0; Channel < MaxChannels; Channel++) {
            for (Byte = 0; Byte < MaxSdram; Byte++) {
              //Lane2Result of 16 Data DLLs
              MrcGetSetBit (MrcData, Controller, Channel, MRC_IGNORE_ARG, Byte, 2, GsmIocDataDqDccLaneStatusResult, ReadCached, &GetSetVal);
              if (TxDqsIndex == 0) {
                RiseFallResults[Index][ArrayIndex] += (INT16) GetSetVal;
              } else {
                RiseFallResults[Index][ArrayIndex] -= (INT16) GetSetVal;
#ifdef MRC_DEBUG_PRINT
                FallResults[Index][ArrayIndex] += (INT16) GetSetVal;
#endif
              }
            }
          }
        }
        //CCC results
        for (Controller = 0; Controller < MAX_CONTROLLER; Controller++) {
          for (Channel = 0; Channel < MaxChannels; Channel++) {
            for (Rank = 0; Rank < RankEnd; Rank++) {
              if (!MrcRankExist (MrcData, Controller, Channel, Rank)) {
                continue;
              }
              //No usage of ChannelArrayIndex since this is aggregating all the results together
              MrcGetSetBit (MrcData, Controller, Channel, Rank, MRC_IGNORE_ARG, 1, GsmIocCccDccLaneStatusResult, ReadUncached, &GetSetVal);
              if (TxDqsIndex == 0) {
                RiseFallResults[Index][ArrayIndex] += (INT16) GetSetVal;
              } else {
                RiseFallResults[Index][ArrayIndex] -= (INT16) GetSetVal;
#ifdef MRC_DEBUG_PRINT
                FallResults[Index][ArrayIndex] += (INT16) GetSetVal;
#endif
              }
            }
          }
        }
      } // for SweepIndex (TxDqPi)
    } // for Index (RxCb)
#ifdef MRC_DEBUG_PRINT
    MRC_DEBUG_MSG (Debug, MSG_LEVEL_ALGO, "%sResults\n RxCb = ", (TxDqsIndex) ? "Fall" : "Rise");
    for (Index = 0; Index < MAX_OPTIMIZED_PI_CB_EN; Index++) {
      MRC_DEBUG_MSG (Debug, MSG_LEVEL_ALGO, "%d\t", Index);
    }
    MRC_DEBUG_MSG (Debug, MSG_LEVEL_ALGO, "\nTxDqPi");
    for (SweepIndex = SweepStart; SweepIndex <= SweepEnd; SweepIndex++) {
      MRC_DEBUG_MSG (Debug, MSG_LEVEL_ALGO, "\n%d", SweepIndex);
      ArrayIndex = SweepIndex - SweepStart;
      for (Index = 0; Index < MAX_OPTIMIZED_PI_CB_EN; Index++) {
        MRC_DEBUG_MSG (Debug, MSG_LEVEL_ALGO, "\t%d", (TxDqsIndex) ? FallResults[Index][ArrayIndex] : RiseFallResults[Index][ArrayIndex]);
      }
    }
    MRC_DEBUG_MSG (Debug, MSG_LEVEL_ALGO, "\n");
#endif
  } // for TxDqsIndex

#ifdef MRC_DEBUG_PRINT
  MRC_DEBUG_MSG (Debug, MSG_LEVEL_ALGO, "\nTotal Rise - Fall\n RxCb = ");
  for (Index = 0; Index < MAX_OPTIMIZED_PI_CB_EN; Index++) {
    MRC_DEBUG_MSG (Debug, MSG_LEVEL_ALGO, "%d\t", Index);
  }
  MRC_DEBUG_MSG (Debug, MSG_LEVEL_ALGO, "\nTxDqPi");
  for (SweepIndex = SweepStart; SweepIndex <= SweepEnd; SweepIndex++) {
    MRC_DEBUG_MSG (Debug, MSG_LEVEL_ALGO, "\n%d", SweepIndex);
    ArrayIndex = SweepIndex - SweepStart;
    for (Index = 0; Index < MAX_OPTIMIZED_PI_CB_EN; Index++) {
      MRC_DEBUG_MSG (Debug, MSG_LEVEL_ALGO, "\t%d", RiseFallResults[Index][ArrayIndex]);
    }
  }
  MRC_DEBUG_MSG (Debug, MSG_LEVEL_ALGO, "\n");
#endif

  //Step J
  MRC_DEBUG_MSG (Debug, MSG_LEVEL_ALGO, "\n");
  MrcCall->MrcSetMem ((UINT8 *) ResultStep, sizeof (ResultStep), 0);
  SweepEnd -= SweepStart;
  for (Index = 0; Index < MAX_OPTIMIZED_PI_CB_EN; Index++) {
    for (SweepIndex = 0; SweepIndex < SweepEnd; SweepIndex++) {
      ResultStep[Index][SweepIndex] += (RiseFallResults[Index][SweepIndex + 1] - RiseFallResults[Index][SweepIndex]);
      MRC_DEBUG_MSG (Debug, MSG_LEVEL_ALGO, "ResultStep[RxCb.%d][%d] = %d\n", Index, SweepIndex, ResultStep[Index][SweepIndex]);
    }
  }

  //Step K
  BestIndex = MAX_OPTIMIZED_PI_CB_EN;
  SmallestPiCbEnResult = MRC_INT32_MAX;
  IdealStep = ((Gear2) ? 4 : 8) * 24;//24 DLLs (16 Data, 8 CCC)
  MRC_DEBUG_MSG (Debug, MSG_LEVEL_ALGO, "\nIdealStep = %d\n", IdealStep);
  for (Index = 0; Index < MAX_OPTIMIZED_PI_CB_EN; Index++) {
    for (SweepIndex = 0; SweepIndex < SweepEnd; SweepIndex++) {
      ResultStep[Index][MAX_OPTIMIZED_PI_SWEEP - 1] += ((ResultStep[Index][SweepIndex] - IdealStep) * (ResultStep[Index][SweepIndex] - IdealStep));
    }
    MRC_DEBUG_MSG (Debug, MSG_LEVEL_ALGO, "RxCb %d, Sum((ResultStep[14:0]-IdealStep)^2) = %d\n", Index, ResultStep[Index][MAX_OPTIMIZED_PI_SWEEP - 1]);
    if (ResultStep[Index][MAX_OPTIMIZED_PI_SWEEP - 1] < SmallestPiCbEnResult) {
      SmallestPiCbEnResult = ResultStep[Index][MAX_OPTIMIZED_PI_SWEEP - 1];
      BestIndex = Index;
      MRC_DEBUG_MSG (Debug, MSG_LEVEL_ALGO, " Smaller Sum found, RxCb %d\n", Index);
    }
  }
  GetSetVal = BestIndex;
  MrcGetSetChStrb (MrcData, MAX_CONTROLLER, MAX_CHANNEL, MAX_SDRAM_IN_DIMM, RxCben, WriteToCache | PrintValue, &GetSetVal);
  MRC_DEBUG_MSG (Debug, MSG_LEVEL_ALGO, "\n%sCB optimization\n", gStopTagStr);

  //Step L
  GetSetVal = DataDccFsmCtl.Bits.MeasPoint;
  MrcGetSetChStrb (MrcData, MAX_CONTROLLER, MAX_CHANNEL, MAX_SDRAM_IN_DIMM, GsmIocDataDccMeasPoint, WriteToCache | PrintValue, &GetSetVal);
  GetSetVal = DataDccFsmCtl.Bits.SaveFullDcc;
  MrcGetSetChStrb (MrcData, MAX_CONTROLLER, MAX_CHANNEL, MAX_SDRAM_IN_DIMM, GsmIocDataDccSaveFullDcc, WriteToCache | PrintValue, &GetSetVal);
  GetSetVal = DataDccFsmCtl.Bits.SkipCRWrite;
  MrcGetSetChStrb (MrcData, MAX_CONTROLLER, MAX_CHANNEL, MAX_SDRAM_IN_DIMM, GsmIocDataDccSkipCRWrite, WriteToCache | PrintValue, &GetSetVal);
  for (Controller = 0; Controller < MAX_CONTROLLER; Controller++) {
    for (Channel = 0; Channel < MaxChannels; Channel++) {
      GetSetVal = (MrcChannelExist (MrcData, Controller, Channel)) ? Outputs->Controller[Controller].Channel[Channel].ValidRankBitMask : 0;
      MrcGetSetChStrb (MrcData, Controller, Channel, MAX_SDRAM_IN_DIMM, GsmIocDataDccRankEn, WriteToCache | PrintValue, &GetSetVal);
    }
  }
  GetSetVal = DataDccFsmCtl.Bits.LaneEn;
  MrcGetSetChStrb (MrcData, MAX_CONTROLLER, MAX_CHANNEL, MAX_SDRAM_IN_DIMM, GsmIocDataDccLaneEn, WriteToCache | PrintValue, &GetSetVal);
  for (CccIndex = 0; CccIndex < MRC_NUM_CCC_INSTANCES; CccIndex++) {
    Offset = OFFSET_CALC_CH (CH0CCC_CR_DCCFSMCONTROL_REG, CH1CCC_CR_DCCFSMCONTROL_REG, CccIndex);
    MrcWriteCR (MrcData, Offset, OrigCCCDccFsmCtl[CccIndex]);

    Offset = OFFSET_CALC_CH (CH0CCC_CR_DDRCRCCCPIDIVIDER_REG, CH1CCC_CR_DDRCRCCCPIDIVIDER_REG, CccIndex);
    MrcWriteCR (MrcData, Offset, OrigCCCPiDivider[CccIndex]);

    Offset = OFFSET_CALC_CH (CH0CCC_CR_DDRCRCCCPICODING0_REG, CH1CCC_CR_DDRCRCCCPICODING0_REG, CccIndex);
    MrcWriteCR (MrcData, Offset, OrigCCCPiCoding0[CccIndex]);
  }
  GetSetVal = 64;
  MrcGetSetStrobe (MrcData, MAX_CONTROLLER, MAX_CHANNEL, 0, MAX_SDRAM_IN_DIMM, TxDqsDelay, WriteToCache | PrintValue, &GetSetVal);
  GetSetVal += (Gear2) ? 96 : 32;
  MrcGetSetStrobe (MrcData, MAX_CONTROLLER, MAX_CHANNEL, 0, MAX_SDRAM_IN_DIMM, TxDqDelay, WriteToCache | PrintValue, &GetSetVal);
  MrcGetSetChStrb (MrcData, MAX_CONTROLLER, MAX_CHANNEL, MAX_SDRAM_IN_DIMM, GsmIocRankOverrideEn, WriteNoCache | PrintValue, &GetSetDis);
  MrcFlushRegisterCachedData (MrcData);
  // Propagate new value
  MrcGetSetNoScope (MrcData, GsmIocForceCmpUpdt, ForceWriteUncached | PrintValue, &GetSetEn);
  ForceRcomp (MrcData);
}

/**
  This function sets up LUT for DCC

  @param[in, out] MrcData - Include all MRC global data.

  @retval mrcSuccess
**/
void
MrcDccLut (
  IN OUT MrcParameters *const MrcData
  )
{
  MrcInput   *Inputs;
  MrcOutput  *Outputs;
  MrcDebug   *Debug;
  const MRC_FUNCTION *MrcCall;
  BOOLEAN    LutHalfStep;
  UINT8      Gear2;
  UINT8      Index;
  UINT8      IndexEnd;
  UINT8      BitIndex;
  UINT8      SweepIndex;
  UINT8      SweepStart;
  UINT8      SweepEnd;
  UINT8      MaxChannels;
  UINT8      MaxSdram;
  UINT8      Count;
  UINT8      Data8;
  UINT8      ShiftValue;
  INT16      SignedData16;
  INT16      Fine[MAX_CONTROLLER][MAX_CHANNEL][MAX_SDRAM_IN_DIMM][MAX_BITS];
  INT16      Coarse[MAX_CONTROLLER][MAX_CHANNEL][MAX_SDRAM_IN_DIMM][MAX_COARSE];
  UINT16     *Data16Ptr;
  UINT16     Data16;
  UINT16     DccLaneStatusResultAvg[MAX_CONTROLLER][MAX_CHANNEL][MAX_SDRAM_IN_DIMM][MAX_DCC_TX_DQS_DQ_PI];
  UINT16     AveDcc[MAX_CONTROLLER][MAX_CHANNEL][MAX_SDRAM_IN_DIMM];
  UINT32     Data32;
  UINT32     Results32;
  UINT32     TransController;
  UINT32     TransChannel;
  UINT32     TransByte;
  UINT32     TransLane;
  UINT32     TransRank;
  UINT32     Controller;
  UINT32     Channel;
  UINT32     Byte;
  UINT32     Bit;
  UINT32     Offset;
  UINT32     GsmMode;
  UINT32     DccOffset[MAX_DCC_TX_DQS_DQ_PI];
  INT64      GetSetEn;
  INT64      GetSetDis;
  INT64      GetSetVal;
  DATA0CH0_CR_DCCPILUT0_STRUCT  DccPiLut0;
  DATA0CH0_CR_DCCPILUT1_STRUCT  DccPiLut1;
  DATA0CH0_CR_DCCPILUT2_STRUCT  DccPiLut2;
  DATA0CH0_CR_DCCPILUT3_STRUCT  DccPiLut3;
  DATA0CH0_CR_DCCPILUT4_STRUCT  DccPiLut4;

  Inputs      = &MrcData->Inputs;
  MrcCall     = Inputs->Call.Func;
  Outputs     = &MrcData->Outputs;
  Debug       = &Outputs->Debug;
  MaxChannels = Outputs->MaxChannels;
  MaxSdram    = Outputs->SdramCount;
  Gear2       = (Outputs->Gear2) ? 1 : 0;
  GetSetEn    = 1;
  GetSetDis   = 0;

  MRC_DEBUG_MSG (Debug, MSG_LEVEL_ALGO, "\n%sDCC LUT - Up to 128 PI settings\n", gStartTagStr);
  MrcCall->MrcSetMem ((UINT8 *) DccLaneStatusResultAvg, sizeof (DccLaneStatusResultAvg), 0);
  MrcCall->MrcSetMem ((UINT8 *) AveDcc, sizeof (AveDcc), 0);
  MrcCall->MrcSetMem ((UINT8 *) Fine, sizeof (Fine), 0);
  MrcCall->MrcSetMem ((UINT8 *) Coarse, sizeof (Coarse), 0);
  MrcCall->MrcSetMem ((UINT8 *) DccOffset, sizeof (DccOffset), 0);
  MrcGetSetChStrb (MrcData, MAX_CONTROLLER, MAX_CHANNEL, MAX_SDRAM_IN_DIMM, GsmIocDataDccSaveFullDcc, WriteToCache | PrintValue, &GetSetDis); //To get DccCount in DCCLaneStatus
  MrcGetSetChStrb (MrcData, MAX_CONTROLLER, MAX_CHANNEL, MAX_SDRAM_IN_DIMM, GsmIocDataDccSkipCRWrite, WriteToCache | PrintValue, &GetSetEn);
  MrcFlushRegisterCachedData (MrcData);

  SweepEnd = MAX_DCC_TX_DQS_DQ_PI / ((Gear2) ? 1 : 2);
  for (SweepIndex = 0; SweepIndex < SweepEnd; SweepIndex++) {
    GetSetVal = SweepIndex;
    MrcGetSetStrobe (MrcData, MAX_CONTROLLER, MAX_CHANNEL, MAX_RANK_IN_CHANNEL, MAX_SDRAM_IN_DIMM, TxDqDelay, WriteCached, &GetSetVal);

    for (Index = 0; Index < 3; Index++) {
      ForceRcomp (MrcData);
    }
    for (Controller = 0; Controller < MAX_CONTROLLER; Controller++) {
      for (Channel = 0; Channel < MaxChannels; Channel++) {
        if (!MrcChannelExist (MrcData, Controller, Channel)) {
          continue;
        }
        for (Byte = 0; Byte < MaxSdram; Byte++) {
          Data16Ptr = &DccLaneStatusResultAvg[Controller][Channel][Byte][SweepIndex];
          for (Bit = 0; Bit < MAX_BITS; Bit++) {
            //Only read the CR when needed (each register holds 3 bits), otherwise read from cache
            GsmMode = (Bit % 3) ? ReadFromCache : ReadCached;
            MrcGetSetBit (MrcData, Controller, Channel, MRC_IGNORE_ARG, Byte, Bit, GsmIocDataDqDccLaneStatusResult, GsmMode, &GetSetVal);
            *Data16Ptr += (UINT16) GetSetVal;
          }
          //Getting the average bits across the byte
          *Data16Ptr = DIVIDEROUND (*Data16Ptr, MAX_BITS);
        }
      }
    }
  }
#ifdef MRC_DEBUG_PRINT
  MRC_DEBUG_MSG (Debug, MSG_LEVEL_ALGO, "DccCount (Average) - TxDq PI\nPI Setting");
  for (SweepIndex = 0; SweepIndex < SweepEnd; SweepIndex++) {
    MRC_DEBUG_MSG (Debug, MSG_LEVEL_ALGO, "\t%d", SweepIndex);
  }
  MRC_DEBUG_MSG (Debug, MSG_LEVEL_ALGO, "\n");

  for (Controller = 0; Controller < MAX_CONTROLLER; Controller++) {
    for (Channel = 0; Channel < MaxChannels; Channel++) {
      if (!MrcChannelExist (MrcData, Controller, Channel)) {
        continue;
      }
      for (Byte = 0; Byte < MaxSdram; Byte++) {
        MRC_DEBUG_MSG (Debug, MSG_LEVEL_ALGO, " MC%d.C%d.S%d", Controller, Channel, Byte);
        for (SweepIndex = 0; SweepIndex < SweepEnd; SweepIndex++) {
          MRC_DEBUG_MSG (Debug, MSG_LEVEL_ALGO, "\t%d", DccLaneStatusResultAvg[Controller][Channel][Byte][SweepIndex]);
        }
        MRC_DEBUG_MSG (Debug, MSG_LEVEL_ALGO, "\n");
      }
    }
  }

  MRC_DEBUG_MSG (Debug, MSG_LEVEL_ALGO, "\nDccOffset - TxDq PI\nPI Setting");
  for (SweepIndex = 0; SweepIndex < SweepEnd; SweepIndex++) {
    MRC_DEBUG_MSG (Debug, MSG_LEVEL_ALGO, "\t%d", SweepIndex);
  }
  MRC_DEBUG_MSG (Debug, MSG_LEVEL_ALGO, "\tAveDcc\n");
#endif
  IndexEnd = SweepEnd / 16;
  for (Controller = 0; Controller < MAX_CONTROLLER; Controller++) {
    for (Channel = 0; Channel < MaxChannels; Channel++) {
      if (!MrcChannelExist (MrcData, Controller, Channel)) {
        continue;
      }
      for (Byte = 0; Byte < MaxSdram; Byte++) {
        MrcGetSetChStrb (MrcData, Controller, Channel, Byte, GsmIocDataDcc2xStep, ReadFromCache, &GetSetVal);
        Data8 = (GetSetVal) ? 2 : 1;
        MrcGetSetChStrb (MrcData, Controller, Channel, Byte, GsmIocDataDccStepSize, ReadFromCache, &GetSetVal);
        Data16 = (UINT16) GetSetVal;
        Data16 *= Data8;

        //AveDcc
        MRC_DEBUG_MSG (Debug, MSG_LEVEL_ALGO, " MC%d.C%d.S%d", Controller, Channel, Byte);
        Data16Ptr = &AveDcc[Controller][Channel][Byte];
        Data32 = 0;
        for (SweepIndex = 0; SweepIndex < SweepEnd; SweepIndex++) {
          Results32 = Data16 * DccLaneStatusResultAvg[Controller][Channel][Byte][SweepIndex];
          MRC_DEBUG_ASSERT (Results32 < MRC_UINT16_MAX, Debug, "Calculation (%d) exceeded DccOffset define size of UINT16 for MC%d.C%d.S%d.SweepIndex%d!\n", Results32, Controller, Channel, Byte, SweepIndex);
          DccOffset[SweepIndex] = (UINT16) Results32;
          MRC_DEBUG_MSG (Debug, MSG_LEVEL_ALGO, "\t%d", DccOffset[SweepIndex]);
          Data32 += DccOffset[SweepIndex];
        }
        *Data16Ptr = (UINT16) (DIVIDEROUND (Data32, SweepEnd));
        MRC_DEBUG_MSG (Debug, MSG_LEVEL_ALGO, "\t%d\n", *Data16Ptr);

        //Fine[x]
        for (BitIndex = 0; BitIndex < MAX_BITS; BitIndex++) {
          Data32 = 0;
          for (Index = 0; Index < IndexEnd; Index++) {
            //Adds up
            Data8 = (Index * 16) + (2 * BitIndex);
            Data32 += DccOffset[Data8] + DccOffset[Data8 + 1];
          }
          Data16 = (UINT16) (DIVIDEROUND (Data32, (IndexEnd * 2)));
          Fine[Controller][Channel][Byte][BitIndex] = (INT16) (Data16 - *Data16Ptr);
        }

        //Coarse
        for (Index = 0; Index < MAX_COARSE; Index++) {
          switch (Index) {
            case 0:
              SignedData16 = Fine[Controller][Channel][Byte][0] + Fine[Controller][Channel][Byte][1];
              Data16 = DIVIDEROUND (SignedData16, 2);
              SignedData16 = (INT16) (*Data16Ptr - Data16);
              SweepStart = 0;
              SweepEnd = 3;
              break;

            case 1:
              SignedData16 = Fine[Controller][Channel][Byte][2] + Fine[Controller][Channel][Byte][3];
              Data16 = DIVIDEROUND (SignedData16, 2);
              SignedData16 = (INT16) (*Data16Ptr - Data16);
              SweepStart = 4;
              SweepEnd = 7;
              break;

            case 2:
              SignedData16 = Fine[Controller][Channel][Byte][4] + Fine[Controller][Channel][Byte][5];
              Data16 = DIVIDEROUND (SignedData16, 2);
              SignedData16 = (INT16) (*Data16Ptr - Data16);
              SweepStart = 8;
              SweepEnd = 11;
              break;

            case 3:
              SignedData16 = Fine[Controller][Channel][Byte][6] + Fine[Controller][Channel][Byte][7];
              Data16 = DIVIDEROUND (SignedData16, 2);
              SignedData16 = (INT16) (*Data16Ptr - Data16);
              SweepStart = 12;
              SweepEnd = 15;
              break;

            case 4:
              SignedData16 = Fine[Controller][Channel][Byte][0] + Fine[Controller][Channel][Byte][1] + Fine[Controller][Channel][Byte][2] + Fine[Controller][Channel][Byte][3];
              Data16 = DIVIDEROUND (SignedData16, 4);
              SignedData16 = (INT16) (*Data16Ptr - Data16);
              SweepStart = 16;
              SweepEnd = 23;
              break;

            case 5:
              SignedData16 = (INT16) (*Data16Ptr);
              SweepStart = 24;
              SweepEnd = 39;
              break;

            case 6:
              SignedData16 = (INT16) (*Data16Ptr);
              SweepStart = 40;
              SweepEnd = 55;
              break;

            case 7:
              SignedData16 = (INT16) (*Data16Ptr);
              SweepStart = 56;
              SweepEnd = 71;
              break;

            case 8:
              SignedData16 = (INT16) (*Data16Ptr);
              SweepStart = 72;
              SweepEnd = 87;
              break;

            case 9:
              SignedData16 = (INT16) (*Data16Ptr);
              SweepStart = 88;
              SweepEnd = 103;
              break;

            case 10:
              SignedData16 = Fine[Controller][Channel][Byte][4] + Fine[Controller][Channel][Byte][5] + Fine[Controller][Channel][Byte][6] + Fine[Controller][Channel][Byte][7];
              Data16 = DIVIDEROUND (SignedData16, 4);
              SignedData16 = (INT16) (*Data16Ptr - Data16);
              SweepStart = 104;
              SweepEnd = 111;
              break;

            case 11:
              SignedData16 = Fine[Controller][Channel][Byte][0] + Fine[Controller][Channel][Byte][1];
              Data16 = DIVIDEROUND (SignedData16, 2);
              SignedData16 = (INT16) (*Data16Ptr - Data16);
              SweepStart = 112;
              SweepEnd = 115;
              break;

            case 12:
              SignedData16 = Fine[Controller][Channel][Byte][2] + Fine[Controller][Channel][Byte][3];
              Data16 = DIVIDEROUND (SignedData16, 2);
              SignedData16 = (INT16) (*Data16Ptr - Data16);
              SweepStart = 116;
              SweepEnd = 119;
              break;

            case 13:
              SignedData16 = Fine[Controller][Channel][Byte][4] + Fine[Controller][Channel][Byte][5];
              Data16 = DIVIDEROUND (SignedData16, 2);
              SignedData16 = (INT16) (*Data16Ptr - Data16);
              SweepStart = 120;
              SweepEnd = 123;
              break;

            case 14:
              SignedData16 = Fine[Controller][Channel][Byte][6] + Fine[Controller][Channel][Byte][7];
              Data16 = DIVIDEROUND (SignedData16, 2);
              SignedData16 = (INT16) (*Data16Ptr - Data16);
              SweepStart = 124;
              SweepEnd = 127;
              break;

            default:
              SignedData16 = (INT16) (*Data16Ptr);
              SweepStart = 0;
              SweepEnd = 0;
          }
          Data8 = (Gear2) ? 1 : 2;
          SweepStart /= Data8;
          SweepEnd /= Data8;
          Data32 = Count = 0;
          for (SweepIndex = SweepStart; SweepIndex <= SweepEnd; SweepIndex++) {
            Data32 += DccOffset[SweepIndex];
            Count++;
          }
          Count = MAX (Count, 1);
          Data16 = (UINT16) (DIVIDEROUND (Data32, Count));
          Coarse[Controller][Channel][Byte][Index] = (INT16) (Data16 - SignedData16);
        } // for MAX_COARSE
      } // for Byte
    } // for Channel
  } // for Controller

#ifdef MRC_DEBUG_PRINT
  MRC_DEBUG_MSG (Debug, MSG_LEVEL_ALGO, "\nFine[x] - TxDq PI\nx = \t");
  for (SweepIndex = 0; SweepIndex < MAX_BITS; SweepIndex++) {
    MRC_DEBUG_MSG (Debug, MSG_LEVEL_ALGO, "\t%d", SweepIndex);
  }
  MRC_DEBUG_MSG (Debug, MSG_LEVEL_ALGO, "\n");

  for (Controller = 0; Controller < MAX_CONTROLLER; Controller++) {
    for (Channel = 0; Channel < MaxChannels; Channel++) {
      if (!MrcChannelExist (MrcData, Controller, Channel)) {
        continue;
      }
      for (Byte = 0; Byte < MaxSdram; Byte++) {
        MRC_DEBUG_MSG (Debug, MSG_LEVEL_ALGO, " MC%d.C%d.S%d", Controller, Channel, Byte);
        for (SweepIndex = 0; SweepIndex < MAX_BITS; SweepIndex++) {
          MRC_DEBUG_MSG (Debug, MSG_LEVEL_ALGO, "\t%d", Fine[Controller][Channel][Byte][SweepIndex]);
        }
        MRC_DEBUG_MSG (Debug, MSG_LEVEL_ALGO, "\n");
      }
    }
  }

  MRC_DEBUG_MSG (Debug, MSG_LEVEL_ALGO, "\nCoarse[x] - TxDq PI\nx = \t");
  for (SweepIndex = 0; SweepIndex < MAX_COARSE; SweepIndex++) {
    MRC_DEBUG_MSG (Debug, MSG_LEVEL_ALGO, "\t%d", SweepIndex);
  }
  MRC_DEBUG_MSG (Debug, MSG_LEVEL_ALGO, "\n");

  for (Controller = 0; Controller < MAX_CONTROLLER; Controller++) {
    for (Channel = 0; Channel < MaxChannels; Channel++) {
      if (!MrcChannelExist (MrcData, Controller, Channel)) {
        continue;
      }
      for (Byte = 0; Byte < MaxSdram; Byte++) {
        MRC_DEBUG_MSG (Debug, MSG_LEVEL_ALGO, " MC%d.C%d.S%d", Controller, Channel, Byte);
        for (SweepIndex = 0; SweepIndex < MAX_COARSE; SweepIndex++) {
          MRC_DEBUG_MSG (Debug, MSG_LEVEL_ALGO, "\t%d", Coarse[Controller][Channel][Byte][SweepIndex]);
        }
        MRC_DEBUG_MSG (Debug, MSG_LEVEL_ALGO, "\n");
      }
    }
  }
#endif

  for (Controller = 0; Controller < MAX_CONTROLLER; Controller++) {
    for (Channel = 0; Channel < MaxChannels; Channel++) {
      if (!MrcChannelExist (MrcData, Controller, Channel)) {
        continue;
      }
      for (Byte = 0; Byte < MaxSdram; Byte++) {
        Data16 = 0;
        for (Index = 0; Index < MAX_BITS; Index++) {
          Data16 = MAX (Data16, ABS (Fine[Controller][Channel][Byte][Index]));
        }
        for (Index = 0; Index < MAX_COARSE; Index++) {
          Data16 = MAX (Data16, ABS (Coarse[Controller][Channel][Byte][Index]));
        }
        LutHalfStep = (Data16 < (31 * 16));

        ShiftValue = (LutHalfStep) ? 4 : 5;

        for (Index = 0; Index < MAX_BITS; Index++) {
          SignedData16 = Fine[Controller][Channel][Byte][Index];
          Fine[Controller][Channel][Byte][Index] = (SignedData16 + (1 << (ShiftValue - 1))) >> ShiftValue;
        }

        for (Index = 0; Index < MAX_COARSE; Index++) {
          SignedData16 = Coarse[Controller][Channel][Byte][Index];
          Coarse[Controller][Channel][Byte][Index] = (SignedData16 + (1 << (ShiftValue - 1))) >> ShiftValue;
        }

        TransController = Controller;
        TransChannel = Channel;
        TransByte = Byte;
        TransLane = 0;
        MrcTranslateSystemToIp (MrcData, &TransController, &TransChannel, &TransRank, &TransByte, &TransLane, TxDqDelay);
        DccPiLut0.Data = DccPiLut1.Data = DccPiLut2.Data = DccPiLut3.Data = DccPiLut4.Data = 0;

        DccPiLut0.Bits.Fine0 = Fine[Controller][Channel][Byte][0];
        DccPiLut0.Bits.Fine1 = Fine[Controller][Channel][Byte][1];
        DccPiLut0.Bits.Fine2 = Fine[Controller][Channel][Byte][2];
        DccPiLut0.Bits.Fine3 = Fine[Controller][Channel][Byte][3];
        DccPiLut0.Bits.Fine4 = Fine[Controller][Channel][Byte][4];
        Offset = OFFSET_CALC_MC_CH (DATA0CH0_CR_DCCPILUT0_REG, DATA0CH1_CR_DCCPILUT0_REG, TransChannel, DATA1CH0_CR_DCCPILUT0_REG, TransByte);
        MrcWriteCR (MrcData, Offset, DccPiLut0.Data);
        MRC_DEBUG_MSG (Debug, MSG_LEVEL_ALGO, "Mc%d.C%d.S%d DccPiLut0 (0x%x) = 0x%x, ", Controller, Channel, Byte, Offset, DccPiLut0.Data);

        DccPiLut1.Bits.Fine5 = Fine[Controller][Channel][Byte][5];
        DccPiLut1.Bits.Fine6 = Fine[Controller][Channel][Byte][6];
        DccPiLut1.Bits.Fine7 = Fine[Controller][Channel][Byte][7];
        Offset = OFFSET_CALC_MC_CH (DATA0CH0_CR_DCCPILUT1_REG, DATA0CH1_CR_DCCPILUT1_REG, TransChannel, DATA1CH0_CR_DCCPILUT1_REG, TransByte);
        MrcWriteCR (MrcData, Offset, DccPiLut1.Data);
        MRC_DEBUG_MSG (Debug, MSG_LEVEL_ALGO, "DccPiLut1 (0x%x) = 0x%x, ", Offset, DccPiLut1.Data);

        DccPiLut2.Bits.Coarse0 = Coarse[Controller][Channel][Byte][0];
        DccPiLut2.Bits.Coarse1 = Coarse[Controller][Channel][Byte][1];
        DccPiLut2.Bits.Coarse2 = Coarse[Controller][Channel][Byte][2];
        DccPiLut2.Bits.Coarse3 = Coarse[Controller][Channel][Byte][3];
        DccPiLut2.Bits.Coarse4 = Coarse[Controller][Channel][Byte][4];
        Offset = OFFSET_CALC_MC_CH (DATA0CH0_CR_DCCPILUT2_REG, DATA0CH1_CR_DCCPILUT2_REG, TransChannel, DATA1CH0_CR_DCCPILUT2_REG, TransByte);
        MrcWriteCR (MrcData, Offset, DccPiLut2.Data);
        MRC_DEBUG_MSG (Debug, MSG_LEVEL_ALGO, "DccPiLut2 (0x%x) = 0x%x, ", Offset, DccPiLut2.Data);

        DccPiLut3.Bits.Coarse5 = Coarse[Controller][Channel][Byte][5];
        DccPiLut3.Bits.Coarse6 = Coarse[Controller][Channel][Byte][6];
        DccPiLut3.Bits.Coarse7 = Coarse[Controller][Channel][Byte][7];
        DccPiLut3.Bits.Coarse8 = Coarse[Controller][Channel][Byte][8];
        DccPiLut3.Bits.Coarse9 = Coarse[Controller][Channel][Byte][9];
        Offset = OFFSET_CALC_MC_CH (DATA0CH0_CR_DCCPILUT3_REG, DATA0CH1_CR_DCCPILUT3_REG, TransChannel, DATA1CH0_CR_DCCPILUT3_REG, TransByte);
        MrcWriteCR (MrcData, Offset, DccPiLut3.Data);
        MRC_DEBUG_MSG (Debug, MSG_LEVEL_ALGO, "DccPiLut3 (0x%x) = 0x%x, ", Offset, DccPiLut3.Data);

        DccPiLut4.Bits.Coarse10 = Coarse[Controller][Channel][Byte][10];
        DccPiLut4.Bits.Coarse11 = Coarse[Controller][Channel][Byte][11];
        DccPiLut4.Bits.Coarse12 = Coarse[Controller][Channel][Byte][12];
        DccPiLut4.Bits.Coarse13 = Coarse[Controller][Channel][Byte][13];
        DccPiLut4.Bits.Coarse14 = Coarse[Controller][Channel][Byte][14];
        DccPiLut4.Bits.LUTHalfStep = (LutHalfStep) ? 1 : 0;
        Offset = OFFSET_CALC_MC_CH (DATA0CH0_CR_DCCPILUT4_REG, DATA0CH1_CR_DCCPILUT4_REG, TransChannel, DATA1CH0_CR_DCCPILUT4_REG, TransByte);
        MrcWriteCR (MrcData, Offset, DccPiLut4.Data);
        MRC_DEBUG_MSG (Debug, MSG_LEVEL_ALGO, "DccPiLut4 (0x%x) = 0x%x\n", Offset, DccPiLut4.Data);
      }
    }
  }
  // Propagate new value
  MrcGetSetNoScope (MrcData, GsmIocForceCmpUpdt, ForceWriteUncached | PrintValue, &GetSetEn);
  ForceRcomp (MrcData);

#ifdef MRC_DEBUG_PRINT
  MRC_DEBUG_MSG (Debug, MSG_LEVEL_ALGO, "\nFinalFine[x] - TxDq PI\nx = \t");
  for (SweepIndex = 0; SweepIndex < MAX_BITS; SweepIndex++) {
    MRC_DEBUG_MSG (Debug, MSG_LEVEL_ALGO, "\t%d", SweepIndex);
  }
  MRC_DEBUG_MSG (Debug, MSG_LEVEL_ALGO, "\n");

  for (Controller = 0; Controller < MAX_CONTROLLER; Controller++) {
    for (Channel = 0; Channel < MaxChannels; Channel++) {
      if (!MrcChannelExist (MrcData, Controller, Channel)) {
        continue;
      }
      for (Byte = 0; Byte < MaxSdram; Byte++) {
        MRC_DEBUG_MSG (Debug, MSG_LEVEL_ALGO, " MC%d.C%d.S%d", Controller, Channel, Byte);
        for (SweepIndex = 0; SweepIndex < MAX_BITS; SweepIndex++) {
          MRC_DEBUG_MSG (Debug, MSG_LEVEL_ALGO, "\t%d", Fine[Controller][Channel][Byte][SweepIndex]);
        }
        MRC_DEBUG_MSG (Debug, MSG_LEVEL_ALGO, "\n");
      }
    }
  }

  MRC_DEBUG_MSG (Debug, MSG_LEVEL_ALGO, "\nFinalCoarse[x] - TxDq PI\nx = \t");
  for (SweepIndex = 0; SweepIndex < MAX_COARSE; SweepIndex++) {
    MRC_DEBUG_MSG (Debug, MSG_LEVEL_ALGO, "\t%d", SweepIndex);
  }
  MRC_DEBUG_MSG (Debug, MSG_LEVEL_ALGO, "\n");

  for (Controller = 0; Controller < MAX_CONTROLLER; Controller++) {
    for (Channel = 0; Channel < MaxChannels; Channel++) {
      if (!MrcChannelExist (MrcData, Controller, Channel)) {
        continue;
      }
      for (Byte = 0; Byte < MaxSdram; Byte++) {
        MRC_DEBUG_MSG (Debug, MSG_LEVEL_ALGO, " MC%d.C%d.S%d", Controller, Channel, Byte);
        for (SweepIndex = 0; SweepIndex < MAX_COARSE; SweepIndex++) {
          MRC_DEBUG_MSG (Debug, MSG_LEVEL_ALGO, "\t%d", Coarse[Controller][Channel][Byte][SweepIndex]);
        }
        MRC_DEBUG_MSG (Debug, MSG_LEVEL_ALGO, "\n");
      }
    }
  }
  MRC_DEBUG_MSG (Debug, MSG_LEVEL_ALGO, "\n%sDCC LUT - Up to 128 PI settings\n", gStopTagStr);
#endif

  GetSetVal = 64 + ((Gear2) ? 96 : 32);
  MrcGetSetStrobe (MrcData, MAX_CONTROLLER, MAX_CHANNEL, MAX_RANK_IN_CHANNEL, MAX_SDRAM_IN_DIMM, TxDqDelay, WriteToCache | PrintValue, &GetSetVal);
  MrcGetSetChStrb (MrcData, MAX_CONTROLLER, MAX_CHANNEL, MAX_SDRAM_IN_DIMM, GsmIocDataDccSkipCRWrite, WriteToCache | PrintValue, &GetSetDis);
  MrcFlushRegisterCachedData (MrcData);
}
/**
  This function configures the Data Invert Nibble feature in the Phy based on the Phy ODT configuration


  @param[in]  MrcData - Pointer to MRC global data.

  @retval MrcStatus - mrcFail if the ODT mode doesn't have a mapping for Data Invert Nibble.
  @retval MrcStatus - mrcSuccess otherwise.
**/
MrcStatus
MrcDataInvertNibble (
  IN  MrcParameters *const MrcData
  )
{
  MrcInput *Inputs;
  MrcOutput *Outputs;
  INT64 GetSetVal;
  INT32 ByteOff;
  INT32 ByteSel;
  UINT32 Controller;
  UINT32 Channel;
  UINT32 Byte;
  UINT32 ByteMax;
  UINT32 SwizOffset;
  UINT32 DataCtrl5Offset;
  UINT8 DniMask;
  UINT8 ByteRead;
  BOOLEAN A0;
  BOOLEAN Lpddr5;
  DATA_CR_DDRCRDATACONTROL5_STRUCT DataControl5;
  DATA_CR_DDRCRWRRETRAINSWIZZLECONTROL_STRUCT WrRetrainSwizzleControl;
  DDRSCRAM_CR_DDRMISCCONTROL7_STRUCT MiscControl7;

  Inputs = &MrcData->Inputs;
  Outputs = &MrcData->Outputs;
  A0 = Inputs->A0;
  ByteMax = (Inputs->UlxUlt) ? 8 : 10;
  Lpddr5 = (Outputs->DdrType == MRC_DDR_TYPE_LPDDR5);
  switch (Outputs->OdtMode) {
    case MrcOdtModeVtt:
      DniMask = (A0) ? 2 : (Lpddr5) ? 3 : 1;
      break;

    case MrcOdtModeVss:
      DniMask = 0;
      break;

    case MrcOdtModeVddq:
      DniMask = (A0) ? 1 : 3;
      break;

    default:
      MRC_DEBUG_MSG (&Outputs->Debug, MSG_LEVEL_ERROR, "Invalid ODT mode for DataInvertNibble: %u\n", Outputs->OdtMode);
      return mrcFail;
  }
  if (Inputs->SafeMode) {
    DniMask = 0;
  }

  // Update the value across the entire phy regardless of population.  Logic won't run on non-populated channels.
  for (Controller = 0; Controller < MAX_CONTROLLER; Controller++) {
    for (Channel = 0; Channel < Outputs->MaxChannels; Channel++) {
      for (Byte = 0; Byte < Outputs->SdramCount; Byte++) {
        GetSetVal = (!A0 && Lpddr5 && ((Byte % 2) == 0)) ? 0 : DniMask;
        MrcGetSetChStrb (MrcData, Controller, Channel, Byte, GsmIocDataInvertNibble, WriteToCache, &GetSetVal);
      }
    }
  }

  if (Inputs->ScramblerSupport) {
    MiscControl7.Data = MrcReadCR (MrcData, DDRSCRAM_CR_DDRMISCCONTROL7_REG);
    MiscControl7.Bits.DataInvertNibble = DniMask;
    MrcWriteCR (MrcData, DDRSCRAM_CR_DDRMISCCONTROL7_REG, MiscControl7.Data);
  }

  if (!A0) {
    for (Channel = 0; Channel < 2; Channel++) {
      for (Byte = 0; Byte < ByteMax; Byte++) {
        SwizOffset = OFFSET_CALC_MC_CH (DATA0CH0_CR_DDRCRWRRETRAINSWIZZLECONTROL_REG, DATA0CH1_CR_DDRCRWRRETRAINSWIZZLECONTROL_REG, Channel, DATA1CH0_CR_DDRCRWRRETRAINSWIZZLECONTROL_REG, Byte);
        WrRetrainSwizzleControl.Data = MrcReadCR (MrcData, SwizOffset);
        if (Outputs->Lpddr) {
          ByteRead = (UINT8) (((Byte % 2) << 1) | WrRetrainSwizzleControl.Bits.DataRetrain_ByteSel);
        } else {
          ByteRead = 0;
        }
        switch (ByteRead) {
          case 1:
            ByteOff = 1;
            break;

          case 3:
            ByteOff = -1;
            break;

          default:
          case 0:
          case 2:
            ByteOff = 0;
            break;
        }
        ByteSel = Byte + ByteOff;

        DataCtrl5Offset = OFFSET_CALC_MC_CH (DATA0CH0_CR_DDRCRDATACONTROL5_REG, DATA0CH1_CR_DDRCRDATACONTROL5_REG, Channel, DATA1CH0_CR_DDRCRDATACONTROL5_REG, ByteSel);
        DataControl5.Data = MrcReadCR (MrcData, DataCtrl5Offset);
        WrRetrainSwizzleControl.Bits.DataInvertNibble = DataControl5.Bits.DataInvertNibble;
        MrcWriteCR (MrcData, SwizOffset, WrRetrainSwizzleControl.Data);
      }
    }
  }

  return mrcSuccess;
}

/**
  DCC Convergence check - Successful convergence is if Status fields return 0x100 (256) +/- 6

  @param[in] MrcData         - Include all MRC global data.
  @param[in] ParamMask       - Bit Mask of parameters to be run.  Bit 0 = DCC_DQ, Bit 1 = DCC_DQS, Bit 2 = DCC_CLK, Bit 3 = DCC_WCK
  @param[in] RankMask        - Which Rank(s) to target
  @param[in] Print           - Whether to print (TRUE) or not (FALSE)

  @retval mrcSuccess if all is converged, mrcFail otherwise.
**/
MrcStatus
MrcDccConvergenceCheck (
  IN MrcParameters *const MrcData,
  IN UINT32               ParamMask,
  IN UINT32               RankMask,
  IN BOOLEAN              Print
  )
{
  MrcOutput   *Outputs;
#ifdef MRC_DEBUG_PRINT
  MrcDebug    *Debug;
#endif
  INT64       GetSetVal;
  UINT32      Controller;
  UINT32      Channel;
  UINT32      Rank;
  UINT32      Byte;
  UINT32      Bit;
  UINT32      GsmReadMode;
  UINT8       MaxByte;
  UINT8       MaxChannels;
  BOOLEAN     TestDq;
  BOOLEAN     TestDqs;
  BOOLEAN     TestClk;
  BOOLEAN     TestWck;
  BOOLEAN     Converged;

  Outputs = &MrcData->Outputs;
  MaxByte = Outputs->SdramCount;
  MaxChannels = Outputs->MaxChannels;
  TestDqs = ParamMask & DCC_DQS_MASK;
  TestDq  = ParamMask & DCC_DQ_MASK;
  TestWck = ParamMask & DCC_WCK_MASK;
  TestClk = ParamMask & DCC_CLK_MASK;
  Converged = TRUE;
  GsmReadMode = (Print) ? PrintValue : 0;
  GsmReadMode |= ReadUncached;
#ifdef MRC_DEBUG_PRINT
  Debug = &Outputs->Debug;
#endif

  for (Rank = 0; Rank < MAX_RANK_IN_CHANNEL; Rank++) {
    if ((RankMask & (1 << Rank)) == 0) {
      continue;
    }
    for (Controller = 0; Controller < MAX_CONTROLLER; Controller++) {
      for (Channel = 0; Channel < MaxChannels; Channel++) {
        if (!MrcRankExist (MrcData, Controller, Channel, Rank)) {
          continue;
        }
        if (TestDq) {
          for (Byte = 0; Byte < MaxByte; Byte++) {
            for (Bit = 0; Bit < MAX_BITS; Bit++) {
              MrcGetSetBit (MrcData, Controller, Channel, Rank, Byte, Bit, GsmIocDataDqDccLaneStatusResult, GsmReadMode, &GetSetVal);
              if ((GetSetVal < (HALF_DUTY_CYCLE - MRC_DCC_CONVERGENCE_RANGE)) || (GetSetVal > (HALF_DUTY_CYCLE + MRC_DCC_CONVERGENCE_RANGE))) {
                Converged = FALSE;
#ifdef MRC_DEBUG_PRINT
                if (Print == FALSE) {
                  //Printing only since this did not converge to show this didnt converge. When Print is not FALSE, all results will be listed regardless of convergence or not
                  MRC_DEBUG_MSG (Debug, MSG_LEVEL_ERROR, "(DQ) Mc%d.Ch%d.R%d.S%d.B%d %d (0x%x) - Failed to converge\n", Controller, Channel, Rank, Byte, Bit, (UINT32) GetSetVal, (UINT32) GetSetVal);
                }
#endif
              }
            }
          }
        }
        if (TestDqs) {
          for (Byte = 0; Byte < MaxByte; Byte++) {
            MrcGetSetBit (MrcData, Controller, Channel, Rank, Byte, MRC_IGNORE_ARG, GsmIocDataDqsDccLaneStatusResult, GsmReadMode, &GetSetVal);
            if ((GetSetVal < (HALF_DUTY_CYCLE - MRC_DCC_CONVERGENCE_RANGE)) || (GetSetVal > (HALF_DUTY_CYCLE + MRC_DCC_CONVERGENCE_RANGE))) {
              Converged = FALSE;
#ifdef MRC_DEBUG_PRINT
              if (Print == FALSE) {
                //Printing only since this did not converge to show this didnt converge. When Print is not FALSE, all results will be listed regardless of convergence or not
                MRC_DEBUG_MSG (Debug, MSG_LEVEL_ERROR, "(DQS) Mc%d.Ch%d.R%d.S%d %d (0x%x) - Failed to converge\n", Controller, Channel, Rank, Byte, (UINT32) GetSetVal, (UINT32) GetSetVal);
              }
#endif
            }
          }
        }
        if (TestClk) {
          MrcGetSetBit (MrcData, Controller, Channel, Rank, MRC_IGNORE_ARG, 2, GsmIocCccDccLaneStatusResult, GsmReadMode, &GetSetVal); //Bit = 2 for Lane5/6 (CLK), Bit = 3 for Lane7/8 (WCK)
          if ((GetSetVal < (HALF_DUTY_CYCLE - MRC_DCC_CONVERGENCE_RANGE)) || (GetSetVal > (HALF_DUTY_CYCLE + MRC_DCC_CONVERGENCE_RANGE))) {
            Converged = FALSE;
#ifdef MRC_DEBUG_PRINT
            if (Print == FALSE) {
              //Printing only since this did not converge to show this didnt converge. When Print is not FALSE, all results will be listed regardless of convergence or not
              MRC_DEBUG_MSG (Debug, MSG_LEVEL_ERROR, "(CLK) Mc%d.Ch%d.R%d %d (0x%x) - Failed to converge\n", Controller, Channel, Rank, (UINT32) GetSetVal, (UINT32) GetSetVal);
            }
#endif
          }
        }
        if (TestWck) {
          MrcGetSetBit (MrcData, Controller, Channel, Rank, MRC_IGNORE_ARG, 3, GsmIocCccDccLaneStatusResult, GsmReadMode, &GetSetVal); //Bit = 2 for Lane5/6 (CLK), Bit = 3 for Lane7/8 (WCK)
          if ((GetSetVal < (HALF_DUTY_CYCLE - MRC_DCC_CONVERGENCE_RANGE)) || (GetSetVal > (HALF_DUTY_CYCLE + MRC_DCC_CONVERGENCE_RANGE))) {
            Converged = FALSE;
#ifdef MRC_DEBUG_PRINT
            if (Print == FALSE) {
              //Printing only since this did not converge to show this didnt converge. When Print is not FALSE, all results will be listed regardless of convergence or not
              MRC_DEBUG_MSG (Debug, MSG_LEVEL_ERROR, "(WCK) Mc%d.Ch%d.R%d %d (0x%x) - Failed to converge\n", Controller, Channel, Rank, (UINT32) GetSetVal, (UINT32) GetSetVal);
            }
#endif
          }
        }
      }
    }
  }
  return (Converged) ? mrcSuccess : mrcFail;
}

/**
  This function returns the target CCC pin parameters for the input Index

  @param[in] Index              - The target group, ony used in for LPDDR5.
                                    0 = CLK
                                    1 = WCK
  @param[out] CccEnFeedbackOut  - GetSet type to use for enabling feedback mode
  @param[out] CccPFallNRiseOut  - GetSet type to use for PFallNRise configuration
  @param[out] CccPRiseNFallOut  - GetSet type to use for PRiseNFall configuration

  @retval mrcSuccess if DCC converges otherwise mrcFail.
**/
void
MrcDccGetTargetCccPins (
  IN  UINT32               Index,
  OUT GSM_GT               *CccEnFeedbackOut,
  OUT GSM_GT               *CccPFallNRiseOut,
  OUT GSM_GT               *CccPRiseNFallOut
  )
{
  GSM_GT CccEnFeedback;
  GSM_GT CccPFallNRise;
  GSM_GT CccPRiseNFall;

  if (Index == 0) {
    CccEnFeedback = GsmIocClkEnFeedback;
    CccPFallNRise = GsmIocClkPFallNRise;
    CccPRiseNFall = GsmIocClkPRiseNFall;
  } else {
    CccEnFeedback = GsmIocWckEnFeedback;
    CccPFallNRise = GsmIocWckPFallNRise;
    CccPRiseNFall = GsmIocWckPRiseNFall;
  }

  if (CccEnFeedbackOut != NULL) {
    *CccEnFeedbackOut = CccEnFeedback;
  }
  if (CccPFallNRiseOut != NULL) {
    *CccPFallNRiseOut = CccPFallNRise;
  }
  if (CccPRiseNFallOut != NULL) {
    *CccPRiseNFallOut = CccPRiseNFall;
  }
}

/**
  This function sets up DCC steps prior to JEDEC init

  @param[in, out] MrcData - Include all MRC global data.

  @retval mrcSuccess if DCC converges otherwise mrcFail.
**/
MrcStatus
MrcDccInitPreJedec (
  IN OUT MrcParameters *const MrcData
  )
{
  MrcInput   *Inputs;
  MrcOutput  *Outputs;
  MrcDebug   *Debug;
  MrcIntOutput  *IntOutputs;
  MrcIntCmdTimingOut *IntCmdTiming;
  const MRC_FUNCTION *MrcCall;
  MrcStatus  Status;
  MrcDdrType  DdrType;
  BOOLEAN    AllTranisitionsFound;
  BOOLEAN    Lpddr;
  BOOLEAN    Lpddr4;
  BOOLEAN    Lpddr5;
  BOOLEAN    Ddr4;
  BOOLEAN    Ddr5;
  BOOLEAN    DtHalo;
  UINT8      Gear2;
  UINT8      ReadIndex;
  UINT8      Index;
  UINT8      IndexEnd;
  UINT8      IndexStart;
  UINT8      RankEnd;
  UINT8      CccIndex;
  UINT8      BitIndex;
  UINT8      BitMask;
  UINT8      SweepIndex;
  UINT8      SweepEnd;
  UINT8      McChBitMask;
  UINT8      MaxChannels;
  UINT8      MaxSdram;
  UINT8      ValidRankMask;
  UINT8      RankMask;
  UINT8      Count;
  UINT8      Data8;
  UINT8      DeskewCal;
  UINT8      FeedbackResult[PN_REPEAT_READS];
  UINT8      PrevFeedbackResult[MAX_CONTROLLER][MAX_CHANNEL][MAX_SDRAM_IN_DIMM];
  UINT8      TransitPoint[MAX_CONTROLLER][MAX_CHANNEL][MAX_SDRAM_IN_DIMM][MAX_TCO_FEEDBACK_BITS];
  UINT16     *Data16Ptr;
  UINT16     Qclkps;
  UINT16     Data16;
  UINT16     PBDStepSizeTicks;
  UINT16     Denominator16;
  UINT16     DataPFallNRise;
  UINT16     DataPRiseNFall;
  UINT16     DccLaneStatusResultAvg[MAX_CONTROLLER][MAX_CHANNEL][MAX_SDRAM_IN_DIMM][MAX_EDGES];
  UINT32     GetSetRankParam;
  UINT32     Data32;
  UINT32     FirstController;
  UINT32     FirstCccChannel;
  UINT32     Controller;
  UINT32     Channel;
  UINT32     ChannelArrayIndex;
  UINT32     Rank;
  UINT32     RankCount;
  UINT32     Byte;
  UINT32     Bit;
  UINT32     Offset;
  UINT32     GsmMode;
  UINT32     OrigCCCDccFsmCtl[MRC_NUM_CCC_INSTANCES];
  INT64      GetSetEn;
  INT64      GetSetDis;
  INT64      GetSetVal;
  INT64      GetSetVal2;
  INT64      GetSetTot;
  INT64      GetSetValPFallNRise;
  INT64      GetSetValPRiseNFall;
  INT64      GetSetStepSize;
  INT64      GetSetMin;
  INT64      GetSetMax;
  INT64      DataTrainFeedbackField;
  GSM_GT     CccEnFeedback;
  GSM_GT     CccPFallNRise;
  GSM_GT     CccPRiseNFall;
  GSM_GT     CccPerBit;
  MC0_REQ0_CR_CPGC_SEQ_CTL_STRUCT  CpgcStartTest;
  MC0_REQ0_CR_CPGC_SEQ_CTL_STRUCT  CpgcStopTest;
  CH0CCC_CR_DCCFSMCONTROL_STRUCT   CCCDccFsmCtl;
  CH0CCC_CR_DDRCRCCCVOLTAGEUSED_STRUCT  CCCVoltageUsed;

  IntOutputs    = (MrcIntOutput *) (MrcData->IntOutputs.Internal);
  Inputs        = &MrcData->Inputs;
  MrcCall       = Inputs->Call.Func;
  Outputs       = &MrcData->Outputs;
  Debug         = &Outputs->Debug;
  MaxChannels   = Outputs->MaxChannels;
  MaxSdram      = Outputs->SdramCount;
  ValidRankMask = Outputs->ValidRankMask;
  Qclkps        = Outputs->Qclkps;
  DdrType       = Outputs->DdrType;
  Lpddr         = Outputs->Lpddr;
  Lpddr4        = (DdrType == MRC_DDR_TYPE_LPDDR4);
  Lpddr5        = (DdrType == MRC_DDR_TYPE_LPDDR5);
  Ddr4          = (DdrType == MRC_DDR_TYPE_DDR4);
  Ddr5          = (DdrType == MRC_DDR_TYPE_DDR5);
  RankEnd       = (Ddr4 || Ddr5) ? MAX_RANK_IN_CHANNEL : 1;
  Gear2         = (Outputs->Gear2) ? 1 : 0;
  GetSetEn      = 1;
  GetSetDis     = 0;
  DtHalo        = Inputs->DtHalo;

  CpgcStartTest.Data = CpgcStopTest.Data = 0;
  CpgcStartTest.Bits.START_TEST = 1;
  CpgcStartTest.Bits.CLEAR_ERRORS = 1;
  CpgcStopTest.Bits.STOP_TEST = 1;
  IntOutputs->SkipZq = TRUE;
  FirstController = (MrcControllerExist (MrcData, cCONTROLLER0)) ? 0 : 1;
  FirstCccChannel = (FirstController * MAX_CHANNEL) + Outputs->Controller[FirstController].FirstPopCh;;

  if (!Inputs->UlxUlt) {
    MRC_DEBUG_MSG (Debug, MSG_LEVEL_ALGO, "\n%sDQS Rise/Fall\n", gStartTagStr);
    MrcGetSetChStrb (MrcData, MAX_CONTROLLER, MAX_CHANNEL, MAX_SDRAM_IN_DIMM, GsmIocDqsRFMode, WriteNoCache, &GetSetEn);
    MrcGetSetLimits (MrcData, TxDqsTcoPFallNRise, NULL, &GetSetMax, NULL);
    SweepEnd = (UINT8) (GetSetMax);

    for (Rank = 0; Rank < MAX_RANK_IN_CHANNEL; Rank++) {
      RankMask = 1 << Rank;
      if ((RankMask & ValidRankMask) == 0) {
        continue;
      }
      McChBitMask = 0;
      for (Controller = 0; Controller < MAX_CONTROLLER; Controller++) {
        for (Channel = 0; Channel < MaxChannels; Channel++) {
          //@todo H/S needs function adjustment for different populated rank across channels
          McChBitMask |= SelectReutRanks (MrcData, (UINT8) Controller, (UINT8) Channel, RankMask, FALSE, 0);
        }
      }
      MRC_DEBUG_MSG (Debug, MSG_LEVEL_ALGO, "\nRank %d\n", Rank);
      SetupIOTestEndless (MrcData, McChBitMask, 10, NSOE, 0, 0, PatWrEndless);


      MrcCall->MrcSetMem ((UINT8 *) PrevFeedbackResult, sizeof (PrevFeedbackResult), 0xFF);
      MrcCall->MrcSetMem ((UINT8 *) TransitPoint, sizeof (TransitPoint), 0xFF);

      for (SweepIndex = 0; SweepIndex <= SweepEnd; SweepIndex++) {
        GetSetValPFallNRise = GetSetMax - SweepIndex;
        GetSetValPRiseNFall = SweepIndex;
        MRC_DEBUG_MSG (Debug, MSG_LEVEL_ALGO, " SweepIndex = 0x%x, GetSetValPFallNRise = 0x%x, GetSetValPRiseNFall = 0x%x\n", SweepIndex, (UINT32) GetSetValPFallNRise, (UINT32) GetSetValPRiseNFall);

        //Check if all transition points have been found for this rank
        AllTranisitionsFound = TRUE;
        for (Controller = 0; Controller < MAX_CONTROLLER; Controller++) {
          for (Channel = 0; Channel < MaxChannels; Channel++) {
            if (!MrcRankExist (MrcData, Controller, Channel, Rank)) {
              continue;
            }
            for (Byte = 0; Byte < MaxSdram; Byte++) {
              for (BitIndex = 0; BitIndex < MAX_TCO_FEEDBACK_BITS; BitIndex++) {
                if (TransitPoint[Controller][Channel][Byte][BitIndex] == 0xFF) {
                  AllTranisitionsFound = FALSE;
                  //For loop exit conditions
                  Controller = MAX_CONTROLLER;
                  Channel = MaxChannels;
                  Byte = MaxSdram;
                  BitIndex = MAX_TCO_FEEDBACK_BITS;
                }
              }
            }
          }
        }
        if (AllTranisitionsFound) {
          MRC_DEBUG_MSG (Debug, MSG_LEVEL_ALGO, " All Transitions Found for this Rank\n");
          break;
        }

        MrcGetSetStrobe (MrcData, MAX_CONTROLLER, MAX_CHANNEL, Rank, MAX_SDRAM_IN_DIMM, TxDqsTcoPFallNRise, WriteToCache, &GetSetValPFallNRise);
        MrcGetSetStrobe (MrcData, MAX_CONTROLLER, MAX_CHANNEL, Rank, MAX_SDRAM_IN_DIMM, TxDqsTcoPRiseNFall, WriteToCache, &GetSetValPRiseNFall);
        MrcFlushRegisterCachedData (MrcData);

        // Propagate new value
        MrcGetSetNoScope (MrcData, GsmIocForceCmpUpdt, ForceWriteUncached | PrintValue, &GetSetEn);

        Cpgc20ControlRegWrite (MrcData, McChBitMask, CpgcStartTest);
        for (Controller = 0; Controller < MAX_CONTROLLER; Controller++) {
          for (Channel = 0; Channel < MaxChannels; Channel++) {
            if (!MrcRankExist (MrcData, Controller, Channel, Rank)) {
              continue;
            }
            for (Byte = 0; Byte < MaxSdram; Byte++) {
              //Record multiple status
              for (ReadIndex = 0; ReadIndex < PN_REPEAT_READS; ReadIndex++) {
                MrcGetSetChStrb (MrcData, Controller, Channel, Byte, GsmIocDataTrainFeedback, ReadUncached, &DataTrainFeedbackField);
                FeedbackResult[ReadIndex] = (UINT8) (DataTrainFeedbackField & TCO_FEEDBACK_MASK);
              }

              //Round the FeebackResult and store data in Array Index = 0
              for (BitIndex = 0; BitIndex < MAX_TCO_FEEDBACK_BITS; BitIndex++) {
                BitMask = 1 << BitIndex;
                Count = 0;
                for (ReadIndex = 0; ReadIndex < PN_REPEAT_READS; ReadIndex++) {
                  Count += (FeedbackResult[ReadIndex] & BitMask) ? 1 : 0;
                }
                if (Count > (PN_REPEAT_READS / 2)) {
                  FeedbackResult[0] |= BitMask;
                } else {
                  FeedbackResult[0] &= ~BitMask;
                }
              }

              MRC_DEBUG_MSG (Debug, MSG_LEVEL_ALGO, "  MC%d.C%d.B%d Feedback = 0x%x\n", Controller, Channel, Byte, FeedbackResult[0]);
              if (SweepIndex == 0) {
                PrevFeedbackResult[Controller][Channel][Byte] = FeedbackResult[0];
              } else {
                for (BitIndex = 0; BitIndex < MAX_TCO_FEEDBACK_BITS; BitIndex++) {
                  BitMask = 1 << BitIndex;
                  if ((PrevFeedbackResult[Controller][Channel][Byte] & BitMask) != (FeedbackResult[0] & BitMask)) {
                    MRC_DEBUG_MSG (Debug, MSG_LEVEL_ALGO, "  MC%d.C%d.B%d - Index %d: Delta vs Original\n", Controller, Channel, Byte, BitIndex);
                    if (TransitPoint[Controller][Channel][Byte][BitIndex] == 0xFF) {
                      TransitPoint[Controller][Channel][Byte][BitIndex] = SweepIndex;
                    }
                  }
                }
              }
            } // For Byte
          } // For Channel
        } // for Controller
        Cpgc20ControlRegWrite (MrcData, McChBitMask, CpgcStopTest);
      } // for SweepIndex

      MRC_DEBUG_MSG (Debug, MSG_LEVEL_ALGO, "TransitPoints - Data\n");
      for (Controller = 0; Controller < MAX_CONTROLLER; Controller++) {
        for (Channel = 0; Channel < MaxChannels; Channel++) {
          if (!MrcRankExist (MrcData, Controller, Channel, Rank)) {
            continue;
          }
          for (Byte = 0; Byte < MaxSdram; Byte++) {
            MRC_DEBUG_MSG (Debug, MSG_LEVEL_ALGO, " MC%d.C%d.B%d\n", Controller, Channel, Byte);
            for (BitIndex = 0; BitIndex < MAX_TCO_FEEDBACK_BITS; BitIndex++) {
              MRC_DEBUG_MSG (Debug, MSG_LEVEL_ALGO, "  BitIndex%d = %d", BitIndex, TransitPoint[Controller][Channel][Byte][BitIndex]);
              if (TransitPoint[Controller][Channel][Byte][BitIndex] > SweepEnd) {
                TransitPoint[Controller][Channel][Byte][BitIndex] = SweepEnd / 2;
                MRC_DEBUG_MSG (Debug, MSG_LEVEL_ALGO, " - Adjusted to %d", TransitPoint[Controller][Channel][Byte][BitIndex]);
              }
              MRC_DEBUG_MSG (Debug, MSG_LEVEL_ALGO, "\n");
            }
          }
        }
      }

      //Average DqsPrNfDelay using BitIndex = 0 + BitIndex = 1 and average DqsPfNrDelay using BitIndex = 2 + BitIndex = 3
      for (Controller = 0; Controller < MAX_CONTROLLER; Controller++) {
        for (Channel = 0; Channel < MaxChannels; Channel++) {
          if (!MrcRankExist (MrcData, Controller, Channel, Rank)) {
            continue;
          }
          for (Byte = 0; Byte < MaxSdram; Byte++) {
            SweepIndex = TransitPoint[Controller][Channel][Byte][0] + TransitPoint[Controller][Channel][Byte][1];
            SweepIndex /= 2;
            GetSetValPRiseNFall = SweepIndex;

            SweepIndex = TransitPoint[Controller][Channel][Byte][2] + TransitPoint[Controller][Channel][Byte][3];
            SweepIndex /= 2;
            GetSetValPFallNRise = GetSetMax - SweepIndex;
            MrcGetSetStrobe (MrcData, Controller, Channel, Rank, Byte, TxDqsTcoPFallNRise, WriteToCache | PrintValue, &GetSetValPFallNRise);
            MrcGetSetStrobe (MrcData, Controller, Channel, Rank, Byte, TxDqsTcoPRiseNFall, WriteToCache | PrintValue, &GetSetValPRiseNFall);
          }
        }
      }
      MrcFlushRegisterCachedData (MrcData);
    } // for Rank

    if (Inputs->UlxUlt || (Ddr4 && Gear2)) {
      for (Controller = 0; Controller < MAX_CONTROLLER; Controller++) {
        for (Channel = 0; Channel < MaxChannels; Channel++) {
          for (Byte = 0; Byte < MaxSdram; Byte++) {
            //Initialize variables
            DataPFallNRise = DataPRiseNFall = Data16 = 0;
            for (Rank = 0; Rank < MAX_RANK_IN_CHANNEL; Rank++) {
              if (!MrcRankExist (MrcData, Controller, Channel, Rank)) {
                continue;
              }
              Data16++; // Ranks populated
              MrcGetSetStrobe (MrcData, Controller, Channel, Rank, Byte, TxDqsTcoPFallNRise, ReadFromCache, &GetSetValPFallNRise);
              MrcGetSetStrobe (MrcData, Controller, Channel, Rank, Byte, TxDqsTcoPRiseNFall, ReadFromCache, &GetSetValPRiseNFall);
              DataPFallNRise += (UINT16) GetSetValPFallNRise;
              DataPRiseNFall += (UINT16) GetSetValPRiseNFall;
            } // For Rank
            if (Data16 > 1) {
              // More than 1 Rank populated
              GetSetValPFallNRise = DataPFallNRise / Data16;
              GetSetValPRiseNFall = DataPRiseNFall / Data16;
              MrcGetSetStrobe (MrcData, Controller, Channel, MAX_RANK_IN_CHANNEL, Byte, TxDqsTcoPFallNRise, WriteToCache | PrintValue, &GetSetValPFallNRise);
              MrcGetSetStrobe (MrcData, Controller, Channel, MAX_RANK_IN_CHANNEL, Byte, TxDqsTcoPRiseNFall, WriteToCache | PrintValue, &GetSetValPRiseNFall);
            }
          } // For Byte
        } // For Channel
      } // For Controller
      MrcFlushRegisterCachedData (MrcData);
    }

    // Propagate new value
    MrcGetSetNoScope (MrcData, GsmIocForceCmpUpdt, ForceWriteUncached | PrintValue, &GetSetEn);
    MRC_DEBUG_MSG (Debug, MSG_LEVEL_ALGO, "\n%sDQS Rise/Fall\n", gStopTagStr);

    MrcGetSetChStrb (MrcData, MAX_CONTROLLER, MAX_CHANNEL, MAX_SDRAM_IN_DIMM, GsmIocDqsRFMode, WriteNoCache | PrintValue, &GetSetDis);
  }

  for (CccIndex = 0; CccIndex < MRC_NUM_CCC_INSTANCES; CccIndex++) {
    Offset = OFFSET_CALC_CH (CH0CCC_CR_DCCFSMCONTROL_REG, CH1CCC_CR_DCCFSMCONTROL_REG, CccIndex);
    CCCDccFsmCtl.Data = MrcReadCR (MrcData, Offset);
    MRC_DEBUG_MSG (Debug, MSG_LEVEL_ALGO, "OrigCccDccFsmCtl[%d] = 0x%x\n", CccIndex, CCCDccFsmCtl.Data);
    //Make sure these are set for Orig array
    CCCDccFsmCtl.Bits.MeasPoint = 0;
    CCCDccFsmCtl.Bits.UpdateTcoComp = 0;
    OrigCCCDccFsmCtl[CccIndex] = CCCDccFsmCtl.Data;
    MRC_DEBUG_MSG (Debug, MSG_LEVEL_ALGO, "Adjusted OrigCccDccFsmCtl[%d] = 0x%x\n", CccIndex, OrigCCCDccFsmCtl[CccIndex]);
  }

  MrcCall->MrcSetMem ((UINT8 *) PrevFeedbackResult, sizeof (PrevFeedbackResult), 0xFF);
  MrcCall->MrcSetMem ((UINT8 *) TransitPoint, sizeof (TransitPoint), 0xFF);

  for (CccIndex = 0; CccIndex < MRC_NUM_CCC_INSTANCES; CccIndex++) {
    Offset = OFFSET_CALC_CH (CH0CCC_CR_DCCFSMCONTROL_REG, CH1CCC_CR_DCCFSMCONTROL_REG, CccIndex);
    CCCDccFsmCtl.Data = OrigCCCDccFsmCtl[CccIndex];

    //Enable clocks to toggle
    CCCDccFsmCtl.Bits.MeasPoint = 2;
    MrcWriteCR (MrcData, Offset, CCCDccFsmCtl.Data);
    MRC_DEBUG_MSG (Debug, MSG_LEVEL_ALGO, "CccDccFsmCtl[%d] = 0x%x\n", CccIndex, CCCDccFsmCtl.Data);
  }

  IndexEnd = (Lpddr5) ? 2 : 1;
  GetSetRankParam = (Lpddr) ? MRC_IGNORE_ARG : MAX_RANK_IN_CHANNEL;
  MrcGetSetLimits (MrcData, GsmIocClkPFallNRise, &GetSetMin, &GetSetMax, NULL);
  SweepEnd = (UINT8) (GetSetMax - GetSetMin);

  for (Index = 0; Index < IndexEnd; Index++) {
    MRC_DEBUG_MSG (Debug, MSG_LEVEL_ALGO, "\n%sCLK Rise/Fall - Index %d (U/Y:0 = CCC56, 1 = CCC78)\n", gStartTagStr, Index);
    for (SweepIndex = 0; SweepIndex <= SweepEnd; SweepIndex++) {
      MRC_DEBUG_MSG (Debug, MSG_LEVEL_ALGO, " SweepIndex = 0x%x\n", SweepIndex);
      GetSetValPRiseNFall = GetSetMin;
      GetSetValPFallNRise = GetSetMax;
      GetSetValPRiseNFall += SweepIndex;
      GetSetValPFallNRise -= SweepIndex;

      //Check if all transition points have been found for this CCC
      AllTranisitionsFound = TRUE;
      for (Controller = 0; Controller < MAX_CONTROLLER; Controller++) {
        for (Channel = 0; Channel < MaxChannels; Channel++) {
          for (Rank = 0; Rank < RankEnd; Rank++) {
            if (!MrcRankExist (MrcData, Controller, Channel, Rank)) {
              continue;
            }
            ChannelArrayIndex = Channel + (Rank * MaxChannels);
            for (BitIndex = 0; BitIndex < MAX_TCO_FEEDBACK_BITS; BitIndex++) {
              if (TransitPoint[Controller][ChannelArrayIndex][Index][BitIndex] == 0xFF) {
                AllTranisitionsFound = FALSE;
                //For loop exit conditions
                Controller = MAX_CONTROLLER;
                Channel = MAX_CHANNEL;
                Rank = RankEnd;
                BitIndex = MAX_TCO_FEEDBACK_BITS;
              }
            }
          }
        }
      }
      if (AllTranisitionsFound) {
        MRC_DEBUG_MSG (Debug, MSG_LEVEL_ALGO, " All Transitions Found for Index %d (0 = CCC56, 1 = CCC78)\n", Index);
        break;
      }

      for (Rank = 0; Rank < RankEnd; Rank++) {
        MrcDccGetTargetCccPins (
          Index,
          &CccEnFeedback,
          &CccPFallNRise,
          &CccPRiseNFall
          );
        // Configure the feedback enable for the target CCC pins
        // First disable the training mode on all ranks
        GetSetVal  = 0;
        MrcGetSetCcc (MrcData, MAX_CONTROLLER, MAX_CHANNEL, GetSetRankParam, 0, CccEnFeedback, WriteNoCache, &GetSetVal);

        // Enable the training mode on the selected rank(s)
        GetSetVal = MRC_ENABLE;
        MrcGetSetCcc (MrcData, Controller, Channel, (DtHalo ? Rank : GetSetRankParam), 0, CccEnFeedback, WriteNoCache, &GetSetVal);

        MrcGetSetCcc (MrcData, MAX_CONTROLLER, MAX_CHANNEL, (DtHalo ? Rank : GetSetRankParam), 0, CccPFallNRise, WriteNoCache | PrintValue, &GetSetValPFallNRise);
        MrcGetSetCcc (MrcData, MAX_CONTROLLER, MAX_CHANNEL, (DtHalo ? Rank : GetSetRankParam), 0, CccPRiseNFall, WriteNoCache | PrintValue, &GetSetValPRiseNFall);
        // Propagate new value
        MrcGetSetNoScope (MrcData, GsmIocForceCmpUpdt, ForceWriteUncached | PrintValue, &GetSetEn);
        ForceRcomp (MrcData);

        //Setup Dummy Traffic
        McChBitMask = 0;
        RankMask = 1 << Rank;
        for (Controller = 0; Controller < MAX_CONTROLLER; Controller++) {
          for (Channel = 0; Channel < MaxChannels; Channel++) {
            McChBitMask |= SelectReutRanks (MrcData, (UINT8) Controller, (UINT8) Channel, RankMask, FALSE, 0);
          }
        }
        SetupIOTestEndless (MrcData, McChBitMask, 10, NSOE, 0, 0, PatWrEndless);
        //Run Dummy Traffic
        Cpgc20ControlRegWrite (MrcData, McChBitMask, CpgcStartTest);

        //Read Status
        for (Controller = 0; Controller < MAX_CONTROLLER; Controller++) {
          for (Channel = 0; Channel < MaxChannels; Channel++) {
            if (!MrcRankExist (MrcData, Controller, Channel, Rank)) {
              continue;
            }
            ChannelArrayIndex = Channel + (Rank * MaxChannels);
            //Record multiple status
            for (ReadIndex = 0; ReadIndex < PN_REPEAT_READS; ReadIndex++) {
              MrcGetSetCcc (MrcData, Controller, Channel, Rank, 0, GsmIocClkPRiseNFallFeedback, ReadUncached, &GetSetVal);
              MrcGetSetCcc (MrcData, Controller, Channel, Rank, 0, GsmIocClkPFallNRiseFeedback, ReadUncached, &DataTrainFeedbackField);
              //Aggregate the Feedback results into one with PRiseNFall as the lower bits and PFallNRise as the upper bits
              FeedbackResult[ReadIndex] = (UINT8) (GetSetVal & (TCO_FEEDBACK_MASK >> CH0CCC_CR_DDRCRCCCPERBITDESKEWPRISENFALL_PrNfFeedback_WID));
              Data8 = (UINT8) (DataTrainFeedbackField & (TCO_FEEDBACK_MASK >> CH0CCC_CR_DDRCRCCCPERBITDESKEWPRISENFALL_PrNfFeedback_WID));
              FeedbackResult[ReadIndex] |= (Data8 << CH0CCC_CR_DDRCRCCCPERBITDESKEWPRISENFALL_PrNfFeedback_WID);
            }

            //Round the FeebackResult and store data in Array Index = 0
            for (BitIndex = 0; BitIndex < (CH0CCC_CR_DDRCRCCCPERBITDESKEWPRISENFALL_PrNfFeedback_WID * 2); BitIndex++) {
              BitMask = 1 << BitIndex;
              Count = 0;
              for (ReadIndex = 0; ReadIndex < PN_REPEAT_READS; ReadIndex++) {
                Count += (FeedbackResult[ReadIndex] & BitMask) ? 1 : 0;
              }
              if (Count > (PN_REPEAT_READS / 2)) {
                FeedbackResult[0] |= BitMask;
              } else {
                FeedbackResult[0] &= ~BitMask;
              }
            }
            MRC_DEBUG_MSG (Debug, MSG_LEVEL_ALGO, "  MC%d.C%d.R%d Feedback = 0x%x\n", Controller, Channel, Rank, FeedbackResult[0]);
            if (SweepIndex == 0) {
              PrevFeedbackResult[Controller][ChannelArrayIndex][Index] = FeedbackResult[0];
            } else {
              for (BitIndex = 0; BitIndex < MAX_TCO_FEEDBACK_BITS; BitIndex++) {
                BitMask = 1 << BitIndex;
                if ((PrevFeedbackResult[Controller][ChannelArrayIndex][Index] & BitMask) == 0) {
                  if ((FeedbackResult[0] & BitMask) == 0) {
                    //No setting has detected an "1" response
                  } else {
                    //Current setting has detected an "1" response, update the PrevFeedback
                    PrevFeedbackResult[Controller][ChannelArrayIndex][Index] |= BitMask;
                    MRC_DEBUG_MSG (Debug, MSG_LEVEL_ALGO, "  MC%d.C%d.R%d - Index %d: Original Updated\n", Controller, Channel, Rank, BitIndex);
                  }
                  //Ignore current result for determining TransitPoint
                  continue;
                }
                if ((PrevFeedbackResult[Controller][ChannelArrayIndex][Index] & BitMask) != (FeedbackResult[0] & BitMask)) {
                  MRC_DEBUG_MSG (Debug, MSG_LEVEL_ALGO, "  MC%d.C%d.R%d - Index %d: Delta vs Original\n", Controller, Channel, Rank, BitIndex);
                  if (TransitPoint[Controller][ChannelArrayIndex][Index][BitIndex] == 0xFF) {
                    TransitPoint[Controller][ChannelArrayIndex][Index][BitIndex] = SweepIndex;
                  }
                }
              }
            }
          } // for Channel
        } // for Controller
        //Stop Dummy Traffic
        Cpgc20ControlRegWrite (MrcData, McChBitMask, CpgcStopTest);
      } // for Rank
    } // for SweepIndex
  } // for Index (CCC56 and CCC78)

#ifdef MRC_DEBUG_PRINT
  MRC_DEBUG_MSG (Debug, MSG_LEVEL_ALGO, "TransitPoints - CLK\n");
  for (Controller = 0; Controller < MAX_CONTROLLER; Controller++) {
    for (Channel = 0; Channel < MaxChannels; Channel++) {
      for (Rank = 0; Rank < RankEnd; Rank++) {
        if (!MrcRankExist (MrcData, Controller, Channel, Rank)) {
          continue;
        }
        ChannelArrayIndex = Channel + (Rank * MaxChannels);
        for (Index = 0; Index < IndexEnd; Index++) {
          MRC_DEBUG_MSG (Debug, MSG_LEVEL_ALGO, " MC%d.C%d.R%d.I%d\n", Controller, Channel, Rank, Index);
          for (BitIndex = 0; BitIndex < MAX_TCO_FEEDBACK_BITS; BitIndex++) {
            MRC_DEBUG_MSG (Debug, MSG_LEVEL_ALGO, "  BitIndex%d = %d", BitIndex, TransitPoint[Controller][ChannelArrayIndex][Index][BitIndex]);
            if (TransitPoint[Controller][ChannelArrayIndex][Index][BitIndex] > SweepEnd) {
              TransitPoint[Controller][ChannelArrayIndex][Index][BitIndex] = SweepEnd / 2;
              MRC_DEBUG_MSG (Debug, MSG_LEVEL_ALGO, " - Adjusted to %d", TransitPoint[Controller][ChannelArrayIndex][Index][BitIndex]);
            }
            MRC_DEBUG_MSG (Debug, MSG_LEVEL_ALGO, "\n");
          }
        }
      } // for Rank
    } // for Channel
  } // for Controller
#endif

  //Average DqsPrNfDelay using BitIndex = 0 + BitIndex = 1 and average DqsPfNrDelay using BitIndex = 2 + BitIndex = 3
  for (Index = 0; Index < IndexEnd; Index++) {
    for (Controller = 0; Controller < MAX_CONTROLLER; Controller++) {
      for (Channel = 0; Channel < MaxChannels; Channel++) {
        for (Rank = 0; Rank < RankEnd; Rank++) {
          if (!MrcRankExist (MrcData, Controller, Channel, Rank)) {
            continue;
          }
          MrcDccGetTargetCccPins (
            Index,
            NULL,
            &CccPFallNRise,
            &CccPRiseNFall
            );
          ChannelArrayIndex = Channel + (Rank * MaxChannels);
          GetSetValPRiseNFall = GetSetMin;
          SweepIndex = TransitPoint[Controller][ChannelArrayIndex][Index][0] + TransitPoint[Controller][ChannelArrayIndex][Index][1];
          SweepIndex /= 2;
          GetSetValPRiseNFall += SweepIndex;

          GetSetValPFallNRise = GetSetMax;
          SweepIndex = TransitPoint[Controller][ChannelArrayIndex][Index][2] + TransitPoint[Controller][ChannelArrayIndex][Index][3];
          SweepIndex /= 2;
          GetSetValPFallNRise -= SweepIndex;

          MrcGetSetCcc (MrcData, Controller, Channel, Rank, 0, CccPFallNRise, WriteNoCache | PrintValue, &GetSetValPFallNRise);
          MrcGetSetCcc (MrcData, Controller, Channel, Rank, 0, CccPRiseNFall, WriteNoCache | PrintValue, &GetSetValPRiseNFall);
        }
      }
    }
  }// for Index
  MRC_DEBUG_MSG (Debug, MSG_LEVEL_ALGO, "\n%sCLK Rise/Fall - Index %d (0 = CCC56, 1 = CCC78)\n", gStopTagStr, Index);

  MrcGetSetCcc (MrcData, MAX_CONTROLLER, MAX_CHANNEL, GetSetRankParam, 0, GsmIocClkEnFeedback, WriteNoCache, &GetSetDis);
  if (Lpddr5) {
    MrcGetSetCcc (MrcData, MAX_CONTROLLER, MAX_CHANNEL, GetSetRankParam, 0, GsmIocWckEnFeedback, WriteNoCache, &GetSetDis);
  }
  // Propagate new value
  MrcGetSetNoScope (MrcData, GsmIocForceCmpUpdt, ForceWriteUncached | PrintValue, &GetSetEn);

  MrcGetSetMc (MrcData, MAX_CONTROLLER, GsmMccEnableDclk, WriteNoCache, &GetSetDis);
  Offset = OFFSET_CALC_CH (CH0CCC_CR_DDRCRCCCVOLTAGEUSED_REG, CH1CCC_CR_DDRCRCCCVOLTAGEUSED_REG, FirstCccChannel);
  CCCVoltageUsed.Data = MrcReadCR (MrcData, Offset);
  CCCVoltageUsed.Bits.DisWckPupDcc = 2;
  MrcWriteCrMulticast (MrcData, CCC_CR_DDRCRCCCVOLTAGEUSED_REG, CCCVoltageUsed.Data);

  MRC_DEBUG_MSG (Debug, MSG_LEVEL_ALGO, "\n%sDccStepSize - Data\n", gStartTagStr);
  MrcCall->MrcSetMem ((UINT8 *) DccLaneStatusResultAvg, sizeof (DccLaneStatusResultAvg), 0);

  //Setup for Comp1
  GetSetVal = 0x20;
  MrcGetSetBit (MrcData, MAX_CONTROLLER, MAX_CHANNEL, MAX_RANK_IN_CHANNEL, MAX_SDRAM_IN_DIMM, MAX_BITS, TxDqBitDelay, WriteToCache | GSM_UPDATE_HOST | PrintValue, &GetSetVal);
  GetSetVal = 16;
  MrcGetSetBit (MrcData, MAX_CONTROLLER, MAX_CHANNEL, MAX_RANK_IN_CHANNEL, MAX_SDRAM_IN_DIMM, MAX_BITS, TxDqDccOffset, WriteToCache | PrintValue, &GetSetVal);
  MrcGetSetChStrb (MrcData, MAX_CONTROLLER, MAX_CHANNEL, MAX_SDRAM_IN_DIMM, GsmIocDataDccMeasPoint, WriteToCache | PrintValue, &GetSetDis);
  MrcGetSetChStrb (MrcData, MAX_CONTROLLER, MAX_CHANNEL, MAX_SDRAM_IN_DIMM, GsmIocDataDccRankEn, WriteToCache | PrintValue, &GetSetEn);
  MrcGetSetChStrb (MrcData, MAX_CONTROLLER, MAX_CHANNEL, MAX_SDRAM_IN_DIMM, GsmIocDataDccSaveFullDcc, WriteToCache | PrintValue, &GetSetDis);
  MrcGetSetChStrb (MrcData, MAX_CONTROLLER, MAX_CHANNEL, MAX_SDRAM_IN_DIMM, GsmIocDataDccSkipCRWrite, WriteToCache | PrintValue, &GetSetEn);
  MrcFlushRegisterCachedData (MrcData);

  // Propagate new value1
  MrcGetSetNoScope (MrcData, GsmIocForceCmpUpdt, ForceWriteUncached | PrintValue, &GetSetEn);
  ForceRcomp (MrcData);

  for (Controller = 0; Controller < MAX_CONTROLLER; Controller++) {
    for (Channel = 0; Channel < MaxChannels; Channel++) {
      if (!MrcChannelExist (MrcData, Controller, Channel)) {
        continue;
      }
      for (Byte = 0; Byte < MaxSdram; Byte++) {
        Data16Ptr = &DccLaneStatusResultAvg[Controller][Channel][Byte][0];
        for (Bit = 0; Bit < MAX_BITS; Bit++) {
          //Only read the CR when needed (each register holds 3 bits), otherwise read from cache
          GsmMode = (Bit % 3) ? ReadFromCache : ReadCached;
          MrcGetSetBit (MrcData, Controller, Channel, MRC_IGNORE_ARG, Byte, Bit, GsmIocDataDqDccLaneStatusResult, GsmMode, &GetSetVal);
          *Data16Ptr += (UINT16) GetSetVal;
        }
        *Data16Ptr /= MAX_BITS;
      }
    }
  }

  //Setup for Comp2, Setup of TxDqBitDelay to 0x20 skipped since done as Comp1 and no change
  GetSetVal = -16;
  MrcGetSetBit (MrcData, MAX_CONTROLLER, MAX_CHANNEL, MAX_RANK_IN_CHANNEL, MAX_SDRAM_IN_DIMM, MAX_BITS, TxDqDccOffset, WriteToCache | PrintValue, &GetSetVal);
  MrcFlushRegisterCachedData (MrcData);

  // Propagate new value1
  MrcGetSetNoScope (MrcData, GsmIocForceCmpUpdt, ForceWriteUncached | PrintValue, &GetSetEn);
  ForceRcomp (MrcData);

  for (Controller = 0; Controller < MAX_CONTROLLER; Controller++) {
    for (Channel = 0; Channel < MaxChannels; Channel++) {
      if (!MrcChannelExist (MrcData, Controller, Channel)) {
        continue;
      }
      for (Byte = 0; Byte < MaxSdram; Byte++) {
        Data16Ptr = &DccLaneStatusResultAvg[Controller][Channel][Byte][1];
        for (Bit = 0; Bit < MAX_BITS; Bit++) {
          //Only read the CR when needed (each register holds 3 bits), otherwise read from cache
          GsmMode = (Bit % 3) ? ReadFromCache : ReadCached;
          MrcGetSetBit (MrcData, Controller, Channel, MRC_IGNORE_ARG, Byte, Bit, GsmIocDataDqDccLaneStatusResult, GsmMode, &GetSetVal);
          *Data16Ptr += (UINT16) GetSetVal;
        }
        *Data16Ptr /= MAX_BITS;
      }
    }
  }

#ifdef MRC_DEBUG_PRINT
  MRC_DEBUG_MSG (Debug, MSG_LEVEL_ALGO, "DccLaneStatusResultAvg - Data\n");
  for (Controller = 0; Controller < MAX_CONTROLLER; Controller++) {
    for (Channel = 0; Channel < MaxChannels; Channel++) {
      if (!MrcChannelExist (MrcData, Controller, Channel)) {
        continue;
      }
      for (Byte = 0; Byte < MaxSdram; Byte++) {
        MRC_DEBUG_MSG (Debug, MSG_LEVEL_ALGO, " MC%d.C%d.S%d - Comp1=%d, Comp2=%d\n", Controller, Channel, Byte, DccLaneStatusResultAvg[Controller][Channel][Byte][0], DccLaneStatusResultAvg[Controller][Channel][Byte][1]);
      }
    }
  }
#endif

  //DccStepSize determination (Data)
  MrcGetSetLimits (MrcData, GsmIocDataDccStepSize, NULL, &GetSetMax, NULL);
  for (Controller = 0; Controller < MAX_CONTROLLER; Controller++) {
    for (Channel = 0; Channel < MaxChannels; Channel++) {
      if (!MrcChannelExist (MrcData, Controller, Channel)) {
        continue;
      }
      for (Byte = 0; Byte < MaxSdram; Byte++) {
        Data16Ptr = DccLaneStatusResultAvg[Controller][Channel][Byte];
        Denominator16 = MAX (ABS (Data16Ptr[0] - Data16Ptr[1]), 1);
        Data16 = UDIVIDEROUND (32 * 32, Denominator16);

        if (Data16 > GetSetMax) {
          GetSetStepSize = Data16 / 2;
          GetSetVal = 1;
        } else {
          GetSetStepSize = Data16;
          GetSetVal = 0;
        }
        MrcGetSetChStrb (MrcData, Controller, Channel, Byte, GsmIocDataDcc2xStep, WriteToCache | PrintValue, &GetSetVal);
        MrcGetSetChStrb (MrcData, Controller, Channel, Byte, GsmIocDataDccStepSize, WriteToCache | PrintValue, &GetSetStepSize);
      }
    }
  }
  //Cleanup of SkipCrWrite
  MrcGetSetChStrb (MrcData, MAX_CONTROLLER, MAX_CHANNEL, MAX_SDRAM_IN_DIMM, GsmIocDataDccSkipCRWrite, WriteToCache | PrintValue, &GetSetDis);
  MrcFlushRegisterCachedData (MrcData);
  // Propagate new value
  MrcGetSetNoScope (MrcData, GsmIocForceCmpUpdt, ForceWriteUncached | PrintValue, &GetSetEn);
  ForceRcomp (MrcData);
  MRC_DEBUG_MSG (Debug, MSG_LEVEL_ALGO, "\n%sDccStepSize - Data\n", gStopTagStr);

  MRC_DEBUG_MSG (Debug, MSG_LEVEL_ALGO, "\n%sDccStepSize - CCC\n", gStartTagStr);
  IndexStart = (Lpddr5) ? 1 : 0;
  IndexEnd = IndexStart + 1;
  for (CccIndex = 0; CccIndex < MRC_NUM_CCC_INSTANCES; CccIndex++) {
    CCCDccFsmCtl.Data = OrigCCCDccFsmCtl[CccIndex];
    if (CCCDccFsmCtl.Bits.RankEn) {
      Offset = OFFSET_CALC_CH (CH0CCC_CR_DCCFSMCONTROL_REG, CH1CCC_CR_DCCFSMCONTROL_REG, CccIndex);

      //Enable clocks to toggle
      CCCDccFsmCtl.Bits.MeasPoint = 0;
      CCCDccFsmCtl.Bits.SaveFullDcc = 0;
      CCCDccFsmCtl.Bits.SkipCRWrite = 1;
      MrcWriteCR (MrcData, Offset, CCCDccFsmCtl.Data);
    }
  }
  MrcCall->MrcSetMem ((UINT8 *) DccLaneStatusResultAvg, sizeof (DccLaneStatusResultAvg), 0);

  Data32 = (!Lpddr) ? MAX_RANK_IN_CHANNEL : MRC_IGNORE_ARG;

  for (Index = IndexStart; Index < IndexEnd; Index++) {
    CccPerBit = (Index == 0) ? ClkPerBitCcc : WckPerBitCcc;
    //Setup for Comp1 (WCK for Index == 1)
    GetSetVal = 0x10;
    GetSetVal2 = 0x20;
    MrcGetSetCccLane (MrcData, MAX_CONTROLLER, MAX_CHANNEL, Data32, 0, CccPerBit, WriteToCache | PrintValue, &GetSetVal); //ClkP/WckP
    MrcGetSetCccLane (MrcData, MAX_CONTROLLER, MAX_CHANNEL, Data32, 1, CccPerBit, WriteToCache | PrintValue, &GetSetVal2); //ClkN/WckN
    MrcFlushRegisterCachedData (MrcData);

    // Propagate new value
    MrcGetSetNoScope (MrcData, GsmIocForceCmpUpdt, ForceWriteUncached | PrintValue, &GetSetEn);
    ForceRcomp (MrcData);
    for (Controller = 0; Controller < MAX_CONTROLLER; Controller++) {
      for (Channel = 0; Channel < MaxChannels; Channel++) {
        for (Rank = 0; Rank < RankEnd; Rank++) {
          if (!MrcRankExist (MrcData, Controller, Channel, Rank)) {
            continue;
          }
          ChannelArrayIndex = Channel + (Rank * MaxChannels);
          MrcGetSetBit (MrcData, Controller, Channel, Rank, MRC_IGNORE_ARG, 2 + Index, GsmIocCccDccLaneStatusResult, ReadUncached | PrintValue, &GetSetVal); //Bit = 2 for Lane5/6 (CLK), Bit = 3 for Lane7/8 (WCK)
          DccLaneStatusResultAvg[Controller][ChannelArrayIndex][0][0] += (UINT16) GetSetVal;
        }
      }
    }

    //Setup for Comp2
    GetSetVal = 0x20;
    GetSetVal2 = 0x10;
    MrcGetSetCccLane (MrcData, MAX_CONTROLLER, MAX_CHANNEL, Data32, 0, CccPerBit, WriteToCache | PrintValue, &GetSetVal); //ClkP/WckP
    MrcGetSetCccLane (MrcData, MAX_CONTROLLER, MAX_CHANNEL, Data32, 1, CccPerBit, WriteToCache | PrintValue, &GetSetVal2); //ClkN/WckN
    MrcFlushRegisterCachedData (MrcData);

    // Propagate new value
    MrcGetSetNoScope (MrcData, GsmIocForceCmpUpdt, ForceWriteUncached | PrintValue, &GetSetEn);
    ForceRcomp (MrcData);
    for (Controller = 0; Controller < MAX_CONTROLLER; Controller++) {
      for (Channel = 0; Channel < MaxChannels; Channel++) {
        for (Rank = 0; Rank < RankEnd; Rank++) {
          if (!MrcRankExist (MrcData, Controller, Channel, Rank)) {
            continue;
          }
          ChannelArrayIndex = Channel + (Rank * MaxChannels);
          MrcGetSetBit (MrcData, Controller, Channel, Rank, MRC_IGNORE_ARG, 2 + Index, GsmIocCccDccLaneStatusResult, ReadUncached | PrintValue, &GetSetVal); //Bit = 2 for Lane5/6 (CLK), Bit = 3 for Lane7/8 (WCK)
          DccLaneStatusResultAvg[Controller][ChannelArrayIndex][0][1] += (UINT16) GetSetVal;
        }
      }
    }
  } // For Index

#ifdef MRC_DEBUG_PRINT
  MRC_DEBUG_MSG (Debug, MSG_LEVEL_ALGO, "DccLaneStatusResultAvg - CCC\n");
  for (Controller = 0; Controller < MAX_CONTROLLER; Controller++) {
    for (Channel = 0; Channel < MaxChannels; Channel++) {
      for (Rank = 0; Rank < RankEnd; Rank++) {
        if (!MrcRankExist (MrcData, Controller, Channel, Rank)) {
          continue;
        }
        ChannelArrayIndex = Channel + (Rank * MaxChannels);
        MRC_DEBUG_MSG (Debug, MSG_LEVEL_ALGO, " MC%d.C%d.R%d - Comp1=%d, Comp2=%d\n", Controller, Channel, Rank, DccLaneStatusResultAvg[Controller][ChannelArrayIndex][0][0], DccLaneStatusResultAvg[Controller][ChannelArrayIndex][0][1]);
      }
    }
  }
#endif

  //DccStepSize determination
  MrcGetSetLimits (MrcData, GsmIocCccDccStepSize, NULL, &GetSetMax, NULL);
  for (Controller = 0; Controller < MAX_CONTROLLER; Controller++) {
    for (Channel = 0; Channel < MaxChannels; Channel++) {
      for (Rank = 0; Rank < RankEnd; Rank++) {
        if (!MrcRankExist (MrcData, Controller, Channel, Rank)) {
          continue;
        }
        ChannelArrayIndex = Channel + (Rank * MaxChannels);
        Data16Ptr = DccLaneStatusResultAvg[Controller][ChannelArrayIndex][0];
        Denominator16 = MAX (ABS (Data16Ptr[0] - Data16Ptr[1]), 1);
        Data16 = UDIVIDEROUND (32 * 32, Denominator16);

        if (Data16 > GetSetMax) {
          GetSetStepSize = Data16 / 2;
          GetSetVal = 1;
        } else {
          GetSetStepSize = Data16;
          GetSetVal = 0;
        }
        MrcGetSetCcc (MrcData, Controller, Channel, Rank, 0, GsmIocCccDcc2xStep, WriteToCache | PrintValue, &GetSetVal);
        MrcGetSetCcc (MrcData, Controller, Channel, Rank, 0, GsmIocCccDccStepSize, WriteToCache | PrintValue, &GetSetStepSize);
      }
    }
  }
  //Cleanup of SkipCrWrite
  for (CccIndex = 0; CccIndex < MRC_NUM_CCC_INSTANCES; CccIndex++) {
    Offset = OFFSET_CALC_CH (CH0CCC_CR_DCCFSMCONTROL_REG, CH1CCC_CR_DCCFSMCONTROL_REG, CccIndex);
    MrcWriteCR (MrcData, Offset, OrigCCCDccFsmCtl[CccIndex]);
  }
  MrcFlushRegisterCachedData (MrcData);
  // Propagate new value
  MrcGetSetNoScope (MrcData, GsmIocForceCmpUpdt, ForceWriteUncached | PrintValue, &GetSetEn);
  ForceRcomp (MrcData);
  MRC_DEBUG_MSG (Debug, MSG_LEVEL_ALGO, "\n%sDccStepSize - CCC\n", gStopTagStr);

  MRC_DEBUG_MSG (Debug, MSG_LEVEL_ALGO, "\n%sPBD codes\n", gStartTagStr);
  MrcGetSetNoScope (MrcData, GsmIocVccDllRxDeskewCal, ReadFromCache | PrintValue, &GetSetVal);
  DeskewCal = (UINT8) GetSetVal;

  for (Controller = 0; Controller < MAX_CONTROLLER; Controller++) {
    for (Channel = 0; Channel < MaxChannels; Channel++) {
      if (!MrcChannelExist (MrcData, Controller, Channel)) {
        continue;
      }
      for (Byte = 0; Byte < MaxSdram; Byte++) {
        MrcGetSetChStrb (MrcData, Controller, Channel, Byte, GsmIocDataDcc2xStep, ReadFromCache, &GetSetVal);
        MrcGetSetChStrb (MrcData, Controller, Channel, Byte, GsmIocDataDccStepSize, ReadFromCache, &GetSetStepSize);
        Data16 = MAX ((UINT16) GetSetStepSize, 1);
        Data16 *= (GetSetVal) ? 2 : 1;
        Data16 /= (DeskewCal) ? 2 : 1; //DccStepSize

        Data32 = (Qclkps * 100) / Data16 / 16; //PBDStepSizeps = QClk / DccStepSize / 16
        PBDStepSizeTicks = 800 / Data16; //PBDStepSizePITicks = 8 / DccStepSize
        PBDStepSizeTicks = MAX (PBDStepSizeTicks, 1);
        MRC_DEBUG_MSG (Debug, MSG_LEVEL_ALGO, " Qclkps=%d, StepSize=%d, PBDStepSizeps (*100) =%d, PBDStepSizePITicks (*100) =%d\n", Qclkps, Data16, Data32, PBDStepSizeTicks);
        GetSetVal = 3000 / Data32; //30ps * 100
        MrcGetSetBit (MrcData, Controller, Channel, MAX_RANK_IN_CHANNEL, Byte, MAX_BITS, RxDqsBitDelay, WriteToCache | GSM_UPDATE_HOST | PrintValue, &GetSetVal);
        GetSetVal = 10000 / Data32; //100ps * 100
        MrcGetSetBit (MrcData, Controller, Channel, MAX_RANK_IN_CHANNEL, Byte, MAX_BITS, TxDqBitDelay, WriteToCache | GSM_UPDATE_HOST | PrintValue, &GetSetVal);
        MrcGetSetStrobe (MrcData, Controller, Channel, MAX_RANK_IN_CHANNEL, Byte, TxDqsBitDelay, WriteToCache | PrintValue, &GetSetVal);

        //RxSALTailCtrl based on DccStepSize (>3.1pS: 0x3, <2.35pS : 0x1, Else : 0x2)
        if (Data32 > 310) {
          GetSetVal = 3;
        } else if (Data32 < 235) {
          GetSetVal = 1;
        } else {
          GetSetVal = 2;
        }
        MrcGetSetChStrb (MrcData, Controller, Channel, Byte, GsmIocRxSALTailCtrl, WriteToCache | PrintValue, &GetSetVal);
      }
      RankCount = 0;
      GetSetTot = 0;
      CccPFallNRise = ClkPerBitCcc;
      for (Rank = 0; Rank < RankEnd; Rank++) {
        if (!MrcRankExist (MrcData, Controller, Channel, Rank)) {
          continue;
        }
        RankCount++;
        ChannelArrayIndex = Channel + (Rank * MaxChannels);
        //Still valid from DccStepSize (CCC)
        Data16Ptr = DccLaneStatusResultAvg[Controller][ChannelArrayIndex][0];
        Denominator16 = MAX (Data16Ptr[0] - Data16Ptr[1], 1);
        Data16 = UDIVIDEROUND (32 * 32, Denominator16);
        Data16 /= (DeskewCal) ? 2 : 1;

        Data32 = (Qclkps * 100) / Data16 / 16;
        PBDStepSizeTicks = 800 / Data16;
        PBDStepSizeTicks = MAX (PBDStepSizeTicks, 1);
        MRC_DEBUG_MSG (Debug, MSG_LEVEL_ALGO, " CCC.Mc%d.C%d.R%d StepSize=%d, PBDStepSizeps (*100) = %d, PBDStepSizePITicks (*100) = %d\n", Controller, Channel, Rank, Data16, Data32, PBDStepSizeTicks);
        if (Lpddr5) {
          //Wck
          CccPFallNRise = WckPerBitCcc;
          GetSetVal2 = 10000 / Data32;
        } else {
          CccPFallNRise = ClkPerBitCcc;
          GetSetVal2 = 5000 / Data32;
        }
        GetSetTot += GetSetVal2;
        Data32 = (!Lpddr) ? Rank : MRC_IGNORE_ARG;
        MrcGetSetCccLane (MrcData, Controller, Channel, Data32, MAX_DDR4_CMD_PINS, CccPFallNRise, WriteToCache | PrintValue, &GetSetVal2);
      }
      if (RankCount) {
        GetSetVal2 = MrcCall->MrcDivU64x64 (GetSetTot, RankCount, NULL);
        for (CccPerBit = MaCaPerBitCcc; CccPerBit <= MiscPerBitCcc; CccPerBit++) {
          if (CccPerBit == CccPFallNRise) {
            //Skip as this is already updated
            continue;
          }
          MrcGetSetCccLane (MrcData, Controller, Channel, Data32, MAX_DDR4_CMD_PINS, CccPerBit, WriteToCache | PrintValue, &GetSetVal2);
        }
      }
      MrcFlushRegisterCachedData (MrcData);
    }
  }
  // Propagate new value
  MrcGetSetNoScope (MrcData, GsmIocForceCmpUpdt, ForceWriteUncached | PrintValue, &GetSetEn);
  ForceRcomp (MrcData);

  for (Controller = 0; Controller < MAX_CONTROLLER; Controller++) {
    for (Channel = 0; Channel < MaxChannels; Channel++) {
      if (MrcChannelExist (MrcData, Controller, Channel)) {
        IntCmdTiming = &IntOutputs->Controller[Controller].CmdTiming[Channel];

        // Initialize CRs shared between CKE/CTL/CMD/CLK
        Offset = (Ddr4) ? MRC_DDR4_CMD_GRP_MAX : 1;
        for (Index = 0; Index < Offset; Index++) {
          GetSetVal = IntCmdTiming->CmdPiCode[Index];
          MrcGetSetCcc (MrcData, Controller, Channel, MRC_IGNORE_ARG, Index, CmdGrpPi, WriteToCache | PrintValue, &GetSetVal);
        }

        for (Rank = 0; Rank < MAX_RANK_IN_CHANNEL; Rank++) {
          if (!MrcRankExist (MrcData, Controller, Channel, Rank)) {
            continue;
          }
          GetSetVal = IntCmdTiming->CtlPiCode[Rank];
          MrcGetSetCcc (MrcData, Controller, Channel, Rank, 0, CtlGrpPi, WriteToCache | PrintValue, &GetSetVal);

          if (Ddr4) {
            GetSetVal = IntCmdTiming->ClkPiCode[Rank];
            MrcGetSetCcc (MrcData, Controller, Channel, Rank, 0, ClkGrpPi, WriteToCache | PrintValue, &GetSetVal);
            GetSetVal = IntCmdTiming->CkePiCode[Rank];
            MrcGetSetCcc (MrcData, Controller, Channel, Rank, 0, CkeGrpPi, WriteToCache | PrintValue, &GetSetVal);
          }
        }
        // Clk/Cke/Wck are per-channel for LPDDR
        if (Lpddr) {
          GetSetVal = IntCmdTiming->ClkPiCode[0];
          MrcGetSetCcc (MrcData, Controller, Channel, MRC_IGNORE_ARG, 0, ClkGrpPi, WriteToCache | PrintValue, &GetSetVal);
        }
        if (Lpddr4) {
          GetSetVal = IntCmdTiming->CkePiCode[0];
          MrcGetSetCcc (MrcData, Controller, Channel, MRC_IGNORE_ARG, 0, CkeGrpPi, WriteToCache | PrintValue, &GetSetVal);
        } else if (Lpddr5) {
          GetSetVal = IntCmdTiming->WckPiCode;
          MrcGetSetCcc (MrcData, Controller, Channel, MRC_IGNORE_ARG, 0, WckGrpPi, WriteToCache | PrintValue, &GetSetVal);
        }
      }
    } // Channel
  } // Controller
  MrcFlushRegisterCachedData (MrcData);
  // Propagate new value1
  MrcGetSetNoScope (MrcData, GsmIocForceCmpUpdt, ForceWriteUncached | PrintValue, &GetSetEn);
  ForceRcomp (MrcData);
  MRC_DEBUG_MSG (Debug, MSG_LEVEL_ALGO, "\n%sPBD codes\n", gStopTagStr);

  Data32 = DCC_DQ_MASK;
  Data32 |= (Lpddr5) ? DCC_WCK_MASK : DCC_CLK_MASK;
  Index = 0;
  do {
    ForceRcomp (MrcData);
    MRC_DEBUG_MSG (Debug, MSG_LEVEL_ALGO, "CompCycle %d\n", Index);
    Index++;
    Status = MrcDccConvergenceCheck (MrcData, Data32, 0xFF, FALSE);
    if ((Index > MRC_MAX_DCC_CONVERGENCE_CYCLES) && (Status != mrcSuccess)) {
      MRC_DEBUG_MSG (Debug, MSG_LEVEL_WARNING, "%s: Failed to converge after %d comp cycles\n", gWarnString, MRC_MAX_DCC_CONVERGENCE_CYCLES);
      Status = mrcSuccess;
      break;
    }
  } while ((Status != mrcSuccess) || (Index < 3));

  for (Controller = 0; Controller < MAX_CONTROLLER; Controller++) {
    for (Channel = 0; Channel < MaxChannels; Channel++) {
      GetSetVal = (MrcChannelExist (MrcData, Controller, Channel)) ? Outputs->Controller[Controller].Channel[Channel].ValidRankBitMask : 0;
      MrcGetSetChStrb (MrcData, Controller, Channel, MAX_SDRAM_IN_DIMM, GsmIocDataDccRankEn, WriteToCache | PrintValue, &GetSetVal);
    }
  }
  MrcFlushRegisterCachedData (MrcData);

  MrcDccCbOptimization (MrcData, FirstCccChannel, OrigCCCDccFsmCtl);

  for (CccIndex = 0; CccIndex < MRC_NUM_CCC_INSTANCES; CccIndex++) {
    CCCDccFsmCtl.Data = OrigCCCDccFsmCtl[CccIndex];
    CCCDccFsmCtl.Bits.UpdateTcoComp = 1;
    Offset = OFFSET_CALC_CH (CH0CCC_CR_DCCFSMCONTROL_REG, CH1CCC_CR_DCCFSMCONTROL_REG, CccIndex);
    MrcWriteCR (MrcData, Offset, CCCDccFsmCtl.Data);
  }

  MrcDccLut (MrcData);

  MrcGetSetMc (MrcData, MAX_CONTROLLER, GsmMccEnableDclk, WriteNoCache, &GetSetEn);
  Offset = OFFSET_CALC_CH (CH0CCC_CR_DDRCRCCCVOLTAGEUSED_REG, CH1CCC_CR_DDRCRCCCVOLTAGEUSED_REG, FirstCccChannel);
  CCCVoltageUsed.Data = MrcReadCR (MrcData, Offset);
  CCCVoltageUsed.Bits.DisWckPupDcc = 0;
  MrcWriteCrMulticast (MrcData, CCC_CR_DDRCRCCCVOLTAGEUSED_REG, CCCVoltageUsed.Data);
  IntOutputs->SkipZq = FALSE;

  return Status;
}

/**
  This function runs DCC steps prior to Early Command Training

  @param[in, out] MrcData - Include all MRC global data.

  @retval mrcSuccess if successful.
**/
MrcStatus
MrcDccPreEct (
  IN OUT MrcParameters *const MrcData
  )
{
  MrcOutput  *Outputs;
  MrcStatus  Status;
  INT64      GetSetVal;
  UINT32     FirstController;
  UINT32     FirstCccChannel;
  UINT32     Offset;
  UINT8      Index;
  CH0CCC_CR_DDRCRPERBITTCO1_STRUCT  CCCPerBitTco1;
  CH0CCC_CR_DDRCRCCCVOLTAGEUSED_STRUCT  CCCVoltageUsed;

  Outputs = &MrcData->Outputs;
  Status = mrcSuccess;

  if (MrcData->Inputs.B0) {
    return Status;
  }

  FirstController = (MrcControllerExist (MrcData, cCONTROLLER0)) ? 0 : 1;
  FirstCccChannel = (FirstController * MAX_CHANNEL) + Outputs->Controller[FirstController].FirstPopCh;;
  Offset = OFFSET_CALC_CH (CH0CCC_CR_DDRCRCCCVOLTAGEUSED_REG, CH1CCC_CR_DDRCRCCCVOLTAGEUSED_REG, FirstCccChannel);
  CCCVoltageUsed.Data = MrcReadCR (MrcData, Offset);
  CCCVoltageUsed.Bits.DisWckPupDcc = 0;
  MrcWriteCrMulticast (MrcData, CCC_CR_DDRCRCCCVOLTAGEUSED_REG, CCCVoltageUsed.Data);

  if (Outputs->DdrType == MRC_DDR_TYPE_LPDDR5) {
    return Status;
  }

  for (Index = 0; Index < MRC_NUM_CCC_INSTANCES; Index++) {
    Offset = OFFSET_CALC_CH (CH0CCC_CR_DDRCRPERBITTCO1_REG, CH1CCC_CR_DDRCRPERBITTCO1_REG, Index);
    CCCPerBitTco1.Data = MrcReadCR (MrcData, Offset); //?Not to be restore?
    CCCPerBitTco1.Bits.CCC5 = 0;
    CCCPerBitTco1.Bits.CCC6 = 0;
    MrcWriteCR (MrcData, Offset, CCCPerBitTco1.Data);
  }
  GetSetVal = 1;
  MrcGetSetNoScope (MrcData, GsmIocForceCmpUpdt, ForceWriteUncached | PrintValue, &GetSetVal);

  Status |= MrcDccTrainingStep (MrcData, MRC_IGNORE_ARG, DCC_CLK_MASK, 3, TRUE);

  if (Outputs->Lpddr == FALSE) {
    Outputs->RestoreMRs = FALSE;
    Outputs->JedecInitDone = FALSE;
    Status |= MrcResetSequence (MrcData);
  }

  return Status;
}

/**
  DCC Training step for running prior to read/write leveling and after command training

  @param[in] MrcData - Include all MRC global data.

  @retval mrcSuccess if DCC converged otherwise mrcFail.
**/
MrcStatus
MrcDccPreLeveling (
  IN MrcParameters *const MrcData
  )
{
  UINT8   CccIndex;
  UINT32  Offset;
  UINT32  FirstController;
  UINT32  FirstCccChannel;
  UINT32  OrigCCCDccFsmCtl[MRC_NUM_CCC_INSTANCES];
  INT64   GetSetVal;
  CH0CCC_CR_DCCFSMCONTROL_STRUCT   CCCDccFsmCtl;

  FirstController = (MrcControllerExist (MrcData, cCONTROLLER0)) ? cCONTROLLER0 : cCONTROLLER1;
  FirstCccChannel = (FirstController * MAX_CHANNEL) + MrcData->Outputs.Controller[FirstController].FirstPopCh;;

  for (CccIndex = 0; CccIndex < MRC_NUM_CCC_INSTANCES; CccIndex++) {
    Offset = OFFSET_CALC_CH (CH0CCC_CR_DCCFSMCONTROL_REG, CH1CCC_CR_DCCFSMCONTROL_REG, CccIndex);
    CCCDccFsmCtl.Data = MrcReadCR (MrcData, Offset);
    CCCDccFsmCtl.Bits.UpdateTcoComp = 0;
    OrigCCCDccFsmCtl[CccIndex] = CCCDccFsmCtl.Data;
  }

  MrcDccCbOptimization (MrcData, FirstCccChannel, OrigCCCDccFsmCtl);

  for (CccIndex = 0; CccIndex < MRC_NUM_CCC_INSTANCES; CccIndex++) {
    CCCDccFsmCtl.Data = OrigCCCDccFsmCtl[CccIndex];
    CCCDccFsmCtl.Bits.UpdateTcoComp = 1;
    Offset = OFFSET_CALC_CH (CH0CCC_CR_DCCFSMCONTROL_REG, CH1CCC_CR_DCCFSMCONTROL_REG, CccIndex);
    MrcWriteCR (MrcData, Offset, CCCDccFsmCtl.Data);
  }

  MrcDccLut (MrcData);

  GetSetVal = 0x20;
  MrcGetSetBit (MrcData, MAX_CONTROLLER, MAX_CHANNEL, MRC_IGNORE_ARG, MAX_SDRAM_IN_DIMM, MAX_BITS, TxTco, WriteToCache | PrintValue, &GetSetVal);
  MrcFlushRegisterCachedData (MrcData);

  //DQS/WCK
  return (MrcDccTrainingStep (MrcData, PatWrEndless, DCC_DQS_MASK | DCC_WCK_MASK, 3, TRUE));
}

/**
  DCC Training step for running prior to read timing

  @param[in] MrcData - Include all MRC global data.

  @retval MrcStatus -  mrcSuccess.
  **/
MrcStatus
MrcDccPreReadTiming (
  IN MrcParameters *const MrcData
  )
{
  MrcOutput          *Outputs;
  MrcDebug           *Debug;
  const MRC_FUNCTION *MrcCall;
  UINT8              McChBitMask;
  UINT8              BestIndex[MAX_CONTROLLER][MAX_CHANNEL][MAX_SDRAM_IN_DIMM];
  UINT8              Index;
  UINT8              IndexMin;
  UINT8              IndexMax;
  UINT8              MaxChannels;
  UINT8              MaxSdram;
  UINT16             TotalCount;
  UINT16             Data16;
  UINT16             *Data16Ptr;
  UINT16             DccLaneStatusResult[MAX_CONTROLLER][MAX_CHANNEL][MAX_SDRAM_IN_DIMM][MAX_EDGES][MAX_RXDQS_VREF_OFFSETS];
  UINT32             Controller;
  UINT32             Channel;
  UINT32             Rank;
  UINT32             Byte;
  UINT32             GsmMode;
  INT64              GetSetVal;
  INT64              GetSetMin;
  INT64              GetSetMax;
  GSM_GT             GetSetEnum;
  MC0_REQ0_CR_CPGC_SEQ_CTL_STRUCT  CpgcStartTest;
  MC0_REQ0_CR_CPGC_SEQ_CTL_STRUCT  CpgcStopTest;

  Outputs = &MrcData->Outputs;
  MrcCall = MrcData->Inputs.Call.Func;
  Debug = &Outputs->Debug;
  MaxChannels = Outputs->MaxChannels;
  MaxSdram = Outputs->SdramCount;
  GetSetEnum = ((Outputs->RxMode == MrcRxModeUnmatchedRxWRload) || (Outputs->RxMode == MrcRxModeUnmatchedRxWPpath)) ? RxDqsUnmatchedAmpOffset : RxDqsAmpOffset;
  MrcGetSetLimits (MrcData, GetSetEnum, &GetSetMin, &GetSetMax, NULL);
  IndexMin = (UINT8) GetSetMin;
  IndexMax = (UINT8) GetSetMax;
  GsmMode = PrintValue;

  CpgcStartTest.Data = CpgcStopTest.Data = 0;
  CpgcStartTest.Bits.START_TEST = 1;
  CpgcStopTest.Bits.STOP_TEST = 1;

  MRC_DEBUG_MSG (Debug, MSG_LEVEL_ALGO, "\n%sDcc_Pre_Read_1D\n", gStartTagStr);

  //RxDqsN/P PI (N is BitLane0, P is BitLane7)
  GetSetVal = 1;
  MrcGetSetChStrb (MrcData, MAX_CONTROLLER, MAX_CHANNEL, MAX_SDRAM_IN_DIMM, GsmIocDataDccMeasPoint, WriteToCache | GsmMode, &GetSetVal);
  MrcFlushRegisterCachedData (MrcData);

  for (Rank = 0; Rank < MAX_RANK_IN_CHANNEL; Rank++) {
    McChBitMask = 0;
    for (Controller = 0; Controller < MAX_CONTROLLER; Controller++) {
      for (Channel = 0; Channel < MaxChannels; Channel++) {
        McChBitMask |= SelectReutRanks (MrcData, (UINT8) Controller, (UINT8) Channel, 1 << Rank, FALSE, 0);
      }
    }
    if (McChBitMask == 0) {
      //No rank
      continue;
    }

    SetupIOTestBasicVA (MrcData, McChBitMask, 1, NSOE, 0, 0, 10, PatRdEndless, 0, 0);
    Cpgc20ControlRegWrite (MrcData, McChBitMask, CpgcStartTest);

    MrcCall->MrcSetMem ((UINT8 *) DccLaneStatusResult, sizeof (DccLaneStatusResult), 0);
    MrcCall->MrcSetMem ((UINT8 *) BestIndex, sizeof (BestIndex), 0);

    for (Index = IndexMin; Index <= IndexMax; Index++) {
      GetSetVal = Index;
      MrcGetSetStrobe (MrcData, MAX_CONTROLLER, MAX_CHANNEL, Rank, MAX_SDRAM_IN_DIMM, GetSetEnum, WriteToCache | GsmMode, &GetSetVal);
      MrcFlushRegisterCachedData (MrcData);
      ForceRcomp (MrcData);

      //Gather the result
      for (Controller = 0; Controller < MAX_CONTROLLER; Controller++) {
        for (Channel = 0; Channel < MaxChannels; Channel++) {
          if (!MrcRankExist (MrcData, Controller, Channel, Rank)) {
            continue;
          }
          for (Byte = 0; Byte < MaxSdram; Byte++) {
            //RxDqsN
            MrcGetSetBit (MrcData, Controller, Channel, MRC_IGNORE_ARG, Byte, 0, GsmIocDataDqDccLaneStatusResult, ReadCached | GsmMode, &GetSetVal);
            DccLaneStatusResult[Controller][Channel][Byte][0][Index] = (UINT16) GetSetVal;

            //RxDqsP
            MrcGetSetBit (MrcData, Controller, Channel, MRC_IGNORE_ARG, Byte, 7, GsmIocDataDqDccLaneStatusResult, ReadCached | GsmMode, &GetSetVal);
            DccLaneStatusResult[Controller][Channel][Byte][1][Index] = (UINT16) GetSetVal;
          }
        }
      }
    }
    Cpgc20ControlRegWrite (MrcData, McChBitMask, CpgcStopTest);
#ifdef MRC_DEBUG_PRINT
    MRC_DEBUG_MSG (Debug, MSG_LEVEL_ALGO, "Rank %d\nVrefOffset\t", Rank);
    for (Index = IndexMin; Index <= IndexMax; Index++) {
      MRC_DEBUG_MSG (Debug, MSG_LEVEL_ALGO, "\t%d", Index);
    }
    MRC_DEBUG_MSG (Debug, MSG_LEVEL_ALGO, "\nRxDqsN-");
#endif
    for (Controller = 0; Controller < MAX_CONTROLLER; Controller++) {
      for (Channel = 0; Channel < MaxChannels; Channel++) {
        if (!MrcRankExist (MrcData, Controller, Channel, Rank)) {
          continue;
        }
        for (Byte = 0; Byte < MaxSdram; Byte++) {
          MRC_DEBUG_MSG (Debug, MSG_LEVEL_ALGO, "Mc%d.C%d.S%d", Controller, Channel, Byte);
          Data16Ptr = DccLaneStatusResult[Controller][Channel][Byte][0];
          for (Index = IndexMin; Index <= IndexMax; Index++) {
            MRC_DEBUG_MSG (Debug, MSG_LEVEL_ALGO, "\t%d", Data16Ptr[Index]);
            Data16 = Data16Ptr[Index];
            Data16Ptr[Index] = ABS (Data16 - HALF_DUTY_CYCLE);
          }
        }
      }
    }
    MRC_DEBUG_MSG (Debug, MSG_LEVEL_ALGO, "\n\nRxDqsP-");
    for (Controller = 0; Controller < MAX_CONTROLLER; Controller++) {
      for (Channel = 0; Channel < MaxChannels; Channel++) {
        if (!MrcRankExist (MrcData, Controller, Channel, Rank)) {
          continue;
        }
        for (Byte = 0; Byte < MaxSdram; Byte++) {
          MRC_DEBUG_MSG (Debug, MSG_LEVEL_ALGO, "Mc%d.C%d.S%d", Controller, Channel, Byte);
          Data16Ptr = DccLaneStatusResult[Controller][Channel][Byte][1];
          for (Index = IndexMin; Index <= IndexMax; Index++) {
            MRC_DEBUG_MSG (Debug, MSG_LEVEL_ALGO, "\t%d", Data16Ptr[Index]);
            Data16 = Data16Ptr[Index];
            Data16Ptr[Index] = ABS (Data16 - HALF_DUTY_CYCLE);
          }
        }
      }
    }
    MRC_DEBUG_MSG (Debug, MSG_LEVEL_ALGO, "\n\nTotalCnt-");

    //Search for the best index that gives us the smallest distance from 50% duty cycle
    for (Controller = 0; Controller < MAX_CONTROLLER; Controller++) {
      for (Channel = 0; Channel < MaxChannels; Channel++) {
        if (!MrcRankExist (MrcData, Controller, Channel, Rank)) {
          continue;
        }
        for (Byte = 0; Byte < MaxSdram; Byte++) {
          MRC_DEBUG_MSG (Debug, MSG_LEVEL_ALGO, "Mc%d.C%d.S%d", Controller, Channel, Byte);
          TotalCount = MRC_UINT16_MAX;
          for (Index = IndexMin; Index <= IndexMax; Index++) {
            Data16 = DccLaneStatusResult[Controller][Channel][Byte][0][Index] + DccLaneStatusResult[Controller][Channel][Byte][1][Index];
            MRC_DEBUG_MSG (Debug, MSG_LEVEL_ALGO, "\t%d", Data16);
            if (Data16 < TotalCount) {
              TotalCount = Data16;
              BestIndex[Controller][Channel][Byte] = Index;
            }
          }
        }
      }
    }
    MRC_DEBUG_MSG (Debug, MSG_LEVEL_ALGO, "\n\n\t\tBest Index\n");
    for (Controller = 0; Controller < MAX_CONTROLLER; Controller++) {
      for (Channel = 0; Channel < MaxChannels; Channel++) {
        if (!MrcRankExist (MrcData, Controller, Channel, Rank)) {
          continue;
        }
        for (Byte = 0; Byte < MaxSdram; Byte++) {
          Data16 = BestIndex[Controller][Channel][Byte];
          MRC_DEBUG_MSG (Debug, MSG_LEVEL_ALGO, "Mc%d.C%d.S%d\t%d\n", Controller, Channel, Byte, Data16);
          GetSetVal = Data16;
          MrcGetSetStrobe (MrcData, Controller, Channel, Rank, Byte, GetSetEnum, WriteToCache | GsmMode, &GetSetVal);
        }
      }
    }
    MrcFlushRegisterCachedData (MrcData);
  } // for Rank

  MRC_DEBUG_MSG (Debug, MSG_LEVEL_ALGO, "\n%sDcc_Pre_Read_1D\n", gStopTagStr);

  //Reset MeasPoint to 0
  GetSetVal = 0;
  MrcGetSetChStrb (MrcData, MAX_CONTROLLER, MAX_CHANNEL, MAX_SDRAM_IN_DIMM, GsmIocDataDccMeasPoint, WriteToCache | GsmMode, &GetSetVal);
  MrcFlushRegisterCachedData (MrcData);

  return mrcSuccess;
}

/**
  DCC Training step for running after training steps

  @param[in] MrcData         - Include all MRC global data.
  @param[in] Traffic         - Type of traffic required (if none, use MRC_IGNORE_ARG)
  @param[in] ParamMask       - Bit Mask of parameters to be run.  (See defines at top of .c file)
  #param[in] NumOfCompCycles - # of Comp Cycles needed
  @param[in] Print           - Whether to print (TRUE) or not (FALSE)

  @retval mrcSuccess if DCC converged otherwise mrcFail
**/
MrcStatus
MrcDccTrainingStep (
  IN MrcParameters *const MrcData,
  IN UINT32               Traffic,
  IN UINT32               ParamMask,
  IN UINT32               NumOfCompCycles,
  IN BOOLEAN              Print
  )
{
  MrcInput          *Inputs;
  MrcOutput         *Outputs;
  MrcDebug          *Debug;
  MrcStatus         Status;
  UINT8             McChBitMask;
  UINT8             CccIndex;
  UINT8             Loop;
  UINT8             LoopMax;
  UINT8             CompCycle;
  UINT8             MaxChannels;
  UINT8             RankEn;
  UINT8             LaneEn;
  UINT32            GsmMode;
  UINT32            Controller;
  UINT32            Channel;
  UINT32            Rank;
  UINT32            Offset;
  UINT32            DbgLevel;
  UINT32            OrigCCCDccFsmCtl[MRC_NUM_CCC_INSTANCES];
  INT64             GetSetMeasPoint;
  INT64             GetSetLaneEn;
  INT64             GetSetDis;
  INT64             GetSetVal;
  BOOLEAN           TestDqs;
  BOOLEAN           TestDq;
  BOOLEAN           TestWck;
  BOOLEAN           TestClk;
  BOOLEAN           Lpddr5;
  BOOLEAN           SpecialDqsSetup;
  MC0_REQ0_CR_CPGC_SEQ_CTL_STRUCT  CpgcStartTest;
  MC0_REQ0_CR_CPGC_SEQ_CTL_STRUCT  CpgcStopTest;
  CH0CCC_CR_DCCFSMCONTROL_STRUCT  CCCDccFsmCtl;

  Inputs  = &MrcData->Inputs;
  Outputs = &MrcData->Outputs;
  Debug   = &Outputs->Debug;
  Lpddr5  = (Outputs->DdrType == MRC_DDR_TYPE_LPDDR5);
  TestDqs = ParamMask & DCC_DQS_MASK;
  TestDq  = ParamMask & DCC_DQ_MASK;
  TestWck = ParamMask & DCC_WCK_MASK;
  TestClk = ParamMask & DCC_CLK_MASK;
  SpecialDqsSetup = !Lpddr5 && Inputs->A0 && TestDqs;
  LoopMax = (SpecialDqsSetup && TestDq) ? 2 : 1;
  MaxChannels = Outputs->MaxChannels;
  GsmMode = (Print) ? PrintValue : 0;
  GetSetDis = 0;
  DbgLevel = (Print) ? MSG_LEVEL_NOTE : MSG_LEVEL_NEVER;
  Status = mrcSuccess;

  if (!(Inputs->TrainingEnables2.DCC && Outputs->Gear2)) {
    //Skip this as DCC is not enabled
    return Status;
  }

  CpgcStartTest.Data = CpgcStopTest.Data = 0;
  CpgcStartTest.Bits.START_TEST = 1;
  CpgcStopTest.Bits.STOP_TEST = 1;

  //Data DccFsm Setup
  GetSetLaneEn = 0;
  if (SpecialDqsSetup) {
    //Special Setup for DQS for Loop 0
    GetSetMeasPoint = 0; //Serializer
    GetSetLaneEn |= DCC_DATA_LANE_EN_DQS;
  } else {
    GetSetMeasPoint = 2; //Pad node
    if (TestDqs) {
      GetSetLaneEn |= DCC_DATA_LANE_EN_DQS;
    }
    if (TestDq) {
      GetSetLaneEn |= DCC_DATA_LANE_EN_DQ;
    }
  }
  if (TestDqs || TestDq) {
    MrcGetSetChStrb (MrcData, MAX_CONTROLLER, MAX_CHANNEL, MAX_SDRAM_IN_DIMM, GsmIocDataDccMeasPoint, WriteToCache | GsmMode, &GetSetMeasPoint);
    MrcGetSetChStrb (MrcData, MAX_CONTROLLER, MAX_CHANNEL, MAX_SDRAM_IN_DIMM, GsmIocDataDccLaneEn, WriteToCache | GsmMode, &GetSetLaneEn);
  } else {
    //Disable RankEn since not testing Data
    MrcGetSetChStrb (MrcData, MAX_CONTROLLER, MAX_CHANNEL, MAX_SDRAM_IN_DIMM, GsmIocDataDccRankEn, WriteToCache | GsmMode, &GetSetDis);
  }
  MrcFlushRegisterCachedData (MrcData);

  //Ccc DccFsm Setup
  CCCDccFsmCtl.Data = 0;
  LaneEn = 0;
  if (TestWck || TestClk) {
    RankEn = 1;
    if (TestWck) {
      LaneEn |= DCC_CCC_LANE_EN_WCK;
    }
    if (TestClk) {
      LaneEn |= DCC_CCC_LANE_EN_CLK;
    }
  } else {
    //Disable RankEn since not testing CCC
    RankEn = 0;
  }
  for (CccIndex = 0; CccIndex < MRC_NUM_CCC_INSTANCES; CccIndex++) {
    Offset = OFFSET_CALC_CH (CH0CCC_CR_DCCFSMCONTROL_REG, CH1CCC_CR_DCCFSMCONTROL_REG, CccIndex);
    CCCDccFsmCtl.Data = OrigCCCDccFsmCtl[CccIndex] = MrcReadCR (MrcData, Offset);

    //Measure at pad
    CCCDccFsmCtl.Bits.MeasPoint = 2;
    CCCDccFsmCtl.Bits.LaneEn = LaneEn;
    CCCDccFsmCtl.Bits.RankEn = RankEn;
    CCCDccFsmCtl.Bits.UpdateTcoComp = (TestClk) ? 0 : 1;
    MrcWriteCR (MrcData, Offset, CCCDccFsmCtl.Data);
    MRC_DEBUG_MSG (Debug, DbgLevel, "CCC MeasPoint %d, CCC LaneEn 0x%x, CCC RankEn 0x%x, CCC UpdateTcoComp 0x%x\n", CCCDccFsmCtl.Bits.MeasPoint, CCCDccFsmCtl.Bits.LaneEn, CCCDccFsmCtl.Bits.RankEn, CCCDccFsmCtl.Bits.UpdateTcoComp);
  }

  for (Loop = 0; Loop < LoopMax; Loop++) {
    //Setup MeasPoint
    if (Loop) {
      //Setup for DQ at pad on Loop 1
      if (Inputs->A0) {
        GetSetMeasPoint = 0;
      } else {
        GetSetMeasPoint = 2; //Pad node
      }
      GetSetLaneEn = DCC_DATA_LANE_EN_DQ;
      MrcGetSetChStrb (MrcData, MAX_CONTROLLER, MAX_CHANNEL, MAX_SDRAM_IN_DIMM, GsmIocDataDccMeasPoint, WriteToCache | GsmMode, &GetSetMeasPoint);
      MrcGetSetChStrb (MrcData, MAX_CONTROLLER, MAX_CHANNEL, MAX_SDRAM_IN_DIMM, GsmIocDataDccLaneEn, WriteToCache | GsmMode, &GetSetLaneEn);
      MrcFlushRegisterCachedData (MrcData);
    }

    for (Rank = 0; Rank < MAX_RANK_IN_CHANNEL; Rank++) {
      McChBitMask = 0;
      for (Controller = 0; Controller < MAX_CONTROLLER; Controller++) {
        for (Channel = 0; Channel < MaxChannels; Channel++) {
          McChBitMask |= SelectReutRanks (MrcData, (UINT8) Controller, (UINT8) Channel, 1 << Rank, FALSE, 0);
        }
      }
      if (McChBitMask == 0) {
        //No rank
        continue;
      }

      if (Traffic != MRC_IGNORE_ARG) {
        SetupIOTestBasicVA (MrcData, McChBitMask, 1, NSOE, 0, 0, 10, (UINT8) Traffic, 0, 0);
        Cpgc20ControlRegWrite (MrcData, McChBitMask, CpgcStartTest);
      }

      for (CompCycle = 0; CompCycle < NumOfCompCycles; CompCycle++) {
        ForceRcomp (MrcData);
#ifdef MRC_DEBUG_PRINT
        if (GsmMode) {
          MRC_DEBUG_MSG (Debug, DbgLevel, "CompCycle %d\n", CompCycle);
        }
#endif
      }
      Status = MrcDccConvergenceCheck (MrcData, ParamMask, 1 << Rank, Print);
      while ((Status != mrcSuccess) && (CompCycle < MRC_MAX_DCC_CONVERGENCE_CYCLES)) {
        ForceRcomp (MrcData);
        CompCycle++;
#ifdef MRC_DEBUG_PRINT
        if (GsmMode) {
          MRC_DEBUG_MSG (Debug, DbgLevel, "CompCycle %d\n", CompCycle);
        }
#endif
        Status = MrcDccConvergenceCheck (MrcData, ParamMask, 1 << Rank, Print);
      }
      if (Status != mrcSuccess) {
        MRC_DEBUG_MSG (Debug, MSG_LEVEL_WARNING, "%s: Failed to converge after %d comp cycles\n", gWarnString, CompCycle);
        Status = mrcSuccess;
      }
      if (Traffic != MRC_IGNORE_ARG) {
        Cpgc20ControlRegWrite (MrcData, McChBitMask, CpgcStopTest);
      }
    }
  }

  //Restore MeasPoint/RankEn/UpdateTcoComp
  for (CccIndex = 0; CccIndex < MRC_NUM_CCC_INSTANCES; CccIndex++) {
    Offset = OFFSET_CALC_CH (CH0CCC_CR_DCCFSMCONTROL_REG, CH1CCC_CR_DCCFSMCONTROL_REG, CccIndex);
    MrcWriteCR (MrcData, Offset, OrigCCCDccFsmCtl[CccIndex]);
  }
  if (TestDqs || TestDq) {
    GetSetMeasPoint = 0;
    GetSetLaneEn = DCC_DATA_LANE_EN_ALL;
    MrcGetSetChStrb (MrcData, MAX_CONTROLLER, MAX_CHANNEL, MAX_SDRAM_IN_DIMM, GsmIocDataDccMeasPoint, WriteToCache | GsmMode, &GetSetMeasPoint);
    MrcGetSetChStrb (MrcData, MAX_CONTROLLER, MAX_CHANNEL, MAX_SDRAM_IN_DIMM, GsmIocDataDccLaneEn, WriteToCache | GsmMode, &GetSetLaneEn);
  } else {
    for (Controller = 0; Controller < MAX_CONTROLLER; Controller++) {
      for (Channel = 0; Channel < MaxChannels; Channel++) {
        GetSetVal = (MrcChannelExist (MrcData, Controller, Channel)) ? Outputs->Controller[Controller].Channel[Channel].ValidRankBitMask : 0;
        MrcGetSetChStrb (MrcData, Controller, Channel, MAX_SDRAM_IN_DIMM, GsmIocDataDccRankEn, WriteToCache | GsmMode, &GetSetVal);
      }
    }
  }
  MrcFlushRegisterCachedData (MrcData);

  return Status;
}

/**
  DCC Training step for running after write leveling

  @param[in] MrcData - Include all MRC global data.

  @retval mrcSuccess if successful
**/
MrcStatus
MrcDccPostWriteLeveling (
  IN MrcParameters *const MrcData
  )
{
  //DQ/DQS/WCK
  return (MrcDccTrainingStep (MrcData, PatWrEndless, DCC_DQ_MASK | DCC_DQS_MASK | DCC_WCK_MASK, 2, TRUE));
}

/**
  DCC Training step for adjusting TcoComp (Lpddr & DDR5)

  @param[in] MrcData - Include all MRC global data.

  @retval mrcSuccess.
**/
MrcStatus
MrcDccAdjustTcoComp (
  IN MrcParameters *const MrcData
  )
{
  const MRC_FUNCTION *MrcCall;
  MrcOutput         *Outputs;
  MrcDebug          *Debug;
  UINT8             CurrentMcChBitMask;
  UINT8             FullMcChBitMask;
  UINT8             McChBitMask;
  UINT8             Index;
  UINT8             MaxChannels;
  UINT8             MaxSdram;
  UINT8             OrigMeasPoint;
  UINT8             OrigSaveFullDcc;
  UINT8             OrigSkipCrWrite;
  UINT8             Rank;
  UINT8             BestIndex[MAX_CONTROLLER][MAX_CHANNEL][MAX_SDRAM_IN_DIMM][MAX_BITS];
  UINT16            HalfDutyCycle;
  UINT16            Data16;
  UINT16            *Data16Ptr;
  UINT16            DataDccLaneStatus[MAX_CONTROLLER][MAX_CHANNEL][MAX_SDRAM_IN_DIMM][MAX_BITS][MAX_TCO_COMP];
  UINT32            Controller;
  UINT32            Channel;
  UINT32            Byte;
  UINT32            Bit;
  INT64             GetSetVal;
  MC0_REQ0_CR_CPGC_SEQ_CTL_STRUCT  CpgcStartTest;
  MC0_REQ0_CR_CPGC_SEQ_CTL_STRUCT  CpgcStopTest;

  MrcCall = MrcData->Inputs.Call.Func;
  Outputs = &MrcData->Outputs;
  Debug = &Outputs->Debug;
  FullMcChBitMask = Outputs->McChBitMask;
  MaxChannels = Outputs->MaxChannels;
  MaxSdram = Outputs->SdramCount;
  CurrentMcChBitMask = 0;

  if (Outputs->DdrType == MRC_DDR_TYPE_DDR4) {
    return mrcSuccess;
  }

  MRC_DEBUG_MSG (Debug, MSG_LEVEL_ALGO, "\n%sDcc_Adjust_TcoComp\n", gStartTagStr);

  CpgcStartTest.Data = CpgcStopTest.Data = 0;
  CpgcStartTest.Bits.START_TEST = 1;
  CpgcStopTest.Bits.STOP_TEST = 1;

  Controller = (MrcControllerExist (MrcData, cCONTROLLER0)) ? cCONTROLLER0 : cCONTROLLER1;
  Channel = Outputs->Controller[Controller].FirstPopCh;
  MrcGetSetChStrb (MrcData, Controller, Channel, 0, GsmIocDataDccMeasPoint, ReadFromCache, &GetSetVal);
  OrigMeasPoint = (UINT8) GetSetVal;
  MrcGetSetChStrb (MrcData, Controller, Channel, 0, GsmIocDataDccSaveFullDcc, ReadFromCache, &GetSetVal);
  OrigSaveFullDcc = (UINT8) GetSetVal;
  MrcGetSetChStrb (MrcData, Controller, Channel, 0, GsmIocDataDccSkipCRWrite, ReadFromCache, &GetSetVal);
  OrigSkipCrWrite = (UINT8) GetSetVal;

  //TxDq Pad node
  GetSetVal = 2;
  MrcGetSetChStrb (MrcData, MAX_CONTROLLER, MAX_CHANNEL, MAX_SDRAM_IN_DIMM, GsmIocDataDccMeasPoint, WriteToCache, &GetSetVal);
  GetSetVal = 0;
  MrcGetSetChStrb (MrcData, MAX_CONTROLLER, MAX_CHANNEL, MAX_SDRAM_IN_DIMM, GsmIocDataDccSaveFullDcc, WriteToCache, &GetSetVal);
  GetSetVal = 1;
  MrcGetSetChStrb (MrcData, MAX_CONTROLLER, MAX_CHANNEL, MAX_SDRAM_IN_DIMM, GsmIocDataDccSkipCRWrite, WriteToCache, &GetSetVal);
  MrcFlushRegisterCachedData (MrcData);

  MrcCall->MrcSetMem ((UINT8 *) DataDccLaneStatus, sizeof (DataDccLaneStatus), 0);
  MrcCall->MrcSetMem ((UINT8 *) BestIndex, sizeof (BestIndex), 0xFF);

  for (Rank = 0; Rank < MAX_RANK_IN_CHANNEL; Rank++) {
    if ((CurrentMcChBitMask & FullMcChBitMask) == FullMcChBitMask) {
      //Done all bytes has been covered
      break;
    }
    McChBitMask = 0;
    for (Controller = 0; Controller < MAX_CONTROLLER; Controller++) {
      for (Channel = 0; Channel < MaxChannels; Channel++) {
        McChBitMask |= SelectReutRanks (MrcData, (UINT8) Controller, (UINT8) Channel, 1 << Rank, FALSE, 0);
      }
    }
    if (((McChBitMask ^ CurrentMcChBitMask) == 0) || (McChBitMask == 0)) {
      //No new bytes being tested or no Rank present
      continue;
    }
    MRC_DEBUG_MSG (Debug, MSG_LEVEL_ALGO, "Rank %d\nTxTco\t", Rank);

    //Dummy LFSR?
    SetupIOTestBasicVA (MrcData, McChBitMask, 1, NSOE, 0, 0, 10, PatWrEndless, 0, 0);
    Cpgc20ControlRegWrite (MrcData, McChBitMask, CpgcStartTest);

    //Sweep TxTco
    for (Index = 0; Index < MAX_TCO_COMP; Index++) {
      MRC_DEBUG_MSG (Debug, MSG_LEVEL_ALGO, "\t%d", Index);
      GetSetVal = Index;
      MrcGetSetBit (MrcData, MAX_CONTROLLER, MAX_CHANNEL, MRC_IGNORE_ARG, MAX_SDRAM_IN_DIMM, MAX_BITS, TxTco, WriteToCache, &GetSetVal);
      MrcFlushRegisterCachedData (MrcData);
      ForceRcomp (MrcData);

      for (Controller = 0; Controller < MAX_CONTROLLER; Controller++) {
        for (Channel = 0; Channel < MaxChannels; Channel++) {
          for (Byte = 0; Byte < MaxSdram; Byte++) {
            if (!MrcByteExist (MrcData, Controller, Channel, Byte)) {
              continue;
            }
            for (Bit = 0; Bit < MAX_BITS; Bit++) {
              MrcGetSetBit (MrcData, Controller, Channel, MRC_IGNORE_ARG, Byte, Bit, GsmIocDataDqDccLaneStatusResult, ReadCached, &GetSetVal);
              DataDccLaneStatus[Controller][Channel][Byte][Bit][Index] = (UINT16) GetSetVal;
            } // for Bit
          } // for Byte
        } // for Channel
      } // for Controller
    } // for Index
    Cpgc20ControlRegWrite (MrcData, McChBitMask, CpgcStopTest);

    //Print out results and find closest to HALF_DUTY_CYCLE
    for (Controller = 0; Controller < MAX_CONTROLLER; Controller++) {
      for (Channel = 0; Channel < MaxChannels; Channel++) {
        for (Byte = 0; Byte < MaxSdram; Byte++) {
          if (!MrcByteExist (MrcData, Controller, Channel, Byte)) {
            continue;
          }
          for (Bit = 0; Bit < MAX_BITS; Bit++) {
            Data16Ptr = DataDccLaneStatus[Controller][Channel][Byte][Bit];
            MRC_DEBUG_MSG (Debug, MSG_LEVEL_ALGO, "\nMc%d.C%d.S%d.B%d", Controller, Channel, Byte, Bit);
            HalfDutyCycle = MRC_UINT16_MAX;
            for (Index = 0; Index < MAX_TCO_COMP; Index++) {
              MRC_DEBUG_MSG (Debug, MSG_LEVEL_ALGO, "\t%d", Data16Ptr[Index]);
              Data16 = ABS (Data16Ptr[Index] - HALF_DUTY_CYCLE);
              if (Data16 < HalfDutyCycle) {
                HalfDutyCycle = Data16;
                BestIndex[Controller][Channel][Byte][Bit] = Index;
              }
            }
          }
        }
      }
    }
    CurrentMcChBitMask |= McChBitMask; //Last one for Rank loop
  }

  for (Controller = 0; Controller < MAX_CONTROLLER; Controller++) {
    for (Channel = 0; Channel < MaxChannels; Channel++) {
      for (Byte = 0; Byte < MaxSdram; Byte++) {
        if (!MrcByteExist (MrcData, Controller, Channel, Byte)) {
          continue;
        }
        for (Bit = 0; Bit < MAX_BITS; Bit++) {
          GetSetVal = BestIndex[Controller][Channel][Byte][Bit];
          MrcGetSetBit (MrcData, MAX_CONTROLLER, MAX_CHANNEL, MRC_IGNORE_ARG, MAX_SDRAM_IN_DIMM, MAX_BITS, TxTco, WriteToCache | PrintValue, &GetSetVal);
        }
      }
    }
  }
  MrcFlushRegisterCachedData (MrcData);

  //Restore DccFsmControl
  GetSetVal = OrigMeasPoint;
  MrcGetSetChStrb (MrcData, MAX_CONTROLLER, MAX_CHANNEL, MAX_SDRAM_IN_DIMM, GsmIocDataDccMeasPoint, WriteToCache, &GetSetVal);
  GetSetVal = OrigSaveFullDcc;
  MrcGetSetChStrb (MrcData, MAX_CONTROLLER, MAX_CHANNEL, MAX_SDRAM_IN_DIMM, GsmIocDataDccSaveFullDcc, WriteToCache, &GetSetVal);
  GetSetVal = OrigSkipCrWrite;
  MrcGetSetChStrb (MrcData, MAX_CONTROLLER, MAX_CHANNEL, MAX_SDRAM_IN_DIMM, GsmIocDataDccSkipCRWrite, WriteToCache, &GetSetVal);
  MrcFlushRegisterCachedData (MrcData);

  MRC_DEBUG_MSG (Debug, MSG_LEVEL_ALGO, "\n%sDcc_Adjust_TcoComp\n", gStopTagStr);

  return mrcSuccess;
}
/**
  This function handles the Lpddr Overshoot problem


  @param[in, out] MrcData - Include all MRC global data.

**/
VOID
MrcLpddrOvershoot (
  IN OUT MrcParameters *const MrcData
  )
{
//  MrcOutput  *Outputs;
//  MrcInput   *Inputs;
//  INT64      OdtSingleSeqEn;
//  INT64      DqsNParkLow;
//  INT64      DqsOdtParkMode;
//  INT64      DisableOdtStatic;
//  INT64      CompDqOdtVrefDn;
//  INT64      CompDqOdtVrefUp;
//  INT64      GetSetVal;
//  INT64      RcompOdt;
//  INT64      GetSetEn;
//  UINT32     FirstController;
//  UINT32     FirstChannel;
//  UINT32     ROdtTarget;
//  UINT32     Var;
//  BOOLEAN    Lpddr4;
//  DDRPHY_COMP_CR_DDRCRDATACOMP0_STRUCT  RcompData0;
//
//  Outputs = &MrcData->Outputs;
//  Inputs  = &MrcData->Inputs;
//  Lpddr4 = (Outputs->DdrType == MRC_DDR_TYPE_LPDDR4);
//  FirstController = (MrcControllerExist (MrcData, cCONTROLLER0)) ? 0 : 1;
//  FirstChannel = Outputs->Controller[FirstController].FirstPopCh;
//  GetSetEn = 1;
//
//  // Read the first populated byte
//  MrcGetSetChStrb (MrcData, FirstController, FirstChannel, 0, GsmIocDataODTSingleSegEn, ReadFromCache, &OdtSingleSeqEn);
//  MrcGetSetChStrb (MrcData, FirstController, FirstChannel, 0, GsmIocDataDqsNParkLow, ReadFromCache, &DqsNParkLow);
//  MrcGetSetChStrb (MrcData, FirstController, FirstChannel, 0, GsmIocDataDqsOdtParkMode, ReadFromCache, &DqsOdtParkMode);
//  MrcGetSetNoScope (MrcData, GsmIocCompOdtStaticDis, ReadFromCache, &DisableOdtStatic);
//  MrcGetSetNoScope (MrcData, CompRcompOdtDn, ReadUncached, &RcompOdt);
//  MrcGetSetNoScope (MrcData, DqOdtVrefDn, ReadUncached, &CompDqOdtVrefDn);
//  MrcGetSetNoScope (MrcData, DqOdtVrefUp, ReadUncached, &CompDqOdtVrefUp);
//  RcompData0.Data = MrcReadCR (MrcData, DDRPHY_COMP_CR_DDRCRDATACOMP0_REG);
//
//#ifdef LOCAL_STUB_FLAG
//  CompDqOdtVrefDn = 0xF;
//  CompDqOdtVrefUp = 0xF;
//#endif
//  ROdtTarget = ((191 * Inputs->RcompResistor) / (UINT32) CompDqOdtVrefUp) - Inputs->RcompResistor;
//  if (ROdtTarget < 55) {
//    GetSetVal = (191 * Inputs->RcompResistor) / (Inputs->RcompResistor + 55); // Increase DqOdt target to atleast 55 Ohms
//    MrcGetSetNoScope (MrcData, DqOdtVrefUp, WriteCached, &GetSetVal);
//  }
//
//  if (Lpddr4) {
//    MrcGetSetChStrb (MrcData, MAX_CONTROLLER, MAX_CHANNEL, MAX_SDRAM_IN_DIMM, GsmIocDataODTSingleSegEn, WriteCached, &GetSetEn);
//  }
//  if (((DqsOdtParkMode == 1) || (DqsOdtParkMode == 2)) && (Outputs->Lpddr) && (Outputs->VccddqVoltage > VDD_0_75)) {
//    MrcGetSetChStrb (MrcData, MAX_CONTROLLER, MAX_CHANNEL, MAX_SDRAM_IN_DIMM, GsmIocDataDqsNParkLow, WriteCached, &GetSetEn);
//  }
//  MrcGetSetChStrb (MrcData, FirstController, FirstChannel, 0, GsmIocDataDqsNParkLow, ReadCached, &GetSetVal);
//
//  if (GetSetVal) {
//    if ((DqsOdtParkMode == 3) && (OdtSingleSeqEn == 0)) {
//      Var = (UINT32) RcompOdt;
//      RcompData0.Bits.RParkOdtDown = Var >> 1;
//      MrcWriteCR (MrcData, DDRPHY_COMP_CR_DDRCRDATACOMP0_REG, RcompData0.Data);
//    } else {
//      MrcGetSetChStrb (MrcData, MAX_CONTROLLER, MAX_CHANNEL, MAX_SDRAM_IN_DIMM, GsmIocDataODTSingleSegEn, WriteCached, &GetSetEn);
//      MrcGetSetNoScope (MrcData, GsmIocCompOdtStaticDis, WriteCached, &GetSetEn);
//      Var = (191 * VDD_0_70) / Outputs->VccddqVoltage;
//      Var = MIN (Var, 191);
//      GetSetVal = Var;
//      MrcGetSetNoScope (MrcData, DqOdtVrefDn, WriteCached, &GetSetVal);
//      ForceRcomp (MrcData);
//      Var = (UINT32) RcompOdt;
//      RcompData0.Bits.RParkOdtDown = Var >> 1;
//      MrcWriteCR (MrcData, DDRPHY_COMP_CR_DDRCRDATACOMP0_REG, RcompData0.Data);
//      MrcGetSetNoScope (MrcData, GsmIocCompOdtStaticDis, WriteCached, &DisableOdtStatic); // Restore DisableOdtStatic
//      ForceRcomp (MrcData);
//      MrcWriteCrMulticast (MrcData, DATA_CR_RCOMPDATA0_REG, RcompData0.Data);
//      MrcGetSetNoScope (MrcData, GsmIocForceCmpUpdt, WriteCached, &GetSetEn);
//    }
//  }
//
//  // Restore
//  MrcGetSetChStrb (MrcData, MAX_CONTROLLER, MAX_CHANNEL, MAX_SDRAM_IN_DIMM, GsmIocDataODTSingleSegEn, WriteToCache, &OdtSingleSeqEn);
//  MrcGetSetChStrb (MrcData, MAX_CONTROLLER, MAX_CHANNEL, MAX_SDRAM_IN_DIMM, GsmIocDataDqsNParkLow, WriteToCache, &DqsNParkLow);
//  MrcGetSetChStrb (MrcData, MAX_CONTROLLER, MAX_CHANNEL, MAX_SDRAM_IN_DIMM, GsmIocDataDqsOdtParkMode, WriteToCache, &DqsOdtParkMode);
//  MrcGetSetNoScope (MrcData, GsmIocCompOdtStaticDis, WriteToCache, &DisableOdtStatic);
//  MrcFlushRegisterCachedData(MrcData);
//  MrcGetSetNoScope (MrcData, CompRcompOdtDn, WriteCached, &RcompOdt);
//  MrcGetSetNoScope (MrcData, DqOdtVrefDn, WriteCached, &CompDqOdtVrefDn);
//  if (ROdtTarget < 55) {
//    MrcGetSetNoScope (MrcData, DqOdtVrefUp, WriteCached, &CompDqOdtVrefUp); //Restore if it was changed
//  }
//  ForceRcomp(MrcData); // Need to re-run the comp after restoring everything
}

/**
  This routine returns the TX FIFO separation based on technology

  @param[in]  MrcData           - Pointer to MRC global data.

  @retval TxFifoSeparation.
**/

INT32
MrcGetTxFifoSeparation (
  IN OUT MrcParameters *const MrcData
  )
{
  INT32      TxFifoSeparation;
  UINT8      Gear2;
  BOOLEAN    Lpddr4;
  BOOLEAN    Lpddr5;
  BOOLEAN    Ddr5;
  BOOLEAN    UlxUlt;
  MrcOutput  *Outputs;
  MrcInput   *Inputs;

  Outputs  = &MrcData->Outputs;
  Inputs   = &MrcData->Inputs;
  Lpddr5   = (Outputs->DdrType == MRC_DDR_TYPE_LPDDR5);
  Lpddr4   = (Outputs->DdrType == MRC_DDR_TYPE_LPDDR4);
  Ddr5     = (Outputs->DdrType == MRC_DDR_TYPE_DDR5);
  Gear2    = (Outputs->Gear2) ? 1 : 0;
  UlxUlt = Inputs->UlxUlt;

  TxFifoSeparation = -1;
  TxFifoSeparation -= ((UlxUlt) ? 1 : 2);
  if (Lpddr4) {
    TxFifoSeparation -= (Gear2 ? 2 : 1);  // Subtract 1 due to 2tCK WRPRE
  } else if (Lpddr5) {
    TxFifoSeparation -= 1;
  } else if (Ddr5) {
    TxFifoSeparation -= 3;
  }

  return TxFifoSeparation;
}

/**
  This routine computes the read and write fifo pointer delays

  @param[in]  MrcData           - Pointer to MRC global data.
  @param[in]  Controller        - Controller to get timings
  @param[in]  Channel           - Controller to get timings
  @param[in]  tCWL              - Write Latency for current channel
  @param[in]  AddTcwl           - Current AddTcwl value
  @param[in]  DecTcwl           - Current DecTcwl value
  @param[out] tCWL4TxDqFifoWrEn - Pointer to the write TX DQ fifo delay
  @param[out] tCWL4TxDqFifoRdEn - Pointer to the read TX DQ fifo delay

  @retval N/A.
**/
VOID
MrcGetTxDqFifoDelays (
  IN OUT MrcParameters *const MrcData,
  IN     UINT32        Controller,
  IN     UINT32        Channel,
  IN     INT32         tCWL,
  IN     UINT32        AddTcwl,
  IN     UINT32        DecTcwl,
     OUT INT64         *tCWL4TxDqFifoWrEn,
     OUT INT64         *tCWL4TxDqFifoRdEn
  )
{
  UINT8      Gear2;
  BOOLEAN    Lpddr;
  BOOLEAN    Lpddr5;
  BOOLEAN    Ddr5;
  MrcOutput  *Outputs;

  Outputs  = &MrcData->Outputs;
  Lpddr    = Outputs->Lpddr;
  Lpddr5   = (Outputs->DdrType == MRC_DDR_TYPE_LPDDR5);
  Ddr5     = (Outputs->DdrType == MRC_DDR_TYPE_DDR5);
  Gear2    = (Outputs->Gear2) ? 1 : 0;

  if (Gear2) {
    *tCWL4TxDqFifoWrEn = tCWL - (2 * DecTcwl) + (2 * AddTcwl) - (Lpddr ? 1 : 3) + (tCWL % 2) - 2 * Lpddr * (tCWL % 2);
    if (Lpddr5) {
      *tCWL4TxDqFifoWrEn -= 8;
    }
    if (Ddr5) {
      *tCWL4TxDqFifoWrEn += 2;
    }
  } else {
    *tCWL4TxDqFifoWrEn  = tCWL - DecTcwl + AddTcwl - 2; // TxDqFifoWrEnTcwlDelay(DClk)
  }
  *tCWL4TxDqFifoRdEn  = *tCWL4TxDqFifoWrEn + MrcGetTxFifoSeparation (MrcData);

}

/**
  This function runs Reset Recovery

  @param[in, out] MrcData - Include all MRC global data.

  @retval mrcSuccess if MC1 not populated or if no Reset Recovery needed otherwise return mrcResetRecovery
**/
MrcStatus
MrcResetRecovery (
  IN OUT MrcParameters *const MrcData
  )
{
  MrcOutput  *Outputs;
  MrcDebug   *Debug;
  MrcStatus  Status;
  UINT8      McChBitMask;
  UINT8      RankMask;
  UINT8      MaxChannel;
  UINT8      Index;
  UINT8      Iteration;
  UINT32     Controller;
  UINT32     Channel;
  UINT32     Rank;
  MC1_STALL_DRAIN_STRUCT  StallDrain;
  MC0_MCDECS_MISC_STRUCT  Mc0McdecsMisc;
  MC0_REQ0_CR_CPGC_SEQ_CTL_STRUCT  CpgcStartTest;
  MC0_REQ0_CR_CPGC_SEQ_CTL_STRUCT  CpgcStopTest;

  Outputs = &MrcData->Outputs;
  Debug = &Outputs->Debug;
  MaxChannel = Outputs->MaxChannels;
  Status = mrcSuccess;
  Iteration = MrcData->Inputs.Iteration;

  if (MrcControllerExist (MrcData, cCONTROLLER1) == FALSE) {
    //No MC1
    return Status;
  }

  SetupIOTestBasicVA (MrcData, Outputs->McChBitMask, 1, NSOE, 0, 0, 8, PatWrRd, 0, 8);

  McChBitMask = 0;
  for (Rank = 0; Rank < MAX_RANK_IN_CHANNEL; Rank++) {
    RankMask = 1 << Rank;
    if ((RankMask & Outputs->ValidRankMask) == 0) {
      continue;
    }
    for (Controller = 0; Controller < MAX_CONTROLLER; Controller++) {
      for (Channel = 0; Channel < MaxChannel; Channel++) {
        McChBitMask |= SelectReutRanks (MrcData, (UINT8) Controller, (UINT8) Channel, RankMask, FALSE, 0);
      }
    }
    break;
  }
  MRC_DEBUG_MSG (Debug, MSG_LEVEL_ERROR, "Issues %d write transactions!", MAX_RESET_RECOVERY_WRITES);
  CpgcStartTest.Data = CpgcStopTest.Data = 0;
  CpgcStartTest.Bits.START_TEST = 1;
  CpgcStopTest.Bits.STOP_TEST = 1;

  for (Index = 0; Index < MAX_RESET_RECOVERY_WRITES; Index++) {
    Cpgc20ControlRegWrite (MrcData, McChBitMask, CpgcStartTest);
    MrcWait (MrcData, 10); //Wait 10ns
    Cpgc20ControlRegWrite (MrcData, McChBitMask, CpgcStopTest);
  }

  StallDrain.Data = MrcReadCR (MrcData, MC1_STALL_DRAIN_REG);
  if (StallDrain.Bits.mc_drained == 0) {
    MRC_DEBUG_MSG (Debug, MSG_LEVEL_ERROR, "%s Detected bad Reset in iter %d!", gErrString, Iteration);
    Status = mrcResetRecovery;
    Mc0McdecsMisc.Data = MrcReadCR (MrcData, MC0_MCDECS_MISC_REG);
    MRC_DEBUG_MSG (Debug, MSG_LEVEL_ERROR, "Current State of Mc0McdecsMisc 0x%x, mrcReqMcReset (Bit0) %d, PunitMCresetDone (Bit1) %d!\nIssue McReset", Mc0McdecsMisc.Data, Mc0McdecsMisc.Bits.Spare_RW & 0x1, (Mc0McdecsMisc.Bits.Spare_RW >> 1) & 0x1);
    //Clear out PunitMCresetDone (Bit1) and set mrcReqMcReset (Bit0)
    Mc0McdecsMisc.Bits.Spare_RW &= ~(1 << 1);
    Mc0McdecsMisc.Bits.Spare_RW |= 1;
    MrcWriteCR (MrcData, MC0_MCDECS_MISC_REG, Mc0McdecsMisc.Data);
    MrcWait (MrcData, 2000); //Wait 2ms for MC reset to occur
    //Poll on Bit1
    do {
      Mc0McdecsMisc.Data = MrcReadCR (MrcData, MC0_MCDECS_MISC_REG);
      Index = (Mc0McdecsMisc.Bits.Spare_RW >> 1) & 0x1;
    } while (Index == 0);
    MRC_DEBUG_MSG (Debug, MSG_LEVEL_ERROR, "Final State of Mc0McdecsMisc 0x%x, mrcReqMcReset (Bit0) %d, PunitMCresetDone (Bit1) %d!", Mc0McdecsMisc.Data, Mc0McdecsMisc.Bits.Spare_RW & 0x1, (Mc0McdecsMisc.Bits.Spare_RW >> 1) & 0x1);
  } else {
    MRC_DEBUG_MSG (Debug, MSG_LEVEL_ERROR, "Reset de-asserted correctly in iter %d!", Iteration);
  }
  return Status;
}

/**
  This function sets up Phy registers that needs clearing after previous SAGV point restore

  @param[in, out] MrcData - Include all MRC global data.
**/
void
MrcDdrPhyRegisterSettings (
  IN OUT MrcParameters *const MrcData
  )
{
  MrcInput   *Inputs;
  MrcOutput  *Outputs;
  UINT32   Controller;
  UINT32   Channel;
  UINT32   IpChannel;
  UINT32   PrevSafeMode;
  BOOLEAN  Lpddr;

  Inputs = &MrcData->Inputs;
  Outputs = &MrcData->Outputs;
  Lpddr = Outputs->Lpddr;
  PrevSafeMode = Inputs->SafeMode;

  Inputs->SafeMode = TRUE;

  for (Controller = 0; Controller < MAX_CONTROLLER; Controller++) {
    for (Channel = 0; Channel < Outputs->MaxChannels; Channel++) {
      if (!MrcChannelExist (MrcData, Controller, Channel) || IS_MC_SUB_CH (Lpddr, Channel)) {
        continue;
      }
      IpChannel = LP_IP_CH (Lpddr, Channel);
      if (Inputs->A0) {
        ConfigA0SpidLowPowerCtl (MrcData, Controller, IpChannel);
      } else {
        // B0 and P0
        ConfigSpidLowPowerCtl (MrcData, Controller, IpChannel);
      }
    }
  }

  Inputs->SafeMode = PrevSafeMode;
}
