/** @file
  .

@copyright
  INTEL CONFIDENTIAL
  Copyright 2018 - 2020 Intel Corporation.

  The source code contained or described herein and all documents related to the
  source code ("Material") are owned by Intel Corporation or its suppliers or
  licensors. Title to the Material remains with Intel Corporation or its suppliers
  and licensors. The Material may contain trade secrets and proprietary and
  confidential information of Intel Corporation and its suppliers and licensors,
  and is protected by worldwide copyright and trade secret laws and treaty
  provisions. No part of the Material may be used, copied, reproduced, modified,
  published, uploaded, posted, transmitted, distributed, or disclosed in any way
  without Intel's prior express written permission.

  No license under any patent, copyright, trade secret or other intellectual
  property right is granted to or conferred upon you by disclosure or delivery
  of the Materials, either expressly, by implication, inducement, estoppel or
  otherwise. Any license under such intellectual property rights must be
  express and approved by Intel in writing.

  Unless otherwise agreed by Intel in writing, you may not remove or alter
  this notice or any other notice embedded in Materials by Intel or
  Intel's suppliers or licensors in any way.

  This file contains an 'Intel Peripheral Driver' and is uniquely identified as
  "Intel Reference Module" and is licensed for Intel CPUs and chipsets under
  the terms of your license agreement with Intel or your vendor. This file may
  be modified by the user, subject to additional terms of the license agreement.

@par Specification Reference:
**/
#include "MrcTypes.h"
#include "MrcInterface.h"
#include "MrcHalRegisterAccess.h"
#include "MrcCommon.h"
#include "MrcChipApi.h"
#include "MrcMemoryApi.h"
#include "MrcDdrIoApiInt.h"
#include "MrcDdr5.h"
#include "MrcDdr5Registers.h"

#define GEN_MRS_FSM_TIMING_SET_ULXULT_MAX  (4)
#define GEN_MRS_FSM_TIMING_SET_DTHALO_MAX  (6)
#define GEN_MRS_FSM_TIMING_SET_MAX         (GEN_MRS_FSM_TIMING_SET_DTHALO_MAX) // MAX (GEN_MRS_FSM_TIMING_SET_ULXULT_MAX, GEN_MRS_FSM_TIMING_SET_DTHALO_MAX)
#define GEN_MRS_FSM_FLAT_TIME_MAX          (127) // BITS[6:0]
#define GEN_MRS_FSM_TIME_SCALAR            (16)
#define GEN_MRS_FSM_TIME_SCALAR_P0_x16     (16)
#define GEN_MRS_FSM_TIME_SCALAR_P0_x32     (32)
#define GEN_MRS_FSM_TIME_SCALAR_P0_x64     (64)
#define GEN_MRS_FSM_TIME_SCALAR_BIT        (0x80)
#define GEN_MRS_FSM_TIME_SCALAR_BIT_P0_x16 (0x080)
#define GEN_MRS_FSM_TIME_SCALAR_BIT_P0_x32 (0x100)
#define GEN_MRS_FSM_TIME_SCALAR_BIT_P0_x64 (0x180)


/**
  This function configures the Generic MRS FSM timing registers and returns the GENERIC_MRS_FSM_TIMING
  register byte index for the requested MR delay. This function uses the input "TimingSet" array to keep
  track of unique MR delay values and their corresponding byte index.

  @param[in]     MrcData            - Pointer to MRC global data.
  @param[in]     Controller         - Handle to the controller
  @param[in]     McCh               - Handle to channel within the memory controller
  @param[in]     Delay              - The required delay in units of tCK
  @param[in]     Valid              - Indicates that the input ControlRegIndx marked as "Valid".
  @param[out]    TimingSetIdx       - Pointer used to return the timing register byte index for the input delay.
  @param[in,out] TimingSet          - Pointer to an array used to keep track of the indexes to unique MrDelay entries
                                      in the timing registers.
  @param[in,out] TimingSetNumValid  - Pointer to a variable used to keep track of the number of valid entries in the
                                      TimingSet array

  @retval mrcSuccess if successful.
  @retval mrcFail if the timing registers are full and the requested Delay could not be allocated.
**/
MrcStatus
ConfigGenMrsFsmTiming  (
  IN     MrcParameters *MrcData,
  IN     UINT32        Controller,
  IN     UINT32        McCh,
  IN     UINT16        Delay,
     OUT UINT32        *TimingSetIdx,
  IN OUT UINT16        (*TimingSet)[GEN_MRS_FSM_TIMING_SET_MAX],
  IN OUT UINT8         *TimingSetNumValid
  )
{
  MrcInput  *Inputs;
  MrcOutput *Outputs;
  UINT32    Offset;
  UINT16    MrDelay;
  UINT32    ScaledTimeMax;
  UINT16    TimingEncode;
  UINT8     NumTimingFields;
  BOOLEAN   Lpddr5;
  MC0_CH0_CR_GENERIC_MRS_FSM_TIMING_STORAGE_STRUCT GenMrsFsmTiming;
  MC0_CH0_CR_GENERIC_MRS_FSM_TIMING_STORAGE_0_P0_STRUCT GenMrsFsmTimingP0;

  Inputs = &MrcData->Inputs;
  Outputs = &MrcData->Outputs;
  Lpddr5 = (Outputs->DdrType == MRC_DDR_TYPE_LPDDR5);

  if (Lpddr5) {
    // Need to scale from tCK to WCK
    MrDelay = Delay * 4;
  } else {
    MrDelay = Delay;
  }

  // Determine if there is a timing already.
  for ((*TimingSetIdx) = 0; (*TimingSetIdx) < (*TimingSetNumValid); (*TimingSetIdx)++) {
    if ((*TimingSet)[*TimingSetIdx] == MrDelay) {
      break;
    }
  }
  // If not, add it if the register is not full.
  if ((*TimingSetIdx) >= (*TimingSetNumValid)) {
    if (Inputs->UlxUlt) {
      ScaledTimeMax = GEN_MRS_FSM_FLAT_TIME_MAX * GEN_MRS_FSM_TIME_SCALAR;
      if (MrDelay > ScaledTimeMax) {
        // Check if the timing value is larger than what the register supports.
        return mrcWrongInputParameter;
      } else if (MrDelay > GEN_MRS_FSM_FLAT_TIME_MAX) {
        // Round to the nearest multiple of the scalar based value and set the bit to enable the scalar
        TimingEncode = ((UINT16) DIVIDECEIL (MrDelay, GEN_MRS_FSM_TIME_SCALAR)) | GEN_MRS_FSM_TIME_SCALAR_BIT;
      } else {
        TimingEncode = MrDelay;
      }
    } else {
      ScaledTimeMax = (GEN_MRS_FSM_FLAT_TIME_MAX * GEN_MRS_FSM_TIME_SCALAR_P0_x64);
      if (MrDelay > ScaledTimeMax) {
        // Check if the timing value is larger than what the register supports.
        return mrcWrongInputParameter;
      } else if (MrDelay > (GEN_MRS_FSM_FLAT_TIME_MAX * GEN_MRS_FSM_TIME_SCALAR_P0_x32)) {
        // Round to the nearest multiple of the scalar based value and set the bit to enable the scalar
        TimingEncode = ((UINT16) DIVIDECEIL (MrDelay, GEN_MRS_FSM_TIME_SCALAR_P0_x64)) | GEN_MRS_FSM_TIME_SCALAR_BIT_P0_x64;
      } else if (MrDelay > (GEN_MRS_FSM_FLAT_TIME_MAX * GEN_MRS_FSM_TIME_SCALAR_P0_x16)) {
        // Round to the nearest multiple of the scalar based value and set the bit to enable the scalar
        TimingEncode = ((UINT16) DIVIDECEIL (MrDelay, GEN_MRS_FSM_TIME_SCALAR_P0_x32)) | GEN_MRS_FSM_TIME_SCALAR_BIT_P0_x32;
      } else if (MrDelay > GEN_MRS_FSM_FLAT_TIME_MAX) {
        // Round to the nearest multiple of the scalar based value and set the bit to enable the scalar
        TimingEncode = ((UINT16) DIVIDECEIL (MrDelay, GEN_MRS_FSM_TIME_SCALAR_P0_x16)) | GEN_MRS_FSM_TIME_SCALAR_BIT_P0_x16;
      } else {
        TimingEncode = MrDelay;
      }
    }

    // Check if we've run out of hardware timing entries
    NumTimingFields = Inputs->UlxUlt ? GEN_MRS_FSM_TIMING_SET_ULXULT_MAX : GEN_MRS_FSM_TIMING_SET_DTHALO_MAX;
    if ((*TimingSetIdx) >= NumTimingFields) {
      return mrcFail;
    }

    // Calculate the register offset
    if (Inputs->UlxUlt) {
      Offset = OFFSET_CALC_MC_CH (
        MC0_CH0_CR_GENERIC_MRS_FSM_TIMING_STORAGE_REG,
        MC1_CH0_CR_GENERIC_MRS_FSM_TIMING_STORAGE_REG,
        Controller,
        MC0_CH1_CR_GENERIC_MRS_FSM_TIMING_STORAGE_REG,
        McCh
        );

      GenMrsFsmTiming.Data = MrcReadCR (MrcData, Offset);
      switch ((*TimingSetIdx) % 4) {
        case 0:
          GenMrsFsmTiming.Bits.TIMING_FIELD_0 = TimingEncode;
          break;

        case 1:
          GenMrsFsmTiming.Bits.TIMING_FIELD_1 = TimingEncode;
          break;

        case 2:
          GenMrsFsmTiming.Bits.TIMING_FIELD_2 = TimingEncode;
          break;

        case 3:
          GenMrsFsmTiming.Bits.TIMING_FIELD_3 = TimingEncode;
          break;

        default:
          return mrcFail;
      }
      MrcWriteCR (MrcData, Offset, GenMrsFsmTiming.Data);
    } else {
      // TGL-H has 6 entries, 3 in each register
      if ((*TimingSetIdx) < 3) {
        Offset = OFFSET_CALC_MC_CH (
          MC0_CH0_CR_GENERIC_MRS_FSM_TIMING_STORAGE_0_P0_REG,
          MC1_CH0_CR_GENERIC_MRS_FSM_TIMING_STORAGE_0_P0_REG,
          Controller,
          MC0_CH1_CR_GENERIC_MRS_FSM_TIMING_STORAGE_0_P0_REG,
          McCh
          );
      } else {
        Offset = OFFSET_CALC_MC_CH (
          MC0_CH0_CR_GENERIC_MRS_FSM_TIMING_STORAGE_1_P0_REG,
          MC1_CH0_CR_GENERIC_MRS_FSM_TIMING_STORAGE_1_P0_REG,
          Controller,
          MC0_CH1_CR_GENERIC_MRS_FSM_TIMING_STORAGE_1_P0_REG,
          McCh
          );
      }

      GenMrsFsmTimingP0.Data = MrcReadCR (MrcData, Offset);
      switch ((*TimingSetIdx) % 3) {
        case 0:
          GenMrsFsmTimingP0.P0Bits.TIMING_FIELD_0 = TimingEncode;
          break;

        case 1:
          GenMrsFsmTimingP0.P0Bits.TIMING_FIELD_1 = TimingEncode;
          break;

        case 2:
          GenMrsFsmTimingP0.P0Bits.TIMING_FIELD_2 = TimingEncode;
          break;

        default:
          return mrcFail;
      }
      MrcWriteCR (MrcData, Offset, GenMrsFsmTimingP0.Data);
    }

    (*TimingSet)[*TimingSetIdx] = MrDelay;
    (*TimingSetNumValid)++;
  }
  return mrcSuccess;
}

/**
  This function configures the Generic MRS FSM shadow registers as specified by the
  input parameters. This function will handle programming of the control
  registers and will program the appropriate MR value register type:
  TGL UlxUly: GENERIC_MRS_FSM_PER_RANK_VALUES is used when MrPerRank == TRUE
              GENERIC_MRS_FSM_SHADOW_VALUES is used when MrPerRank == FALSE
  TGL DtHalo: GENERIC_MRS_FSM_STORAGE_VALUES is used

  @param[in]     MrcData       - Pointer to MRC global data.
  @param[in]     Controller    - Handle to the controller
  @param[in]     McCh          - Handle to channel within the memory controller
  @param[in]     GenMrRef      - Pointer to an entry from the MrData array belonging to the ControlRegIdx values to be set.
  @param[in]     MrPerRank     - The DRAM Mode Register requires unique values for each rank.
  @param[in]     TimingSetIdx  - The timing register byte index to use for the input ControlRegIdx.
  @param[in]     ControlRegIdx - The MrData array entry to use when configuring the shadow registers.
  @param[in]     MrData        - Pointer to an array of MR data to configure the MRS FSM with.
  @param[in,out] StorageIdx    - TGL UlxUlt: StorageIdx points to the next available GENERIC_MRS_FSM_PER_RANK_VALUES
                                 register.
                                 TGL DtHalo: StorageIdx points to the next available GENERIC_MRS_FSM_STORAGE_VALUES
                                 byte.

  @retval mrcSuccess if successful.
  @retval mrcFail if the shadow registers are out of free entries.
**/
MrcStatus
ConfigGenMrsFsmValue  (
  IN     MrcParameters           *MrcData,
  IN     UINT32                  Controller,
  IN     UINT32                  McCh,
  IN     MRC_GEN_MRS_FSM_MR_TYPE *GenMrRef,
  IN     BOOLEAN                 MrPerRank,
  IN     UINT32                  TimingSetIdx,
  IN     UINT8                   ControlRegIdx,
  IN     MRC_GEN_MRS_FSM_MR_TYPE MrData[MAX_CONTROLLER][MAX_CHANNEL][MAX_RANK_IN_CHANNEL][MAX_MR_GEN_FSM],
  IN OUT UINT8                   *StorageIdx
  )
{
  MrcDebug  *Debug;
  MrcInput  *Inputs;
  MrcOutput *Outputs;
  UINT32    ControlRegNum;
  UINT32    Channel;
  UINT32    MaxSubCh;
  UINT32    SubCh;
  UINT32    Offset;
  UINT32    Rank;
  UINT32    PerRankSwitch;
  UINT8     ShadowFieldIdx;
  BOOLEAN   Done;
  BOOLEAN   Lpddr;
  MRC_GEN_MRS_FSM_MR_TYPE                               *GenMrPtr;
  MC0_CH0_CR_GENERIC_MRS_FSM_CONTROL_0_STRUCT           GenMrsFsmCtl;
  MC0_CH0_CR_GENERIC_MRS_FSM_SHADOW_VALUES_0_STRUCT     GenMrsFsmChVal;
  MC0_CH0_CR_GENERIC_MRS_FSM_PER_RANK_VALUES_0_STRUCT   GenMrsFsmPerRankVal;
  MC0_CH0_CR_GENERIC_MRS_FSM_CONTROL_0_P0_STRUCT        GenMrsFsmCtlP0;
  MC0_CH0_CR_GENERIC_MRS_FSM_STORAGE_VALUES_0_P0_STRUCT GenMrsFsmChValP0;

  Inputs = &MrcData->Inputs;
  Outputs = &MrcData->Outputs;
  Debug = &Outputs->Debug;
  Lpddr = Outputs->Lpddr;
  MaxSubCh = (Lpddr) ? MAX_SUB_CHANNEL : 1;

  // StorageIdx is used differently on TGL UlxUlt and DtHalo
  // UlxUlt: StorageIdx points to the next available GENERIC_MRS_FSM_PER_RANK_VALUES register.
  // DtHalo: StorageIdx points to the next available GENERIC_MRS_FSM_STORAGE_VALUES byte. TGL DtHalo
  //         treats the byte fields in GENERIC_MRS_FSM_STORAGE_VALUES registers as an array of bytes.
  //         The GENERIC_MRS_FSM_CONTROL.GENERIC_MRS_STORAGE_POINTER must be programmed with a byte
  //         index into this array. Example: GENERIC_MRS_FSM_STORAGE_VALUES_27.BYTE_1 corresponds
  //         to byte index (27 * 4) + 1 = 109.
  if (Inputs->UlxUlt) {
    // Configure control register for this MR.
    GenMrsFsmCtl.Data = 0;
    GenMrsFsmCtl.Bits.ADDRESS = GenMrRef->MrAddr;
    GenMrsFsmCtl.Bits.ACTIVE = GenMrRef->Valid;
    GenMrsFsmCtl.Bits.TIMING_VALUE_POINTER = TimingSetIdx;
    GenMrsFsmCtl.Bits.FSP_CONTROL = (GenMrRef->FspWrToggle ? 1 : (GenMrRef->FspOpToggle ? 2 : 0));
    if (MrPerRank) {
      GenMrsFsmCtl.Bits.PER_RANK = 1;
      GenMrsFsmCtl.Bits.PER_RANK_STORAGE_POINTER = *StorageIdx;
    }
    Offset = MC0_CH0_CR_GENERIC_MRS_FSM_CONTROL_0_REG +
      ((MC1_CH0_CR_GENERIC_MRS_FSM_CONTROL_0_REG - MC0_CH0_CR_GENERIC_MRS_FSM_CONTROL_0_REG) * Controller) +
      ((MC0_CH1_CR_GENERIC_MRS_FSM_CONTROL_0_REG - MC0_CH0_CR_GENERIC_MRS_FSM_CONTROL_0_REG) * McCh) +
      ((MC0_CH0_CR_GENERIC_MRS_FSM_CONTROL_1_REG - MC0_CH0_CR_GENERIC_MRS_FSM_CONTROL_0_REG) * ControlRegIdx);
    MrcWriteCR (MrcData, Offset, GenMrsFsmCtl.Data);

    // Configure MR Value in the Shadow or Per Rank registers.
    if (MrPerRank) {
      GenMrsFsmPerRankVal.Data = 0;
      for (SubCh = 0; SubCh < MaxSubCh; SubCh++) {
        Channel = (McCh * MaxSubCh) + SubCh;
        for (Rank = 0; Rank < MAX_RANK_IN_DIMM; Rank++) {
          if (!MrcRankExist (MrcData, Controller, Channel, Rank)) {
            continue;
          }

          GenMrPtr = &MrData[Controller][Channel][Rank][ControlRegIdx];
          if (!GenMrPtr->Valid) {
            continue;
          }

          PerRankSwitch = (SubCh * MAX_RANK_IN_DIMM) + Rank;
          switch (PerRankSwitch) {
            case 0:
              GenMrsFsmPerRankVal.Bits.SHADOW_0 = GenMrPtr->MrData;
              break;

            case 1:
              GenMrsFsmPerRankVal.Bits.SHADOW_1 = GenMrPtr->MrData;
              break;

            case 2:
              GenMrsFsmPerRankVal.Bits.SHADOW_2 = GenMrPtr->MrData;
              break;

            case 3:
              GenMrsFsmPerRankVal.Bits.SHADOW_3 = GenMrPtr->MrData;
              break;

            default:
              MRC_DEBUG_MSG (Debug, MSG_LEVEL_ERROR, "Invalid case in Per Rank Switch: %d\n", PerRankSwitch);
              return mrcFail;
          }
        }
      }
      MRC_DEBUG_ASSERT ((*StorageIdx < GEN_MRS_FSM_PER_RANK_MAX), Debug, "Number of per-rank instances has exceeded the max supported by the MC\n");
      Offset = MC0_CH0_CR_GENERIC_MRS_FSM_PER_RANK_VALUES_0_REG +
        ((MC1_CH0_CR_GENERIC_MRS_FSM_PER_RANK_VALUES_0_REG - MC0_CH0_CR_GENERIC_MRS_FSM_PER_RANK_VALUES_0_REG) * Controller) +
        ((MC0_CH1_CR_GENERIC_MRS_FSM_PER_RANK_VALUES_0_REG - MC0_CH0_CR_GENERIC_MRS_FSM_PER_RANK_VALUES_0_REG) * McCh) +
        ((MC0_CH0_CR_GENERIC_MRS_FSM_PER_RANK_VALUES_1_REG - MC0_CH0_CR_GENERIC_MRS_FSM_PER_RANK_VALUES_0_REG) * (*StorageIdx));
      MrcWriteCR (MrcData, Offset, GenMrsFsmPerRankVal.Data);
      (*StorageIdx)++;
    } else {
      // Else, map it into the correct field of the Shadow Value (X % 4) where X is the control register instance.
      ControlRegNum = ControlRegIdx / 4;
      ShadowFieldIdx = ControlRegIdx % 4;

      // Find the first populated SubChannel or Rank and use that array entry to configure the shadow value.
      Done = FALSE;
      GenMrPtr = NULL;
      for (SubCh = 0; ((SubCh < MaxSubCh) && !Done); SubCh++) {
        for (Rank = 0; ((Rank < MAX_RANK_IN_CHANNEL) && !Done); Rank++) {
          Channel = (McCh * MaxSubCh) + SubCh;
          if (MrcRankExist (MrcData, Controller, Channel, Rank)) {
            GenMrPtr = &MrData[Controller][Channel][Rank][ControlRegIdx];
            Done = TRUE;
          }
        }
      }
      MRC_DEBUG_ASSERT ((GenMrPtr != NULL), Debug, "Did not find a populated SubCh/Rank\n");

      Offset = MC0_CH0_CR_GENERIC_MRS_FSM_SHADOW_VALUES_0_REG +
        ((MC1_CH0_CR_GENERIC_MRS_FSM_SHADOW_VALUES_0_REG - MC0_CH0_CR_GENERIC_MRS_FSM_SHADOW_VALUES_0_REG) * Controller) +
        ((MC0_CH1_CR_GENERIC_MRS_FSM_SHADOW_VALUES_0_REG - MC0_CH0_CR_GENERIC_MRS_FSM_SHADOW_VALUES_0_REG) * McCh) +
        ((MC0_CH0_CR_GENERIC_MRS_FSM_SHADOW_VALUES_1_REG - MC0_CH0_CR_GENERIC_MRS_FSM_SHADOW_VALUES_0_REG) * ControlRegNum);
      GenMrsFsmChVal.Data = MrcReadCR (MrcData, Offset);
      switch (ShadowFieldIdx) {
        case 0:
          GenMrsFsmChVal.Bits.SHADOW_0 = GenMrPtr->MrData;
          break;

        case 1:
          GenMrsFsmChVal.Bits.SHADOW_1 = GenMrPtr->MrData;
          break;

        case 2:
          GenMrsFsmChVal.Bits.SHADOW_2 = GenMrPtr->MrData;
          break;

        case 3:
          GenMrsFsmChVal.Bits.SHADOW_3 = GenMrPtr->MrData;
          break;
      }
      MrcWriteCR (MrcData, Offset, GenMrsFsmChVal.Data);
    }
  } else {
    // Inputs->DtHalo
    // Configure control register for this MR.
    GenMrsFsmCtlP0.Data = 0;
    GenMrsFsmCtlP0.P0Bits.COMMAND_TYPE = GenMrRef->CmdType;
    GenMrsFsmCtlP0.P0Bits.ADDRESS = GenMrRef->MrAddr;
    GenMrsFsmCtlP0.P0Bits.ACTIVE = GenMrRef->Valid;
    GenMrsFsmCtlP0.P0Bits.TIMING_VALUE_POINTER = TimingSetIdx;
    GenMrsFsmCtlP0.P0Bits.GENERIC_MRS_STORAGE_POINTER = *StorageIdx;
    GenMrsFsmCtlP0.P0Bits.FSP_CONTROL = (GenMrRef->FspWrToggle ? 1 : (GenMrRef->FspOpToggle ? 2 : 0));
    if (MrPerRank) {
      GenMrsFsmCtlP0.P0Bits.PER_RANK = 1;
    }
    Offset = MC0_CH0_CR_GENERIC_MRS_FSM_CONTROL_0_P0_REG +
      ((MC1_CH0_CR_GENERIC_MRS_FSM_CONTROL_0_P0_REG - MC0_CH0_CR_GENERIC_MRS_FSM_CONTROL_0_P0_REG) * Controller) +
      ((MC0_CH1_CR_GENERIC_MRS_FSM_CONTROL_0_P0_REG - MC0_CH0_CR_GENERIC_MRS_FSM_CONTROL_0_P0_REG) * McCh) +
      ((MC0_CH0_CR_GENERIC_MRS_FSM_CONTROL_1_P0_REG - MC0_CH0_CR_GENERIC_MRS_FSM_CONTROL_0_P0_REG) * ControlRegIdx);
    MrcWriteCR (MrcData, Offset, GenMrsFsmCtlP0.Data);

    // Configure MR Value registers.
    for (SubCh = 0; SubCh < MaxSubCh; SubCh++) {
      Channel = (McCh * MaxSubCh) + SubCh;
      for (Rank = 0; Rank < MAX_RANK_IN_DIMM; Rank++) {
        if (!MrcRankExist (MrcData, Controller, Channel, Rank)) {
          continue;
        }

        GenMrPtr = &MrData[Controller][Channel][Rank][ControlRegIdx];
        if (!GenMrPtr->Valid) {
          continue;
        }

        // Per Rank configurations may span multiple registers, including registers
        // where previous byte indexes are occupied by non per-rank values. To prevent
        // overwriting existing entries, do a read-modify-write for each new value.
        // Crossing register boundaries will be automatically handled by incrementing
        // StorageIdx for each new value written.
        MRC_DEBUG_ASSERT ((*StorageIdx < GEN_MRS_FSM_BYTE_P0_MAX), Debug, "Number of byte storage instances has exceeded the max supported by the MC\n");
        ControlRegNum = (*StorageIdx) / 4;
        Offset = MC0_CH0_CR_GENERIC_MRS_FSM_STORAGE_VALUES_0_P0_REG +
          ((MC1_CH0_CR_GENERIC_MRS_FSM_STORAGE_VALUES_0_P0_REG - MC0_CH0_CR_GENERIC_MRS_FSM_STORAGE_VALUES_0_P0_REG) * Controller) +
          ((MC0_CH1_CR_GENERIC_MRS_FSM_STORAGE_VALUES_0_P0_REG - MC0_CH0_CR_GENERIC_MRS_FSM_STORAGE_VALUES_0_P0_REG) * McCh) +
          ((MC0_CH0_CR_GENERIC_MRS_FSM_STORAGE_VALUES_1_P0_REG - MC0_CH0_CR_GENERIC_MRS_FSM_STORAGE_VALUES_0_P0_REG) * ControlRegNum);
        GenMrsFsmChValP0.Data = MrcReadCR (MrcData, Offset);

        // The Registers in a dual rank configuration must be stored in the following order
        // assuming n is the byte address specified in GENERIC_MRS_STORAGE_POINTER.
        // n   = Sub-channel 0, Rank 0
        // n+1 = Sub-channel 0, Rank 1
        // n+2 = Sub-channel 1, Rank 0
        // n+3 = Sub-channel 1, Rank 1
        ShadowFieldIdx = (*StorageIdx) % 4;
        switch (ShadowFieldIdx) {
          case 0:
            GenMrsFsmChValP0.P0Bits.BYTE_0 = GenMrPtr->MrData;
            break;

          case 1:
            GenMrsFsmChValP0.P0Bits.BYTE_1 = GenMrPtr->MrData;
            break;

          case 2:
            GenMrsFsmChValP0.P0Bits.BYTE_2 = GenMrPtr->MrData;
            break;

          case 3:
            GenMrsFsmChValP0.P0Bits.BYTE_3 = GenMrPtr->MrData;
            break;
        }
        MrcWriteCR (MrcData, Offset, GenMrsFsmChValP0.Data);
        (*StorageIdx)++;
        // Only a single byte field is needed for non per-rank MrData array entries.
        if (!MrPerRank) {
          break;
        }
      } // Rank
      // Only a single byte field is needed for non per-rank MrData array entries.
      if (!MrPerRank) {
        break;
      }
    } // SubCh
  }
  return mrcSuccess;
}

/**
  This function configures the Generic MRS FSM shadow registers based on the MrData inputs.
  It will determine if it needs to use the per-rank feature if the MR value differs across ranks.

  @param[in] MrcData - Pointer to MRC global data.
  @param[in] MrData  - Pointer to an array of MR data to configure the MRS FSM with.

  @retval mrcSuccess if successful.
  @retval mrcFail if MrData pointer is null, the timing or per-rank registers are out of free entries.
**/
MrcStatus
MrcGenMrsFsmConfig (
  IN  MrcParameters *MrcData,
  IN  MRC_GEN_MRS_FSM_MR_TYPE MrData[MAX_CONTROLLER][MAX_CHANNEL][MAX_RANK_IN_CHANNEL][MAX_MR_GEN_FSM]
  )
{
  const MRC_FUNCTION  *MrcCall;
  MrcDebug  *Debug;
  MrcOutput *Outputs;
  MrcStatus Status;
  UINT32    Controller;
  UINT32    Channel;
  UINT32    McCh;
  UINT32    MaxSubCh;
  UINT32    Offset;
  UINT32    Rank;
  UINT32    SubCh;
  UINT32    TimingSetIdx;
  UINT16    InitData;
  UINT16    TimingSet[GEN_MRS_FSM_TIMING_SET_MAX];
  UINT8     ControlRegIdx;
  UINT8     StorageIdx;
  UINT8     TimingSetNumValid;
  UINT8     FspOpSwitchIdx;
  BOOLEAN   SagvBreakPoint;
  BOOLEAN   Lpddr;
  BOOLEAN   MrPerRank;
  BOOLEAN   DoZqCal;
  MRC_GEN_MRS_FSM_MR_TYPE *GenMrPtr;
  MC0_CH0_CR_MRS_FSM_CONTROL_STRUCT MrsFsmCtl;

  Outputs = &MrcData->Outputs;
  Debug = &Outputs->Debug;
  MrcCall = MrcData->Inputs.Call.Func;
  Lpddr = Outputs->Lpddr;
  MaxSubCh = (Lpddr) ? MAX_SUB_CHANNEL : 1;
  Rank = 0;
  Channel = 0;
  GenMrPtr = NULL;
  SagvBreakPoint = FALSE;
  FspOpSwitchIdx = 0;
  DoZqCal = TRUE;

  for (Controller = 0; Controller < MAX_CONTROLLER; Controller++) {
    if (!MrcControllerExist (MrcData, Controller)) {
      // Skip unpopulated controllers.
      continue;
    }
    // Determine if the the MR value is the same across Sub-Channels and Ranks
    for (McCh = 0; McCh < MAX_CHANNEL_SHARE_REGS; McCh++) {
      MrcCall->MrcSetMemWord (TimingSet, GEN_MRS_FSM_TIMING_SET_MAX, 0);
      TimingSetNumValid = 0;
      StorageIdx = 0;
      MRC_MC_IP_DEBUG_MSG (Debug, MSG_LEVEL_ALGO, "Mc%d.IpCh%d\n\tStorageIdx = %d\tTimingSetNumValid = %d\n", Controller, McCh, StorageIdx, TimingSetNumValid);
      for (ControlRegIdx = 0; ControlRegIdx < MAX_MR_GEN_FSM; ControlRegIdx++) {
        MrPerRank = FALSE;
        InitData  = MRC_UINT16_MAX;
        // Gate SubCh and Rank loops with MrPerRank == FALSE.
        // Once we find that they are different per rank; break out and start the register configuration
        for (SubCh = 0; ((SubCh < MaxSubCh) && (!MrPerRank)); SubCh++) {
          Channel = (McCh * MaxSubCh) + SubCh;
          for (Rank = 0; ((Rank < MAX_RANK_IN_CHANNEL) && (!MrPerRank)); Rank++) {
            if (!MrcRankExist (MrcData, Controller, Channel, Rank)) {
              continue;
            }

            GenMrPtr = &MrData[Controller][Channel][Rank][ControlRegIdx];
            if (!GenMrPtr->Valid) {
              continue;
            }

            // Base Case
            if (InitData == MRC_UINT16_MAX) {
              InitData = GenMrPtr->MrData;
            } else if (InitData != GenMrPtr->MrData) { // Check that MR data is matching or needs to be per Rank
              MrPerRank = TRUE;
            }

            if ((GenMrPtr->CmdType == GmfCmdMpc) && (GenMrPtr->MrData == DDR5_MPC_ZQCAL_LATCH)) {
              // Turn off automatic ZQCAL when issued as part of the input sequence
              DoZqCal = FALSE;
            }
          } // Rank
        } // SubCh

        // If InitData is not another value besides MRC_UINT16_MAX, then this ControlRegIdx
        // did not have a valid entry in the array for the subch's or ranks.  Skip this index.
        if (InitData == MRC_UINT16_MAX) {
          continue;
        }

        if (GenMrPtr == NULL) {
          MRC_DEBUG_MSG (Debug, MSG_LEVEL_ERROR, "GenMrPtr is Null\n");
          return mrcFail;
        }

        Status = ConfigGenMrsFsmTiming (
                  MrcData,
                  Controller,
                  McCh,
                  GenMrPtr->Delay,
                  &TimingSetIdx,
                  &TimingSet,
                  &TimingSetNumValid
                  );
        if (Status != mrcSuccess) {
          MRC_DEBUG_MSG (Debug, MSG_LEVEL_ERROR, "Generic FSM ");
          if (Status == mrcWrongInputParameter) {
            MRC_DEBUG_MSG (Debug, MSG_LEVEL_ERROR, "Delay Overflow");
          } else {
            MRC_DEBUG_MSG (Debug, MSG_LEVEL_ERROR, "Timing is full");
          }
          MRC_DEBUG_MSG (
            Debug,
            MSG_LEVEL_ERROR,
            " - NumValid: %u, Index: %u, ControlRegIdx: %u, Mc: %u, IpCh: %u\n",
            TimingSetNumValid, TimingSetIdx, ControlRegIdx, Controller, McCh);
          return Status;
        }
        Status = ConfigGenMrsFsmValue (
                  MrcData,
                  Controller,
                  McCh,
                  GenMrPtr,
                  MrPerRank,
                  TimingSetIdx,
                  ControlRegIdx,
                  MrData,
                  &StorageIdx
                  );
        if (Status != mrcSuccess) {
          return Status;
        }

        if (GenMrPtr->FreqSwitchPoint) {
          SagvBreakPoint = TRUE;
          FspOpSwitchIdx = ControlRegIdx;
        }
      } // ControlRegIdx
    } // McCh
  } // Controller

  // Finally configure the MRS_FSM to use the Generic version if this is the first time sending the sequence
  MrsFsmCtl.Data = 0;
  MrsFsmCtl.Bits.GENERIC_MRS_FSM_Enable = 1;
  if (Lpddr && SagvBreakPoint) {
    MrsFsmCtl.Bits.LPDDR4_split_transition = 1;
    MrsFsmCtl.Bits.LPDDR4_switch_FSP = 1;
    MrsFsmCtl.Bits.GENERIC_MRS_FSM_Breakpoint_Address = FspOpSwitchIdx;
    MrsFsmCtl.Bits.GV_auto_enable = 1;
  }
  if (DoZqCal) {
    MrsFsmCtl.Bits.do_ZQCL = 1;
  }

  for (Controller = 0; Controller < MAX_CONTROLLER; Controller++) {
    if (MrcControllerExist (MrcData, Controller)) {
      Offset = MC0_BC_CR_MRS_FSM_CONTROL_REG +
        ((MC1_BC_CR_MRS_FSM_CONTROL_REG - MC0_BC_CR_MRS_FSM_CONTROL_REG) * Controller);
      MrcWriteCR64 (MrcData, Offset, MrsFsmCtl.Data);
    }
  }

  return mrcSuccess;
}

/**
  This function executes the MRS FSM and waits for the FSM to complete.
  If the FSM does not complete after 10 seconds, it will return an error message.

  @param[in] MrcData - Pointer to MRC global data.

  @retval mrcFail if the FSM is not idle.
  @retval mrcSuccess otherwise.
**/
MrcStatus
MrcGenMrsFsmRun (
  IN  MrcParameters *MrcData
  )
{
  const MRC_FUNCTION  *MrcCall;
  MrcOutput *Outputs;
  UINT64    Timeout;
  UINT32    Controller;
  UINT32    Channel;
  UINT32    IpCh;
  UINT32    Offset;
  UINT8     MaxChannel;
  BOOLEAN   Lpddr;
  BOOLEAN   Flag;
  MC0_CH0_CR_MRS_FSM_RUN_STRUCT MrsFsmRun;

  Outputs = &MrcData->Outputs;
  MrcCall = MrcData->Inputs.Call.Func;
  Lpddr = Outputs->Lpddr;
  MaxChannel = Outputs->MaxChannels;

  // Start all FSM's in parallel
  for (Controller = 0; Controller < MAX_CONTROLLER; Controller++) {
    for (Channel = 0; Channel < MaxChannel; Channel++) {
      if (MrcChannelExist (MrcData, Controller, Channel)) {
        IpCh = LP_IP_CH (Lpddr, Channel);
        Offset = OFFSET_CALC_MC_CH (MC0_CH0_CR_MRS_FSM_RUN_REG, MC1_CH0_CR_MRS_FSM_RUN_REG, Controller, MC0_CH1_CR_MRS_FSM_RUN_REG, IpCh);

        MrsFsmRun.Data = 0;
        MrsFsmRun.Bits.Run = 1;
        MrcWriteCR (MrcData, Offset, MrsFsmRun.Data);

        // If we're LPDDR and we are on Channel 0 or 2, increment channel additionally so we move to the next MC Channel.
        if (Lpddr && ((Channel % 2) == 0)) {
          Channel++;
        }
      }
    }
  }

  // Check that all FSM's are done
  Timeout = MrcCall->MrcGetCpuTime () + MRC_WAIT_TIMEOUT; // 10 seconds timeout
  for (Controller = 0; Controller < MAX_CONTROLLER; Controller++) {
    for (Channel = 0; Channel < MaxChannel; Channel++) {
      if (MrcChannelExist (MrcData, Controller, Channel)) {
        IpCh = LP_IP_CH (Lpddr, Channel);
        Offset = OFFSET_CALC_MC_CH (MC0_CH0_CR_MRS_FSM_RUN_REG, MC1_CH0_CR_MRS_FSM_RUN_REG, Controller, MC0_CH1_CR_MRS_FSM_RUN_REG, IpCh);

        do {
          MrsFsmRun.Data = MrcReadCR (MrcData, Offset);
          Flag = (MrsFsmRun.Bits.Run == 1);
        } while (Flag && (MrcCall->MrcGetCpuTime () < Timeout));
        if (Flag) {
          MRC_DEBUG_MSG (&Outputs->Debug, MSG_LEVEL_ERROR, "Generic FSM is not idle\n");
          return mrcFail;
        }

        // If we're LPDDR and we are on Channel 0 or 2, increment channel additionally so we move to the next MC Channel.
        if (Lpddr && ((Channel % 2) == 0)) {
          Channel++;
        }
      }
    }
  }

  return mrcSuccess;
}

UINT32
MrcQclkToTck (
  IN  MrcParameters *const MrcData,
  IN  UINT32               Qclk
  )
{
  MrcOutput *Outputs;
  UINT32    RetVal;
  UINT32    Scalar;
  BOOLEAN   Gear2;

  Outputs = &MrcData->Outputs;
  Gear2 = Outputs->Gear2;

  if (Outputs->DdrType == MRC_DDR_TYPE_LPDDR5) {
    Scalar = Gear2 ? 4 : 8;
  } else {
    Scalar = Gear2 ? 1 : 2;
  }

  RetVal = Qclk / Scalar;

  return RetVal;
}

/**
  Program MC/DDRIO registers to Gear1 or Gear2 mode.
  This only includes Gear2 mode enable/disable, not other registers that are impacted by gear mode.

  @param[in] MrcData - The MRC general data.
  @param[in] Gear2   - TRUE for Gear2, FALSE for Gear1

  @retval None
**/
VOID
MrcSetGear2 (
  IN MrcParameters *const MrcData,
  IN BOOLEAN              Gear2
  )
{
  INT64     GetSetVal;

  GetSetVal = Gear2 ? 1 : 0;
  MrcGetSetMcCh (MrcData, MAX_CONTROLLER, MAX_CHANNEL, GsmMccGear2, WriteNoCache, &GetSetVal);

  DdrIoConfigGear2 (MrcData, Gear2);
}

UINT32
MrcTckToQclk (
  IN  MrcParameters *const MrcData,
  IN  UINT32               Tck
  )
{
  MrcOutput *Outputs;
  UINT32    RetVal;
  UINT32    Scalar;
  BOOLEAN   Gear2;

  Outputs = &MrcData->Outputs;
  Gear2 = Outputs->Gear2;

  if (Outputs->DdrType == MRC_DDR_TYPE_LPDDR5) {
    Scalar = Gear2 ? 4 : 8;
  } else {
    Scalar = Gear2 ? 1 : 2;
  }

  RetVal = Tck * Scalar;

  return RetVal;
}

/**
  Programming of CCC_CR_DDRCRCCCCLKCONTROLS_BlockTrainRst

  @param[in] MrcData - The MRC global data.
  @param[in] BlockTrainReset - TRUE to BlockTrainReset for most training algos.  FALSE for specific training algos that need PiDivider sync.

**/
VOID
MrcBlockTrainResetToggle (
  IN MrcParameters *const MrcData,
  IN BOOLEAN              BlockTrainReset
)
{
  UINT8   Value;
  UINT8   Index;
  UINT32  Offset;
  CH0CCC_CR_DDRCRCCCCLKCONTROLS_STRUCT  CccClkControls;

  Value = (BlockTrainReset) ? 1 : 0;

  for (Index = 0; Index < MAX_SYS_CHANNEL; Index++) {
    Offset = OFFSET_CALC_CH (CH0CCC_CR_DDRCRCCCCLKCONTROLS_REG, CH1CCC_CR_DDRCRCCCCLKCONTROLS_REG, Index);
    CccClkControls.Data = MrcReadCR (MrcData, Offset);
    if (CccClkControls.Bits.BlockTrainRst == Value) {
      continue;
    }
    CccClkControls.Bits.BlockTrainRst = Value;
    MrcWriteCR (MrcData, Offset, CccClkControls.Data);
  }
}

/**
  This function configures the DDRCRCMDBUSTRAIN register to values for normal mode.

  @param[in]  MrcData - Pointer to global MRC data.

  @retval none.
**/
VOID
MrcSetWritePreamble (
  IN  MrcParameters *const  MrcData
  )
{
  MrcOutput         *Outputs;
  MrcDdrType        DdrType;
  BOOLEAN           Ddr5;
  INT64             NumMaskedPulses;
  INT64             NumOfPulses;

  Outputs           = &MrcData->Outputs;
  DdrType           = Outputs->DdrType;
  Ddr5              = (DdrType == MRC_DDR_TYPE_DDR5);


  NumMaskedPulses = (Ddr5 ? 1 : 0);
  NumOfPulses = 3;

  MrcGetSetChStrb (MrcData, MAX_CONTROLLER, MAX_CHANNEL, MAX_SDRAM_IN_DIMM, GsmIocDqsMaskPulseCnt, WriteToCache, &NumMaskedPulses);
  MrcGetSetChStrb (MrcData, MAX_CONTROLLER, MAX_CHANNEL, MAX_SDRAM_IN_DIMM, GsmIocDqsPulseCnt,     WriteToCache, &NumOfPulses);
  MrcFlushRegisterCachedData (MrcData);
}

// TGL P0 has unsorted CCC channel instances. This table
// maps a System Channel index to a CCC Channel Instance
// modifier. This modifier indicates the position of the
// corresponding register offset relative to the P0 CCC
// CH0 instance.
//
// P0 CCC CH Instances sorted by register offset:
// CH Instance     | CH2 CH0 CH3 CH1 CH5 CH7 Ch4 CH6
// -------------------------------------------------
// Relative Offset |  -1   0   1   2   3   4   5   6
// to CH0 Instance |
INT8 CccChannelInstanceOffsetP0[MAX_SYS_CHANNEL] = {
  0, 2, -1, 1, 5, 3, 6, 4
};

// The offset between each CCC channel instance is constant across all CCC Registers. The offset
// only differs between TGL-U\Y and TGL-H.
#define TGL_ULXULT_CCC_CH_OFFSET (CH1CCC_CR_DDRCRCACOMP_REG - CH0CCC_CR_DDRCRCACOMP_REG)       // 0x100
#define TGL_H_CCC_CH_OFFSET      (CH0CCC_CR_DDRCRCACOMP_P0_REG - CH2CCC_CR_DDRCRCACOMP_P0_REG) //  0x80

/**
  Calculate the Unicast CCC Channel Register address for the input channel on the current SoC. The
  input register CH0 Offset must be the A0\B0 offset. This offset will be automatically adjusted for P0.

  @param[in] MrcData      - Pointer to MRC global data.
  @param[in] RegOffsetCh0 - The register offset for the CH0 instance of a CCC register. This must be
                            a TGL A0\B0 register offset.
  @param[in] Channel      - The system channel that must be accessed by the returned register offset.

  @retval UINT32 - The register offset to the requested channel for the current SoC.
**/
UINT32
MrcCrOffsetProjAdjCccCh (
  IN MrcParameters *const  MrcData,
  IN UINT32                RegOffsetCh0,
  IN UINT32                Channel
  )
{
  const MrcInput *Inputs;
  MrcOutput      *Outputs;
  MrcDebug       *Debug;
  UINT32         Offset;

  Inputs        = &MrcData->Inputs;
  Outputs       = &MrcData->Outputs;
  Debug         = &Outputs->Debug;

  Offset = 0xFFFFFFFF;
  if (RegOffsetCh0 < CH0CCC_CR_DDRCRCACOMP_REG && RegOffsetCh0 > CH0CCC_CR_DCCLANESTATUS2_REG) {
    // Check that the inputs are winthin the known register offset range
    // for TGL A0\B0 CCC CH0 register instances.
    MRC_DEBUG_ASSERT (
      (RegOffsetCh0 >= CH0CCC_CR_DDRCRCACOMP_REG) || (RegOffsetCh0 <= CH0CCC_CR_DCCLANESTATUS2_REG), // 0x00003600 to 0x00003674
      Debug,
      "Error: Not a valid CCC Channel Register\n"
      );
  } else {
    if (Inputs->UlxUlt) {
      Offset = RegOffsetCh0 + (TGL_ULXULT_CCC_CH_OFFSET * Channel);
    } else {
      Offset = (RegOffsetCh0 | MRC_BIT7) + (TGL_H_CCC_CH_OFFSET * CccChannelInstanceOffsetP0[Channel]);
    }
  }

  return Offset;
}

/**
  Returns the currently configured DRAM Command Intput Rate NMode.
  Note: In DDR4 3N is used during training (in Gear1), but this
  function won't report this.

  @param[in] MrcData    - Include all MRC global data.

  @retval 1 = 1N Mode
  @retval 2 = 2N Mode
**/
UINT32
MrcGetNMode (
  IN MrcParameters *const MrcData
  )
{
  MrcInput          *Inputs;
  MrcOutput         *Outputs;
  MrcChannelOut     *ChannelOut;
  MrcControllerOut  *ControllerOut;
  MrcProfile        Profile;
  UINT32            FirstController;
  UINT32            FirstChannel;
  UINT32            NMode;


  Inputs          = &MrcData->Inputs;
  Outputs         = &MrcData->Outputs;
  Profile         = Inputs->MemoryProfile;
  FirstController = (MrcControllerExist (MrcData, cCONTROLLER0)) ? 0 : 1;
  ControllerOut   = &Outputs->Controller[FirstController];
  FirstChannel    = ControllerOut->FirstPopCh;
  ChannelOut      = &ControllerOut->Channel[FirstChannel];

  // In DDR4 3N is used during training (in Gear1), but this
  // function won't report this.
  NMode = ChannelOut->Timing[Profile].NMode;

  return NMode;
}

/**
  Configure the MC to issue multicycle CS_n MPC commands.
  The DRAM must be configured separately by either setting
  DDR5 MR2.OP[4] = 0 or by resetting the DRAM.

  @param[in] MrcData    - Include all MRC global data.

  @retval None
**/
void
EnableMcMulticycleCs (
  IN MrcParameters *const MrcData
  )
{
  MrcOutput *Outputs;
  UINT32    Gear;
  UINT8     Controller;
  UINT8     Channel;
  UINT32    Offset;
  UINT32    NMode;
  INT64     MultiCycCmd;
  MC0_CH0_CR_TC_MPC_P0_STRUCT TcMpc;

  Outputs           = &MrcData->Outputs;
  Gear              = Outputs->Gear2 ? 2 : 1;
  NMode             = MrcGetNMode (MrcData);

  for (Controller = 0; Controller < MAX_CONTROLLER; Controller++) {
    for (Channel = 0; Channel < MAX_CHANNEL; Channel++) {
      if (MrcChannelExist (MrcData, Controller, Channel)) {
        Offset = OFFSET_CALC_MC_CH(MC0_CH0_CR_TC_MPC_P0_REG, MC1_CH0_CR_TC_MPC_P0_REG, Controller, MC0_CH1_CR_TC_MPC_P0_REG, Channel);
        TcMpc.Data = 0;
        TcMpc.P0Bits.MPC_Setup  = (NMode == 2) ? (DDR5_tMC_MPC_SETUP_MIN + Gear): DDR5_tMC_MPC_SETUP_MIN;
        TcMpc.P0Bits.MPC_Hold   = DDR5_tMC_MPC_HOLD_MIN;
        TcMpc.P0Bits.MultiCycCS = DDR5_tMPC_CS;
        MultiCycCmd             = 1;
        MrcWriteCR (MrcData, Offset, TcMpc.Data);
        MrcGetSetMcCh (MrcData, Controller, Channel, GsmMccMultiCycCmd, WriteNoCache, &MultiCycCmd);
      }
    } // Channel
  } // Controller
}

/**
  Configure the MC and DRAM for single cycle CS_n MPC
  commands. An MRW is issued to the DRAM to configure
  DDR5 MR2[4] = 1.

  @param[in] MrcData    - Include all MRC global data.

  @retval None
**/
void
DisableMcMulticycleCs (
  IN MrcParameters *const MrcData
  )
{
  MrcOutput         *Outputs;
  MrcControllerOut  *ControllerOut;
  MrcChannelOut     *ChannelOut;
  UINT32            Controller;
  UINT32            Channel;
  UINT32            Rank;
  UINT16            *MrPtr;
  MrcModeRegister   MrAddress;
  UINT32            MrIndex;
  INT64             MultiCycCmd;
  DDR5_MODE_REGISTER_2_TYPE Mr2;
  UINT32        Dimm;
  UINT32        DimmRank;

  Outputs = &MrcData->Outputs;

  // Set the MC register
  MultiCycCmd = 0;
  MrcGetSetMcCh (MrcData, MAX_CONTROLLER, MAX_CHANNEL, GsmMccMultiCycCmd, WriteNoCache, &MultiCycCmd);

  // Find MR array index for MR2
  MrAddress = mrMR2;
  MrIndex = MrcMrAddrToIndex (MrcData, &MrAddress);
  if (MrIndex >=  MAX_MR_IN_DIMM) {
    // Error condition
    return;
  }

  // Set the DRAM Mode Register
  for (Controller = 0; Controller < MAX_CONTROLLER; Controller++) {
    ControllerOut = &Outputs->Controller[Controller];
    for (Channel = 0; (Channel < Outputs->MaxChannels); Channel++) {
      if (!(MrcChannelExist (MrcData, Controller, Channel))) {
        continue;
      }
      ChannelOut = &ControllerOut->Channel[Channel];
      for (Rank = 0; Rank < MAX_RANK_IN_CHANNEL; Rank++) {
        if (!(MrcRankExist (MrcData, Controller, Channel, Rank))) {
          continue;
        }
        Dimm = RANK_TO_DIMM_NUMBER (Rank);
        DimmRank = Rank % MAX_RANK_IN_DIMM;
        MrPtr = ChannelOut->Dimm[Dimm].Rank[DimmRank].MR;
        Mr2.Data8 = (UINT8) MrPtr[MrIndex];
        Mr2.Bits.CsAssertionDuration = 1; // Only a single cycle of CS assertion supported for MPC and VrefCA commands
        MrPtr[MrIndex] = Mr2.Data8;
        // Issue MRW to DDR5 MR2
        MrcIssueMrw (
          MrcData,
          Controller,
          Channel,
          Rank,
          mrMR2,
          Mr2.Data8,
          MRC_PRINTS_ON   // DebugPrint
          );
      }
    }
  }
}

/**
  Translate a TGL U\Y Register Offset to a TGL H Register Offset when the
  current processor is TGL H. The output is not guranteed to be correct if
  the input register offset is not a TGL U\Y register offset. There are known
  TGL-H register offsets which alias to TGL-U\Y register offsets.

  @param[in] MrcData      - Pointer to MRC global data.
  @param[in] RegOffset    - The register offset to tanslate. This must be a TGL A0\B0
                            register offset. Set BIT31 to avoid translation.

  @retval UINT32 - The register offset to the register for the current SoC.
**/
UINT32
MrcCrOffsetProjAdj (
  IN MrcParameters *const  MrcData,
  IN UINT32                RegOffset
  )
{
  const MrcInput *Inputs;
  UINT32         Offset;
  UINT32         Channel;
  UINT32         CccRegOffset;

  Inputs        = &MrcData->Inputs;

  if ((RegOffset & NO_ADJ) != 0) {
    // BIT 31 set indicates that no offset adjustment is needed. Clear the bit and return the offset.
    Offset = RegOffset & ~(NO_ADJ);
  } else if (Inputs->UlxUlt) {
    // No Translation is needed for TGL-U\Y
    Offset = RegOffset;
  } else {
    if (RegOffset >= DDRVTT0_CR_DDRCRVTTGENCONTROL_REG && RegOffset <= DDRVTT3_CR_DDRCRVTTCOMPOFFSET2_REG) {
      // 0x00002900 to 0x00002A8C
      Offset = RegOffset + (DDRVTT0_CR_DDRCRVTTGENCONTROL_P0_REG - DDRVTT0_CR_DDRCRVTTGENCONTROL_REG); // 4352
    } else if (RegOffset == DDRVREF_CR_DDRCRVREFADJUST2_REG) {
      // 0x00002B0C
      Offset = DDRVREF_CR_DDRCRVREFADJUST2_P0_REG;
    } else if (RegOffset >= CH0CCC_CR_DDRCRCACOMP_REG && RegOffset <= CH7CCC_CR_DCCLANESTATUS2_REG) {
      // 0x00003600 to 0x00003D74
      // CCC Channel Registers
      // First figure out the base register offset and channel instance from the input offset
      // Bits [10:8] provide the channel
      Channel = ((RegOffset - 0x00003600) & (MRC_BIT10 | MRC_BIT9 | MRC_BIT8)) >> 8;
      // Bits [7:0] provide the base register instance
      CccRegOffset = (RegOffset & 0xFF) | 0x00003600;

      // Check that the calculated base register address is within bounds
      // of known TGL U\Y CCC CH0 registers
      if (CccRegOffset >= CH0CCC_CR_DDRCRCACOMP_REG && CccRegOffset <= CH0CCC_CR_DCCLANESTATUS2_REG) {
        Offset = MrcCrOffsetProjAdjCccCh (MrcData, CccRegOffset, Channel);
      } else {
        Offset = RegOffset;
      }
    } else if ((RegOffset >= MC0_CH0_CR_TC_PRE_REG && RegOffset <= MC0_CH0_CR_WCK_CONFIG_REG) ||
               (RegOffset >= MC1_CH0_CR_TC_PRE_REG && RegOffset <= MC1_CH0_CR_WCK_CONFIG_REG)) {
      // 0x00004000 to 0x000040F8
      // 0x00014000 to 0x000140F8
      Offset = RegOffset + (MC0_CH0_CR_TC_PRE_P0_REG - MC0_CH0_CR_TC_PRE_REG); // 40960
    } else if ((RegOffset >= MC0_CH0_CR_GENERIC_MRS_FSM_CONTROL_0_REG && RegOffset <= MC0_CH0_CR_GENERIC_MRS_FSM_CONTROL_27_REG) ||
               (RegOffset >= MC1_CH0_CR_GENERIC_MRS_FSM_CONTROL_0_REG && RegOffset <= MC1_CH0_CR_GENERIC_MRS_FSM_CONTROL_27_REG)) {
      // 0x00004120 to 0x0000418C
      // 0x00014120 to 0x0001418C
      Offset = RegOffset + (MC0_CH0_CR_GENERIC_MRS_FSM_CONTROL_0_P0_REG - MC0_CH0_CR_GENERIC_MRS_FSM_CONTROL_0_REG); // 42224
    } else if (RegOffset == MC0_CH0_CR_MCMNTS_RDDATA_CTL_REG || RegOffset == MC1_CH0_CR_MCMNTS_RDDATA_CTL_REG) {
      // 0x000041E8
      // 0x000141E8
      Offset = RegOffset + (MC0_CH0_CR_MCMNTS_RDDATA_CTL_P0_REG - MC0_CH0_CR_MCMNTS_RDDATA_CTL_REG); // 42008
    } else if ((RegOffset >= MC0_CH0_CR_REUT_CH_MISC_REFRESH_CTRL_REG && RegOffset <= MC0_CH0_CR_MCMNTS_SPARE_REG) ||
               (RegOffset >= MC1_CH0_CR_REUT_CH_MISC_REFRESH_CTRL_REG && RegOffset <= MC1_CH0_CR_MCMNTS_SPARE_REG)) {
      // 0x00004204 to 0x000043FC
      // 0x00014204 to 0x000143FC
      if (RegOffset == MC0_CH0_CR_TC_MR2_SHADDOW_REG || RegOffset == MC1_CH0_CR_TC_MR2_SHADDOW_REG) {
        // These Regs are unique to TGL U\Y
        // 0x0000424C
        // 0x0001424C
        Offset = 0xFFFFFFFF;
      } else if (RegOffset == MC0_CH0_CR_LPDDR_PASR_REG || RegOffset == MC1_CH0_CR_LPDDR_PASR_REG) {
        // 0x00004290
        // 0x00014290
        Offset = RegOffset + (MC0_CH0_CR_LPDDR_PASR_P0_REG - MC0_CH0_CR_LPDDR_PASR_REG); // 41532
      } else {
        Offset = RegOffset + (MC0_CH0_CR_REUT_CH_MISC_REFRESH_CTRL_P0_REG - MC0_CH0_CR_REUT_CH_MISC_REFRESH_CTRL_REG); // 41472
      }
    } else if ((RegOffset >= MC0_CH1_CR_TC_PRE_REG && RegOffset <= MC0_CH1_CR_WCK_CONFIG_REG) ||
               (RegOffset >= MC0_BC_CR_TC_PRE_REG && RegOffset <= MC0_BC_CR_WCK_CONFIG_REG) ||
               (RegOffset >= MC1_CH1_CR_TC_PRE_REG && RegOffset <= MC1_CH1_CR_WCK_CONFIG_REG) ||
               (RegOffset >= MC1_BC_CR_TC_PRE_REG && RegOffset <= MC1_BC_CR_WCK_CONFIG_REG)) {
      // 0x00004400 to 0x000044F8
      // 0x00004C00 to 0x00004CF8
      // 0x00014400 to 0x000144F8
      // 0x00014C00 to 0x00014CF8
      Offset = RegOffset + (MC0_CH1_CR_TC_PRE_P0_REG - MC0_CH1_CR_TC_PRE_REG); // 41984
    } else if ((RegOffset >= MC0_CH1_CR_GENERIC_MRS_FSM_CONTROL_0_REG && RegOffset <= MC0_CH1_CR_GENERIC_MRS_FSM_CONTROL_27_REG) ||
               (RegOffset >= MC0_BC_CR_GENERIC_MRS_FSM_CONTROL_0_REG && RegOffset <= MC0_BC_CR_GENERIC_MRS_FSM_CONTROL_27_REG) ||
               (RegOffset >= MC1_CH1_CR_GENERIC_MRS_FSM_CONTROL_0_REG && RegOffset <= MC1_CH1_CR_GENERIC_MRS_FSM_CONTROL_27_REG) ||
               (RegOffset >= MC1_BC_CR_GENERIC_MRS_FSM_CONTROL_0_REG && RegOffset <= MC1_BC_CR_GENERIC_MRS_FSM_CONTROL_27_REG)) {
      // 0x00004520 to 0x0000458C
      // 0x00004D20 to 0x00004D8C
      // 0x00014520 to 0x0001458C
      // 0x00014D20 to 0x00014D8C
      Offset = RegOffset + (MC0_CH1_CR_GENERIC_MRS_FSM_CONTROL_0_P0_REG - MC0_CH1_CR_GENERIC_MRS_FSM_CONTROL_0_REG); // 43248
    } else if (RegOffset == MC0_CH1_CR_MCMNTS_RDDATA_CTL_REG || RegOffset == MC0_BC_CR_MCMNTS_RDDATA_CTL_REG ||
               RegOffset == MC1_CH1_CR_MCMNTS_RDDATA_CTL_REG || RegOffset == MC1_BC_CR_MCMNTS_RDDATA_CTL_REG) {
      // 0x000045E8
      // 0x00004DE8
      // 0x000145E8
      // 0x00014DE8
      Offset = RegOffset + (MC0_CH1_CR_MCMNTS_RDDATA_CTL_P0_REG - MC0_CH1_CR_MCMNTS_RDDATA_CTL_REG); // 43032
    } else if ((RegOffset >= MC0_CH1_CR_REUT_CH_MISC_REFRESH_CTRL_REG && RegOffset <= MC0_CH1_CR_MCMNTS_SPARE_REG) ||
               (RegOffset >= MC0_BC_CR_REUT_CH_MISC_REFRESH_CTRL_REG && RegOffset <= MC0_BC_CR_MCMNTS_SPARE_REG) ||
               (RegOffset >= MC1_CH1_CR_REUT_CH_MISC_REFRESH_CTRL_REG && RegOffset <= MC1_CH1_CR_MCMNTS_SPARE_REG) ||
               (RegOffset >= MC1_BC_CR_REUT_CH_MISC_REFRESH_CTRL_REG && RegOffset <= MC1_BC_CR_MCMNTS_SPARE_REG)) {
      // 0x00004604 to 0x000047FC
      // 0x00004E04 to 0x00004FFC
      // 0x00014604 to 0x000147FC
      // 0x00014E04 to 0x00014FFC
      if (RegOffset == MC0_CH1_CR_TC_MR2_SHADDOW_REG || RegOffset == MC0_BC_CR_TC_MR2_SHADDOW_REG ||
          RegOffset == MC1_CH1_CR_TC_MR2_SHADDOW_REG || RegOffset == MC1_BC_CR_TC_MR2_SHADDOW_REG) {
        // These register are unique to TGL U\Y
        // 0x0000464C
        // 0x00004E4C
        // 0x0001464C
        // 0x00014E4C
        Offset = 0xFFFFFFFF;
      } else if (RegOffset == MC0_CH1_CR_LPDDR_PASR_REG || RegOffset == MC0_BC_CR_LPDDR_PASR_REG ||
                 RegOffset == MC1_CH1_CR_LPDDR_PASR_REG || RegOffset == MC1_BC_CR_LPDDR_PASR_REG) {
        // 0x00004690
        // 0x00004E90
        // 0x00014690
        // 0x00014E90
        Offset = RegOffset + (MC0_CH1_CR_LPDDR_PASR_P0_REG - MC0_CH1_CR_LPDDR_PASR_REG); // 42556
      } else {
        Offset = RegOffset + (MC0_CH1_CR_REUT_CH_MISC_REFRESH_CTRL_P0_REG - MC0_CH1_CR_REUT_CH_MISC_REFRESH_CTRL_REG); // 42496
      }
    } else if ((RegOffset >= MC0_MAD_INTER_CHANNEL_REG && RegOffset <= MC0_PARITY_CONTROL_REG) ||
               (RegOffset >= MC1_MAD_INTER_CHANNEL_REG && RegOffset <= MC1_PARITY_CONTROL_REG)) {
      // 0x00005000 to 0x000051B4
      // 0x00015000 to 0x000151B4
      if (RegOffset == MC0_PWM_GLB_DRV_OFF_COUNT_REG || RegOffset == MC1_PWM_GLB_DRV_OFF_COUNT_REG) {
        // 0x000050C0
        // 0x000150C0
        Offset = RegOffset + (MC0_PWM_GLB_DRV_OFF_COUNT_P0_REG - MC0_PWM_GLB_DRV_OFF_COUNT_REG); // 35000
      } else {
        Offset = RegOffset + (MC0_MAD_INTER_CHANNEL_P0_REG - MC0_MAD_INTER_CHANNEL_REG); // 34816
      }
    } else if ((RegOffset >= MC0_DDRPL_CFG_DTF_REG && RegOffset <= MC0_DDRPL_VISA_CFG_DTF_REG) ||
               (RegOffset >= MC1_DDRPL_CFG_DTF_REG && RegOffset <= MC1_DDRPL_VISA_CFG_DTF_REG)) {
      // 0x00005200 to 0x00005210
      // 0x00015200 to 0x00015210
      Offset = RegOffset + (MC0_DDRPL_CFG_DTF_P0_REG - MC0_DDRPL_CFG_DTF_REG); // 36096
    } else if (RegOffset >= CCC_CR_DDRCRCACOMP_REG && RegOffset <= CCC_CR_DCCLANESTATUS2_REG) {
      // 0x00008A00 to 0x00008A74
      Offset = RegOffset - (CCC_CR_DDRCRCACOMP_REG - CCC_CR_DDRCRCACOMP_P0_REG); // 512
    } else {
      // Check if the input register offset is unique to TGL B0
      if (RegOffset == DDRVREF_CR_DDRCRVREFADJUST1_REG) {
        // 0x00002B08
        Offset = 0xFFFFFFFF;
      } else if ((RegOffset >= MC0_CH0_CR_GENERIC_MRS_FSM_PER_RANK_VALUES_0_REG && RegOffset <= MC0_CH0_CR_GENERIC_MRS_FSM_TIMING_STORAGE_REG) ||
                 (RegOffset >= MC0_CH1_CR_GENERIC_MRS_FSM_PER_RANK_VALUES_0_REG && RegOffset <= MC0_CH1_CR_GENERIC_MRS_FSM_TIMING_STORAGE_REG) ||
                 (RegOffset >= MC0_BC_CR_GENERIC_MRS_FSM_PER_RANK_VALUES_0_REG && RegOffset <= MC0_BC_CR_GENERIC_MRS_FSM_TIMING_STORAGE_REG) ||
                 (RegOffset >= MC1_CH0_CR_GENERIC_MRS_FSM_PER_RANK_VALUES_0_REG && RegOffset <= MC1_CH0_CR_GENERIC_MRS_FSM_TIMING_STORAGE_REG) ||
                 (RegOffset >= MC1_CH1_CR_GENERIC_MRS_FSM_PER_RANK_VALUES_0_REG && RegOffset <= MC1_CH1_CR_GENERIC_MRS_FSM_TIMING_STORAGE_REG) ||
                 (RegOffset >= MC1_BC_CR_GENERIC_MRS_FSM_PER_RANK_VALUES_0_REG && RegOffset <= MC1_BC_CR_GENERIC_MRS_FSM_TIMING_STORAGE_REG)) {
        // 0x00004190 to 0x000041E0
        // 0x00004590 to 0x000045E0
        // 0x00004D90 to 0x00004DE0
        // 0x00014190 to 0x000141E0
        // 0x00014590 to 0x000145E0
        // 0x00014D90 to 0x00014DE0
        Offset = 0xFFFFFFFF;
      } else {
        // If none of the conditions are met, return the input register offset unchanged
        Offset = RegOffset;
      }
    }
  } // DtHalo

  return Offset;
}
/**
  This function calculates the Roundtrip latency based off of technology

  @param[in] MrcData - Include all MRC global data.
  @param[in] Roundtrip - Common calculated roundtrip among technolgies

  @retval Roundtrip - Calcuated Roundtrip specific to technology
**/
UINT32
MrcGetRoundTrip (
  IN MrcParameters *const  MrcData,
  IN UINT32                RoundTrip
  )
{
  MrcInput          *Inputs;
  MrcOutput         *Outputs;
  UINT32            Gear2;
  BOOLEAN           Ddr4;
  BOOLEAN           Ddr5;
  BOOLEAN           UlxUlt;
  BOOLEAN           Lpddr4;

  Outputs = &MrcData->Outputs;
  Inputs = &MrcData->Inputs;
  Gear2 = (Outputs->Gear2) ? 1 : 0;
  UlxUlt = Inputs->UlxUlt;
  Lpddr4 = (Outputs->DdrType == MRC_DDR_TYPE_LPDDR4);
  Ddr4 = (Outputs->DdrType == MRC_DDR_TYPE_DDR4);
  Ddr5 = (Outputs->DdrType == MRC_DDR_TYPE_DDR5);

  if (Gear2) {
    if (Ddr4) {
      RoundTrip += ((UlxUlt) ? 12 : 11);
    } else {
      RoundTrip += ((Lpddr4) ? 12 : ((Ddr5) ? 8 : 17));
    }
  } else {
    if (Ddr4) {
      RoundTrip += ((UlxUlt) ? 16 : 15);
    } else {
      RoundTrip += 18;
    }
  }
  return RoundTrip;
}

/**
  This funtion finds the maximum possible increment and  decrement values for write leveling trainings

  @param[in] MrcData - Pointer to MRC global data.
  @param[out] MaxAdd  - Max possible increment value for write leveling trainings
  @param[out] MaxDec  - Max possible decrement value for write leveling trainings

  @retval Success/mrcWrongInputParameter
**/
MrcStatus
MrcMcTxCycleLimits (
  IN MrcParameters *const MrcData,
  OUT UINT32              *MaxAdd,
  OUT UINT32              *MaxDec
  )
{
  UINT32              tCWL;
  INT32               TxFifoSeparation;
  INT64               GetSetVal;
  UINT8               FirstChannel;
  UINT8               FirstController;
  BOOLEAN             Lpddr4;
  MrcDebug            *Debug;
  MrcOutput           *Outputs;

  Outputs          = &MrcData->Outputs;
  Debug            = &Outputs->Debug;
  Lpddr4           = (Outputs->DdrType == MRC_DDR_TYPE_LPDDR4);
  FirstController  = (MrcControllerExist(MrcData, cCONTROLLER0)) ? 0 : 1;
  FirstChannel     = Outputs->Controller[FirstController].FirstPopCh;

  if ((MaxAdd == NULL) || (MaxDec == NULL)) {
    MRC_DEBUG_MSG (Debug, MSG_LEVEL_ERROR, "%s - NULL output pointer\n", gErrString);
    return mrcWrongInputParameter;
  }

  *MaxAdd = MAX_ADD_DELAY * (Outputs->Gear2 ? 2 : 1) + MAX_ADD_RANK_DELAY;
  *MaxDec = 0;

  MrcGetSetMcCh (MrcData, FirstController, FirstChannel, GsmMctCWL, ReadFromCache, &GetSetVal);
  tCWL = (UINT32) GetSetVal;
  if (Lpddr4) {
    tCWL++;     // Due to tDQSS on LP4
  }

  TxFifoSeparation = MrcGetTxFifoSeparation (MrcData);
  // Make sure TxDqFifoRdEnTcwlDelay won't become negative
  if (Outputs->Gear2) {
    *MaxDec = (tCWL - (Lpddr4 ? 1 : 3) + (tCWL % 2) - 2 * Lpddr4 * (tCWL % 2) + ((TxFifoSeparation < 0) ? (TxFifoSeparation + 1) : 0)) / 2;
  } else {
    *MaxDec = tCWL - 2 + ((TxFifoSeparation < 0) ? TxFifoSeparation : 0);
  }
  MRC_WRLVL_FLYBY_DEBUG_MSG (Debug, MSG_LEVEL_NOTE, "%s : MaxAdd = %d, MaxDec = %d \n", __FUNCTION__, *MaxAdd, *MaxDec);

  return mrcSuccess;
}


/**
  Find the minimum and maximum PI setting across Tx DQ/DQS on a given Rank, on all channels.
  Determine how far we can use the PI value to shift the Cycle. Min value will use DQS,
  Max value will use DQ (DQ is DQS + 32 (gear1) or DQS + 96 (gear2), and also plus tDQS2DQ for LP4).

  @param[in]  MrcData       - Pointer to MRC global data.
  @param[in]  Rank          - Current working rank
  @param[in]  MaxAdd        - Max possible increment value for write leveling trainings
  @param[in]  MaxDec        - Max possible decrement value for write leveling trainings
  @param[out] StartOffset   - Starting offset
  @param[out] EndOffset     - End of limit
  @param[out] SavedTxDqsVal - Pointer to array where current Tx Dqs timings will be stored

  @retval MrcStatus - If it succeeded, return mrcSuccess
**/

MrcStatus
MrcIoTxLimits (
  IN  MrcParameters *const MrcData,
  IN  UINT32               Rank,
  IN  UINT32               MaxAdd,
  IN  UINT32               MaxDec,
  OUT INT32               *StartOffset,
  OUT INT32               *EndOffset,
  OUT UINT16              SavedTxDqsVal[MAX_CONTROLLER][MAX_CHANNEL][MAX_SDRAM_IN_DIMM]
  )
{
  UINT32      Controller;
  UINT32      Channel;
  INT64       GetSetVal;
  UINT16      TxDqsValue;
  UINT16      TxDqMaxVal;
  UINT16      MaxCode;
  UINT16      MinCode;
  UINT8       Byte;
  UINT8       PiCycleDec;
  UINT8       PiCycleAdd;
  MrcOutput   *Outputs;
  MrcDebug    *Debug;

  MaxCode         = 0;
  Outputs         = &MrcData->Outputs;
  Debug           = &Outputs->Debug;

  if ((EndOffset == NULL) || (StartOffset == NULL) || (SavedTxDqsVal == NULL)) {
    MRC_DEBUG_MSG (Debug, MSG_LEVEL_ERROR, "%s - NULL output pointer\n", gErrString);
    return mrcWrongInputParameter;
  }

  MrcGetSetLimits (MrcData, TxDqDelay, NULL, &GetSetVal, NULL);
  TxDqMaxVal  = (UINT16) GetSetVal;
  MinCode     = TxDqMaxVal;

  for (Controller = 0; Controller < MAX_CONTROLLER; Controller++) {
    for (Channel = 0; Channel < Outputs->MaxChannels; Channel++) {
      if (!(MrcRankExist (MrcData, Controller, Channel, Rank))) {
        continue;
      }
      for (Byte = 0; Byte < Outputs->SdramCount; Byte++) {
        MrcGetSetStrobe (MrcData, Controller, Channel, Rank, Byte, TxDqDelay, ReadFromCache, &GetSetVal);
        MaxCode = MAX (MaxCode, (UINT16) GetSetVal);

        MrcGetSetStrobe (MrcData, Controller, Channel, Rank, Byte, TxDqsDelay, ReadFromCache, &GetSetVal);
        TxDqsValue = (UINT16) GetSetVal;
        // Save current TcDqs value
        SavedTxDqsVal[Controller][Channel][Byte] = TxDqsValue;
        MinCode = MIN (MinCode, TxDqsValue);
      }
    }
  } //Controller

  PiCycleDec = (UINT8) (MinCode / 128);
  PiCycleAdd = (UINT8) ((TxDqMaxVal - MaxCode) / 128);

  if (Outputs->Gear2) {
    PiCycleDec = 0;
    *StartOffset = - 2 * MaxDec;
  } else {
    *StartOffset = -PiCycleDec - MaxDec;  // We can go more negative offsets using PI settings
  }
  *EndOffset   = MaxAdd + PiCycleAdd;


  MRC_DEBUG_MSG (Debug, MSG_LEVEL_NOTE, "Params : MaxDec: %d, MaxAdd: %d, PiCycleDec: %d, PiCycleAdd: %d\n", MaxDec, MaxAdd, PiCycleDec, PiCycleAdd);
  MRC_DEBUG_MSG (Debug, MSG_LEVEL_NOTE, "MinCode: %d, MaxCode: %d, Start: %d, End: %d\n", MinCode, MaxCode, *StartOffset, *EndOffset);

  return mrcSuccess;
}

/**
  Programs new delay offsets to DQ/DQS timing

  @param[in]  MrcData       - Pointer to MRC global data.
  @param[in]  Rank          - Current working rank
  @param[in]  MaxAdd        - Max possible increment value for write leveling trainings
  @param[in]  MaxDec        - Max possible decrement value for write leveling trainings
  @param[in]  SavedTxDqsVal - array of the initial TX DQS Delays
  @param[in]  TxDelayOffset - Delay to be programmed

  @retval mrcSuccess or mrcTimingError if minimum TX fifo separation is not met
**/

MrcStatus
SetWriteCycleDelay (
  IN  MrcParameters *const  MrcData,
  IN  UINT32                Rank,
  IN  UINT32                MaxAdd,
  IN  UINT32                MaxDec,
  IN  UINT16                SavedTxDqsVal[MAX_CONTROLLER][MAX_CHANNEL][MAX_SDRAM_IN_DIMM],
  IN  INT32                 TxDelayOffset
  )
{
  UINT32                        Controller;
  UINT32                        Channel;
  UINT32                        IpChannel;
  UINT32                        Offset;
  UINT32                        tCWL;
  UINT8                         Byte;
  UINT32                        FirstChannel;
  UINT32                        FirstController;
  INT32                         GlobalByteOff;
  INT64                         FifoFlybyDelay;
  INT64                         GetSetVal;
  INT64                         TxDqDqsDelayAdj;
  INT64                         TxDqFifoWrEn;
  INT64                         TxDqFifoRdEn;
  INT32                         CycleDelay[MAX_CONTROLLER][MAX_CHANNEL];
  UINT32                        AddOneQclk[MAX_CONTROLLER][MAX_CHANNEL];
  BOOLEAN                       Lpddr;
  BOOLEAN                       Lpddr4;
  const MRC_FUNCTION            *MrcCall;
  const MrcInput                *Inputs;
  MrcOutput                     *Outputs;
  MrcDebug                      *Debug;
  MC0_CH0_CR_SC_WR_DELAY_STRUCT ScWrDelay;

  Inputs           = &MrcData->Inputs;
  Outputs          = &MrcData->Outputs;
  Debug            = &Outputs->Debug;
  MrcCall          = Inputs->Call.Func;
  Lpddr            = Outputs->Lpddr;
  // 0.5UI for Gear 1, 1.5UI for Gear 2
  TxDqDqsDelayAdj  = Outputs->Gear2 ? 96 : 32;
  Lpddr4           = (Outputs->DdrType == MRC_DDR_TYPE_LPDDR4);
  FirstController  = (MrcControllerExist(MrcData, cCONTROLLER0)) ? 0 : 1;
  FirstChannel     = MrcData->Outputs.Controller[FirstController].FirstPopCh;

  MrcCall->MrcSetMem ((UINT8 *) CycleDelay, sizeof (CycleDelay), 0);
  MrcCall->MrcSetMem ((UINT8 *) AddOneQclk, sizeof (AddOneQclk), 0);

  MrcGetSetMcCh (MrcData, FirstController, FirstChannel, GsmMctCWL, ReadFromCache, &GetSetVal);
  tCWL = (UINT32) GetSetVal;
  if (Lpddr4) {
    tCWL++;     // Due to tDQSS on LP4
  }

  // Program new delay offsets to DQ/DQS timing:
  for (Controller = 0; Controller < MAX_CONTROLLER; Controller++) {
    for (Channel = 0; Channel < Outputs->MaxChannels; Channel++) {
      if (!(MrcRankExist (MrcData, Controller, Channel, Rank))) {
        continue;
      }
      GlobalByteOff = 0;
      if (Outputs->Gear2) {
        AddOneQclk[Controller][Channel] = 1;
        if (TxDelayOffset > (INT32) MaxAdd) {                 // Will use DQ/DQS PI to go right
          CycleDelay[Controller][Channel] = MAX_ADD_DELAY + MAX_ADD_RANK_DELAY;
          GlobalByteOff = 128 * (TxDelayOffset - MaxAdd);
        } else if (TxDelayOffset < -1 - 2 * (INT32) MaxDec) {     // Will use DQ/DQS PI to go left
          CycleDelay[Controller][Channel] = -1 * MaxDec;
          GlobalByteOff = 128 * (TxDelayOffset + 2 * MaxDec);
        } else {
          if (TxDelayOffset > 2 * MAX_ADD_DELAY) {            // Will use TxDqFifoRdEnFlybyDelay
            CycleDelay[Controller][Channel] = TxDelayOffset - MAX_ADD_DELAY;
          } else {                                            // Will use Dec_tCWL / Add_tCWL
            AddOneQclk[Controller][Channel] = ((TxDelayOffset % 2) == 0) ? 1 : 0;  // Add one QCLK in Gear2 every other cycle offset (when CurrentOffset is even)
            if (TxDelayOffset > 0) {
              CycleDelay[Controller][Channel] = (TxDelayOffset + 1) / 2;
            } else {
              CycleDelay[Controller][Channel] = TxDelayOffset / 2;
            }
          }
        }
      } else {  // Gear1
        if (TxDelayOffset > (INT32) MaxAdd) {
          CycleDelay[Controller][Channel] = MaxAdd;
          GlobalByteOff = 128 * (TxDelayOffset - MaxAdd);
        } else if (TxDelayOffset < -1 * (INT32) MaxDec) {
          CycleDelay[Controller][Channel] = -1 * MaxDec;
          GlobalByteOff = 128 * (TxDelayOffset + MaxDec);
        } else {
          CycleDelay[Controller][Channel] = TxDelayOffset;
        }
      }

      // Write Tx DQ/DQS Flyby delays
      // Note that we program these also in case GlobalByteOff is zero, because it might have been non-zero in the previous cycle.
      if (GlobalByteOff != 0) {
        MRC_WRLVL_FLYBY_DEBUG_MSG (Debug, MSG_LEVEL_NOTE, "\nAdd GlobalByteOff = %d to TxDQS Flyby delay: Mc %d Ch %d \n", GlobalByteOff, Controller, Channel);
      }
      for (Byte = 0; Byte < Outputs->SdramCount; Byte++) {
        GetSetVal = SavedTxDqsVal[Controller][Channel][Byte] + GlobalByteOff;
        MrcGetSetStrobe (MrcData, Controller, Channel, Rank, Byte, TxDqsDelay, WriteToCache, &GetSetVal);
        GetSetVal += TxDqDqsDelayAdj;
        MrcGetSetStrobe (MrcData, Controller, Channel, Rank, Byte, TxDqDelay, WriteToCache, &GetSetVal);
      }
      MrcFlushRegisterCachedData (MrcData);

      // Write SC_WR_DELAY
      FifoFlybyDelay = 0;
      ScWrDelay.Data = 0;
      if (CycleDelay[Controller][Channel] < 0) {
        ScWrDelay.Bits.Dec_tCWL = ABS (CycleDelay[Controller][Channel]);
        ScWrDelay.Bits.Add_tCWL = 0;
      } else {
        if (CycleDelay[Controller][Channel] > MAX_ADD_DELAY) {
          ScWrDelay.Bits.Add_tCWL = MAX_ADD_DELAY;
          FifoFlybyDelay = CycleDelay[Controller][Channel] - MAX_ADD_DELAY;
        } else {
          ScWrDelay.Bits.Add_tCWL = CycleDelay[Controller][Channel];
        }
        ScWrDelay.Bits.Dec_tCWL = 0;
      }
      IpChannel = LP_IP_CH (Lpddr, Channel);
      Offset = OFFSET_CALC_MC_CH (MC0_CH0_CR_SC_WR_DELAY_REG, MC1_CH0_CR_SC_WR_DELAY_REG, Controller, MC0_CH1_CR_SC_WR_DELAY_REG, IpChannel);
      MrcWriteCR (MrcData, Offset, ScWrDelay.Data);

      MrcGetSetMcChRnk (MrcData, Controller, Channel, Rank, TxDqFifoRdEnFlybyDelay, WriteToCache, &FifoFlybyDelay);
      GetSetVal = 0;
      MrcGetSetMcCh (MrcData, Controller, Channel, GsmIocTxDqFifoRdEnPerRankDelDis, WriteToCache, &GetSetVal);

      MrcGetTxDqFifoDelays (MrcData, Controller, Channel, tCWL, ScWrDelay.Bits.Add_tCWL, ScWrDelay.Bits.Dec_tCWL, &TxDqFifoWrEn, &TxDqFifoRdEn);
      TxDqFifoRdEn += AddOneQclk[Controller][Channel]; // TxDqFifoRdEnTcwlDelay(DCLK)

      if (TxDqFifoRdEn < 0) {
        MRC_DEBUG_MSG (Debug, MSG_LEVEL_ERROR, "\n%s Not enough Tx FIFO separation! TxDqFifoWrEn: %d, TxFifoSeparation: %d\n", gErrString, (UINT32) TxDqFifoWrEn, MrcGetTxFifoSeparation (MrcData));
        return mrcTimingError;
      }
      MrcGetSetMcCh (MrcData, Controller, Channel, TxDqFifoWrEnTcwlDelay, WriteToCache, &TxDqFifoWrEn);
      MrcGetSetMcCh (MrcData, Controller, Channel, TxDqFifoRdEnTcwlDelay, WriteToCache, &TxDqFifoRdEn);
      MrcFlushRegisterCachedData (MrcData);
    } // for Channel
  } //Controller

  return mrcSuccess;
}
