/** @file
  This module configures the memory controller address decoder.

@copyright
  INTEL CONFIDENTIAL
  Copyright 1999 - 2020 Intel Corporation.

  The source code contained or described herein and all documents related to the
  source code ("Material") are owned by Intel Corporation or its suppliers or
  licensors. Title to the Material remains with Intel Corporation or its suppliers
  and licensors. The Material may contain trade secrets and proprietary and
  confidential information of Intel Corporation and its suppliers and licensors,
  and is protected by worldwide copyright and trade secret laws and treaty
  provisions. No part of the Material may be used, copied, reproduced, modified,
  published, uploaded, posted, transmitted, distributed, or disclosed in any way
  without Intel's prior express written permission.

  No license under any patent, copyright, trade secret or other intellectual
  property right is granted to or conferred upon you by disclosure or delivery
  of the Materials, either expressly, by implication, inducement, estoppel or
  otherwise. Any license under such intellectual property rights must be
  express and approved by Intel in writing.

  Unless otherwise agreed by Intel in writing, you may not remove or alter
  this notice or any other notice embedded in Materials by Intel or
  Intel's suppliers or licensors in any way.

  This file contains an 'Intel Peripheral Driver' and is uniquely identified as
  "Intel Reference Module" and is licensed for Intel CPUs and chipsets under
  the terms of your license agreement with Intel or your vendor. This file may
  be modified by the user, subject to additional terms of the license agreement.

@par Specification Reference:
**/
#include "MrcAddressDecodeConfiguration.h"

/**
  This function configures the zone configuration registers MAD-CR and MAD-ZR-CR.

  @param[in, out] MrcData     - Include all MRC global data.
  @param[in]      Controller  - Controller to configure.

  @retval Nothing.
**/
void
ZoneConfiguration (
  IN OUT MrcParameters *const MrcData,
  IN     UINT32               Controller
  )
{
  static const UINT8  ChWidDecode[MAX_MRC_DDR_TYPE] = {2, 1, 0, 0, 0xFF};
  MrcInput            *Inputs;
  MrcOutput           *Outputs;
  MrcDebug            *Debug;
  MrcControllerOut    *ControllerOut;
  MrcChannelOut       *ChannelOut;
  INT64               ChLMap;
  INT64               StkdMode;
  INT64               StkdModeCh1;
  INT64               EnhancedChMode;
  INT64               StkdModeChHashBits;
  INT64               ChSSize;
  INT64               DramType;
  INT64               HashMode;
  INT64               LsbMaskBit;
  INT64               HashMask;
  INT64               McChWidth;
  UINT32              ChannelSizeMin;
  UINT32              ChannelSize[MAX_CHANNEL];
  UINT32              CapacityDivisor;
  UINT32              Channel;
  UINT32              Dimm;
  UINT32              SystemChannel;
  UINT32              SmallChIdx;
  BOOLEAN             FullLpChPop;
  MC0_CHANNEL_EHASH_STRUCT  ChannelEHash;

  Inputs              = &MrcData->Inputs;
  Outputs             = &MrcData->Outputs;
  Debug               = &Outputs->Debug;
  ControllerOut       = &Outputs->Controller[Controller];
  CapacityDivisor     = 512;
  StkdModeChHashBits  = 0;
  StkdMode            = 0;
  EnhancedChMode      = Outputs->EnhancedChannelMode;
  StkdModeCh1         = 0;
  DramType            = Outputs->DdrType;
  HashMode            = 0;
  SystemChannel       = (EnhancedChMode) ? ctChannel2 : ctChannel1;

  // Add up the amount of memory in each channel.
  FullLpChPop = TRUE;   // Assume all LPDDR channels (4x16) are populated in the controller.
  for (Channel = 0; Channel < MAX_CHANNEL; Channel++) {
    ChannelOut = &ControllerOut->Channel[Channel];
    ChannelSize[Channel] = 0;
    if (ChannelOut->Status == CHANNEL_PRESENT) {
      for (Dimm = 0; Dimm < MAX_DIMMS_IN_CHANNEL; Dimm++) {
        if (ChannelOut->Dimm[Dimm].Status == DIMM_PRESENT) {
          ChannelSize[Channel] += ChannelOut->Dimm[Dimm].DimmCapacity;
        }
      }
    } else {
      // If any channel is not present in the controller, set the flag to false for LPDDR.
      FullLpChPop = FALSE;
    }
  }

  if (ChannelSize[SystemChannel] <= ChannelSize[ctChannel0]) {
    ChLMap = 0;
    ChannelSizeMin = ChannelSize[SystemChannel];
    SmallChIdx = SystemChannel;
  } else {
    //  ChannelSize0 < ChannelSize1
    ChLMap = 1;
    ChannelSizeMin = ChannelSize[ctChannel0];
    SmallChIdx = ctChannel0;
  }
  // Need to add the capacity of the second LP Channel in Enhanced Channel Mode
  // Channels must be homogeneous
  if (Outputs->EnhancedChannelMode) {
    SmallChIdx++;
    if (MrcChannelExist (MrcData, Controller, SmallChIdx)) {
      ChannelSizeMin += ChannelSize[SmallChIdx];
    }
  }
  ChSSize = ChannelSizeMin / CapacityDivisor;
    // Interleaved mode
    // Check for any Channel hash support
    // Channel hash override enabled then take values from input parameters
    if (Inputs->ChHashOverride == TRUE) {
      if (Inputs->ChHashEnable) {
        HashMask = Inputs->ChHashMask;
        LsbMaskBit = Inputs->ChHashInterleaveBit;
        HashMode = 1;
        MRC_DEBUG_MSG(Debug, MSG_LEVEL_NOTE, "Channel HASH Enabled\n");
      }
    } else if (DramType == MRC_DDR_TYPE_DDR4) {
      HashMask = 0;
      LsbMaskBit = 0;
      HashMode = 0;
      MRC_DEBUG_MSG (Debug, MSG_LEVEL_NOTE, "Channel HASH Auto - Default to DDR4\n");
    } else {
      HashMask = 0x830;
      LsbMaskBit = 2;
      HashMode = 1;
      MRC_DEBUG_MSG (Debug, MSG_LEVEL_NOTE, "Channel HASH Auto Enabled - Default to DramType:%d\n", DramType);
    }
    MrcGetSetMc (MrcData, Controller, GsmMccLsbMaskBit, WriteNoCache | PrintValue, &LsbMaskBit);
    MrcGetSetMc (MrcData, Controller, GsmMccHashMask, WriteNoCache | PrintValue, &HashMask);

  MrcGetSetMc (MrcData, Controller, GsmMccHashMode, WriteNoCache | PrintValue, &HashMode);

  ChannelEHash.Data = 0;
  if (EnhancedChMode && FullLpChPop) {
    // EHASH can only be enabled if all LPDDR channels are populated on the controller.
    // If RI is off, EHASH must be off.
    ChannelEHash.Bits.EHASH_MODE = Inputs->RankInterleave;
    ChannelEHash.Bits.EHASH_LSB_MASK_BIT = 2;
    ChannelEHash.Bits.EHASH_MASK = 0x830;
  }
  MrcWriteCR (MrcData, OFFSET_CALC_CH (MC0_CHANNEL_EHASH_REG, MC1_CHANNEL_EHASH_REG, Controller), ChannelEHash.Data);

  McChWidth   = ChWidDecode[Outputs->DdrType];

  MrcGetSetMc (MrcData, Controller, GsmMccAddrDecodeDdrType,  ForceWriteCached | PrintValue, &DramType);
  MrcGetSetMc (MrcData, Controller, GsmMccChWidth,            WriteToCache     | PrintValue, &McChWidth);
  if (Inputs->A0) {
    // The registers associated with these GetSets only exist on TGL A0
    MrcGetSetMc (MrcData, Controller, GsmMccStackedMode,        WriteToCache     | PrintValue, &StkdMode);
    MrcGetSetMc (MrcData, Controller, GsmMccStackChMap,         WriteToCache     | PrintValue, &StkdModeCh1);
    MrcGetSetMc (MrcData, Controller, GsmMccStackedChHash,      WriteToCache     | PrintValue, &StkdModeChHashBits);
  }
  MrcGetSetMc (MrcData, Controller, GsmMccSChannelSize,       WriteToCache     | PrintValue, &ChSSize);
  MrcGetSetMc (MrcData, Controller, GsmMccLChannelMap,        WriteToCache     | PrintValue, &ChLMap);
  MrcFlushRegisterCachedData (MrcData);
}

/**
  This function configures the zone configuration registers MAD_SLICE.

  @param[in, out] MrcData - Include all MRC global data.

  @retval Nothing.
**/

void
ControllerZoneConfiguration (
  IN OUT MrcParameters *const MrcData
  )
{
  MrcInput          *Inputs;
  MrcOutput         *Outputs;
  MrcDebug          *Debug;
  MrcControllerOut  *ControllerOut;
  MrcChannelOut     *ChannelOut;
  INT64             MsLMap[MRC_2LM_CMF];
  INT64             StkdMode[MRC_2LM_CMF];
  INT64             StkdModeMs1[MRC_2LM_CMF];
  INT64             StkdModeMsHashBits[MRC_2LM_CMF];
  INT64             MsSSize[MRC_2LM_CMF];
  INT64             HashMode[MRC_2LM_CMF];
  INT64             LsbMaskBit[MRC_2LM_CMF];
  INT64             HashMask[MRC_2LM_CMF];
  INT64             SliceDisable[MAX_CONTROLLER][MRC_2LM_CMF - 1]; //No TOPOLOGY_CFG_REG for Near Memory
  INT64             Sel_2lm_1lm[MRC_2LM_CMF - 1];
  UINT32            ControllerSizeMin;
  UINT32            ControllerSize[MAX_CONTROLLER];
  UINT32            CapacityDivisor;
  UINT8             Channel;
  UINT8             Controller;
  UINT8             Dimm;
  UINT8             Cmf;
  UINT8             CmfMax;
  const MRC_FUNCTION  *MrcCall;
  Inputs              = &MrcData->Inputs;
  Outputs             = &MrcData->Outputs;
  Debug               = &Outputs->Debug;
  MrcCall             = Inputs->Call.Func;
  CapacityDivisor     = 512;
  ControllerSize[cCONTROLLER0] = 0;
  ControllerSize[cCONTROLLER1] = 0;

  // Add up the amount of memory in each controller.
  for (Controller = 0; Controller < MAX_CONTROLLER; Controller++) {
    ControllerOut = &Outputs->Controller[Controller];
    ControllerSize[Controller] = 0;
    if (ControllerOut->Status == CONTROLLER_PRESENT) {
      for (Channel = 0; Channel < MAX_CHANNEL; Channel++) {
        ChannelOut            = &ControllerOut->Channel[Channel];
        if (ChannelOut->Status == CHANNEL_PRESENT) {
          if (ChannelOut->ValidSubChBitMask != 3) {
          }
          for (Dimm = 0; Dimm < MAX_DIMMS_IN_CHANNEL; Dimm++) {
            if (ChannelOut->Dimm[Dimm].Status == DIMM_PRESENT) {
              ControllerSize[Controller] += ChannelOut->Dimm[Dimm].DimmCapacity;
            }
          }
        }
      }
    }
  }
  // Initializing all fields values arrays with zeroes
  // MAD_SLICE_REG fields
  MrcCall->MrcSetMem ((UINT8 *)MsSSize, sizeof (MsSSize), 0);
  MrcCall->MrcSetMem ((UINT8 *)MsLMap, sizeof (MsLMap), 0);
  MrcCall->MrcSetMem ((UINT8 *)StkdMode, sizeof (StkdMode), 0);
  MrcCall->MrcSetMem ((UINT8 *)StkdModeMsHashBits, sizeof (StkdModeMsHashBits), 0);
  MrcCall->MrcSetMem ((UINT8 *)StkdModeMs1, sizeof (StkdModeMs1), 0);
  // MEMORY_SLICE_HASH_REG fields
  MrcCall->MrcSetMem ((UINT8 *)HashMode, sizeof (HashMode), 0);
  MrcCall->MrcSetMem ((UINT8 *)HashMask, sizeof (HashMask), 0);
  MrcCall->MrcSetMem ((UINT8 *)LsbMaskBit, sizeof (LsbMaskBit), 0);
  // TOPOLOGY_GLOBAL_CFG_0 fields
  MrcCall->MrcSetMem ((UINT8 *)SliceDisable, sizeof (LsbMaskBit), 0);
  MrcCall->MrcSetMem ((UINT8 *)Sel_2lm_1lm, sizeof (Sel_2lm_1lm), 0);

  CmfMax = (Inputs->MemBootMode == MEM_BOOT_MODE_2LM) ? MRC_2LM_CMF : MRC_1LM_CMF; // Chose how many CMF registers
  Cmf =  (Inputs->MemBootMode == MEM_BOOT_MODE_2LM) ? MRC_CMF_NM : MRC_CMF0; // Program CMF1_NM or CMF0 registers only

  if (ControllerSize[cCONTROLLER1] <= ControllerSize[cCONTROLLER0]) {
    ControllerSizeMin = ControllerSize[cCONTROLLER1];
  } else {
    // MemorySliceSize0 < MemorySliceSize1
    MsLMap[Cmf] = 1;
    ControllerSizeMin = ControllerSize[cCONTROLLER0];
  }
  // Interleaved mode
    if (Inputs->MsHashEnable) { // Check for any Memory Slice hash support
      HashMask[Cmf] = Inputs->MsHashMask;
      MrcGetSetCmf (MrcData, MRC_IGNORE_ARG, Cmf, GsmCmiHashMask, WriteNoCache | PrintValue, &HashMask[Cmf]);
      LsbMaskBit[Cmf] = Inputs->MsHashInterleaveBit;
      MrcGetSetCmf (MrcData, MRC_IGNORE_ARG, Cmf, GsmCmiLsbMaskBit, WriteNoCache | PrintValue, &LsbMaskBit[Cmf]);
      HashMode[Cmf] = 1;
      MRC_DEBUG_MSG (Debug, MSG_LEVEL_NOTE, "Controller HASH Enabled\n");
    }
  if (Inputs->MemBootMode == MEM_BOOT_MODE_2LM) {
    MRC_DEBUG_MSG (Debug, MSG_LEVEL_NOTE, "Running in 2LM mode\n");
    MsSSize[MRC_CMF0] = Inputs->FarMemorySize / 2;
    MsSSize[MRC_CMF1] = Inputs->FarMemorySize / 2;
    MsSSize[MRC_CMF_NM] = ControllerSizeMin / CapacityDivisor;
    Sel_2lm_1lm[MRC_CMF0] = 1;
    Sel_2lm_1lm[MRC_CMF1] = 1;
  } else {
    MsSSize[MRC_CMF0] = ControllerSizeMin / CapacityDivisor; // Only Program CMF0 ignore everything else
  }

  Cmf = (Inputs->MemBootMode == MEM_BOOT_MODE_2LM) ? MRC_CMF1 : MRC_CMF0; //For 2LM disable bits are in CMF1
  for (Controller = 0; Controller < MAX_CONTROLLER; Controller++) {
    ControllerOut = &Outputs->Controller[Controller];
    if (ControllerOut->Status != CONTROLLER_PRESENT) {
      SliceDisable[Controller][Cmf] = 1;
      MrcGetSetCmf (MrcData, Controller, Cmf, GsmCmiSliceDisable,  WriteToCache | PrintValue, &SliceDisable[Controller][Cmf]);
    }
    MrcGetSetCmf (MrcData, MRC_IGNORE_ARG, Cmf, GsmCmiSel_2lm_1lm,     WriteToCache | PrintValue, &Sel_2lm_1lm[Cmf]);
  }
  if ((ControllerSize[cCONTROLLER1] != 0) && (ControllerSize[cCONTROLLER0] != 0)) {
    Outputs->MemoryMapData.ILMem = ControllerSizeMin * 2;
  } else {
    Outputs->MemoryMapData.ILMem = 0;
  }
  MRC_DEBUG_MSG (Debug, MSG_LEVEL_NOTE, "Interleaved Memory between controllers %d Megabytes\n", Outputs->MemoryMapData.ILMem);

  for (Cmf = 0; Cmf <= CmfMax - 1; Cmf++) {
    MRC_DEBUG_MSG (Debug, MSG_LEVEL_NOTE, "Updating the registers for CMF%d\n", Cmf);
    MrcGetSetCmf (MrcData, MRC_IGNORE_ARG, Cmf, GsmCmiHashMode, WriteNoCache | PrintValue, &HashMode[Cmf]);
    MrcGetSetCmf (MrcData, MRC_IGNORE_ARG, Cmf, GsmCmiStackedMode,       WriteToCache | PrintValue, &StkdMode[Cmf]);
    MrcGetSetCmf (MrcData, MRC_IGNORE_ARG, Cmf, GsmCmiStackMsMap,        WriteToCache | PrintValue, &StkdModeMs1[Cmf]);
    MrcGetSetCmf (MrcData, MRC_IGNORE_ARG, Cmf, GsmCmiStackedMsHash,     WriteToCache | PrintValue, &StkdModeMsHashBits[Cmf]);
    MrcGetSetCmf (MrcData, MRC_IGNORE_ARG, Cmf, GsmCmiSMadSliceSize,     WriteToCache | PrintValue, &MsSSize[Cmf]);
    MrcGetSetCmf (MrcData, MRC_IGNORE_ARG, Cmf, GsmCmiLMadSliceMap,      WriteToCache | PrintValue, &MsLMap[Cmf]);
  }
  MrcFlushRegisterCachedData (MrcData);
}

/**
  This function configures the MAD_DIMM_CH0/1 register.

  @param[in] MrcData    - Include all MRC global data.
  @param[in] Controller - Controller to configure.
  @param[in] Channel    - Channel to configure.

  @retval Nothing.
**/
void
ChannelAddressDecodeConfiguration (
  IN MrcParameters *const MrcData,
  IN const UINT32         Controller,
  IN const UINT32         Channel
  )
{
  const MrcInput      *Inputs;
  MrcOutput           *Outputs;
  MrcChannelOut       *ChannelOut;
  MrcControllerOut    *ControllerOut;
  MrcDimmOut          *Dimm0Out;
  MrcDimmOut          *Dimm1Out;
  MrcDimmOut          *DimmL;
  MrcDimmOut          *DimmS;
  MrcDimmOut          *LpOrDdrDimm1;
  MrcDdrType          DdrType;
  INT64               GetSetVal;
  INT64               DimmLMap;
  INT64               DLNOR;
  INT64               DSNOR;
  INT64               RankInterleave;
  INT64               EnhancedInterleave;
  INT64               EccMode;
  INT64               DimmLSize;
  INT64               DimmSSize;
  UINT32              DimmLDramWidth;
  UINT32              DimmSDramWidth;
  UINT32              Dimm0Capacity;
  UINT32              Dimm1Capacity;
  UINT32              Offset;
  UINT32              CapacityDivisor;
  UINT32              IpChannel;
  UINT32              DimmSize;
  UINT32              Ddr5Device8Gb;
  BOOLEAN             Lpddr;
  MC0_CH0_CR_DDR_MR_PARAMS_STRUCT    DdrMrParams;

  Inputs            = &MrcData->Inputs;
  Outputs           = &MrcData->Outputs;
  ControllerOut     = &Outputs->Controller[Controller];
  ChannelOut        = &ControllerOut->Channel[Channel];
  DdrType           = Outputs->DdrType;
  DdrMrParams.Data  = 0;
  CapacityDivisor   = 512;
  DimmLDramWidth    = 0;
  DimmSDramWidth    = 0;
  DimmLSize         = 0;
  DimmSSize         = 0;
  DLNOR             = 0;
  DSNOR             = 0;
  Ddr5Device8Gb     = 1;

  // If we are in EnhancedChannelMode treat LPDDR Ch 1/3 as the second DIMM in the MC Channel.
  // Otherwise use DIMM1 in the same struct.
  LpOrDdrDimm1  = (Outputs->EnhancedChannelMode) ? &ControllerOut->Channel[Channel + 1].Dimm[dDIMM0] : &ChannelOut->Dimm[dDIMM1];
  Dimm0Out      = &ChannelOut->Dimm[dDIMM0];
  Dimm1Out      = &ChannelOut->Dimm[dDIMM1];
  DdrType       = MrcData->Outputs.DdrType;
  Lpddr         = Outputs->Lpddr;
  IpChannel     = LP_IP_CH (Lpddr, Channel);

  if (Dimm0Out->Status == DIMM_PRESENT) {
    Dimm0Capacity = Dimm0Out->DimmCapacity;
    // ddr5_device_8Gb:   0 - DDR5 capacity is more than 8Gb
    //                    1 - DDR5 capacity is 8Gb (default value)
    Ddr5Device8Gb = (((SdramCapacityTable[Dimm0Out->DensityIndex] * 8) > 8192) ? 0 : 1);
  } else {
    Dimm0Capacity = 0;
    Ddr5Device8Gb = (((SdramCapacityTable[Dimm1Out->DensityIndex] * 8) > 8192) ? 0 : 1);;
  }

  if (LpOrDdrDimm1->Status == DIMM_PRESENT) {
    Dimm1Capacity = (MAX_DIMMS_IN_CHANNEL > 1) ? LpOrDdrDimm1->DimmCapacity : 0;
  } else {
    Dimm1Capacity = 0;
  }

  // larger dimm will be located to Dimm L and small dimm will be located to dimm S
  if (Dimm1Capacity <= Dimm0Capacity) {
    DimmL = Dimm0Out;
    DimmS = LpOrDdrDimm1;
    // larger DIMM in capacity 0 - DIMM 0 or 1 - DIMM 1
    DimmLMap = 0;
  } else {
    DimmL = LpOrDdrDimm1;
    DimmS = Dimm0Out;
    // larger DIMM in capacity 0 - DIMM 0 or 1 - DIMM 1
    DimmLMap = 1;
  }
  // Dimm L
  if ((0 < DimmL->RankInDimm) && (DimmL->Status == DIMM_PRESENT)) {
    DimmLSize = DimmL->DimmCapacity / CapacityDivisor;
    if ((Lpddr) && (Inputs->CpuModel == cmTGL_ULX_ULT)) {
      // For non-power of two device sizes we need to program DIMM_L/S_SIZE to the nearest power of two.
      DimmSize = (UINT32) DimmLSize;
      if (MrcCountBitsEqOne (DimmSize) == 2) { // 3, 6, 12, 24 etc.
        DimmLSize = (DimmSize * 4) / 3;        // Round up to the nearest power of two
      }
    }
    // RankInDimm must be 1 or 2, we test the case that the value is 0
    DLNOR = DimmL->RankInDimm - 1;
    // SDRAM width
    switch (DimmL->SdramWidth) {
      case 8:
        DimmLDramWidth = 0;
        break;

      case 16:
        DimmLDramWidth = 1;
        break;

      case 32:
      case 64:
        DimmLDramWidth = 2;
        break;
    }
  }
  // Dimm S
  if ((MAX_DIMMS_IN_CHANNEL > 1) && (0 < DimmS->RankInDimm) && (DimmS->Status == DIMM_PRESENT)) {
    DimmSSize = DimmS->DimmCapacity / CapacityDivisor;
    if ((Lpddr) && (Inputs->CpuModel == cmTGL_ULX_ULT)) {
      // For non-power of two device sizes we need to program DIMM_L/S_SIZE to the nearest power of two.
      DimmSize = (UINT32) DimmSSize;
      if (MrcCountBitsEqOne (DimmSize) == 2) { // 3, 6, 12, 24 etc.
        DimmSSize = (DimmSize * 4) / 3;        // Round up to the nearest power of two
      }
    }
    // RankInDimm must be 1 or 2, we test the case that this value is 0.
    DSNOR = DimmS->RankInDimm - 1;
    // SDRAM width
    switch (DimmS->SdramWidth) {
      case 8:
        DimmSDramWidth = 0;
        break;

      case 16:
        DimmSDramWidth = 1;
        break;

      case 32:
      case 64:
        DimmSDramWidth = 2;
        break;
    }
  }

  RankInterleave = (Inputs->RankInterleave) ? 1 : 0;
  EnhancedInterleave = (Inputs->EnhancedInterleave) ? 1 : 0;

  if (Outputs->EccSupport) {
    // Set to '01' - ECC is active in IO, ECC logic is not active.
    EccMode = emEccIoActive;
  } else {
    EccMode = 0;
  }

  // DIMM Config
  MrcGetSetMcCh (MrcData, Controller, Channel, GsmMccLDimmSize,      WriteToCache | PrintValue, &DimmLSize);
  MrcGetSetMcCh (MrcData, Controller, Channel, GsmMccLDimmRankCnt,   WriteToCache | PrintValue, &DLNOR);
  MrcGetSetMcCh (MrcData, Controller, Channel, GsmMccSDimmSize,      WriteToCache | PrintValue, &DimmSSize);
  MrcGetSetMcCh (MrcData, Controller, Channel, GsmMccSDimmRankCnt,   WriteToCache | PrintValue, &DSNOR);
  GetSetVal = DimmLDramWidth;
  MrcGetSetMcCh (MrcData, Controller, Channel, GsmMccLDimmDramWidth, WriteToCache | PrintValue, &GetSetVal);
  GetSetVal = DimmSDramWidth;
  MrcGetSetMcCh (MrcData, Controller, Channel, GsmMccSDimmDramWidth, WriteToCache | PrintValue, &GetSetVal);

  if (DdrType == MRC_DDR_TYPE_DDR5) {
    GetSetVal = Ddr5Device8Gb;
    MrcGetSetMcCh (MrcData, Controller, Channel, GsmMccSDdr58GbDevice, WriteToCache | PrintValue, &GetSetVal);
  }

  //IntraChannel Config
  MrcGetSetMcCh (MrcData, Controller, Channel, GsmMccLDimmMap,           WriteToCache | PrintValue, &DimmLMap);
  if (Inputs->A0) {
    // The registers associated with this GetSet only exist on TGL A0
    MrcGetSetMcCh (MrcData, Controller, Channel, GsmMccRankInterleave,     WriteToCache | PrintValue, &RankInterleave);
  }
  MrcGetSetMcCh (MrcData, Controller, Channel, GsmMccEnhancedInterleave, WriteToCache | PrintValue, &EnhancedInterleave);
  MrcGetSetMcCh (MrcData, Controller, Channel, GsmMccEccMode,            WriteToCache | PrintValue, &EccMode);

  MrcFlushRegisterCachedData (MrcData);

  if (!Inputs->MrcSafeConfig) {
    DdrMrParams.Bits.MR4_PERIOD = 0x200D;

    if (DdrType == MRC_DDR_TYPE_DDR4) {
      DdrMrParams.Bits.DDR4_TS_readout_en = 1;
    }
  }
  if (DimmLMap == 0) {
    DdrMrParams.Bits.Rank_0_width = DimmLDramWidth;
    DdrMrParams.Bits.Rank_1_width = DimmLDramWidth;
    DdrMrParams.Bits.Rank_2_width = DimmSDramWidth;
    DdrMrParams.Bits.Rank_3_width = DimmSDramWidth;
  } else {
    DdrMrParams.Bits.Rank_0_width = DimmSDramWidth;
    DdrMrParams.Bits.Rank_1_width = DimmSDramWidth;
    DdrMrParams.Bits.Rank_2_width = DimmLDramWidth;
    DdrMrParams.Bits.Rank_3_width = DimmLDramWidth;
  }

  Offset = OFFSET_CALC_MC_CH (MC0_CH0_CR_DDR_MR_PARAMS_REG, MC1_CH0_CR_DDR_MR_PARAMS_REG, Controller, MC0_CH1_CR_DDR_MR_PARAMS_REG, IpChannel);
  MrcWriteCR (MrcData, Offset, DdrMrParams.Data);

}

/**
  Initialize the MARS feature (Memory AwaRe Scrubber)

  @param[in] MrcData - Include all MRC global data.

  @retval none
**/
void
MrcMarsConfig (
  IN MrcParameters *const MrcData
  )
{
  /*
  UINT32  Index;
  UINT32  Data32;
  static const struct {
    UINT16 McReg;
    UINT16 IdpReg;
  } RegTable[] = {
    { MC0_MAD_INTER_CHANNEL_REG,  CCF_IDP_MAD_INTER_CHANNEL_REG },
    { MC0_MAD_INTRA_CH0_REG,      CCF_IDP_MAD_INTRA_CH0_REG     },
    { MC0_MAD_INTRA_CH1_REG,      CCF_IDP_MAD_INTRA_CH1_REG     },
    { MC0_MAD_DIMM_CH0_REG,       CCF_IDP_MAD_DIMM_CH0_REG      },
    { MC0_MAD_DIMM_CH1_REG,       CCF_IDP_MAD_DIMM_CH1_REG      },
    { MC0_CHANNEL_HASH_REG,       CCF_IDP_CHANNEL_HASH_REG      },
    { MC0_CHANNEL_EHASH_REG,      CCF_IDP_CHANNEL_EHASH_REG     }
  };
  CCF_IDP_MARS_ENABLE_STRUCT  MarsEnable;

  if (MrcData->Inputs.Mars == 0) {
    return;
  }
  // Copy from MC address decoding registers to CCF IDP register.
  // They have exactly the same format, so can copy as is.
  // @todo Some future SKU might have 2 IDP blocks, need to program both in this case.
  for (Index = 0; Index < ARRAY_COUNT (RegTable); Index++) {
    Data32 = MrcReadCR (MrcData, RegTable[Index].McReg);
    MrcWriteCR (MrcData, RegTable[Index].IdpReg, Data32);
  }

  MarsEnable.Data = 0;
  MarsEnable.Bits.MARS_Enable = 1;
  MrcWriteCR (MrcData, CCF_IDP_MARS_ENABLE_REG, MarsEnable.Data);
  MRC_DEBUG_MSG (&MrcData->Outputs.Debug, MSG_LEVEL_NOTE, "MARS Enabled\n");
  */
  return;
}

/**
  This function is the main address decoding configuration function.

  @param[in] MrcData - Include all MRC global data.

  @retval Nothing.
**/
void
MrcAdConfiguration (
  IN MrcParameters *const MrcData
  )
{
  UINT32      Controller;
  UINT32      Channel;
  BOOLEAN     Lpddr;

  Lpddr = MrcData->Outputs.Lpddr;

  for (Controller = 0; Controller < MAX_CONTROLLER; Controller++) {
    if (!MrcControllerExist (MrcData, Controller)) {
      continue;
    }
    ZoneConfiguration (MrcData, Controller);
    for (Channel = 0; Channel < MAX_CHANNEL; Channel++) {
      if ((!MrcChannelExist (MrcData, Controller, Channel)) || IS_MC_SUB_CH (Lpddr, Channel)) {
        // For LPDDR4/5, only program register on even channels.
        continue;
      }
      ChannelAddressDecodeConfiguration (MrcData, Controller, Channel);
    } // for Channel
  } // for Controller

  // Initialize the MARS feature (Memory AwaRe Scrubber)
  // This has to be done once all the MC Address Decode register are programmed
  MrcMarsConfig (MrcData);

  return;
}
