/** @file
  Implementation of DDR5 Specific functions, and constants.

@copyright
  INTEL CONFIDENTIAL
  Copyright 2019 - 2020 Intel Corporation.

  The source code contained or described herein and all documents related to the
  source code ("Material") are owned by Intel Corporation or its suppliers or
  licensors. Title to the Material remains with Intel Corporation or its suppliers
  and licensors. The Material may contain trade secrets and proprietary and
  confidential information of Intel Corporation and its suppliers and licensors,
  and is protected by worldwide copyright and trade secret laws and treaty
  provisions. No part of the Material may be used, copied, reproduced, modified,
  published, uploaded, posted, transmitted, distributed, or disclosed in any way
  without Intel's prior express written permission.

  No license under any patent, copyright, trade secret or other intellectual
  property right is granted to or conferred upon you by disclosure or delivery
  of the Materials, either expressly, by implication, inducement, estoppel or
  otherwise. Any license under such intellectual property rights must be
  express and approved by Intel in writing.

  Unless otherwise agreed by Intel in writing, you may not remove or alter
  this notice or any other notice embedded in Materials by Intel or
  Intel's suppliers or licensors in any way.

  This file contains an 'Intel Peripheral Driver' and is uniquely identified as
  "Intel Reference Module" and is licensed for Intel CPUs and chipsets under
  the terms of your license agreement with Intel or your vendor. This file may
  be modified by the user, subject to additional terms of the license agreement.

@par Specification Reference:
  DDR5 JEDEC Spec
**/

#include "MrcDdr5.h"
#include "MrcDdrCommon.h"
#include "MrcHalRegisterAccess.h"
#include "MrcApi.h"
#include "MrcCommon.h"
#include "MrcChipApi.h"
#include "MrcDdr5Registers.h"
#include "MrcMemoryApi.h"


///
/// Initialization Timing Parameters
///
#define MRC_DDR5_tINIT0_MS   20      ///< Max voltage-ramp time
#define MRC_DDR5_tINIT1_US   200     ///< Min RESET_n LOW time after completion of voltage Ramp
#define MRC_DDR5_tINIT2_NS   10      ///< Min CS_n LOW time before RESET_n HIGH
#define MRC_DDR5_tINIT3_MS   4       ///< Min CS_n LOW time after RESET_n HIGH
#define MRC_DDR5_tINIT4_US   2       ///< Minimum time for DRAM to register EXIT on CS_n HIGH
#define MRC_DDR5_tINIT5_NCK  3       ///< Minimum cycles required after CS_n HIGH

/**
  Override CS to the input CsOverrideVal value.

  @param[in] MrcData       - Pointer to MRC global data.
  @param[in] CsOverrideVal - Value to override CS to. Any non-zero value sets CS high.
  @param[in] CsOverrideEn  - Input specifying whether to enable or disable CS Override.

  @retval N/A
**/
void
OverrideCs (
  IN MrcParameters *const MrcData,
  IN UINT8                CsOverrideVal,
  IN BOOLEAN              CsOverrideEn
  )
{
  MrcOutput     *Outputs;
  INT64         ValidRankBitMask;
  INT64         OverrideValRankBitMask;
  INT64         OverrideEnRankBitMask;
  INT32         Channel;
  UINT32        Controller;

  Outputs = &MrcData->Outputs;

  for (Controller = 0; Controller < MAX_CONTROLLER; Controller++) {
    for (Channel = 0; Channel < MAX_CHANNEL; Channel++) {
      if (MrcChannelExist (MrcData, Controller, Channel)) {
        ValidRankBitMask = Outputs->Controller[Controller].Channel[Channel].ValidRankBitMask;
        OverrideValRankBitMask = (CsOverrideVal == 0) ? 0 : ValidRankBitMask;
        OverrideEnRankBitMask = (CsOverrideEn == FALSE) ? 0 : ValidRankBitMask;
        MrcGetSetMcCh (MrcData, Controller, Channel, GsmMccCsOverrideVal, WriteToCache, &OverrideValRankBitMask);
        MrcGetSetMcCh (MrcData, Controller, Channel, GsmMccCsOverride, WriteToCache, &OverrideEnRankBitMask);
      }
    } // Channel
  } // Controller
  MrcFlushRegisterCachedData (MrcData);
}


/**
  This function converts from the integer defined Read Latency to the Mode Register (MR0)
  encoding of the timing in DDR5.

  @param[in]  MrcData - Pointer to global MRC data.
  @param[in]  Value   - Requested Read Latency value.
  @param[out] EncVal  - Encoded Mode Register value.

  @retval MrcStatus - mrcSuccess if the latency is supported.  Else mrcWrongInputParameter.
**/
MrcStatus
EncodeReadLatencyDdr5 (
  IN  MrcParameters *MrcData,
  IN  UINT16        Value,
  OUT UINT8         *EncVal
  )
{
  MrcDebug    *Debug;
  MrcStatus   Status;
  DDR5_MR0_RL MrValue;

  Debug = &MrcData->Outputs.Debug;
  Status = mrcSuccess;

  if (Value < 22 ||  Value > 66 || (Value %2 != 0)) {
    MrValue = Ddr5RlMax;
  } else {
    MrValue = (Value - 22) / 2;
  }

  if (MrValue < Ddr5Rl_22 || MrValue >= Ddr5RlMax) {
    MRC_DEBUG_MSG (Debug, MSG_LEVEL_ERROR, "%s Invalid %s Latency Value: %d\n", gErrString, gRdString, Value);
    Status = mrcWrongInputParameter;
  }


  if (EncVal != NULL) {
    *EncVal = MrValue;
  } else {
    MRC_DEBUG_MSG (Debug, MSG_LEVEL_ERROR, "%s %s\n", __FUNCTION__, gNullPtrErrStr);
    Status = mrcWrongInputParameter;
  }

  return Status;
}

/**
  This function converts from the integer defined Write Recovery to the Mode Register (MR6)
  encoding of the timing in DDR5.

  @param[in]  MrcData - Pointer to global MRC data.
  @param[in]  Value   - Requested Write Recovery value.
  @param[out] EncVal  - Encoded Mode Register value.

  @retval MrcStatus - mrcSuccess if the latency is supported.  Else mrcWrongInputParameter.
**/
MrcStatus
EncodeWriteRecoveryDdr5 (
  IN  MrcParameters *MrcData,
  IN  UINT16        Value,
  OUT UINT8         *EncVal
  )
{
  MrcDebug    *Debug;
  MrcStatus   Status;
  DDR5_MR6_WR MrValue;

  Debug = &MrcData->Outputs.Debug;
  Status = mrcSuccess;

  if (Value < 48 || Value > 96 || (Value % 6 != 0)) {
    MrValue = Ddr5WrMax;
  } else {
    MrValue = (Value - 48) / 6;
  }

  if (MrValue < Ddr5Wr_48 || MrValue >= Ddr5WrMax) {
    MRC_DEBUG_MSG (Debug, MSG_LEVEL_ERROR, "%s Unsupported Write Recovery value\n", gErrString);
    Status = mrcWrongInputParameter;
  }

  if (EncVal != NULL) {
    *EncVal = MrValue;
  } else {
    MRC_DEBUG_MSG (Debug, MSG_LEVEL_ERROR, "%s %s\n", __FUNCTION__, gNullPtrErrStr);
    Status = mrcWrongInputParameter;
  }

  return Status;
}


/**
  This function converts from the integer defined tRTP timing to the Mode Register (MR6)
  encoding of the timing in DDR5.

  @param[in]  MrcData - Pointer to global MRC data.
  @param[in]  Value   - Requested tRTP value.
  @param[out] EncVal  - Encoded Mode Register value.

  @retval MrcStatus - mrcSuccess if the timing is supported.  Else mrcWrongInputParameter.
**/
MrcStatus
EncodeTrtpDdr5 (
  IN  MrcParameters *MrcData,
  IN  UINT16        Value,
  OUT UINT8         *EncVal
  )
{
  MrcDebug      *Debug;
  MrcStatus     Status;
  DDR5_MR6_TRTP MrValue;

  Debug = &MrcData->Outputs.Debug;
  Status = mrcSuccess;


  switch (Value) {
    case 12:
      MrValue = Ddr5Trtp_12;
      break;

    case 14:
      MrValue = Ddr5Trtp_14;
      break;

    case 15:
      MrValue = Ddr5Trtp_15;
      break;

    case 17:
      MrValue = Ddr5Trtp_17;
      break;

    case 18:
      MrValue = Ddr5Trtp_18;
      break;

    case 20:
      MrValue = Ddr5Trtp_20;
      break;

    case 21:
      MrValue = Ddr5Trtp_21;
      break;

    case 23:
      MrValue = Ddr5Trtp_23;
      break;

    case 24:
      MrValue = Ddr5Trtp_24;
      break;

    default:
      MrValue = Ddr5TrtpMax;
      Status = mrcWrongInputParameter;
      break;
  }

  if (MrValue < Ddr5TrtpMax) {
    if (EncVal != NULL) {
      *EncVal = MrValue;
    } else {
      MRC_DEBUG_MSG (Debug, MSG_LEVEL_ERROR, "%s %s\n", __FUNCTION__, gNullPtrErrStr);
      Status = mrcWrongInputParameter;
    }
  } else {
    MRC_DEBUG_MSG (Debug, MSG_LEVEL_ERROR, "%s Invalid %s Latency Value: %d\n", gErrString, gWrString, Value);
    Status = mrcWrongInputParameter;
  }

  return Status;
}

/**
  This function returns the tCCD_L/tCCD_L_WR/tDLLK MR13 encoded value
  for the input frequency value.

  @param[in]  MrcData   - Pointer to global MRC data.
  @param[in]  Frequency - Data Rate
  @param[out] EncVal    - Encoded Mode Register value.

  @retval MrcStatus - mrcSuccess if the timing is supported.  Else mrcWrongInputParameter.
**/
MrcStatus
EncodeDdr5TccdlTdllk (
  IN  MrcParameters *MrcData,
  IN  MrcFrequency  Frequency,
  OUT UINT8         *EncVal
  )
{
  MrcStatus Status;
  MrcDebug  *Debug;
  UINT8     MrValue;

  Debug = &MrcData->Outputs.Debug;
  Status = mrcSuccess;

  if (((f2000 < Frequency) && (Frequency <= f2100)) || (Frequency == f3200)) {
    MrValue = 0;
  } else if ((f3200 < Frequency) && (Frequency <= f3600)) {
    MrValue = 1;
  } else if ((f3600 < Frequency) && (Frequency <= f4000)) {
    MrValue = 2;
  } else if ((f4000 < Frequency) && (Frequency <= f4400)) {
    MrValue = 3;
  } else if ((f4400 < Frequency) && (Frequency <= f4800)) {
    MrValue = 4;
  } else if ((f4800 < Frequency) && (Frequency <= f5200)) {
    MrValue = 5;
  } else if ((f5200 < Frequency) && (Frequency <= f5600)) {
    MrValue = 6;
  } else if ((f5600 < Frequency) && (Frequency <= f6000)) {
    MrValue = 7;
  } else if ((f6000 < Frequency) && (Frequency <= f6400)) {
    MrValue = 8;
  } else {
    MrValue = 0xFF;
  }

  if (MrValue != 0xFF) {
    if (EncVal != NULL) {
      *EncVal = MrValue;
    } else {
      MRC_DEBUG_MSG (Debug, MSG_LEVEL_ERROR, "%s %s\n", __FUNCTION__, gNullPtrErrStr);
      Status = mrcWrongInputParameter;
    }
  } else {
    MRC_DEBUG_MSG (Debug, MSG_LEVEL_ERROR, "%s Invalid Frequency Value: %d\n", gErrString, Frequency);
    Status = mrcWrongInputParameter;
  }

  return Status;
}

/**
  This function returns the Read Preamble Setting MR8 encoded value.

  @param[in]  MrcData   - Pointer to global MRC data.
  @param[out] EncVal    - Encoded Mode Register value.

  @retval MrcStatus - mrcSuccess if the timing is supported.  Else mrcWrongInputParameter.
**/
MrcStatus
MrcDdr5GetReadPreambleSetting (
  IN  MrcParameters  *MrcData,
  OUT DDR5_MR8_TRPRE *EncVal
  )
{
  MrcStatus      Status;
  MrcDebug       *Debug;
  UINT32         tRPRE;
  DDR5_MR8_TRPRE ReadPreambleSettings;

  Debug = &MrcData->Outputs.Debug;
  Status = mrcSuccess;

  tRPRE = MrcGetRpre (MrcData);
  if (tRPRE == 1) {
    ReadPreambleSettings = Ddr5tRPRE_1tCK_10;
  } else if (tRPRE == 2) {
    ReadPreambleSettings = Ddr5tRPRE_2tCK_1110;
  } else if (tRPRE == 3) {
    ReadPreambleSettings = Ddr5tRPRE_3tCK_000010;
  } else if (tRPRE == 4) {
    ReadPreambleSettings = Ddr5tRPRE_4tCK_00001010;
  } else {
    ReadPreambleSettings = 0;
    MRC_DEBUG_MSG (Debug, MSG_LEVEL_ERROR, "%s Invalid tRPRE %d\n", gErrString, tRPRE);
    Status = mrcWrongInputParameter;
  }

  if (Status == mrcSuccess && EncVal != NULL) {
    *EncVal = ReadPreambleSettings;
  }

  return Status;
}

/**
  This function returns the Write Preamble Setting MR8 encoded value.

  @param[in]  MrcData   - Pointer to global MRC data.
  @param[out] EncVal    - Encoded Mode Register value.

  @retval MrcStatus - mrcSuccess if the timing is supported.  Else mrcWrongInputParameter.
**/
MrcStatus
MrcDdr5GetWritePreambleSetting (
  IN  MrcParameters  *MrcData,
  OUT DDR5_MR8_TWPRE *EncVal
  )
{
  MrcStatus      Status;
  MrcDebug       *Debug;
  UINT32         tWPRE;
  DDR5_MR8_TWPRE WritePreambleSettings;

  Debug = &MrcData->Outputs.Debug;
  Status = mrcSuccess;

  tWPRE = MrcGetWpre (MrcData);
  if (tWPRE == 2) {
    WritePreambleSettings = Ddr5tWPRE_2tCK_0010;
  } else if (tWPRE == 3) {
    WritePreambleSettings = Ddr5tWPRE_3tCK_000010;
  } else if (tWPRE == 4) {
    WritePreambleSettings = Ddr5tWPRE_4tCK_00001010;
  } else {
    WritePreambleSettings = 0;
    MRC_DEBUG_MSG (Debug, MSG_LEVEL_ERROR, "%s Invalid tWPRE %d\n", gErrString, tWPRE);
    Status = mrcWrongInputParameter;
  }

  if (Status == mrcSuccess && EncVal != NULL) {
    *EncVal = WritePreambleSettings;
  }

  return Status;
}

/**
  This function returns an MR value or MPC command for the input MrRegNum.
  Some MR values (such as timing or vref MRs) will be different depending
  on the system configuration specified in the MrcData variable.

  @param[in]  MrcData      - Pointer to global MRC data.
  @param[in]  MrRegNum     - Requested MrRegiser
  @param[out] MrRegValOut  - The value for the requested MR Register
  @param[in]  Controller   - Index of the requested Controler
  @param[in]  Channel      - Index of the requested Channel

  @retval MrcStatus - mrcSuccess if the MrRegNum is supported. Else mrcWrongInputParameter.
**/
MrcStatus
Ddr5JedecInitVal (
  IN  MrcParameters *const MrcData,
  IN  MrcModeRegister      MrRegNum,
  OUT UINT16               *MrRegValOut,
  IN  UINT32               Controller,
  IN  UINT32               Channel
  )
{
  MrcStatus      Status;
  MrcInput       *Inputs;
  MrcOutput      *Outputs;
  MrcDebug       *Debug;
  MrcChannelOut  *ChannelOut;
  MrcTiming      *TimingPtr;
  UINT8          RetVal;
  UINT8          RlEnc;
  UINT8          WrEnc;
  UINT8          TrtpEnc;
  UINT8          TdllkEnc;
  UINT32         Profile;
  DDR5_MR8_TRPRE RpreVal;
  DDR5_MR8_TWPRE WpreVal;
  DDR5_MODE_REGISTER_0_TYPE *Mr0;
  DDR5_MODE_REGISTER_2_TYPE *Mr2;
  DDR5_MODE_REGISTER_4_TYPE *Mr4;
  DDR5_MODE_REGISTER_6_TYPE *Mr6;
  DDR5_MODE_REGISTER_8_TYPE *Mr8;
  DDR5_MODE_REGISTER_10_TYPE *Mr10;
  DDR5_MODE_REGISTER_11_TYPE *Mr11;
  DDR5_MODE_REGISTER_12_TYPE *Mr12;
  DDR5_MODE_REGISTER_13_TYPE *Mr13;
  DDR5_MODE_REGISTER_34_TYPE *Mr34;
  DDR5_MODE_REGISTER_35_TYPE *Mr35;
  DDR5_MODE_REGISTER_37_TYPE *Mr37;
  DDR5_MODE_REGISTER_38_TYPE *Mr38;
  DDR5_MODE_REGISTER_39_TYPE *Mr39;
  DDR5_MODE_REGISTER_45_TYPE *Mr45;

  Inputs     = &MrcData->Inputs;
  Outputs    = &MrcData->Outputs;
  Debug      = &Outputs->Debug;
  Profile    = Inputs->MemoryProfile;
  ChannelOut = &Outputs->Controller[Controller].Channel[Channel];
  TimingPtr  = &ChannelOut->Timing[Profile];
  Status     = mrcSuccess;
  RetVal     = 0;

  // We will only return one value for the input MR Num.
  Mr0  = (DDR5_MODE_REGISTER_0_TYPE *) &RetVal;
  Mr2  = (DDR5_MODE_REGISTER_2_TYPE *) &RetVal;
  Mr4  = (DDR5_MODE_REGISTER_4_TYPE *) &RetVal;
  Mr6  = (DDR5_MODE_REGISTER_6_TYPE *) &RetVal;
  Mr8  = (DDR5_MODE_REGISTER_8_TYPE *) &RetVal;
  Mr10 = (DDR5_MODE_REGISTER_10_TYPE *) &RetVal;
  Mr11 = (DDR5_MODE_REGISTER_11_TYPE *) &RetVal;
  Mr12 = (DDR5_MODE_REGISTER_12_TYPE *) &RetVal;
  Mr13 = (DDR5_MODE_REGISTER_13_TYPE *) &RetVal;
  Mr34 = (DDR5_MODE_REGISTER_34_TYPE *) &RetVal;
  Mr35 = (DDR5_MODE_REGISTER_35_TYPE *) &RetVal;
  Mr37 = (DDR5_MODE_REGISTER_37_TYPE *) &RetVal;
  Mr38 = (DDR5_MODE_REGISTER_38_TYPE *) &RetVal;
  Mr39 = (DDR5_MODE_REGISTER_39_TYPE *) &RetVal;
  Mr45 = (DDR5_MODE_REGISTER_45_TYPE *) &RetVal;

  switch (MrRegNum) {
    case mrMR0:
      // Mr0->Bits.BurstLength = 0 (BL16)
      if (EncodeReadLatencyDdr5 (MrcData, TimingPtr->tCL, &RlEnc) != mrcSuccess) {
        Status = mrcWrongInputParameter;
      }
      Mr0->Bits.CasLatency = RlEnc;
      break;

    case mrMR2:
      // Mr2->Bits.ReadPreambleTraining = 0 (Normal Mode)
      // Mr2->Bits.WriteLevelingTraining = 0 (Normal Mode)
      // Mr2->Bits.Mode1n = 0 (2N Mode) (Read Only)
      // Mr2->Bits.MaxPowerSavingsMode = 0 (Disable);
      Mr2->Bits.CsAssertionDuration = 0; // (Multiple cycles of CS assertion supported)
      // Mr2->Bits.Device15Mpsm = 0 (Disable)
      // Mr2->Bits.InternalWriteTiming = 0 (Disable)
      break;

    case mrMR3:
      // Mr3->Bits.WriteLevelingInternalCycleLowerByte = 0
      // Mr3->Bits.WriteLevelingInternalCycleUpperByte = 0
      break;

    case mrMR4:
      Mr4->Bits.RefreshRate = 0x2; // Read Only, setting to default value
      Mr4->Bits.RefreshTrfcMode = 1; // MC only supports Fine Granularity Refresh (FGR)
      // Mr4->Bits.Tuf = 0 [Read Only]
      break;

    case mrMR5:
      // Mr5->Bits.DataOutputDisable = 0; (Default)
      // Mr5->Bits.PullUpOutputDriverImpedance = 0; (RZQ/7)
      // Mr5->Bits.TdqsEnable = 0; (Default)
      // Mr5->Bits.DmEnable = 0; (Default)
      // Mr5->Bits.PullDownOutputDriverImpedance = 0; (RZQ/7)
      break;

    case mrMR6:
      if (EncodeWriteRecoveryDdr5 (MrcData, TimingPtr->tWR, &WrEnc) != mrcSuccess) {
        Status = mrcWrongInputParameter;
      }
      if (EncodeTrtpDdr5 (MrcData, TimingPtr->tRTP, &TrtpEnc) != mrcSuccess) {
        Status = mrcWrongInputParameter;
      }
      Mr6->Bits.WriteRecoveryTime = WrEnc;
      Mr6->Bits.tRTP = TrtpEnc;
      break;

    case mrMR8:
      if (MrcDdr5GetReadPreambleSetting (MrcData, &RpreVal) != mrcSuccess) {
        Status = mrcWrongInputParameter;
      }
      Mr8->Bits.ReadPreambleSettings = RpreVal;

      if (MrcDdr5GetWritePreambleSetting (MrcData, &WpreVal) != mrcSuccess) {
        Status = mrcWrongInputParameter;
      }
      Mr8->Bits.WritePreambleSettings = WpreVal;

      // Mr8->Bits.ReadPostambleSettings = 0;  // 0.5 tCK - 0 Pattern
      // Mr8->Bits.WritePostambleSettings = 0; // 0.5 tCK - 0 Pattern
      break;

    case mrMR10:
      Mr10->Bits.VrefDqCalibrationValue = Ddr5Vref_75p0; // 75% of VDDQ (Default)
      break;

    case mrMR11:
      // Read Only: Configured via VrefCA Command
      Mr11->Bits.VrefCaCalibrationValue = DDR5_VREFCA (Ddr5Vref_75p0); // 75% of VDDQ (Default)
      break;

    case mrMR12:
      // Read Only: Configured via VrefCS Command
      Mr12->Bits.VrefCsCalibrationValue = DDR5_VREFCS (Ddr5Vref_75p0); // 75% of VDDQ (Default)
      break;

    case mrMR13:
      if (EncodeDdr5TccdlTdllk (MrcData, Outputs->HighFrequency, &TdllkEnc) != mrcSuccess) {
        Status = mrcWrongInputParameter;
      }
      Mr13->Bits.tCCD_L_tDLLK = TdllkEnc;
      break;

    case mpcMR13:
      if (EncodeDdr5TccdlTdllk (MrcData, Outputs->HighFrequency, &TdllkEnc) != mrcSuccess) {
        Status = mrcWrongInputParameter;
      }
      RetVal = DDR5_MPC_CFG_TDLLK_TCCD_L (TdllkEnc);
      break;

    case mrMR37:
      Mr37->Bits.OdtlOnWrOffset = OdtlOnWrOffsetMinus3;
      Mr37->Bits.OdtlOffWrOffset = OdtlOffWrOffsetPlus2;
      break;

    case mrMR38:
      Mr38->Bits.OdtlOnWrNtOffset = OdtlOnWrOffsetMinus1;
      Mr38->Bits.OdtlOffWrNtOffset = OdtlOffWrOffset0;
      break;

    case mrMR39:
      Mr39->Bits.OdtlOnRdNtOffset = OdtlOnRdOffsetMinus1;
      Mr39->Bits.OdtlOffRdNtOffset = OdtlOffRdOffset0;
      break;

    case mrMR40:
      // Mr40->Bits.ReadDqsOffsetTiming = 0 (0 Clock - Default)
      break;

    case mrMR45:
      Mr45->Bits.DqsIntervalTimerRunTime = 4; // Stop after 64 clocks
      break;

    case mrMR48:
      //Mr48->Bits.WritePatternMode = 0 (default)
      break;

    case mpcMR32a0:
      // MR32 Group A RTT_CK
      RetVal = DDR5_MPC_GROUP_A_RTT_CK (CkCsCaOdt_RTT_OFF);
      break;

    case mpcMR32a1:
      // MR32 Group A RTT_CS
      RetVal = DDR5_MPC_GROUP_A_RTT_CS (CkCsCaOdt_RTT_OFF);
      break;

    case mpcMR32b0:
      // MR32 Group B RTT_CK
      RetVal = DDR5_MPC_GROUP_B_RTT_CK (CkCsCaOdt_RZQ_6_40);
      break;

    case mpcMR32b1:
      // MR32 Group B RTT_CS
      RetVal = DDR5_MPC_GROUP_B_RTT_CS (CkCsCaOdt_RZQ_6_40);
      break;

    case mpcMR33a0:
      // MR32 Group A RTT_CA
      RetVal = DDR5_MPC_GROUP_A_RTT_CA (CkCsCaOdt_RTT_OFF);
      break;

    case mpcMR33b0:
      // MR32 Group B RTT_CA
      RetVal = DDR5_MPC_GROUP_B_RTT_CA (CkCsCaOdt_RZQ_3_80);
      break;

    case mpcMR33:
      // MR33 DQS_RTT_PARK
      RetVal = DDR5_MPC_SET_DQS_RTT_PARK (Rtt_RTT_OFF);
      break;

    case mpcMR34:
      // MR34 RTT_PARK
      RetVal = DDR5_MPC_SET_RTT_PARK (Rtt_RTT_OFF);
      break;

    case mpcApplyVrefCa:
      // Apply Vrefca and RTT_CA/CS/CK
      RetVal = DDR5_MPC_APPLY_VREF_RTT;
      break;

    case mrMR34:
      // RTT_PARK must be configured via MPC command:
      // DDR5_MPC_SET_RTT_PARK
      // Mr34->Bits.RttPark = Rtt_RTT_OFF; // Default
      Mr34->Bits.RttWr = Rtt_RZQ_240; // Default
      break;

    case mrMR35:
      Mr35->Bits.RttNomWr = Rtt_RZQ_3_80;
      Mr35->Bits.RttNomRd = Rtt_RZQ_3_80;
      break;

    case mrMR36:
      // Mr36->Bits.RttLoopback = 0 (RTT_OFF Default)
      break;

    case mpcDllReset:
      RetVal = DDR5_MPC_DLL_RESET;
      break;

    case mpcZqCal:
      RetVal = DDR5_MPC_ZQCAL_START;
      break;

    case mpcZqLat:
      RetVal = DDR5_MPC_ZQCAL_LATCH;
      break;

    case mpcEnterCaTrainMode:
      RetVal = DDR5_MPC_ENTER_CA_TRAINING_MODE;
      break;

    case mpcSetCmdTiming:
      RetVal = (TimingPtr->NMode == 2) ? DDR5_MPC_SET_2N_COMMAND_TIMING : DDR5_MPC_SET_1N_COMMAND_TIMING;
      break;

    default:
      Status = mrcWrongInputParameter;
      break;
  }

  if (MrRegValOut != NULL) {
    *MrRegValOut = RetVal;
  } else {
    Status = mrcWrongInputParameter;
  }

  if (Status != mrcSuccess) {
    MRC_DEBUG_MSG (Debug, MSG_LEVEL_ERROR, "%s failed for MrRegNum = %d\n", __FUNCTION__, MrRegNum);
  }

  return Status;
}


/**
  This function returns the Generic MRS FSM Command Type and Delay Type associated with the
  input MrcModeRegister value.

  @param[in]  MrcData      - Pointer to global MRC data.
  @param[in]  MrRegNum     - Requested MrRegiser
  @param[out] CmdTypeOut   - The Generic MRS FSM Command Type to use for the input MrRegister
  @param[out] DelayTypeOut - The type of delay to use for the input MrRegister

  @retval MrcStatus - mrcSuccess if the MrRegNum is supported. Else mrcWrongInputParameter.
**/
MrcStatus
MrcDdr5GetGmfAttributes (
  IN  MrcParameters *const MrcData,
  IN  MrcModeRegister      MrRegNum,
  OUT GmfCmdType           *CmdTypeOut,  OPTIONAL
  OUT GmfDelayType         *DelayTypeOut OPTIONAL
  )
{
  MrcStatus     Status;
  MrcOutput     *Outputs;
  MrcDebug      *Debug;
  GmfCmdType    CmdType;
  GmfDelayType  DelayType;

  Outputs    = &MrcData->Outputs;
  Debug      = &Outputs->Debug;
  Status     = mrcSuccess;
  CmdType    = GmfCmdMrw;
  DelayType  = GmfDelay_tMOD;

  switch (MrRegNum) {
    case mrMR0:
    case mrMR2:
    case mrMR3:
    case mrMR4:
    case mrMR5:
    case mrMR6:
    case mrMR8:
    case mrMR10:
    case mrMR13:
    case mrMR37:
    case mrMR38:
    case mrMR39:
    case mrMR40:
    case mrMR45:
    case mrMR48:
    case mrMR34:
    case mrMR35:
    case mrMR36:
      CmdType    = GmfCmdMrw;
      DelayType  = GmfDelay_tMOD;
      break;

    case mrMR11:
    case mrMR12:
      CmdType   = GmfCmdVref;
      DelayType = GmfDelay_tVREFCA;
      break;

    case mpcApplyVrefCa:
    case mpcMR13:
    case mpcMR32a0:
    case mpcMR32a1:
    case mpcMR32b0:
    case mpcMR32b1:
    case mpcMR33a0:
    case mpcMR33b0:
    case mpcMR33:
    case mpcMR34:
    case mpcDllReset:
    case mpcSetCmdTiming:
      CmdType   = GmfCmdMpc;
      DelayType = GmfDelay_tMOD;
      break;

    case mpcZqCal:
      CmdType = GmfCmdMpc;
      DelayType = GmfDelay_tZQCAL;
      break;

    case mpcZqLat:
    case mpcEnterCaTrainMode:
      CmdType = GmfCmdMpc;
      DelayType = GmfDelay_tZQLAT;
      break;

    default:
      Status = mrcWrongInputParameter;
      break;
  }

  if (CmdTypeOut != NULL) {
    *CmdTypeOut = CmdType;
  }

  if (DelayTypeOut != NULL) {
    *DelayTypeOut = DelayType;
  }

  if (Status != mrcSuccess) {
    MRC_DEBUG_MSG (Debug, MSG_LEVEL_ERROR, "%s failed for MrRegNum = %d\n", __FUNCTION__, MrRegNum);
  }

  return Status;
}

/**
  This function returns the requested DelayType timing in nCK units.

  @param[in]  MrcData      - Pointer to global MRC data.
  @param[in]  DelayType    - Requested delay type
  @param[in]  tCK          - the memory clock period in Femotosecond units
  @param[out] TimingNckOut - Output variable for the requested delay timing in nCK units

  @retval MrcStatus - mrcSuccess if the DelayType is supported. Else mrcWrongInputParameter.
**/
MrcStatus
Ddr5GmfDelayTimings (
  IN  MrcParameters *const MrcData,
  IN  GmfDelayType         DelayType,
  IN  UINT32               tCK,
  OUT UINT32               *TimingNckOut
  )
{
  UINT32        TimingNck;

  if (TimingNckOut == NULL) {
    return mrcWrongInputParameter;
  }

  switch (DelayType)
  {
    case GmfDelay_tMOD:
      // DDR5 "tMRD"
      // max(14ns, 16nCK)
      TimingNck = tMODGet(MrcData, tCK);
      break;

    case GmfDelay_tZQCAL:
      TimingNck = tZQCALGet(MrcData, tCK);
      break;

    case GmfDelay_tZQLAT:
      TimingNck = tZQCSGet(MrcData, tCK);
      break;

    case GmfDelay_tDFE:
      // DDR5 DFE Mode Register Write Update Delay Time
      // Min 80ns
      TimingNck = DIVIDECEIL (80*1000*1000, tCK);
      break;

    case GmfDelay_tVREFCA:
      TimingNck = tMODGet(MrcData, tCK);
      break;

    default:
      return mrcWrongInputParameter;
      break;
  }

  *TimingNckOut = TimingNck;
  return mrcSuccess;
}

/**
  This function will setup the default MR values for DDR5 based on
  DRAM Timings and Frequency in MRC global data.
  Only populated Channels and Ranks are initialized.

  @param[in, out]  MrcData   -  Pointer to MRC global data.
  @param[in]       MrAddress -  mrEndOfSequence terminated array of MrcModeRegister addresses to initialize

  @retval MrcStatus - mrcSuccess if successful, else an error status.
**/
MrcStatus
InitMrwDdr5 (
  IN OUT MrcParameters *const MrcData,
  IN     const UINT16  *MrAddress
  )
{
  MrcOutput     *Outputs;
  MrcChannelOut *ChannelOut;
  UINT16        *MrPtr;
  UINT32        MrIndex;
  MrcStatus     Status;
  UINT32        Controller;
  UINT32        Channel;
  UINT32        Dimm;
  UINT32        DimmRank;
  UINT32        Rank;
  UINT32        Index;
  MrcModeRegister CurMrAddr;

  Outputs = &MrcData->Outputs;
  Status = mrcSuccess;

  for (Controller = 0; Controller < MAX_CONTROLLER; Controller++) {
    for (Channel = 0; Channel < Outputs->MaxChannels; Channel++) {
      for (Rank = 0; Rank < MAX_RANK_IN_CHANNEL; Rank++) {
        if (!MrcRankExist (MrcData, Controller, Channel, Rank)) {
          continue;
        }
        Dimm = RANK_TO_DIMM_NUMBER (Rank);
        DimmRank = Rank % MAX_RANK_IN_DIMM;
        for (Index = 0; MrAddress[Index] != mrEndOfSequence; Index++) {
          CurMrAddr = MrAddress[Index];
          ChannelOut = &Outputs->Controller[Controller].Channel[Channel];
          MrPtr = ChannelOut->Dimm[Dimm].Rank[DimmRank].MR;
          MrIndex = MrcMrAddrToIndex (MrcData, &CurMrAddr);
          // Fetch the init value
          Status = Ddr5JedecInitVal (MrcData, CurMrAddr, &MrPtr[MrIndex], Controller, Channel);
          if (Status != mrcSuccess) {
            return Status;
          }
        } // Index
      } // Rank
    } // Channel
  } // Controller

  return Status;
}


/**
  Perform JEDEC Init sequence for DDR5.

  @param[in] MrcData  - Pointer to MRC global data.
  @param[in] Sequence - Array of MrcModeRegister values defining the sequence
                        of MRW and MPC commands to perform. The entire array
                        is index until the sentinel value mrEndOfSequence
                        is reached.

  @retval MrcStatus
**/
MrcStatus
PerformGenericMrsFsmSequence (
  IN MrcParameters *const MrcData,
  IN const UINT16  *const Sequence
  )
{
  const MRC_FUNCTION  *MrcCall;
  MrcStatus     Status;
  MrcInput      *Inputs;
  MrcOutput     *Outputs;
  MrcChannelOut *ChannelOut;
  MrcDebug      *Debug;
  MrcTiming     *Timing;
  UINT32        Channel;
  UINT32        Controller;
  UINT32        Dimm;
  UINT32        DimmRank;
  UINT32        Rank;
  UINT32        CurDelay;
  UINT32        OutIdx;
  UINT8         Data;
  UINT32        MrIndex;
  UINT32        Index;
  UINT16        *MrPtr;
  GmfCmdType    CmdType;
  GmfDelayType  DelayType;
  MrcModeRegister CurMrAddr;
  MRC_GEN_MRS_FSM_MR_TYPE *GenMrsFsmMr;
  MRC_GEN_MRS_FSM_MR_TYPE MrData[MAX_CONTROLLER][MAX_CHANNEL][MAX_RANK_IN_CHANNEL][MAX_MR_GEN_FSM];

  Inputs  = &MrcData->Inputs;
  Outputs = &MrcData->Outputs;
  Debug   = &Outputs->Debug;
  MrcCall = MrcData->Inputs.Call.Func;
  Status  = mrcSuccess;

  if (!Outputs->RestoreMRs) {
    // Setup Init MR's to a default value, and update MrcData.
    Status = InitMrwDdr5 (MrcData, Sequence);
    if (Status != mrcSuccess) {
      return Status;
    }
  }

  // Clear out our array
  MrcCall->MrcSetMem ((UINT8 *) MrData, sizeof (MrData), 0);

  for (Controller = 0; Controller < MAX_CONTROLLER; Controller++) {
    for (Channel = 0; Channel < Outputs->MaxChannels; Channel++) {
      ChannelOut = &Outputs->Controller[Controller].Channel[Channel];
      Timing     = &ChannelOut->Timing[Inputs->MemoryProfile];
      for (Rank = 0; Rank < MAX_RANK_IN_CHANNEL; Rank++) {
        if (!MrcRankExist (MrcData, Controller, Channel, Rank)) {
          continue;
        }
        Dimm = RANK_TO_DIMM_NUMBER (Rank);
        DimmRank = Rank % MAX_RANK_IN_DIMM;
        MrPtr = ChannelOut->Dimm[Dimm].Rank[DimmRank].MR;
        for (Index = 0, OutIdx = 0; Sequence[Index] != mrEndOfSequence; Index++) {
          CurMrAddr = Sequence[Index];
          MrIndex = MrcMrAddrToIndex (MrcData, &CurMrAddr);
          if (MrIndex <  MAX_MR_IN_DIMM) {
            // Fetch the Command and Delay type
            Status = MrcDdr5GetGmfAttributes (MrcData, CurMrAddr, &CmdType, &DelayType);
            if (Status != mrcSuccess) {
              return Status;
            }

            // Fetch the delay value
            Status = Ddr5GmfDelayTimings (MrcData, DelayType, Timing->tCK, &CurDelay);
            if (Status != mrcSuccess) {
              return Status;
            }

            // Fetch the data
            Data = (UINT8) MrPtr[MrIndex];

            // Populate the Generic MRS FSM entry
            GenMrsFsmMr = &MrData[Controller][Channel][Rank][OutIdx];
            GenMrsFsmMr->MrData  = Data;
            if (CurMrAddr > MRC_UINT8_MAX) {
              // Special case for MPC commands
              GenMrsFsmMr->MrAddr = 0;
            } else {
              GenMrsFsmMr->MrAddr = (UINT8) CurMrAddr;
            }
            GenMrsFsmMr->Valid   = TRUE;
            GenMrsFsmMr->Delay   = (UINT16) CurDelay;
            GenMrsFsmMr->CmdType = CmdType;
          } else {
            MRC_DEBUG_MSG (Debug, MSG_LEVEL_ERROR, "MR index(%d) exceeded MR array length(%d)\n", MrIndex, MAX_MR_IN_DIMM);
            Status = mrcWrongInputParameter;
            return Status;
          }
          OutIdx++;
        } // Index
      } // Rank
    } // Channel
  } // Controller

  // Program Generic MRS FSM Per Controller/Channel
  Status = MrcGenMrsFsmConfig (MrcData, MrData);
  if (Status != mrcSuccess) {
    return Status;
  }

  // Run Generic FSM
  // Since we use the MRS FSM, we've configured it to send ZQ at the end if the sequence.
  // DDR5 ZQ is skipped if the sequence includes a ZQCAL MPC command.
  Status = MrcGenMrsFsmRun (MrcData);
  if (Status != mrcSuccess) {
    return Status;
  }


  return Status;
}

/**
  Perform JEDEC Init sequence for DDR5.

  @param[in] MrcData - Pointer to MRC global data.

  @retval MrcStatus
**/
MrcStatus
MrcJedecInitDdr5 (
  IN MrcParameters *const MrcData
  )
{
  static const UINT16 JedecInitSequence[] = {
    mpcSetCmdTiming, mpcMR32a0, mpcMR32a1, mpcMR33a0, mpcMR32b0, mpcMR32b1, mpcMR33b0,
    mrMR11, mrMR12, mpcMR33, mpcMR34, mrMR0,  mrMR2,  mrMR3,  mrMR4, mrMR5, mrMR6,
    mrMR8, mrMR10, mrMR13, mrMR34, mrMR35, mrMR36, mrMR37, mrMR38, mrMR39, mrMR45,
    mpcApplyVrefCa, mrEndOfSequence
    };
  return PerformGenericMrsFsmSequence (MrcData, JedecInitSequence);
}

/**
  Perform JEDEC reset sequence for DDR5.

  @param[in] MrcData - Include all MRC global data.

  @retval - mrcSuccess if no errors encountered, mrcFail otherwise
**/
MrcStatus
MrcJedecResetDdr5 (
  IN MrcParameters *const MrcData
  )
{
  MrcStatus         Status;
  MrcInput          *Inputs;
  MrcOutput         *Outputs;
  MRC_FUNCTION      *MrcCall;
  MrcTiming         *Timing;
  UINT32            PciEBar;
  INT64             GetSetVal;
  UINT32            tInit1;
  UINT32            tInit3;
  UINT32            tInit4;
  UINT32            tInit5;
  UINT32            CsOverrideDelay;
  UINT32            tXS;
  UINT32            tXPR;
  UINT32            FirstController;
  UINT8             FirstChannel;
  MrcProfile        Profile;
  static const UINT16 JedecResetExitSequence[] = {
    mpcMR13,
    mpcDllReset,
    mpcZqCal,
    mpcZqLat,
    mrEndOfSequence
    };

  Inputs            = &MrcData->Inputs;
  Outputs           = &MrcData->Outputs;
  MrcCall           = Inputs->Call.Func;
  PciEBar           = Inputs->PciEBaseAddress;
  FirstController   = (MrcControllerExist(MrcData, cCONTROLLER0)) ? 0 : 1;
  FirstChannel      = MrcData->Outputs.Controller[FirstController].FirstPopCh;
  Profile           = Inputs->MemoryProfile;
  Timing            = &Outputs->Controller[FirstController].Channel[FirstChannel].Timing[Profile];
  tXS               = tXSRGet (MrcData, Timing);
  Status            = mrcSuccess;

  // Init Timings in Nanoseconds
  tInit1 = MRC_DDR5_tINIT1_US * MRC_TIMER_1US;
  tInit3 = MRC_DDR5_tINIT3_MS * MRC_TIMER_1MS;
  tInit4 = MRC_DDR5_tINIT4_US * MRC_TIMER_1US;
  // Convert nCK to FS
  tInit5 = (MRC_DDR5_tINIT5_NCK * Outputs->MemoryClock);
  tInit5 = DIVIDECEIL (tInit5, FEMTOSECONDS_PER_NANOSECOND);
  tXPR   = (tXS * Outputs->MemoryClock);
  tXPR   = DIVIDECEIL (tXPR, FEMTOSECONDS_PER_NANOSECOND);
  CsOverrideDelay = (3 * Outputs->MemoryClock); //3tCK
  CsOverrideDelay = DIVIDECEIL (CsOverrideDelay, FEMTOSECONDS_PER_NANOSECOND);

  // Enable Multicycle CS on the MC which is the DRAM's default MPC mode after DRAM Reset
  EnableMcMulticycleCs (MrcData);

  // Force CKE_ON before DRAM Reset
  // Even though there is no CKE pin on DDR5, CKE override is used
  // to prevent power down entry in all technologies.
  MrcCkeOnProgramming (MrcData);
  GetSetVal = MC0_CH0_CR_REUT_CH_MISC_CKE_CTRL_CKE_Override_MAX;
  MrcGetSetMcCh (MrcData, MAX_CONTROLLER, MAX_CHANNEL, GsmMccCkeOverride, WriteNoCache, &GetSetVal);

  // CA Tristate is diabled in MrcMcConfiguration
  // Disabling CA forces CA[4:0] high when the CA bus is inactive.
  // This allows us to issue multiple cycles of NOP commands by simply
  // driving CS_N low using OverrideCs();

  // Check if this is the first time we are performing JEDEC RESET
  if (Outputs->JedecInitDone){
    // Reset Initialization with Stable Power:
    // Only need to assert RESET for 1us (tPW_RESET)
    tInit1 = MRC_TIMER_1US;
  }

  // Override CS_N to go LOW
  OverrideCs (MrcData, 0, TRUE);

  // Assert DIMM reset signal
  MrcCall->MrcDramReset (PciEBar, 0);

  // After tINIT1, bring Dram out of Reset
  MrcWait (MrcData, tInit1);
  MrcCall->MrcDramReset (PciEBar, 1);

  // After tINIT3, override CS_N to go HIGH
  MrcWait (MrcData, tInit3);
  OverrideCs (MrcData, 1, TRUE);

  // Wait for tINIT4 time for CS_N to remain HIGH
  MrcWait (MrcData, tInit4);

  // At tINIT4, issue NOP command by driving CS_N low for tINIT5
  OverrideCs (MrcData, 0, TRUE);
  MrcWait (MrcData, tInit5);

  // After tINIT5, override CS_N to go HIGH
  OverrideCs (MrcData, 1, TRUE);

  // Disabling the CS Override causes CA[5:13] to go low.
  // Wait at least 3tCK after setting CS_N high
  // to avoid CA transitioning during the NOP.
  MrcWait (MrcData, CsOverrideDelay);
  OverrideCs (MrcData, 1, FALSE);

  // Step8: Wait for tXPR timings before sending any command
  MrcWait (MrcData, tXPR);

  if (PerformGenericMrsFsmSequence (MrcData, JedecResetExitSequence) != mrcSuccess) {
    Status = mrcFail;
  }

  return Status;
}

/**
  This function issues the PDA Enumeration ID MPC. This will first send a MPC to enter PDA Enumeration Mode.
  It will then iterate through all DRAM devices and send an MPC to assign it an ID number, starting at 0.
  Once the iteration is done, a MPC will be sent to exit PDA Mode. This will wait tMPC Delay between PDA Entry
  and the first PDA Enumerate ID MPC's. This will wait tPDA_DELAY between the PDA Enumerate ID MPC's.

  @param[in]  MrcData    - Pointer to global MRC data
  @retval MrcStatus - mrcTimeout if the FSM does not complete after 1s
  @retval MrcStatus - mrcSuccess if the MPC is sent successfuly
  @retval MrcStatus - mrcFail for unexepected failures
**/
MrcStatus
MrcPdaEnumeration (
  IN MrcParameters *const MrcData
)
{
  MrcStatus     Status;
  MrcOutput     *Outputs;
  MrcInput      *Inputs;
  MrcChannelOut *ChannelOut;
  MrcDimmOut    *DimmOut;
  MrcDebug      *Debug;
  MrcTiming     *Timing;
  INT64         GetSetVal;
  INT64         tCWL4TxDqFifoWrEnSave;
  INT64         tCWL4TxDqFifoRdEnSave;
  INT64         tCWL4TxDqFifoWrEn;
  INT64         tCWL4TxDqFifoRdEn;
  INT64         TcwlSave;
  INT64         TcwlDecSave;
  INT64         TxDqsSave;
  INT64         OdtParkModeSave;
  UINT32        tMpcNck;
  UINT32        tMpcNckFs;
  UINT32        tMpcNs;
  UINT32        DecTcwl;
  UINT32        Controller;
  UINT32        Channel;
  UINT32        CpuByte;
  UINT32        CpuBytex16;
  UINT32        TcwlCalc;
  UINT32        FirstController;
  UINT32        TckPs;
  UINT32        Gear2;
  UINT32        BLns;
  UINT32        tpda_delay;
  UINT8         Rank;
  UINT8         NumDevices;
  UINT8         Device;
  UINT8         DimmIdx;
  UINT8         FirstChannel;
  BOOLEAN       Ddr5;

  Outputs           = &MrcData->Outputs;
  Inputs            = &MrcData->Inputs;
  Status            = mrcSuccess;
  Debug             = &Outputs->Debug;
  FirstController   = Outputs->FirstPopController;
  FirstChannel      = Outputs->Controller[FirstController].FirstPopCh;
  Ddr5              = (Outputs->DdrType == MRC_DDR_TYPE_DDR5);
  TckPs             = Outputs->tCKps;
  Gear2             = Outputs->Gear2 ? 1 : 0; // Used to add extra clock(s) in Gear2 mode
  BLns              = DIVIDECEIL ((TckPs * Outputs->BurstLength), 1000);
  CpuBytex16        = 0;
  // Only run PDA Enumeration on DDR5
  if (!Ddr5) {
    MRC_DEBUG_MSG (Debug, MSG_LEVEL_NOTE, "Skip PDA Enumeration, only supported for DDR5 \n");
    return Status;
  }

  // Restore ODTParkMode
  MrcGetSetChStrb(MrcData, FirstController, FirstChannel, 0, GsmIocDataDqsOdtParkMode, ReadFromCache | PrintValue, &OdtParkModeSave);
  // Drive DQS low
  GetSetVal = 1;
  MrcGetSetChStrb (MrcData, FirstController, FirstChannel, 0, GsmIocDataDqsOdtParkMode, WriteToCache, &GetSetVal);

  // Set DQ Pins HIGH, to a known state
  GetSetVal = 0xFF;
  MrcGetSetChStrb (MrcData, MAX_CONTROLLER, MAX_CHANNEL, MAX_SDRAM_IN_DIMM, GsmIocDqOverrideData, WriteToCache, &GetSetVal);
  MrcGetSetChStrb (MrcData, MAX_CONTROLLER, MAX_CHANNEL, MAX_SDRAM_IN_DIMM, GsmIocDqOverrideEn, WriteToCache, &GetSetVal);

  // Save all tCWL settings
  MrcGetSetMcCh (MrcData, FirstController, FirstChannel, GsmMctCWL, ReadFromCache | PrintValue, &TcwlSave);
  MrcGetSetMcCh (MrcData, FirstController, FirstChannel, GsmMctCWLDec, ReadFromCache | PrintValue, &TcwlDecSave);
  MrcGetSetMcCh (MrcData, FirstController, FirstChannel, TxDqFifoWrEnTcwlDelay, ReadFromCache | PrintValue, &tCWL4TxDqFifoWrEnSave);
  MrcGetSetMcCh (MrcData, FirstController, FirstChannel, TxDqFifoRdEnTcwlDelay, ReadFromCache | PrintValue, &tCWL4TxDqFifoRdEnSave);
  MrcGetSetStrobe ( MrcData, FirstController, FirstChannel, 0, 0, TxDqsDelay, ReadFromCache | PrintValue, &TxDqsSave);

  // Set custom tCWL settings for PDA Enumeration
  DecTcwl = 0;
  GetSetVal = 0;
  MrcGetSetMcCh (MrcData, FirstController, FirstChannel, GsmMctCWLDec, WriteToCache | PrintValue, &GetSetVal);
  MrcGetSetStrobe (MrcData, MAX_CONTROLLER, MAX_CHANNEL, MAX_RANK_IN_CHANNEL, MAX_SDRAM_IN_DIMM, TxDqsDelay, WriteToCache | PrintValue, &GetSetVal);

  // tCWL = 11.5ns - This is the middle of the tPDA_DQS_Delay range
  TcwlCalc = DIVIDECEIL (11500, TckPs);

  // Add 3ns to adjust for internal delays
  TcwlCalc += DIVIDECEIL (3000, TckPs);

  if (Gear2) {
    tCWL4TxDqFifoWrEn = TcwlCalc - (2 * DecTcwl) - 3 + (TcwlCalc % 2);
  } else {
    tCWL4TxDqFifoWrEn  = TcwlCalc - DecTcwl - 2; // TxDqFifoWrEnTcwlDelay(DClk)
  }
  tCWL4TxDqFifoRdEn  = tCWL4TxDqFifoWrEn - 1; // TxDqFifoRdEnTcwlDelay(DCLK)
  tCWL4TxDqFifoRdEn -= 3;

  GetSetVal = TcwlCalc;
  MrcGetSetMcCh (MrcData, FirstController, FirstChannel, GsmMctCWL, WriteToCache | PrintValue, &GetSetVal);
  MrcGetSetMcCh (MrcData, MAX_CONTROLLER, MAX_CHANNEL, TxDqFifoWrEnTcwlDelay, WriteToCache | PrintValue, &tCWL4TxDqFifoWrEn);
  MrcGetSetMcCh (MrcData, MAX_CONTROLLER, MAX_CHANNEL, TxDqFifoRdEnTcwlDelay, WriteToCache | PrintValue, &tCWL4TxDqFifoRdEn);

  // Force ON TX in the Data bytes
  GetSetVal = 1;
  MrcGetSetChStrb (MrcData, MAX_CONTROLLER, MAX_CHANNEL, MAX_SDRAM_IN_DIMM, GsmIocTxOn, WriteToCache, &GetSetVal);
  MrcGetSetChStrb (MrcData, MAX_CONTROLLER, MAX_CHANNEL, MAX_SDRAM_IN_DIMM, GsmIocTxPiPwrDnDis, WriteToCache, &GetSetVal);

  MrcFlushRegisterCachedData (MrcData);

  for (Controller = 0; Controller < MAX_CONTROLLER; Controller++) {
    for (Channel = 0; Channel < Outputs->MaxChannels; Channel++) {
      for (Rank = 0; Rank < MAX_RANK_IN_CHANNEL; Rank++) {
        if (!MrcRankExist (MrcData, Controller, Channel, Rank)) {
          continue;
        }
        ChannelOut = &Outputs->Controller[Controller].Channel[Channel];
        DimmIdx = RANK_TO_DIMM_NUMBER (Rank);
        DimmOut = &ChannelOut->Dimm[DimmIdx];
        Timing = &ChannelOut->Timing[Inputs->MemoryProfile];

        tMpcNck = tMODGet (MrcData, Timing->tCK);
        tMpcNckFs = tMpcNck * Outputs->MemoryClock;
        tMpcNs = DIVIDECEIL (tMpcNckFs, FEMTOSECONDS_PER_NANOSECOND);

        // Channel Width (ddr5 = 32 Bytes) / SdramWidth (x8 or x16)
        NumDevices = DimmOut->PrimaryBusWidth / DimmOut->SdramWidth;
        if ((DimmOut->SdramWidth == 8) && DimmOut->EccSupport) {
          NumDevices += 1;
        }

        // Enter PDA Enumerate ID
        Status = MrcIssueMpc (MrcData, Controller, Channel, Rank, DDR5_MPC_ENTER_PDA_ENUM_PROG_MODE, TRUE);
        // Wait tMPC Delay
        MrcWait (MrcData, tMpcNs);

        // Set up each device
        for (Device = 0; Device < NumDevices; Device++) {
          CpuByte = Device * ((DimmOut->SdramWidth == 16) ? 2 : 1);

          // Set DQ Pins LOW
          GetSetVal = 0x00;
          MrcGetSetChStrb (MrcData, Controller, Channel, CpuByte, GsmIocDqOverrideData, WriteToCache, &GetSetVal);
          GetSetVal = 0xFF;   // Send all the bits
          MrcGetSetChStrb (MrcData, Controller, Channel, CpuByte, GsmIocDqOverrideEn, WriteToCache, &GetSetVal);

          if (DimmOut->SdramWidth == 16) {
            CpuBytex16 = CpuByte + 1;
            // Set DQ Pins LOW
            GetSetVal = 0x00;
            MrcGetSetChStrb (MrcData, Controller, Channel, CpuBytex16, GsmIocDqOverrideData, WriteToCache, &GetSetVal);
            GetSetVal = 0xFF;   // Send all the bits
            MrcGetSetChStrb (MrcData, Controller, Channel, CpuBytex16, GsmIocDqOverrideEn, WriteToCache, &GetSetVal);
          }
          MrcFlushRegisterCachedData (MrcData);

          // Send PDA Enumerate ID
          Status = MrcIssueMpc (MrcData, Controller, Channel, Rank, DDR5_MPC_PDA_ENUMERATE_ID (Device), TRUE);

          GetSetVal = 0xFF;
          // Set DQ Pins back to HIGH to a known state
          MrcGetSetChStrb (MrcData, Controller, Channel, CpuByte, GsmIocDqOverrideData, WriteToCache, &GetSetVal);
          MrcGetSetChStrb (MrcData, Controller, Channel, CpuByte, GsmIocDqOverrideEn, WriteToCache, &GetSetVal);

          if (DimmOut->SdramWidth == 16) {
            // Set DQ Pins back to HIGH to a known state
            GetSetVal = 0xFF;
            MrcGetSetChStrb (MrcData, Controller, Channel, CpuBytex16, GsmIocDqOverrideData, WriteToCache, &GetSetVal);
            MrcGetSetChStrb (MrcData, Controller, Channel, CpuBytex16, GsmIocDqOverrideEn, WriteToCache, &GetSetVal);
          }
          MrcFlushRegisterCachedData (MrcData);

          // wait tpda_delay = tPDA_DQS_Delay_max + BL / 2 + 19ns
          tpda_delay = 18 + BLns + 19;
          MrcWait (MrcData, tpda_delay * MRC_TIMER_1NS);
        }
        //Exit PDA Enumerate ID
        Status = MrcIssueMpc (MrcData, Controller, Channel, Rank, DDR5_MPC_EXIT_PDA_ENUM_PROG_MODE, TRUE);
        // Wait tMPC Delay
        MrcWait (MrcData, tMpcNs);
      }
    }
  }
  // Disable DQ override
  GetSetVal = 0x00;
  MrcGetSetChStrb (MrcData, MAX_CONTROLLER, MAX_CHANNEL, MAX_SDRAM_IN_DIMM, GsmIocDqOverrideEn, WriteToCache, &GetSetVal);

  // Restore ODTParkMode
  MrcGetSetChStrb (MrcData, MAX_CONTROLLER, MAX_CHANNEL, 0, GsmIocDataDqsOdtParkMode, WriteToCache, &OdtParkModeSave);

  // Restore all tCWL settings
  MrcGetSetMcCh (MrcData, MAX_CONTROLLER, MAX_CHANNEL, GsmMctCWL, WriteToCache | PrintValue, &TcwlSave);
  MrcGetSetMcCh (MrcData, MAX_CONTROLLER, MAX_CHANNEL, GsmMctCWLDec, WriteToCache | PrintValue, &TcwlDecSave);
  MrcGetSetMcCh (MrcData, MAX_CONTROLLER, MAX_CHANNEL, TxDqFifoWrEnTcwlDelay, WriteToCache | PrintValue, &tCWL4TxDqFifoWrEnSave);
  MrcGetSetMcCh (MrcData, MAX_CONTROLLER, MAX_CHANNEL, TxDqFifoRdEnTcwlDelay, WriteToCache | PrintValue, &tCWL4TxDqFifoRdEnSave);
  MrcGetSetStrobe (MrcData, MAX_CONTROLLER, MAX_CHANNEL, MAX_RANK_IN_CHANNEL, MAX_SDRAM_IN_DIMM, TxDqsDelay, WriteToCache | PrintValue, &TxDqsSave);

  GetSetVal = 0;
  MrcGetSetChStrb (MrcData, MAX_CONTROLLER, MAX_CHANNEL, MAX_SDRAM_IN_DIMM, GsmIocTxOn, WriteToCache, &GetSetVal);
  MrcGetSetChStrb (MrcData, MAX_CONTROLLER, MAX_CHANNEL, MAX_SDRAM_IN_DIMM, GsmIocTxPiPwrDnDis, WriteToCache, &GetSetVal);

  MrcFlushRegisterCachedData (MrcData);
  return Status;
}

/**
  Enable/Disable DDR5 Write Leveling mode on DRAM

  @param[in] MrcData     - Include all MRC global data.
  @param[in] WriteLvMode - Write leveling mode to enable (Internal/External)
                          Ddr5ExternalWrLvMode:
                            MR2[1]: Write Leveling = 1
                            MR2[7]: Internal Write Timing = 0
                          Ddr5InternalWrLvMode:
                            MR2[1]: Write Leveling = 1
                            MR2[7]: Internal Write Timing = 1
                          Ddr5DisableWrLvMode (Normal Mode):
                            MR2[1]: Write Leveling = 0
                            MR2[7]: Internal Write Timing = 1

  @retval - mrcSuccess if no errors encountered, mrcFail otherwise
**/
MrcStatus
MrcDdr5SetDramWrLvMode (
  IN MrcParameters  *const MrcData,
  IN Ddr5WrLvlMode   const  WrLevelingMode
  )

{
  UINT32                      Controller;
  UINT32                      Channel;
  UINT8                       Rank;
  UINT8                       RankMask;
  UINT8                       RankHalf;
  UINT8                       RankMod2;
  UINT8                       ValidRankMask;
  MrcStatus                   Status;
  MrcStatus                   MrsStatus;
  MrcChannelOut               *ChannelOut;
  MrcOutput                   *Outputs;
  DDR5_MODE_REGISTER_2_TYPE   MR2;

  Outputs       = &MrcData->Outputs;
  ValidRankMask = Outputs->ValidRankMask;
  Status        = mrcSuccess;


  for (Rank = 0; Rank < MAX_RANK_IN_CHANNEL; Rank++) {
    RankMask = (1 << Rank);
    if (!(RankMask & ValidRankMask)) {
      // Skip if all channels empty
      continue;
    }
    RankHalf      = Rank / MAX_RANK_IN_DIMM;
    RankMod2      = Rank % MAX_RANK_IN_DIMM;

    for (Controller = 0; Controller < MAX_CONTROLLER; Controller++) {
      for (Channel = 0; Channel < MAX_CHANNEL; Channel++) {
        if (!(MrcRankExist (MrcData, Controller, Channel, Rank))) {
          continue;
        }
        ChannelOut = &Outputs->Controller[Controller].Channel[Channel];
        MR2.Data8 = (UINT8) ChannelOut->Dimm[RankHalf].Rank[RankMod2].MR[mrMR2];
        MR2.Bits.WriteLevelingTraining = (WrLevelingMode == Ddr5WrLvlModeDisable) ? 0 : 1;
        MR2.Bits.InternalWriteTiming   = (WrLevelingMode == Ddr5WrLvlModeExternal) ? 0 : 1;
        MrsStatus = MrcWriteMRS (MrcData, Controller, Channel, 1 << Rank, mrMR2, (UINT16) MR2.Data8);
        if (MrsStatus != mrcSuccess) {
          Status |= MrsStatus;
        }
        ChannelOut->Dimm[RankHalf].Rank[RankMod2].MR[mrMR2] = MR2.Data8;
      }  // for Channel
    } // Controller
  }

  return Status;
}

/**
  Set CAS latency on on All DRAMs in the system

  @param[in] MrcData     - Include all MRC global data.
  @param[in] CasLatency  - CAS latency value:
                            00000B: 22
                            00001B: 24
                            00010B: 26
                            00011B: 28
                            ...
                            10011B: 60
                            10100B: 62
                            10101B: 64
                            10110B: 66
                            All other encodings reserved.

  @retval - mrcSuccess if no errors encountered, mrcFail otherwise
**/
MrcStatus
MrcDdr5SetCasLatency (
  IN MrcParameters  *const MrcData,
  UINT8                    CasLatency
  )
{
  UINT32                      Controller;
  UINT32                      Channel;
  UINT8                       Rank;
  UINT8                       RankHalf;
  UINT8                       RankMod2;
  UINT8                       ValidRankMask;
  UINT8                       RankMask;
  MrcStatus                   Status;
  MrcStatus                   MrsStatus;
  MrcChannelOut               *ChannelOut;
  MrcOutput                   *Outputs;
  DDR5_MODE_REGISTER_0_TYPE   MR0;

  Outputs       = &MrcData->Outputs;
  ValidRankMask = Outputs->ValidRankMask;
  Status        = mrcSuccess;


  for (Rank = 0; Rank < MAX_RANK_IN_CHANNEL; Rank++) {
    RankMask = (1 << Rank);
    if (!(RankMask & ValidRankMask)) {
      // Skip if all channels empty
      continue;
    }

    RankHalf      = Rank / MAX_RANK_IN_DIMM;
    RankMod2      = Rank % MAX_RANK_IN_DIMM;

    for (Controller = 0; Controller < MAX_CONTROLLER; Controller++) {
      for (Channel = 0; Channel < MAX_CHANNEL; Channel++) {
        if (!(MrcRankExist (MrcData, Controller, Channel, Rank))) {
          continue;
        }
        ChannelOut = &Outputs->Controller[Controller].Channel[Channel];
        MR0.Data8 = (UINT8) ChannelOut->Dimm[RankHalf].Rank[RankMod2].MR[mrMR0];
        MR0.Bits.CasLatency = CasLatency;
        MrsStatus = MrcWriteMRS (MrcData, Controller, Channel, 1 << Rank, mrMR0, (UINT16) MR0.Data8);
        if (MrsStatus != mrcSuccess) {
          Status |= MrsStatus;
        }
      }  // for Channel
    } // Controller
  }

  return Status;
}

/**
  This function sets Write Leveling Internal Cycle delay through MR3 commands
  to the provided controller/channels

  @param[in] MrcData          - Include all MRC global data.
  @param[in] Controller       - Index of the requested Controler
  @param[in] Channel          - Index of the requested Channel
  @param[in] Rank             - Current working rank
  @param[in] IntCycleAlignLow - Write Leveling Internal Cycle Alignment (Lower Byte)
  @param[in] IntCycleAlignUp  - Write Leveling Internal Cycle Alignment (Upper Byte)
                                0000B: Disable (Default)
                                0001B: -1 tCK
                                0010B: -2 tCK
                                0011B: -3 tCK
                                0100B: -4 tCK
                                0101B: -5 tCK
                                0110B: -6 tCK
                                0111B: -7 tCK
                                1000B~ 1111B: RFU

  @retval - mrcSuccess if no errors encountered, mrcFail otherwise
**/
MrcStatus
MrcSetMR3_DDR5 (
  IN MrcParameters  *const MrcData,
  IN UINT32                Controller,
  IN UINT32                Channel,
  IN UINT32                Rank,
  IN UINT8                 IntCycleAlignLow,
  IN UINT8                 IntCycleAlignUp
  )
{
  UINT32                      ControllerStart;
  UINT32                      ControllerEnd;
  UINT32                      ControllerLoop;
  UINT32                      ChannelStart;
  UINT32                      ChannelEnd;
  UINT32                      ChannelLoop;
  UINT32                      RankHalf;
  UINT32                      RankMod2;
  MrcStatus                   Status;
  MrcStatus                   MrsStatus;
  MrcChannelOut               *ChannelOut;
  MrcOutput                   *Outputs;
  DDR5_MODE_REGISTER_3_TYPE   MR3;

  Outputs       = &MrcData->Outputs;
  RankHalf      = Rank / MAX_RANK_IN_DIMM;
  RankMod2      = Rank % MAX_RANK_IN_DIMM;
  Status        = mrcSuccess;

  if (Controller >= MAX_CONTROLLER) {
    ControllerStart = 0;
    ControllerEnd = MAX_CONTROLLER;
  } else {
    ControllerStart = Controller;
    ControllerEnd = Controller + 1;
  }
  if (Channel >= Outputs->MaxChannels) {
    ChannelStart = 0;
    ChannelEnd = (UINT32) Outputs->MaxChannels;
  } else {
    ChannelStart = Channel;
    ChannelEnd = Channel + 1;
  }

  for (ControllerLoop = ControllerStart; ControllerLoop < ControllerEnd; ControllerLoop++) {
    for (ChannelLoop = ChannelStart; ChannelLoop < ChannelEnd; ChannelLoop++) {
      if (!(MrcRankExist (MrcData, ControllerLoop, ChannelLoop, Rank))) {
        continue;
      }
      ChannelOut = &Outputs->Controller[ControllerLoop].Channel[ChannelLoop];
      MR3.Data8 = (UINT8) ChannelOut->Dimm[RankHalf].Rank[RankMod2].MR[mrMR3];
      // Lower Byte WL Internal Cycle Alignment is intended for x4, x8, and x16 configurations.
      // Upper Byte WL Internal Cycle Alignment is intended for x16 configuration only.
      MR3.Bits.WriteLevelingInternalCycleLowerByte = IntCycleAlignLow;
      MR3.Bits.WriteLevelingInternalCycleUpperByte = IntCycleAlignUp;
      MrsStatus = MrcWriteMRS (MrcData, ControllerLoop, ChannelLoop, 1 << Rank, mrMR3, (UINT16) MR3.Data8);
      if (MrsStatus != mrcSuccess) {
        Status |= MrsStatus;
      }
      ChannelOut->Dimm[RankHalf].Rank[RankMod2].MR[mrMR3] = MR3.Data8;
    }  // for ChannelLoop
  } // ControllerLoop

  return Status;
}
