/** @file
  Definition of DDR5 Specific functions, constants, defines,
  and data structures.

@copyright
  INTEL CONFIDENTIAL
  Copyright 2019 - 2020 Intel Corporation.

  The source code contained or described herein and all documents related to the
  source code ("Material") are owned by Intel Corporation or its suppliers or
  licensors. Title to the Material remains with Intel Corporation or its suppliers
  and licensors. The Material may contain trade secrets and proprietary and
  confidential information of Intel Corporation and its suppliers and licensors,
  and is protected by worldwide copyright and trade secret laws and treaty
  provisions. No part of the Material may be used, copied, reproduced, modified,
  published, uploaded, posted, transmitted, distributed, or disclosed in any way
  without Intel's prior express written permission.

  No license under any patent, copyright, trade secret or other intellectual
  property right is granted to or conferred upon you by disclosure or delivery
  of the Materials, either expressly, by implication, inducement, estoppel or
  otherwise. Any license under such intellectual property rights must be
  express and approved by Intel in writing.

  Unless otherwise agreed by Intel in writing, you may not remove or alter
  this notice or any other notice embedded in Materials by Intel or
  Intel's suppliers or licensors in any way.

  This file contains an 'Intel Peripheral Driver' and is uniquely identified as
  "Intel Reference Module" and is licensed for Intel CPUs and chipsets under
  the terms of your license agreement with Intel or your vendor. This file may
  be modified by the user, subject to additional terms of the license agreement.

@par Specification Reference:
  DDR5 JEDEC Spec
**/
#ifndef _MRC_DDR_5_H_
#define _MRC_DDR_5_H_

#include "MrcTypes.h"
#include "MrcInterface.h"

// DDR5 MPC Command definitions
#define DDR5_MPC_EXIT_CS_TRAINING_MODE    0x00
#define DDR5_MPC_ENTER_CS_TRAINING_MODE   0x01
#define DDR5_MPC_DLL_RESET                0x02
#define DDR5_MPC_ENTER_CA_TRAINING_MODE   0x03
#define DDR5_MPC_ZQCAL_LATCH              0x04
#define DDR5_MPC_ZQCAL_START              0x05
#define DDR5_MPC_STOP_DQS_INTERVAL_OSC    0x06
#define DDR5_MPC_START_DQS_INTERVAL_OSC   0x07
#define DDR5_MPC_SET_2N_COMMAND_TIMING    0x08
#define DDR5_MPC_SET_1N_COMMAND_TIMING    0x09
#define DDR5_MPC_EXIT_PDA_ENUM_PROG_MODE  0x0A
#define DDR5_MPC_ENTER_PDA_ENUM_PROG_MODE 0x0B
#define DDR5_MPC_MANUAL_ECS_OPERATION     0x0C
// 0x0D - 0x1E = Reserved for Future Use
#define DDR5_MPC_APPLY_VREF_RTT           0x1F
// MPC Masks
// These MPC commands must be masked with the data
// or ID codes to be used. Refer to the DDR5 spec
// MPC Opcodes section.

// MPC commands using OP[2:0] to specify MR value
#define DDR5_MPC_GROUP_A_RTT_CK(val)      (0x20 | (0x7 & (val)))
#define DDR5_MPC_GROUP_B_RTT_CK(val)      (0x28 | (0x7 & (val)))
#define DDR5_MPC_GROUP_A_RTT_CS(val)      (0x30 | (0x7 & (val)))
#define DDR5_MPC_GROUP_B_RTT_CS(val)      (0x38 | (0x7 & (val)))
#define DDR5_MPC_GROUP_A_RTT_CA(val)      (0x40 | (0x7 & (val)))
#define DDR5_MPC_GROUP_B_RTT_CA(val)      (0x48 | (0x7 & (val)))
#define DDR5_MPC_SET_DQS_RTT_PARK(val)    (0x50 | (0x7 & (val)))
#define DDR5_MPC_SET_RTT_PARK(val)        (0x58 | (0x7 & (val)))
// MPC commands using OP[3:0] to specify ID value
#define DDR5_MPC_PDA_ENUMERATE_ID(val)    (0x60 | (0xF & (val)))
#define DDR5_MPC_PDA_SELECT_ID(val)       (0x70 | (0xF & (val)))
#define DDR5_MPC_CFG_TDLLK_TCCD_L(val)    (0x80 | (0xF & (val)))

// VREF commands using OP[6:0] to specify VREF calibration setting
#define DDR5_VREFCA(val)                  (0x7F & (val))
#define DDR5_VREFCS(val)                  ((0x7F & (val)) | MRC_BIT7)


///
/// Timings
///

///  Precharge to Precharge Delay for all Frequencies in tCK
#define MRC_DDR5_tPPD_ALL_FREQ (2)

// DDR5 Muti-cycle CS timings (nCK) from JEDEC spec
#define DDR5_tMC_MPC_SETUP_MIN             3 // tMC_MPC_Setup
#define DDR5_tMC_MPC_HOLD_MIN              3 // tMC_MPC_Hold
#define DDR5_tMPC_CS                       6 // tMPC_CS

/// CAS-to-CAS delay for all frequencies in tCK
#define MRC_DDR5_tCCD_ALL_FREQ (8)

/// tDQSCK (pS)
#define MRC_DDR5_tDQSCK_MIN (135)
#define MRC_DDR5_tDQSCK_MAX (135)

/// tDQSS (1000nCK)
#define MRC_DDR5_tDQSS_MIN (375)
#define MRC_DDR5_tDQSS_MAX (375)

// tDQS2DQmin (pS)
#define MRC_DDR5_tDQS2DQ_MIN (200)

/// tXP value for DDR5: max(7.5ns, 8nCK)
#define tXP_DDR5_FS  7500000

/// tRPRE value (nCK)
#define TRPRE_ALL_FREQ_DDR5_1tCK  (1)
#define TRPRE_ALL_FREQ_DDR5_2tCK  (2)

/// tWPRE value (nCK)
#define TWPRE_ALL_FREQ_DDR5  (2)

/// tODTon / tODToff values in [nCK]
#define MRC_DDR5_tODT_ON_WR_OFFSET  (-3)
#define MRC_DDR5_tODT_OFF_WR_OFFSET (2)

/// CKE min pulse width/ Power down Entry to Exit (FS)
#define MRC_DDR5_tCKE_MIN           (7500000)
#define MRC_DDR5_tCKE_MIN_NCK       (8)

#define DDR5_CL_30                         4

// Write Leveling Pulse Enable - Time from Write Leveling Training Enable MRW to
// when Internal Write Leveling Pulse logic level is valid (NS)
#define MRC_DDR5_tWLPEN_NS             (15)

// MAX tWLO acording to DDR5 spec is 9.5 ns
#define tWLO_MAX_FS     (95 * 100 * 1000)

// DDR5 Write leveling modes
typedef enum {
  Ddr5WrLvlModeExternal,
  Ddr5WrLvlModeInternal,
  Ddr5WrLvlModeDisable
} Ddr5WrLvlMode;

/**
  Perform JEDEC Init sequence for DDR5.

  @param[in] MrcData - Pointer to MRC global data.

  @retval MrcStatus
**/
MrcStatus
MrcJedecInitDdr5 (
  IN MrcParameters *const MrcData
  );

/**
  Perform JEDEC reset sequence for DDR5.

  @param[in] MrcData - Include all MRC global data.

  @retval - mrcSuccess if no errors encountered, mrcFail otherwise
**/
MrcStatus
MrcJedecResetDdr5 (
  IN MrcParameters *const MrcData
  );

/**
  Issue NOP command using MRH (Mode Register Handler).

  @param[in] MrcData    - Include all MRC global data.
  @param[in] Controller - the controller to work on
  @param[in] Channel    - The channel to work on
  @param[in] Rank       - The rank to work on
  @param[in] DebugPrint - When TRUE, will print debugging information

  @retval mrcSuccess               - NOP was sent successfully
  @retval mrcDeviceBusy            - Timed out waiting for MRH
  @retval mrcUnsupportedTechnology - The memory technology is not supported
**/
MrcStatus
MrcIssueNop (
  IN MrcParameters *const MrcData,
  IN UINT32               Controller,
  IN UINT32               Channel,
  IN UINT32               Rank,
  IN BOOLEAN              DebugPrint
  );

/**
  Issue VREFCA command using MRH (Mode Register Handler).

  @param[in] MrcData    - Include all MRC global data.
  @param[in] Controller - the controller to work on
  @param[in] Channel    - The channel to work on
  @param[in] Rank       - The rank to work on
  @param[in] Opcode     - VREFCA Opcode OP[7:0]
  @param[in] DebugPrint - When TRUE, will print debugging information

  @retval mrcSuccess               - VREFCA was sent successfully
  @retval mrcDeviceBusy            - Timed out waiting for MRH
  @retval mrcUnsupportedTechnology - The memory technology is not supported
**/
MrcStatus
MrcIssueVrefCa (
  IN MrcParameters *const MrcData,
  IN UINT32               Controller,
  IN UINT32               Channel,
  IN UINT32               Rank,
  IN UINT32               Opcode,
  IN BOOLEAN              DebugPrint
  );

/**
  Enable/Disable DDR5 Write Leveling mode on DRAM

  @param[in] MrcData     - Include all MRC global data.
  @param[in] WriteLvMode - Write leveling mode to enable (Internal/External)
                          Ddr5ExternalWrLvMode:
                            MR2[1]: Write Leveling = 1
                            MR2[7]: Internal Write Timing = 0
                          Ddr5InternalWrLvMode:
                            MR2[1]: Write Leveling = 1
                            MR2[7]: Internal Write Timing = 1
                          Ddr5DisableWrLvMode (Normal Mode):
                            MR2[1]: Write Leveling = 0
                            MR2[7]: Internal Write Timing = 1

  @retval - mrcSuccess if no errors encountered, mrcFail otherwise
**/
MrcStatus
MrcDdr5SetDramWrLvMode (
  IN MrcParameters  *const MrcData,
  IN Ddr5WrLvlMode   const  WrLevelingMode
  );

/**
  Set CAS latency on on DRAM

  @param[in] MrcData     - Include all MRC global data.
  @param[in] CasLatency  - CAS latency value:
                            00000B: 22
                            00001B: 24
                            00010B: 26
                            00011B: 28
                            ...
                            10011B: 60
                            10100B: 62
                            10101B: 64
                            10110B: 66
                            All other encodings reserved.

  @retval - mrcSuccess if no errors encountered, mrcFail otherwise
**/
MrcStatus
MrcDdr5SetCasLatency (
  IN MrcParameters  *const MrcData,
  UINT8                    CasLatency
  );

/**
  This function sets Write Leveling Internal Cycle delay through MR3 command

  @param[in] MrcData          - Include all MRC global data.
  @param[in] Controller       - Index of the requested Controler
  @param[in] Channel          - Index of the requested Channel
  @param[in] Rank             - Current working rank
  @param[in] IntCycleAlignLow - Write Leveling Internal Cycle Alignment (Lower Byte)
  @param[in] IntCycleAlignUp  - Write Leveling Internal Cycle Alignment (Upper Byte)
                                0000B: Disable (Default)
                                0001B: -1 tCK
                                0010B: -2 tCK
                                0011B: -3 tCK
                                0100B: -4 tCK
                                0101B: -5 tCK
                                0110B: -6 tCK
                                0111B: -7 tCK
                                1000B~ 1111B: RFU

  @retval - mrcSuccess if no errors encountered, mrcFail otherwise
**/
MrcStatus
MrcSetMR3_DDR5 (
  IN MrcParameters  *const MrcData,
  IN UINT32                Controller,
  IN UINT32                Channel,
  IN UINT32                Rank,
  IN UINT8                 IntCycleAlignLow,
  IN UINT8                 IntCycleAlignUp
  );

#endif // _MRC_DDR_5_H_
