/** @file
  This file contains common initialization code for PCIe SIP controller

@copyright
  INTEL CONFIDENTIAL
  Copyright 2020 Intel Corporation.

  The source code contained or described herein and all documents related to the
  source code ("Material") are owned by Intel Corporation or its suppliers or
  licensors. Title to the Material remains with Intel Corporation or its suppliers
  and licensors. The Material may contain trade secrets and proprietary and
  confidential information of Intel Corporation and its suppliers and licensors,
  and is protected by worldwide copyright and trade secret laws and treaty
  provisions. No part of the Material may be used, copied, reproduced, modified,
  published, uploaded, posted, transmitted, distributed, or disclosed in any way
  without Intel's prior express written permission.

  No license under any patent, copyright, trade secret or other intellectual
  property right is granted to or conferred upon you by disclosure or delivery
  of the Materials, either expressly, by implication, inducement, estoppel or
  otherwise. Any license under such intellectual property rights must be
  express and approved by Intel in writing.

  Unless otherwise agreed by Intel in writing, you may not remove or alter
  this notice or any other notice embedded in Materials by Intel or
  Intel's suppliers or licensors in any way.

  This file contains an 'Intel Peripheral Driver' and is uniquely identified as
  "Intel Reference Module" and is licensed for Intel CPUs and chipsets under
  the terms of your license agreement with Intel or your vendor. This file may
  be modified by the user, subject to additional terms of the license agreement.

@par Specification Reference:
**/

#include <Library/PeiPcieSipInitLib.h>
#include <Library/DebugLib.h>

#include <Register/PcieSipRegs.h>

/**
  Configure single LTR subtraction register based on the
  configuration.

  @param[in] RpDev                 Pointer to the root port device
  @param[in] LtrSubRegisterOffset  Offset of the particular LTR subtraction register
  @param[in] LtrSubConfig          Pointer to LTR subtraction configuration
**/
VOID
PcieSipConfigureLtrSubtractionSingle (
  IN PCIE_ROOT_PORT_DEV      *RpDev,
  IN UINT32                  LtrSubRegisterOffset,
  IN LTR_SUBTRACTION_CONFIG  *LtrSubConfig
  )
{
  UINT32  Data32And;
  UINT32  Data32Or;

  if (LtrSubConfig->NoSnoopEnable) {
    Data32And = ~(B_PCIE_LTRSUB_LTRNSLSUBEN |
                  B_PCIE_LTRSUB_LTRNSLSSUBV_MASK |
                  B_PCIE_LTRSUB_LTRNSLSUBV_MASK);

    Data32Or =  (B_PCIE_LTRSUB_LTRNSLSUBEN |
                 (LtrSubConfig->NoSnoopScale << N_PCIE_LTRSUB_LTRNSLSSUBV) |
                 (LtrSubConfig->NoSnoopValue << N_PCIE_LTRSUB_LTRNSLSUBV));

    RpDev->PciCfgAccess.AndThenOr32 (
      RpDev,
      LtrSubRegisterOffset,
      Data32And,
      Data32Or
      );
  }

  if (LtrSubConfig->SnoopEnable) {
    Data32And = (UINT32) ~(B_PCIE_LTRSUB_LTRSLSUBEN |
                           B_PCIE_LTRSUB_LTRSLSSUBV_MASK |
                           B_PCIE_LTRSUB_LTRSLSUBV_MASK);

    Data32Or = (B_PCIE_LTRSUB_LTRSLSUBEN |
                (LtrSubConfig->SnoopScale << N_PCIE_LTRSUB_LTRSLSSUBV) |
                (LtrSubConfig->SnoopValue << N_PCIE_LTRSUB_LTRSLSUBV));

    RpDev->PciCfgAccess.AndThenOr32 (
      RpDev,
      LtrSubRegisterOffset,
      Data32And,
      Data32Or
      );
  }
}

/**
  Configure all LTR subtraction registers according to configuration

  @param[in] RpDev  Pointer to the root port device
  @param[in] L1StandardConfig  Pointer to L1Standard LTR subtraction config
  @param[in] L1p1Config        Pointer to L1.1 LTR subtraction config
  @param[in] L1p2Config        Pointer to L1.2 LTR subtraction config
**/
VOID
PcieSipConfigureLtrSubstraction (
  IN PCIE_ROOT_PORT_DEV      *RpDev,
  IN LTR_SUBTRACTION_CONFIG  *L1StandardConfig,
  IN LTR_SUBTRACTION_CONFIG  *L1p1Config,
  IN LTR_SUBTRACTION_CONFIG  *L1p2Config
  )
{
  if (RpDev->SipVersion < PcieSip16) {
    DEBUG ((DEBUG_WARN, "PCIe SIP < %d does not support LTR subtraction\n", PcieSip16));
    return;
  }

  PcieSipConfigureLtrSubtractionSingle (RpDev, R_PCIE_LTRSUBL1STD, L1StandardConfig);
  PcieSipConfigureLtrSubtractionSingle (RpDev, R_PCIE_LTRSUBL11, L1p1Config);
  PcieSipConfigureLtrSubtractionSingle (RpDev, R_PCIE_LTRSUBL12, L1p2Config);
}

/**
  Enables retimer presence and two retimers presence detect on supported
  SIP versions.

  @param[in] RpDev  Pointer to the root port device
**/
VOID
PcieSipConfigureRetimerSupport (
  IN PCIE_ROOT_PORT_DEV  *RpDev
  )
{
  if (RpDev->SipVersion < PcieSip16) {
    DEBUG ((DEBUG_WARN, "PCIe SIP < %d does not support retimer presence detect\n", PcieSip16));
    return;
  }

  RpDev->PciCfgAccess.Or32 (
    RpDev,
    R_PCIE_LCAP2,
    (UINT32) (B_PCIE_LCAP2_TRPDS | B_PCIE_LCAP2_RPDS)
    );
}

/**
  Configure 10-Bit Tag

  @param[in] RpDev               Pointer to the root port device
  @param[in] CompleterSupported  If TRUE it indicates 10-Bit Tag Completer is supported.
  @param[in] RequesterSupported  If TRUE it indicates 10-Bit Tag Requester is supported.

**/
VOID
PcieSipConfigure10BitTag (
  IN PCIE_ROOT_PORT_DEV      *RpDev,
  IN BOOLEAN                 CompleterSupported,
  IN BOOLEAN                 RequesterSupported
  )
{
  if (RpDev->SipVersion < PcieSip16) {
    DEBUG ((DEBUG_WARN, "PCIe SIP < %d does not support 10-bit tag\n", PcieSip16));
    return;
  }

  if (CompleterSupported == FALSE) {
    //
    // Program 10-Bit Tag Completer Supported PX10BTCS = 0x0
    //
    RpDev->PciCfgAccess.And32 (RpDev, R_PCIE_DCAP2, (UINT32)~B_PCIE_DCAP2_PX10BTCS);
  } else {
    //
    // Program 10-Bit Tag Completer Supported PX10BTCS = 0x1
    //
    RpDev->PciCfgAccess.Or32 (RpDev, R_PCIE_DCAP2, B_PCIE_DCAP2_PX10BTCS);
  }

  if (RequesterSupported == FALSE) {
    //
    // Program 10-Bit Tag Requester Enable PX10BTRE = 0x0
    // Program 10-Bit Tag Requester Supported PX10BTRS = 0x0
    //
    RpDev->PciCfgAccess.And32 (RpDev, R_PCIE_DCTL2, (UINT32)~B_PCIE_DCTL2_PX10BTRE);
    RpDev->PciCfgAccess.And32 (RpDev, R_PCIE_DCAP2, (UINT32)~B_PCIE_DCAP2_PX10BTRS);
  } else {
    //
    // Program 10-Bit Tag Requester Enable PX10BTRE = 0x1
    // Program 10-Bit Tag Requester Supported PX10BTRS = 0x1
    //
    RpDev->PciCfgAccess.Or32 (RpDev, R_PCIE_DCTL2, B_PCIE_DCTL2_PX10BTRE);
    RpDev->PciCfgAccess.Or32 (RpDev, R_PCIE_DCAP2, B_PCIE_DCAP2_PX10BTRS);
  }

  //
  // Program Fabric 10-bit Tag Support Enable F10BTSE = 0x0
  //
  RpDev->PciCfgAccess.And32 (RpDev, R_PCIE_ADVMCTRL, (UINT32)~B_PCIE_ADVMCTRL_F10BTSE);
}

/**
  Configure Peer Disable

  @param[in] RpDev               Pointer to the root port device
  @param[in] CfgReads            TRUE/FALSE, enable/disable forwarding of Upstream Posted Memory Reads
  @param[in] CfgWrites           TRUE/FALSE, enable/disable forwarding of Upstream Posted Memory Writes
**/
VOID
PcieSipConfigurePeerDisable (
  IN PCIE_ROOT_PORT_DEV      *RpDev,
  IN BOOLEAN                 CfgReads,
  IN BOOLEAN                 CfgWrites
  )
{
  UINT32  Data32And;
  UINT32  Data32Or;

  Data32And = 0;
  Data32Or = 0;

  if (RpDev->SipVersion < PcieSip16) {
    DEBUG ((DEBUG_WARN, "PCIe SIP < %d does not support Peer Disable\n", PcieSip16));
    return;
  }

  Data32And &= (UINT32) ~((B_PCIE_CFG_CCFG_UPMWPD) | (B_PCIE_CFG_CCFG_UMRPD));
  if (CfgReads) {
    Data32Or |=(UINT32) (B_PCIE_CFG_CCFG_UMRPD);
  }
  if (CfgWrites) {
    Data32Or |= (UINT32) (B_PCIE_CFG_CCFG_UPMWPD);
  }

  RpDev->PciCfgAccess.AndThenOr32 (
    RpDev,
    R_PCIE_CFG_CCFG,
    Data32And,
    Data32Or
    );
}

/**
  Configure PCIe Completion Coalescing

  @param[in] RpIndex             Index of the PCI root port
**/
VOID
PcieConfigureCoalescing (
  IN  PCIE_ROOT_PORT_DEV    *RpDev
  )
{
  UINT32    Data32And;

  if (RpDev->SipVersion < PcieSip16) {
    DEBUG ((DEBUG_WARN, "PCIe SIP < %d does not support Completion Coalescing\n", PcieSip16));
    return;
  }

  Data32And = (RpDev->PciCfgAccess.Read32 (RpDev, R_PCIE_COCTL) &
              ~(UINT32)(B_PCIE_COCTL_PWCE | B_PCIE_COCTL_DDCE |
                        B_PCIE_COCTL_CT_MASK | B_PCIE_COCTL_CTE |
                        B_PCIE_COCTL_ROAOP | B_PCIE_COCTL_PCLM_MASK |
                        B_PCIE_COCTL_NPCLM_MASK));

  RpDev->PciCfgAccess.And32 (RpDev, R_PCIE_COCTL, Data32And);
  DEBUG((DEBUG_INFO, "COCTL = %x\n", RpDev->PciCfgAccess.Read32 (RpDev, R_PCIE_COCTL)));
  return;
}
