/** @file
  PCH private PEI PMC Library Ver3.

@copyright
  INTEL CONFIDENTIAL
  Copyright 2019 - 2020 Intel Corporation.

  The source code contained or described herein and all documents related to the
  source code ("Material") are owned by Intel Corporation or its suppliers or
  licensors. Title to the Material remains with Intel Corporation or its suppliers
  and licensors. The Material may contain trade secrets and proprietary and
  confidential information of Intel Corporation and its suppliers and licensors,
  and is protected by worldwide copyright and trade secret laws and treaty
  provisions. No part of the Material may be used, copied, reproduced, modified,
  published, uploaded, posted, transmitted, distributed, or disclosed in any way
  without Intel's prior express written permission.

  No license under any patent, copyright, trade secret or other intellectual
  property right is granted to or conferred upon you by disclosure or delivery
  of the Materials, either expressly, by implication, inducement, estoppel or
  otherwise. Any license under such intellectual property rights must be
  express and approved by Intel in writing.

  Unless otherwise agreed by Intel in writing, you may not remove or alter
  this notice or any other notice embedded in Materials by Intel or
  Intel's suppliers or licensors in any way.

  This file contains an 'Intel Peripheral Driver' and is uniquely identified as
  "Intel Reference Module" and is licensed for Intel CPUs and chipsets under
  the terms of your license agreement with Intel or your vendor. This file may
  be modified by the user, subject to additional terms of the license agreement.

@par Specification Reference:
**/

#include <Base.h>
#include <Uefi/UefiBaseType.h>
#include <Library/IoLib.h>
#include <Library/DebugLib.h>
#include <Library/BaseLib.h>
#include <Ppi/SiPolicy.h>
#include <Library/PchPcrLib.h>
#include <Library/DciPrivateLib.h>
#include <Library/PmcPrivateLib.h>
#include <Register/PmcRegs.h>
#include <Register/PmcRegsVer2.h>
#include <Register/PmcRegsFivr.h>
#include <Library/BaseMemoryLib.h>
#include <Register/PchRegs.h>
#include <Library/DciPrivateLib.h>
#include <Library/PciSegmentLib.h>
#include <Library/GpioPrivateLib.h>
#include <Library/PchInfoLib.h>
#include <Library/CpuPlatformLib.h>

#include "PmcPrivateLibInternal.h"

/**
  S0ix settings

  @param[in] PmConfig                 PmConfig
**/
VOID
PmcSlpS0Config (
  IN PCH_PM_CONFIG                    *PmConfig
  )
{
  UINT32                              PchPwrmBase;

  PchPwrmBase = PmcGetPwrmBase ();
  ///
  /// PWRMBASE + 0x1B1C = 0x801E
  ///
  MmioOr32 (PchPwrmBase + R_PMC_PWRM_1B1C, 0x801E);
  ///
  /// PWRMBASE + 0x1B20 = 0x00138801
  ///
  MmioWrite32 (PchPwrmBase + R_PMC_PWRM_CPPM_MISC_CFG, 0x00138801);
}

/**
  Configure Low Power Mode S0ix sub-state support

  @param[in] PmConfig                 PmConfig
**/
VOID
PmcConfigureLpmS0ixSupport (
  IN PCH_PM_CONFIG                         *PmConfig
  )
{
  //
  // LPM unsupported on EBG PCH
  //
}

/**
  Enable ModPHY SUS Power Gating
**/
VOID
EnableModPhySusWellPg (
  VOID
  )
{
  // PG unsupported for EBG
  return;
}

/**
  Configure CPPM Force Alignment Control
**/
VOID
PmcConfigureCppmForceAlignmentControl (
  VOID
  )
{
  UINT32 PwrmBase;
  UINT32 Data32Or;
  UINT32 Data32And;

  PwrmBase = PmcGetPwrmBase ();

  //
  // Recommendation on programming is delivered by IP
  // team.
  //
  Data32Or = 0xD80A8000;
  Data32And = ~0u;

  if (PchStepping () == PCH_Z0) {
    Data32And &= ~(UINT32)BIT28;
  }

  MmioAndThenOr32 (
    PwrmBase + R_PMC_PWRM_1BD0,
    Data32And,
    Data32Or
    );

  //
  // Lock PCH Forced alignment config
  //
  MmioOr32 (PwrmBase + R_PMC_PWRM_1BD0, BIT30);
}

/**
  This function is part of PMC init and configures which clock wake signals should
  set the SLOW_RING, SA, FAST_RING_CF and SLOW_RING_CF indication sent up to the CPU/PCH
**/
VOID
PmcInitClockWakeEnable (
  VOID
  )
{
  UINT32                    PchPwrmBase;

  PchPwrmBase = PmcGetPwrmBase ();

  //
  // Configure which backbone clock wake signals should set the SLOW_RING indication
  // sent up over the PM_SYNC
  // Enabled for:
  // All PCIe ports
  // All SATA ports
  // GbE
  // CNVi
  // HDA
  // USB OTG
  // VLW
  // Legacy cluster
  // P2SB
  // ISH
  // ME
  // PCIe NAND
  // SMBUS
  // CHAP
  // Thermal sensor
  // SPI
  // PMC
  // Display
  // Southport A, B, C and D
  // Primary channel
  // MEIO
  // GPIO
  // Audio DSP
  // LPSS
  // SCS
  // THC 0 and 1
  //
  MmioWrite32 (PchPwrmBase + R_PMC_PWRM_EN_CW_SLOW_RING, 0x2F8FBB01);
  MmioWrite32 (PchPwrmBase + R_PMC_PWRM_EN_CW_SLOW_RING2, 0x1A80C7E0);

  //
  // Configure which backbone clock wake signals should set the SA indication
  // sent up over the PM_SYNC. This configuration matches the SLOW_RING indication
  // configuration.
  //
  MmioWrite32 (PchPwrmBase + R_PMC_PWRM_EN_CW_SA, 0x2F8FBB01);
  MmioWrite32 (PchPwrmBase + R_PMC_PWRM_EN_CW_SA2, 0x1A80C7E0);

  //
  // Configure which clock wake signals should set the SLOW_RING_CF indication
  // sent up over the PM_SYNC
  // Enabled for:
  // Legacy cluster
  // P2SB
  //
  MmioWrite32 (PchPwrmBase + R_PMC_PWRM_EN_CW_SLOW_RING_CF, 0x00018000);
}

/**
  Configures PM_SYNC state hysteresis according to IP recommendation
  for SoC.
**/
VOID
PmcConfigurePmSyncHysteresis (
  VOID
  )
{
  //
  // Configure hysteresis for each state indication on PM_SYNC
  // This is recommendation from PMC IP team.
  //
  MmioWrite32 (PmcGetPwrmBase () + R_PMC_PWRM_PM_SYNC_STATE_HYS, 0x00200840);
}

/**
  This function configures PCH FIVR FET Ramp time config
**/
VOID
PmcFivrFetRampTimeConfig (
  VOID
  )
{
  //
  // Lock V1p05 PHY FET Ramp Time settings
  //
  MmioOr32 (PmcGetPwrmBase () + R_PMC_PWRM_EXT_FET_RAMP_CONFIG, B_PMC_PWRM_EXT_FET_RAMP_CONFIG_V105_PHY_FRT_LOCK | B_PMC_PWRM_EXT_FET_RAMP_CONFIG_V105_PHY_IS_FRT_LOCK);
}

/**
  Get the CPU IOVR ramp duration POR for the SoC

  @retval Ramp duration in the unit of 10us
**/
UINT8
PmcGetCpuIovrRampDuration (
  VOID
  )
{
  return 0;
}

/**
  Perform generation specific locking configuration.

  @param[in] SiPolicy  Pointer to SiPolicy
**/
VOID
PmcGenerationSpecificLock (
  IN SI_POLICY_PPI  *SiPolicy
  )
{
}

/**
  Initialize LDO settings.
**/
VOID
PmcInitLdo (
  VOID
  )
{
  //
  // Enable USB2/TS LDO dynamic shutdown and enable CNVi LDO low power mode for all SKUs
  //
  MmioOr32 (PmcGetPwrmBase () + R_PMC_PWRM_PMLDOCTRL, B_PMC_PWRM_PMLDOCTRL_CNVIP24LDOLPEN | B_PMC_PWRM_PMLDOCTRL_USB2TS1P3LDODSEN);
}

/**
  Will send IPC command to PMC to write Private Register through proxy access

  @param[in]  PrivateRegisterIndex  Private Register Index
  @param[in]  WriteValue            Write value

  @retval     EFI_SUCCESS           Value changed successfully
**/
STATIC
EFI_STATUS
PmcWritePrivateRegister32 (
  IN  UINT8        PrivateRegisterIndex,
  IN  UINT32       WriteValue
  )
{
  EFI_STATUS                Status;
  PMC_IPC_COMMAND_BUFFER    Wbuf;

  if (PrivateRegisterIndex >= V_PMC_PRIVATE_REGISTER_ID_MAX) {
    DEBUG ((
      DEBUG_ERROR,
      "Unsupported PMC Private Register Index provided: %d, max is %d\n",
      PrivateRegisterIndex, V_PMC_PRIVATE_REGISTER_ID_MAX
      ));
    return EFI_UNSUPPORTED;
  }

  Status = EFI_SUCCESS;
  ZeroMem (&Wbuf, sizeof (PMC_IPC_COMMAND_BUFFER));

  Wbuf.Buf0 = PrivateRegisterIndex;
  Wbuf.Buf1 = WriteValue;
  Status = PmcSendCommand (
    V_PMC_PWRM_IPC_CMD_COMMAND_PMC_PRV_REG_ACCESS,
    0x1,  // Read
    0x2,  // Write buffer size
    &Wbuf,
    NULL
    );
  if (EFI_ERROR (Status)) {
    DEBUG ((DEBUG_ERROR, "Error while writing PMC Private Register id %d - %r\n", PrivateRegisterIndex, Status));
    return Status;
  }

  return EFI_SUCCESS;
}

/**
  This function shall program Global Reset Event/Trigger masks according to values
  provided in PMC Config Block. Programming shall be done through proxy access to
  those Private Configuration Registers

  @param[in] SiPolicyPpi  Pointer to SI_POLICY_PPI instance
**/
STATIC
VOID
PmcProgramGlobalResetMasks (
  IN SI_POLICY_PPI  *SiPolicyPpi
  )
{
  PCH_PM_CONFIG             *PmConfig;
  EFI_STATUS                Status;

  DEBUG ((DEBUG_INFO, "%a start\n", __FUNCTION__));

  Status = GetConfigBlock ((VOID *) SiPolicyPpi, &gPmConfigGuid, (VOID *) &PmConfig);
  ASSERT_EFI_ERROR (Status);
  if (EFI_ERROR (Status)) {
    DEBUG ((DEBUG_ERROR, "Error while getting PmConfig - %r\n", Status));
    return;
  }

  if (!PmConfig->GlobalResetMasksOverride) {
    DEBUG ((DEBUG_INFO, "Global Resets Masks override is disabled.\n"));
    return;
  }

  Status = PmcWritePrivateRegister32 (
             V_PMC_PRIVATE_REGISTER_ID_GBLRST_TRIG_MASK0,
             PmConfig->GlobalResetTriggerMask.Value
             );
  if (EFI_ERROR (Status)) {
    return;
  }

  Status = PmcWritePrivateRegister32 (
             V_PMC_PRIVATE_REGISTER_ID_GBLRST_EVENT_MASK0,
             PmConfig->GlobalResetEventMask.Value
             );
  if (EFI_ERROR (Status)) {
    return;
  }
}

/**
  Performs generation specific programing.

  @param[in] SiPolicyPpi  Pointer to SI_POLICY_PPI instance
**/
VOID
PmcGenerationSpecificPrograming (
  IN SI_POLICY_PPI  *SiPolicyPpi
  )
{
  //
  // W/A for IIO enumerator resetting PMC BAR address.
  // After we set PWRM and ACPI base in early PEI but before we
  // hide PMC config space access IIO PCI enumerator will
  // disable PMC bus master and IO space enable bit which
  // will disable ACPI IO BAR access causing boot issues.
  // As a temporary w/a we reenable the access here.
  //
  PmcEnableIOSpaceAndBusMaster ();

  //
  // Will program Global Reset Trigger and Events masks
  // based on value provided in PMC config block
  //
  PmcProgramGlobalResetMasks (SiPolicyPpi);
}

/**
  Checks if current PMC supports timed GPIO functionality

  @retval TRUE   Timed GPIO functionality is supported
  @retval FALSE  Timed GPIO functionality is not supported
**/
BOOLEAN
PmcIsTimedGpioSupported (
  VOID
  )
{
  return TRUE;
}


/**
  Checks if given PMC version supports S0ix auto demotion.

  @retval TRUE  S0ix auto demotion supported
  @retval FALSE S0ix auto demotion not supported
**/
BOOLEAN
PmcIsS0ixAutoDemotionSupported (
  VOID
  )
{
  return FALSE;
}

/**
  Checks if the CPPM qualifiers are supported on the
  SoC.

  @retval TRUE  CPPM qualification is supported
  @retval FALSE CPPM qualification is not supported
**/
BOOLEAN
PmcIsCppmQualificationSupported (
  VOID
  )
{
  return FALSE;
}
