/** @file
  Initilize TME in PEI

@copyright
  INTEL CONFIDENTIAL
  Copyright 2020 Intel Corporation.

  The source code contained or described herein and all documents related to the
  source code ("Material") are owned by Intel Corporation or its suppliers or
  licensors. Title to the Material remains with Intel Corporation or its suppliers
  and licensors. The Material may contain trade secrets and proprietary and
  confidential information of Intel Corporation and its suppliers and licensors,
  and is protected by worldwide copyright and trade secret laws and treaty
  provisions. No part of the Material may be used, copied, reproduced, modified,
  published, uploaded, posted, transmitted, distributed, or disclosed in any way
  without Intel's prior express written permission.

  No license under any patent, copyright, trade secret or other intellectual
  property right is granted to or conferred upon you by disclosure or delivery
  of the Materials, either expressly, by implication, inducement, estoppel or
  otherwise. Any license under such intellectual property rights must be
  express and approved by Intel in writing.

  Unless otherwise agreed by Intel in writing, you may not remove or alter
  this notice or any other notice embedded in Materials by Intel or
  Intel's suppliers or licensors in any way.

  This file contains an 'Intel Peripheral Driver' and is uniquely identified as
  "Intel Reference Module" and is licensed for Intel CPUs and chipsets under
  the terms of your license agreement with Intel or your vendor. This file may
  be modified by the user, subject to additional terms of the license agreement.

@par Specification Reference:
**/
#include <Library/PeiServicesLib.h>
#include <Library/BaseMemoryLib.h>
#include <Library/DebugLib.h>
#include <Library/TmeLib.h>
#include <Library/MsrFruLib.h>
#include <Library/PmcLib.h>
#include <CpuAccess.h>
#include <Register/Cpuid.h>
#include <Register/ArchMsr.h>
#include <Register/PmcRegs.h>

typedef struct {
  UINT64 TmeEnable          :1;
  UINT64 KeySelect          :1;
  UINT64 SaveKeyForStandby  :1;
} TME_CONFIG;

/**
  Perform Multi-Key Total Memory Encryption initialization.

  @param[in] TmeEnable      - TME policy enable
  @param[in] TmeExcludeBase - Base physical address to be excluded for TME encryption
  @param[in] TmeExcludeSize - Size of range to be excluded from TME encryption

  @retval VOID - No value to return
**/
VOID
TmeInit (
  IN UINT32 TmeEnable,
  IN UINT64 TmeExcludeBase,
  IN UINT64 TmeExcludeSize
  )
{
  EFI_CPUID_REGISTER             CpuidRegs;
  MSR_TME_CAPABILITY_REGISTER    TmeCapability;
  MSR_TME_ACTIVATE_REGISTER      TmeActivate;
  MSR_TME_ACTIVATE_REGISTER      TmeValidate;
  MSR_TME_EXCLUDE_MASK_REGISTER  ExcludeMask;
  MSR_TME_EXCLUDE_BASE_REGISTER  ExcludeBase;
  BOOLEAN                        TmeLocked;
  BOOLEAN                        TmeSupported;
  EFI_BOOT_MODE                  BootMode;
  EFI_STATUS                     Status;
  UINT8                          Index;
  UINT32                         RegisterVal32;
  BOOLEAN                        WarmReset;

  TmeCapability.Uint64   = 0;
  TmeActivate.Uint64     = 0;
  TmeValidate.Uint64     = 0;
  TmeLocked              = 0;
  TmeSupported           = 0;
  BootMode               = 0;
  Status                 = 0;
  Index                  = 0;
  WarmReset              = FALSE;

  DEBUG ((DEBUG_INFO, "Total Memory Encryption (TME) Initialization\n"));

  ///
  /// Detect TME capability and encryption policy is supported in the hardware
  ///

  ///
  /// Verify TME supported through CPUID.7.0.ECX.13
  ///
  AsmCpuidEx (CPUID_STRUCTURED_EXTENDED_FEATURE_FLAGS, 0, &CpuidRegs.RegEax,&CpuidRegs.RegEbx,&CpuidRegs.RegEcx,&CpuidRegs.RegEdx);
  if ((CpuidRegs.RegEcx & BIT13) != BIT13) {
    DEBUG ((DEBUG_INFO, "TME is not supported by the processor. Skipping TME activation.\n"));
    return;
  }

  ///
  /// Read MSR_TME_CAPABILITY (981H) to determine supported encryption policy
  /// TGL currently only supports AES-XTS
  ///
  TmeCapability.Uint64 = AsmReadMsr64 (MSR_TME_CAPABILITY);
  if (TmeCapability.Bits.AesXts != 0) {
    DEBUG ((DEBUG_INFO, "TME supports AES-XTS encryption policy\n"));
    ///
    /// Set TME policy
    ///
    TmeActivate.Bits.TmePolicy = V_TME_ACTIVATE_TME_POLICY_AES_XTS;

    ///
    /// TME is supported in the hardware
    ///
    TmeSupported = TRUE;
  } else {
    DEBUG ((DEBUG_INFO, "TME does not support selected encryption policy. Skipping TME activation.\n"));
    return;
  }

  if (TmeSupported) {
    ///
    /// Enable TME if BIOS policy is enabled
    ///
    if (TmeEnable) {
      ///
      /// Configure MSR_TME_EXCLUDE_MASK (983H) with TME Mask and Enable bit
      ///
      if (TmeExcludeSize != 0) {
        ExcludeMask.Uint64 = TmeExcludeSize;
        ExcludeMask.Bits.Enable = 1;
        AsmWriteMsr64 (MSR_TME_EXCLUDE_MASK, ExcludeMask.Uint64);
      }

      ///
      /// Configure MSR_TME_EXCLUDE_BASE (984H) with TME Base
      ///
      if (TmeExcludeBase != 0) {
        ExcludeBase.Uint64 = TmeExcludeBase;
        AsmWriteMsr64 (MSR_TME_EXCLUDE_BASE, ExcludeBase.Uint64);
      }

      ///
      /// Set TME Key Select and Save Key for Warm Reset, Standby and Flash Update Mode
      /// Key Select
      ///   - Set for Warm Reset, S3 resume flow and flash update flow to restore
      ///     TME keys from PCH
      ///   - Clear for cold boot to create new TME keys
      /// Save Key
      ///   - Clear for Warm Reset, S3 resume flow and flash update flow
      ///   - Set for cold boot to save key into storage
      ///
      Status = PeiServicesGetBootMode (&BootMode);
      ASSERT_EFI_ERROR (Status);

      ///
      /// Detect Warm Reset boot
      ///
      RegisterVal32 = MmioRead32 ((UINTN) (PmcGetPwrmBase () + R_PMC_PWRM_GEN_PMCON_A));
      if ((((RegisterVal32 & B_PMC_PWRM_GEN_PMCON_A_MEM_SR) != 0) &&
          ((RegisterVal32 & B_PMC_PWRM_GEN_PMCON_A_DISB) != 0))) {
        WarmReset = TRUE;
      }

      if ((BootMode == BOOT_ON_S3_RESUME) ||
          (BootMode == BOOT_ON_FLASH_UPDATE) ||
          (WarmReset == TRUE)) {
        TmeActivate.Bits.KeySelect = 1;
        TmeActivate.Bits.SaveKey = 0;
      } else {
        TmeActivate.Bits.KeySelect = 0;
        TmeActivate.Bits.SaveKey = 1;
      }

      ///
      /// Set TME Enable
      ///
      TmeActivate.Bits.TmeEnable = 1;
    } else {
      ///
      /// Clear TME Enable
      ///
      TmeActivate.Bits.TmeEnable = 0;
      DEBUG ((DEBUG_INFO, "TME is disabled. Skipping TME activation.\n"));
    }

    ///
    /// Configure MSR_TME_ACTIVATE (982H) with TME Enable, Key Select, Save Key, and TME Policy
    /// Lock bit will be set upon successful WRMSR for MSR_TME_ACTIVATE.
    ///   - First SMI will also set the Lock
    ///   - This will also lock MSR_TME_EXCLUDE_MASK and MSR_TME_EXCLUDE_BASE
    ///
    DEBUG ((DEBUG_INFO, "Initialize MSR_TME_ACTIVATE: %016llx\n", TmeActivate.Uint64));

    for (Index = 0; Index < TME_ACTIVATE_RETRY_LIMIT; Index++) {
      AsmWriteMsr64 (MSR_TME_ACTIVATE, TmeActivate.Uint64);

      TmeValidate.Uint64 = AsmReadMsr64 (MSR_TME_ACTIVATE);
      if (TmeValidate.Bits.Lock == 1) {
        DEBUG ((DEBUG_INFO, "MSR_TME_ACTIVATE was initialized and locked: %016llx\n", TmeValidate));
        TmeLocked = TRUE;
        break;
      }
    }

    ///
    /// Ensure Lock bit of MSR_TME_ACTIVATE (982H) is set
    /// Lock bit will not be set if creation of ephemeral number using DRNG action failed.
    ///
    if (!TmeLocked) {
      DEBUG ((DEBUG_INFO, "TME was not enabled. Locking MSR_TME_ACTIVATE.\n"));
      TmeActivate.Bits.TmeEnable = 0;
      AsmWriteMsr64 (MSR_TME_ACTIVATE, TmeActivate.Uint64);
      TmeValidate.Uint64 = AsmReadMsr64 (MSR_TME_ACTIVATE);
      DEBUG ((DEBUG_INFO, "MSR_TME_ACTIVATE: %016llx\n", TmeValidate));
    }
  }
}

/**
  Reports if TME is enabled or not

  @retval TRUE             If TME is enabled
  @retval FALSE            If TME is not enabled
**/
BOOLEAN
IsTmeEnabled (
  VOID
  )
{
  MSR_TME_ACTIVATE_REGISTER      TmeActivate;

  TmeActivate.Uint64 = 0;
  TmeActivate.Uint64 = AsmReadMsr64 (MSR_TME_ACTIVATE);

  if((TmeActivate.Bits.Lock == 1) && (TmeActivate.Bits.TmeEnable == 1)) {
      return TRUE;
  }

  return FALSE;
}

/**
  Reports if Lock bit in TME Activate MSR locked or not

  @retval TRUE             If TME Activate MSR is locked
  @retval FALSE            If TME Activate MSR is not locked
**/
BOOLEAN
IsTmeActivateLockSet (
  VOID
  )
{
  MSR_TME_ACTIVATE_REGISTER      TmeActivate;

  TmeActivate.Uint64 = 0;
  TmeActivate.Uint64 = AsmReadMsr64 (MSR_TME_ACTIVATE);
  return TmeActivate.Bits.Lock != 0;
}
