/** @file
  PEIM to initialize Dynamic PMem NVDIMM CACHE PPI

@copyright
  INTEL CONFIDENTIAL
  Copyright 2020 Intel Corporation.

  The source code contained or described herein and all documents related to the
  source code ("Material") are owned by Intel Corporation or its suppliers or
  licensors. Title to the Material remains with Intel Corporation or its suppliers
  and licensors. The Material may contain trade secrets and proprietary and
  confidential information of Intel Corporation and its suppliers and licensors,
  and is protected by worldwide copyright and trade secret laws and treaty
  provisions. No part of the Material may be used, copied, reproduced, modified,
  published, uploaded, posted, transmitted, distributed, or disclosed in any way
  without Intel's prior express written permission.

  No license under any patent, copyright, trade secret or other intellectual
  property right is granted to or conferred upon you by disclosure or delivery
  of the Materials, either expressly, by implication, inducement, estoppel or
  otherwise. Any license under such intellectual property rights must be
  express and approved by Intel in writing.

  Unless otherwise agreed by Intel in writing, you may not remove or alter
  this notice or any other notice embedded in Materials by Intel or
  Intel's suppliers or licensors in any way.

  This file contains an 'Intel Peripheral Driver' and is uniquely identified as
  "Intel Reference Module" and is licensed for Intel CPUs and chipsets under
  the terms of your license agreement with Intel or your vendor. This file may
  be modified by the user, subject to additional terms of the license agreement.

@par Specification Reference:
**/
#include <Library/NvdimmCacheInfoLib.h>



EFI_STATUS
EFIAPI
GetNvdimmCacheInfo (
  IN CONST EFI_NVDIMM_CACHE_PPI   *This,
  IN NVDIMM_CACHE_INFO  *NvdimmCacheInfo
  )
{
  UINTN                     Edrambar;
  UINT32                    MchBar;
  TWOLM_INFO_HOB            *TwoLmInfoHob;

  if (IsTwoLmEnabled ()== FALSE) {
    DEBUG ((DEBUG_ERROR, "2LM: TwoLm not enabled\n"));
    return EFI_UNSUPPORTED;
  }

  TwoLmInfoHob = (TWOLM_INFO_HOB *) GetFirstGuidHob (&gTwoLmInfoHobGuid);
  if (TwoLmInfoHob == NULL) {
    DEBUG ((DEBUG_ERROR, "2LM: TwoLmInfoHob not found\n"));
    return EFI_UNSUPPORTED;
  }
  if (TwoLmInfoHob->TwoLmInfo.PMemSize == 0) {
    DEBUG ((DEBUG_ERROR, "PMEM Disabled, skip populating Nvdimm Cache Info Hob\n"));
    return EFI_UNSUPPORTED;
  }

  MchBar = PegPciSegmentRead32 (PCI_SEGMENT_LIB_ADDRESS (SA_SEG_NUM, SA_PEG_BUS_NUM, 0, 0, 0) + R_SA_MCHBAR) & ~BIT0;
  Edrambar = MmioRead64 (MchBar +R_SA_MCHBAR_EDRAMBAR_OFFSET) & ~BIT0;

  DEBUG ((DEBUG_INFO, "2LM: EDRAM Base is 0X %x\n",Edrambar ));

  NvdimmCacheInfo->Revision                = NVDIMM_CACHE_INFO_REVISION;
  NvdimmCacheInfo->DeepFlushRegister       = Edrambar + ASTRO_OFFSET + R_ASTRO_DEEP_FLUSH_TASK_REG0;
  NvdimmCacheInfo->DeepFlushSizeRegister   = Edrambar + ASTRO_OFFSET + R_ASTRO_DEEP_FLUSH_TASK_REG1;
  NvdimmCacheInfo->DeepFlushStatusRegister = Edrambar + ASTRO_OFFSET + R_ASTRO_DEEP_FLUSH_STATUS;
  NvdimmCacheInfo->MaxDeepFlushSize        = (0x100000000 - 0x1000); // 4GB -4K from DurabilityDsms.asl
  NvdimmCacheInfo->NvdimmCacheBaseAddress  = TwoLmInfoHob->TwoLmInfo.PMemBase;
  NvdimmCacheInfo->NvdimmCacheLength       = LShiftU64(TwoLmInfoHob->TwoLmInfo.PMemSize, 20); // In bytes

  DEBUG ((DEBUG_INFO, "2LM: DeepFlushRegister 0X%x NvdimmCacheBaseAddress 0X%llx\n",NvdimmCacheInfo->DeepFlushRegister, NvdimmCacheInfo->NvdimmCacheBaseAddress));
  DEBUG ((DEBUG_INFO, "2LM: NvdimmCacheLength 0X%lx\n", TwoLmInfoHob->TwoLmInfo.PMemSize));

  return EFI_SUCCESS;
}


EFI_STATUS
EFIAPI
NvdimmCacheInfoPpiHobInstall(
  VOID
  )
{
  EFI_STATUS                Status;
  EFI_PEI_PPI_DESCRIPTOR    *NvdimmCachePpiDescriptor;
  EFI_NVDIMM_CACHE_PPI      *NvdimmCachePpi;
  NVDIMM_CACHE_HOB          *NvdimmCacheInfoHob;
  NVDIMM_CACHE_INFO         *NvdimmCacheInfo;
  SI_PREMEM_POLICY_PPI      *SiPreMemPolicyPpi;
  TWOLM_PREMEM_CONFIG       *TwoLmPreMemConfig;


  ///
  /// Get policy settings through the SaInitConfigBlock PPI
  ///
  Status = PeiServicesLocatePpi(
             &gSiPreMemPolicyPpiGuid,
             0,
             NULL,
             (VOID **)&SiPreMemPolicyPpi
  );
  ASSERT_EFI_ERROR(Status);
  if ((Status != EFI_SUCCESS) || (SiPreMemPolicyPpi == NULL)) {
    return EFI_NOT_FOUND;
  }

  Status = GetConfigBlock((VOID *)SiPreMemPolicyPpi, &gTwoLmPreMemConfigGuid, (VOID *)&TwoLmPreMemConfig);
  ASSERT_EFI_ERROR(Status);

  if (!TwoLmPreMemConfig->DpmemSupport) {
    DEBUG ((DEBUG_ERROR, "2LM: DPmem support policy is not Enabled\n"));
    return EFI_UNSUPPORTED;
  }

  Status = PeiServicesLocatePpi (
             &gNvdimmCachePpiGuid,
             0,
             NULL,
             (VOID **)&NvdimmCachePpi
             );

  if (Status != EFI_SUCCESS) {

    NvdimmCachePpiDescriptor = (EFI_PEI_PPI_DESCRIPTOR *) AllocateZeroPool (sizeof (EFI_PEI_PPI_DESCRIPTOR));
    NvdimmCachePpi = (EFI_NVDIMM_CACHE_PPI *) AllocateZeroPool (sizeof (EFI_NVDIMM_CACHE_PPI));
    NvdimmCacheInfo = (NVDIMM_CACHE_INFO *) AllocateZeroPool (sizeof (NVDIMM_CACHE_INFO));

    if ((NvdimmCachePpiDescriptor == NULL) || (NvdimmCachePpi == NULL) || (NvdimmCacheInfo == NULL)) {
      DEBUG ((DEBUG_ERROR, "2LM: NvdimmCacheInfoPpiHobInstall : Buffer allocation failure\n"));
      return EFI_BUFFER_TOO_SMALL;
    }
    NvdimmCachePpi->GetNvdimmCacheInfo = GetNvdimmCacheInfo;
    NvdimmCachePpiDescriptor->Flags = EFI_PEI_PPI_DESCRIPTOR_PPI | EFI_PEI_PPI_DESCRIPTOR_TERMINATE_LIST;
    NvdimmCachePpiDescriptor->Guid  = &gNvdimmCachePpiGuid;
    NvdimmCachePpiDescriptor->Ppi   = NvdimmCachePpi;

    ///
    /// Install the Nvdimm Cache PPI
    ///
    Status = PeiServicesInstallPpi (NvdimmCachePpiDescriptor);

    if (Status != EFI_SUCCESS) {
      DEBUG ((DEBUG_ERROR, "Nvdimm Cache PPI Installation failed!!\n"));
      return Status;
    }
    DEBUG ((DEBUG_INFO, "Nvdimm Cache PPI Installed\n"));

    Status = GetNvdimmCacheInfo(NvdimmCachePpi, NvdimmCacheInfo);
    if (Status != EFI_SUCCESS) {
      DEBUG ((DEBUG_ERROR, "Get Nvdimm Cache Info failure- Skipping Nvdimm Cache Hob creation!!\n"));
      return Status;
    }

    //
    // Create HOB for DPMem INFO
    //
    Status = PeiServicesCreateHob (
              EFI_HOB_TYPE_GUID_EXTENSION,
              sizeof (NVDIMM_CACHE_HOB),
              (VOID **) &NvdimmCacheInfoHob
              );
    ASSERT_EFI_ERROR (Status);

    ///
    /// Initialize DPMem HOB data
    NvdimmCacheInfoHob->EfiHobGuidType.Name = gNvdimmCacheInfoHobGuid;
    CopyMem(&NvdimmCacheInfoHob->NvdimmCacheInfo, NvdimmCacheInfo, sizeof(NVDIMM_CACHE_INFO));
    DEBUG ((DEBUG_INFO, "Nvdimm Cache Hob Created\n"));
  }
  else {
    DEBUG ((DEBUG_INFO, "Nvdimm Cache PPI is already installed. Skipping...\n"));
  }
  return Status;
}
