/** @file
  @This file implements the parser functions to parse 3DXP device Mailbox output
  payload and populate the 2LM Info HOB.

@copyright
  INTEL CONFIDENTIAL
  Copyright 2020 Intel Corporation.

  The source code contained or described herein and all documents related to the
  source code ("Material") are owned by Intel Corporation or its suppliers or
  licensors. Title to the Material remains with Intel Corporation or its suppliers
  and licensors. The Material may contain trade secrets and proprietary and
  confidential information of Intel Corporation and its suppliers and licensors,
  and is protected by worldwide copyright and trade secret laws and treaty
  provisions. No part of the Material may be used, copied, reproduced, modified,
  published, uploaded, posted, transmitted, distributed, or disclosed in any way
  without Intel's prior express written permission.

  No license under any patent, copyright, trade secret or other intellectual
  property right is granted to or conferred upon you by disclosure or delivery
  of the Materials, either expressly, by implication, inducement, estoppel or
  otherwise. Any license under such intellectual property rights must be
  express and approved by Intel in writing.

  Unless otherwise agreed by Intel in writing, you may not remove or alter
  this notice or any other notice embedded in Materials by Intel or
  Intel's suppliers or licensors in any way.

  This file contains an 'Intel Peripheral Driver' and is uniquely identified as
  "Intel Reference Module" and is licensed for Intel CPUs and chipsets under
  the terms of your license agreement with Intel or your vendor. This file may
  be modified by the user, subject to additional terms of the license agreement.

@par Specification Reference:
**/
#include <Uefi.h>
#include <Pi/PiBootMode.h>
#include <Pi/PiHob.h>
#include <Library/DebugLib.h>
#include <Library/TwoLmInitLib.h>

/**
This function would issue Identify2lmModuleInfo  Mailbox Command
And parse the output payload to store the required information in 2lm info HOB
to be used by the other boot stages.

@param[in]      TwoLmInfo            2LM HOB which will have necessary values to initialize FMHC.

@retval EFI_SUCCESS            Successfully retrieve 2LM module info
**/
EFI_STATUS
EFIAPI
GetTwoLmModuleInfo (
  IN TWOLM_INFO_HOB *TwoLmInfoHob
)
{
  UINT8                   DeviceReturnStatus;
  UINT8                   OutputPayloadSizeDw;
  DEVICE_2LM_INFORMATION   Device2lmInfo;
  EFI_STATUS               Status;

  DeviceReturnStatus  = 0;
  OutputPayloadSizeDw = sizeof(Device2lmInfo); //or maybe #define size as per the latest FIS version
  // Converting size in DW
  if ((OutputPayloadSizeDw & 0x3) == 0) {
    OutputPayloadSizeDw = OutputPayloadSizeDw / 4;
  } else {
    OutputPayloadSizeDw = (OutputPayloadSizeDw / 4) + 1;
  }


  Status = Identify2lmModuleCommand (&DeviceReturnStatus, (UINT32 *)(&Device2lmInfo), OutputPayloadSizeDw);
  if (Status != EFI_SUCCESS){
    //@tbd check if need to RETRY for certain timeout value
    return Status;
  }
  if (DeviceReturnStatus) { // if the return Status is other than 00h (Success)
    return EFI_DEVICE_ERROR;
  }

  // Getting Device Health Status
  TwoLmInfoHob->TwoLmInfo.DeviceHealthStatus = (UINT16)Device2lmInfo.CAPABILITIES_2LM.DeviceHealthStatus;

  // a. BIOS checks Device Health Status register (byte 139) Device Healthy field (bit[0])
  //    If not healthy (bit[0] = 1), BIOS declare boot fails and boot as 1LM. Else, continue with step b below
  if (TwoLmInfoHob->TwoLmInfo.DeviceHealthStatus & B_DEVICE_HELTH_STATUS) {
    DEBUG ((DEBUG_ERROR, "2LM: Device Health Error, booting in 1LM mode\n"));
    ASSERT (FALSE);
    return EFI_UNSUPPORTED;
  }

  // b.  BIOS checks "Additional Feature Support" register (byte 148-141) "Max Power Consumption Adjustment Supported" field (bit[2])
  //  i.  If this bit is 1b, BIOS shall read bytes 171-164 for platform to determine what is the max allowable power consumption for WLD
  //  ii. BIOS shall send "set_slot_power" message to WLD prior to step 10 (enable 2LM mode) in order to relax the power consumption of the device and increase performance
  //  iii.  If this bit is 0b, continue to next step
  if (Device2lmInfo.CAPABILITIES_2LM.AdditionalFeatureSupport & B_MAX_POWER_CONSUMPTION_ADJUSTMENT_SUPPORTED) {
    DEBUG ((DEBUG_INFO, "2LM: Max Power Consumption Adjustment is Supported\n"));
  }

  // c.  BIOS checks "Additional Feature Support" register (byte 148-141) "Min Power Consumption Adjustment Supported" field (bit[3])
  //  i.  If this bit is 1b, BIOS shall read bytes 175-168 for platform to determine what is the min allowable power consumption for WLD
  //  ii. BIOS shall send "set_slot_power" message to WLD prior to step 10 (enable 2LM mode) in order to relax the power consumption of the device and increase performance
  //  iii.  If this bit is 0b, continue to next step
  if (Device2lmInfo.CAPABILITIES_2LM.AdditionalFeatureSupport & B_MIN_POWER_CONSUMPTION_ADJUSTMENT_SUPPORTED) {
    DEBUG ((DEBUG_INFO, "2LM: Min Power Consumption Adjustment is Supported\n"));
  }

  // Getting the Region Capabilities
  TwoLmInfoHob->TwoLmInfo.PlatformRegionRequired = (UINT8)Device2lmInfo.REGION_CAPABILITIES_Union.REGION_CAPABILITIES_Struct.PlatformRegionRequired;

  // Getting provisioning Status
  TwoLmInfoHob->TwoLmInfo.DeviceProvisioningStatus = Device2lmInfo.CAPABILITIES_2LM.Byte130_Union.Byte_130_Struct.RegionProvisioned;
  // Getting TotalNumberofRegions
  TwoLmInfoHob->TwoLmInfo.TotalNumberofRegions = Device2lmInfo.CAPABILITIES_2LM.Byte130_Union.Byte_130_Struct.TotalNumberOfRegions;
  // Getting encryption Status for 2LM
  TwoLmInfoHob->TwoLmInfo.MemoryEncryptionSupported = (UINT8)Device2lmInfo.MODULE_2LM_SKU_Union.MODULE_2LM_SKU_Struct.MemoryEncryption;

  // Getting max outstanding credit information
  TwoLmInfoHob->TwoLmInfo.Mrts = (UINT8)Device2lmInfo.CAPABILITIES_2LM.Mrts;
  TwoLmInfoHob->TwoLmInfo.Mwss = (UINT8)Device2lmInfo.CAPABILITIES_2LM.Mwss;
  TwoLmInfoHob->TwoLmInfo.Mcsrts = (UINT8)Device2lmInfo.CAPABILITIES_2LM.Mcsrts;

  // Getting device version information
  TwoLmInfoHob->TwoLmInfo.VendorId   = (UINT16)Device2lmInfo.VendorID;
  TwoLmInfoHob->TwoLmInfo.DevId      = (UINT16)Device2lmInfo.DeviceID;
  TwoLmInfoHob->TwoLmInfo.RevisionId = (UINT16)Device2lmInfo.RevisionID;
  TwoLmInfoHob->TwoLmInfo.Ifc        = (UINT16)Device2lmInfo.IFC;
  TwoLmInfoHob->TwoLmInfo.SerialNum  = *(UINT32*)Device2lmInfo.SerialNumber;

  // Getting 2LM Module SKU Information
  TwoLmInfoHob->TwoLmInfo.MemoryModeSupported = (UINT8)Device2lmInfo.MODULE_2LM_SKU_Union.MODULE_2LM_SKU_Struct.MemoryModeSupported;
  TwoLmInfoHob->TwoLmInfo.StorageModeSupported = (UINT8)Device2lmInfo.MODULE_2LM_SKU_Union.MODULE_2LM_SKU_Struct.StorageModeSupported;
  TwoLmInfoHob->TwoLmInfo.PMemModeSupported = (UINT8)Device2lmInfo.MODULE_2LM_SKU_Union.MODULE_2LM_SKU_Struct.PMemModeSupported;
  TwoLmInfoHob->TwoLmInfo.DieSparingCapable = (UINT8)Device2lmInfo.MODULE_2LM_SKU_Union.MODULE_2LM_SKU_Struct.DieSparingCapable;
  // Reading the Random write speed from device
  TwoLmInfoHob->TwoLmInfo.RandomWriteSpeed = (UINT8)Device2lmInfo.CAPABILITIES_2LM.BwReporting_DefaultPowerTPS0.RandomWriteSpeed;
  return EFI_SUCCESS;
}


/**
This function would issue QueryRegion  Mailbox Command
And parse the output payload to store the required information in 2lm info HOB
to be used by the other boot stages.

@param[in]      TwoLmInfo            2LM HOB which will have necessary values to initialize FMHC.

@retval EFI_SUCCESS            Successfully retrieve region info
**/
EFI_STATUS
EFIAPI
GetQueryRegionInfo (
  IN TWOLM_INFO_HOB *TwoLmInfoHob
)
{
  UINT8                       DeviceReturnStatus;
  UINT8                       OutputPayloadSizeDw;
  UINT8                       RegionId;
  REGION_LAYOUT_INFORMATION   RegionLayoutInfo[MAX_REGIONS];
  EFI_STATUS                  Status;
  int                         i;

  DeviceReturnStatus  = 0;
  RegionId            = 0xFF; // To query all the regions
  OutputPayloadSizeDw = sizeof(REGION_LAYOUT_INFORMATION)*MAX_REGIONS; //or maybe #define size as per the latest FIS version
  // Converting size in DW
  if ((OutputPayloadSizeDw & 0x3) == 0) {
    OutputPayloadSizeDw = OutputPayloadSizeDw / 4;
  } else {
    OutputPayloadSizeDw = (OutputPayloadSizeDw / 4) + 1;
  }

  if (!TwoLmInfoHob->TwoLmInfo.DeviceProvisioningStatus) {
    return EFI_DEVICE_ERROR;
  }
  if (TwoLmInfoHob->TwoLmInfo.TotalNumberofRegions > MAX_REGIONS) {
    return EFI_BUFFER_TOO_SMALL;
  }


  Status = QueryRegionLayoutCommand (&DeviceReturnStatus, (UINT32 *)RegionLayoutInfo, OutputPayloadSizeDw, RegionId);

  if (Status != EFI_SUCCESS){
    //@tbd check if need to RETRY for certain timeout value
    return Status;
  }
  if (DeviceReturnStatus) { // if the return Status is other than 00h (Success)
    return EFI_DEVICE_ERROR;
  }

  // Find out region details
  for (i=0; i < MAX_REGIONS; ++i) {
    if (!RegionLayoutInfo[i].DW0Data_Union.DW0Bits.RegionValidindicator) { // Check if region is valid
      continue;  //continue to find next valid region
    }
    switch (RegionLayoutInfo[i].DW0Data_Union.DW0Bits.BiosRegionType) {

      case NVMeStorage:
        if (RegionLayoutInfo[i].DW0Data_Union.DW0Bits.RegionAttr_Mem_Storage != StorageMode){
          return EFI_DEVICE_ERROR;
        }
        TwoLmInfoHob->TwoLmInfo.RegionIdStorage = (UINT8)RegionLayoutInfo[i].DW0Data_Union.DW0Bits.RegionId;
        TwoLmInfoHob->TwoLmInfo.StorageSize = RegionLayoutInfo[i].RegionSize;
        break;

      case VolatileMemory:
        if (RegionLayoutInfo[i].DW0Data_Union.DW0Bits.RegionAttr_Mem_Storage != MemoryMode){
          return EFI_DEVICE_ERROR;
        }
        TwoLmInfoHob->TwoLmInfo.RegionId2lm = (UINT8)RegionLayoutInfo[i].DW0Data_Union.DW0Bits.RegionId;
        TwoLmInfoHob->TwoLmInfo.FarMemSize = RegionLayoutInfo[i].RegionSize;
        TwoLmInfoHob->TwoLmInfo.FarMemFZS  = (UINT8)RegionLayoutInfo[i].DW0Data_Union.DW0Bits.RegionAttr_FastZero;
        break;

      case IntelReservedRegionMemory:
        if (RegionLayoutInfo[i].DW0Data_Union.DW0Bits.RegionAttr_Mem_Storage != MemoryMode){
          return EFI_DEVICE_ERROR;
        }
        TwoLmInfoHob->TwoLmInfo.RegionIdIrr = (UINT8)RegionLayoutInfo[i].DW0Data_Union.DW0Bits.RegionId;
        TwoLmInfoHob->TwoLmInfo.IrrSize = RegionLayoutInfo[i].RegionSize;
        break;

      case PersistenceMemory:
        if (RegionLayoutInfo[i].DW0Data_Union.DW0Bits.RegionAttr_Mem_Storage != MemoryMode){
          return EFI_DEVICE_ERROR;
        }
        TwoLmInfoHob->TwoLmInfo.RegionIdPMem = (UINT8)RegionLayoutInfo[i].DW0Data_Union.DW0Bits.RegionId;
        TwoLmInfoHob->TwoLmInfo.PMemSize = RegionLayoutInfo[i].RegionSize;
        break;

      default:
         break;
    } // End of switch-case

    // Check if this is the last region provisioned
    if (RegionLayoutInfo[i].DW0Data_Union.DW0Bits.LastRegionIndicator) {
      return EFI_SUCCESS;
    }
  } // End of for

return EFI_SUCCESS;
}


/**
This function would issue SetHwParametersCommand  Mailbox Command to set the HW parameters
of far memory device as per internal/external user selection.
HW Parameter must be sent for every PERST cycle (cold or warm boot).
The value is only stick over low power state but not PERST

@param[in]      TwoLmConfig            2LM Config block.

@retval         EFI_SUCCESS            Successful completion of command
**/
EFI_STATUS
EFIAPI
SetHardwareParameters (
   IN TWOLM_PREMEM_CONFIG      *TwoLmPreMemConfig
)
{
  UINT8                   DeviceReturnStatus;
  UINT8                   InputPayloadSizeDw;
  HW_PARAMETERS           HwParameters;
  EFI_STATUS               Status;

  DeviceReturnStatus  = 0;
  InputPayloadSizeDw = sizeof(HwParameters);
  // Converting size in DW
  if ((InputPayloadSizeDw & 0x3) == 0) {
    InputPayloadSizeDw = InputPayloadSizeDw / 4;
  } else {
    InputPayloadSizeDw = (InputPayloadSizeDw / 4) + 1;
  }

  // Set the HwParemeters command buffer as per the user selection
  if (TwoLmPreMemConfig->SetHwParameters) {
    HwParameters.Ltr_L1D2_ThVal   =   (UINT16)TwoLmPreMemConfig->Ltr_L1D2_ThVal;
    HwParameters.Ltr_L1D2_ThScale   =   (UINT16)TwoLmPreMemConfig->Ltr_L1D2_ThScale;
    HwParameters.SysPwrState   =   (UINT8)TwoLmPreMemConfig->SysPwrState;
    HwParameters.MediaDeathNotification   =  (UINT16) TwoLmPreMemConfig->MediaDeathNotification;
    HwParameters.HealthLogNotification   =   (UINT16) TwoLmPreMemConfig->HealthLogNotification;
    HwParameters.TempBelowThrottleNotification   = (UINT16)  TwoLmPreMemConfig->TempBelowThrottleNotification;
    HwParameters.TempAboveThrottleNotification   =  (UINT16) TwoLmPreMemConfig->TempAboveThrottleNotification;
    HwParameters.MissingCommitBitNotification   =  (UINT16) TwoLmPreMemConfig->MissingCommitBitNotification;
  } else {
    return EFI_SUCCESS;
  }


  Status = SetHwParametersCommand (&DeviceReturnStatus, (UINT32 *)&HwParameters, InputPayloadSizeDw);
  if (Status != EFI_SUCCESS){
    return Status;
  }
  if (DeviceReturnStatus) { // if the return Status is other than 00h (Success)
    return EFI_DEVICE_ERROR;
  }

  return EFI_SUCCESS;
}
