/** @file
  Source file for 2LM feature initialization.

@copyright
  INTEL CONFIDENTIAL
  Copyright 2020 Intel Corporation.

  The source code contained or described herein and all documents related to the
  source code ("Material") are owned by Intel Corporation or its suppliers or
  licensors. Title to the Material remains with Intel Corporation or its suppliers
  and licensors. The Material may contain trade secrets and proprietary and
  confidential information of Intel Corporation and its suppliers and licensors,
  and is protected by worldwide copyright and trade secret laws and treaty
  provisions. No part of the Material may be used, copied, reproduced, modified,
  published, uploaded, posted, transmitted, distributed, or disclosed in any way
  without Intel's prior express written permission.

  No license under any patent, copyright, trade secret or other intellectual
  property right is granted to or conferred upon you by disclosure or delivery
  of the Materials, either expressly, by implication, inducement, estoppel or
  otherwise. Any license under such intellectual property rights must be
  express and approved by Intel in writing.

  Unless otherwise agreed by Intel in writing, you may not remove or alter
  this notice or any other notice embedded in Materials by Intel or
  Intel's suppliers or licensors in any way.

  This file contains an 'Intel Peripheral Driver' and is uniquely identified as
  "Intel Reference Module" and is licensed for Intel CPUs and chipsets under
  the terms of your license agreement with Intel or your vendor. This file may
  be modified by the user, subject to additional terms of the license agreement.

@par Specification Reference:
**/
#include <Library/TwoLmInitLib.h>
#include <Library/NvdimmCacheInfoLib.h>
#include <Register/TwoLmRegs.h>
#include <Library/PmcPrivateLib.h>
#include <Library/PeiServicesLib.h>
#include <Library/BaseMemoryLib.h>
#include <Library/PeiServicesTablePointerLib.h>
#include <IndustryStandard/Pci22.h>
#include <Include/PcieRegs.h>
#include <Register/SaRegsHostBridge.h>
#include <Library/TimerLib.h>
#include <Library/PciExpressHelpersLib.h>
#include <Library/PcieHelperLib.h>
#include <CpuRegs.h>
#include <Register/CommonMsr.h>
#include <CpuSbInfo.h>
#include <Library/IoLib.h>
#include <Library/CpuPlatformLib.h>
#include <Register/TwoLmRegs.h>
#include <Register/CpuPcieRegs.h>
#include <Library/PeiTwoLmInitFruLib.h>
#include <Library/RngLib.h>
#include <Library/MsrFruLib.h>
#include <Library/SystemTimeLib.h>


#define VC_128PHASE_ARBITRATION     0x4   //100b for 128 phases
#define RETRY_COUNT                 3


/**
  Wrapper for BzmMemoryInitDoneCallback.

  @param[in] PeiServices          General purpose services available to every PEIM.
  @param[in] NotifyDescriptor     The notification structure this PEIM registered on install.
  @param[in] Ppi                  The memory discovered PPI.  Not used.

  @retval EFI_SUCCESS             Succeeds.
          EFI_UNSUPPORTED         if the feature is not supported.
**/
EFI_STATUS
EFIAPI
MemoryInitDoneCallback (
  IN  EFI_PEI_SERVICES             **PeiServices,
  IN  EFI_PEI_NOTIFY_DESCRIPTOR    *NotifyDescriptor,
  IN  VOID                         *Ppi
  );

/**
  BZM Initialization is done after the MrcMemoryInitDonePpi is produced.

  @param[in] PeiServices          General purpose services available to every PEIM.
  @param[in] NotifyDescriptor     The notification structure this PEIM registered on install.
  @param[in] Ppi                  The memory discovered PPI.  Not used.

  @retval EFI_SUCCESS             Succeeds.
          EFI_UNSUPPORTED         if the feature is not supported.
**/
EFI_STATUS
EFIAPI
BzmMemoryInitDoneCallback (
  IN  EFI_PEI_SERVICES             **PeiServices,
  IN  EFI_PEI_NOTIFY_DESCRIPTOR    *NotifyDescriptor,
  IN  VOID                         *Ppi
  );
static EFI_PEI_NOTIFY_DESCRIPTOR  gMemInitDoneNotifyList[] = {
  {
    EFI_PEI_PPI_DESCRIPTOR_NOTIFY_CALLBACK | EFI_PEI_PPI_DESCRIPTOR_TERMINATE_LIST,
    &gMrcMemoryInitDonePpiGuid,
    MemoryInitDoneCallback
  }
};


/**
This function initializes Far Memory Host Controller.

@param[in]      TwoLmInfo            2LM HOB which will have necessary values to initialize FMHC.
@param[in]      TwoLmConfig              2LM Config block.

@retval EFI_SUCCESS            Successfully initialized Far Memory Host Controller.
**/
STATIC
EFI_STATUS
EFIAPI
FmhcInit(
   IN TWOLM_INFO               *TwoLmInfo
  ,IN TWOLM_PREMEM_CONFIG      *TwoLmConfig
)
{
  UINT8             PmControlStatusOffset;
  UINT8             PcieCapabilityOffset;
  UINT16            MfvcCapabilityOffset;
  UINT16            L1PmCapabilityOffset;
  UINT16            SecondaryPcieCapabilityOffset;
  UINT16            DataLinkFeatureCapabilityOffset;
  UINT16            PhyLayerCapabilityOffset;
  UINT16            PhyLayerMarginingCapabilityOffset;
  UINT32            Data32;
  UINT32            Data32Or;
  UINT32            Data32And;
  UINT32            DeviceBaseAddress;

  DEBUG ((DEBUG_INFO, "2LM: FMHC Initialization Start\n"));

  //
  // Program Persistent Memory register if it is enabled.
  //
  if (TwoLmInfo->PMemSize != 0) {
    CpuRegbarWrite64 (CPU_SB_PID_FMHC, R_FMHC_PMRBL, LShiftU64(TwoLmInfo->FarMemSize, 20));
    CpuRegbarWrite64 (CPU_SB_PID_FMHC, R_FMHC_PMRLL, (LShiftU64(TwoLmInfo->FarMemSize, 20) + LShiftU64(TwoLmInfo->PMemSize, 20) - 1));
  }

  //
  // Enable PMC reqeust
  //
  Data32 = CpuRegbarRead32 (CPU_SB_PID_FMHC, R_FMHC_PCE);
  CpuRegbarWrite32 (CPU_SB_PID_FMHC, R_FMHC_PCE, (Data32 | B_FMHC_PCE_PMCRE));

  //
  // Initialize FM Max Supported Outstanding Request register
  //
  Data32And = ~(UINT32)(B_FMHC_FMAXOUTR_MRTS_MASK | B_FMHC_FMAXOUTR_MWSS_MASK | B_FMHC_FMAXOUTR_MCSRTS_MASK);
  Data32Or = B_FMHC_FMAXOUTR_LOADCV | (TwoLmInfo->Mrts << B_FMHC_FMAXOUTR_MRTS_OFFSET) | (TwoLmInfo->Mwss << B_FMHC_FMAXOUTR_MWSS_OFFSET) | (TwoLmInfo->Mcsrts << B_FMHC_FMAXOUTR_MCSRTS_OFFSET);
  CpuRegbarAndThenOr32 (CPU_SB_PID_FMHC, R_FMHC_FMAXOUTR, Data32And, Data32Or);

  //
  // Initize 2LM, PMem and IRR region IDs
  //
  Data32And = ~(UINT32)(B_FMHC_FMAP_PMRID_MASK | B_FMHC_FMAP_IRRID_MASK | B_FMHC_FMAP_MRID_MASK);
  Data32Or = (TwoLmInfo->RegionIdPMem << B_FMHC_FMAP_PMRID_OFFSET) | (TwoLmInfo->RegionIdIrr << B_FMHC_FMAP_IRRID_OFFSET) | (TwoLmInfo->RegionId2lm << B_FMHC_FMAP_MRID_OFFSET);
  CpuRegbarAndThenOr32 (CPU_SB_PID_FMHC, R_FMHC_FMAP, Data32And, Data32Or);


  //
  // Find Power Management Status/Control register capability offset, PCIe capability offset, Multi Function virtual channel extended capability offset,
  // L1 PM substates extended capability offset and update these values in RCIEPPCIE and RCIEPL1PMMFVC registers.
  //
  PmControlStatusOffset  = PcieFindCapId (SA_SEG_NUM, SA_PEG_BUS_NUM, TWOLM_DEVICE_NUM, TWOLM_FUNC_0, PM_CAPID);
  PcieCapabilityOffset  = PcieFindCapId (SA_SEG_NUM, SA_PEG_BUS_NUM, TWOLM_DEVICE_NUM, TWOLM_FUNC_0, PCIE_CAPID);
  MfvcCapabilityOffset  = PcieFindExtendedCapId (SA_SEG_NUM, SA_PEG_BUS_NUM, TWOLM_DEVICE_NUM, TWOLM_FUNC_0, MFVC_CAPID);
  L1PmCapabilityOffset = PcieFindExtendedCapId (SA_SEG_NUM, SA_PEG_BUS_NUM, TWOLM_DEVICE_NUM, TWOLM_FUNC_0, L1PM_CAPID);

  if ( PmControlStatusOffset && PcieCapabilityOffset ) {
    Data32And = ~(UINT32) (B_FMHC_RCIEPPCIE_PMCSR_CAPOFF_MASK | B_FMHC_RCIEPPCIE_DEVTYP_MASK | B_FMHC_RCIEPPCIE_PCIE_CAPOFF_MASK);
    Data32Or = (PmControlStatusOffset << B_FMHC_RCIEPPCIE_PMCSR_CAPOFF_OFFSET) | (V_FMHC_RCIEPPCIE_DEVTYP << B_FMHC_RCIEPPCIE_DEVTYP_OFFSET) | (PcieCapabilityOffset << B_FMHC_RCIEPPCIE_PCIE_CAPOFF_OFFSET) | B_FMHC_RCIEPPCIE_PCIE_CAPOFFEN;
    CpuRegbarAndThenOr32 (CPU_SB_PID_FMHC, R_FMHC_RCIEPPCIE, Data32And, Data32Or);

    Data32And = ~(UINT32) (B_FMHC_RCIEPPCIE2_PCIE_CAPOFFLIM_MASK);
    Data32Or = ((PcieCapabilityOffset + PCIE_CAP_SIZE -1)  << B_FMHC_RCIEPPCIE2_PCIE_CAPOFFLIM_OFFSET);
    CpuRegbarAndThenOr32 (CPU_SB_PID_FMHC, R_FMHC_RCIEPPCIE2, Data32And, Data32Or);
  }

  if ( MfvcCapabilityOffset && L1PmCapabilityOffset ) {
    Data32And = ~(UINT32) (B_FMHC_RCIEPL1PMMFVC_MFVC_CAPOFF_MASK | B_FMHC_RCIEPL1PMMFVC_L1PM_CAPOFF_MASK);
    Data32Or = B_FMHC_RCIEPL1PMMFVC_MFVC_CAPOFFEN | (MfvcCapabilityOffset << B_FMHC_RCIEPL1PMMFVC_MFVC_CAPOFF_OFFSET) | B_FMHC_RCIEPL1PMMFVC_L1PM_CAPOFFEN | (L1PmCapabilityOffset << B_FMHC_RCIEPL1PMMFVC_L1PM_CAPOFF_OFFSET);
    CpuRegbarAndThenOr32 (CPU_SB_PID_FMHC, R_FMHC_RCIEPL1PMMFVC, Data32And, Data32Or);

    Data32And = ~(UINT32) (B_FMHC_RCIEPL1PMMFVC2_MFVC_CAPOFFLIM_MASK | B_FMHC_RCIEPL1PMMFVC2_L1PM_CAPOFFLIM_MASK);
    Data32Or = ((MfvcCapabilityOffset + MFVC_CAP_SIZE -1)  << B_FMHC_RCIEPL1PMMFVC2_MFVC_CAPOFFLIM_OFFSET) | ((L1PmCapabilityOffset + L1PM_CAP_SIZE - 1) << B_FMHC_RCIEPL1PMMFVC2_L1PM_CAPOFFLIM_OFFSET);
    CpuRegbarAndThenOr32 (CPU_SB_PID_FMHC, R_FMHC_RCIEPL1PMMFVC2, Data32And, Data32Or);
  }

  SecondaryPcieCapabilityOffset  = PcieFindExtendedCapId (SA_SEG_NUM, SA_PEG_BUS_NUM, TWOLM_DEVICE_NUM, TWOLM_FUNC_0, SEC_PCIE_CAPID);
  DataLinkFeatureCapabilityOffset  = PcieFindExtendedCapId (SA_SEG_NUM, SA_PEG_BUS_NUM, TWOLM_DEVICE_NUM, TWOLM_FUNC_0, DATA_LINK_FEATURE_CAPID);

  if ( SecondaryPcieCapabilityOffset && DataLinkFeatureCapabilityOffset ) {
    Data32And = ~(UINT32) (B_FMHC_RCIEPLCAPAB_CAPSTRUCT_B_CAPOFF_MASK | B_FMHC_RCIEPLCAPAB_CAPSTRUCT_A_CAPOFF_MASK);
    Data32Or = B_FMHC_RCIEPLCAPAB_CAPSTRUCT_B_CAPOFFEN | (DataLinkFeatureCapabilityOffset << B_FMHC_RCIEPLCAPAB_CAPSTRUCT_B_CAPOFF_OFFSET) | B_FMHC_RCIEPLCAPAB_CAPSTRUCT_A_CAPOFFEN | (SecondaryPcieCapabilityOffset << B_FMHC_RCIEPLCAPAB_CAPSTRUCT_A_CAPOFF_OFFSET);
    CpuRegbarAndThenOr32 (CPU_SB_PID_FMHC, R_FMHC_RCIEPLCAPAB, Data32And, Data32Or);

    Data32And = ~(UINT32) (B_FMHC_RCIEPLCAPAB_CAPSTRUCT_B_CAPOFFLIM_MASK | B_FMHC_RCIEPLCAPAB_CAPSTRUCT_A_CAPOFFLIM_MASK);
    Data32Or = ((SecondaryPcieCapabilityOffset + SEC_PCIE_CAP_SIZE -1)  << B_FMHC_RCIEPLCAPAB_CAPSTRUCT_B_CAPOFFLIM_OFFSET) | ((DataLinkFeatureCapabilityOffset + DATA_LINK_FEATURE_CAP_SIZE - 1) << B_FMHC_RCIEPLCAPAB_CAPSTRUCT_A_CAPOFFLIM_OFFSET);
    CpuRegbarAndThenOr32 (CPU_SB_PID_FMHC, R_FMHC_RCIEPLCAPAB2, Data32And, Data32Or);
  }

  PhyLayerCapabilityOffset  = PcieFindExtendedCapId (SA_SEG_NUM, SA_PEG_BUS_NUM, TWOLM_DEVICE_NUM, TWOLM_FUNC_0, PHY_LAYER_16_CAPID);
  PhyLayerMarginingCapabilityOffset  = PcieFindExtendedCapId (SA_SEG_NUM, SA_PEG_BUS_NUM, TWOLM_DEVICE_NUM, TWOLM_FUNC_0, PHY_LAYER_16_MARGINING_CAPID);

  if ( PhyLayerCapabilityOffset && PhyLayerMarginingCapabilityOffset ) {
    Data32And = ~(UINT32) (B_FMHC_RCIEPLCAPCD_CAPSTRUCT_D_CAPOFF_MASK | B_FMHC_RCIEPLCAPCD_CAPSTRUCT_C_CAPOFF_MASK);
    Data32Or = B_FMHC_RCIEPLCAPCD_CAPSTRUCT_D_CAPOFFEN | (PhyLayerMarginingCapabilityOffset << B_FMHC_RCIEPLCAPCD_CAPSTRUCT_D_CAPOFF_OFFSET) | B_FMHC_RCIEPLCAPCD_CAPSTRUCT_C_CAPOFFEN | (PhyLayerCapabilityOffset << B_FMHC_RCIEPLCAPCD_CAPSTRUCT_C_CAPOFF_OFFSET);
    CpuRegbarAndThenOr32 (CPU_SB_PID_FMHC, R_FMHC_RCIEPLCAPAB, Data32And, Data32Or);

    Data32And = ~(UINT32) (B_FMHC_RCIEPLCAPCD_CAPSTRUCT_D_CAPOFFLIM_MASK | B_FMHC_RCIEPLCAPCD_CAPSTRUCT_C_CAPOFFLIM_MASK);
    Data32Or = ((PhyLayerCapabilityOffset + PHY_LAYER_16_CAP_SIZE - 1) << B_FMHC_RCIEPLCAPCD_CAPSTRUCT_D_CAPOFFLIM_OFFSET) | ((PhyLayerMarginingCapabilityOffset + PHY_LAYER_16_MARGINING_CAP_SIZE - 1) << B_FMHC_RCIEPLCAPCD_CAPSTRUCT_C_CAPOFFLIM_OFFSET);
    CpuRegbarAndThenOr32 (CPU_SB_PID_FMHC, R_FMHC_RCIEPLCAPAB2, Data32And, Data32Or);
  }

  //
  // Program TPower on Pmstates register based on the value from Willard device.
  //
  CpuRegbarWrite32 (CPU_SB_PID_FMHC, R_TPOWER_ON_PMSTATES, (TwoLmInfo->TpowerOnSupport & (~(BIT2 | BIT10))));

  //
  // Program FMHC internal clock gating
  //
  Data32And = ~(UINT32) (B_FMHC_GC03_DCG_AES_DIS | B_FMHC_GC03_DCG_MTU_DIS | B_FMHC_GC03_DCG_STU_DIS | B_FMHC_GC03_DCG_CMIEP_DIS | B_FMHC_GC03_DCG_HBIF_DIS | B_FMHC_GC03_DCG_SBREG_DIS | B_FMHC_GC03_DCG_PMU_DIS | B_FMHC_GC03_DCG_IPDYN_DIS);
  CpuRegbarAndThenOr32 (CPU_SB_PID_FMHC, R_FMHC_GC03, Data32And, 0);

  Data32And = ~(UINT32) (B_FMHC_PRI_CDCB_CCD | B_FMHC_PRI_CDCB_DDCGD);
  CpuRegbarAndThenOr32 (CPU_SB_PID_FMHC, R_FMHC_PRI_CDCB, Data32And, 0);

  Data32And = ~(UINT32) (B_FMHC_SB_CDCB_CCD | B_FMHC_SB_CDCB_DDCGD);
  CpuRegbarAndThenOr32 (CPU_SB_PID_FMHC, R_FMHC_SB_CDCB, Data32And, 0);

  // BIOS to program PCIE_CAPLCTL_7_0SHADOW field [7:0] in SoCPCIECAP (0xc8) register.
  // BIOS shall program this field to shadow the content of FMSS PCI Express controller Link Control Register (0x50) bits[7:0].
  // BIOS does not need to update bit[31:8] as the default values match the FMSS PCIe root port's L1SCTL2 and LCTL  reg. offsets.
  DeviceBaseAddress = PCI_SEGMENT_LIB_ADDRESS (SA_SEG_NUM, SA_PEG_BUS_NUM, SA_PEG3_DEV_NUM, SA_PEG3_FUN_NUM, 0);
  Data32 = PegPciSegmentRead32 (DeviceBaseAddress + R_PCIE_LCTL); // Read PCIe Rp controller's link control reg
  Data32And = (UINT32)~(B_FMHC_PCIE_CAPLCTL_7_0SHADOW_MASK);
  Data32Or = (Data32 & 0x000000FF); // only [7:0] bit field
  CpuRegbarAndThenOr32 (CPU_SB_PID_FMHC, R_FMHC_SoCPCIECAP, Data32And, Data32Or);
  DEBUG ((DEBUG_INFO, "2LM: Value for FMHC Reg SoCPCIECAP is 0X %x\n",CpuRegbarRead32(CPU_SB_PID_FMHC, R_FMHC_SoCPCIECAP)));

  // BIOS writes "0000 0800h" to FMHC MMIO PGCU.
  // Meant to enable PGCB clock gating & wait for 8 clks after idle before clk-gating.
  CpuRegbarWrite32 (CPU_SB_PID_FMHC, R_FMHC_PCGU, 0x00000800);

  // FMHC time outs on handshakes with 2LM Dev or FMSS PCIe controller
  // GC[23:4] = 22222h, The values shall be programmed to 2ms which allows Pcode to time out first and then FMHC times out.
  Data32And = (UINT32)~(B_FMHC_GC_TIMEOUTS_MASK);
  Data32Or  = (0x22222 << B_FMHC_GC_TIMEOUTS_OFFSET); //Bits [23:4]
  CpuRegbarAndThenOr32 (CPU_SB_PID_FMHC, R_FMHC_GC, Data32And, Data32Or);
  //
  // Lock FMHC registers and REGACC register itself
  //
  CpuRegbarOr32 (CPU_SB_PID_FMHC, R_FMHC_REGACC, B_FMHC_REGACC_MAPLCK | B_FMHC_REGACC_RMPBARLCK | B_FMHC_REGACC_PMBARLCK | B_FMHC_REGACC_LCK);

  return EFI_SUCCESS;
}


/**
This function initializes Willard device for Powermanagent and 2LM enable.

@param[in]      TwoLmConfig            2LM Config block.
@param[in]      TWOLM_INFO_HOB         *TwoLmInfoHob

@retval         EFI_SUCCESS            Successfully initialized Willard device.
**/
STATIC
EFI_STATUS
EFIAPI
WillardInit (
  IN TWOLM_PREMEM_CONFIG      *TwoLmPreMemConfig,
  IN TWOLM_INFO_HOB           *TwoLmInfoHob
  )
{
  UINT32            Data;
  UINT32            SocData;
  UINT16            SocL1PmCapabilityOffset;
  UINT32            SocValue;
  UINT32            SocScale;
  UINT32            DevValuePm4;
  UINT32            DevScalePm4;
  UINT32            DevValuePm5;
  UINT32            DevScalePm5;
  UINT32            SocBaseAddress;
  UINT8             DeviceReturnStatus;
  EFI_BOOT_MODE     BootMode;

  EFI_STATUS  Status;

  DEBUG ((DEBUG_INFO, "2LM: WillardDeviceInit Start\n"));

  //
  // 1. Read VDM Device HW Status register (150ms for now from platform Power up)
  //
  Status = RegReadVdm (R_SA_FMSS_VDM_MAIL_BOX_DEVHWS, &Data);
  ASSERT_EFI_ERROR (Status);
  if (!(Data & B_SA_FMSS_VDM_MAIL_BOX_DEVHWS_FWLOADED)) {
    DEBUG ((DEBUG_INFO, "2LM: Device FW error, booting in 1LM mode\n"));
    ASSERT (FALSE);
    return EFI_UNSUPPORTED;
  }

  //
  // 2.  BIOS check RID for VDM & mailbox, fail then boot as 1LM
  //
  Status = RegReadVdm (R_SA_FMSS_VDM_MAIL_BOX_RID, &Data);
  ASSERT_EFI_ERROR (Status);
  if (((Data & (B_MB_INTERFACE_RID_MASK << B_MB_INTERFACE_RID_OFFSET)) != MB_INTERFACE_RID)
   || ((Data & (B_VDM_INTERFACE_RID_MASK << B_VDM_INTERFACE_RID_OFFSET)) != VDM_INTERFACE_RID)) {
    DEBUG ((DEBUG_INFO, "2LM: Device Rivision error, booting in 1LM mode\n"));
    ASSERT (FALSE);
    return EFI_UNSUPPORTED;
  }

  // 3.  VDM write to PSC register (0x0014h) for every boot :
  //  a.  Write 10 to bit[31:30] if booting from S3 resume.
  //  b.  Write 01 to bit[31:30] for all other boots.
  //  c.  Write '000' to bit[2:0] as indicates not entering low power state
  // Note : BIOS must write to bit[31:30] of the PSC register for every boot.
  Status = PeiServicesGetBootMode (&BootMode);
  ASSERT_EFI_ERROR (Status);

  Status = RegReadVdm (R_SA_FMSS_VDM_MAIL_BOX_POWER_STATE_CONTROL, &Data);
  ASSERT_EFI_ERROR (Status);
  Data &= ~(BIT31 | BIT30 | BIT2 | BIT1 | BIT0);
  if (BootMode == BOOT_ON_S3_RESUME) {
    Status = RegWriteVdm (R_SA_FMSS_VDM_MAIL_BOX_POWER_STATE_CONTROL, (Data | BIT31));
    ASSERT_EFI_ERROR (Status);
  }
  else {
    Status = RegWriteVdm (R_SA_FMSS_VDM_MAIL_BOX_POWER_STATE_CONTROL, (Data | BIT30));
    ASSERT_EFI_ERROR (Status);
  }

  //
  // 4.  Send VDM admin command Identify 2LM Module to check device status
  //
  Status = GetTwoLmModuleInfo (TwoLmInfoHob);
  if (!TwoLmInfoHob->TwoLmInfo.DeviceProvisioningStatus) {
    TwoLmPreMemConfig->MemBootMode = MEM_BOOT_MODE_PROVISIONING;
    TwoLmInfoHob->TwoLmInfo.CurrentMemBootMode = MEM_BOOT_MODE_PROVISIONING;
    //Reseting Astro Reg to 0
    AstroInit (&TwoLmInfoHob->TwoLmInfo);
    return EFI_UNSUPPORTED;
  }

  //
  // 5. Send VDM admin command Query Region Layout
  //
  GetQueryRegionInfo (TwoLmInfoHob);

  TwoLmPreMemConfig->FarMemorySize =  TwoLmInfoHob->TwoLmInfo.FarMemSize;
  if (!TwoLmInfoHob->TwoLmInfo.FarMemSize) {
    return EFI_UNSUPPORTED;
  }

  TwoLmPreMemConfig->PMemSize =  TwoLmInfoHob->TwoLmInfo.PMemSize;

  TwoLmInfoHob->TwoLmInfo.PMemBase = LShiftU64(TwoLmInfoHob->TwoLmInfo.FarMemSize, 20); // In bytes

  // 6.  BIOS issues Clear region FIS command
  //  a.  Required if Identify FIS command shows there is a provisioned Volatile Memory region
  //  b.  This step is required for BIOS bootup for cold boot, warm reset, S4 or S5 low power cases only.
  // Note:If BZM is not supported, issuing the Clear Region command is not technically required,
  // But if there is not a latency impact to issuing Clear Region regardless just leave it on to the volatile memory region.
  // It adds another layer of security (in addition to the FME key being rotated).
  //
  if ((BootMode != BOOT_ON_S3_RESUME) && (BootMode != BOOT_ON_FLASH_UPDATE)) {
    // Issue Clear region FIS comand for 2LM region
    if (!(TwoLmInfoHob->TwoLmInfo.FarMemFZS)) {
      DEBUG ((DEBUG_INFO, "Fast Zero is not supported for 2LM region\n"));
      ASSERT (FALSE);
    }
    Status = ClearRegionCommand (&DeviceReturnStatus, TwoLmInfoHob->TwoLmInfo.RegionId2lm);
    if (Status != EFI_SUCCESS){
      DEBUG ((DEBUG_INFO, "Mailbox Communication Error\n"));
      ASSERT (FALSE);
      return Status;
    }
    if (DeviceReturnStatus) { // if the return Status is other than 00h (Success)
      DEBUG ((DEBUG_INFO, "3DXP Device Error\n"));
      ASSERT (EFI_DEVICE_ERROR);
      return EFI_DEVICE_ERROR;
    }
  }

  //
  // 7. VDM admin command write set log page -> HW Parameters
  //
  if (BootMode != BOOT_ON_S3_RESUME) {
    Status = SetHardwareParameters (
             TwoLmPreMemConfig
             );
    ASSERT_EFI_ERROR (Status);
  }

  //
  // 8.  VDM read device Tpoweron supported and compare with SOC program with max value to VDM Tpoweron value register
  //  a.  Read VDM 2LM L1.2 Tpoweron Support register (offset 30h)
  //  b.  Compare with SOC Tpoweron values
  //  c.  Write VDM 2LM L1,2 Tpoweron Control register (offset 34h)
  //  d.  write the max value to host as well
  Status = RegReadVdm (R_SA_FMSS_VDM_MAIL_BOX_TPOWER_ON_SUPPORT, &Data);
  ASSERT_EFI_ERROR (Status);

  DevScalePm4 =  Data & (B_T_POWER_ON_SCALE_PM4_MASK << B_T_POWER_ON_SCALE_PM4_OFFSET);
  DevScalePm5 =  Data & (B_T_POWER_ON_SCALE_PM5_MASK << B_T_POWER_ON_SCALE_PM5_OFFSET);

  DevValuePm4 =  Data & (B_T_POWER_ON_VALUE_PM4_MASK << B_T_POWER_ON_VALUE_PM4_OFFSET);
  DevValuePm5 =  Data & (B_T_POWER_ON_VALUE_PM5_MASK << B_T_POWER_ON_VALUE_PM5_OFFSET);

  SocBaseAddress = PCI_SEGMENT_LIB_ADDRESS (SA_SEG_NUM, SA_MC_BUS, SA_PEG3_DEV_NUM, 0, 0);
  SocL1PmCapabilityOffset = PcieFindExtendedCapId (SA_SEG_NUM, SA_MC_BUS, TWOLM_DEVICE_NUM, TWOLM_FUNC_0, L1PM_CAPID);
  SocData = PegPciSegmentRead32 (SocBaseAddress + SocL1PmCapabilityOffset + R_L1PM_SUBSTATES_CAPABILITIES_REGISTER);
  SocScale = SocData & (B_PORT_T_POWER_ON_SCALE_MASK << B_PORT_T_POWER_ON_SCALE_OFFSET);
  SocValue = SocData & (B_PORT_T_POWER_ON_VALUE_MASK << B_PORT_T_POWER_ON_VALUE_OFFSET);

  // PM4
  if (((SocScale + 1) * SocValue) > ((DevScalePm4 + 1) * DevValuePm4)) { // program with the soc settings
    Data |= (SocScale << B_T_POWER_ON_SCALE_PM4_OFFSET) | (SocValue << B_T_POWER_ON_VALUE_PM4_OFFSET);
  } else { // program dev with the dev settings
    Data |= (DevScalePm4 << B_T_POWER_ON_SCALE_PM4_OFFSET) | (DevValuePm4 << B_T_POWER_ON_VALUE_PM4_OFFSET);
  }
  // PM5
  if (((SocScale + 1) * SocValue) > ((DevScalePm5 + 1) * DevValuePm5)) { // program with the soc settings
    Data |= (SocScale << B_T_POWER_ON_SCALE_PM5_OFFSET) | (SocValue << B_T_POWER_ON_VALUE_PM5_OFFSET);
  } else { // program dev with the dev settings
    Data |= (DevScalePm5 << B_T_POWER_ON_SCALE_PM5_OFFSET) | (DevValuePm5 << B_T_POWER_ON_VALUE_PM5_OFFSET);
  }

  Status = RegWriteVdm (R_SA_FMSS_VDM_MAIL_BOX_TPOWER_ON_CONTROL, Data);
  ASSERT_EFI_ERROR (Status);

  // write the max value to host as well
  if (((DevScalePm5 + 1) * DevValuePm5) > ((DevScalePm4 + 1) * DevValuePm4)) { // program with the soc settings
    SocData |= (DevScalePm5 << B_PORT_T_POWER_ON_SCALE_OFFSET) | (DevValuePm5 << B_PORT_T_POWER_ON_VALUE_OFFSET);
  } else { // program dev with the dev settings
    SocData |= (DevScalePm4 << B_PORT_T_POWER_ON_SCALE_OFFSET) | (DevValuePm4 << B_PORT_T_POWER_ON_VALUE_OFFSET);
  }
  PegPciSegmentWrite32 (SocBaseAddress + SocL1PmCapabilityOffset + R_L1PM_SUBSTATES_CAPABILITIES_REGISTER, SocData);

  //
  // 9.  VDM write to FMSS Host BDF register (offset 0xCh) with host BDF value if needed. Default value is 0.
  // Note that BIOS only needs to program this VDM register if host BDF is non-zero.

  // 10.  VDM write to FMSS Power Management Mode register (offset 10h)
  //  a.  Set FMPMMEN bit to 1 (enable 2LM Power Management Mode), this bit is set after FMHC init
  //  b.  Allow customer optional to set NVMe_hold_disable bit (default to 0)
  //
  Status = RegReadVdm (R_SA_FMSS_VDM_MAIL_BOX_FMPMM, &Data);
  ASSERT_EFI_ERROR (Status);
  Data |= TwoLmPreMemConfig->NVMeHoldDisableBit &  B_SA_FMSS_VDM_MAIL_BOX_NVME_HOLD_DISABLE;
  Status = RegWriteVdm (R_SA_FMSS_VDM_MAIL_BOX_FMPMM, Data);
  ASSERT_EFI_ERROR (Status);

  return Status;
}


/**
This function initializes 2LM specific config registers of 2LM Device PCIe Function 0 and Function 7.
As per Willard  BWG v1.52

@param[in]      TwoLmConfig            2LM Config block.

@retval         EFI_SUCCESS            Successfully initialized 3DXP Device's PCIe Controller.
@retval         EFI_NOT_FOUND          Error condition when the RP is not enabled or the capability is not found.
**/
STATIC
EFI_STATUS
EFIAPI
WillardPcieDeviceInit (
  IN TWOLM_PREMEM_CONFIG  *TwoLmPreMemConfig
)
{
  UINT32            DeviceBaseAddress;
  UINT32            SocBaseAddress;
  UINT32            Data32Or;
  UINT16            Data16Or;
  UINT16            Data16And;
  UINT16            MfvcCapabilityOffset;
  UINT8             DevPcieCapabilityOffset;
  UINT8             SocPcieCapabilityOffset;
  UINT16            SocL1PmCapabilityOffset;
  UINT16            DevL1PmCapabilityOffset;
  UINT8             Mps;
  UINT32            SocValue;
  UINT32            SocScale;
  UINT32            DevValue;
  UINT32            DevScale;
  UINT8             i;
  UINT8             j;

  DEBUG ((DEBUG_INFO, "2LM: WillardPcieDeviceInit Start\n"));

  // Get the Soc base address for Pcie North Port (Peg60)
  SocBaseAddress = PCI_SEGMENT_LIB_ADDRESS (SA_SEG_NUM, SA_MC_BUS, SA_PEG3_DEV_NUM, 0, 0);
  if (PegPciSegmentRead16 (SocBaseAddress + PCI_VENDOR_ID_OFFSET) == 0xFFFF) {
    return EFI_NOT_FOUND;
  }

  //
  // Following configurations are required for Fun7
  //
  DeviceBaseAddress = PCI_SEGMENT_LIB_ADDRESS (SA_SEG_NUM, SA_PEG_BUS_NUM, TWOLM_DEVICE_NUM, TWOLM_FUNC_7, 0);
  if (PegPciSegmentRead16 (DeviceBaseAddress + PCI_VENDOR_ID_OFFSET) == 0xFFFF) {
    return EFI_NOT_FOUND;
  }

  //
  // Fun7: Program Base Address Register (10h to 14h):BAR[63:56]=80h BAR[55:0] = 00000000000000h
  // note: just configuring the 64bit Bar register here, there won't be any 64bit mmio access in PEI phase.
  //

  PegPciSegmentAnd32 (DeviceBaseAddress + PCI_BASE_ADDRESSREG_OFFSET, 0);
  PegPciSegmentOr32 (DeviceBaseAddress + PCI_BASE_ADDRESSREG_OFFSET + 0x4, BIT31);


  //
  // Fun7: Program MSE and SERRE to 1 in command register (offset 04h)
  //
  Data32Or = (UINT32)(EFI_PCI_COMMAND_SERR + EFI_PCI_COMMAND_MEMORY_SPACE);
  PegPciSegmentOr32 (DeviceBaseAddress + PCI_COMMAND_OFFSET, Data32Or);

  //
  // 2. Configure PCIe Express Capability Structure as below
  //
  DevPcieCapabilityOffset  = PegPcieFindCapId (SA_SEG_NUM, SA_PEG_BUS_NUM, TWOLM_DEVICE_NUM, TWOLM_FUNC_7, PCIE_CAPID);

  SocPcieCapabilityOffset  = PegPcieFindCapId (SA_SEG_NUM, SA_MC_BUS, SA_PEG3_DEV_NUM, 0, PCIE_CAPID);

    //
    // Fun7: Device Control Register (08h):

    // Max_payload_size: Read MPS support from both SOC and Device and program the common min to PF7 (expected must be 512B = 010b)
  if (DevPcieCapabilityOffset) {
    if (0) { // this is a temp WA to avoid Mps assert. Need to enable this check when Mps programming of Peg60 controller is moved to pre-mem.
      SocValue = PegPciSegmentRead32 (SocBaseAddress + SocPcieCapabilityOffset + R_DEVICE_CAPABILITIES_REGISTER) & 0x7;
      DevValue = PegPciSegmentRead32 (DeviceBaseAddress + DevPcieCapabilityOffset + R_DEVICE_CAPABILITIES_REGISTER)& 0x7;
      Mps = (UINT8)(SocValue < DevValue ? SocValue : DevValue); // Mps is set to the common minimum
      if ( Mps < 0x2) {
        DEBUG ((DEBUG_INFO, "2LM: Max Payload size is less than 512B\n"));
        ASSERT (FALSE);
        return EFI_UNSUPPORTED;
      }
    } else {
      Mps = 0x2; // Set to 512B
    }
    // Correctable error reporting enable = 1b, Non fatal error reporting enable = 1b, Fatal error reporting enable = 1b
    // Unsupported request reporting enable = 1b, Relaxed Ordering Enable = 1
    //
    Data16Or = (UINT16)(Mps << B_MAX_PAYLOAD_SIZE_OFFSET) |
    (B_ENABLE_RELAXED_ORDERING + B_UNSUPPORTED_REQUEST_EN + B_FATAL_ERROR_EN + B_NON_FATAL_ERROR_EN + B_CORRECTABLE_ERROR_EN);
    PegPciSegmentOr16 (DeviceBaseAddress + DevPcieCapabilityOffset + R_DEVICE_CONTROL_REGISTER, Data16Or);

    //
    // Fun7: Write to link control register of PF7 to enable ASPM L0s and L1.
    //
    PegPciSegmentOr16 (DeviceBaseAddress + DevPcieCapabilityOffset + R_LINK_CONTROL_REGISTER, (L1_SET | L0_SET));
    DEBUG ((DEBUG_VERBOSE, "2LM: ASPM value for 0/3/7 at offset 0x%x  is %d \n", DevPcieCapabilityOffset, PegPciSegmentRead16(DeviceBaseAddress + DevPcieCapabilityOffset + R_LINK_CONTROL_REGISTER)));

    //
    // Fun7: Device Control 2 Register (28h): IDO Completion Enable = 1
    //
    Data16Or = (UINT16) (B_IDO_COMPLETION_EN);
    PegPciSegmentOr16 (DeviceBaseAddress + DevPcieCapabilityOffset + R_DEVICE_CONTROL2_REGISTER, Data16Or);
  }
  //
  // Following configurations are required for Fun0
  //

  DeviceBaseAddress = PCI_SEGMENT_LIB_ADDRESS (SA_SEG_NUM, SA_PEG_BUS_NUM, TWOLM_DEVICE_NUM, TWOLM_FUNC_0, 0);
  DevPcieCapabilityOffset  = PcieFindCapId (SA_SEG_NUM, SA_PEG_BUS_NUM, TWOLM_DEVICE_NUM, TWOLM_FUNC_0, PCIE_CAPID);

  if ( DevPcieCapabilityOffset ) {
    //
    // Fun0: Link Control Register (10h): Disable ASPM (L0s & L1)
    //
    Data16And = (UINT16) ( ~ (B_L1_ENTRY_ENABLED + B_L0_ENTRY_ENABLED));
    PegPciSegmentAnd16 (DeviceBaseAddress + DevPcieCapabilityOffset + R_LINK_CONTROL_REGISTER, Data16And);
    DEBUG ((DEBUG_VERBOSE, "2LM: Fun0: ASPM is disabled, R_LINK_CONTROL_REGISTER  =  0x%x \n", PegPciSegmentRead16 (DeviceBaseAddress + DevPcieCapabilityOffset + R_LINK_CONTROL_REGISTER)));

    //
    // Fun0: Device Control 2 Register (28h): LTR Mechanism Enable = 1b
    //
    Data16Or = (UINT16) (B_LTR_MECHANISM_EN);
    PegPciSegmentOr16 (DeviceBaseAddress + DevPcieCapabilityOffset + R_DEVICE_CONTROL2_REGISTER, Data16Or);
  }

  // 5. Check the RC and PF0 PM substate capability, and program ASPM/PM L1.1 and L1.2 enabled
  // (based on support bit)(expected ASPM L1.2 enabled only)
  SocL1PmCapabilityOffset = PegPcieFindExtendedCapId (SA_SEG_NUM, SA_MC_BUS, TWOLM_DEVICE_NUM, TWOLM_FUNC_0, L1PM_CAPID);
  DevL1PmCapabilityOffset = PegPcieFindExtendedCapId (SA_SEG_NUM, SA_MC_BUS, SA_PEG3_DEV_NUM, TWOLM_FUNC_0, L1PM_CAPID);

  if (SocL1PmCapabilityOffset && DevL1PmCapabilityOffset) {
    SocValue = PegPciSegmentRead32 (SocBaseAddress + SocL1PmCapabilityOffset + R_L1PM_SUBSTATES_CAPABILITIES_REGISTER)
                & (0x3 << B_ASPM_L1_2_SUPPORTED_OFFSET);
    DevValue = PegPciSegmentRead32 (DeviceBaseAddress + DevL1PmCapabilityOffset + R_L1PM_SUBSTATES_CAPABILITIES_REGISTER)
                & (0x3 << B_ASPM_L1_2_SUPPORTED_OFFSET);
    // write the common value
    Data16Or = (UINT16) ((SocValue & DevValue) << B_ASPM_L1_2_SUPPORTED_OFFSET);
    // Check if ASPM L1.2 is enabled as expected
    if (!(Data16Or & (0x1 << B_ASPM_L1_2_SUPPORTED_OFFSET))) {
      ASSERT (FALSE);
      return EFI_NOT_FOUND;
    }
    PegPciSegmentOr16 (DeviceBaseAddress + DevL1PmCapabilityOffset + R_L1PM_SUBSTATES_CONTROL_1_REGISTER, Data16Or);

    // 6. Read PF0 Tpoweron supported for both SOC and device, program with max value to PF0
    // TPOS is (scale + 1) * value, calculate it for both SOC and Dev and then compare and program based on the max value

    SocScale = PegPciSegmentRead32 (SocBaseAddress + SocL1PmCapabilityOffset + R_L1PM_SUBSTATES_CAPABILITIES_REGISTER)
                & (B_PORT_T_POWER_ON_SCALE_MASK << B_PORT_T_POWER_ON_SCALE_OFFSET);
    DevScale = PegPciSegmentRead32 (DeviceBaseAddress + DevL1PmCapabilityOffset + R_L1PM_SUBSTATES_CAPABILITIES_REGISTER)
                & (B_PORT_T_POWER_ON_SCALE_MASK << B_PORT_T_POWER_ON_SCALE_OFFSET);

    SocValue = PegPciSegmentRead32 (SocBaseAddress + SocL1PmCapabilityOffset + R_L1PM_SUBSTATES_CAPABILITIES_REGISTER)
                  & (B_PORT_T_POWER_ON_VALUE_MASK << B_PORT_T_POWER_ON_VALUE_OFFSET);
    DevValue = PegPciSegmentRead32 (DeviceBaseAddress + DevL1PmCapabilityOffset + R_L1PM_SUBSTATES_CAPABILITIES_REGISTER)
                  & (B_PORT_T_POWER_ON_VALUE_MASK << B_PORT_T_POWER_ON_VALUE_OFFSET);

    if (((SocScale+1) * SocValue) > ((DevScale+1) * DevValue)) {
      // program dev with the Soc settings
      Data32Or = (SocScale << B_PORT_T_POWER_ON_SCALE_CONTROL_OFFSET) | (SocValue << B_PORT_T_POWER_ON_VALUE_CONTROL_OFFSET);
    } else { // program dev with the dev settings
      Data32Or = (DevScale << B_PORT_T_POWER_ON_SCALE_CONTROL_OFFSET) | (DevValue << B_PORT_T_POWER_ON_VALUE_CONTROL_OFFSET);
    }
    // program Dev PF0
    PegPciSegmentOr32 (DeviceBaseAddress + DevL1PmCapabilityOffset + R_L1PM_SUBSTATES_CONTROL_2_REGISTER, Data32Or);
  }
  //
  // Following programming is required for the MFVC capability (Fun0)
  // The Extended Capability ID for the MFVC Capability is 0008h (as per PCIe spec v4)
  //
  MfvcCapabilityOffset  = PegPcieFindExtendedCapId (SA_SEG_NUM, SA_PEG_BUS_NUM, TWOLM_DEVICE_NUM, TWOLM_FUNC_0, MFVC_CAPID);

  if ( MfvcCapabilityOffset ) {
    if (TwoLmPreMemConfig->MfvcWrrArb) {
      //7.b  Provide capability to configure MFVC WRR arbitration table
      // i.  Write VC1 or VC0 to all 128 phases of these registers
      for (i = 0; i < MFVC_VC_ARB_TABLE_INDEX; ++i) {
          Data32Or = 0x0;
          for (j = 0; j < 8; ++j) {
            if (TwoLmPreMemConfig->VcId_7_0[i] & (0x1 << j)) {
              switch(j) {
                case 0:
                  Data32Or |= (0x1 << B_VC_ID0_OFFSET);
                  break;
                case 1:
                  Data32Or |= (0x1 << B_VC_ID1_OFFSET);
                  break;
                case 2:
                  Data32Or |= (0x1 << B_VC_ID2_OFFSET);
                  break;
                case 3:
                  Data32Or |= (0x1 << B_VC_ID3_OFFSET);
                  break;
                case 4:
                  Data32Or |= (0x1 << B_VC_ID4_OFFSET);
                  break;
                case 5:
                  Data32Or |= (0x1 << B_VC_ID5_OFFSET);
                  break;
                case 6:
                  Data32Or |= (0x1 << B_VC_ID6_OFFSET);
                  break;
                case 7:
                  Data32Or |= (0x1 << B_VC_ID7_OFFSET);
                  break;
              } // end of switch
            } // end of if
          } // end of for
        } // end of for
      PegPciSegmentOr32 (DeviceBaseAddress + MfvcCapabilityOffset + R_MFVC_VC_ARB_TABLE_0_REGISTER + i*4, Data32Or);

      // 7.a.ii. Write to MFVC Port VC Control and Status register (offset 0Ch) :
      //    1.  Set load_vc_arbitration_table (bit[0]) to 1
      //    2.  Set vc_arbitration_select (bit[3:1]) to 100b (128 phases)
      Data16Or = (VC_128PHASE_ARBITRATION << B_VC_ARBITRATION_SELECT_OFFSET) | B_LOAD_VC_ARBITRATION_TABLE;
      PegPciSegmentOr16 (DeviceBaseAddress + MfvcCapabilityOffset + R_PORT_VC_CONTROL_REGISTER, Data16Or);
      // 7.a.iii.  Read the MFVC Port VC Control and Status register (offset 0Ch) to make sure bit[0] is clear
      // (This bit always returns 0b when read.)
      // This should be very fast compare to config write then read.
      // set max 3 times read if not 0 and declare failure and proceed
      for (i = 0; i < RETRY_COUNT; ++i) {
        if (PegPciSegmentRead16 (DeviceBaseAddress + MfvcCapabilityOffset + R_PORT_VC_STATUS_REGISTER) & 0x1) {
          if (i == 2) {
            ASSERT(FALSE);
          }
        }
        else {
          i = 3;
        }
      }
    } else { // end of arbitration table programming
      // 7.b.  If table don't need to be programmed (default with all 128 phases set to VC1)
      //    i. Write to MFVC Port VC Control and Status register (offset 0Ch) :
      //      1.  Set vc_arbitration_select (bit[3:1]) to 100b (128 phases)
      Data16Or = (VC_128PHASE_ARBITRATION << B_VC_ARBITRATION_SELECT_OFFSET);
      PegPciSegmentOr16 (DeviceBaseAddress + MfvcCapabilityOffset + R_PORT_VC_CONTROL_REGISTER, Data16Or);
    }

    // 8.a.  Write to MFVC VC1 Resource Control Register (offset 20h-23h)
    //    i. Set mfvc tc_vc_map (bit[7:0]) to 1000_0000b (bit[7] is TC7)
    //   ii. Set mfvc_vc_id (bit[26:24]) to 1b
    //  iii.  Set mfvc vc1_enable (bit[31]) to 1b
    PegPciSegmentWrite32 (DeviceBaseAddress + MfvcCapabilityOffset + R_VC1_RESOURCE_CONTROL_REGISTER, 0x81000080);

    // 8.b.  Read  MFVC VC1 Resource Status Register (offset 24h-27) to confirm vc_negotiate_pending (bit[17]) is clear to 0 (wait for 1ms)
    MicroSecondDelay(1000);
    if ((PegPciSegmentRead32 (DeviceBaseAddress + MfvcCapabilityOffset + R_VC1_RESOURCE_STATUS_REGISTER) >> B_VC1_NEGOTIATION_PENDING_OFFSET) & 0x1) {
      ASSERT(FALSE);
      return EFI_NOT_FOUND;
    }
  } // end of MFVC capability

  DEBUG ((DEBUG_VERBOSE, "2LM: WillardPcieDeviceInit Initialization End\n"));
  return EFI_SUCCESS;
}

/**
This function initializes 2LM specific configurations for the SOC Host PCIe RP controller.

@param[in]      TwoLmConfig            2LM Config block.

@retval         EFI_SUCCESS            Successfully initialized SOC Device's PCIe Controller.
**/
STATIC
EFI_STATUS
EFIAPI
HostPcieDeviceInit (
  IN TWOLM_PREMEM_CONFIG  *TwoLmPreMemConfig
)
{
  UINT32            DeviceBaseAddress;
  UINT32            Data32Or;
  UINT32            Data32And;

  DEBUG ((DEBUG_INFO, "2LM: HostPcieDeviceInit Start\n"));

  DeviceBaseAddress = PCI_SEGMENT_LIB_ADDRESS (SA_SEG_NUM, SA_PEG_BUS_NUM, SA_PEG3_DEV_NUM, SA_PEG3_FUN_NUM, 0);
  if (PegPciSegmentRead16 (DeviceBaseAddress + PCI_VENDOR_ID_OFFSET) == 0xFFFF) {
    return EFI_NOT_FOUND;
  }

  //
  // Program BME and MSE in command register
  //
  Data32Or = (UINT32)(EFI_PCI_COMMAND_BUS_MASTER + EFI_PCI_COMMAND_MEMORY_SPACE);
  PegPciSegmentOr32 (DeviceBaseAddress + PCI_COMMAND_OFFSET, Data32Or);

  //
  // Program 0 to Primary, Secondary and Subordinate bus registers.
  //
  PegPciSegmentWrite16 (DeviceBaseAddress + PCI_BRIDGE_PRIMARY_BUS_REGISTER_OFFSET, 0);
  PegPciSegmentWrite8 (DeviceBaseAddress + PCI_BRIDGE_SUBORDINATE_BUS_REGISTER_OFFSET, 0);

  //
  // Program Memory Base and Limit register with value 0xFFF to Bits15:4 and 0 to Bits 31:20
  //
  Data32And = 0xF000F;
  Data32Or = 0xFFF0;
  PegPciSegmentAndThenOr32 (DeviceBaseAddress + R_PCI_BRIDGE_MBL, Data32And, Data32Or);

  //
  // Program Prefetchable Memory Base and Limit register with value 0xFFF to Bits15:4 and 0 to Bits 31:20
  //
  Data32And = 0xF000F;
  Data32Or = 0xFFF0;
  PegPciSegmentAndThenOr32 (DeviceBaseAddress + R_PCI_BRIDGE_PMBL, Data32And, Data32Or);

  //
  // Program all 1s Prefetchable Memory Base Upper and Limit Upper.
  //
  PegPciSegmentWrite32 (DeviceBaseAddress + R_PCI_BRIDGE_PMBU32, 0xFFFFFFFF);
  PegPciSegmentWrite32 (DeviceBaseAddress + R_PCI_BRIDGE_PMLU32, 0);

  //
  // Enable ASPM Optionality Compliance, ASPM L0 and L1 Capabilities
  //
  Data32Or = B_PCIE_LCAP_ASPMOC + B_PCIE_LCAP_APMS_L1 + B_PCIE_LCAP_APMS_L0S;
  PegPciSegmentOr32 (DeviceBaseAddress + R_PCIE_LCAP, Data32Or);

  ///
  /// Active State Link PM Control (ASPM)  Indicates whether the  root port should enter  L0s or L1 or both
  /// 2LM BIOS to  program to 0x2 for [1:0] (10 L1 Entry Enabled)
  ///
  PegPciSegmentOr32 (DeviceBaseAddress + R_PCIE_LCTL, L1_SET);

  //
  // Enable 10-Bit Tag Completer Support capabilities
  //
  Data32Or = B_PCIE_DCAP2_PX10BTCS;
  PegPciSegmentOr32 (DeviceBaseAddress + R_PCIE_DCAP2, Data32Or);

  //
  // Initialize L0s Periodic Flow Control Update Frequency
  //
  Data32Or = 3 << B_PCIE_CONTROL1_L0SPFCUF_OFFSET;
  PegPciSegmentOr32 (DeviceBaseAddress + R_PCIE_CONTROL1, Data32Or);

  //
  // Initialize Transaction Layer Advance Control and Status register.
  //
  Data32And = (UINT32)~(B_PCIE_TLADVCTLST_VC0TXARBGC_MASK + B_PCIE_TLADVCTLST_VC0REQPUTARBGC_MASK);
  Data32Or = (7 << B_PCIE_TLADVCTLST_VC1TXARBGC_OFFSET) + (7 << B_PCIE_TLADVCTLST_VC1REQPUTARBGC_OFFSET);
  PegPciSegmentAndThenOr32 (DeviceBaseAddress + R_PCIE_TLADVCTLST, Data32And, Data32Or);

  //
  // Initialize Far Memory Mode Control and Status register
  //
  Data32And = (UINT32)(~B_PCIE_FMMCTLST_FMMPS_MASK);
  Data32Or = 2 << B_PCIE_FMMCTLST_FMMPS_OFFSET;
  PegPciSegmentAndThenOr32 (DeviceBaseAddress + R_PCIE_FMMCTLST, Data32And, Data32Or);

  //
  // Initialize VC1 link
  //
  // Todo. Below code has to be revisited once it is added as part of PEG code.
  PegPciSegmentWrite32 (DeviceBaseAddress + 0x280, 0x10009);
  PegPciSegmentWrite32 (DeviceBaseAddress + 0x294, 0x8000007f);
  PegPciSegmentWrite32 (DeviceBaseAddress + 0x2a0, 0x81000080);
  PegPciSegmentOr32    (DeviceBaseAddress + 0x284, 0x1);    //Bits[2:0] BIOS to prog 0x1 for VC support config
  DEBUG ((DEBUG_VERBOSE, "2LM: HostPcieDeviceInit End\n"));
  return EFI_SUCCESS;
}

/**
  Silicon Initializes after Policy PPI produced, All required polices must be installed before the callback

  @param[in] PeiServices          General purpose services available to every PEIM.
  @param[in] NotifyDescriptor     The notification structure this PEIM registered on install.
  @param[in] Ppi                  The memory discovered PPI.  Not used.

  @retval EFI_SUCCESS             If Successful.
  @retval EFI_NOT_FOUND           If SiPreMemPolicyPpi not located
  @retval EFI_UNSUPPORTED         TwoLmInfoHob not found.
**/
EFI_STATUS
EFIAPI
MemoryInitDoneCallback (
  IN  EFI_PEI_SERVICES             **PeiServices,
  IN  EFI_PEI_NOTIFY_DESCRIPTOR    *NotifyDescriptor,
  IN  VOID                         *Ppi
)
{
  return BzmMemoryInitDoneCallback (PeiServices, NotifyDescriptor, Ppi);
}
/**
  Function to modify EPOC bit to support 1LM/2LM mode switching
  Changing EPOC bit will trigger System-Reboot
**/
VOID
ModifyEpoc2lmBit (
  VOID
  )
{
  TWOLM_EPOC_SET TwoLmEpocValue;


  TwoLmEpocValue.Uint32 = PmcGetCpuEpoc ();
  if (TwoLmEpocValue.Bits.TwoLm) {
    TwoLmEpocValue.Bits.TwoLm = 0; // Set the EPOC bit to 1LM
  } else {
    TwoLmEpocValue.Bits.TwoLm = 1; // Set the EPOC bit to 2LM
  }
  PmcSetCpuEpoc (TwoLmEpocValue.Uint32);
  //
  //RESET is Required since the EPOC value has changed
  //
  (*GetPeiServicesTablePointer ())->ResetSystem2 (EfiResetCold, EFI_SUCCESS, 0, NULL);
}

/**
  Function to check the possibility of booting in 2LM mode based on multiple parameters

  @param[in] MemBootMode           - Boot Mode Setup option for 2LM

  @retval UINT8                    - return Boot policy based on the boot mode decision
**/
UINT8
Detect2Lm (
  IN UINT8        MemBootMode
  )
{
  UINT16          DevId;
  UINT8           Dev;
  UINT8           Func;
  UINT8           HeaderType;
  UINT8           MaxFunction;
  UINT32          DeviceBaseAddress;
  UINT32          Data32Or;
  UINT32          Data32And;

  UINT32          PcieBusNum  = 0x00010100;

  DEBUG ((DEBUG_INFO, "2LM: Detecting 2LM device\n"));

  // Enable the config space for Dev3, Func0 and Func7 (enabling it irrespective of the status to avoid the if condition check)
  PegPciSegmentOr32 (PCI_SEGMENT_LIB_ADDRESS (SA_SEG_NUM,SA_MC_BUS, SA_MC_DEV, SA_MC_FUN, R_SA_DEVEN), (UINT32) B_SA_DEVEN_D3F0EN_MASK);
  PegPciSegmentOr32 (PCI_SEGMENT_LIB_ADDRESS (SA_SEG_NUM,SA_MC_BUS, SA_MC_DEV, SA_MC_FUN, R_SA_DEVEN), (UINT32) B_SA_DEVEN_D3F7EN_MASK);

  //
  // Step1: check CAPID and Setup option for 2LM, reset EPOC bit if required
  // @todo: Add check for 2LM CAPID
  //
  if (MemBootMode == MEM_BOOT_MODE_1LM){

    if (IsTwoLmEnabled() == TRUE) {
      ModifyEpoc2lmBit ();
    }
    return MEM_BOOT_MODE_1LM;
  }
  //
  // Step2: At this stage, check the possibility to boot in 2LM mode based on the 2LM device discovery
  // @todo take care of provisioning mode
  //

  //
  // Step3: BootMode:2LM, EPOC = ? check if the EPOC bit is supporting 2LM mode
  //
  if (IsTwoLmEnabled() == FALSE) {
    DeviceBaseAddress = PCI_SEGMENT_LIB_ADDRESS (SA_SEG_NUM, SA_MC_BUS, SA_PEG3_DEV_NUM, 0, 0);
    if (PegPciSegmentRead16 (DeviceBaseAddress + PCI_VENDOR_ID_OFFSET) == 0xFFFF) {
      return MEM_BOOT_MODE_1LM;
    }
    //
    // Enumerate and check if any of the devices connected to 0/6/0 is 2LM device
    // VID= 0x8086 and DID = 0x41C0 to 0x41CF
    //
    PegPciSegmentWrite32 (DeviceBaseAddress + PCI_BRIDGE_PRIMARY_BUS_REGISTER_OFFSET, PcieBusNum); // Assigning temp Bus
    for (Dev = 0; Dev <= 31; ++Dev) {
      //
      // Read Vendor ID to check if device exists
      // if no device exists, then check next device
      //
      DeviceBaseAddress = PCI_SEGMENT_LIB_ADDRESS (SA_SEG_NUM, 1, Dev, 0, 0);
      if (PegPciSegmentRead16 (DeviceBaseAddress + PCI_VENDOR_ID_OFFSET) == 0xFFFF) {
        continue;
      }
      //
      // Check for a multifunction device
      //
      HeaderType = PegPciSegmentRead8 (DeviceBaseAddress + PCI_HEADER_TYPE_OFFSET);
      if ((HeaderType & HEADER_TYPE_MULTI_FUNCTION) != 0) {
        MaxFunction = 7;
      } else {
        MaxFunction = 0;
      }
      for (Func = 0; Func <= MaxFunction; ++Func) {
        DeviceBaseAddress = PCI_SEGMENT_LIB_ADDRESS (SA_SEG_NUM, 1, Dev, Func, 0);
        if (PegPciSegmentRead16 (DeviceBaseAddress + PCI_VENDOR_ID_OFFSET) == 0xFFFF) {
          continue;
        } else if (PegPciSegmentRead16 (DeviceBaseAddress + PCI_VENDOR_ID_OFFSET) == 0x8086) {
          DevId = (PegPciSegmentRead16 (DeviceBaseAddress + PCI_DEVICE_ID_OFFSET));
          if ( DevId >> 4 == 0x41C ) {
            DEBUG ((DEBUG_INFO, "2LM: Device detected on Peg port is 2LM device\n"));
            ModifyEpoc2lmBit ();
          }
        }
      }
    }// End of this phase, BootMode:2LM, EPOC = 1LM, Device = 1LM/No device
    //
    // Need to clear the assigned temporary bus no
    //
    DeviceBaseAddress = PCI_SEGMENT_LIB_ADDRESS (SA_SEG_NUM, SA_MC_BUS, SA_PEG3_DEV_NUM, 0, 0);
    PegPciSegmentWrite32 (DeviceBaseAddress + PCI_BRIDGE_PRIMARY_BUS_REGISTER_OFFSET, 0);
    return MEM_BOOT_MODE_1LM;
  }

  //
  // M_CREDITS_CONFIG_DONE bits in FMHC has to be set before first cfg access to 2LM device and VDM register access.
  //
  CpuRegbarOr32 (CPU_SB_PID_FMHC, R_FMHC_CCRDC, B_FMHC_CCRDC_M_CREDITS_CONFIG_DONE);
  CpuRegbarOr32 (CPU_SB_PID_FMHC, R_FMHC_HBIFCREDIT_VC1, B_FMHC_HBIFCREDIT_VC1_M_CREDITS_CONFIG_DONE);

  //
  // Write value 16 for the MRCRD [12:8] of CCRDC register
  //
  Data32And = ~(UINT32)(B_FMHC_CCRDC_MRCRD_MASK);
  Data32Or =  (0x10 << B_FMHC_CCRDC_MRCRD_OFFSET);
  CpuRegbarAndThenOr32 (CPU_SB_PID_FMHC, R_FMHC_CCRDC, Data32And, Data32Or);
  DEBUG ((DEBUG_INFO, "Reg val for CCRDC is  0x%x\n", CpuRegbarRead32 (CPU_SB_PID_FMHC, R_FMHC_CCRDC)));
  //
  // Step4: BootMode:2LM, EPOC = 2LM, Device = ?
  // Check if FMHC is enabled
  //
  DeviceBaseAddress = PCI_SEGMENT_LIB_ADDRESS (SA_SEG_NUM, SA_MC_DEV, TWOLM_DEVICE_NUM, 0, 0);
  if (PegPciSegmentRead16 (DeviceBaseAddress + PCI_VENDOR_ID_OFFSET) != 0xFFFF) {
    //
    // FMHC is enabled, do the device discovery for 0/3/Y and check for 2LM device
    // Check for a multifunction device
    //
    HeaderType = PegPciSegmentRead8 (DeviceBaseAddress + PCI_HEADER_TYPE_OFFSET);
    if ((HeaderType & HEADER_TYPE_MULTI_FUNCTION) != 0) {
      MaxFunction = 7;
    } else {
      MaxFunction = 0;
    }
    for (Func = 0; Func <= MaxFunction; ++Func) {
      DeviceBaseAddress = PCI_SEGMENT_LIB_ADDRESS (SA_SEG_NUM, SA_MC_DEV, TWOLM_DEVICE_NUM, Func, 0);
      if (PegPciSegmentRead16 (DeviceBaseAddress + PCI_VENDOR_ID_OFFSET) == 0xFFFF) {
        continue;
      } else if (PegPciSegmentRead16 (DeviceBaseAddress + PCI_VENDOR_ID_OFFSET) == 0x8086) {
        DevId = (PegPciSegmentRead16 (DeviceBaseAddress + PCI_DEVICE_ID_OFFSET));
        if ( DevId >> 4 == 0x41C ) {
          // At this stage => Setup:2LM, EPOC = 2LM, Device = 2LM, Update boot policy to 2LM
          DEBUG ((DEBUG_INFO, "2LM: 2LM Device detected, Boot the system in 2LM mode\n"));
          return MEM_BOOT_MODE_2LM;
        }
      }
    }
  }
  DEBUG ((DEBUG_INFO, "2LM: No 2LM device found\n"));
  ModifyEpoc2lmBit ();
  return MEM_BOOT_MODE_1LM;
}

/**
This function performs any early configuration of 2LM if needed.

**/
VOID
EFIAPI
TwoLmEarlyConfig (
  )
{
  UINT32            Data32;

  DEBUG ((DEBUG_INFO, "2LM: TwoLmEarlyConfig Start\n"));
  Data32 = PciSegmentRead32 (PCI_SEGMENT_LIB_ADDRESS (SA_SEG_NUM, SA_MC_BUS, SA_MC_DEV, SA_MC_FUN, R_SA_MC_CAPID0_B));
  if (Data32 & BIT14) {
    Data32 = PciSegmentRead32 (PCI_SEGMENT_LIB_ADDRESS (SA_SEG_NUM, SA_MC_BUS, SA_MC_DEV, SA_MC_FUN, R_SA_DEVEN));

    //
    // Hide RP controller Bus 0 Dev 6 Fun 0 if it is a 2LM enabled boot.
    //
    Data32 &= ~B_SA_DEVEN_D6EN_MASK;
    PciSegmentWrite32 (PCI_SEGMENT_LIB_ADDRESS (SA_SEG_NUM, SA_MC_BUS, SA_MC_DEV, SA_MC_FUN, R_SA_DEVEN), Data32);
  } else {
    Data32 = PciSegmentRead32 (PCI_SEGMENT_LIB_ADDRESS (SA_SEG_NUM, SA_MC_BUS, SA_MC_DEV, SA_MC_FUN, R_SA_DEVEN));
    //
    // Hide Bus 0 Dev 3 Fun 0/7 in case if it is enabled
    //
    Data32 &= ~B_SA_DEVEN_D3F0EN_MASK & ~B_SA_DEVEN_D3F7EN_MASK;
    PciSegmentWrite32 (PCI_SEGMENT_LIB_ADDRESS (SA_SEG_NUM, SA_MC_BUS, SA_MC_DEV, SA_MC_FUN, R_SA_DEVEN), Data32);
  }
  Data32 = PciSegmentRead32 (PCI_SEGMENT_LIB_ADDRESS (SA_SEG_NUM, SA_MC_BUS, SA_MC_DEV, SA_MC_FUN, R_SA_DEVEN));

  //Todo: below code has been added temporarily to enable 2LM and 1LM NVME boot in official BIOS.
  // Once it has been implemented in SA driver, below four lines of code can be removed.
    PegPciSegmentWrite32 (PCI_SEGMENT_LIB_ADDRESS (SA_SEG_NUM, SA_PEG_BUS_NUM, SA_PEG3_DEV_NUM, SA_PEG3_FUN_NUM, 0xD0), 0x00000077);
    PegPciSegmentWrite32 (PCI_SEGMENT_LIB_ADDRESS (SA_SEG_NUM, SA_PEG_BUS_NUM, SA_PEG3_DEV_NUM, SA_PEG3_FUN_NUM, 0xF0), 0x00000770);
    PegPciSegmentWrite32 (PCI_SEGMENT_LIB_ADDRESS (SA_SEG_NUM, SA_PEG_BUS_NUM, SA_PEG3_DEV_NUM, SA_PEG3_FUN_NUM, 0x6CC), 0);
    PegPciSegmentWrite32 (PCI_SEGMENT_LIB_ADDRESS (SA_SEG_NUM, SA_PEG_BUS_NUM, SA_PEG3_DEV_NUM, SA_PEG3_FUN_NUM, 0x594), 0x1E40F);
}

/**
This function does the basic configurations of IPs and issues the Mailbox commands
to find out the Far Memory region and its size which would be used by the MRC phase
in creating the memory map.

@param[in]      TwoLmConfig              2LM Config block.

@retval EFI_SUCCESS            Successfully initialized Pre-Mem configurations.
@retval EFI_UNSUPPORTED        in case if all the requirements to support 2LM boot mode are not met.
**/

EFI_STATUS
EFIAPI
TwoLmPreMemInit (
   IN TWOLM_PREMEM_CONFIG      *TwoLmPreMemConfig
)
{
  EFI_STATUS                   Status;
  TWOLM_INFO_HOB               *TwoLmInfoHob;

  DEBUG ((DEBUG_INFO, "2LM: TwoLmPreMemInit Start\n"));

  //
  // Create HOB for 2LM INFO
  //
  Status = PeiServicesCreateHob (
             EFI_HOB_TYPE_GUID_EXTENSION,
             sizeof (TWOLM_INFO_HOB),
             (VOID **) &TwoLmInfoHob
             );
  ASSERT_EFI_ERROR (Status);

  //
  // Initialize default HOB data
  //
  TwoLmInfoHob->EfiHobGuidType.Name = gTwoLmInfoHobGuid;
  ZeroMem (&(TwoLmInfoHob->TwoLmInfo), sizeof (TWOLM_INFO));

  if (Detect2Lm (TwoLmPreMemConfig->MemBootMode) == MEM_BOOT_MODE_1LM) {
    TwoLmPreMemConfig->MemBootMode = MEM_BOOT_MODE_1LM;
    TwoLmInfoHob->TwoLmInfo.CurrentMemBootMode = MEM_BOOT_MODE_1LM;
    return EFI_UNSUPPORTED;
  }


  // Initialize Hob with the current policy and EPOC value
  TwoLmInfoHob->TwoLmInfo.CurrentMemBootMode = TwoLmPreMemConfig->MemBootMode;

  //
  // Initialize Host PCIe controller
  //
  Status = HostPcieDeviceInit (TwoLmPreMemConfig);
  if (Status != EFI_SUCCESS) {
      TwoLmInfoHob->TwoLmInfo.CurrentMemBootMode = MEM_BOOT_MODE_1LM;
      return EFI_UNSUPPORTED;
  }

  //
  // Initialize 2LM device PCIe controller
  //
  Status = WillardPcieDeviceInit (TwoLmPreMemConfig);
  if (Status != EFI_SUCCESS) {
      TwoLmInfoHob->TwoLmInfo.CurrentMemBootMode = MEM_BOOT_MODE_1LM;
      return EFI_UNSUPPORTED;
  }

  //
  // Initialize Willard device for Power management and 2LM enable.
  //
  Status = WillardInit (
            TwoLmPreMemConfig
           ,TwoLmInfoHob
          );
  if (Status != EFI_SUCCESS) {
      TwoLmInfoHob->TwoLmInfo.CurrentMemBootMode = MEM_BOOT_MODE_1LM;
      return EFI_UNSUPPORTED;
  }

  //
  // Register Silicon init call back after PlatformPolicy PPI produced
  //
  Status = PeiServicesNotifyPpi (gMemInitDoneNotifyList);
  ASSERT_EFI_ERROR (Status);

  NvdimmCacheInfoPpiHobInstall();


  return EFI_SUCCESS;
}

/**
  Perform Far Memory Encryption initialization.

  @retval VOID - No value to return
**/
VOID
FmeInit (
  )
{
  MSR_FME_ACTIVATE_REGISTER   MsrFmeActivate;
  EFI_BOOT_MODE               BootMode;
  EFI_STATUS                  Status;

  MsrFmeActivate.Uint64 = 0;

  DEBUG ((DEBUG_INFO, "2LM: Far Memory Encryption (FME) Initialization\n"));

  //
  // Check if FME_ACTIVATE MSR can be setup.
  //
  if (MsrIsSxp2lmEnabled ()) {

    //
    // Set Fme enable - BIT[1]
    //
    MsrFmeActivate.Bits.FmeEnable = 1;
    //
    // Set FME Key Select for various boot modes.
    // Key Select - BIT[2]
    //   - Set for S3 resume and flash update flow to restore FME keys from previous boot.
    //   - Clear for cold/warm boot to create new FME keys
    //
    Status = PeiServicesGetBootMode (&BootMode);
    ASSERT_EFI_ERROR (Status);

    if ((BootMode == BOOT_ON_S3_RESUME) || (BootMode == BOOT_ON_FLASH_UPDATE)) {
      MsrFmeActivate.Bits.KeySelect = 1;
    } else {
      MsrFmeActivate.Bits.KeySelect = 0;
    }

    //
    // Save key for standby - BIT[3]
    //
    MsrFmeActivate.Bits.SaveKeyForStandby = 1;

    //
    // Lock FME Activate MSR - BIT[0]
    //
    MsrFmeActivate.Bits.Lock = 1;

    //
    // Configure IA32_FME_ACTIVATE MSR (81H) with FME Enable, Key Select, Save Key
    //
    AsmWriteMsr64 (MSR_FME_ACTIVATE, MsrFmeActivate.Uint64);
  } else {
  }
}

/**
This function does the basic configurations of FMHC and Astro controller
This function will be called only if the 2LM mode is enabled in the policy.

@param[in]      NearMemorySize       Total DRAM/Near mem size in MB
@param[in]      Mc0Size              MC0 mem size in MB
@param[in]      Mc1Size              MC1 mem size in MB
@param[in]      ToludBase            Value of ToludBase
@param[in]      MemBootMode          Mem boot mode value as a result of detected near mem size

@retval EFI_SUCCESS            Successfully initialized Post-Mem configurations.
**/
EFI_STATUS
EFIAPI
TwoLmPostDdrInit (
  IN UINT32 NearMemorySize,
  IN UINT32 Mc0Size,
  IN UINT32 Mc1Size,
  IN UINT32 ToludBase,
  IN UINT8 MemBootMode
  )
{
  EFI_STATUS                    Status;
  TWOLM_INFO_HOB                *TwoLmInfoHob;
  TWOLM_PREMEM_CONFIG           *TwoLmPreMemConfig;
  SI_PREMEM_POLICY_PPI          *SiPreMemPolicyPpi;
  UINT32                        Data;

  ///
  /// Get policy settings through the SaInitConfigBlock PPI
  ///
  Status = PeiServicesLocatePpi(
             &gSiPreMemPolicyPpiGuid,
             0,
             NULL,
             (VOID **)&SiPreMemPolicyPpi
  );
  ASSERT_EFI_ERROR(Status);
  if ((Status != EFI_SUCCESS) || (SiPreMemPolicyPpi == NULL)) {
    return EFI_NOT_FOUND;
  }

  Status = GetConfigBlock((VOID *)SiPreMemPolicyPpi, &gTwoLmPreMemConfigGuid, (VOID *)&TwoLmPreMemConfig);
  ASSERT_EFI_ERROR(Status);


  DEBUG ((DEBUG_INFO, "2LM: TwoLmPostDdrInit start\n"));

  TwoLmInfoHob = (TWOLM_INFO_HOB *) GetFirstGuidHob (&gTwoLmInfoHobGuid);
  if (TwoLmInfoHob == NULL) {
    return EFI_UNSUPPORTED;
  }

  if (MemBootMode != MEM_BOOT_MODE_2LM) {
    // updating policy and Hob info based on the near mem detection
    TwoLmPreMemConfig->MemBootMode = MemBootMode;
    TwoLmInfoHob->TwoLmInfo.CurrentMemBootMode = MemBootMode;
    return EFI_UNSUPPORTED;
  }

  // Update Hob with inputs from MRC
  TwoLmInfoHob->TwoLmInfo.NearMemSize = NearMemorySize;
  TwoLmInfoHob->TwoLmInfo.Mc0Size     = Mc0Size;
  TwoLmInfoHob->TwoLmInfo.Mc1Size     = Mc1Size;
  TwoLmInfoHob->TwoLmInfo.ToludBase = ToludBase;
  // Initialize the Far Memory Host Controller
  //
  Status = FmhcInit (
            &TwoLmInfoHob->TwoLmInfo
            ,TwoLmPreMemConfig
           );
  if (Status != EFI_SUCCESS) {
    // updating policy and Hob info to 1LM
    TwoLmPreMemConfig->MemBootMode = MEM_BOOT_MODE_1LM;
    TwoLmInfoHob->TwoLmInfo.CurrentMemBootMode = MEM_BOOT_MODE_1LM;
    return EFI_UNSUPPORTED;
  }
  // 10.  VDM write to FMSS Power Management Mode register (offset 10h)
  //  a.  Set FMPMMEN bit to 1 (enable 2LM Power Management Mode)
  // Note : When FMPMMEM is set to 1, 2LM mode is enabled which means Willard is RCiEP device.
  // And few capabilities are hidden for PF0.
  //
  Status = RegReadVdm (R_SA_FMSS_VDM_MAIL_BOX_FMPMM, &Data);
  ASSERT_EFI_ERROR (Status);
  Data |= B_SA_FMSS_VDM_MAIL_BOX_FMPMM_FMPMMEN;
  Status = RegWriteVdm (R_SA_FMSS_VDM_MAIL_BOX_FMPMM, Data);
  ASSERT_EFI_ERROR (Status);

  //
  // Initialize the Astro Controller
  //
  Status = AstroInit (&TwoLmInfoHob->TwoLmInfo);
  if (Status != EFI_SUCCESS) {
    // updating policy and Hob info to 1LM
    TwoLmPreMemConfig->MemBootMode = MEM_BOOT_MODE_1LM;
    TwoLmInfoHob->TwoLmInfo.CurrentMemBootMode = MEM_BOOT_MODE_1LM;
    return EFI_UNSUPPORTED;
  }

  //
  // Initialize Far Memory Encrytion feature
  //
  FmeInit ();
  return EFI_SUCCESS;
}


/**
This function hanldes all post mem activities for 2LM feature.

@retval EFI_SUCCESS      Successfully initialized Post-Mem configurations.
**/
EFI_STATUS
EFIAPI
TwoLmPostMemInit (
  )
{
  TWOLM_INFO_HOB         *TwoLmInfoHob;
  UINT64                 McD0BaseAddress;
  UINT32                 DevEn;
  UINT64                 NonceValue;
  UINT8                  DeviceReturnStatus;
  EFI_STATUS             Status;
  SYSTEM_TIME            SystemTime;

  DEBUG ((DEBUG_INFO, "2LM: TwoLmPostMemInit start\n"));

  TwoLmInfoHob = (TWOLM_INFO_HOB *)GetFirstGuidHob (&gTwoLmInfoHobGuid);
  if (TwoLmInfoHob == NULL) {
    return EFI_UNSUPPORTED;
  }

  McD0BaseAddress = PCI_SEGMENT_LIB_ADDRESS (SA_SEG_NUM, SA_MC_BUS, 0, 0, 0);
  DevEn = PegPciSegmentRead32 (McD0BaseAddress + R_SA_DEVEN);

  //
  // Hide NVME function Bus 0 Dev 3 Fun 0, if there is no storage available
  //
  if (TwoLmInfoHob->TwoLmInfo.StorageSize == 0) {
    DevEn &= ~B_SA_DEVEN_D3F0EN_MASK;
  }

  //
  // Hide 2LM PCIe function Bus 0 Dev 3 Fun 7
  //
  DevEn &= ~B_SA_DEVEN_D3F7EN_MASK;

  //
  // Hide RP controller Bus 0 Dev 6 Fun 0
  //
  DevEn &= ~B_SA_DEVEN_D6EN_MASK;

  PegPciSegmentWrite32 (McD0BaseAddress + R_SA_DEVEN, DevEn);

  DEBUG ((DEBUG_INFO, "2LM: Willard PostMemInit start\n"));
  // Issuing set system time command
  GetSystemTime (&SystemTime);
  Status = SetSystemTimeCommand (&DeviceReturnStatus, ConvertSystemTimeToUnixFormat (&SystemTime));
  if (Status != EFI_SUCCESS) {
    DEBUG ((DEBUG_INFO, "Mailbox Communication Error while setting system time\n"));
    ASSERT_EFI_ERROR (Status);
  }
  if (DeviceReturnStatus) { // if the return Status is other than 00h (Success)
    DEBUG ((DEBUG_INFO, "3DXP Device Error while setting system time with DeviceReturnStatus = %x\n", DeviceReturnStatus));
    ASSERT (EFI_DEVICE_ERROR);
  }

  // Generating security nonce value, no need to store this.
  if (GetRandomNumber64(&NonceValue) != FALSE) {
    // Issuing set security nonce FIS command
    Status = SetSecurityBiosNonceCommand (&DeviceReturnStatus, NonceValue);
    if (Status != EFI_SUCCESS){
      DEBUG ((DEBUG_INFO, "Mailbox Communication Error while setting nonce\n"));
      ASSERT_EFI_ERROR (Status);
    }
    if (DeviceReturnStatus) { // if the return Status is other than 00h (Success)
      DEBUG ((DEBUG_INFO, "3DXP Device Error while setting nonce with DeviceReturnStatus = %x\n", DeviceReturnStatus));
      ASSERT (EFI_DEVICE_ERROR);
    }

    // Issuing Config Lockdown
    Status = SetConfigLockdownCommand (&DeviceReturnStatus, NonceValue);
    if (Status != EFI_SUCCESS){
      DEBUG ((DEBUG_INFO, "Mailbox Communication Error while setting config lock\n"));
      ASSERT_EFI_ERROR (Status);
    }
    if (DeviceReturnStatus) { // if the return Status is other than 00h (Success)
      DEBUG ((DEBUG_INFO, "3DXP Device Error while setting config lock with DeviceReturnStatus = %x\n", DeviceReturnStatus));
      ASSERT (EFI_DEVICE_ERROR);
    }
    NonceValue = 0;   // Clearing the Nonce variable
  }

  return EFI_SUCCESS;
}
