/** @file
  This code provides Initialization of Nfit tables for 2LM/PMem.

@copyright
  INTEL CONFIDENTIAL
  Copyright 2019-2020 Intel Corporation.

  The source code contained or described herein and all documents related to the
  source code ("Material") are owned by Intel Corporation or its suppliers or
  licensors. Title to the Material remains with Intel Corporation or its suppliers
  and licensors. The Material may contain trade secrets and proprietary and
  confidential information of Intel Corporation and its suppliers and licensors,
  and is protected by worldwide copyright and trade secret laws and treaty
  provisions. No part of the Material may be used, copied, reproduced, modified,
  published, uploaded, posted, transmitted, distributed, or disclosed in any way
  without Intel's prior express written permission.

  No license under any patent, copyright, trade secret or other intellectual
  property right is granted to or conferred upon you by disclosure or delivery
  of the Materials, either expressly, by implication, inducement, estoppel or
  otherwise. Any license under such intellectual property rights must be
  express and approved by Intel in writing.

  Unless otherwise agreed by Intel in writing, you may not remove or alter
  this notice or any other notice embedded in Materials by Intel or
  Intel's suppliers or licensors in any way.

  This file contains an 'Intel Peripheral Driver' and is uniquely identified as
  "Intel Reference Module" and is licensed for Intel CPUs and chipsets under
  the terms of your license agreement with Intel or your vendor. This file may
  be modified by the user, subject to additional terms of the license agreement.

@par Specification Reference:
**/

#include "NfitDxe.h"
#include <Library/UefiBootServicesTableLib.h>
#include <Protocol/AcpiTable.h>
#include <Protocol/AcpiSystemDescriptionTable.h>
#include <Uefi/UefiSpec.h>
#include <Pi/PiFirmwareFile.h>
#include <Protocol/FirmwareVolume2.h>
#include <Library/PeiDxeSmmTwoLmLib.h>
#include <Library/DebugLib.h>
#include <Library/MemoryAllocationLib.h>
#include <Library/BaseLib.h>
#include <Library/HobLib.h>
#include <Library/NvdimmCacheInfoHob.h>

extern EFI_GUID               gTwoLmInfoHobGuid;
GLOBAL_REMOVE_IF_UNREFERENCED   EFI_2LM_OPTANE_DEVICE_PROTOCOL TwoLmOptaneInterface;

EFI_ACPI_NFIT_TABLE NfitTable = {

    NVDIMM_FW_INTERFACE_TABLE_SIGNATURE,      // Signature
    sizeof (EFI_ACPI_NFIT_TABLE),             // Length
    NVDIMM_FW_INTERFACE_TABLE_REVISION,       // Revision  [01]
    0,                                        // Checksum
    { 'I', 'N', 'T', 'E', 'L', ' ' },
    NVDIMM_FW_INTERFACE_TABLE_OEM_TABLE_ID,     // OemTblId
    NVDIMM_FW_INTERFACE_TABLE_OEM_RIVISION,     // OemRevision
    NVDIMM_FW_INTERFACE_TABLE_OEM_CREATOR_ID,   // CreatorId
    NVDIMM_FW_INTERFACE_TABLE_CREATOR_RIVISION, // CreatorRev
    0,                                          // Reserved

  {
    EFI_ACPI_6_2_NFIT_NVDIMM_REGION_MAPPING_STRUCTURE_TYPE,         // Type
    sizeof (EFI_ACPI_6_2_NFIT_NVDIMM_REGION_MAPPING_STRUCTURE),     // Length
    {0x1},                                                          // NFITDeviceHandle
    17,                                                             // Type17 handle from BIOS boot
    1,                                                              // NVDIMMRegionID
    1,                                                              // SPARangeStructureIndex
    1,                                                              // NVDIMMControlRegionStructureIndex
    0,                                                              // NVDIMMRegionSize
    0,                                                              // RegionOffset
    0,                                                              // NVDIMMPhysicalAddressRegionBase
    0,                                                              // InterleaveStructureIndex
    0x1,                                                            // InterleaveWays
    0x20,                                                           // NVDIMMStateFlags
    0                                                               // Reserved
  },

  {
    EFI_ACPI_6_2_NFIT_NVDIMM_CONTROL_REGION_STRUCTURE_TYPE,                   // Type
    32,                                     // Length
    1,                                    // NVDIMMControlRegionStructureIndex
    0x8086,                                    // VendorID
    0x41c0,                                    // DeviceID
    0x0,                                    // RevisionID
    0x8086,                                    // SubsystemVendorID
    0,                                    // SubsystemDeviceID
    0,                                    // SubsystemRevisionID
    0x1,                                    // ValidFields
    0xa2,                                    // ManufacturingLocation
    0x18,                                    // ManufacturingDate
    {0,0},                                    //  Reserved_22[2]
    0,                                    //  SerialNumber
    0,                                    //  RegionFormatInterfaceCode
    0                                    //  NumberOfBlockControlWindows
  },

  {
    EFI_ACPI_6_1_NFIT_SYSTEM_PHYSICAL_ADDRESS_RANGE_STRUCTURE_TYPE,     // Type
    sizeof (EFI_ACPI_6_1_NFIT_SYSTEM_PHYSICAL_ADDRESS_RANGE_STRUCTURE), // Length
    1,                                                                  // SPARangeStructureIndex
    0x0,                                                                // Flags
    0,                                                                  // Reserved
    0,                                                                  // ProximityDomain
    EFI_ACPI_6_2_NFIT_GUID_BYTE_ADDRESSABLE_PERSISTENT_MEMORY_REGION,   // AddressRangeTypeGUID
    0,                                                                  // SystemPhysicalAddressRangeBase
    0,                                                                  // SystemPhysicalAddressRangeLength
    0                                                                   // AddressRangeMemoryMappingAttribute
  }

  /*{
    EFI_ACPI_6_2_NFIT_PLATFORM_CAPABILITIES_STRUCTURE_TYPE,      // Type
    sizeof (EFI_ACPI_6_2_NFIT_PLATFORM_CAPABILITIES_STRUCTURE),  // Length
    0,                                                           // HighestValidCapability
    0,                                                           //  Rsvd1
    0,                                                           // Rsvd2
    0,                                                           //  Capabilities
    0                                                            //  Rsvd3
  }*/

};

/**
  Update the NFIT table

  @param[in, out] TableHeader         - The table to be set
  @param[in]      TwoLmInfoHob        - Version to publish
**/
VOID
NfitTableUpdate (
  IN       TWOLM_INFO_HOB        *TwoLmInfoHob
  )
{
  EFI_ACPI_NFIT_TABLE *NfitTablePtr;
  NVDIMM_CACHE_HOB    *NvdimmCacheInfoHob;

  NfitTablePtr   = (EFI_ACPI_NFIT_TABLE *) &NfitTable;

  NvdimmCacheInfoHob = (NVDIMM_CACHE_HOB *) GetFirstGuidHob (&gNvdimmCacheInfoHobGuid);


  //
  // Update the dynamic fields
  //
  NfitTablePtr->NfitSpa.SystemPhysicalAddressRangeBase   = TwoLmInfoHob->TwoLmInfo.PMemBase;
  NfitTablePtr->NfitSpa.SystemPhysicalAddressRangeLength = LShiftU64(TwoLmInfoHob->TwoLmInfo.PMemSize, 20); // In bytes
  NfitTablePtr->NfitSpa.AddressRangeMemoryMappingAttribute = (EFI_MEMORY_NV | EFI_MEMORY_WB);

  if (TwoLmInfoHob->TwoLmInfo.PMemSize == 0 || NvdimmCacheInfoHob != NULL) {
    NfitTable.Length = sizeof (EFI_ACPI_NFIT_TABLE) - sizeof (EFI_ACPI_6_2_NFIT_SYSTEM_PHYSICAL_ADDRESS_RANGE_STRUCTURE);
    DEBUG ((DEBUG_INFO, "PMEM size is 0, intializing SPA as zero\n"));
    NfitTablePtr->NfitRms.SPARangeStructureIndex = 0;
  }

  NfitTablePtr->NfitRms.NVDIMMRegionSize           =  LShiftU64(TwoLmInfoHob->TwoLmInfo.PMemSize, 20); // In bytes

  NfitTablePtr->NfitNcrs.VendorID                  =  TwoLmInfoHob->TwoLmInfo.VendorId;
  NfitTablePtr->NfitNcrs.DeviceID                  =  TwoLmInfoHob->TwoLmInfo.DevId;
  NfitTablePtr->NfitNcrs.RevisionID                =  TwoLmInfoHob->TwoLmInfo.RevisionId;
  NfitTablePtr->NfitNcrs.RegionFormatInterfaceCode =  TwoLmInfoHob->TwoLmInfo.Ifc;
  NfitTablePtr->NfitNcrs.SerialNumber              =  TwoLmInfoHob->TwoLmInfo.SerialNum;
}

/**
  2LM Passthrough Command

  @param  OpCode                 Low word contains command OpCode and high word contains command SubOpcode.
  @param  OpCodeDataLength       Length of OpCodeData buffer in bytes.
  @param  OpCodeData             8 bytes of Command register value followed by input data. Max input data size is 512.
  @param  OutputDataLength       Length of OutputDataLength buffer in bytes.
  @param  OutputData             8 bytes of Status register value followed by 512 bytes of output data.

  @retval EFI_SUCCESS            The 2LM Passthrough command executed successfully.
  @retval EFI_INVALID_PARAMETER  Invalid parameters passed to this protocol.
  @retval Others                 Fails to execute the 2LM Passthrough command.
**/
EFI_STATUS
EFIAPI
TwoLmPassThruCommand (
IN         UINT32                   OpCode,
IN         UINT32                   OpCodeDataLength,
IN         VOID                     *OpCodeData,
IN         UINT32                   OutputDataLength,
OUT        VOID                     *OutputData
  )
{
  MAILBOX_CMD MbCmd;
  EFI_STATUS  Status;
  UINT8       MbStatus;
  UINT16      Opcode;
  UINT16      SubOpcode;
  UINT64      StatusRegister;

  //
  // Basic sanity check for parameters
  //
  if (!OutputData || !(OutputDataLength)) {
    DEBUG ((DEBUG_ERROR, "Invalid parameters\n"));
    return EFI_INVALID_PARAMETER;
  }
  if (OutputDataLength < 8) { // 8 bytes
    DEBUG ((DEBUG_ERROR, "Payload size is not enough to hold complete output of this command\n"));
    return EFI_INVALID_PARAMETER;
  }
  if (OpCodeDataLength < 8){
    DEBUG ((DEBUG_ERROR, "Opcode Data Length is insufficient to contain the Command registers\n"));
    return EFI_INVALID_PARAMETER;
  }

  MbCmd.StatusRegister = &StatusRegister;
  MbCmd.BiosNonceValue = 0; // Nonce value is not required
  MbCmd.MbCmd0Reg.Data = ((UINT32 *)OpCodeData)[0];
  MbCmd.MbCmd1Reg.Data = ((UINT32 *)OpCodeData)[1];
  SubOpcode = (OpCode & 0xFFFF0000)>>16;
  Opcode = OpCode & 0x0000FFFF;
  if (Opcode != MbCmd.MbCmd1Reg.Bits.FwMbOpcode ||
      SubOpcode != MbCmd.MbCmd1Reg.Bits.FwMbSubOpcode) {
    DEBUG ((DEBUG_ERROR, "Invalid parameters. Opcode doesn't match the Opcode in the Command Register\n"));
    return EFI_INVALID_PARAMETER;
  }
  MbCmd.OutputPayloadPtr = &(((UINT32 *)OutputData)[2]);
  if (OutputDataLength % sizeof(UINT32) !=0) {
    DEBUG ((DEBUG_ERROR, "2LM: TwoLmPassThruCommand - Output Data Length not DWORD Aligned\n"));
    return EFI_INVALID_PARAMETER;
  } else {
    MbCmd.OutputPayloadSizeDw = (UINT8) ((OutputDataLength-8)/4);
  }
  MbCmd.InputPayloadPtr = &(((UINT32 *)OpCodeData)[2]);
  if (OpCodeDataLength % sizeof(UINT32) !=0) {
    DEBUG ((DEBUG_ERROR, "2LM: TwoLmPassThruCommand - Opcode Data Length not DWORD Aligned\n"));
    return EFI_INVALID_PARAMETER;
  } else {
    MbCmd.InputPayloadSizeDw = (UINT8)((OpCodeDataLength-8)/4);
  }
  MbCmd.DeviceReturnStatus = &MbStatus;

  DEBUG ((DEBUG_INFO, "2LM Passthrough Command to 3DXP\n"));
  Status = IssueMailboxCommand(MbCmd);
  *((UINT64 *)OpCodeData) = StatusRegister;
  return Status;
}


/**
  This function installs Nfit ACPI table

  @param[in] ImageHandle             Handle for the image of this driver
  @param[in] SystemTable             Pointer to the EFI System Table

  @retval EFI_SUCCESS             The function completed successfully
  @retval EFI_OUT_OF_RESOURCES    No enough buffer to allocate
**/
EFI_STATUS
EFIAPI
NfitEntryPoint (
  IN EFI_HANDLE         ImageHandle,
  IN EFI_SYSTEM_TABLE   *SystemTable
  )
{
  EFI_STATUS                      Status;
  UINTN                           AcpiTableHandle;
  EFI_ACPI_TABLE_PROTOCOL         *AcpiTable;
  TWOLM_INFO_HOB                  *TwoLmInfoHob;

  AcpiTable     = NULL;

  if (IsTwoLmEnabled ()== FALSE) {
    return EFI_UNSUPPORTED;
  }

  DEBUG ((DEBUG_INFO, "NfitEntryPoint \n"));

  TwoLmInfoHob = (TWOLM_INFO_HOB *) GetFirstGuidHob (&gTwoLmInfoHobGuid);
  if (TwoLmInfoHob == NULL) {
    DEBUG ((DEBUG_ERROR, "2LM: TwoLmInfoHob not found\n"));
    return EFI_UNSUPPORTED;
  }

  ///
  /// Fix Nfit Table always created, skip install when PMem is not enabled
  ///
  if (TwoLmInfoHob->TwoLmInfo.PMemSize == 0) {
    DEBUG ((DEBUG_INFO, "Nfit Table installed for PMEM size zero\n"));
  }

  TwoLmOptaneInterface.Revision = 0x01;
  TwoLmOptaneInterface.PassthroughCommand = TwoLmPassThruCommand;
  Status = gBS->InstallProtocolInterface (
                &ImageHandle,
                &gTwoLmPassThruProtocolGuid,
                EFI_NATIVE_INTERFACE,
                &TwoLmOptaneInterface
                );

  ///
  /// Locate ACPI support protocol
  ///
  Status = gBS->LocateProtocol (&gEfiAcpiTableProtocolGuid, NULL, (VOID **) &AcpiTable);
  if (EFI_ERROR (Status)) {
    return EFI_NOT_FOUND;
  }

  NfitTableUpdate (TwoLmInfoHob);
  ///
  /// Update the Nfit table in the ACPI tables.
  ///
  AcpiTableHandle = 0;
  if (&NfitTable != NULL) {
    Status = AcpiTable->InstallAcpiTable (
                          AcpiTable,
                          &NfitTable,
                          NfitTable.Length,
                          &AcpiTableHandle
                          );
    DEBUG ((DEBUG_INFO, "2LM: Nfit Table installed\n"));
  }

  return Status;
}
